;;; Copyright 2025
;;; /packaging/guix/ctoolbox.scm is free software; you can redistribute it and/or modify it
;;; under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 3 of the License, or (at
;;; your option) any later version.
;;;
;;; /packaging/guix/ctoolbox.scm is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with /packaging/guix/ctoolbox.scm.  If not, see <http://www.gnu.org/licenses/>.

;; CURRENT STATUS: Blocked by dependencies that require rustc 1.89, while Guix
;; only offers 1.85.1.

(use-modules (system base compile) (srfi srfi-1))

(define-module (ctoolbox)
    #:use-module (guix licenses)
    #:use-module (guix build-system cargo)
    #:use-module (guix build-system copy)
    #:use-module (guix packages)
    #:use-module (guix download)
    #:use-module (guix git-download)
    #:use-module (guix gexp)
    #:use-module (gnu packages)
    #:use-module (ctoolbox-rust-crates)
)

;; Building a package that's hidden (untested): guix build -e '(@ (gnu packages module-name) package-name)'
;; Building a package that's not exported: guix build --sources -e '(@@ (gnu packages rust-crates) rust-whatever-0.1.0)'

(define-public ctoolbox
    (package
        (name "ctoolbox")
        (version "0.1.0")
        (source
            (origin
                ;; FIXME: Kind of a hack. `git archive` puts all files in the
                ;; root directory, so zipbomb wraps it up in a subdirectory.
                (method url-fetch/zipbomb)
                (uri (string-append "file://" (getcwd) "/built/src/src.zip"))
                (sha256
                (base32 "0placeholder1"))
                (modules '((guix build utils)))
                (snippet '(begin
                    (for-each delete-file-recursively '(
                        ".github"
                        ".vscode"
                        "built"
                        "packaging/guix/generated"
                        "docs/eite/eite.js"
                        "docs/eite/implementation/platform-support/web/soccer.css"
                        "docs/eite/implementation/platform-support/web/soccer.otf"
                        "docs/eite/implementation/platform-support/web/soccer.svg"
                        "docs/mockups/Home screen, open, search/index.html"
                        "docs/mockups/Home screen, open, search/assets/apps"
                        "docs/mockups/Home screen, open, search/assets/fonts"
                        "docs/mockups/Home screen, open, search/assets/html5shim"
                        "docs/mockups/Home screen, open, search/assets/jquery-ui-1.9.2.custom"
                        "docs/mockups/Home screen, open, search/assets/json2"
                        "docs/mockups/Home screen, open, search/assets/react"
                        "docs/mockups/Home screen, open, search/assets/scripts.js"
                        "ctoolbox/Cargo.lock"
                    ))
                    (substitute* "ctoolbox/Cargo.toml"
                        (("^rouille .*") "rouille = { features = [ \"rustls\" ], version = \"*\" }\n")
                    )
                    #t
                ))
            )
        )
        (build-system cargo-build-system)
        (native-inputs (list
            (specification->package "unzip")
        ))
        (inputs (cons*
            (specification->package "bash")
            rust-rouille-3.6.2.31f772c
            (cargo-inputs 'ctoolbox #:module '(ctoolbox-rust-crates))
        ))
        (arguments
            (list
                #:phases
                #~(modify-phases %standard-phases
                    (add-before 'build 'pre-build
                        (lambda* (#:key inputs #:allow-other-keys)
                            (display (string-append "file://" (getcwd)))
                            (copy-recursively "guix-vendor/rust-rouille-multipart-0.18.0.31f772c-checkout" "guix-vendor/rouille-multipart-0.18.0")
                            (invoke "bash" "-c" "echo 'rouille-multipart = { path = \"../guix-vendor/rouille-multipart-0.18.0/rouille-multipart\" }'$'\\n''rouille = { path = \"../guix-vendor/rouille-3.6.2\" }' >> ctoolbox/Cargo.toml")
                            (invoke "packaging/pre-build")
                            (copy-file "src.zip" "built/src/src.zip")
                            ;; Guix needs the Rust package to be in the source
                            ;; directory, and the lib.rs expects the source
                            ;; files to be one level up from where the
                            ;; Cargo.toml is located.
                            ;; (invoke "bash" "-c" "mv ./* ../; mv ../ctoolbox/* ./")
                            (chdir "ctoolbox")
                        )
                    )
                )
            )
        )
        (synopsis "Collective Toolbox: tools for egalitarian, compassionate, community-minded computing")
        (description
        "Collective Toolbox is a set of tools designed to enable egalitarian, compassionate, community-minded computing.")
        (home-page "https://www.example.org/")
        (license (list agpl3+ expat silofl1.1))
    )
)

(define-public ctoolbox-vendored
    (hidden-package
        (package
            (name "ctoolbox-vendored")
            (version "0.1.0")
            (source
                (origin
                    (method url-fetch)
                    (uri (string-append "file://" (getcwd) "/built/src/vendor.tar"))
                    (sha256
                    (base32 "0placeholder2"))
                    (modules '((guix build utils)))
                )
            )
            (build-system copy-build-system)
            (synopsis "Collective Toolbox: tools for egalitarian, compassionate, community-minded computing (vendored dependency tarball for rouille)")
            (description
            "Collective Toolbox is a set of tools designed to enable egalitarian, compassionate, and community-minded computing. (vendored dependency tarball for rouille)")
            (home-page "https://www.example.org/")
            (license (list expat asl2.0))
        )
    )
)

(define-public rust-rouille-3.6.2.31f772c
    (
    let (
        (commit "31f772c40007503d25e60e9aba77836a4273ec26")
        (revision "0")
    )
    (hidden-package
        (package
            (name "rust-rouille")
            ;; FIXME: rouille-multipart should be split into a separate package
            ;; see https://codeberg.org/guix/guix/src/commit/6dc9cee1dd51a69f8940215f4db294b0cde02c92/guix/build/cargo-build-system.scm#L79
            (version (git-version "3.6.2" revision commit))
            ;; tiny_http is vendored, so it's not available in rust_crates, even
            ;; though it's depended on by rouille
            (source
                (origin
                    (method git-fetch)
                    (uri (git-reference
                        (url "https://github.com/tomaka/rouille.git")
                        (commit commit)))
                    (file-name (git-file-name name version))
                    ;; Updating hash:
                    ;; git clone https://github.com/tomaka/rouille.git
                    ;; cd rouille
                    ;; git checkout 31f772c40007503d25e60e9aba77836a4273ec26
                    ;; guix hash -x --serializer=nar .
                    (sha256
                    (base32 "138kclylm6g4dpmc2ds45maal7yhcdsrrhblwynx05gpph9swq82"))
                    (modules '((guix build utils)))
                    (snippet '(begin
                        ;; Turn off openssl from rouille tiny_http dependency
                        ;; ctoolbox uses rustls instead
                        (substitute* "Cargo.toml"
                            (("^ssl =.*") "\n"))
                        ;; Remove optional dependencies that cause build to fail
                        ;; due to dependency not being available
                        (substitute* "Cargo.toml"
                            (("^postgres .*") "\n"))
                        (delete-file-recursively "examples/database.rs")
                        (substitute* "rouille-multipart/Cargo.toml"
                            (("^clippy .*") "\n"))
                        (substitute* "rouille-multipart/Cargo.toml"
                            (("^hyper .*") "\n"))
                        (substitute* "rouille-multipart/Cargo.toml"
                            (("^iron .*") "\n"))
                        (substitute* "rouille-multipart/Cargo.toml"
                            (("^nickel .*") "\n"))
                        (substitute* "rouille-multipart/Cargo.toml"
                            (("^rocket .*") "\n"))
                        (substitute* "rouille-multipart/Cargo.toml"
                            (("^env_logger .*") "\n"))
                        (substitute* "rouille-multipart/Cargo.toml"
                            (("^tiny_http .*") "tiny_http = { version = \"0.12\", features = [ \"zeroize\" ], optional = true }\n"))
                        (substitute* "rouille-multipart/Cargo.toml"
                            (("^default.*") "default = [\"client\", \"mock\", \"server\", \"tiny_http\"]\n"))
                        ;; Patch rouille to use vendored tiny http
                        (substitute* "Cargo.toml"
                            (("^tiny_http .*") "tiny_http = { version = \"0.12\", features = [ \"zeroize\" ] }\n"))
                        ;; Append patch for vendored tiny_http after members
                        ;; section
                        (substitute* "Cargo.toml"
                            (("^members .*") "members = [\"rouille-multipart\"]\n\n[patch.crates-io]\ntiny_http = { path = \"vendor/tiny_http\" }\n"))
                        #t
                    ))
                )
            )
            (build-system cargo-build-system)
            (inputs (cons*
                (specification->package "bash")
                ctoolbox-vendored
                (cargo-inputs 'ctoolbox #:module '(ctoolbox-rust-crates))
            ))
            (arguments
                (list
                    #:phases
                    #~(modify-phases %standard-phases
                        (add-before 'build 'pre-build
                            (lambda* (#:key inputs #:allow-other-keys)
                                (display (string-append "file://" (getcwd)))
                                (mkdir-p "vendor")
                                (copy-recursively (string-append (assoc-ref inputs "ctoolbox-vendored") "/tiny_http") "vendor/tiny_http")
                                (copy-recursively "rouille-multipart" "guix-vendor/rouille-multipart")
                                (copy-recursively "vendor/tiny_http" "guix-vendor/tiny_http")
                                (invoke "bash" "-c" "echo '{\"files\":{}}' > guix-vendor/rouille-multipart/.cargo-checksum.json")
                                (invoke "bash" "-c" "echo '{\"files\":{}}' > guix-vendor/tiny_http/.cargo-checksum.json")
                            )
                        )
                    )
                )
            )
            (home-page "https://github.com/tomaka/rouille/")
            (synopsis "Rust web framework Rouille")
            (description "This package provides Rust web framework Rouille.")
            (license (list expat asl2.0 cc-by-sa3.0))
        )
    )
    )
)

ctoolbox
