use crate::cli::{self, Cli, Invocation};
use crate::io::print;
use crate::io::webui::start_webui_server;
use crate::storage::{self, get_storage_dir};
use crate::utilities::logging::setup_logger;
use crate::utilities::resource_lock::check_filesystem_lock_support;
use crate::utilities::*;
use crate::workspace::ipc::auth::capability::{
    CapabilityBundle, CapabilitySet, CapabilityToken, MethodRule, MethodSelector, ServiceName
};
use crate::workspace::ipc::process_manager::{
    ChildHandle, ProcessManager, SpawnParams, TokioProcessManager,
};
use crate::workspace::ipc::router::{ConnectionContext, SimpleRouter};
use crate::workspace::ipc::services::workspace::api::{
    ProcessService, ShutdownTreeRequest, ShutdownTreeResponse,
};
use crate::workspace::ipc::types::{ChildKind, ConnectionId, ProcessId};
use crate::workspace::ipc::IPC_API;
use std::collections::HashMap;
use std::sync::Arc;

pub mod ipc;
// pub mod ipc_old;
use anyhow::{bail, ensure};
use once_cell::sync::OnceCell;

// Some services should be single instance processes: e.g. the workspace itself,
// storage manager, local web server, etc.
// Other services should be multi-instance, e.g. renderer processes, format
// converters, etc.

static PROCESS_MANAGER: OnceCell<Arc<dyn ProcessManager>> = OnceCell::new();

/// Initializes the global process manager singleton.
pub fn set_global_process_manager(
    pm: Arc<dyn ProcessManager>,
) -> anyhow::Result<()> {
    #[cfg(test)]
    {
        if PROCESS_MANAGER.get().is_some() {
            return Ok(());
        }
    }
    PROCESS_MANAGER
        .set(pm)
        .map_err(|_| anyhow::anyhow!("ProcessManager already set"))
}

/// Gets a reference to the global process manager.
pub fn get_global_process_manager() -> anyhow::Result<Arc<dyn ProcessManager>> {
    PROCESS_MANAGER
        .get()
        .cloned()
        .ok_or_else(|| anyhow::anyhow!("ProcessManager not initialized"))
}

pub struct WorkspaceState {
    pub process_manager: Arc<dyn ProcessManager>,
    pub router: Arc<SimpleRouter>,
}

/// Process service implementation for the workspace.
#[derive(Debug)]
struct WorkspaceProcessService {
    process_manager: Arc<dyn ProcessManager>,
}

#[async_trait::async_trait]
impl ProcessService for WorkspaceProcessService {
    async fn shutdown_tree(
        &self,
        request: ShutdownTreeRequest,
    ) -> Result<ShutdownTreeResponse, crate::workspace::ipc::error::Error> {
        log!(
            "Shutdown requested: {:?}",
            request.reason.as_deref().unwrap_or("no reason provided")
        );
        Ok(ShutdownTreeResponse { acknowledged: true })
    }
}

/// Initializes the process manager and router, returning the `WorkspaceState`.
pub async fn initialize_workspace_state(
    invocation: &cli::Invocation,
) -> anyhow::Result<WorkspaceState> {
    let process_manager: Arc<dyn ProcessManager> = TokioProcessManager::new();

    set_global_process_manager(process_manager.clone())?;

    let process_service = Arc::new(WorkspaceProcessService {
        process_manager: process_manager.clone(),
    });

    let router = Arc::new(
        SimpleRouter::new().with_process_service(process_service),
    );

    if !matches!(invocation, cli::Invocation::Subprocess(_)) {
        setup_panic_hooks(&process_manager);
    }

    Ok(WorkspaceState {
        process_manager,
        router,
    })
}

fn setup_panic_hooks(pm: &Arc<dyn ProcessManager>) {
    let pm_clone = pm.clone();
    std::panic::set_hook(Box::new(move |info| {
        eprintln!("Panic occurred: {info}");
        // In a real implementation, we'd terminate all children here
    }));
}

pub async fn entry() -> anyhow::Result<()> {
    check_filesystem_lock_support()?;

    // Parse either a subprocess invocation or a user CLI command set.
    let invocation = cli::parse_invocation()?;

    // If it's a subprocess, initialize and exit early.
    if let cli::Invocation::Subprocess(subproc) = &invocation {
        setup_logger(&invocation)?;
        subprocess_init(&invocation)?;
        return Ok(());
    }
    // At this point we know it's a user invocation.

    let cli = invocation.expect_cli();

    // Try lightweight tools (hex2dec, etc.).
    if let Some(code) = cli::maybe_run_lightweight(cli).await? {
        std::process::exit(code);
    }

    // Proceed with full application boot.
    setup_logger(&invocation)?;
    let state = initialize_workspace_state(&invocation).await?;
    boot(state, cli).await?;
    Ok(())
}

/// Main boot logic for normal application startup.
async fn boot(state: WorkspaceState, args: &Cli) -> Result<()> {
    get_storage_dir().unwrap();

    // Create capability bundle for renderer subprocess
    let renderer_capabilities = create_renderer_capabilities();
    let renderer_params = SpawnParams {
        kind: ChildKind::Renderer,
        program: None,
        args: vec!["renderer".to_string()],
        env: vec![],
        cwd: None,
        capabilities: renderer_capabilities,
    };

/*    -    call_ipc(&localwebui.channel, "start_local_web_ui", "".into()).await?;
-    print(strtovec(
-        &call_ipc(&renderer.channel, "test_echo", "Hello, world!".into())
-            .await?,
-    ));*/

    let renderer = state.process_manager.spawn_child(renderer_params).await?;
    log!("Spawned renderer process: {:?}", renderer.pid);

    // Create capability bundle for IO/webui subprocess
    let io_capabilities = create_io_capabilities();
    let io_params = SpawnParams {
        kind: ChildKind::Io,
        program: None,
        args: vec!["io".to_string()],
        env: vec![],
        cwd: None,
        capabilities: io_capabilities,
    };

    let io_process = state.process_manager.spawn_child(io_params).await?;
    log!("Spawned IO process: {:?}", io_process.pid);

    let doc =
        storage::get_asset("intro.html").expect("Could not load intro.html");
    // let doc = strtovec("0");
    let doc_str = vectostr(&doc);
    print(strtovec(format!("Document: {doc_str}").as_str()));*/
    // let pid = hc_renderer::start(hc_formats::convert_from(doc, strtovec("html")));
    // hc_renderer::start(hc_formats::convert_from(doc, strtovec("html")));*/
    // data_channel_test().await;


    // Main event loop
    start_webui_server().await?;
    loop {
        sleep(10);
    }
}

pub fn subprocess_init(invocation: &Invocation) -> Result<()> {
    // This is run in subprocesses only

    let args = match invocation {
        Invocation::Subprocess(args) => args,
        _ => bail!("Not a subprocess invocation"),
    };

    let service = &args.service_name;

    ensure!(IPC_API.contains(service.as_str()), "Unknown service");
    ensure!("process" != service.as_str(), "Unknown service");

    debug!(format!("Service started: {}", service));
    loop {
        let message = json!({ "type": "poll_for_task" });
        let response = send_message(&channel, message.to_string().as_str());

        let response_type = jq(".type", response.as_str()).unwrap();

        if response_type == "new_task" {
            let new_task_method = jq(".method", response.as_str()).unwrap();
            let new_task_args =
                get_as_json_string_from_json_string(response.as_str(), "args")?;
            let msgid = jq(".msgid", response.as_str())
                .unwrap()
                .parse::<u64>()
                .unwrap();
            let channel_copy =
                channel_from_json_string(json!(channel).as_str()).unwrap();
            std::thread::spawn({
                let service = service.clone();
                let args = new_task_args.clone();
                move || {
                    dispatch_call(
                        service.as_str(),
                        &channel_copy,
                        msgid,
                        new_task_method.as_str(),
                        if let Some(args) = args {
                            args.clone()
                        } else {
                            "null".to_string()
                        }
                        .as_str(),
                    );
                }
            });
        } else if response_type == "no_new_tasks" {
            usleep(100000);
        } else {
            log!(
                format!("Received unexpected IPC response: {response}")
                    .as_str()
            );
            usleep(100000);
        }
    }
}


/// Create capability bundle for renderer processes.
fn create_renderer_capabilities() -> CapabilityBundle {
    let mut allowed: HashMap<ServiceName, Vec<MethodRule>> = HashMap::new();

    // Renderers can request process shutdown
    allowed.insert(
        ServiceName("process".to_string()),
        vec![MethodRule {
            method: MethodSelector::Exact("shutdown_tree".into()),
            quotas: None,
        }],
    );

    CapabilityBundle {
        capabilities: CapabilitySet {
            allowed,
            global_limits: None,
        },
        token: CapabilityToken("TODO".to_string()),
    }
}

/// Create capability bundle for IO/webui processes.
fn create_io_capabilities() -> CapabilityBundle {
    let mut allowed: HashMap<ServiceName, Vec<MethodRule>> = HashMap::new();

    // IO can request process shutdown
    allowed.insert(
        ServiceName("process".to_string()),
        vec![MethodRule {
            method: MethodSelector::Exact("shutdown_tree".into()),
            quotas: None,
        }],
    );

    // IO can access network service
    allowed.insert(
        ServiceName("network".to_string()),
        vec![MethodRule {
            method: MethodSelector::Any,
            quotas: None,
        }],
    );

    CapabilityBundle {
        capabilities: CapabilitySet {
            allowed,
            global_limits: None,
        },
        token: CapabilityToken("TODO".to_string()),
    }
}

pub fn workspace_restart(pm: &Arc<dyn ProcessManager>) {
    std::process::exit(0);
}

pub fn workspace_shutdown(pm: &Arc<dyn ProcessManager>) {
    std::process::exit(0);
}

/// Boot logic for tests: sets up process manager and router.
fn impl_boot_for_test() -> Result<Arc<dyn ProcessManager>> {
    setup_logger(&cli::Invocation::User(Cli {
        ctoolbox_ipc_port: None,
        command: None,
    }))?;

    let process_manager: Arc<dyn ProcessManager> = TokioProcessManager::new();

    set_global_process_manager(process_manager.clone())?;

    Ok(process_manager)
}

pub struct TestHarness {
    pub pm: Arc<dyn ProcessManager>,
}

static HARNESS: OnceCell<TestHarness> = OnceCell::new();

impl TestHarness {
    #[tracing::instrument]
    /// Attempts to initialize the `TestHarness`, returning a result.
    pub fn try_init() -> Result<&'static TestHarness> {
        if let Some(harness) = HARNESS.get() {
            return Ok(harness);
        }

        let pm = impl_boot_for_test();
        match pm {
            Ok(pm_arc) => Ok(HARNESS.get_or_init(|| TestHarness { pm: pm_arc })),
            Err(e) => Err(e),
        }
    }
}

// Not meant to be used directly - use #[crate::ctb_test] instead.
pub fn boot_for_test() -> Option<&'static TestHarness> {
    if let Ok(harness) = TestHarness::try_init() {
        Some(harness)
    } else {
        None
    }
}
