use crate::utilities::ipc::{Channel, channel_from_json_string};
use crate::utilities::json::jq;
use crate::utilities::process::ProcessManager;
use crate::utilities::{Context, Result, json};
use crate::workspace::ipc_old::dispatch::ipc_call_method;
use clap::Parser;
use serde::{Deserialize, Serialize};
use std::collections::HashMap;
use std::sync::Arc;
use std::sync::Mutex;
use tiny_http::{Response, Server};

#[derive(PartialEq, Clone, Serialize, Deserialize)]
enum CallState {
    Created,
    Claimed,
    Success,
    Error,
}

#[derive(Clone, Serialize, Deserialize)]
struct Call {
    method: String,
    args: String,
    response: Option<String>,
    state: CallState,
}

#[derive(Clone, Serialize, Deserialize)]
struct CallWithId {
    id: u64,
    call: Call,
}

struct Calls {
    calls: HashMap<u64, Call>,
    last_id: u64,
}

impl Calls {
    fn new() -> Calls {
        Calls {
            calls: HashMap::new(),
            last_id: 0,
        }
    }
    fn get(&mut self, id: &u64) -> &mut Call {
        self.calls.get_mut(id).unwrap()
    }
}

pub fn start_ipc_server(
    process_manager: Arc<Mutex<ProcessManager>>,
    port: u16,
    authentication_key: String,
) {
    // log!("Starting IPC server");

    let mut queued_calls: HashMap<String, Calls> = HashMap::new();

    let mut known_channels: Vec<Channel> = vec![Channel {
        name: "ipc_control".to_string(),
        port,
        authentication_key: authentication_key.clone(),
    }];

    let server = Server::http(format!("127.0.0.1:{port}")).unwrap();

    for request in server.incoming_requests() {
        handle_request(
            process_manager.clone(),
            request,
            &mut known_channels,
            &mut queued_calls,
        );
    }
}

fn process_request(
    process_manager: Arc<Mutex<ProcessManager>>,
    target_channel: Channel,
    content: String,
    known_channels: &mut Vec<Channel>,
    queued_calls: &mut HashMap<String, Calls>,
) -> Result<String> {
    let provided_key = target_channel.authentication_key.clone();

    let expected_key: String = known_channels
        .iter()
        .find(|c| c.name == target_channel.name)
        .map_or(String::new(), |c| c.authentication_key.clone());

    if expected_key != provided_key {
        return Err(anyhow::anyhow!("Invalid authentication key"));
    }

    let channel_calls: &mut Calls = queued_calls
        .entry(target_channel.name.clone())
        .or_insert(Calls::new());

    let message_type = jq(".type", content.as_str()).unwrap_or_default();

    let mut response = None;

    // log!(format!("Received message: {}", content).as_str());

    if message_type == "workspace_call" {
        let new_task_method = jq(".method", content.as_str()).unwrap();
        // get Serde value from jq .args
        let new_task_args = jq(".args", content.as_str()).unwrap();
        let task_args_val =
            serde_json::from_str::<serde_json::Value>(new_task_args.as_str())
                .with_context(|| "Could not parse args as JSON")?;
        ipc_call_method(
            &new_task_method,
            &task_args_val,
            Some(process_manager),
        );
        response = Some(json!({"type": "success"}).to_string());
    } else if message_type == "add_channel" {
        let channel: Channel = serde_json::from_str::<Channel>(
            jq(".channel", content.as_str())
                .with_context(|| "Could not get channel from body json")?
                .as_str(),
        )
        .with_context(|| "Could not parse channel from body json")?;
        known_channels.push(channel);
        response = Some(json!({"type": "success"}).to_string());
    } else if message_type == "remove_channel" {
        let name = jq(".channel.name", content.as_str())
            .with_context(|| "Could not get channel from json")?;
        let name = name.as_str();
        known_channels.retain(|c| c.name != name);
        response = Some(json!({"type": "success"}).to_string());
    } else if message_type == "ipc_ping" {
        response = Some(json!({"type": "ipc_ready"}).to_string());
    } else if message_type == "poll_for_result" {
        let msgid = jq(".msgid", content.as_str())
            .with_context(|| "Could not jq msgid")?;
        if msgid == "null" {
            return Err(anyhow::anyhow!("No msgid provided"));
        }
        let msgid = msgid
            .parse::<u64>()
            .with_context(|| "Could not parse msgid")?;
        let message = &channel_calls.get(&msgid);
        if message.state == CallState::Success {
            response = Some(
                json!({"type": "result", "content": message.response})
                    .to_string(),
            );
            channel_calls.calls.remove(&msgid);
        } else if message.state == CallState::Error {
            response = Some(json!({"type": "error"}).to_string());
        } else {
            response = Some(json!({"type": "pending"}).to_string());
        }
    } else if message_type == "poll_for_task" {
        response = Some(json!({"type": "no_new_tasks"}));
        for (msgid, call) in &mut channel_calls.calls {
            if call.state == CallState::Created {
                call.state = CallState::Claimed;
                response = Some(
                    json!({
                        "type": "new_task",
                        "method": call.method,
                        "args": call.args,
                        "msgid": msgid,
                    })
                    .to_string(),
                );
                break;
            }
        }
    } else if message_type == "call" {
        let method = jq(".method", content.as_str())
            .with_context(|| "Could not query method")?;
        let args = jq(".args", content.as_str())
            .with_context(|| "Could not query args")?;
        let msgid = channel_calls.last_id + 1;
        channel_calls.calls.insert(
            msgid,
            Call {
                method: method.clone(),
                args: args.clone(),
                response: None,
                state: CallState::Created,
            },
        );
        channel_calls.last_id = msgid;
        response =
            Some(json!({"type": "call_pending", "msgid": msgid}).to_string());
    } else if message_type == "response" {
        let msgid: u64 = jq(".msgid", content.as_str())
            .with_context(|| "Could not jq msgid")?
            .parse::<u64>()
            .with_context(|| "Could not parse msgid")?;

        let content = jq(".content", content.as_str())
            .with_context(|| "Could not jq content")?;
        channel_calls.get(&msgid).response = Some(content.clone());
        channel_calls.get(&msgid).state = CallState::Success;
        response = Some(json!({"type": "success"}).to_string());
    }

    response.with_context(|| format!("Invalid message type {message_type}"))
}

fn error_response(request: tiny_http::Request, message: &str) {
    match request.respond(Response::from_string(
        json!({"type": "error", "message": message}).to_string(),
    )) {
        Ok(()) => {}
        Err(e) => {
            eprintln!("Failed to send error response: {e}");
        }
    }
}

fn handle_request(
    process_manager: Arc<Mutex<ProcessManager>>,
    mut request: tiny_http::Request,
    known_channels: &mut Vec<Channel>,
    queued_calls: &mut HashMap<String, Calls>,
) {
    let target_channel = channel_from_json_string(
        request
            .headers()
            .iter()
            .find(|h| h.field.equiv("X-CollectiveToolbox-IPCAuth"))
            .map_or_else(
                || "No channel provided".to_string(),
                |h| h.value.clone().to_string(),
            )
            .as_str(),
    )
    .context("Invalid channel header");

    if target_channel.is_err() {
        error_response(request, "Invalid channel");
        return;
    }

    let mut content = String::new();
    if request
        .as_reader()
        .read_to_string(&mut content)
        .context("Could not read request body")
        .is_err()
    {
        error_response(request, "Could not read request body");
        return;
    }

    let response = process_request(
        process_manager,
        target_channel.unwrap(),
        content,
        known_channels,
        queued_calls,
    );

    let response = match response {
        Ok(r) => r,
        Err(e) => {
            json!({"type": "error", "message": e.to_string()}).to_string()
        }
    };

    request.respond(Response::from_string(response)).unwrap();
}
