//! IPC dispatch logic for workspace services.

use crate::formats::base64::bytes_to_standard_base64;
use crate::utilities::ipc::Channel;
use crate::utilities::process::ProcessManager;
use crate::utilities::serde_value::{get_as_bytes, get_as_u64};
use crate::utilities::{json, send_message};
use crate::workspace::ipc_old::IPC_API;
use crate::{json_value, renderer};
use serde_json::Value;
use std::sync::Arc;
use std::sync::Mutex;

mod db;

pub fn dispatch_call(
    service_name: &str,
    to_channel: &Channel,
    msgid: u64,
    method: &str,
    args: &str,
) {
    if IPC_API.contains_key(service_name) {
        let Some(allowed_methods) = IPC_API.get(service_name) else {
            let response = json!({
                "type": "response",
                "msgid": msgid,
                "error": format!("Service {service_name} not allowed"),
            });
            send_message(to_channel, &response.to_string());
            return;
        };
        if allowed_methods.contains(&method) {
            // Parse args as serde_json::Value
            let args_value: Value = match serde_json::from_str(args) {
                Ok(val) => val,
                Err(e) => {
                    let response = json!({
                        "type": "response",
                        "msgid": msgid,
                        "error": format!("Invalid args: {}", e),
                    });
                    send_message(to_channel, &response.to_string());
                    return;
                }
            };

            let result = ipc_call_method(method, &args_value, None);

            let response = json!({
                "type": "response",
                "msgid": msgid,
                "content": result,
            });
            send_message(to_channel, &response.to_string());

            return;
        }
    }
    let response = json!({
        "type": "response",
        "msgid": msgid,
        "error": format!("Method {method} not allowed or not implemented for {service_name}"),
    });
    send_message(to_channel, &response.to_string());
}

/// Dispatches an IPC method call with structured arguments.
#[allow(clippy::too_many_lines)]
pub fn ipc_call_method(
    method: &str,
    args: &Value,
    process_manager: Option<Arc<Mutex<ProcessManager>>>,
) -> Value {
    match method {
        // IO
        "start_local_web_ui" => {
            json_value!({ "value" => crate::io::webui::start_webui_server() })
        }

        // Database operations
        "get_str_u64" => {
            let db = args.get("db").and_then(Value::as_str).unwrap_or_default();
            let key =
                args.get("key").and_then(Value::as_str).unwrap_or_default();
            if let Some(val) = db::get_str_u64(db, key) {
                json_value!({ "value" => val })
            } else {
                json_value!({ "none" => true })
            }
        }
        "get_all_u64_keys" => {
            let db = args.get("db").and_then(Value::as_str).unwrap_or_default();
            match db::get_all_u64_keys(db) {
                Ok(val) => json_value!({ "value" => val }),
                Err(e) => {
                    json_value!({ "error" => format!("{}: {:?}", e.to_string(), e) })
                }
            }
        }
        "put_str_u64" => {
            let db = args.get("db").and_then(Value::as_str).unwrap_or_default();
            let key =
                args.get("key").and_then(Value::as_str).unwrap_or_default();
            let value = args.get("value").and_then(Value::as_u64).unwrap_or(0);
            match db::put_str_u64(db, key, value) {
                Ok(()) => json_value!({ "value" => "ok" }),
                Err(e) => {
                    json_value!({ "error" => format!("{}: {:?}", e.to_string(), e) })
                }
            }
        }
        "delete_str_u64" => {
            let db = args.get("db").and_then(Value::as_str).unwrap_or_default();
            let key =
                args.get("key").and_then(Value::as_str).unwrap_or_default();
            match db::delete_str_u64(db, key) {
                Ok(()) => json_value!({ "value" => "ok" }),
                Err(e) => {
                    json_value!({ "error" => format!("{}: {:?}", e.to_string(), e) })
                }
            }
        }
        "get_u64_str" => {
            let db = args.get("db").and_then(Value::as_str).unwrap_or_default();
            let key = args.get("key").and_then(Value::as_u64).unwrap_or(0);
            if let Some(val) = db::get_u64_str(db, key) {
                json_value!({ "value" => val })
            } else {
                json_value!({ "none" => true })
            }
        }
        "put_u64_str" => {
            let db = args.get("db").and_then(Value::as_str).unwrap_or_default();
            let key = args.get("key").and_then(Value::as_u64).unwrap_or(0);
            let value = args
                .get("value")
                .and_then(Value::as_str)
                .unwrap_or_default();
            match db::put_u64_str(db, key, value) {
                Ok(()) => json_value!({ "value" => "ok" }),
                Err(e) => {
                    json_value!({ "error" => format!("{}: {:?}", e.to_string(), e) })
                }
            }
        }
        "delete_u64_str" => {
            let db = args.get("db").and_then(Value::as_str).unwrap_or_default();
            let key = args.get("key").and_then(Value::as_u64).unwrap_or(0);
            match db::delete_u64_str(db, key) {
                Ok(()) => json_value!({ "value" => "ok" }),
                Err(e) => {
                    json_value!({ "error" => format!("{}: {:?}", e.to_string(), e) })
                }
            }
        }
        "get_u64_bytes" => {
            let db = args.get("db").and_then(Value::as_str).unwrap_or_default();
            let key = args.get("key").and_then(Value::as_u64).unwrap_or(0);
            if let Some(val) = db::get_u64_bytes(db, key) {
                json_value!({ "value" => bytes_to_standard_base64(&val) })
            } else {
                json_value!({ "none" => true })
            }
        }
        "put_u64_bytes" => {
            let db = args.get("db").and_then(Value::as_str).unwrap_or_default();
            let key = get_as_u64(args, "key").unwrap_or(0);
            let value = get_as_bytes(args, "value").unwrap_or_default();
            match db::put_u64_bytes(db, key, &value) {
                Ok(()) => json_value!({ "value" => "ok" }),
                Err(e) => {
                    json_value!({ "error" => format!("{}: {:?}", e.to_string(), e) })
                }
            }
        }
        "delete_u64_bytes" => {
            let db = args.get("db").and_then(Value::as_str).unwrap_or_default();
            let key = args.get("key").and_then(Value::as_u64).unwrap_or(0);
            match db::delete_u64_bytes(db, key) {
                Ok(()) => json_value!({ "value" => "ok" }),
                Err(e) => {
                    json_value!({ "error" => format!("{}: {:?}", e.to_string(), e) })
                }
            }
        }

        // Renderer
        "test_echo" => {
            json_value!({ "value" => renderer::test_echo(args.to_string().as_str()) })
        }
        // Workspace calls are handled by the server, not the usual IPC path
        "workspace_restart" => {
            if let Some(pm) = process_manager {
                crate::workspace::workspace_restart(&pm);
                json_value!({ "value" => "" })
            } else {
                json_value!({ "error" => "process manager required" })
            }
        }
        "workspace_shutdown" => {
            if let Some(pm) = process_manager {
                crate::workspace::workspace_shutdown(&pm);
                json_value!({ "value" => "" })
            } else {
                json_value!({ "error" => "process manager required" })
            }
        }
        &_ => {
            json_value!({ "error" => format!("Method {method} not allowed or not implemented") })
        }
    }
}
