use crate::cli::Invocation;
use crate::utilities::ipc::{Channel, channel_from_json_string};
use crate::utilities::json::{jq, jqq};
use crate::utilities::serde_value::get_as_json_string_from_json_string;
use crate::utilities::*;
use crate::workspace::ipc_old::dispatch::dispatch_call;
use crate::workspace::{get_global_process_manager, initialize_process_state};
use anyhow::{anyhow, bail, ensure};
use clap::Parser;
use serde_json::Value;
use std::collections::HashMap;
use std::error::Error;
use std::sync::LazyLock;

pub mod dispatch;
pub mod server;

pub static IPC_API: LazyLock<HashMap<String, Vec<&str>>> =
    LazyLock::new(|| {
        HashMap::from([
            ("formats".to_string(), vec!["convert_if_needed"]),
            (
                "io".to_string(),
                vec!["print", "poll_hid", "start_local_web_ui"],
            ),
            ("network".to_string(), vec!["get_asset", "get_url"]),
            (
                "renderer".to_string(),
                vec!["test_echo", "start", "get_frame"],
            ),
            ("storage".to_string(), vec!["set", "get"]),
            (
                "workspace".to_string(),
                vec![
                    "pc_restart",
                    "pc_shutdown",
                    "workspace_restart",
                    "workspace_shutdown",
                ],
            ),
        ])
    });

pub fn resolve_workspace_channel() -> Result<Channel> {
    let pm = get_global_process_manager()?;
    let pm = pm.lock();
    if pm.is_err() {
        bail!("Failed to lock process manager");
    }
    let pm = pm.unwrap();
    let channel = pm.workspace_channel()?;
    Ok(channel)
}

pub fn subprocess_init(invocation: &Invocation) -> Result<()> {
    // This is run in subprocesses only

    initialize_process_state(invocation)?;

    let args = match invocation {
        Invocation::Subprocess(args) => args,
        _ => bail!("Not a subprocess invocation"),
    };

    let channel = resolve_workspace_channel()?;

    let service = &args.service_name;

    ensure!(IPC_API.contains_key(service), "Unknown service");

    debug!(format!("Service started: {}", service));
    loop {
        let message = json!({ "type": "poll_for_task" });
        let response = send_message(&channel, message.to_string().as_str());

        let response_type = jq(".type", response.as_str()).unwrap();

        if response_type == "new_task" {
            let new_task_method = jq(".method", response.as_str()).unwrap();
            let new_task_args =
                get_as_json_string_from_json_string(response.as_str(), "args")?;
            let msgid = jq(".msgid", response.as_str())
                .unwrap()
                .parse::<u64>()
                .unwrap();
            let channel_copy =
                channel_from_json_string(json!(channel).as_str()).unwrap();
            std::thread::spawn({
                let service = service.clone();
                let args = new_task_args.clone();
                move || {
                    dispatch_call(
                        service.as_str(),
                        &channel_copy,
                        msgid,
                        new_task_method.as_str(),
                        if let Some(args) = args {
                            args.clone()
                        } else {
                            "null".to_string()
                        }
                        .as_str(),
                    );
                }
            });
        } else if response_type == "no_new_tasks" {
            usleep(100000);
        } else {
            log!(
                format!("Received unexpected IPC response: {response}")
                    .as_str()
            );
            usleep(100000);
        }
    }
}

pub fn send_message(channel: &Channel, message: &str) -> String {
    maybe_send_message(channel, message).expect("Failed to send IPC message")
}

pub fn maybe_send_message(
    channel: &Channel,
    message: &str,
) -> Result<String, Box<dyn Error>> {
    let message_type = jqq(".type", message).unwrap_or_default();
    if message_type != "add_channel"
        && message_type != "poll_for_task"
        && message_type != "poll_for_result"
    {
        log!(
            format!("Sending message to channel {}: {}", channel.name, message)
                .as_str()
        );
    }

    let response = ureq::post(
        format!("http://127.0.0.1:{}/hc_ipc", channel.port).as_str(),
    )
    .header("X-CollectiveToolbox-IPCAuth", json!(channel))
    .send(message.as_bytes())?
    .body_mut()
    .read_to_string()?;
    Ok(response)
}

pub fn wait_for_message_sync(channel: &Channel, msgid: u64) -> String {
    loop {
        let message = json!({ "type": "poll_for_result", "msgid": msgid });
        let response = send_message(channel, message.to_string().as_str());
        let response_type = jq(".type", response.as_str()).unwrap();

        if response_type == "result" {
            return jq(".content", response.as_str()).unwrap();
        } else if response_type == "pending" {
            usleep(100000);
        } else {
            log!(
                format!("Received unexpected IPC response: {response}")
                    .as_str()
            );
            usleep(100000);
        }
    }
}

pub fn listen_for_message(channel: &Channel, msgid: u64) -> String {
    wait_for_message_sync(channel, msgid)
}

pub fn call_ipc_sync(
    channel: &Channel,
    method: &str,
    args: Value,
) -> Result<String> {
    #[cfg(test)]
    {
        // In test mode, run IPC calls in-process.
        // You may want to mock or stub the result for test mode.

        use crate::workspace::get_global_process_manager;
        use crate::workspace::ipc_old::dispatch::ipc_call_method;
        let pm = get_global_process_manager()?;
        let result = ipc_call_method(method, &args, Some(pm));
        return Ok(json!(result).to_string());
    }

    let message = json!({ "type": "call", "method": method, "args": args });
    let response = send_message(channel, &message.to_string());
    let response_type = jqq(".type", &response).unwrap_or_default();

    if response_type != "call_pending" {
        return Err(anyhow!("Unexpected IPC response: {response}"));
    }

    let msgid = jq(".msgid", &response)
        .map_err(|e| {
            anyhow!("Failed to get msgid from response: {response}, error: {e}")
        })?
        .parse::<u64>()?;

    Ok(wait_for_message_sync(channel, msgid))
}

pub async fn wait_for_message(channel: &Channel, msgid: u64) -> String {
    loop {
        let message = json!({ "type": "poll_for_result", "msgid": msgid });
        let response = send_message(channel, message.to_string().as_str());
        let response_type = jq(".type", response.as_str()).unwrap();

        if response_type == "result" {
            return jq(".content", response.as_str()).unwrap();
        } else if response_type == "pending" {
            tokio::time::sleep(std::time::Duration::from_micros(100_000)).await;
        } else {
            log!(
                format!("Received unexpected IPC response: {response}")
                    .as_str()
            );
            tokio::time::sleep(std::time::Duration::from_micros(100_000)).await;
        }
    }
}

pub async fn call_ipc(
    channel: &Channel,
    method: &str,
    args: Value,
) -> Result<String> {
    #[cfg(test)]
    {
        use crate::workspace::get_global_process_manager;
        use crate::workspace::ipc_old::dispatch::ipc_call_method;

        let pm = get_global_process_manager()?;
        let result = ipc_call_method(method, &args, Some(pm));
        return Ok(json!(result).to_string());
    }

    let message = json!({ "type": "call", "method": method, "args": args });
    let response = send_message(channel, &message.to_string());
    let response_type = jqq(".type", &response).unwrap_or_default();

    if response_type != "call_pending" {
        return Err(anyhow!("Unexpected IPC response: {response}"));
    }

    let msgid = jq(".msgid", &response)
        .map_err(|e| {
            anyhow!("Failed to get msgid from response: {response}, error: {e}")
        })?
        .parse::<u64>()?;

    Ok(wait_for_message(channel, msgid).await)
}
