//! Process service API definitions.
//!
//! This service provides operations that any process can handle, such as
//! graceful shutdown of its process tree.

use anyhow::Result;
use async_trait::async_trait;
use serde::{Deserialize, Serialize};

/// Service name for process-level operations.
pub const SERVICE_NAME: &str = "process";

/// Method name for `shutdown_tree`.
pub const METHOD_SHUTDOWN_TREE: &str = "shutdown_tree";

/// Request to shutdown a process and its children.
#[derive(Debug, Clone, Default, Serialize, Deserialize)]
pub struct ShutdownTreeRequest {
    /// Optional reason for shutdown (for logging/audit).
    pub reason: Option<String>,
}

/// Response from `shutdown_tree`.
#[derive(Debug, Clone, Default, Serialize, Deserialize)]
pub struct ShutdownTreeResponse {
    /// Whether the shutdown was acknowledged.
    pub acknowledged: bool,
}

/// Trait for process-level service operations.
///
/// Any process that participates in the IPC system should implement this
/// trait to handle graceful shutdown requests.
#[async_trait]
pub trait ProcessService: Send + Sync + std::fmt::Debug {
    /// Request graceful shutdown of this process and its children.
    async fn shutdown_tree(
        &self,
        request: ShutdownTreeRequest,
    ) -> Result<ShutdownTreeResponse, crate::workspace::ipc::error::Error>;
}

#[derive(Debug)]
pub struct MockProcessService;

#[async_trait]
impl ProcessService for MockProcessService {
    async fn shutdown_tree(
        &self,
        _request: ShutdownTreeRequest,
    ) -> Result<ShutdownTreeResponse, crate::workspace::ipc::error::Error> {
        Ok(ShutdownTreeResponse { acknowledged: true })
    }
}
