//! Thin IPC-facing Network service adapter.
//!
//! This module defines a minimal `NetworkService` trait and postcard-encoded
//! request/response types for the "network" IPC service. Implementations should
//! delegate to `crate::network::NetworkBackend` (or similar), keeping business
//! logic out of IPC.

use anyhow::Result;
use async_trait::async_trait;
use serde::{Deserialize, Serialize};

use crate::network::NetworkBackend;

/// Service name used for routing and authorization.
pub const SERVICE_NAME: &str = "network";

/// Method identifiers supported by the network service.
pub const METHOD_FETCH: &str = "fetch";
pub const METHOD_READ_FILE: &str = "read_file";

/// Request payloads (postcard-encoded).
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct FetchRequest {
    pub url: String,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct ReadFileRequest {
    pub path: String,
}

/// Response payloads (postcard-encoded).
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct BytesResponse {
    /// Raw bytes result.
    pub bytes: Vec<u8>,
}

/// IPC-facing network service trait. Methods mirror `crate::network` APIs.
///
#[async_trait]
pub trait NetworkService: Send + Sync + std::fmt::Debug {
    /// Fetch bytes from a URL (http/https).
    ///
    /// Delegates to the underlying networking backend.
    async fn fetch(&self, req: FetchRequest) -> Result<BytesResponse>;

    /// Read a local file into memory.
    ///
    /// Delegates to the underlying networking backend.
    async fn read_file(&self, req: ReadFileRequest) -> Result<BytesResponse>;
}

/// Default network service using `DefaultNetworkBackend`.
#[derive(Debug, Default)]
pub struct DefaultNetworkService {
    backend: crate::network::DefaultNetworkBackend,
}

impl DefaultNetworkService {
    /// Create a new default network service.
    pub fn new() -> Self {
        Self::default()
    }
}

#[async_trait]
impl NetworkService for DefaultNetworkService {
    async fn fetch(&self, req: FetchRequest) -> Result<BytesResponse> {
        let bytes = self.backend.fetch(req.url).await?;
        Ok(BytesResponse { bytes })
    }

    async fn read_file(&self, req: ReadFileRequest) -> Result<BytesResponse> {
        let bytes = self.backend.read_file(req.path).await?;
        Ok(BytesResponse { bytes })
    }
}

/// Mock network service using `MockNetworkBackend`.
#[derive(Debug, Default)]
pub struct MockNetworkService {
    backend: crate::network::MockNetworkBackend,
}

impl MockNetworkService {
    /// Create a new mock network service.
    pub fn new() -> Self {
        Self::default()
    }
}

#[async_trait]
impl NetworkService for MockNetworkService {
    async fn fetch(&self, req: FetchRequest) -> Result<BytesResponse> {
        let bytes = self.backend.fetch(req.url).await?;
        Ok(BytesResponse { bytes })
    }

    async fn read_file(&self, req: ReadFileRequest) -> Result<BytesResponse> {
        let bytes = self.backend.read_file(req.path).await?;
        Ok(BytesResponse { bytes })
    }
}
