use crate::workspace::ipc::auth::capability::CapabilitySet;
use crate::workspace::ipc::error::Error;
use crate::workspace::ipc::process_manager::ProcessManager;
use crate::workspace::ipc::protocol::{
    Event, MethodId, Request, Response, RpcError,
};
use crate::workspace::ipc::services::network::api::{
    METHOD_FETCH, METHOD_READ_FILE, NetworkService,
    SERVICE_NAME as NETWORK_SERVICE_NAME,
};
use crate::workspace::ipc::services::process::api::{
    METHOD_SHUTDOWN_TREE, ProcessService, SERVICE_NAME as PROCESS_SERVICE_NAME,
    ShutdownTreeRequest,
};
use crate::workspace::ipc::types::{ConnectionId, ProcessId};
use async_trait::async_trait;
use std::collections::HashMap;
use std::sync::{Arc, Mutex};
use std::time::Duration;
use tokio::time::{self, Instant};

use tracing::Instrument as _;

/// Key for rate limiting (connection, service, method).
#[derive(Clone, Debug, Eq, PartialEq, Hash)]
struct RateKey {
    conn_id: ConnectionId,
    service: String,
    method: String,
}

impl RateKey {
    fn bytes(conn_id: ConnectionId, service: String, method: String) -> Self {
        Self {
            conn_id,
            service,
            method,
        }
    }
}

/// Simple token bucket for rate limiting.
#[derive(Debug)]
struct TokenBucket {
    capacity: u128,
    tokens: u128,
    last_refill: Instant,
    rate_bytes_per_sec: u64,
}

impl TokenBucket {
    fn new(rate: u64, capacity: u64, now: Instant) -> Self {
        Self {
            capacity: u128::from(capacity),
            tokens: u128::from(capacity),
            last_refill: now,
            rate_bytes_per_sec: rate,
        }
    }

    fn try_take(&mut self, amount: u64, now: Instant) -> bool {
        let elapsed = now.duration_since(self.last_refill);
        let elapsed_ns = elapsed.as_nanos();
        let added_tokens =
            (elapsed_ns * u128::from(self.rate_bytes_per_sec)) / 1_000_000_000;
        self.tokens = (self.tokens + added_tokens).min(self.capacity);
        self.last_refill = now;

        let amount_u128 = u128::from(amount);
        if self.tokens >= amount_u128 {
            self.tokens -= amount_u128;
            true
        } else {
            false
        }
    }
}

/// Request-scoped cancellation state for cooperative cancellation.
///
/// Services can query cancellation via [`is_cancelled`]. Cancellation is
/// intentionally best-effort and requires cooperative checks.
#[derive(Debug, Clone)]
pub struct RequestCancellation {
    cancelled: std::sync::Arc<std::sync::atomic::AtomicBool>,
}

impl RequestCancellation {
    /// Create a new, non-cancelled token.
    pub fn new() -> Self {
        Self {
            cancelled: std::sync::Arc::new(std::sync::atomic::AtomicBool::new(
                false,
            )),
        }
    }

    /// Mark this token as cancelled.
    pub fn cancel(&self) {
        self.cancelled
            .store(true, std::sync::atomic::Ordering::SeqCst);
    }

    /// Whether cancellation has been requested.
    pub fn is_cancelled(&self) -> bool {
        self.cancelled.load(std::sync::atomic::Ordering::SeqCst)
    }
}

impl Default for RequestCancellation {
    fn default() -> Self {
        Self::new()
    }
}

#[derive(Debug, Clone)]
struct RequestContext {
    cancellation: RequestCancellation,
}

tokio::task_local! {
    static IPC_REQUEST_CONTEXT: RequestContext;
}

/// Run `fut` with a request cancellation context installed.
pub async fn scope_request_cancellation<T>(
    cancellation: RequestCancellation,
    fut: impl std::future::Future<Output = T>,
) -> T {
    IPC_REQUEST_CONTEXT
        .scope(RequestContext { cancellation }, fut)
        .await
}

/// Returns true if the current task is executing within an IPC request and has
/// been cancelled.
pub fn is_cancelled() -> bool {
    IPC_REQUEST_CONTEXT
        .try_with(|ctx| ctx.cancellation.is_cancelled())
        .unwrap_or(false)
}

/// Central router interface responsible for:
/// - binding connections to capability sets
/// - enforcing authorization
/// - dispatching requests to services
/// - emitting events
#[async_trait]
pub trait Router: Send + Sync {
    /// Register a new connection after handshake.
    async fn register_connection(
        &self,
        ctx: ConnectionContext,
    ) -> Result<(), Error>;

    /// Resolve and dispatch a request to a target service method.
    async fn dispatch(
        &self,
        ctx: &ConnectionContext,
        request: Request,
    ) -> Result<Response, Error>;

    /// Emit an event to a connection or broadcast to all with appropriate policies.
    async fn emit_event(&self, event: Event) -> Result<(), Error>;

    /// Check whether a given method is allowed by a connection’s capabilities.
    fn is_authorized(
        &self,
        ctx: &ConnectionContext,
        method: &MethodId,
    ) -> Result<(), RpcError>;

    /// Observe a cancellation request for auditing/metrics.
    ///
    /// Implementations may ignore this. Cancellation itself is enforced by the
    /// IPC server loop via cooperative checks and/or task abort.
    async fn observe_cancel(
        &self,
        _ctx: &ConnectionContext,
        _id: u64,
    ) -> Result<(), Error> {
        Ok(())
    }
}

/// Context bound to a connection for authorization and audit.
#[derive(Debug, Clone)]
pub struct ConnectionContext {
    pub id: ConnectionId,
    pub capabilities: CapabilitySet,
    /// Optional additional metadata (process kind, user, document id, etc.)
    pub metadata: Option<serde_json::Value>,
}

/// Tracks connection heartbeats and terminates associated processes when
/// liveness timeouts are exceeded.
#[derive(Debug)]
pub struct HeartbeatTracker {
    process_manager: Arc<dyn ProcessManager>,
    check_interval: Duration,
    /// Maximum silence before a connection is considered dead. `None` disables
    /// timeouts.
    max_silence: Option<Duration>,
    state: Mutex<HashMap<ConnectionId, HeartbeatEntry>>,
}

#[derive(Debug, Clone)]
struct HeartbeatEntry {
    pid: ProcessId,
    last_heartbeat: Instant,
}

impl HeartbeatTracker {
    /// Create a new tracker and start a background checker task.
    ///
    /// `allowed_missed_intervals` controls after how many consecutive missed
    /// intervals the connection is considered dead.
    pub fn new(
        process_manager: Arc<dyn ProcessManager>,
        check_interval: Duration,
        allowed_missed_intervals: u32,
    ) -> Arc<Self> {
        let max_silence = check_interval.checked_mul(allowed_missed_intervals);
        let tracker = Arc::new(Self {
            process_manager,
            check_interval,
            max_silence,
            state: Mutex::new(HashMap::new()),
        });
        Self::spawn_checker(tracker.clone());
        tracker
    }

    fn spawn_checker(tracker: Arc<Self>) {
        let _ = tokio::spawn(async move {
            tracker.run_checker().await;
        });
    }

    async fn run_checker(self: Arc<Self>) {
        let mut interval = time::interval(self.check_interval);

        loop {
            interval.tick().await;

            let Some(max_silence) = self.max_silence else {
                continue;
            };

            let now = Instant::now();
            let to_terminate: Vec<(ConnectionId, ProcessId)> = {
                let state_lock = self.state.lock();
                let state = match state_lock {
                    Ok(s) => s,
                    Err(_) => break, // Poisoned
                };
                state
                    .iter()
                    .filter_map(|(conn_id, entry)| {
                        let since = now.duration_since(entry.last_heartbeat);
                        if since > max_silence {
                            Some((*conn_id, entry.pid))
                        } else {
                            None
                        }
                    })
                    .collect()
            };

            if to_terminate.is_empty() {
                continue;
            }

            for (conn_id, pid) in to_terminate {
                let span = tracing::info_span!(
                    "ipc.heartbeat.timeout",
                    conn_id = %conn_id,
                    process_id = %pid
                );

                async {
                    let _ =
                        self.process_manager.terminate_tree(pid, true).await;
                    if let Ok(mut state) = self.state.lock() {
                        state.remove(&conn_id);
                    }
                }
                .instrument(span)
                .await;
            }
        }
    }

    /// Start tracking a connection and its owning process.
    pub fn track_connection(&self, connection: ConnectionId, pid: ProcessId) {
        let entry = HeartbeatEntry {
            pid,
            last_heartbeat: Instant::now(),
        };

        if let Ok(mut state) = self.state.lock() {
            state.insert(connection, entry);
        }
    }

    /// Record an observed heartbeat from the given connection.
    pub fn record_heartbeat(&self, connection: &ConnectionId) {
        if let Ok(mut state) = self.state.lock() {
            if let Some(entry) = state.get_mut(connection) {
                entry.last_heartbeat = Instant::now();
            }
        }
    }

    /// Stop tracking the given connection.
    pub fn remove(&self, connection: &ConnectionId) {
        if let Ok(mut state) = self.state.lock() {
            state.remove(connection);
        }
    }

    /// Check if a connection is currently tracked (test-only).
    #[cfg(test)]
    fn is_tracked(&self, connection: &ConnectionId) -> bool {
        if let Ok(state) = self.state.lock() {
            state.contains_key(connection)
        } else {
            false
        }
    }
}

/// A simple in-memory router that registers connections, authorizes requests,
/// and returns a canned 'not implemented' response for authorized calls.
#[derive(Debug)]
pub struct IpcRouter {
    // Keep a minimal registry. Avoid panics; ignore duplicates by replacing.
    connections: std::sync::RwLock<Vec<ConnectionContext>>,
    // New: thin network service adapter
    network_service: Option<Arc<dyn NetworkService>>,
    process_service: Option<Arc<dyn ProcessService>>,
    rate_limiter: tokio::sync::Mutex<HashMap<RateKey, TokenBucket>>,
}

impl Default for IpcRouter {
    fn default() -> Self {
        Self::new()
    }
}

impl IpcRouter {
    /// Create a new `SimpleRouter`.
    pub fn new() -> Self {
        Self {
            connections: std::sync::RwLock::new(Vec::new()),
            network_service: None,
            process_service: None,
            rate_limiter: tokio::sync::Mutex::new(HashMap::new()),
        }
    }

    /// Set the process service.
    pub fn with_process_service(
        mut self,
        svc: Arc<dyn ProcessService>,
    ) -> Self {
        self.process_service = Some(svc);
        self
    }

    /// Set the network service (thin adapter).
    pub fn with_network_service(
        mut self,
        svc: Arc<dyn NetworkService>,
    ) -> Self {
        self.network_service = Some(svc);
        self
    }

    fn should_rate_limit(service: &str) -> bool {
        service == "io" || service == "network"
    }

    fn match_quotas(
        ctx: &ConnectionContext,
        method: &MethodId,
    ) -> Result<
        Option<crate::workspace::ipc::auth::capability::QuotaSet>,
        RpcError,
    > {
        use crate::workspace::ipc::auth::capability::{
            MethodRule, ServiceName,
        };

        let service_key = ServiceName(method.service.clone());
        let Some(rules) = ctx.capabilities.allowed.get(&service_key) else {
            return Err(RpcError::unauthorized(format!(
                "service '{}' is not allowed",
                method.service
            )));
        };

        let matched: Option<&MethodRule> = rules
            .iter()
            .find(|rule| rule.method.matches(&method.service, &method.method));

        let Some(rule) = matched else {
            return Err(RpcError::unauthorized(format!(
                "method '{}.{}' is not allowed",
                method.service, method.method
            )));
        };

        Ok(rule.quotas.clone())
    }

    async fn enforce_rate_limits(
        &self,
        ctx: &ConnectionContext,
        method: &MethodId,
        request_bytes: usize,
    ) -> Result<(), RpcError> {
        if !Self::should_rate_limit(method.service.as_str()) {
            return Ok(());
        }

        let quotas = Self::match_quotas(ctx, method)?;
        let Some(quotas) = quotas else {
            return Ok(());
        };

        let now = Instant::now();

        if let Some(rate) = quotas.bytes_per_sec {
            let Some(burst) = quotas.effective_burst_bytes() else {
                return Ok(());
            };

            let cost_u64 = u64::try_from(request_bytes).map_err(|e| {
                RpcError::capability_denied(format!(
                    "request size too large for limiter: {e}"
                ))
            })?;

            let key = RateKey::bytes(
                ctx.id,
                method.service.clone(),
                method.method.clone(),
            );

            let mut guard = self.rate_limiter.lock().await;
            let bucket = guard
                .entry(key)
                .or_insert_with(|| TokenBucket::new(rate, burst, now));

            if !bucket.try_take(cost_u64, now) {
                return Err(RpcError::capability_denied(format!(
                    "rate limit exceeded for {}.{} (bytes/sec)",
                    method.service, method.method
                )));
            }
        }

        Ok(())
    }

    fn decode_shutdown_request(
        args: Vec<u8>,
    ) -> Result<ShutdownTreeRequest, RpcError> {
        if args.is_empty() {
            return Ok(ShutdownTreeRequest::default());
        }
        postcard::from_bytes::<ShutdownTreeRequest>(&args).map_err(|e| {
            RpcError {
                code: "invalid_args".into(),
                message: format!("invalid workspace.shutdown args: {e}"),
            }
        })
    }
    /// Helper to create an error response.
    fn error_response(id: u64, code: &str, message: &str) -> Response {
        Response {
            id,
            ok: false,
            result: None,
            error: Some(RpcError {
                code: code.into(),
                message: message.into(),
            }),
        }
    }

    /// Helper to create a success response with serialized bytes.
    fn success_response(id: u64, bytes: Vec<u8>) -> Response {
        Response {
            id,
            ok: true,
            result: Some(bytes),
            error: None,
        }
    }

    /// Helper to check if a service is configured.
    fn check_service<T>(
        &self,
        service: &Option<T>,
        service_name: &str,
        id: u64,
    ) -> Result<(), Response> {
        if service.is_none() {
            return Err(Self::error_response(
                id,
                "not_implemented",
                &format!("{service_name} service not configured"),
            ));
        }
        Ok(())
    }

    /// Helper to decode a request from bytes.
    fn decode_request<T: serde::de::DeserializeOwned>(
        &self,
        args: &[u8],
        method: &str,
        id: u64,
    ) -> Result<T, Response> {
        postcard::from_bytes(args).map_err(|e| {
            Self::error_response(
                id,
                "invalid_args",
                &format!("invalid {method}.args: {e}"),
            )
        })
    }

    /// Helper to handle service call and serialize response.
    async fn handle_service_call<T: serde::Serialize, E: std::fmt::Display>(
        &self,
        call: impl std::future::Future<Output = Result<T, E>>,
        id: u64,
    ) -> Result<Response, Error> {
        match call.await {
            Ok(resp) => {
                let bytes = postcard::to_stdvec(&resp).map_err(|e| {
                    anyhow::anyhow!("failed to serialize response: {e}")
                })?;
                Ok(Self::success_response(id, bytes))
            }
            Err(e) => Ok(Self::error_response(id, "internal", &e.to_string())),
        }
    }

    async fn dispatch_process(
        &self,
        request: Request,
    ) -> Result<Response, Error> {
        if let Err(resp) =
            self.check_service(&self.process_service, "process", request.id)
        {
            return Ok(resp);
        }
        let svc = self.process_service.as_ref().unwrap();

        // FIXME: A process should only be able to shut down itself or its own
        // subtrees.
        match request.method.method.as_str() {
            METHOD_SHUTDOWN_TREE => {
                let shutdown_req = Self::decode_shutdown_request(request.args)
                    .map_err(|e| {
                        Self::error_response(request.id, &e.code, &e.message)
                    })?;
                self.handle_service_call(
                    svc.shutdown_tree(shutdown_req),
                    request.id,
                )
                .await
            }
            _ => Ok(Self::error_response(
                request.id,
                "not_implemented",
                "method not implemented",
            )),
        }
    }

    async fn dispatch_network(
        &self,
        request: Request,
    ) -> Result<Response, Error> {
        if let Err(resp) =
            self.check_service(&self.network_service, "network", request.id)
        {
            return Ok(resp);
        }
        let svc = self.network_service.as_ref().unwrap();

        match request.method.method.as_str() {
            METHOD_FETCH => {
                let req = self.decode_request(
                    &request.args,
                    "network.fetch",
                    request.id,
                )?;
                self.handle_service_call(svc.fetch(req), request.id).await
            }
            METHOD_READ_FILE => {
                let req = self.decode_request(
                    &request.args,
                    "network.read_file",
                    request.id,
                )?;
                self.handle_service_call(svc.read_file(req), request.id)
                    .await
            }
            _ => Ok(Self::error_response(
                request.id,
                "not_implemented",
                "method not implemented",
            )),
        }
    }
}

#[async_trait]
impl Router for IpcRouter {
    /// Register a new connection after handshake.
    async fn register_connection(
        &self,
        ctx: ConnectionContext,
    ) -> Result<(), Error> {
        if let Ok(mut guard) = self.connections.write() {
            if let Some(pos) = guard.iter().position(|c| c.id == ctx.id) {
                guard[pos] = ctx;
            } else {
                guard.push(ctx);
            }
        }
        Ok(())
    }

    /// Resolve and dispatch a request to a target service method.
    async fn dispatch(
        &self,
        ctx: &ConnectionContext,
        request: Request,
    ) -> Result<Response, Error> {
        let span = tracing::info_span!(
            "ipc.dispatch",
            conn_id = %ctx.id,
            request_id = request.id,
            service = %request.method.service,
            method = %request.method.method
        );

        async move {
            match self.is_authorized(ctx, &request.method) {
                Ok(()) => {
                    if let Err(e) = self
                        .enforce_rate_limits(
                            ctx,
                            &request.method,
                            request.args.len(),
                        )
                        .await
                    {
                        return Ok(Response {
                            id: request.id,
                            ok: false,
                            result: None,
                            error: Some(e),
                        });
                    }

                    if request.method.service == PROCESS_SERVICE_NAME {
                        return self.dispatch_process(request).await;
                    }

                    if request.method.service == NETWORK_SERVICE_NAME {
                        return self.dispatch_network(request).await;
                    }

                    Ok(Response {
                        id: request.id,
                        ok: false,
                        result: None,
                        error: Some(RpcError {
                            code: "not_implemented".into(),
                            message: "method not implemented".into(),
                        }),
                    })
                }
                Err(e) => Ok(Response {
                    id: request.id,
                    ok: false,
                    result: None,
                    error: Some(e),
                }),
            }
        }
        .instrument(span)
        .await
    }

    async fn emit_event(&self, event: Event) -> Result<(), Error> {
        let span = tracing::info_span!("ipc.emit_event", topic = ?event.topic);
        async move {
            // No-op for now.
            Ok(())
        }
        .instrument(span)
        .await
    }

    /// Check whether a given method is allowed by a connection’s capabilities.
    fn is_authorized(
        &self,
        ctx: &ConnectionContext,
        method: &MethodId,
    ) -> Result<(), RpcError> {
        use crate::workspace::ipc::auth::capability::{
            MethodRule, ServiceName,
        };

        let service_key = ServiceName(method.service.clone());
        let Some(rules) = ctx.capabilities.allowed.get(&service_key) else {
            return Err(RpcError {
                code: "unauthorized".into(),
                message: format!("service '{}' is not allowed", method.service),
            });
        };

        let allowed = rules.iter().any(|rule: &MethodRule| {
            rule.method.matches(&method.service, &method.method)
        });

        if allowed {
            Ok(())
        } else {
            Err(RpcError {
                code: "unauthorized".into(),
                message: format!(
                    "method '{}.{}' is not allowed",
                    method.service, method.method
                ),
            })
        }
    }

    async fn observe_cancel(
        &self,
        _ctx: &ConnectionContext,
        _id: u64,
    ) -> Result<(), Error> {
        Ok(())
    }
}

impl IpcRouter {
    /// Retrieve a registered connection by id.
    pub fn get(&self, id: &ConnectionId) -> Option<ConnectionContext> {
        if let Ok(guard) = self.connections.read() {
            guard.iter().find(|c| c.id == *id).cloned()
        } else {
            None
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    use crate::workspace::ipc::auth::capability::{
        CapabilitySet, MethodRule, MethodSelector, QuotaSet, ServiceName,
    };
    use crate::workspace::ipc::process_manager::{
        ChildHandle, ProcessManager, SpawnParams,
    };
    use crate::workspace::ipc::protocol::{MethodId, Request};
    use crate::workspace::ipc::services::network::api::{
        BytesResponse as NetBytesResponse, FetchRequest, METHOD_FETCH,
        METHOD_READ_FILE, MockNetworkService, ReadFileRequest,
        SERVICE_NAME as NETWORK_SERVICE_NAME,
    };
    use crate::workspace::ipc::services::process::MockProcessService;
    use crate::workspace::ipc::{
        assert_ipc_response_error, assert_ipc_response_ok,
    };

    use crate::workspace::ipc::types::{ChildKind, ProcessId};
    use anyhow::Result;
    use std::collections::HashMap;

    fn ctx_with_rules(
        service: &str,
        rules: Vec<MethodRule>,
    ) -> ConnectionContext {
        let mut allowed: HashMap<ServiceName, Vec<MethodRule>> = HashMap::new();
        allowed.insert(ServiceName(service.to_string()), rules);
        ConnectionContext {
            id: Default::default(),
            capabilities: CapabilitySet {
                allowed,
                global_limits: None,
            },
            metadata: None,
        }
    }

    fn req(service: &str, method: &str) -> Request {
        Request {
            id: Default::default(),
            method: MethodId {
                service: service.into(),
                method: method.into(),
            },
            args: vec![],
        }
    }

    /// Exact selector should allow only the exact method.
    #[crate::ctb_test(tokio::test)]
    async fn auth_exact_allows() -> Result<()> {
        let router = IpcRouter::new();
        let ctx = ctx_with_rules(
            "svc",
            vec![MethodRule {
                method: MethodSelector::Exact("do_work".into()),
                quotas: None,
            }],
        );

        let resp = router.dispatch(&ctx, req("svc", "do_work")).await?;
        assert!(!resp.ok);
        assert_eq!(resp.error.unwrap().code, "not_implemented");

        let resp2 = router.dispatch(&ctx, req("svc", "other")).await?;
        assert!(!resp2.ok);
        assert_eq!(resp2.error.unwrap().code, "unauthorized");
        Ok(())
    }

    /// Prefix selector should allow matching prefix and deny others.
    #[crate::ctb_test(tokio::test)]
    async fn auth_prefix() -> Result<()> {
        let router = IpcRouter::new();
        let ctx = ctx_with_rules(
            "svc",
            vec![MethodRule {
                method: MethodSelector::Prefix("do_".into()),
                quotas: None,
            }],
        );

        let ok_resp = router.dispatch(&ctx, req("svc", "do_stuff")).await?;
        assert!(!ok_resp.ok);
        assert_eq!(ok_resp.error.unwrap().code, "not_implemented");

        let deny_resp = router.dispatch(&ctx, req("svc", "list")).await?;
        assert!(!deny_resp.ok);
        assert_eq!(deny_resp.error.unwrap().code, "unauthorized");
        Ok(())
    }

    /// Any selector allows all methods within the service. Absence of service
    /// entry denies all methods.
    #[crate::ctb_test(tokio::test)]
    async fn auth_any_and_missing_service() -> Result<()> {
        let router = IpcRouter::new();

        // Any allows all for 'svc'
        let ctx_any = ctx_with_rules(
            "svc",
            vec![MethodRule {
                method: MethodSelector::Any,
                quotas: None,
            }],
        );

        let ok_resp = router.dispatch(&ctx_any, req("svc", "x")).await?;
        assert!(!ok_resp.ok);
        assert_eq!(ok_resp.error.unwrap().code, "not_implemented");

        // No entry for 'other' -> unauthorized
        let deny_resp = router.dispatch(&ctx_any, req("other", "x")).await?;
        assert!(!deny_resp.ok);
        assert_eq!(deny_resp.error.unwrap().code, "unauthorized");
        Ok(())
    }

    /// Selectors with fully-qualified strings should also match.
    #[crate::ctb_test(tokio::test)]
    async fn auth_fully_qualified_selectors() -> Result<()> {
        let router = IpcRouter::new();
        let ctx = ctx_with_rules(
            "svc",
            vec![
                MethodRule {
                    method: MethodSelector::Exact("svc.do_a".into()),
                    quotas: None,
                },
                MethodRule {
                    method: MethodSelector::Prefix("svc.do_".into()),
                    quotas: None,
                },
            ],
        );

        // Exact match
        let r1 = router.dispatch(&ctx, req("svc", "do_a")).await?;
        assert_eq!(r1.error.unwrap().code, "not_implemented");

        // Prefix match
        let r2 = router.dispatch(&ctx, req("svc", "do_b")).await?;
        assert_eq!(r2.error.unwrap().code, "not_implemented");

        // Non-matching method
        let r3 = router.dispatch(&ctx, req("svc", "list")).await?;
        assert_eq!(r3.error.unwrap().code, "unauthorized");

        Ok(())
    }

    /// Register_connection stores the ConnectionContext.
    #[crate::ctb_test(tokio::test)]
    async fn register_stores_context() -> Result<()> {
        let router = IpcRouter::new();
        let ctx = ConnectionContext {
            id: ConnectionId::default(),
            capabilities: CapabilitySet::default(),
            metadata: Some(serde_json::json!({"k":"v"})),
        };
        router.register_connection(ctx.clone()).await?;
        let got = router.get(&ctx.id).expect("stored");
        assert_eq!(got.id, ctx.id);
        assert_eq!(got.metadata, ctx.metadata);
        Ok(())
    }

    #[derive(Debug, Default)]
    struct MockProcessManager {
        terminated: std::sync::Mutex<Vec<(ProcessId, bool)>>,
    }

    impl MockProcessManager {
        fn terminations(&self) -> Vec<(ProcessId, bool)> {
            if let Ok(guard) = self.terminated.lock() {
                guard.clone()
            } else {
                Vec::new()
            }
        }
    }

    #[async_trait]
    impl ProcessManager for MockProcessManager {
        async fn spawn_child(
            &self,
            _params: SpawnParams,
        ) -> Result<ChildHandle, Error> {
            Ok(ChildHandle {
                pid: ProcessId::default(),
                kind: ChildKind::Renderer,
                connection: None,
            })
        }

        async fn attach_connection(
            &self,
            _pid: ProcessId,
            _conn: ConnectionId,
        ) -> Result<(), Error> {
            Ok(())
        }

        async fn list_children(&self) -> Result<Vec<ChildHandle>, Error> {
            Ok(Vec::new())
        }

        async fn terminate_tree(
            &self,
            pid: ProcessId,
            force: bool,
        ) -> Result<(), Error> {
            if let Ok(mut guard) = self.terminated.lock() {
                guard.push((pid, force));
            }
            Ok(())
        }
    }

    /// Heartbeat miss should trigger termination and cleanup.
    #[crate::ctb_test(tokio::test)]
    async fn heartbeat_miss_triggers_termination_and_cleanup() -> Result<()> {
        let manager = Arc::new(MockProcessManager::default());
        let tracker = HeartbeatTracker::new(
            manager.clone(),
            Duration::from_millis(20),
            1,
        );

        let conn = ConnectionId::default();
        let pid = ProcessId::default();
        tracker.track_connection(conn, pid);

        // Do not record any heartbeat; allow multiple intervals to pass.
        tokio::time::sleep(Duration::from_millis(100)).await;

        let terms = manager.terminations();
        assert_eq!(terms.len(), 1);
        assert_eq!(terms[0].0, pid);
        assert!(terms[0].1);
        assert!(!tracker.is_tracked(&conn));

        Ok(())
    }

    #[crate::ctb_test(tokio::test)]
    async fn routes_workspace_shutdown() -> Result<()> {
        let router =
            IpcRouter::new().with_process_service(Arc::new(MockProcessService));

        let mut allowed: HashMap<ServiceName, Vec<MethodRule>> = HashMap::new();
        allowed.insert(
            ServiceName("process".to_string()),
            vec![MethodRule {
                method: MethodSelector::Exact("shutdown_tree".into()),
                quotas: None,
            }],
        );

        let ctx = ConnectionContext {
            id: Default::default(),
            capabilities: CapabilitySet {
                allowed,
                global_limits: None,
            },
            metadata: None,
        };

        let req = Request {
            id: Default::default(),
            method: MethodId {
                service: "process".into(),
                method: "shutdown_tree".into(),
            },
            args: vec![],
        };

        let resp = router.dispatch(&ctx, req).await?;
        assert_ipc_response_ok(&resp);
        assert!(resp.error.is_none());
        Ok(())
    }

    /// Exceeding bytes/sec should yield a capability_denied error.
    #[crate::ctb_test(tokio::test)]
    async fn rate_limit_bytes_per_sec_is_enforced_for_io_and_network()
    -> Result<()> {
        let router = IpcRouter::new();

        let mut allowed: HashMap<ServiceName, Vec<MethodRule>> = HashMap::new();
        allowed.insert(
            ServiceName("io".to_string()),
            vec![MethodRule {
                method: MethodSelector::Exact("read".into()),
                quotas: Some(QuotaSet {
                    bytes_per_sec: Some(10),
                    ops_per_sec: None,
                    burst: Some(10),
                }),
            }],
        );

        let ctx = ConnectionContext {
            id: ConnectionId::default(),
            capabilities: CapabilitySet {
                allowed,
                global_limits: None,
            },
            metadata: None,
        };

        let mk_req = |n: usize| Request {
            id: Default::default(),
            method: MethodId {
                service: "io".into(),
                method: "read".into(),
            },
            args: vec![0u8; n],
        };

        // First request consumes full burst.
        let r1 = router.dispatch(&ctx, mk_req(10)).await?;
        assert_eq!(r1.error.unwrap().code, "not_implemented");

        // Second request should be denied immediately.
        let r2 = router.dispatch(&ctx, mk_req(1)).await?;
        assert_ipc_response_error(&r2);
        assert_eq!(r2.error.unwrap().code, "capability_denied");

        Ok(())
    }

    /// Routes network.fetch and returns serialized bytes.
    #[crate::ctb_test(tokio::test)]
    async fn routes_network_fetch() -> anyhow::Result<()> {
        let router = IpcRouter::new()
            .with_network_service(Arc::new(MockNetworkService::new()));

        let mut allowed: HashMap<ServiceName, Vec<MethodRule>> = HashMap::new();
        allowed.insert(
            ServiceName(NETWORK_SERVICE_NAME.to_string()),
            vec![MethodRule {
                method: MethodSelector::Exact(METHOD_FETCH.into()),
                quotas: None,
            }],
        );

        let ctx = ConnectionContext {
            id: Default::default(),
            capabilities: CapabilitySet {
                allowed,
                global_limits: None,
            },
            metadata: None,
        };

        let args = postcard::to_stdvec(&FetchRequest {
            url: "https://example.com".into(),
        })?;
        let req = Request {
            id: Default::default(),
            method: MethodId {
                service: NETWORK_SERVICE_NAME.into(),
                method: METHOD_FETCH.into(),
            },
            args,
        };

        let resp = router.dispatch(&ctx, req).await?;
        assert_ipc_response_ok(&resp);
        let decoded: NetBytesResponse =
            postcard::from_bytes(&resp.result.unwrap())?;
        assert_eq!(
            decoded.bytes,
            vec![
                69, 120, 97, 109, 112, 108, 101, 32, 72, 84, 84, 80, 83, 32,
                70, 101, 116, 99, 104
            ]
        );
        Ok(())
    }

    /// Routes network.read_file and returns serialized bytes.
    #[crate::ctb_test(tokio::test)]
    async fn routes_network_read_file() -> anyhow::Result<()> {
        let router = IpcRouter::new()
            .with_network_service(Arc::new(MockNetworkService::new()));

        let mut allowed: HashMap<ServiceName, Vec<MethodRule>> = HashMap::new();
        allowed.insert(
            ServiceName(NETWORK_SERVICE_NAME.to_string()),
            vec![MethodRule {
                method: MethodSelector::Exact(METHOD_READ_FILE.into()),
                quotas: None,
            }],
        );

        let ctx = ConnectionContext {
            id: Default::default(),
            capabilities: CapabilitySet {
                allowed,
                global_limits: None,
            },
            metadata: None,
        };

        let args: Vec<u8> = postcard::to_stdvec(&ReadFileRequest {
            path: "/tmp/example/file.txt".into(),
        })?;
        let req = Request {
            id: Default::default(),
            method: MethodId {
                service: NETWORK_SERVICE_NAME.into(),
                method: METHOD_READ_FILE.into(),
            },
            args,
        };

        let resp = router.dispatch(&ctx, req).await?;
        assert_ipc_response_ok(&resp);
        let decoded: NetBytesResponse =
            postcard::from_bytes(&resp.result.unwrap())?;
        assert_eq!(
            decoded.bytes,
            vec![
                69, 120, 97, 109, 112, 108, 101, 32, 70, 105, 108, 101, 32, 82,
                101, 97, 100
            ]
        );
        Ok(())
    }
}
