//! Tokio-based process management with platform-specific supervision.
//!
//! On Unix, children are placed in their own process group and (on Linux) are
//! configured with a parent-death signal. Tree termination uses killpg.
//!
//! On Windows, children are placed into a Job Object configured to kill all
//! processes on close; tree termination uses `TerminateJobObject`.

use crate::workspace::ipc::auth::capability::CapabilityBundle;
use crate::workspace::ipc::error::Error;
use crate::workspace::ipc::services::process::api::ShutdownTreeResponse;
use crate::workspace::ipc::types::ChildKind;
use crate::workspace::ipc::types::{ConnectionId, ProcessId};
use async_trait::async_trait;
use serde::{Deserialize, Serialize};
use std::future::Future;
use std::time::Duration;
use tokio::time;

pub mod unix;
pub mod windows;

/// Parameters for spawning a child.
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct SpawnParams {
    pub kind: ChildKind,
    /// Executable or command line.
    pub program: Option<String>,
    /// Arguments for the child.
    pub args: Vec<String>,
    /// Environment variables.
    pub env: Vec<(String, String)>,
    /// Working directory.
    pub cwd: Option<String>,
    /// Capability bundle bound to the child's control connection.
    pub capabilities: CapabilityBundle,
}

/// A handle with metadata tracked by the workspace.
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct ChildHandle {
    pub pid: ProcessId,
    pub kind: ChildKind,
    pub connection: Option<ConnectionId>,
}

/// Supervisor/process manager for spawning and tracking child trees with
/// OS-level supervision.
#[async_trait]
pub trait ProcessManager: Send + Sync + std::fmt::Debug {
    async fn spawn_child(
        &self,
        params: SpawnParams,
    ) -> Result<ChildHandle, Error>;

    async fn attach_connection(
        &self,
        pid: ProcessId,
        conn: ConnectionId,
    ) -> Result<(), Error>;

    async fn list_children(&self) -> Result<Vec<ChildHandle>, Error>;

    /// Terminate a process tree. This is invoked by liveness checks such as
    /// the heartbeat tracker when a connection is considered dead.
    async fn terminate_tree(
        &self,
        pid: ProcessId,
        force: bool,
    ) -> Result<(), Error>;

    /// Wait for a process to exit, up to `timeout`.
    ///
    /// Returns `Ok(true)` if the process is known to have exited within the
    /// timeout, `Ok(false)` if the wait timed out.
    ///
    /// Platform managers should override this to actually reap/wait on the
    /// underlying child handle. The default implementation is conservative and
    /// simply waits out the timeout.
    async fn wait_for_exit(
        &self,
        _pid: ProcessId,
        timeout: Duration,
    ) -> Result<bool, Error> {
        time::sleep(timeout).await;
        Ok(false)
    }

    /// Force kill a process tree.
    ///
    /// Default behavior delegates to `terminate_tree(pid, true)`.
    async fn kill_tree(&self, pid: ProcessId) -> Result<(), Error> {
        self.terminate_tree(pid, true).await
    }
}

/// Outcome of a graceful shutdown attempt.
#[derive(Debug, Clone, Copy, PartialEq, Eq)]
pub struct GracefulShutdownOutcome {
    /// Whether the shutdown request was acknowledged within `ack_timeout`.
    pub acknowledged: bool,
    /// Whether the process exited within the relevant timeout.
    pub exited: bool,
    /// Whether the shutdown path required a forced tree kill.
    pub forced: bool,
}

/// Attempt a graceful shutdown of `pid`.
///
/// The caller provides `send_shutdown`, which should represent the RPC to the
/// target process' `process.shutdown_tree` handler, returning when the target
/// acknowledges the request.
///
/// Behavior:
/// - wait for ack up to `ack_timeout`
/// - wait for exit up to `exit_timeout`
/// - if either step times out, force-kill the tree and wait again up to
///   `exit_timeout`
///
/// This is intentionally transport-agnostic: IPC wiring lives elsewhere.
pub async fn graceful_shutdown_tree<F, Fut>(
    process_manager: &dyn ProcessManager,
    pid: ProcessId,
    send_shutdown: F,
    ack_timeout: Duration,
    exit_timeout: Duration,
) -> Result<GracefulShutdownOutcome, Error>
where
    F: FnOnce() -> Fut + Send,
    Fut: Future<Output = Result<ShutdownTreeResponse, Error>> + Send,
{
    let acknowledged = match time::timeout(ack_timeout, send_shutdown()).await {
        Ok(Ok(resp)) => resp.acknowledged,
        Ok(Err(_)) => false,
        Err(_elapsed) => false,
    };

    if acknowledged {
        let exited = process_manager.wait_for_exit(pid, exit_timeout).await?;
        if exited {
            return Ok(GracefulShutdownOutcome {
                acknowledged,
                exited,
                forced: false,
            });
        }
    }

    process_manager.kill_tree(pid).await?;
    let exited = process_manager.wait_for_exit(pid, exit_timeout).await?;

    Ok(GracefulShutdownOutcome {
        acknowledged,
        exited,
        forced: true,
    })
}

#[cfg(unix)]
pub use unix::TokioProcessManager;
#[cfg(windows)]
pub use windows::TokioProcessManager;

#[cfg(test)]
mod tests {
    use super::*;
    use crate::workspace::ipc::types::ConnectionId;
    use anyhow::Result;
    use std::sync::Arc;
    use std::sync::atomic::{AtomicUsize, Ordering};
    use tokio::sync::Notify;

    #[derive(Debug)]
    struct MockProcessManager {
        killed: AtomicUsize,
        waited: AtomicUsize,
        exited: Arc<Notify>,
    }

    impl MockProcessManager {
        fn new(exited: Arc<Notify>) -> Self {
            Self {
                killed: AtomicUsize::new(0),
                waited: AtomicUsize::new(0),
                exited,
            }
        }
    }

    #[async_trait]
    impl ProcessManager for MockProcessManager {
        async fn spawn_child(
            &self,
            _params: SpawnParams,
        ) -> Result<ChildHandle, Error> {
            Ok(ChildHandle {
                pid: ProcessId::default(),
                kind: ChildKind::Renderer,
                connection: None,
            })
        }

        async fn attach_connection(
            &self,
            _pid: ProcessId,
            _conn: ConnectionId,
        ) -> Result<(), Error> {
            Ok(())
        }

        async fn list_children(&self) -> Result<Vec<ChildHandle>, Error> {
            Ok(Vec::new())
        }

        async fn terminate_tree(
            &self,
            _pid: ProcessId,
            _force: bool,
        ) -> Result<(), Error> {
            self.killed.fetch_add(1, Ordering::SeqCst);
            let exited = self.exited.clone();
            tokio::spawn(async move {
                time::sleep(Duration::from_millis(1)).await;
                exited.notify_waiters();
            });
            Ok(())
        }

        async fn wait_for_exit(
            &self,
            _pid: ProcessId,
            timeout: Duration,
        ) -> Result<bool, Error> {
            self.waited.fetch_add(1, Ordering::SeqCst);
            match time::timeout(timeout, self.exited.notified()).await {
                Ok(()) => Ok(true),
                Err(_elapsed) => Ok(false),
            }
        }
    }

    #[crate::ctb_test(tokio::test)]
    async fn graceful_shutdown_cooperative_child() -> Result<()> {
        let exited = Arc::new(Notify::new());
        let pm = MockProcessManager::new(exited.clone());
        let pid = ProcessId::default();

        let outcome = graceful_shutdown_tree(
            &pm,
            pid,
            || async {
                let exited = exited.clone();
                let _ = tokio::spawn(async move {
                    time::sleep(Duration::from_millis(10)).await;
                    exited.notify_waiters();
                });
                Ok(ShutdownTreeResponse { acknowledged: true })
            },
            Duration::from_millis(50),
            Duration::from_millis(200),
        )
        .await?;

        assert_eq!(
            outcome,
            GracefulShutdownOutcome {
                acknowledged: true,
                exited: true,
                forced: false,
            }
        );
        assert_eq!(pm.killed.load(Ordering::SeqCst), 0);
        assert!(pm.waited.load(Ordering::SeqCst) >= 1);
        Ok(())
    }

    #[crate::ctb_test(tokio::test)]
    async fn graceful_shutdown_ignoring_child_forces_kill() -> Result<()> {
        let exited = Arc::new(Notify::new());
        let pm = MockProcessManager::new(exited.clone());
        let pid = ProcessId::default();

        let outcome = graceful_shutdown_tree(
            &pm,
            pid,
            || async {
                std::future::pending::<Result<ShutdownTreeResponse, Error>>()
                    .await
            },
            Duration::from_millis(20),
            Duration::from_millis(200),
        )
        .await?;

        assert!(!outcome.acknowledged);
        assert!(outcome.forced);
        assert!(outcome.exited);
        assert_eq!(pm.killed.load(Ordering::SeqCst), 1);
        assert!(pm.waited.load(Ordering::SeqCst) >= 1);
        Ok(())
    }
}
