#[cfg(windows)]
use anyhow::Context;

#[cfg(windows)]
use windows_sys::Win32::Foundation::{
    CloseHandle, GetLastError, HANDLE, INVALID_HANDLE_VALUE,
};

#[cfg(windows)]
use windows_sys::Win32::System::Memory::{
    CreateFileMappingW, FILE_MAP_READ, FILE_MAP_WRITE, MapViewOfFile,
    PAGE_READWRITE, UnmapViewOfFile,
};

#[cfg(windows)]
use windows_sys::Win32::System::Threading::{
    DUPLICATE_SAME_ACCESS, DuplicateHandle, GetCurrentProcess,
};

#[allow(unsafe_code)]
#[cfg(windows)]
fn last_err(msg: &str) -> Error {
    let code = unsafe { GetLastError() };
    anyhow::anyhow!("{msg} (GetLastError={code})").into()
}

#[cfg(windows)]
fn to_handle(raw: u64) -> Result<HANDLE, Error> {
    // HANDLE is pointer-sized; reject truncation on 32-bit.
    let h = raw as usize;
    anyhow::ensure!(h as u64 == raw, "HANDLE value does not fit in usize");
    Ok(h as HANDLE)
}

#[allow(unsafe_code)]
#[cfg(windows)]
pub fn create_file_mapping(size: u64) -> Result<u64, Error> {
    let high = (size >> 32) as u32;
    let low = (size & 0xffff_ffff) as u32;

    // Pagefile-backed mapping: INVALID_HANDLE_VALUE.
    let handle = unsafe {
        CreateFileMappingW(
            INVALID_HANDLE_VALUE,
            std::ptr::null(),
            PAGE_READWRITE,
            high,
            low,
            std::ptr::null(),
        )
    };
    if handle == 0 {
        return Err(last_err("CreateFileMappingW failed"));
    }
    Ok(handle as u64)
}

#[allow(unsafe_code)]
#[cfg(windows)]
pub fn duplicate_handle_current(handle: u64) -> Result<u64, Error> {
    let src = unsafe { GetCurrentProcess() };
    let h = to_handle(handle)?;
    let mut out: HANDLE = 0;

    let ok = unsafe {
        DuplicateHandle(
            src,
            h,
            src,
            &mut out as *mut HANDLE,
            0,
            0,
            DUPLICATE_SAME_ACCESS,
        )
    };
    if ok == 0 {
        return Err(last_err("DuplicateHandle failed"));
    }
    Ok(out as u64)
}

#[allow(unsafe_code)]
#[cfg(windows)]
pub fn close_handle(handle: u64) -> Result<(), Error> {
    let h = to_handle(handle)?;
    let ok = unsafe { CloseHandle(h) };
    if ok == 0 {
        return Err(last_err("CloseHandle failed"));
    }
    Ok(())
}

#[cfg(windows)]
pub struct MappingView {
    handle: HANDLE,
    ptr: *mut u8,
    len: usize,
}

#[cfg(windows)]
impl MappingView {
    pub fn as_ptr(&self) -> *const u8 {
        self.ptr as *const u8
    }
}

#[allow(unsafe_code)]
#[cfg(windows)]
impl Drop for MappingView {
    fn drop(&mut self) {
        unsafe {
            if !self.ptr.is_null() {
                let _ = UnmapViewOfFile(self.ptr as *const _);
            }
            if self.handle != 0 {
                let _ = CloseHandle(self.handle);
            }
        }
    }
}

#[allow(unsafe_code)]
#[cfg(windows)]
pub fn map_view_read(handle: u64, len: usize) -> Result<MappingView, Error> {
    // Duplicate so the returned view owns a handle distinct from any registry
    // handle retained by the allocator.
    let dup = duplicate_handle_current(handle)?;
    let h = to_handle(dup)?;

    let ptr = unsafe { MapViewOfFile(h, FILE_MAP_READ, 0, 0, len) } as *mut u8;
    if ptr.is_null() {
        // Ensure the duplicated handle is closed.
        let _ = unsafe { CloseHandle(h) };
        return Err(last_err("MapViewOfFile(FILE_MAP_READ) failed"));
    }

    Ok(MappingView {
        handle: h,
        ptr,
        len,
    })
}

#[allow(unsafe_code)]
#[cfg(windows)]
pub fn write_mapping(handle: u64, data: &[u8]) -> Result<(), Error> {
    let len = data.len();
    let dup = duplicate_handle_current(handle)?;
    let h = to_handle(dup)?;

    let ptr = unsafe { MapViewOfFile(h, FILE_MAP_WRITE, 0, 0, len) } as *mut u8;
    if ptr.is_null() {
        let _ = unsafe { CloseHandle(h) };
        return Err(last_err("MapViewOfFile(FILE_MAP_WRITE) failed"));
    }

    // Safety: ptr is valid for `len` bytes for this view; copy in-bounds.
    unsafe {
        std::ptr::copy_nonoverlapping(data.as_ptr(), ptr, len);
        let _ = UnmapViewOfFile(ptr as *const _);
        let _ = CloseHandle(h);
    }

    Ok(())
}
