//! Session and RPC client implementations for framed IPC with postcard.
//!
//! This module provides:
//! - A Session implementation that serializes and deserializes Message values
//!   using postcard over a framed transport.
//! - A default `RpcClient` that correlates request/response pairs by `RequestId`,
//!   forwards cancellations, and exposes an events stream with bounded
//!   backpressure.
//!
//! `StreamManager` is transport-only; data processing is delegated to services.

use crate::workspace::ipc::auth::capability::TokenValidator;
use crate::workspace::ipc::error::Error;
use crate::workspace::ipc::protocol::RpcError;
use crate::workspace::ipc::protocol::{
    Cancel, Event, Message, Request, Response, StreamControl,
};
use crate::workspace::ipc::protocol::{Hello, HelloErr, HelloOk};
use crate::workspace::ipc::router::{ConnectionContext, Router};
use crate::workspace::ipc::transport::FramedConnection;
use crate::workspace::ipc::types::ConnectionId;
use crate::workspace::ipc::types::{RequestId, StreamId};
use anyhow::{Context as _, anyhow};
use async_trait::async_trait;
use bytes::Bytes;
use futures_core::Stream;
use std::collections::HashMap;
use std::pin::Pin;
use std::sync::Arc;
use tokio::sync::{mpsc, oneshot};
use tokio_stream::wrappers::ReceiverStream;
use tracing::Instrument as _;

/// RPC client interface layered on a Session, with request/response correlation and cancellation.
#[async_trait]
pub trait RpcClient: Send + Sync {
    /// Send a request; returns immediately after sending.
    async fn send_request(&self, req: Request) -> Result<(), Error>;

    /// Await a response with the given id.
    async fn recv_response(&self, id: RequestId) -> Result<Response, Error>;

    /// Cancel a request by id.
    async fn cancel(&self, cancel: Cancel) -> Result<(), Error>;

    /// Receive server-initiated events.
    fn events(&self) -> Pin<Box<dyn Stream<Item = Event> + Send>>;
}

/// Streaming control plane to coordinate substreams or blob-backed flows.
#[async_trait]
pub trait StreamManager: Send + Sync {
    async fn start(&self, ctl: StreamControl) -> Result<(), Error>;
    async fn next(&self, ctl: StreamControl) -> Result<(), Error>;
    async fn end(&self, ctl: StreamControl) -> Result<(), Error>;

    /// Optional helper for control-plane streaming (chunked via frames).
    fn stream_incoming(
        &self,
        id: StreamId,
    ) -> Pin<Box<dyn Stream<Item = Bytes> + Send>>;
}

/// A higher-level session that speaks the Message protocol over a `FramedConnection`.
#[async_trait]
pub trait Session: Send + Sync {
    /// Send a structured message.
    async fn send(&self, msg: &Message) -> Result<(), Error>;

    /// Receive the next structured message (deserializing from frames). None on EOF.
    async fn recv(&self) -> Result<Option<Message>, Error>;

    /// Client-side handshake helper.
    ///
    /// Sends `Hello` and waits for `HelloOk` or `HelloErr`. Returns bound capability
    /// set on success. Avoids panics; errors map to [`Error`].
    async fn client_handshake(
        &self,
        hello: Hello,
    ) -> Result<crate::workspace::ipc::auth::capability::CapabilitySet, Error>
    where
        Self: Sized,
    {
        self.send(&Message::Hello(hello)).await?;
        loop {
            let Some(msg) = self.recv().await? else {
                return Err(to_ipc_error(anyhow!("eof before HelloOk")));
            };
            match msg {
                Message::HelloOk(ok) => {
                    return Ok(ok.bound_capabilities);
                }
                Message::HelloErr(err) => {
                    return Err(to_ipc_error(anyhow!(
                        "handshake failed: {}",
                        err.message
                    )));
                }
                // Ignore unrelated messages during handshake.
                _ => {}
            }
        }
    }

    /// Server-side handshake helper.
    ///
    /// Waits for Hello, validates the token, replies with HelloOk/HelloErr,
    /// and registers the connection on success.
    async fn server_handshake<V: TokenValidator, R: Router>(
        &self,
        validator: &V,
        router: &R,
        connection_id: ConnectionId,
    ) -> Result<crate::workspace::ipc::auth::capability::CapabilitySet, Error>
    where
        Self: Sized,
    {
        let Some(msg) = self.recv().await? else {
            return Err(to_ipc_error(anyhow!("eof awaiting Hello")));
        };
        let Message::Hello(hello) = msg else {
            return Err(to_ipc_error(anyhow!("expected Hello")));
        };

        let bound = match validator.validate(&hello.token) {
            Ok(set) => set,
            Err(e) => {
                // Send HelloErr and return error.
                let _ = self
                    .send(&Message::HelloErr(HelloErr::new(format!("{e:#}"))))
                    .await;
                return Err(to_ipc_error(anyhow!("token invalid: {e:#}")));
            }
        };

        // Send success response.
        self.send(&Message::HelloOk(HelloOk::new(bound.clone())))
            .await?;

        // Register connection context.
        let ctx = ConnectionContext {
            id: connection_id,
            capabilities: bound.clone(),
            metadata: hello.client_info.as_ref().map(|ci| {
                serde_json::json!({
                    "name": ci.name,
                    "version": ci.version,
                    "process_kind": ci.process_kind
                })
            }),
        };
        router.register_connection(ctx).await?;

        Ok(bound)
    }
}

const DEFAULT_EVENTS_CAPACITY: usize = 64;

/// Adapter implementing Session on top of a framed connection that transports
/// raw Bytes. Messages are encoded/decoded with postcard.
pub struct FramedSession<T> {
    inner: T,
}

impl<T: Clone> FramedSession<T> {
    /// Create a new framed session from a lower-level transport.
    pub fn new(inner: T) -> Self {
        Self { inner }
    }
}

#[async_trait]
impl<T> Session for FramedSession<T>
where
    T: FramedConnection + Clone + 'static,
{
    /// Serialize Message with postcard and send as a single frame.
    async fn send(&self, msg: &Message) -> Result<(), Error> {
        let bytes = postcard::to_stdvec(msg)
            .context("postcard serialize Message")
            .map_err(to_ipc_error)?;
        self.inner.send_frame(Bytes::from(bytes)).await
    }

    /// Receive one frame, deserialize with postcard into Message.
    async fn recv(&self) -> Result<Option<Message>, Error> {
        let maybe = self.inner.recv_frame().await?;
        let Some(frame) = maybe else {
            return Ok(None);
        };
        let msg = postcard::from_bytes::<Message>(&frame)
            .context("postcard deserialize Message")
            .map_err(to_ipc_error)?;
        Ok(Some(msg))
    }
}

/// Convert `anyhow::Error` into crate Error, preserving context.
fn to_ipc_error(e: anyhow::Error) -> Error {
    e.into()
}

/// Simple blob store interface used for blob-backed flows.
///
/// In production this would map to a real data plane (shared memory, file,
/// socket, etc). For now it is used for integration tests.
#[async_trait]
pub trait BlobStore: Send + Sync {
    async fn put(
        &self,
        token: crate::workspace::ipc::types::BlobToken,
        bytes: Bytes,
    ) -> Result<(), Error>;

    async fn take(
        &self,
        token: &crate::workspace::ipc::types::BlobToken,
    ) -> Result<Bytes, Error>;

    async fn len(&self) -> Result<usize, Error>;
}

/// In-memory `BlobStore` used by tests (and as a reference implementation).
#[derive(Debug, Default)]
pub struct InMemoryBlobStore {
    inner: tokio::sync::Mutex<
        HashMap<crate::workspace::ipc::types::BlobToken, Bytes>,
    >,
}

#[async_trait]
impl BlobStore for InMemoryBlobStore {
    async fn put(
        &self,
        token: crate::workspace::ipc::types::BlobToken,
        bytes: Bytes,
    ) -> Result<(), Error> {
        let mut guard = self.inner.lock().await;
        guard.insert(token, bytes);
        Ok(())
    }

    async fn take(
        &self,
        token: &crate::workspace::ipc::types::BlobToken,
    ) -> Result<Bytes, Error> {
        let mut guard = self.inner.lock().await;
        let Some(bytes) = guard.remove(token) else {
            return Err(to_ipc_error(anyhow!("unknown blob token")));
        };
        Ok(bytes)
    }

    async fn len(&self) -> Result<usize, Error> {
        let guard = self.inner.lock().await;
        Ok(guard.len())
    }
}

// ---- Default RpcClient implementation ----

/// Default `RpcClient` built over a Session. It:
/// - Correlates responses by `RequestId`.
/// - Forwards cancellation requests (server-driven cancellation is expected).
/// - Emits server Events via a bounded mpsc channel with backpressure.
///
/// The client spawns a background task to dispatch incoming Messages.
pub struct DefaultRpcClient<S: Session> {
    session: Arc<S>,
    connection_id: Option<ConnectionId>,
    inflight: tokio::sync::Mutex<HashMap<RequestId, oneshot::Sender<Response>>>,
    pending: tokio::sync::Mutex<HashMap<RequestId, Response>>,
    events_tx: mpsc::Sender<Event>,
    events_rx: std::sync::Mutex<Option<mpsc::Receiver<Event>>>,
    streams: Arc<tokio::sync::Mutex<HashMap<StreamId, IncomingStreamState>>>,
}

impl<S: Session + 'static> DefaultRpcClient<S> {
    /// Construct a new `RpcClient` over the given Session. The events channel is
    /// bounded by `events_capacity` to provide backpressure.
    pub fn new(
        session: std::sync::Arc<S>,
        events_capacity: usize,
    ) -> std::sync::Arc<Self> {
        Self::new_with_connection_id(session, None, events_capacity)
    }

    /// Construct a new `RpcClient` with an optional connection id for logging.
    pub fn new_with_connection_id(
        session: std::sync::Arc<S>,
        connection_id: Option<ConnectionId>,
        events_capacity: usize,
    ) -> std::sync::Arc<Self> {
        let (events_tx, events_rx) = mpsc::channel(events_capacity);
        let client = std::sync::Arc::new(Self {
            session: session.clone(),
            connection_id,
            inflight: tokio::sync::Mutex::new(HashMap::new()),
            pending: tokio::sync::Mutex::new(HashMap::new()),
            events_tx,
            events_rx: std::sync::Mutex::new(Some(events_rx)),
            streams: Arc::new(tokio::sync::Mutex::new(HashMap::new())),
        });

        Self::spawn_dispatcher(client.clone());
        client
    }

    async fn handle_stream_control(
        &self,
        ctl: StreamControl,
    ) -> Result<(), Error> {
        let id = match &ctl {
            StreamControl::Start { id, .. } => *id,
            StreamControl::Next { id, .. } => *id,
            StreamControl::End { id, .. } => *id,
        };

        let span = if let Some(conn_id) = self.connection_id {
            tracing::info_span!("ipc.rpc.stream_control", conn_id = %conn_id, stream_id = id)
        } else {
            tracing::info_span!("ipc.rpc.stream_control", stream_id = id)
        };

        async move {
            match ctl {
                StreamControl::Start { .. } => {
                    let mut guard = self.streams.lock().await;
                    guard
                        .entry(id)
                        .or_insert_with(|| IncomingStreamState::new(64));
                    Ok(())
                }
                StreamControl::Next { chunk, .. } => {
                    let Some(chunk) = chunk else {
                        // Blob-backed or out-of-band; nothing to deliver on control plane.
                        return Ok(());
                    };

                    let mut guard = self.streams.lock().await;
                    let st = guard
                        .entry(id)
                        .or_insert_with(|| IncomingStreamState::new(64));
                    st.tx.send(Bytes::from(chunk)).await.map_err(|e| {
                        to_ipc_error(anyhow!("stream send failed: {e}"))
                    })?;
                    Ok(())
                }
                StreamControl::End { .. } => {
                    let mut guard = self.streams.lock().await;
                    guard.remove(&id);
                    Ok(())
                }
            }
        }
        .instrument(span)
        .await
    }

    #[cfg(test)]
    async fn stream_count(&self) -> usize {
        let guard = self.streams.lock().await;
        guard.len()
    }
}

struct IncomingStreamState {
    tx: mpsc::Sender<Bytes>,
    rx: Option<mpsc::Receiver<Bytes>>,
}

impl IncomingStreamState {
    fn new(capacity: usize) -> Self {
        let (tx, rx) = mpsc::channel(capacity);
        Self { tx, rx: Some(rx) }
    }
}

impl<S: Session + 'static> DefaultRpcClient<S> {
    fn spawn_dispatcher(client: std::sync::Arc<Self>) {
        tokio::spawn(async move {
            loop {
                let msg = match client.session.recv().await {
                    Ok(Some(m)) => m,
                    Ok(None) => {
                        // EOF: fail all inflight waiters with an error.
                        let mut inflight = client.inflight.lock().await;
                        for (id, sender) in inflight.drain() {
                            let _ = sender.send(Response {
                                id,
                                ok: false,
                                result: None,
                                error: Some(RpcError {
                                    code: "eof".to_string(),
                                    message:
                                        "session closed while awaiting response"
                                            .to_string(),
                                }),
                            });
                        }
                        break;
                    }
                    Err(e) => {
                        // On session error, fail all inflight waiters.
                        let mut inflight = client.inflight.lock().await;
                        for (id, sender) in inflight.drain() {
                            let _ = sender.send(Response {
                                id,
                                ok: false,
                                result: None,
                                error: Some(RpcError {
                                    code: "session_recv_error".to_string(),
                                    message: format!("{e:#}"),
                                }),
                            });
                        }
                        break;
                    }
                };

                match msg {
                    Message::Response(resp) => {
                        let id = response_id(&resp);
                        let span = if let Some(conn_id) = client.connection_id {
                            tracing::info_span!("ipc.rpc.recv_response_msg", conn_id = %conn_id, request_id = id)
                        } else {
                            tracing::info_span!(
                                "ipc.rpc.recv_response_msg",
                                request_id = id
                            )
                        };
                        async {
                            if let Some(sender) =
                                client.inflight.lock().await.remove(&id)
                            {
                                let _ = sender.send(resp);
                            } else {
                                client.pending.lock().await.insert(id, resp);
                            }
                        }
                        .instrument(span)
                        .await;
                    }
                    Message::Event(ev) => {
                        // Apply backpressure by awaiting send.
                        if let Err(_e) = client.events_tx.send(ev).await {
                            // Receiver dropped; ignore but keep draining input.
                        }
                    }
                    Message::Stream(ctl) => {
                        let _ = client.handle_stream_control(ctl).await;
                    }
                    // Not handled by client:
                    Message::Hello(_)
                    | Message::HelloOk(_)
                    | Message::HelloErr(_)
                    | Message::Request(_)
                    | Message::Cancel(_) => {}
                }
            }
        });
    }

    async fn handle_pending_response(
        &self,
        id: RequestId,
    ) -> Result<Response, Error> {
        if let Some(resp) = self.pending.lock().await.remove(&id) {
            return Ok(resp);
        }

        let (tx, rx) = oneshot::channel();
        {
            let mut inflight = self.inflight.lock().await;
            if let Some(resp) = self.pending.lock().await.remove(&id) {
                return Ok(resp);
            }
            inflight.insert(id, tx);
        }

        match rx.await {
            Ok(resp) => Ok(resp),
            Err(e) => Err(to_ipc_error(anyhow!(
                "response waiter dropped for id={id:?}: {e}"
            ))),
        }
    }
}

#[async_trait]
impl<S: Session + 'static> RpcClient for DefaultRpcClient<S> {
    /// Send the request envelope to the remote endpoint.
    async fn send_request(&self, req: Request) -> Result<(), Error> {
        let span = if let Some(conn_id) = self.connection_id {
            tracing::info_span!("ipc.rpc.send_request", conn_id = %conn_id, request_id = req.id, service = %req.method.service, method = %req.method.method)
        } else {
            tracing::info_span!("ipc.rpc.send_request", request_id = req.id, service = %req.method.service, method = %req.method.method)
        };
        async move { self.session.send(&Message::Request(req)).await }
            .instrument(span)
            .await
    }

    /// Await the response correlated by `RequestId`. If the response arrived
    /// before this method was called, it is returned immediately.
    async fn recv_response(&self, id: RequestId) -> Result<Response, Error> {
        let span = if let Some(conn_id) = self.connection_id {
            tracing::info_span!("ipc.rpc.recv_response", conn_id = %conn_id, request_id = id)
        } else {
            tracing::info_span!("ipc.rpc.recv_response", request_id = id)
        };
        async move { self.handle_pending_response(id).await }
            .instrument(span)
            .await
    }

    /// Forward the cancellation to the server. Server-driven cancellation is
    /// expected to materialize as a Response that the dispatcher routes to the
    /// waiter. This method does not fail the waiter eagerly.
    async fn cancel(&self, cancel: Cancel) -> Result<(), Error> {
        let span = if let Some(conn_id) = self.connection_id {
            tracing::info_span!("ipc.rpc.cancel", conn_id = %conn_id, request_id = cancel.id)
        } else {
            tracing::info_span!("ipc.rpc.cancel", request_id = cancel.id)
        };
        async move {
            // Resolve any local waiter immediately.
            if let Some(sender) = self.inflight.lock().await.remove(&cancel.id)
            {
                let _ = sender.send(Response {
                    id: cancel.id,
                    ok: false,
                    result: None,
                    error: Some(RpcError::cancelled(
                        "request cancelled by client",
                    )),
                });
            }
            let _ = self.pending.lock().await.remove(&cancel.id);

            self.session.send(&Message::Cancel(cancel)).await
        }
        .instrument(span)
        .await
    }

    fn events(&self) -> Pin<Box<dyn Stream<Item = Event> + Send>> {
        let mut guard = match self.events_rx.lock() {
            Ok(g) => g,
            Err(_) => return Box::pin(futures_util::stream::empty()),
        };

        if let Some(rx) = guard.take() {
            Box::pin(ReceiverStream::new(rx))
        } else {
            Box::pin(futures_util::stream::empty())
        }
    }
}

#[async_trait]
impl<S: Session + 'static> StreamManager for DefaultRpcClient<S> {
    async fn start(&self, ctl: StreamControl) -> Result<(), Error> {
        self.session.send(&Message::Stream(ctl)).await
    }

    async fn next(&self, ctl: StreamControl) -> Result<(), Error> {
        self.session.send(&Message::Stream(ctl)).await
    }

    async fn end(&self, ctl: StreamControl) -> Result<(), Error> {
        self.session.send(&Message::Stream(ctl)).await
    }

    fn stream_incoming(
        &self,
        id: StreamId,
    ) -> Pin<Box<dyn Stream<Item = Bytes> + Send>> {
        let streams = self.streams.clone();
        let (tx, rx) = mpsc::channel::<Bytes>(1);

        tokio::spawn(async move {
            let mut guard = streams.lock().await;
            if let Some(state) = guard.get_mut(&id) {
                if let Some(mut original_rx) = state.rx.take() {
                    drop(guard);
                    while let Some(bytes) = original_rx.recv().await {
                        if tx.send(bytes).await.is_err() {
                            break;
                        }
                    }
                }
            }
        });

        Box::pin(ReceiverStream::new(rx))
    }
}

// ---- Helpers (minimal assumptions about protocol types) ----

/// Extract the `RequestId` from a Response.
fn response_id(resp: &Response) -> RequestId {
    resp.id
}

// ---- Tests: mocked Session, correlation, out-of-order, cancellation ----
// These tests use a mock Session and the public RpcClient API.

#[cfg(test)]
mod tests {
    use crate::workspace::ipc::protocol::MethodId;

    use super::*;
    use anyhow::{Context as _, Result};

    /// A simple in-memory Session mock backed by channels carrying Message.
    struct MockSession {
        to_server: mpsc::Sender<Message>,
        from_server: tokio::sync::Mutex<mpsc::Receiver<Message>>,
    }

    #[async_trait]
    impl Session for MockSession {
        async fn send(&self, msg: &Message) -> Result<(), Error> {
            self.to_server
                .send(msg.clone())
                .await
                .map_err(|e| to_ipc_error(anyhow!("send failed: {}", e)))
        }

        async fn recv(&self) -> Result<Option<Message>, Error> {
            let mut rx = self.from_server.lock().await;
            Ok(rx.recv().await)
        }
    }

    fn make_mock_session() -> (
        std::sync::Arc<MockSession>,
        mpsc::Receiver<Message>,
        mpsc::Sender<Message>,
    ) {
        let (to_server_tx, to_server_rx) = mpsc::channel(16);
        let (from_server_tx, from_server_rx) = mpsc::channel(16);
        let session = std::sync::Arc::new(MockSession {
            to_server: to_server_tx,
            from_server: tokio::sync::Mutex::new(from_server_rx),
        });
        (session, to_server_rx, from_server_tx)
    }

    /// Sends two concurrent requests and verifies that out-of-order responses
    /// are matched to the correct waiters.
    #[crate::ctb_test(tokio::test)]
    async fn rpc_correlation_out_of_order() -> Result<()> {
        let (session, mut to_server_rx, from_server_tx) = make_mock_session();
        let client = DefaultRpcClient::new(session, DEFAULT_EVENTS_CAPACITY);

        // Create two requests
        let req1 = Request {
            id: 1,
            method: MethodId {
                service: "test".to_string(),
                method: "method1".to_string(),
            },
            args: vec![],
        };
        let req2 = Request {
            id: 2,
            method: MethodId {
                service: "test".to_string(),
                method: "method2".to_string(),
            },
            args: vec![],
        };

        // Send requests concurrently
        let send1 = client.send_request(req1.clone());
        let send2 = client.send_request(req2.clone());
        tokio::try_join!(send1, send2)?;

        let sent_req1 = to_server_rx.recv().await.context("missing req1")?;
        let sent_req2 = to_server_rx.recv().await.context("missing req2")?;
        assert!(matches!(sent_req1, Message::Request(r) if r.id == 1));
        assert!(matches!(sent_req2, Message::Request(r) if r.id == 2));

        // Spawn tasks to await responses
        let client2 = client.clone();
        let recv1_handle =
            tokio::spawn(async move { client.recv_response(1).await });
        let recv2_handle =
            tokio::spawn(async move { client2.recv_response(2).await });

        // Send responses out of order (id 2 first, then id 1)
        let resp2 = Response {
            id: 2,
            ok: true,
            result: Some(vec![0xBB]),
            error: None,
        };
        let resp1 = Response {
            id: 1,
            ok: true,
            result: Some(vec![0xAA]),
            error: None,
        };
        from_server_tx.send(Message::Response(resp2)).await?;
        from_server_tx.send(Message::Response(resp1)).await?;

        let (resp1_recv, resp2_recv) =
            tokio::try_join!(recv1_handle, recv2_handle)?;
        let resp1_recv = resp1_recv?;
        let resp2_recv = resp2_recv?;
        assert_eq!(resp1_recv.id, 1);
        assert_eq!(resp1_recv.result, Some(vec![0xAA]));
        assert_eq!(resp2_recv.id, 2);
        assert_eq!(resp2_recv.result, Some(vec![0xBB]));
        Ok(())
    }

    /// Cancels a request and verifies downstream behavior (e.g., server sends
    /// a cancellation response routed to the correct waiter).
    #[crate::ctb_test(tokio::test)]
    async fn rpc_cancellation() -> Result<()> {
        let (session, mut to_server_rx, from_server_tx) = make_mock_session();
        let client = DefaultRpcClient::new(session, DEFAULT_EVENTS_CAPACITY);

        // Send a request
        let req = Request {
            id: 1,
            method: MethodId {
                service: "test".to_string(),
                method: "method".to_string(),
            },
            args: vec![],
        };
        client.send_request(req).await?;

        // Verify request was sent
        let sent_req = to_server_rx.recv().await.context("missing request")?;
        assert!(matches!(sent_req, Message::Request(r) if r.id == 1));

        // Cancel the request
        let cancel = Cancel { id: 1 };
        client.cancel(cancel).await?;

        // Verify cancel was sent
        let sent_cancel =
            to_server_rx.recv().await.context("missing cancel")?;
        assert!(matches!(sent_cancel, Message::Cancel(c) if c.id == 1));

        // Server sends a cancellation response too (client should handle either).
        let resp = Response {
            id: 1,
            ok: false,
            result: None,
            error: Some(RpcError::cancelled("request cancelled")),
        };
        from_server_tx.send(Message::Response(resp)).await?;

        let recv_resp = client.recv_response(1).await?;
        assert_eq!(recv_resp.id, 1);
        assert!(!recv_resp.ok);
        assert_eq!(recv_resp.error.unwrap().code, "cancelled");
        Ok(())
    }

    /// Cancel an in-flight "fetch" and ensure work stops and a cancelled response is sent.
    #[ignore = "test is incomplete"]
    #[crate::ctb_test(tokio::test)]
    async fn server_cancel_stops_inflight_fetch() -> Result<()> {
        Ok(())
        /*let (session, mut to_server_rx, from_server_tx) = make_mock_session();

        let steps = std::sync::Arc::new(AtomicUsize::new(0));
        let router = SimpleRouter::new();
        router.with_network_service(Arc::new(MockNetworkService::new()));

        let ctx = ConnectionContext {
            id: ConnectionId::default(),
            capabilities: CapabilitySet::default(),
            metadata: None,
        };

        let server_task = tokio::spawn(serve_router(session.clone(), router, ctx));

        // Send a fetch request into the server loop.
        from_server_tx
            .send(Message::Request(Request {
                id: 7,
                method: MethodId {
                    service: "network".into(),
                    method: "fetch".into(),
                },
                args: vec![1, 2, 3],
            }))
            .await?;

        // Ensure it started doing work.
        tokio::time::sleep(std::time::Duration::from_millis(40)).await;
        let before = steps.load(Ordering::SeqCst);

        // Cancel it.
        from_server_tx.send(Message::Cancel(Cancel::new(7))).await?;

        // Observe the server emitting a cancelled response.
        let msg = to_server_rx.recv().await.context("missing response")?;
        let Message::Response(resp) = msg else {
            anyhow::bail!("expected Response");
        };
        assert_eq!(resp.id, 7);
        assert_eq!(resp.error.context("missing error")?.code, "cancelled");

        // Work should stop growing soon after cancellation.
        tokio::time::sleep(std::time::Duration::from_millis(50)).await;
        let after = steps.load(Ordering::SeqCst);
        assert!(after >= before);

        // Stop the server task by dropping channels/session.
        drop(from_server_tx);
        let _ = server_task.await?;
        Ok(())*/
    }
}
