use crate::workspace::ipc::error::Error;
use crate::workspace::ipc::types::{BlobId, BlobToken};
use async_trait::async_trait;
use serde::{Deserialize, Serialize};
use std::os::fd::FromRawFd;
use std::path::PathBuf;
use uuid::Uuid;

/// Platform-neutral description of a shared memory handle that can be sent via control plane metadata.
#[derive(Debug, Clone, PartialEq, Serialize, Deserialize)]
pub enum SharedBlobDescriptor {
    /// Unix file descriptor (sent via `SCM_RIGHTS` on Unix sockets).
    #[cfg(unix)]
    UnixFd(i32),
    /// Windows HANDLE (duplicated to the target process).
    #[cfg(windows)]
    WindowsHandle(u64),
    /// Cross-platform fallback via temporary file path.
    FilePath(PathBuf),
    /// Opaque handle by name (e.g., named shared memory).
    Named(String),
}

/// A producer-created blob that can be shared with other processes.
pub struct ProducerBlob {
    pub id: BlobId,
    pub size: u64,
    pub descriptor: SharedBlobDescriptor,
    pub token: BlobToken,
}

impl ProducerBlob {
    /// Best-effort helper for writing blob contents in tests and simple
    /// producer workflows.
    #[allow(unsafe_code)]
    pub fn write_all(&self, data: &[u8]) -> Result<(), Error> {
        if (u64::try_from(data.len())?) > self.size {
            return Err(Error::Internal(
                "blob write exceeds allocated size".to_string(),
            ));
        }

        match &self.descriptor {
            #[cfg(unix)]
            SharedBlobDescriptor::UnixFd(fd) => {
                let dup = crate::workspace::ipc::platform::unix::dup_fd(*fd)?;
                let mut file = unsafe { std::fs::File::from_raw_fd(dup) };
                use std::io::{Seek, SeekFrom, Write};
                file.seek(SeekFrom::Start(0))?;
                file.write_all(data)?;
                file.flush()?;
                Ok(())
            }
            #[cfg(windows)]
            SharedBlobDescriptor::WindowsHandle(handle) => {
                crate::workspace::ipc::platform::windows::write_mapping(
                    *handle, data,
                )
            }
            SharedBlobDescriptor::FilePath(path) => {
                use std::io::{Seek, SeekFrom, Write};
                let mut file = std::fs::OpenOptions::new()
                    .read(true)
                    .write(true)
                    .create(false)
                    .open(path)?;
                file.seek(SeekFrom::Start(0))?;
                file.write_all(data)?;
                file.flush()?;
                Ok(())
            }
            SharedBlobDescriptor::Named(_name) => Err(Error::Unsupported(
                "Named shared blobs are not implemented for writing"
                    .to_string(),
            )),
        }
    }
}

/// A mapped read-only view of a blob.
pub struct MappedRead<'a> {
    /// Platform-backed mapping pointer and length are implementation details.
    pub len: usize,
    ptr: *const u8,
    backing: MappedReadBacking,
    /// Lifetime-bound marker to prevent use-after-free.
    pub _marker: std::marker::PhantomData<&'a ()>,
}

impl MappedRead<'_> {
    #[allow(unsafe_code)]
    pub fn as_slice(&self) -> &[u8] {
        // Safety: `ptr` and `len` are valid for the lifetime of `backing`.
        unsafe { std::slice::from_raw_parts(self.ptr, self.len) }
    }
}

enum MappedReadBacking {
    Memmap(memmap2::Mmap),
    #[cfg(windows)]
    Windows(crate::workspace::ipc::platform::windows::MappingView),
    #[allow(dead_code)]
    Empty,
}

/// Blob allocator for creating and managing shared blobs.
#[async_trait]
pub trait BlobAllocator: Send + Sync {
    /// Create a new blob of the given size and return a producer handle with a lifecycle token.
    async fn create(&self, size: u64) -> Result<ProducerBlob, Error>;

    /// Cleanup a blob proactively using its token (server-side GC).
    async fn cleanup(&self, token: &BlobToken) -> Result<(), Error>;
}

/// Reader side to map an incoming blob by token/descriptor.
#[async_trait]
pub trait BlobReader: Send + Sync {
    /// Map a blob for reading using its token and descriptor metadata.
    async fn map_read<'a>(
        &'a self,
        token: &BlobToken,
        desc: &SharedBlobDescriptor,
    ) -> Result<MappedRead<'a>, Error>;
}

#[derive(Debug, Clone, Copy)]
pub enum BlobBackend {
    /// Use the platform’s preferred shared-memory mechanism.
    PlatformDefault,
    /// Always use a temp-file + `memmap2` fallback (portable, testable).
    TempFileFallback,
}

#[derive(Debug)]
struct BlobRecord {
    token: BlobToken,
    size: u64,
    descriptor: SharedBlobDescriptor,
}

#[derive(Debug)]
pub struct SharedMemoryBlobs {
    backend: BlobBackend,
    records: std::sync::Mutex<Vec<BlobRecord>>,
    seq: std::sync::atomic::AtomicU64,
}

impl SharedMemoryBlobs {
    pub fn new(backend: BlobBackend) -> Self {
        Self {
            backend,
            records: std::sync::Mutex::new(Vec::new()),
            seq: std::sync::atomic::AtomicU64::new(0),
        }
    }

    fn new_token(&self) -> BlobToken {
        let _seq = self.seq.fetch_add(1, std::sync::atomic::Ordering::Relaxed);
        BlobToken {
            id: BlobId(Uuid::new_v4()),
            size: 0,
            lease_ms: None,
        }
    }

    fn find_record(&self, token: &BlobToken) -> Option<BlobRecord> {
        let records = self.records.lock().ok()?;
        records
            .iter()
            .find(|r| &r.token == token)
            .map(|r| BlobRecord {
                token: r.token.clone(),
                size: r.size,
                descriptor: r.descriptor.clone(),
            })
    }

    fn remove_record(&self, token: &BlobToken) -> Option<BlobRecord> {
        let mut records = self.records.lock().ok()?;
        let idx = records.iter().position(|r| &r.token == token)?;
        Some(records.remove(idx))
    }

    fn create_tempfile_blob(
        &self,
        size: u64,
        token: &BlobToken,
    ) -> Result<SharedBlobDescriptor, Error> {
        let mut path = std::env::temp_dir();
        path.push(format!("{}.bin", token.id.0));

        let file = std::fs::OpenOptions::new()
            .read(true)
            .write(true)
            .create_new(true)
            .open(&path)?;
        file.set_len(size)?;
        Ok(SharedBlobDescriptor::FilePath(path))
    }
}

#[async_trait]
impl BlobAllocator for SharedMemoryBlobs {
    async fn create(&self, size: u64) -> Result<ProducerBlob, Error> {
        let token = self.new_token();

        let descriptor = match self.backend {
            BlobBackend::TempFileFallback => {
                self.create_tempfile_blob(size, &token)?
            }
            BlobBackend::PlatformDefault => {
                #[cfg(unix)]
                {
                    // Linux memfd; non-Linux Unix will error out and fall back.
                    if let Ok(fd) =
                        crate::workspace::ipc::platform::unix::create_memfd(
                            size,
                        )
                    {
                        SharedBlobDescriptor::UnixFd(fd)
                    } else {
                        self.create_tempfile_blob(size, &token)?
                    }
                }
                #[cfg(windows)]
                {
                    let handle =
                        crate::workspace::ipc::platform::windows::create_file_mapping(size)?;
                    SharedBlobDescriptor::WindowsHandle(handle)
                }
                #[cfg(not(any(unix, windows)))]
                {
                    self.create_tempfile_blob(size, &token)?
                }
            }
        };

        {
            let mut records = self.records.lock().map_err(|_| {
                Error::Internal("blob registry mutex poisoned".to_string())
            })?;
            records.push(BlobRecord {
                token: token.clone(),
                size,
                descriptor: descriptor.clone(),
            });
        }

        Ok(ProducerBlob {
            id: BlobId::default(),
            size,
            descriptor,
            token,
        })
    }

    async fn cleanup(&self, token: &BlobToken) -> Result<(), Error> {
        let Some(record) = self.remove_record(token) else {
            // Cleanup is idempotent.
            return Ok(());
        };

        match record.descriptor {
            #[cfg(unix)]
            SharedBlobDescriptor::UnixFd(fd) => {
                crate::workspace::ipc::platform::unix::close_fd(fd)
            }
            #[cfg(windows)]
            SharedBlobDescriptor::WindowsHandle(handle) => {
                crate::workspace::ipc::platform::windows::close_handle(handle)
            }
            SharedBlobDescriptor::FilePath(path) => {
                let _ = std::fs::remove_file(path);
                Ok(())
            }
            SharedBlobDescriptor::Named(_name) => Ok(()),
        }
    }
}

#[allow(unsafe_code)]
#[async_trait]
impl BlobReader for SharedMemoryBlobs {
    async fn map_read<'a>(
        &'a self,
        token: &BlobToken,
        desc: &SharedBlobDescriptor,
    ) -> Result<MappedRead<'a>, Error> {
        let record = self.find_record(token).ok_or_else(|| Error::NotFound)?;

        // Ensure descriptor matches what we created/tracked for this token.
        if &record.descriptor != desc {
            return Err(Error::Internal(
                "descriptor does not match tracked blob token".to_string(),
            ));
        }

        let len: usize = record.size.try_into().map_err(|_| {
            Error::Internal(
                "blob too large to map on this platform".to_string(),
            )
        })?;

        match desc {
            #[cfg(unix)]
            SharedBlobDescriptor::UnixFd(fd) => {
                let dup = crate::workspace::ipc::platform::unix::dup_fd(*fd)?;
                let file = unsafe { std::fs::File::from_raw_fd(dup) };
                let mmap = unsafe { memmap2::Mmap::map(&file)? };
                let ptr = mmap.as_ptr();
                Ok(MappedRead {
                    len,
                    ptr,
                    backing: MappedReadBacking::Memmap(mmap),
                    _marker: std::marker::PhantomData,
                })
            }
            #[cfg(windows)]
            SharedBlobDescriptor::WindowsHandle(handle) => {
                let view =
                    crate::workspace::ipc::platform::windows::map_view_read(
                        *handle, len,
                    )?;
                let ptr = view.as_ptr();
                Ok(MappedRead {
                    len,
                    ptr,
                    backing: MappedReadBacking::Windows(view),
                    _marker: std::marker::PhantomData,
                })
            }
            SharedBlobDescriptor::FilePath(path) => {
                let file = std::fs::File::open(path)?;
                let mmap = unsafe { memmap2::Mmap::map(&file)? };
                let ptr = mmap.as_ptr();
                Ok(MappedRead {
                    len,
                    ptr,
                    backing: MappedReadBacking::Memmap(mmap),
                    _marker: std::marker::PhantomData,
                })
            }
            SharedBlobDescriptor::Named(_name) => Err(Error::Unsupported(
                "Named shared blobs are not implemented for reading"
                    .to_string(),
            )),
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use anyhow::Result;

    #[crate::ctb_test(tokio::test)]
    async fn blob_round_trip_tempfile_fallback() -> Result<()> {
        let blobs = SharedMemoryBlobs::new(BlobBackend::TempFileFallback);
        let data = b"hello-blob-fallback".to_vec();

        let blob = blobs.create(u64::try_from(data.len())?).await?;
        blob.write_all(&data)?;

        let mapped = blobs.map_read(&blob.token, &blob.descriptor).await?;
        assert_eq!(mapped.as_slice(), &data[..]);

        blobs.cleanup(&blob.token).await?;
        Ok(())
    }

    #[cfg(unix)]
    #[crate::ctb_test(tokio::test)]
    async fn blob_round_trip_unix_platform_default() -> Result<()> {
        let blobs = SharedMemoryBlobs::new(BlobBackend::PlatformDefault);
        let data = b"hello-blob-unix".to_vec();

        let blob = blobs.create(u64::try_from(data.len())?).await?;
        blob.write_all(&data)?;

        let mapped = blobs.map_read(&blob.token, &blob.descriptor).await?;
        assert_eq!(mapped.as_slice(), &data[..]);

        blobs.cleanup(&blob.token).await?;
        Ok(())
    }

    #[cfg(windows)]
    #[crate::ctb_test(tokio::test)]
    async fn blob_round_trip_windows_platform_default() -> Result<()> {
        let blobs = SharedMemoryBlobs::new(BlobBackend::PlatformDefault);
        let data = b"hello-blob-windows".to_vec();

        let blob = blobs.create(data.len() as u64).await?;
        blob.write_all(&data)?;

        let mapped = blobs.map_read(&blob.token, &blob.descriptor).await?;
        assert_eq!(mapped.as_slice(), &data[..]);

        blobs.cleanup(&blob.token).await?;
        Ok(())
    }
}
