use serde::{Deserialize, Serialize};
use std::collections::HashMap;

/// An unguessable capability token bound to a single connection/process.
#[derive(
    Debug, Clone, PartialEq, Eq, Hash, Serialize, Deserialize, Default,
)]
pub struct CapabilityToken(pub String);

/// A fully-resolved capability set derived from a token.
#[derive(Debug, Clone, Serialize, Deserialize, Default)]
pub struct CapabilitySet {
    /// Allowed methods per service, including optional quotas/limits.
    pub allowed: HashMap<ServiceName, Vec<MethodRule>>,
    /// Optional global quotas or ceilings.
    pub global_limits: Option<GlobalLimits>,
}

/// Logical service identifier (human-readable, stable across versions).
#[derive(Debug, Clone, PartialEq, Eq, Hash, Serialize, Deserialize)]
pub struct ServiceName(pub String);

/// Pattern/rule controlling access to a method with optional quotas.
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct MethodRule {
    pub method: MethodSelector,
    pub quotas: Option<QuotaSet>,
}

/// A method selector can be exact or wildcard.
#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum MethodSelector {
    /// Exact service.method string, e.g., "storage.get".
    Exact(String),
    /// Prefix match, e.g., "network." allows "network.get" and "network.post".
    Prefix(String),
    /// Allow all methods in the service.
    Any,
}

impl MethodSelector {
    /// Check if this selector matches the given service and method.
    pub fn matches(&self, service: &str, method: &str) -> bool {
        let full_name = format!("{service}.{method}");
        match self {
            MethodSelector::Exact(s) => s == method || s == &full_name,
            MethodSelector::Prefix(prefix) => {
                method.starts_with(prefix) || full_name.starts_with(prefix)
            }
            MethodSelector::Any => true,
        }
    }
}

/// Quotas applicable to a method.
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct QuotaSet {
    /// Optional bytes/sec rate limit.
    pub bytes_per_sec: Option<u64>,
    /// Optional ops/sec limit.
    pub ops_per_sec: Option<u64>,
    /// Optional burst capacity.
    pub burst: Option<u64>,
}

impl QuotaSet {
    /// Compute the effective burst capacity for a bytes/sec token bucket.
    ///
    /// If `burst` is not specified, this defaults to allowing a 1-second burst
    /// at the configured rate.
    pub fn effective_burst_bytes(&self) -> Option<u64> {
        let Some(rate) = self.bytes_per_sec else {
            return None;
        };
        Some(self.burst.unwrap_or(rate))
    }

    /// Compute the effective burst capacity for an ops/sec token bucket.
    ///
    /// If `burst` is not specified, this defaults to allowing a 1-second burst
    /// at the configured rate.
    pub fn effective_burst_ops(&self) -> Option<u64> {
        let Some(rate) = self.ops_per_sec else {
            return None;
        };
        Some(self.burst.unwrap_or(rate))
    }
}

/// Global limits across the connection/process.
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct GlobalLimits {
    pub max_concurrent_requests: Option<u32>,
    pub max_streams: Option<u32>,
    pub max_blob_bytes: Option<u64>,
}

impl GlobalLimits {
    /// Whether this set contains any configured limits.
    pub fn is_empty(&self) -> bool {
        self.max_concurrent_requests.is_none()
            && self.max_streams.is_none()
            && self.max_blob_bytes.is_none()
    }
}

/// A capability bundle given to the workspace to spawn a child process.
#[derive(Debug, Clone, Serialize, Deserialize, Default)]
pub struct CapabilityBundle {
    /// Initial capability token to authenticate the child’s control connection.
    pub token: CapabilityToken,
    /// Initial capability set bound to the connection.
    pub capabilities: CapabilitySet,
}

/// Validates a capability token and derives a bound `CapabilitySet`.
/// Implementations should avoid panics and return errors for invalid tokens.
pub trait TokenValidator: Send + Sync {
    /// Validate a token, producing a `CapabilitySet` on success.
    fn validate(
        &self,
        token: &CapabilityToken,
    ) -> Result<CapabilitySet, anyhow::Error>;
}

#[cfg(test)]
mod tests {
    use super::*;
    use anyhow::Result;

    /// A simple fake validator for tests. Accepts "ok", rejects others.
    struct FakeTokenValidator;

    impl TokenValidator for FakeTokenValidator {
        fn validate(
            &self,
            token: &CapabilityToken,
        ) -> Result<CapabilitySet, anyhow::Error> {
            if token.0 == "ok" {
                Ok(CapabilitySet::default())
            } else {
                Err(anyhow::anyhow!("invalid token"))
            }
        }
    }

    #[crate::ctb_test]
    fn fake_validator_ok() -> Result<()> {
        let v = FakeTokenValidator;
        let set = v.validate(&CapabilityToken("ok".into()))?;
        let _ = set; // success
        Ok(())
    }

    #[crate::ctb_test]
    fn fake_validator_err() -> Result<()> {
        let v = FakeTokenValidator;
        let res = v.validate(&CapabilityToken("bad".into()));
        assert!(res.is_err());
        Ok(())
    }
}
