//! Utilities for process management, including tracking and spawning subprocesses.

use anyhow::{Result, anyhow};
use serde::{Deserialize, Serialize};
use std::collections::HashMap;
use std::env;
use std::io::Write;
use std::process::{Command, Stdio};
use std::sync::Arc;
use std::sync::Mutex;

use crate::json;
use crate::utilities::ipc::{Channel, IPC_ARG};
use crate::utilities::{generate_authentication_key, send_message};

/// Represents a managed process.
#[derive(Clone, Serialize, Deserialize, Debug)]
pub struct Process {
    pub pid: u64,
    pub system_pid: u64,
    pub channel: Channel,
    /// PID of the workspace or renderer process that started this process.
    pub parent_pid: u64,
}

/// Represents a group or job handle for managing process subtrees.
#[derive(Clone, Serialize, Deserialize, Default, Debug)]
pub struct ProcessGroup {
    pub group_id: u64,
    pub members: Vec<u64>,
    // Add more fields as needed for job control.
}

/// Manages all workspace and subprocesses.
#[derive(Clone, Serialize, Deserialize, Default, Debug)]
pub struct ProcessManager {
    pub port: u16,
    pub ipc_control_channel: Channel,
    pub processes: HashMap<u64, Process>,
    /// Monotonic counter for process IDs.
    pub last_id: u64,
    /// Tracks process groups/job handles for subtree management.
    pub process_groups: HashMap<u64, ProcessGroup>,
}

impl ProcessManager {
    /// Returns the last assigned process ID.
    pub fn last_index(&self) -> u64 {
        self.last_id
    }
    /// Returns the next process ID to be assigned.
    pub fn next_index(&self) -> u64 {
        self.last_id + 1
    }
    /// Returns the last process added.
    pub fn last_process(&self) -> Result<Process> {
        self.processes
            .get(&self.last_index())
            .cloned()
            .ok_or_else(|| anyhow!("Last process not found"))
    }
    /// Returns the workspace channel. I think this is confusingly named,
    /// because each subprocess talks to the workspace, not to other
    /// subprocesses. Only the workspace itself should need to know its own
    /// channel.
    pub fn workspace_channel(&self) -> Result<Channel> {
        let ch = self
            .processes
            .get(&0)
            .map(|p| p.channel.clone())
            .ok_or_else(|| anyhow!("Workspace process not found"))?;
        // if (ch.name != "ipc_control" && ch.name != "") {
        //     return Err(anyhow!("Workspace channel name mismatch"));
        // }
        Ok(ch)
    }
    /// Adds a new process group.
    pub fn add_process_group(&mut self, group: ProcessGroup) {
        self.process_groups.insert(group.group_id, group);
    }
    /// Adds a process to a group.
    pub fn add_process_to_group(&mut self, group_id: u64, pid: u64) {
        if let Some(group) = self.process_groups.get_mut(&group_id) {
            group.members.push(pid);
        }
    }
}

/// Only the workspace should call this.
/// Spawns a new process and tracks it with a monotonic ID.
pub async fn start_process(
    manager: &Arc<Mutex<ProcessManager>>,
    parent_pid: u64,
    args: Vec<String>,
) -> Result<Process> {
    let manager = manager
        .lock()
        .map_err(|_| anyhow!("Failed to lock process manager"))?;
    let mut manager = manager;
    let next_id = manager.next_index();
    let first_arg = args
        .first()
        .ok_or_else(|| anyhow!("No command specified to start_process"))?;
    let subprocess_channel = Channel {
        name: format!("com.ctoolbox.p{next_id}.{first_arg}"),
        port: manager.port,
        authentication_key: generate_authentication_key(),
    };

    let path = env::current_exe()
        .map_err(|e| anyhow!("failed to get current executable path: {e}"))?;
    let mut new_args = vec![
        IPC_ARG.to_string(),
        subprocess_channel.to_arg_string(),
        next_id.to_string(),
    ];
    new_args.extend(args);
    #[allow(clippy::zombie_processes)]
    let mut _process = Command::new(path.clone())
        .args(new_args)
        .stdin(Stdio::piped())
        .stdout(Stdio::inherit())
        .stderr(Stdio::inherit())
        .spawn()
        .map_err(|e| anyhow!("failed to execute server process: {e}"))?;
    let subprocess_stdin = _process
        .stdin
        .as_mut()
        .ok_or_else(|| anyhow!("Failed to get subprocess stdin"))?;
    subprocess_stdin
        .write_all(subprocess_channel.authentication_key.as_bytes())
        .map_err(|e| {
            anyhow!("Failed to write authentication key to subprocess: {e}")
        })?;
    let system_pid = u64::from(_process.id());
    send_message(
        &manager.ipc_control_channel,
        json!({"type": "add_channel", "channel": subprocess_channel})
            .to_string()
            .as_str(),
    );
    /*let ready = listen_for_message(&channel).await;
    debug!("Subprocess reply: {} {}", jq(".type", ready.as_str()).unwrap(), ready.as_str());
    if jq(".type", ready.as_str()).unwrap() == "ready" {
        debug!("Subprocess ready: {} {}", channel.name, get_service_name());
    }

    sleep(10);
    let message = "hello from parent".to_string();
    log("Sending hello message to child");
    send_message(channel.clone(), message.as_str());*/
    let new_process = Process {
        pid: next_id,
        system_pid,
        channel: subprocess_channel,
        parent_pid,
    };

    manager.processes.insert(next_id, new_process.clone());
    manager.last_id = next_id;

    // Optionally, add to a process group here if needed.
    // manager.add_process_to_group(group_id, next_id);

    manager.last_process()
}
