use anyhow::Result;
use serde::{Deserialize, Serialize};

use crate::cli::IpcEndpoint;

pub static IPC_ARG: &str = "--76c89de8-96b3-4372-ab16-cd832871fce3";

#[derive(Clone, Serialize, Deserialize, Default, Debug)]
pub struct Channel {
    pub name: String,
    pub port: u16,
    pub authentication_key: String,
}

impl Channel {
    pub fn to_arg_string(&self) -> String {
        format!("{}:{}", self.port, self.name)
    }

    // All processes are allowed to call workspace I guess and it avoids
    // needing to pass around a channel reference to just let it bypass the auth
    // which doesn't add any security in that case anyway
    pub fn workspace() -> Self {
        Channel {
            name: "workspace".to_string(),
            port: 0,
            authentication_key: String::new(),
        }
    }
}

pub fn channel_from_args_and_key(
    name_and_port: &IpcEndpoint,
    authentication_key: String,
) -> Channel {
    Channel {
        name: name_and_port.identity.clone(),
        port: name_and_port.port,
        authentication_key,
    }
}

pub fn channel_from_json_string(json_string: &str) -> Result<Channel> {
    serde_json::from_str(json_string)
        .map_err(|e| anyhow::anyhow!("{e}: {json_string}"))
}
