pub use anyhow::{Context, Result};
use clap::{crate_name, crate_version};
use core::panic;
use fork::{Fork, daemon};
use passwords::PasswordGenerator;
use serde::Serialize;
use serde_json::Value;
use std::backtrace::Backtrace;
use std::collections::HashMap;
use std::error::Error;
use std::path::PathBuf;
use std::process::Command;
use std::{env, fmt, fs};
use sysinfo::{Pid, Process, System};
use unicode_segmentation::UnicodeSegmentation;

pub mod debug_tools;
pub mod ipc;
pub mod json;
pub mod logging;
pub mod panic_hooks;
pub mod password;
pub mod process;
pub mod reader;
pub mod resource_lock;
pub mod serde_value;
use crate::cli::Invocation;
use crate::formats::unicode::scalars_to_string_lossy;
use crate::utilities::ipc::Channel;
use crate::utilities::process::ProcessManager;
pub use crate::utilities_json_json as json;
pub use ctb_test_macro::ctb_test;

pub const COLUMN_UUID_DELIM: &str = "d13420ff-b2d7-4e52-a390-7a0d6159e8d6";

pub fn strtovec(s: &str) -> Vec<u8> {
    s.as_bytes().to_owned()
}

pub fn vectostr(v: &[u8]) -> String {
    String::from_utf8_lossy(v).to_string()
}

pub fn strtohex<T>(s: T) -> String
where
    T: AsRef<[u8]>,
{
    hex::encode(s)
}

pub fn vectohex<T>(s: T) -> String
where
    T: AsRef<[u8]>,
{
    hex::encode(s)
}

pub fn substr_mb(s: &str, start: i128, end: i128) -> String {
    let chars: Vec<&str> = s.graphemes(true).collect();
    let len = i128::try_from(chars.len()).expect("usize did not fit in i128");

    // Convert negative indices to positive ones
    let start = if start < 0 { len + start } else { start };
    let end = if end < 0 { len + end } else { end };

    // Clamp indices within valid character bounds
    let start = start.max(0).min(len); // looks backwards but yes
    let end = end.max(0).min(len);

    // Extract substring
    chars[usize::try_from(start).expect("Did not fit")
        ..usize::try_from(end).expect("Did not fit")]
        .join("")
}

// edited from https://stackoverflow.com/a/59401721
pub fn find_first_matching_key_for_value(
    map: HashMap<Vec<u8>, Vec<u8>>,
    needle: Vec<u8>,
) -> Option<Vec<u8>> {
    map.iter().find_map(|(key, val)| {
        if *val == needle {
            Some(key.clone())
        } else {
            None
        }
    })
}

pub use log::{
    debug as hc_internal_log_debug, error as hc_internal_log_error,
    info as hc_internal_log_info, warn as hc_internal_log_warn,
};

// Random UUID to allow tracing to pull the column out after
#[macro_export]
macro_rules! log {
    ($($arg:expr),*) => {
        $crate::hc_internal_log_debug!("{:?}{}{}", ($($arg),*), $crate::COLUMN_UUID_DELIM, column!());
    }
}
pub use crate::log;

#[macro_export]
macro_rules! debug {
    ($($arg:expr),*) => {
        $crate::hc_internal_log_debug!("{:?}{}{}", ($($arg),*), $crate::COLUMN_UUID_DELIM, column!());
    }
}
pub use crate::debug;

#[macro_export]
macro_rules! info {
    ($($arg:expr),*) => {
        $crate::hc_internal_log_info!("{:?}{}{}", ($($arg),*), $crate::COLUMN_UUID_DELIM, column!());
    }
}
pub use crate::info;

#[macro_export]
macro_rules! warn {
    ($($arg:expr),*) => {
        $crate::hc_internal_log_warn!("{:?}{}{}", ($($arg),*), $crate::COLUMN_UUID_DELIM, column!());
    }
}
pub use crate::warn;

#[macro_export]
macro_rules! error {
    ($($arg:expr),*) => {
        $crate::hc_internal_log_error!("{:?}{}{}", ($($arg),*), $crate::COLUMN_UUID_DELIM, column!());
    }
}
pub use crate::error;

#[macro_export]
macro_rules! log_fmt {
    ($fmt:expr, $($arg:tt)*) => {
        $crate::hc_internal_log_debug!("{}{}{}", format!($fmt, $($arg)*), $crate::COLUMN_UUID_DELIM, column!());
    };
    ($msg:expr) => {
        $crate::hc_internal_log_debug!("{}{}{}", format!($msg), $crate::COLUMN_UUID_DELIM, column!());
    };
}
pub use crate::log_fmt;

#[macro_export]
macro_rules! debug_fmt {
    ($fmt:expr, $($arg:tt)*) => {
        $crate::hc_internal_log_debug!("{}{}{}", format!($fmt, $($arg)*), $crate::COLUMN_UUID_DELIM, column!());
    };
    ($msg:expr) => {
        $crate::hc_internal_log_debug!("{}{}{}", format!($msg), $crate::COLUMN_UUID_DELIM, column!());
    };
}
pub use crate::debug_fmt;

#[macro_export]
macro_rules! info_fmt {
    ($fmt:expr, $($arg:tt)*) => {
        $crate::hc_internal_log_info!("{}{}{}", format!($fmt, $($arg)*), $crate::COLUMN_UUID_DELIM, column!());
    };
    ($msg:expr) => {
        $crate::hc_internal_log_info!("{}{}{}", format!($msg), $crate::COLUMN_UUID_DELIM, column!());
    };
}
pub use crate::info_fmt;

#[macro_export]
macro_rules! warn_fmt {
    ($fmt:expr, $($arg:tt)*) => {
        $crate::hc_internal_log_warn!("{}{}{}", format!($fmt, $($arg)*), $crate::COLUMN_UUID_DELIM, column!());
    };
    ($msg:expr) => {
        $crate::hc_internal_log_warn!("{}{}{}", format!($msg), $crate::COLUMN_UUID_DELIM, column!());
    };
}
pub use crate::warn_fmt;

#[macro_export]
macro_rules! log_string {
    ($document:expr) => {
        log!($document);
    };
}

#[macro_export]
macro_rules! log_type {
    ($t:ty) => {
        log!(std::any::type_name::<$t>());
    };
}

#[macro_export]
macro_rules! json_value {
    // Match key-value pairs: json_value!({ "foo" => bar, ... })
    ({ $($key:expr => $value:expr),* $(,)? }) => {
        {
            #[allow(unused_mut)]
            let mut map = serde_json::Map::new();
            $(
                map.insert($key.into(), serde_json::to_value($value).expect("Failed to convert to JSON value"));
            )*
            serde_json::Value::Object(map)
        }
    };
    // Match a single value: json_value!("foo")
    ($key:expr) => {
        {
            serde_json::to_value($key).expect("Failed to convert to JSON value")
        }
    };
}

// Use this by defining an error_abort macro in other code for it to expand to, e.g. in lib.rs
#[macro_export]
macro_rules! unwrap_or_custom_error {
    ($result:expr, $error:expr) => {
        match $result {
            Ok(value) => value,
            Err(_) => {
                // log!($error);
                error_abort!($result.context($error))
            }
        }
    };
}

#[macro_export]
macro_rules! unwrap_or_result_error {
    ($result:expr, $error:expr) => {
        unwrap_or_custom_error!(
            $result.map_err(|e| anyhow::anyhow!("{}", e)),
            $error
        )
    };
}

#[macro_export]
macro_rules! bail_if_none {
    // case: no message — produce a default anyhow::Error
    ($opt:expr) => {
        $opt.ok_or_else(|| anyhow::anyhow!(format!("unexpected None, at {} line: {}, column: {}", file!(), line!(), column!())))?
    };

    // case: single literal/message without formatting
    ($opt:expr, $msg:expr) => {
        $opt.ok_or_else(|| anyhow::anyhow!(format!("{:?}, at {} line: {}, column: {}", $msg, file!(), line!(), column!())))?
    };

    // case: format-like message with args
    ($opt:expr, $fmt:expr, $($args:tt)+) => {
        $opt.ok_or_else(|| anyhow::anyhow!($fmt, $($args)+))?
    };
}

pub fn backtrace_string() -> String {
    format!("{}", Backtrace::capture())
}

pub fn backtrace_print() {
    println!("Backtrace: {}", backtrace_string());
}

pub fn this_pid() -> Vec<u8> {
    std::process::id().to_string().into_bytes()
}

pub fn in_array(needle: Vec<u8>, map: HashMap<u32, Vec<u8>>) -> bool {
    map.iter().any(|(_, val)| *val == needle)
}

pub fn is_subprocess(invocation: &Invocation) -> bool {
    invocation.is_subprocess()
}

pub fn get_service_name(invocation: &Invocation) -> String {
    if is_subprocess(invocation) {
        invocation
            .subprocess()
            .map(|s| s.service_name.clone())
            .unwrap_or_default()
    } else {
        String::new()
    }
}

pub fn remove_suffix<'a>(string: &'a str, suffix: &str) -> &'a str {
    if string.to_string().ends_with(suffix) {
        &string[..string.len() - suffix.len()]
    } else {
        string
    }
}

pub fn sleep(seconds: u64) {
    std::thread::sleep(std::time::Duration::from_secs(seconds));
}

pub fn usleep(microseconds: u64) {
    std::thread::sleep(std::time::Duration::from_micros(microseconds));
}

pub fn setup_panic_hooks(manager: &ProcessManager) {
    panic_hooks::setup_panic_hooks(manager)
}

pub fn send_message(channel: &Channel, message: &str) -> String {
    crate::workspace::ipc_old::send_message(channel, message)
}

pub fn maybe_send_message(
    channel: &Channel,
    message: &str,
) -> Result<String, Box<dyn Error>> {
    crate::workspace::ipc_old::maybe_send_message(channel, message)
}

pub async fn wait_for_message(channel: &Channel, msgid: u64) -> String {
    crate::workspace::ipc_old::wait_for_message(channel, msgid).await
}

pub fn listen_for_message(channel: &Channel, msgid: u64) -> String {
    crate::workspace::ipc_old::listen_for_message(channel, msgid)
}

pub fn call_ipc_sync(
    channel: &Channel,
    method: &str,
    args: Value,
) -> Result<String> {
    crate::workspace::ipc_old::call_ipc_sync(channel, method, args)
}

pub async fn call_ipc(
    channel: &Channel,
    method: &str,
    args: Value,
) -> Result<String> {
    crate::workspace::ipc_old::call_ipc(channel, method, args).await
}

pub fn shutdown(process_manager: &ProcessManager) {
    println!("Shutting down");
    cleanup_processes(process_manager);
    std::process::exit(1);
}

pub fn upgrade_in_place(
    temp_path: &PathBuf,
    target_path: &PathBuf,
) -> Result<()> {
    fs::copy(temp_path, target_path)
        .with_context(|| "Failed to copy new executable over old one")?;
    Ok(())
}

pub fn fork(path: &PathBuf, args: Vec<&str>) {
    if let Ok(Fork::Child) = daemon(false, false) {
        Command::new(path)
            .args(args)
            .output()
            .expect("failed to execute process");
    }
}

// This API is annoying, I can't figure out how to get it to take just the PID
fn get_ctoolbox_process(s: &mut System, pid: u32) -> Option<&Process> {
    s.refresh_all();
    let process = s.process(Pid::from_u32(pid))?;
    let subprocess_exe = process.exe()?;
    let this_exe = env::current_exe().unwrap();
    if this_exe != subprocess_exe {
        return None;
    }
    Some(process)
}

pub fn get_this_executable() -> PathBuf {
    let mut s = System::new_all();
    s.refresh_all();
    env::current_exe().unwrap()
}

pub fn wait_for_ctoolbox_process_exit<'a>(pid: u32) {
    let mut s = System::new_all();
    let mut process = get_ctoolbox_process(&mut s, pid);
    while process.is_some() {
        process = get_ctoolbox_process(&mut s, pid);
        std::thread::sleep(std::time::Duration::from_millis(100));
    }
}

pub fn wait_for_ctoolbox_exit_and_clean_up(pid: u32) {
    wait_for_ctoolbox_process_exit(pid);
}

pub fn cleanup_processes(process_manager: &ProcessManager) {
    // let json = json!(process_manager);
    // println!("Cleaning up processes: {}", json);
    process_manager
        .processes
        .iter()
        .for_each(|(_id, process_record)| {
            let mut s = System::new_all();
            if let Some(process) = get_ctoolbox_process(
                &mut s,
                process_record
                    .system_pid
                    .try_into()
                    .expect("Invalid PID for this platform"),
            ) {
                // println!("Killing process {}", process.pid());
                process.kill();
            }
        });
}

pub fn generate_authentication_key() -> String {
    let pg = PasswordGenerator {
        length: 64,
        numbers: true,
        lowercase_letters: true,
        uppercase_letters: true,
        symbols: true,
        spaces: true,
        exclude_similar_characters: false,
        strict: true,
    };

    pg.generate_one().unwrap()
}

pub fn u8_vec_to_formatted_hex(values: &[u8]) -> String {
    let mut out = String::new();
    for &v in values {
        out.push_str(&format!("{v:02x} "));
    }
    out.to_uppercase().trim().to_string()
}

pub fn u32_vec_to_formatted_hex(values: &[u32]) -> String {
    let mut out = String::new();
    for &v in values {
        out.push_str(&format!("{v:02x} "));
    }
    out.to_uppercase().trim().to_string()
}

// Test helpers to compare slices with clearer diff on failure.
pub fn assert_vec_u32_eq(expected: &[u32], actual: &[u32]) -> Vec<u32> {
    assert!(
        expected == actual,
        "Vectors (u32) differ.\n{}",
        fmt_mismatch_vec_u32(expected, actual)
    );
    actual.to_vec()
}

pub fn assert_vec_u8_eq(expected: &[u8], actual: &[u8]) -> Vec<u8> {
    assert!(
        expected == actual,
        "Vectors (u8) differ.\n{}",
        fmt_mismatch_vec_u8(expected, actual)
    );
    actual.to_vec()
}

pub fn fmt_mismatch_vec_u8(expected: &[u8], actual: &[u8]) -> String {
    format!(
        "Expected: {:?}\nActual:   {:?}\nExpected (hex): {:?}\nActual   (hex): {:?}\nExpected (lossy): {:?}\nActual   (lossy): {:?}",
        expected,
        actual,
        u8_vec_to_formatted_hex(expected),
        u8_vec_to_formatted_hex(actual),
        String::from_utf8_lossy(expected),
        String::from_utf8_lossy(actual)
    )
}

pub fn fmt_mismatch_vec_u32(expected: &[u32], actual: &[u32]) -> String {
    format!(
        "Expected: {:?}\nActual:   {:?}\nExpected (hex): {:?}\nActual   (hex): {:?}\nExpected (lossy): {:?}\nActual   (lossy): {:?}",
        expected,
        actual,
        u32_vec_to_formatted_hex(expected),
        u32_vec_to_formatted_hex(actual),
        scalars_to_string_lossy(expected),
        scalars_to_string_lossy(actual)
    )
}

pub fn fmt_mismatch_string(expected: &str, actual: &str) -> String {
    format!(
        "Expected: {:?}\nActual:   {:?}\nExpected (hex): {:?}\nActual   (hex): {:?}",
        expected,
        actual,
        u8_vec_to_formatted_hex(expected.as_bytes()),
        u8_vec_to_formatted_hex(actual.as_bytes()),
    )
}

pub fn feature(feature_name: &str) -> bool {
    let current_settings =
        crate::storage::pc_settings::PcSettings::load().unwrap_or_default();
    match feature_name {
        "login" => current_settings.feature_login,
        "registration" => current_settings.feature_registration,
        _ => false,
    }
}

thread_local! {
    pub static CURRENT_TEST_NAME: std::cell::RefCell<Option<String>> = std::cell::RefCell::new(None);
}

#[cfg(test)]
pub fn set_current_test_name(p: String) {
    CURRENT_TEST_NAME.with(|c| *c.borrow_mut() = Some(p));
}

pub fn get_current_test_name() -> String {
    assert!(cfg!(test), "get_current_test_name called outside of test");

    if let Some(p) = CURRENT_TEST_NAME.with(|c| c.borrow().clone()) {
        p
    } else {
        panic!("Test name not set");
    }
}

#[derive(Serialize)]
pub struct BuildInfo {
    pub(crate) name: String,
    pub(crate) version: String,
    pub(crate) build_date: String,
    pub(crate) commit: String,
}

pub fn build_info() -> BuildInfo {
    BuildInfo {
        name: crate_name!().to_string(),
        version: crate_version!().to_string(),
        build_date: env!("VERGEN_BUILD_TIMESTAMP").to_string(),
        commit: env!("VERGEN_GIT_SHA").to_string(),
    }
}

pub fn assert_vec_u32_ok_eq(
    expected: &[u32],
    actual: anyhow::Result<Vec<u32>>,
) -> Vec<u32> {
    match actual {
        Ok(v) => assert_vec_u32_eq(expected, &v),
        Err(e) => panic!("Expected Ok(Vec<u32>), got Err: {e:?}"),
    }
}

pub fn assert_vec_u8_ok_eq(
    expected: &[u8],
    actual: anyhow::Result<Vec<u8>>,
) -> Vec<u8> {
    match actual {
        Ok(v) => assert_vec_u8_eq(expected, &v),
        Err(e) => panic!("Expected Ok(Vec<u8>), got Err: {e:?}"),
    }
}

pub fn assert_string_contains(expected: &str, actual: &str) {
    assert!(
        actual.contains(expected),
        "String '{actual}' does not contain expected substring '{expected}'."
    );
}

pub fn assert_string_not_contains(expected: &str, actual: &str) {
    assert!(
        !actual.contains(expected),
        "String '{actual}' unexpectedly contains forbidden substring '{expected}'."
    );
}
