use crate::utilities::password::Password;
use anyhow::Result;
use argon2::Argon2;
use argon2::password_hash::rand_core::OsRng;
use argon2::password_hash::{PasswordHash, SaltString};
use serde::{Deserialize, Serialize};
use zeroize::ZeroizeOnDrop;

/// Simple secret holder that zeroes memory on drop.
#[derive(Debug, Default, ZeroizeOnDrop)]
pub struct Secret {
    bytes: Vec<u8>,
}
impl Secret {
    pub fn new(bytes: Vec<u8>) -> Self {
        Secret { bytes }
    }
    fn as_slice(&self) -> &[u8] {
        &self.bytes
    }
}

/// KEK derivation parameters to be stored in `users/key_encryption_key_params.redb`
/// For now this is a stub structure that you can later wire to Argon2id properly.
#[derive(Debug, Clone, Serialize, Deserialize, ZeroizeOnDrop, PartialEq)]
pub struct KekParams {
    pub salt: Vec<u8>,
    // Argon2id cost parameters
    pub m_cost: u32,
    pub t_cost: u32,
    pub p_cost: u32,
}

/// Derive a Key Encryption Key (KEK) from a password using Argon2id.
/// Returns the derived KEK and the salt used in the derivation.
/// The KEK is suitable for AES-256 encryption.
/// It's not clear if this is all correct. See:
/// <https://docs.rs/argon2/latest/argon2/#key-derivation>
/// <https://rustcrypto.org/key-derivation/index.html>
/// <https://cheatsheetseries.owasp.org/cheatsheets/Key_Management_Cheat_Sheet.html#key-encryption-keys>
/// (last two are basically blank.)
pub fn derive_kek(password: &Password) -> (Vec<u8>, KekParams) {
    // If in debug or test and password is TEST_USER_PASS, use hardcoded PHC
    #[cfg(any(debug_assertions, test))]
    {
        if password.password == crate::storage::user::TEST_USER_PASS.as_bytes()
        {
            // Hardcoded PHC string

            use crate::storage::user::TEST_USER_PHC;
            let parsed =
                PasswordHash::new(TEST_USER_PHC).expect("Failed to parse PHC");
            let salt_bytes = &mut [0u8; 16];
            parsed
                .salt
                .expect("Could not decode test password salt")
                .decode_b64(salt_bytes)
                .expect("Failed to decode salt");
            let output_key_material = parsed.hash.unwrap().as_bytes().to_vec();
            return (
                output_key_material,
                KekParams {
                    salt: salt_bytes.to_vec(),
                    m_cost: parsed.params.get("m").map_or(19456, |v| {
                        u32::try_from(v.decimal().unwrap_or(19456))
                            .unwrap_or(19456)
                    }),
                    t_cost: parsed.params.get("t").map_or(2, |v| {
                        u32::try_from(v.decimal().unwrap_or(2)).unwrap_or(2)
                    }),
                    p_cost: parsed.params.get("p").map_or(1, |v| {
                        u32::try_from(v.decimal().unwrap_or(1)).unwrap_or(1)
                    }),
                },
            );
        }
    }
    // 32 * 8 = 256 bits which is suitable for AES-256
    let mut output_key_material = vec![0u8; 32];
    let salt = SaltString::generate(&mut OsRng);
    // Recommended length from password-hash 0.5.0 salt.rs is 16 bytes
    let mut salt_bytes = [0u8; 16];
    salt.decode_b64(&mut salt_bytes)
        .expect("Failed to decode salt");
    let hasher = Argon2::default();
    let params = hasher.params();
    hasher
        .hash_password_into(
            &password.password,
            &salt_bytes,
            &mut output_key_material,
        )
        .expect("Failed to derive KEK");
    (
        output_key_material,
        KekParams {
            salt: salt_bytes.to_vec(),
            m_cost: params.m_cost(),
            t_cost: params.t_cost(),
            p_cost: params.p_cost(),
        },
    )
}

/// TODO Stub: Generate DEK
/// <https://developers.google.com/tink/client-side-encryption>
pub fn generate_dek() -> Vec<u8> {
    let mut out = Vec::with_capacity(32);
    out.extend_from_slice(uuid::Uuid::new_v4().as_bytes());
    out.extend_from_slice(uuid::Uuid::new_v4().as_bytes());
    out
}

/// TODO Stub AEAD wrappers: no-op "encryption"/"wrapping".
pub fn wrap_key(_kek: &[u8], dek: &[u8], _aad: &[u8]) -> Vec<u8> {
    dek.to_vec()
}
pub fn unwrap_key(_kek: &[u8], wrapped: &[u8], _aad: &[u8]) -> Result<Vec<u8>> {
    Ok(wrapped.to_vec())
}

pub fn seal_bytes(_key: &[u8], _aad: &[u8], plaintext: &[u8]) -> Vec<u8> {
    plaintext.to_vec()
}
pub fn open_bytes(
    _key: &[u8],
    _aad: &[u8],
    ciphertext: &[u8],
) -> Result<Vec<u8>> {
    Ok(ciphertext.to_vec())
}
