//! User management: creation, deletion, login, metadata.
//! This *tries* to lock the user during write operations so it won't end up in
//! inconsistent states, etc. but that doesn't currently seem to work reliably.

use crate::storage::db::{
    delete_user_auth, delete_user_display_name, delete_user_id,
    delete_user_kek_params, delete_user_picture, delete_user_pubkey,
    delete_user_uuid, delete_user_wrapped_dek, get_user_auth, get_user_id,
    get_user_kek_params, get_user_name, get_user_pictures, get_user_pubkey,
    get_user_wrapped_dek, put_user_auth, put_user_id, put_user_kek_params,
    put_user_uuid, put_user_wrapped_dek,
};
use crate::storage::graph::Graph;
use crate::storage::pc_settings::ensure_pc_settings;
use crate::storage::user::auth::{KekParams, Secret};
use crate::utilities::password::{Password, hash, verify};
use crate::utilities::resource_lock::{Lock, ResourceLock};
use crate::{bail_if_none, error, get_storage_dir, json, log};
use anyhow::{Context, Result, anyhow, bail};
use serde::{Deserialize, Serialize, de::DeserializeOwned};
use std::collections::HashMap;
use std::ffi::OsString;
use std::fs;
use std::path::Path;

pub mod auth;

pub const TEST_USER_PASS: &str =
    "test_password_523ed9db-1885-439d-83d5-61c869e5223d";
pub const TEST_USER_PHC: &str = "$argon2id$v=19$m=19456,t=2,p=1$RXiQwP6rcmLh98qumvyu0g$Pc9dem17Dgwz6uoiCuceMh+VYQxZ8WqSK36gfpAZYXY";

#[derive(Debug)]
pub struct NameAndIdLock {
    pub name_lock: ResourceLock,
    pub id_lock: ResourceLock,
}

impl Lock for NameAndIdLock {}

impl NameAndIdLock {
    pub fn new(name_lock: ResourceLock, id_lock: ResourceLock) -> Self {
        Self { name_lock, id_lock }
    }
}

#[derive(Debug, Default, Serialize, Deserialize)]
pub struct UserLocalConfig {
    pub default_store_location: OsString,
}

#[derive(Debug, Default, Serialize)]
pub struct UserPublicInfo {
    local_id: u64,
    name: String,
    display_name: Option<Vec<u8>>,
    uuid: Vec<u8>,
    picture: Option<Vec<u8>>,
    #[serde(skip)]
    lock: Option<ResourceLock>,
}

impl UserPublicInfo {
    pub fn get_by_name(name: &str) -> Result<Option<Self>> {
        let _name_lock = ResourceLock::acquire(USER_NAME_LOCK, &name)?;
        let mut user_info = UserPublicInfo::default();
        let id = get_user_id_by_name(name);
        let Some(id) = id else {
            return Ok(None);
        };
        let user_info_lock = bail_if_none!(
            ResourceLock::acquire(USER_PUBLIC_INFO_LOCK, &id).ok()
        );
        // Re-check ID  to ensure mapping stability
        let id_check = bail_if_none!(get_user_id_by_name(name));
        if id != id_check {
            bail!(format!(
                "User ID changed during get_by_name for '{name}': {id} -> {id_check}"
            ));
        }
        user_info.lock = Some(user_info_lock);
        user_info.local_id = id;
        // Get display name and picture  if needed in future
        user_info.name = name.to_string();
        if let Some(picture) = get_user_picture_by_id(id) {
            user_info.picture = Some(picture);
        }
        Ok(Some(user_info))
    }

    pub fn get_by_id(id: u64) -> Option<Self> {
        let mut user_info = UserPublicInfo::default();
        let user_info_lock =
            ResourceLock::acquire(USER_PUBLIC_INFO_LOCK, &id).ok()?;
        user_info.lock = Some(user_info_lock);
        user_info.local_id = id;

        // Perform DB lookups , after acquiring public info lock (consistent ordering: never acquire other locks while holding DB lock).
        user_info.name = get_user_name(id)?;

        if let Some(picture) = { get_user_picture_by_id(id) } {
            user_info.picture = Some(picture);
        }
        Some(user_info)
    }

    pub fn local_id(&self) -> u64 {
        self.local_id
    }
    pub fn name(&self) -> &str {
        &self.name
    }
    pub fn display_name(&self) -> Option<&[u8]> {
        self.display_name.as_deref()
    }
    pub fn uuid(&self) -> &Vec<u8> {
        &self.uuid
    }
    pub fn user_picture(&self) -> Option<&[u8]> {
        self.picture.as_deref()
    }

    pub fn list_all() -> Result<Vec<Self>> {
        let user_ids = get_all_user_ids()?;
        let mut users = Vec::new();
        for id in user_ids {
            if let Some(user) = UserPublicInfo::get_by_id(id) {
                users.push(user);
            }
        }
        Ok(users)
    }
}

#[derive(Debug, Default)]
pub struct User {
    public_info: UserPublicInfo,
    remote_id: Option<u64>,
    local_config: UserLocalConfig,
    graphs: Vec<Graph>,
    dek: Option<Secret>,
    lock: Option<NameAndIdLock>, // Guard that keeps user locked while User is alive.
}

const USER_LOCK: &str = "user";
const USER_NAME_LOCK: &str = "user_name";
const USER_PUBLIC_INFO_LOCK: &str = "user_public_info";

impl User {
    pub fn name(&self) -> String {
        self.public_info.name.clone()
    }

    pub fn local_id(&self) -> u64 {
        self.public_info.local_id
    }

    pub fn remote_id(&self) -> Option<u64> {
        self.remote_id
    }

    pub fn display_name(&self) -> Option<&[u8]> {
        self.public_info.display_name.as_deref()
    }

    pub fn user_picture(&self) -> Option<&[u8]> {
        self.public_info.picture.as_deref()
    }

    pub fn uuid(&self) -> &Vec<u8> {
        &self.public_info.uuid
    }

    /// Create a new local user.
    pub fn create(name: &str, password: &Password) -> Result<Self> {
        // 1) Acquire name lock first to serialize operations on this user name.
        let name_lock = ResourceLock::acquire(USER_NAME_LOCK, &name)?;

        error!(format!("Creating user '{name}'"));
        ensure_base_layout().context("Failed to ensure base layout")?;

        // 2) Check existence.
        if get_user_id_by_name(name).is_some() {
            return Err(anyhow!("User '{name}' already exists"));
        }

        // 3) Allocate a new user id using the file-based atomic counter.
        let user_id = next_user_id().context("Unable to allocate user id")?;

        // 4) Acquire per-user lock (exclusive) for entire creation.
        let id_lock = lock_by_id(user_id)?;
        let user_lock = NameAndIdLock { name_lock, id_lock };

        let (kek, kek_params) = auth::derive_kek(password);
        let dek = auth::generate_dek();
        let phc = hash(password)?;

        // Wrap DEK under KEK with aad=user_id
        // AAD = "additional authenticated data" in AEAD, used to avoid reusing
        // the key with different user ID, etc.
        // So, to allow the user record to be copied between local machine and
        // server, also allocate a UUID for the user and use that as the AAD.
        // The UUID must be persistent when the user record is copied (unlike
        // the local user ID).

        // Allocate UUID for AAD
        let uuid = uuid::Uuid::new_v4();
        let uuid = uuid.as_bytes();

        let aad = uuid;
        let wrapped_dek = auth::wrap_key(&kek, &dek, aad);

        // 5) Persist all user metadata.
        let root = get_storage_dir().context("No storage dir")?;
        let users_dir = root.join("users");
        fs::create_dir_all(&users_dir).ok();

        put_user_id(name, user_id)?;
        put_user_uuid(user_id, uuid)?;
        put_user_auth(user_id, &phc)?;
        put_user_kek_params(user_id, json!(kek_params).as_bytes())?;
        put_user_wrapped_dek(user_id, &wrapped_dek)?;

        ensure_pc_settings().context("Failed to ensure pc_settings.json")?;

        {
            let root = get_storage_dir().context("No storage dir")?;
            let graphs_dir = root.join("graphs").join(user_id.to_string());
            fs::create_dir_all(&graphs_dir).with_context(|| {
                format!("Failed to create graphs dir {}", graphs_dir.display())
            })?;
        }

        let mut user = User::default();
        user.public_info.local_id = user_id;
        user.public_info.name = name.to_string();
        user.public_info.uuid = uuid.to_vec();
        user.local_config = user.local_config();
        user.dek = Some(Secret::new(dek));
        user.lock = Some(user_lock);

        Ok(user)
    }

    /// Delete this user
    pub fn delete(&self) -> Result<()> {
        ensure_base_layout().context("Failed to ensure base layout")?;
        let user_id = self.local_id();
        let name = self.name();

        // If caller already holds locks (e.g., from create), respect them and only serialize DB work.
        if self.lock.is_some() {
            Self::_delete_user(user_id, &name)
        } else {
            // Acquire locks in consistent order: name -> id -> db
            let _name_lock = ResourceLock::acquire(USER_NAME_LOCK, &name)?;
            let _id_lock = lock_by_id(user_id)?;

            Self::_delete_user(user_id, &name)
        }
    }

    /// Delete a user by name
    pub fn delete_by_name(name: &str) -> Result<()> {
        ensure_base_layout().context("Failed to ensure base layout")?;
        log!("Deleting user '{name}'");

        // Acquire locks in consistent order: name -> id -> db
        let name_lock = ResourceLock::acquire(USER_NAME_LOCK, &name)?;

        let user_id = get_user_id_by_name(name)
            .ok_or_else(|| anyhow!("User ID not found for name {name}"))?;

        let _id_lock = lock_by_id(user_id)?;

        // Perform deletion .
        let res = {
            // Keep name_lock in scope to maintain consistent ownership while deleting mapping.
            let _keep_name_lock = &name_lock;
            Self::_delete_user(user_id, name)
        };

        // Verify deletion .
        {
            if get_user_id_by_name(name).is_some() {
                bail!("Failed to delete user {name} with id {user_id}");
            }
        }
        res
    }

    /// Internal (no-DB-lock) deletion logic extracted to avoid double-locking.
    /// Do NOT call this without holding the DB lock. This function assumes the caller holds:
    /// - `USER_NAME_LOCK` (for `name`)
    /// - `USER_LOCK` (for `user_id`)
    fn _delete_user(user_id: u64, name: &str) -> Result<()> {
        let root = get_storage_dir().context("No storage dir")?;

        delete_user_uuid(user_id)?;
        delete_user_auth(user_id)?;
        delete_user_picture(user_id)?;
        delete_user_display_name(user_id)?;
        delete_user_kek_params(user_id)?;
        delete_user_wrapped_dek(user_id)?;
        delete_user_pubkey(user_id)?;
        delete_user_id(name, user_id)?;

        // Remove user's graphs directory (filesystem, not DB)
        let graphs_dir = root.join("graphs").join(user_id.to_string());
        if graphs_dir.exists() {
            fs::remove_dir_all(&graphs_dir).with_context(|| {
                format!("Failed to remove graphs dir {graphs_dir:?}")
            })?;
        }
        Ok(())
    }

    /// Login: Verify password, derive KEK, verify DEK.
    /// TODO: Possibly this should also log in to the server and start syncing any new user data.
    pub fn login(
        public_info: UserPublicInfo,
        password: &Password,
    ) -> Result<Self> {
        ensure_base_layout().context("Failed to ensure base layout")?;

        if public_info.local_id == 0 {
            return Err(anyhow!(
                "UserPublicInfo.local_id was 0. Provide a valid user_id or use UserPublicInfo::get_by_name."
            ));
        }

        let user_id = public_info.local_id;

        // Acquire per-user lock for consistency while reading user metadata.
        // Lock ordering: id lock first, DB lock only around DB calls.
        let _user_lock = lock_by_id(user_id)?;

        let (phc, kek_params, wrapped_dek, uuid) = {
            let phc = get_user_password_by_id(user_id).ok_or_else(|| {
                anyhow!("Auth entry for user_id {user_id} not found")
            })?;

            let kek_params = get_user_kek_params_by_id(user_id)
                .and_then(|bytes| {
                    serde_json::from_slice::<KekParams>(&bytes).ok()
                })
                .ok_or_else(|| {
                    anyhow!("KEK params not found for user_id {user_id}")
                })?;

            let wrapped_dek =
                get_user_wrapped_dek_by_id(user_id).ok_or_else(|| {
                    anyhow!("Wrapped DEK not found for user_id {user_id}")
                })?;

            // Make sure we have UUID in public_info for AAD.
            if public_info.uuid.is_empty() {
                // If missing, try to reload public fields if needed in the future.
            }

            (phc, kek_params, wrapped_dek, public_info.uuid.clone())
        };

        if !(verify(password, &phc)?) {
            return Err(anyhow!("Invalid password"));
        }

        // Derive KEK. If the auth module exposes a derive-with-params, it should be used here.
        // Fallback to existing derive to preserve behavior.
        let (kek, _ignored_params) = auth::derive_kek(password);

        let dek = auth::unwrap_key(&kek, &wrapped_dek, &uuid)
            .context("Failed to unwrap DEK")?;

        let mut user = User::default();
        user.public_info = public_info;
        user.local_config = user.local_config();
        user.dek = Some(Secret::new(dek));
        // We intentionally do not hold the user lock beyond login.
        Ok(user)
    }

    pub fn local_config(&self) -> UserLocalConfig {
        let cache_dir = get_storage_dir().unwrap();
        let user_cache_dir = std::path::Path::new(&cache_dir).join(self.name());
        std::fs::create_dir_all(&user_cache_dir)
            .expect("Failed to create per-user cache directory");
        let config_path = user_cache_dir.join("local_config");
        if !config_path.exists() {
            let default_config = UserLocalConfig {
                default_store_location: cache_dir.into_os_string(),
            };
            let json = serde_json::to_string_pretty(&default_config).unwrap();
            std::fs::write(&config_path, json)
                .expect("Failed to write default local_config");
            return default_config;
        }
        let json = std::fs::read_to_string(&config_path)
            .expect("Failed to read local_config");
        serde_json::from_str(&json).expect("Failed to deserialize local_config")
    }

    pub fn create_graph(&mut self, label: &str) -> Result<&Graph> {
        let new_graph_id = if self.graphs.is_empty() {
            1
        } else {
            self.graphs
                .iter()
                .map(|g| g.graph_id)
                .max()
                .ok_or_else(|| anyhow!("Failed to get max graph id"))?
                + 1
        };
        let new_graph = Graph::new(new_graph_id, label, self);
        self.graphs.push(new_graph);
        self.graphs
            .last()
            .ok_or_else(|| anyhow!("Failed to get max graph id"))
    }

    pub fn get_graph_count(&self) -> usize {
        self.graphs.len()
    }

    pub fn get_graph_by_id(&self, id: u32) -> Option<&Graph> {
        self.graphs.iter().find(|g| g.graph_id == id)
    }
}

pub fn lock_by_name(
    name: &str,
) -> Result<(ResourceLock, Option<ResourceLock>)> {
    // Always acquire name lock first
    let name_lock = ResourceLock::acquire(USER_NAME_LOCK, &name)?;
    // Then, resolve id (if any)
    let maybe_user = UserPublicInfo::get_by_name(name)?;
    let mut id_lock = None;
    if let Some(user) = maybe_user {
        let id = user.local_id;
        id_lock = Some(lock_by_id(id)?);
        // Double-check mapping hasn't changed while acquiring id lock
        let user_check = UserPublicInfo::get_by_name(name)?;
        if let Some(user_check) = user_check {
            if id != user_check.local_id {
                log!(format!(
                    "User ID changed during get_by_name for '{}': {} -> {}",
                    name, id, user_check.local_id
                ));
                bail!("User ID changed during lock acquisition");
            }
        }
    }
    Ok((name_lock, id_lock))
}

pub fn lock_by_id(user_id: u64) -> Result<ResourceLock> {
    let id_lock = ResourceLock::acquire(USER_LOCK, &user_id.to_string())?;
    Ok(id_lock)
}

fn get_user_id_by_name(name: &str) -> Option<u64> {
    get_user_id(name)
}

fn get_user_password_by_id(user_id: u64) -> Option<String> {
    get_user_auth(user_id)
}

fn get_user_picture_by_id(user_id: u64) -> Option<Vec<u8>> {
    get_user_pictures(user_id)
}

fn get_user_kek_params_by_id(user_id: u64) -> Option<Vec<u8>> {
    get_user_kek_params(user_id)
}

fn get_user_wrapped_dek_by_id(user_id: u64) -> Option<Vec<u8>> {
    get_user_wrapped_dek(user_id)
}

fn get_user_pubkey_by_id(user_id: u64) -> Option<Vec<u8>> {
    get_user_pubkey(user_id)
}

pub fn get_all_user_ids() -> Result<Vec<u64>> {
    ensure_base_layout().context("Failed to ensure base layout")?;

    crate::storage::db::get_all_user_ids()
}

// --------- Internal helpers ----------

fn ensure_base_layout() -> Result<()> {
    let root = get_storage_dir().context("No storage dir")?;
    let config_dir = root.join("config");
    let users_dir = root.join("users");
    let graphs_dir = root.join("graphs");
    fs::create_dir_all(&config_dir)?;
    fs::create_dir_all(&users_dir)?;
    fs::create_dir_all(&graphs_dir)?;
    Ok(())
}

/// Returns the next available user ID, guaranteed to be monotonically increasing and never reused.
/// Uses a persistent file to store the last allocated user ID.
fn next_user_id() -> Result<u64> {
    use fs2::FileExt;
    use std::fs::OpenOptions;
    use std::io::{Read, Seek, Write};

    let root = get_storage_dir().context("No storage dir")?;
    let last_id_path = root.join("users").join("last_user_id");

    // Open the file (create if not exists) for read/write. Do NOT truncate.
    let mut file = OpenOptions::new()
        .read(true)
        .write(true)
        .truncate(false)
        .create(true)
        .open(&last_id_path)
        .with_context(|| {
            format!("Failed to open {}", last_id_path.display())
        })?;

    // Acquire exclusive lock for atomicity.
    file.lock_exclusive().with_context(|| {
        format!("Failed to lock {}", last_id_path.display())
    })?;

    // Seek to start and read the current content.
    file.seek(std::io::SeekFrom::Start(0))?;
    let mut content = String::new();
    file.read_to_string(&mut content)?;
    let last_id = if content.trim().is_empty() {
        0
    } else {
        content.trim().parse::<u64>()?
    };

    // Increment and handle overflow.
    let next_id = last_id
        .checked_add(1)
        .ok_or_else(|| anyhow!("User ID overflow"))?;

    // Seek to start, truncate, and write the new value.
    file.seek(std::io::SeekFrom::Start(0))?;
    file.set_len(0)?; // Truncate the file.
    file.write_all(next_id.to_string().as_bytes())
        .with_context(|| {
            format!("Failed to write {}", last_id_path.display())
        })?;

    // Unlock is automatic on drop.
    Ok(next_id)
}

fn read_json_map<T: DeserializeOwned>(
    path: &Path,
) -> Result<HashMap<String, T>> {
    if !path.exists() {
        return Ok(HashMap::new());
    }
    let data = fs::read(path)?;
    if data.is_empty() {
        return Ok(HashMap::new());
    }
    let map =
        serde_json::from_slice::<HashMap<String, T>>(&data).unwrap_or_default();
    Ok(map)
}

fn write_json_map<T: Serialize>(
    path: &Path,
    map: &HashMap<String, T>,
) -> Result<()> {
    let tmp = serde_json::to_vec_pretty(map)?;
    let tmp_path = path.with_extension("tmp");
    fs::write(&tmp_path, tmp)?;
    fs::rename(tmp_path, path)?;
    Ok(())
}

#[cfg(test)]
pub fn get_test_user(name: &str) -> User {
    use crate::debug;

    let _ = lock_by_name(name).expect("Could not lock name");
    debug!(
        "Thread {} acquired lock for test user '{}'",
        std::thread::current().name().unwrap_or("unnamed"),
        name
    );
    User::delete_by_name(name).ok();

    assert!(
        !get_user_id_by_name(name).is_some(),
        "Failed to delete test user."
    );
    debug!(
        "Thread {} sees that test user not exists '{}'",
        std::thread::current().name().unwrap_or("unnamed"),
        name
    );

    let password = get_test_password();
    let user = User::create(name, &password).expect(
        format!(
            "User creation failed {}",
            std::thread::current().name().unwrap_or("unnamed")
        )
        .as_str(),
    );
    debug!(
        "Thread {} was able to create user '{}'",
        std::thread::current().name().unwrap_or("unnamed"),
        name
    );

    user
}

#[cfg(test)]
fn get_test_password() -> Password {
    Password::from_string(TEST_USER_PASS)
}

#[cfg(test)]
#[allow(clippy::unwrap_in_result, clippy::panic_in_result_fn)]
mod tests {
    use super::*;
    use crate::{bail_if_none, debug};

    fn get_test_user(name: &str) -> User {
        super::get_test_user(name)
    }

    #[crate::ctb_test]
    fn test_user_name_and_local_id() {
        let user = User {
            public_info: UserPublicInfo {
                local_id: 42,
                name: "alice".to_string(),
                ..Default::default()
            },
            ..Default::default()
        };
        assert_eq!(user.name(), "alice");
        assert_eq!(user.local_id(), 42);
    }

    #[crate::ctb_test]
    fn test_user_picture_none() {
        let user = User {
            public_info: UserPublicInfo {
                picture: None,
                ..Default::default()
            },
            ..Default::default()
        };
        assert!(user.user_picture().is_none());
    }

    #[crate::ctb_test]
    fn test_user_picture_some() {
        let pic = vec![1, 2, 3];
        let user = User {
            public_info: UserPublicInfo {
                picture: Some(pic.clone()),
                ..Default::default()
            },
            ..Default::default()
        };
        assert_eq!(user.user_picture(), Some(pic.as_slice()));
    }

    #[crate::ctb_test]
    fn test_create_and_get_graph_by_id() -> Result<()> {
        let mut user = User {
            public_info: UserPublicInfo {
                local_id: 1,
                name: "test_graph".to_string(),
                ..Default::default()
            },
            ..Default::default()
        };
        let label = "test_graph_label";
        let graph = user.create_graph(label)?;
        assert_eq!(graph.label, label);
        let id = graph.graph_id;
        let fetched = bail_if_none!(user.get_graph_by_id(id));
        assert_eq!(fetched.label, label);
        Ok(())
    }

    #[crate::ctb_test]
    fn test_get_graph_by_id_none() {
        let user = User::default();
        assert!(user.get_graph_by_id(12345).is_none());
    }

    #[crate::ctb_test]
    fn test_get_all_user_ids() -> Result<()> {
        let user = get_test_user(format!("{}_1", function_name!()).as_str());
        let user2 = get_test_user(format!("{}_2", function_name!()).as_str());
        let ids = get_all_user_ids()?;
        assert!(ids.contains(&user.local_id()));
        assert!(ids.contains(&user2.local_id()));
        user.delete()?;
        user2.delete()?;
        Ok(())
    }

    #[crate::ctb_test]
    fn test_create_and_login_user() -> Result<()> {
        let name = function_name!();
        User::delete_by_name(name).ok(); // .ok() swallows error

        let password = Password::from_string(TEST_USER_PASS);
        let user = super::get_test_user(name);
        assert_eq!(user.name(), name);

        let public_info = bail_if_none!(UserPublicInfo::get_by_name(name)?);
        let logged_in = User::login(public_info, &password)?;
        assert_eq!(logged_in.name(), name);

        // Hold the lock until cleanup is done
        logged_in.delete()?;
        Ok(())
    }

    #[crate::ctb_test]
    fn test_create_and_delete_user() -> Result<()> {
        let user = get_test_user(function_name!());
        user.delete()?;
        assert!(get_user_id_by_name(&user.name()).is_none());
        debug!(get_user_password_by_id(user.local_id()));
        assert!(get_user_password_by_id(user.local_id()).is_none());
        assert!(get_user_picture_by_id(user.local_id()).is_none());
        assert!(get_user_kek_params_by_id(user.local_id()).is_none());
        assert!(get_user_wrapped_dek_by_id(user.local_id()).is_none());
        assert!(get_user_pubkey_by_id(user.local_id()).is_none());
        assert!(!fs::exists(
            get_storage_dir()?
                .join("graphs")
                .join(user.local_id().to_string())
        )?);
        Ok(())
    }

    #[crate::ctb_test]
    fn test_create_duplicate_user_fails() -> Result<()> {
        let name = function_name!();
        let _ = lock_by_name(name).expect("Could not lock name");
        let user = get_test_user(name);
        drop(user);
        let res = User::create(name, &get_test_password());
        assert!(res.is_err());
        get_test_user(name).delete()?;
        Ok(())
    }

    #[crate::ctb_test]
    fn test_login_invalid_password_fails() -> Result<()> {
        let user = get_test_user(function_name!());

        // Clone the picture if it exists, to avoid holding a reference after
        // user is dropped. Not sure if there's an easier way to do this.
        let picture = user.user_picture();
        let new_picture: Option<Vec<u8>> = if picture.is_none() {
            None
        } else {
            Some(
                user.user_picture()
                    .ok_or(anyhow::anyhow!("Failed to get user picture"))?
                    .to_vec(),
            )
        };

        let new_public_info = UserPublicInfo {
            local_id: user.local_id(),
            name: user.name().clone(),
            display_name: user.display_name().clone().map(|v| v.to_vec()),
            uuid: user.uuid().clone(),
            picture: new_picture,
            lock: None,
        };

        // Drop the original user (and lock) before attempting login
        drop(user);

        let res = User::login(
            new_public_info,
            &Password::from_string("wrong_password"),
        );
        assert!(res.is_err());

        Ok(())
    }

    #[crate::ctb_test]
    fn test_local_config_roundtrip() -> Result<()> {
        let user = get_test_user(function_name!());

        let config = user.local_config();
        assert_eq!(
            config.default_store_location,
            get_storage_dir()?.into_os_string()
        );
        user.delete()?;
        Ok(())
    }
}
