//! Persistent configuration for PC settings.
//! Provides serialization and deserialization to a file in the app's cache
//! directory.
//!
//! - Always lock the file before reading or writing to avoid race conditions
//!   between processes.
//! - TODO: Consider reloading settings after external changes since settings
//!   can be changed by other processes.

use anyhow::{Context, Result};
use fs2::FileExt;
use serde::{Deserialize, Serialize};
use std::fs::OpenOptions;
use std::io::{Read, Seek, SeekFrom, Write};
use std::path::PathBuf;

use crate::storage::get_storage_dir;
use crate::utilities::get_current_test_name;

pub static DEFAULT_SHOW_USERS: bool = true;
pub static DEFAULT_SERVER_URL: &str = "https://collectivetoolbox.com";
pub static DEFAULT_BIND_TO_IP: &str = "127.0.0.1";
pub static DEFAULT_KIOSK_MODE: bool = false;
pub static DEFAULT_LOG_STACK_FILE: bool = false;
pub static DEFAULT_FEATURE_LOGIN: bool = true;
pub static DEFAULT_FEATURE_REGISTRATION: bool = true;

#[derive(Serialize, Deserialize, Debug, Clone, PartialEq, Eq)]
#[allow(clippy::struct_excessive_bools)]
#[serde(default)]
pub struct PcSettings {
    pub show_users: bool,
    pub bind_to_ip: String,
    pub server_url: String,
    pub domain_name: Option<String>,
    pub fixed_http_port: Option<u16>,
    pub fixed_https_port: Option<u16>,
    pub http_redirect: bool,
    pub tls_certificate: Option<String>,
    pub tls_private_key: Option<String>,
    pub admin_users: Vec<u64>,
    // Until admin_users can be properly implemented, just use a single admin password
    pub admin_password_hash: Option<String>,
    pub kiosk_mode: bool,
    pub log_stack_file: bool,
    pub feature_login: bool,
    pub feature_registration: bool,
    pub version: u32,
    pub note: String,
}

impl Default for PcSettings {
    fn default() -> Self {
        PcSettings {
            show_users: DEFAULT_SHOW_USERS,
            bind_to_ip: DEFAULT_BIND_TO_IP.into(),
            server_url: DEFAULT_SERVER_URL.into(),
            domain_name: None,
            fixed_http_port: None,
            fixed_https_port: None,
            http_redirect: true,
            tls_certificate: None,
            tls_private_key: None,
            admin_users: Vec::new(),
            admin_password_hash: None,
            kiosk_mode: DEFAULT_KIOSK_MODE,
            log_stack_file: DEFAULT_LOG_STACK_FILE,
            feature_login: DEFAULT_FEATURE_LOGIN,
            feature_registration: DEFAULT_FEATURE_REGISTRATION,
            version: 1,
            note: "Unsigned default settings (stub).".into(),
        }
    }
}

pub fn get_str_setting(setting: &str) -> Option<String> {
    let settings = get_settings();
    match setting {
        "bind_to_ip" => Some(settings.bind_to_ip),
        "server_url" => Some(settings.server_url),
        "domain_name" => settings.domain_name,
        _ => None,
    }
}

pub fn get_bool_setting(setting: &str) -> bool {
    let settings = get_settings();
    match setting {
        "show_users" => settings.show_users,
        "http_redirect" => settings.http_redirect,
        "kiosk_mode" => settings.kiosk_mode,
        "log_stack_file" => settings.log_stack_file,
        _ => unimplemented!(),
    }
}

impl PcSettings {
    /* /// Returns the path to the settings file in the cache directory.
    fn settings_path() -> Result<PathBuf> {
        let mut path = get_storage_dir()
            .context("Failed to get cache directory")?
            .join("config");
        std::fs::create_dir_all(&path)?;
        if cfg!(test) {
            path.push("test_pc_settings.json");
        } else {
            path.push("pc_settings.json");
        }
        Ok(path)
    }*/

    /// Returns the path to the settings file in the cache directory.
    fn settings_path() -> anyhow::Result<PathBuf> {
        // If running tests, prefer thread-local override
        if cfg!(test) {
            
        }

        let mut path = get_storage_dir()?.join("config");
        std::fs::create_dir_all(&path)?;
        let filename = if cfg!(test) {
            format!("{}_pc_settings.json", get_current_test_name())
        } else {
            "pc_settings.json".to_string()
        };
        path.push(filename);
        Ok(path)
    }

    /// Loads `PcSettings` from the settings file, creating one if absent, or
    /// returns an error.
    /// TODO: Add signature verification.
    pub fn load() -> Result<Self> {
        let path = Self::settings_path()?;
        if !std::fs::exists(path.as_path())? {
            PcSettings::default().save()?;
        }
        let file =
            OpenOptions::new().read(true).open(&path).with_context(|| {
                format!("Failed to open settings file: {}", path.display())
            })?;

        // Lock for shared reading (blocks if another process is writing)
        file.lock_shared()
            .context("Failed to acquire shared lock on settings file")?;

        let mut contents = String::new();
        std::io::Read::read_to_string(&mut &file, &mut contents)
            .context("Failed to read settings file")?;

        // Release lock before parsing
        file.unlock()?;

        if contents.trim().is_empty() {
            anyhow::bail!("Settings file is empty");
        }

        let settings = serde_json::from_str(&contents)
            .context("Failed to parse settings JSON")?;
        Ok(settings)
    }

    /// Saves `PcSettings` to the settings file, locking it for exclusive write.
    pub fn save(&self) -> Result<()> {
        let path = Self::settings_path()?;
        let mut file = OpenOptions::new()
            .write(true)
            .create(true)
            .truncate(true)
            .open(&path)
            .with_context(|| {
                format!(
                    "Failed to open settings file for writing: {}",
                    path.display()
                )
            })?;

        // Lock for exclusive writing
        file.lock_exclusive()
            .context("Failed to acquire exclusive lock on settings file")?;

        let data = serde_json::to_string_pretty(self)
            .context("Failed to serialize settings")?;
        file.seek(SeekFrom::Start(0))?;
        file.write_all(data.as_bytes())
            .context("Failed to write settings file")?;
        file.set_len(
            u64::try_from(data.len()).context("Failed to set file length")?,
        )?;

        // Release lock after writing
        file.unlock()?;

        Ok(())
    }
}

pub fn ensure_pc_settings() -> Result<()> {
    PcSettings::load()?;
    Ok(())
}

pub fn get_settings() -> PcSettings {
        crate::storage::pc_settings::PcSettings::load().unwrap_or_default()
}

#[cfg(test)]
#[allow(clippy::unwrap_in_result, clippy::panic_in_result_fn)]
mod tests {
    use super::PcSettings;
    use anyhow::Result;

    #[crate::ctb_test]
    fn test_save_and_load_settings() -> Result<()> {
        let old_settings = PcSettings::load()?;
        let settings = PcSettings {
            bind_to_ip: "0.0.0.0".into(),
            fixed_http_port: Some(8080),
            fixed_https_port: Some(8443),
            admin_users: vec![1, 2, 3],
            ..Default::default()
        };
        settings.save()?;

        let loaded_settings = PcSettings::load()?;
        assert_eq!(settings, loaded_settings);
        old_settings.save()?;
        assert_eq!(old_settings, PcSettings::load()?);
        Ok(())
    }
}
