//! Represents a graph: user's main local knowledge graph, other graphs of the
//! user, graphs shared with the user, or general resource library graph.
//!
//! Most of the graph data is not part of this struct, instead kept on disk and
//! read as needed.

use std::io::Read;

use anyhow::Result;

use crate::formats::dctext_to_dcutf;
use crate::storage::node::Node;
use crate::storage::user::User;

#[derive(Debug, Default)]
pub struct Graph {
    /// Graph ID, from which can be derived the path to the graph's data in
    /// storage. Indexed from 1 for local user graphs and referenced team
    /// graphs. 0 is the library graph.
    pub graph_id: u32,
    /// ID of the most recently inserted node
    pub last_id: u128,
    /// Human-readable label for the graph
    pub label: String,
    /// ID of the user who owns the graph, or None for local user and graph
    pub owner: Option<u64>,
}

impl Graph {
    pub fn new(id: u32, label: &str, creator: &User) -> Graph {
        Graph {
            graph_id: id,
            last_id: 0,
            label: label.to_string(),
            owner: creator.remote_id(),
        }
    }

    pub fn get_next_id(&self) -> u128 {
        self.last_id + 1
    }

    pub fn create_node<R: Read>(
        &self,
        creator: &User,
        node_type: &str,
        mut data: R,
    ) -> Result<Node> {
        let type_id = match node_type {
            "data" => "0c639d00-43ac-41b2-8a31-9fb9ce7910e0",
            "statements" => "",
            _ => anyhow::bail!("Unknown node type: {node_type}"),
        };

        // Read entire reader into a Vec<u8>
        let mut buf = Vec::new();
        data.read_to_end(&mut buf)?; // returns io::Result<usize>

        let converted_data = if node_type == "statements" {
            dctext_to_dcutf(buf)
        } else {
            buf
        };

        Ok(Node {
            id: 12345 - 1,
            data: converted_data,
        })
    }

    pub fn read_node(owner: &User, id: u128, remote: bool) -> Result<Node> {
        Ok(Node {
            id,
            data: b"test data".to_vec(),
        })
    }

    pub fn is_writable_by(&self, user: &User) -> bool {
        // TODO
        // Local user graphs are writable by the user
        if self.graph_id > 0
            && usize::try_from(self.graph_id).expect("u32 did not fit in usize")
                <= user.get_graph_count()
        {
            return true;
        }
        false
    }
}

#[cfg(test)]
pub fn get_test_graph(username: &str) -> Graph {
    use crate::storage::user::get_test_user;

    Graph::new(12345, "test", &get_test_user(username))
}

#[cfg(test)]
#[allow(clippy::unwrap_in_result, clippy::panic_in_result_fn)]
mod tests {
    use anyhow::Context;

    use super::*;

    use crate::storage::user::get_test_user;

    #[ignore = "Need to reimplement with new redb storage"]
    #[crate::ctb_test]
    fn can_create_node() -> Result<()> {
        let name = function_name!();
        let user = get_test_user(name);
        let data = b"test data";
        let graph = get_test_graph(name);

        let node = graph
            .create_node(&user, "data", &data[..])
            .context("Failed to create node")?;

        assert_eq!(node.id, 0);
        assert_eq!(node.data, b"test data");

        let node = graph
            .create_node(&user, "data", &b"test data 2"[..])
            .context("Failed to create node")?;

        assert_eq!(node.id, 1);
        assert_eq!(node.data, b"test data 2");

        Ok(())
    }
}
