//! Public API for db operations. Dispatches automatically over IPC to the real
//! implementations in `ipc::dispatch::db`.

use crate::formats::base64::{
    bytes_to_standard_base64, standard_base64_to_bytes,
};
use crate::utilities::call_ipc_sync;
use crate::utilities::json::jq;
use crate::workspace::ipc_old::resolve_workspace_channel;
use crate::{bail_if_none, debug_fmt, error, json_value, log, warn};
use anyhow::Result;

/// Helper to call IPC and extract a value using jq.
fn ipc_get<T: std::str::FromStr>(
    method: &str,
    args: serde_json::Value,
    jq_path: &str,
) -> Option<T> {
    let channel = resolve_workspace_channel();
    if channel.is_err() {
        return None;
    }
    let channel = channel.unwrap();
    let ipc_result = call_ipc_sync(&channel, method, args);
    let ipc_result = match ipc_result {
        Ok(r) => r,
        Err(_) => return None,
    };

    let ipc_result = ipc_result.as_str();

    // Parse JSON and check for "value" key
    let parsed: serde_json::Value = serde_json::from_str(ipc_result).ok()?;
    let value = parsed.get("value");
    let none = parsed.get("none");
    if value.is_none() && none.is_none() {
        warn!("ipc_put_delete: {} returned neither value nor none", method);
        return None;
    }
    if none.is_some() {
        return None;
    }

    let value = jq(jq_path, ipc_result).ok()?;
    let s = value.as_str();
    T::from_str(s).ok()
}

/// Helper to call IPC for put/delete operations.
fn ipc_put_delete(method: &str, args: serde_json::Value) -> Result<()> {
    let channel = resolve_workspace_channel()?;
    let ipc_result = call_ipc_sync(&channel, method, args)?;
    let parsed: serde_json::Value = serde_json::from_str(ipc_result.as_str())
        .ok()
        .ok_or(anyhow::anyhow!("Failed to parse IPC response"))?;
    let value = parsed.get("value");
    let error = parsed.get("error");
    if value.is_none() && error.is_none() {
        warn!(
            "ipc_put_delete: {} returned neither value nor error",
            method
        );
        return Err(anyhow::anyhow!("IPC response missing value; {method}"));
    }
    if error.is_some() {
        let error_msg = error
            .and_then(|e| e.as_str())
            .unwrap_or("Unknown error")
            .to_string();
        error!("ipc_put_delete: {} returned error: {}", method, &error_msg);
        return Err(anyhow::anyhow!("IPC error from {method}: {error_msg}"));
    }
    Ok(())
}

/// Helper to call IPC for `Vec<u64>`.
fn ipc_get_vec_u64(method: &str, args: serde_json::Value) -> Result<Vec<u64>> {
    let channel = resolve_workspace_channel()?;
    let ipc_result = call_ipc_sync(&channel, method, args);
    let ipc_result = match ipc_result {
        Ok(r) => r,
        Err(e) => return Err(anyhow::anyhow!("Failed to call IPC sync: {e}")),
    };
    debug_fmt!("ipc_get_vec_u64: {} -> {}", method, ipc_result);
    let parsed: serde_json::Value = serde_json::from_str(ipc_result.as_str())?;
    let value = parsed.get("value");
    let error = parsed.get("error");
    if value.is_none() && error.is_none() {
        warn!(
            "ipc_get_vec_u64: {} returned neither value nor error: {:?}",
            method, ipc_result
        );
        return Err(anyhow::anyhow!("IPC response missing value; {method}"));
    }
    if error.is_some() {
        return Err(anyhow::anyhow!("IPC error from {method}: {ipc_result}"));
    }
    let value = bail_if_none!(value);
    let arr = bail_if_none!(value.as_array());
    let mut result = Vec::new();
    for v in arr {
        if let Some(n) = v.as_u64() {
            result.push(n);
        } else {
            error!("ipc_get_vec_u64: {} returned non-u64 value", method);
            return Err(anyhow::anyhow!(
                "IPC response contained non-u64 value; {method}"
            ));
        }
    }
    Ok(result)
}

/// Get a u64 value by string key.
pub fn get_str_u64(db: &str, key: &str) -> Option<u64> {
    ipc_get::<u64>(
        "get_str_u64",
        json_value!({ "db".to_string() =>  db, "key".to_string() =>  key }),
        ".value",
    )
}

pub fn get_all_u64_keys(db: &str) -> Result<Vec<u64>> {
    ipc_get_vec_u64(
        "get_all_u64_keys",
        json_value!({ "db".to_string() =>  db }),
    )
}

/// Put a u64 value by string key.
pub fn put_str_u64(db: &str, key: &str, value: u64) -> Result<()> {
    ipc_put_delete(
        "put_str_u64",
        json_value!({ "db".to_string() =>  db, "key".to_string() =>  key, "value".to_string() =>  value }),
    )
}

/// Delete a u64 value by string key.
pub fn delete_str_u64(db: &str, key: &str) -> Result<()> {
    ipc_put_delete(
        "delete_str_u64",
        json_value!({ "db".to_string() =>  db, "key".to_string() =>  key }),
    )
}

/// Get a string value by u64 key.
pub fn get_u64_str(db: &str, key: u64) -> Option<String> {
    ipc_get::<String>(
        "get_u64_str",
        json_value!({ "db".to_string() =>  db, "key".to_string() =>  key }),
        ".value",
    )
}

/// Put a string value by u64 key.
pub fn put_u64_str(db: &str, key: u64, value: &str) -> Result<()> {
    ipc_put_delete(
        "put_u64_str",
        json_value!({ "db".to_string() =>  db, "key".to_string() =>  key, "value".to_string() =>  value }),
    )
}

/// Delete a string value by u64 key.
pub fn delete_u64_str(db: &str, key: u64) -> Result<()> {
    ipc_put_delete(
        "delete_u64_str",
        json_value!({ "db".to_string() =>  db, "key".to_string() =>  key }),
    )
}

/// Get a byte vector by u64 key.
pub fn get_u64_bytes(db: &str, key: u64) -> Option<Vec<u8>> {
    let channel = resolve_workspace_channel();
    if channel.is_err() {
        return None;
    }
    let channel = channel.ok()?;
    let ipc_result = call_ipc_sync(
        &channel,
        "get_u64_bytes",
        json_value!({ "db".to_string() =>  db, "key".to_string() =>  key }),
    );
    let ipc_result = match ipc_result {
        Ok(r) => r,
        Err(_) => return None,
    };
    let parsed: serde_json::Value =
        serde_json::from_str(ipc_result.as_str()).ok()?;
    let value = parsed.get("value")?.as_str()?;
    // Seems like I ought to be able to do something like this, but apparently
    // not:
    // let r: Option<Vec<u8>> = serde_json::from_str(value).ok();
    let r = standard_base64_to_bytes(value.to_string()).ok();
    log!(format!(
        "ipc_get: get_u64_bytes {} {} -> {} bytes",
        db,
        key,
        String::from_utf8_lossy(r.clone()?.as_slice()),
    ));
    r
}

/// Put a byte slice by u64 key.
pub fn put_u64_bytes(db: &str, key: u64, value: &[u8]) -> Result<()> {
    return ipc_put_delete(
        "put_u64_bytes",
        json_value!({ "db".to_string() =>  db, "key".to_string() =>  key, "value".to_string() =>  value }),
    );
    let encoded = bytes_to_standard_base64(value);
    ipc_put_delete(
        "put_u64_bytes",
        json_value!({ "db".to_string() =>  db, "key".to_string() =>  key, "value".to_string() =>  encoded }),
    )
}

/// Delete a byte slice by u64 key.
pub fn delete_u64_bytes(db: &str, key: u64) -> Result<()> {
    ipc_put_delete(
        "delete_u64_bytes",
        json_value!({ "db".to_string() =>  db, "key".to_string() =>  key }),
    )
}

/// username -> `user_id`
pub fn get_user_id(name: &str) -> Option<u64> {
    get_str_u64("users/ids", name)
}

pub fn get_all_user_ids() -> Result<Vec<u64>> {
    get_all_u64_keys("users/ids_rev")
}

pub fn get_user_name(user_id: u64) -> Option<String> {
    get_u64_str("users/ids_rev", user_id)
}

pub fn put_user_id(name: &str, user_id: u64) -> Result<()> {
    put_str_u64("users/ids", name, user_id)?;
    put_u64_str("users/ids_rev", user_id, name)
}

pub fn delete_user_id(name: &str, user_id: u64) -> Result<()> {
    delete_str_u64("users/ids", name)?;
    delete_u64_str("users/ids_rev", user_id)
}

/// username -> `user_uuid`
pub fn get_user_uuid(id: u64) -> Option<Vec<u8>> {
    get_u64_bytes("users/uuids", id)
}

pub fn put_user_uuid(id: u64, uuid: &[u8]) -> Result<()> {
    put_u64_bytes("users/uuids", id, uuid)
}

pub fn delete_user_uuid(id: u64) -> Result<()> {
    delete_u64_bytes("users/uuids", id)
}

/// `user_id` -> PHC password hash (verification only)
pub fn get_user_auth(user_id: u64) -> Option<String> {
    get_u64_str("users/auth", user_id)
}

pub fn put_user_auth(user_id: u64, phc_hash: &str) -> Result<()> {
    put_u64_str("users/auth", user_id, phc_hash)
}

pub fn delete_user_auth(user_id: u64) -> Result<()> {
    delete_u64_str("users/auth", user_id)
}

/// `user_id` -> user display name (binary data)
pub fn get_user_display_name(user_id: u64) -> Option<Vec<u8>> {
    get_u64_bytes("users/display_names", user_id)
}

pub fn put_user_display_name(user_id: u64, display_name: &[u8]) -> Result<()> {
    put_u64_bytes("users/display_names", user_id, display_name)
}

pub fn delete_user_display_name(user_id: u64) -> Result<()> {
    delete_u64_bytes("users/display_names", user_id)
}
/// `user_id` -> user picture (binary data)
pub fn get_user_pictures(user_id: u64) -> Option<Vec<u8>> {
    get_u64_bytes("users/pictures", user_id)
}

pub fn put_user_picture(user_id: u64, picture_data: &[u8]) -> Result<()> {
    put_u64_bytes("users/pictures", user_id, picture_data)
}

pub fn delete_user_picture(user_id: u64) -> Result<()> {
    delete_u64_bytes("users/pictures", user_id)
}

/// `user_id` -> KEK params (Argon2id PHC string)
pub fn get_user_kek_params(user_id: u64) -> Option<Vec<u8>> {
    get_u64_bytes("users/key_encryption_key_params", user_id)
}

pub fn put_user_kek_params(user_id: u64, kek_params: &[u8]) -> Result<()> {
    put_u64_bytes("users/key_encryption_key_params", user_id, kek_params)
}

pub fn delete_user_kek_params(user_id: u64) -> Result<()> {
    delete_u64_bytes("users/key_encryption_key_params", user_id)
}

/// `user_id` -> wrapped data encryption key (DEK)
pub fn get_user_wrapped_dek(user_id: u64) -> Option<Vec<u8>> {
    get_u64_bytes("users/wrapped_dek", user_id)
}

pub fn put_user_wrapped_dek(user_id: u64, wrapped_dek: &[u8]) -> Result<()> {
    put_u64_bytes("users/wrapped_dek", user_id, wrapped_dek)
}

pub fn delete_user_wrapped_dek(user_id: u64) -> Result<()> {
    delete_u64_bytes("users/wrapped_dek", user_id)
}

/// `user_id` -> public key (ed25519)
pub fn get_user_pubkey(user_id: u64) -> Option<Vec<u8>> {
    get_u64_bytes("users/pubkeys", user_id)
}

pub fn put_user_pubkey(user_id: u64, pubkey: &[u8]) -> Result<()> {
    put_u64_bytes("users/pubkeys", user_id, pubkey)
}

pub fn delete_user_pubkey(user_id: u64) -> Result<()> {
    delete_u64_bytes("users/pubkeys", user_id)
}
