//! Defines the web routes for the application.

use axum::{Router, routing::get, routing::post};

use crate::debug;
use crate::io::webui::AppState;
#[allow(clippy::wildcard_imports)]
use crate::io::webui::controllers::*;

pub fn build_routes(state: AppState) -> Router {
    debug!("Building router");
    // Build router
    Router::new()
        // --- web controller routes ---
        .route("/", get(web::get_index))
        .route("/privacy-policy", get(web::privacy_policy))
        .route("/subscribe", get(web::subscribe))
        // --- auth controller routes ---
        .route("/login", get(auth::get_login).post(auth::post_login))
        .route("/login-password", post(auth::post_login_password))
        .route("/registration", post(auth::post_registration))
        // --- app controller routes ---
        .route(
            "/pc-settings",
            get(app::get_public_pc_settings).post(app::post_public_pc_settings),
        )
        .route("/home", get(app::get_home))
        // --- search controller routes ---
        .route("/search", get(search::get_index))
        // --- graph controller routes ---
        .route("/nodes", get(graph::get_nodes_index))
        .route("/nodes/view", get(graph::get_nodes_view))
        .route(
            "/nodes/create",
            get(graph::get_nodes_create).post(graph::post_nodes_create),
        )
        .route(
            "/nodes/upload",
            get(graph::get_nodes_upload).post(graph::post_nodes_upload),
        )
        // --- base controller routes (docs, css, static, fallback) ---
        .route("/docs", get(base::get_doc_index))
        .route("/docs/{*path}", get(base::get_doc_page))
        .route("/app.css", get(base::get_app_css))
        .route("/src.zip", get(base::get_src_zip))
        .route("/dependencies.zip", get(base::get_dependencies_zip))
        .route("/installer-linux-x64", get(base::get_installer_linux_x64))
        // static files or 404 page fallback
        .fallback(get(base::static_or_404))
        .with_state(state)
}

pub fn base_url() -> String {
    // TODO: Load from settings. See webui.rs for how to do this
    "http://localhost:8080".to_string()
}