use std::io::Read;

use axum::Form;
use axum::body::Bytes;
use axum::extract::{Query, State};
use axum::response::{IntoResponse, Redirect, Response};
use axum_typed_multipart::{TryFromMultipart, TypedMultipart};
use serde::Deserialize;

use crate::io::webui::{
    AppState, AuthenticatedUser, PageQuery, RequestState, error_400, error_403,
    respond_page,
};
use crate::io::webui::controllers::base::{redirect_temporary};
use crate::utilities::strtovec;
use crate::{get_user_and_graph, json_value};

pub async fn get_nodes_index(
    State(state): State<AppState>,
    req: RequestState,
    _q: Query<PageQuery>,
) -> Response {
    respond_page(&state, req, "nodes.index", &json_value!({}))
}

pub async fn get_nodes_view(
    State(state): State<AppState>,
    req: RequestState,
    _q: Query<PageQuery>,
) -> Response {
    respond_page(&state, req, "nodes.view", &json_value!({}))
}

pub async fn get_nodes_create(
    State(state): State<AppState>,
    req: RequestState,
) -> Response {
    respond_page(&state, req, "nodes.create", &json_value!({}))
}

#[derive(Deserialize)]
pub struct CreateNodeForm {
    graph: u32,
    node_type: String,
    node_content: String,
}

pub async fn post_nodes_create(
    State(state): State<AppState>,
    req: RequestState,
    sess: AuthenticatedUser,
    Form(input): Form<CreateNodeForm>,
) -> Response {
    get_user_and_graph!(&state, &req, sess, input.graph, user, graph);

    if let Err(e) = graph.create_node(
        &user,
        input.node_type.as_str(),
        strtovec(input.node_content.as_str()).as_slice(),
    ) {
        return error_400(&state, &req, e);
    }

    redirect_temporary(req.is_js_request, "/nodes")
}

pub async fn get_nodes_upload(
    State(state): State<AppState>,
    req: RequestState,
) -> Response {
    respond_page(&state, req, "nodes.upload", &json_value!({}))
}

#[derive(TryFromMultipart)]
#[try_from_multipart(strict)]
pub struct UploadNodeForm {
    graph: u32,
    node_type: String,
    #[form_data(limit = "unlimited")]
    node_content: Bytes,
}

/// Wrapper around axum:body:Bytes that implements Read. Untested.
struct ReadableBytes {
    inner: Bytes,
}

// Untested
impl Read for ReadableBytes {
    fn read(&mut self, buf: &mut [u8]) -> std::io::Result<usize> {
        let len = std::cmp::min(buf.len(), self.inner.len());
        buf[..len].copy_from_slice(&self.inner[..len]);
        self.inner = self.inner.slice(len..);
        Ok(len)
    }
}

impl ReadableBytes {
    fn new(bytes: Bytes) -> Self {
        Self { inner: bytes }
    }
}

pub async fn post_nodes_upload(
    State(state): State<AppState>,
    req: RequestState,
    sess: AuthenticatedUser,
    TypedMultipart(form): TypedMultipart<UploadNodeForm>,
) -> Response {
    get_user_and_graph!(&state, &req, sess, form.graph, user, graph);
    let readable_bytes = ReadableBytes::new(form.node_content);

    if let Err(e) =
        graph.create_node(&user, form.node_type.as_str(), readable_bytes)
    {
        return error_400(&state, &req, e);
    }

    redirect_temporary(req.is_js_request, "/nodes")
}
