//! Controller for static assets and other routes shared between the app and the
//! general web site.
use std::borrow::Cow;

use anyhow::anyhow;
use async_trait::async_trait;
use axum::{
    body::Body,
    extract::State,
    http::{HeaderValue, Uri, header},
    response::{IntoResponse, Redirect, Response},
};
use axum_extra::response::file_stream::FileStream;
use mime_guess;
use tokio::fs::File;
use tokio_util::io::ReaderStream;

use crate::storage::get_asset;
use crate::storage::pc_settings::DEFAULT_SERVER_URL;
use crate::utilities::build_info;
use crate::{
    io::webui::{AppState, RequestState, error_400, error_404, render_view},
    json_value,
};

/// Build a JSON redirect response with X-CollectiveToolbox-IsJsRedirect header.
fn json_redirect_response(location: &str) -> Response {
    let mut resp = Response::new(Body::from(
        serde_json::json!({ "url": location }).to_string(),
    ));
    resp.headers_mut().insert(
        "X-CollectiveToolbox-IsJsRedirect",
        HeaderValue::from_static("true"),
    );
    resp.headers_mut().insert(
        header::CONTENT_TYPE,
        HeaderValue::from_static("application/json"),
    );
    *resp.status_mut() = axum::http::StatusCode::OK;
    resp
}

/// Redirect helper for 307 (Temporary Redirect, preserves method).
/// If X-CollectiveToolbox-IsJsRequest header is present, returns JSON with
/// target URL and X-CollectiveToolbox-IsJsRedirect response header.
pub fn redirect_temporary_preserve_method(
    is_js_req: bool,
    location: &str,
) -> Response {
    if is_js_req {
        json_redirect_response(location)
    } else {
        axum::response::Redirect::temporary(location).into_response()
    }
}

/// Redirect helper for 303 (See Other, changes POST to GET).
/// If X-CollectiveToolbox-IsJsRequest header is present, returns JSON with
/// target URL and X-CollectiveToolbox-IsJsRedirect response header.
pub fn redirect_temporary(
    is_js_req: bool,
    location: &str,
) -> Response {
    if is_js_req {
        json_redirect_response(location)
    } else {
        axum::response::Redirect::to(location).into_response()
    }
}

/// Redirect helper for 308 (Permanent Redirect, preserves method).
/// If X-CollectiveToolbox-IsJsRequest header is present, returns JSON with
/// target URL and X-CollectiveToolbox-IsJsRedirect response header.
pub fn redirect_permanent(
    is_js_req: bool,
    location: &str,
) -> Response {
    if is_js_req {
        json_redirect_response(location)
    } else {
        axum::response::Redirect::permanent(location).into_response()
    }
}

pub async fn get_app_css(
    State(state): State<AppState>,
    req: RequestState,
) -> Response {
    match render_view(
        &state.hbs,
        "encre_css".to_string(),
        &req,
        &json_value!({}),
    ) {
        Ok(css) => {
            let mut resp = Response::new(Body::from(css));
            resp.headers_mut().insert(
                header::CONTENT_TYPE,
                HeaderValue::from_static("text/css"),
            );
            resp
        }
        Err(e) => error_400(&state, &req, e),
    }
}

pub async fn get_installer_linux_x64(
    State(state): State<AppState>,
    req: RequestState,
) -> Response {
    // Resolve the current executable path first
    let path = match std::env::current_exe() {
        Ok(p) => p,
        Err(e) => return error_400(&state, &req, anyhow!(e)),
    };

    // Open the file asynchronously and stream it back
    let file = match File::open(&path).await {
        Ok(f) => f,
        Err(e) => return error_400(&state, &req, anyhow!(e)),
    };

    let stream = ReaderStream::new(file);

    let build_info = build_info();
    let version = build_info.version;
    let commit = build_info.commit;

    let file_name = format!("ctoolbox-installer-linux-x64-{version}-{commit}");

    let file_stream_resp = FileStream::new(stream).file_name(file_name);
    let mut resp = file_stream_resp.into_response();
    resp.headers_mut().insert(
        header::CONTENT_TYPE,
        HeaderValue::from_static("application/x-executable"),
    );
    resp
}

fn get_asset_zip(state: &AppState, req: RequestState, path: &str) -> Response {
    let path = format!("{path}.zip");
    let asset = get_asset(&path);

    if let Some(bytes) = asset {
        let build_info = build_info();
        let version = build_info.version;
        let commit = build_info.commit;

        let file_name = format!("ctoolbox-{path}-{version}-{commit}.zip");

        let mut resp = Response::new(Body::from(bytes));
        resp.headers_mut().insert(
            header::CONTENT_TYPE,
            HeaderValue::from_static("application/zip"),
        );

        // Set Content-Disposition so the response is downloaded as `file_name`.
        if let Ok(disposition) = HeaderValue::from_str(&format!(
            "attachment; filename=\"{file_name}\""
        )) {
            resp.headers_mut()
                .insert(header::CONTENT_DISPOSITION, disposition);
        }

        return resp;
    }

    // Otherwise render 404 page
    error_404(
        state,
        &req,
        format!("The requested URL '{path}' was not found."),
    )
}

pub async fn get_src_zip(
    State(state): State<AppState>,
    req: RequestState,
) -> Response {
    return get_asset_zip(&state, req, "src");
}

// Trying to bundle this caused OOM on compilation
/*pub async fn get_dependencies_zip(
    State(state): State<AppState>,
    req: RequestState,
) -> Response {
    return get_asset_zip(&state, req, "dependencies");
}*/

pub async fn get_dependencies_zip(
    State(state): State<AppState>,
    req: RequestState,
) -> Response {
    // Resolve the current executable path first
    let exe_path = match std::env::current_exe() {
        Ok(p) => p,
        Err(e) => return error_400(&state, &req, anyhow!(e)),
    };
    let build_info = build_info();
    let version = build_info.version;
    let commit = build_info.commit;
    let deps_original_filename = format!("ctoolbox-dependencies-{commit}.zip");
    let deps_filename = format!("ctoolbox-dependencies-{version}-{commit}.zip");
    let deps_primary_url = format!("{DEFAULT_SERVER_URL}/dependencies.zip");

    let deps_path = match exe_path.parent() {
        Some(parent) => parent.join(&deps_original_filename),
        None => {
            return redirect_temporary(req.is_js_request, &deps_primary_url);
        }
    };

    // Try to open dependencies.zip; if missing, redirect; other errors -> error_400.
    match File::open(&deps_path).await {
        Ok(file) => {
            let stream = ReaderStream::new(file);
            let file_stream_resp =
                FileStream::new(stream).file_name(deps_filename);
            let mut resp = file_stream_resp.into_response();
            resp.headers_mut().insert(
                header::CONTENT_TYPE,
                HeaderValue::from_static("application/zip"),
            );
            resp
        }
        Err(e) => {
            if e.kind() == std::io::ErrorKind::NotFound {
                redirect_temporary(req.is_js_request, &deps_primary_url)
            } else {
                error_400(&state, &req, anyhow!(e))
            }
        }
    }
}

pub async fn get_doc_index(
    State(state): State<AppState>,
    req: RequestState,
) -> Response {
    redirect_temporary(req.is_js_request, "/docs/rust/ctoolbox/index.html")
}

pub async fn get_doc_page(
    State(state): State<AppState>,
    req: RequestState,
    path: axum::extract::Path<String>,
) -> Response {
    let path = path.as_str();

    asset_or_404(&state, req, format!("/docs/{path}").as_str())
}

pub async fn static_or_404(
    State(state): State<AppState>,
    req: RequestState,
    uri: Uri,
) -> Response {
    asset_or_404(&state, req, uri.path())
}

fn asset_or_404(state: &AppState, req: RequestState, path: &str) -> Response {
    let path = path.trim_start_matches('/');

    // Try serving embedded asset
    let mut asset = get_asset(path);
    if asset.is_none() {
        asset = get_asset(format!("web/{path}").as_str());
    }

    if let Some(bytes) = asset {
        let mime_guess = mime_guess::from_path(path).first();
        let mime_guess_str: Cow<'static, str> = match mime_guess {
            Some(mime) => Cow::Owned(mime.essence_str().to_string()),
            None => Cow::Borrowed("application/octet-stream"),
        };

        let mut resp = Response::new(Body::from(bytes));
        resp.headers_mut().insert(
            header::CONTENT_TYPE,
            HeaderValue::from_str(mime_guess_str.as_ref()).unwrap_or_else(
                |_| HeaderValue::from_static("application/octet-stream"),
            ),
        );
        return resp;
    }

    // Otherwise render 404 page
    error_404(
        state,
        &req,
        format!("The requested URL '{path}' was not found."),
    )
}

#[cfg(test)]
mod tests {
    use crate::io::webui::test_helpers::{
        test_get_no_login, test_get_no_login_json, test_get_redirect_no_login,
    };
    use crate::utilities::{
        assert_string_contains, assert_string_not_contains,
    };

    #[crate::ctb_test(tokio::test)]
    async fn can_get_doc_index() {
        let (status, location) = test_get_redirect_no_login("/docs").await;
        assert_eq!(status, 303);
        assert_eq!(location, "/docs/rust/ctoolbox/index.html");
    }

    #[crate::ctb_test(tokio::test)]
    async fn can_get_rust_index() {
        let (status, body) =
            test_get_no_login("/docs/rust/ctoolbox/index.html").await;
        assert_eq!(status, 200);
        assert!(body.contains("<title>ctoolbox - Rust</title>"));
    }

    #[crate::ctb_test(tokio::test)]
    async fn can_get_css() {
        let (status, body) = test_get_no_login("/app.css").await;
        assert_eq!(status, 200);
        assert!(body.contains("Abstract-Polygon-Background")); // Check for app code
        assert!(body.contains("SFMono-Regular")); // Check for encrecss code
    }

    #[crate::ctb_test(tokio::test)]
    async fn can_download_source() {
        let (status, body) = test_get_no_login("/src.zip").await;
        assert_eq!(status, 200);
        assert!(body.starts_with("PK"));
    }

    #[crate::ctb_test(tokio::test)]
    async fn can_get_404() {
        let (status, body) = test_get_no_login("/nonexistent").await;
        assert_eq!(status, 404);
        assert_string_contains("was not found.</h1>", &body);
    }

    #[crate::ctb_test(tokio::test)]
    async fn can_get_404_json() {
        let (status, body) = test_get_no_login_json("/nonexistent").await;
        assert_eq!(status, 404);
        assert_string_contains("was not found.", &body);
        assert_string_not_contains("was not found.</h1>", &body);
    }
}
