use anyhow::Result;
use axum::debug_handler;
use axum::extract::State;
use axum::response::{IntoResponse, Redirect, Response};
use axum_extra::extract::CookieJar;
use axum_extra::extract::cookie::Cookie;
use axum_typed_multipart::TryFromMultipart;
use maplit::btreemap;
use serde::{Deserialize, Serialize};

use crate::io::webui::flexible_form::FlexibleForm;
use crate::io::webui::session_auth::Session;
use crate::io::webui::{
    AppState, RequestState, error_404, recoverable_error, respond_dialog,
};
use crate::io::webui::controllers::base::{redirect_temporary};
use crate::storage::pc_settings::get_bool_setting;
use crate::storage::user::{User, UserPublicInfo};
use crate::utilities::feature;
use crate::utilities::password::Password;
use crate::{debug, error, info, json_value};

fn login_disabled(
    State(state): State<AppState>,
    req: RequestState,
) -> Response {
    let kiosk_mode = get_bool_setting("kiosk_mode");
    respond_dialog(
        &state,
        req,
        "login_disabled",
        &btreemap! { "kiosk_mode".to_string() => kiosk_mode },
    )
}

fn registration_disabled(
    State(state): State<AppState>,
    req: RequestState,
) -> Response {
    let kiosk_mode = get_bool_setting("kiosk_mode");
    respond_dialog(
        &state,
        req,
        "registration_disabled",
        &btreemap! { "kiosk_mode".to_string() => kiosk_mode },
    )
}

pub async fn get_login(
    State(state): State<AppState>,
    req: RequestState,
) -> Response {
    if (!feature("login")) {
        return login_disabled(State(state), req);
    }

    respond_dialog(&state, req, "login", &json_value!({}))
}

#[derive(TryFromMultipart, Serialize, Deserialize)]
#[try_from_multipart(strict)]
pub struct LoginForm {
    username: String,
}

pub async fn post_login(
    State(state): State<AppState>,
    req: RequestState,
    FlexibleForm(input): FlexibleForm<LoginForm>,
) -> Response {
    if (!feature("login")) {
        return login_disabled(State(state), req);
    }

    let username = input.username;

    let local_exists = local_account_exists(&username);
    let remote_exists = remote_account_exists(&username);

    // FIXME  TODO
    /*if !local_exists && !remote_exists {
        // Registration view with suggested username
        return respond_page(
            &state,
            req,
            "register",
            &btreemap! { "username".to_string() => username.clone() },
        );
    } else if !local_exists && remote_exists {
        // Log in with remote account only: auto-create local account and do initial sync
        panic!("Not implemented: log in with remote account only");
    }*/

    // Either both exist, or only local exists. No problem, continue
    respond_dialog(
        &state,
        req,
        "login_password",
        &btreemap! { "username".to_string() => username.clone() },
    )
}

#[derive(TryFromMultipart, Serialize, Deserialize)]
#[try_from_multipart(strict)]
pub struct LoginPasswordForm {
    username: String,
    password: String, // Hidden, just kept to maintain state
}
pub async fn post_login_password(
    State(state): State<AppState>,
    req: RequestState,
    jar: CookieJar,
    FlexibleForm(input): FlexibleForm<LoginPasswordForm>,
) -> impl IntoResponse {
    if (!feature("login")) {
        return (jar, login_disabled(State(state), req));
    }

    let username = input.username;
    let password = Password {
        password: input.password.as_bytes().to_vec(),
    };
    debug!(format!(
        "post_login_password: username={}, password.len={}",
        username.clone(),
        password.password.len()
    ));
    let user_public_info = UserPublicInfo::get_by_name(&username);
    let Ok(Some(user_public_info)) = user_public_info else {
        return (
            jar,
            error_404(
                &state,
                &req,
                "The account seems to have disappeared or been removed? This may indicate a bug",
            ),
        );
    };

    let logged_in = User::login(user_public_info, &password);
    let Ok(logged_in) = logged_in else {
        return (
            jar,
            recoverable_error(
                &state,
                req,
                "The password is most likely incorrect, or perhaps there was an error looking up the user.",
            ),
        );
    };

    // Log in was ok; return redirect home with session cookie set
    let session = Session::new(&mut state.clone(), logged_in).await;
    let updated_jar = jar.add(Cookie::new("session", session.id()));

    (updated_jar, redirect_temporary(req.is_js_request, "/home"))
}

#[derive(TryFromMultipart, Serialize, Deserialize)]
#[try_from_multipart(strict)]
pub struct RegistrationForm {
    username: String,
    password: String,
    password_confirm: String,
}
/// Handles registration form submission.
/// Logs detailed errors if multipart parsing fails or if any step fails.
#[debug_handler]
pub async fn post_registration(
    State(state): State<AppState>,
    req: RequestState,
    jar: CookieJar,
    FlexibleForm(input): FlexibleForm<RegistrationForm>,
) -> impl IntoResponse {
    if (!feature("registration")) {
        return registration_disabled(State(state), req);
    }

    debug!("post_registration: received registration request");
    info!("post_registration: received registration request");

    let username = input.username.clone();
    let password = Password {
        password: input.password.as_bytes().to_vec(),
    };
    let password_confirm = Password {
        password: input.password_confirm.as_bytes().to_vec(),
    };

    if password != password_confirm {
        error!(
            "post_registration: passwords did not match for user '{}'",
            username
        );
        return recoverable_error(
            &state,
            req,
            "Passwords did not match".to_string(),
        );
    }

    error!("post_registration: creating user '{}'", &username);
    let user = match User::create(&username, &password) {
        Ok(user) => user,
        Err(e) => {
            error!(
                "post_registration: failed to create user '{}': {:?}",
                username, &e
            );
            return recoverable_error(
                &state,
                req,
                format!("Failed to create user: {e:?}"),
            );
        }
    };
    let Ok(Some(user_info)) = UserPublicInfo::get_by_name(&username) else {
        error!(
            "post_registration: failed to get user info for '{}'",
            &username
        );
        return recoverable_error(
            &state,
            req,
            format!("Failed to get user info for '{username}'"),
        );
    };
    if user_info.name() != username {
        error!(
            "post_registration: user info name mismatch: '{}' != '{}'",
            user_info.name(),
            &username
        );
        return recoverable_error(
            &state,
            req,
            format!(
                "User info name mismatch: '{}' != '{}'",
                user_info.name(),
                &username
            ),
        );
    }
    info!("post_registration: created user '{}'", &username);

    post_login_password(
        axum::extract::State(state),
        req,
        jar,
        // FIXME: Avoid cloning password string. The zeroizing can't actually
        // clear out all the copies if it's being passed around everywhere like
        // this, but that might just be inevitable.
        FlexibleForm(LoginPasswordForm {
            username: username.clone(),
            password: password.as_string_not_zeroizing(),
        }),
    )
    .await
    .into_response()
}

// ================ Auth and user helpers (stubs) ================

fn local_account_exists(username: &str) -> bool {
    if let Ok(Some(_)) = UserPublicInfo::get_by_name(username) {
        return true;
    }
    false
}

fn remote_account_password_matches(
    _username: &String,
    _password: &String,
) -> bool {
    true
}

fn remote_account_exists(_username: &String) -> bool {
    false
}

#[cfg(test)]
#[allow(clippy::unwrap_in_result, clippy::panic_in_result_fn)]
mod auth_controller_tests {
    use super::*;
    use crate::io::webui::test_helpers::{
        assert_eq_or_print_body, assert_or_print_body,
        assert_successful_and_return_resp, test_app, test_get_no_login,
        test_post_no_login, test_request, test_request_get_response,
    };
    use crate::storage::user::{get_test_user, lock_by_name};
    use axum::http::StatusCode;
    use ctb_test_macro::ctb_test;
    use http::Method;

    // In the interest of actually testing this flow, these few tests should NOT
    // use the default test password (which bypasses the hash check for
    // performance)

    const THIS_TEST_USER_PASS: &str = "test_password_auth_controller";

    #[ctb_test(tokio::test)]
    async fn test_get_login_route() {
        let (status, body) = test_get_no_login("/login").await;
        assert_eq!(status, StatusCode::OK);
        assert!(body.contains("login"));
    }

    #[ctb_test(tokio::test)]
    async fn test_registration_and_login_flow() -> Result<()> {
        let name = function_name!();
        let _ = lock_by_name(name)?;
        User::delete_by_name(name).ok();
        let (_state, app) = test_app();
        #[derive(serde::Serialize)]
        struct RegistrationForm<'a> {
            username: &'a str,
            password: &'a str,
            password_confirm: &'a str,
        }
        let reg_form = RegistrationForm {
            username: name,
            password: THIS_TEST_USER_PASS,
            password_confirm: THIS_TEST_USER_PASS,
        };
        let resp = test_request_get_response(
            &app,
            axum::http::Method::POST,
            "/registration",
            None,
            None,
            None,
            None,
            Some(&reg_form),
        )
        .await;
        let resp = assert_successful_and_return_resp(resp, true).await;
        let cookie = resp.headers().get("Set-Cookie");
        assert!(cookie.is_some(), "No Set-Cookie header found");
        let user_info = UserPublicInfo::get_by_name(name)?
            .expect("Failed to get user info");
        assert!(user_info.name() == name);

        #[derive(serde::Serialize)]
        struct LoginForm<'a> {
            username: &'a str,
        }
        let login_form = LoginForm { username: name };
        let (status, body) =
            test_post_no_login("/login", None, None, Some(&login_form)).await;
        assert_eq!(status, StatusCode::OK);
        assert!(body.contains("login-password"));

        #[derive(serde::Serialize)]
        struct LoginPasswordForm<'a> {
            username: &'a str,
            password: &'a str,
        }
        let login_pw_form = LoginPasswordForm {
            username: name,
            password: THIS_TEST_USER_PASS,
        };
        let resp = test_request_get_response(
            &app,
            axum::http::Method::POST,
            "/login-password",
            None,
            None,
            None,
            None,
            Some(&login_pw_form),
        )
        .await;
        let resp = assert_successful_and_return_resp(resp, true).await;
        let cookie = resp.headers().get("Set-Cookie");
        assert!(cookie.is_some(), "No Set-Cookie header found");
        let cookie = cookie
            .and_then(|v| v.to_str().ok())
            .ok_or_else(|| anyhow::anyhow!("Could not get cookie"));
        assert!(cookie.is_ok(), "Could not get cookie");
        let cookie = cookie.expect("checked").to_string();

        let (status, body) = test_request::<()>(
            &app,
            Method::GET,
            "/search",
            None,
            Some(&cookie),
            None,
            None,
            None,
        )
        .await;
        assert_eq_or_print_body(status, 200, &body);
        assert_or_print_body(body.contains("name=\"search-text\""), &body);
        Ok(())
    }

    #[ctb_test(tokio::test)]
    async fn test_post_login_password_route_wrong() -> Result<()> {
        let name = function_name!();
        let _ = lock_by_name(name)?;
        let (_state, app) = test_app();
        #[derive(serde::Serialize)]
        struct LoginPasswordForm<'a> {
            username: &'a str,
            password: &'a str,
        }
        // Make sure the user exists
        get_test_user(name);
        let login_pw_form = LoginPasswordForm {
            username: name,
            password: "WRONG_test_password",
        };
        let resp = test_request_get_response(
            &app,
            axum::http::Method::POST,
            "/login-password",
            None,
            None,
            None,
            None,
            Some(&login_pw_form),
        )
        .await;
        let resp = assert_successful_and_return_resp(resp, false).await;
        let cookie = resp.headers().get("Set-Cookie");
        assert!(cookie.is_none(), "No Set-Cookie header found");
        Ok(())
    }
}
