//! Controller for general app UI pages.

use axum::response::{IntoResponse, Redirect, Result};
use axum::{extract::State, response::Response};
use axum_typed_multipart::TryFromMultipart;
use serde::{Deserialize, Serialize};

use crate::formats::ip::get_regex_ipv4_ipv6_exact;
use crate::io::webui::controllers::base::redirect_temporary;
use crate::io::webui::error::{WebErr, WebError, WebResult};
use crate::io::webui::flexible_form::FlexibleForm;
use crate::io::webui::session_auth::AuthenticatedUser;
use crate::io::webui::{AppState, RequestState, error_403, respond_page};
use crate::storage::pc_settings::{DEFAULT_BIND_TO_IP, DEFAULT_SERVER_URL};
use crate::storage::user::UserPublicInfo;
use crate::utilities::build_info;
use crate::utilities::password::{Password, hash, verify};
use crate::{debug_fmt, json_value};

/// This is *not* the index page; it's the home page after logging in.
pub async fn get_home(
    State(state): State<AppState>,
    req: RequestState,
    user: AuthenticatedUser,
) -> Response {
    respond_page(&state, req, "home", &json_value!({}))
}

/// The first-run setup and public PC settings page. For multi-user systems,
/// public servers, kiosks, etc. this should typically be locked down, but it
/// needs to be available for long enough after install for the administrator to
/// get the computer set up. FIXME: May not exactly be accurate? TBD.
pub async fn get_public_pc_settings(
    State(state): State<AppState>,
    req: RequestState,
) -> WebResult<Response> {
    #[derive(Serialize, Debug)]
    struct UserWithAdmin {
        admin: bool,
        local_id: u64,
        username: String,
        display_name: Option<Vec<u8>>,
        picture: Option<Vec<u8>>,
    }

    let build_info = build_info();
    let current_settings =
        crate::storage::pc_settings::PcSettings::load().unwrap_or_default();

    /*let all_users = crate::storage::user::UserPublicInfo::list_all()
    .web_err(&state, &req)?;*/
    // FIXME IPC is broken, so temporarily disable user listing
    let all_users = Vec::<UserPublicInfo>::new();

    debug_fmt!("All users: {all_users:#?}");

    let users: Vec<UserWithAdmin> = all_users
        .into_iter()
        .map(|u| UserWithAdmin {
            admin: current_settings.admin_users.contains(&u.local_id()),
            local_id: u.local_id(),
            username: u.name().to_string(),
            display_name: u.display_name().map(<[u8]>::to_vec),
            picture: u.user_picture().map(<[u8]>::to_vec),
        })
        .collect();

    debug_fmt!("Rendering PC settings page with users: {users:#?}");

    Ok(respond_page(
        &state,
        req,
        "settings.pc-settings",
        &json_value! ({
            "show_users" => current_settings.show_users,
            "bind_to_ip" => current_settings.bind_to_ip,
            "bind_to_ip_regex" => get_regex_ipv4_ipv6_exact(),
            "server_url" => current_settings.server_url,
            "domain_name" => current_settings.domain_name.unwrap_or_default(),
            "fixed_http_port" => current_settings.fixed_http_port.unwrap_or(0),
            "fixed_https_port" => current_settings.fixed_https_port.unwrap_or(0),
            "tls_certificate" => current_settings.tls_certificate.unwrap_or_default(),
            "kiosk_mode" => current_settings.kiosk_mode,
            "log_stack_file" => current_settings.log_stack_file,

            // Feature flags
            "feature_login" => current_settings.feature_login,
            "feature_registration" => current_settings.feature_registration,

            // Secrets not shown, just whether they are set
            "tls_private_key_set" => current_settings.tls_private_key.is_some(),
            "admin_password_set" => current_settings.admin_password_hash.is_some(),

            // All users, for building list
            "users" => users,

            // Provide build info for display
            "crate_name" => build_info.name,
            "crate_version" => build_info.version,
            "build_date" => build_info.build_date,
            "commit" => build_info.commit
        }),
    ))
}

#[derive(TryFromMultipart, Serialize, Deserialize)]
#[try_from_multipart(strict)]
pub struct PcSettingsForm {
    show_users: Option<bool>,
    bind_to_ip: Option<String>,
    server_url: Option<String>,
    domain_name: String,
    fixed_http_port: u16,
    fixed_https_port: u16,
    http_redirect: Option<bool>,
    tls_certificate: Option<String>,
    tls_private_key: Option<String>,
    admin_users: Vec<u64>,
    admin_password: Option<String>,
    kiosk_mode: Option<bool>,
    log_stack_file: Option<bool>,
    feature_login: Option<bool>,
    feature_registration: Option<bool>,
}

#[allow(clippy::too_many_lines)]
pub async fn post_public_pc_settings(
    State(state): State<AppState>,
    req: RequestState,
    form: FlexibleForm<PcSettingsForm>,
) -> WebResult<Response> {
    let PcSettingsForm {
        show_users,
        bind_to_ip,
        server_url,
        domain_name,
        fixed_http_port,
        fixed_https_port,
        http_redirect,
        tls_certificate,
        tls_private_key,
        admin_users,
        admin_password,
        kiosk_mode,
        log_stack_file,
        feature_login,
        feature_registration,
    } = form.0;

    let admin_password: Option<Password> = admin_password
        .map(|admin_password| Password::from_string(&admin_password));

    let current_settings =
        crate::storage::pc_settings::PcSettings::load().unwrap_or_default();
    // If an admin password is set, require it
    if let Some(ref current_pass_hash) = current_settings.admin_password_hash {
        match admin_password.clone() {
            Some(provided)
                if verify(&provided, current_pass_hash.as_str()).map_err(
                    |e| WebError::new(e, state.clone(), req.clone()),
                )? => {}
            _ => {
                return Ok(error_403(
                    &state,
                    &req,
                    "Invalid admin password".to_string(),
                ));
            }
        }
    }
    // If no admin password is set and one is provided, hash it and save it
    let admin_password_hash =
        if current_settings.admin_password_hash.is_none() {
            if let Some(provided) = admin_password {
                Some(hash(&provided).map_err(|e| {
                    WebError::new(e, state.clone(), req.clone())
                })?)
            } else {
                None
            }
        } else {
            current_settings.admin_password_hash.clone()
        };

    let tls_private_key = if tls_private_key.is_some() {
        tls_private_key
    } else {
        current_settings.tls_private_key.clone()
    };

    let settings = crate::storage::pc_settings::PcSettings {
        show_users: if let Some(show_users) = show_users {
            show_users
        } else {
            current_settings.show_users
        },
        bind_to_ip: if let Some(bind_to_ip) = bind_to_ip {
            if bind_to_ip.is_empty() {
                DEFAULT_BIND_TO_IP.to_string()
            } else {
                bind_to_ip
            }
        } else {
            current_settings.bind_to_ip
        },
        server_url: if let Some(server_url) = server_url {
            if server_url.is_empty() {
                DEFAULT_SERVER_URL.to_string()
            } else {
                server_url
            }
        } else {
            current_settings.server_url
        },
        domain_name: if domain_name.is_empty() {
            None
        } else {
            Some(domain_name)
        },
        fixed_http_port: if fixed_http_port == 0 {
            None
        } else {
            Some(fixed_http_port)
        },
        fixed_https_port: if fixed_https_port == 0 {
            None
        } else {
            Some(fixed_https_port)
        },
        http_redirect: if let Some(http_redirect) = http_redirect {
            http_redirect
        } else {
            current_settings.http_redirect
        },
        tls_certificate,
        tls_private_key,
        admin_users,
        admin_password_hash,
        kiosk_mode: if let Some(kiosk_mode) = kiosk_mode {
            kiosk_mode
        } else {
            current_settings.kiosk_mode
        },
        log_stack_file: if let Some(log_stack_file) = log_stack_file {
            log_stack_file
        } else {
            current_settings.log_stack_file
        },
        feature_login: if let Some(feature_login) = feature_login {
            feature_login
        } else {
            current_settings.feature_login
        },
        feature_registration: if let Some(feature_registration) =
            feature_registration
        {
            feature_registration
        } else {
            current_settings.feature_registration
        },
        ..Default::default()
    };

    settings.save().web_err(&state, &req)?;
    Ok(redirect_temporary(req.is_js_request, "/pc-settings"))
}

#[cfg(test)]
#[allow(clippy::unwrap_in_result, clippy::panic_in_result_fn)]
mod tests {
    use crate::io::webui::test_helpers::{
        assert_body_contains, assert_eq_or_print_body, test_get_no_login,
        test_post_no_login,
    };
    use crate::storage::pc_settings::PcSettings;
    use crate::storage::user::get_test_user;

    #[crate::ctb_test(tokio::test)]
    async fn can_get_pc_settings() {
        let test_user_1 =
            get_test_user(format!("{}_1", function_name!()).as_str());
        let test_user_2 =
            get_test_user(format!("{}_2", function_name!()).as_str());
        let test_user_3 =
            get_test_user(format!("{}_3", function_name!()).as_str());

        // Set known settings
        let settings = PcSettings {
            show_users: true,
            bind_to_ip: "123.456.789".to_string(),
            domain_name: Some("foo.com".to_string()),
            fixed_http_port: Some(1234),
            fixed_https_port: Some(4567),
            admin_users: vec![test_user_1.local_id(), test_user_3.local_id()],
            ..Default::default()
        };
        settings.save().unwrap();

        let (status, body) = test_get_no_login("/pc-settings").await;
        assert_eq_or_print_body(status, 200, &body);
        assert_body_contains(
            "These options apply to all users on this computer",
            &body,
        );
        // Check that the form contains the current values
        assert_body_contains("value=\"foo.com\"", &body);
        assert_body_contains("value=\"1234\"", &body);
        // FIXME unimplemented due to broken IPC
        /*assert_body_contains(
            &format!("\"{}\"  checked", test_user_1.local_id()),
            &body,
        );
        assert_body_not_contains(
            &format!("\"{}\"  checked", test_user_2.local_id()),
            &body,
        );
        assert_body_contains(
            &format!("\"{}\"  checked", test_user_3.local_id()),
            &body,
        );*/
        // assert_body_contains("checked", &body); // show_users checked
    }

    #[crate::ctb_test(tokio::test)]
    async fn can_save_settings_and_read_back() {
        // Prepare form data
        #[derive(serde::Serialize)]
        struct Form {
            show_users: bool,
            bind_to_ip: Option<String>,
            domain_name: Option<String>,
            fixed_http_port: Option<u16>,
            fixed_https_port: Option<u16>,
            admin_users: Vec<u64>,
            feature_login: bool,
        }
        let form = Form {
            show_users: false,
            bind_to_ip: Some("127.0.0.1".to_string()),
            domain_name: Some("example.com".to_string()),
            fixed_http_port: Some(5678),
            fixed_https_port: Some(5679),
            admin_users: vec![18, 19],
            feature_login: true,
        };

        let (status, body) =
            test_post_no_login("/pc-settings", None, Some(&form), None).await;
        assert_eq_or_print_body(status, 303, &body); // Should redirect

        // Now load settings and check values
        let loaded = PcSettings::load().unwrap();
        assert!(!loaded.show_users);
        assert_eq!(loaded.bind_to_ip, "127.0.0.1".to_string());
        assert_eq!(loaded.domain_name, Some("example.com".to_string()));
        assert_eq!(loaded.fixed_http_port, Some(5678));
        assert_eq!(loaded.fixed_https_port, Some(5679));
        assert_eq!(loaded.admin_users, vec![18, 19]);
        assert!(loaded.feature_login);
    }
}
