use std::collections::HashMap;
use std::net::{Ipv4Addr, SocketAddr};
use std::sync::Arc;

use anyhow::{Context, Result};
use axum::Router;
use axum::extract::FromRequestParts;
use axum::response::{Html, IntoResponse, Response};
use axum_server::tls_rustls::RustlsConfig;
use handlebars::Handlebars;
use http::StatusCode;
use maplit::btreemap;
use portpicker::pick_unused_port;
use serde::{Deserialize, Serialize};
use serde_json::{Value, json, to_value};
use std::time::Duration;
use tokio::sync::Mutex;
use tower_http::compression::CompressionLayer;
use tower_http::cors::CorsLayer;

use crate::formats::markdown::markdown2html;
use crate::io::webui::access_log_layer::AccessLogLayer;
use crate::io::webui::routes::build_routes;
use crate::io::webui::session_auth::{AuthenticatedUser, Session, SharedUser};
use crate::json_value;
use crate::storage::{get_asset, register_views};
use crate::utilities::serde_value::insert_key;
use crate::utilities::*;

pub mod access_log_layer;
pub mod error;
pub mod flexible_form;
pub mod routes;
pub mod session_auth;
pub mod test_helpers;
pub mod webview;
pub mod controllers {
    pub mod app;
    pub mod auth;
    pub mod base;
    pub mod graph;
    pub mod search;
    pub mod web;
}

// Shared application state
#[derive(Clone)]
pub struct AppState {
    hbs: Arc<Handlebars<'static>>,
    sessions: Arc<Mutex<HashMap<Vec<u8>, Session>>>,
    sessions_by_user: Arc<Mutex<HashMap<u64, Vec<Vec<u8>>>>>,
    users: Arc<Mutex<HashMap<u64, SharedUser>>>,
}

impl Default for AppState {
    fn default() -> Self {
        let hbs = register_views();
        Self {
            hbs: Arc::new(hbs),
            sessions: Arc::new(Mutex::new(HashMap::new())),
            users: Arc::new(Mutex::new(HashMap::new())),
            sessions_by_user: Arc::new(Mutex::new(HashMap::new())),
        }
    }
}

/// Trait for types that can serve as a context for a Handlebars view.
pub trait ViewContext: Serialize {
    /// Used to add or override keys for layouts.
    fn with_content(self, content: String) -> serde_json::Value
    where
        Self: Sized,
    {
        // Compose self and content into a merged JSON object.
        let mut map = serde_json::to_value(self)
            .expect("context to be serializable")
            .as_object()
            .cloned()
            .unwrap_or_default();
        map.insert("content".to_string(), Value::String(content));
        Value::Object(map)
    }
}

// Blanket impl for all Serialize types
impl<T: Serialize> ViewContext for T {}

// --- 2. Example context structs ---

#[derive(Serialize)]
struct ErrorContext {
    message: String,
    message_details: String,
}

pub fn start_webui_server() -> u16 {
    log!("Starting local web UI server");
    let current_settings =
        crate::storage::pc_settings::PcSettings::load().unwrap_or_default();
    let protocol: String =
        if let Some(ref _cert) = current_settings.tls_certificate {
            log!("Using HTTPS");
            "https".to_string()
        } else {
            log!("Using HTTP");
            "http".to_string()
        };
    let relevant_port = if protocol == "http" {
        current_settings.fixed_http_port
    } else {
        current_settings.fixed_https_port
    };
    let port: u16 = if let Some(port) = relevant_port {
        log!("Using fixed port from settings: {}", port);
        port
    } else {
        pick_unused_port().expect("No ports free")
    };
    let bind_to_ip: String = current_settings.bind_to_ip;
    log!("Using server address: {}", bind_to_ip.clone());
    let domain: String = if let Some(domain) = current_settings.domain_name {
        log!("Using configured domain name: {}", domain.clone());
        domain
    } else {
        bind_to_ip.to_string()
    };
    let protocol_clone = protocol.clone();
    let bind_to_ip_clone = bind_to_ip.clone();
    let domain_clone = domain.clone();
    let tls_certificate = current_settings.tls_certificate.clone();
    let tls_private_key = current_settings.tls_private_key.clone();
    std::thread::spawn(move || {
        if let Err(e) = start_webui_server_inner(
            port,
            protocol_clone,
            bind_to_ip_clone,
            Some(domain_clone),
            tls_certificate,
            tls_private_key,
        ) {
            log!(format!("Web UI server failed to start: {e:?}"));
        }
    });

    // If using HTTPS and not on port 80, also start up a HTTP->HTTPS redirector
    if protocol == "https" && port != 80 && current_settings.http_redirect {
        let redirect_from_port = 80;
        let bind_to_ip_clone = bind_to_ip.clone();
        std::thread::spawn(move || {
            // Check if we can bind to port 80 on the given IP
            let can_bind = bind_to_ip_clone
                .parse::<Ipv4Addr>()
                .ok()
                .and_then(|ip| {
                    std::net::TcpListener::bind((ip, redirect_from_port)).ok()
                })
                .is_some();
            if !can_bind {
                log!(
                    "Cannot bind to port 80 for HTTP->HTTPS redirector, skipping"
                );
                return;
            }

            let rt = match tokio::runtime::Builder::new_current_thread()
                .enable_all()
                .thread_name("localwebui-redirect")
                .build()
            {
                Ok(rt) => rt,
                Err(e) => {
                    log!(format!(
                        "Failed building redirector tokio runtime: {e:?}"
                    ));
                    return;
                }
            };

            let result = rt.block_on(http_to_https(
                bind_to_ip_clone,
                redirect_from_port,
                Some(port),
            ));
            if let Err(e) = result {
                log!(format!("HTTP->HTTPS redirector failed: {e:?}"));
            }
        });
    }

    let url = format!("{protocol}://{domain}:{port}");
    let result = webbrowser::open(url.as_str());
    if let Err(e) = result {
        log!(format!("Failed to open web browser automatically: {e:?}"));
        log!(format!(
            "Please open your web browser and navigate to {url}"
        ));
    } else {
        log!(format!("Web browser opened to {url}"));
    }

    port
}

async fn http_to_https(
    bind_to_ip: String,
    redirect_from_port: u16,
    relevant_port: Option<u16>,
) -> Result<()> {
    let ip = bind_to_ip.parse::<Ipv4Addr>().with_context(|| {
        format!("Could not parse bind IP address: {bind_to_ip}")
    })?;
    let addr = SocketAddr::from((ip, redirect_from_port));

    axum_server::bind(addr)
        .serve(
            Router::new()
                .fallback(axum::routing::any(
                    move |req: axum::http::Request<_>| async move {
                        let host = req
                            .headers()
                            .get("host")
                            .and_then(|h| h.to_str().ok())
                            .unwrap_or("");
                        let uri = req.uri().to_string();
                        let redirect_to_port = if let Some(relevant_port) =
                            relevant_port
                            && relevant_port != 443
                        {
                            format!(":{}", relevant_port)
                        } else {
                            "".to_string()
                        };
                        let redirect_url = format!(
                            "https://{}{}{}",
                            host, redirect_to_port, uri
                        );
                        axum::response::Redirect::permanent(&redirect_url)
                    },
                ))
                .into_make_service(),
        )
        .await
        .context("Error in HTTP to HTTPS redirector")?;

    Ok(())
}

const SLOW_TTFB_THRESHOLD: Duration = Duration::from_millis(150);

pub fn build_app_router(state: AppState) -> Router {
    build_routes(state)
        .layer(AccessLogLayer::new(SLOW_TTFB_THRESHOLD))
        .layer(CompressionLayer::new())
        .layer(CorsLayer::permissive())
}

fn start_webui_server_inner(
    port: u16,
    protocol: String,
    bind_to_ip: String,
    domain_name: Option<String>,
    tls_certificate: Option<String>,
    tls_private_key: Option<String>,
) -> Result<()> {
    // Build templates once and share via state
    let hbs = register_views();
    let state = AppState {
        hbs: Arc::new(hbs),
        sessions: Arc::new(Mutex::new(HashMap::new())),
        users: Arc::new(Mutex::new(HashMap::new())),
        sessions_by_user: Arc::new(Mutex::new(HashMap::new())),
    };

    let app = build_app_router(state);

    // Run on a dedicated runtime in this thread
    let rt = tokio::runtime::Builder::new_multi_thread()
        .enable_all()
        .thread_name("localwebui-axum")
        .build()
        .context("failed building tokio runtime")?;

    rt.block_on(async move {
        let ip = bind_to_ip.parse::<Ipv4Addr>().with_context(|| {
            format!("Could not parse bind IP address: {bind_to_ip}")
        })?;
        let addr = SocketAddr::from((ip, port));

        // NOTE: into_make_service_with_connect_info enables the ConnectInfo<SocketAddr> extraction
        let make_service =
            app.into_make_service_with_connect_info::<SocketAddr>();

        if protocol == "http" {
            axum_server::bind(addr)
                .serve(make_service)
                .await
                .context("HTTP server exited with error")?;
            return Ok(());
        }

        let cert_vec = tls_certificate
            .context("TLS certificate not provided, cannot start HTTPS server")?
            .into_bytes();
        let key_vec = tls_private_key
            .context("TLS private key not provided, cannot start HTTPS server")?
            .into_bytes();

        let config = RustlsConfig::from_pem(cert_vec, key_vec)
            .await
            .context("Failed to build RustlsConfig from PEM")?;

        axum_server::bind_rustls(addr, config)
            .serve(make_service)
            .await
            .context("HTTPS server exited with error")?;

        Ok(())
    })
}

#[derive(Serialize, Clone)]
pub struct RequestState {
    route: String,
    method: String,
    accept: Option<String>,
    is_js_request: bool,
}

impl<S> FromRequestParts<S> for RequestState
where
    S: Send + Sync,
{
    type Rejection = StatusCode;
    async fn from_request_parts(
        parts: &mut axum::http::request::Parts,
        _state: &S,
    ) -> Result<Self, Self::Rejection> {
        Ok(RequestState {
            route: parts.uri.path().to_string(),
            method: parts.method.to_string(),
            accept: parts
                .headers
                .get(axum::http::header::ACCEPT)
                .map(|v| v.to_str().unwrap().to_string()),
            is_js_request: parts
            .headers
            .get("X-CollectiveToolbox-IsJsRequest")
            .and_then(|v| v.to_str().ok())
            .map(|s| s.eq_ignore_ascii_case("true"))
            .unwrap_or(false)
        })
    }
}

#[derive(Deserialize)]
/// The `PageQuery` struct is meant for extracting query parameters from the request URL, specifically a ?page=N parameter. For example, /nodes?page=2.
pub struct PageQuery {
    page: Option<String>,
}

// ================ Render helpers ================

fn respond_general<T: serde::Serialize>(
    state: &AppState,
    req: RequestState,
    view: &str,
    data: &T,
) -> Response {
    match render_page(&state.hbs, None, view.to_string(), &req, data) {
        Ok(html) => Html(html).into_response(),
        Err(e) => error_400(state, &req, e),
    }
}

fn respond_page<T: serde::Serialize>(
    state: &AppState,
    req: RequestState,
    view: &str,
    data: &T,
) -> Response {
    match render_page(&state.hbs, Some("page"), view.to_string(), &req, data) {
        Ok(html) => Html(html).into_response(),
        Err(e) => error_400(state, &req, e),
    }
}

fn respond_markdown_page(
    state: &AppState,
    req: RequestState,
    view: &str,
) -> Response {
    let md = get_asset(format!("views/pages/{view}.md").as_str());

    if let Some(md) = md {
        let page = markdown2html(md);

        return match render_page(
            &state.hbs,
            Some("page"),
            "pages.markdown".to_string(),
            &req,
            &json_value!({ "page" => String::from_utf8_lossy(&page).to_string() }),
        ) {
            Ok(html) => Html(html).into_response(),
            Err(e) => error_400(state, &req, e),
        };
    } else {
        return error_404(
            state,
            &req,
            format!("Markdown page not found: {}", view),
        );
    }
}

fn respond_dialog<T: serde::Serialize>(
    state: &AppState,
    req: RequestState,
    view: &str,
    data: &T,
) -> Response {
    match render_page(
        &state.hbs,
        Some("dialog"),
        format!("dialogs.{view}"),
        &req,
        data,
    ) {
        Ok(html) => Html(html).into_response(),
        Err(e) => error_400(state, &req, e),
    }
}

// ================ Error helpers ================

fn error_500<E: std::fmt::Debug + std::fmt::Display>(
    state: &AppState,
    req: &RequestState,
    e: E,
) -> Response {
    error_response(state, req, e, StatusCode::INTERNAL_SERVER_ERROR)
}

fn error_400<E: std::fmt::Debug + std::fmt::Display>(
    state: &AppState,
    req: &RequestState,
    e: E,
) -> Response {
    error_response(state, req, e, StatusCode::BAD_REQUEST)
}

fn error_401<E: std::fmt::Debug + std::fmt::Display>(
    state: &AppState,
    req: &RequestState,
    e: E,
) -> Response {
    error_response(state, req, e, StatusCode::UNAUTHORIZED)
}

fn error_403<E: std::fmt::Debug + std::fmt::Display>(
    state: &AppState,
    req: &RequestState,
    e: E,
) -> Response {
    error_response(state, req, e, StatusCode::FORBIDDEN)
}

fn error_404<E: std::fmt::Debug + std::fmt::Display>(
    state: &AppState,
    req: &RequestState,
    e: E,
) -> Response {
    error_response(state, req, e, StatusCode::NOT_FOUND)
}

fn recoverable_error<E: std::fmt::Display>(
    state: &AppState,
    req: RequestState,
    e: E,
) -> Response {
    // FIXME: Use JS to intercept this (if JS is running) and show a modal dialog instead of a full page
    let mut response = respond_page(
        state,
        req,
        "layouts._recoverable-error",
        &btreemap! { "recoverable_error_message".to_string() =>  e.to_string()},
    );
    let status = response.status_mut();
    *status = StatusCode::BAD_REQUEST;
    response
}

fn error_response<E: std::fmt::Debug + std::fmt::Display>(
    state: &AppState,
    req: &RequestState,
    e: E,
    status_code: StatusCode,
) -> Response {
    let accept = req.accept.clone();

    let (message, details) = {
        let message = e.to_string();
        let details = format!("{e:?}");
        (message, details)
    };

    if let Some(accept) = accept
        && accept.contains("application/json")
    {
        // Return JSON error
        return error_response_json_with_details(
            message.clone(),
            details.clone(),
            status_code,
        );
    }

    // Default or for "text/html"
    match render_page(
        &state.hbs,
        Some("page"),
        "error".to_string(),
        req,
        &ErrorContext {
            message: message.clone(),
            message_details: format!(
                "{message}\nHTTP Status: {status_code}\ndetails:\n{details}"
            ),
        },
    ) {
        Ok(html) => {
            let mut resp = Html(html).into_response();
            *resp.status_mut() = status_code;
            resp
        }
        Err(e) => error_response_json_with_details(
            format!("Error rendering error response {e:?}"),
            details,
            status_code,
        ),
    }
}

/// Returns a JSON error response including message and details.
fn error_response_json_with_details<E: std::fmt::Display>(
    message: E,
    details: String,
    status_code: StatusCode,
) -> Response {
    let body = json!({
        "type": "error",
        "message": message.to_string(),
        "message_details": details,
    });
    (status_code, axum::Json(body)).into_response()
}

// ================ Template rendering ================

fn render_view<T: serde::Serialize>(
    hbs: &Handlebars<'_>,
    view: String,
    req: &RequestState,
    data: &T,
) -> Result<String> {
    hbs_render(hbs, &view, req, data).context("Could not render view")
}

fn render_page<T: serde::Serialize>(
    hbs: &Handlebars<'_>,
    layout: Option<&str>,
    view: String,
    req: &RequestState,
    data: &T,
) -> Result<String> {
    let view_rendered = hbs_render(hbs, view.as_str(), req, data)?;

    let layout_rendered = if let Some(layout) = layout {
        hbs_render(
            hbs,
            format!("layouts.{layout}").as_str(),
            req,
            &data.with_content(view_rendered),
        )
    } else {
        Ok(view_rendered)
    }?;

    hbs_render(hbs, "layouts.app", req, &data.with_content(layout_rendered))
}

fn hbs_render<T: serde::Serialize>(
    hbs: &Handlebars<'_>,
    view: &str,
    req: &RequestState,
    data: &T,
) -> Result<String> {
    // Convert data to a HashMap
    let req_value = to_value(req).context("Could not serialize request")?;
    insert_key(data, "_request", req_value);

    hbs.render(view, &data).map_err(|e| {
        anyhow::anyhow!(
            "Could not render template: {}\n\
             Template: {:?}\n\
             Line: {:?}\n\
             Column: {:?}\n\
             Reason: {}",
            view,
            e.template_name,
            e.line_no,
            e.column_no,
            e.reason(),
        )
    })
}

// Macros for user and graph access

#[macro_export]
macro_rules! get_user {
    ($shared_user:expr, $req:expr, $user:ident) => {
        let $user = $shared_user.blocking_lock(); // keep guard in scope
    };
}

#[macro_export]
macro_rules! get_user_and_graph {
    ($state:expr, $req:expr, $shared_user:expr, $graph_id:expr, $user:ident, $graph:ident) => {
        let $user = $shared_user.user.blocking_lock(); // keep guard in scope
        let graph = $user.get_graph_by_id($graph_id);
        if (graph.is_none()) {
            return error_400($state, $req, "Graph not found");
        }
        let $graph = graph.unwrap();
        if !$graph.is_writable_by(&*$user) {
            return error_403($state, $req, "User can't write to graph");
        }
    };
}

#[cfg(test)]
#[allow(clippy::unwrap_in_result, clippy::panic_in_result_fn)]
mod tests {
    use super::*;

    #[crate::ctb_test]
    fn test_is_send_and_sync() {
        fn is_send_and_sync<T: Send + Sync>() {}
        is_send_and_sync::<AppState>();
    }
}
