use anyhow::{Result, anyhow};
use clap::{Parser, Subcommand};
use futures::StreamExt;
use std::path::PathBuf;

use crate::cli::base_conversion::{BaseArgs, run_base_convert};
use crate::cli::{StringInput, ToolResult, generate_help_bytes};
use crate::utilities::{
    fork, get_this_executable, upgrade_in_place,
    wait_for_ctoolbox_exit_and_clean_up,
};

/// Return true if this is a command that can be run without booting.
pub fn is_lightweight_command(command: String) -> bool {
    matches!(
        String::as_str(&command),
        "base2base"
            | "hex2dec"
            | "dec2hex"
            | "hexfmt"
            | "gdb_instructions_generate"
            | "help"
    )
}

#[derive(Subcommand, Debug)]
pub enum Command {
    /// Wait for the provided PID to shutdown, then clean up
    #[command(name = "waitshutdown")]
    WaitShutdown {
        /// Process ID to wait for
        #[arg(long)]
        pid: u32,
    },
    /// Wait for the provided PID to shutdown, then start ctoolbox with the
    /// given port
    #[command(name = "waitrestart")]
    WaitRestart {
        /// Process ID to wait for
        #[arg(long)]
        pid: u32,
        /// Port to pass to the new ctoolbox instance
        #[arg(long)]
        port: u16,
    },
    /// Wait for the provided PID to shutdown, then upgrade the ctoolbox
    /// instance in place, then restart it. (Will need to copy the executable
    /// before upgrading it probably because Windows locks running executables.)
    #[command(name = "waitupgrade")]
    WaitUpgrade {
        /// Process ID to wait for
        #[arg(long)]
        pid: u32,
        /// Path to the temporary file holding the new ctoolbox executable
        #[arg(long)]
        temp_path: PathBuf,
        /// Path to the installed ctoolbox executable
        #[arg(long)]
        target_path: PathBuf,
        /// Port to pass to the new ctoolbox instance
        #[arg(long)]
        port: u16,
    },
    /// Convert from one base to another (for base <= 36)
    #[command(name = "base2base")]
    Base2Base {
        /// All positional arguments for custom parsing
        #[arg(required = true)]
        args: Vec<String>,
        #[command(flatten)]
        base_args: BaseArgs,
    },
    /// Convert from hexadecimal to decimal
    #[command(name = "hex2dec")]
    Hex2Dec {
        #[command(flatten)]
        string_input: StringInput,
        #[command(flatten)]
        base_args: BaseArgs,
    },
    /// Convert from decimal to hexadecimal
    #[command(name = "dec2hex")]
    Dec2Hex {
        #[command(flatten)]
        string_input: StringInput,
        #[command(flatten)]
        base_args: BaseArgs,
    },
    /// Reformat hexdumps
    #[command(name = "hexfmt")]
    Hexfmt {
        #[command(flatten)]
        string_input: StringInput,
        #[command(flatten)]
        base_args: BaseArgs,
    },
    /// Generate GDB instructions from symbols
    #[command(name = "gdb_instructions_generate")]
    GdbInstructionsGenerate {},
    /// Show help (lightweight)
    Help,
    /// Unimplemented, just an example
    ShowNode {
        /// Example parameter
        #[arg(short, long)]
        id: i128,
    },
}

// ---------------------------
// Command Execution
// ---------------------------

pub async fn run_lightweight_command(cmd: &Command) -> Result<ToolResult> {
    match cmd {
        Command::WaitShutdown { pid } => {
            wait_for_ctoolbox_exit_and_clean_up(*pid);
            Ok(ToolResult::immediate_ok(Vec::new()))
        }
        Command::WaitRestart { pid, port } => {
            wait_for_ctoolbox_exit_and_clean_up(*pid);
            fork(
                &get_this_executable(),
                vec!["--ctoolbox-ipc-port", &port.to_string().as_str()],
            );
            Ok(ToolResult::immediate_ok(Vec::new()))
        }
        Command::WaitUpgrade {
            pid,
            temp_path,
            target_path,
            port,
        } => {
            wait_for_ctoolbox_exit_and_clean_up(*pid);
            upgrade_in_place(temp_path, target_path)?;
            fork(
                target_path,
                vec!["--ctoolbox-ipc-port", &port.to_string().as_str()],
            );
            Ok(ToolResult::immediate_ok(Vec::new()))
        }
        Command::Base2Base { args, base_args } => {
            let (input, from_base, to_base) = match args.len() {
                1 => (&args[0], None, None),
                3 => {
                    let from_base = Some(args[0].parse::<u8>()?);
                    let to_base = Some(args[1].parse::<u8>()?);
                    let input = &args[2];
                    (input, from_base, to_base)
                }
                _ => {
                    eprintln!(
                        "Invalid arguments! Usage: base2base [FROM_BASE TO_BASE INPUT] or [INPUT]"
                    );
                    std::process::exit(1);
                }
            };

            run_base_convert(
                &from_base,
                &to_base,
                &StringInput {
                    input: input.clone(),
                },
                base_args,
            )
        }
        Command::Hex2Dec {
            string_input,
            base_args,
        } => run_base_convert(&Some(16), &Some(10), string_input, base_args),
        Command::Dec2Hex {
            string_input,
            base_args,
        } => run_base_convert(&Some(10), &Some(16), string_input, base_args),
        Command::Hexfmt {
            string_input,
            base_args,
        } => run_base_convert(&Some(16), &Some(16), string_input, base_args),
        Command::GdbInstructionsGenerate {} => {
            // FIXME: Use a streaming ToolResult here
            crate::utilities::debug_tools::generate_gdb_instructions_streaming(
            )?;
            Ok(ToolResult::immediate_ok(Vec::new()))
        }
        Command::Help => Ok(ToolResult::immediate_ok(generate_help_bytes())),
        Command::ShowNode { .. } => Err(anyhow!(
            "ShowNode should not be run as lightweight command; it needs the full environment"
        )),
    }
}
