use anyhow::{Result, anyhow, bail};
use clap::Parser;
use futures::{Stream, StreamExt};
use std::env;
use std::pin::Pin;
use std::str::FromStr;

use crate::cli::routing::{
    Command, is_lightweight_command, run_lightweight_command,
};
use crate::storage::get_help_for_tty;
use crate::utilities::ipc::IPC_ARG;

pub mod base_conversion;
pub mod routing;

// -----------------------------------------------------------------------------
// Invocation Enum – top-level discrimination between subprocess & user CLI
// -----------------------------------------------------------------------------

#[derive(Debug)]
pub enum Invocation {
    Subprocess(SubprocessArgs),
    User(Cli),
}

impl Default for Invocation {
    fn default() -> Self {
        Invocation::User(Cli {
            ctoolbox_ipc_port: None,
            command: None,
        })
    }
}

impl Invocation {
    pub fn is_subprocess(&self) -> bool {
        matches!(self, Invocation::Subprocess(_))
    }

    pub fn subprocess(&self) -> Option<&SubprocessArgs> {
        if let Invocation::Subprocess(s) = self {
            Some(s)
        } else {
            None
        }
    }

    pub fn expect_cli(&self) -> &Cli {
        match self {
            Invocation::User(cli) => cli,
            Invocation::Subprocess(_) => {
                panic!("Called expect_cli() on a subprocess invocation")
            }
        }
    }
}

// -----------------------------------------------------------------------------
// Subprocess Argument Structures
// -----------------------------------------------------------------------------

#[derive(Debug, Clone)]
pub struct SubprocessArgs {
    pub ipc: IpcEndpoint,
    pub subprocess_index: u32,
    pub service_name: String,
    pub extra: Vec<String>,
}

#[derive(Debug, Clone)]
pub struct IpcEndpoint {
    pub port: u16,
    pub identity: String,
}

impl FromStr for IpcEndpoint {
    type Err = anyhow::Error;

    fn from_str(s: &str) -> Result<Self> {
        // Expect "<port>:<identity>"
        let mut parts = s.splitn(2, ':');
        let port_part = parts
            .next()
            .ok_or_else(|| anyhow!("Missing port in IPC specification"))?;
        let identity_part = parts
            .next()
            .ok_or_else(|| anyhow!("Missing identity in IPC specification"))?;
        let port: u16 = port_part
            .parse()
            .map_err(|e| anyhow!("Invalid port '{port_part}': {e}"))?;
        Ok(IpcEndpoint {
            port,
            identity: identity_part.to_string(),
        })
    }
}

// Manual parsing of subprocess arguments (since the magic token won't play
// nicely as a Clap subcommand).
fn parse_subprocess(raw: &[String]) -> Result<SubprocessArgs> {
    // raw[0] is program name
    // raw[1] == SUBPROCESS_MAGIC
    if raw.len() < 5 {
        bail!(
            "Subprocess invocation requires at least 4 arguments after program name: \
             <magic> <port:identity> <index> <type> [extra..]"
        );
    }
    let ipc = IpcEndpoint::from_str(&raw[2])?;
    let subprocess_index: u32 = raw[3]
        .parse()
        .map_err(|e| anyhow!("Invalid subprocess index '{}': {}", raw[3], e))?;
    let service_name = raw[4].clone();
    let extra = if raw.len() > 5 {
        raw[5..].to_vec()
    } else {
        Vec::new()
    };
    Ok(SubprocessArgs {
        ipc,
        subprocess_index,
        service_name,
        extra,
    })
}

// Public parsing entry point used by lib::entry().
pub fn parse_invocation() -> Result<Invocation> {
    let raw: Vec<String> = env::args().collect();
    if raw.get(1).is_some_and(|s| s == IPC_ARG) {
        let sub = parse_subprocess(&raw)?;
        return Ok(Invocation::Subprocess(sub));
    }
    // Fallback: user CLI
    let cli = Cli::parse(); // Clap handles errors & help display
    Ok(Invocation::User(cli))
}

// -----------------------------------------------------------------------------
// Regular (human CLI use or desktop app main process) CLI definition
// -----------------------------------------------------------------------------

#[derive(Parser, Debug)]
#[command(
    name = "ctoolbox",
    version,
    about = "Collective Toolbox",
    disable_help_subcommand = true
)]
pub struct Cli {
    #[arg(long)]
    pub ctoolbox_ipc_port: Option<u16>,

    #[command(subcommand)]
    pub command: Option<Command>,
}

// ---------------------------
// Lightweight Execution Gate
// ---------------------------

// Decides if we exit early.
// Returns Ok(Some(exit_code)) if a lightweight command was executed.
// Returns Ok(None) if we should proceed to heavy boot.
// Errors bubble up as Err(...).
pub async fn maybe_run_lightweight(cli: &Cli) -> Result<Option<i32>> {
    let Some(cmd) = &cli.command else {
        return Ok(None); // no command => proceed to full app
    };

    let args: Vec<String> = std::env::args().collect();
    let first = args.get(1);
    if first.is_some() && !is_lightweight_command(first.unwrap().to_string()) {
        return Ok(None);
    }

    let result = run_lightweight_command(cmd).await?;
    let exit_code = dispatch_tool_result(result).await?;
    Ok(Some(exit_code))
}

// ---------------------------
// Tool Result Abstractions
// ---------------------------

pub enum ToolResult {
    // Immediate, single-buffer outputs
    Immediate {
        stdout: Vec<u8>,
        stderr: Vec<u8>,
        exit_code: i32,
    },
    // Streaming output (future extensibility)
    Streaming {
        stream: Pin<Box<dyn Stream<Item = OutputChunk> + Send>>,
        exit_code: i32,
    },
}

pub enum OutputChunk {
    Stdout(Vec<u8>),
    Stderr(Vec<u8>),
}

impl ToolResult {
    pub fn immediate_ok(stdout: Vec<u8>) -> Self {
        ToolResult::Immediate {
            stdout,
            stderr: Vec::new(),
            exit_code: 0,
        }
    }
    pub fn immediate_err(stderr: Vec<u8>, code: i32) -> Self {
        ToolResult::Immediate {
            stdout: Vec::new(),
            stderr,
            exit_code: code,
        }
    }
}

// Central dispatcher for writing a ToolResult to real stdio.
async fn dispatch_tool_result(result: ToolResult) -> Result<i32> {
    use std::io::{Write, stderr, stdout};

    match result {
        ToolResult::Immediate {
            stdout: out,
            stderr: err,
            exit_code,
        } => {
            let mut so = stdout().lock();
            let mut se = stderr().lock();
            if !out.is_empty() {
                so.write_all(&out)?;
            }
            if !err.is_empty() {
                se.write_all(&err)?;
            }
            Ok(exit_code)
        }
        ToolResult::Streaming {
            mut stream,
            exit_code,
        } => {
            let mut so = stdout().lock();
            let mut se = stderr().lock();
            while let Some(chunk) = stream.next().await {
                match chunk {
                    OutputChunk::Stdout(d) => so.write_all(&d)?,
                    OutputChunk::Stderr(d) => se.write_all(&d)?,
                }
            }
            Ok(exit_code)
        }
    }
}

// ---------------------------
// Shared Arg Structures
// ---------------------------

#[derive(clap::Args, Debug)]
pub struct StringInput {
    /// Input number or string
    pub input: String,
}

// Utilities

fn generate_help_bytes() -> Vec<u8> {
    // Could introspect Clap auto-generated help if desired:
    // let mut cmd = Cli::command();
    // let mut buf = Vec::new();
    // cmd.write_help(&mut buf).unwrap();
    // buf
    get_help_for_tty(get_width())
}

/// Return the width of the terminal
pub fn get_width() -> u16 {
    termsize::get().map_or(80, |s| s.cols)
}

#[cfg(test)]
mod tests {
    use super::*;

    #[crate::ctb_test]
    fn test_get_help_bytes() {
        let help_bytes = generate_help_bytes();
        assert!(String::from_utf8_lossy(&help_bytes).contains("## Synopsis"));
    }
}
