use anyhow::Result;
use std::path::Path;
use std::process::Command;
use std::{env, fs};
use vergen_gix::{
    BuildBuilder, CargoBuilder, Emitter, GixBuilder, RustcBuilder,
    SysinfoBuilder,
};

/// Recursively copies a directory from `src` to `dst`.
fn copy_dir_recursive(src: &Path, dst: &Path) -> Result<()> {
    if src.is_dir() {
        fs::create_dir_all(dst)?;
        for entry in fs::read_dir(src)? {
            let entry = entry?;
            let src_path = entry.path();
            let dst_path = dst.join(entry.file_name());
            if src_path.is_dir() {
                copy_dir_recursive(&src_path, &dst_path)?;
            } else {
                fs::copy(&src_path, &dst_path)?;
            }
        }
    } else {
        fs::copy(src, dst)?;
    }
    Ok(())
}

/// Build script to prepare assets and metadata for the project.
fn main() -> Result<()> {
    println!("cargo:rerun-if-changed=../assets");

    let path = env::var("CARGO_MANIFEST_DIR")?;
    let current = Path::new(path.as_str());
    let parent = &current
        .parent()
        .ok_or_else(|| anyhow::anyhow!("No parent directory"))?;
    env::set_current_dir(parent)?;


    println!("Preparing build output directory...");
    if fs::metadata("built").is_ok() {
        fs::remove_dir_all("built")?;
    }
    fs::create_dir("built")?;

    println!("Copying assets...");
    copy_dir_recursive(Path::new("assets"), Path::new("built/assets"))?;

    println!("Creating .keep file in assets...");
    fs::File::create("built/assets/.keep")?;

    println!("Creating src directory...");
    fs::create_dir_all("built/src")?;

    println!("Switching to ctoolbox directory...");
    std::env::set_current_dir("ctoolbox")?;

    println!("Copying base documentation...");
    copy_dir_recursive(
        Path::new("../docs"),
        Path::new("../built/assets/docs"),
    )?;

    println!("Copying Rust documentation if available...");
    if fs::metadata("target/doc")
        .ok()
        .map(|m| m.is_dir())
        .unwrap_or(false)
    {
        copy_dir_recursive(
            Path::new("target/doc"),
            Path::new("../built/assets/docs/rust"),
        )?;
    }

    println!("Returning to project root...");
    std::env::set_current_dir("..")?;

    println!("Copying license files for supported platforms...");
    let platforms = ["linux-x64", "web"];
    for platform in platforms {
        fs::create_dir_all(format!("built/{}", platform))?;
        for entry in fs::read_dir(".")? {
            let entry = entry?;
            let path = entry.path();
            if let Some(name) = path.file_name().and_then(|n| n.to_str())
                && name.starts_with("LICENSE")
            {
                let dest = format!("built/{}/{}", platform, name);
                fs::copy(&path, &dest)?;
            }
        }
    }

    println!("Archiving source code...");
    Command::new("git")
        .args(["archive", "HEAD", "-o", "built/src/src.zip"])
        .status()?;

    std::env::set_current_dir("ctoolbox")?;

    // NOTE: This will output everything, and requires all features enabled.
    // NOTE: See the specific builder documentation for configuration options.
    let build = BuildBuilder::all_build()?;
    let cargo = CargoBuilder::all_cargo()?;
    let gix = GixBuilder::all_git()?;
    let rustc = RustcBuilder::all_rustc()?;
    let si = SysinfoBuilder::all_sysinfo()?;

    Emitter::default()
        .add_instructions(&build)?
        .add_instructions(&cargo)?
        .add_instructions(&gix)?
        .add_instructions(&rustc)?
        .add_instructions(&si)?
        .emit()?;

    Ok(())
}
