const colorThemeStorageKey = "color-theme-setting";

document.addEventListener("DOMContentLoaded", () => {
    function updateScrollbarSize() {
        // Set a --scrollbar css variable on body with the width of the scrollbar
        // FIXME replace this with JS once CSS container style queries are widely supported
        let html = document.documentElement;
        html.classList.add("calculating-scrollbar-size");
        const scrollbarWidth = window.innerWidth - document.querySelector('body').clientWidth;
        document.body.style.setProperty('--scrollbar', `${scrollbarWidth}px`);
        html.classList.remove("calculating-scrollbar-size");
    }
    updateScrollbarSize();
    window.addEventListener("resize", () => {
        updateScrollbarSize();
    });

    // ---- Small DOM helpers -------------------------------------------------
    const $ = (sel, root = document) => root.querySelector(sel);
    const $$ = (sel, root = document) => Array.from(root.querySelectorAll(sel));

    const isModifiedClick = (event /*, link */) => {
        return event.ctrlKey || event.metaKey || event.shiftKey || event.altKey;
    };

    const markLoading = (v) => {
        if (v) document.body.classList.add("content-loading");
        else document.body.classList.remove("content-loading");
    };

    const isCrossOriginUrl = (url) => {
        try {
            return new URL(url, window.location.href).origin !== window.location.origin;
        } catch {
            return false;
        }
    };

    // ---- Theme handling ----------------------------------------------------
    const rememberThemeRadioState = () => {
        // Restore theme selection from sessionStorage
        const savedTheme = sessionStorage.getItem(colorThemeStorageKey);
        if (savedTheme) {
            const radioToCheck = $(
                `input[name="color-theme-setting"][value="${savedTheme}"]`
            );
            if (radioToCheck) radioToCheck.checked = true;
        }
    };

    // ---- Error modal -------------------------------------------------------
    function showError({ message, isJson }) {
        const errorDiv = $(".error-modal");
        const errorText = $(".error-modal-content");
        if (!errorDiv || !errorText) {
            // If the error modal isn't present on this page, just stop loading.
            markLoading(false);
            return;
        }

        errorDiv.classList.add("flex");
        errorDiv.classList.remove("hidden");

        if (isJson) {
            try {
                const obj = JSON.parse(message);
                errorText.textContent = JSON.stringify(obj, null, 2).replaceAll(
                    "\\n",
                    "\\n\n"
                );
            } catch {
                errorText.textContent = message;
            }
        } else {
            errorText.textContent = message;
        }

        markLoading(false);
    }

    // ---- Body swap ---------------------------------------------------------
    function updateBody(htmlString) {
        const parser = new DOMParser();
        const newDoc = parser.parseFromString(htmlString, "text/html");
        document.body.innerHTML = newDoc.body.innerHTML;

        rememberThemeRadioState();
        markLoading(false);

        if (window.location.hash) {
            history.replaceState(
                null,
                "",
                window.location.pathname + window.location.search
            );
        }

        document.body.scrollTo({ top: 0, behavior: "instant" });

        setupSidebarToggle();
        applyExternalLinkAttrs();
    }

    // ---- Fetch helpers -----------------------------------------------------
    async function fetchAndSwap(url, options = {}) {
        markLoading(true);

        try {
            const response = await fetch(url, {
                // ...existing headers behavior...
                headers: {
                    Accept: "text/html;q=0.9,application/json;q=0.8,*/*;q=0.7",
                    "X-CollectiveToolbox-IsJsRequest": "true",
                    ...(options.headers || {}),
                },
                ...options,
            });

            const contentType = response.headers.get("content-type") || "";
            const isJson = contentType.includes("application/json");
            const isJsRedirect = response.headers.get("X-CollectiveToolbox-IsJsRedirect");

            // Handle JSON redirects from JS-aware handlers
            if (isJsRedirect) {
                const text = await response.text();
                try {
                    const json = JSON.parse(text);
                    const redirectUrl = json.url;

                    if (isCrossOriginUrl(redirectUrl)) {
                        markLoading(false);
                        window.open(redirectUrl, "_blank");
                        return;
                    }

                    await fetchAndSwap(redirectUrl);
                    return;
                } catch (e) {
                    showError({
                        message: "Invalid redirect response: " + e.message,
                        isJson: false,
                    });
                    return;
                }
            }

            // If final URL after redirects is cross-origin, open externally.
            const finalUrl = response.url || url;
            if (isCrossOriginUrl(finalUrl)) {
                markLoading(false);
                window.open(finalUrl, "_blank");
                return;
            }

            const text = await response.text();
            if (!response.ok) showError({ message: text, isJson });
            else updateBody(text);
        } catch (err) {
            showError({ message: err.message, isJson: false });
        }
    }

    // ---- Modal iframe ------------------------------------------------------
    const modalState = {
        open: false,
    };

    function getModalElements() {
        const iframe = document.getElementById("modal-content-frame");
        const wrapper = iframe ? iframe.parentElement : null;
        const root = document.getElementById("root");
        return { iframe, wrapper, root };
    }

    function modalOpen(url) {
        const { iframe, wrapper, root } = getModalElements();
        if (!iframe || !wrapper || !root) return;

        iframe.src = url;
        iframe.scrollIntoView({ behavior: "smooth" });

        wrapper.classList.remove("hidden");
        wrapper.classList.add("flex");
        root.style.overflow = "hidden";

        modalState.open = true;
    }

    window.modalContentClose = function () {
        const { iframe, wrapper, root } = getModalElements();
        if (!iframe || !wrapper || !root) return;

        iframe.src = "about:blank";
        wrapper.classList.add("hidden");
        wrapper.classList.remove("flex");
        root.style.overflow = "auto";

        modalState.open = false;
    };

    window.modalContentPopout = function () {
        const { iframe } = getModalElements();
        if (iframe && iframe.src && iframe.src !== "about:blank") {
            window.open(iframe.src, "_blank");
        }
        window.modalContentClose();
    };

    // ---- Sidebar toggle ----------------------------------------------------
    function setupSidebarToggle() {
        const toggle = $$('[href="#sidebar"]')[0];
        const sidebar = document.getElementById("sidebar");
        if (!toggle || !sidebar) return;

        const removeFragment = () => {
            if (window.location.hash === "#sidebar") {
                history.replaceState(
                    null,
                    "",
                    window.location.pathname + window.location.search
                );
            }
        };

        // Reset to "closed" on each body swap (matches previous behavior).
        sidebar.classList.add("sidebar-js-enabled");
        sidebar.style.display = "none";
        sidebar.style.transform = "translateX(110%)";
        sidebar.style.transition = sidebar.style.transition || "transform 200ms ease-in-out";
        sidebar.style.willChange = "transform";
        sidebar.setAttribute("aria-hidden", "true");
        toggle.setAttribute("aria-expanded", "false");
        sidebar.classList.remove("sidebar-open");

        removeFragment();
    }

    function sidebarSetOpen(v) {
        const toggle = $$('[href="#sidebar"]')[0];
        const sidebar = document.getElementById("sidebar");
        if (!toggle || !sidebar) return;

        const open = !!v;
        if (open) {
            sidebar.style.display = "block";
            window.setTimeout(() => {
                sidebar.style.transform = "translateX(0)";
                sidebar.setAttribute("aria-hidden", "false");
                toggle.setAttribute("aria-expanded", "true");
                sidebar.classList.add("sidebar-open");
            }, 10);
        } else {
            sidebar.style.transform = "translateX(110%)";
            sidebar.setAttribute("aria-hidden", "true");
            toggle.setAttribute("aria-expanded", "false");
            sidebar.classList.remove("sidebar-open");
            window.setTimeout(() => {
                sidebar.style.display = "none";
            }, 200);
        }
    }

    // ---- External link attrs ----------------------------------------------
    function applyExternalLinkAttrs() {
        // Make all .external-link open in new tab
        $$(".link-external").forEach((link) => {
            link.setAttribute("target", "_blank");
            link.setAttribute("rel", "noopener noreferrer");
        });
    }

    // ---- Delegated listeners ----------------------------------------------
    // Theme radio changes
    document.addEventListener("change", (event) => {
        const radio = event.target;
        if (!radio || radio.tagName !== "INPUT") return;
        if (radio.name !== "color-theme-setting") return;

        if (radio.checked) {
            sessionStorage.setItem(colorThemeStorageKey, radio.value);
            document.querySelector("html").classList.add(`theme-${radio.value}`);
        }
    });

    // Forms
    document.addEventListener("submit", (event) => {
        const form =
            event.target && event.target.closest ? event.target.closest("form") : null;
        if (!form) return;

        event.preventDefault();

        const formData = new FormData(form);
        const action = form.action || window.location.href;
        const method = (form.method || "GET").toUpperCase();

        fetchAndSwap(action, {
            method,
            body: method === "GET" ? null : formData,
        });
    });

    // Clicks (links + modal + SPA nav)
    document.addEventListener("click", (event) => {
        const target = event.target;

        // Close modal if open and click is outside the iframe.
        if (modalState.open) {
            const { iframe } = getModalElements();
            if (iframe && target && !iframe.contains(target)) {
                window.modalContentClose();
            }
        }

        const link = target && target.closest ? target.closest("a[href]") : null;
        if (!link) return;

        // Smooth scrolly for hash links
        const hrefAttr = link.getAttribute("href") || "";
        if (hrefAttr.startsWith("#")) {
            if (hrefAttr === "#") {
                event.preventDefault();
                document.body.scrollTo({ top: 0, behavior: "smooth" });
                history.pushState(null, "", window.location.pathname + window.location.search);
                return;
            }
            const targetId = hrefAttr.substring(1);
            const targetElement = document.getElementById(targetId);
            if (targetElement) {
                event.preventDefault();
                targetElement.scrollIntoView({ behavior: "smooth" });
                history.pushState(null, "", window.location.pathname + window.location.search);
            }
            return;
        }

        // Download links: default behavior
        if (link.classList.contains("link-download")) return;

        // External links: default behavior (attrs are enforced elsewhere)
        if (link.classList.contains("link-external")) return;

        // Open-in-frame: modal behavior
        if (link.classList.contains("link-open-in-frame")) {
            if (isModifiedClick(event)) return;
            event.preventDefault();
            // Critical: don't let this same click be seen as "outside" modal and
            // immediately close it (which aborts the iframe navigation).
            event.stopPropagation();
            modalOpen(link.href);
            return;
        }

        // Ignore javascript: links
        if (hrefAttr.startsWith("javascript")) return;

        // SPA navigation for internal links; allow explicit target=_blank to bypass.
        if (isModifiedClick(event) || link.target === "_blank") return;

        event.preventDefault();
        fetchAndSwap(link.href);
    });

    // Escape closes modal and sidebar
    document.addEventListener("keydown", (event) => {
        if (event.key !== "Escape") return;

        if (modalState.open) {
            window.modalContentClose();
            return;
        }

        const sidebar = document.getElementById("sidebar");
        const isOpen = sidebar && sidebar.classList.contains("sidebar-open");
        if (isOpen) sidebarSetOpen(false);
    });

    // Sidebar click delegation (toggle + outside-to-close)
    document.addEventListener("click", (event) => {
        const toggle =
            event.target && event.target.closest
                ? event.target.closest('[href="#sidebar"]')
                : null;

        const sidebar = document.getElementById("sidebar");
        if (!sidebar) return;

        const removeFragment = () => {
            if (window.location.hash === "#sidebar") {
                history.replaceState(
                    null,
                    "",
                    window.location.pathname + window.location.search
                );
            }
        };

        if (toggle) {
            removeFragment();
            event.preventDefault();
            const isOpen = sidebar.classList.contains("sidebar-open");
            sidebarSetOpen(!isOpen);
            return;
        }

        const isOpen = sidebar.classList.contains("sidebar-open");
        if (!isOpen) return;

        const hamburger = $$('[href="#sidebar"]')[0];
        if (
            hamburger &&
            !sidebar.contains(event.target) &&
            !hamburger.contains(event.target)
        ) {
            removeFragment();
            sidebarSetOpen(false);
        }
    });

    // ---- Initial page setup ------------------------------------------------
    rememberThemeRadioState();
    applyExternalLinkAttrs();
    setupSidebarToggle();
});