---
applyTo: '**'
---

- Use the `anyhow` crate for error handling, and avoid panics.
- Avoid statefulness where possible.
- Docblocks should wrap at 80 columns. For modules, they should start with `//!`, and for functions, they should start with `///`.
- Leave existing comments and documentation intact when making unrelated edits.
- Don't try to fix errors or compilation failures when applying edits, unless explicitly requested.
- Please leave existing comments in place.
- Instead of unwrap, expect, or similar constructions, prefer returning a Result with anyhow::Error, or other constructions like `let Some(user) = user else { return Err(...); };` to handle errors gracefully.
- Never use `#[test]` or `#[tokio::test]`. Always use `#[crate::ctb_test]` or `#[crate::ctb_test(tokio::test)]` respectively instead.
- Never use `as` for type conversions. Use `try_from` or similar instead.

## Architecture Overview
- Multi-process app: main workspace process spawns subprocesses (renderer, io/webui) via IPC using `interprocess` crate.
- Async runtime: tokio; web server: axum; storage: redb database.
- Data flow: IPC channels for inter-process comms; web UI via HTTP; assets embedded via `include_dir`.
- Key modules: [cli.rs](cli.rs) for parsing, [workspace.rs](workspace.rs) for process mgmt, [io/webui.rs](io/webui.rs) for web interface.

## Critical Workflows
- Build: `./build linux-x64` (runs tests, compiles release binary to `built/linux-x64/`).
- Run: `./run-linux` (builds if needed, executes with backtraces).
- Test: Use `#[crate::ctb_test]` macro; run via `cargo test` in `ctoolbox/` dir.
- Deps: Update with `cargo update`; check licenses with `cargo deny check`; vendor with `cargo vendor`.
- Format/Lint: `cargo fmt`, `cargo clippy --fix`.

## Project-Specific Patterns
- Invocation handling: Parse CLI or subprocess args into `Invocation` enum; subprocesses use magic IPC token.
- Process manager: Start subprocesses with `start_process`.
- Error handling: Always return `Result<T, anyhow::Error>`; no panics or unwraps.
- Logging: Use `tracing` with `setup_logger`; avoid `println!`, use `log!` macro provided by `crate::utilities::*`.
- Assets: Static includes like `PROJECT_DIR` for embedded files; copy to `built/assets/` during build.

## Integration Points
- Web UI: Axum server; serves HTML/JS from `assets/`; opens in browser via `webbrowser`.
- IPC: Old system uses `Channel` struct with port/auth key; messages as JSON via `call_ipc`. Replacing with `interprocess` crate and socket-based IPC.
- Storage: Redb key-value databases with API in `workspace/storage/db.rs`.
- External: matchbox_socket for networking; zeroize for secrets.