// Copyright © 2025 Mikhail Hogrefe
//
// This file is part of Malachite.
//
// Malachite is free software: you can redistribute it and/or modify it under the terms of the GNU
// Lesser General Public License (LGPL) as published by the Free Software Foundation; either version
// 3 of the License, or (at your option) any later version. See <https://www.gnu.org/licenses/>.

use malachite_base::iterators::prefix_to_string;
use malachite_base::num::basic::traits::Zero;
use malachite_nz::natural::Natural;
use malachite_q::Rational;
use malachite_q::exhaustive::exhaustive_rationals_with_denominator_range_to_negative_infinity;
use std::str::FromStr;

fn helper(d: &str, a: &str, out: &str) {
    assert_eq!(
        prefix_to_string(
            exhaustive_rationals_with_denominator_range_to_negative_infinity(
                Natural::from_str(d).unwrap(),
                Rational::from_str(a).unwrap()
            ),
            20
        ),
        out
    );
}

#[test]
fn test_exhaustive_rationals_with_denominator_range_to_negative_infinity() {
    helper(
        "1",
        "0",
        "[0, -1, -2, -3, -4, -5, -6, -7, -8, -9, -10, -11, -12, -13, -14, -15, -16, -17, -18, \
        -19, ...]",
    );
    helper(
        "2",
        "0",
        "[-1/2, -3/2, -5/2, -7/2, -9/2, -11/2, -13/2, -15/2, -17/2, -19/2, -21/2, -23/2, -25/2, \
        -27/2, -29/2, -31/2, -33/2, -35/2, -37/2, -39/2, ...]",
    );
    helper(
        "3",
        "0",
        "[-1/3, -2/3, -4/3, -5/3, -7/3, -8/3, -10/3, -11/3, -13/3, -14/3, -16/3, -17/3, -19/3, \
        -20/3, -22/3, -23/3, -25/3, -26/3, -28/3, -29/3, ...]",
    );
    helper(
        "4",
        "0",
        "[-1/4, -3/4, -5/4, -7/4, -9/4, -11/4, -13/4, -15/4, -17/4, -19/4, -21/4, -23/4, -25/4, \
        -27/4, -29/4, -31/4, -33/4, -35/4, -37/4, -39/4, ...]",
    );
    helper(
        "5",
        "0",
        "[-1/5, -2/5, -3/5, -4/5, -6/5, -7/5, -8/5, -9/5, -11/5, -12/5, -13/5, -14/5, -16/5, \
        -17/5, -18/5, -19/5, -21/5, -22/5, -23/5, -24/5, ...]",
    );
    helper(
        "6",
        "0",
        "[-1/6, -5/6, -7/6, -11/6, -13/6, -17/6, -19/6, -23/6, -25/6, -29/6, -31/6, -35/6, -37/6, \
        -41/6, -43/6, -47/6, -49/6, -53/6, -55/6, -59/6, ...]",
    );

    helper(
        "1",
        "1/2",
        "[0, -1, -2, -3, -4, -5, -6, -7, -8, -9, -10, -11, -12, -13, -14, -15, -16, -17, -18, \
        -19, ...]",
    );
    helper(
        "2",
        "1/2",
        "[1/2, -1/2, -3/2, -5/2, -7/2, -9/2, -11/2, -13/2, -15/2, -17/2, -19/2, -21/2, -23/2, \
        -25/2, -27/2, -29/2, -31/2, -33/2, -35/2, -37/2, ...]",
    );
    helper(
        "3",
        "1/2",
        "[1/3, -1/3, -2/3, -4/3, -5/3, -7/3, -8/3, -10/3, -11/3, -13/3, -14/3, -16/3, -17/3, \
        -19/3, -20/3, -22/3, -23/3, -25/3, -26/3, -28/3, ...]",
    );
    helper(
        "4",
        "1/2",
        "[1/4, -1/4, -3/4, -5/4, -7/4, -9/4, -11/4, -13/4, -15/4, -17/4, -19/4, -21/4, -23/4, \
        -25/4, -27/4, -29/4, -31/4, -33/4, -35/4, -37/4, ...]",
    );
    helper(
        "5",
        "1/2",
        "[1/5, -1/5, 2/5, -2/5, -3/5, -4/5, -6/5, -7/5, -8/5, -9/5, -11/5, -12/5, -13/5, -14/5, \
        -16/5, -17/5, -18/5, -19/5, -21/5, -22/5, ...]",
    );
    helper(
        "6",
        "1/2",
        "[1/6, -1/6, -5/6, -7/6, -11/6, -13/6, -17/6, -19/6, -23/6, -25/6, -29/6, -31/6, -35/6, \
        -37/6, -41/6, -43/6, -47/6, -49/6, -53/6, -55/6, ...]",
    );

    helper(
        "1",
        "245850922/78256779",
        "[0, 1, -1, 2, -2, 3, -3, -4, -5, -6, -7, -8, -9, -10, -11, -12, -13, -14, -15, -16, ...]",
    );
    helper(
        "2",
        "245850922/78256779",
        "[1/2, -1/2, 3/2, -3/2, 5/2, -5/2, -7/2, -9/2, -11/2, -13/2, -15/2, -17/2, -19/2, -21/2, \
        -23/2, -25/2, -27/2, -29/2, -31/2, -33/2, ...]",
    );
    helper(
        "3",
        "245850922/78256779",
        "[1/3, -1/3, 2/3, -2/3, 4/3, -4/3, 5/3, -5/3, 7/3, -7/3, 8/3, -8/3, -10/3, -11/3, -13/3, \
        -14/3, -16/3, -17/3, -19/3, -20/3, ...]",
    );
    helper(
        "4",
        "245850922/78256779",
        "[1/4, -1/4, 3/4, -3/4, 5/4, -5/4, 7/4, -7/4, 9/4, -9/4, 11/4, -11/4, -13/4, -15/4, \
        -17/4, -19/4, -21/4, -23/4, -25/4, -27/4, ...]",
    );
    helper(
        "5",
        "245850922/78256779",
        "[1/5, -1/5, 2/5, -2/5, 3/5, -3/5, 4/5, -4/5, 6/5, -6/5, 7/5, -7/5, 8/5, -8/5, 9/5, -9/5, \
        11/5, -11/5, 12/5, -12/5, ...]",
    );
    helper(
        "6",
        "245850922/78256779",
        "[1/6, -1/6, 5/6, -5/6, 7/6, -7/6, 11/6, -11/6, 13/6, -13/6, 17/6, -17/6, -19/6, -23/6, \
        -25/6, -29/6, -31/6, -35/6, -37/6, -41/6, ...]",
    );

    helper(
        "1",
        "-245850922/78256779",
        "[-4, -5, -6, -7, -8, -9, -10, -11, -12, -13, -14, -15, -16, -17, -18, -19, -20, -21, \
        -22, -23, ...]",
    );
    helper(
        "2",
        "-245850922/78256779",
        "[-7/2, -9/2, -11/2, -13/2, -15/2, -17/2, -19/2, -21/2, -23/2, -25/2, -27/2, -29/2, \
        -31/2, -33/2, -35/2, -37/2, -39/2, -41/2, -43/2, -45/2, ...]",
    );
    helper(
        "3",
        "-245850922/78256779",
        "[-10/3, -11/3, -13/3, -14/3, -16/3, -17/3, -19/3, -20/3, -22/3, -23/3, -25/3, -26/3, \
        -28/3, -29/3, -31/3, -32/3, -34/3, -35/3, -37/3, -38/3, ...]",
    );
    helper(
        "4",
        "-245850922/78256779",
        "[-13/4, -15/4, -17/4, -19/4, -21/4, -23/4, -25/4, -27/4, -29/4, -31/4, -33/4, -35/4, \
        -37/4, -39/4, -41/4, -43/4, -45/4, -47/4, -49/4, -51/4, ...]",
    );
    helper(
        "5",
        "-245850922/78256779",
        "[-16/5, -17/5, -18/5, -19/5, -21/5, -22/5, -23/5, -24/5, -26/5, -27/5, -28/5, -29/5, \
        -31/5, -32/5, -33/5, -34/5, -36/5, -37/5, -38/5, -39/5, ...]",
    );
    helper(
        "6",
        "-245850922/78256779",
        "[-19/6, -23/6, -25/6, -29/6, -31/6, -35/6, -37/6, -41/6, -43/6, -47/6, -49/6, -53/6, \
        -55/6, -59/6, -61/6, -65/6, -67/6, -71/6, -73/6, -77/6, ...]",
    );
}

#[test]
#[should_panic]
fn exhaustive_rationals_with_denominator_range_to_negative_infinity_fail() {
    exhaustive_rationals_with_denominator_range_to_negative_infinity(
        Natural::ZERO,
        Rational::from_unsigneds(1u32, 3),
    );
}
