// Copyright © 2025 Mikhail Hogrefe
//
// This file is part of Malachite.
//
// Malachite is free software: you can redistribute it and/or modify it under the terms of the GNU
// Lesser General Public License (LGPL) as published by the Free Software Foundation; either version
// 3 of the License, or (at your option) any later version. See <https://www.gnu.org/licenses/>.

use itertools::Itertools;
use malachite_base::num::basic::traits::{One, Zero};
use malachite_base::strings::ToDebugString;
use malachite_q::Rational;
use malachite_q::exhaustive::exhaustive_rational_inclusive_range;
use std::str::FromStr;

fn exhaustive_rational_inclusive_range_helper(a: &str, b: &str, values: &str) {
    let a = Rational::from_str(a).unwrap();
    let b = Rational::from_str(b).unwrap();
    let xs = exhaustive_rational_inclusive_range(a, b)
        .take(50)
        .collect_vec()
        .to_debug_string();
    assert_eq!(xs, values);
}

#[test]
fn test_exhaustive_rational_inclusive_range() {
    exhaustive_rational_inclusive_range_helper(
        "0",
        "2",
        "[0, 1/2, 1, 1/3, 2, 3/2, 2/3, 1/6, 4/3, 1/4, 5/3, 1/5, 3/4, 2/5, 5/4, 1/8, 7/4, 3/5, \
        4/5, 1/7, 6/5, 5/6, 7/5, 3/8, 8/5, 7/6, 9/5, 2/7, 11/6, 3/7, 4/7, 1/12, 5/7, 5/8, 6/7, \
        1/9, 8/7, 7/8, 9/7, 1/10, 10/7, 9/8, 11/7, 2/9, 12/7, 11/8, 13/7, 1/11, 13/8, 4/9]",
    );
    exhaustive_rational_inclusive_range_helper(
        "1/3",
        "1/2",
        "[1/2, 1/3, 2/5, 3/8, 3/7, 4/9, 4/11, 5/14, 5/11, 5/12, 5/13, 7/15, 6/13, 7/16, 6/17, \
        8/21, 7/17, 7/18, 8/17, 7/19, 8/19, 7/20, 9/19, 9/22, 9/20, 10/21, 8/23, 9/25, 9/23, \
        11/24, 10/23, 11/28, 11/23, 11/25, 12/25, 10/27, 9/26, 11/27, 11/26, 10/29, 13/27, 13/28, \
        11/29, 11/31, 12/29, 11/30, 13/29, 13/33, 14/29, 13/30]",
    );
    exhaustive_rational_inclusive_range_helper(
        "99/100",
        "101/100",
        "[1, 99/100, 101/100, 101/102, 100/101, 103/102, 102/101, 103/104, 102/103, 105/104, \
        104/103, 104/105, 106/105, 105/106, 107/106, 109/110, 106/107, 107/108, 108/107, 108/109, \
        109/108, 110/109, 111/110, 112/113, 110/111, 111/112, 112/111, 114/113, 113/112, 113/114, \
        115/114, 118/119, 114/115, 115/116, 116/115, 116/117, 117/116, 118/117, 117/118, 120/121, \
        119/118, 120/119, 119/120, 121/122, 121/120, 122/121, 123/122, 125/126, 122/123, 123/124]",
    );
    exhaustive_rational_inclusive_range_helper(
        "268876667/98914198",
        "245850922/78256779",
        "[3, 11/4, 14/5, 20/7, 17/6, 23/8, 25/8, 30/11, 25/9, 29/10, 26/9, 31/11, 28/9, 31/10, \
        32/11, 41/15, 34/11, 35/12, 37/12, 39/14, 36/13, 41/14, 37/13, 45/16, 38/13, 43/14, \
        40/13, 43/15, 44/15, 47/16, 46/15, 57/20, 47/15, 49/16, 47/17, 52/19, 48/17, 49/18, \
        49/17, 59/20, 50/17, 53/18, 52/17, 53/19, 53/17, 55/18, 54/19, 63/23, 55/19, 61/20]",
    );

    exhaustive_rational_inclusive_range_helper(
        "-2",
        "0",
        "[0, -1/2, -1, -1/3, -2, -3/2, -2/3, -1/6, -4/3, -1/4, -5/3, -1/5, -3/4, -2/5, -5/4, \
        -1/8, -7/4, -3/5, -4/5, -1/7, -6/5, -5/6, -7/5, -3/8, -8/5, -7/6, -9/5, -2/7, -11/6, \
        -3/7, -4/7, -1/12, -5/7, -5/8, -6/7, -1/9, -8/7, -7/8, -9/7, -1/10, -10/7, -9/8, -11/7, \
        -2/9, -12/7, -11/8, -13/7, -1/11, -13/8, -4/9]",
    );
    exhaustive_rational_inclusive_range_helper(
        "-1/2",
        "-1/3",
        "[-1/2, -1/3, -2/5, -3/8, -3/7, -4/9, -4/11, -5/14, -5/11, -5/12, -5/13, -7/15, -6/13, \
        -7/16, -6/17, -8/21, -7/17, -7/18, -8/17, -7/19, -8/19, -7/20, -9/19, -9/22, -9/20, \
        -10/21, -8/23, -9/25, -9/23, -11/24, -10/23, -11/28, -11/23, -11/25, -12/25, -10/27, \
        -9/26, -11/27, -11/26, -10/29, -13/27, -13/28, -11/29, -11/31, -12/29, -11/30, -13/29, \
        -13/33, -14/29, -13/30]",
    );
    exhaustive_rational_inclusive_range_helper(
        "-101/100",
        "-99/100",
        "[-1, -99/100, -101/100, -101/102, -100/101, -103/102, -102/101, -103/104, -102/103, \
        -105/104, -104/103, -104/105, -106/105, -105/106, -107/106, -109/110, -106/107, -107/108, \
        -108/107, -108/109, -109/108, -110/109, -111/110, -112/113, -110/111, -111/112, -112/111, \
        -114/113, -113/112, -113/114, -115/114, -118/119, -114/115, -115/116, -116/115, -116/117, \
        -117/116, -118/117, -117/118, -120/121, -119/118, -120/119, -119/120, -121/122, -121/120, \
        -122/121, -123/122, -125/126, -122/123, -123/124]",
    );
    exhaustive_rational_inclusive_range_helper(
        "-245850922/78256779",
        "-268876667/98914198",
        "[-3, -11/4, -14/5, -20/7, -17/6, -23/8, -25/8, -30/11, -25/9, -29/10, -26/9, -31/11, \
        -28/9, -31/10, -32/11, -41/15, -34/11, -35/12, -37/12, -39/14, -36/13, -41/14, -37/13, \
        -45/16, -38/13, -43/14, -40/13, -43/15, -44/15, -47/16, -46/15, -57/20, -47/15, -49/16, \
        -47/17, -52/19, -48/17, -49/18, -49/17, -59/20, -50/17, -53/18, -52/17, -53/19, -53/17, \
        -55/18, -54/19, -63/23, -55/19, -61/20]",
    );

    exhaustive_rational_inclusive_range_helper(
        "-2",
        "3",
        "[0, 1/2, 1, 1/3, -1, -1/2, 2, 1/4, -2, 3/2, 3, -1/3, -3/2, 2/3, 5/2, 1/6, -2/3, -1/4, \
        4/3, 1/5, -4/3, 3/4, 5/3, -1/6, -5/3, -3/4, 7/3, -1/5, 8/3, 5/4, -5/4, 1/9, 7/4, 2/5, \
        -7/4, 5/6, 9/4, -2/5, 11/4, 1/7, 3/5, -5/6, -3/5, -1/7, 4/5, 7/6, -4/5, -1/9, 6/5, -7/6]",
    );
    exhaustive_rational_inclusive_range_helper(
        "-1/2",
        "1/3",
        "[0, -1/2, 1/3, 1/5, -1/3, 1/4, -1/4, 1/7, -1/5, 1/6, -2/5, -1/7, -1/6, 2/7, -2/7, 1/11, \
        -3/7, 1/8, -1/8, 1/10, -3/8, 1/9, -1/9, 1/12, 2/9, -1/10, -2/9, -1/11, -4/9, 3/10, -3/10, \
        1/15, 2/11, -1/12, -2/11, 1/13, 3/11, -5/12, -3/11, 1/14, -4/11, -1/13, -5/11, -1/14, \
        2/13, 3/14, -2/13, 1/17, 3/13, -3/14]",
    );
    exhaustive_rational_inclusive_range_helper(
        "-101/100",
        "99/100",
        "[0, 1/2, -1, 1/3, -1/2, -1/3, 2/3, 1/6, -2/3, 1/4, -1/4, -1/6, 3/4, 1/5, -3/4, 1/8, \
        -1/5, 5/6, 2/5, 1/7, -2/5, -5/6, 3/5, -1/8, -3/5, -1/7, 4/5, 3/8, -4/5, 2/7, -2/7, 1/12, \
        3/7, -3/8, -3/7, 1/9, 4/7, 5/8, -4/7, 1/10, 5/7, -5/8, -5/7, -1/9, 6/7, 7/8, -6/7, 1/11, \
        -7/8, 2/9]",
    );
    exhaustive_rational_inclusive_range_helper(
        "-245850922/78256779",
        "268876667/98914198",
        "[0, 1/2, 1, 1/3, -1, -1/2, 2, 1/4, -2, 3/2, -3, -1/3, -3/2, 2/3, 5/2, 1/6, -5/2, -2/3, \
        4/3, 1/5, -4/3, -1/4, 5/3, -1/6, -5/3, 3/4, 7/3, -1/5, -7/3, -3/4, 8/3, 1/8, -8/3, 5/4, \
        -5/4, 5/6, 7/4, 2/5, -7/4, 1/7, 9/4, -2/5, -9/4, -5/6, -11/4, 3/5, -3/5, 1/9, 4/5, 7/6]",
    );
    exhaustive_rational_inclusive_range_helper("0", "0", "[0]");
}

#[test]
#[should_panic]
fn exhaustive_rational_inclusive_range_fail() {
    exhaustive_rational_inclusive_range(Rational::ONE, Rational::ZERO);
}
