// Copyright © 2025 Mikhail Hogrefe
//
// This file is part of Malachite.
//
// Malachite is free software: you can redistribute it and/or modify it under the terms of the GNU
// Lesser General Public License (LGPL) as published by the Free Software Foundation; either version
// 3 of the License, or (at your option) any later version. See <https://www.gnu.org/licenses/>.

use itertools::{Itertools, repeat_n};
use malachite_base::num::basic::integers::PrimitiveInt;
use malachite_base::num::basic::traits::{One, Zero};
use malachite_base::num::basic::unsigneds::PrimitiveUnsigned;
use malachite_base::num::conversion::traits::{
    ConvertibleFrom, Digits, ExactFrom, PowerOf2Digits, SaturatingFrom, WrappingFrom,
};
use malachite_base::slices::{slice_leading_zeros, slice_trailing_zeros};
use malachite_base::test_util::generators::common::GenConfig;
use malachite_base::test_util::generators::{
    unsigned_pair_gen_var_10, unsigned_vec_unsigned_pair_gen_var_5,
    unsigned_vec_unsigned_pair_gen_var_12,
};
use malachite_nz::natural::Natural;
use malachite_nz::natural::conversion::digits::general_digits::*;
use malachite_nz::platform::Limb;
use malachite_nz::test_util::generators::{
    natural_unsigned_pair_gen_var_5, natural_vec_natural_pair_gen_var_1,
    natural_vec_natural_pair_gen_var_2, natural_vec_natural_pair_gen_var_3,
    natural_vec_natural_pair_gen_var_4, unsigned_vec_unsigned_vec_unsigned_triple_gen_var_2,
    unsigned_vec_unsigned_vec_unsigned_triple_gen_var_3,
};
use std::iter::empty;
use std::panic::catch_unwind;
use std::str::FromStr;

fn verify_limbs_from_digits_small_base<T: PrimitiveUnsigned>(
    original_out: &[Limb],
    xs: &[T],
    base: u64,
    out_len: usize,
    out: &[Limb],
    preserve_unused_out: bool,
) where
    Natural: From<T>,
{
    let mut expected_limbs = from_digits_desc_naive_primitive(xs, T::exact_from(base))
        .unwrap()
        .into_limbs_asc();
    assert!(expected_limbs.len() <= out_len);
    expected_limbs.resize(out_len, 0);
    assert_eq!(expected_limbs, &out[..out_len]);
    if preserve_unused_out {
        assert_eq!(&original_out[out_len..], &out[out_len..]);
    }
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_from_digits_small_base_basecase() {
    fn test(out_before: &[Limb], xs: &[u8], base: u64, out_after: &[Limb]) {
        let mut out = out_before.to_vec();
        let out_len = limbs_from_digits_small_base_basecase(&mut out, xs, base).unwrap();
        assert_eq!(&out[..out_len], out_after);
        verify_limbs_from_digits_small_base(out_before, xs, base, out_len, &out, true);
    }
    // - res_digit == 0
    test(&[10; 2], &[0], 9, &[]);
    // - base != 10
    // - size == 0 second time
    // - res_digit != 0
    test(&[10; 2], &[1], 9, &[1]);
    test(&[10; 2], &[2, 0, 0, 2, 1, 1, 0, 0, 1, 1, 0], 3, &[123456]);
    test(&[10; 2], &[1, 2, 4, 2, 2, 3, 1, 1], 5, &[123456]);
    test(&[10; 2], &[2, 3, 5, 1, 3, 2, 0], 6, &[123456]);
    test(&[10; 2], &[1, 0, 2, 2, 6, 3, 4], 7, &[123456]);
    test(&[10; 2], &[2, 0, 7, 3, 1, 3], 9, &[123456]);
    // - base == 10
    test(&[10; 2], &[1, 2, 3, 4, 5, 6], 10, &[123456]);
    test(&[10; 2], &[8, 4, 8, 3, 3], 11, &[123456]);
    test(&[10; 2], &[5, 11, 5, 4, 0], 12, &[123456]);
    test(&[10; 2], &[4, 4, 2, 6, 8], 13, &[123456]);
    test(&[10; 2], &[3, 2, 13, 12, 4], 14, &[123456]);
    test(&[10; 2], &[2, 6, 8, 10, 6], 15, &[123456]);
    test(&[10; 2], &[12, 34, 56], 100, &[123456]);
    test(&[10; 2], &[8, 19, 87], 123, &[123456]);
    test(&[10; 2], &[1, 229, 36], 255, &[123456]);
    // - j < xs_len
    // - size == 0 first time
    // - y != 0
    // - size != 0 second time
    // - cy_limb != 0 second time
    test(&[10; 4], &[27, 90, 34, 55, 72, 93], 104, &[4056409437, 78]);
    // - size != 0 first time
    // - cy_limb != 0 first time
    test(
        &[10; 13],
        &[93, 88, 7, 58, 53, 12, 72, 49, 56, 91],
        98,
        &[419414959, 1047946650, 4],
    );
    // - cy_limb == 0 second time
    test(
        &[10; 7],
        &[14, 117, 113, 119, 39, 15, 71, 111, 83, 69, 36, 65, 47, 105, 10, 57, 101],
        121,
        &[3340886628, 3137387930, 2143550403, 399066],
    );
    // - cy_limb == 0 first time
    test(
        &[10; 15],
        &[
            116, 130, 52, 119, 76, 102, 131, 43, 138, 137, 117, 78, 23, 136, 111, 20, 132, 103,
            126, 38, 96, 23, 23, 128, 17, 123, 30, 135, 39, 114, 36, 49, 22, 90, 79, 87, 32, 21,
            67, 55, 138, 33, 44, 84, 72, 104, 10, 118, 28, 63, 85, 36, 110, 108, 61, 16, 115, 75,
            26,
        ],
        144,
        &[
            3776613706, 1267825649, 3972333409, 93554931, 973182892, 3884362880, 1799682409,
            3576731056, 1717911507, 3669574309, 502648924, 825650249, 3326096263, 105,
        ],
    );
    // - y == 0
    test(&[10; 3], &[0; 8], 17, &[]);
}

fn limbs_from_digits_small_base_basecase_properties_helper<T: PrimitiveUnsigned>()
where
    Limb: WrappingFrom<T>,
    Natural: From<T>,
{
    let mut config = GenConfig::new();
    config.insert("mean_stripe_n", 4 << Limb::LOG_WIDTH);
    config.insert("mean_digit_count_n", 32);
    config.insert("mean_excess_limb_count_n", 32);
    unsigned_vec_unsigned_vec_unsigned_triple_gen_var_3::<T, Limb>().test_properties_with_config(
        &config,
        |(mut out, xs, base)| {
            let t_base = T::exact_from(base);
            assert_eq!(
                limbs_from_digits_small_base_basecase(&mut out, &xs, base).is_some(),
                xs.iter().all(|&x| x < t_base)
            );
        },
    );

    unsigned_vec_unsigned_vec_unsigned_triple_gen_var_2::<T, Limb>().test_properties_with_config(
        &config,
        |(mut out, xs, base)| {
            let old_out = out.clone();
            let out_len = limbs_from_digits_small_base_basecase(&mut out, &xs, base).unwrap();
            verify_limbs_from_digits_small_base(&old_out, &xs, base, out_len, &out, true);
        },
    );
}

#[test]
fn limbs_from_digits_small_base_basecase_properties() {
    apply_fn_to_unsigneds!(limbs_from_digits_small_base_basecase_properties_helper);
}

#[test]
fn test_limbs_from_digits_small_base() {
    fn test(out_before: &[Limb], xs: &[u8], base: u64, out_after: &[Limb]) {
        let mut out = out_before.to_vec();
        let out_len = limbs_from_digits_small_base(&mut out, xs, base).unwrap();
        assert_eq!(&out[..out_len], out_after);
        verify_limbs_from_digits_small_base(out_before, xs, base, out_len, &out, false);
    }
    #[cfg(feature = "32_bit_limbs")]
    {
        // - xs_len < SET_STR_PRECOMPUTE_THRESHOLD
        test(&[10; 2], &[0], 9, &[]);
        test(&[10; 2], &[1], 9, &[1]);
        test(&[10; 2], &[2, 0, 0, 2, 1, 1, 0, 0, 1, 1, 0], 3, &[123456]);
        // - xs_len >= SET_STR_PRECOMPUTE_THRESHOLD
        // - xs_len > len_lo in limbs_from_digits_small_base_divide_and_conquer
        // - len_hi < SET_STR_DC_THRESHOLD in limbs_from_digits_small_base_divide_and_conquer
        // - out_len_hi > 0 in limbs_from_digits_small_base_divide_and_conquer
        // - len_lo < SET_STR_DC_THRESHOLD in limbs_from_digits_small_base_divide_and_conquer
        // - out_len_lo != 0 in limbs_from_digits_small_base_divide_and_conquer
        test(
            &[10; 21],
            &[
                73, 23, 120, 45, 108, 147, 113, 90, 129, 11, 86, 0, 102, 81, 22, 17, 32, 121, 29,
                82, 27, 25, 39, 9, 139, 59, 51, 13, 44, 3, 37, 104, 41, 40, 87, 66, 83, 146, 40,
                132, 15, 50, 66, 80, 73, 61, 75, 25, 78, 34, 128, 86, 68, 26, 62, 135, 21, 9, 27,
                74, 12, 39, 67, 15, 41, 50, 144, 35, 98, 140, 124, 132, 10, 37, 101, 99, 14,
            ],
            150,
            &[
                807199004, 1221740022, 1059231306, 812955315, 3612990781, 1076930217, 4174550874,
                1465864994, 3608544731, 124397213, 3447302311, 1159387221, 1234841936, 1231519323,
                2930655271, 132298638, 866385569, 3068,
            ],
        );
        // - len_hi >= SET_STR_DC_THRESHOLD in limbs_from_digits_small_base_divide_and_conquer
        // - len_lo >= SET_STR_DC_THRESHOLD in limbs_from_digits_small_base_divide_and_conquer
        test(
            &[10; 26],
            &[
                93, 88, 7, 58, 53, 12, 72, 49, 56, 91, 68, 47, 3, 50, 89, 6, 66, 53, 64, 90, 97,
                65, 54, 17, 8, 15, 9, 17, 0, 92, 68, 96, 25, 27, 74, 39, 36, 75, 93, 95, 11, 5, 8,
                56, 32, 50, 23, 67, 15, 26, 56, 87, 92, 1, 68, 19, 55, 67, 40, 44, 29, 60, 89, 4,
                11, 34, 46, 46, 78, 89, 47, 40, 30, 93, 57, 42, 7, 0, 75, 96, 70, 82, 10, 74, 29,
                36, 20, 51, 86, 4, 45, 56, 29, 38, 58, 77, 34, 43, 32, 92, 38, 20, 95, 35, 7, 56,
                87, 54, 61,
            ],
            98,
            &[
                3323139349, 754115856, 3445503528, 536428460, 3075474891, 2424049884, 1254229577,
                3821259306, 2296236337, 3400357558, 2698888552, 1812174, 3958831496, 3237209566,
                3557655660, 684907780, 2294464631, 693096664, 1238680639, 1345304633, 2472224180,
                2869650732, 125880,
            ],
        );
        test(
            &[10; 1336],
            &[
                6, 14, 15, 18, 4, 19, 14, 2, 21, 21, 20, 4, 17, 8, 15, 12, 21, 0, 2, 8, 22, 0, 20,
                2, 9, 2, 12, 7, 18, 21, 3, 4, 6, 0, 15, 12, 1, 1, 10, 19, 16, 8, 20, 11, 4, 2, 3,
                9, 8, 9, 14, 18, 11, 21, 22, 22, 5, 8, 10, 14, 18, 13, 21, 14, 7, 9, 21, 17, 6, 4,
                17, 22, 16, 16, 18, 22, 3, 5, 10, 9, 10, 0, 5, 17, 9, 14, 16, 10, 4, 5, 21, 8, 8,
                9, 16, 7, 15, 15, 4, 11, 14, 1, 3, 20, 15, 4, 2, 6, 7, 4, 3, 21, 15, 7, 12, 9, 21,
                15, 11, 10, 2, 8, 19, 12, 22, 15, 3, 1, 20, 14, 8, 8, 12, 11, 17, 1, 5, 22, 21, 21,
                11, 13, 17, 0, 10, 20, 1, 14, 17, 20, 14, 3, 6, 11, 14, 9, 21, 12, 9, 12, 17, 18,
                11, 7, 19, 5, 0, 13, 20, 12, 17, 8, 3, 5, 19, 5, 18, 10, 10, 7, 22, 20, 3, 21, 2,
                18, 15, 15, 16, 1, 13, 15, 21, 16, 4, 15, 18, 4, 4, 7, 16, 20, 0, 7, 2, 6, 15, 10,
                16, 0, 0, 2, 4, 21, 5, 7, 16, 8, 0, 20, 3, 12, 18, 8, 9, 8, 3, 1, 10, 8, 9, 17, 9,
                17, 18, 17, 14, 9, 19, 22, 1, 21, 9, 22, 8, 19, 0, 5, 16, 10, 17, 2, 0, 17, 17, 6,
                20, 3, 0, 4, 21, 0, 14, 13, 16, 12, 20, 9, 1, 18, 21, 11, 13, 18, 13, 2, 2, 12, 3,
                0, 16, 12, 21, 22, 19, 15, 0, 8, 9, 0, 12, 6, 19, 12, 7, 4, 3, 6, 18, 17, 1, 5, 22,
                22, 18, 3, 1, 14, 9, 1, 20, 10, 17, 11, 3, 19, 3, 16, 5, 11, 2, 17, 0, 21, 1, 14,
                2, 22, 11, 17, 22, 3, 4, 17, 19, 3, 7, 20, 19, 17, 19, 22, 12, 15, 22, 1, 12, 6, 5,
                4, 1, 4, 12, 13, 2, 2, 21, 5, 16, 3, 21, 15, 18, 0, 6, 11, 12, 1, 7, 14, 11, 8, 1,
                12, 12, 18, 0, 12, 12, 6, 0, 5, 4, 18, 6, 20, 3, 18, 20, 17, 20, 22, 11, 3, 7, 15,
                8, 11, 14, 21, 2, 11, 3, 22, 19, 2, 5, 21, 10, 11, 18, 9, 17, 18, 7, 12, 13, 1, 22,
                11, 11, 3, 3, 21, 16, 7, 17, 3, 18, 2, 5, 4, 12, 15, 16, 22, 4, 18, 8, 20, 20, 15,
                18, 7, 9, 18, 3, 0, 13, 15, 0, 19, 10, 10, 10, 7, 11, 10, 21, 19, 14, 20, 0, 17,
                22, 8, 3, 19, 17, 10, 13, 14, 15, 20, 4, 10, 12, 8, 0, 14, 12, 22, 18, 1, 7, 14,
                11, 7, 9, 13, 2, 0, 6, 17, 17, 1, 19, 14, 11, 10, 2, 17, 20, 6, 1, 22, 17, 14, 11,
                0, 5, 14, 11, 16, 7, 0, 0, 13, 2, 12, 7, 8, 12, 3, 0, 3, 13, 20, 16, 16, 5, 8, 9,
                22, 7, 18, 6, 10, 21, 17, 0, 8, 21, 14, 5, 6, 4, 15, 22, 22, 13, 6, 12, 13, 21, 5,
                1, 10, 21, 12, 15, 21, 7, 6, 5, 5, 7, 11, 8, 8, 14, 8, 17, 10, 15, 14, 19, 5, 2, 5,
                22, 19, 10, 20, 16, 11, 11, 19, 6, 20, 13, 18, 3, 9, 14, 1, 12, 2, 5, 12, 10, 6, 6,
                4, 7, 19, 9, 18, 8, 18, 2, 7, 21, 11, 0, 2, 13, 0, 0, 9, 2, 21, 15, 20, 7, 18, 13,
                22, 2, 11, 4, 21, 18, 10, 22, 18, 9, 4, 16, 19, 4, 16, 10, 22, 14, 8, 20, 6, 1, 1,
                18, 18, 20, 13, 22, 19, 10, 19, 19, 21, 0, 10, 21, 15, 11, 21, 13, 1, 4, 12, 22,
                15, 13, 20, 17, 3, 15, 14, 12, 9, 15, 20, 9, 1, 8, 18, 20, 21, 3, 17, 9, 21, 16,
                13, 1, 20, 11, 14, 21, 13, 21, 11, 17, 5, 9, 18, 14, 11, 9, 6, 10, 16, 3, 10, 7,
                22, 2, 1, 2, 9, 2, 8, 2, 11, 18, 18, 11, 2, 17, 20, 22, 10, 18, 16, 6, 17, 7, 9,
                14, 16, 1, 7, 15, 13, 14, 16, 10, 4, 2, 19, 5, 2, 21, 12, 6, 9, 13, 10, 13, 5, 5,
                20, 14, 20, 17, 15, 15, 6, 9, 11, 6, 7, 12, 1, 10, 10, 15, 3, 8, 11, 3, 17, 1, 11,
                18, 3, 16, 5, 20, 22, 9, 10, 2, 5, 6, 6, 13, 1, 7, 9, 22, 4, 10, 4, 3, 12, 8, 16,
                6, 8, 17, 2, 16, 19, 11, 10, 21, 8, 20, 12, 19, 1, 17, 8, 2, 8, 22, 7, 10, 11, 4,
                5, 22, 21, 2, 20, 19, 21, 13, 16, 22, 0, 4, 16, 4, 20, 0, 11, 20, 3, 17, 13, 3, 16,
                1, 13, 18, 18, 14, 0, 0, 14, 6, 20, 13, 14, 5, 5, 13, 0, 20, 3, 6, 2, 20, 9, 15, 7,
                15, 3, 6, 3, 20, 6, 1, 7, 3, 13, 14, 9, 5, 11, 4, 17, 16, 3, 7, 4, 17, 13, 10, 2,
                18, 5, 13, 21, 11, 21, 21, 19, 10, 7, 20, 4, 11, 5, 10, 18, 10, 18, 15, 12, 7, 15,
                17, 14, 6, 17, 20, 3, 0, 5, 12, 11, 20, 6, 22, 9, 18, 6, 10, 2, 22, 21, 16, 5, 20,
                20, 9, 3, 17, 11, 16, 20, 2, 9, 1, 9, 3, 22, 17, 10, 16, 18, 1, 11, 11, 7, 10, 14,
                22, 18, 1, 5, 5, 21, 5, 2, 21, 16, 1, 21, 16, 11, 4, 13, 20, 1, 20, 15, 0, 21, 11,
                5, 0, 6, 4, 8, 17, 8, 21, 20, 22, 8, 8, 2, 10, 8, 3, 18, 6, 0, 19, 15, 6, 20, 19,
                4, 10, 10, 8, 3, 14, 15, 16, 9, 13, 7, 14, 11, 21, 4, 10, 4, 7, 11, 0, 12, 16, 11,
                11, 1, 20, 11, 5, 0, 5, 5, 0, 20, 18, 20, 21, 0, 9, 10, 5, 8, 0, 17, 20, 16, 2, 22,
                5, 19, 8, 18, 22, 18, 19, 5, 6, 17, 13, 9, 12, 4, 22, 17, 1, 12, 14, 22, 6, 8, 10,
                15, 8, 4, 20, 1, 22, 6, 0, 3, 18, 2, 0, 11, 11, 12, 4, 3, 19, 20, 20, 1, 9, 10, 9,
                1, 17, 0, 12, 6, 7, 11, 1, 9, 19, 0, 2, 10, 11, 13, 0, 9, 14, 5, 12, 1, 17, 11, 11,
                9, 11, 8, 11, 17, 16, 14, 21, 22, 4, 0, 19, 19, 20, 10, 5, 19, 5, 14, 20, 2, 14,
                15, 20, 19, 18, 22, 20, 11, 8, 17, 5, 11, 1, 7, 21, 22, 15, 16, 5, 12, 22, 15, 18,
                3, 5, 9, 1, 21, 20, 9, 12, 8, 18, 13, 6, 16, 12, 11, 12, 2, 8, 11, 6, 15, 14, 17,
                5, 4, 14, 10, 10, 16, 4, 2, 16, 16, 5, 18, 19, 20, 2, 2, 12, 14, 7, 5, 19, 18, 17,
                9, 22, 8, 2, 13, 20, 6, 13, 5, 3, 14, 13, 4, 20, 12, 8, 12, 20, 1, 1, 3, 12, 22,
                18, 5, 15, 9, 21, 20, 12, 15, 7, 20, 11, 7, 6, 9, 1, 20, 8, 13, 11, 4, 14, 15, 14,
                16, 7, 19, 10, 5, 10, 9, 17, 6, 8, 20, 15, 20, 5, 16, 22, 14, 3, 4, 15, 18, 18, 6,
                15, 1, 10, 9, 1, 5, 4, 4, 4, 16, 4, 16, 6, 14, 8, 3, 7, 21, 5, 8, 20, 2, 10, 15, 1,
                7, 11, 7, 3, 4, 19, 3, 10, 10, 16, 20, 19, 0, 6, 10, 13, 9, 18, 17, 1, 0, 15, 2,
                13, 22, 13, 15, 5, 17, 6, 12, 18, 2, 1, 20, 22, 4, 14, 21, 0, 21, 21, 13, 20, 8,
                17, 16, 6, 8, 7, 8, 4, 20, 19, 0, 18, 4, 12, 19, 9, 14, 18, 19, 21, 9, 2, 11, 1,
                21, 21, 17, 1, 6, 1, 1, 12, 0, 10, 11, 14, 7, 2, 19, 13, 5, 10, 10, 7, 7, 14, 11,
                12, 11, 0, 18, 0, 17, 17, 3, 21, 5, 0, 7, 17, 12, 11, 16, 20, 11, 21, 16, 22, 8,
                10, 6, 14, 3, 14, 7, 2, 0, 9, 11, 15, 13, 14, 3, 17, 13, 20, 14, 10, 16, 9, 12, 14,
                16, 2, 10, 15, 6, 16, 21, 15, 2, 15, 11, 19, 19, 13, 9, 4, 15, 7, 16, 19, 8, 1, 3,
                20, 9, 21, 16, 3, 8, 7, 13, 20, 6, 2, 22, 20, 4, 4, 21, 8, 16, 7, 0, 7, 18, 22, 5,
                6, 22, 5, 0, 5, 10, 11, 3, 1, 16, 1, 5, 17, 17, 6, 11, 9, 21, 1, 3, 3, 11, 12, 21,
                18, 15, 8, 0, 16, 13, 2, 7, 5, 19, 9, 18, 14, 15, 0, 11, 22, 4, 8, 18, 20, 9, 1, 0,
                15, 1, 5, 0, 0, 14, 8, 3, 17, 17, 14, 9, 16, 11, 13, 20, 22, 8, 3, 10, 1, 11, 21,
                2, 17, 5, 5, 1, 9, 0, 13, 1, 10, 9, 4, 5, 3, 14, 21, 19, 12, 6, 6, 5, 13, 13, 6,
                19, 3, 3, 4, 8, 13, 22, 3, 19, 4, 18, 18, 16, 18, 22, 10, 10, 1, 15, 17, 18, 8, 7,
                11, 7, 10, 5, 21, 9, 0, 15, 8, 3, 16, 7, 21, 0, 19, 13, 6, 16, 11, 12, 13, 5, 13,
                7, 14, 15, 3, 12, 4, 4, 22, 16, 11, 4, 0, 18, 4, 14, 18, 11, 4, 18, 7, 7, 9, 14, 8,
                3, 11, 11, 0, 1, 3, 8, 5, 11, 7, 8, 3, 0, 14, 7, 1, 3, 20, 21, 5, 3, 2, 21, 12, 0,
                21, 5, 21, 3, 18, 10, 9, 0, 3, 13, 1, 11, 13, 10, 7, 13, 18, 15, 14, 8, 14, 6, 0,
                15, 6, 16, 17, 10, 2, 8, 0, 2, 22, 4, 20, 15, 15, 22, 8, 2, 2, 12, 20, 20, 20, 1,
                0, 10, 2, 12, 14, 13, 13, 8, 22, 7, 2, 9, 15, 22, 19, 9, 17, 14, 10, 1, 21, 5, 22,
                18, 6, 18, 1, 9, 9, 12, 7, 3, 4, 13, 13, 12, 10, 1, 3, 16, 5, 22, 21, 15, 10, 22,
                19, 10, 3, 21, 17, 3, 7, 13, 18, 18, 14, 11, 15, 16, 8, 5, 20, 12, 8, 21, 10, 22,
                17, 7, 18, 9, 10, 19, 6, 14, 2, 22, 17, 4, 9, 9, 13, 19, 0, 21, 1, 14, 3, 19, 10,
                7, 3, 19, 1, 12, 19, 21, 20, 13, 4, 7, 7, 18, 1, 13, 15, 3, 10, 5, 6, 16, 5, 20,
                20, 8, 11, 14, 6, 1, 11, 0, 9, 22, 8, 14, 22, 16, 5, 19, 7, 17, 18, 7, 17, 3, 18,
                20, 18, 12, 19, 11, 4, 8, 19, 4, 6, 19, 17, 12, 1, 10, 8, 16, 12, 4, 1, 17, 21, 12,
                7, 22, 21, 15, 11, 13, 21, 19, 21, 9, 14, 5, 18, 10, 8, 10, 20, 5, 6, 5, 11, 11, 1,
                1, 1, 6, 0, 21, 14, 16, 15, 20, 13, 12, 2, 9, 8, 16, 7, 10, 17, 3, 1, 3, 21, 14, 1,
                7, 10, 19, 9, 8, 3, 9, 16, 19, 12, 6, 19, 14, 1, 2, 20, 22, 10, 9, 8, 21, 13, 9, 8,
                7, 1, 14, 9, 17, 12, 12, 4, 3, 4, 4, 22, 12, 6, 10, 4, 20, 21, 11, 3, 17, 1, 14, 0,
                0, 13, 2, 3, 18, 10, 14, 16, 1, 16, 22, 7, 0, 5, 12, 9, 19, 14, 22, 2, 20, 14, 15,
                16, 0, 7, 2, 20, 11, 6, 11, 8, 18, 0, 6, 20, 8, 3, 15, 20, 22, 14, 12, 12, 0, 20,
                6, 18, 11, 0, 9, 16, 18, 8, 9, 12, 7, 10, 16, 0, 8, 4, 8, 14, 17, 16, 17, 2, 3, 13,
                22, 22, 1, 2, 10, 4, 2, 10, 17, 20, 9, 13, 17, 1, 18, 17, 19, 14, 13, 13, 15, 11,
                17, 22, 17, 2, 18, 16, 0, 15, 22, 2, 6, 2, 6, 0, 10, 14, 4, 20, 17, 9, 7, 12, 22,
                17, 11, 9, 10, 0, 22, 18, 19, 21, 7, 10, 1, 17, 19, 8, 5, 11, 4, 3, 6, 13, 6, 17,
                17, 19, 3, 12, 4, 1, 21, 10, 13, 6, 15, 0, 8, 8, 7, 8, 9, 5, 19, 1, 2, 11, 15, 18,
                3, 0, 3, 15, 16, 13, 9, 2, 21, 21, 10, 13, 20, 7, 16, 16, 1, 3, 11, 4, 22, 10, 10,
                22, 8, 16, 6, 17, 5, 1, 22, 16, 11, 5, 6, 22, 14, 21, 16, 4, 7, 10, 12, 12, 1, 20,
                18, 13, 10, 19, 17, 7, 0, 7, 3, 22, 22, 7, 9, 19, 22, 12, 6, 13, 15, 19, 16, 12, 6,
                7, 11, 10, 15, 4, 9, 9, 11, 20, 6, 21, 9, 13, 2, 2, 10, 22, 2, 10, 2, 2, 8, 22, 4,
                3, 9, 0, 6, 10, 10, 15, 7, 12, 7, 12, 10, 1, 4, 7, 10, 8, 21, 16, 12, 12, 8, 12,
                22, 20, 0, 8, 1, 20, 20, 3, 6, 4, 15, 20, 3, 11, 17, 19, 8, 7, 6, 21, 15, 22, 15,
                5, 6, 2, 16, 11, 20, 7, 8, 6, 1, 9, 6, 12, 2, 4, 4, 11, 21, 0, 7, 15, 8, 0, 6, 13,
                6, 17, 2, 18, 10, 21, 21, 10, 4, 10, 9, 12, 8, 12, 10, 15, 5, 6, 10, 17, 15, 16, 9,
                8, 12, 14, 10, 13, 4, 2, 8, 11, 9, 19, 14, 11, 17, 6, 5, 21, 13, 19, 21, 2, 10, 10,
                11, 12, 13, 2, 17, 18, 9, 11, 8, 18, 9, 4, 8, 11, 9, 14, 0, 13, 3, 3, 15, 21, 15,
                8, 6, 0, 15, 8, 10, 2, 3, 2, 21, 16, 5, 14, 20, 19, 11, 12, 10, 16, 11, 6, 12, 6,
                7, 3, 2, 12, 15, 2, 21, 7, 7, 7, 7, 3, 10, 22, 7, 13, 16, 7, 16, 7, 18, 6, 20, 2,
                22, 14, 19, 13, 11, 0, 2, 21, 4, 3, 14, 1, 14, 4, 2, 13, 7, 0, 10, 13, 6, 8, 22,
                10, 13, 21, 5, 13, 1, 4, 21, 12, 16, 12, 14, 13, 16, 20, 18, 20, 8, 22, 19, 3, 2,
                15, 16, 19, 11, 16, 6, 17, 19, 22, 13, 11, 20, 9, 20, 21, 9, 8, 22, 22, 18, 17, 13,
                8, 18, 21, 2, 9, 13, 13, 15, 6, 14, 17, 2, 18, 3, 0, 9, 12, 11, 19, 0, 12, 0, 3,
                14, 14, 3, 8, 9, 12, 12, 10, 3, 16, 1, 12, 15, 1, 4, 3, 19, 20, 2, 14, 4, 17, 20,
                20, 21, 1, 22, 19, 3, 6, 21, 0, 8, 21, 20, 7, 13, 20, 14, 2, 15, 6, 5, 4, 0, 6, 10,
                16, 22, 11, 6, 2, 21, 5, 12, 7, 20, 21, 10, 3, 0, 8, 17, 20, 22, 2, 1, 1, 15, 18,
                9, 5, 8, 17, 18, 7, 18, 16, 1, 9, 5, 8, 15, 12, 1, 9, 11, 17, 9, 1, 4, 6, 18, 0,
                14, 3, 20, 16, 11, 2, 11, 8, 7, 22, 13, 2, 20, 21, 15, 7, 15, 4, 5, 0, 18, 18, 13,
                16, 17, 15, 3, 17, 11, 4, 5, 16, 13, 20, 15, 11, 17, 10, 21, 3, 11, 3, 3, 7, 4, 7,
                19, 8, 3, 17, 7, 21, 0, 10, 1, 3, 17, 10, 13, 16, 18, 11, 10, 22, 16, 21, 6, 12, 2,
                7, 1, 20, 3, 11, 11, 1, 1, 4, 0, 8, 17, 14, 15, 18, 1, 19, 9, 17, 20, 1, 2, 20, 9,
                16, 15, 14, 13, 7, 18, 5, 19, 22, 10, 15, 9, 21, 16, 19, 9, 12, 5, 21, 4, 21, 12,
                20, 6, 13, 19, 3, 2, 18, 14, 9, 21, 4, 19, 3, 9, 21, 4, 13, 14, 14, 10, 6, 13, 6,
                0, 14, 7, 15, 2, 18, 20, 22, 0, 12, 16, 7, 15, 2, 15, 9, 17, 9, 7, 11, 12, 8, 1,
                15, 2, 19, 0, 22, 19, 22, 14, 7, 18, 13, 9, 2, 6, 2, 15, 1, 19, 4, 9, 20, 6, 14,
                15, 17, 3, 5, 15, 21, 19, 7, 7, 6, 12, 13, 17, 10, 3, 21, 3, 10, 14, 6, 13, 21, 10,
                7, 15, 20, 6, 21, 7, 15, 4, 12, 20, 0, 19, 21, 14, 3, 12, 22, 3, 20, 11, 21, 3, 6,
                0, 1, 19, 4, 6, 22, 17, 3, 6, 20, 17, 7, 3, 0, 17, 5, 18, 19, 12, 15, 9, 16, 8, 20,
                7, 14, 2, 15, 7, 4, 3, 4, 7, 2, 21, 1, 17, 16, 5, 6, 11, 12, 22, 4, 20, 14, 13, 3,
                1, 6, 13, 10, 9, 1, 13, 5, 2, 20, 9, 16, 20, 19, 7, 12, 9, 4, 19, 6, 18, 19, 15,
                11, 6, 14, 16, 19, 14, 20, 11, 11, 0, 8, 12, 0, 10, 12, 0, 21, 17, 17, 11, 3, 6,
                13, 21, 6, 16, 6, 14, 9, 14, 5, 7, 19, 10, 12, 10, 9, 16, 21, 8, 0, 20, 19, 16, 14,
                8, 8, 13, 19, 20, 18, 7, 10, 2, 12, 4, 3, 17, 3, 15, 12, 13, 16, 4, 14, 6, 20, 9,
                15, 12, 13, 19, 12, 19, 4, 2, 2, 18, 19, 11, 11, 20, 4, 3, 15, 7, 0, 12, 5, 8, 0,
                2, 16, 21, 6, 4, 5, 14, 6, 5, 22, 3, 8, 1, 10, 14, 1, 19, 9, 18, 21, 20, 5, 5, 3,
                18, 9, 21, 15, 19, 8, 22, 20, 12, 8, 19, 22, 4, 9, 5, 13, 6, 17, 5, 10, 0, 7, 9,
                19, 22, 16, 9, 16, 9, 2, 3, 10, 13, 11, 8, 3, 22, 21, 3, 1, 20, 16, 20, 19, 21, 16,
                0, 6, 2, 14, 12, 6, 6, 0, 0, 1, 22, 19, 12, 11, 10, 1, 15, 8, 0, 3, 21, 6, 6, 21,
                5, 14, 15, 1, 5, 11, 8, 9, 15, 3, 9, 12, 22, 15, 19, 6, 22, 21, 21, 11, 6, 10, 0,
                9, 9, 4, 20, 4, 21, 18, 7, 6, 9, 3, 11, 8, 17, 17, 9, 22, 12, 21, 3, 1, 16, 22, 13,
                6, 15, 18, 4, 22, 8, 10, 7, 11, 4, 0, 13, 13, 17, 11, 7, 5, 20, 18, 3, 0, 11, 21,
                16, 15, 19, 20, 8, 5, 1, 14, 13, 8, 18, 18, 14, 4, 22, 15, 0, 20, 22, 13, 9, 4, 3,
                6, 8, 14, 7, 4, 8, 15, 20, 8, 15, 13, 4, 22, 2, 0, 22, 22, 14, 5, 2, 5, 11, 1, 8,
                22, 2, 3, 11, 9, 9, 6, 9, 14, 14, 20, 18, 18, 12, 1, 18, 12, 8, 19, 7, 6, 6, 11,
                14, 3, 16, 7, 13, 11, 17, 3, 18, 9, 2, 19, 22, 11, 20, 0, 17, 4, 14, 10, 1, 19, 3,
                21, 4, 12, 6, 13, 4, 21, 20, 4, 2, 10, 13, 11, 21, 15, 7, 17, 13, 4, 2, 3, 1, 17,
                8, 19, 15, 1, 6, 11, 20, 5, 4, 17, 6, 20, 15, 19, 16, 16, 16, 1, 18, 9, 22, 10, 8,
                12, 2, 18, 14, 9, 12, 18, 9, 8, 13, 19, 16, 3, 15, 8, 2, 14, 18, 2, 12, 4, 1, 5, 3,
                13, 6, 21, 5, 19, 20, 15, 4, 9, 18, 9, 14, 15, 19, 1, 19, 7, 11, 15, 9, 11, 22, 0,
                6, 17, 17, 6, 1, 9, 0, 6, 5, 22, 11, 18, 13, 4, 9, 17, 3, 20, 13, 14, 6, 1, 7, 14,
                16, 13, 22, 1, 15, 7, 2, 17, 22, 1, 22, 21, 0, 18, 9, 21, 22, 1, 22, 16, 7, 22, 4,
                18, 2, 9, 2, 4, 16, 12, 9, 22, 18, 18, 20, 21, 8, 21, 1, 1, 7, 22, 20, 16, 5, 21,
                14, 4, 15, 11, 19, 0, 17, 12, 12, 3, 22, 4, 22, 3, 11, 0, 7, 13, 1, 21, 0, 6, 16,
                5, 18, 5, 5, 14, 15, 4, 0, 9, 9, 17, 3, 17, 0, 2, 14, 12, 6, 1, 17, 6, 21, 3, 15,
                4, 2, 5, 22, 7, 6, 21, 6, 16, 3, 3, 5, 6, 9, 8, 19, 5, 9, 16, 3, 18, 19, 20, 18,
                10, 5, 15, 17, 20, 16, 5, 1, 13, 20, 8, 18, 14, 12, 20, 18, 18, 4, 1, 0, 2, 8, 16,
                18, 17, 15, 21, 14, 5, 13, 17, 14, 4, 10, 16, 0, 0, 22, 2, 9, 10, 13, 20, 1, 12,
                11, 10, 22, 15, 16, 19, 10, 8, 1, 21, 13, 11, 3, 21, 7, 11, 3, 14, 8, 0, 4, 2, 6,
                22, 22, 2, 5, 9, 19, 0, 13, 6, 1, 15, 21, 11, 11, 14, 7, 14, 4, 21, 11, 22, 4, 14,
                6, 13, 5, 10, 19, 8, 21, 22, 2, 12, 16, 6, 2, 1, 12, 7, 6, 20, 9, 22, 7, 12, 19, 6,
                10, 8, 10, 19, 16, 17, 9, 5, 8, 16, 14, 16, 11, 4, 2, 17, 6, 9, 5, 5, 12, 15, 15,
                18, 12, 10, 8, 1, 7, 8, 11, 2, 21, 20, 20, 22, 10, 10, 3, 9, 5, 5, 12, 12, 11, 4,
                7, 6, 1, 22, 7, 18, 20, 15, 12, 13, 10, 12, 17, 12, 12, 21, 0, 12, 19, 2, 8, 4, 17,
                5, 20, 8, 7, 16, 22, 7, 18, 17, 5, 13, 21, 7, 3, 22, 20, 15, 1, 13, 3, 6, 6, 18, 8,
                0, 16, 10, 2, 3, 8, 12, 0, 14, 1, 7, 17, 4, 17, 0, 14, 5, 22, 11, 10, 16, 19, 12,
                11, 18, 13, 22, 19, 17, 19, 0, 6, 18, 13, 11, 17, 21, 9, 21, 10, 0, 17, 14, 10, 5,
                13, 5, 14, 7, 6, 0, 14, 20, 16, 11, 9, 4, 3, 19, 3, 19, 16, 13, 21, 22, 21, 2, 12,
                13, 6, 7, 19, 18, 20, 6, 2, 9, 6, 2, 14, 6, 18, 0, 0, 14, 9, 22, 3, 22, 3, 19, 5,
                10, 2, 13, 2, 5, 15, 9, 11, 19, 10, 6, 14, 11, 9, 0, 20, 22, 17, 16, 4, 14, 7, 2,
                20, 10, 21, 8, 19, 5, 9, 16, 4, 17, 7, 17, 3, 0, 12, 0, 7, 9, 11, 4, 9, 19, 20, 19,
                17, 10, 19, 4, 20, 0, 9, 3, 7, 1, 19, 9, 13, 7, 15, 9, 17, 13, 21, 19, 15, 2, 11,
                6, 4, 16, 9, 22, 22, 19, 9, 10, 21, 5, 13, 9, 4, 20, 18, 7, 1, 1, 19, 10, 3, 9, 8,
                21, 0, 18, 7, 7, 7, 1, 8, 1, 1, 11, 16, 19, 18, 17, 10, 21, 18, 5, 5, 0, 4, 0, 0,
                7, 9, 7, 5, 11, 18, 16, 20, 14, 19, 10, 20, 5, 20, 13, 20, 7, 15, 2, 4, 21, 14, 1,
                15, 19, 17, 4, 21, 4, 14, 20, 14, 20, 20, 1, 6, 12, 11, 6, 7, 3, 0, 19, 2, 2, 17,
                14, 6, 6, 16, 12, 1, 11, 6, 22, 21, 19, 22, 19, 19, 12, 22, 10, 21, 11, 19, 18, 6,
                2, 17, 11, 11, 18, 9, 13, 0, 13, 12, 13, 6, 19, 20, 11, 11, 4, 20, 15, 22, 3, 16,
                7, 0, 17, 0, 0, 12, 19, 1, 1, 8, 10, 4, 8, 18, 20, 17, 12, 9, 13, 21, 3, 12, 4, 21,
                0, 1, 1, 15, 8, 22, 14, 20, 0, 6, 8, 22, 20, 3, 11, 14, 5, 11, 7, 20, 2, 1, 5, 7,
                9, 10, 1, 19, 19, 19, 12, 12, 20, 11, 7, 14, 0, 22, 13, 9, 3, 13, 12, 9, 10, 18, 0,
                19, 13, 6, 4, 14, 5, 15, 17, 1, 2, 18, 13, 18, 1, 22, 17, 15, 3, 2, 16, 4, 4, 21,
                10, 6, 12, 17, 3, 11, 9, 18, 15, 9, 9, 19, 13, 19, 7, 6, 5, 19, 8, 11, 9, 12, 10,
                14, 9, 7, 16, 12, 6, 17, 20, 16, 21, 5, 17, 16, 9, 14, 3, 9, 6, 12, 11, 18, 22, 14,
                4, 22, 17, 4, 9, 7, 6, 2, 18, 20, 21, 10, 9, 0, 12, 11, 3, 0, 5, 8, 15, 13, 8, 17,
                13, 20, 14, 17, 4, 8, 2, 3, 15, 15, 20, 12, 10, 11, 14, 21, 4, 22, 4, 11, 5, 20,
                11, 10, 4, 19, 18, 8, 4, 17, 4, 12, 5, 20, 12, 22, 6, 11, 3, 4, 20, 9, 10, 20, 21,
                9, 20, 21, 0, 9, 11, 6, 13, 16, 2, 8, 14, 7, 16, 11, 1, 14, 4, 8, 14, 14, 20, 1,
                17, 9, 8, 15, 4, 19, 19, 1, 16, 7, 20, 18, 18, 21, 0, 1, 8, 21, 12, 9, 14, 10, 0,
                3, 5, 5, 2, 16, 17, 14, 15, 4, 19, 4, 21, 15, 1, 21, 6, 22, 8, 6, 10, 2, 19, 20,
                19, 13, 8, 21, 21, 9, 11, 13, 11, 10, 16, 6, 14, 20, 4, 15, 11, 22, 8, 10, 2, 14,
                12, 4, 15, 10, 10, 8, 5, 5, 5, 17, 22, 6, 7, 17, 21, 14, 6, 10, 0, 0, 13, 15, 12,
                11, 15, 10, 18, 5, 18, 19, 0, 11, 10, 6, 13, 0, 21, 22, 1, 6, 22, 20, 1, 21, 4, 19,
                12, 20, 8, 7, 10, 6, 13, 21, 7, 8, 1, 1, 5, 1, 19, 15, 15, 3, 19, 11, 11, 10, 14,
                22, 2, 3, 9, 18, 10, 22, 22, 21, 14, 4, 12, 21, 20, 12, 1, 5, 10, 14, 4, 5, 0, 1,
                2, 13, 13, 8, 8, 16, 14, 18, 17, 9, 4, 2, 17, 6, 15, 16, 14, 15, 12, 19, 7, 15, 14,
                7, 0, 12, 14, 5, 12, 14, 13, 21, 3, 7, 9, 13, 21, 11, 17, 1, 22, 17, 4, 10, 22, 1,
                19, 10, 3, 20, 3, 12, 0, 21, 19, 18, 1, 4, 1, 5, 4, 17, 20, 10, 11, 18, 9, 9, 20,
                0, 13, 4, 13, 22, 11, 9, 18, 19, 18, 10, 1, 2, 8, 11, 19, 5, 2, 19, 5, 10, 14, 18,
                11, 18, 6, 19, 10, 16, 3, 12, 13, 13, 2, 19, 21, 15, 2, 15, 21, 11, 20, 15, 14, 5,
                7, 7, 10, 8, 3, 9, 4, 0, 21, 17, 21, 0, 11, 11, 14, 16, 10, 14, 21, 4, 17, 8, 3,
                10, 10, 8, 16, 13, 0, 21, 14, 8, 3, 11, 11, 3, 15, 10, 1, 1, 18, 15, 6, 2, 1, 10,
                20, 12, 5, 4, 0, 14, 13, 3, 20, 8, 9, 16, 16, 12, 7, 2, 4, 14, 7, 16, 10, 4, 11, 4,
                5, 9, 19, 22, 8, 15, 13, 4, 8, 10, 9, 15, 20, 16, 0, 17, 13, 1, 12, 18, 22, 10, 8,
                14, 9, 16, 1, 6, 12, 19, 22, 3, 2, 17, 17, 13, 7, 8, 18, 15, 8, 21, 3, 14, 1, 10,
                13, 1, 13, 3, 6, 20, 7, 21, 13, 0, 8, 13, 5, 19, 19, 11, 5, 17, 3, 13, 12, 22, 10,
                5, 6, 10, 1, 20, 13, 11, 17, 14, 13, 7, 14, 19, 22, 12, 12, 15, 3, 11, 12, 15, 7,
                22, 17, 4, 14, 9, 4, 18, 17, 11, 7, 5, 19, 16, 8, 4, 2, 11, 19, 3, 15, 22, 0, 3,
                20, 0, 7, 3, 2, 6, 16, 22, 15, 19, 18, 14, 2, 17, 15, 3, 13, 10, 20, 12, 8, 21, 11,
                5, 20, 22, 22, 11, 9, 20, 14, 12, 20, 8, 6, 9, 12, 6, 16, 16, 20, 22, 0, 20, 8, 2,
                12, 22, 19, 16, 15, 6, 11, 7, 1, 8, 14, 21, 14, 22, 18, 1, 22, 8, 8, 14, 22, 3, 5,
                22, 6, 9, 7, 13, 7, 12, 10, 6, 6, 22, 8, 5, 20, 12, 0, 13, 7, 2, 17, 0, 6, 20, 3,
                18, 13, 21, 11, 21, 12, 19, 7, 21, 13, 21, 21, 6, 22, 7, 9, 20, 13, 12, 20, 2, 1,
                12, 16, 5, 8, 5, 5, 5, 16, 13, 3, 2, 12, 5, 7, 8, 12, 4, 10, 17, 5, 17, 3, 18, 22,
                3, 6, 3, 10, 18, 18, 3, 3, 2, 4, 18, 3, 18, 17, 2, 11, 3, 14, 4, 16, 2, 12, 4, 7,
                18, 21, 22, 5, 18, 9, 7, 10, 17, 16, 7, 17, 7, 2, 5, 11, 12, 4, 22, 7, 21, 0, 8,
                16, 0, 16, 1, 9, 11, 13, 11, 6, 19, 13, 0, 10, 15, 1, 3, 8, 21, 7, 6, 21, 11, 15,
                13, 1, 10, 20, 17, 22, 7, 16, 8, 9, 9, 22, 6, 18, 12, 21, 17, 14, 10, 10, 20, 18,
                10, 10, 7, 2, 7, 11, 21, 1, 1, 16, 3, 22, 2, 1, 6, 12, 16, 13, 1, 15, 18, 4, 2, 16,
                15, 5, 5, 2, 4, 13, 9, 10, 1, 4, 3, 16, 9, 4, 17, 11, 10, 9, 1, 9, 1, 18, 0, 6, 6,
                17, 14, 17, 20, 19, 7, 21, 4, 9, 17, 5, 0, 22, 19, 10, 20, 5, 13, 12, 19, 13, 21,
                14, 14, 2, 15, 19, 2, 8, 20, 9, 11, 6, 14, 4, 12, 21, 9, 7, 10, 12, 3, 14, 1, 2, 6,
                18, 14, 20, 11, 19, 13, 3, 21, 10, 22, 12, 4, 19, 12, 2, 12, 6, 4, 5, 2, 15, 2, 17,
                11, 21, 3, 8, 14, 7, 3, 7, 4, 16, 9, 2, 13, 2, 13, 14, 9, 3, 6, 11, 13, 22, 8, 20,
                11, 2, 6, 13, 22, 15, 1, 14, 7, 16, 18, 15, 2, 21, 21, 5, 19, 5, 12, 5, 14, 7, 2,
                1, 12, 22, 8, 16, 13, 12, 15, 10, 6, 21, 8, 9, 6, 14, 11, 11, 8, 3, 1, 21, 9, 3,
                17, 21, 6, 8, 8, 15, 20, 9, 19, 18, 16, 22, 20, 18, 4, 19, 18, 7, 20, 17, 16, 5,
                17, 8, 22, 13, 9, 6, 13, 2, 22, 15, 1, 1, 14, 2, 11, 18, 19, 8, 10, 7, 5, 16, 5,
                13, 19, 17, 2, 15, 14, 8, 14, 20, 6, 16, 9, 13, 16, 16, 19, 9, 2, 12, 15, 17, 19,
                21, 4, 6, 13, 15, 18, 22, 17, 12, 20, 15, 5, 7, 18, 20, 12, 19, 14, 19, 1, 8, 20,
                5, 18, 4, 11, 5, 10, 11, 11, 2, 20, 20, 2, 5, 4, 0, 15, 6, 7, 10, 14, 14, 0, 13,
                20, 5, 17, 17, 0, 2, 0, 18, 0, 10, 17, 17, 2, 19, 5, 2, 21, 1, 8, 17, 16, 0, 6, 4,
                22, 8, 0, 16, 16, 2, 20, 8, 12, 15, 17, 21, 12, 8, 21, 10, 2, 21, 13, 11, 18, 5,
                19, 9, 2, 10, 11, 6, 4, 10, 20, 20, 9, 7, 11, 13, 20, 3, 3, 8, 7, 7, 10, 14, 10,
                14, 10, 16, 3, 14, 0, 3, 19, 18, 21, 3, 21, 10, 17, 22, 12, 18, 21, 4, 14, 16, 2,
                14, 15, 2, 8, 14, 17, 14, 6, 3, 0, 17, 2, 15, 4, 4, 20, 14, 14, 1, 19, 0, 12, 14,
                6, 5, 16, 10, 8, 20, 5, 1, 10, 8, 2, 3, 22, 16, 0, 7, 0, 15, 5, 12, 9, 11, 5, 19,
                12, 12, 14, 4, 22, 6, 9, 3, 18, 12, 9, 10, 10, 19, 19, 21, 14, 10, 5, 4, 9, 3, 22,
                17, 10, 17, 19, 8, 6, 14, 14, 5, 5, 0, 9, 3, 11, 2, 17, 15, 7, 4, 19, 8, 21, 8, 7,
                11, 2, 16, 3, 2, 19, 13, 19, 17, 7, 21, 4, 4, 11, 15, 3, 1, 19, 19, 22, 10, 2, 5,
                8, 15, 17, 6, 5, 15, 15, 9, 7, 2, 0, 14, 11, 14, 1, 3, 13, 14, 4, 3, 11, 6, 14, 0,
                0, 1, 2, 19, 8, 6, 14, 10, 9, 19, 22, 11, 12, 1, 15, 11, 10, 7, 15, 13, 8, 3, 12,
                8, 1, 13, 13, 22, 17, 0, 15, 0, 1, 4, 13, 17, 8, 10, 20, 17, 17, 18, 3, 4, 13, 7,
                16, 7, 17, 22, 0, 0, 13, 13, 21, 6, 4, 0, 19, 12, 9, 20, 1, 21, 20, 2, 14, 20, 19,
                0, 18, 5, 14, 1, 12, 0, 9, 21, 8, 4, 11, 3, 22, 20, 15, 22, 5, 22, 20, 11, 21, 18,
                8, 13, 12, 6, 10, 22, 12, 13, 7, 6, 13, 1, 13, 2, 19, 1, 13, 1, 0, 21, 5, 6, 17,
                22, 14, 20, 3, 12, 8, 19, 6, 21, 19, 3, 2, 8, 3, 6, 2, 13, 19, 20, 8, 15, 19, 10,
                5, 22, 16, 13, 0, 4, 4, 4, 2, 15, 13, 5, 10, 3, 21, 3, 3, 4, 1, 4, 22, 10, 2, 2, 2,
                8, 18, 8, 13, 19, 13, 9, 7, 16, 5, 8, 6, 3, 10, 10, 5, 19, 7, 4, 5, 5, 1, 9, 4, 6,
                12, 8, 8, 2, 3, 7, 10, 10, 22, 2, 9, 14, 14, 0, 3, 2, 12, 18, 6, 8, 8, 8, 5, 20,
                12, 15, 3, 8, 12, 13, 21, 18, 17, 12, 18, 13, 13, 8, 10, 2, 3, 7, 15, 4, 13, 15,
                22, 0, 6, 12, 10, 12, 3, 20, 22, 11, 15, 0, 12, 3, 16, 11, 5, 0, 6, 19, 20, 8, 15,
                21, 10, 10, 9, 16, 3, 20, 3, 13, 2, 16, 16, 11, 20, 12, 6, 8, 0, 1, 15, 12, 9, 12,
                18, 20, 14, 13, 17, 8, 12, 1, 10, 16, 1, 0, 7, 5, 0, 8, 17, 19, 20, 5, 15, 4, 9,
                16, 21, 16, 2, 22, 12, 11, 2, 9, 2, 5, 7, 14, 10, 8, 14, 9, 4, 3, 12, 9, 7, 1, 4,
                10, 22, 12, 11, 4, 5, 14, 11, 11, 22, 21, 15, 12, 17, 0, 12, 8, 6, 6, 7, 16, 3, 2,
                4, 21, 18, 10, 15, 17, 10, 22, 9, 21, 14, 12, 15, 6, 13, 10, 5, 6, 16, 17, 1, 13,
                12, 3, 4, 9, 3, 4, 20, 9, 4, 15, 16, 15, 22, 6, 1, 20, 12, 5, 12, 17, 17, 14, 21,
                12, 2, 0, 11, 17, 9, 16, 4, 9, 7, 9, 10, 22, 4, 9, 21, 1, 1, 15, 16, 12, 15, 21, 1,
                10, 19, 22, 14, 4, 17, 5, 2, 1, 13, 21, 22, 12, 5, 13, 8, 22, 15, 22, 5, 5, 11, 12,
                9, 16, 7, 4, 0, 15, 14, 18, 15, 5, 7, 13, 10, 13, 2, 5, 19, 8, 3, 16, 6, 13, 1, 1,
                16, 20, 22, 3, 18, 6, 9, 15, 8, 21, 2, 16, 5, 19, 11, 13, 4, 18, 4, 12, 13, 17, 4,
                10, 7, 8, 7, 2, 22, 9, 20, 16, 10, 10, 18, 22, 22, 5, 15, 8, 10, 15, 14, 5, 12, 20,
                21, 5, 13, 9, 10, 0, 16, 17, 20, 0, 6, 8, 11, 2, 9, 17, 22, 11, 14, 10, 10, 17, 3,
                7, 8, 20, 1, 11, 17, 18, 1, 14, 1, 16, 14, 12, 19, 14, 15, 13, 22, 9, 18, 17, 16,
                5, 15, 22, 7, 19, 14, 21, 12, 22, 15, 16, 9, 7, 11, 18, 6, 17, 22, 21, 15, 11, 12,
                10, 17, 1, 1, 17, 4, 15, 3, 4, 5, 13, 9, 16, 14, 4, 19, 21, 20, 20, 2, 15, 19, 11,
                13, 9, 1, 17, 16, 7, 5, 8, 13, 14, 5, 17, 13, 21, 19, 14, 14, 20, 7, 21, 18, 22,
                18, 2, 9, 4, 5, 7, 10, 2, 14, 18, 12, 10, 4, 5, 15, 0, 0, 4, 6, 14, 0, 2, 0, 21,
                19, 22, 22, 3, 15, 6, 11, 22, 15, 21, 14, 13, 0, 7, 8, 4, 20, 12, 12, 11, 15, 12,
                15, 0, 5, 13, 6, 0, 22, 11, 10, 1, 1, 13, 11, 19, 2, 0, 12, 8, 8, 20, 9, 14, 10,
                12, 20, 8, 20, 9, 10, 6, 21, 2, 17, 17, 11, 1, 14, 13, 5, 11, 4, 2, 13, 2, 21, 18,
                9, 13, 14, 13, 2, 3, 19, 8, 17, 15, 8, 17, 12, 17, 8, 15, 4, 20, 22, 7, 7, 21, 5,
                6, 14, 17, 0, 15, 16, 22, 11, 17, 17, 20, 21, 13, 0, 13, 14, 2, 5, 14, 16, 3, 14,
                11, 15, 4, 9, 0, 6, 5, 5, 18, 7, 22, 4, 21, 17, 3, 1, 17, 18, 12, 12, 18, 14, 18,
                0, 0, 8, 14, 1, 14, 0, 6, 21, 8, 8, 18, 6, 1, 17, 0, 0, 16, 0, 13, 12, 7, 18, 2, 6,
                16, 13, 11, 0, 5, 1, 6, 8, 21, 22, 3, 6, 8, 21, 12, 19, 16, 11, 3, 17, 13, 15, 8,
                1, 11, 22, 2, 8, 22, 7, 14, 14, 20, 18, 18, 20, 22, 17, 3, 20, 4, 14, 13, 17, 2, 5,
                20, 19, 17, 16, 18, 17, 20, 6, 19, 2, 2, 9, 0, 10, 1, 11, 18, 0, 7, 19, 16, 8, 0,
                1, 21, 14, 21, 3, 18, 0, 1, 2, 4, 0, 9, 17, 12, 1, 9, 17, 21, 7, 17, 20, 9, 10, 7,
                6, 10, 10, 15, 14, 2, 9, 20, 20, 8, 15, 2, 6, 3, 0, 7, 11, 21, 21, 8, 6, 4, 7, 9,
                5, 3, 15, 4, 1, 1, 10, 21, 13, 22, 5, 22, 2, 5, 11, 6, 22, 10, 11, 3, 12, 3, 20, 7,
                22, 5, 11, 20, 0, 5, 15, 20, 10, 7, 11, 0, 21, 7, 21, 8, 0, 16, 16, 22, 12, 16, 13,
                10, 5, 22, 13, 15, 12, 9, 12, 13, 2, 7, 0, 5, 16, 18, 0, 1, 0, 6, 7, 0, 1, 7, 22,
                4, 4, 13, 9, 3, 11, 11, 10, 5, 20, 17, 16, 18, 15, 6, 20, 15, 1, 8, 4, 2, 3, 21,
                19, 3, 14, 6, 16, 5, 17, 10, 20, 15, 5, 5, 20, 21, 3, 6, 14, 9, 22, 18, 11, 14, 2,
                21, 5, 9, 16, 16, 21, 3, 4, 19, 11, 14, 19, 18, 21, 4, 0, 18, 8, 16, 21, 7, 19, 4,
                2, 14, 10, 17, 18, 18, 11, 17, 2, 14, 22, 20, 2, 22, 6, 4, 12, 6, 6, 16, 5, 16, 3,
                15, 12, 22, 7, 15, 19, 14, 9, 2, 10, 19, 5, 3, 0, 20, 3, 12, 6, 14, 2, 20, 20, 18,
                22, 3, 11, 20, 10, 1, 7, 17, 18, 17, 7, 0, 20, 8, 22, 1, 21, 7, 22, 4, 18, 7, 16,
                13, 10, 7, 13, 0, 1, 11, 7, 20, 14, 0, 5, 10, 2, 9, 17, 5, 1, 13, 14, 5, 18, 17, 9,
                16, 12, 0, 18, 7, 16, 17, 14, 10, 9, 19, 20, 0, 15, 10, 20, 15, 22, 8, 13, 19, 4,
                19, 6, 14, 1, 21, 3, 4, 7, 16, 5, 12, 9, 2, 4, 17, 22, 16, 18, 3, 19, 19, 21, 14,
                22, 13, 19, 2, 18, 0, 4, 9, 12, 12, 3, 19, 5, 18, 9, 14, 9, 4, 21, 12, 18, 13, 17,
                1, 5, 20, 5, 14, 7, 19, 16, 22, 12, 6, 2, 21, 10, 5, 12, 12, 19, 11, 0, 5, 0, 4, 4,
                8, 22, 20, 18, 20, 17, 8, 14, 0, 3, 12, 9, 21, 17, 14, 15, 10, 13, 20, 15, 21, 10,
                7, 2, 7, 13, 11, 1, 22, 9, 19, 14, 4, 0, 18, 9, 22, 10, 6, 11, 10, 21, 8, 1, 10,
                13, 21, 0, 11, 6, 8, 14, 22, 6, 13, 11, 11, 18, 4, 3, 8, 10, 20, 16, 15, 16, 10,
                14, 18, 7, 4, 16, 1, 6, 17, 4, 22, 19, 2, 9, 16, 0, 18, 15, 2, 9, 15, 5, 20, 20, 0,
                22, 10, 2, 21, 22, 14, 11, 10, 19, 18, 10, 11, 14, 12, 5, 7, 16, 15, 5, 22, 12, 16,
                13, 6, 11, 21, 18, 21, 15, 19, 7, 9, 2, 13, 2, 14, 6, 9, 2, 19, 2, 2, 0, 5, 19, 13,
                12, 10, 11, 9, 6, 21, 17, 1, 16, 15, 9, 4, 5, 12, 11, 0, 21, 21, 12, 20, 11, 20, 7,
                10, 4, 21, 20, 0, 17, 8, 14, 3, 17, 1, 7, 17, 16, 16, 3, 19, 18, 7, 13, 17, 2, 8,
                2, 17, 4, 5, 7, 22, 2, 5, 8, 5, 8, 21, 15, 14, 17, 11, 16, 20, 8, 1, 5, 10, 8, 5,
                7, 14, 5, 16, 6, 17, 8, 14, 4, 21, 5, 22, 6, 18, 22, 4, 16, 0, 0, 8, 22, 3, 9, 13,
                21, 13, 5, 9, 5, 13, 17, 11, 19, 17, 15, 6, 5, 11, 12, 4, 2, 7, 13, 16, 3, 8, 0, 5,
                20, 12, 11, 1, 15, 16, 0, 9, 7, 9, 16, 15, 21, 19, 19, 8, 3, 12, 14, 21, 18, 22, 5,
                13, 12, 20, 11, 13, 11, 7, 16, 12, 7, 13, 15, 14, 22, 10, 17, 0, 11, 6, 20, 3, 2,
                14, 13, 22, 21, 18, 20, 7, 18, 9, 5, 14, 22, 1, 9, 19, 3, 12, 20, 1, 15, 18, 6, 7,
                16, 2, 20, 17, 2, 4, 11, 15, 19, 4, 14, 16, 19, 9, 2, 17, 1, 17, 5, 5, 10, 5, 20,
                4, 22, 2, 8, 1, 12, 21, 22, 2, 4, 22, 5, 15, 18, 5, 4, 4, 22, 15, 20, 22, 19, 19,
                8, 17, 16, 15, 1, 11, 12, 5, 0, 17, 14, 0, 10, 0, 12, 18, 16, 14, 4, 21, 16, 2, 11,
                3, 6, 10, 12, 14, 1, 21, 10, 16, 3, 19, 11, 14, 19, 8, 4, 20, 9, 14, 6, 5, 16, 19,
                21, 9, 7, 16, 19, 14, 19, 7, 17, 9, 11, 8, 2, 3, 21, 19, 14, 14, 0, 10, 8, 20, 13,
                11, 18, 4, 1, 4, 9, 13, 21, 18, 12, 19, 11, 20, 16, 11, 6, 13, 15, 3, 22, 2, 5, 16,
                11, 18, 3, 16, 6, 17, 9, 20, 9, 0, 14, 15, 13, 18, 22, 2, 14, 11, 4, 12, 18, 6, 16,
                5, 4, 12, 5, 4, 22, 19, 4, 22, 2, 2, 16, 4, 16, 13, 4, 5, 15, 1, 3, 3, 14, 19, 20,
                20, 3, 1, 7, 18, 6, 8, 6, 19, 4, 19, 4, 6, 2, 14, 13, 11, 2, 22, 19, 19, 13, 22,
                12, 18, 19, 16, 11, 13, 6, 9, 0, 1, 9, 10, 22, 18, 0, 2, 22, 9, 8, 4, 20, 21, 9,
                11, 9, 19, 21, 18, 17, 9, 0, 1, 2, 10, 6, 4, 16, 14, 8, 3, 1, 7, 3, 18, 16, 9, 14,
                10, 4, 1, 14, 6, 7, 10, 18, 13, 13, 17, 16, 14, 14, 11, 9, 13, 6, 10, 0, 8, 14, 19,
                6, 12, 10, 2, 9, 1, 22, 7, 11, 6, 5, 18, 9, 20, 8, 12, 10, 21, 12, 4, 22, 1, 19,
                22, 7, 11, 18, 3, 16, 3, 19, 10, 14, 8, 10, 18, 6, 22, 5, 4, 21, 15, 13, 22, 19,
                21, 6, 11, 9, 1, 11, 4, 1, 5, 0, 0, 16, 3, 11, 1, 10, 18, 2, 14, 12, 13, 17, 11, 3,
                15, 10, 22, 13, 5, 19, 2, 2, 6, 17, 3, 11, 19, 17, 5, 4, 14, 18, 14, 0, 6, 17, 2,
                13, 4, 1, 10, 17, 4, 19, 4, 3, 13, 19, 6, 19, 2, 17, 20, 10, 6, 15, 17, 4, 14, 19,
                14, 6, 9, 16, 2, 13, 3, 9, 7, 8, 12, 16, 15, 12, 15, 1, 22, 16, 3, 21, 17, 18, 2,
                6, 14, 16, 15, 20, 13, 19, 19, 4, 8, 10, 17, 8, 6, 19, 8, 8, 9, 22, 21, 13, 16, 16,
                9, 20, 10, 12, 21, 4, 11, 6, 11, 18, 7, 6, 21, 21, 2, 20, 17, 12, 1, 2, 5, 2, 5, 9,
                19, 9, 15, 4, 10, 10, 4, 22, 0, 9, 15, 0, 2, 14, 20, 21, 3, 0, 8, 13, 12, 21, 15,
                21, 20, 9, 21, 0, 18, 0, 22, 15, 11, 11, 16, 21, 14, 19, 14, 22, 8, 1, 19, 2, 18,
                4, 4, 9, 20, 11, 7, 20, 10, 4, 20, 20, 6, 20, 16, 22, 3, 4, 3, 18, 5, 7, 20, 3, 3,
                4, 3, 6, 4, 15, 6, 1, 3, 9, 22, 13, 0, 5, 6, 15, 13, 3, 3, 21, 7, 3, 16, 0, 9, 1,
                11, 22, 18, 12, 22, 6, 22, 6, 8, 14, 5, 14, 17, 19, 17, 21, 12, 8, 2, 11, 11, 17,
                2, 11, 3, 21, 1, 7, 6, 11, 8, 17, 12, 0, 4, 10, 9, 9, 16, 13, 9, 22, 20, 1, 17, 6,
                22, 17, 12, 2, 16, 11, 1, 6, 21, 14, 4, 16, 4, 0, 21, 4, 8, 20, 19, 15, 9, 2, 15,
                2, 16, 7, 8, 11, 12, 19, 7, 19, 8, 11, 1, 11, 0, 13, 16, 12, 12, 3, 8, 7, 15, 5,
                18, 4, 15, 18, 7, 12, 2, 2, 7, 7, 21, 1, 13, 19, 11, 5, 1, 15, 3, 9, 7, 2, 21, 7,
                19, 3, 15, 1, 8, 18, 6, 16, 16, 0, 18, 3, 2, 11, 12, 7, 10, 21, 18, 18, 11, 12, 6,
                20, 7, 1, 5, 18, 14, 8, 14, 21, 6, 15, 10, 14, 16, 5, 8, 6, 12, 22, 16, 11, 13, 1,
                2, 19, 16, 7, 22, 14, 22, 10, 11, 16, 4, 14, 22, 16, 2, 12, 2, 8, 3, 1, 2, 13, 2,
                17, 10, 15, 11, 15, 10, 10, 7, 4, 5, 15, 10, 18, 4, 13, 8, 6, 6, 20, 7, 0, 15, 20,
                8, 18, 21, 5, 7, 16, 1, 16, 15, 5, 12, 20, 19, 22, 12, 14, 1, 11, 12, 5, 4, 20, 3,
                12, 18, 21, 22, 10, 10, 16, 9, 5, 14, 22, 13, 21, 22, 2, 7, 8, 6, 3, 18, 18, 10,
                15, 14, 15, 8, 5, 21, 3, 20, 0, 4, 6, 4, 2, 7, 17, 14, 1, 4, 13, 8, 4, 22, 19, 13,
                20, 16, 21, 0, 2, 3, 2, 12, 19, 22, 15, 16, 19, 11, 9, 17, 8, 17, 6, 20, 4, 2, 7,
                14, 7, 4, 5, 9, 17, 5, 1, 3, 14, 11, 16, 1, 7, 19, 12, 14, 0, 22, 13, 0, 17, 8, 13,
                19, 11, 11, 0, 22, 19, 2, 0, 5, 9, 12, 14, 17, 22, 7, 4, 4, 15, 13, 20, 16, 16, 6,
                3, 16, 4, 6, 11, 13, 9, 20, 6, 17, 10, 17, 22, 6, 9, 16, 0, 12, 11, 18, 10, 11, 0,
                0, 15, 21, 7, 14, 22, 6, 6, 4, 13, 11, 6, 22, 8, 12, 19, 22, 10, 16, 14, 11, 5, 3,
                9, 1, 8, 20, 14, 18, 4, 22, 7, 6, 13, 20, 12, 4, 1, 8, 4, 1, 16, 4, 22, 8, 20, 4,
                11, 20, 16, 9, 12, 21, 1, 2, 3, 22, 6, 10, 16, 17, 6, 0, 7, 21, 0, 19, 14, 13, 11,
                0, 6, 21, 12, 21, 4, 7, 22, 18, 6, 20, 5, 22, 20, 14, 13, 9, 14, 4, 2, 19, 6, 10,
                14, 19, 22, 22, 18, 16, 1, 5, 16, 9, 18, 14, 4, 7, 5, 6, 4, 2, 3, 7, 4, 2, 11, 21,
                14, 3, 1, 12, 14, 11, 1, 18, 13, 4, 7, 20, 8, 8, 13, 3, 14, 18, 9, 10, 1, 2, 1, 9,
                6, 3, 5, 19, 0, 17, 18, 22, 8, 6, 10, 12, 11, 18, 16, 6, 20, 7, 7, 22, 8, 7, 16,
                17, 14, 12, 12, 18, 13, 21, 2, 16, 2, 0, 16, 3, 21, 0, 21, 14, 3, 4, 13, 1, 8, 12,
                20, 1, 1, 8, 3, 6, 18, 21, 3, 19, 18, 8, 3, 2, 21, 5, 9, 1, 18, 11, 19, 4, 12, 3,
                5, 6, 3, 1, 20, 21, 18, 14, 12, 6, 14, 12, 18, 6, 22, 3, 5, 4, 22, 9, 9, 0, 7, 19,
                1, 5, 8, 3, 20, 3, 14, 3, 19, 10, 10, 13, 17, 2, 14, 15, 7, 14, 1, 1, 20, 3, 22, 2,
                22, 15, 5, 6, 13, 15, 22, 19, 2, 21, 1, 14, 19, 5, 18, 7, 14, 15, 17, 14, 11, 3,
                13, 3, 10, 13, 7, 3, 11, 20, 0, 3, 8, 20, 5, 20, 14, 15, 1, 21, 11, 1, 9, 10, 14,
                19, 13, 20, 5, 15, 7, 17, 5, 21, 0, 14, 13, 16, 15, 11, 13, 18, 1, 6, 13, 5, 7, 15,
                16, 1, 0, 4, 10, 0, 9, 9, 5, 13, 6, 22, 1, 2, 17, 13, 17, 16, 9, 15, 1, 16, 10, 8,
                3, 18, 19, 18, 4, 19, 7, 1, 12, 9, 16, 15, 9, 2, 0, 20, 6, 6, 14, 22, 4, 2, 17, 15,
                13, 22, 17, 11, 5, 2, 13, 14, 22, 0, 6, 16, 15, 17, 22, 2, 9, 16, 12, 19, 15, 12,
                1, 9, 0, 14, 13, 10, 16, 14, 10, 22, 1, 11, 15, 10, 9, 19, 13, 15, 13, 22, 11, 15,
                11, 9, 3, 6, 18, 20, 13, 20, 1, 15, 22, 18, 18, 7, 18, 16, 22, 12, 4, 18, 9, 11, 6,
                1, 14, 19, 19, 0, 22, 5, 18, 3, 5, 10, 13, 15, 2, 4, 6, 2, 14, 11, 21, 5, 0, 11,
                14, 0, 19, 14, 1, 20, 4, 12, 1, 9, 5, 12, 15, 12, 3, 1, 2, 14, 22, 18, 20, 5, 1, 6,
                0, 5, 14, 12, 3, 0, 10, 22, 6, 4, 4, 22, 20, 9, 8, 14, 0, 2, 6, 14, 13, 0, 15, 18,
                10, 17, 8, 20, 21, 17, 3, 6, 2, 13, 4, 0, 15, 4, 22, 19, 20, 4, 3, 3, 4, 8, 5, 3,
                10, 6, 8, 7, 9, 18, 4, 12, 9, 9, 3, 15, 21, 2, 6, 4, 7, 12, 1, 15, 0, 18, 6, 5, 10,
                11, 12, 15, 15, 2, 12, 4, 11, 15, 3, 19, 22, 19, 22, 9, 10, 15, 17, 8, 17, 0, 15,
                15, 18, 12, 0, 1, 4, 10, 6, 21, 16, 5, 0, 20, 21, 21, 4, 22, 7, 10, 17, 13, 10, 7,
                7, 21, 13, 5, 3, 20, 5, 14, 5, 9, 9, 6, 3, 21, 6, 7, 15, 10, 1, 4, 15, 7, 16, 3,
                13, 14, 4, 21, 6, 14, 11, 18, 5, 8, 13, 20, 10, 10, 13, 17, 7, 0, 9, 21, 1, 3, 4,
                16, 16, 20, 5, 3, 5, 14, 22, 1, 20, 21, 11, 16, 13, 10, 21, 8, 8, 16, 12, 12, 8,
                12, 9, 6, 20, 15, 15, 12, 1, 15, 7, 11, 22, 4, 2, 7, 2, 3, 0, 8, 7, 10, 16, 16, 22,
                10, 21, 3, 13, 0, 10, 7, 17, 21, 0, 22, 4, 15, 6, 10, 14, 0, 7, 0, 22, 16, 11, 2,
                19, 15, 5, 2, 11, 13, 3, 14, 9, 19, 19, 21, 8, 19, 1, 21, 7, 17, 12, 18, 17, 7, 5,
                13, 6, 13, 5, 21, 14, 8, 4, 13, 21, 14, 0, 4, 15, 4, 7, 0, 11, 11, 0, 17, 1, 4, 4,
                18, 21, 19, 15, 21, 11, 2, 6, 18, 10, 10, 21, 11, 19, 13, 7, 2, 3, 12, 7, 16, 16,
                16, 16, 17, 22, 20, 20, 10, 13, 16, 19, 6, 1, 18, 19, 12, 3, 11, 7, 17, 16, 7, 17,
                1, 15, 18, 16, 11, 19, 1, 16, 3, 16, 9, 11, 18, 1, 20, 1, 20, 8, 13, 11, 3, 15, 13,
                15, 21, 13, 22, 22, 2, 6, 7, 21, 12, 6, 8, 18, 21, 11, 1, 0, 15, 21, 11, 15, 19,
                20, 4, 0, 7, 4, 18, 5, 4, 17, 11, 1, 17, 4, 12, 11, 22, 5, 5, 21, 14, 17, 7, 12, 7,
                17, 18, 17, 8, 3, 20, 3, 5, 1, 5, 3, 19, 6, 18, 20, 13, 17, 19, 3, 19, 17, 3, 20,
                10, 20, 3, 8, 16, 7, 8, 3, 22, 18, 12, 21, 10, 19, 12, 12, 16, 12, 0, 5, 12, 8, 19,
                20, 2, 5, 21, 18, 19, 1, 1, 20, 22, 20, 8, 11, 15, 17, 13, 20, 9, 16, 21, 4, 12, 2,
                1, 5, 4, 4, 17, 19, 0, 19, 9, 14, 12, 21, 16, 16, 8, 6, 10, 12, 0, 13, 12, 22, 11,
                1, 17, 6, 12, 19, 3, 13, 8, 13, 16, 6, 2, 6, 18, 9, 18, 17, 11, 20, 15, 2, 15, 22,
                17, 17, 11, 22, 1, 8, 12, 13, 10, 3, 5, 12, 21, 14, 10, 17, 19, 17, 15, 11, 20, 20,
                21, 19, 14, 6, 8, 17, 1, 18, 13, 22, 11, 3, 1, 21, 6, 10, 22, 7, 5, 0, 3, 12, 14,
                8, 3, 4, 9, 9, 21, 11, 10, 9, 11, 9, 4, 19, 7, 10, 22, 13, 2, 2, 22, 19, 9, 16, 21,
                1, 9, 13, 16, 11, 22, 17, 11, 12, 22, 0, 13, 6, 8, 1, 0, 0, 4, 18, 3, 1, 12, 5, 5,
                3, 9, 6, 6, 17, 9, 4, 19, 14, 0, 9, 1, 19, 8, 0, 14, 18, 18, 9, 16, 15, 10, 10, 11,
                12, 17, 19, 11, 15, 0, 10, 22, 5, 21, 21, 2, 20, 15, 5, 1, 7, 7, 6, 15, 12, 6, 13,
                20, 4, 21, 10, 15, 1, 13, 13, 9, 21, 18, 0, 5, 9, 20, 21, 7, 22, 2, 4, 10, 10, 0,
                11, 4, 3, 15, 12, 20, 9, 9, 11, 15, 15, 21, 18, 1, 4, 11, 10, 0, 6, 18, 6, 11, 20,
                3, 11, 11, 13, 16, 9, 1, 1, 10, 6, 20, 14, 2, 7, 4, 18, 9, 12, 16, 22, 20, 9, 19,
                2, 8, 3, 4, 15, 18, 3, 14, 7, 15, 1, 15, 8, 19, 5, 8, 2, 22, 19, 13, 18, 0, 19, 12,
                17, 21, 22, 19, 6, 12, 2, 2, 20, 18, 21, 18, 18, 7, 19, 16, 11, 1, 21, 10, 5, 12,
                15, 18, 17, 10, 0, 20, 20, 19, 6, 6, 15, 8, 5, 21, 11, 0, 6, 10, 13, 8, 20, 7, 2,
                11, 11, 3, 14, 11, 9, 8, 15, 0, 3, 22, 0, 12, 6, 0, 12, 21, 11, 8, 7, 5, 0, 6, 4,
                15, 18, 9, 21, 5, 4, 20, 14, 5, 9, 19, 3, 11, 12, 22, 7, 18, 21, 9, 17, 3, 2, 11,
                15, 6, 19, 2, 3, 8, 10, 7, 6, 8, 13, 2, 2, 9, 20, 3, 7, 5, 13, 13, 9, 0, 18, 21, 1,
                22, 6, 7, 18, 5, 14, 14, 19, 0, 6, 4, 3, 19, 20, 22, 20, 7, 22, 22, 2, 11, 16, 16,
                20, 17, 13, 20, 1, 21, 17, 22, 3, 4, 6, 7, 9, 11, 21, 17, 13, 21, 8, 18, 22, 4, 19,
                8, 7, 11, 13, 1, 10, 14, 18, 17, 14, 15, 22, 3, 14, 20, 15, 3, 22, 15, 22, 12, 12,
                18, 14, 16, 14, 21, 17, 1, 16, 12, 8, 2, 3, 6, 4, 9, 7, 5, 3, 15, 21, 14, 0, 20, 9,
                7, 19, 10, 4, 5, 11, 8, 19, 0, 22, 7, 3, 22, 8, 12, 5, 22, 4, 15, 11, 2, 4, 13, 2,
                4, 9, 7, 5, 17, 14, 19, 7, 14, 4, 8, 17, 18,
            ],
            23,
            &[
                2432543987, 2276527024, 3303037865, 3040931222, 124530589, 2075117878, 373679459,
                1719887982, 927313221, 2350422398, 2410320624, 2621140451, 898468917, 1050104246,
                3314171259, 2202176491, 3934792658, 1945547354, 752254890, 68461992, 1501238000,
                1916284043, 2385775512, 1516051704, 1572036976, 177218342, 4078653370, 3944978370,
                3993325163, 35949890, 28445911, 2989181316, 3711931852, 3969713797, 853651131,
                1048673303, 164841509, 2591275506, 2676954415, 910103144, 2881135462, 1328960554,
                3814928621, 1570089463, 3936037561, 3706141104, 3622208665, 1736839730, 818356637,
                903565935, 3997690004, 2013074628, 245593157, 2604723513, 2614342663, 1992293635,
                1023889602, 2910391935, 1146365901, 3356081083, 3777393537, 2858374037, 2863919743,
                1056257782, 809068644, 672989257, 3747695201, 2522210474, 525888082, 1848521587,
                2468974825, 653259135, 1975306186, 2456732937, 1299155795, 1794805560, 569248853,
                102439111, 256768309, 4037408132, 77496396, 1983342746, 2296048037, 3348374173,
                1213807447, 1373478223, 2235179986, 1490259265, 2112796854, 2547714374, 869600236,
                2852487281, 2064710386, 447426889, 767950649, 3612583010, 1780520436, 3627700369,
                3905832623, 412784141, 2186300632, 1687791207, 3268878142, 2439345575, 544593515,
                2488911021, 1860939518, 3160552347, 3520538477, 2254621553, 3817356405, 1093773391,
                104168916, 2026826706, 3008411949, 731612881, 400526524, 1319326349, 994927338,
                1642377367, 89069495, 4052976262, 407108435, 1706274710, 3618323759, 2999112327,
                3472562229, 594407960, 4257849311, 1499146458, 3343189692, 3332254596, 3845543202,
                3993385369, 1501681435, 608654406, 2482649339, 3990535200, 1449270641, 4016403422,
                342445879, 4228145020, 4030280967, 891466632, 1454359846, 1886612732, 3712575231,
                3934358913, 1100305416, 1249102960, 2133206589, 3528060174, 1115362965, 871488833,
                2959657656, 1791653312, 1587540254, 1527651781, 2515721439, 2549398069, 1442586984,
                3746591659, 1219966156, 1140399539, 3198235519, 2830212264, 3671456437, 2604181784,
                2742431758, 3688451693, 1352937420, 2958171794, 3484353914, 1323974235, 635848732,
                3435591916, 1147399220, 3055499602, 3459025755, 1441809200, 3288677176, 1356946570,
                2922613843, 3435973582, 3169578850, 2010271922, 2926310610, 4220927069, 3147037801,
                3177585452, 3767093688, 3797368144, 2993412434, 52467448, 2638704998, 500593776,
                435767039, 75611890, 173252622, 369536399, 3001199800, 1696483020, 403435509,
                2676770836, 135469730, 4294502870, 3467161889, 487127268, 144987877, 3830430272,
                4176107046, 547748139, 2535372546, 3407597209, 4273742947, 3591467776, 869590956,
                1405915818, 3698534773, 260295134, 4044017583, 4035779488, 463440960, 4200854080,
                657217412, 3110364826, 2029115239, 2200955452, 779832230, 1588101134, 1711375700,
                3878103627, 1709250626, 2858094475, 509701948, 717581521, 3068900665, 1481026334,
                1218138746, 4054497212, 3753193645, 2200558797, 2971038189, 3966112697, 558197857,
                2787894924, 744571221, 1568885442, 2788507093, 1552043446, 172898599, 1046802271,
                845471067, 3408324900, 3882264867, 1306206395, 1165216268, 1191839803, 1527610198,
                4188064533, 3649876341, 2133981933, 1122433329, 166725368, 2113942545, 262674194,
                2983616752, 3664915710, 2528883697, 72691628, 1094952676, 3781397836, 2543857595,
                1262396255, 1541616154, 695881242, 4267754168, 2929769840, 1150669979, 2444044368,
                2085087061, 1011358281, 414070849, 1990902057, 2504781194, 762251676, 2992215331,
                321648205, 749759701, 2683064829, 371937346, 1916620629, 3504232544, 2210693036,
                1759776303, 1252482988, 2327728963, 855162264, 1839954590, 1657033791, 790197996,
                232471839, 2461249283, 117939746, 560149963, 1044762152, 1013702332, 932681941,
                1205754842, 1751214724, 860371808, 3898533721, 39404571, 1091417891, 593738330,
                3258375503, 4277934029, 688697039, 4083389785, 1292128636, 1251798962, 235181476,
                3030160145, 2367582304, 2609657981, 3483636249, 2810692598, 1851185576, 2778801808,
                1690833980, 4279865527, 2021420402, 1898436492, 2626314401, 3873432257, 1672600024,
                2803701169, 66336423, 1619175812, 1947174922, 427990001, 2831016152, 3944679881,
                1055966210, 1089542534, 2005082069, 1971269850, 1924433077, 4217952611, 2749232796,
                1481402693, 3203808543, 3753215487, 1988955035, 4002936550, 1944840595, 2089249464,
                226215659, 1145745745, 2617193917, 2071905246, 2716825521, 756996862, 803384898,
                2498333605, 3766458014, 2625315151, 1231183373, 3407435990, 2004334218, 136726326,
                3694458272, 439206563, 1697431856, 109034171, 3129361133, 2148702911, 1603279218,
                665749557, 2453579770, 1778010870, 3980259964, 3820850075, 423551518, 4244040519,
                771818147, 2473197470, 2184490663, 66706499, 373610607, 2507344693, 1182035514,
                2294279416, 348192110, 2147447089, 1429312800, 3521861641, 2941695755, 2373618812,
                3122839309, 3164597812, 4174895272, 3446714428, 4179964099, 3369606920, 3245057821,
                512812868, 3801957629, 3866197478, 1983232922, 1164421658, 3492043893, 4102973269,
                3482785988, 3613848786, 994234819, 3925964490, 2520653440, 306189184, 3378720004,
                325986226, 663667879, 2399249015, 716637187, 2390212162, 3760061962, 582012888,
                3300637312, 3552019520, 2310968716, 1848698860, 2484373755, 729280744, 833802096,
                3737502652, 3910738649, 2057433799, 2248378565, 268720583, 3069947043, 3763100719,
                3510420884, 1101435572, 462921272, 3929831289, 348032532, 3764507470, 3983733230,
                2585114124, 2116017165, 3240263985, 1205053072, 1665443971, 2423786772, 1138900192,
                2363229724, 1887946492, 3218510326, 3268927353, 1385744919, 2498515502, 1516552001,
                3939009406, 3701533621, 913382028, 3344976363, 3384856500, 3870149157, 1329228053,
                1291707300, 3899335066, 3940597462, 3153019193, 2408594961, 2671044621, 3551857958,
                2209058090, 3493940212, 222503531, 2114494914, 565799347, 1971473416, 4277831935,
                2358519001, 490963986, 3852565277, 984935336, 1133370208, 1617043634, 2857541004,
                832503739, 108040238, 3880146835, 2807229258, 3575769310, 2774854680, 868093555,
                2575005082, 2956555327, 3737955779, 4249638030, 1940414434, 3753480496, 49729337,
                1193299683, 3201089034, 420648789, 3352785498, 288688721, 1521187011, 1386674633,
                1364869610, 885037528, 1252279155, 3649552262, 2152649024, 3204905942, 1783619473,
                2861412105, 3540375905, 3746939744, 156102264, 3466170698, 11866214, 1012760070,
                1958489233, 3665112630, 3509540297, 2512443114, 2128112533, 1628693203, 2042312955,
                1656593755, 2564164318, 3270705760, 1680356097, 5025885, 2796521567, 2263503167,
                2420040603, 3394606541, 214460118, 992057652, 3266983797, 2159688810, 1993456089,
                4175543350, 3963528935, 2513264991, 3658064350, 156097811, 2826997767, 865018090,
                1321677484, 2168436280, 1124406880, 3231042910, 1275491774, 9015246, 3130741698,
                2078413139, 1824585545, 1955698270, 1768370940, 613758713, 3783594115, 2662841648,
                3623245683, 3253698278, 4191335005, 3430738147, 3732087559, 237862677, 3240225475,
                2509625707, 3371804946, 561137032, 2487087488, 566174513, 55477126, 1492417125,
                3600485779, 3489229753, 1660001711, 903932105, 2666760808, 3267530948, 1804351992,
                2931928329, 1701287766, 847625244, 2066564451, 2319056965, 2848820614, 854672647,
                535121151, 2288435608, 2524958641, 3296694570, 1364335301, 709714126, 2360287547,
                4106461964, 329631027, 849263407, 2311850705, 1653642573, 2959033037, 3845765450,
                2411305922, 1012439121, 974968898, 3631461599, 777873009, 920353902, 725744567,
                1725097098, 333882177, 3584334013, 3078291222, 3960551469, 2531623496, 1211920485,
                2390442828, 2791068256, 1910957250, 1632728636, 3797597081, 3611429309, 1566721977,
                2509740862, 4176296323, 590851749, 3457912421, 2303299273, 2528501850, 889922996,
                536952148, 2891183341, 3609423256, 2878185261, 3218672709, 956031557, 1687988839,
                920703413, 4209253359, 1815357979, 3709650040, 1017145721, 1810276897, 1020365743,
                2313969064, 1539434460, 4255594610, 1458431491, 680535340, 1647259761, 1004107429,
                2415667132, 31092371, 935775706, 396967039, 27911921, 2657305445, 2248738175,
                2497727679, 1483667419, 1951176417, 2145607203, 3224667691, 2438573057, 2942868990,
                2027000102, 2913871289, 3685393236, 2425696100, 782260165, 483022313, 3940181670,
                3798075211, 1219099101, 203528657, 529875562, 276233372, 264759552, 51482773,
                4105461152, 708360569, 559014934, 735004662, 2304945262, 2287817402, 1399360640,
                4034225857, 1458299505, 1242064138, 3475741745, 2971620406, 2364951756, 1272349036,
                2302805830, 298664638, 3662585622, 2397537734, 3026808232, 1114868390, 673264043,
                3318150460, 1138305753, 3797784655, 2881309047, 174048567, 3800745317, 3055674757,
                2607943906, 944193168, 2003534871, 1434769036, 458792317, 2282245931, 636767187,
                1532218286, 724269210, 2874117375, 1439654243, 698007041, 2836800209, 1025277967,
                1527769433, 2515862986, 1373613061, 287099492, 2691879042, 2428315581, 2367162320,
                1217630768, 952926109, 3408276082, 3496973185, 3089422616, 1048409122, 707017659,
                2780132225, 2844071654, 1836018831, 3441433162, 624841709, 3116228289, 1956737587,
                3778477996, 336613926, 2937889239, 384836527, 1341826476, 3712358886, 2306774444,
                1341535315, 1777669116, 1199737655, 266426998, 2697021776, 2817808455, 2023340729,
                3851312363, 3980251913, 3207784360, 333886678, 689050764, 331336434, 3431528012,
                654836012, 3282279155, 249795473, 4101667665, 994019464, 1551695192, 919619198,
                2822963861, 1058136685, 347412103, 8043636, 548175678, 160303852, 4191967974,
                114867036, 2223903475, 3083148884, 2624932033, 2791621092, 2277501601, 2800576686,
                1130603172, 566089914, 1444023227, 1500498349, 150928386, 2908319814, 2257652416,
                1193412204, 3157603623, 4288978904, 2684571944, 2504105716, 3900496848, 344319926,
                2148470734, 3941199614, 801816176, 2970106595, 2551130561, 1139061079, 1765357801,
                2801395705, 1452414774, 1670219825, 2423177094, 321332024, 2032854264, 2554376386,
                537964156, 1300131358, 2011623640, 4208848750, 2877603209, 896201049, 954025445,
                1801424491, 3220899647, 3629652945, 1882282457, 1857761982, 2758423637, 787777871,
                225484364, 713108959, 2092789353, 2948984687, 269006096, 76419691, 569618318,
                1650706474, 4007329961, 270502651, 708871540, 4193563743, 2210670738, 2545734314,
                2838989169, 2119352897, 3507676809, 192990507, 447022484, 1538552337, 3536011095,
                1971356807, 1997639220, 4240677277, 2042803185, 747190487, 690822786, 3425774253,
                1970444075, 1519418519, 2502800132, 456092197, 3089519109, 2494760604, 1753613710,
                3680543205, 1225101795, 2697514748, 2933361013, 4175549155, 2216849000, 3929431645,
                4230737093, 2641643251, 4199674746, 1962717842, 3676951090, 2505693179, 850730013,
                3866789620, 152159307, 2179605002, 687124397, 2450265390, 445267719, 1990658425,
                2317584166, 1464123246, 2333847393, 3958580546, 589512689, 1683476060, 528323569,
                4040487824, 1531042664, 2122382129, 3943959015, 3860235235, 1153301333, 566968208,
                2417583538, 1103445908, 1261825307, 2550203026, 1658486984, 3038335860, 1841338780,
                3360364021, 2660553389, 2924187195, 4241529036, 2271736721, 3765073952, 2580215584,
                1438318228, 2787276148, 454821732, 355379320, 2482867526, 1411322874, 4211251252,
                1898863052, 471267359, 3485276742, 1285961552, 3923197508, 3768783839, 4025502468,
                18284057, 2673735163, 2221768523, 681186346, 88325179, 2889050541, 1256119319,
                1858502698, 2398900641, 976659823, 2961773032, 3868539432, 2535077451, 1763974232,
                2135036948, 1399244834, 1152496492, 2401894286, 1267776170, 2133978424, 3666460581,
                2519353074, 2831901651, 3888910459, 1900327248, 1654981227, 1101450299, 3445846720,
                3449825460, 3791261482, 432004847, 3066863554, 2698800432, 1641108, 707956864,
                1870786679, 3055496006, 3728988665, 398291770, 2212074450, 2155438841, 3431922058,
                3971811313, 1719261068, 1735009605, 4213793259, 58186599, 3855304568, 3067682717,
                228831702, 3278881736, 4153465006, 1952694150, 3651972132, 3534641986, 1121445172,
                169071915, 4212735773, 3505561766, 1395317315, 736925405, 3125143948, 3297296490,
                3127331986, 3616368376, 2056503535, 788835758, 788439625, 1630085743, 2747839891,
                2343131865, 1081156327, 3120208131, 394640347, 3903778496, 4252716449, 2283235409,
                483957986, 83131712, 518164125, 3367314863, 3912497177, 1979817847, 2541067002,
                3406633266, 2525836540, 2905407837, 927633307, 357014464, 1712382708, 2975844776,
                3790058589, 1514112964, 366170767, 328664775, 4185303074, 1820638444, 2924908138,
                2047784738, 1987583135, 2372860973, 3609968385, 3143257714, 3427362641, 3072122247,
                1141840336, 2310485042, 368865492, 4065735384, 3981077989, 2624946734, 3489009884,
                2263206997, 2463456412, 2712152858, 1734251492, 2655931795, 3696764806, 3404614825,
                4011135114, 1018397103, 572166022, 2062711851, 2569156569, 3460399925, 1702645790,
                1098604909, 2069896475, 3646476155, 820160451, 147337310, 1765191975, 60191201,
                626121624, 2726686045, 2538949724, 187866767, 2388425781, 935215588, 871611286,
                3906412946, 1246311689, 3306250646, 4170514946, 2323420983, 2057780630, 628784453,
                1410213509, 372936645, 501063446, 175500703, 258466122, 62857995, 3386238628,
                2083632783, 2605300978, 1118667897, 1446000958, 79053812, 3654798093, 1881412132,
                4168572524, 1027075259, 1028158997, 1654169038, 2157952826, 2032022698, 1151156532,
                1065611895, 3911407574, 2900828134, 1637425467, 2817709633, 1736551320, 3741593941,
                3202948790, 1599337068, 3398371980, 2224875086, 1229925548, 1303814094, 3574632434,
                3493928291, 884975947, 1246641330, 2088204280, 2837698631, 1824887371, 1450894602,
                266444701, 3075497931, 3834303497, 3027392031, 2531258637, 2642373781, 1193002026,
                1767288754, 1311625744, 1001761926, 3062663859, 52529591, 3745995278, 438887272,
                700662189, 4212368956, 2107562915, 319883829, 215035253, 1488253283, 1301013358,
                433922795, 1753607220, 794705122, 1689562210, 3154133389, 2179734736, 2971286634,
                2392964366, 2792619323, 1432651454, 3418370805, 3131519322, 3349726128, 2474689584,
                466365332, 811867264, 3052861325, 4097469989, 93032001, 3926176274, 2350247983,
                1674779175, 291602062, 2846268401, 3277208194, 1497830566, 3625189672, 9072038,
                2951339513, 1792718546, 1776258935, 1029608682, 1412608523, 3989131994, 3101063512,
                1207282153, 2951261525, 3563115848, 1682346995, 4283726597, 3927997318, 4070406195,
                3269446148, 2053006428, 3883337744, 2238918050, 3463951652, 3364024419, 2879606446,
                2715677768, 3398302025, 1565164133, 3468031718, 3995385067, 2354095405, 70728059,
                1501387287, 531457926, 4071000954, 2691373653, 130456474, 162613531, 3278227001,
                1871081118, 2108414937, 1956838987, 3201051305, 1099789655, 2688103265, 3376545054,
                4194764973, 3048141369, 92313010, 2714344608, 2836237280, 2306143043, 157665707,
                4202292074, 1667642970, 2709322690, 33398946, 1993933511, 876610905, 4199861858,
                4070029492, 4112921551, 4157395415, 1744841967, 375479353, 782950108, 3732824359,
                3468562129, 1534389810, 3406467972, 231940487, 27701910, 379953968, 1660493409,
                409076658, 839620823, 2991390226, 3717064548, 2195983177, 1378317571, 2131651037,
                264521891, 107725272, 2543867771, 1340163093, 1806455525, 1349163961, 2698063614,
                1300591146, 2489572412, 1792603038, 8170771, 1532472015, 4089778540, 1087892505,
                4011589612, 2513998273, 2753333194, 2374394, 3072400605, 2057302330, 206942689,
                930325233, 1494197348, 2396885015, 3519563996, 1389464877, 212652146, 3546128890,
                1601747989, 676352508, 2556438908, 3274662607, 595702540, 3961163985, 214798108,
                158276567, 2000671252, 1909961436, 3608100180, 1725922971, 1921296001, 3248181224,
                2681071530, 4234758481, 3214059028, 1675447675, 2089523164, 470493498, 340308561,
                4095769841, 3045163996, 1700180965, 1833518916, 3218401197, 3764107983, 760622304,
                2715854634, 2566771885, 2855925414, 2596118107, 2718892126, 2919661660, 1777435166,
                704222621, 4188226574, 1287796296, 32153441, 2088095808, 2504036947, 481481625,
                1981638127, 2659395414, 4019699467, 3022239223, 1432279555, 3716343314, 2101658277,
                2538588325, 2228739909, 325848107, 3728753696, 226572131, 70936434, 2326533081,
                4197100817, 2130486321, 3647628851, 2365959588, 146012,
            ],
        );
        // - out_len_hi == 0 in limbs_from_digits_small_base_divide_and_conquer
        // - out_len_lo == 0 in limbs_from_digits_small_base_divide_and_conquer
        test(&[10; 739], &[0; 7100], 10, &[0; 369]);
    }
    #[cfg(not(feature = "32_bit_limbs"))]
    {
        test(
            &[
                424216987022722268,
                8103283177193628673,
                4470414499436387861,
                13801449255692046935,
                9547961461378503966,
                5606458381418993750,
                17091516772434375724,
                11977748006393229058,
                6812658461074707191,
                12015828441561040006,
            ],
            &[
                51, 153, 87, 70, 104, 138, 111, 23, 85, 119, 176, 8, 65, 118, 3, 156, 55, 121, 87,
                162,
            ],
            182,
            &[4980635571795936200, 9374876857207216767, 1331096],
        );
        test(
            &[
                5518555633093191759,
                12836059490899113620,
                12661099089298342811,
                1992484047621263128,
                3777109748949388481,
                4126346840817077875,
                7608898082196224717,
                6664909716221346590,
                679023819841892642,
            ],
            &[41, 16, 48, 34, 6, 51, 35, 56, 47, 44, 44, 20, 51, 39, 36, 33, 4, 14, 25, 3],
            61,
            &[16759895520031570107, 18665423842034105],
        );
        // - xs_len <= len_lo in limbs_from_digits_small_base_divide_and_conquer
        // - xs_len >= SET_STR_DC_THRESHOLD in limbs_from_digits_small_base_divide_and_conquer
        test(
            &[10; 10],
            &[
                2, 3, 0, 1, 0, 1, 0, 2, 0, 0, 1, 2, 3, 2, 0, 0, 3, 0, 3, 1, 3, 2, 4, 4, 2, 1, 0, 1,
                1, 3, 1, 2, 2, 4, 1, 4, 1, 0, 2, 3, 1, 1, 4, 4, 0, 3, 3, 3, 0, 0, 0, 3, 1, 0, 1, 1,
                3, 1, 1, 0, 3, 3, 2, 4, 4, 4, 3, 4, 2, 0, 1, 1, 3, 4, 4, 3, 0, 4, 3, 0, 2, 2, 3, 4,
                3, 0, 4, 2, 2, 0, 3, 4, 0, 1, 0, 4, 3, 1, 1, 2, 1, 2, 0, 1, 3, 0, 2, 2, 1, 1, 2, 2,
                2, 0, 4, 3, 3, 3, 1, 2, 2, 2, 2, 0, 1, 0, 2, 4, 4, 4, 1, 3, 2, 2, 0, 1, 3, 1, 1, 4,
                1, 3, 4, 2, 3, 0, 1, 2, 2, 0, 0, 4, 1, 1, 1, 3, 1, 3, 0, 2, 0, 2, 0, 2, 1, 1, 2, 4,
                0, 3, 2, 3, 0, 4, 3, 1, 2, 4, 4, 1, 3, 3, 3, 4, 2, 0, 3, 0, 3, 4, 3, 1, 4, 3, 3, 2,
                2, 1, 1, 1, 0, 3, 2, 0, 4, 1, 1, 1, 0, 3, 0, 3, 4, 4, 4, 4, 4, 0, 4, 2, 1, 4, 4,
            ],
            5,
            &[
                277482782106898878,
                8398439881293320527,
                6237275848328729287,
                17623578066584208310,
                2887235280090375858,
                8886426738771697150,
                6137418971026265383,
                15927966713276205188,
                28,
            ],
        );
    }
}

fn limbs_from_digits_small_base_properties_helper<T: PrimitiveUnsigned>()
where
    Limb: WrappingFrom<T>,
    Natural: From<T>,
{
    let mut config = GenConfig::new();
    config.insert("mean_stripe_n", 4 << Limb::LOG_WIDTH);
    config.insert("mean_digit_count_n", 2048);
    config.insert("mean_excess_limb_count_n", 32);
    unsigned_vec_unsigned_vec_unsigned_triple_gen_var_3::<T, Limb>().test_properties_with_config(
        &config,
        |(mut out, xs, base)| {
            let t_base = T::exact_from(base);
            assert_eq!(
                limbs_from_digits_small_base(&mut out, &xs, base).is_some(),
                xs.iter().all(|&x| x < t_base)
            );
        },
    );

    unsigned_vec_unsigned_vec_unsigned_triple_gen_var_2::<T, Limb>().test_properties_with_config(
        &config,
        |(mut out, xs, base)| {
            let old_out = out.clone();
            let out_len = limbs_from_digits_small_base(&mut out, &xs, base).unwrap();
            verify_limbs_from_digits_small_base(&old_out, &xs, base, out_len, &out, false);
        },
    );
}

#[test]
fn limbs_from_digits_small_base_properties() {
    apply_fn_to_unsigneds!(limbs_from_digits_small_base_properties_helper);
}

#[test]
fn test_from_digits_desc_basecase() {
    fn test_ok(xs: &[u8], base: Limb, n: &str) {
        let n = Natural::from_str(n).unwrap();
        assert_eq!(from_digits_desc_basecase(xs, base).unwrap(), n);
        assert_eq!(
            from_digits_desc_naive_primitive(xs, u8::exact_from(base)).unwrap(),
            n
        );
    }
    test_ok(&[0], 9, "0");
    test_ok(&[1], 9, "1");
    test_ok(&[2, 0, 0, 2, 1, 1, 0, 0, 1, 1, 0], 3, "123456");
    test_ok(&[0, 0, 2, 0, 0, 2, 1, 1, 0, 0, 1, 1, 0], 3, "123456");
    test_ok(
        &[
            73, 23, 120, 45, 108, 147, 113, 90, 129, 11, 86, 0, 102, 81, 22, 17, 32, 121, 29, 82,
            27, 25, 39, 9, 139, 59, 51, 13, 44, 3, 37, 104, 41, 40, 87, 66, 83, 146, 40, 132, 15,
            50, 66, 80, 73, 61, 75, 25, 78, 34, 128, 86, 68, 26, 62, 135, 21, 9, 27, 74, 12, 39,
            67, 15, 41, 50, 144, 35, 98, 140, 124, 132, 10, 37, 101, 99, 14,
        ],
        150,
        "176685760608531978263731938997517835399219565848609872558191310084297042793489341607854254\
        709347189745346071475819587247558105442098729883999424898641968281841439662364",
    );

    fn test_err(xs: &[u8], base: Limb) {
        assert!(from_digits_desc_basecase(xs, base).is_none());
        assert!(from_digits_desc_naive_primitive(xs, u8::exact_from(base)).is_none());
    }
    test_err(&[10, 11, 12], 10);
}

fn from_digits_desc_basecase_fail_helper<T: ConvertibleFrom<Limb> + PrimitiveUnsigned>()
where
    Limb: WrappingFrom<T>,
{
    assert_panic!(from_digits_desc_basecase::<T>(&[], 0));
    assert_panic!(from_digits_desc_basecase::<T>(&[], 1));
}

#[test]
fn from_digits_desc_basecase_fail() {
    apply_fn_to_unsigneds!(from_digits_desc_basecase_fail_helper);
}

fn from_digits_desc_basecase_properties_helper<T: ConvertibleFrom<Limb> + PrimitiveUnsigned>()
where
    Limb: SaturatingFrom<T> + WrappingFrom<T>,
    Natural: From<T>,
{
    let mut config = GenConfig::new();
    config.insert("mean_digit_count_n", 32);
    config.insert("mean_length_n", 32);
    unsigned_vec_unsigned_pair_gen_var_12::<T, Limb>().test_properties_with_config(
        &config,
        |(xs, base)| {
            let t_base = T::exact_from(base);
            assert_eq!(
                from_digits_desc_basecase(&xs, base).is_some(),
                xs.iter().all(|&x| x < t_base)
            );
        },
    );

    unsigned_vec_unsigned_pair_gen_var_5::<T, Limb>().test_properties_with_config(
        &config,
        |(xs, base)| {
            let n = from_digits_desc_basecase(&xs, base);
            assert_eq!(
                from_digits_desc_naive_primitive(&xs, T::exact_from(base)),
                n
            );
        },
    );
}

#[test]
fn from_digits_desc_basecase_properties() {
    apply_fn_to_unsigneds!(from_digits_desc_basecase_properties_helper);
}

#[test]
fn test_from_digits_asc_limb() {
    fn test_ok(xs: &[u32], base: Limb, n: &str) {
        let n = Natural::from_str(n).unwrap();
        assert_eq!(from_digits_asc_limb(xs.iter().copied(), base).unwrap(), n);
        assert_eq!(
            from_digits_desc_naive_primitive(
                &xs.iter().copied().rev().collect_vec(),
                u32::exact_from(base)
            )
            .unwrap(),
            n
        );
    }
    test_ok(&[], 9, "0");
    test_ok(&[0], 9, "0");
    test_ok(&[1], 9, "1");
    test_ok(&[0, 1, 1, 0, 0, 1, 1, 2, 0, 0, 2], 3, "123456");
    test_ok(&[0, 1, 1, 0, 0, 1, 1, 2, 0, 0, 2, 0, 0], 3, "123456");
    test_ok(&[2, 4, 6, 4, 2], 8, "10658");
    test_ok(&[789, 456, 123], 1000, "123456789");
    test_ok(
        &[
            14, 99, 101, 37, 10, 132, 124, 140, 98, 35, 144, 50, 41, 15, 67, 39, 12, 74, 27, 9, 21,
            135, 62, 26, 68, 86, 128, 34, 78, 25, 75, 61, 73, 80, 66, 50, 15, 132, 40, 146, 83, 66,
            87, 40, 41, 104, 37, 3, 44, 13, 51, 59, 139, 9, 39, 25, 27, 82, 29, 121, 32, 17, 22,
            81, 102, 0, 86, 11, 129, 90, 113, 147, 108, 45, 120, 23, 73,
        ],
        150,
        "176685760608531978263731938997517835399219565848609872558191310084297042793489341607854254\
        709347189745346071475819587247558105442098729883999424898641968281841439662364",
    );
    test_ok(
        &[302, 2359, 150, 1581, 2859, 1843, 2403, 2039, 27, 1598],
        3543,
        "140578615308984594421852296827289425",
    );
    test_ok(
        &[1187762660, 83185796, 570510527, 293681571, 1518538399, 1153431348],
        1525385058,
        "9525530906278526930121302445905223566866929778026945776",
    );
    test_ok(
        &[
            1535724679, 31832127, 1494323667, 798341655, 1169948427, 1204675417, 1214721934,
            1599722999, 1842176041, 1659733906, 824969631, 1046252719,
        ],
        1895460450,
        "118713086621740109729266002848273602283691336549480820142301294962906804654789322428658177\
        1782212965464227294329",
    );

    fn test_err(xs: &[u32], base: Limb) {
        assert!(from_digits_asc_limb(xs.iter().copied(), base).is_none());
        assert!(
            from_digits_desc_naive_primitive(
                &xs.iter().copied().rev().collect_vec(),
                u32::exact_from(base)
            )
            .is_none()
        );
    }
    test_err(&[12, 11, 10], 10);
}

fn from_digits_asc_limb_fail_helper<T: ConvertibleFrom<Limb> + PrimitiveUnsigned>()
where
    Limb: TryFrom<T> + WrappingFrom<T>,
    Natural: From<T> + PowerOf2Digits<T>,
{
    assert_panic!(from_digits_asc_limb::<_, T>(empty(), 0));
    assert_panic!(from_digits_asc_limb::<_, T>(empty(), 1));
}

#[test]
fn from_digits_asc_limb_fail() {
    apply_fn_to_unsigneds!(from_digits_asc_limb_fail_helper);
}

fn from_digits_asc_limb_properties_helper<T: ConvertibleFrom<Limb> + PrimitiveUnsigned>()
where
    Limb: ExactFrom<T> + SaturatingFrom<T> + WrappingFrom<T>,
    Natural: From<T> + PowerOf2Digits<T>,
{
    let mut config = GenConfig::new();
    config.insert("mean_digit_count_n", 32);
    unsigned_vec_unsigned_pair_gen_var_12::<T, Limb>().test_properties_with_config(
        &config,
        |(xs, base)| {
            let t_base = T::exact_from(base);
            assert_eq!(
                from_digits_asc_limb(xs.iter().copied(), base).is_some(),
                xs.iter().all(|&x| x < t_base)
            );
        },
    );

    unsigned_vec_unsigned_pair_gen_var_5::<T, Limb>().test_properties_with_config(
        &config,
        |(xs, base)| {
            let n = from_digits_asc_limb(xs.iter().copied(), base);
            assert_eq!(
                from_digits_desc_naive_primitive(
                    &xs.into_iter().rev().collect_vec(),
                    T::exact_from(base)
                ),
                n
            );
        },
    );
}

#[test]
fn from_digits_asc_limb_properties() {
    apply_fn_to_unsigneds!(from_digits_asc_limb_properties_helper);
}

#[test]
fn test_from_digits_desc_limb() {
    fn test_ok(xs: &[u32], base: Limb, n: &str) {
        let n = Natural::from_str(n).unwrap();
        assert_eq!(from_digits_desc_limb(xs.iter().copied(), base).unwrap(), n);
        assert_eq!(
            from_digits_desc_naive_primitive(xs, u32::exact_from(base)).unwrap(),
            n
        );
    }
    test_ok(&[], 9, "0");
    // - Some(log_base) != base.checked_log_base_2()
    // - base < 256
    test_ok(&[0], 9, "0");
    test_ok(&[1], 9, "1");
    test_ok(&[2, 0, 0, 2, 1, 1, 0, 0, 1, 1, 0], 3, "123456");
    test_ok(&[0, 0, 2, 0, 0, 2, 1, 1, 0, 0, 1, 1, 0], 3, "123456");
    // - Some(log_base) = base.checked_log_base_2()
    test_ok(&[2, 4, 6, 4, 2], 8, "10658");
    // - base >= 256
    // - from_digits_desc_divide_and_conquer_limb; power_index == 0 || b <
    //   FROM_DIGITS_DIVIDE_AND_CONQUER_THRESHOLD
    // - from_digits_desc_divide_and_conquer_limb; base <= SQRT_MAX_LIMB
    test_ok(&[123, 456, 789], 1000, "123456789");
    test_ok(
        &[
            73, 23, 120, 45, 108, 147, 113, 90, 129, 11, 86, 0, 102, 81, 22, 17, 32, 121, 29, 82,
            27, 25, 39, 9, 139, 59, 51, 13, 44, 3, 37, 104, 41, 40, 87, 66, 83, 146, 40, 132, 15,
            50, 66, 80, 73, 61, 75, 25, 78, 34, 128, 86, 68, 26, 62, 135, 21, 9, 27, 74, 12, 39,
            67, 15, 41, 50, 144, 35, 98, 140, 124, 132, 10, 37, 101, 99, 14,
        ],
        150,
        "176685760608531978263731938997517835399219565848609872558191310084297042793489341607854254\
        709347189745346071475819587247558105442098729883999424898641968281841439662364",
    );
    // - power_index != 0 && b >= FROM_DIGITS_DIVIDE_AND_CONQUER_THRESHOLD
    // - from_digits_desc_divide_and_conquer_limb; xs_len > p
    test_ok(
        &[1598, 27, 2039, 2403, 1843, 2859, 1581, 150, 2359, 302],
        3543,
        "140578615308984594421852296827289425",
    );
    // - from_digits_desc_divide_and_conquer_limb; base > SQRT_MAX_LIMB
    test_ok(
        &[1153431348, 1518538399, 293681571, 570510527, 83185796, 1187762660],
        1525385058,
        "9525530906278526930121302445905223566866929778026945776",
    );
    // - from_digits_desc_divide_and_conquer_limb; xs_len <= p
    test_ok(
        &[
            1046252719, 824969631, 1659733906, 1842176041, 1599722999, 1214721934, 1204675417,
            1169948427, 798341655, 1494323667, 31832127, 1535724679,
        ],
        1895460450,
        "118713086621740109729266002848273602283691336549480820142301294962906804654789322428658177\
        1782212965464227294329",
    );

    fn test_err(xs: &[u32], base: Limb) {
        assert!(from_digits_desc_limb(xs.iter().copied(), base).is_none());
        assert!(from_digits_desc_naive_primitive(xs, u32::exact_from(base)).is_none());
    }
    test_err(&[10, 11, 12], 10);
}

fn from_digits_desc_limb_fail_helper<T: ConvertibleFrom<Limb> + PrimitiveUnsigned>()
where
    Limb: TryFrom<T> + WrappingFrom<T>,
    Natural: From<T> + PowerOf2Digits<T>,
{
    assert_panic!(from_digits_desc_limb::<_, T>(empty(), 0));
    assert_panic!(from_digits_desc_limb::<_, T>(empty(), 1));
}

#[test]
fn from_digits_desc_limb_fail() {
    apply_fn_to_unsigneds!(from_digits_desc_limb_fail_helper);
}

fn from_digits_desc_limb_properties_helper<T: ConvertibleFrom<Limb> + PrimitiveUnsigned>()
where
    Limb: ExactFrom<T> + SaturatingFrom<T> + WrappingFrom<T>,
    Natural: From<T> + PowerOf2Digits<T>,
{
    let mut config = GenConfig::new();
    config.insert("mean_digit_count_n", 32);
    unsigned_vec_unsigned_pair_gen_var_12::<T, Limb>().test_properties_with_config(
        &config,
        |(xs, base)| {
            let t_base = T::exact_from(base);
            assert_eq!(
                from_digits_desc_limb(xs.iter().copied(), base).is_some(),
                xs.iter().all(|&x| x < t_base)
            );
        },
    );

    unsigned_vec_unsigned_pair_gen_var_5::<T, Limb>().test_properties_with_config(
        &config,
        |(xs, base)| {
            let n = from_digits_desc_limb(xs.iter().copied(), base);
            assert_eq!(
                from_digits_desc_naive_primitive(&xs, T::exact_from(base)),
                n
            );
        },
    );
}

#[test]
fn from_digits_desc_limb_properties() {
    apply_fn_to_unsigneds!(from_digits_desc_limb_properties_helper);
}

#[test]
fn test_from_digits_asc_large() {
    fn test_ok(xs: &[&str], base: &str, n: &str) {
        let xs = xs.iter().map(|x| Natural::from_str(x).unwrap());
        let base = Natural::from_str(base).unwrap();
        let n = Natural::from_str(n).unwrap();
        assert_eq!(from_digits_asc_large(xs.clone(), &base).unwrap(), n);
        assert_eq!(
            from_digits_desc_naive(&xs.rev().collect_vec(), &base).unwrap(),
            n
        );
    }
    test_ok(&["0", "100"], "10000000000", "1000000000000");
    test_ok(
        &["27917287424", "18657454436", "8470329472"],
        "34359738368",
        "10000000000000000000000000000000",
    );
    test_ok(
        &[
            "4373186134",
            "2564485756",
            "2124820161",
            "4270626619",
            "5254372654",
            "713959034",
            "4750044302",
            "5833014701",
            "978351288",
            "4288991795",
            "972424917",
            "1439538405",
            "5308114100",
            "1115837958",
            "2267585072",
            "4579628351",
            "3319271253",
            "139021832",
        ],
        "6000000000",
        "235317521501133049587746364812444472287442159306443086833887479789539173449622133054745814\
        7574478578278803560754066959663745455193666960506455349780493525811386914540373186134",
    );
    test_ok(
        &[
            "10459983243",
            "21532249186",
            "9820491776",
            "2837355685",
            "9767368393",
            "3483032332",
            "21535703589",
            "11033729126",
            "9179503556",
            "8692905086",
            "4911199976",
            "15555287795",
            "16865310802",
            "20703615271",
            "16296043356",
            "2287104975",
            "7356592443",
            "19932263435",
            "22157300197",
        ],
        "24107150480",
        "167549316476609610254199818013888896761159964082484503741352784741144483211685148473082817\
        5855188918761759968447473283739550817700685872668984640703760436616510553858385622841396530\
        38089626212682923",
    );

    fn test_err(xs: &[&str], base: &str) {
        let xs = xs.iter().map(|x| Natural::from_str(x).unwrap());
        let base = Natural::from_str(base).unwrap();
        assert!(from_digits_asc_large(xs.clone(), &base).is_none());
        assert!(from_digits_desc_naive(&xs.rev().collect_vec(), &base).is_none());
    }
    test_err(
        &["1000000000000000000000001", "1000000000000000000000002"],
        "1000000000000000000000000",
    );
}

#[test]
fn from_digits_asc_large_fail() {
    assert_panic!(from_digits_asc_large(empty(), &Natural::ZERO));
    assert_panic!(from_digits_asc_large(empty(), &Natural::ONE));
}

#[test]
fn from_digits_asc_large_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_digit_count_n", 32);
    natural_vec_natural_pair_gen_var_3().test_properties_with_config(&config, |(xs, base)| {
        assert_eq!(
            from_digits_asc_large(xs.iter().cloned(), &base).is_some(),
            xs.iter().all(|x| x < &base)
        );
    });

    natural_vec_natural_pair_gen_var_1().test_properties_with_config(&config, |(xs, base)| {
        let n = from_digits_asc_large(xs.iter().cloned(), &base);
        assert_eq!(
            from_digits_desc_naive(&xs.into_iter().rev().collect_vec(), &base),
            n
        );
    });
}

#[test]
fn test_from_digits_desc_large() {
    fn test_ok(xs: &[&str], base: &str, n: &str) {
        let xs = xs.iter().map(|x| Natural::from_str(x).unwrap());
        let base = Natural::from_str(base).unwrap();
        let n = Natural::from_str(n).unwrap();
        assert_eq!(from_digits_desc_large(xs.clone(), &base).unwrap(), n);
        assert_eq!(from_digits_desc_naive(&xs.collect_vec(), &base).unwrap(), n);
    }
    // - Some(log_base) != base.checked_log_base_2()
    // - from_digits_desc_divide_and_conquer; power_index == 0 || u64::exact_from(xs_len) *
    //   base.significant_bits() < FROM_DIGITS_DIVIDE_AND_CONQUER_THRESHOLD
    test_ok(&["100", "0"], "10000000000", "1000000000000");
    // - Some(log_base) = base.checked_log_base_2()
    test_ok(
        &["8470329472", "18657454436", "27917287424"],
        "34359738368",
        "10000000000000000000000000000000",
    );
    // - from_digits_desc_divide_and_conquer; power_index != 0 && u64::exact_from(xs_len) *
    //   base.significant_bits() >= FROM_DIGITS_DIVIDE_AND_CONQUER_THRESHOLD
    // - from_digits_desc_divide_and_conquer; xs_len > p
    test_ok(
        &[
            "139021832",
            "3319271253",
            "4579628351",
            "2267585072",
            "1115837958",
            "5308114100",
            "1439538405",
            "972424917",
            "4288991795",
            "978351288",
            "5833014701",
            "4750044302",
            "713959034",
            "5254372654",
            "4270626619",
            "2124820161",
            "2564485756",
            "4373186134",
        ],
        "6000000000",
        "235317521501133049587746364812444472287442159306443086833887479789539173449622133054745814\
        7574478578278803560754066959663745455193666960506455349780493525811386914540373186134",
    );
    // - from_digits_desc_divide_and_conquer; xs_len <= p
    test_ok(
        &[
            "22157300197",
            "19932263435",
            "7356592443",
            "2287104975",
            "16296043356",
            "20703615271",
            "16865310802",
            "15555287795",
            "4911199976",
            "8692905086",
            "9179503556",
            "11033729126",
            "21535703589",
            "3483032332",
            "9767368393",
            "2837355685",
            "9820491776",
            "21532249186",
            "10459983243",
        ],
        "24107150480",
        "167549316476609610254199818013888896761159964082484503741352784741144483211685148473082817\
        5855188918761759968447473283739550817700685872668984640703760436616510553858385622841396530\
        38089626212682923",
    );

    fn test_err(xs: &[&str], base: &str) {
        let xs = xs.iter().map(|x| Natural::from_str(x).unwrap());
        let base = Natural::from_str(base).unwrap();
        assert!(from_digits_desc_large(xs.clone(), &base).is_none());
        assert!(from_digits_desc_naive(&xs.collect_vec(), &base).is_none());
    }
    test_err(
        &["1000000000000000000000000", "1000000000000000000000001"],
        "1000000000000000000000000",
    );
}

#[test]
fn from_digits_desc_large_fail() {
    assert_panic!(from_digits_desc_large(empty(), &Natural::ZERO));
    assert_panic!(from_digits_desc_large(empty(), &Natural::ONE));
}

#[test]
fn from_digits_desc_large_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_digit_count_n", 32);
    natural_vec_natural_pair_gen_var_3().test_properties_with_config(&config, |(xs, base)| {
        assert_eq!(
            from_digits_desc_large(xs.iter().cloned(), &base).is_some(),
            xs.iter().all(|x| x < &base)
        );
    });

    natural_vec_natural_pair_gen_var_1().test_properties_with_config(&config, |(xs, base)| {
        let n = from_digits_desc_large(xs.iter().cloned(), &base);
        assert_eq!(from_digits_desc_naive(&xs, &base), n);
    });
}

#[test]
fn from_digits_asc_unsigned() {
    fn test_ok(xs: &[Limb], base: Limb, n: &str) {
        let n = Natural::from_str(n).unwrap();
        assert_eq!(
            Natural::from_digits_asc(&base, xs.iter().copied()).unwrap(),
            n
        );
    }
    test_ok(&[], 9, "0");
    test_ok(&[0], 9, "0");
    test_ok(&[1], 9, "1");
    test_ok(&[0, 1, 1, 0, 0, 1, 1, 2, 0, 0, 2], 3, "123456");
    test_ok(&[0, 1, 1, 0, 0, 1, 1, 2, 0, 0, 2, 0, 0], 3, "123456");
    test_ok(&[2, 4, 6, 4, 2], 8, "10658");
    test_ok(&[789, 456, 123], 1000, "123456789");
    test_ok(
        &[
            14, 99, 101, 37, 10, 132, 124, 140, 98, 35, 144, 50, 41, 15, 67, 39, 12, 74, 27, 9, 21,
            135, 62, 26, 68, 86, 128, 34, 78, 25, 75, 61, 73, 80, 66, 50, 15, 132, 40, 146, 83, 66,
            87, 40, 41, 104, 37, 3, 44, 13, 51, 59, 139, 9, 39, 25, 27, 82, 29, 121, 32, 17, 22,
            81, 102, 0, 86, 11, 129, 90, 113, 147, 108, 45, 120, 23, 73,
        ],
        150,
        "176685760608531978263731938997517835399219565848609872558191310084297042793489341607854254\
        709347189745346071475819587247558105442098729883999424898641968281841439662364",
    );
    test_ok(
        &[302, 2359, 150, 1581, 2859, 1843, 2403, 2039, 27, 1598],
        3543,
        "140578615308984594421852296827289425",
    );
    test_ok(
        &[1187762660, 83185796, 570510527, 293681571, 1518538399, 1153431348],
        1525385058,
        "9525530906278526930121302445905223566866929778026945776",
    );
    test_ok(
        &[
            1535724679, 31832127, 1494323667, 798341655, 1169948427, 1204675417, 1214721934,
            1599722999, 1842176041, 1659733906, 824969631, 1046252719,
        ],
        1895460450,
        "118713086621740109729266002848273602283691336549480820142301294962906804654789322428658177\
        1782212965464227294329",
    );

    fn test_err(xs: &[Limb], base: Limb) {
        assert!(Natural::from_digits_asc(&base, xs.iter().copied()).is_none());
    }
    test_err(&[10, 11, 12], 10);
}

fn from_digits_asc_unsigned_fail_helper<T: PrimitiveUnsigned>()
where
    Natural: Digits<T>,
{
    assert_panic!(Natural::from_digits_asc(&T::ONE, empty()));
    assert_panic!(Natural::from_digits_asc(&T::ZERO, empty()));
}

#[test]
fn from_digits_asc_unsigned_fail() {
    apply_fn_to_unsigneds!(from_digits_asc_unsigned_fail_helper);
}

fn from_digits_asc_unsigned_helper<T: PrimitiveUnsigned + SaturatingFrom<T> + WrappingFrom<T>>()
where
    Natural: Digits<T>,
{
    let mut config = GenConfig::new();
    config.insert("mean_digit_count_n", 32);
    unsigned_vec_unsigned_pair_gen_var_12::<T, T>().test_properties_with_config(
        &config,
        |(xs, base)| {
            let t_base = T::exact_from(base);
            assert_eq!(
                Natural::from_digits_asc(&base, xs.iter().copied()).is_some(),
                xs.iter().all(|&x| x < t_base)
            );
        },
    );

    unsigned_vec_unsigned_pair_gen_var_5::<T, T>().test_properties_with_config(
        &config,
        |(digits, base)| {
            let n = Natural::from_digits_asc(&base, digits.iter().copied()).unwrap();
            assert_eq!(
                Natural::from_digits_desc(&base, digits.iter().rev().copied()).unwrap(),
                n
            );
            let trailing_zeros = slice_trailing_zeros(&digits);
            assert_eq!(
                n.to_digits_asc(&base),
                &digits[..digits.len() - trailing_zeros]
            );
        },
    );

    unsigned_pair_gen_var_10::<T, T, usize>().test_properties(|(base, u)| {
        assert_eq!(
            Natural::from_digits_asc(&base, repeat_n(T::ZERO, u)),
            Some(Natural::ZERO)
        );
    });
}

#[test]
fn from_digits_asc_unsigned_properties() {
    apply_fn_to_unsigneds!(from_digits_asc_unsigned_helper);
}

#[test]
fn from_digits_desc_unsigned() {
    fn test_ok(xs: &[Limb], base: Limb, n: &str) {
        let n = Natural::from_str(n).unwrap();
        assert_eq!(
            Natural::from_digits_desc(&base, xs.iter().copied()).unwrap(),
            n
        );
        assert_eq!(from_digits_desc_naive_primitive(xs, base).unwrap(), n);
    }
    test_ok(&[], 9, "0");
    test_ok(&[0], 9, "0");
    test_ok(&[1], 9, "1");
    test_ok(&[2, 0, 0, 2, 1, 1, 0, 0, 1, 1, 0], 3, "123456");
    test_ok(&[0, 0, 2, 0, 0, 2, 1, 1, 0, 0, 1, 1, 0], 3, "123456");
    test_ok(&[2, 4, 6, 4, 2], 8, "10658");
    test_ok(&[123, 456, 789], 1000, "123456789");
    test_ok(
        &[
            73, 23, 120, 45, 108, 147, 113, 90, 129, 11, 86, 0, 102, 81, 22, 17, 32, 121, 29, 82,
            27, 25, 39, 9, 139, 59, 51, 13, 44, 3, 37, 104, 41, 40, 87, 66, 83, 146, 40, 132, 15,
            50, 66, 80, 73, 61, 75, 25, 78, 34, 128, 86, 68, 26, 62, 135, 21, 9, 27, 74, 12, 39,
            67, 15, 41, 50, 144, 35, 98, 140, 124, 132, 10, 37, 101, 99, 14,
        ],
        150,
        "176685760608531978263731938997517835399219565848609872558191310084297042793489341607854254\
        709347189745346071475819587247558105442098729883999424898641968281841439662364",
    );
    test_ok(
        &[1598, 27, 2039, 2403, 1843, 2859, 1581, 150, 2359, 302],
        3543,
        "140578615308984594421852296827289425",
    );
    test_ok(
        &[1153431348, 1518538399, 293681571, 570510527, 83185796, 1187762660],
        1525385058,
        "9525530906278526930121302445905223566866929778026945776",
    );
    test_ok(
        &[
            1046252719, 824969631, 1659733906, 1842176041, 1599722999, 1214721934, 1204675417,
            1169948427, 798341655, 1494323667, 31832127, 1535724679,
        ],
        1895460450,
        "118713086621740109729266002848273602283691336549480820142301294962906804654789322428658177\
        1782212965464227294329",
    );

    fn test_err(xs: &[Limb], base: Limb) {
        assert!(Natural::from_digits_desc(&base, xs.iter().copied()).is_none());
        assert!(from_digits_desc_naive_primitive(xs, base).is_none());
    }
    test_err(&[10, 11, 12], 10);
}

fn from_digits_desc_unsigned_fail_helper<T: PrimitiveUnsigned>()
where
    Natural: Digits<T>,
{
    assert_panic!(Natural::from_digits_desc(&T::ONE, empty()));
    assert_panic!(Natural::from_digits_desc(&T::ZERO, empty()));
}

#[test]
fn from_digits_desc_unsigned_fail() {
    apply_fn_to_unsigneds!(from_digits_desc_unsigned_fail_helper);
}

fn from_digits_desc_unsigned_helper<T: PrimitiveUnsigned + SaturatingFrom<T> + WrappingFrom<T>>()
where
    Natural: Digits<T> + From<T>,
{
    let mut config = GenConfig::new();
    config.insert("mean_digit_count_n", 32);
    unsigned_vec_unsigned_pair_gen_var_12::<T, T>().test_properties_with_config(
        &config,
        |(xs, base)| {
            let t_base = T::exact_from(base);
            assert_eq!(
                Natural::from_digits_desc(&base, xs.iter().copied()).is_some(),
                xs.iter().all(|&x| x < t_base)
            );
        },
    );

    unsigned_vec_unsigned_pair_gen_var_5::<T, T>().test_properties_with_config(
        &config,
        |(digits, base)| {
            let n = Natural::from_digits_desc(&base, digits.iter().copied()).unwrap();
            assert_eq!(
                Natural::from_digits_asc(&base, digits.iter().rev().copied()).unwrap(),
                n
            );
            let leading_zeros = slice_leading_zeros(&digits);
            assert_eq!(n.to_digits_desc(&base), &digits[leading_zeros..]);
            assert_eq!(
                from_digits_desc_naive_primitive(&digits, T::exact_from(base)).unwrap(),
                n
            );
        },
    );

    unsigned_pair_gen_var_10::<T, T, usize>().test_properties(|(base, u)| {
        assert_eq!(
            Natural::from_digits_desc(&base, repeat_n(T::ZERO, u)),
            Some(Natural::ZERO)
        );
    });
}

#[test]
fn from_digits_desc_unsigned_properties() {
    apply_fn_to_unsigneds!(from_digits_desc_unsigned_helper);
}

#[test]
fn from_digits_asc() {
    fn test_ok(xs: &[&str], base: &str, n: &str) {
        let xs = xs.iter().map(|x| Natural::from_str(x).unwrap());
        let base = Natural::from_str(base).unwrap();
        let n = Natural::from_str(n).unwrap();
        assert_eq!(Natural::from_digits_asc(&base, xs.clone()).unwrap(), n);
    }
    test_ok(&[], "9", "0");
    test_ok(&["0"], "9", "0");
    test_ok(&["1"], "9", "1");
    test_ok(
        &["0", "1", "1", "0", "0", "1", "1", "2", "0", "0", "2"],
        "3",
        "123456",
    );
    test_ok(
        &["0", "1", "1", "0", "0", "1", "1", "2", "0", "0", "2", "0", "0"],
        "3",
        "123456",
    );
    test_ok(&["2", "4", "6", "4", "2"], "8", "10658");
    test_ok(&["789", "456", "123"], "1000", "123456789");
    test_ok(
        &[
            "14", "99", "101", "37", "10", "132", "124", "140", "98", "35", "144", "50", "41",
            "15", "67", "39", "12", "74", "27", "9", "21", "135", "62", "26", "68", "86", "128",
            "34", "78", "25", "75", "61", "73", "80", "66", "50", "15", "132", "40", "146", "83",
            "66", "87", "40", "41", "104", "37", "3", "44", "13", "51", "59", "139", "9", "39",
            "25", "27", "82", "29", "121", "32", "17", "22", "81", "102", "0", "86", "11", "129",
            "90", "113", "147", "108", "45", "120", "23", "73",
        ],
        "150",
        "176685760608531978263731938997517835399219565848609872558191310084297042793489341607854254\
        709347189745346071475819587247558105442098729883999424898641968281841439662364",
    );
    test_ok(
        &["302", "2359", "150", "1581", "2859", "1843", "2403", "2039", "27", "1598"],
        "3543",
        "140578615308984594421852296827289425",
    );
    test_ok(
        &["1187762660", "83185796", "570510527", "293681571", "1518538399", "1153431348"],
        "1525385058",
        "9525530906278526930121302445905223566866929778026945776",
    );
    test_ok(
        &[
            "1535724679",
            "31832127",
            "1494323667",
            "798341655",
            "1169948427",
            "1204675417",
            "1214721934",
            "1599722999",
            "1842176041",
            "1659733906",
            "824969631",
            "1046252719",
        ],
        "1895460450",
        "118713086621740109729266002848273602283691336549480820142301294962906804654789322428658177\
        1782212965464227294329",
    );
    test_ok(&["0", "100"], "10000000000", "1000000000000");
    test_ok(
        &["27917287424", "18657454436", "8470329472"],
        "34359738368",
        "10000000000000000000000000000000",
    );
    test_ok(
        &[
            "4373186134",
            "2564485756",
            "2124820161",
            "4270626619",
            "5254372654",
            "713959034",
            "4750044302",
            "5833014701",
            "978351288",
            "4288991795",
            "972424917",
            "1439538405",
            "5308114100",
            "1115837958",
            "2267585072",
            "4579628351",
            "3319271253",
            "139021832",
        ],
        "6000000000",
        "235317521501133049587746364812444472287442159306443086833887479789539173449622133054745814\
        7574478578278803560754066959663745455193666960506455349780493525811386914540373186134",
    );
    test_ok(
        &[
            "10459983243",
            "21532249186",
            "9820491776",
            "2837355685",
            "9767368393",
            "3483032332",
            "21535703589",
            "11033729126",
            "9179503556",
            "8692905086",
            "4911199976",
            "15555287795",
            "16865310802",
            "20703615271",
            "16296043356",
            "2287104975",
            "7356592443",
            "19932263435",
            "22157300197",
        ],
        "24107150480",
        "167549316476609610254199818013888896761159964082484503741352784741144483211685148473082817\
        5855188918761759968447473283739550817700685872668984640703760436616510553858385622841396530\
        38089626212682923",
    );

    fn test_err(xs: &[&str], base: &str) {
        let xs = xs.iter().map(|x| Natural::from_str(x).unwrap());
        let base = Natural::from_str(base).unwrap();
        assert!(Natural::from_digits_asc(&base, xs.clone()).is_none());
    }
    test_err(&["101", "102"], "100");
}

#[test]
fn from_digits_asc_fail() {
    assert_panic!(Natural::from_digits_asc(&Natural::ZERO, empty()));
    assert_panic!(Natural::from_digits_asc(&Natural::ONE, empty()));
}

#[test]
fn from_digits_asc_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_bits_n", 64);
    config.insert("mean_digit_count_n", 32);
    natural_vec_natural_pair_gen_var_4().test_properties_with_config(&config, |(xs, base)| {
        assert_eq!(
            Natural::from_digits_asc(&base, xs.iter().cloned()).is_some(),
            xs.iter().all(|x| x < &base)
        );
    });

    natural_vec_natural_pair_gen_var_2().test_properties_with_config(&config, |(digits, base)| {
        let n = Natural::from_digits_asc(&base, digits.iter().cloned()).unwrap();
        assert_eq!(
            Natural::from_digits_desc(&base, digits.iter().rev().cloned()).unwrap(),
            n
        );
        let trailing_zeros = slice_trailing_zeros(&digits);
        assert_eq!(
            n.to_digits_asc(&base),
            &digits[..digits.len() - trailing_zeros]
        );
    });

    natural_unsigned_pair_gen_var_5().test_properties(|(base, u)| {
        assert_eq!(
            Natural::from_digits_asc(&base, repeat_n(Natural::ZERO, u)),
            Some(Natural::ZERO)
        );
    });
}

#[test]
fn from_digits_desc() {
    fn test_ok(xs: &[&str], base: &str, n: &str) {
        let xs = xs.iter().map(|x| Natural::from_str(x).unwrap());
        let base = Natural::from_str(base).unwrap();
        let n = Natural::from_str(n).unwrap();
        assert_eq!(Natural::from_digits_desc(&base, xs.clone()).unwrap(), n);
        assert_eq!(from_digits_desc_naive(&xs.collect_vec(), &base).unwrap(), n);
    }
    test_ok(&[], "9", "0");
    test_ok(&["0"], "9", "0");
    test_ok(&["1"], "9", "1");
    test_ok(
        &["2", "0", "0", "2", "1", "1", "0", "0", "1", "1", "0"],
        "3",
        "123456",
    );
    test_ok(
        &["0", "0", "2", "0", "0", "2", "1", "1", "0", "0", "1", "1", "0"],
        "3",
        "123456",
    );
    test_ok(&["2", "4", "6", "4", "2"], "8", "10658");
    test_ok(&["123", "456", "789"], "1000", "123456789");
    test_ok(
        &[
            "73", "23", "120", "45", "108", "147", "113", "90", "129", "11", "86", "0", "102",
            "81", "22", "17", "32", "121", "29", "82", "27", "25", "39", "9", "139", "59", "51",
            "13", "44", "3", "37", "104", "41", "40", "87", "66", "83", "146", "40", "132", "15",
            "50", "66", "80", "73", "61", "75", "25", "78", "34", "128", "86", "68", "26", "62",
            "135", "21", "9", "27", "74", "12", "39", "67", "15", "41", "50", "144", "35", "98",
            "140", "124", "132", "10", "37", "101", "99", "14",
        ],
        "150",
        "176685760608531978263731938997517835399219565848609872558191310084297042793489341607854254\
        709347189745346071475819587247558105442098729883999424898641968281841439662364",
    );
    test_ok(
        &["1598", "27", "2039", "2403", "1843", "2859", "1581", "150", "2359", "302"],
        "3543",
        "140578615308984594421852296827289425",
    );
    test_ok(
        &["1153431348", "1518538399", "293681571", "570510527", "83185796", "1187762660"],
        "1525385058",
        "9525530906278526930121302445905223566866929778026945776",
    );
    test_ok(
        &[
            "1046252719",
            "824969631",
            "1659733906",
            "1842176041",
            "1599722999",
            "1214721934",
            "1204675417",
            "1169948427",
            "798341655",
            "1494323667",
            "31832127",
            "1535724679",
        ],
        "1895460450",
        "118713086621740109729266002848273602283691336549480820142301294962906804654789322428658177\
        1782212965464227294329",
    );
    test_ok(&["100", "0"], "10000000000", "1000000000000");
    test_ok(
        &["8470329472", "18657454436", "27917287424"],
        "34359738368",
        "10000000000000000000000000000000",
    );
    test_ok(
        &[
            "139021832",
            "3319271253",
            "4579628351",
            "2267585072",
            "1115837958",
            "5308114100",
            "1439538405",
            "972424917",
            "4288991795",
            "978351288",
            "5833014701",
            "4750044302",
            "713959034",
            "5254372654",
            "4270626619",
            "2124820161",
            "2564485756",
            "4373186134",
        ],
        "6000000000",
        "235317521501133049587746364812444472287442159306443086833887479789539173449622133054745814\
        7574478578278803560754066959663745455193666960506455349780493525811386914540373186134",
    );
    test_ok(
        &[
            "22157300197",
            "19932263435",
            "7356592443",
            "2287104975",
            "16296043356",
            "20703615271",
            "16865310802",
            "15555287795",
            "4911199976",
            "8692905086",
            "9179503556",
            "11033729126",
            "21535703589",
            "3483032332",
            "9767368393",
            "2837355685",
            "9820491776",
            "21532249186",
            "10459983243",
        ],
        "24107150480",
        "167549316476609610254199818013888896761159964082484503741352784741144483211685148473082817\
        5855188918761759968447473283739550817700685872668984640703760436616510553858385622841396530\
        38089626212682923",
    );

    fn test_err(xs: &[&str], base: &str) {
        let xs = xs.iter().map(|x| Natural::from_str(x).unwrap());
        let base = Natural::from_str(base).unwrap();
        assert!(Natural::from_digits_desc(&base, xs.clone()).is_none());
        assert!(from_digits_desc_naive(&xs.collect_vec(), &base).is_none());
    }
    test_err(&["101", "102"], "100");
}

#[test]
fn from_digits_desc_fail() {
    assert_panic!(Natural::from_digits_desc(&Natural::ZERO, empty()));
    assert_panic!(Natural::from_digits_desc(&Natural::ONE, empty()));
}

#[test]
fn from_digits_desc_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_bits_n", 64);
    config.insert("mean_digit_count_n", 32);
    natural_vec_natural_pair_gen_var_4().test_properties_with_config(&config, |(xs, base)| {
        assert_eq!(
            Natural::from_digits_desc(&base, xs.iter().cloned()).is_some(),
            xs.iter().all(|x| x < &base)
        );
    });

    natural_vec_natural_pair_gen_var_2().test_properties_with_config(&config, |(digits, base)| {
        let n = Natural::from_digits_desc(&base, digits.iter().cloned()).unwrap();
        assert_eq!(
            Natural::from_digits_asc(&base, digits.iter().rev().cloned()).unwrap(),
            n
        );
        let leading_zeros = slice_leading_zeros(&digits);
        assert_eq!(n.to_digits_desc(&base), &digits[leading_zeros..]);
        assert_eq!(from_digits_desc_naive(&digits, &base).unwrap(), n);
    });

    natural_unsigned_pair_gen_var_5().test_properties(|(base, u)| {
        assert_eq!(
            Natural::from_digits_desc(&base, repeat_n(Natural::ZERO, u)),
            Some(Natural::ZERO)
        );
    });
}
