// Copyright © 2025 Mikhail Hogrefe
//
// This file is part of Malachite.
//
// Malachite is free software: you can redistribute it and/or modify it under the terms of the GNU
// Lesser General Public License (LGPL) as published by the Free Software Foundation; either version
// 3 of the License, or (at your option) any later version. See <https://www.gnu.org/licenses/>.

use malachite_base::num::arithmetic::traits::{
    CeilingDivNegMod, DivMod, Mod, ModAssign, ModIsReduced, NegMod, NegModAssign,
};
use malachite_base::num::basic::integers::PrimitiveInt;
use malachite_base::num::basic::traits::{One, Zero};
use malachite_base::num::conversion::traits::{ExactFrom, JoinHalves};
use malachite_base::test_util::generators::common::GenConfig;
use malachite_base::test_util::generators::{
    unsigned_pair_gen_var_12, unsigned_vec_pair_gen_var_11, unsigned_vec_pair_gen_var_18,
    unsigned_vec_triple_gen_var_57, unsigned_vec_unsigned_pair_gen_var_22,
    unsigned_vec_unsigned_pair_gen_var_26, unsigned_vec_unsigned_pair_gen_var_27,
    unsigned_vec_unsigned_pair_gen_var_28,
};
use malachite_nz::natural::Natural;
use malachite_nz::natural::arithmetic::div_mod::{
    limbs_div_mod_barrett_scratch_len, limbs_two_limb_inverse_helper,
};
use malachite_nz::natural::arithmetic::mod_op::{
    limbs_mod, limbs_mod_barrett, limbs_mod_by_two_limb_normalized, limbs_mod_divide_and_conquer,
    limbs_mod_limb, limbs_mod_limb_alt_1, limbs_mod_limb_alt_2, limbs_mod_limb_any_leading_zeros_1,
    limbs_mod_limb_any_leading_zeros_2, limbs_mod_limb_at_least_1_leading_zero,
    limbs_mod_limb_at_least_2_leading_zeros, limbs_mod_limb_small_normalized,
    limbs_mod_limb_small_unnormalized, limbs_mod_schoolbook, limbs_mod_three_limb_by_two_limb,
    limbs_mod_to_out,
};
use malachite_nz::platform::{DoubleLimb, Limb};
use malachite_nz::test_util::generators::{
    large_type_gen_var_12, natural_gen, natural_gen_var_2, natural_pair_gen_var_5,
    natural_pair_gen_var_6, natural_triple_gen_var_4, unsigned_sextuple_gen_var_2,
    unsigned_vec_quadruple_gen_var_5, unsigned_vec_unsigned_vec_unsigned_triple_gen_var_17,
};
use malachite_nz::test_util::natural::arithmetic::mod_op::{limbs_mod_limb_alt_3, rug_neg_mod};
use num::{BigUint, Integer};
use rug;
use rug::ops::RemRounding;
use std::str::FromStr;

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_mod_limb() {
    let test = |ns: &[Limb], d: Limb, r: Limb| {
        assert_eq!(limbs_mod_limb::<DoubleLimb, Limb>(ns, d), r);
        assert_eq!(
            limbs_mod_limb_any_leading_zeros_1::<DoubleLimb, Limb>(ns, d),
            r
        );
        assert_eq!(
            limbs_mod_limb_any_leading_zeros_2::<DoubleLimb, Limb>(ns, d),
            r
        );
        assert_eq!(limbs_mod_limb_alt_1::<DoubleLimb, Limb>(ns, d), r);
        assert_eq!(limbs_mod_limb_alt_2::<DoubleLimb, Limb>(ns, d), r);
        assert_eq!(limbs_mod_limb_alt_3(ns, d), r);
    };
    test(&[0, 0], 2, 0);
    // - shift != 0 in limbs_mod_limb_any_leading_zeros_1
    // - r_hi < b in limbs_mod_limb_any_leading_zeros_1
    // - n == 2 in limbs_mod_limb_any_leading_zeros_1
    // - !divisor.get_highest_bit() in limbs_mod_limb_alt_2
    // - !divisor.get_highest_bit() && len < MOD_1U_TO_MOD_1_1_THRESHOLD in limbs_mod_limb_alt_2
    test(&[6, 7], 1, 0);
    test(&[6, 7], 2, 0);
    // - n > 2 in limbs_mod_limb_any_leading_zeros_1
    // - !divisor.get_highest_bit() && MOD_1U_TO_MOD_1_1_THRESHOLD <= len <
    //   MOD_11_TO_MOD_1_2_THRESHOLD in limbs_mod_limb_alt_2
    test(&[100, 101, 102], 10, 8);
    test(&[123, 456], 789, 636);
    test(&[0, 0], 0xa0000000, 0);
    // - shift == 0 in limbs_mod_limb_any_leading_zeros_1
    // - divisor.get_highest_bit() in limbs_mod_limb_alt_2
    // - divisor.get_highest_bit() && len < MOD_1N_TO_MOD_1_1_THRESHOLD in limbs_mod_limb_alt_2
    test(&[6, 7], 0x80000000, 6);
    test(&[6, 7], 0xa0000000, 0x20000006);
    // - divisor.get_highest_bit() && len >= MOD_1N_TO_MOD_1_1_THRESHOLD in limbs_mod_limb_alt_2
    test(&[100, 101, 102], 0xabcddcba, 2152689614);
    // - r_hi >= b in limbs_mod_limb_any_leading_zeros_1
    test(&[u32::MAX, u32::MAX], 2, 1);
    test(&[u32::MAX, u32::MAX], 3, 0);
    test(&[u32::MAX, u32::MAX], u32::MAX, 0);
    test(&[u32::MAX, u32::MAX], 0xa0000000, 1610612735);
    test(&[100, 101, 102], u32::MAX, 303);
    test(&[1, 2, 3, 4], 6, 1);
    // - !divisor.get_highest_bit() && len >= MOD_11_TO_MOD_1_2_THRESHOLD && (len <
    //   MOD_12_TO_MOD_1_4_THRESHOLD || divisor & HIGHEST_TWO_BITS_MASK != 0) in
    //   limbs_mod_limb_alt_2
    test(
        &[
            3713432036, 2475243626, 3960734766, 244755020, 3760002601, 301563516, 2499010086,
            1451814771, 1299826235, 3628218184, 2565364972, 3729936002,
        ],
        565832495,
        295492150,
    );
    // - !divisor.get_highest_bit() && len >= MOD_12_TO_MOD_1_4_THRESHOLD && divisor &
    //   HIGHEST_TWO_BITS_MASK == 0 in limbs_mod_limb_alt_2
    test(
        &[
            540286473, 1475101238, 1863380542, 2517905739, 81646271, 3172818884, 2759300635,
            852345965, 3647245071, 3875987988, 4229899590, 4100778302, 1641902155, 1289745333,
            3414845068, 119899697, 2175381145, 2490291811, 3047506964, 1815484255, 3379971995,
            1695675424, 1418284338,
        ],
        436775226,
        165213921,
    );
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_mod_limb_fail_1() {
    limbs_mod_limb::<DoubleLimb, Limb>(&[10], 10);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_mod_limb_fail_2() {
    limbs_mod_limb::<DoubleLimb, Limb>(&[10, 10], 0);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_mod_limb_any_leading_zeros_1_fail_1() {
    limbs_mod_limb_any_leading_zeros_1::<DoubleLimb, Limb>(&[10], 10);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_mod_limb_any_leading_zeros_1_fail_2() {
    limbs_mod_limb_any_leading_zeros_1::<DoubleLimb, Limb>(&[10, 10], 0);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_mod_limb_any_leading_zeros_2_fail_1() {
    limbs_mod_limb_any_leading_zeros_2::<DoubleLimb, Limb>(&[10], 10);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_mod_limb_any_leading_zeros_2_fail_2() {
    limbs_mod_limb_any_leading_zeros_2::<DoubleLimb, Limb>(&[10, 10], 0);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_mod_limb_small_normalized() {
    let test = |ns: &[Limb], d: Limb, r: Limb| {
        assert_eq!(
            limbs_mod_limb_small_normalized::<DoubleLimb, Limb>(ns, d),
            r
        );
    };
    test(&[0x80000123], 0x80000000, 0x123);
    test(&[0, 0], 0xa0000000, 0);
    test(&[6, 7], 0x80000000, 6);
    test(&[6, 7], 0xa0000000, 0x20000006);
    test(&[100, 101, 102], 0xabcddcba, 2152689614);
    test(&[u32::MAX, u32::MAX], u32::MAX, 0);
    test(&[u32::MAX, u32::MAX], 0xa0000000, 1610612735);
    test(&[100, 101, 102], u32::MAX, 303);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_mod_limb_small_normalized_fail_1() {
    limbs_mod_limb_small_normalized::<DoubleLimb, Limb>(&[], u32::MAX);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_mod_limb_small_normalized_fail_2() {
    limbs_mod_limb_small_normalized::<DoubleLimb, Limb>(&[10, 10], 0);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_mod_limb_small_unnormalized() {
    let test = |ns: &[Limb], d: Limb, r: Limb| {
        assert_eq!(
            limbs_mod_limb_small_unnormalized::<DoubleLimb, Limb>(ns, d),
            r
        );
        assert_eq!(
            limbs_mod_limb_at_least_1_leading_zero::<DoubleLimb, Limb>(ns, d),
            r
        );
    };
    test(&[0, 0], 2, 0);
    test(&[0], 2, 0);
    // - remainder >= divisor in limbs_mod_limb_small_unnormalized
    // - len.odd() in limbs_mod_limb_at_least_1_leading_zero
    // - len == 1 in limbs_mod_limb_at_least_1_leading_zero
    test(&[6], 2, 0);
    test(&[6], 4, 2);
    // - len.even() in limbs_mod_limb_at_least_1_leading_zero
    // - len < 4 in limbs_mod_limb_at_least_1_leading_zero
    test(&[6, 7], 1, 0);
    test(&[6, 7], 2, 0);
    // - len.odd() && len != 1 in limbs_mod_limb_at_least_1_leading_zero
    test(&[100, 101, 102], 10, 8);
    // - remainder < divisor in limbs_mod_limb_small_unnormalized
    test(&[123, 456], 789, 636);
    test(&[u32::MAX, u32::MAX], 2, 1);
    test(&[u32::MAX, u32::MAX], 3, 0);
    // - len >= 4 in limbs_mod_limb_at_least_1_leading_zero
    test(&[1, 2, 3, 4, 5], 6, 3);
    test(&[1, 2, 3, 4], 6, 1);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_mod_limb_small_unnormalized_fail_1() {
    limbs_mod_limb_small_unnormalized::<DoubleLimb, Limb>(&[], 10);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_mod_limb_small_unnormalized_fail_2() {
    limbs_mod_limb_small_unnormalized::<DoubleLimb, Limb>(&[10, 10], u32::MAX);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_mod_limb_at_least_1_leading_zero_fail_1() {
    limbs_mod_limb_at_least_1_leading_zero::<DoubleLimb, Limb>(&[], 10);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_mod_limb_at_least_1_leading_zero_fail_2() {
    limbs_mod_limb_at_least_1_leading_zero::<DoubleLimb, Limb>(&[10, 10], u32::MAX);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_mod_limb_at_least_2_leading_zeros() {
    let test = |ns: &[Limb], d: Limb, r: Limb| {
        assert_eq!(
            limbs_mod_limb_at_least_2_leading_zeros::<DoubleLimb, Limb>(ns, d),
            r
        );
    };
    test(&[0, 0], 2, 0);
    test(&[0], 2, 0);
    // - len ≡ 1 mod 4
    // - len < 4
    test(&[6], 2, 0);
    test(&[6], 4, 2);
    // - len ≡ 2 mod 4
    test(&[6, 7], 1, 0);
    test(&[6, 7], 2, 0);
    // - len ≡ 3 mod 4
    test(&[100, 101, 102], 10, 8);
    test(&[123, 456], 789, 636);
    test(&[u32::MAX, u32::MAX], 2, 1);
    test(&[u32::MAX, u32::MAX], 3, 0);
    // - len ≡ 0 mod 4
    test(&[1, 2, 3, 4], 6, 1);
    // - len >= 4
    test(&[1, 2, 3, 4, 5], 6, 3);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_mod_limb_at_least_2_leading_zeros_fail_1() {
    limbs_mod_limb_at_least_2_leading_zeros::<DoubleLimb, Limb>(&[], 10);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_mod_limb_at_least_2_leading_zeros_fail_2() {
    limbs_mod_limb_at_least_2_leading_zeros::<DoubleLimb, Limb>(&[10, 10], 0x7fffffff);
}

fn verify_limbs_mod_three_limb_by_two_limb(
    n_2: Limb,
    n_1: Limb,
    n_0: Limb,
    d_1: Limb,
    d_0: Limb,
    r: DoubleLimb,
) {
    let n = Natural::from_owned_limbs_asc(vec![n_0, n_1, n_2]);
    let d = Natural::from(DoubleLimb::join_halves(d_1, d_0));
    let r = Natural::from(r);
    assert_eq!(n % &d, r);
    assert!(r < d);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_mod_three_limb_by_two_limb() {
    let test = |n_2, n_1, n_0, d_1, d_0, r| {
        assert_eq!(
            limbs_mod_three_limb_by_two_limb(
                n_2,
                n_1,
                n_0,
                d_1,
                d_0,
                limbs_two_limb_inverse_helper(d_1, d_0)
            ),
            r
        );
        verify_limbs_mod_three_limb_by_two_limb(n_2, n_1, n_0, d_1, d_0, r);
    };
    test(1, 2, 3, 0x80000004, 5, 0x7ffffffdfffffffe);
    test(2, 0x40000000, 4, 0x80000000, 0, 0x4000000000000004);
    test(
        1614123406,
        3687984980,
        2695202596,
        2258238141,
        1642523191,
        274277675918877623,
    );
}

fn verify_limbs_mod_by_two_limb_normalized(ns: &[Limb], ds: &[Limb], r_0: Limb, r_1: Limb) {
    let n = Natural::from_limbs_asc(ns);
    let d = Natural::from_limbs_asc(ds);
    let expected_r = n % &d;
    let r = Natural::from_owned_limbs_asc(vec![r_0, r_1]);
    assert_eq!(r, expected_r);
    assert!(r < d);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_mod_by_two_limb_normalized() {
    let test = |ns: &[Limb], ds: &[Limb], r_0: Limb, r_1: Limb| {
        assert_eq!(limbs_mod_by_two_limb_normalized(ns, ds), (r_0, r_1));
        verify_limbs_mod_by_two_limb_normalized(ns, ds, r_0, r_1);
    };
    test(&[1, 2], &[3, 0x80000000], 1, 2);
    test(&[1, 2, 3, 4, 5], &[3, 0x80000000], 166, 2147483626);
    test(
        &[4142767597, 2922703399, 3921445909],
        &[2952867570, 2530544119],
        3037232599,
        1218898013,
    );
}

#[test]
#[should_panic]
fn limbs_mod_by_two_limb_normalized_fail_1() {
    limbs_mod_by_two_limb_normalized(&[1, 2], &[3, 4]);
}

#[test]
#[should_panic]
fn limbs_mod_by_two_limb_normalized_fail_2() {
    limbs_mod_by_two_limb_normalized(&[1, 2], &[3, 0x80000000, 4]);
}

fn verify_limbs_mod_1(ns_in: &[Limb], ds: &[Limb], ns_out: &[Limb]) {
    let n = Natural::from_limbs_asc(ns_in);
    let d = Natural::from_limbs_asc(ds);
    let expected_r = n % &d;
    let r = Natural::from_limbs_asc(&ns_out[..ds.len()]);
    assert_eq!(r, expected_r);
    assert!(r < d);
}

#[test]
fn test_limbs_mod_schoolbook() {
    let test = |ns_in: &[Limb], ds: &[Limb], ns_out: &[Limb]| {
        let mut ns = ns_in.to_vec();
        let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
        limbs_mod_schoolbook(&mut ns, ds, d_inv);
        assert_eq!(ns, ns_out);
        verify_limbs_mod_1(ns_in, ds, &ns);
    };
    #[cfg(feature = "32_bit_limbs")]
    {
        test(&[1, 2, 3], &[3, 4, 0x80000000], &[1, 2, 3]);
        test(
            &[1, 2, 3, 4, 5, 6],
            &[3, 4, 0x80000000],
            &[268, 328, 2147483575, 4294967251, 5, 6],
        );
        test(
            &[0, 0, 0, 1],
            &[1, 0, 0x80000000],
            &[u32::MAX, u32::MAX, 0x7fffffff, 1],
        );
        test(
            &[
                2460989955, 642673607, 1971681331, 2338977519, 475442971, 2516527409, 3470931196,
                4262052990,
            ],
            &[
                1430162564, 1873089879, 3090281851, 2861182896, 2502166555, 2486624904, 960387080,
                3073064701,
            ],
            &[
                1030827391, 3064551024, 3176366775, 3772761918, 2268243711, 29902504, 2510544116,
                1188988289,
            ],
        );
        test(
            &[
                0,
                0x4000000,
                0xfffff000,
                63,
                4294443008,
                u32::MAX,
                u32::MAX,
                8388575,
                4294836224,
                0x3fffffff,
            ],
            &[33554304, 4294443008, u32::MAX],
            &[
                0,
                0,
                0,
                4294443072,
                0xfff00000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                4294836224,
                0x3fffffff,
            ],
        );
    }
    #[cfg(not(feature = "32_bit_limbs"))]
    {
        test(
            &[
                9995257893114397114,
                9401504468144459131,
                558615837638945228,
                10733662027974786928,
                18295107704289976677,
                1814706268673753787,
                12474943759854623335,
                8814778832826774208,
                9159057654048965906,
                4451260977376821357,
                18241701617364042056,
                6169989192350218482,
                15071965537117101028,
                13509168527678537782,
                12224278653171635329,
                16077066393714953826,
                1433938684868066489,
                13014970036232570373,
                899282336249563956,
                3089487642230339536,
                3787737519477527148,
                16667686214395942740,
                8787122953224574943,
                7841835218775877827,
                9693303502025838409,
                16122224776459879427,
                144327425397945219,
            ],
            &[
                2350654041004706911,
                7834348511584604247,
                12756796070221345724,
                3842923787777653903,
                12373799197090248752,
                9712029403347085570,
                1426676505264168302,
                10586232903332693517,
                8387833601131974459,
                6290888746273553243,
                9503969704425173615,
            ],
            &[
                12688955427180652274,
                7641660693922643933,
                8789985477567049482,
                5698832637416200787,
                14684840547760545685,
                2822100467869581421,
                3557573565928866957,
                4409631974409684922,
                16994214656621423610,
                4513108841166793667,
                9009005527785483287,
                4330767427200269309,
                11409205475757922767,
                12430752173702915207,
                11990819624778098799,
                4145020291153594556,
                7099200056207569977,
                3510167930325480168,
                4731667122118327121,
                10720310942075546738,
                5799804483118787221,
                17268037247251138479,
                13305947798457087249,
                1405091439325174594,
                13072181651983436371,
                16122224776459879427,
                144327425397945219,
            ],
        );
    }
}

#[test]
#[should_panic]
fn limbs_mod_schoolbook_fail_1() {
    let ds = &[3, 0x80000000];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_mod_schoolbook(&mut [1, 2, 3], ds, d_inv);
}

#[test]
#[should_panic]
fn limbs_mod_schoolbook_fail_2() {
    let ds = &[3, 4, 5, 0x80000000];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_mod_schoolbook(&mut [1, 2, 3], ds, d_inv);
}

#[test]
#[should_panic]
fn limbs_mod_schoolbook_fail_3() {
    let ds = &[3, 4, 5];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_mod_schoolbook(&mut [1, 2, 3], ds, d_inv);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_mod_divide_and_conquer() {
    let test = |qs_in: &[Limb], ns_in: &[Limb], ds: &[Limb], ns_out: &[Limb]| {
        let mut qs = qs_in.to_vec();
        let mut ns = ns_in.to_vec();
        let ds_len = ds.len();
        let d_inv = limbs_two_limb_inverse_helper(ds[ds_len - 1], ds[ds_len - 2]);
        limbs_mod_divide_and_conquer(&mut qs, &mut ns, ds, d_inv);
        assert_eq!(&ns[..ds_len], ns_out);
        verify_limbs_mod_1(ns_in, ds, &ns);
    };
    test(
        &[10; 4],
        &[1, 2, 3, 4, 5, 6, 7, 8, 9],
        &[3, 4, 5, 6, 7, 0x80000000],
        &[718, 910, 1080, 1286, 1492, 2147483434],
    );
    test(
        &[0; 3],
        &[0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0],
        &[0, 0, 0, 0, 1, 0, 0, 0x80000000],
        &[0, 0, 0, 0, u32::MAX, u32::MAX, u32::MAX, 0x7fffffff],
    );
    test(
        &[
            1341443830, 680228019, 2358294753, 4240552485, 4220791420, 3445360969, 1267691556,
            3340881424, 992535572, 1421700486, 4136881179,
        ],
        &[
            342694080, 2041952014, 1126110022, 386493367, 2712278177, 1646877232, 868323005,
            1128141558, 2508964077, 2270408768, 413397861, 276293957, 2413870149, 3904853505,
            1276303079, 2556190902, 3967785388, 2803311564, 1963309088, 2281251697, 108392083,
            3821172642, 2048710986, 445614759, 2901896991, 3601903082, 1748962330, 658337341,
            3756770944, 2926642090, 7482898, 2241569433, 3381544088, 3275875595, 3036252835,
            2075586864, 1267727773, 221607050, 3946214050, 921175728, 655433083, 3594825352,
            1857831165, 3460211778, 2710641188, 1271591399, 263125866, 3619333728, 1041602820,
            3231202083, 259949419, 4156557841, 3419638674, 3270070813, 2084018311, 2951772182,
            860428573, 1138531819, 71044931, 3406637788, 3662652535, 3741808197, 86584983,
            1543795459, 160951322, 1039367499, 2428548772, 3604743617, 1138310711, 2881638514,
            2899388867, 3501166651, 161644747, 2395978477, 2682256494, 3417971200, 624066885,
            1772236754, 48275686, 3979745914, 2458268763, 2470110876, 3143857674, 3359279240,
            46246098, 2912257532, 2916568936,
        ],
        &[
            4267671814, 3181376526, 4204274026, 2772982826, 171341955, 3239339748, 83269891,
            2325049318, 746305994, 1290717192, 1805639230, 1856363646, 3445599715, 1388597950,
            2699498310, 246990054, 1396771726, 265087723, 3372840761, 3260397695, 2010115470,
            2211074582, 1883962263, 256363687, 4012920099, 2367091978, 1485341992, 1836339208,
            1434180757, 2616877872, 1145425491, 1261399773, 3926766027, 3115015865, 1741783714,
            3802513218, 4081750672, 3856503286, 2137512595, 1192985123, 1863041957, 700637154,
            2503934926, 2021744810, 157763047, 453501781, 1898727251, 3741166728, 105996483,
            3390681922, 1127471458, 4229032301, 254843851, 2729237155, 1707011850, 1756734415,
            1864969309, 96539023, 220456923, 3061776324, 1741151628, 1199010536, 2099212259,
            3095715999, 1369052975, 2575251120, 1821995904, 4169452262, 3299830336, 666854626,
            3647011057, 3965571435, 1551218489, 726151433, 1730753749, 750433250, 2084548637,
            475638490, 3209308986, 1536371286, 3458731894, 451976715, 1463077982, 2401462738,
        ],
        &[
            50882392, 3543059939, 2606418187, 171797978, 3702339267, 3400795540, 1675838746,
            3131956910, 3246773095, 2366066134, 1942572319, 200951604, 2751524401, 2229461380,
            953596498, 2149080321, 992246008, 2665146762, 3598754770, 101403369, 2364937647,
            3687287854, 3518388658, 3773344884, 2952272227, 2179289728, 3985568008, 2985468384,
            1733586261, 4006520661, 592636589, 1738983882, 3771432433, 334405279, 30528997,
            3353300809, 3694124666, 1257341879, 1142716002, 3178475445, 1431232435, 2733787829,
            2350134046, 162973337, 604966704, 1355497537, 2006066960, 1053669332, 2166548589,
            1297102232, 878976420, 1030095675, 3202690801, 955256572, 1487108550, 2945568597,
            3875785235, 1390419282, 1216776596, 786261001, 3784540042, 1845909951, 1777517834,
            12237310, 475982412, 3059699536, 4125207911, 2924612871, 3238196381, 2479221661,
            1483884124, 433754513, 447002430, 2684532044, 3494064078, 619277683, 2680056541,
            538552294, 1984267024, 700348592, 1278341418, 1614761367, 3944408375, 1269851701,
        ],
    );
    test(
        &[
            3656551823, 3383257247, 550091805, 1932339117, 3279901067, 2864941409, 3440756420,
            466783416, 651341619, 2318022162, 1201487629, 3064175522, 2345469826, 3581748650,
            2606599593, 3742095121,
        ],
        &[
            3689695988, 3117118992, 1237207334, 1180101208, 2947083336, 2827697081, 221923985,
            2507264800, 619323315, 1516562901, 3186068769, 1013769776, 1271582351, 1669896397,
            2497959088, 3504047880, 4212568439, 1608150067, 4150555892, 374787916, 889323122,
            3833122938, 4262272989,
        ],
        &[
            3226059476, 722714803, 1915993039, 2160412527, 1484228011, 1715426198, 1611850737,
            3097214810,
        ],
        &[
            1074470868, 2174113464, 1715956530, 3963027838, 1871758691, 1809783936, 3173233408,
            1926929270,
        ],
    );
    test(
        &[
            2108009976, 2838126990, 827008974, 4157613011, 3782799311, 839921672, 879731301,
            1926496935, 3712149888, 2363043408, 3863476261, 1703718215, 499165670, 2993223230,
            4279694017, 2976095205, 2193242813, 3593708162, 586354096, 296757650, 242624274,
            199363552, 2802933005, 2759687879, 3925865458, 4217980786, 621040829, 4008464542,
            3795907109, 270168250, 3625538473, 759155103, 3959828159, 1835721901, 1818377089,
            114690984, 2967220963, 2377698568, 2241245638, 2252300906, 3116551983, 3644576818,
            2608501946, 2181270779, 1815210824, 1268283136, 484429506, 1180516809, 559610343,
            2543044915, 3719740023, 558939794, 3989231279, 1089033996, 3509826828, 3545449066,
            3353643945, 2822587915, 2300900020, 309191900, 2371088678, 1169317833, 2688462322,
            704882387, 1429447898, 2514029776, 1955158460, 3862686818, 1584130288, 482000750,
            4224830364, 2104097699, 1183456674, 2270729014, 2607655660, 2966545947, 2931252595,
            759652385, 3970839358, 2487392344, 1507124762, 758697388, 3840603988, 1293583672,
            252424514, 646352064, 1756992845, 2309291248, 3392597937, 2946557682, 3659552460,
            1720311830, 3395887210, 2674135679, 3196424053, 2895118562, 1256983146, 2482795730,
            1922474903, 2105232035, 3692384401, 2891711374, 3496171341, 515080177, 4073860792,
            2051799135, 2279442367, 3258846627, 70977295, 1937954703, 2800340027, 1760719187,
            646417623, 3158839049, 3176203007, 213391749, 869819866, 3906922929, 1586646665,
            1614123406, 3687984980, 1177302158, 272342927, 1441818617, 2083855650, 1541781406,
            3311904586, 106907162, 4108429222, 662974012, 4140594255, 1984524179, 3259771450,
            3863443649, 3307256449, 3597197635, 3771753969, 3551628070, 3550518548, 2546464879,
            3812123817, 2299164636, 2258249267, 815945886, 3725185601, 738801212, 3862546442,
            207778311, 1535617953, 2150997157, 729581295, 143725958, 1410055499, 4204574368,
            2185442793, 3653861199, 1767186378, 3537808267, 1766021033, 2239408696, 1986025224,
            3625288948, 4046347844, 2490523057, 3491861717, 1312039469, 597857291, 206880668,
            1812928615, 3684502664, 3820772609, 1327777451, 1237379465, 764273090, 3594853086,
            2859643842, 1243019386, 3444655918, 1596721727, 2221238575, 2483346164, 2064518283,
            2282844430, 1960424205, 4028198846, 2342293800, 1645322170, 3508868511, 2376856302,
            1764497492, 2461899059, 3430260828, 1083983032, 1448541077, 2081767380, 1131293272,
            2121152826, 4212689224, 2039367869, 2193844060, 1864700761, 1182989130, 1481869846,
            969270162, 3997775597, 1483663077, 394034269, 1586435653, 2281434261, 1937512315,
            836935557, 59742568, 1854699653, 2116629465, 328821302, 1999851100, 604863759,
            556389113, 2057620108, 1992554416, 427465103, 813735424, 487055225, 402647361,
            3006998394, 2146302389, 2308517156, 1063328348, 2382344201, 2665908320, 481118926,
            2376219143, 1164055989, 128362737, 2711666601, 1898018298, 2928344090, 3862251511,
            749517272, 2790599579, 3506539103, 2592038317, 1682088920, 1332639030, 3053921374,
            3066891587, 242466377, 3536065834, 2016487096, 1321862427, 2409873623, 3088797737,
            2840818323, 2052021858, 4079299898, 452968540, 3880575723, 2680668784, 1502544423,
            1619882634, 1068420458, 3236073318, 1740351058, 1183565882, 2761799103, 246817911,
            2967032269, 4003972840, 969974269, 3193561842, 1002645557, 2347161653, 4272895273,
            1522072178, 972383725, 3026624523, 434293193, 2855690795, 2455822196, 2413168892,
            2999634230, 784531194, 249556803, 2559695464, 3131798530, 668810109, 2114036394,
            2861740558, 370716909, 2126014070, 3913884200, 1577476936, 855642956, 2074597201,
            3461659621, 3292762038, 2153122643, 1658901071, 1562860969, 501980289, 1611843440,
            2433026108, 3791871978,
        ],
        &[
            2325147333, 1406197332, 1502532645, 137754493, 1488912421, 1014222449, 1678300831,
            2078887154, 2311629707, 3855402290, 1904906031, 1350026944, 314061674, 919972212,
            1429252796, 544807358, 2607111216, 3001219316, 160533550, 2875169678, 1402474615,
            3234840532, 1662112023, 1943050521, 4132256985, 2915369430, 699620367, 59590583,
            2461837586, 3986106860, 2659659366, 3589796255, 4168995860, 2337625655, 4292546333,
            1793085295, 2647770730, 2871508164, 2660729400, 3630964395, 3874212033, 1240042620,
            1713051992, 2105061240, 968807145, 3026819276, 2442427316, 2718248469, 2572140009,
            1793916894, 2112926236, 1058005225, 1528978151, 98430379, 3348888463, 2121431457,
            3545708238, 3975070402, 764453596, 192861224, 2698588322, 1493819664, 2502294971,
            3139295505, 1318649192, 1552623529, 2989328476, 1592829691, 1175278450, 182489613,
            1858350437, 2440903999, 1622786339, 162763913, 4106583944, 609833788, 1344925844,
            3910333895, 1369398307, 51929280, 2214052160, 78601532, 642016948, 3966440550,
            4161875340, 3764174238, 2944905670, 832634369, 2082700552, 3444794582, 962611192,
            38510846, 2733538376, 2141542785, 621670923, 3130866640, 1666281368, 3628581169,
            3965100367, 2889091703, 762371524, 3752935086, 656901001, 2374421435, 3084212494,
            1186140919, 3420046356, 775236355, 1337865306, 2051118555, 713071112, 1946850844,
            1427068060, 3949026319, 3075128554, 438166063, 3975561046, 858907746, 26964870,
            3843362808, 1461820702, 3090970200, 1037540471, 289966660, 2768763862, 50890120,
            580189324, 2911422550, 3684738914, 2025034397, 414622319, 2287243961, 3268939669,
            3547523099, 4011963324, 1768695320, 1859343614, 2123856143, 4141054481, 765801396,
            359993985, 3668605792, 3613648266, 1778728280, 2547397231, 2456061149, 2562918666,
            2903450513, 1994190773, 99234624, 3722083920, 4262323306, 202219441, 4201857695,
            3988878636, 1533308334, 401400520, 1069756554, 2457773969, 2892388936, 3423117995,
            1944069442, 492036629, 3426800580, 2282483359, 4006366620, 1695364515, 2555180845,
            1669287836, 349290429, 778467450, 2020203604, 2218159817, 1450404019, 1278304750,
            2412695340, 1592154884, 3868182043, 2240370481, 3859902860, 1008825116, 412233394,
            2475457637, 3664379433, 4204584226, 2750684469, 4113507475, 2916584959, 285955744,
            739598569, 18278051, 3768126932, 2181905109, 2612988076, 1827656088, 1160380415,
            4160443718, 1846086671, 3050604645, 2547108010, 2828666778, 3252702258, 3885923576,
            2331974758, 730724707, 1528859315, 4288784328, 3677151116, 445199233, 3304488688,
            3566979465, 3541025426, 2491779846, 3112990742, 2583249486, 3403111749, 1930721237,
            3428792463, 2896462048, 2985885576, 1819460734, 21206096, 3560441846, 987100555,
            2844904275, 84854892, 1268249628, 3963306788, 3338670067, 2504599089, 65588657,
            321493327, 4249673617, 4150876068, 721566898, 2186945060, 922948272, 1502464627,
            1426914435, 2906888275, 3454987739, 2609132626, 2073366782, 1058809001, 1226951003,
            2624503637, 282722778, 1407178266, 3304346308, 3613017687, 2397594777, 161128076,
            1938083248, 3042822216, 3958145306, 1431161184, 4147630074, 1744618540, 3729847703,
            2366216428, 2101731625, 2812677939, 3587016062, 1626785569, 3117017254, 3720581461,
            4254564252, 2400911475, 750362271, 3062375802, 2182368701, 268751862, 2682134876,
            274793576, 2417658182, 2321240923, 4200603818, 1349142841, 4066976957, 1043247680,
            679021323, 2933095037, 1621487581, 225868669, 1030678959, 1236344244, 972640788,
            1784301152, 578117349, 3509551550, 214420003, 3786992737, 802372148, 3562194086,
            2187046946, 3600373521, 4275090943, 2120016813, 4177241875, 3185774231, 2397692077,
            1015362399, 2178889151, 3433916223, 1688082118, 1971242178, 236388706, 3802829765,
            521309115, 2299816689, 3207614143, 1053195464, 3584561145, 1178690670, 2940812254,
            3321982035, 2754825123, 3073598062, 202404806, 547895545, 1188944547, 1056841779,
            529463805, 204665384, 850370055, 2063320161, 3724100092, 1180272690, 1398467003,
            2814052449, 1311768018, 659771105, 3226477227, 4230080238, 2134344405, 1461172705,
            2728018383, 1816821358, 3231137250, 2012377728, 2206916761, 3121807673, 3037967028,
            3653505029, 4164239058, 1337501466, 2487452346, 428839025, 3321493092, 2920064796,
            2750126813, 3635762100, 2659829309, 1992243088, 2359420187, 861816585, 4182313312,
            1389528266, 2890138969, 2195001695, 576635805, 1974585967, 851480113, 1896534852,
            3143828462, 4276040597, 3921135859, 2394233682, 2692972118, 2103941456, 3931376544,
            3790764632, 2278428793, 3381340311, 3516241824, 2923157557, 1276268033, 3471729699,
            2370137091, 2637023305, 4027814477, 3711009696, 3667995314, 459819000, 1002878772,
            2658190707, 1189561796, 1831813150, 2874008201,
        ],
        &[
            1703326352, 1625259628, 3642322228, 911402341, 2158835226, 939248485, 3607511108,
            2863853568, 1611642161, 1312857772, 1839433327, 567060478, 3139863681, 3642698184,
            3744632443, 712538472, 2692932947, 576185818, 156934113, 518107105, 2803035863,
            2284220097, 3447382922, 2400125006, 3565062840, 160044186, 3644393084, 4196433258,
            3391883838, 1115703759, 2380388002, 962895870, 4001772616, 2311278419, 2620271020,
            3047789793, 3229254302, 3182628087, 2718480927, 2872538422, 1963990826, 2856537226,
            1729736384, 372544200, 1482170753, 3370665422, 305555060, 4060802437, 1456459795,
            2848113958, 1036273297, 3245835346, 3355862097, 3242734427, 3313897419, 4230036415,
            1425594704, 2428756491, 420147714, 3218149930, 1791217142, 2881741659, 3231612869,
            845372829, 1635665046, 1525482197, 3779462557, 2279005731, 626908159, 2963734610,
            215448882, 2517668355, 3294128769, 1167990371, 2344651228, 2882002195, 3350056381,
            3749638142, 476010196, 2304276366, 1088082897, 1927672185, 3316329053, 1174989749,
            1101596429, 3054511229, 1685228241, 2301284206, 2526016461, 684171580, 3869889559,
            2606492401, 2114658735, 1419214215, 2779052449, 2594333668, 2225446358, 1211258458,
            4029343999, 2816277230, 4098142949, 1987363952, 2732004911, 2516355975, 2365350298,
            1747416524, 3951513077, 3526462790,
        ],
        &[
            1707334789, 979340170, 3926273349, 2029694660, 686358037, 1260123629, 689471236,
            4127009836, 1697317324, 3024229658, 3968952300, 2265029759, 733830093, 2234057965,
            799627406, 1641742523, 2511563041, 678204198, 60484965, 1732955108, 1348641815,
            1389334445, 3762643904, 3914179461, 4076462931, 3234877741, 2806715190, 3058911852,
            3068848122, 80861952, 660210165, 2035863667, 3882844515, 2652987274, 207029959,
            4170201248, 1224653886, 3554890284, 323203565, 969583090, 2280910221, 2548241625,
            2048045350, 3691775573, 3750377304, 2623340073, 1726676400, 2551234664, 3420452119,
            1785849707, 3581615935, 3521097022, 4008380099, 4200557552, 3795120169, 488463511,
            3784467967, 2462685342, 1684084055, 1072125823, 765304928, 2900468163, 3209242917,
            399160769, 3897765190, 3241572534, 1027203705, 4127226109, 154941175, 1755153596,
            4225252328, 1209101989, 2906888698, 1481930532, 2884731435, 2416462752, 553651049,
            2247421040, 3347507436, 1936621186, 3156059073, 4059864280, 3191940359, 3872902453,
            783002816, 3711047515, 1471051688, 357992570, 1061754510, 3765779801, 4106840335,
            3213049794, 1824410844, 2555550067, 1066295382, 3248869963, 4006861446, 3297772017,
            3418388327, 2284151457, 766215224, 1091253635, 2070830666, 1435103841, 1408352671,
            338399542, 2946293497, 412804347,
        ],
    );
    test(
        &[
            3333140561, 2349469031, 666934289, 3646788197, 3373340607, 3062489357, 1781577064,
            3971231072, 2027939726, 18149756, 461121573, 1026270057, 4154481498, 3854995191,
            2829169318, 2604923553, 1399568275, 4214150257, 1269657447, 527209086, 559827800,
            1978807620, 1942748659, 3544969161, 760896836, 694370989, 3323597771, 1950755821,
            1740588707, 2716546795, 1636681281, 131422512, 2683077565, 4107286368, 3475458482,
            1406752921, 2741526831, 625075474, 1356017187, 660778033, 4078598031, 2127023485,
            1900179860, 2413773277, 2442990944, 850894581, 1061627184, 2255197913, 490496693,
            400007380, 3092474488, 1228905086, 1949288324, 1866222251, 4084867584, 219026194,
            1513678816, 2415719293, 671740632, 2331463444, 1649508273, 622524186, 36198287,
            385821590, 3891184984, 3888867762, 823766845, 871168771, 4220135146, 3182319759,
            3811051122, 2244319794, 1994956782, 3515960993, 3523512148, 4142104035, 11404747,
            3407068856, 2661473296, 1343912700, 1278923149, 1319603671, 3928437020, 2678906106,
            1133029551, 3498992572, 3960506675, 2156562886, 2180928089, 4027866586, 3363079195,
            2699625088, 10633112, 1776348429, 2657266039, 366882369, 3755249663, 1513004482,
            385462618, 1235118123, 2498536948, 2963795538, 373052379, 4211662453, 1125414031,
            3132282357, 2555861787, 3949045845, 2694335672, 335933961, 2882374566, 3487775198,
            1816750435, 3017829161, 964613564, 159555703, 1822850997, 2275865751, 4158541302,
            2501990511, 3721844651, 2781838144, 1795741626, 377630941, 2137086277, 150771080,
            3410699853, 2181744662, 3165518125, 156617039, 3099687100, 2159351188, 340108091,
            2408725718, 1639869303, 1752843886, 1941447467, 2157946488, 1515895552, 3624132653,
            3556726921, 2697391196, 3650363912, 2596921683, 3641623430, 3884964637, 4229700714,
            1936259996, 3765315225, 4087511128, 746291101, 1752759843, 699518598, 2102672534,
            2335948897, 4138231225, 3162828440, 2881768368, 1254885146, 2977691172, 2250072411,
            2614007259, 2771551612, 776074685, 3162839614, 1471012556, 523323297, 1856466918,
            4090405884, 813410161, 100155385, 1084252628, 1599609446, 139407955, 4121369443,
            1963225032, 2735275753, 100107317, 1064061529, 50786861, 1654370440, 3334472166,
            1385803654, 3881645932, 3538258401, 1238925138, 1307191791, 2439918414, 1853442725,
            3584719283, 2139830944, 1008033673, 2159214266, 1728151904, 2054140804, 2246372905,
            1296034828, 732486414, 4197166746, 2149038695, 2421556216, 2209917678, 2053724924,
            3695308867, 2810551805, 3486105675, 979515789, 2065617241, 3321246575, 4275036932,
            3028497318, 1883398852, 1301060583, 2486427309, 592113155, 2482121819, 53211642,
            3067158128, 2316354328, 2457370797, 1766767168, 4130364650, 1394191393, 893956183,
            3188688185, 1875859783, 3686866863, 3294062742, 2146834313, 2406586432, 1574602180,
            3208002016, 1067708846, 1620126309, 593911816, 1573785767, 2208656217, 4037191927,
            3437397556, 425721135, 3749387007, 407468749, 4203518167, 557223936, 3186718435,
            3473504989, 2673679280, 1478635903, 4172750097, 1469776418, 2542136327, 941914936,
            3438581078, 3116581940, 3479694655, 159558945, 3809792710, 2458053073, 945749848,
            3386290936, 950397544, 148027778, 275178526, 1664599931, 2000347485, 3312866545,
            4123761723, 2408292910, 4050298, 4225958969, 2745635153, 1877283873, 90573827,
            3031684040, 2275034810, 476532317, 4006527249, 2818283214, 661127690, 2880950746,
            3285389422, 965419004, 1273458580, 1611437669, 4165036259, 503131762, 337440890,
            1274837150, 3486913543, 1110808141, 3573281492, 3543428578, 4165550725, 79661970,
            1465954224, 4160343768, 3753977684, 849800837, 1568284940, 1214508489, 777072754,
            3766232553, 4163385462, 3428999259, 906027128, 4231567110, 1070253805, 2975789822,
            1794042990, 2937404746, 3315145973, 3978541255, 191961431, 2940436214, 614874984,
            868388988, 4078685796, 2613877234, 3380127687, 2664922311, 3262235038, 741871774,
            1144269180, 4258950699, 3668553647, 3162392321, 927199539, 557637561, 1114912670,
            84346866, 4241634616, 592125931, 1914166161, 1225886409, 722682712, 2163011354,
            2585075265, 2786722853, 1993844439, 3433052694, 3718814733, 3881165856, 1749980152,
            2115861966, 3664836486, 1644814678, 603487648, 821270634, 3041437740, 925329308,
            414275079, 3675481275, 583965775, 3319812843, 270816609, 1752806861, 3191564196,
            3197116565, 2440000620, 4074382088, 3346183865, 176525856, 3011091744, 746158938,
            1522027948, 930706422, 2574400675, 4141639266, 758742087, 1418306156, 493509688,
            4102390641, 3757183059, 2174506350, 3715933078, 3450125642, 1345548124, 3845070193,
            1563795423, 1304049315, 2604880108, 1998271877, 4078353958, 966179722, 3401781407,
            1708183977, 3890171386, 698646214, 604975097, 4189977050, 1847675543, 2745048720,
            1585177380, 3886892753, 2534712423, 1830125662, 796996186, 3789574770, 2219095884,
            38427113, 3653683439, 1791769536, 3941686320, 2991883118, 1081207663, 3863639548,
            3017351034, 741393256, 2333573607, 2783158305, 2015776314, 3034707152, 1871224010,
            1795881414, 928539629, 3211747317, 461147496, 1512738445, 1178891684, 2603472964,
            1694950214, 3121022285, 91788635, 2303571910, 3259041724, 1240577113, 827064078,
            3847765430, 1254285617, 2609289648, 82835530, 2235563901, 2939680484, 3327268666,
            3659752183, 1783109461, 2828509671, 3804296197, 406800911, 2037292519, 941994402,
            3350211940, 4004870359, 1105941782, 2367553097, 2440982990, 1468367, 2369391483,
            3613174135, 2337161925, 2600055725, 1612773644, 1037201309, 919122459, 1114016493,
            670586694, 594109706, 113464329, 4093636009, 307208587, 3876235186, 3370598978,
            934478190, 2843794891,
        ],
        &[
            2368917411, 3160758175, 1531480588, 2746326450, 3122359045, 1995296172, 310451197,
            1100329853, 2141732304, 3562890277, 4172526924, 4143185198, 2397101926, 620168044,
            828452963, 3271569899, 2726758945, 1647557222, 2748652901, 1029871186, 1773072595,
            1311795717, 2792240418, 3100733307, 298902661, 2608361440, 642743765, 3026335712,
            1586592828, 1251923561, 2152803283, 3359308047, 3979690761, 2548031009, 2148504694,
            2514911217, 401310800, 1418125404, 2325652800, 716233458, 422262103, 376190943,
            1713380879, 2534508846, 2080413009, 2690193951, 2909422387, 2435837201, 176977642,
            224472737, 1590904934, 1664979624, 1748982641, 1284257790, 2779881254, 385265989,
            1148527382, 800142050, 3595556318, 3950265146, 3931111523, 3399894595, 4200004994,
            3727110364, 1510525540, 429323681, 2586345405, 3441619670, 94654169, 2266545045,
            1451808026, 286040435, 364003484, 2106693078, 1916214330, 2622096560, 3504008387,
            1867458297, 4209615436, 2899684845, 3149003214, 2731979274, 2481557740, 234269740,
            1239420776, 2726912636, 1844267393, 1488044058, 1587005400, 145015793, 1637696129,
            1990069464, 3053970062, 676336554, 625771493, 1492378707, 1231018082, 3059699722,
            4113615249, 2967103567, 3335726303, 2137774253, 273460502, 1561489611, 1227429414,
            778315897, 3061631993, 2437789514, 2566029814, 3017507824, 3219754602, 35255136,
            1934306764, 1655248959, 3655963814, 2821596564, 1431977748, 3114012833, 3915822059,
            343010167, 3704265251, 2691314567, 2602475242, 2459094882, 58857240, 2230690665,
            1357656795, 3025543277, 3488674640, 408985076, 80729036, 246851391, 1419653386,
            530010128, 3529087076, 3980421141, 4129023783, 1630771696, 2057278583, 2501545742,
            790764428, 3984347668, 2562445985, 2262348292, 2515967925, 1103371818, 3735595401,
            3210477145, 3387257181, 538884374, 2695206119, 4211749041, 1006274735, 2406061865,
            2654644297, 1268724640, 2399689302, 157759259, 564853502, 865205768, 2174783618,
            3167863384, 1770172407, 1906775255, 3171669377, 2455086709, 1355327864, 3351895167,
            4184858376, 2699533565, 2939169294, 1702277740, 3447680482, 3322351998, 4248517760,
            4100425921, 1169799041, 524849931, 1743345401, 3987645088, 3022177240, 3263737545,
            3656340460, 239445170, 4089162190, 1208712978, 1453215235, 3390225374, 3337889603,
            717395279, 1066458381, 2162685522, 4009460245, 2575879990, 2578471337, 4218103221,
            3046325563, 1352991515, 1561949281, 3488911174, 295633326, 1993756395, 3677304657,
            796898320, 45622345, 2999482773, 2078651788, 1552765091, 428170722, 1748054862,
            4158222865, 1050292437, 249107, 2651086861, 1540178674, 1275557298, 651050585,
            1895599156, 4150309716, 2064336046, 1021257319, 2422595446, 3116755258, 2756656575,
            2328011578, 3632905157, 2576200202, 830418644, 2430246370, 3913080080, 2371749061,
            2683067461, 3611558701, 1603392737, 796477803, 604567756, 1376069347, 1491246154,
            2728262664, 4138498935, 3008562381, 1061547384, 285679033, 2358943172, 1884649492,
            1783482693, 1010268161, 176114433, 794165875, 2362278477, 3903204233, 326754905,
            1988607850, 3187254334, 1749797209, 2986661384, 1759716588, 3137467938, 4067743599,
            1134210801, 3799848836, 1955405545, 3881788427, 3097574490, 3844756657, 3183850151,
            2496328910, 1468671385, 2888878911, 2306353811, 1498824361, 4152891378, 1588217107,
            79108222, 2883552792, 2390312777, 1587172303, 2070384343, 2265280181, 4013380367,
            2742676878, 2654283484, 1471778694, 970959698, 1006151052, 1276307400, 2874101774,
            3169092608, 244587925, 2402787407, 1635687252, 835856534, 321407542, 2307278464,
            2272745321, 2574317642, 729437319, 1682288870, 1482920833, 776000268, 3908963888,
            214874919, 4233311318, 1441385448, 2358127573, 2753681514, 467574656, 4139948165,
            1538275035, 3244920878, 2576965792, 584068468, 3054546876, 2629688518, 4253271747,
            3723872815, 1652066683, 990821089, 2335421805, 1989570928, 240486517, 2872315587,
            3869991906, 3870517664, 1540804424, 397183643, 3750033565, 1433260634, 1506168711,
            3616651625, 512028445, 3746712828, 3278592880, 2611514549, 1214563129, 1259227909,
            1067976218, 3425169051, 741795595, 893688343, 2674408703, 3694908868, 2478153735,
            2220661625, 1022546736, 3719214155, 3161293211, 4131981986, 1473264088, 1651777063,
            1438502715, 290022167, 59234682, 3458968160, 2552001459, 3451530289, 3800073253,
            717882913, 845719525, 1038699111, 3058303772, 1117505279, 3682430977, 2869037104,
            2562493618, 960519305, 4147639705, 1817463351, 3166022129, 3200769866, 789666262,
            2654485924, 3686362402, 2179867687, 3980226915, 3671542918, 1896992204, 1514962591,
            815867715, 3924270086, 4262628477, 3977258034,
        ],
        &[
            491933121, 4095154087, 1190388341, 2123387578, 3299683037, 3840239224, 3891252433,
            3823858915, 548023871, 3189021809, 130119335, 1406023029, 1691542279, 2101470388,
            367145009, 521045073, 1370995123, 202251222, 2377400220, 3656022509, 2413445446,
            3583191945, 150872893, 1253881977, 3664619565, 661519973, 672952411, 1111416155,
            2582282747, 2253466637, 3993637585, 1023965093, 4055175549, 1721196160, 4271105372,
            1844335262, 794907961, 3970777280, 1279126024, 3389187110, 6388771, 3557792034,
            2503975364, 1216157736, 4081863507, 1212796362, 22717034, 3517296783, 3126258972,
            454147562, 4182968994, 685295412, 2996150679, 840677609, 2775806132, 2555856748,
            2855553012, 2971949846, 3057579317, 954558068, 1251032539, 2673204563, 697957636,
            1312479140, 2686740988, 2005131101, 664387193, 2169938189, 4081160067, 3469684725,
            123152292, 2899902365, 1142266692, 2066860665, 647803613, 3574778631, 1302596512,
            678200322, 3509569151, 849744577, 1521760114, 3195673981, 111241164, 623461570,
            3571445453, 200189218, 4138515602, 665845464, 4184871542, 1353865395, 2913489035,
            3076207781, 2549062531, 3446687793, 685940105, 1430268106, 2488363196, 1172253595,
            4151501269, 3292280286, 2025935004, 3388911026, 1080796771, 3386986959, 4175562557,
            2440556214, 9871988, 5488492, 4179517076, 908443542, 364965294, 2303037125, 369102673,
            1570100894, 3615415741, 3133015360, 2623933135, 211314818, 3377760249, 2285528104,
            3096807957, 3213907502, 3633343924, 2065769028, 300167487, 1745300631, 1943126607,
            1065862406, 2422304458, 523765517, 57630113, 520436682, 581485167, 527613705,
            1677813796, 2698743484, 2448718551, 1681977324, 4183961145, 333724319, 2667028740,
            3403027352, 1499647517, 2965583197, 1090547876, 1536215624, 3407325498, 1392474450,
            3354142052, 3326326224, 1138672330, 1577104875, 235932133, 3877789753, 1822119722,
            305251772, 3802044423, 608489509, 2727503310, 865373313, 3763994776, 3206318655,
            88097048, 4241064408, 1171067683, 4250936686, 729121178, 763857558, 450636528,
            2096197907, 1427186250, 1251064262, 3036762405, 2966424923, 880639864, 1995310474,
            2023015792, 4198637421, 3996151657, 3070092174, 331733799, 2680348059, 2601390369,
            92920881, 600583651, 1323663334, 4258529164, 2493258032, 3313489774, 2050494797,
            1494880112, 1342788885, 2493538385, 1096813856, 27701253, 2762164402, 3934044073,
            2289089808, 721323988, 2246162231, 2514451824, 3095057964, 1921436785, 2847276160,
            1369952734, 4246781184, 3442030977, 3930627222, 614706399, 3051396564, 3044240928,
            3450585985, 2120836933, 3006035710, 178202067, 736149219, 623937260, 1958379885,
            3104814268, 3269365015, 914726129, 2951180538, 3940965189, 1869582492, 2599100173,
            1009213592, 1410445689, 1179880005, 3583285938, 1921636631, 434725812, 735782942,
            2548761027, 3733379485, 3466163816, 2455801926, 1673952134, 2422059957, 4005305357,
            2795894062, 3245947918, 4171997564, 2269244989, 1975479474, 2449387935, 2550893401,
            4033149, 3723976428, 3871853952, 3210178057, 2062131312, 1434821634, 1130065536,
            4276609753, 41976415, 968125282, 853099726, 3675357389, 3545982836, 1237895428,
            88970686, 4259275077,
        ],
        &[
            3428605691, 3747865452, 3179948115, 2350061225, 568002466, 1150456745, 398797276,
            1289784855, 1713342581, 1243415582, 2262476431, 2640854581, 2126611553, 279688048,
            1750838292, 4147801215, 1090003542, 3022523567, 193306367, 3095010320, 4106129220,
            2615828356, 2229875117, 1066352114, 1628397898, 370050060, 4157038478, 3551524420,
            3356007005, 71309641, 801692593, 3448737794, 2187893878, 3694205688, 3993669226,
            1704384353, 437976508, 281233142, 3685881986, 557022935, 2623696682, 3473835466,
            2373703925, 4102552649, 3005124737, 1604665678, 2894612216, 113726438, 326916434,
            2354584210, 2207672568, 1272863868, 1643657755, 453968751, 3873638645, 2388412558,
            2026577065, 3555348237, 2342844423, 564662317, 3301275728, 2205414701, 1980322846,
            2111846734, 3351655044, 3986317737, 3746472405, 4065813457, 3783710419, 2063617629,
            3337397526, 396000532, 1367889075, 3493608147, 3502046919, 4063752078, 2026966957,
            1730343114, 1889308216, 2064908820, 1350757250, 953211432, 3171180512, 3683852864,
            2865517189, 9728788, 4075807873, 230559850, 3447666295, 3875782338, 2964025959,
            3279832060, 1701408562, 1524777719, 2018628821, 877683542, 2894422993, 2842238026,
            2368190181, 2574790540, 3301078931, 519264490, 2640226714, 1382986517, 2712031077,
            1268830384, 2018243821, 1374899410, 1932087493, 4236194373, 2079152365, 1069353434,
            2794047491, 3019576051, 1360117274, 3615353679, 3816194597, 563906666, 1827758233,
            3391740972, 4039283822, 474920421, 602146150, 3520459365, 2217822485, 1815675107,
            1257332805, 101198736, 3156910259, 1281759597, 2577040428, 1316988841, 790502779,
            2183328888, 3514409721, 3117638288, 2751392867, 625199304, 4183171081, 810287943,
            199775540, 2896399552, 1276626819, 3536203521, 3977249862, 1553852925, 1596788767,
            4199876212, 4164476281, 1700598937, 4111801568, 897035781, 3318378574, 4049506305,
            2522517679, 2296942227, 91515126, 2738354614, 3444113221, 776723413, 2322159108,
            258188842, 4014626824, 2677530556, 2630906411, 2348643583, 2619411399, 1783064149,
            3684137491, 4249816060, 2342260236, 4186117248, 3931377058, 283592002, 2818946531,
            2155019855, 1212182985, 937681548, 520081464, 1686760025, 4003314294, 120323668,
            3282730481, 1331500167, 2103184161, 453960707, 3079516933, 4193898972, 279508919,
            2459211929, 4262295127, 1413249938, 2652447340, 1521334653, 3641279725, 1047788888,
            2569690437, 3517124875, 527659101, 950373677, 562851515, 3518587604, 2668894245,
            2265163836, 2975852422, 3168107907, 3219343979, 3364338005, 4122186542, 3517854206,
            626303838, 201513242, 4042863292, 4203206994, 154681851, 1046188872, 1014035958,
            1565481708, 783475007, 1419399714, 1439418252, 3671223860, 338436030, 2221870804,
            275182251, 790279697, 3288665213, 3033639795, 2320556485, 1648356934, 326278183,
            429784181, 104482651, 1051376296, 3823684587, 3057907898, 4214146560, 2770254088,
            2132180852, 515232230, 3723417512, 3994583184, 1170935740, 565152109, 1486190861,
            2906472005, 3154973145, 1571950735, 1391459913, 2972445791, 1670706254, 2522911431,
            2665886703, 2875123532, 623767904, 1291947264, 3935970871, 2500309617, 3931020115,
            3436644927, 3672390796, 2307254280,
        ],
    );
    test(
        &[
            386353625, 2283334827, 253851108, 4279287864, 2561872983, 1000131216, 216965099,
            1733322743, 3814906874, 520558483, 2573981410, 3011109093, 1361665859, 2981910167,
            2501833654, 2673979284, 2696912361, 797890797, 97183495, 2324611911, 3471887001,
            1013155374, 524729310, 3971281568, 2873934524, 1076401482, 1645093822, 423739674,
            3400177353, 3716049850, 2858905860, 3351077705, 1478568656, 3370413874, 3966773377,
            1660817988, 3770638166, 698194415, 92466505, 399941585, 3301477423, 2578917060,
            3012966571, 1915213695, 3062576400, 2979801504, 3838450560, 975685759, 3154985631,
            1986911427, 3559636850, 2134674515, 3128102806, 1097660929, 1212690783, 3611467219,
            1232001761, 904465835, 671601217, 2829397960, 3910014855, 2035732477, 3262086074,
            3077756473, 1064556527, 3095006074, 436186642, 3844079107, 2798261317, 3470373096,
            218408908, 4184804463, 2826021777, 1939569756, 1452808097, 1149547097, 316749819,
            1631475409, 4023391942, 513096309, 3730431754, 3391919570, 4153138767, 1328779762,
            3015228258, 2487205968, 213982794, 1927310263, 897606115,
        ],
        &[
            133654450, 371159391, 40768202, 2113895976, 2732923987, 3215138425, 2391931662,
            2672734577, 195120416, 939150931, 1941696822, 2919872876, 2564539452, 3080820908,
            2846583634, 3349568397, 3465385329, 3550405793, 1724556069, 151917586, 3772273289,
            4156003461, 2380288065, 356475766, 4123619567, 1619017859, 3739529130, 490749645,
            267879469, 2890847693, 3854244893, 3384266657, 115723042, 103663103, 267682519,
            965965574, 1841386548, 2476428873, 6965648, 3347378817, 4174270937, 911558590,
            1958087705, 1467103561, 1925556384, 607664404, 2051263137, 3092553737, 3239342455,
            1360744647, 331798903, 938411111, 3063265275, 4144610940, 624250586,
        ],
        &[
            3712980585, 4273305215, 2588121912, 1482202110, 247010829, 2649827458, 2673293530,
            50266420, 3870816552,
        ],
        &[
            3140142351, 3638334268, 927590030, 3257445174, 190628332, 423978494, 2125526010,
            2335881748, 2011775260,
        ],
    );
    test(
        &[
            3451729766, 1173985848, 93029266, 2489920009, 3973680219, 780152687, 1953113811,
            856318718, 3314090115, 2569865885, 10849561, 743853677, 1946859719, 3063924404,
            3352045106, 101532091, 429420318, 3158186124, 1342100634, 1680882573, 4042659776,
            565667221, 989688746, 4541446, 834463803, 3523014310, 3432977627, 1818084880,
            3544409611, 1926948225, 3663387900, 1035293697, 285923430, 3096314023, 664014848,
            2054069437, 3124990832, 115905130, 1379323203, 2244054884, 2012040660, 1620087465,
            1907788850, 132467852, 3020885577, 238576412, 567018667, 3426013542, 3145012002,
            2070201540, 1421424259, 2074360440, 810615969, 153078550, 825376847, 2436604908,
            1365021492, 288298963, 3753461462, 4215347696, 514661745, 3484458912, 4247537127,
            1579462876, 3865189996, 1018915575, 2974009562, 3501816358, 2821759336, 2722697353,
            517866351, 632193686, 1959130106, 2570573609, 3726681184, 1695074321, 1803440017,
            268435105, 2059636298, 1870796488, 3616705235, 779907146, 3840642834, 3690603635,
            889240128, 4220497978, 1506894274, 3744062951, 3031958651, 4062058444, 2309758423,
            1411665710, 1638698058, 1784343146, 373693019, 3239595939, 1656022725, 1985062241,
            2188775097, 395478226, 2732514978, 2804731752, 4294707867, 2796289890, 4286815139,
            2694259304, 3400027223, 2866208426, 712658287, 3005677674, 3799280383, 333964254,
            426632823, 1071506508, 424615774, 1216913008, 1528084851, 2440789637, 8647650,
            1482353199, 1079815951, 1741478231, 335212928, 3230225069, 812456049, 1107265661,
            309740350, 39813940, 1038050523, 3708554333, 2276389162, 3145208114, 286364489,
            2078653938, 2343246544, 761958231, 2400118006, 2820784413, 91083352, 868359282,
            2976918690, 1899156995, 2927253740, 2971404198, 955553520, 2163090892, 1779162381,
            3706789234, 1427323973, 2892945053, 3765307938, 1888304940, 4281185143, 4063224403,
            3895821018, 292451283, 1315405597, 1278053061, 2659048964, 1651475603, 2464063324,
            2470117886, 885060568, 2223396118, 983203622, 2153273837, 909540554, 11662729,
            1290088356, 3558268851, 2470462290, 3816443571, 1219777270, 1663771088, 2059674235,
            3741754207, 918293007, 1170814255, 2592844500, 34078497, 868179007, 2916133370,
            224466504, 1457353443, 743181292, 2045016399, 4268051489, 2582113750, 108054210,
            1045580699, 1021681246, 4287699909, 2442965928, 2358040249, 2188665290, 2493816549,
            1868457202, 1538714165, 3141445884, 1886427078, 2413842865, 3754082182, 1242353411,
            837326189, 1891378866, 2004934119, 2718308820, 1293740965, 3706050263, 660923758,
            4037329521, 1659464686, 3767070633, 2733667846, 1303579884, 720201259, 4178102594,
            1521257108, 3676354960, 2861869547, 1694835235, 1663377877, 1033893232, 1788049008,
            2131980496, 1380320795, 1706971693, 1547085187, 4249747381, 4022331904, 3773862935,
            186851005, 1122661765, 2869769810, 714482846, 122240297, 1990969517, 1505355345,
            1418846887, 2157766299, 3944099174, 752193837, 2717991695, 3106584127, 2176113460,
            2424010054, 2202537322, 735567917, 2635174646, 57167007, 1501943516, 3981020168,
            2264071537, 1811387936, 3230232665, 2056500724, 2470649052, 298208196, 361093257,
            837772742,
        ],
        &[
            3759203346, 4116788486, 472182301, 1657732406, 1379670583, 1302865697, 694415200,
            281678172, 1654699562, 3014506006, 1741431230, 3860463155, 1145719988, 3566676616,
            2320845102, 2321995000, 536415919, 2387437419, 3567575351, 89009414, 1112333426,
            3147195251, 3149954561, 1503279897, 3064049622, 172526224, 3226367243, 4252316042,
            2985921401, 3319920602, 2971463135, 662010514, 2716903751, 3114900209, 4294888231,
            2302528059, 2985636330, 4227410510, 523042380, 1128527265, 2237860860, 2902190407,
            2028066064, 43980114, 85965472, 1423570354, 370469915, 2799642260, 678694776,
            4007712353, 1236341524, 2105667795, 1513858354, 3031596159, 3550734930, 4152331215,
            778814882, 736616127, 1334786757, 193672910, 4005017476, 49451649, 274162308,
            193472860, 3849716547, 1147344616, 3779102036, 636364760, 49841072, 4203872775,
            2917594088, 2516158563, 1718950767, 2409600701, 4112139783, 968713133, 2367692503,
            1383637739, 1447827357, 70112330, 1597216524, 1036612170, 47442040, 3535655506,
            3217988958, 1524147015, 1542229372, 4278993048, 1177373055, 2455939135, 3977288741,
            2323766505, 2631150922, 2494146789, 1766908673, 2014027584, 3170299898, 2200469205,
            447095672, 3553741345, 1471247393, 1705231371, 3597764506, 2179927417, 2057038135,
            2877684583, 4157842671, 3988568107, 136943619, 1758053719, 133243005, 1316787143,
            3912460869, 3007503154, 673776671, 2300353234, 2544200665, 1154464703, 2377559298,
            2085288775, 570803259, 1718587627, 1887771341, 2141823671, 812587685, 473768462,
            3537328856, 2051785610, 1867482529, 386483431, 2102139021, 3579998391, 1082229057,
            4197498026, 2004997074,
        ],
        &[3960768807, 1512612890, 3846503597, 4289005374, 3202073567, 4260281911, 4219386150],
        &[1749675900, 83311145, 2319870768, 2301367900, 1271820700, 747192890, 2232034578],
    );
    test(
        &[
            1444823481, 1895962470, 1145820971, 951685031, 1619286897, 645659681, 3225126137,
            1237595037, 528204561, 3332610202, 2935828481, 3511627759, 3794280700, 504366468,
            2867551534, 432130908, 1682537913, 2112976328, 291513836, 2849031909, 3326433036,
            1925364297, 3779561553, 487877982, 435659240, 1540143453, 3711236017, 1440132082,
            825660149, 1624471959, 1941539144,
        ],
        &[
            110690520, 2114031203, 436730283, 2624140005, 3364550071, 806172129, 2495686202,
            3508099340, 1915440069, 3196245075, 3757278096, 786923263, 1820433641, 3283619351,
            2326488762, 119217121, 3545365539, 3063896557, 1866040341, 3325517573, 713948948,
            2863092530, 2356199798, 3131342786, 4122497749, 3637172320, 2586791774, 2794104974,
            1738360752, 1591954820, 2633103962, 398797289, 30804220, 2682142316, 1858898375,
            2406882215, 999588447, 645011646, 3469603876, 3657151690, 2477958807, 286833025,
            1802487901, 2709688493, 2318425877, 3516543869,
        ],
        &[
            3866576925, 1873138890, 1768109056, 2705255084, 4241776445, 658244374, 286815054,
            153522406, 2020517918, 87408861, 541283899, 2327698817, 696800449, 1357243166,
            2373189783,
        ],
        &[
            1748815118, 2402822363, 2091637344, 2074482742, 2316188658, 2370992052, 3607116276,
            2077769165, 1723590682, 2807481888, 338472032, 1807369086, 4009280963, 3941733223,
            1094544382,
        ],
    );
    test(
        &[0; 7],
        &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1],
        &[0, 0, 0, 1, 0, 0x80000000],
        &[0, 0, 0, 0xfffffff9, u32::MAX, 0x7fffffff],
    );
    test(
        &[
            2516900350, 3628420684, 45459239, 1513614219, 989392388, 453507217, 1512254264,
            2426221826, 1233076550, 244898124, 2764583004, 3329101881, 1699874673, 3208045667,
            2210327365, 948719692, 29636434, 309163724, 626642377, 3476255186, 2213282765,
            3561515942, 4189534597, 950518908, 139348404, 1259661856, 694161725, 2634491186,
            3727710582, 788362676, 3045474119, 4091304988, 2805201294, 3987583308, 1169494314,
            1793738933, 1926962859, 143323906, 3767356737, 2736821279, 3155234958, 369913573,
            2104538395, 1964464892, 120182245, 1607533355, 3512159772, 428709975, 3273911427,
            1151542509, 2860816013, 1136173381, 662332459, 3578077057, 3243603823, 818341966,
            1912641848, 3483224197, 605429894, 746864868, 3712846888, 2818736253, 462940792,
            3732576280, 1660947800, 1011233610, 843846147, 2308498613, 1878649446, 1442264468,
            2962205673, 2832862233, 1244894861, 1156622958, 1582973750, 1574948490, 3846938629,
            935089728, 1705244040, 4043433438, 3114254174, 1062986960, 3905677705, 1930766407,
            1354562200, 955778452, 2932639939, 249007379, 3763599833, 32702674, 3759419411,
            4281057196, 1688108074, 3861261668, 444663411, 2875533494, 971084229, 1671724356,
            681694042, 1336342690, 3989885039, 3351736702, 3656186109, 2518436399, 3272921467,
            235621572, 641659042, 2119882595, 1983842751, 1488017726, 3187465143, 2506358055,
            3348535321, 999852594, 4293611527, 1073384603, 250481194, 4148825621, 3839501948,
            3818994640, 1414056232, 2701465891, 499495991, 1361504858, 1308748876, 1650859371,
            3722040369, 223548436, 2431095012, 3829963049, 367284736, 451104830, 3192609374,
            959276880, 2266256342, 4176815348, 3580491737, 674113379, 4084740134, 883805748,
            892822580, 2297912891, 2348080766, 3777075710, 2743249275, 1929349295, 2464988855,
            1327032420, 1398791640, 2085211945, 889984414, 883598216, 1429348413, 154972877,
            3622910749, 1670306325, 2187005411, 651154286, 2688016708, 1041316505, 2776658464,
            161390395, 3846865057, 3267383835, 1010302540, 3607316170, 1418199870, 3954492322,
            922710287, 1323651551, 2431904946, 3054195005, 902509698, 1670133092, 1092971517,
            1075109776,
        ],
        &[
            1426636939, 2291732338, 3067548371, 1958595483, 4265832187, 1189668824, 2839754982,
            3301139255, 530760593, 3029188645, 1298922175, 1055363550, 2070850046, 992858009,
            1206997984, 1510007185, 2204463849, 2600547912, 3402640587, 372285673, 54684613,
            211924658, 2690037178, 1114469505, 374316665, 319939537, 1363579227, 25099128,
            2808696886, 2318162131, 1283473588, 4173665536, 1302236115, 2981103428, 364658024,
            3500679214, 73401477, 2399140443, 2401409782, 974668784, 3260945138, 700031094,
            619920306, 2267288646, 225106679, 2520705554, 760523499, 2971006600, 2155228491,
            967349835, 2899641188, 2567983244, 1086821788, 208644014, 2327565301, 3873773388,
            1254243651, 3371628757, 1486945462, 3153661571, 3596439034, 3448680976, 2884772533,
            2113693162, 3571562346, 3271064301, 2352298427, 2624271304, 2147301202, 1628031526,
            2268489536, 2460058239, 3824853526, 1114448457, 1711950597, 2805133745, 3886047152,
            1482425544, 3015333985, 1726209429, 1945238011, 2536476005, 1974758758, 3373425305,
            2797746190, 1649833243, 2549697741, 2772403629, 1276934258, 1197267921, 480328120,
            3145095762, 647795990, 605209910, 1896440782, 2021024361, 4280085434, 384535302,
            228210377, 1342562609, 550195641, 304643910, 803682708, 2640613597, 214734489,
            3610647775, 3787633026, 3169459037, 2718345417, 1486978425, 4213419405, 2452166506,
            2369416903, 2146131022, 3632701374, 857903632, 2077812289, 2633196639, 1727541399,
            1445819648, 1663740433, 2573746586, 150961985, 2843003920, 2121435332, 503235641,
            2162411412, 2641993335, 2306838455, 3013627797, 187911601, 2832638744, 3245525711,
            4114681913, 4248626985, 892023341, 4180070411, 462293938, 2339665566, 886499165,
            2680618581, 1275413281, 1653569893, 3506841220, 805196234, 2798162194, 2468871331,
            3396930973, 2515921922, 897743120, 2034157178, 2373161268, 3355723218, 3914014661,
            994426990, 2992401161, 2581503631, 697892172, 82419160, 2053001658, 3320533046,
            3791389127, 1839288008, 270908689, 1413534997, 429338594, 997502210, 3103236226,
            2401510054, 2663896427, 3727426794, 2724686279, 3944143754,
        ],
        &[
            3845376373, 2532376922, 273377135, 3677509657, 2349096840, 865593650, 3913447585,
            3605798835, 2383627001, 256434907, 1598943794, 582857664, 2095029213, 493203050,
            4072709797, 2373943493, 3435129133, 3493851471, 1582101091, 2328189763, 853133742,
            2400321806, 3544763029, 3255311766, 2487877428, 2345846021, 3540974504, 1867377852,
            2499885368, 3386759756, 3964579463, 3290899719, 2797957335, 2863338106, 3380325251,
            1361967176, 2820693326, 1270065483, 2316076985, 20080038, 2796142674, 1493193006,
            3582143536, 1315905237, 884491638, 2352205614, 3522766470, 1928358187, 1698951621,
            4079830723, 3210021170, 3465329392, 3919799352, 4222905913, 255999498, 707431500,
            4259794201, 604859264, 3884712299, 2790776367, 3466495862, 1352500875, 1421557254,
            3449668508, 339288181, 2600471637, 2364459698, 1593691125, 3908068568, 1590428078,
            3967743945, 361284513, 1947639764, 2146594709, 698658216, 2934691142, 3852641925,
            2677524346, 3136105113, 3755616420, 393422782, 4055467971, 836734774, 1342622278,
            1842869702, 3646198604, 3250586544, 1155965487, 3143510276, 264136749, 764849704,
            2833843864, 3628999351, 1603529516, 3414178524, 2241268229, 1653241000, 3792840587,
        ],
        &[
            2033854275, 3346500754, 1026391195, 2931755468, 4132970223, 507488150, 3787819697,
            4267307150, 2229023955, 3535124792, 2115982537, 3261552465, 1403786253, 4168838966,
            2905077003, 3343366362, 4058434477, 3051762274, 2754465216, 2139644668, 2121257326,
            147231635, 1411313351, 1942773925, 1518792733, 4026871852, 1608765600, 2674641821,
            3388658059, 4045496133, 4161170911, 1328400056, 1288717066, 741487928, 2130937044,
            3073981280, 486422437, 2174741216, 4004413829, 1284627600, 1179437164, 1177246499,
            2634583310, 2950709339, 3500520165, 502702338, 3556716680, 3168781392, 201139498,
            1773610178, 1336109110, 75169920, 4261358748, 3946913462, 1177416531, 1904421971,
            529501654, 760059661, 2718436151, 1695275660, 769729199, 1160292778, 1476806846,
            913674448, 1863234151, 2549859956, 2693241338, 2672780028, 1641057282, 3939169714,
            1414934690, 318354912, 300864018, 400872379, 1081977009, 3047385148, 3103660917,
            175274494, 1331622900, 1558233166, 3851938640, 205628404, 992837552, 615234521,
            3588694980, 848421132, 513268312, 468730797, 1030322804, 1991013271, 1629772980,
            3351179781, 2259742484, 957712992, 1217297297, 2285959262, 947300257, 3115413281,
        ],
    );
}

#[test]
#[should_panic]
fn limbs_mod_divide_and_conquer_fail_1() {
    let ds = &[3, 4, 5, 6, 0x80000000];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_mod_divide_and_conquer(&mut [10; 4], &mut [1, 2, 3, 4, 5, 6, 7, 8, 9], ds, d_inv);
}

#[test]
#[should_panic]
fn limbs_mod_divide_and_conquer_fail_2() {
    let ds = &[3, 4, 5, 6, 7, 0x80000000];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_mod_divide_and_conquer(&mut [10; 4], &mut [1, 2, 3, 4, 5, 6, 7, 8], ds, d_inv);
}

#[test]
#[should_panic]
fn limbs_mod_divide_and_conquer_fail_3() {
    let ds = &[3, 4, 5, 6, 7, 8];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_mod_divide_and_conquer(&mut [10; 4], &mut [1, 2, 3, 4, 5, 6, 7, 8, 9], ds, d_inv);
}

fn verify_limbs_mod_2(rs_in: &[Limb], ns: &[Limb], ds: &[Limb], rs_out: &[Limb]) {
    let n = Natural::from_limbs_asc(ns);
    let d = Natural::from_limbs_asc(ds);
    let d_len = ds.len();
    let expected_r = n % &d;
    let r = Natural::from_limbs_asc(&rs_out[..d_len]);
    assert_eq!(r, expected_r);
    assert_eq!(&rs_in[d_len..], &rs_out[d_len..]);
    assert!(r < d);
}

#[test]
fn test_limbs_mod_barrett() {
    let test = |qs_in: &[Limb], rs_in: &[Limb], ns: &[Limb], ds: &[Limb], rs_out: &[Limb]| {
        let mut qs = qs_in.to_vec();
        let mut rs = rs_in.to_vec();
        let mut scratch = vec![0; limbs_div_mod_barrett_scratch_len(ns.len(), ds.len())];
        limbs_mod_barrett(&mut qs, &mut rs, ns, ds, &mut scratch);
        assert_eq!(rs, rs_out);
        verify_limbs_mod_2(rs_in, ns, ds, &rs);
    };
    #[cfg(feature = "32_bit_limbs")]
    {
        test(
            &[10, 10],
            &[10, 10, 10],
            &[1, 2, 3],
            &[1, 0x80000000],
            &[0xfffffffb, 1, 10],
        );
        test(
            &[
                3347432287, 667250836, 2974407817, 429996943, 2750501459, 981155607, 2784121354,
                3415523349, 1074478768, 1820977205, 231390348, 1372589414, 2069009514, 2430571486,
                1857553253, 1536218533, 3666738280, 3477095993, 1509486443, 3092180985, 1588218081,
                2448257324, 672729993, 2374023188, 2913701567, 2541072953, 762935357, 1333399576,
                2676378567, 2962748936, 744662462, 1379794240, 2290146289, 4080012384, 2739115667,
                701035190, 4208500661, 2190862255, 3061320303, 3853993751, 2324138270, 2878407932,
                3552024846, 3816909905, 3815506441, 1255343574, 1901570312, 91219029, 1399539132,
                3118241152, 3325617992, 2881339798, 2520199899, 3667594206, 1853889834, 866669545,
                4104229459, 2070282300, 1484152931, 3098893530, 358952500, 2164190448, 4138312509,
                727032566, 2750086730, 3119399990, 1390731909,
            ],
            &[
                1037101017, 1727794380, 1490452070, 1098486148, 4011045557, 2946764072, 697788893,
                2534857680, 3261736561, 2991036431, 3203952375, 678147993, 1718071908, 1324397528,
                3227412704, 2808109049, 2498489092, 1483895999, 1974627636, 774482390, 1844543122,
                3947438938, 1357670917, 3021715202, 2044056785, 26339702, 2002972036, 2442219903,
                3581261805, 291932047, 3070844846, 1561991523, 2591235485, 4062653222, 432295028,
                3024980126, 3174836607, 1645982854, 968272899, 3101889669, 3996099656, 1637081358,
                2998072999, 2235890151, 553838585, 1688691976, 4092350095, 2065416776, 3709647283,
                1072755910, 2385936305, 2615766642, 136864038, 757727106, 2199116578, 2066024664,
                3784388877, 896467499, 298040025, 315565009, 4174702765, 3225704704, 23210826,
                2346706257, 1990744556, 2040289114, 1584842987, 1498510914, 2668413122, 2503096531,
                2005961409, 2272795985, 394426800, 2285275525, 2471642108, 2784697818, 1258928804,
                151106578, 3449120105, 169880528, 3307735530, 1735274873, 1875100806, 2033402873,
                1707359177, 2748179285, 3611534985, 3379681783, 1722155183, 1210860589, 1583939074,
                1413313200, 1958033089, 1382323587, 3321046184, 2567768608, 2706224116, 1925292443,
                486456966, 33556563, 4003036360, 624876701, 2847313485, 3451353450, 3432337713,
                1567344163, 2879538252, 624051431, 1206504435, 3713613381, 1062990555, 498008949,
                1798769609, 3937669287, 4147083150, 2479873449, 1454286408, 665997230, 331654711,
                1482116196, 2545259161, 1591656626, 904758973, 1600224317, 3734588228, 2386781283,
                4212676921, 1479533912, 2157734799, 3588439224,
            ],
            &[
                3380102021, 2452922302, 1962966574, 3357220828, 3890634007, 1037328979, 1799391942,
                1004666291, 3982464365, 4257152024, 187444313, 3439720141, 1266693503, 724678963,
                3437259008, 1572104078, 4039559445, 3285865736, 3569601379, 1308348984, 1718729371,
                284719758, 444404442, 4151851178, 2692003849, 1055982587, 4035467383, 3921915216,
                3487944745, 1817447325, 1542763265, 3937726417, 2223825009, 1394339428, 1214099970,
                4192217805, 3753293552, 3811191236, 706896668, 1075933013, 3658129436, 156288657,
                1259909922, 1889374713, 2941065257, 81950085, 649933990, 3336906952, 2183193381,
                1175410937, 2580459452, 3904196981, 2723001189, 1789637716, 2009908016, 3170700121,
                18873175, 2237687603, 1424198121, 980763182, 1043721413, 1070004656, 3240828354,
                1874904174, 1859325685, 914946875, 3171367435, 891888508, 94557928, 2528939804,
                1026794028, 412407977, 1504097351, 2321903212, 4049474173, 2376576188, 2102495452,
                1976457360, 2692728936, 3130170922, 2123163736, 3537525633, 1557920518, 2628772698,
                453031939, 451448604, 2142259, 3310964267, 2745320950, 2734144127, 3699687158,
                784837860, 1503636827, 811700689, 582299446, 2429787592, 1210273821, 2702547973,
                2699574096, 3370153605, 1236597584, 1088015704, 2939670102, 165924895, 18853284,
                3700200624, 3220442711, 2965715000, 1488424876, 3464105223, 217616508, 315230760,
                1429465794, 1372241659, 4248780864, 767395681, 4105247821, 1532652179, 2100932260,
                2674903482, 484464190, 3609058420, 1471559178, 4122200129, 4198439348, 1623393770,
                4254969136, 2270601790, 4239377868, 588832613, 2695326914, 1952240370, 2369526722,
                669955508, 1561380904, 1661607015, 491163324, 4063133286, 339180970, 1218080293,
                2858927543, 2211268294, 4242441810, 4166777931, 459461625, 2722830085, 1984526624,
                2371328931, 3795073119, 313183280, 3291949415, 1843511154, 809173981, 3006056205,
                3286577997, 74654418, 3553288229, 2169290280, 151023939, 1586116459, 3040308252,
                4142315068, 2646953656, 4225002928, 4051564438, 1032308582, 4166856711,
            ],
            &[
                2427812263, 4051563058, 1632406385, 3107965286, 2723923069, 1032718410, 1300669246,
                1686783039, 2900234860, 381961320, 3860838802, 3653570244, 1848772343, 2404802446,
                17732494, 2990806772, 1910668969, 3421521828, 1829199569, 2997918403, 3557175274,
                3519619382, 740250212, 998547797, 140767728, 1892932539, 668118324, 497299250,
                937663333, 2612689662, 4010699478, 389960621, 3786256383, 1395295829, 1276717484,
                1869006600, 2093853956, 1555847120, 101097965, 231988325, 1792259822, 29327628,
                408838665, 3646864419, 2451151869, 3859710466, 2245716151, 2883931737, 1198282734,
                2659505911, 354206954, 3482662941, 3212522494, 4106878300, 1901956875, 645864641,
                1078147214, 1797412778, 3856838800, 2428567766, 662568175, 698243503, 4146493635,
                150220967, 3723014797, 1944885174, 2282033009, 3395489773, 473095209, 3085161317,
                2999589203, 3878818308, 1223719023, 352990164, 249223561, 4143029861, 2622703186,
                796827870, 1054862138, 677020780, 1289629518, 3449239587, 3541454564, 3311081075,
                583050818, 491332673, 1997521794, 627632367, 2864835203, 2880916227, 233376741,
                1067584403, 2454677969, 4048829167, 2955413920, 651548741, 1667224967, 659069317,
                4019573675, 3305249558, 3965166031, 2657374006, 1899323245, 2478966232, 2865170658,
                819362488, 2414458118, 203150817, 3555808793, 3366944226,
            ],
            &[
                2882761542, 785407498, 1124133487, 572386909, 1169675686, 3567625455, 4199715839,
                3851396630, 3274146684, 3477301663, 3302596192, 1480559142, 100842294, 3530609196,
                4086393700, 1190308308, 4263459677, 1142007717, 2247036858, 4120633014, 2294603452,
                2399429000, 2624162280, 1454610696, 270143073, 3957844361, 3046891288, 1549007386,
                233535161, 3109659080, 209449866, 498073310, 2626910641, 195499996, 3976787267,
                3922992490, 349292363, 3307104635, 365992265, 4190381097, 3755383481, 2478897620,
                1546723896, 4001998632, 1128630761, 2676599750, 2475921888, 3233603626, 1483504432,
                2551193799, 3557144097, 313815765, 1349744193, 1374844288, 2400211342, 2986607421,
                3195157472, 1299836347, 1861689827, 742292964, 2823852296, 448143737, 1667945403,
                633976860, 2902355589, 2157443145, 243118113, 2840619921, 2919500120, 2742747945,
                845926917, 220468159, 1355718767, 2330428847, 3717678284, 1787470596, 2523167223,
                2214932965, 2055827539, 27888400, 2211687684, 526284755, 994343204, 904008663,
                487598580, 2347713123, 3532282785, 3549720901, 2459046510, 338578128, 1168895967,
                3291681132, 3787087696, 1555047248, 894667165, 1085939503, 4100902874, 309142266,
                1088553857, 2096128611, 1104848719, 3907470805, 3414980158, 3260046959, 2704772147,
                288217831, 1781865616, 179692408, 1428188291, 2562193479, 1062990555, 498008949,
                1798769609, 3937669287, 4147083150, 2479873449, 1454286408, 665997230, 331654711,
                1482116196, 2545259161, 1591656626, 904758973, 1600224317, 3734588228, 2386781283,
                4212676921, 1479533912, 2157734799, 3588439224,
            ],
        );
        test(
            &mut [10, 10],
            &mut [10, 10],
            &[0, 0, 0, 1],
            &[1, 0x80000000],
            &[1, 0x7ffffffe],
        );
        test(
            &mut [10; 3],
            &mut [10, 10],
            &[0; 5],
            &[0, 0x80000000],
            &[0, 0],
        );
        test(
            &mut [10; 125],
            &mut [10; 405],
            &[
                2824467771, 3299124311, 2818671068, 3680778570, 981687343, 2406693669, 659467593,
                2993414243, 45406089, 1478779191, 1711786852, 1750419133, 2909274013, 511107722,
                3561218251, 1136025710, 993075881, 3516743656, 2114794292, 1997914, 3812643652,
                2505455342, 1465178283, 1978590884, 2983823507, 1628362808, 1695987706, 2508198478,
                518693670, 835288758, 1012607974, 749782274, 577737644, 2333209317, 4167594129,
                2270730042, 3116080031, 2989072664, 3700594452, 2080277190, 1740782490, 2022893854,
                2980102855, 751636825, 3922671264, 3391149880, 614930929, 120867901, 2567507861,
                1524427749, 2580678371, 2199894921, 1776614185, 1166041925, 1079838738, 110556090,
                2831653230, 2555307604, 1415160216, 250952335, 1727168666, 4114643415, 3895205129,
                692242739, 1888273350, 3407947612, 2500185766, 2392325992, 454908683, 1510488259,
                627605677, 2579656597, 503296566, 732883994, 1804534661, 1024632105, 2582676903,
                2186193446, 2035546222, 1537608294, 2871563506, 1842379464, 4285710796, 1145782664,
                2168389602, 2821015070, 3799000303, 1910989354, 2712163170, 2373188937, 94374379,
                745462864, 1273218817, 599427865, 3025428597, 4170147632, 654664457, 999597566,
                1301587861, 785951171, 1404519917, 226625774, 114784173, 1303174335, 1991080402,
                953508169, 2955695351, 4272100018, 4152229685, 505603829, 1367830508, 275280730,
                1743334800, 136640107, 543638242, 2867205034, 3205072197, 122133232, 2907110177,
                2577983454, 4044792269, 1426164254, 3391219052, 4077004042, 60445643, 361888880,
                1370511290, 4140611, 1545574856, 337863511, 2696753299, 882513095, 2220102704,
                4040224968, 3895008884, 3144563393, 2298197836, 2781817257, 1704369652, 3931316078,
                315544063, 3497851423, 3664143565, 952326443, 3938628137, 2331019960,
            ],
            &[
                3222658839, 1719980949, 49002116, 576454969, 2480526346, 4090562449, 3826526799,
                1957153338, 2225171960, 2687643162, 708981609, 3176715792, 2962973059, 2652817195,
                1197341180, 2939993186, 3554367730, 1570613844, 2536409086, 499280456, 3247067732,
                1595825640, 2368929206, 663346056, 3132360609, 1474802683, 2339072148, 2202454503,
                3728785029, 3203437738, 904057293, 3413710811, 2642403758, 3344563036, 4027597111,
                1888090288, 1863907782, 2464957062, 2688706102, 569142547, 397802070, 245946780,
                3489512252, 2617239243, 779718731, 179625066, 4023449096, 3996006819, 953561942,
                1127504598, 1992436862, 3073123794, 1835220635, 3142367806, 3159289192, 1927251431,
                2830198394, 910503635, 2897643083, 2932878381, 1507827586, 3067247947, 885528755,
                2017570727, 3134837176, 499511463, 1535166690, 1643153113, 2968177930, 3952804609,
                3989699184, 256167900, 567717900, 3896800262, 2310644620, 2351013373, 1964719080,
                1905779649, 3311747907, 1340733718, 2870734940, 3144732377, 1107086597, 3017452910,
                32224803, 2995638379, 1782145516, 3064028223, 405412838, 4063680301, 3415165225,
                1788170832, 3805938672, 1902801261, 2231309030, 1810036873, 3470233158, 2413346472,
                3530489706, 110880408, 887205258, 3604865827, 2857625992, 3018122209, 2631997677,
                1125977084, 2461114861, 2161802286, 273825728, 2086061888, 541221199, 1806844610,
                1376732414, 3802142465, 3535151037, 2954021790, 2116798495, 3735046347, 1186632354,
                2833281426, 3457883657, 3962766016, 13814577, 1650009216, 4147309188, 2302630822,
                3980940746, 4258997501,
            ],
            &[
                671156763, 3522970087, 1803674928, 705095332, 2537912304, 2452859779, 3450873050,
                2679364048, 3526396093, 4177976105, 266611217, 1151557614, 2132817166, 3877142811,
                507210298, 2972523503, 3073806840, 3566048488, 880991998, 3345680417, 3235170656,
                2403723979, 1313154315, 2771932402, 1330267204, 3414255915, 1179382708, 3233207082,
                2503609750, 249317411, 2632726993, 130324620, 2032071659, 3005726146, 3875282024,
                887247870, 1998016163, 4254819443, 3740865454, 3553274754, 2446121384, 3275444724,
                233325872, 539446483, 1087952312, 1835245616, 3627109091, 2510529404, 185630261,
                3706778570, 3098183261, 1931085512, 798171652, 3040444481, 1161846676, 1038636293,
                3421904104, 2901743603, 300733872, 3719079820, 3468813384, 2881344572, 2125343174,
                1936937822, 598427076, 1712995528, 4084208860, 3733468538, 1669487237, 1385196430,
                4053752992, 3005943092, 2525935674, 282240792, 4279794411, 4125983631, 2329629976,
                2704369810, 1464421682, 1313908108, 1226002425, 1808406752, 3275768064, 3175836384,
                931684775, 589156533, 3002460872, 2667576199, 1619914000, 3662683434, 3389418364,
                772131109, 3768421507, 3396906380, 2736076174, 2900071166, 538472464, 373492649,
                1143475222, 591038056, 3736209846, 856338986, 232250025, 3427987228, 159577157,
                485970599, 624042727, 2427226926, 758605639, 309500365, 1623348751, 4085018849,
                1073894303, 3272403714, 3990448709, 1089031712, 1648217954, 1406121363, 1462876615,
                1399073841, 3264088864, 3357032142, 1754184006, 598504776, 2188603491, 1888083714,
                2827226198, 3414026126, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
            ],
        );
        test(
            &[10; 115],
            &[10; 1254],
            &[
                2401497277, 3741262706, 691871239, 1949640889, 266070694, 3295242537, 3037066187,
                3875417448, 1358853731, 4158125741, 737486219, 2625351318, 2205206855, 501024225,
                2060376077, 1953521053, 3762056723, 163485007, 769999701, 2385831686, 2388009068,
                3120433785, 609866149, 1014929431, 4255522153, 3908104955, 1296186549, 1400337892,
                2481196402, 1530226708, 1736756203, 4267671814, 3181376526, 4204274026, 2772982826,
                171341955, 3239339748, 83269891, 2325049318, 746305994, 1290717192, 1805639230,
                1856363646, 3445599715, 1388597950, 2699498310, 246990054, 1396771726, 265087723,
                3372840761, 3260397695, 2010115470, 2211074582, 1883962263, 256363687, 4012920099,
                2367091978, 1485341992, 1836339208, 1434180757, 2616877872, 1145425491, 1261399773,
                3926766027, 3115015865, 1741783714, 3802513218, 4081750672, 3856503286, 2137512595,
                1192985123, 1863041957, 700637154, 2503934926, 2021744810, 157763047, 453501781,
                1898727251, 3741166728, 105996483, 3390681922, 1127471458, 4229032301, 254843851,
                2729237155, 1707011850, 1756734415, 1864969309, 96539023, 220456923, 3061776324,
                1741151628, 1199010536, 2099212259, 3095715999, 1369052975, 2575251120, 1821995904,
                4169452262, 3299830336, 666854626, 3647011057, 3965571435, 1551218489, 726151433,
                1730753749, 750433250, 2084548637, 475638490, 3209308986, 1536371286, 3458731894,
                451976715, 1463077982, 3275775647, 1176722184, 946473888, 295088963, 1154092407,
                103838303, 3743650178, 4149308777, 1617840168, 3465913339, 4133656500, 186966677,
                1624567957, 3264214229, 1285455678, 3951367256, 1685500601, 2890483019, 3692192743,
                503129144, 3972113270, 2472468030, 3386552094, 520995657, 318761291, 3652507947,
                876136924, 451086694, 2855321222, 557040372, 805677163, 2526778273, 2320694361,
                1256424719, 1952008095, 4056766614, 4133706458, 3948099450, 343436346, 4192911370,
                284826582, 674589589, 2186532130, 3868256489, 3551620239, 3301954255, 1174676675,
                603478538, 275555563, 2405977791, 1847784099, 896249737, 2255526478, 1787445221,
                239988612, 1297053793, 4022115314, 3399153034, 1717528213, 1660404772, 3074772866,
                2806710693, 1643510791, 3184377187, 1540838982, 134943412, 4022881239, 1840380980,
                3861497294, 2510066751, 1433230269, 2045075002, 2415627101, 1223566402, 3367143819,
                612631145, 1154322627, 3247079704, 1778868389, 272795417, 933297265, 3458460741,
                1389431679, 2907745644, 2086192882, 2684128325, 2650239535, 1784197442, 3949401651,
                1783533373, 4288056634, 888203755, 437001866, 1407050668, 3135687315, 3463879635,
                1702062054, 1204976730, 4177166610, 739673691, 2277342971, 3478834440, 4081300483,
                914246861, 2121339885, 1961961557, 92095006, 2260302382, 2461228008, 3993959318,
                2950464602, 1950783601, 1224172324, 2576903297, 576707684, 4125788986, 2560012161,
                1956729376, 1629575657, 4036954933, 300086502, 4022406600, 3237472700, 1646970397,
                2062807673, 1759249491, 1956866526, 2025198625, 1297980907, 2709447351, 3977405586,
                3105592010, 4167079730, 2234262082, 3282437095, 3492171389,
            ],
            &[
                414354569, 1366704839, 3104154714, 3052961192, 1917209478, 527538773, 96279538,
                131392026, 2197344782, 2544900655, 3721689046, 1511768113, 2703760155, 1445726112,
                1858496576, 1575565932, 3793123807, 1385830602, 411844037, 2236175811, 573323765,
                2565409615, 2461667516, 3785284514, 4260465727, 3052818977, 1895970970, 3792821387,
                4109605593, 1615504561, 3239716852, 1706559703, 3187779838, 3278892149, 3231274931,
                4198044737, 1513165468, 245082668, 1270047387, 3396787938, 1940745611, 3740768753,
                2072323188, 285189880, 644016853, 3495689253, 1168360917, 2400702434, 984958722,
                222994595, 3006232190, 2064852513, 2152929014, 1163603446, 3191172136, 1835493220,
                3277922296, 3636603619, 1531000264, 3057499684, 3823394467, 2647235938, 1717867123,
                499456838, 4094707383, 2454454468, 3671059285, 3201191048, 2439390676, 3022007460,
                4061878398, 4243531226, 3389689292, 558186288, 3777835601, 3123712634, 3503180938,
                3691679764, 1701324443, 835309072, 3906212426, 4049064285, 1768153688, 2228746280,
                3594304220, 2961136397, 213262897, 80889120, 4109164941, 1704207190, 149603550,
                4140450876, 835389387, 2885880052, 439529491, 1516869648, 2050960198, 926155485,
                475035022, 1187115633, 894340078, 343754206, 3955497730, 1344991704, 598758584,
                4056958135, 2153619389, 2541228501, 3084209032, 689848357, 1722405264, 3359713626,
                1572612273, 740065643, 1582253391, 4070651569, 1908850964, 495396647, 4057999777,
                3077955309, 4157175191, 2258556901, 2113839, 3880102604, 2790333280, 2943303426,
                3912791781, 4284145483, 1840413180, 3097912758, 1064553745, 1126983800, 2256085399,
                2448922218, 981100464, 266358012, 1971087790, 1988527723, 237308225, 1936681335,
                4258331432, 1348277821, 529864588, 2497818319, 4244738664, 564460927, 1832999941,
                65438986, 1185234412, 1316363591, 1608606945, 1000964619, 1732529786, 527797972,
                150140396, 2156815817, 4220448391, 1051855867, 4280728484, 1058663428, 4250194724,
                1036504424, 3657621656, 189944192, 57605355, 3072751931, 752978294, 1461890065,
                2724185615, 141061473, 1466705961, 304569847, 2188875717, 617186800, 3058558545,
                1591798676, 88078245, 623288082, 132838333, 1534710499, 2200810995, 2857112538,
                3863089059, 3060974440, 16773497, 1918012630, 3327340967, 3661341852, 2395761457,
                1535964139, 746843178, 4060399981, 4287382590, 1336021602, 1599090737, 1174816250,
                481099617, 918021470, 1173582832, 1368734491, 888221269, 510154884, 1262941262,
                2567813829, 1077623012, 2261930141, 2307570205, 2069365574, 3586190102, 1698424580,
                615530592, 4037104804, 1283475105, 4214495193, 3288288012, 3585802470, 261207221,
                3506861813, 1536248070, 3151134123, 645179327, 490399820, 2479385864, 3709733877,
                396138285, 2627168580, 4028642442, 2012757271, 1884810033, 4051927252, 2652348818,
                2069059332, 2416554871, 28369250, 1321756354, 3314148643, 2588326180, 3141604522,
                1974534297, 4170320959, 2683069809, 1278314467, 312763173, 1481522722, 2846196729,
            ],
            &[
                3521949631, 142488398, 2558755806, 546733190, 2792635139, 1330279357, 3430917364,
                206259042, 1647608238, 3662226174, 2677354502, 501681320, 3339919968, 736768019,
                4258570929, 4202792087, 1651152858, 415339868, 1437647081, 1371825040, 1535867295,
                1113678172, 4268196599, 2619516870, 4117420686, 2491883937, 1711537149, 3681498741,
                487272454, 3249094136, 2833249206, 1984417080, 608254353, 2280645468, 3395986408,
                3532393236, 285299266, 249977069, 1222542802, 1102282847, 2598354195, 439643199,
                2126654464, 607909799, 2936741523, 3763227596, 4235714917, 1698786602, 1862433691,
                2965568505, 2196568398, 3068583313, 2467355500, 2575639252, 3699104542, 1440347634,
                2202871610, 1085633678, 2985993155, 1566387171, 2653399887, 3340197769, 2614511546,
                565555655, 4058797643, 1841325854, 671657716, 560998985, 334170781, 85067287,
                1935383603, 3452109092, 3454188157, 154367468, 3636444053, 3436201211, 2489786062,
                1247734460, 4255062573, 1485745449, 2725636745, 3754874076, 4135899179, 2423709040,
                392644331, 2072210265, 339068725, 3933463273, 4288485419, 4212008073, 3733609991,
                4120877958, 1466616375, 2958829060, 83050688, 4089523843, 2719660596, 3345635470,
                4218331277, 176804352, 2942885054, 1103429906, 1232548981, 3336211973, 2320933704,
                2354076844, 2529582693, 3959342251, 1769056129, 3021361560, 1001511651, 766365010,
                303546733, 2838329868, 696771802, 2198903959, 3831700812, 4192525926, 2867372730,
                3200189248, 2637281548, 3606733360, 2672897488, 2624221199, 1766256987, 3059086417,
                3482969658, 2697305862, 415920845, 1575345078, 931452186, 2064246173, 3556532318,
                1127897667, 4176776745, 1103206019, 759429637, 3840446634, 2669415208, 4098858695,
                3968696636, 474286523, 3603742455, 3683233930, 3910949095, 1016216353, 1693878242,
                2936976170, 2258461087, 1543964790, 1257134506, 2705742284, 619723369, 2313052342,
                1916338183, 3578695856, 2388468521, 1417556272, 2630850792, 2327790132, 3594026023,
                1659979016, 477242076, 1948563308, 2803059042, 2392360966, 2007275313, 272222339,
                264018609, 1147760126, 4131679216, 833601075, 1429546506, 3426172207, 1125974375,
                3584187622, 1582857679, 1084294484, 2193982331, 3084073169, 892928835, 2823566821,
                3544239939, 172135078, 2430777811, 1620416268, 1792703584, 2899867927, 3974701277,
                1890638601, 3367278100, 4278027820, 3171315428, 3696212814, 1038269559, 2729624661,
                173005716, 3652244541, 724827867, 3530325019, 2203073321, 2587791340, 1909841295,
                1412031121, 1429200221, 3614105795, 3265396657, 1362525000, 1906071103, 1060149404,
                2733619757, 2675449568, 3414488370, 210765135, 2800530139, 3014478492, 3230858260,
                3409460781, 3411785589, 2543986390, 3934724152, 3235965806, 2053739672, 4085596199,
                1563589410, 2817526968, 4048637993, 3055675422, 2173262993, 2970495556, 514198452,
                1591956633, 3852865086, 124440700, 1002712372, 596340824, 3999538417, 117476573,
                2037586050, 3291573274, 1511567035, 1614566437, 3821078763, 961133253, 2796121934,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10,
            ],
        );
    }
    #[cfg(not(feature = "32_bit_limbs"))]
    {
        test(
            &[
                9467139676469209549,
                590281264913108801,
                2444597864339828042,
                14864081623018167053,
                17247631583397892818,
                14553244820490552393,
                12829855212365838534,
                3531466761266135925,
                11372171159031384620,
                5870847048926628622,
                9193240591510160396,
                4025304033900694645,
                17261252553070218810,
                1960954339685170046,
                9689788042403522315,
                6777781555383721706,
                564338697137211139,
                2236230241770389317,
                7260612298284603217,
                7551831827404764167,
                14240930170717885933,
                5403113159439788814,
                11890192816726889700,
                4571074562843162754,
                6585174915018781919,
                16820621164336832115,
                2906428786044704407,
                14549789283221663094,
                8574533295629171239,
                2554371108614275803,
                12102140726062325305,
                5927759880327312421,
                1947106727211751169,
                4243634376196439953,
                3958901784387088795,
                11997496505606868854,
                8614259227764305665,
                2851316343853334425,
                7360014330841540867,
                12833167596911644967,
                16420023392297553774,
                448987443751486030,
                1127916262354476550,
                998365641524414747,
                771182118275428982,
                3644906482543632509,
                12008471662501141232,
                6735530311778854203,
                14937996347189247818,
                10310266533399488799,
                14173112110010990829,
                10048196983951682150,
                7660343487007291390,
                5442254482173747309,
                11894682918116055490,
                555474340707057713,
                5454210195350763050,
                6335921756850263208,
                11125242160747149811,
                5950900297278200050,
                4972887779003626968,
                15099558014605218082,
                8070946285642924813,
                5661924633602804698,
                4017981546647826153,
                3298784914124356106,
                1048634509662954303,
            ],
            &[
                1601810336850541391,
                6414377018375174391,
                919960012376992260,
                7432535796251013733,
                14687204344100976419,
                4239980047441451729,
                3419961342059496953,
                16241811836659790563,
                582511433240649668,
                246415310737356671,
                4213289580820322712,
                6570486144808547472,
                649071692625916450,
                16246395526862594705,
                5418002168615819323,
                8787804720479728592,
                16606257073152622475,
                1410262373214068670,
                2846556520446018460,
                12702224195603835801,
                14824788470038136235,
                8070823597295684045,
                10064047777457825217,
                12473928276513608352,
                4502242984490214594,
                8213242804638977717,
                11407293275124988149,
                15303970978422100227,
                5734015081210218763,
                6780255848655226189,
                5308681743121306497,
                3590094480707706491,
                924657275311967241,
                13313245703725306506,
                13755314570936606477,
                1722612846595274627,
                11310713460998976321,
                17860713973992950196,
                4832972642835413575,
                6159638888337886550,
                16436259186156623772,
                8219825892743153908,
                12825816757113080127,
                17576561025816321316,
                16426316569762062979,
                7041069634919241163,
                14958139263401063299,
                8541928253694453091,
                14623281385826596559,
                5881920493887697974,
                2856125878289959548,
                13138961086309035424,
                1445182795958550281,
                989371283719803801,
                14414401959190662319,
                12694741659414145902,
                4944169755440522083,
                4850330269090259826,
                7263731888738583042,
                3847985956261979907,
                10886313908655224591,
                16884924304355496352,
                17434369248195376612,
                9624045826924140661,
                9440358242723121635,
                44413621672380228,
                14501853932445164832,
                4412509400967439905,
                4059150452772511000,
                14121559478607349396,
                6939247432141568945,
                9144772592231287902,
                9035665952014395555,
                14897716038428818227,
                15076823799682351749,
                16277606343067785741,
                1901372957955104636,
                9089098420745572372,
                7424125507905187829,
                12368330388186365917,
                12778631331588513256,
                7648700244963873530,
                3776028467270636181,
                16107369772612157292,
                9046543866366629415,
                13251838453492673272,
                2975138303374961664,
                2462966081502544324,
                814917744063498594,
                13005222969141014201,
                5026855542811222762,
                2382391535955253463,
                5899528506680208740,
                13336141787604226636,
                2972106516856532181,
                14045206722662122716,
                13361481863470745757,
                1303122885509130861,
                2127042842590220699,
                6185514996199626930,
                16622745329653187578,
                1256533115604594696,
                11243434057073050028,
                9066405370642929758,
                4552716624777987017,
                1128092841571320177,
                3756894170446405064,
                3328979948330012406,
                15049042231988504879,
                3530903968533512703,
                17905884378428276079,
                3713606437359152813,
                9370125751473669209,
                10012381228375856292,
                6916709082927033810,
                17606888822444197359,
                10818646898854586102,
                9927520028322801489,
                17015575684851383261,
                15287563104761130407,
                4246630569287612899,
                6737522138244324583,
                16956805675707509108,
                10507692348129622148,
                1144006384258648355,
                4260138925314579646,
                14508270375905482517,
                3559046762122289814,
                15199903498605585717,
                16432575538338639756,
            ],
            &[
                5592822181652430458,
                17506309594623475426,
                14050063092843436505,
                7714948654849123602,
                12960251261153655218,
                12293354553127881932,
                18236552269356335752,
                8116592444680865353,
                5788148289215893900,
                8971586422059718981,
                7830225577778842728,
                16542468928391674395,
                7685364916096821326,
                10621040870461365797,
                13017786142160529410,
                15623634767986429431,
                3570280230795725028,
                5462496964935066825,
                13096782281304038571,
                1028538565298248606,
                1306240401949593430,
                9752885768751332733,
                16834186448734276157,
                7637206024963845142,
                14347989823115406647,
                11694534965309339857,
                17506164867360865472,
                18226657528377127028,
                17113377911080849277,
                12836900580850373696,
                7624296086468816307,
                3241150049341673184,
                11432625191496196476,
                7201112746916642381,
                14159401858377741999,
                13042389876849434472,
                6539044773109830458,
                5626612565019858620,
                16855355146181557220,
                2896027096805100269,
                1749121842377065409,
                14486576285464434664,
                13003134176160772443,
                4520736241988979615,
                14386190542886985088,
                5203873393963986621,
                2767011034432552285,
                6993214728799607735,
                16219726698174042484,
                12883752315794694947,
                4638588568194864850,
                13671093412063473486,
                16507766724668892429,
                508640878850638089,
                10821378407060575406,
                7345992160360704865,
                10043908069076742056,
                18320857698273010360,
                2218460838923946432,
                12515547441597943704,
                8373193888106171069,
                11138535537637434655,
                2449630875466873477,
                8767699022869197665,
                6446940169111329267,
                12481583138381941451,
                18405856492243480913,
                8856983257605794349,
                11977198292403090397,
                4992072464455161558,
                1429124422280954077,
                16695629052720642207,
                15320583853113777709,
                9270873446273846783,
                9544568698292818385,
                16150036388405871492,
                4707553634577955164,
                12819651092171657742,
                6194167822526464289,
                3302239600303656809,
                13520149877311082646,
                3658928982143500918,
                6938149248133961941,
                3561482775788486836,
                11608588618646326460,
                9312042305619109305,
                7752033026671842884,
                17638989744464665362,
                11720586895422616384,
                17488161719714908794,
                14359533380966089634,
                9112047402314539693,
                9045007769242360827,
                14039981061261051366,
                12161985564133395359,
                12552005864162233454,
                13987473146854769743,
                18218863257920884509,
                527093394921779615,
                10809320813728048297,
                9176568449151735783,
                16181777654712188911,
                3708863559833042684,
                15705373009786311560,
                12044345266545331965,
                15745097802473032619,
                3448641903062070968,
                7079863925626111888,
                10936059842602692227,
                3949990826310278419,
                8781569794504977516,
                2710649351206510739,
                8045244454460377457,
                16793164706992819994,
                14171071447733414359,
                6442975288102390424,
                13408292761208182437,
                18190891873633561623,
                14416452326133420106,
                11467995109126255854,
                10355921853391241663,
                3362337025912778005,
                12828210017750944427,
                6724529289281531448,
                17910365619735459537,
                18061583666579344364,
                7385510916096038449,
                14735084634699542626,
                2983574817812405387,
                1160726944282883582,
                2119364765206141036,
                6464791768964988391,
                8824016806539484116,
                8270030586906660422,
                15002907845345901548,
                5776511587881140498,
                8026215921305305807,
                4715947225014261487,
                1386404302622265399,
                155879991043344208,
                7029054386547293524,
                11220489073691424833,
                2733588115435171341,
                5063187558440228172,
                14208382232307280669,
                3376720278729662133,
                14690710213721258877,
                11094816467682244196,
                13929756812886477086,
                14698847015580325706,
                15992372099283576304,
                5818288394714584003,
                5038324365106419021,
                506432342213631013,
                7781480413773762085,
                2275557418573742756,
                8703301345083257514,
                11424227525030059244,
                8529590924998378026,
                5577211436885609271,
                16859602499387351883,
                5848360527348246100,
                2118117857825986323,
                3178709503818913442,
                14068618324826971348,
                7820101807986528976,
                16849757836429474618,
            ],
            &[
                10900792384749518304,
                1752565570529908396,
                3402229115647561238,
                2472381872242532960,
                15748089475115162936,
                1958855681762413475,
                12100416912810779188,
                12256578057348862042,
                6556831077371185734,
                15914846823057329492,
                17346954154793811255,
                17566187606614467459,
                1433606366066775495,
                9089332045922722756,
                10056944581186126460,
                5324425019386643029,
                5281765195814058625,
                1449711238109407238,
                5903959110668039125,
                3336955200427408551,
                751494194154096512,
                15350321905800137137,
                12407374450431165353,
                8705815621686854350,
                18038286270431178148,
                11671842546699641930,
                9343865367071815679,
                13401838367914321671,
                18365991333043790435,
                17428290065100096976,
                6040216493892400727,
                4224515713015397505,
                16578741590625036060,
                11835373548777581169,
                18413478850867685366,
                8453265724386285209,
                5394500720474148965,
                1927463313122594080,
                4177838821929605731,
                10680620304882583021,
                180005403771618203,
                2256408572502279608,
                11718426532525535626,
                14260315287467647015,
                4035759666841010016,
                16259497729863184485,
                7772704202422133476,
                6815813069474359325,
                11207378575459431371,
                18308033894506293455,
                9875145231436590806,
                15354934628544213986,
                761822562304640527,
                7391550101325083295,
                4023926600201752832,
                922969942182092752,
                12110946035817932140,
                16574399923422896843,
                7087993004495856759,
                8299428112066197874,
                4589434828506762129,
                13978205413565566735,
                15675366647238478172,
                7819770375827015142,
                6823625407675143456,
                2042269662440457350,
                11521115322912473140,
                13703874674141705702,
                1295561690992462505,
                12464082489717915012,
                11378922861990148970,
                2076282285705918066,
                1390689690731346588,
                13670979351308531000,
                12980996477862222169,
                10496970808504864546,
                14015388605987660396,
                4171129107047347396,
                1656857204469415571,
                17492457435753920912,
                10132937897450237781,
                5065601177732655021,
                17498367701449356268,
                9552937910825811119,
                6213399497134928078,
                12865519292113075754,
                8548871019474664332,
                12973212090641168109,
                3018095992673320728,
                4102580256148037725,
                11411505815957245048,
                8044142604358855954,
                6163064685377006161,
                7676133172626151339,
                15177331097378985405,
                923128391789363540,
                8405355494789853124,
                8409013636286216842,
                17519952046647436442,
                12690425880634822079,
                7295927951214020420,
                5103171252065286692,
                4531269626482776566,
                17509133779966482098,
                16771567673323510549,
                9940518318209913958,
                2566490491730418524,
                4997841530198583881,
                11759671980624847072,
                12804335234851198898,
            ],
            &[
                11770973102809246906,
                5654064914644075203,
                14154195031168297145,
                3196816914765589663,
                13269534898382935070,
                6602763211194544985,
                16018516851544910792,
                3180625223944165224,
                5350941027330432714,
                6398743157953739295,
                1906167046060045935,
                3556291732895289080,
                17581917949159063577,
                11917842024560925369,
                5513291200181693371,
                2805207190711928628,
                10488935147153720707,
                9353681503295456349,
                11923702266366452872,
                16445534228693736707,
                1321572029020662220,
                2088179387423767363,
                2940575338916006997,
                5660751381096282465,
                9379032325552965866,
                6155754375040762615,
                17757845483758379498,
                4791726927249442785,
                2271154604380575561,
                2810516188026226067,
                17085166475121940699,
                99572451658953140,
                5696266492111718158,
                10862208603440047636,
                17394577092287756526,
                8594037627948854376,
                15384378431083587966,
                8688372331869182594,
                6898185140759317855,
                9871173932216986045,
                12528452101456458662,
                13557460938391962098,
                4450262203153213242,
                9384263431241833822,
                9013347668307504163,
                11311595079831956259,
                936915772753461789,
                8433063029106192581,
                16896552532247465449,
                436920621494623275,
                17030913520242164150,
                9892849888317096483,
                10846215292694646349,
                8237341484379572099,
                16094501558496758251,
                1945916543779639318,
                18090461816104963907,
                4146339453090156488,
                16829166437621172993,
                3078631608488886297,
                4439257821416126689,
                7705380192038328855,
                3538871580337510846,
                13746211561839200096,
                11905905983562326697,
                13358208869614303744,
                5843566918809605864,
                9427926296554685300,
                14598289187890400060,
                11008519655031485653,
                8593121017807548632,
                9396415039249055836,
                6268435325877250293,
                17232661262414298575,
                72834733148045272,
                10210620080322964005,
                4785045324389620807,
                12743978039683515133,
                1532018623508178307,
                1626858325392940984,
                14509988664710915642,
                11484970178386722683,
                9637331423956040786,
                16727570268905595418,
                17007646468471636895,
                11784309337440826435,
                8264087540904628249,
                5386650781848750246,
                2634849477861627672,
                13571530405202238935,
                957020390364360749,
                10551951282480361215,
                5860903308359481633,
                7956790956638521305,
                8737258119314118945,
                12342128112549159916,
                16715653993533680030,
                16954242660912593528,
                2489318600286989195,
                7710331526401780284,
                14799339212986303952,
                15607538409164651487,
                16839239842720015730,
                12565794445330275268,
                11582072386774199841,
                7158773820903321984,
                1329934199003039189,
                6297267245832566486,
                6153026408185751289,
                3259942857823462030,
                17905884378428276079,
                3713606437359152813,
                9370125751473669209,
                10012381228375856292,
                6916709082927033810,
                17606888822444197359,
                10818646898854586102,
                9927520028322801489,
                17015575684851383261,
                15287563104761130407,
                4246630569287612899,
                6737522138244324583,
                16956805675707509108,
                10507692348129622148,
                1144006384258648355,
                4260138925314579646,
                14508270375905482517,
                3559046762122289814,
                15199903498605585717,
                16432575538338639756,
            ],
        );
        test(&[0], &[0, 0], &[0, 0, 1], &[0, 0x8000000000000000], &[0, 0]);
    }
}

#[test]
#[should_panic]
fn limbs_mod_barrett_fail_1() {
    let ns = &[1, 2, 3];
    let ds = &[0x80000000];
    let mut scratch = vec![0; limbs_div_mod_barrett_scratch_len(ns.len(), ds.len())];
    limbs_mod_barrett(&mut [10, 10], &mut [10, 10, 10], ns, ds, &mut scratch);
}

#[test]
#[should_panic]
fn limbs_mod_barrett_fail_2() {
    let ns = &[1, 2];
    let ds = &[1, 0x80000000];
    let mut scratch = vec![0; limbs_div_mod_barrett_scratch_len(ns.len(), ds.len())];
    limbs_mod_barrett(&mut [10, 10], &mut [10, 10, 10], ns, ds, &mut scratch);
}

#[test]
#[should_panic]
fn limbs_mod_barrett_fail_3() {
    let ns = &[1, 2, 3];
    let ds = &[1, 2];
    let mut scratch = vec![0; limbs_div_mod_barrett_scratch_len(ns.len(), ds.len())];
    limbs_mod_barrett(&mut [10, 10], &mut [10, 10, 10], ns, ds, &mut scratch);
}

#[test]
fn test_limbs_mod() {
    let test = |rs_in: &[Limb], ns: &[Limb], ds: &[Limb], rs_out: &[Limb]| {
        let mut rs = rs_in.to_vec();
        limbs_mod_to_out(&mut rs, ns, ds);
        assert_eq!(rs, rs_out);

        let rs = limbs_mod(ns, ds);
        let rs_limit = ds.len();
        assert_eq!(rs, &rs_out[..rs_limit]);
        assert_eq!(&rs_in[rs_limit..], &rs_out[rs_limit..]);

        verify_limbs_mod_2(rs_in, ns, ds, rs_out);
    };
    #[cfg(feature = "32_bit_limbs")]
    {
        let test_only_verify = |rs_in: &[Limb], ns: &[Limb], ds: &[Limb]| {
            let mut rs = rs_in.to_vec();
            limbs_mod_to_out(&mut rs, ns, ds);
            let rs_out_alt = rs;

            let rs = limbs_mod(ns, ds);
            let rs_limit = ds.len();
            let mut rs_out = rs_in.to_vec();
            rs_out[..rs_limit].copy_from_slice(&rs);

            assert_eq!(rs_out, rs_out_alt);
            verify_limbs_mod_2(rs_in, ns, ds, &rs_out);
        };
        test(&[10; 4], &[1, 2], &[3, 4], &[1, 2, 10, 10]);
        test(&[10; 4], &[1, 2, 3], &[4, 5], &[2576980381, 2, 10, 10]);
        test(
            &[10; 4],
            &[1, 2, 3],
            &[4, 0x80000000],
            &[4294967273, 1, 10, 10],
        );
        test(&[10; 3], &[0; 3], &[0, 0, 1], &[0; 3]);
        test(&[10; 3], &[0, 0, 1], &[0, 0, 1], &[0; 3]);
        test(&[10; 3], &[0; 4], &[0, 0, 1], &[0; 3]);
        test(&[10; 3], &[0, 0, 1], &[0, 1, 1], &[0, 0, 1]);
        test(
            &[10; 30],
            &[
                3099176493, 2686179191, 2963763290, 1498905807, 2459138342, 883505904, 186294937,
                1240988195, 1152000807, 3485254691, 3058980612, 4051918579, 84687528, 3474110821,
                825051253, 2495113655, 4269053297, 531158278, 2131423786, 735959410, 1116746011,
                2099249419, 427789428, 2182383446, 3493666451, 2086611578, 2349523902, 1865831092,
                104597580, 720145483, 1653677313, 2449086005, 1225118075, 1458687427, 2284889737,
                1178000580, 1204354477, 1109887135, 2302764777, 1833071533, 2749496868, 3989183692,
                3112262804, 2143914125, 71393498, 3686808425, 568295620, 856601035, 205234115,
                617861388, 102368116, 561132401, 733711249, 2931194848, 1371887453, 2942553563,
                977874680, 3567773306, 2225947571, 668306082, 3150218776, 3697051793, 4272816685,
                2926797142, 2177250555, 2840203688, 51052169, 1663516163, 3085068676, 471736374,
                3127282104, 1358199856, 250868071, 54398826, 2107868776, 1621910930, 1677400830,
                20889047, 4180454819, 908186988, 1625301399, 817226551, 4039437379, 261787419,
                1484678288, 1183594253, 3154399910, 888455895, 1713982625, 2925178936, 2076694919,
                2288237521, 620657938, 2457761902, 2906110636, 649773865, 1387981473, 1600821385,
                3512543520, 2803028516, 1584293468, 1443490691, 2589741655, 3170284085, 1902835632,
                1603116322, 2850443356, 60694796, 779743237, 4006039758, 3272304347, 2442366350,
                3601524312, 720808564, 3291979084, 1675880280, 4139440559, 3672978467, 3938651783,
                3257918674, 2077369764, 3027135446, 3940308694, 2533432781, 4105883289, 626796131,
                608794713, 810463080, 2363401530, 2800065942, 2679046183, 4063827982, 2327835929,
                522607612, 1660561124, 817356463, 2984129158, 146372510, 4293731514, 2558750639,
                3948304613, 1448810261, 826590770, 2423032883, 1050524737, 2770168039, 265887860,
                3353243705, 2674401993, 1496121248, 2053790031, 1863298286, 1068977659, 2875518960,
                4137037281, 2888534695, 473250405, 3368783492, 3797281057, 132028186, 316202261,
                381223941, 3366394660, 425124513, 4083952857, 2301832392, 3321645917, 509444341,
                361894284, 3976042471, 2957832871, 250836826, 1408114735, 1614725824, 4214313885,
                4081964203, 2663894263, 145380453, 2329325761, 2055092903, 126874014, 1000460254,
                3910319503, 2561073113, 4231784000, 1042785346, 3511487116, 1270832223, 3456586087,
                1210528079, 4119713948, 3299598065, 4178946856, 1623358249, 1001721969, 3964704699,
                4133343553, 2193492316, 407893767, 2484237727, 3782842008, 2959959832, 449539588,
                2978317444, 4008576738, 3478248618, 98697842, 3986898689, 1622966191, 3594042026,
                3280563264, 270713233, 3999415188, 401848066, 623805281, 1710411200, 1801291443,
                640646012, 2283134929, 1043531051,
            ],
            &[
                3120245340, 1558358768, 2311090615, 3377497278, 2047683134, 1453801550, 1974940808,
                1545066418, 2840151982, 2463969682, 2025473894, 697610529, 906262319,
            ],
            &[
                4239744213, 626950715, 3916878897, 2974637695, 488695410, 1094975104, 2496621051,
                205419221, 4192241096, 3583664917, 511224033, 2016541230, 718692763, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
            ],
        );
        test(
            &[10; 215],
            &[
                59488434, 3053025912, 3197673861, 28499967, 919132002, 3254889605, 3510992086,
                4155414397, 1401076337, 3663475665, 3859836963, 838452584, 1714061888, 756807500,
                721664913, 2122189093, 190377396, 772161692, 337532878, 2801246946, 3347165247,
                504261973, 4207433768, 3203836949, 1637702776, 3950126612, 3575550053, 630469114,
                2468674532, 3237190898, 2699368075, 2126693510, 2005217541, 990977199, 211751826,
                4095342386, 2390539499, 553381774, 1202910267, 3124334908, 3631027545, 3118560130,
                3505258080, 2533393565, 3897399802, 1644233433, 2966724466, 1228506720, 4266980301,
                1975806255, 3087455989, 430105567, 2496389669, 2559019944, 1343075842, 515604272,
                2623722375, 3549762134, 3517683279, 3564893038, 3816995397, 21727446, 2930818039,
                2672793843, 1589653728, 2917937865, 665591277, 1222288512, 1197537333, 2425822187,
                3647080994, 3180811677, 3800694132, 4006852228, 1392314914, 1823809919, 3315974102,
                2004045597, 2557007205, 532899885, 1088861606, 3969820933, 711573681, 3744617855,
                3618206965, 3834838995, 601286347, 1494520421, 2007503431, 3672534190, 3437739925,
                3075991743, 1108097829, 438830502, 854842946, 798336207, 2497251014, 985928236,
                1930130581, 260199427, 3446617597,
            ],
            &[4005818469, 1809723722],
            &[
                1137651599, 55996911, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10,
            ],
        );
        test(
            &[10; 178],
            &[
                3300759985, 47489222, 1596412201, 3681499224, 721942549, 2798268772, 1725678834,
                1018213306, 278963757, 2162182910, 4238919550, 1480123217, 3550714331, 2714414047,
                1584331762, 3790545549, 2452172382, 3823866136, 3793112677,
            ],
            &[
                2532879920, 749327979, 378994598, 3512427187, 1768486327, 874381062, 3750599339,
                1824619676, 4102262081, 3872688397, 4080392661, 1233784995, 3967512105, 536874,
                1334357500, 533131065, 3802334171, 2688028352,
            ],
            &[
                687097313, 233246948, 2670005176, 2918696316, 2059690051, 3559547370, 982825692,
                2828616847, 3397764024, 1640472335, 1827782800, 239756415, 4169556349, 1129911247,
                2514993175, 2469589340, 3137350801, 1512435665, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10,
            ],
        );
        test(
            &[10; 6],
            &[1, 2, 3, 4, 5, 6, 7, 8, 9],
            &[3, 4, 5, 6, 7, 0x80000000],
            &[718, 910, 1080, 1286, 1492, 2147483434],
        );
        test(
            &[10; 4],
            &[0, 0, 0, 3, u32::MAX, 0x80000000],
            &[3, 4, u32::MAX, 0x80000000],
            &[3, 4, 0xfffffffc, 0x7fffffff],
        );
        test(
            &[10; 10],
            &[500160962, 3482059973, 3833374734, 2382534866, 7345183],
            &[1962915382, 2761901894, 931109938],
            &[3497463394, 501014622, 297308821, 10, 10, 10, 10, 10, 10, 10],
        );
        test(
            &[10; 20],
            &[
                3406830026, 653096149, 665840651, 3475733215, 2990001405, 3962996135, 1125790437,
                510807273, 2268497988, 3801098676, 4251717337, 4009243500, 3093915781, 2221063229,
                1145961534, 1520950302, 4156341537, 4051749345, 1887697916, 2938053362, 1317017771,
                2551451381, 1165783725, 1692722530,
            ],
            &[
                757200076, 1850648953, 1310285965, 43796838, 2811750593, 2350568971, 1162649579,
                3588922014, 843685186, 327689313, 822131258, 632127361, 4287259347, 855554137,
                2978907761, 2140676852,
            ],
            &[
                1472562202, 467285767, 3874015352, 3774608073, 3547803297, 3252310888, 379016022,
                3915526178, 172347662, 1424794357, 2354655682, 1763403834, 1622341466, 1527845266,
                321699454, 1968063162, 10, 10, 10, 10,
            ],
        );
        test(
            &[10; 3],
            &[3156053040, 2869923590, 2300216426, 1369503],
            &[177914576, 1102898054, 1547778],
            &[0, 0, 0],
        );
        test(
            &[10; 255],
            &[
                4077004042, 60445643, 361888880, 1370511290, 4140611, 1545574856, 337863511,
                2696753299, 882513095, 2220102704, 4040224968, 3895008884, 3144563393, 2298197836,
                2781817257, 1704369652, 3931316078, 315544063, 3497851423, 3664143565, 952326443,
                3938628137, 2331019960, 3709840874, 886597817, 2718223965, 2271519236, 1069952925,
                2973484971, 694036791, 3948658826, 1091836994, 3976014345, 2590014773, 1030558914,
                3050284515, 4064531951, 227487465, 1614818201, 2853653798, 312516670, 1131265287,
                1079324522, 2790732553, 2857570668, 2459258033, 3676592728, 1861660079, 3846516218,
                2487383610, 820186200, 670229005, 795815915, 4236865449, 388462685, 3530199852,
                4067109237, 228011739, 3243897206, 2102622551, 3066279582, 2307396469, 3918457987,
                3177100158, 1165801255, 3470877002, 730654273, 225744395, 303984647, 71923315,
                2631641933, 642193977, 214491076, 4083126324, 1300818039, 3145270519, 3071074628,
                533164498, 2260983885, 2398373112, 2291850622, 2549903002, 2876921663, 2087750942,
                3012568578, 2072377257, 2300996389, 1724754966, 3663537905, 2452259556, 681943280,
                495483009, 1806854180, 2022877157, 3545541285, 3155213185, 1520127898, 881038528,
                2682219847, 2061051159, 4091777429, 2973569703, 2243163157, 3812478845, 2868568231,
                538252316, 915738068, 3645567840, 557499318, 586868707, 334386881, 1009292772,
                9683309, 1394382805, 1469121902, 3538152891, 192398621, 3298083372, 291538032,
                454155529, 3467440904, 107593274, 2581455983, 4068643130, 1289926312, 2620830750,
                3599696168, 845286964, 93649777, 211929373, 405469122, 2046160625, 4269019707,
                3153165259, 3273675158, 740768718, 1724953971, 4107531497, 3669145824, 2308427932,
                4057339865, 30691770, 4215434099, 2209368084, 3410096812, 3930010459, 4058257897,
                1401228000, 3205298856, 919645629, 1080800085, 3531478127, 4244298042, 1920158944,
                4150416695, 2942997287, 621554087, 2193963845, 3535984780, 1948458592, 1237268195,
                4059838770, 4178023707, 2225635923, 1078133440, 1904752780, 1129557671, 2721601376,
                3991770285, 2491529378, 2424674576, 4164361438, 1566434777, 3112804218, 2185237562,
                2557399204, 967876237, 2243639679, 2436745635, 2376273731, 3848434966, 1970386149,
                294413738, 1420657192, 694358663, 647017168, 4094657777, 3013362236, 590143693,
                2285757920, 2393504081, 116938458, 891518480, 696284131, 2710811430, 3317562620,
                1647436225, 1605219555, 3358343198, 2183786684, 1105785201, 248367120, 1398964063,
                3309346508, 3273593185, 2032930327, 2675472140, 557577078, 692086759, 1588022257,
                1734092444, 927029618, 3696557822, 3370092072, 2151067302, 2650483754, 3670460764,
                2763985465, 1126450086, 3029345718, 1504895755, 3694166108, 578785502, 294874178,
                110079616, 1327583117, 2661938072, 2345632606, 3722933355, 656745495, 671269721,
                2346516327, 1169529001, 4238866211, 949351039, 3500460463, 4199863511, 2082398726,
                1281275042, 2560151133, 1138051905, 468589983, 727511715, 2554428159, 2014177155,
                3346720979, 1317583392, 1714879892, 4249909774, 2158352329, 3040382616, 964600510,
                319055149, 890296957, 2071841867, 2811829970, 3945344817, 4075668231, 1336582250,
                1632615222, 2996499016, 1247703908, 2161089989, 1757474064, 1957983140, 2427088146,
                3819098678, 1414774317, 1685786700, 2842967660, 2434489739, 2748571761, 2393986575,
                585646425, 733271380, 4096908487, 866235842, 1811304066, 2337315068, 4064326552,
                2517677609, 595634311, 929413085, 2556167349, 2814648343, 3447175234, 4171314821,
                3354462286, 2190450759, 3693163187, 817168953, 753817804, 2679932775, 2584000650,
                511478235, 2858863655, 2320104001, 2367392444, 1724096919, 1152655320, 3131579385,
                2076635048, 3557452129, 3931367674, 1011051727, 3910860016, 3742755232, 3977120817,
                178461096, 1938068862, 1480470517, 2997286985, 196403685, 3581556360, 3800228840,
                2583177118, 3801520456, 1254931401, 3392702841, 1046934176, 412728369, 2864978525,
                970771733, 672204372, 1528132877, 2673063996, 740759188, 1840165684, 1821213771,
                3111387733, 2386566079, 2682984985, 808050061, 4160818440, 1656303941, 2884647798,
                3997066586, 3727980283, 4181671626, 2736389654, 1808865173, 1642983278, 875419327,
                3960909969, 3751011271, 2807559080, 1535615269, 2321637137, 2745054513, 3020092780,
                1359015906, 2466514344, 591997660, 2754906030, 3487658632, 3722637404, 1613546148,
                915678357, 1842708127, 701921987, 1207101857, 1974760740, 2410755764, 3324705651,
                3105821672, 335929431, 1887715703, 4035171395, 2815003797, 3632947325, 3664189665,
                2211203246, 1473602679, 3802261181, 3699100807, 3964975584, 2757250875, 3240191191,
                3824907808, 1582581356, 116402974, 2321914135, 933007824, 136161346, 2857937718,
                1044097090, 477414473, 3966936277, 21169877, 2518410639, 1938987054, 4164953085,
                3287593768, 96004465, 3316812094, 2635456314, 2931969475, 3535012794, 3954299175,
                4188146823, 1054040153, 4020697192, 3043463112, 487456176, 1536604607, 2191156801,
                2869189759, 650124563, 54254415, 910946976, 3240585021, 4240104628, 989174439,
                3235043341, 607641178, 2910679971, 4072297259, 2664931864, 2560203429, 3992282762,
                480443057, 3944140064, 1601223002, 3984987075, 3894479377, 1082591102, 2444518988,
                2566734417, 1272094512, 2581007089, 3838472665, 2810473520, 1590808097, 143027202,
                1667662742, 3686892725, 228309572, 4091365295, 1107025920, 938240502, 567559985,
                2237721627, 1939039548, 3053102548, 3379831217, 3536253061, 1586694963, 931323468,
                593457460, 1981974171, 443146100, 11888347, 2403661012, 2646485528, 528884126,
                1040587284, 2828170682, 512577110, 3534338212, 1642316111, 124785841,
            ],
            &[
                3924577409, 3801138880, 2554871927, 1797946680, 1773117295, 3509783608, 281892096,
                2358224613, 4217828194, 1339769957, 439570060, 3151617679, 1384372613, 4141908033,
                3034076248, 201556707, 1940999750, 1134129834, 1062995144, 3653931665, 1653789920,
                2219526834, 849219893, 3448402897, 890163003, 378127096, 3324725657, 2858888153,
                4101277783, 1736762052, 4083759525, 1398269743, 946713296, 1332088349, 1241448676,
                820326205, 1554372778, 2902257209, 2531752530, 593903741, 283370156, 1184475111,
                4048000423, 3262885223, 3065903568, 2181798675, 970937186, 2831985588, 3318099557,
                125775489, 2602810229, 1265835529, 370295842, 3885440035, 332195328, 803805465,
                2339766805, 3074827324, 176637217, 2717782694, 3228882886, 2079180306, 1985828369,
                1451458086, 2647399511, 653380804, 2816232893, 580123271, 1284828784, 1488365849,
                3992136885, 1712696753, 3200712671, 4019248351, 2502831049, 3353947366, 3805407296,
                3975182552, 1255379188, 3079539536, 2754826248, 2654365805, 1932758080, 2018940849,
                3987435655, 2107608701, 1979416324, 2553264951, 2332482333, 3854067399, 7383141,
                2032883058, 209316404, 1509786266, 891329562, 2446773803, 981357189, 289424276,
                1563495883, 1124421740, 4019623418, 229810471, 1293459273, 685378813, 763089605,
                543868756, 4294383498, 4256223402, 3521141578, 1300426165, 4203249157, 2446225842,
                3483103616, 2930204118, 3459214772, 2242535858, 2845721016, 2523036196, 1191008269,
                1438983871, 1990994626, 1369473842, 3325530252, 2436280648, 3247037040, 1827745637,
                1440370214, 565710731, 918521819, 3174181412, 2021188910, 1292274207, 3161583133,
                1940037031, 1320244184, 1944735324, 1624924970, 1914256377, 1558283566, 1557557944,
                819014915, 3477749819, 2317549528, 1815788616, 3410581584, 865672422, 3902451420,
                47231850, 2052078859, 2128580726, 2861615294, 2875806926, 2504678981, 2922243612,
                2567002225, 3176460060, 3501319351, 256937154, 986867411, 2877370533, 3829593455,
                169449010, 1762966206, 126552381, 3303194639, 437611604, 2065984359, 215674808,
                2549495983, 226202732, 19733282, 265226462, 732006741, 2011526642, 2592527346,
                3141322109, 980881336, 4085884669, 2038689570, 3704380296, 1688979285, 1869912321,
                1472588169, 3395855158, 2409096192, 2603014340, 2571643809, 1244430574, 2331499515,
                234553812, 3949996932, 1906820531, 1849149876, 4204335779, 1113287279, 1165814564,
                3834491692, 3992932111, 1545656810, 1251393152,
            ],
            &[
                1301325451, 2202463531, 827674839, 2990077551, 3515177881, 1362651403, 481136630,
                3397298241, 3518449263, 76128572, 4225087952, 1094548681, 282440964, 1243437992,
                2536206935, 1312305087, 2350135248, 1191467757, 772049416, 1651785710, 13757971,
                4177684085, 4225218737, 3829973687, 4224488792, 835349925, 807373018, 2003072240,
                1998180420, 1893011542, 2772647530, 1845722545, 3927750947, 1582656468, 3929302887,
                2309901890, 2219253077, 74452907, 3721812597, 3102504646, 1837697683, 4169869788,
                1109227870, 1893389948, 1637025892, 401123956, 2362840033, 3249816285, 1331916977,
                11330013, 1790389738, 4046514714, 4242300773, 386281183, 4154390602, 2907471583,
                596332521, 1031920105, 1416982457, 936732589, 620289351, 2021344934, 4134683536,
                254271151, 3153428294, 2775130118, 767392350, 1524517878, 1803212363, 1935475282,
                476597337, 1665283842, 385826494, 1418089792, 1583155781, 904332458, 2167494919,
                4213977373, 2749948942, 1366862995, 974249813, 2151831423, 2426239456, 4154619782,
                1909704093, 82889380, 1836307000, 1860128761, 2497608852, 302875545, 2512570768,
                3465381901, 3664521586, 2113690348, 3208498261, 211553098, 1122435807, 3989762258,
                1790955068, 1274985561, 801567833, 206607829, 1945509947, 2689438654, 630881630,
                3166357111, 950757161, 2525660644, 2782583030, 2082925684, 4017471838, 2770389652,
                3152723497, 2730012549, 375610667, 331640140, 3189394626, 1169047456, 1269402316,
                366259196, 659129976, 1835642575, 1109092533, 1409510258, 1055228915, 3865844484,
                1951513725, 1561115766, 1535328235, 566121235, 1218990885, 2397023975, 110038834,
                410651474, 2293988363, 1341825237, 3049031992, 1068909343, 3433095008, 226980250,
                234888265, 3753182189, 712474545, 3392300015, 358854770, 2053632965, 2314494072,
                3270641699, 1763895670, 4027033942, 3610557903, 1952033954, 3311300088, 2108677074,
                2293958755, 649449413, 1137877459, 1632674803, 3368118971, 4027363661, 3533599056,
                4112294692, 1992613374, 2157460184, 2344709644, 3126961604, 3915648450, 163946806,
                3452706048, 3278493797, 3569418312, 619110361, 1791108219, 53432167, 1675611199,
                3468608269, 3380246190, 3977636639, 2303818017, 2959386722, 4259696814, 2826759181,
                1786462977, 974733518, 2879253522, 3102475706, 2731858652, 2537398244, 2096104946,
                3000514581, 821433471, 3175348116, 2836803346, 3386301193, 400757797, 3528093517,
                689295706, 494008951, 3664544857, 1053036864, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10,
            ],
        );
        test(
            &[10; 1098],
            &[
                1563750116, 2761722522, 2234492700, 664378669, 2289696554, 166287485, 476834844,
                2920420087, 2528304999, 1938173917, 3950825207, 4243596790, 2986888741, 3513584208,
                1621935302, 3010962100, 97331004, 3497602837, 3156993605, 4246943277, 3773399914,
                1791046376, 1330596670, 2066725731, 3080890361, 1411737887, 1547876370, 3080590448,
                1532248923, 1435492412, 3448185497, 1125870178, 3236487585, 411921826, 2813696973,
                3217407722, 3223519146, 3065092232, 2369951103, 2509352358, 926337125, 3014682136,
                2966028824, 1505359010, 1713762975, 4092602064, 637732943, 1302647295, 2902550792,
                889992967, 3865589903, 1419953312, 9792449, 3415320263, 564771153, 1688162383,
                2484779041, 309493832, 218454065, 3482805065, 3854968150, 917520015, 394395502,
                1316087688, 4013170326, 3611139371, 3969996396, 845435642, 1250928392, 3564637530,
                3534519817, 4193257039, 1210838207, 1018604374, 1039438522, 3241956599, 283401613,
                3791643370, 2495704071, 2807365421, 66163757, 3720427495, 929326153, 1056427022,
                2224017890, 3983649303, 3242577483, 2252715700, 3763817420, 3945056819, 2536219802,
                2347365037, 1332467980, 932313386, 1344905435, 1546705437, 2064541576, 1511380419,
                3830839286, 1360902083, 3503290266, 3917441903, 1833414880, 3943874358, 1755117544,
                2923194969, 2839238526, 1436154878, 2022570233, 2627252135, 514344885, 3629007310,
                2734265902, 3693108522, 3208247898, 2234133971, 1972552128, 2954899309, 2020593517,
                2046949131, 3731357545, 1937584680, 3590346900, 3944796673, 1847972007, 2243279972,
                659814707, 354026232, 3897554349, 2861167827, 616720453, 3566033773, 667580062,
                2925065641, 1209421025, 2590558623, 4061392256, 1200913167, 951116272, 3677973046,
                3503505276, 3890324717, 588682794, 1822470598, 3062466072, 1622960923, 2217967478,
                1671943310, 3797078111, 306673750, 416365363, 1127402537, 4051830565, 1295357578,
                3597600014, 3944475003, 289083572, 792152829, 1866204223, 154676033, 1521441452,
                3508161103, 925444108, 1492828246, 661274700, 3234551268, 2848116256, 2684114954,
                1278505794, 1135004416, 1528837298, 903435517, 897822285, 1306717602, 1475128383,
                1820901356, 1682104357, 2694156349, 1295051939, 3518824442, 250688398, 2216356021,
                3513738778, 1822646861, 2230691522, 3766112863, 615919379, 1974329303, 1351423644,
                2477642991, 1213629777, 1378792281, 2617577340, 2140073178, 2191340227, 2566413020,
            ],
            &[
                3993985187, 418743558, 1313360511, 1682822200, 2190606963, 1153956890, 2732042229,
                1286070126, 662863720, 4186279397, 540453600, 3865675059, 1267823400, 3515598059,
                2652841176, 1202274130, 1747727592, 4024332644, 1774850346, 681999022, 3602470822,
                2758087563, 1879963947, 3431635945, 246837220, 425774791, 1105979956, 2942553343,
                1698889280, 2350552370, 3068812671, 3185616175, 3088938498, 878065968, 4173406915,
                3359260462, 3381157370, 170884832, 4229318755, 2494013993, 1549083413, 4140845217,
                4084551031, 3363271174, 234069829, 579586268, 409399673, 3823445787, 644521777,
                2518846537, 536802144, 2920279334, 18872050, 1166062818, 330384075, 567479240,
                242105288, 2256634487, 2099809688, 3182064703, 3455323968, 3796049588, 913224553,
                1273986744, 1216752331, 3471953608, 4141617369, 113668956, 2271813874, 3836046471,
                1442113087, 1985184453, 1605355495, 4002876308, 3555195054, 1327862010, 2467826608,
                2786784068, 2229364499, 4162880057, 2362120653, 2428764072, 3253302211, 4041072194,
                3342725375, 5091414, 4236090390, 645428080, 2998645452, 2029624491, 56740124,
                2165228859, 3087533984, 1636492749, 2209776976, 3692581237, 607934780, 2356087899,
                844604833, 3795358717, 1191407440, 2348446542, 2260870238, 3095317646, 2239633241,
                1510395276, 1414456297, 1992793921, 2093060671, 3555947012, 2097207883, 2341507439,
                1797902178, 3894103463, 589765482, 4279776370, 1462654158, 1306903445, 2072601153,
                2881422521, 41492691, 12234573, 1317588012, 460035424, 2087095783, 1325294692,
                639610198, 163158835, 583584804, 2753511772, 3964488699, 2486983401, 2238895215,
                1588375790, 2681620680, 4165955199,
            ],
            &[
                847465617, 4013314980, 3846912964, 910246757, 2972404216, 2762134840, 1953196515,
                4029750073, 4055861164, 1261155057, 4100430088, 2785511074, 497236120, 1367918422,
                3738577206, 736241263, 650868678, 2349520844, 3387563635, 741561301, 2423229549,
                2724745547, 894805326, 2982356553, 4201925489, 3718492797, 3268940283, 1316855216,
                535979702, 1867135512, 2894454664, 1953954059, 3331547089, 3629484636, 809197002,
                3453878941, 2710176999, 2236099000, 2474407754, 508610082, 3021250908, 950343151,
                2347617926, 3116087366, 387686701, 1840236979, 608794169, 80987566, 2929948001,
                3099591933, 1498594442, 976518465, 2412591727, 4182908032, 3079365048, 2450010282,
                2281968581, 2243779001, 309401913, 3783895152, 2093682035, 2006514294, 2836109544,
                964684103, 1645279117, 3042605589, 3518893342, 3753614018, 3245078642, 976146102,
                470909372, 417907929, 584991811, 4141480919, 3596625836, 3530716922, 1149488478,
                595234233, 353336372, 638993563, 341110766, 984515266, 1262652725, 843520981,
                594906936, 3922870845, 2571018879, 2329734185, 4233366705, 2923004114, 168356226,
                666545721, 3625364230, 2627653737, 2717371138, 3344260869, 893744508, 407453169,
                1376334915, 1670800848, 320471216, 2114539766, 1671037181, 886024377, 1940914459,
                1457430737, 1427559259, 3091470472, 2856003945, 2098123517, 1418768818, 2088924969,
                3884270171, 233325771, 2475658630, 3306400235, 1042747488, 2251057616, 288186037,
                1069520858, 1716749834, 2851756715, 3107695069, 4055132959, 1800778936, 1250813397,
                4225902318, 3898227912, 2436303557, 4099682647, 4014271299, 2579688947, 2474049743,
                1793988451, 1750223744, 2886556309, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10,
            ],
        );
        test(
            &[10; 915],
            &[
                685114319, 3257424506, 414393229, 2860628494, 121479429, 2579870766, 1597023839,
                2474392621, 3223958133, 1457435159, 3048194581, 3047568591, 328216793, 3688729284,
                1474966660, 3229714080, 1674178552, 3622356479, 2813923308, 420491140, 1181947036,
                2552013519, 2545590339, 977030817, 1410754865, 418734423, 3219708398, 3814271579,
                856825330, 886986018, 2340527841, 1574000724, 1326246490, 2345646501, 3649082184,
                1757236341, 3333117097, 4191792834, 771549916, 4014890173, 1767239699, 1537408864,
                860021926, 3301854273, 2439584990, 3450574632, 2067936331, 2217599411, 1936116337,
                3184038132, 3501118169, 1312763670, 1815889084, 4185547518, 1921708290, 4252193163,
                733366199, 1748333822, 3613571030, 2021878401, 674325326, 1834274504, 1974211381,
                2155793730, 666543182, 3988638747, 719903603, 4243752700, 3417033998, 578937389,
                1954345891, 438767411, 1067012960, 2140679028, 2616731558, 3608791372, 234168266,
                115663374, 37715525, 3155431063, 1484074906, 2987669067, 1980354536, 201618151,
                2443603378, 442745319, 2757914412, 1266139308, 4142086597, 1989465976, 3517367864,
                1441253229, 1295109068, 2757883716, 1533532909, 4121897334, 3324479034, 33282683,
                1821922930, 1002968212, 762196862, 13770263, 826603273, 4072569825, 780821896,
                788706413, 2104768306, 1607373740, 341951230, 1675442736, 3679554432, 4040499065,
                571466582, 467434507, 1883560688, 3831540337, 740943368, 2376257013, 1304728970,
                917813781, 3342830532, 3374539547, 1366263565, 1830393744, 3801219913, 3174561984,
                3973286677, 1398849159, 369072692, 656722452, 2994544393, 2007585192, 3393313477,
                2976738602, 1184554260, 1566038994, 826752733, 477094709, 3837269061, 2769881480,
                2709841643, 2163313442, 1223013930, 2855285371, 472880962, 695734454, 3106659328,
                336220721, 2424316775, 1005851508, 3997249632, 3813922059, 4109122372, 1011074885,
                44571353, 3135429322, 2678006854, 1812501650, 531726754, 684688016, 82728273,
                2816638159, 2837354685, 3655899911, 36796549, 4088908020, 1199108102, 2363335387,
                2702162409, 1359179115, 4038752961, 4132030616, 3472925013, 918360670, 1612076468,
                1617334280, 3399902835, 1794719516, 1364930290, 3884047381, 1715844217, 2543581627,
                1948226880, 3734808909, 1123962649, 6885664, 4055662667, 2036545059, 1825684950,
                626135857, 3682021373, 2923868404, 1141437069, 301320286, 2038697946, 4203441370,
                2625080149, 2416510088, 3453059660, 2196830401, 1003239496, 766384828, 1454135529,
                3753927217, 289196672, 3027589815, 386319177, 4286570851, 34998813, 2808034465,
                654631613, 2919774640, 3980646343, 3390105294, 3876707444, 342623382, 3063311246,
                2951194817, 2409427609, 277380170, 1128962197, 512899487, 1130696384, 337608154,
                4248250968, 2538526153, 408791364, 1355901969, 930023605, 619907788, 1270155017,
                2669635170,
            ],
            &[
                933436633, 2314417619, 1779862136, 2339830299, 3359221691, 1983848875, 2097698892,
                1645402710, 49190984, 3806363526, 2374325643, 638588450, 3467828663, 2693267297,
                3081019625, 2568134532, 3644457728, 2630819968, 707790566, 1984505565, 3749563552,
                3700374589, 2579510542, 4246015133, 1527325532, 3034605869, 2134963426, 3613350443,
                2082268909, 3145097012, 497158738, 1750605816, 1683654362, 1392048080, 2595287102,
                1859642116, 3880370994, 773829956, 2727575776, 868421082, 910865307, 4010486592,
                72360528, 2330397412, 2764112435, 2833247271, 1236763483, 1139702723, 3454679019,
                3138128998, 3268622050, 3622582141, 1726130545, 2831795892, 391680447, 1964919870,
                2937963749, 260385042, 1893950828, 1182888075, 3460030123, 2590897592, 3391091057,
                3700415037, 1619162545, 1524016666, 1313243906, 3716478858, 1054838129, 1929422210,
                4093970820, 1243478860, 3650034984, 3598628391, 99479112, 2103638976, 3422493162,
                3133671222, 3540259323, 1874029880, 1634209319, 2379655185, 1946213151, 2520479253,
            ],
            &[
                4135839129, 3082159379, 2808741151, 2746492478, 1317271925, 1590248590, 1871245480,
                883232624, 3985674442, 3997584338, 2338792001, 1776575346, 2531584902, 261625854,
                3578066182, 2058830353, 684820033, 3683099826, 1776182826, 2182228087, 919424929,
                4095708279, 1423878550, 172604911, 255380658, 4104949687, 3178922494, 1444103543,
                1691042525, 1011488353, 3955571774, 2253259467, 3143874569, 377143694, 2390377782,
                1070304427, 1715840158, 2972468795, 455414172, 3300053546, 2552753048, 3276730351,
                1581696761, 1405031742, 2503564946, 1980393840, 2686153828, 3279538716, 1074513470,
                2389426005, 592338809, 348493719, 3669366843, 2086362650, 1888752201, 1485850549,
                3098846363, 839653456, 2380177511, 1732519385, 1998829691, 3296699081, 2705709135,
                2848494034, 4155180828, 1425421469, 3752183557, 2319259329, 2757221818, 1921158733,
                3302049214, 1696454223, 3356952349, 3100878977, 324054921, 2131412976, 1078305944,
                698318350, 4151030129, 2259288990, 762849915, 3134288938, 4090864118, 1223661238,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
            ],
        );
        test(
            &[10; 1458],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[
                1699768805, 1493014143, 2592376845, 190926105, 542951397, 3841252648, 1343830055,
                2073311615, 2471501748, 1744126095, 4269392231, 3721081862, 3530443208, 1084284148,
                467429630, 306942799, 1316059964, 2458728017, 4217570360, 2624704102, 3521450981,
                2608224726, 3040435965, 2267671055, 471534848, 3629410298, 1363200167, 203206966,
                3414918917, 3335382360, 1913622484, 1218762755, 3956682428, 1037359525, 5531615,
                3407338094, 3864111971, 3057551672, 1048359741, 3074041931, 1285559492, 2147056083,
                4091000908, 3900432305, 3556431023, 1451794634, 2393864263, 2505694122, 2968009980,
                1048797969, 1048151529, 817832869, 2521994756, 2426006163, 3275649087, 941801312,
                1575422758, 361314564, 722834359, 4247301247, 2186131251, 3429599060, 2212966801,
                1219957676, 1702525945, 940680423, 333505183, 2493537355, 354760063, 274436852,
                1639243309, 2924727204, 4285739645, 4001422648, 105606253, 3112142556, 3975066309,
                409404932, 3040839303, 4022137216, 276092852, 3345019055, 1650156126, 169273103,
                1920493641, 197903490, 4009771827, 838073342, 3939112618, 3406907996, 4120898973,
                2720730518, 2834644027, 3448317034, 3673080760, 1810888516, 2181591183, 2952080694,
                3794819074, 676604950, 3118739900, 606918192, 1316167449, 2730639798, 1815557579,
                1808982050, 2109827633, 1671035061, 7558450, 1554665087, 520056416, 4218246290,
                1161585267, 1878255675, 2486870832, 2088887262, 1303298097, 164712340, 2377403159,
                525890592, 3347413040, 1697811557, 3373912443, 1800652721, 1430587466, 630390988,
                1443110580, 572173114, 3006613569, 163802577, 3661300485, 2066285319, 3197778371,
                1266266830, 3617690296, 4233259050, 3805452611, 2682754452, 1121170085, 4239996815,
                2574739406, 3293943958, 589250712, 694935820, 2394149134, 3507180662, 2403010680,
                1341157799, 688040930, 1064943055, 1576175762, 2748814503, 3365706670, 2331616371,
                3891308187, 3625939659, 834228833, 2747861390, 4238259694, 2400594789, 1064448439,
                1457694712, 2503273199, 848323770, 3879018391, 419366498, 2787183492, 1572178433,
                1143352485, 4132728989, 3611364165, 2042218208, 4202715626, 4222254124, 3573214358,
                2530238089, 3214459960, 199438106, 1990033748, 3797350881, 2039620692, 2196170141,
                2867974320, 218349677, 2334991902, 2199173454, 2635928703, 1960733130, 3298739914,
                1745262170, 2022670546, 1773942006, 4022957449, 3487536364, 4203797706, 321674131,
                2963478216, 482717037, 423207073, 2529853593, 115650857, 1559406958, 2515253401,
                4253176221, 1494891359, 54215779, 145831030, 2534290332, 722379383, 3288965822,
                3478280697, 114345927, 438460383, 4120362834, 3417392370, 3871144709, 2774707897,
                2662801524, 3802201761, 1044223137, 817062608, 964570171, 2627601973, 2648686193,
                2728498120, 1589690733, 3285610776, 295629246, 3995872256, 2573726546, 231960020,
                4274030532, 3444536058, 3882840099, 2252235022, 3959235737, 3485371602, 1365773288,
                33313646, 1392617408, 1844121885, 993486158, 821274239, 2676605019, 3812510576,
                3432776114, 495146668, 216967234, 3172737228, 3417198461, 1629229154, 1821830950,
                1340735610, 3211463144, 3560398150, 1050123642, 874441227, 3169516221, 2783241998,
                99843330, 1175436161, 2825581162, 3259620982, 3350111857, 1606464613, 1911911079,
                4127551535, 3229370127, 3828823712, 2744685123, 2859704702, 1106636072, 369804135,
                2989745290, 2768260583, 3275220193, 1104864623, 2290350054, 1049972093, 1592992521,
                3598788064, 3350912109, 2954408081, 4230173468, 2901042813, 1614811380, 1930643862,
                4235254046, 3520012632, 1160373738, 3249852125, 923106635, 2511906301, 2055083218,
                3193170540, 843255473, 3274357481, 1069334404, 2160558830, 1247811861, 1960090087,
                1260505746, 273679994, 628898405, 3731946987, 3821611375, 1479367897, 406962212,
                1951524671, 151905808, 2697376333, 2827375721, 3738186384, 2398267382, 1189552516,
                3271847863, 3695955312, 839640611, 391563549, 2851226168, 533382637, 4005632192,
                779063015, 972248299, 4160069328, 1551848869, 2901173293, 1788886403, 7742522,
                1152154246, 3162815742, 3933026131, 2093435260, 293643707, 537139803, 166837469,
                1353454767, 647497063, 3056417780, 4130500121, 2563320691, 3004161478, 4266673072,
                1525376895, 2692236203, 612090721, 3803502732, 1472545930, 1285255741, 3563385061,
                354899818, 3985901869, 1077755288, 3764626839, 1736779714, 1388617683, 373259603,
                406803142, 2250511459, 3661744930, 1712371282, 3755716304, 1001652736, 1437790589,
                2722214878, 3509431789, 1045623456, 1477384299, 1529044191, 3485492831, 777726776,
                3111945645, 87325680, 3674053827, 3777562490, 2467629264, 3920974102, 1118313420,
                518650065, 2779852693, 3938004723, 351444334, 2301762313, 1696242659, 915234550,
                3426770333, 2623516555, 2367502869, 1764681654, 3012087599, 2970366387, 1214823325,
                1416807413, 1002068525, 902786108, 4219021066, 3685197343, 627261248, 275319515,
                1048180581, 184895903, 4054317442, 1164782510, 2969271356, 448119019, 3225650628,
                3586253056, 13844949, 4265695528, 3156376136, 2094798072, 857381027, 2996376430,
                2313835774, 421527113, 577824604, 2027011737, 1936987926, 1716119129, 2495416975,
                1566342556, 1974265475, 100699876, 3818621196, 864167162, 264481383, 2779691848,
                2519054856, 356904428, 1425193402, 2029146838, 3791198981, 1181286580, 2106244436,
                4074963191, 1156817098, 2002193018, 584536494, 1252244761, 3576951572, 2017920970,
                1604293290, 3010503460, 1201492606, 1555982397, 2556964369, 3428819809, 4168489079,
                4171058202, 404720822, 2480856269, 1813722898, 2093837710, 3164853594, 3430042720,
                2545385245, 1253945679, 2067139932, 1049755734, 4162345450, 1330690410, 2567802121,
                3494789533, 4027320953, 1359086506, 697144652, 2171860846, 1885681379, 3200348033,
                779514913, 3892431243, 3292022849, 3206367229, 141294896, 1247341423, 1286221471,
                2030904879, 133272354, 1441910339, 3982161305, 2696309947, 3801057775, 4053369379,
                3233631762, 3173498696, 19486328, 3282282805, 2117833655, 416934200, 2476837107,
                3550654071, 835595228, 2784887835, 1849982594, 1215825494, 120169702, 3628150453,
                813273996, 3164608875, 2585886019, 3468354974, 2529654702, 3937155612, 2948360421,
                3839784361, 3626976122, 3884938510, 2182665318, 1984077334, 3592802345, 974314095,
                2658877268, 3320515471, 3220348329, 2057524489, 3330170042, 1859432638, 180803537,
                1712128971, 2278567221, 2233418239, 2029256422, 183505236, 1671411329, 207176584,
                2036148357, 2313901094, 3988184694, 2436690588, 211724920, 3032356297, 3340214696,
                117345076, 1500403818, 1365050888, 2323945197, 2919933846, 3023641486, 599606728,
                3628676259, 1581385213, 1427038937, 807104522, 3978940995, 3629484758, 47151915,
                1224789415, 2759574529, 2839954032, 1315873434, 3168847820, 1610459325, 3726598118,
                896590825, 2419742875, 3430079217, 3778931934, 3687222980, 332999836, 1043316180,
                4189864653, 685607150, 3695627010, 196835870, 3941428183, 676485145, 986494888,
                1049128467, 1888162207, 2801108003, 2947315222, 1894059346, 454711531, 3589935315,
                1803051198, 3655137369, 983821858, 3081875497, 3803393764, 2428490033, 1679378736,
                246071720, 1483050983, 1382043974, 895942294, 2442009263, 68111122, 1626094364,
                1007851423, 4064984075, 1533309302, 3360582064, 2740180933, 883885210, 2989133634,
                678055765, 2661431250, 92559696, 1032783565, 22899740, 2361882691, 2393385114,
                992907787, 3373832203, 343145453, 1165167516, 290287284, 1707286378, 634939907,
                1875541381, 1297012104, 3157634640, 1929967474, 1519414443, 3881762803, 2681995819,
                1775206192, 2755218098, 3910825543, 1860602393, 1832806570, 415596447, 2186155905,
                2791826564, 1816344456, 4023525966, 243382839, 748453584, 512556810, 3922005623,
                1061467548, 276741166, 2229273612, 1738207771, 4128865796, 1984054190, 1324502121,
                2297662740, 1222235249, 3182342363, 1665477641, 2147473685, 2421806673, 2322492629,
                3459362839, 293710623, 3706578807, 1598383617, 3666283079, 1011133678, 2189483475,
                938644636, 847088475, 195518118, 544927250, 1332819612, 2366720962, 3995717811,
                1985243513, 948001598, 634366393, 3212334449, 4000960249, 3974258285, 4013403482,
                418753555, 1101654041, 1090819318, 1675158019, 3683152713, 1204756106, 3841987860,
                2260147808, 102023094, 238154404, 533560693, 4086126380, 642534617, 1632145671,
                1006291121, 1435987078, 1313038313, 4186175328, 843360286, 1839639056, 1353648132,
                2221125136, 3732368512, 3339185832, 662563454, 2744469998, 1331973650, 1343096038,
                3251304725, 3601378173, 3422205067, 172706680, 908147887, 3728762498, 3795011129,
                3044217576, 67031330, 2499472547, 4147725229, 2529069049, 3741434149, 4201254595,
                3779595001, 42489268, 2053548825, 985282652, 3980674776, 3248961215, 3376433654,
                38182659, 2602101994, 1925777855, 806407427, 2317267910, 1380652265, 3701097444,
                1220848862, 2025812459, 3482791264, 1753046753, 598143651, 2616070898, 2479609320,
                1868138196, 945178683, 3832269010, 2314371642, 2400732781, 2048750780, 54093670,
                327716566, 3334970102, 28861124, 4118278855, 3714357594, 541833330, 2000911880,
                2666752754, 499968133, 1548730866, 1814521338, 2487369228, 1138644653, 739248636,
                126190040, 3978033791, 1817304821, 1225794794, 2871919793, 940731169, 504076316,
                995594649, 3329757458, 2846947354, 3891292748, 3959049218, 2127628616, 810346662,
                2271875593, 3880247584, 1164066271, 1750445267, 338639562, 1699074958, 959302743,
                2956590033, 867798509, 627451437, 3510106002, 2473252158, 1509206912, 3352377241,
                3802665519, 2147957812, 56184472, 1718654148, 3672624374, 3511414009, 2682054173,
                3922316656, 2414259176, 1496250883, 3967344788, 3214918603, 1624761599, 2724006253,
                978208159, 1113357772, 3487098045, 4270740839, 633061828, 2261921853, 3462914738,
                1530768806, 1364864048, 2876294873, 2510835104, 4242821662, 2474987381, 3633586220,
                979932165, 4155189874, 2525154569, 1335483880, 639760624, 1660790427, 2226690500,
                4135706134, 150618757, 593063255, 3258479253, 3728374085, 3007800944, 3679685736,
                694904689, 102552353, 1428747995, 1176285881, 641509845, 3270509599, 272169854,
                3047261598, 1287492008, 3351279172, 2404469180, 459751592, 1333387953, 735177161,
                215716037, 536597459, 3665567562, 822815388, 3602692316, 3168229037, 1054147108,
                4146505577, 1580773238, 1088501561, 3629868530, 2978204283, 3610240644, 2223237525,
                3153209697, 448561701, 895234984, 1592358205, 57857782, 851682344, 4181525201,
                1988009551, 3651424110, 3655716749, 3787661870, 2925252014, 3708997912, 3309060091,
                4188222270, 1673276025, 2192528846, 958274526, 1258766035, 3525801758, 1215614889,
                3051826051, 2354974337, 1541657893, 1271755575, 496264166, 2626820052, 936952045,
                3785181421, 1294902361, 1958901697, 1604821791, 218282718, 2246953449, 538193118,
                1919817946, 1243076425, 799521120, 279827487, 2722496365, 4019574708, 990869332,
                2181585056, 2705356597, 610157367, 4119248513, 3343890269, 3121261960, 4085000780,
                1022014736, 4240976369, 2817889889, 4075894073, 560049242, 373205120, 334714162,
                1892034277, 51733004, 1776487312, 1265439929, 529285398, 2048981213, 1833004632,
                827301005, 1648393113, 4281016481, 2048185380, 784315559, 3245679058, 3513265995,
                1369181664, 4269143782, 113931975, 1398194472, 540409306, 216516009, 2627890586,
                3694809441, 2573974797, 1396611872, 2646365320, 988053471, 84792061, 963667070,
                2114579379, 3564371585, 3893773169, 4197725787, 2378021272, 3273607654, 1738197474,
                2402812179, 1278628015, 2317178895, 3160300732, 603898477, 1287079046, 392763347,
                445875312, 318139180, 1903469143, 3585413276, 885483094, 1674549044, 3911672420,
                1575430246, 1867282418, 2115410448, 4189894183, 3512999498, 2833930381, 4284803257,
                2594863293, 3053172089, 442647755, 2456733276, 3207156565, 1248598530, 3135346976,
                2511563725, 2761431370, 3552853409, 3064928230, 1668127734, 2151818142, 1573413967,
                2528903432, 2255579949, 4086541389, 1008056620, 651544198, 4099217330, 2147059795,
                4189054906, 3160158060, 3084688966, 1829066970, 366751766, 1086760266, 1823021570,
                3035776668, 3725549438, 303569416, 1637955291, 3070457854, 2756834184, 1117066585,
                2815830458, 1229154243, 978732541, 3666377420, 2646214554, 3084488532, 2757010866,
                1756160517, 2475577838, 467860047, 3125348085, 990351000, 3303850398, 1120462919,
                1920695807, 2980611720, 142863055, 2378648555, 2707355873, 3519612422, 3266835497,
                2512195477, 1941306513, 365527762, 592954943, 2552580189, 2606501901, 1933724422,
                2729691276, 28289820, 3017728757, 3650961916, 3696374759, 4228636665, 1241471494,
                3355869165, 1455843481, 2403725149, 829901760, 2618851388, 2623034964, 2752098284,
                962418521, 964160519, 1174284358, 2043907493, 848741317, 2520932535, 53649688,
                4228750878, 3694987632, 2694050164, 1097113047, 1221352269, 3997202409, 2858697527,
                2874569664, 82995009, 2474870382, 608319794, 2793235942, 1762763510, 593568738,
                4140942494, 3399850155, 3632742015, 1495257590, 1987803076, 3040781503, 4185563847,
                2095097137, 2984180367, 2385825018, 276655462, 2186224265, 242785858, 25971964,
                960934651, 4259630073, 2869516035, 1099452330, 1108772812, 2208212843, 138327599,
                3047969124, 2406962821, 3234115454, 2013090243, 1253595634, 379537518, 966627542,
                2289594467, 3820588844, 3307962615, 2251395356, 1086612191, 2411787092, 716861072,
                488537062, 4279016079, 1024670198, 3162868375, 2993498248, 318817475, 1204805151,
                2917216152, 202544687, 4055006202, 2166902046, 777932120, 1954248209, 1436706075,
                392807099, 3560992122, 3690436481, 4086456539, 1672219993, 718561908, 1329443408,
                41759831, 3830929272, 468558885, 2850993192, 1203438200, 173099196, 3100895691,
                3212371819, 2727223413, 265778175, 1306749738, 1945372531, 3409735664, 1829111612,
                73857789, 1967847248, 111126146, 1941085939, 2157306354, 932967688, 1524770100,
                3562820290, 1815865396, 618928660, 1954521715, 2791055048, 1472833149, 2745012885,
                2590581857, 764068138, 3810864565, 2068720839, 423731968, 2781829173, 671016197,
                3626671545, 578830019, 2263629204, 3196999908, 979097653, 3960854009, 2652280123,
                2014180777, 3212225669, 1901793582, 791028866, 1535961645, 3425925260, 1728220859,
                906987059, 1320441954, 103740087, 138471919, 2751991892, 3763793139, 1930252328,
                2302777807, 4166228863, 1898736904, 2679190175, 1902812084, 3494412200, 2003091421,
                647088593, 1275527793, 1571337951, 1296166873, 952173553, 3873665860, 2863399704,
                1687749991, 1019315430, 262989750, 1995806398, 3977438566, 3065387570, 194327737,
                2433254350, 3852864758, 3086016127, 1848276257, 872733825, 4293282833, 3839899541,
                1287823073, 2617969813, 1718468275, 640897626, 2796238324, 2471345611, 3038580905,
                1824539690, 2178312422, 3642675324, 4168141874, 4093223514, 1058804935, 1645178046,
                3320651392, 2520046692, 1724703883, 821899232, 481110608, 2492775734, 922020621,
                2923405792, 3392950548, 76237439, 3904683294, 256504571, 727968315, 59423766,
                3078236506, 2465675774, 2942973709, 2930861254, 2455418902, 2986403934, 3056400696,
                3599867309, 595501194, 2042151651, 2763604081, 65000450, 398696589, 3026108404,
                4199385741, 3349995311, 129915120, 2486649758, 775313272, 2784093349, 1582665104,
                2775814261, 1584569957, 2195691054, 2575677337, 3244017237, 2457054839, 3897157261,
                3195991591, 2030342150, 3727062402, 3706029571, 1179068874, 94821848, 2230566710,
                3669075264, 2425831169, 2438414276, 1720898289, 138216286, 3807634131, 346144588,
                1780209932, 694722911, 297024116, 2795490416, 1836169972, 2915769303, 591959219,
                2377921602, 4158591575, 1182337705, 231710027, 2274507465, 2241869617, 300827881,
                4150367209, 3585840019, 4167111741, 1023965277, 1674153048, 887595527, 1601672948,
                1765479313, 4065509701, 3915091640, 1297871355, 3479625154, 2831036720, 3785201761,
                1221164796, 1647627464, 2551945655, 1768755122, 2079192478, 347747455, 4045445050,
                2954562195, 3495749867, 2264323612, 3116421719, 2118887029, 432847575, 3976298717,
                30083877, 78749340, 2656872748, 3171516677, 944561854, 202462342, 339834758,
                1199920281, 488096540, 3380838041, 3470160182, 1755758102, 803450485, 738480354,
                357618351, 8551902, 812925594, 3218358491, 2918846080, 3174772578, 822290689,
                51990462, 3350891640, 2729296982, 2723805763, 3540769319, 1145905142, 1754039404,
                4289454572, 4153901476, 1336760032, 2717506348, 2938254966, 1476941276, 3458786809,
                2378593502, 1537570700, 476723401, 1207306095, 1092996894, 1187377294, 3666979500,
                3948059620, 4150681293, 719413909, 1118296918, 3753268822, 905882560, 1638884713,
                1259563753, 1063300422, 1502884288, 2285369399, 1874551590, 2040785043, 1775679918,
                1144757520, 3854262168, 3821097323, 282903083, 197139966, 976017372, 3684024526,
                3746169537, 108937132, 2444167905, 3366454633, 1002555971, 3961863482, 1693512346,
                4219424106, 2731825813, 2173055658, 3589347522, 1180897582, 349307065, 132252171,
                1286185962, 2906205536, 2843606103, 27822898, 2481699072, 2948630201, 1774092707,
                1171123309, 3404061713, 2905420837, 2351973006, 3971568799, 3248015376, 2297549471,
                2624549152, 2864086950, 2757145051, 284981704, 4116824485, 786416861, 455364234,
                810567289, 1304615212, 2127708286, 3066389895, 3906659140, 1621009466, 4060905456,
                2498550541, 2021095657, 189923381, 3015918335, 394513300, 395082750, 1997152191,
                3900214471, 2076041515, 2762331608, 3059576182, 634566583, 2077367009, 239466771,
                3512081528, 3492757347, 1343506243, 144657866, 1186993011, 2942254420, 3813487876,
                640076454, 3107898063, 4057144155, 2951251839, 4059833593, 2265248880, 2051308911,
                3838642653, 1427962716, 1138966813, 1697969541, 3885404282, 2547515947, 2006341172,
                1338884621, 3673075887, 2250647769, 2303605646, 4029999904, 2015620651, 429025216,
                4230380695, 3438741639,
            ],
            &[
                1988498380, 3050132581, 84001963, 3445692001, 2734783387, 1774219496, 2162032114,
                770345120, 3457703796, 3559428633, 4257360364, 1090466591, 3567817053, 2497380424,
                3534691541, 1279435419, 2742018051, 2205075879, 641106, 1754923528, 58973331,
                1715000559, 1117715270, 2272741525, 2115935795, 1483835360, 1296235, 1796957756,
                3968651424, 152068196, 176938861, 570875261, 3336099773, 1193842175, 3048757774,
                764666488, 3843941623, 1548464729, 1897867882, 2878929765, 448571520, 2406065734,
                3568376411, 108079736, 153179622, 376615879, 462260105, 973865240, 3413391101,
                2811718165, 3572533591, 3909460695, 3134008033, 3897396582, 1040616570, 2998174467,
                4068999806, 2523162074, 948788147, 2600716643, 3959424266, 2966714, 526886335,
                3296693425, 2243375988, 4143935802, 1111766991, 1144066805, 2311007495, 1010976381,
                845073483, 1135483459, 3990880342, 294797346, 2679966602, 2533930988, 1664564718,
                268281267, 2092673357, 469524764, 3945653712, 326073803, 2694037849, 3507600917,
                3290201609, 121045908, 22654279, 1505718612, 551101589, 2556674091, 3163892929,
                1910818775, 3418844366, 3372741206, 130020156, 715267628, 4274925122, 1048026555,
                2835427841, 3505581892, 1602295761, 2414351668, 1553499591, 4229635398, 2519192505,
                2021787900, 2301738189, 2691817702, 5305850, 1565221152, 3538706617, 3057801765,
                2874943612, 3131932103, 2612726193, 1166707551, 2820888127, 3160538692, 3124232750,
                481384322, 2617755783, 3630694751, 1672329146, 3497041986, 441500257, 1563794131,
                666530710, 1149258814, 4274365529, 169180791, 4187510295, 2714548495, 3562799169,
                2904591043, 1774457729, 800521106, 1411048330, 2663300606, 22655328, 3774883767,
                3819965706, 3800295936, 3668492475, 1379825015, 4213938165, 1132023495, 3760304231,
                2917225913, 189390673, 1938798483, 3140766517, 2153396970, 1928404388, 2785977327,
                934591575, 2405323840, 3637359453, 981340132, 242310679, 981811654, 2288564752,
                2987925303, 4031514738, 197635692, 2859303437, 2533609681, 1151769485, 2644644277,
                2635940433, 3366453887, 4277743333, 929550085, 40133408, 1833321431, 2429701519,
                1464545187, 3066929948, 3904082769, 373116082, 2430829492, 2571513045, 3885018135,
                603068030, 1172828581, 4065558289, 1163895893, 2468059427, 1548489900, 1717402470,
                4016751470, 1013738008, 1034029588, 3482329597, 3435207029, 1673410325, 397717038,
                1500823297, 1911172810, 1420629560, 3358784452, 1312197637, 1152173034, 367120884,
                384630941, 3440282377, 2522765605, 1597847376, 2683717257, 2561199311, 639683785,
                3817861808, 463544224, 3991704969, 3376721583, 105154089, 1533594425, 335823085,
                1107739913, 1452695631, 954081147, 1472744072, 109401385, 3210541127, 1847806577,
                327707567, 2422910059, 2867854042, 1286261864, 2777291397, 2491134001, 1866376440,
                1442628329, 1148774257, 327348168, 796722022, 1651402005, 2839518531, 707220227,
                442580375, 614584592, 4054371638, 313021875, 3191805300, 2207878775, 3933190445,
                2035546077, 381129617, 3161098198, 1019615010, 2743759521, 1458405016, 1891243747,
                3502084250, 951344904, 2669441803, 966435550, 1450947158, 2445618755, 2629179958,
                1786188217, 1157343233, 512400759, 3058846955, 1691540553, 3902487730, 53457416,
                827127510, 109080803, 2065162700, 2595989450, 514516885, 3571421189, 1946474067,
                3695201586, 529285628, 2120794437, 1831163308, 1518439076, 3361874260, 3805558145,
                2288973775, 2352901588, 4206307376, 1343461937, 1115914255, 241429811, 2386351727,
                3283851422, 1570726296, 4171557460, 2197857248, 3493510408, 2254067927, 3407035296,
                1925479341, 3186474620, 953208782, 715061374, 1181448968, 1220358590, 1370257748,
                2925654278, 3323438388, 1322650109, 3766704524, 3870430557, 3257441173, 2439781482,
                1554488089, 800150735, 1514175574, 3153400949, 1583649143, 1280421056, 3882604400,
                2181270798, 2782475677, 3148486479, 4071326639, 1764147111, 3505719068, 1862656169,
                1261162987, 2211270974, 3217710591, 2927566184, 4232715549, 3582658271, 1363726747,
                3233373581, 3022128605, 3193686951, 1851291940, 2618042857, 2848579530, 4293396464,
                1928252986, 528145280, 300907511, 3891802439, 1267856769, 1165928777, 4227625628,
                540012700, 469250948, 966533455, 2875072197, 2230103081, 2000106078, 3086794985,
                4244235827, 3081744548, 3713073740, 2925851679, 2315339278, 2558156736, 3723155058,
                3227292432, 2941681952, 4041413976, 2097513703, 3042683335, 3624088138, 3936977773,
                2820887559, 1664662915, 2334141648, 4092839529, 274159708, 4055649255, 3512716978,
                1365039076, 3916570667, 2158939813, 403036637, 942532489, 378445989, 2167306547,
                753402418, 1193358264, 3877024670, 3743152347, 116435136, 1948333248, 569098908,
                2922999784, 917671206, 2718173867, 1010784137, 1804409345, 2242631895, 3989814639,
                2044343096, 723486672, 1841507274, 2333301606, 4260854855, 2763867469, 2805327422,
                2351089720, 1719837442, 4008440919, 16200186, 1228294632, 3833322142, 151876299,
                3340352893, 647820547, 3228338641, 3940895065, 1697980005, 2505263582, 77077868,
                1311758352, 2346165371, 2652028800, 3480066477, 1481299332, 2948248752, 501377681,
                3276784059, 4102724530, 1207318829, 2947644680, 3069480791, 1116349810, 3395241135,
                3570300879, 3836110678, 23881082, 2523984619, 86893874, 2919930037, 3241130876,
                3697730978, 1459589531, 486161579, 3036213671, 2106273230, 391770200, 1135761788,
                3542580424, 2902564186, 4169447111, 1908429065, 498329772, 2010302804, 1930725702,
                1614128757, 1901201146, 2340750074, 2621544559, 1554979341, 2490973900, 3039157328,
                2525878574, 2064002895, 2981842962, 513591182, 48663763, 346106995, 1067873617,
                1664056855, 3497080122, 2640223678, 4006771320, 1595836468, 2008498009, 1036368219,
                238997308, 168075897, 876079826, 2934854956, 1075263443, 3803042525, 2802898866,
                2169905202, 3377165843, 2465797619, 1978983742, 2322310751, 1590628498, 1459551643,
                4156365433, 3054380889, 1819890111, 2928185712, 2553214234, 3066671630, 3394771139,
                1734126526, 247246953, 3320484300, 579355057, 1177404814, 1327413352, 2035170753,
                1052379386, 3339678481, 2430828601, 3360847369, 2961791848, 484983472, 3181472945,
                3105431626, 1283580906, 3703197182, 1961222326, 3649286491, 2664156595, 2335129028,
                230491131, 468504878, 3710487036, 3159611165, 3544126878, 1088644958, 2961497435,
                2785759884, 1537524580, 563225344, 2114093362, 4212602497, 1169631938, 2088081811,
                4006629680, 3089709789, 3749609771, 1969435408, 1786420402, 3870446123, 2169555554,
                1571013574, 506991779, 2174983408, 2376333115, 1813451470, 2875689985, 2056697043,
                2022446139, 828430986, 710475734, 785980495, 505758805, 3139780897, 3708680832,
                3390307357, 2434318195, 330093210, 3019701899, 3546042185, 202048370, 3017694172,
                3813711930, 1950710894, 2336832114, 3123649938, 3799964456, 1278675670, 1776925346,
                23381263, 909581672, 3084101661, 1592521095, 3095643203, 1245783006, 917301336,
                1175072489, 1493603974, 537178477, 3098462691, 848961674, 4083634813, 485866548,
                2460787176, 1957075662, 1653349530, 193311723, 1510554197, 615759127, 3054909658,
                3810118423, 3275067767, 2822189856, 1822027915, 641453111, 3902949794, 1707895715,
                187255999, 1547540130, 3996925138, 3744594623, 279929032, 2815355330, 1197018567,
                334914949, 104288985, 152451615, 2257137946, 495821725, 3891425071, 2698902656,
                4248123041, 3994796663, 2283493355, 1792145295, 771358160, 57983054, 2699162282,
                3252719646, 901108453, 21411013, 1214186627, 3971974103, 4284646962, 505661368,
                2014789373, 709655680, 4019528811, 3456428712, 3896941342, 25228952, 3267679573,
                2554048052, 1140488725, 796840032, 1008803984, 1769445882, 450164846, 183418978,
                1557823191, 2205206958, 2186287277, 4041341207, 2277585274, 2647704834, 3299210809,
                465486816, 81165601, 3688958209, 4134919427, 308497409, 1670228519, 1054621084,
                3997139209, 642331675, 877436795, 3750152836, 2099953927, 2407318768, 4064392686,
                3499776748, 2890558934, 4257002037, 409497686, 1871363314, 3488166608, 368834184,
                683374402, 3315975032, 3919319927, 1636872711, 3746724350, 411713899, 3127168268,
                1541472093, 481068370, 3914726958, 3809187727, 1019234471, 4257529799, 1795024406,
                2169201644, 180192691, 3146544995, 3086567728, 1371853848, 3442647104, 1956649571,
                3221872492, 2599010979, 3825880037, 129532942, 1962882180, 2981643769, 501519665,
                1104150124, 3577388513, 530338682, 2379351921, 476796974, 2079075205, 523122306,
                4175790511, 1769173929, 3684943051, 203952644, 2367960727, 2956929713, 724249999,
                3868724734, 3128867062, 788369620, 183457300, 4030548412, 320471199, 818557389,
                3673114423, 3427092352, 965641427, 4165737446, 546271097, 3179039741, 1968478116,
                233505213, 3523513681, 3185397073, 1639252860, 192368536, 2476919576, 1286359266,
                3468793964, 3927932569, 1554017778, 381977140, 2630912557, 3248408028, 1380148387,
                434027229, 3679247941, 2320186711, 4049616334, 1306803801, 3657216807, 4072237788,
                2409653033, 16652557, 777239076, 3435314631, 953899982, 375200832, 3240441496,
                1403201300, 3463532889, 2152357282, 1492290956, 1756116611, 2979347831, 157726282,
                597994889, 3571510881, 4022595441, 3689069225, 3371053456, 1105664287, 381184864,
                3760869170, 2128986335, 4138730626, 2108903255, 3330167716, 1193420433, 4081108869,
                2371248791, 2008259868, 329286806, 834934063, 1587339743, 3835392552, 3027794970,
                2946227510, 2759445311, 4153403869, 3246082301, 2169502676, 4274409702, 1618245891,
                3538641406, 2440237498, 291928949, 1309678695, 2248858018, 1259269551, 3062553916,
                2375747923, 929738114, 3593316299, 395934754, 1240422334, 403599738, 3596058407,
                4199195184, 708535732, 1045565478, 2985960024, 2935035441, 1902361158, 1360356647,
                655818324, 1464367881, 1402852252, 1605910196, 1553177831, 2270937291, 3867336542,
                1242299751, 4201495317, 1894193944, 2952284338, 2983597634, 3331327699, 2688007694,
                1534358324, 151501193, 874435351, 113820939, 3053243228, 2461547215, 3493411866,
                2931307581, 243458129, 2419713971, 2259098050, 4153522826, 1347753364, 789286836,
                3723853308, 282228699, 31083181, 212832201, 3992535406, 3841973318, 3150180785,
                966530634, 1983757001, 4224102508, 4103842551, 2315916292, 797697662, 915301841,
                727577536, 3413485655, 917582931, 843507358, 1529131982, 1843003612, 3002449886,
                346610535, 991891575, 2565267970, 3912371508, 2710618386, 2330075864, 2735484155,
                2546043717, 3241769509, 2210455214, 4169254026, 1482724431, 674379398, 565771066,
                3142137141, 515638418, 939683105, 1345730468, 2404207571, 249108973, 1883806778,
                2156914196, 1564711040, 3370156804, 1857590348, 1147780810, 3733908955, 2181248732,
                1904993207, 3694175594, 619779923, 3491824168, 3612713452, 1216174279, 663457690,
                4014509322, 2569761593, 14109664, 1828397087, 2936522368, 4158885908, 657033012,
                3173212007, 472151655, 684251460, 4075235611, 4224354174, 1004777987, 505536516,
                1454022092, 824505586, 3067613896, 2235424889, 3665322087, 3481517352, 2363451123,
                2771040650, 3235122422, 1154036889, 969497501, 2563704447, 3797108, 1395020021,
                2762840789, 1908946824, 3014468569, 925225402, 1138335619, 3624023401, 14380823,
                2839985590, 1786726385, 2046596657, 862236402, 214661254, 79111383, 1583211853,
                1641485501, 2710635102, 4088715922, 3860998541, 1323605849, 3597942622, 1491675000,
                1599602967, 3388494990, 3354632479, 2528582795, 215618636, 394082738, 1141528681,
                1784788922, 1221028471, 3234711669, 904205099, 1676230442, 3127792899, 3994795553,
                2452526892, 2165805680, 1335373003, 192721187, 4240993835, 3133423197, 1689272558,
                3673814847, 1422539041, 3736049673, 1833170900, 282944559, 2338828666, 2779222702,
                3327509347, 103394172, 3158363803, 1906876457, 2941521396, 4107834947, 3417951529,
                880661309, 2072508295, 589009441, 1335393037, 4277398556, 2493403024, 3409655003,
                3109850219, 1180552996, 2381399690, 2298638016, 3501746890, 846617313, 2905524779,
                2707401285, 2041915730, 2296396459, 1041902038, 889847207, 1989421094, 1389388870,
                3827587250, 1783862700, 3828138938, 1868614698, 4248902541, 3581264817, 3916285777,
                1776264454, 2214767964, 2937276417, 1736659895, 1395637227, 280854206, 226975266,
                3550562380, 1121092319, 159004679, 3748222278, 1260252989, 1422903228, 3336410666,
                194417341, 2723594163, 2281242077, 1784282179, 1034680840, 1402111826, 335654645,
                294940873, 2853511062, 378984905, 2532157416, 2195182123, 3873081897, 1625154299,
                260395831, 3540310196, 4273320806, 2622305394, 2711543735, 200100618, 3414809217,
                2926348222, 1329786033, 4245332557, 3574371092, 2777917577, 321179615, 3642895588,
                1496048120, 797710555, 3671936109, 2438718323, 45015662, 2137354414, 874208938,
                3450439142, 1916685905, 1662667234, 2856738964, 3829688032, 3404462996, 2848035045,
                2061762938, 1260861712, 3010572066, 3394836279, 2342878342, 1271875691, 4111464444,
                2126598368, 2380694046, 2430271490, 1145499017, 3787409979, 1555528697, 3376084869,
                642452482, 2589187231, 1081315770, 2087941360, 3364823763, 3805619618, 486395332,
                908395706, 2499268457, 2420556587, 474512813, 2481646657, 3492741661, 1968018988,
                1074530387, 2014914730, 2803826225, 3220982875, 1438277839, 2775829138, 1528403642,
                3794191286, 2608342366, 410785526, 2637600256, 3490750019, 1440381245, 814635701,
                2260916755, 2650730282, 415890751, 2524080685, 3474570208, 2446217936, 2397550701,
                231181743, 736834280, 1383997656, 1496783958, 623278112, 1645711091, 2659144562,
                2936379758, 424018844, 1549465529, 2669736270, 1664977313, 1523334854, 193915346,
                917134660, 1861003559, 2211424692, 3594457673, 521847126, 2663400187, 3341032570,
                1640978768, 3374727890, 2984430165, 3295971473, 3727310438, 4148801904, 2931771022,
                3471915299, 3774018978, 243266066, 719450229, 786371556, 1967721517, 3662692002,
                2660462613, 3406551440, 3689369626, 4170908863, 927580754, 1492537107, 1444056133,
                934562830, 964303678, 1533941876, 4122966823, 3705199737, 1112924448, 95199848,
                343531614, 594658489, 808416147, 2905432520, 3584653211, 1387698320, 3110254075,
                179755886, 2585492770, 826193502, 633023598, 1166009098, 1290016012, 672935203,
                442560997, 2496163987, 4194310358, 522918013, 4222433924, 1620670288, 1584642952,
                1110116518, 2050444989, 3738181405, 2449666200, 1322361471, 3346873860, 1704204055,
                2765918523, 1681401174, 1734783149, 2990922980, 2845873797, 2655480068, 1134013632,
                627992748, 2305459149, 890878648, 3702690433, 2308583593, 1647897272, 3079544696,
                2470471677, 4040208261, 2073585273, 1793034905, 1713771355, 2220715251, 2553773388,
                1442482611, 3113497415, 2768408881, 1270025121, 1399831313, 1630862433, 3377364946,
                1442504714, 2789927137, 3191151633, 2578873407, 208337033, 3096561372, 3943415949,
                3118056636, 1664092654, 533371474, 1132049460, 1378502069, 1205939688, 2646775436,
                873517579, 3647947118, 4249353240, 1234899426, 3562671303, 1028908207, 3806236229,
                2688126461, 2379248861, 4273316716, 1028554767, 960050373, 34458263, 2497415615,
                2000699114, 1654490516, 3970420809, 430138864, 2839090700, 2992085491, 2048751357,
                747829112, 2102051019, 2747342438, 3939874657, 204736053, 132477025, 2895769009,
                4049016784, 4006488678, 3010059929, 3869487365, 820665998, 3637576575, 1400083196,
                3176270933, 1580718861, 1862589245, 3687231820, 2811111046, 288642712, 2708675068,
                3659920550, 1043267703, 2675211709, 2471783225, 3908671780, 1796097076, 3884725302,
                2618455344, 2727437605, 3198512809, 77833978, 4002905580, 2475905855, 4285041054,
                1379496519, 2810710199, 3524329171, 2422823286, 3888601537, 1921960588, 4141779429,
                3945205304, 2680621131, 4186120628, 1952951538, 2875169441, 3303243339, 574383361,
                2010030917, 3924461786, 1497240891, 3972138842, 3082719894, 700821923, 1225799274,
                39941891, 1579154501, 2895091775, 2026419054, 3180814760, 1239600240, 3443816247,
                4103641786, 1778116375, 1356356349, 3003002432, 2464906412, 3106084532, 620250446,
                2199567717, 4285388064, 1443224417, 1183702872, 2361871288, 2889920918, 151923059,
                3665604400, 902272748, 3673929087, 777413599, 2880183228, 3116820884, 12649513,
                2151951398, 2517689255, 2603024997, 1540902312, 413276528, 463295145, 1270795006,
                3197387059, 1235198896, 1591251569, 1536841283, 183104831, 1099570929, 4157586543,
                4245916264, 1187513801, 2725120513, 2263045835, 3616676335, 1441932591, 859970322,
                1785738074, 2632201495, 3074325275, 3739260875, 3210655551, 3115242275, 2776972168,
                1857654859, 3388697322, 816121986, 4034608581, 3645370625, 3901309336, 3655082618,
                694485749, 2189293828, 4003306605, 2104718709, 2248125819, 1242466666, 3425122185,
                2526773969, 1997783773, 3366295660, 3638946293, 236310604, 3074020533, 1544508523,
                2720105666, 4275312048, 2125511485, 2928099726, 3115904574, 1659470574, 2302631502,
                2782437446, 3351759933, 3997936888, 2966997408, 3158966556, 2819300721, 1647111112,
                3003598038, 2858959466, 2179510155, 2584211049, 2202151208, 3064305858, 898246753,
                1503685985, 3011437597, 3645693595, 2240988222, 3440343576, 4238868402, 3504605984,
                693969911, 37266154, 53849195, 240646457, 1676768678, 3078948456, 353299888,
                3398599422, 1225876435, 3474721352, 2919885255, 1645273187, 1329251097, 63136291,
                3744051145, 3578975497, 743346836, 2992151082, 2097992510, 2029339826, 1064760489,
                4287623009, 690583934, 2137261889, 2523731672, 2989051237, 3080671706, 891097198,
                705467787, 3974619780, 2507988307, 3580098689, 2900468355, 2065761093, 1093422169,
                3699292714, 3679511724, 1376356943, 190833907, 1757023356, 2392875706, 1341718208,
                3312756053, 1335629721, 3842952633, 1585023162, 3483717351, 1166983906, 3797224124,
                2704194560, 3663191348, 1943345575, 2900370013, 98538425, 2337184175, 665720157,
                3985681094, 1514818198,
            ],
        );
        test_only_verify(&[10; 100], &[123; 20000], &[123; 100]);
    }
    #[cfg(not(feature = "32_bit_limbs"))]
    {
        test(
            &[10; 17],
            &[
                9995257893114397114,
                9401504468144459131,
                558615837638945228,
                10733662027974786928,
                18295107704289976677,
                1814706268673753787,
                12474943759854623335,
                8814778832826774208,
                9159057654048965906,
                4451260977376821357,
                18241701617364042056,
                6169989192350218482,
                15071965537117101028,
                13509168527678537782,
                12224278653171635329,
                16077066393714953826,
                1433938684868066489,
                13014970036232570373,
                899282336249563956,
                3089487642230339536,
                3787737519477527148,
                16667686214395942740,
                8787122953224574943,
                7841835218775877827,
                9693303502025838409,
                16122224776459879427,
                144327425397945219,
            ],
            &[
                2350654041004706911,
                7834348511584604247,
                12756796070221345724,
                3842923787777653903,
                12373799197090248752,
                9712029403347085570,
                1426676505264168302,
                10586232903332693517,
                8387833601131974459,
                6290888746273553243,
                9503969704425173615,
            ],
            &[
                12688955427180652274,
                7641660693922643933,
                8789985477567049482,
                5698832637416200787,
                14684840547760545685,
                2822100467869581421,
                3557573565928866957,
                4409631974409684922,
                16994214656621423610,
                4513108841166793667,
                9009005527785483287,
                10,
                10,
                10,
                10,
                10,
                10,
            ],
        );
        test(
            &[10; 23],
            &[
                748159237152854524,
                14199895651244313572,
                9044210482484213648,
                3880401870711113518,
                1694971440240542063,
                13547801197479934494,
                5244069077418598572,
                17329479401291658084,
                12613311850003558282,
                5618071535926791206,
                16954511293879569524,
                8600749590433482901,
                11708546551548237376,
                10879843710159659952,
                9101678715417935644,
                12126242459863584426,
                17259866272884195621,
                4418382641453775715,
                542305129955142216,
                6563442437678466173,
                12794875758080454756,
                7461769876910639905,
                17925257245127463276,
                5137728719899113924,
                12905981752247605071,
            ],
            &[
                2654882163556630563,
                2047318842992691178,
                17944530594807555614,
                17278864523505748498,
                1160166728089482341,
                18368953657130322418,
                3937719995815345698,
                12007028340444721520,
                1496744539933999053,
                1476923054783110845,
                6551619938265612084,
                16801911333947266527,
                13986495313155597995,
                6571595571877061463,
                10140569634762389822,
                16210530410764331582,
                15172903143228403872,
                5831780706385794192,
                12288937301416472500,
                16224579586702000460,
                14545605105156691376,
                8614987803254853144,
                16629891239728134900,
            ],
            &[
                18140975738986113396,
                16765596268029991308,
                14497740378349400824,
                8834432760455669008,
                2081502095596466916,
                16785570606386467383,
                5299348241512211807,
                17503170383548190207,
                16775442261989831354,
                8131705923782084593,
                266320274487676679,
                6602256474512308593,
                2102043233085822823,
                11614561527212258722,
                17915538208051341722,
                5710195504177465517,
                2094480568485157388,
                14339014023087152780,
                6947889352398323832,
                10985139413433625547,
                12373170520775701923,
                9198039438688117621,
                15475638737141339650,
            ],
        );
        test(
            &[10; 56],
            &[
                14660214196707223375,
                14265972253040120215,
                15506320303100465818,
                17085621003033826581,
                11203337550022453944,
                15493204961705835371,
                5803021083410871755,
                8112917457002746745,
                12663484193891261040,
                1721048899893287199,
                8062187621610464306,
                13431761655884620090,
                7331427712144411262,
                3626934647030185267,
                13231383914073320042,
                11637171044660683638,
                15189928975258171045,
                941827519265124224,
                2992792486091076914,
                2044203374633195985,
                8310380355675814732,
                1677894573715118386,
                1863631713396879617,
                13750903464355877990,
                13561054993991137710,
                6643134394212488277,
                9782189322903525535,
                7987880548748269544,
                17396502810230452231,
                9355336424066456608,
                6974435047841500624,
                4695995454788932008,
                9790410161672155866,
                7324176676989916049,
                14873447357313289350,
                17933513319573948354,
                16221633809094225356,
                1119296061370324791,
                13659405622992751643,
                10536448431317839371,
                15771892335411705715,
                6450515195565208913,
                12583173873673842188,
                8943105588740166659,
                16781237121411387206,
                7355272525679995848,
                8924936502454129260,
                9464007023044637842,
                2392086820925613645,
                6952992660961663836,
                15709161892606831425,
                15961199354349516091,
                8170938350051511007,
                10106337242460916657,
                4519632767875399815,
                13966478644099829332,
                18146666299243951179,
                18001892575388798951,
                17442461326088111501,
                12996149925790510613,
                15125238000270787220,
                13458137050174539117,
                7565676737178758148,
                7820895745333505106,
                18391820881894926862,
                17227107494212736312,
                16170524482788524562,
                18292226432698054709,
                16409124153431213414,
                2622798522164114141,
                2030148142272451724,
                12631034221630749586,
                12521714531249855181,
                4869764655816857917,
                18312880399388298885,
                1881841240505020002,
                16686085102712131293,
                1638984612454565124,
                5980766772519196081,
                14473546029553426533,
                2610255570241349719,
                4121823778233332328,
                15196027812344512481,
                17634932614139407184,
                14566629132274047837,
                6629067916649366603,
                39453246491293667,
                4118307938296638515,
                176389639877922730,
                2385844666265721927,
                14424300909552701177,
                2596064544694255252,
                9262830285738421829,
                8366979142044016136,
                12451088247268499723,
                16456341544263224076,
                405434591376297036,
                5989071471671786526,
                17922319711997177283,
                12402685985480014221,
                11440567647536028583,
                17109382986734751589,
                1165111999013207871,
                9042409351611763515,
                335396288523389342,
                6889397323074150916,
                13998858741906849976,
                15927944587197048898,
                10995067153735213576,
                13255077995174337515,
                11985913648073551062,
                16606199253171990948,
                16615211378568935152,
                13000672060735124358,
            ],
            &[
                6726150808576237754,
                9590776370558469124,
                4613857594775205869,
                5605914158178321857,
                12627075307783464761,
                456502911636413728,
                6201419543988208076,
                12457367465345491402,
                9194484469177303126,
                14469237774454463326,
                8872571916644400618,
                10371861714649740250,
                9551882050917532587,
                1418647961867356190,
                11742587182398063873,
                11015016132415914044,
                8777839015232205587,
                11080046461630228193,
                13740325869131645472,
                17716201322003396844,
                2184375889136968144,
                2744007897878529583,
                10107840174031679018,
                6807210551800087042,
                3927845063936277496,
                4657264236265855475,
                18202437017170404187,
                5332422779150911238,
                15515262280249200267,
                248667350560422394,
                3473467338029486524,
                5450666559053310869,
                9114347711968955703,
                1001965327187909086,
                9391480248060184246,
                9069754537718985217,
                6108113375902101471,
                615335597740998377,
                7341924484422171664,
                7557688311245960406,
                10629369615492290302,
                6551022068682485711,
                13009629572214277263,
                9801266711191462998,
                12475469715378400041,
                16817728089246511388,
                5318131496704799888,
                14034696640350324685,
                173195053797772988,
                9465580662794117123,
                9395502290798332505,
                172507413604644051,
                13462235362634225088,
                9267822876689174860,
                12978933587961252639,
            ],
            &[
                16979197013852036393,
                4534519222829727882,
                5127955051936920534,
                5669732551578654322,
                13787946500638697314,
                2666880029397285003,
                18286001525339884787,
                3747928243980886079,
                5670276194023029484,
                15201258611907138387,
                6046915833599742673,
                13282924752646783062,
                18026143804639821221,
                10186643213552896189,
                17209309200088910354,
                13215180252119768256,
                1246399679408038126,
                4186715523775575401,
                16756959752065842207,
                6600048850655585015,
                4543693866439677976,
                15594233518271892275,
                15247811862837572166,
                6322126320582019533,
                649809830609098083,
                5229876751712742127,
                17719948521867410031,
                10737539927122287433,
                12476905306147178753,
                1539850235988803702,
                13572545877865905325,
                11163694899331373883,
                7882148214994127637,
                8164419266634080608,
                5782587821804107698,
                12155391719814216620,
                8020222143449740150,
                8489927257914490530,
                15688922762526028920,
                207673185831465902,
                13825819490340731785,
                14207999229863934400,
                10163751595898713958,
                17777080404153962435,
                17016927136773389232,
                3820023214020965653,
                1892439588667561762,
                16909683715900311832,
                11919385779232783009,
                11201007117990222527,
                8700983269916503928,
                5034192113764375450,
                12790439085134048151,
                17790018876931315900,
                5953092655978688336,
                10,
            ],
        );
        test(
            &[0; 1031],
            &[
                18285600144381611747,
                2607658718107379546,
                9747604544742966856,
                2960103437306950033,
                11333576199308712527,
                14238003202284088532,
                16265332193148604979,
                1379942304307770309,
                16938783330207677640,
                2745252988447228002,
                6791234087903673829,
                12518174636027607009,
                15674280970395865818,
                13974288954585950791,
                14841204964098561443,
                8748517132169388864,
                17525897240098709564,
                16349048229068678114,
                17968982428715890523,
                9358013674764648602,
                5199229606169954727,
                422943872008603371,
                9250194632016078797,
                10021555560097390790,
                2684614891063859627,
                7440654004485895845,
                15497536988658560429,
                16091759101488274182,
                7624403524401510649,
                10159556756468762769,
                2567573898157529790,
                15609657766585643253,
                16236943306986434793,
                5890548198293774422,
                17362921261219419202,
                2808858681174890249,
                6640440299319480962,
                4326748503634915406,
                11456055721778818238,
                10707187402450482341,
                10505948868316563545,
                18366980076276187881,
                15068095720303300745,
                4952650647951470154,
                15911634980226720120,
                10740745397838202379,
                3200127162032008025,
                3763543940730164508,
                3723483274937556495,
                17871783614248259146,
                8786059543177366813,
                13050056601003380055,
                5623281846050325544,
                8511114651096961385,
                10891975825673952044,
                16080052858418260073,
                7171941023859325850,
                2503165103241590389,
                11967819128596062687,
                17746426492658731021,
                1524291256148953421,
                2076038155471824980,
                5880985691677435840,
                4713693495653829540,
                8577899432290161344,
                11976146392187565938,
                2529575305527670284,
                3765314673643814808,
                7234028569335439232,
                12375682646064399341,
                7101569992574162906,
                11098657045492285566,
                1153652218824818047,
                1210454437960036069,
                2694638310421669121,
                7870059579873173861,
                3508868899875540240,
                13609981818610730741,
                10703439073683625892,
                13071420953159015459,
                13716502475013538938,
                10634244764110852465,
                16403497596959564382,
                15306415780193612609,
                17601327182445704540,
                18298349615021078405,
                18111916165749794999,
                7537294468737253688,
                7386987937618099419,
                10775482015504178918,
                8525379645714838775,
                8788609183023289620,
                895207535126054757,
                17118106622782691546,
                16749800484842219686,
                10933310771357651927,
                17045784651422893806,
                10055875188754575183,
                9519498699655590213,
                17322601174638910336,
                12118730165352641482,
                5930728976613565641,
                17159697904528121051,
                15309680603600886131,
                2354852814839706361,
                1866864355164982520,
                14799678913936204761,
                114516480202556771,
                13374320542339641716,
                17948849241297689882,
                11937303202723535389,
                8708253685264852524,
                14784022410383824926,
                15741150074911773529,
                4751508515383369506,
                6753819133127024063,
                7615256392835158829,
                14330334546462075477,
                5916715905113385520,
                16699087896061936717,
                2161501163497237132,
                7657705954734728178,
                13842299391403386537,
                6172027960597539813,
                5307601271781038790,
                4555363624168980660,
                8897561972602250232,
                492135721000809790,
                196374542645899516,
                14990889098212601995,
                6381940097804056460,
                7088456247368733765,
                174918913838309902,
                14001140535865588809,
                16643163063827173806,
                1238954875765713073,
                13202778059857945377,
                13892311264186089515,
                13771970066908683469,
                14995846051430041367,
                17421145547571170005,
                10821698378431695436,
                5316647583697276386,
                14051654090444941132,
                18388943018780729150,
                11259779711269878259,
                8742581111664031470,
                8889827116461862655,
                17319531559654594882,
                6500455467003400955,
                3959184730122082747,
                4054561166041887982,
                17316966724931034861,
                1607293446195331113,
                1556357414629043861,
                11011377910730701009,
                13217502480646592162,
                5556764467305459566,
                12021589498137324260,
                531690648537094954,
                1038223775874752655,
                10112707642670009067,
                16909364180557997068,
                13549292338140530580,
                14089674983696767092,
                14960833416043686252,
                3361732906192010803,
                15411952217752187419,
                475547270098839964,
                8227352829235176532,
                4333410121135356569,
                16974375358381043398,
                7462799035629267008,
                3961814110661375506,
                3191195952723002366,
                15267463856820101363,
                2682345599906528119,
                9202835087842166762,
                5753494002443879006,
                13496926465554347490,
                15537050413782275199,
                4005699994847427717,
                13011783168070790177,
                16932430119108925312,
                10469912374234875027,
                5996081763274312546,
                13299322992254047535,
                13566444816251120959,
                13441242515242491906,
                6531470165275421029,
                6998803913624387519,
                5190923639408924125,
                1301326112306077751,
                2724835656803633652,
                14868270067404848561,
                16428689823791581153,
                6363562758622431773,
                5795421246045082577,
                13738077193279334339,
                17818084837848061577,
                7409215264364848571,
                2893214122000373991,
                4066741278513513286,
                6208913273098798518,
                5323770029909018782,
                5551337708190405618,
                10281953507639521630,
                7674048265312145606,
                13913805095945202464,
                11645578449852204207,
                5216827728522108969,
                18149089235876418482,
                4193193659542496963,
                4384037638695340997,
                6264405355262516799,
                5923041840648053673,
                18445416708501282570,
                1449616427668184913,
                17814282379016188485,
                11394546105942760801,
                514272123860845956,
                5487867953647208460,
                9765982880054791980,
                15269612196572547812,
                6933672683776753268,
                10897766962490536850,
                7963503740124897799,
                1249067770622312125,
                8521634516253063564,
                15047118858576393194,
                3052485574284313450,
                15762145598668916146,
                18313105704770544279,
                6541634279146063106,
                3775400747866844137,
                16496810669859720010,
                8379319521731425820,
                10800415684660017782,
                6142541123129235279,
                8861768666560539302,
                15721050291326133652,
                11134786092605171040,
                13017548541999335364,
                16909509841464459084,
                8484654924270783553,
                15005512331960977142,
                4074672441814183738,
                4078543398182843194,
                6773998236248908357,
                631977413504183184,
                9491750397437496755,
                3377999220825657249,
                3346615552296940898,
                17604589720878522247,
                5769086022977878966,
                17347842858220401404,
                11931418629108149400,
                14490915772427782374,
                11545152969051431084,
                18194530049847403452,
                14802318193842666424,
                16217304261948074309,
                1412773250833563909,
                7479345161017490691,
                14305628603682480019,
                14047111540339531390,
                2202019933845371318,
                213291111871207451,
                9537101822050445095,
                360483935799398585,
                15821799940301882413,
                13587044781648265473,
                11158232757574497759,
                12493764616797894572,
                12454593638118113923,
                16175783796857016626,
                11972175061436709392,
                14628387199635430458,
                5253379621963585234,
                6809274079261887574,
                1124840670828961539,
                5990829238947221675,
                16084193379498378212,
                6869638953004510310,
                17906409028525617365,
                2739791577758847487,
                480131078003075884,
                16023370915391221306,
                15817682603522356534,
                12423449398808644199,
                14130320721960901031,
                7934273914237433979,
                786249691033407308,
                318299827682676802,
                17097143809787660310,
                17521193394806881361,
                8643211685837497345,
                1204030043754228035,
                2676537878172957478,
                1780844249865247048,
                1857038044392024228,
                15537174113724189402,
                15216184620024948614,
                18295257315773355758,
                5049495368122551869,
                511689711728309335,
                16940513174832634299,
                12104504465679211266,
                15914220570778735059,
                6279446781938212962,
                601029894044123455,
                17804281207827746870,
                433540216199633732,
                7944481812184613751,
                3853777775611712418,
                12126936213601232312,
                7065913778676807607,
                43238211455364752,
                8192260579196809431,
                8266744677261407642,
                4334793912650253838,
                12956232517663155469,
                5369404211871099568,
                3235414488456415006,
                12567595135095181283,
                13364037675200086203,
                2815554933029326376,
                6185051196142844265,
                16989212187178652008,
                3878956585408701458,
                708958585074504373,
                16920922562467588927,
                18249699025766906652,
                18027337540317897651,
                2618966708286475266,
                1113776870378384593,
                5988471205905783011,
                11449408349082147321,
                6427498821108845882,
                8757628250113942992,
                3713195263318788838,
                601232850676293890,
                9498794059653193118,
                6534298968814526870,
                12486527138072517297,
                15783607638066166551,
                17817123635789116996,
                1248377855370621035,
                11141540237607695650,
                11310946791190236066,
                12846390650128150341,
                15186900836698527641,
                3867690890119912186,
                4589172752928224161,
                18406516299291763072,
                11413251654981660385,
                16024801440525265600,
                4278405775100889986,
                8755603474239908996,
                3160069975842943114,
                9906744402561181341,
                9965035381218832722,
                6752543667701180453,
                9346639648731189767,
                10661232594877011901,
                14936909082854570803,
                1969452932415658459,
                1696145335740060250,
                18164995195644723626,
                5001142711409868759,
                502680569449519136,
                3022117952710893422,
                10177903082752941255,
                15652975217635236869,
                2966845407346038740,
                475611270044318476,
                16066077773226714888,
                15255919358962415817,
                11732084421089027684,
                4677384405854396649,
                4610350216539332701,
                10328878907789481841,
                6555568935797623159,
                3017832515483377819,
                14707077320201566872,
                12590581217649731730,
                3161603588075729925,
                2563665003397527163,
                5979170469664323613,
                5626012705821413010,
                14527466196441398570,
                9288633927607394750,
                14475257195709120279,
                11865264300717431518,
                17136126069177251144,
                1473429666850111981,
                7510246272136643144,
                18310164183004169194,
                3593138027022744772,
                6350576143506118040,
                12773591294872180021,
                4700465661075210908,
                18143542497507479548,
                7719757932210686713,
                4338121025681527358,
                15571485830471842974,
                16406722845577898376,
                5721314859539898709,
                6589287198671761501,
                9745964795756887415,
                10659452091068591333,
                6690465060747950210,
                728586915321808599,
                8500603684008829799,
                6301121250492508617,
                1984503548524816268,
                18268190305907234300,
                7454276134026583117,
                8547492859943863047,
                9499474055416732782,
                300834181303701699,
                6423420221300563233,
                10906772834067949181,
                12139236772746843296,
                15479617127388528892,
                7255634393137870745,
                422813312578980582,
                1148428423488055078,
                2405372466822606988,
                18202859592780650711,
                15123472874738759666,
                2160560009262204586,
                1220679802754985663,
                8755041458565252857,
                10511683773218681884,
                3102123453859609904,
                2170460789136594842,
                10540612896965819281,
                11507452270722296921,
                3888033478974547053,
                2913172211662715309,
                5819754293645132724,
                5417423241393498910,
                1121133064615112577,
                8899735394209007976,
                17939515994389118968,
                3057735335191827376,
                6167592388365551096,
                7859032161115089000,
                16392857948131842701,
                9672772119042623246,
                6861392074297507574,
                10079830057029967718,
                1610021695530123734,
                11499290474395739628,
                18357287834621605730,
                7842767805813604560,
                17595414094496571456,
                7727166516961623663,
                5489874195960679366,
                16978517381214022058,
                11807176188692933608,
                15778362166265505627,
                12826269652469979971,
                7799517220815997172,
                6941343467007860979,
                14716681029145080878,
                5685272761645196256,
                13841850638623453280,
                18117357664247936176,
                11360524534267913074,
                18408319184374782489,
                7834872787814949745,
                16527576874764684677,
                16576008924894848236,
                12304125447826256834,
                18301056937365686021,
                4783517752793787963,
                2014695633627143026,
                8288284640512774782,
                14976342552024779031,
                2436344641520555003,
                13990286257760881060,
                18118120610252892651,
                8432102189510702685,
                9013208900764457155,
                15956895100219663594,
                11965405775699735022,
                11479704069694999185,
                1315304162999565265,
                7619720903122481473,
                10798255112249544769,
                9067939483463584201,
                8242338589636256000,
                11595456341434427053,
                14057071507676286987,
                13492895734414905617,
                8167664880787891961,
                13065353170057145433,
                1390434145929843432,
                15097573448811012551,
                10002618775785350116,
                13037941058513405166,
                10812662834979031696,
                7926490047107184245,
                5399553758270644040,
                15603376937288512903,
                3817798784005377844,
                12622533130989118638,
                12217900933744511855,
                11495811053435066455,
                17088770095627295990,
                6282760152901843934,
                11339571359698956951,
                10010250182369906159,
                11956984034096448616,
                230656938835165573,
                155123462325204273,
                16323688331673735090,
                12405389233556129322,
                16866268093283320126,
                8076215192615100819,
                17788391733691877517,
                13738263070347903116,
                17514698963264490076,
                14361153193344045623,
                10697613194968260318,
                398797896589607036,
                11354436663860273814,
                3988804407101251650,
                11718246560466989132,
                10607955713165345856,
                12293428353299936422,
                2774193885283164233,
                12111880818145942696,
                5269109320979566091,
                17700804638146972952,
                13483325589659749714,
                7018249448187819229,
                16605502462498173206,
                14334331196281817568,
                17997264741331536966,
                16942067749941194095,
                18326105716816079857,
                2873713554755956975,
                89972251875212547,
                11011206003344840167,
                16263607631557095555,
                3956845690962774124,
                15843733851407790844,
                4532327333478987933,
                14523922448792187179,
                671512032483507444,
                9051952980308527779,
                5771887136664598504,
                7318486125138347367,
                2056863287621362849,
                5319340499967729915,
                6771692072764865726,
                4993497900077789612,
                14846578576184294087,
                8749976795074478239,
                13944161778625889141,
                13939659589211557709,
                8674498121103871446,
                13858955528517010167,
                6854968788299690525,
                9785571842093558996,
                6616581398547260916,
                11173134838307567854,
                15020736287331849950,
                213678875278769463,
                8136043007787109258,
                963815884865140222,
                14383997603758711218,
                2441395136452256841,
                2285295004115032555,
                14143394511135418137,
                15622142144369079201,
                16493347221813991820,
                17474381401274268608,
                936058552355041089,
                12942246561895670820,
                10317584384347004208,
                11782536454405046773,
                12350945172113506786,
                7391214981076111463,
                7073023377646396731,
                4011509139909188128,
                2257833537102580311,
                7274600296374037369,
                2769150843750451970,
                9102426399850639067,
                8214223108585585549,
                11637757222275081002,
                9169116734789273638,
                13195915003973211388,
                5475411329499291917,
                5086118280372242975,
                17718728186337638844,
                16105088227769019766,
                6209815346219807551,
                2881314910093736440,
                12490322405428340882,
                12004395323760866113,
                9968698365927837252,
                17316938449682856319,
                15625883316417529399,
                12226632472013770838,
                14161693092908388449,
                17778030898846793856,
                13976693691407275877,
                3477953777282076567,
                13740936608896808297,
                17180555612721269478,
                13416907629329031924,
                5869124347540464645,
                13384341358033251916,
                14313718444333031397,
                3376219002404246289,
                339811206717616611,
                2616093060711965335,
                8204344482414301835,
                9186789611379643507,
                14338727533336579685,
                12784860191396059227,
                10489777731949698480,
                14621154301243573986,
                13692422118348399108,
                3784610479458294357,
                7735714461152647872,
                12931998015666009092,
                12887123618479150667,
                4005389619579094200,
                12949896351841026182,
                4274952326880371758,
                2252542616091249178,
                11223191016227664360,
                12276921256190460942,
                9311103011248164424,
                14618190821987444804,
                16794211813689377361,
                17066759525543500866,
                7695930135720339816,
                15636206560014700744,
                7080595571863441999,
                16002741097850788650,
                7266069110373243930,
                18266417640341330095,
                12863061635472560818,
                2804143153090216476,
                5360410260257436881,
                10259005860191386447,
                3129239898744374569,
                5704118451966324874,
                2617683975038617110,
                1419008983250810424,
                18176995508939903574,
                3428325565738056333,
                15807295691769067887,
                17098475885649833094,
                435386771621069983,
                3749240361804507548,
                3484935049619675621,
                14151734623085313892,
                7107401439542109862,
                9140252282652202082,
                17139755393802157327,
                1840890360164731160,
                13945742659422648167,
                2993927710006218010,
                5123393630446195909,
                12126601606897174993,
                3757237441333108331,
                13430475145736272456,
                141286591810884028,
                13716994059020442018,
                16256272733681835404,
                13021661888272553737,
                10938682475722641190,
                16670594791610413108,
                7581768967654565510,
                7258449528404661741,
                16972464178815325271,
                13103916238623071625,
                14383222643604888683,
                9134614536502072501,
                4484749605592970811,
                7617523826265603571,
                286970167644433897,
                17350844883213337166,
                9372418001867268906,
                11036524012081184976,
                14599471605351296864,
                14353873970303354160,
                13804851709719425521,
                12542207662034387249,
                5455404902951618435,
                18208534021141654431,
                10760289922882806669,
                10594303509090562894,
                10523612649331418075,
                10175165805141093468,
                3845925331470243881,
                9968712325133293994,
                12632170237051093440,
                14407500129795275129,
                7039838041956792694,
                8486437057635491253,
                12990896208114377714,
                2169038646525679305,
                17731701498975515823,
                12645000352324144895,
                6940383608615873690,
                17112601588170442293,
                9898015000882658910,
                14088035355956248952,
                507013420654526970,
                16055894587901325067,
                17646639921892423762,
                9750921155184562446,
                854792204257329267,
                1907596925043325411,
                13186585073232938657,
                17324756184357468589,
                13215141704644982246,
                9990033614756242368,
                10703741257062175210,
                17316690423805264994,
                10087888281338300084,
                1089239266776577821,
                12176439490507890322,
                12695856964012192215,
                11884436036930363286,
                16729977556989227363,
                10923433806234038450,
                12992592892604117864,
                1268144030942582108,
                10300209358150898566,
                7832544414491392865,
                6657397193574515418,
                17331856360490769436,
                6685863352121534775,
                4872607912521754323,
                13925034245904306548,
                15794800294630432773,
                4150702655832044776,
                1922931668614047821,
                12163445688451808604,
                6814128619400384333,
                1631984743731389500,
                3778862171994919347,
                15454466802087055901,
                9264991358295251267,
                9936918682375797148,
                18157798989566244240,
                5581635369737696387,
                4479378944085838523,
                1634001250673190201,
                9492710507275012642,
                9234172590874940829,
                8783494515876021547,
                10628896245581316049,
                9175567334027325902,
                18216027702703135409,
                11634672812751315604,
                5707660447334077643,
                10202156378293455124,
                13892983828551272913,
                9908193203293924452,
                17571530454157193212,
                15072879288037447844,
                127116006100922087,
                11771185510908260613,
                2445074823174851909,
                10670868476525820983,
                7425728342067716996,
                2985971528087667631,
                7395252277468977294,
                2997333798387301672,
                17435969755936090886,
                2284549800776567351,
                1549552636061955968,
                2562232096971944509,
                18175473291905577073,
                2694680735549112623,
                1791226765063856042,
                6639691018975209776,
                14450608736185529781,
                776581885259523327,
                481545710834071552,
                17904221072369375896,
                7658678246624118033,
                14858939803632465766,
                16638621759785444057,
                7852705140401778814,
                15504288742303235325,
                16123480377693163548,
                17580674642691781753,
                2060059788952429927,
                8044488423463356775,
                12159915697800522747,
                9979462535316088472,
                11464540989399692426,
                9723421225194612320,
                6449580914984939699,
                9406550367550615810,
                5311403373873226147,
                3066770575413693918,
                633416401733094941,
                11712977725629062725,
                4883265930195070560,
                16189146570612520152,
                1698905934763835706,
                950927367170890858,
                8829376333788190736,
                7907286046365739950,
                3164878452294289455,
                10511755255842705560,
                2924155429164172446,
                16124299828159551950,
                6010099555943764977,
                12724759932455911822,
                8127212736140639283,
                10128696763620279450,
                18035084640302368027,
                8781918799260800755,
                14441294204200284647,
                1927193541320801341,
                17060847707989835700,
                8195209044950545498,
                4420757359342589946,
                3727277256040667922,
                385534260624886499,
                3266025280223616765,
                8307518494652840749,
                10628505726158916146,
                12481632216062021685,
                8443059664409740255,
                17083655987996225144,
                730490563149619104,
                16115455384269067104,
                2382469211629823819,
                15299473012699513736,
                2545726390341274697,
                9566636451267734313,
                17404798143522728759,
                12853265630229752228,
                2172148070178466503,
                11003520300346660378,
                1134402141777052949,
                11771658227786580781,
                12088790715698048033,
                17088312113295353834,
                9858438140481896937,
                4732279313243969474,
                9302577460868509370,
                10413475674288905591,
                13021872760990946904,
                9638439145954399186,
                2342341335564065825,
                18026701375952328694,
                5788988813536004381,
                6112690485750689530,
                12277817288452946380,
                7006316162927315897,
                12091459534025657274,
                2152473017451714429,
                7117646493299604201,
                1591263581497951670,
                12905706257967317715,
                9739330975167177950,
                14245120246238342654,
                5445126088677322792,
                8475451974915724213,
                16532703318231814798,
                11346158079313546034,
                2587775018559680831,
                16042335036595321112,
                2783584919636232962,
                9477074304137314302,
                6679970828349102077,
                14914134423793746235,
                8835496380779847657,
                13649036745393349622,
                11800533244055495841,
                10329924636422821002,
                12857463828194064326,
                376873602561854006,
                13991526512239117933,
                4244930150917248420,
                9377193637737929560,
                13470983451570137106,
                13709193869228195237,
                873834737899882109,
                17298475531540644799,
                5157431525573101837,
                15694890039292357223,
                7263973093257735703,
                12810611197004081294,
                17159179774131359627,
                16320123751204284184,
                6659167957772819702,
                3559977382115542403,
                12297726256906381994,
                10545494770323931829,
                14423760345131502963,
                11951650428815262297,
                3552160721881779344,
                3921553500551057972,
                8827773853996085356,
                13035185276040540279,
                2431229196115555640,
                8867095302754162942,
                5824531700307273486,
                17795693614810306555,
                5327408766240893952,
                16204495625665346776,
                4520750388436708886,
                16011790401268552431,
                13186223796877430470,
                5894651091707364204,
                7349710328825984565,
                12990495590567591633,
                15789659485048909512,
                8110395238510363562,
                7157879492847706094,
                1855521204456327951,
                9387090323123631218,
                6983791028671987424,
                17264360937175821593,
                8964335962673307125,
                4438686713691513133,
                4609268036048721815,
                13573223706586323329,
                4370839728814507274,
                7190667190262397984,
                10585250032143309718,
                5262324401338557970,
                18436298550586124762,
                615281048933821162,
                7892081066718287534,
                4153026337684348988,
                1998277297931299322,
                9570951836837442866,
                4214601792245837204,
                12544321637650748092,
                9861635514343412559,
                5163007832201827055,
                8357508302373244198,
                14941521234529596324,
                3025149840692592346,
                1565659965409485275,
                8113415648323406877,
                18047137088865383715,
                4761751436090139639,
                410742697284005702,
                8123281234742203725,
                2300465198396824513,
                7573276443955885792,
                18354607580020177721,
                13826724636992006395,
                8759796250561912860,
                15446866439938431144,
                6466830870344794620,
                7248618897690538684,
                10968843377874574640,
                4538534083565844065,
                10463149345079834509,
                7862094636026465434,
                9363380870281847317,
                17750231516234915849,
                10605066658889662478,
                8651524573226517322,
                18099624785064356749,
                9012260115980472797,
                5112736631921184699,
                16433387801586396924,
                11364983784749187252,
                4592720300056375272,
                17400478621152089229,
                9960304372235946193,
                6810543488590694603,
                9421463587360187007,
                11110051932845796606,
                15820198248392525417,
                17370225890188484838,
                5511348958827419764,
                2047354351196049067,
                8842128025677787612,
                2546794526658883557,
                9990966093285021784,
                10968573027006701235,
                8153649946073344163,
                3435896373516278367,
                8854351008479838087,
                15172945879518770400,
                16101272223167010392,
                6080119325930559085,
                18336117387797080742,
                11495477657407490192,
                13799378353686928991,
                814519040394412554,
                4055424121543431351,
                17971320561372651,
                3045424347018335501,
                9144749175919467258,
                17693139582687085722,
                15549618811805620741,
                10042978303090035766,
                11250820672049577391,
                14650606162160854310,
                681053092156505443,
                1467784211081816986,
                14292070206822113207,
                3356550259250142987,
                8897529911380186072,
                15774124171111764637,
                3608035896189707114,
                14945841364780812283,
                13638982384303383036,
                5615554819198447325,
                6604755778043315386,
                13154764512915240378,
                15781224930202608619,
                5254282127938948208,
                15314350610960881010,
                127785624547347482,
                553562291991422824,
                2665385793464509946,
                11960374868185251273,
                11044390920977985650,
                4524189753866435423,
                11922516208188195244,
                7322728958220573958,
                4566122595518047958,
                9257927878541886507,
                11778087706650674305,
                12675707724087743775,
                14040780876086782366,
                10310608200410344735,
                14747492163773576994,
                14763717547881637874,
                4519119406094329744,
                5416745750548195790,
                8698677371585009606,
                10540691781438295688,
                7621321253979898000,
                12870309033904607776,
                8766530226422390333,
                14410571192410377800,
                5017391940288233504,
                8156621488957733235,
                5701635921482105032,
                8697544584237000467,
                14463840701596743299,
                1132393248669674482,
                17783802461705335925,
                3379973927717106230,
                170233915427480249,
                7721198001501959996,
                5984888435057942981,
                11741252369856674270,
                2667584213579254971,
                12391311788904160680,
                6403858222659082204,
                1706835705335237525,
                6951651160442697621,
                3395001754770416725,
                16336009550495654242,
                1588611969581977597,
                3369833199693136814,
                16679631221398536370,
                17143410383401198876,
                10718378066292328788,
                5098802788047763943,
                5003995899412543356,
                7797753734199884695,
                9804992578690571817,
                852561252664521182,
                676543942240459737,
                12047623372899199507,
                12688884395525848756,
                2909502726897081899,
                6031120528123066491,
                10187664230696218588,
                1634984456433807424,
                13461263282968829173,
                562535253767547543,
                3262317265760851658,
                10159933570831396212,
                9610081805557490182,
                10645180788472548796,
                3825134762043109512,
                11454131642474959474,
                9401675919629678678,
                14450914272139752267,
                7874801920832590192,
                18054728540681521894,
                15303403280502540543,
                15467520708986418749,
                7705766485597949742,
                204013254901132813,
                13967733294005655167,
                4001477583895779972,
                5972807400755250985,
                1217989087590450380,
                10862887725324735466,
                1706534929358469389,
                13264418588578493513,
                10626892394654962607,
                2832066031146486501,
                6506140673085843042,
                14326905332890415966,
                4447185119806812807,
                10289677945299645724,
                4459612687423833871,
                10867968914888905042,
                6485974059834599556,
                14142727197114410913,
                8587964753565667799,
                8972765350531032618,
                2292328948393940314,
                935062418987300570,
                6736600688398044784,
                7931198222933701097,
                12299736294122528410,
                14482610418638535345,
                14434918807907278467,
                2439305975663118437,
                4480608229958318339,
                3952066067287563861,
                14258451016047506475,
                1231074371015334625,
                10331248234059926878,
                748201218435577445,
                17462982229802307244,
                15686869687340151573,
                17053063210448785862,
                3599854718778131864,
                9301737618251045483,
                7627446349617553731,
                12835054936982245416,
                6336754664191040283,
                5870029311999289016,
                6027766907695007223,
                4082089270941640253,
                10932978191544782698,
                858542788191451824,
                18173438258887704458,
                15950175591359254382,
                10496438189070905854,
                3761196574957125185,
                8070795822731288097,
                14333885717356195593,
                12907535393235068762,
                1300746091836895017,
                16095775835363872226,
                5221409894943186677,
                10801943559441670680,
                2226518879270450694,
                14144518871296969095,
                8329328663632421754,
                14151247548153947290,
                7005388744793033852,
                3731783672032466610,
                5642923046890143070,
                2243566518435928649,
                17827722312011386647,
                3288494091174335807,
                12802408479052688767,
                3941775622555625432,
                13974527790491109973,
                7184506709498460272,
                11905130726430415767,
                12898188848602614942,
                3336329541205892765,
                16740350675338109671,
                335449286526469870,
                16670399530115701446,
                3370759555155341636,
                13153054802454960430,
                17020347152181004363,
                16005180766456295251,
                16035747705790056677,
                1904525470716808389,
                1362939978621142721,
                14268989953945718594,
                3131759537359596393,
                12623877369046640758,
                16034480904634718634,
                7205849095551422163,
                18352760423874563428,
                9294184574071126998,
                4855176006968938932,
                13740659884417148386,
                9678792914697707031,
                17346691148499799327,
                3661296281536466662,
                1858171995289042067,
                18199380542019467146,
                18435270579529426778,
                8792805062170870691,
                2216926146372155391,
                3336698928747087340,
                15361162057883195491,
                2230334966886787987,
                6790919905652555693,
                9523654540872868833,
                2373039885555601800,
                4493652711092482273,
                4848735304745253118,
                4615650132005261688,
                7011332091469251441,
                15051588471309214410,
                16436499359483898767,
                5534529923924019676,
                8031782014972324826,
                9587715117105518067,
                5201232572863714631,
                6949484852216655045,
                11242297859309211887,
                5445327566402416511,
                11759527029723480062,
                7401655792007047741,
                12981437329323060492,
                16627917972301844941,
                16086064209851186236,
                18317605428363897776,
                5479987106722795675,
                470044962552825995,
                3263508437049006276,
                8532199872517472836,
                179963713893634086,
                14750710828805195780,
                8566890372208465530,
                457434420254160615,
                9026428119711082888,
                5051697286542220522,
                4066845298260571291,
                15266943848838025449,
                7790231715639606132,
                18047039423184779192,
                7097787949459808218,
                12326346119319424869,
                14895758855340221400,
                270818421368673258,
                10953991395430365501,
                10722834322274726220,
                7175276538034253286,
                12293270138776886211,
                10868230614957475384,
                15568806031450505970,
                2173332627995000239,
                14888402241228040414,
                18405500854031799916,
                9396217995786697609,
                8638079772902395024,
                8700301862255582170,
                16669714149683986916,
                11791960995856983997,
                6941956874526614041,
                471913500417437253,
                12645657659237731696,
                7344954555441957686,
                8576764504255171238,
                17045229481365431866,
                10710908163870404781,
                11301570355963434173,
                5971177578976838887,
                115032958122130333,
                16996195561179864691,
                5303610406571283411,
                1557215249726500571,
                429079935597742432,
                6007572072647948719,
                8793841786368287656,
                209772504980789061,
                12514491936429933001,
                7014555801251009174,
                5854792368971550756,
                867919854085761466,
                11854155159921686895,
                13142074624110571417,
                7907850875814485970,
                16861742288872511551,
                1208590626489658408,
                8601279929820923503,
                2258924690462246171,
                9585184984627008623,
                5315828210383196353,
                1297805663260559339,
                7398320505542428444,
                17073967214624996598,
                17704120650146094638,
                1299385581680024947,
                1443489762904405882,
                13518221354231615324,
                9070126881206870042,
                993730884724745294,
                14081041013869681380,
                8359943552596347499,
                6866696380166200245,
                3837542845849508667,
                14602652064433499913,
                14314340020057352466,
                8839559535694012406,
                14227230979046046062,
                10298228782846904508,
                13298718693240900028,
                13909155817605919274,
                8780642200132615642,
                9412091595152829592,
                6822145839018577251,
                8785978531052338151,
                4948528066049804851,
                1172628369060862431,
                5666685907911162463,
                14153629140204265152,
                10500845530057771412,
                14608765724027887761,
                5087095504941752721,
                8701919577673940506,
                17137520717957261114,
                9199637801040704714,
                7095886304415555902,
                15925336741311965253,
                13077337771490692636,
                145710441006655783,
                13157185292334340173,
                13407008567503772150,
                14215960022966892561,
                11985156240556559472,
                17945288516894513884,
                14097688271959790972,
                15505832877271523166,
                5843174730859038007,
                16205337090179264073,
                17606308108912062454,
                669419080191010051,
                15851467430769620452,
                4936909049694472990,
                9426661750545745060,
                15134074240235574428,
                7927257239677744604,
                13865931916371651548,
                9055694221969554578,
                18105938911030359772,
                17344312477772739728,
                7984780979580024449,
                17946659128143746086,
                3659014353178730517,
                10881946920507249755,
                9643240829875041416,
                9979006042478131099,
                3624673912637648606,
                10183097162206305022,
                14705929380474025749,
                8560872261129163553,
                8542571307034562924,
                2849336599501641328,
                1427173016960915063,
                10561422500956348533,
                13953646915815013961,
                14711009426065697276,
                17629582482698947067,
                5068672373293663589,
                10328236305608973133,
                11390625625852214740,
                16171676010227409920,
                12007752012156277746,
                8787339600696104751,
                17147558071877953304,
                6243146155416198819,
                16781815705425825152,
                9276534213532622927,
                5681999116735330218,
                16852010554258505062,
                1705868855435043353,
                14818213196970587370,
                9256091199810743680,
                10438037318676504268,
                16794099631419865392,
                9815261792343983998,
                8286783569926236889,
                337573434858179240,
                16820096306284770618,
                15966431763494632363,
                6481208570789422743,
                8429344246220730373,
                27262454150324780,
                9516508385964067765,
                12876008564654096078,
                4071754764426708210,
                11593726207040321843,
                10464906996437266409,
                17795281436982488931,
                7949644313068982658,
                7004988298640898067,
                12037768710549869791,
                16790409316897618369,
                7084103797769437957,
                2419445775321185382,
                8074279743518467644,
                8069323749479463736,
                17160974715554385442,
                6446198417065372686,
                909199233418090118,
                17988046405451941411,
                15619199660230837101,
                8540771809211736979,
                15004059727603767665,
                9970067037279458294,
                18037810830226270966,
                7395784361237814976,
                441608911830412969,
                15193617525949907726,
                8329750585903900376,
                9232838481221040709,
                13862722871242295122,
                2651785001097390772,
                13513885689469842635,
                2916635381190975954,
                14673366675350976320,
                16796162449704211211,
                15311238965453782301,
                17048079530325153354,
                6357896971279650981,
                6912501188214061635,
                15767706566209693421,
                4950547784710662682,
                6818275469676220067,
                6311416403137682772,
                14192827820659451230,
                5519753761822532883,
                6327790738344616319,
                1571492847028444517,
                519836968001291628,
                8880684714326015791,
                4414274058554177288,
                6693694082558848552,
                9180328506312734588,
                11851754793034588133,
                17892308745832908148,
                12122989628255788470,
                9813496966285232956,
                10955347908495147392,
                7570827623140202413,
                12207236341183232786,
                4777412212063025006,
                12385252236023114447,
                9508166476039999651,
                114906234504385867,
                9207258060511755437,
                9210800729712345963,
                5771423404639548192,
                15194614200877751860,
                4338352328776016735,
                2610599495858282641,
                16733979028371628176,
                1929910361255473211,
                14137591116693762330,
                11276030669834631098,
                8384908437274376663,
                9932945040858512861,
                6671057894832572720,
                15600691762738444868,
                1090424207156031951,
                7633234269493162481,
                1452672955493924651,
                8042177265524791320,
                5918487395429243286,
                6071477929355428400,
                8446553879294519239,
                17238305490194970122,
                8987047329207310489,
                12398791106166157316,
                15174911221193868718,
                16946345338721827201,
                10949948458544444544,
                17318691300002579588,
                3998415180681624527,
                12750967780890975278,
                288347833998723232,
                16095212211245790724,
                7822321122528030405,
                1109063169170634828,
                4652215944166612902,
                16803163450585461189,
                2776655336762420387,
                15642000619067002463,
                2018031858324511467,
                11237756472829950832,
                17565143038179107620,
                16281738469740252935,
                17600775725533342946,
                14750124410220606835,
                12883103971220877489,
                15936542280450981255,
                9714323865324229767,
                1469778854158587075,
                9593652974682074282,
                11762461414294133530,
                1404229929827068795,
                16503900992928085410,
                14684251465993475323,
                17706137844161464268,
                3909560365123413776,
                9861484314350641439,
                12009083324415037686,
                18169123171153377641,
                2393555352820887974,
                12841835092045423984,
                5564097118032313065,
                12777929502410633133,
                5078336225337553091,
                8694020844593690446,
                11388328638584884970,
                13970084235020262491,
                829772823417380360,
                8286179729255644084,
                957139202750627952,
                3221879212666198953,
                7645002751366998370,
                16715354970243034507,
                18304845819423800855,
                9447343833119941047,
                15163436489241423867,
                2479924668890227186,
                7818020209349584296,
                10580682200967646169,
                13911404593504958709,
                8718275660345998055,
                10124290490841409455,
                7651303722187917624,
                8976889705380896709,
                11163660337280624646,
                10388784795451265724,
                3541771203499561964,
                6075152751154669243,
                2218667547472367527,
                3995516111323252367,
                11180655997089000245,
                5073383660790586917,
                13458152992621454755,
                11676714551890338118,
                1494799466123513971,
                7327872328474238339,
                2389581909384645325,
                16687753253842922782,
                11699615151482669193,
                14774804959571915475,
                947896829667145566,
                10573954710522391072,
                7072571908216975157,
                1921454675444217366,
                11232731567954977537,
                14840397076140631292,
                5060980672779376677,
                12975040048834693618,
                10752224255692434704,
                1306885046650589222,
                1930876755170305305,
                8739276995254053408,
                17062175909187112484,
                13994219820475631601,
                15241051299228638903,
                13189674117437271785,
                3224946313788614445,
                2946953495584212148,
                12515092889993340415,
                8549243107224308830,
                6555967774604787675,
                17074340742411084655,
                15860252206376208136,
                2452901849982359121,
                1217176756850333549,
                15260442969744536089,
                15266619117940483626,
                3092040621234773226,
                8592622982489602359,
                17268283100249965729,
                9118863795573669565,
                5004589504450812591,
                6062182260926951061,
                17997691517876082652,
                4684918577412859882,
                15422706458772273904,
                10373853167444581181,
                9008368442714488639,
                3624691764843327336,
                1028359463583846893,
                859230892908817598,
                4558109131236939269,
                9197636145007824268,
                3397702897203024845,
                17645318682095346025,
                297912191657103235,
                16707811002284900618,
                9544713268645707384,
                18174588433767285273,
                14115044574398526220,
                9550382630286578375,
                5669098498675234593,
                1419240760313025600,
                2624357144105750680,
                1883325833631791743,
                12465653732071455581,
                14016850785680848114,
                10853862374929837476,
                9213307118918963573,
                17446748283231651532,
                6944359057113621208,
                4376214699313790455,
                4272171302826636053,
                1361571807732899930,
                15928311517924806859,
                10979193193168197302,
                14315748858106823250,
                11415251230794226377,
                14341688199410222112,
                13372316746789815573,
                12205797915700991132,
                12054245058919264469,
                7420566807537884143,
                5432588773028217373,
                6350587654471565537,
                6531829368743229731,
                2207976572300046135,
                12703622166413274712,
                18356491416550058865,
                11743861089192075323,
                9975481076082250423,
                5270519543247843824,
                16596875614345327407,
                264970638689993857,
                12731851329939038802,
                4100374548344342900,
                11696626303514932259,
                5795347109604371486,
                4018536166291403802,
                11605839014492166891,
                4046122976220313619,
                7173051512574373511,
                10760222355399775007,
                2397452537849365173,
                7000689557970901078,
                3508218483424754056,
                11249130935335670874,
                15485017380487364008,
                14838383309738250533,
                1654265944033471910,
                6550408986795246586,
                11574017534658799615,
                4938545842279886871,
                6987608227293293739,
                8490763818451756454,
                15553940026252549950,
                6498239556873819401,
                8894315085699368208,
                1767452481216158212,
                17391068041642225358,
                12099770647562592260,
                3232275359468560052,
                9875417733360604587,
                7344425724555447442,
                18431729052975636742,
                2951164042491135904,
                15903903429564563858,
                17834292681718510486,
                4198460729209826867,
                18283343945753908147,
                13114879247072451198,
                6308546439134988873,
                8448793227573510403,
                12883523838880467178,
                12608354838733343984,
                12284418193790956340,
                3099459343498815410,
                3160460005758859474,
                17209423043124247155,
                10299954565801063517,
                17312198705527288316,
                1159079074290265282,
                13803764368072178660,
                12357928111570535493,
                9319226950413258817,
                7861389264222959398,
                4208357822974886032,
                4724453996000826506,
                9219506857747054473,
                16650755835313196333,
                4602357483273387357,
                8421668150799174361,
                12464268335381289971,
                16842331782835636444,
                2164847798539863920,
                11848181286892607238,
                8483204622866252940,
                5177013341061819166,
                6794893916803855304,
                830729528874930746,
                10838023688403983614,
                2936119661394195602,
                16749442537019148931,
                6282125361565649956,
                12522652324110588825,
                706946363036865451,
                287377191870510960,
                4079038197618045672,
                65600267963984525,
                15852196146124192732,
                7791496333170096323,
                1197307315361468886,
                12564376588501612052,
                18287064409980818748,
                16513413097113895910,
                13772586176619885924,
                17412579692431697526,
                9943582674770693553,
                18017460763820483721,
                12774668415999817484,
                17419661422487800791,
                8542296650831811427,
                11030794367359113173,
                3091992835617551759,
                6647390664467556203,
                4760797422569339325,
                15821430759187595540,
                15905919720868415262,
                6964495470980486330,
                11481098497210815485,
                13606157196874073055,
                2545548939497776096,
                17492549168216383449,
                976625893298873591,
                3307888858281262315,
                4406331424039826265,
                7251492960589745703,
                1462803411594068207,
                14615947077492017745,
                14477853980032711958,
                6580067844089943084,
                8164600499091902269,
                4672103301095852237,
                355042099720565535,
                11026766258596173656,
                9849960052517135155,
                16725670149953992537,
                15683028676578831515,
                10104782290810509683,
                4905343590849641899,
                17726532108833531580,
                13910286230069122723,
                9139857611408342006,
                4873918044036171351,
                5211230260928194081,
                4862798578450111352,
                8040863435679951765,
                9157211979753234216,
                5969001182224545740,
                14666007377730777715,
                746412171062253704,
                5788296420214704699,
                16052313287704740168,
                15016980204491105152,
                128634126176999080,
                12474550947728802418,
                17798334047543705551,
                16546317352521543345,
                5935130659172645630,
                9017751516052971809,
                8734332868677069231,
                4489180421214586024,
                10852153455030219222,
                9452853792027507,
                7518566350390737179,
                226398539584556263,
                11348207575144575782,
                11286381051277848898,
                2699324198517442237,
                5176288375717881953,
                7814733078102607208,
                2870574509136744908,
                13870395192961519685,
                16224938596653908092,
                1164830155402073602,
                6289706313355675934,
                10093838410108560175,
                2551691589709651401,
                15882696485463451559,
                3395731646324122865,
                12473003610421288231,
                7149619960361615882,
                9872807622306918213,
                6663012576266477470,
                17016549792515115205,
                7914178572800158023,
                17014701567637435932,
                14346020259284370257,
                5074365976986728463,
                4760168726990175515,
                9895948792206852207,
                18273255159788978236,
                15744547484672365588,
                16864050091650583468,
                7152138044891906219,
                3133231435713995056,
                14654289506261133725,
                14216037513719362689,
                6883938067921952514,
                13637751749205284541,
                3930907102025842836,
                7818503413066881742,
                16512268473175446131,
                13618638131220285482,
                3526560852689406895,
                16317348279030286281,
                5781706287563855208,
                16877706843053345782,
                9787389520151320440,
                1672625892740702444,
                17775094144895143457,
                7284734471708236532,
                6265912061217026214,
                16428427284586036469,
                7239300285490611080,
                12257099262814629130,
                1443098110450745157,
                7111333716613874437,
                10912463389630114261,
                16046472632120791101,
                11071308475967108461,
                15827031925467657247,
                8451701545861351835,
                2032718007638091987,
                9275305708809424137,
                10133532607349132521,
                62031173551766505,
                1069668934694816298,
                10021315903394494037,
                18062320514241791111,
                16614273374620642998,
                3874528406029009574,
                11609554407744278526,
                6664429329059042685,
                2466789265733227250,
                13791862987730476433,
                3100076428479694646,
                1940573727807547478,
                1717992312877296193,
                3858278897688455120,
                2708391244967786582,
                5897077897033357390,
                13903090124093303714,
                5210172753029188249,
                7461814148277300830,
                9825991953137124992,
                8749849231144845954,
                15733619176610014569,
                2536045915051793710,
                2795322555877650589,
                15164137179922407098,
                11311210084140344802,
                10571285191092929225,
                4809833504711226607,
                15053596169163492600,
                12405664553095020547,
                16471715640263755395,
                3828351876877550724,
                8499888452131579133,
                5279727535149011805,
                788643833745941437,
                11012285726544455006,
                15140257364221864448,
                1834285009459528474,
                6667182475201903438,
                15294024395589453460,
                10526482467856757788,
                11877982762793807927,
                16230316864465358123,
                15723393603594181628,
                18424513935220466565,
                2247141586434891619,
                9561372576131838874,
                8725104655684175177,
                4143398848385382705,
                15142179196637056843,
                17330386941741492571,
                8185673125837710067,
                14939737481973173962,
                10478395787666924945,
                2774689508274990026,
                12451742284278860222,
                10819437460867617255,
                13731567691766219490,
                2010973683387239919,
                9908150371329956596,
                5142510757399393995,
                8739166923706997151,
                16903649040425907246,
                6887419514989237100,
                1875697363324020800,
                16410802797521308346,
                135862499096678312,
                1002710565280860416,
                5506348106905501784,
                11368287034717703998,
                12947266586015982813,
                5318254909926585664,
                9296468205284792595,
                8004814724160061752,
                366393468717204226,
                10790734687437783398,
                4968864942084278227,
                15798662312565535644,
                1358717464422009379,
                12888686466434430386,
                370520483338984073,
                9184642588343372758,
                9427179068402613262,
                10546786545869440465,
                12655417731444306355,
                9241957630409402290,
                821497951821470195,
                3197970711092278947,
                2184424336239199737,
                5106022623290447939,
                16287048031456496603,
                14037056338746814782,
                7474000199931218562,
                12188148167023924203,
                4192028392289466906,
                7847291769851591899,
                3599561900723047299,
                851922034760320751,
                13587138821847499752,
                15960408996225570897,
                3252001517847791314,
                794174927197992513,
                13500786923616416327,
                10492520307372562282,
                17778963725733766896,
                9940193524688369200,
                7839429243680619309,
                11152685327758330011,
                4793514629906514769,
                12571317405831554973,
                14236764890442489777,
                14285226682975878252,
                4752465601541667238,
                7069073862761309134,
                18297621617638524964,
                16921089529724869008,
                15310629395363439387,
                13959904965987734388,
                5530325767118220975,
                955257971161171732,
                9249687876412175486,
                6437014646360143974,
                1355675591008680076,
                14363814817404705228,
                11444699168208672882,
                16890344077647521215,
                13663194688247169944,
                9311758702620596945,
                9111333905244008262,
                3739308534752587737,
                10082399017494306243,
                14502104680453962352,
                3240447658814963588,
                1945616413338215240,
                12702436196323839746,
                15056103164643567233,
                14232906369316952845,
                8850081479512611348,
                78742167026295742,
                10346207199790592003,
                8330318603106182342,
                7555172829812303483,
                9135554534047252825,
                2213978715965424041,
                14487841485121669529,
                14201073224315526620,
                12002576656710628646,
                9216161229854384642,
                15287364979646883311,
                4707577895128497224,
                14025232799632084648,
                5317706395334588893,
                1489525665074354154,
                9157439364142092486,
                7033808256892914411,
                2856904746309353752,
                11315255790030198921,
                12167475608624586287,
                14898366696798509538,
                9918128939648387398,
                15155651994814693417,
                7300960472494546400,
                2127526231515960912,
                8923406206829661118,
                16210393114988425815,
                6582813395371196199,
                15621762854392338472,
                3789550044424910854,
                16988798023618902953,
                16490593537106458722,
                14139545158028230044,
                9923635478455782946,
                8897217280317716469,
                8453814653444266828,
                7491325054712040563,
                12316700761490127631,
                3768457975900229126,
                3000037930881673503,
                16933057766750635435,
                15031174928418481583,
                3019987484089942716,
                18385609936539305759,
                17538768527945166806,
                5447221007766299602,
                4022798439534764827,
                17074805847175217989,
                9477194110020170032,
                10200259846152838876,
                6455055232471034376,
                5679758327656142352,
                7374147007380920949,
                10200477410022066193,
                2678287199491553725,
                5011120062607229052,
                7214257460239674372,
                9631753064650306646,
                1151983673963114572,
                2455508156322019163,
                13344796472161923955,
                17003695385004214505,
                11166307372098954131,
                2221946328882472137,
                7061489614039858629,
                3790971501091116413,
                7833733804721228481,
                193935756315502298,
                10256338066976304411,
                6065749049589877362,
                10736472690437134411,
                10155365416269683464,
                8189737882773858983,
                13671211115390707327,
                9578025336797949473,
                4003423028251739147,
                4500512304316856151,
                16693811644887033093,
                7707767344560596984,
                5599541458797859603,
                11592325835943024402,
                13355669588132489385,
                15932001697488084536,
                9870357379952328535,
                18310499762741838752,
                16196774945614723862,
                15790487345250753382,
                11990590655367464148,
                6718674416650039299,
                14857329469228715563,
                15423479737409206383,
                17779878703537243863,
                2210183662909405344,
                2815540909030329829,
                5940765014986549489,
                16884893974961992656,
                4351638805125702042,
                17084353252120736247,
                5724252085645496824,
                17794301429798973952,
                7949129492433028333,
                17290229503720706399,
                12839841549866653659,
                14863383260888850679,
                15788917858243172461,
                12477975711782044218,
                12362210923589911101,
                12864920132785914749,
                10860162458674426789,
                15634366676531407135,
                6076026769874036504,
                12186624835981223643,
                477838969646790588,
                10209086843687255723,
                4453597694185557465,
                9792638779490749472,
                8664773568287178159,
                18322711946104316656,
                5061590004040013161,
                11611936838758532137,
                3299611430381151917,
                3978890638406515286,
                5081581149863744934,
                6181936955807580784,
                3372188069238850634,
                17854265102071578827,
                1268076978660794505,
                4722477313014336378,
                5471980643922086499,
                16788648796682094709,
                13328529160256523927,
                17816434316065398701,
                4038311782651024619,
                8255869964978262613,
                8890037824421975027,
                7319879894513022697,
                15914698902244750946,
                8857389029776818687,
                441863541258666465,
                3208037884158506723,
                6113734693574039276,
                264986606662698668,
                1166237501799711319,
                1823307447602740420,
                5025672756710206999,
                18278689422657324853,
                4594507497487291210,
                1280886041213346013,
                11896598892872950706,
                5576693076164380614,
                10907097907235031727,
                317196941106545936,
                10598515286008372146,
                10232665246931238263,
                16641631582624438065,
                1996845757059763068,
                8450189738658666487,
                16874433980334624325,
                11462456373743157203,
                17922058749640783486,
                10312126789368172997,
                13340565145612845891,
                2211332034685909397,
                6280821632118062402,
                11495261020831147497,
                7525751008510944548,
                16481618138878161375,
                17687291635686050126,
                4726039693402439590,
                6224312974372200845,
                177270674822437819,
                12844674102136119940,
                2765095317735756740,
                6375785787962152345,
                4765877337254522185,
                5699318866607950771,
                7726266014961936643,
                10439969907866586583,
                11215604933445164846,
                11144874084048847665,
                16677507722859793772,
                14926605406115980653,
                11021418173114133993,
                18143508384673173911,
                17869384978881128193,
                539841771736061948,
                813906843235746774,
                2120916504386756315,
                17563694758070096975,
                5807011249822640473,
                12074412278826937750,
                10494691749052118094,
                4034710777981084065,
                12945928140391853828,
                13460531663356368909,
                9805968628270501084,
                8389235258735557083,
                4653641631812286778,
                8961436573446073348,
                6695344158613046716,
                16832827001219019801,
                5227374831316662089,
                5088308882683339576,
                3279167959625059741,
                10436103497832123501,
                2797296687506317311,
                18305760283738580632,
                3531912382829235411,
                8745021258521548248,
                2062796519113317445,
                7005506890190360164,
                15260828944473012523,
                11736618288708934540,
                15523033936060000556,
                9071328583193116055,
                2949954518191771020,
                9219149478710608069,
                1788704922178497999,
                3917123331723493156,
                8283987826876682192,
                3926515941902592230,
                4576777655155350003,
                2895291773949819710,
                5409491793062473762,
                7827016058731337158,
                2360244091777092634,
                17765210336693396066,
                15413391993237156529,
                9542656815625824892,
                9430563399204843121,
                11559365450247449510,
                15796909749180552904,
                62820771556683371,
                17275496805851130274,
                6527142162623953967,
                4324187305635963626,
                8775772925418898915,
                4283973075245351015,
                8846184539722500106,
                2860502690824226817,
                1240652940592194,
                15064048316358894661,
                15864254034666279286,
                4098943802367114793,
                2956682521101031793,
                1877556925464957894,
                9589792900125026809,
                12161426439049578986,
                6435540582736037161,
                10575856639119978060,
                14751225739652781433,
                3642975302329400055,
                10213336036038833464,
                7137807512192531866,
                15715238576819145647,
                17891816262406767004,
                2885759842806217828,
                10001842985502100171,
                16889965688337087910,
                7192241521508241891,
                12459052609846245573,
                16045056395445218777,
                14263408894637697194,
                6319194484923808205,
                6887168986137924562,
                4723017923445345268,
                8785445940747788332,
                2221153194351157100,
                4112058037714640844,
                16624774310058492682,
                18155682498135649179,
                4932684590312312624,
                6466580926084050461,
                13095557293811313234,
                5629819270691955218,
                6705036397567572049,
                15096504303120814344,
                17813708390329054153,
                2867801270346878637,
                4428369643259246081,
                14356128800268363075,
                2105538175211522946,
                8157169805154179676,
                13281934124532911335,
                721993975496515367,
                9731311970803693633,
                17408383560695508306,
                3661032291356469843,
                17118601234529144108,
                14173384340046827967,
                13956269658151533336,
                14193288209519401944,
                10854924829501094491,
                9362722838007437994,
                13139075107441548779,
                53888779219483732,
                9991228977857148506,
                11866192603390104699,
                1903043806364214756,
                14664090553522048890,
                94655028324146279,
                5944941626254245313,
                3067815658183085252,
                10088835779690593635,
                12586879497500128294,
                17338285387265829132,
                13546679410630118150,
                4684155179104389445,
                17373141188216408963,
                10683275227724384825,
                11727464505463619204,
                16890013009763926299,
                12256558554118856945,
                1692369148468240661,
                14260594858617396870,
                8079067481737035693,
                17302479952759473929,
                3465288033646104738,
                7033532051975356541,
                5750806504528053243,
                16142598027708695045,
                3763071556050151375,
                747332911728793689,
                7125512446937882000,
                1711762381076986594,
                15200743445044769163,
                567070148436076824,
                11150986733649743405,
                1751542514779667937,
                14904426643112052682,
                16243812463903474525,
                4313436371286419450,
                18281797249258270726,
                13863976627696030069,
                12604750965802575212,
                10295616727116267978,
                7795942372543067322,
                6713137871239810667,
                13830865828728404463,
                11139257890131452244,
                2759969414377994154,
                151123411588221984,
                1548247161690787301,
                16234894685549118874,
                13392592243844300663,
                4341718471344294648,
                7137644906875460827,
                5416631993939117377,
                11693386548832817142,
                11851058952064231493,
                9690204938813699131,
                68528268163302809,
                13565043622050152230,
                7983220844142817842,
                6794808910151851013,
                2118666482985725107,
                7247597207626321306,
                16937693375274789158,
                10616884815388828659,
                11696015248644704721,
                3352478958667054947,
                3225871086863590754,
                180230908445925268,
                6654741577584823985,
                11710009687907650028,
                628457965752902770,
                1735690146666342885,
                16556237906505825388,
                12440271636242780312,
                8713058872068144699,
                8975441040195700002,
                10850963832660773043,
                86339822673122508,
                4000323666157343936,
                12142567722725546249,
                3374710818178624781,
                6392232680788016964,
                12790199390015157427,
                4671003891873822839,
                4452182512321524727,
                3905095889403963026,
                16726139916064776463,
                3453332349452466770,
                16221710379420353742,
                4632551944540970365,
                8387501246722919769,
                9326344266173346561,
                14810846036626389828,
                7283658938651037849,
                5813262143148998917,
                11727805907650385268,
                9857096480403873519,
                6356324418997451940,
                4218579215729792881,
                10470999063206203293,
                622196070040738252,
                17021144211023071215,
                272053703337182123,
                1450758744433386057,
                8397337301039096971,
                1756739264090010567,
                14804942979114893233,
                12762190712783408010,
                4486114271404551982,
                13538690353341899530,
                6182493449368831960,
                2768435790455158079,
                8491588540734758297,
                5001428683933385935,
                16861055012236061995,
                10578636081126268849,
                14449779720168257980,
                9370646703732609468,
                476896208068404545,
                11604128769892258429,
                3178840918267575500,
                3407583825966649895,
                17913378662961900390,
                1138652617523020638,
                4702124733398535376,
                16197108716716127538,
                16852598869081929632,
                5020614776073585872,
                1188708373797737364,
                13456434315549496452,
                3798071449732352219,
                6593816705053129140,
                8463318160472344568,
                10114730768365942765,
                2093191144313711189,
                12720945689550243678,
                17481264965133664645,
                17640221870440170657,
                3490097599238691289,
                3624992904734580452,
                5383687377060455481,
                10776924742527744182,
                15746021472289883888,
                17925392760128571522,
                12689005861958409137,
                5231100489367895272,
                14007286820825332338,
                2397858260918981403,
                3320067181759583430,
                18227252228223404573,
                18011328143837002368,
                4798104660333169602,
                10865877383638846865,
                3767167656736085179,
                13252620599128939387,
                6636747284560413010,
                13514887050922862847,
                422456370340431886,
                9612933469182009435,
                8572209183721381907,
                8056048067037472186,
                16714990476240297166,
                3520645146861432874,
                4395550491627332110,
                9462218612700999094,
                6214250154185857186,
                7628194292741759764,
                3556878717144745691,
                18413027616459454704,
                10160691680912469038,
                13717170760433051685,
                15180833335295580609,
                1819382907555618141,
                13320476638466170165,
                10422294181974936444,
                7164181547072169074,
                4206492057553506325,
                12633015044784588629,
                379171732810097117,
                10351100759422011738,
                3673330411049649463,
                15927757725693044281,
                13690250449236882125,
                15900551197649269941,
                18329662994600175312,
                2110669219011944740,
                2226752393294287183,
                1014550150137152480,
                9887871066385753226,
                4568683160401484222,
                10951431797661196524,
                2312645127290056695,
                16937439673591720324,
                6396609076318661377,
                8389300110486808887,
                1923677274274786580,
                3549825163974227141,
                8566399761898493308,
                639620030714191084,
                768228320555967661,
                8108418530085400912,
                10272182341228792152,
                16750158829727849854,
                4444117530817286053,
                586462195320680600,
                4278069062735193163,
                4273315131401359275,
                11611787428091079232,
                11835985289253260719,
                3096230712379885797,
                5322239580881209060,
                11938666223318366819,
                5060592382016062327,
                109409443906808448,
                2380457102117134635,
                15857731461784222805,
                6584665638956888535,
                5947254465908568523,
                9420786545084124002,
                4503761263873856534,
                6641486141146012370,
                9121700908113990684,
                9720958769231915561,
                7003782806520614311,
                16920307773004245532,
                12763580883987437513,
                13231043349317133789,
                16203109836791501802,
                698049430375939280,
                3030494131166195541,
                10815604140749178453,
                13804995432448225904,
                2055424018541357006,
                17073648888173819706,
                15463924428711597599,
                13403700103090470402,
                16763806955533746160,
                10509071477142331263,
                7405199613665848444,
                1289961083639614668,
                13019156453931533504,
                511380020193300140,
                15833618630547184883,
                16532434650886462678,
                7779349734049489490,
                6380411322746302134,
                9492908436973173105,
                7167088077189974813,
                17396059932024334263,
                16483611260630861934,
                9669468321319436238,
                12712646425834594093,
                9288405016973860570,
                11325843021373113165,
                11258335193384041947,
                1997545595799365359,
                17390606419645620788,
                5130387948734042681,
                3121742342628196576,
                6099609462914905352,
                6524840389448497409,
                3251588966299685263,
                12515453442966781968,
                4351131781828315306,
                7915089733711494046,
                8605641240710406093,
                11102717433954517200,
                8816805538627940082,
                17227753413606850602,
                10263075023716628113,
                7686631682618468388,
                10330530590800536028,
                2587341603576202527,
                11500396296312301825,
                13531587115648414633,
                15808917747546864365,
                14333112763098420691,
                14761783111500015307,
                53964954401492398,
                16533467491022296104,
                3096485428305684908,
                6318844018415466634,
                7132560604178571060,
                16428604738409136626,
                4375929108222206025,
                10112133284375140971,
                3256631261503314080,
                16136845689782815155,
                15928971771356595498,
                14645130502697393764,
                14481748170304871772,
                1698922547070724229,
                9468607121868930261,
                11617214057068823437,
                1382928294652688250,
                12680084220201263431,
                15926122141331194025,
                7979147799621842755,
                9803599832824191350,
                11150566171126503035,
                11946016594155356252,
                8452284422264862510,
                8621047342992105610,
                8991106617593209637,
                4607652189428377711,
                312850746210123282,
                13020253158865282886,
                9665527964173507749,
                18176337504665053651,
                10689513521305445515,
                1839607582702602917,
                13793618456770339778,
                12156395506174115903,
                15230345865671590927,
                4715973552531673115,
                14606516029379763286,
                17293771405588962821,
                16175698567845501145,
                3789965395433901756,
                7250432289837425960,
                2507460993020556336,
                16416011798948565038,
                6033312224978543324,
                14568892906298810894,
                9207153067892339499,
                14537938431481818790,
                16550516704016530385,
                9907670198667127499,
                11918889385815695319,
                5981318855234384164,
                6303359496211585544,
                897784439258088201,
                7533487805017595619,
                5938457201719040924,
                10019150195260257017,
                12026554931880038202,
                9224513724236448133,
                18208973061276109233,
                14240550756243930801,
                15387974810151313549,
                193781859214087276,
                11713954174101604255,
                1341093990140003013,
                6012491122871297010,
                13892530467037865965,
                1368649256526022373,
                7020819021722718671,
                15787587617733723291,
                7785870968748246057,
                1947448600658942561,
                16530328214840567615,
                6317941331086952570,
                7139842027589405175,
                10933153414206777090,
                9672586073744099620,
                17228494407277874159,
                11578033579087247602,
                15459493510831207600,
                8222705886823487604,
                578413881840029844,
                2972330100500591542,
                7515621234174823899,
                6704318956697481184,
                7539293268516643247,
                1580608997901534862,
                1201050878740265829,
                10170365701874480907,
                9870363666224664306,
                17305296184407747996,
                4224805619364289032,
                15085837243579345311,
                2443632602903912157,
                13046843267919167284,
                12116351035376725179,
                263232896451917215,
                7772071638110042980,
                16549817644785034122,
                15257628965480096285,
                13004189693733577363,
                3864285740392230962,
                3468607145721577697,
                354566527372793754,
                12925883576993237617,
                12693175375557728813,
                622461988842871628,
                11397527121006480165,
                12911631110359477732,
                13101149248297235307,
                3443924800339223639,
                11021054325079132422,
                5789359016899607332,
                15670032899436160802,
                17231336489372520758,
                15011195019101462587,
                17697169511730379360,
                6214385185627447169,
                10180011765464756731,
                11047248145840743090,
                2742832730133990272,
                14029491892088421031,
                7977004417662816272,
                14199429721808423163,
                18279997024327708476,
                13359704350594785266,
                18410294947752215697,
                4900459096393373794,
                17759944268317835227,
                7514273237508120753,
                5036409225635108741,
                18108376710275555530,
                2676982762515162946,
                510220139774948017,
                9208245069742717667,
                10376186455595199891,
                421637325546718875,
                16791119924695165602,
                5075099022211468458,
                12517850612529158688,
                14010730112031892170,
                9181830456575508952,
                8562092376348428853,
                13106552497603917850,
                14302339172367595976,
                3354417171377350434,
                11375931064094009915,
                7367612779037974062,
                17960646160089319914,
                7765039110674214207,
                11258859351125514864,
                6790503465712539668,
                15367600556170700044,
                6616796092052086761,
                12472062233228758417,
                6844102383328993237,
                12811249515443122455,
                5102740036611216259,
                15944761788498002902,
                14694830622589958200,
                2106296585469395117,
                12001390986454100522,
                3886258101634881021,
                7626447484564536615,
                12121778508662936673,
                4637632611996699763,
                284639519767470911,
                6111767225293430109,
                14899978058938638435,
                9492490138059005865,
                3457376194893217608,
                6384991674167924954,
                16479663265444998572,
                2182180597909020672,
                15349832040611960272,
                6995769868581799282,
                7434433066635564427,
                7330931577518389882,
                11026383041313003706,
                12870325652218422628,
                1151311929447712791,
                9598200806509616050,
                17161065747263888006,
                6901765437989220429,
                17524807833762947972,
                7424089904607803796,
                10169273845585922571,
                427683750176118191,
                16198716196396797311,
                4234469982734082109,
                10433113399561032205,
                8455659057075573351,
                5339947829324400827,
                8994757262942082927,
                16110373093388044843,
                8165219755821135739,
                11079951015921397003,
                17913162205445436558,
                18307573113704258140,
                5184334744599569992,
                8209592006521651537,
                15233048304519008983,
                3893904742760952245,
                12843184579518607189,
                16317789833580002355,
                3932965922048250109,
                39643566078340112,
                14454778211007988105,
                11325761588512978910,
                7326308723838667358,
                918638912645150633,
                11749550850384896237,
                11656774623671896072,
                4102579533296639675,
                1131200342214858221,
                17002635763021940679,
                12699501461265202766,
                18366533628748257986,
                4211496006430696197,
                15592383694738584641,
                12132073284414780422,
                14112124548036073660,
                640983069936058034,
                12843837916043035998,
                12619927090786907296,
                14878814042191779303,
                26802691526432636,
                13542740387876579301,
                3027951464265029786,
                16531319427892075667,
                7433366770688989904,
                17675108936726475823,
                12991848794171434707,
                11308307557943773008,
                8441361375349434971,
                16048668185725998079,
                6848195560782317706,
                15732631277486546880,
                8796031012662685103,
                471880524013342791,
                2847727284878021151,
                8909402802655050731,
                9155393982976367055,
                5339419398360353054,
                12394943475850749453,
                1082481852253596422,
                2475925758143360234,
                16780844326338173281,
                9955442783741736414,
                16398521725460554238,
                11754389093333746016,
                9510081579192799387,
                10740614081401142716,
                6751836211534532930,
                3310216743613517077,
                4711354618706526528,
                11761966320912417805,
                36323519866736684,
                2431075662700945654,
                15771975356730131291,
                14504980176985527547,
                5551404397884447150,
                7328632262405216781,
                6659335151951442348,
                12904376258123706403,
                14766024566003307212,
                2729153030346049029,
                9776447889426450664,
                8565088926995480737,
                12733048698734167426,
                16140690590167093278,
                11999311158468044755,
                13598288633972733571,
                13732432400180070386,
                15194135950298716633,
                13160554968820068964,
                10539080943199862762,
                15115912129273478029,
                17652613656477189074,
                7952212396658049573,
                16131569825649579263,
                14739342068703048684,
                2756008787672952449,
                17980612557340090024,
                4505357043573157185,
                5886347178459794981,
                13467166348185532762,
                7542340141940109840,
                8604473128836312462,
                14161530301987102049,
                7768636854450163149,
                14487815318986749283,
                6851630924543230233,
                15816812718841829697,
                13309960537373415159,
                13533823018337269511,
                4066246092121693604,
                9157035362257855573,
                412214665099375028,
                14477789736289307578,
                8471884807077804890,
                5401116457851954216,
                448871182302722508,
                11307725734317055468,
                10916897930231800956,
                12320324583264666269,
                9855253221365914708,
                10341139850853598251,
                473763070798790310,
                2738091302514798630,
                5726660080813664052,
                4685085520987362,
                142836371012422078,
                12594070839012993809,
                10074822048682622819,
                12827059438505836814,
                16625605990012653363,
                5223703331085192665,
                9833285584664723263,
                8290425612857589870,
                6306954960404098112,
                9412474111491552241,
                3143257994045884457,
                16371722385530408059,
                155878147658248091,
                14824551048836933161,
                6944511999081736143,
                12540758531163865907,
                11983135069038529053,
                4545712278933588398,
                15077557719097293644,
                8824726955621677512,
                9467059989026717476,
                1832994947759543686,
                9395963500530779164,
                10187513723496037742,
                9614996490866381781,
                4074827755791565698,
                12833908036330960194,
                16292730478901857762,
                17035820163674794684,
                989273775053529110,
                10590449105705689408,
                2922575348536557726,
                9097036857789013295,
                12287791133096088844,
                11574082336078807433,
                16687006015927882120,
                8596744673168310178,
                8970905056348840750,
                5517972678784462826,
                881430805465529877,
                3412610955059774968,
                7028178389869167183,
                5170343795610265286,
                11012307439076817531,
                12613254778376420629,
                4047156878552340678,
                14789560647177930279,
                4218958252980142613,
                6013379809704688444,
                10780282427110087598,
                3110875217681031780,
                1760113645124538802,
                4510301077917148010,
                11245657134256982381,
                3975465112728195039,
                10147869619910151776,
                9521542758217967325,
                9796643369778561059,
                18438740993593422996,
                5755122647194246655,
                13089871854524075038,
                6020811568278170849,
                2905460035802413789,
                5770692989150859960,
                9121733696504177003,
                9054740045617104954,
                6651061222026801570,
                6979411400071458744,
                3760933926638598638,
                16334302020381288285,
                2057660950359678276,
                11293316249797872974,
                11604535997697431331,
                11921660539545426320,
                9756548788646446777,
                7269421896006778050,
                3975439838039732758,
                4483475473549910496,
                11780853832789946501,
                8179840661787229204,
                2659883347338825753,
                11257903019053621162,
                9583945287877466892,
                16363933527036814071,
                1290785704105238692,
                8655198258069320180,
                12518299182873150764,
                7361462000368712194,
                9733702303272955573,
                2362863984326388251,
                14125653338074659438,
                8452977297932581636,
                12482787099763665262,
                14889071888466818806,
                13718685624353930921,
                5045136353562987858,
                5296643588295054890,
                13213371806126402767,
                3277278393660323717,
                5669363365118437907,
                15016463550423536857,
                16936344489863263921,
                2565955327304029560,
                4796734652908453706,
                1378655232854232950,
                9591923038316868621,
                16194271618909991898,
                8523814065658657216,
                968783876319889044,
                11876875087504947472,
                15343430958924511022,
                4726016088940609325,
                15814602580682745621,
                17553495950903097582,
                3845404925865098583,
                18216981879371013309,
                4102006890010720614,
                15988113496959934091,
                6019266456198657999,
                11837451541533231967,
                1877408958200970989,
                11155878481768284249,
                16966216363889316426,
                16449662703095636488,
                11722657415321325203,
                8151686270422518591,
                6829105588528021796,
                5605072424730907975,
                9610630778445319623,
                12490688233524207364,
                15924383595417258143,
                6293909456050977691,
                1520031459958370315,
                16878056482786965582,
                1567491557266512441,
                11553059083075601436,
                16588286805077499576,
                2831700015193252058,
                9291679435178297105,
                16696183961702972685,
                6110717331497256774,
                3508641981673281301,
                2888222167255453397,
                6382434826249612462,
                1615495003744037219,
                3450690972855989121,
                7373699525385823603,
                1487214282399378665,
                5054239246312770486,
                6325543252810070667,
                604916284274232643,
                5705249803789781860,
                1347389337137481378,
                17341491803371701023,
                7562092053824458983,
                11528860291431916871,
                10058885809804159773,
                12028682695619691748,
                2072309850975849144,
                17510902039090560383,
                10964691654809950146,
                8557150693328114885,
                10018287661176094357,
                12067793619240607034,
                12258652923874874737,
                14315661805556678059,
                1532344907755129707,
                11548306890701017011,
                9836569326549702085,
                13196528743664879413,
                16910466377189639446,
                305247665277886260,
                10234652344969601959,
                6426209661333831948,
                13316514153882647236,
                320668129269757278,
                9513807142666087825,
                12132502292274229629,
                7907062113630478532,
                6769256578404078947,
                11077216337588272062,
                11192173094112274045,
                8665424743913923338,
                17273052641666860613,
                8242738008086379196,
                13851918804734763571,
                7049674271488004214,
                1410604294587433641,
                13544384176158464923,
                12468330754572317265,
                1317196613823108318,
                406832464837758362,
                6039154084839734004,
                1995025881371264773,
                15399759218648596543,
                7323767619167721842,
                14228968351368149001,
                14556483677198851362,
                10822573411785744110,
                13457687797488671733,
                15113597365244568644,
                5830040833200595947,
                10782120448577939853,
                5923233368326767564,
                13565433709591402733,
                7409835447323460538,
                4942580795960776301,
                8068857330020361381,
                11063008871093537743,
                6583328726807166886,
                2194893842926463044,
                9586490095482756942,
                11811119386143060256,
                3644614927094824958,
                4762559862573353606,
                7054841812193239255,
                104169344287246978,
                13956225391862408166,
                4507288715283113798,
                11772996098884686986,
                12792586168475881829,
                15798457727837504001,
                16896795677207949298,
                10265172010623416483,
                14814049521182301273,
                14251567652536773005,
                11041033214099412656,
                6589290323087713529,
                17080306026857684487,
                8536449894500168140,
                13538904949191682258,
                998478434926587432,
                14670273496850215516,
                15311327973305354729,
                12644104051633342488,
                12970733262467631495,
                4625272733914046269,
                4694472538276404068,
                3167371560249938056,
                12856489429820568743,
                15490218655488023802,
                16928081564771690940,
                8200145788070169286,
                3346452930233714534,
                12795482436352374291,
                3314315705851242852,
                3285152211139935017,
                11224563416094698786,
                9705270177443534531,
                17644295182363680941,
                13520301373099144288,
                3541351715009433425,
                16020276637771846482,
                6027464490277968416,
                4251343941648935652,
                17763014962498696031,
                720132515041512177,
                5173394266280593900,
                15204667210087004583,
                17335914961708883300,
                13218634015062319702,
                128827822406486046,
                13559392235200971420,
                17252636495073021415,
                750007296061102031,
                18254099543935170871,
                7509241234704231102,
                8157193578702246807,
                10298529352072110821,
                5298418177375924856,
                4741852588831255524,
                14527267204533561544,
                9095259534397569417,
                344527310846137539,
                13247449783957233078,
                2586644247732041192,
                3100790169250176954,
                16510173867908259851,
                11333281937378448547,
                10849352377567806999,
                18082377466933472402,
                13561723472079052715,
                14547157815982897407,
                9391369342410485871,
                2348830231178979433,
                16863733530713957956,
                18119733445408743443,
                13725121556809406883,
                9229975878895702895,
                7401016155946807916,
                4811692959864213177,
                6237359194264984939,
                1904440073749912075,
                6353600963009324869,
                4984137724622228146,
                1196090958128593834,
                5119707990981594180,
                3310978073014066771,
                12983216966379762511,
                6612762118768660390,
                4003198331951059920,
                6385796970903716569,
                17680686381236190132,
                7090591178254990847,
                13123476457069922276,
                11991645307762636710,
                5129610078187868363,
                5604361132348700067,
                11043762682351635495,
                6391120985682487625,
                4455513730428273542,
                3762654197950669235,
                9310420262887992163,
                11993439152830872164,
                15439986350739535069,
                1488229606606016585,
                8189570287966130481,
                16920767496097942040,
                7797006069350129228,
                17531953262576833935,
                17024439366185018969,
                8083356010952666614,
                6837066330579419154,
                6443183984423798969,
                4776996086855661713,
                9156255850041051387,
                6083831993070398816,
                14543764421371224439,
                11977571598909514982,
                10188060030334373973,
                9449009160086154971,
                1300044225109284268,
                8825712025626564634,
                5510762923192959602,
                7324962531338046625,
                11942013042939202021,
                13598796371758253088,
                3794998887633576199,
                5084084225782194038,
                15889660289095533051,
                15428522601238467679,
                13696067867927948600,
                1072542240072425197,
                9785760286300726328,
                17624535067298229622,
                3938570677885383673,
                3705077412606219929,
                5340068085943913770,
                4210056340846145048,
                1976665594829638831,
                13311220586145473347,
                6943938959824844891,
                7268540691149845607,
                4375885954089016506,
                14963703931344653483,
                2580585000938123597,
                1068177107148635952,
                2341910556569696351,
                9142288039937977251,
                8517576956327184753,
                3376248516190748921,
                13301443214809490744,
                13173593921898127720,
                17359893842944679338,
                7047222128454518811,
                5947406210583265494,
                7833766002049023813,
                18234948673625328162,
                16320032712674678534,
                12642441645902018157,
                16283613144322847005,
                7863579138098504089,
                6035051266438089663,
                5453172248583632847,
                13775470103081167926,
                1182333285767268205,
                10495826772905540632,
                3056962127861644896,
                3638669906264037789,
                11198631986907530450,
                13541149168510941603,
                3250675262588623536,
                1880073565717482485,
                11936216765079705804,
                7530067836231811508,
                16311420158606635425,
                14900155531445760709,
                9999132967166761272,
                1476154866153451041,
                7195014637072196640,
                7248062023586150152,
                12914546489578466976,
                16344645330454865081,
                13063707322424725098,
                15370611307815629185,
                2118132721549701162,
                14789791523379164567,
                13011177800824348207,
                18047963884795850285,
                13689837669785098672,
                5906277653382226335,
                2699555162272597747,
                14303717239113776133,
                3323572003892592160,
                2809044788581830880,
                12842135035209068787,
                12962493426638468571,
                7435779585695058098,
                677266868482172083,
                10678064418979058702,
                10945593620822631517,
                8023058850263770330,
                13198960821944542699,
                11380271889111358356,
                8834711603621437912,
                4878802483786297049,
                13441769216931046849,
                2621489868509896160,
                2739596448875968624,
                18316605028815878992,
                6393166579696267066,
                16501151485480928243,
                5164267149115550765,
                2594692795895679355,
                14280335185447698254,
                4970220064329069703,
                12700511247281457031,
                9357661341818532747,
                5976227213729484842,
                2603370613961010712,
                12878876453986011658,
                12614332416265768442,
                11711218014924643648,
                10655032017324143953,
                17992359291438428544,
                2028340169188763801,
                11337597956814839013,
                3790580704158447112,
                15661080312640132051,
                5990952720662828532,
                2238062688033035456,
                7214374085319154262,
                6225683889287881657,
                14352811622893565131,
                14441940256427869428,
                11264789359569297809,
                13541671887968904175,
                8127321364325418809,
                1617127834649530192,
                11585884571784117082,
                3211445923374162685,
                17806365145235708527,
                14310538800447604307,
                6443392190904066779,
                12304012177210398685,
                9644770647148205176,
                10349883319707014538,
                13335242320392313944,
                11550748494455916510,
                14451727464612357396,
                12354315710763337118,
                5654383747200879937,
                12534212795765636178,
                5562388391253840440,
                18294236533364599656,
                1155120808307866205,
                7354961778452510694,
                18411509148244472731,
                16833067451811570610,
                11664219918607046336,
                1931829024033366436,
                9466235535733377083,
                16125981341055192255,
                13123158579309508422,
                2811914968022899296,
                13666576380287885316,
                12073873146611787542,
                13976563976935754757,
                16819184991876208483,
                7126372516019046104,
                3771351679962818186,
                14749464410272499350,
                1158403298912255932,
                2704550233949493414,
                2361919431011664920,
                5147560855044821678,
                316960590350675551,
                8956506393667806523,
                14816316691312040724,
                5483483435418414420,
                16492702091910529628,
                11531197856172089811,
                9119688446381120307,
                9148465870444231525,
                11348956949863877257,
                16515747438741224650,
                14748627773020492829,
                6792333258618815286,
                12745044341734710216,
                9118601693019210121,
                3426576167844415246,
                4516979855142897567,
                18383203113196616305,
                6037335393642080610,
                5178029553332071284,
                120594988188384758,
                16911068593865363851,
                5611330513352860577,
                3672922004740508910,
                9348962843954154258,
                13167081398063848866,
                8662579917294749869,
                7714235959241362618,
                1992168626444259154,
                2892890698408203389,
                4458208962709454498,
                16006038729069481615,
                9254965750786898182,
                9178915422232682008,
                7329871300650615008,
                3409678840041114839,
                8437730364512235388,
                12654053093050306243,
                17603379767897285990,
                11140578823362012333,
                18400373855086378549,
                13179220706346381456,
                9393189516096637793,
                7962379756613380226,
                3887762359860195719,
                4702005148801552936,
                13554138609046523902,
                16949160381904484821,
                11391648750970778464,
                3884921049575088732,
                2209098818625079997,
                17799308759029324494,
                8498222108115834352,
                5958053849320451390,
                2912489977852727908,
                9002784252893780496,
                9103973893872873428,
                8800570704285521553,
                8871184059044310741,
                16063400495158915832,
                4031307968282193036,
                8573254083908951390,
                16828584717166299815,
                6494527269040260826,
                7249207279315565170,
                13665554008066255134,
                16839236341242902217,
                4123411447375871691,
                10234357340093040168,
                3518390908608069874,
                1349715014789845667,
                4808798653471270523,
                10095338955926329757,
                12695489128540790910,
                16406780457966718375,
                9629301538534411032,
                15498997222980997073,
                14953211317262356769,
                13024285896073029281,
                14612862358802352485,
                13035854789120349150,
                7106552057026833281,
                14174041880363702304,
                10636381469539675255,
                7187203466667480841,
                17084717625060872520,
                14126252750523939037,
                15698822748718907462,
                7397904980198721162,
                3690934666532247692,
                15065052457896220534,
                18426160025856536769,
                8825186243264577791,
                1029026778404309644,
                6950954458719738495,
                8771914568967577350,
                9728022865849579333,
                17765239852650572501,
                8343956648157842199,
                6334778100080475126,
                14664915370166297479,
                12941375852754193937,
                6405436436050091550,
                9909891363532277572,
                17675467189601777836,
                12324897459248787941,
                13981335142409795938,
                17744674612593350856,
                2757587566489447009,
                8956578028983169012,
                17018240562727015214,
                16391639968111798359,
                10795774151465209457,
                2093281970351304070,
                4858229909673020125,
                9933967343419708401,
                6016509820559456727,
                8502253989054157587,
                1829270427581251166,
                15709494579608995279,
                8568909950760020667,
                10892820698227375197,
                14181785622385373392,
                1741244869916907800,
                5584342639358924336,
                10920058455798795209,
                8773147161499562392,
                17372651285045475439,
                4317270331835178992,
                12072221230432227872,
                5409998193192839065,
                5774604676764375748,
                6270673186535390929,
                11593578005123149455,
                1350291030727318811,
                1734041789149423611,
                17594245397221493182,
                10230281830065256754,
                17395251244463936984,
                6363273717700984685,
                6898868518537597264,
                5078809256797027276,
                16014780232233692324,
                12698021273486013402,
                10034628326465398981,
                803921083181133037,
                15053987885932962839,
                2543409146477711493,
                1443575144201918042,
                15527634810280511429,
                4846659158470140918,
                12183993101313012462,
                12953651110928456969,
                15038003468538497849,
                16252394591537250522,
                4193562522090296620,
                12800591763579590396,
                15408603400942806170,
                17682640208554596681,
                12409260763379977486,
                9315397818726387830,
                14193643034683668324,
                16508104272720261755,
                13905815394730317134,
                3679293709974358468,
                5907121386159753765,
                5304769534414875015,
                18444287322445993596,
                16494909212649576365,
                1411980536871151577,
                16554121151776992767,
                10463210072047518059,
                18023249470213998238,
                12383793606215823679,
                800485989566329537,
                15071139529581604306,
                1895628018455249296,
                1105126069974836026,
                663584933815778847,
                3312928249076628720,
                16646825576352631842,
                12151072254529924642,
                8642785126263747138,
                15111584853907028239,
                2776308895820674677,
                10812673714292641746,
                15054903323219264732,
                2933094921150563585,
                17830062919092998452,
                10432330036767081552,
                1033494976738144353,
                9761049718456631119,
                2978550270506139328,
                2266577450295768406,
                1214959603409486419,
                4338607014040436,
                4115159162283844446,
                16888561205519247430,
                4412890126763454080,
                85232547048588047,
                9970948529228850752,
                3713393126801217778,
                10760802738778841380,
                4895931449000106045,
                7255762665740428559,
                335936537100116164,
                17662778773720603199,
                12685690660905184006,
                14808811012699062151,
                10663380197090780332,
                7067049757688041329,
                5878922448066220307,
                14490189620281922843,
                14473260627169887886,
                14832232348370447833,
                5951438093661647765,
                4872796166881199876,
                10429350570401478982,
                15251172935852223950,
                7325621106097392986,
                11834629549993290294,
                11569303197879748368,
                977753391334101798,
                6969686311950446853,
                7366200137172124620,
                14212998724743560415,
                8081354270400804456,
                880037272912518632,
                380852277882293501,
                8947498467697222234,
                12757931014561147509,
                667810441349799809,
                2059689120439565076,
                4750342010666522618,
                12492546560830555305,
                10465223395020225996,
                9585389043436680319,
                16512031046174235646,
                9200703296926083957,
                2032951915453711444,
                8002381174072382487,
                2389280604724196215,
                5721524692632905547,
                7647110976590069211,
                12059178411864741586,
                11975912934455308497,
                17753420811972470476,
                9192795740173474360,
                12105166740957629248,
                1250032644270746132,
                4765787287696278663,
                13422669569705597563,
                11373393845567526876,
                9153761787497611672,
                12030450058080806969,
                16537144145501712177,
                5983397454957073922,
                2683673672010015032,
                3861849825629132671,
                12629475949497494729,
                9966234113604834798,
                8466461060632654110,
                3335857898642241962,
                10682117780482341111,
                39787149161941136,
                5109208035333406420,
                15684027554962652735,
                11748970404336820351,
                4095159288133322954,
                11402790565133565981,
                7401785106763668233,
                6248303083190138658,
                8738855753439299071,
                15339296752899687482,
                3182643107247815384,
                9378566535341755855,
                14162009846460499887,
                4109114270719890973,
                1511416153435046138,
                9020508522805341634,
                7477142615652523025,
                18385803953807265839,
                2546195602154738889,
                7405815335061777589,
                669094957884330982,
                6056073383962448124,
                7353716808540962518,
                4360012167404246482,
                9748894070631625514,
                533481526701689863,
                10374030482918918166,
                8591056344610511021,
                5753611323076772742,
                17682325790000893453,
                17410739240868104811,
                5651721564405833200,
                6771155463622294161,
                16102506172900055409,
                3922155114242412611,
                8047195046803908182,
                10181935025035061310,
                7790459581343029499,
                12616035204180788696,
                11350000801651108812,
                9664257517083996508,
                6877290837507516646,
                2900668296021062883,
                5478674374212655343,
                6296162153147759087,
                16234745308754151636,
                7633277882818614712,
                12843201503731712898,
                14050083812260020416,
                95836250787434633,
                8997148048216279615,
                5687865929885690952,
                7684029056960298539,
                17362423790832901707,
                15489463223618769754,
                489477444088932230,
                17185715982289417959,
                11823522798262762160,
                16827938740128105724,
                13257410922007807743,
                4679309639860400639,
                10164917239224603264,
                7304922446473936277,
                5597481581945817575,
                3820424595621388213,
                6884972150866725771,
                2694324376629657414,
                16833520400918984369,
                10912563532735352632,
                7352008235482304863,
                15885791964406330114,
                11790453162438620892,
                7239285021069261812,
                10125071555494474444,
                3796211625840968848,
                193266766691985397,
                14792644085266904226,
                12987152514130686235,
                9436744123646480376,
                8251406129814681521,
                6271920831800751835,
                12738844182063426283,
                15330691163818996222,
                1166041665727565335,
                13315209544138143871,
                2216677316767409466,
                12690036582164080137,
                3031307738126062683,
                11944440105997765289,
                2026188631449062025,
                9212773879860499293,
                3341981308024472580,
            ],
            &[
                13169652012768583147,
                11662749290968821290,
                15540028649212863382,
                15222191772599122928,
                15852211273603082347,
                2960464712590969339,
                9805532347048503730,
                11483156842014743008,
                12223751000089978338,
                2069577308821362462,
                448020966454441311,
                3493610387626232963,
                5182130948708681868,
                4456536535034921689,
                14139535083059739910,
                16341604127053129275,
                8476750032358968964,
                10942274856214572646,
                4426573651999058804,
                17958371714565221264,
                4238987860855800392,
                14450925264923938948,
                4364481289454625563,
                13074790925168694865,
                17725186031491616550,
                3252385755044260850,
                2489691844698347065,
                12584453984545921259,
                16484939371755776909,
                17591699265602016514,
                387926126025483991,
                2463760292646247776,
                1563580628888369628,
                12118979090065897855,
                7930230430393343357,
                8834023487181578700,
                3705061516543521950,
                4053176902619603587,
                6269266280509284877,
                5236641881717368169,
                6899670106034156441,
                12840225294220778634,
                3029440279850048735,
                9595166473221718273,
                14550471357175468027,
                1255830945916836371,
                5122133954630663971,
                1502740582962090101,
                9086275497101221342,
                5421916064800438697,
                2179464833779544632,
                8247585817871008274,
                10322070671895980987,
                10962237209348590213,
                1167315820290398061,
                4563405457798620948,
                5324869835018392710,
                3941515430275954463,
                1740094742017128848,
                6138471019874045522,
                9176213051434147108,
                12501904812453849864,
                11388325168842467058,
                17687408972570957542,
                4497137255970523312,
                11770269946165024215,
                10156397926378383551,
                2019377424484113393,
                3590983554593693717,
                15657928449196312716,
                14894403732533901822,
                14625564293418317463,
                17710716015671252818,
                638852277383858770,
                13800670797826096071,
                5519007738268375392,
                1013914690263468265,
                13249504241724137521,
                12893315643913486917,
                2437939332985629386,
                9866735673756110580,
                7894145839618709904,
                7802203558085957599,
                2394814001366981818,
                7648858197866754106,
                45932857419714984,
                9659675215715090679,
                12803466849037463477,
                10717587339591792614,
                15131328571226415781,
                17805628124155724278,
                13655343682572989974,
                17362327502804463995,
                1075287460052064029,
                2626850396158965217,
                6885136253881452991,
                18424853270962528365,
                17198461501972637926,
                6379806426022159044,
                8783653573177405838,
                11457364326293914599,
                4889546163609369345,
                8345334679410921999,
                8128212263869118047,
                5243835377297724594,
                13077589216209892913,
                14947690960428829294,
                7468326754846892750,
                8665069680383178636,
                17267364262089018920,
                14794644088627434581,
                5781944628494317535,
                13467037542503882140,
                5504287258083448304,
                3101706185952735851,
                16528687565738693574,
                6853101285149030710,
                958150364135525317,
                13159192403688468400,
                9000130047394330226,
                12454106782612240528,
                10410037892467703265,
                2755063393447593674,
                16909229563067880679,
                14030067308250726680,
                304063827283258286,
                18020451823918033658,
                4591035690657100956,
                13162350837887011458,
                7701399978207767086,
                15815279200816529309,
                15597387052720630170,
                4740390593486726162,
                2909712595092459460,
                15729877265226222100,
                12096570103612063938,
                13702753332858235272,
                1411835182384377603,
                17487917594195511154,
                996305544975114980,
                12929066524383774152,
                16956610067055251962,
                13012122099231548957,
                14395689158572846177,
                7935429077563080059,
                17542764935243101497,
                18413828249925519615,
                17271321257298840331,
                406766324155888956,
                14818331828215117974,
                8187386216945380849,
                16792638575276899182,
                6540004189057298818,
                465689003663102769,
                11974785539062893556,
                11523099967255714584,
                14591636912283935445,
                4871757987380755363,
                9308130069290886865,
                10612219179153523310,
                5420071244571425562,
                12724302122592532888,
                5173545157589181140,
                1429324130148847596,
                18414282026575416657,
                1721766939064217260,
                185288146264311331,
                8027581238524003859,
                12629121378225318691,
                5422022678890177669,
                6300007666394297907,
                13771277516427401556,
                4461063943424152501,
                5814271695863693557,
                9833762282568603070,
                5534898830643720899,
                11414581203473717153,
                13252208296859650317,
                6070377927034194834,
                468982118171070321,
                15569817073538385644,
                8515954578625472255,
                17703147536676285546,
                2713926565223832109,
                16501613896332875878,
                12360649233716580846,
                6951122318765494287,
                1255018572838364315,
                5022530899188934293,
                14471021290055621346,
                3965972119993826976,
                10914640319991729939,
                4671893019190076511,
                8132288757883741990,
                7379730619550447967,
                7082846353163966035,
                7749284037330602929,
                17090786448482076935,
                3623093583024522445,
                6775174256836032571,
                5932859643874401763,
                9263718069255818484,
                11941172789559596258,
                7505495305758940713,
                8374557854631657796,
                5030384219708396008,
                3609081759463861677,
                2444176790044307758,
                12883357531875110193,
                2545675590276006076,
                1091049428613676247,
                3001645527995991696,
                9613284625002680960,
                13486970893816936976,
                11564610387350917328,
                7838720830134816136,
                3368189672511472263,
                16407993343222005051,
                3965440550685537978,
                4708478243468404768,
                1875742626798710446,
                109477476572266497,
                9185060711428219960,
                2062557879153655855,
                15504010890019002364,
                1436468774610332740,
                470895386030379118,
                11192666441535176072,
                7094623340817246353,
                16041820192281853667,
                5048179711756437977,
                1908765243120848557,
                10810219542319634593,
                15971282719319513661,
                8395180060853137311,
                17415231760777435363,
                10489441496511966940,
                3213813993411776195,
                13309643264471443871,
                14152370013037392509,
                9805694808649718454,
                2991106746654603900,
                2279780759950444362,
                13989789024965052794,
                11942900996372546394,
                15051646099368945987,
                7583267444740224154,
                5192555551306992142,
                13216959612015226558,
                14397593124921798192,
                4374707725758052604,
                15529039788482882168,
                16475619415913302762,
                2649655538625316107,
                11899566009797061232,
                16892018353302708322,
                12364166218684092426,
                10372683350152669575,
                5328477050361662527,
                10949313792626687170,
                11924230090558408019,
                636095257275314085,
                10210914536395617961,
                6515368615710817299,
                17001040330117317563,
                4519581466335837391,
                12514298524110577503,
                1744739741016504912,
                9689951195577973993,
                2774540351629210180,
                7646006525601223889,
                7530479442266342421,
                10538917317906671989,
                6031253606146099659,
                4606167752034374890,
                17763804757970320617,
                6871166461842469695,
                3069238346134678776,
                6898492556629873366,
                18330913680324098797,
                11469514731644700404,
                541525751461369940,
                18096353837092553451,
                16390355202675291281,
                11980660197830118439,
                15349236833293249382,
                1901608480726107605,
                10754477505423276549,
                1780265741680781169,
                6204486525934807168,
                2675596223932028138,
                1082656654749778403,
                15313246555910578407,
                16597928615331243076,
                9259667019969423039,
                13966308744186245944,
                3711817850123753976,
                10380302371138242940,
                1102076039759449071,
                1796134829015781040,
                8320143967311972799,
                12603759772660781109,
                9343508021191710450,
                11833581848018650785,
                5036078561764059265,
                14477046704375156568,
                17958581882146395247,
                6312863763728381603,
                5890775652924120757,
                3024100139720533291,
                15594980805126031556,
                9929023026443104564,
                6003618257535256748,
                10584762467178432077,
                8070692469310128137,
                3448506857138757663,
                12067250229870443464,
                8289617863280099780,
                7705347156484755083,
                8239333463449775936,
                12953560490361273923,
                7863343728067574413,
                15499170354027597711,
                16789849673988523596,
                193554686279189480,
                9324166766841362509,
                17865442848307712220,
                4017570072096420828,
                359910874540192895,
                14283136222667718022,
                15545842610688758167,
                9161703351540143861,
                10856046010222875891,
                270665191423043189,
                15816860058997167317,
                10604344774996324885,
                3421439666836669232,
                15362489920565158049,
                901986887674190509,
                8102011985696643347,
                9831189590417175739,
                7694208525827500802,
                17761503253302918415,
                16843825659097088549,
                15968418888658984723,
                13213298841864639514,
                10768746828601368994,
                15843374988242905966,
                10372481985078119452,
                7145529267057724453,
                588445411774656377,
                6907651744680403237,
                13729433346623178186,
                7466282812151723072,
                2397976897839660773,
                3595288154116844222,
                3771046205661211665,
                18292138992930644589,
                5076094080207419189,
                16955129624357257007,
                3929215337030354093,
                11057894513779502294,
                11863471976993485758,
                9408615240131003867,
                2448439901554368107,
                7669176413900589628,
                13955883899642443946,
                3559761514651432863,
                5999853120304570098,
                2807928142993089395,
                6880972979489023884,
                8540438276045396759,
                13336393887543928327,
                14948024915319356336,
                754486632845103448,
                16252590658474672770,
                11838852632524903679,
                16841746824793599992,
                4866584459095974245,
                13046184814853980498,
                8710985684981006226,
                2355658489901240093,
                17682251256218590491,
                12738885399031396318,
                3641463259326479601,
                11573802402437143897,
                4785854649163125380,
                17654700608592820265,
                7689738254866306083,
                3911334929659733907,
                3805687817679677472,
                3372180752527728873,
                12516023017017683246,
                6541569134387572029,
                14932792460356112158,
                274079842264421275,
                14741476297422044947,
                7640237864595124592,
                7990415001403780690,
                14485175485850127358,
                15701126719940798119,
                7497786401585040464,
                2046586308065249596,
                4234230752844127257,
                3490034085431124756,
                7410205974894450493,
                13594277608316567296,
                13163068385450899386,
                6911330603633557461,
                4165273348787835236,
                13196815825235476993,
                15409229411685315073,
                4238586327560550200,
                4274172509763128574,
                8233070080684386642,
                1152981557203525396,
                2258263369774438742,
                16860590243592954334,
                10642146327347114407,
                16394076595581895042,
                3547145813374894294,
                7303429257652622247,
                4599869911214805642,
                1470127264048118954,
                6553553520469198052,
                624221776008514758,
                16024993613079197687,
                12072685851575104500,
                4289561477780298769,
                14522030035108027997,
                10025959010474970695,
                4338629041423254482,
                12668451231751642706,
                7041673834301289685,
                1101300709714299914,
                15830324192849592102,
                6361267392222383309,
                12743724653827284764,
                13913765213720067888,
                12472686594479717483,
                17041667280903689118,
                12612588174971368395,
                4625038945277105759,
                4288983616264520473,
                13257170061268337928,
                15378358854815534796,
                1307879339751355693,
                9045256242044980161,
                13066137780318064542,
                17483698129188252816,
                5537781126683250458,
                15814088167067959984,
                17124111106622331645,
                2228813159198082553,
                12771677329276373795,
                7473600359686104579,
                10172401847101693206,
                8817322474066906889,
                10002046533039894942,
                18350636570159627372,
                2608194133937884904,
                16496103406635747044,
                3203166370447920771,
                1416097169966651887,
                860034031549905008,
                15435212006888252138,
                4982756044520683355,
                14527835149012332766,
                15648655845991912781,
                15220061209366144565,
                10713592116389348982,
                11856331444132914701,
                13011883287657858557,
                18375660967945668491,
                7611177834315648116,
                11475292136381607303,
                12142625427672665818,
                1644376245508948388,
                14986147846200059747,
                10358191212479538383,
                2124663276626493099,
                15025396046176607689,
                7566356641233415240,
                3834259550197202335,
                7422439917941038432,
                14335746041650957139,
                11010643115481291768,
                16639917992723217687,
                3400935192914355446,
                6526469202356090658,
                2911450085827434145,
                2592450403666061908,
                6573976403851247795,
                4285219873414520276,
                15329532146725863869,
                6799347789229688901,
                6639341479260471078,
                5137054100466749121,
                5102420602612942914,
                12454937665202983967,
                12962650067948965377,
                17483657740097839287,
                12223498331603994543,
                14079145217780475833,
                3310333423646124225,
                8148231670879135605,
                9541038636961784750,
                11089617618703446533,
                3560209342825276058,
                2574634823119268479,
                2588311243176990131,
                15013509191556503430,
                15455072885984268053,
                13986137485131324765,
                5720067926322634989,
                320358790111393789,
                15410939285136639329,
                15697272961739331244,
                1355846111750777085,
                2546959351276906905,
                14534695021313396715,
                12100942362109396400,
                1909839808518887758,
                15057164163203198957,
                10816292376571145141,
                17932667027515587541,
                8894763284691124395,
                17376767493810914651,
                1054656077023869452,
                8186894411081110112,
                4918609723063820894,
                10472223930221187526,
                8210872174259857628,
                8260857529094096436,
                11707883934354468821,
                7798711246011152830,
                12313898141945763999,
                4092899057830195664,
                12768752632246510730,
                7011904089799177666,
                18269846218072729005,
                4472126139715717850,
                7263280326862818361,
                4617033187864299345,
                16062484525314928005,
                7764862313171817272,
                5507098054488704246,
                15417948890339509732,
                3199184308759380889,
                12873647785813929264,
                15851634423390314715,
                3379562369756927370,
                2914796073219663247,
                9453518290639493724,
                10389507097848983202,
                6987583823829734148,
                5644521590882572085,
                16385036288302112854,
                14202037992186130818,
                1662729350231436066,
                16086866498209553899,
                11668110277201388817,
                2581723866463834707,
                14466562150912612524,
                18444381539786477861,
                11792751584642451686,
                644172128540161319,
                15714856399314491326,
                6336413224068482580,
                15648040294508822482,
                3581892413107868745,
                2087860646799841458,
                12928312782299116049,
                4630539989871908829,
                6915603518697601787,
                14918733616941773093,
                15087665207194981362,
                613969264471068110,
                3544544803540968193,
                5220360717428049381,
                3375260723217895506,
                16778447987047186697,
                11894812810172861433,
                14542979312432827779,
                13561316753657099179,
                1968473618327458185,
                5456934299993876358,
                7469735400543018284,
                7068657239043349097,
                9254040800524479410,
                5214107275553666760,
                5001330682797736624,
                17026132790954295091,
                10969354337912897232,
                869351389198733586,
                9574587810937905907,
                6923373747285449001,
                12719903474522918771,
                12250857818862900982,
                2153256573644871951,
                13709729097688338391,
                2899377834668376960,
                6841869401787596550,
                10339715141881541659,
                718655127116157511,
                13741406681906111321,
                15392288868670671619,
                17545786767738322758,
                11089794771839702758,
                219094231550242583,
                9674358307241496741,
                4002620491443309348,
                8012290324302967070,
                8714326673970415123,
                12380264119087982419,
                11108551912665576265,
                3708955946231240253,
                11607264285750874901,
                4216424611300123940,
                8256067698160491644,
                7506994298913059376,
                11501100133000306066,
                7549037119651533111,
                5112970978157051505,
                13518442004335850902,
                7350008695599366666,
                9542638447580599221,
                2158188779042279647,
                1604170908932337076,
                6391929341437538244,
                10779746559124306734,
                12656450449103176181,
                10121044395803332310,
                11651522650180060700,
                16328514328541821784,
                10523761782294701174,
                1822303728984452144,
                7260452386953933551,
                12204494523152059466,
                17100382786462731446,
                12548225060484474439,
                11326872465335361323,
                5596285187831113760,
                5909751003547848587,
                992306619579834385,
                2228664329259510492,
                15196270917670055982,
                2131889695890586742,
                9819776790840463517,
                1473593931814601068,
                14939513955417919643,
                7685117665289355354,
                6724244559482510387,
                101460542568704969,
                7352190705745517803,
                2150669404455836808,
                455126870518119052,
                11553577781469330615,
                10384536636375637150,
                14364597940339752635,
                2495319672618636831,
                8180783418013670444,
                3773765470722916252,
                10227754258298095569,
                4811293038656815364,
                2304664986814074602,
                2418992918773594204,
                6388448639180737685,
                6893696275605311774,
                13851958766811817366,
                4793074330182947851,
                4813793152033309122,
                1098414269347239038,
                11751908186279088029,
                15167430159498163699,
                10528686090760108130,
                17120804521852235295,
                5538072916971524268,
                18284870874128074659,
                224523015148941267,
                14496093443691170173,
                5490655299006793623,
                11584315652765207551,
                3229579698633170375,
                5527375504543907996,
                6239324650299686862,
                2232148196848710005,
                9031256656667465794,
                40023723987633148,
                14627741678268078967,
                11950626749898431344,
                14785414395108523186,
                870832948163923646,
                13004928204786500706,
                11732704426571300600,
                907514934656616373,
                11587740830029349245,
                7583505136677874861,
                2623787766244261370,
                10256163025245964672,
                7257287909572091733,
                4812915812660879618,
                3468364439026649335,
                6604147596013682093,
                16816894664989923710,
                3964120271755066308,
                6086548275463043980,
                10447117175471304144,
                12856607441078849807,
                10592398111690221432,
                14269275707379118835,
                4396127772639422927,
                10489721798852691549,
                6681412646421684356,
                12490106130106455109,
                3624408102113036819,
                17246634770535451304,
                73750230380742720,
                3858795004528971618,
                18037249274864620640,
                9013680198221453825,
                11909298583725721915,
                7512555591920604211,
                17449244834240874515,
                7793591661081767445,
                16652407448095073621,
                11804971167281860328,
                16410605854679152138,
                1713343542446776550,
                16483548344281970818,
                4628627981197409087,
                9083762764219706193,
                17065951854876058290,
                11490185594933330698,
                9891012968004512035,
                368256788081374064,
                470621950908252,
                17956316299265730194,
                8221822257224301188,
                6380872247545425488,
                9031471961851140451,
                17956391258409007260,
                13017604911584754606,
                9245778199486992293,
                8056793204522970093,
                2531655295742236851,
                16039488662052065607,
                4695551407385442957,
                16581782264738209135,
                9457268466305801158,
                2502271077174158302,
                836797013558343943,
                1224523470498102285,
                374165187087964522,
                10212314508075547184,
                7669093970332858967,
                9176723313583927295,
                8839526249801396044,
                9460240515540152245,
                526138750665184089,
                6908994573320912372,
                12176043050383940227,
                8431292525679019707,
                7800865258551716648,
                2930208417676545581,
                12001139080437682925,
                2848294057310329443,
                3495763019164815081,
                13998099468531902475,
                2334207795701343601,
                1265352082029868252,
                5352951968929077427,
                11045950825318827473,
                17705541510574119706,
                10388016860076689091,
                6188336841660552180,
                3874842388902329486,
                3586889024655840994,
                7577730940669490050,
                3828324032996703471,
                7382564057518691083,
                18020764642442034007,
                3255848346667024023,
                913981917700616307,
                6544220161742236189,
                628453794376152858,
                5273139986123145851,
                12804359385937816703,
                15490118777013724970,
                5004956137979193113,
                11537600208633214816,
                6041778305931749161,
                18021165191120757752,
                4439552721639804321,
                9445924893904633163,
                6333608419766441318,
                6265966168082228187,
                5090746176805509189,
                12079779788101746835,
                3405057163683809716,
                18176431495575685240,
                5416316679824315388,
                5666993498240944478,
                9396996717121195736,
                15077940112899068312,
                15769845674133811095,
                4172092555141541227,
                10535661541737621064,
                10760049649331328984,
                9044328015087752668,
                4208484814661836163,
                4524519455413453106,
                8416163180819044235,
                76133776812564214,
                8938397440798861336,
                7699462000599225384,
                9542247023407697767,
                1397476485288102889,
                12838783590422826565,
                11111769246893411506,
                17719085637529819037,
                18350048131892648764,
                647133208572620551,
                3545832683987398107,
                17392149180006432089,
                4645454948531556636,
                4638633830898005179,
                15832113029610700219,
                12312766094616504216,
                11054531676785061901,
                12882000175930708962,
                17065292675540451547,
                7348135285662513163,
                10368855461513742486,
                17213692472848349009,
                5448237816391429929,
                17475251701204191843,
                12352004908893419844,
                4421019229248084817,
                5233890583141520327,
                15415498665325430242,
                8699482400205838797,
                11081065321342975843,
                13692414560354688711,
                2057100631763259620,
                1768191442150902149,
                12463118341434320148,
                14697940362261450576,
                7225350934202123472,
                9180626944525820669,
                14817691250517499688,
                9524098384486815580,
                4894140555324164954,
                2309262794362375113,
                8645740617004046828,
                16829903653299752341,
                4278553392732422821,
                16174995535301769275,
                7274361875865992211,
                5483861345936568593,
                16555617280692737205,
                2106788877369503099,
                6355572984153420404,
                15413046204382700864,
                15184587830187202860,
                17638950466015943646,
                10142359495639047596,
                2852278080143788489,
                14902051473257283997,
                12684082230048932446,
                5127878269602498356,
                1458833335160283623,
                18343902965575788886,
                15956410439937236477,
                10143815431586875695,
                8519328419973108156,
                9379288137799805556,
                11116700304696206296,
                3269635235146511468,
                7446145908658980471,
                8423421258962502269,
                2960959498368537205,
                8029834119982365442,
                12951331847284209016,
                13335373530776006943,
                2051038411438343086,
                14988843915712852160,
                11070472159205332532,
                2405233098821133668,
                12872299485466251244,
                4146849588411590424,
                8652997027024585292,
                9417874732592565535,
                2771366489480367357,
                16381568131097774387,
                1160190173665475303,
                12142110843989704382,
                17157977103678150822,
                12240974252728213375,
                8005232037362794911,
                62748564819661432,
                15562900352067429326,
                13819933941831285726,
                8700733640462543781,
                17417693392111317966,
                11195518897113567806,
                8656545261572504126,
                2665140739762930045,
                2523114891549547532,
                15947075740878853027,
                16617762624958355772,
                9189411410511795645,
                9417682881040880723,
                1220008371097876826,
                4473435335583972063,
                12416336276688398745,
                9439155074539900208,
                17599514971068491442,
                1382978212068751289,
                12012964446858847311,
                13059411911626363940,
                9000605160655962560,
                18444429515577739106,
                1678358883660355393,
                5074103451266196440,
                1108189786283239561,
                10612698979413855931,
                4989895863474068758,
                653620819971957060,
                12060263383072344947,
                15076531277958824549,
                10597595939754188762,
                15724450558403473627,
                16382650125476798913,
                8738893762787150302,
                9733212779214420390,
                5558001311635500417,
                2720376062751869999,
                2936217033743496825,
                3059230718610361194,
                5589670835881847684,
                12297086381101763222,
                12830838307506907529,
                7727866232017902927,
                16978583538503170044,
                5646531489086592542,
                5696883024644775459,
                1778382958859079024,
                9219341275099378906,
                17593963150311257769,
                198176573650528594,
                10425519151629874209,
                13457055667833759545,
                17213583299444020650,
                14862117423842849302,
                16906392281688863338,
                4290302946266596427,
                13505570400341984024,
                2056510386911305907,
                11571082458161054292,
                7566281435617775832,
                10210241606684674096,
                10697498687651578080,
                4947318501826974509,
                5861434792576988890,
                9869040491958929661,
                5532013052186573089,
                12416594018569715230,
                10820734898252214278,
                8537468997207455166,
                439343392893541672,
                14734630723072977466,
                9898041261033982210,
                17393034157801306127,
                13758525217758656857,
                2713096067589327874,
                12442874178280670969,
                14498768890161377807,
                161967360889456606,
                11930601082339405958,
                11362787206913160786,
                12545070484550723467,
                14815095129086834610,
                8605787829002095856,
                12923597973881209899,
                10587151577465292584,
                14183134664818382968,
                5172374334040473114,
                9364926097792520560,
                6171919845492918332,
                4386231931762251578,
                2155663571428617218,
                7765511219465701910,
                12061535530010910259,
                16100330205655290696,
                9974384126080194942,
                9674175848724908032,
                1687977953933677553,
                2052842863240418404,
                18296704913226861337,
                4023434438910601547,
                9899743965354682742,
                5310198234766113901,
                5162456985274720081,
                8646509974119784542,
                6113717383160009894,
                6014191554047499022,
                8484348411248636487,
                2876047496375093909,
                6660788054730924011,
                1664197372146636482,
                2820993293910648101,
                11665512417100144808,
                6502457677692154814,
                2170150732610996776,
                15522116097990839236,
                15440086643427998964,
                3675113412536301313,
                8316358419658521853,
                189389437052919038,
                6981753429704993440,
                12802645856607643857,
                7243825699014665843,
                12796799374814419834,
                8068467907492367066,
                16310382387876189712,
                1486061732225930791,
                382533986053029739,
                16359148449656380800,
                17246074348224240328,
                11700746273206874174,
            ],
            &[
                12622393184452988185,
                6576838566809301648,
                5775205735049728716,
                8287278668518156534,
                12192798229753293112,
                13075940713888694545,
                12454048030060066070,
                811912065849436865,
                14514498508890956207,
                3718975340013031462,
                5748599663081118833,
                7103827618714489700,
                1300608812711622052,
                6273491418082608362,
                10602503410062294488,
                3181236738912952810,
                11220558709822910991,
                1757050625501318151,
                13622421981172568598,
                12832273400791276781,
                5855047466166966026,
                8265514107974512007,
                4674109636920327418,
                15064411300962651052,
                10245194862302150019,
                14697255588820327266,
                15300902461722524376,
                4133685444341227543,
                10059303764976223858,
                12025228906916712453,
                3342699448606724702,
                4043427178402639607,
                8354440664524965265,
                14685529799335576021,
                1998565942493742826,
                424458568212209524,
                17175147416410219809,
                8429313419702361615,
                13100329766043560847,
                16630087004934016029,
                1975643079807205422,
                14257249811747812832,
                16201214544277583293,
                10834834555324096005,
                12974966771898232184,
                10175033989408956356,
                13198699716510876245,
                12667628170354632487,
                13500517333069730437,
                16670408922706885903,
                16358625308423139600,
                3004604642549199974,
                15303830789363557104,
                8922231358023374966,
                5889284410855917356,
                14457653243881523958,
                1394588608017240060,
                15159743360841977286,
                16367826019458727864,
                13291054711118709892,
                5739609590986339029,
                16727185121517285485,
                5857477526908140920,
                7564539623883188584,
                15028512432506130151,
                9451165299139733038,
                7720569164934553601,
                12250119607377112932,
                13838627703516838495,
                4013168924318862931,
                15619946084866502269,
                4850331740702252734,
                4489013954543776517,
                10588792414395604443,
                3902488335149478186,
                5227683403359005885,
                7076009259772820031,
                8343084757976224303,
                5177402640814818565,
                2803445024434417300,
                4085994593957453089,
                17829839529352422672,
                3578735210913423849,
                2792806554222369562,
                15534236200548977629,
                3124727634409390443,
                1084234245147417462,
                1598571627717804884,
                752329673301761003,
                12688955135473243162,
                3217100306345888294,
                14566509229614296336,
                11270786911681876633,
                6662721337309084196,
                12158803014774283480,
                8518459513237421390,
                5910661800660978266,
                744628824726848011,
                5469395445007916291,
                17702754906843708743,
                7968296670638284359,
                7510310566936008628,
                1793306228498108099,
                6353661129197447108,
                3126760188195263395,
                18414397606186055674,
                17486033377523869654,
                12070987014285382902,
                9991875370105670503,
                16129470720148507308,
                1039596679504335478,
                12158960746785703045,
                14081886766117011421,
                8985649855006063812,
                16817043795641114882,
                3052278422002310841,
                11364643265564647695,
                5842131153676887611,
                17671187237412914007,
                8666277097534907268,
                1983817527033510933,
                3541460822071490249,
                6007905868789893806,
                652610536509913563,
                11037032573713184751,
                11009156643744878354,
                17591018917017615621,
                11710867183921879437,
                12251167927128949787,
                8974964125845019147,
                16969705468711169632,
                677161721939989175,
                1648428070692386013,
                17084411431327447982,
                16079687398141624486,
                2070436743667033809,
                2763689681570019260,
                5133447300857161972,
                15255425835157483742,
                5966697992253286903,
                361687726752744332,
                422697286210467434,
                7041898431113518823,
                13795043150550399420,
                583755960417947241,
                2875531821527607804,
                4394534528084370577,
                12535851821369757920,
                7213044591651422167,
                1000544905146371455,
                4653559762494073142,
                7779502532203369012,
                16702927718605692181,
                12982197318021037408,
                11786209078045506353,
                15564445991051693791,
                168023569501370168,
                14266568007896652015,
                5832135998445910960,
                5245116375679737807,
                13311176288056325459,
                3321928379396402727,
                2233377424088736972,
                16828319963915513015,
                9909739734979137218,
                9397081158643758651,
                16237610644843443174,
                12049861989729092874,
                963173337050875289,
                1393288980328672002,
                8369734762526854678,
                10087190917060030061,
                16611332898402499111,
                17383970934682687319,
                1806005829620081904,
                2572338165105796552,
                12096685872124824149,
                10625877836202904943,
                3565777311969108583,
                2012783419915002762,
                9017825065020723376,
                16320469478354954310,
                11806049513756650829,
                7727243155747891778,
                15158785566684134320,
                2456938594466970469,
                7873271366166028790,
                7686875950801483839,
                5617367656140550179,
                13464612212949386527,
                14757590544232715739,
                11128260848489879698,
                17626653494839808091,
                4716501168031180893,
                564851231412074685,
                8132887702611477369,
                17552681344611428819,
                3132957846371517857,
                11213590168035816092,
                5431508051609838979,
                4515078710323627036,
                13543555744861125477,
                309309746143903518,
                16853539194117891693,
                12430835367581296760,
                9255793325320254163,
                6591016556383892120,
                17100720673447110167,
                6061633534359930907,
                2424801126737847877,
                143356278625650878,
                13554246058973057,
                2522638740117423042,
                17137049738161235473,
                13176139598975768388,
                1166120501092890058,
                13609465226914230670,
                1875448353263386203,
                11743691709994844759,
                11061975822835649567,
                3910967390258776729,
                13577511757143681922,
                13827967505806592907,
                2051032579834721288,
                13464396192578309450,
                12800325352123601126,
                5662172040792660351,
                4668403599689432265,
                13491732430526501935,
                14157448317801936638,
                9727633765236455195,
                9719667367393442680,
                65474725188749034,
                541181075058713986,
                1362959920153019155,
                909082441655242031,
                10411762570666372898,
                11947296633429547042,
                2358875224077331054,
                18328103363666444110,
                9866396237081281387,
                8956667376401863125,
                3553975952663161681,
                14635072569697080107,
                15544243098851103691,
                1716867379502200816,
                8776276410599591217,
                15666047021340369577,
                18147839669015605124,
                5572968214423121059,
                3098271794032834805,
                9673511458079712674,
                16010905689567823114,
                16267201338365627673,
                3382936832397700934,
                14957330839370249294,
                2642620075223240608,
                1037193656622159510,
                9012211029341477206,
                11431820793941846484,
                8851179188286979417,
                1398798117665992200,
                10789412881967821793,
                9415310270308545401,
                3878698623022480586,
                5863277386811827416,
                7839335917417032151,
                2682939031547768972,
                13397811690933306272,
                6260148512861761963,
                250324816893616712,
                9085404939459661953,
                9291403781568194640,
                4943156069611799558,
                4012150457352746082,
                6743913241329187121,
                12017429880940425975,
                12929043722801309882,
                10207563294594880008,
                11011363281303738187,
                15710481255331302269,
                16065095211995167995,
                7729419910130307261,
                14687299092290139624,
                14212774840097619295,
                2696933303282739820,
                18380487173546723450,
                16340353901250449060,
                16974439742831865800,
                950395719691498193,
                14294722598295985858,
                9819629466618736444,
                5134213478899186274,
                15000357065088523575,
                10866834082882265182,
                5562209566130970606,
                11574047289359876875,
                14220066737185616618,
                7779431549476206578,
                6164322776781074672,
                12698699879532635460,
                9775735470622210670,
                3418783774094779723,
                11711261272918394032,
                130965081793297038,
                17951168177437733692,
                10982891057320326220,
                15446335221541982820,
                3245348832258032003,
                16936841548783354130,
                17845797372140897632,
                15738639878561184724,
                8036219019342274839,
                1452217306593250967,
                6524640126112677080,
                9487637664603922399,
                9816896718276865827,
                6660361176840108508,
                7155686979915548241,
                5363459444036855846,
                14211791847250927632,
                3758021753984352623,
                14892848306282510880,
                5622159652308652067,
                3013224027720379126,
                6593658810977588348,
                8737392006680055212,
                13098306207053606600,
                10321069935686406569,
                7922126603433897812,
                16345323120534317859,
                2051345375893155420,
                2642889174797452752,
                15490895676079176387,
                3421728094884350553,
                1403789733153795522,
                18095248776246397645,
                1141680856759996549,
                18100437303596844925,
                10084704427684498223,
                6594067795036706889,
                6973150786346631215,
                18179961484385983667,
                13461945752721759987,
                11607193624374158079,
                15870559753884484047,
                5909010369355025391,
                6576523373028484144,
                16932784188428621626,
                4532369146195665092,
                8439234310823545929,
                1340079530194594172,
                14887691618471650731,
                12957273899514368091,
                12528827801491392116,
                13543620352724873404,
                15027641031340635112,
                3497080345802388600,
                12532486649114668085,
                6567958427571461048,
                15524967599943619324,
                9687676309746098775,
                17837290175131112775,
                6562577965471546738,
                15069045445640377298,
                15845870183628003510,
                4882896822391168376,
                299291354580119732,
                1253349370757830450,
                4019946893924061554,
                4835687088435778153,
                560543554021366834,
                9073367421711646017,
                7210725233620419378,
                1298874732344651234,
                1572176424217571034,
                13459119860690231055,
                14894361022235972365,
                12479129811186551951,
                14014563421012570478,
                5045150035901968604,
                8669967307441751567,
                1905648106987249698,
                15427548300560765880,
                16929690218389356941,
                2257784955568148289,
                16671033335294376858,
                17747845134586030566,
                1166533040996500790,
                15593722083313803843,
                9396109408892612308,
                11499981530721896824,
                16970499445754392592,
                8812544874418820841,
                8398666252211387757,
                2704923715823740363,
                5314497299504523542,
                5160833323750080202,
                4489614019342255168,
                16950479847425158028,
                6792145380267486390,
                14740232145636169668,
                621507344077794685,
                3751460578157408141,
                8129084486654095901,
                10001447130583054783,
                12074020860950174718,
                4077799248444467034,
                6822188364356126108,
                600699026381065222,
                4751815635139776884,
                15367619962178612777,
                5104893889903277375,
                4661242055023047595,
                2154923710109688891,
                17429472085810639176,
                15724063920670689806,
                15216015843686884607,
                12102731527843294992,
                5923322163550614145,
                578844978773347240,
                10464767667513753088,
                4422886088759698238,
                16103295972923438952,
                11952617258811913803,
                6497649448616698694,
                7393337205916981343,
                18096118845330092692,
                9417521917459925024,
                10361801380161480586,
                12227883220061157093,
                12402861272457739535,
                16959542957232270118,
                3100109669354124792,
                8252021294756255295,
                6601544514524806222,
                2299228880186395084,
                17206687535661702962,
                11126365220307664227,
                14027475381759030281,
                13930713973102273385,
                15470781455897247423,
                11132792425729103210,
                12743536155285333682,
                18404252734721007574,
                4471349713039799575,
                1097648964215121725,
                3481722566194833071,
                17004865376724342545,
                4412003432895082804,
                7207210335443122075,
                11185557399280223037,
                17791545657588716507,
                15001384756479157715,
                7068904123591490511,
                15342280096772936670,
                2962241209356984664,
                11085614675955826760,
                17304628576563637894,
                13276820937207248747,
                8496585032541849567,
                16867719644448479323,
                3556128556368553119,
                4135507236401264417,
                8300362706308478874,
                12391805295598536754,
                212746375693666561,
                17668657797254926684,
                11472227395456264735,
                3346261809041337938,
                10494844376481252742,
                1039434633412734920,
                3537248641106173003,
                15722028107965146028,
                3957344489070663630,
                171520814732107885,
                15701539439111209401,
                16266903088897196785,
                8162180034652464302,
                751396451910146673,
                6284978536639446110,
                9475038153254288249,
                3366181839966720435,
                38511352552288459,
                5091650762875915484,
                15684434521130749089,
                3313175693887858849,
                3512862245616351559,
                5836672242124333533,
                7050189281309530509,
                6693570573437863470,
                1180929186779755232,
                13670569646431885520,
                12219171873489819591,
                4177796076111463096,
                12523988008299904003,
                2573301004696669683,
                14000913768617402515,
                1901514323758988374,
                15680204389035820646,
                4057543360329026133,
                10484558774556926484,
                16880563214620981386,
                9806529946108790649,
                64858070620446021,
                13058265791054866751,
                14208780393297513060,
                132444534110050230,
                18171930401783150790,
                3788701567251119654,
                8068556048675251486,
                7698349927972499523,
                4200940425301802259,
                5859679210997281495,
                1196917998587841402,
                293846739593611503,
                742395111532740587,
                7299921458271130779,
                4013584019577681930,
                13051907051578937053,
                15460522300561557538,
                11739346111540426460,
                6000392060961693921,
                2093513337710768996,
                2971782196230136142,
                14650675906567946197,
                13044422653436935289,
                5393154276677237552,
                9414604975305044529,
                14383902696216976975,
                3526706571033049996,
                10453515348838320881,
                1300027919459661427,
                18220108229306772305,
                6736376168692276304,
                14279821335279300725,
                1137238529965897445,
                10675553998207815811,
                10380811037741532194,
                8166481736795163463,
                1781828658320585086,
                12391009694347491828,
                917764047682931238,
                5513578507525454078,
                17200064132172477743,
                17870911035870211136,
                14336835701074601920,
                11250050928475332766,
                11433063323885652386,
                7083842474651714165,
                2491363132897582954,
                6822759191074732531,
                1975703285786766592,
                4664247346348091399,
                6200670517431733578,
                7625423344374743344,
                12752241152270105853,
                1792181265050143868,
                9684341866777183803,
                2519544645709853691,
                10916780105681637442,
                10344955420168669639,
                9942180203006299987,
                12084291513489188332,
                3948266540776100311,
                9154092837114088215,
                5346858599403446633,
                16526650534031004365,
                1000379885777408481,
                9962324609830950859,
                17358968356608808633,
                16077501442923026330,
                16373391819852480638,
                7285769258546345039,
                9258849101785002591,
                5742154400061770227,
                6000603993323989058,
                14647716329474708008,
                9759234782041772304,
                10140774772642173806,
                12417308406161902675,
                787743125041003665,
                16505409716496421861,
                13496346584666506754,
                6161155548184037990,
                6052945594671650378,
                3943721555742518361,
                1724411800658421894,
                5539278770081464366,
                15345354647968324153,
                11115348357173195281,
                14941897866359059705,
                13232123053046857932,
                10293530598380524712,
                6785485284866633954,
                2274573151581626897,
                7914418682634630883,
                9050484888129037258,
                5596144246310386066,
                16947403501604289983,
                4629498933991963153,
                12809658963208366731,
                15794201530500048648,
                13688081233074642819,
                13734318539544366298,
                17338613223770563244,
                8949201706050038962,
                7457293230715991355,
                3392962986871707645,
                2665311774547334828,
                14769130061609548831,
                14798608201523559143,
                4248204495368930031,
                2118780687916441347,
                7587741700881576871,
                14435287502544940055,
                5661100765027979237,
                16339955561566207846,
                8140260981243250155,
                8056431096984667176,
                7147782631749791333,
                2489306669890900889,
                4475802923302768991,
                16154685423804744738,
                7934898789690822040,
                6879516481280076285,
                9089400825108258886,
                17930990199515864021,
                14768387486978797965,
                18129863777024625000,
                10818394317143489199,
                15491447326600259816,
                16981166881989734434,
                16302476576745273904,
                6518490351683715742,
                8412288793616962739,
                6351653340417195445,
                13150568765377194334,
                13805040795477806115,
                5934147261272121494,
                1994383077577350943,
                16635675973993021427,
                12605935823956703560,
                17476993101523741670,
                6478809187431271696,
                2732525537481875395,
                9879991725842211511,
                6807932982002010721,
                7352337837011779903,
                2817471936912129199,
                13262278618157850043,
                10769791684153169602,
                16373519399949306992,
                2899723188139359765,
                10536102085057036916,
                6542494148698745845,
                8208357062363683807,
                7135654628053388596,
                9239547683832094589,
                5857943902095897523,
                2988676895399293972,
                10864744325417622458,
                5315919323785706256,
                17726255358609389283,
                548416274855739353,
                14341121194849694922,
                7541833390688838500,
                12990342594130240843,
                18010249397570629115,
                12270534655132211820,
                11089591099925577121,
                8904086294203345138,
                360354335597452271,
                15390384602644239058,
                9646385913080766351,
                7807991425432087787,
                9178671661816533936,
                14127492076118037414,
                2392162322212242991,
                14036941142798507711,
                6502240351436565564,
                7893899872624843337,
                14288594129943657110,
                14040928483141897687,
                1592502089611546139,
                16817286687244219231,
                1288452307437286948,
                14067519453346137912,
                10075887672168075195,
                16769912598561257346,
                15456884559283078393,
                8711311950138380878,
                17440519414852901585,
                4031578079792007763,
                17459032506776526507,
                6047439267180308295,
                13036838382774047407,
                5995888809898483332,
                5666540365944124863,
                9669239227306108902,
                13524757761279567441,
                9100467949589737947,
                10630607884662362720,
                7656432890734400844,
                14587775638685276238,
                13512825387557617884,
                7244019832094584382,
                14996712660826603119,
                15920594541485478011,
                18302888448007121275,
                2860853013354641768,
                16346890198034118818,
                3995008638584894259,
                15431133287252532218,
                12007621448583591973,
                12654616832949992670,
                17218932927375784084,
                5108339176271521031,
                1475805094516082093,
                12897404584908775312,
                12902004423289195678,
                632420305961082974,
                3939829721363017782,
                8159027656096857323,
                5289724432039220494,
                8105874082275331898,
                7058967362148243623,
                7084554565361601863,
                10713529750643129586,
                16053444010682289763,
                16789097964014031314,
                2561050582281530128,
                5001739943623024256,
                5563376828383980501,
                7029523460044084584,
                3734303822831180195,
                17604507458906309331,
                15410396100097854167,
                12072617536058855445,
                9343913967197010858,
                17464576053666161449,
                14222234943442423138,
                3029903299346511292,
                2439504202412922232,
                17909602716813870812,
                7163148054936239235,
                1203376532559407808,
                13308342071396905822,
                9659238717159340331,
                9298790467854663751,
                13879727520175998029,
                3540668590936352635,
                9319928603109010852,
                7135941863720734900,
                4755935102616669000,
                16186952942091333092,
                868417045309212144,
                4614771873250033976,
                12695043568719791838,
                11555375308916964213,
                7608623603624110550,
                2657517781272739364,
                16275540663546067547,
                13686833504204824152,
                12138906647042213779,
                10867610696309111099,
                14724502273153125344,
                16861685577613202946,
                8743302596210517799,
                14587474243501868159,
                2572487334058012273,
                603248717087965084,
                5059320325228404990,
                16238530369175832077,
                4519037757700474392,
                12378671739975925965,
                12407484273586632994,
                15313162588758450641,
                7715370120812724109,
                16244624049595468844,
                16509798106801620258,
                8670816985658531869,
                17161563202276681629,
                6760859082822229695,
                6807294103798050445,
                12102809409961330094,
                15733867019982720404,
                4500829814752383546,
                276727174991039753,
                7166981968731078898,
                6839153022681710561,
                9052935609363333850,
                180023864532346968,
                15036321241551927313,
                8004861917338601611,
                10433381600254688194,
                58897018650060884,
                12176800300096084156,
                1677464574753824642,
                9033479160050603494,
                15102533506708978663,
                14084179961157892415,
                15862200228768947299,
                13406816340465864482,
                4480435244557703388,
                9328135622344307924,
                8569086569674090389,
                117119659942779713,
                53818961148559993,
                17499848839935720391,
                9136357190515762310,
                9027105858395986940,
                17989890318601901443,
                18097540417946691193,
                14525610539500917353,
                2154941454922905428,
                5904339583092243168,
                13443891604996267,
                17266860992297653711,
                8744756382159865790,
                18024305737878696983,
                2546630949169889692,
                14096991573794098342,
                6037463192406505117,
                16222309559186427347,
                17465566247484999322,
                12659653001889101916,
                17145147704704336749,
                270028953681259627,
                10687721792402042274,
                7909714047373529261,
                1516466812193933964,
                18277293846935305743,
                16998707418464340047,
                10456180009273456482,
                10894339053818039134,
                7931553102963941505,
                5469019641482786787,
                15559448848625861221,
                13379845259726008039,
                15222258432394099771,
                9220502339943331578,
                11041492625928826773,
                13723142119880743788,
                13406366780219048563,
                16242021890137344914,
                3242681820103600264,
                7433022400269598514,
                13800053899816348927,
                10971582708599334044,
                16780079125754812721,
                7499355964098143157,
                6683080229097120149,
                4643838630497962703,
                2452976410357582431,
                13628797043976089046,
                12208582264105438774,
                10972823060808439197,
                5834946779865455467,
                7061987228048265540,
                1207502404391064867,
                4999557813060557947,
                6110696626660879360,
                16155026384042239167,
                14271653985173561903,
                15409968780617538981,
                12583134000526518362,
                14425350186270986364,
                4240634153006170791,
                15186331103475807077,
                3441220359690995595,
                7730377008237943835,
                4028688972105728738,
                2120709666343944557,
                15275844229510750384,
                18298631304947594122,
                13289696428719138891,
                12902251465979698800,
                17784642778353395327,
                8932078122697958597,
                2343784433129198521,
                2625325561581989570,
                6226848264428179755,
                9341106183231718139,
                9098935814035218596,
                7352506358300392706,
                2344104557164367420,
                12260380423821950355,
                5334772679936050516,
                4625031027053042238,
                16182107767362335138,
                16366374610370031102,
                4836384899789893883,
                7955273434480297668,
                6911864392544134089,
                16417538807637628387,
                7095497393911852501,
                13038578327075160812,
                3237033976078933963,
                12153345193878928470,
                8529208509214849086,
                3557796804144548743,
                4802066316596054332,
                15433632645238893171,
                7546572980385595878,
                11926731397057053166,
                12459060237931625488,
                14141685648156683211,
                7288151748928762022,
                4432041251568429780,
                15563480646640553472,
                16053290806405898796,
                15137506699232809093,
                8005443786172873610,
                2583037632942587270,
                17849394642972838924,
                14162302110936479805,
                2509458716192360405,
                5356417310559321341,
                13186254752172487693,
                11386136498514963040,
                17709391199907333437,
                10287520916427793726,
                16609632513291109233,
                2167580474521363269,
                12093450528168333086,
                7863381164011802024,
                3333719694645745323,
                11940256871115614834,
                15608703623597862105,
                12353758136546081384,
                6684006108125243153,
                12729735272407622593,
                5729918571797570178,
                836259910264806143,
                14839853436536328663,
                3503216322132876395,
                12081845346843956683,
                11452606043277089033,
                3278971557269095943,
                14460255683089324678,
                26212371609850095,
                5791996297577934908,
                11791872952518605557,
                4303537542168010174,
                16090584062354399534,
                4908264648595320164,
                7751540775663330461,
                6257171389619325857,
                10426410760581414335,
                17538696865214967217,
                7467505509612490667,
                9596349531846363101,
                10432645685361251072,
                4554200269336177591,
                14164090620513306305,
                12703411361203184202,
                10381679821480302262,
                17407993808772229676,
                13447212664492178323,
                6368763612994722491,
                6985317877139191413,
                3304210073159911148,
                11815333377693368073,
                5807352926666783905,
                15497939312564010706,
                6997355376158126843,
                8894976899204659904,
                6062763749482474444,
                12366982438149692528,
                5811941270446802002,
                1365585056061256612,
                13475218612693439913,
                3101226957898170052,
                7483152820787337316,
                11966132556998778392,
                15847594708317668544,
                8536570052230898519,
                17755327099574675677,
                4573875060013426627,
                10571694422987443237,
                13886938922370949941,
                13243541345182765869,
                2407150453734919381,
                3504658143291885737,
                7031357511394950972,
                10000971933511773390,
                13912116329153795479,
                15031415497603451939,
                4599320279014918550,
                11777085389799050565,
                10789465428143426974,
                13380525436974205384,
                481819144611018854,
                5450316677878014202,
                15246244400696673845,
                2539248742679727539,
                13402910818618694139,
                17131842085718107785,
                1254150638176089903,
                4995981122218138624,
                7087931693181888823,
                15049301175005363928,
                108059457248600792,
                6415264753538529207,
                8176266644263202488,
                9913473944910878449,
                1695436935634773000,
                17297145186449223950,
                12551494433741244515,
                9893572477404506151,
                5018441835213902689,
                2942536196417517781,
                16905131659211501459,
                6905710711318770295,
                7616899203962297179,
                14893845862286767619,
                9802013654239316989,
                17646160800663090718,
                688334719842856096,
                2007794688466179654,
                654395123253282904,
                8101085388166846547,
                129085352126721490,
                5656513500529597182,
                5068380157352571940,
                16823659606299635586,
                18201953606184755353,
                14292322613095109457,
                16683857805140576716,
                2453074754987128823,
            ],
        );
    }
}

#[test]
#[should_panic]
fn limbs_mod_fail_1() {
    limbs_mod(&[1, 2, 3], &[4]);
}

#[test]
#[should_panic]
fn limbs_mod_fail_2() {
    limbs_mod(&[1], &[4, 5]);
}

#[test]
#[should_panic]
fn limbs_mod_fail_3() {
    limbs_mod(&[1, 2, 3], &[4, 0]);
}

#[test]
#[should_panic]
fn limbs_mod_to_out_fail_1() {
    limbs_mod_to_out(&mut [10; 4], &[1, 2, 3], &[4]);
}

#[test]
#[should_panic]
fn limbs_mod_to_out_fail_2() {
    limbs_mod_to_out(&mut [10; 4], &[1], &[4, 5]);
}

#[test]
#[should_panic]
fn limbs_mod_to_out_fail_3() {
    limbs_mod_to_out(&mut [10; 4], &[1, 2, 3], &[4, 0]);
}

#[test]
fn test_mod() {
    let test = |s, t, remainder| {
        let u = Natural::from_str(s).unwrap();
        let v = Natural::from_str(t).unwrap();

        let mut x = u.clone();
        x.mod_assign(v.clone());
        assert!(x.is_valid());
        assert!(x.mod_is_reduced(&v));
        assert_eq!(x.to_string(), remainder);

        let mut x = u.clone();
        x.mod_assign(&v);
        assert!(x.is_valid());
        assert_eq!(x.to_string(), remainder);

        let r = u.clone().mod_op(v.clone());
        assert!(r.is_valid());
        assert_eq!(r.to_string(), remainder);

        let r = u.clone().mod_op(&v);
        assert!(r.is_valid());
        assert_eq!(r.to_string(), remainder);

        let r = (&u).mod_op(v.clone());
        assert!(r.is_valid());
        assert_eq!(r.to_string(), remainder);

        let r = (&u).mod_op(&v);
        assert!(r.is_valid());
        assert_eq!(r.to_string(), remainder);

        let mut x = u.clone();
        x %= v.clone();
        assert!(x.is_valid());
        assert_eq!(x.to_string(), remainder);

        let mut x = u.clone();
        x %= &v;
        assert!(x.is_valid());
        assert_eq!(x.to_string(), remainder);

        let r = u.clone() % v.clone();
        assert!(r.is_valid());
        assert_eq!(r.to_string(), remainder);

        let r = u.clone() % &v;
        assert!(r.is_valid());
        assert_eq!(r.to_string(), remainder);

        let r = &u % v.clone();
        assert!(r.is_valid());
        assert_eq!(r.to_string(), remainder);

        let r = &u % &v;
        assert!(r.is_valid());
        assert_eq!(r.to_string(), remainder);

        let num_u = BigUint::from_str(s).unwrap();
        let num_v = BigUint::from_str(t).unwrap();

        let r = num_u.mod_floor(&num_v);
        assert_eq!(r.to_string(), remainder);

        let r = num_u % num_v;
        assert_eq!(r.to_string(), remainder);

        let rug_u = rug::Integer::from_str(s).unwrap();
        let rug_v = rug::Integer::from_str(t).unwrap();

        let r = (rug_u.clone()).rem_floor(rug_v.clone());
        assert_eq!(r.to_string(), remainder);

        let r = rug_u % rug_v;
        assert_eq!(r.to_string(), remainder);

        assert_eq!(u.div_mod(v).1.to_string(), remainder);
    };
    test("0", "1", "0");
    test("0", "123", "0");
    test("1", "1", "0");
    test("123", "1", "0");
    test("123", "123", "0");
    test("123", "456", "123");
    test("456", "123", "87");
    test("4294967295", "4294967295", "0");
    test("4294967295", "4294967295", "0");
    test("1000000000000", "1", "0");
    test("1000000000000", "3", "1");
    test("1000000000000", "123", "100");
    test("1000000000000", "4294967295", "3567587560");
    test("1000000000000000000000000", "1", "0");
    test("1000000000000000000000000", "3", "1");
    test("1000000000000000000000000", "123", "37");
    test("1000000000000000000000000", "4294967295", "3167723695");
    test("1000000000000000000000000", "1234567890987", "530068894399");
    test(
        "100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000\
        0",
        "1234567890987654321234567890987654321",
        "779655053998040854338961591319296066",
    );
    test(
        "100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000\
        0",
        "316049380092839506236049380092839506176",
        "37816691783627670491375998320948925696",
    );
    test(
        "253640751230376270397812803167",
        "2669936877441",
        "1520301762334",
    );
    test(
        "3768477692975601",
        "11447376614057827956",
        "3768477692975601",
    );
    test("3356605361737854", "3081095617839357", "275509743898497");
    test(
        "1098730198198174614195",
        "953382298040157850476",
        "145347900158016763719",
    );
    test(
        "69738658860594537152875081748",
        "69738658860594537152875081748",
        "0",
    );
    test(
        "1000000000000000000000000",
        "1000000000000000000000000",
        "0",
    );
    test("0", "1000000000000000000000000", "0");
    test("123", "1000000000000000000000000", "123");
}

#[test]
#[should_panic]
fn mod_assign_fail() {
    Natural::from(10u32).mod_assign(Natural::ZERO);
}

#[test]
#[should_panic]
fn mod_assign_ref_fail() {
    Natural::from(10u32).mod_assign(&Natural::ZERO);
}

#[test]
#[should_panic]
fn mod_fail() {
    Natural::from(10u32).mod_op(Natural::ZERO);
}

#[test]
#[should_panic]
fn mod_val_ref_fail() {
    Natural::from(10u32).mod_op(&Natural::ZERO);
}

#[test]
#[should_panic]
fn mod_ref_val_fail() {
    (&Natural::from(10u32)).mod_op(Natural::ZERO);
}

#[test]
#[should_panic]
fn mod_ref_ref_fail() {
    (&Natural::from(10u32)).mod_op(&Natural::ZERO);
}

#[test]
#[should_panic]
fn rem_assign_fail() {
    let mut n = Natural::from(10u32);
    n %= Natural::ZERO;
}

#[test]
#[should_panic]
fn div_rem_assign_ref_fail() {
    let mut n = Natural::from(10u32);
    n %= &Natural::ZERO;
}

#[test]
#[should_panic]
#[allow(unused_must_use, clippy::unnecessary_operation)]
fn rem_fail() {
    Natural::from(10u32) % Natural::ZERO;
}

#[test]
#[should_panic]
#[allow(unused_must_use, clippy::unnecessary_operation)]
fn rem_val_ref_fail() {
    Natural::from(10u32) % &Natural::ZERO;
}

#[test]
#[should_panic]
#[allow(unused_must_use, clippy::unnecessary_operation)]
fn rem_ref_val_fail() {
    &Natural::from(10u32) % Natural::ZERO;
}

#[test]
#[should_panic]
#[allow(unused_must_use, clippy::unnecessary_operation)]
fn rem_ref_ref_fail() {
    &Natural::from(10u32) % &Natural::ZERO;
}

#[test]
fn test_neg_mod() {
    let test = |s, t, remainder| {
        let u = Natural::from_str(s).unwrap();
        let v = Natural::from_str(t).unwrap();

        let mut x = u.clone();
        x.neg_mod_assign(v.clone());
        assert!(x.is_valid());
        assert!(x.mod_is_reduced(&v));
        assert_eq!(x.to_string(), remainder);

        let mut x = u.clone();
        x.neg_mod_assign(&v);
        assert!(x.is_valid());
        assert_eq!(x.to_string(), remainder);

        let r = u.clone().neg_mod(v.clone());
        assert!(r.is_valid());
        assert_eq!(r.to_string(), remainder);

        let r = u.clone().neg_mod(&v);
        assert!(r.is_valid());
        assert_eq!(r.to_string(), remainder);

        let r = (&u).neg_mod(v.clone());
        assert!(r.is_valid());
        assert_eq!(r.to_string(), remainder);

        let r = (&u).neg_mod(&v);
        assert!(r.is_valid());
        assert_eq!(r.to_string(), remainder);

        let r = rug_neg_mod(
            rug::Integer::from_str(s).unwrap(),
            rug::Integer::from_str(t).unwrap(),
        );
        assert_eq!(r.to_string(), remainder);

        assert_eq!(u.ceiling_div_neg_mod(v).1.to_string(), remainder);
    };
    test("0", "1", "0");
    test("0", "123", "0");
    test("1", "1", "0");
    test("123", "1", "0");
    test("123", "123", "0");
    test("123", "456", "333");
    test("456", "123", "36");
    test("4294967295", "1", "0");
    test("4294967295", "4294967295", "0");
    test("1000000000000", "1", "0");
    test("1000000000000", "3", "2");
    test("1000000000000", "123", "23");
    test("1000000000000", "4294967295", "727379735");
    test("1000000000000000000000000", "1", "0");
    test("1000000000000000000000000", "3", "2");
    test("1000000000000000000000000", "123", "86");
    test("1000000000000000000000000", "4294967295", "1127243600");
    test("1000000000000000000000000", "1234567890987", "704498996588");
    test(
        "10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000\
         00",
        "1234567890987654321234567890987654321",
        "454912836989613466895606299668358255",
    );
    test(
        "10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000\
         00",
        "316049380092839506236049380092839506176",
        "278232688309211835744673381771890580480",
    );
    test(
        "253640751230376270397812803167",
        "2669936877441",
        "1149635115107",
    );
    test(
        "3768477692975601",
        "11447376614057827956",
        "11443608136364852355",
    );
    test("3356605361737854", "3081095617839357", "2805585873940860");
    test(
        "1098730198198174614195",
        "953382298040157850476",
        "808034397882141086757",
    );
    test(
        "69738658860594537152875081748",
        "69738658860594537152875081748",
        "0",
    );
    test(
        "1000000000000000000000000",
        "1000000000000000000000000",
        "0",
    );
    test("0", "1000000000000000000000000", "0");
    test(
        "123",
        "1000000000000000000000000",
        "999999999999999999999877",
    );
}

#[test]
#[should_panic]
fn neg_mod_assign_fail() {
    Natural::from(10u32).neg_mod_assign(Natural::ZERO);
}

#[test]
#[should_panic]
fn neg_mod_assign_ref_fail() {
    Natural::from(10u32).neg_mod_assign(&Natural::ZERO);
}

#[test]
#[should_panic]
fn neg_mod_fail() {
    Natural::from(10u32).neg_mod(Natural::ZERO);
}

#[test]
#[should_panic]
fn neg_mod_val_ref_fail() {
    Natural::from(10u32).neg_mod(&Natural::ZERO);
}

#[test]
#[should_panic]
fn neg_mod_ref_val_fail() {
    (&Natural::from(10u32)).neg_mod(Natural::ZERO);
}

#[test]
#[should_panic]
fn neg_mod_ref_ref_fail() {
    (&Natural::from(10u32)).neg_mod(&Natural::ZERO);
}

#[test]
fn limbs_mod_limb_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    unsigned_vec_unsigned_pair_gen_var_22().test_properties_with_config(&config, |(ns, d)| {
        let r = limbs_mod_limb::<DoubleLimb, Limb>(&ns, d);
        assert_eq!(Natural::from_limbs_asc(&ns) % Natural::from(d), r);
        assert_eq!(
            limbs_mod_limb_any_leading_zeros_1::<DoubleLimb, Limb>(&ns, d),
            r
        );
        assert_eq!(
            limbs_mod_limb_any_leading_zeros_2::<DoubleLimb, Limb>(&ns, d),
            r
        );
        assert_eq!(limbs_mod_limb_alt_1::<DoubleLimb, Limb>(&ns, d), r);
        assert_eq!(limbs_mod_limb_alt_2::<DoubleLimb, Limb>(&ns, d), r);
        assert_eq!(limbs_mod_limb_alt_3(&ns, d), r);
    });
}

#[test]
fn limbs_mod_limb_small_normalized_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    unsigned_vec_unsigned_pair_gen_var_26().test_properties_with_config(&config, |(ns, d)| {
        let r = limbs_mod_limb_small_normalized::<DoubleLimb, Limb>(&ns, d);
        assert_eq!(r, Natural::from_limbs_asc(&ns) % Natural::from(d));
        if ns.len() == 1 {
            assert_eq!(r, ns[0] % d);
        } else {
            assert_eq!(r, limbs_mod_limb::<DoubleLimb, Limb>(&ns, d));
        }
    });
}

#[test]
fn limbs_mod_limb_small_unnormalized_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    unsigned_vec_unsigned_pair_gen_var_27().test_properties_with_config(&config, |(ns, d)| {
        let r = limbs_mod_limb_small_unnormalized::<DoubleLimb, Limb>(&ns, d);
        assert_eq!(
            r,
            limbs_mod_limb_at_least_1_leading_zero::<DoubleLimb, Limb>(&ns, d)
        );
        assert_eq!(r, Natural::from_limbs_asc(&ns) % Natural::from(d));
        if ns.len() == 1 {
            assert_eq!(r, ns[0] % d);
        } else {
            assert_eq!(r, limbs_mod_limb::<DoubleLimb, Limb>(&ns, d));
        }
    });
}

#[test]
fn limbs_mod_limb_at_least_2_leading_zeros_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    unsigned_vec_unsigned_pair_gen_var_28().test_properties_with_config(&config, |(ns, d)| {
        let r = limbs_mod_limb_at_least_2_leading_zeros::<DoubleLimb, Limb>(&ns, d);
        assert_eq!(r, Natural::from_limbs_asc(&ns) % Natural::from(d));
        if ns.len() == 1 {
            assert_eq!(r, ns[0] % d);
        } else {
            assert_eq!(r, limbs_mod_limb::<DoubleLimb, Limb>(&ns, d));
        }
    });
}

#[test]
fn limbs_mod_three_limb_by_two_limb_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    unsigned_sextuple_gen_var_2().test_properties_with_config(
        &config,
        |(n_2, n_1, n_0, d_1, d_0, inverse)| {
            let r = limbs_mod_three_limb_by_two_limb(n_2, n_1, n_0, d_1, d_0, inverse);
            verify_limbs_mod_three_limb_by_two_limb(n_2, n_1, n_0, d_1, d_0, r);
        },
    );
}

#[test]
fn limbs_mod_by_two_limb_normalized_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    unsigned_vec_pair_gen_var_18().test_properties_with_config(&config, |(ns, ds)| {
        let (r_0, r_1) = limbs_mod_by_two_limb_normalized(&ns, &ds);
        verify_limbs_mod_by_two_limb_normalized(&ns, &ds, r_0, r_1);
    });
}

#[test]
fn limbs_mod_schoolbook_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    unsigned_vec_unsigned_vec_unsigned_triple_gen_var_17().test_properties_with_config(
        &config,
        |(mut ns, ds, inverse)| {
            let ns_old = ns.clone();
            limbs_mod_schoolbook(&mut ns, &ds, inverse);
            verify_limbs_mod_1(&ns_old, &ds, &ns);
        },
    );
}

#[test]
fn limbs_mod_divide_and_conquer_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 128);
    config.insert("mean_stripe_n", 64 << Limb::LOG_WIDTH);
    large_type_gen_var_12().test_properties_with_config(
        &config,
        |(mut qs, mut ns, ds, inverse)| {
            let ns_old = ns.clone();
            limbs_mod_divide_and_conquer(&mut qs, &mut ns, &ds, inverse);
            verify_limbs_mod_1(&ns_old, &ds, &ns);
        },
    );
}

#[test]
fn limbs_mod_barrett_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 512);
    config.insert("mean_stripe_n", 128 << Limb::LOG_WIDTH);
    unsigned_vec_quadruple_gen_var_5().test_properties_with_config(
        &config,
        |(mut qs, mut rs, ns, ds)| {
            let rs_old = rs.clone();
            let mut scratch = vec![0; limbs_div_mod_barrett_scratch_len(ns.len(), ds.len())];
            limbs_mod_barrett(&mut qs, &mut rs, &ns, &ds, &mut scratch);
            verify_limbs_mod_2(&rs_old, &ns, &ds, &rs);
        },
    );
}

fn verify_limbs_mod_3(ns: &[Limb], ds: &[Limb], rs: &[Limb]) {
    let n = Natural::from_limbs_asc(ns);
    let d = Natural::from_limbs_asc(ds);
    let expected_r = n % &d;
    let r = Natural::from_limbs_asc(rs);
    assert_eq!(r, expected_r);
    assert!(r < d);
    assert_eq!(rs.len(), ds.len());
}

#[test]
fn limbs_mod_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 512);
    config.insert("mean_stripe_n", 128 << Limb::LOG_WIDTH);
    unsigned_vec_pair_gen_var_11().test_properties_with_config(&config, |(ns, ds)| {
        verify_limbs_mod_3(&ns, &ds, &limbs_mod(&ns, &ds));
    });
}

#[test]
fn limbs_mod_to_out_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 512);
    config.insert("mean_stripe_n", 128 << Limb::LOG_WIDTH);
    unsigned_vec_triple_gen_var_57().test_properties_with_config(&config, |(mut rs, ns, ds)| {
        let rs_old = rs.clone();
        limbs_mod_to_out(&mut rs, &ns, &ds);
        verify_limbs_mod_2(&rs_old, &ns, &ds, &rs);
    });
}

#[allow(clippy::needless_pass_by_value)]
fn mod_properties_helper(x: Natural, y: Natural) {
    let mut mut_x = x.clone();
    mut_x.mod_assign(&y);
    assert!(mut_x.is_valid());
    let remainder = mut_x;
    assert!(remainder.mod_is_reduced(&y));

    let mut mut_x = x.clone();
    mut_x.mod_assign(y.clone());
    let remainder_alt = mut_x;
    assert!(remainder_alt.is_valid());
    assert_eq!(remainder_alt, remainder);

    let remainder_alt = (&x).mod_op(&y);
    assert!(remainder_alt.is_valid());
    assert_eq!(remainder_alt, remainder);

    let remainder_alt = (&x).mod_op(y.clone());
    assert!(remainder_alt.is_valid());
    assert_eq!(remainder_alt, remainder);

    let remainder_alt = x.clone().mod_op(&y);
    assert!(remainder_alt.is_valid());
    assert_eq!(remainder_alt, remainder);

    let remainder_alt = x.clone().mod_op(y.clone());
    assert!(remainder_alt.is_valid());
    assert_eq!(remainder_alt, remainder);

    let mut remainder_alt = x.clone();
    remainder_alt %= &y;
    assert!(remainder_alt.is_valid());
    assert_eq!(remainder_alt, remainder);

    let mut remainder_alt = x.clone();
    remainder_alt %= y.clone();
    assert!(remainder_alt.is_valid());
    assert_eq!(remainder_alt, remainder);

    let remainder_alt = &x % &y;
    assert!(remainder_alt.is_valid());
    assert_eq!(remainder_alt, remainder);

    let remainder_alt = &x % y.clone();
    assert!(remainder_alt.is_valid());
    assert_eq!(remainder_alt, remainder);

    let remainder_alt = x.clone() % &y;
    assert!(remainder_alt.is_valid());
    assert_eq!(remainder_alt, remainder);

    let remainder_alt = x.clone() % y.clone();
    assert!(remainder_alt.is_valid());
    assert_eq!(remainder_alt, remainder);

    let remainder_alt = (&x).div_mod(&y).1;
    assert_eq!(remainder_alt, remainder);

    let num_remainder = BigUint::from(&x).mod_floor(&BigUint::from(&y));
    assert_eq!(Natural::from(&num_remainder), remainder);

    let num_remainder = BigUint::from(&x) % &BigUint::from(&y);
    assert_eq!(Natural::from(&num_remainder), remainder);

    let rug_remainder = rug::Integer::from(&x).rem_floor(rug::Integer::from(&y));
    assert_eq!(Natural::exact_from(&rug_remainder), remainder);

    let rug_remainder = rug::Integer::from(&x) % rug::Integer::from(&y);
    assert_eq!(Natural::exact_from(&rug_remainder), remainder);

    assert!(remainder < y);
    assert_eq!(&remainder % y, remainder);
}

#[test]
fn mod_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_bits_n", 2048);
    config.insert("mean_stripe_n", 256 << Limb::LOG_WIDTH);
    natural_pair_gen_var_5().test_properties_with_config(&config, |(x, y)| {
        mod_properties_helper(x, y);
    });

    natural_pair_gen_var_6().test_properties_with_config(&config, |(x, y)| {
        mod_properties_helper(x, y);
    });

    natural_gen().test_properties(|n| {
        assert_eq!(n % Natural::ONE, 0);
    });

    natural_gen_var_2().test_properties(|ref n| {
        assert_eq!(n % n, 0);
        assert_eq!(Natural::ZERO % n, 0);
        if *n > 1 {
            assert_eq!(Natural::ONE % n, 1);
        }
    });

    natural_triple_gen_var_4().test_properties(|(ref x, ref y, ref z)| {
        assert_eq!((x + y) % z, (x % z + y % z) % z);
        assert_eq!(x * y % z, (x % z) * (y % z) % z);
    });

    unsigned_pair_gen_var_12::<Limb, Limb>().test_properties(|(x, y)| {
        assert_eq!(Natural::from(x) % Natural::from(y), x % y);
    });
}

#[allow(clippy::needless_pass_by_value)]
fn neg_mod_properties_helper(x: Natural, y: Natural) {
    let mut mut_x = x.clone();
    mut_x.neg_mod_assign(&y);
    assert!(mut_x.is_valid());
    let remainder = mut_x;
    assert!(remainder.mod_is_reduced(&y));

    let mut mut_x = x.clone();
    mut_x.neg_mod_assign(y.clone());
    let remainder_alt = mut_x;
    assert!(remainder_alt.is_valid());
    assert_eq!(remainder_alt, remainder);

    let remainder_alt = (&x).neg_mod(&y);
    assert!(remainder_alt.is_valid());
    assert_eq!(remainder_alt, remainder);

    let remainder_alt = (&x).neg_mod(y.clone());
    assert!(remainder_alt.is_valid());
    assert_eq!(remainder_alt, remainder);

    let remainder_alt = x.clone().neg_mod(&y);
    assert!(remainder_alt.is_valid());
    assert_eq!(remainder_alt, remainder);

    let remainder_alt = x.clone().neg_mod(y.clone());
    assert!(remainder_alt.is_valid());
    assert_eq!(remainder_alt, remainder);

    let remainder_alt = (&x).neg_mod(&y);
    assert_eq!(remainder_alt, remainder);

    let rug_remainder = rug_neg_mod(rug::Integer::from(&x), rug::Integer::from(&y));
    assert_eq!(Natural::exact_from(&rug_remainder), remainder);

    assert!(remainder < y);
}

#[test]
fn neg_mod_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_bits_n", 2048);
    config.insert("mean_stripe_n", 256 << Limb::LOG_WIDTH);
    natural_pair_gen_var_5().test_properties_with_config(&config, |(x, y)| {
        neg_mod_properties_helper(x, y);
    });

    natural_pair_gen_var_6().test_properties_with_config(&config, |(x, y)| {
        neg_mod_properties_helper(x, y);
    });

    natural_gen().test_properties(|n| {
        assert_eq!(n.neg_mod(Natural::ONE), 0);
    });

    natural_gen_var_2().test_properties(|n| {
        assert_eq!((&n).neg_mod(&n), 0);
        assert_eq!(Natural::ZERO.neg_mod(&n), 0);
        assert_eq!(Natural::ONE.neg_mod(&n), n - Natural::ONE);
    });

    natural_triple_gen_var_4().test_properties(|(ref x, ref y, ref z)| {
        assert_eq!((x + y).neg_mod(z), (x % z + y % z).neg_mod(z));
        assert_eq!((x * y).neg_mod(z), ((x % z) * (y % z)).neg_mod(z));
    });

    unsigned_pair_gen_var_12::<Limb, Limb>().test_properties(|(x, y)| {
        assert_eq!(Natural::from(x).neg_mod(Natural::from(y)), x.neg_mod(y));
    });
}
