// Copyright © 2025 Mikhail Hogrefe
//
// This file is part of Malachite.
//
// Malachite is free software: you can redistribute it and/or modify it under the terms of the GNU
// Lesser General Public License (LGPL) as published by the Free Software Foundation; either version
// 3 of the License, or (at your option) any later version. See <https://www.gnu.org/licenses/>.

use malachite_base::num::arithmetic::traits::{
    CoprimeWith, DivExact, DivisibleBy, Gcd, GcdAssign, Lcm, ModPowerOf2,
};
use malachite_base::num::basic::integers::PrimitiveInt;
use malachite_base::num::basic::traits::{One, Zero};
use malachite_base::num::conversion::traits::ExactFrom;
use malachite_base::test_util::generators::common::{GenConfig, TINY_LIMIT};
use malachite_base::test_util::generators::{
    unsigned_gen_var_11, unsigned_pair_gen_var_27, unsigned_quadruple_gen_var_11,
    unsigned_vec_unsigned_pair_gen_var_23,
};
use malachite_nz::natural::Natural;
use malachite_nz::natural::arithmetic::gcd::half_gcd::{
    HalfGcdMatrix, limbs_gcd_div, limbs_gcd_reduced, limbs_half_gcd_matrix_1_mul_vector,
};
use malachite_nz::natural::arithmetic::gcd::limbs_gcd_limb;
use malachite_nz::natural::arithmetic::gcd::matrix_2_2::{
    limbs_matrix_2_2_mul, limbs_matrix_mul_2_2_scratch_len,
};
use malachite_nz::platform::Limb;
use malachite_nz::test_util::generators::{
    large_type_gen_var_5, large_type_gen_var_6, large_type_gen_var_7, large_type_gen_var_8,
    natural_gen, natural_pair_gen, natural_pair_gen_var_4, natural_triple_gen,
    unsigned_vec_pair_gen_var_10,
};
#[cfg(feature = "32_bit_limbs")]
use malachite_nz::test_util::natural::arithmetic::gcd::half_gcd_matrix_create;
use malachite_nz::test_util::natural::arithmetic::gcd::{
    OwnedHalfGcdMatrix, gcd_binary_nz, gcd_euclidean_nz, half_gcd_matrix_1_to_naturals,
    half_gcd_matrix_to_naturals, limbs_gcd_div_alt, limbs_gcd_div_naive,
};
use num::BigUint;
use num::Integer as rug_integer;
use std::str::FromStr;

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_gcd_limb() {
    let test = |xs, y, out| {
        assert_eq!(limbs_gcd_limb(xs, y), out);
    };
    test(&[0, 1], 3, 1);
    test(&[0, 1], 4, 4);
    test(&[3744702173, 1327092413], 1932313754, 7);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_half_gcd_matrix_init() {
    let test = |n, out| {
        let scratch_len = HalfGcdMatrix::min_init_scratch(n);
        assert_eq!(OwnedHalfGcdMatrix::init(n, vec![0; scratch_len]), out);
    };
    test(
        1,
        OwnedHalfGcdMatrix {
            data: vec![1, 0, 0, 0, 0, 0, 1, 0],
            s: 2,
            two_s: 4,
            three_s: 6,
            n: 1,
        },
    );
    test(
        3,
        OwnedHalfGcdMatrix {
            data: vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0],
            s: 3,
            two_s: 6,
            three_s: 9,
            n: 1,
        },
    );
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_half_gcd_matrix_update_q() {
    let test = |mut m: OwnedHalfGcdMatrix, qs: &[Limb], column, m_after| {
        let mut scratch = vec![0; m.update_q_scratch_len(qs.len())];
        m.update_q(qs, column, &mut scratch);
        assert_eq!(m, m_after);
    };
    // - qs_len > 1
    // - n + qs_len > self.n
    // - reached nonzero limb
    // - zero carry
    test(
        half_gcd_matrix_create(
            16,
            8,
            vec![
                3364358997, 3754657515, 2983848742, 3936755874, 1784338974, 2546784265, 1325228501,
                2948540251, 0, 0, 0, 0, 0, 0, 0, 0, 48883882, 3336115747, 2904559986, 3732488562,
                2492471725, 1228174159, 2572912965, 1596092594, 0, 0, 0, 0, 0, 0, 0, 0, 2193991530,
                2899278504, 3717617329, 1249076698, 879590153, 4210532297, 3303769392, 1147691304,
                0, 0, 0, 0, 0, 0, 0, 0, 3624392894, 1881877405, 1728780505, 931615955, 1096404509,
                1326003135, 370549396, 1987183422, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ),
        &[851586836, 2796582944, 2985872407],
        0,
        half_gcd_matrix_create(
            16,
            11,
            vec![
                2035904157, 337140004, 3757354024, 670541357, 1156551026, 2954076405, 2306281186,
                901391702, 26359260, 3840226876, 1109607712, 0, 0, 0, 0, 0, 48883882, 3336115747,
                2904559986, 3732488562, 2492471725, 1228174159, 2572912965, 1596092594, 0, 0, 0, 0,
                0, 0, 0, 0, 808024130, 607404974, 3634401411, 320381711, 4246353409, 2141048780,
                3541800429, 3925651923, 2116051089, 3713928088, 1381495070, 0, 0, 0, 0, 0,
                3624392894, 1881877405, 1728780505, 931615955, 1096404509, 1326003135, 370549396,
                1987183422, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ),
    );
    // - qs_len == 1
    test(
        half_gcd_matrix_create(
            3,
            1,
            vec![1918546714, 0, 0, 3992705367, 0, 0, 4279371403, 0, 0, 3759197804, 0, 0],
        ),
        &[3543156725],
        1,
        half_gcd_matrix_create(
            3,
            2,
            vec![
                1918546714, 0, 0, 2304079673, 1582715589, 0, 4279371403, 0, 0, 3341917811,
                3530290808, 0,
            ],
        ),
    );
    // - didn't reach nonzero limb
    test(
        half_gcd_matrix_create(
            11,
            5,
            vec![
                0, 4293918720, 4294967295, 134217727, 0, 0, 0, 0, 0, 0, 0, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 0, 0, 0, 0, 0, 0, 268435455, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 4294967295, 4294967295, 134217727, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ),
        &[4294967295, 4294967295],
        1,
        half_gcd_matrix_create(
            11,
            6,
            vec![
                0, 4293918720, 4294967295, 134217727, 0, 0, 0, 0, 0, 0, 0, 4294967295, 1048575, 0,
                4159700992, 4294967295, 134217728, 0, 0, 0, 0, 0, 268435455, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 4026531840, 4294967295, 402653182, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ),
    );
    // - nonzero carry
    test(
        half_gcd_matrix_create(
            26,
            14,
            vec![
                4294967295, 536870911, 0, 0, 0, 3758096384, 4294967295, 4294967295, 4294967295,
                4294967295, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 524287, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 4294967288, 4294967295, 4294967295, 4294967295, 4294967295, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ),
        &[4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295],
        1,
        half_gcd_matrix_create(
            26,
            18,
            vec![
                4294967295, 536870911, 0, 0, 0, 3758096384, 4294967295, 4294967295, 4294967295,
                4294967295, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3758096384,
                4294967295, 4294967295, 4294967295, 536870911, 0, 4294967295, 536870911, 0,
                4294967295, 4294967295, 3758096383, 4294967295, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                524287, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
                4294443007, 4294967295, 4294967287, 4294967295, 4294967295, 4294967295, 4294967295,
                524288, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ),
    );
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_matrix_2_2_mul() {
    let test = |xs_len,
                xs00: &[Limb],
                xs01: &[Limb],
                xs10: &[Limb],
                xs11: &[Limb],
                ys00: &[Limb],
                ys01: &[Limb],
                ys10: &[Limb],
                ys11: &[Limb],
                xs00_out: &[Limb],
                xs01_out: &[Limb],
                xs10_out: &[Limb],
                xs11_out: &[Limb]| {
        let x00 = Natural::from_limbs_asc(&xs00[..xs_len]);
        let x01 = Natural::from_limbs_asc(&xs01[..xs_len]);
        let x10 = Natural::from_limbs_asc(&xs10[..xs_len]);
        let x11 = Natural::from_limbs_asc(&xs11[..xs_len]);
        let mut xs00 = xs00.to_vec();
        let mut xs01 = xs01.to_vec();
        let mut xs10 = xs10.to_vec();
        let mut xs11 = xs11.to_vec();
        let mut scratch = vec![0; limbs_matrix_mul_2_2_scratch_len(xs_len, ys00.len())];
        limbs_matrix_2_2_mul(
            &mut xs00,
            &mut xs01,
            &mut xs10,
            &mut xs11,
            xs_len,
            ys00,
            ys01,
            ys10,
            ys11,
            &mut scratch,
        );
        assert_eq!(xs00, xs00_out);
        assert_eq!(xs01, xs01_out);
        assert_eq!(xs10, xs10_out);
        assert_eq!(xs11, xs11_out);

        let y00 = Natural::from_limbs_asc(ys00);
        let y01 = Natural::from_limbs_asc(ys01);
        let y10 = Natural::from_limbs_asc(ys10);
        let y11 = Natural::from_limbs_asc(ys11);
        let z00 = Natural::from_limbs_asc(&xs00);
        let z01 = Natural::from_limbs_asc(&xs01);
        let z10 = Natural::from_limbs_asc(&xs10);
        let z11 = Natural::from_limbs_asc(&xs11);
        assert_eq!(z00, &x00 * &y00 + &x01 * &y10);
        assert_eq!(z01, x00 * &y01 + x01 * &y11);
        assert_eq!(z10, &x10 * y00 + &x11 * y10);
        assert_eq!(z11, x10 * y01 + x11 * y11);
    };
    // - small arguments
    // - small arguments, xs_len >= ys_len
    test(
        6,
        &[
            1824026683, 855383832, 2690657440, 2097392823, 1744856106, 145210032, 2603321546,
            3714268226, 2572752468, 3552024211, 2105680113, 94008293, 3599160935,
        ],
        &[
            2100861089, 4058927494, 3716614958, 1374886937, 1422228766, 4125067565, 2659645894,
            3516145724, 333198217, 178588388, 4107267309, 1484604263, 2137933301,
        ],
        &[
            1307794677, 4133495830, 1895462954, 3236069911, 3579790936, 2823361555, 1496151737,
            2535182824, 534572820, 3734537343, 1327464147, 146259718, 609826618,
        ],
        &[
            2641001264, 3832561802, 3044312402, 2136702112, 2248874160, 4213317499, 3447231898,
            3536677714, 1313296525, 2266853686, 2052041169, 347717049, 3434352423,
        ],
        &[17796630, 1709267377, 3881785007, 955800997, 1047963095, 1178960613],
        &[3543460331, 2385517496, 2086915063, 4055235164, 948436119, 2870150085],
        &[773552019, 2621197289, 344580017, 3644778487, 274842105, 3931875648],
        &[3510151397, 4026412532, 4195503381, 682501213, 1600717006, 4252677869],
        &[
            2986138245, 1099838926, 919425337, 2682286770, 2754856871, 2818192521, 1654309759,
            3195744726, 1787512677, 270014583, 3333887406, 3816198933, 0,
        ],
        &[
            343075118, 2459980459, 1480149398, 379857868, 2605135184, 4087866241, 313289235,
            2201953346, 57359095, 3003278162, 1714985241, 4181488913, 0,
        ],
        &[
            2571956126, 963369470, 3125970281, 2288117049, 561041416, 923295190, 2931384231,
            3709683058, 128286972, 2292494833, 488801956, 337168684, 1,
        ],
        &[
            2992440279, 2112449146, 2902973305, 109063322, 1501017829, 475736264, 25775141,
            3410631721, 3956905026, 3154770531, 1888093339, 1763601187, 1,
        ],
    );
    // - small arguments, xs_len < ys_len
    test(
        2,
        &[
            342664039, 3864205283, 1256491723, 1548927821, 721895559, 1910261306, 1235301118,
            799978473, 549823983, 2512819824, 2068299963, 2595078570, 2035992824, 1064296193,
            3249660883,
        ],
        &[
            3453614628, 3979759903, 3415299769, 13499660, 1318795951, 4144584485, 2603705982,
            1273366949, 1110716009, 4026346288, 4104893234, 1485704330, 993867679, 3893686890,
            3439773462,
        ],
        &[
            1413063068, 1416612149, 250032189, 2057599093, 512591148, 2867423455, 157645789,
            33515858, 259890804, 2083013527, 2993123902, 3443423003, 2587947916, 2060570216,
            358183429,
        ],
        &[
            3771316904, 1123102423, 3617951287, 3593485506, 3547215643, 4279031677, 3253560754,
            4217820186, 342159470, 3956944812, 4058756711, 3961349923, 734903752, 3497757471,
            2783111581,
        ],
        &[
            560210975, 3955544118, 1525693335, 2423056590, 1672468134, 1425293729, 3576282506,
            2808604793, 2173831628, 3419991459, 1679121946, 1623263180,
        ],
        &[
            269931918, 1859534151, 1418011147, 3930467901, 3120401510, 2878732263, 2440621113,
            3698526898, 1960607269, 766485754, 1276737094, 1400250704,
        ],
        &[
            31094149, 3099789406, 924340153, 1040795176, 864813453, 1273500430, 1433881359,
            3857073234, 725256613, 302279104, 1991322247, 3304275709,
        ],
        &[
            1816819354, 1423518943, 1640383514, 2897037119, 3299791289, 2007192829, 3211337188,
            141201204, 2347531699, 2806328153, 4130308450, 1398383850,
        ],
        &[
            2476191277, 2109850628, 475699680, 1695888498, 997959785, 3844200037, 3615832345,
            3189739607, 4090633814, 4172440182, 2429164332, 1687921897, 3000987025, 227266471, 1,
        ],
        &[
            4293289930, 3312097786, 3225023610, 1814982734, 4123961773, 3202664630, 3288402967,
            3423618017, 2648236802, 1235962392, 3785214889, 4142071340, 2656497128, 2555569668, 0,
        ],
        &[
            550585132, 1958217637, 1079695233, 2070051690, 1642769243, 2223520962, 2790719352,
            1102872554, 1250178031, 2079404531, 2578751592, 3643898046, 2569241678, 1399445906, 0,
        ],
        &[
            1489643928, 2454326491, 2000310088, 504396801, 1710204547, 859777848, 2758305889,
            1268262193, 931603943, 2447640352, 2661054782, 1778884540, 664908352, 827512809, 0,
        ],
    );
    // - large arguments
    // - !x11_sign first
    // - !x01_sign first && *xs01_lo_last == 0
    // - t0_sign first
    // - *t0_last == 0
    // - x01_sign == t0_sign
    // - !t0_sign second && *t0_last == 0
    // - !x01_sign second
    // - !x11_sign second
    // - !t0_sign third
    test(
        13,
        &[
            2327266888, 159606339, 3865198656, 559105571, 2679446641, 3007100930, 1196374243,
            2596728943, 1430648584, 25111289, 3084098662, 482968659, 31861624, 1410839754,
            2868430508, 2675304181, 2820971697, 8433363, 356205126, 104388181, 4288829918,
            3309134473, 419097037, 2074236894, 2540543830, 4244884836, 1103759092,
        ],
        &[
            2582817524, 4164383668, 1706646291, 1800000652, 1666270692, 2628289120, 568570643,
            1583069261, 439682583, 3837085850, 247464905, 3856266079, 200842326, 965872188,
            2600055254, 1543566961, 1164241353, 2239343733, 1797130797, 3623500339, 1808008532,
            2868955656, 564142102, 1020590314, 4158719441, 3395391357, 181182939,
        ],
        &[
            2210310100, 69043551, 2823304128, 859345289, 809080897, 120879541, 848904185,
            3835017916, 4066448129, 827700778, 641878366, 728444321, 22823142, 834400226,
            2533421831, 507928329, 398885658, 2332167275, 2030642299, 2377794489, 4277961295,
            2703891174, 3551408972, 450706094, 3991414091, 2973081680, 2714910760,
        ],
        &[
            3640661014, 3206858101, 1737126835, 2106218066, 1425703988, 2338047020, 712809585,
            1420991435, 802147205, 1773895924, 2923903580, 4180731388, 625628901, 81750990,
            3245267790, 2492995722, 457141292, 1264357942, 2610553127, 4085223468, 2725553562,
            3215595517, 3513268137, 4258341409, 1316871298, 2379198914, 1468344750,
        ],
        &[
            3008762500, 1175160375, 2716597715, 2933871453, 475199659, 3313171252, 201726752,
            3994474734, 3542696350, 475569587, 3061861352, 2795963138, 283981201,
        ],
        &[
            4169211551, 1045673686, 3266173725, 1121123371, 2838095204, 3904211332, 2018560297,
            1147750373, 444496996, 2142529574, 3971639805, 3751979387, 2535175321,
        ],
        &[
            1047770371, 1327165786, 2797828578, 2685696909, 3147360099, 2201558235, 1866861337,
            3761155252, 1834270154, 2215232253, 3198169502, 1924553977, 1837870685,
        ],
        &[
            2114181245, 3857180605, 3554288771, 865294166, 1626355571, 408295120, 177441714,
            1458573842, 2220915043, 163377253, 1218043108, 2602503433, 528893379,
        ],
        &[
            2717110780, 986237674, 4250650056, 1544812173, 3986397690, 3836431188, 1613821249,
            2248597207, 2448784182, 2786929074, 1893569847, 1881917363, 2790152996, 3179057496,
            3799013333, 237221218, 3475954500, 4281253174, 3125290097, 2813956421, 1855289745,
            377172866, 3911404025, 3636398044, 3080186791, 88049640, 0,
        ],
        &[
            3505418204, 565465608, 1097112481, 3850268278, 2024670176, 2587290050, 3411014407,
            245345981, 2788535930, 2809261912, 2745706761, 556134486, 3944352700, 2638013922,
            4093507304, 1504147190, 2989086350, 3911331623, 2934713592, 3270681569, 2138704930,
            3059900964, 978968136, 2445899049, 3968088878, 43539092, 0,
        ],
        &[
            3865656210, 3194879205, 414139316, 177501159, 2684513621, 1999012183, 2643222146,
            294716300, 4274795829, 1085676658, 3215144153, 1601970312, 3171535086, 643673394,
            3946671909, 4065144781, 215713465, 172238316, 2724659711, 799017611, 153635789,
            429414755, 1974259637, 1237183745, 2501881650, 269223554, 0,
        ],
        &[
            858223466, 4042145127, 1530910355, 223572531, 3154483634, 1177157565, 2140895357,
            1246913012, 2944555982, 1472007124, 1255831930, 1641310063, 2184842279, 2314841816,
            3300111001, 1618884985, 908327583, 2089112945, 1411422173, 2387337957, 3072912755,
            3289360053, 2808553404, 730378617, 671725420, 90513297, 0,
        ],
    );
    // - x11_sign first
    // - !t0_sign first
    // - *t0_last != 0
    // - *xs01_lo_last == 0
    // - !t0_sign second && *t0_last != 0
    test(
        15,
        &[
            1698534354, 3051849227, 453937331, 3597744389, 4188491352, 1540429554, 1541686219,
            53413474, 2449460985, 2817396669, 2111985345, 2509026402, 1809740433, 3289293410,
            2617592565, 2791033656, 2892380542, 4279121468, 1127688708, 1053029299, 916223848,
            2015034641, 342194872, 1618165570, 3500431510, 192527413, 2131355967, 3029734551,
            2974882323, 1660068342, 2973244676,
        ],
        &[
            1838750278, 66304311, 2535657427, 2850224201, 698579880, 1262199901, 3281530768,
            2112263091, 2707934779, 1567684797, 1075812175, 3518220523, 1324578919, 2267335289,
            3456142629, 3821323209, 1326553164, 564294454, 4194035902, 1211624633, 4195909292,
            2658461285, 789623662, 2974487150, 1275698083, 2125737295, 4192486278, 3187284022,
            88896961, 2297316271, 3463163938,
        ],
        &[
            3818564698, 3079932038, 3482135970, 684071506, 1279597152, 608056186, 4111845555,
            4244378225, 1830629760, 3169670996, 2225259632, 2264284180, 3411037700, 2316989665,
            4067022578, 3527348537, 1977664147, 2506129832, 2221226065, 1330889781, 1701021791,
            644923460, 1849689221, 2715236311, 2773306484, 239859281, 3230697771, 19267319,
            558009472, 1404018120, 3728834892,
        ],
        &[
            1670192276, 1652365370, 735393172, 511768757, 2457280859, 3609810335, 3935660653,
            1797012973, 2672222932, 1933650427, 649516978, 512827590, 782690911, 2062492600,
            2388214063, 3703253089, 904911621, 3390829428, 3844496615, 3067716449, 2596709309,
            1917646504, 1265079680, 3900449910, 367095064, 999066040, 1909678899, 1128678763,
            1656559628, 4266623984, 3645416988,
        ],
        &[
            1921432021, 3518700209, 2079139530, 1137029652, 2300568548, 1222502019, 4223478985,
            2972106290, 3031541529, 2100809044, 1022823954, 194891273, 3689971092, 3296679225,
            4094754776,
        ],
        &[
            948939782, 2806265501, 2304000452, 2075826159, 909827325, 768415732, 2455484163,
            1934928621, 980278935, 2767893850, 1683388302, 1684144848, 3942412863, 331912133,
            3187438755,
        ],
        &[
            1356019384, 2887360290, 1007932454, 3678465615, 2580178110, 3043732489, 3324841406,
            937504067, 386593536, 1673041889, 3131783437, 1641423898, 4040080328, 3081806329,
            1375763815,
        ],
        &[
            1047656638, 2177659443, 4241187176, 2747048355, 1596389859, 4248315455, 3794436052,
            415309719, 1967342941, 3004037899, 374134042, 1698070105, 2693572544, 3158142963,
            4229995607,
        ],
        &[
            960652810, 221555597, 2254803219, 2541562745, 754445106, 359755753, 480947080,
            1224152909, 4273400475, 2309775298, 69810482, 2561255677, 3113057442, 3555394132,
            2689777384, 3342925543, 3147365260, 135644488, 3684637040, 3040946646, 1628023293,
            4023411971, 908469819, 409304848, 3324164633, 1421103084, 368154340, 3701822194,
            274487004, 3602643412, 0,
        ],
        &[
            1997266656, 4144548866, 3822293605, 3926602038, 1609713691, 2892681355, 3749833208,
            2219312564, 1687251603, 1588930279, 710521538, 1644589528, 4054199977, 1672634472,
            3997104967, 956657984, 1345800224, 3871498877, 1120533029, 4151694583, 3146989749,
            3919396264, 3231441052, 3162857783, 2091524393, 124782745, 1890937176, 4230835488,
            2323009602, 1051495796, 1,
        ],
        &[
            2429170498, 799964546, 2388325391, 317823710, 1840557881, 1282801560, 3546799442,
            457834840, 566997058, 2439965867, 3515007146, 3574214064, 4144082196, 3712070930,
            4088900464, 931383261, 170089945, 3405025888, 3812686020, 3504707725, 3656419747,
            3267430265, 3757212521, 606804174, 1643444794, 1555142733, 2148558239, 3200013733,
            3246813347, 347461213, 1,
        ],
        &[
            1131485172, 3765288921, 453666926, 1551065789, 1216395236, 2385231809, 2742879582,
            2426027378, 4167822390, 2229883144, 4228139351, 1293005830, 2665193524, 202666761,
            3531978819, 2231355982, 1235148096, 3915389935, 1798246474, 1563880136, 2945683561,
            1359457669, 1027112316, 3946769090, 4219178471, 794179363, 621594377, 2736938672,
            401053543, 1075392661, 1,
        ],
    );
    // - x01_sign first
    // - t0_sign second
    // - x01_sign second
    // - x11_sign second
    // - t0_sign third
    test(
        13,
        &[
            1025572927, 4037308532, 206447616, 1333004461, 2310395791, 2869951650, 3992227745,
            2130697188, 3157273254, 1749550525, 2868602361, 3060642701, 2426631225, 1974455379,
            2987014830, 3508635696, 4024076244, 1755792892, 787921452, 4166340149, 3450397006,
            1108082683, 457400481, 879113144, 70830365, 2916933914, 2341310905, 851535845,
            216319323,
        ],
        &[
            69731978, 32563822, 946916236, 4016159116, 1224868021, 1718595702, 3605404890,
            1865781628, 1881971935, 1029571366, 841091130, 1677586018, 810666231, 2272746683,
            422109939, 1896472571, 3472541307, 1431848293, 116371044, 608472959, 213411238,
            4154745720, 3801332950, 4224350156, 448206049, 1568183327, 3637228341, 3718344479,
            2263647818,
        ],
        &[
            3460082903, 1733690143, 3780228761, 375677842, 1691069946, 689934922, 2437077094,
            3214349377, 2429121920, 653102152, 649867420, 921104070, 4125433869, 2564034244,
            2631897297, 1501813458, 3052965366, 2782742208, 4166256185, 995218213, 2136996300,
            710550319, 2748268062, 846266401, 385079156, 4270529993, 3575888617, 1142776416,
            876703144,
        ],
        &[
            3175747111, 594393429, 2604257881, 3289051141, 2317754576, 2939871916, 3748706204,
            1013852187, 3343723329, 930022353, 505449760, 486033689, 320591743, 2258238434,
            731770059, 3003208735, 1775573735, 3729639165, 3460733502, 3012496005, 1974364986,
            642896119, 3342553315, 3822794210, 229391052, 1989115715, 2908530532, 1629235212,
            297906894,
        ],
        &[
            2771639655, 2551822689, 1209354903, 2178693592, 1193377488, 793947593, 1366547443,
            2939017203, 963848668, 851106223, 1557851261, 997940551, 2665723642, 1647387186,
            2335120168,
        ],
        &[
            500011960, 1984266544, 857628644, 2215374614, 1925463875, 3051446098, 614991784,
            3952563722, 3226462656, 3978687345, 2589261427, 3942462520, 1207531315, 1120805743,
            1003360645,
        ],
        &[
            451008257, 2688612018, 4236929273, 890846394, 3566713276, 489065477, 715681827,
            3122623659, 3596792199, 1985137091, 2004754993, 1883145970, 1355971815, 187148563,
            3883316291,
        ],
        &[
            1364526373, 1859255461, 3407839641, 1532219679, 3952961366, 1871608475, 4041336906,
            3915677605, 4046491136, 2645312057, 1120032684, 1046405474, 1502799372, 4217018138,
            1023990345,
        ],
        &[
            570058979, 1182257372, 2005332174, 4070895605, 782477793, 360549018, 4128785555,
            3018190268, 2520343653, 2142831802, 2730358685, 1347549043, 898050628, 2236297907,
            3902264192, 3422936759, 3756013281, 352639804, 1061655526, 1986719339, 1128230524,
            2860027389, 1088162212, 2410209429, 3802902058, 2977799885, 3237963571, 2052297093, 0,
        ],
        &[
            1879768634, 2114652211, 1774713818, 554539813, 3731694105, 2702797297, 2844034165,
            2300392570, 1941578090, 3901846543, 983591421, 898885286, 822227747, 3851095686,
            2719131109, 3757259511, 2102411795, 1640921620, 3665191858, 2874213628, 2733705524,
            3691904983, 3823452407, 1169915867, 1621234879, 3630743707, 3478309891, 760168923, 0,
        ],
        &[
            3001545896, 250983613, 389728947, 3814529625, 1132206990, 1065795545, 2212114490,
            2023039590, 3597610210, 1780520334, 3734977828, 2468915322, 4140659688, 115504217,
            819490125, 2708811829, 3384020868, 1515546393, 3633787370, 1821986409, 2218993449,
            3207510568, 3670555307, 3845772841, 3486603021, 1022756175, 2686130697, 2532811595, 0,
        ],
        &[
            2918481451, 3610609934, 2847451215, 2220769302, 2338571396, 2165440207, 147236177,
            3730198517, 3208724339, 649659344, 1519241706, 1867099736, 1460711278, 2220689669,
            3279748484, 1549511341, 227389699, 855766629, 3771440986, 3432825737, 1053106079,
            1117588071, 1313153141, 571072082, 2914006120, 3516294505, 1320473469, 1040189722, 0,
        ],
    );
    // - !x01_sign first && *xs01_lo_last != 0
    test(
        10,
        &[
            2010512066, 3816652533, 3172500500, 565989349, 3299472792, 2270934281, 1434273491,
            2883068162, 1560071521, 2704546576, 4072612274, 1051271061, 3502419446, 355770508,
            10573273, 270934222, 2810553666, 2643486939, 744991576, 1870415987, 2808955936,
        ],
        &[
            465514465, 2659174522, 3761352398, 3377965188, 1124434853, 3313355157, 619366950,
            3258308994, 2397177830, 4225454449, 3612285332, 2249675933, 913650729, 1455947505,
            2849031252, 3527524335, 2078373570, 3478300565, 3188941397, 3478789236, 3675330696,
        ],
        &[
            1404705627, 2788199838, 3229806456, 2206730708, 2050218206, 2991653989, 3066690310,
            3453635714, 3845109791, 627857052, 3843810702, 1520846308, 2495828552, 1637354383,
            2549944931, 3600368060, 108395402, 3341126373, 947408628, 2185962205, 1308735610,
        ],
        &[
            1053200638, 3763337935, 4268597333, 3193422669, 3607631108, 3846186640, 1111342197,
            552132609, 1307213459, 4257590159, 1367177524, 4201506028, 476759064, 680921272,
            2519538412, 238268545, 631677622, 549510100, 771370867, 3109499678, 1448324944,
        ],
        &[
            3267186431, 1424299202, 1039441915, 934588642, 50593817, 2147729982, 1547157070,
            1504136284, 1637263862, 4122900145,
        ],
        &[
            3236915806, 1590507024, 3040977659, 1225520269, 2190057676, 3504088852, 3648192167,
            1810940202, 83552157, 2321737355,
        ],
        &[
            4225004158, 590791745, 643591931, 3221398002, 1827972871, 1242243388, 2513726409,
            1495850896, 4192093161, 50221236,
        ],
        &[
            2036607814, 1957760917, 1453625187, 3689677164, 3487789049, 491100324, 1044724096,
            3399811921, 4055989690, 1295009692,
        ],
        &[
            3263538684, 120256485, 4060593063, 3438619735, 3929215534, 2588618000, 2277735492,
            1072816982, 3840198566, 1043147899, 1235395843, 4200415399, 3035243806, 1304983761,
            3872141601, 1404040725, 4044445427, 429312851, 3176929590, 2645604084, 0,
        ],
        &[
            4122626498, 107203592, 3628216803, 2047112948, 2210020383, 4096415682, 1367835443,
            2351498305, 4206740014, 996512353, 1147432853, 2660487268, 2975278601, 828643899,
            3534602756, 3189739581, 1167342346, 2278425545, 830131806, 2736051400, 0,
        ],
        &[
            2978640553, 1655222107, 3504154601, 1909395468, 3162278697, 2260916194, 3267701344,
            1089677643, 751892566, 4225581520, 2977844572, 882459562, 3829381149, 4213372565,
            1864175453, 3748132665, 2980365850, 1250882260, 3505860930, 652487710, 0,
        ],
        &[
            3447523550, 464011238, 1942725626, 1838669852, 3904733671, 258179587, 3703737903,
            1567067340, 3691418914, 1104958587, 1817293529, 2218949245, 4009018118, 2970219724,
            3485275615, 2079452863, 3685497725, 785093512, 267920582, 1623141509, 0,
        ],
    );
    // - x01_sign != t0_sign
    test(
        11,
        &[
            105108810, 2939008753, 43370164, 1242908972, 1159070628, 84952645, 4165954274,
            906659711, 1019556239, 2534013318, 721285587, 1325925355, 1632336316, 4096108923,
            1763507067, 3809084118, 2331120351, 2754738494, 2103185448, 1244905580, 1907939448,
            2609958778, 730167872,
        ],
        &[
            2265464388, 1834541318, 2372682679, 3313972430, 3918258678, 2394770661, 1037415216,
            274650728, 3739985994, 2108171684, 2199795287, 566429532, 1157864107, 2882221652,
            910859296, 451698206, 2649007716, 1415208273, 2646592820, 3034704049, 2120666349,
            979747662, 322674205,
        ],
        &[
            3874939837, 1233304929, 106185835, 37158351, 2802916915, 3304535373, 696577680,
            3544731437, 3157476288, 1443616905, 3457867645, 4024841761, 4062836214, 2279777825,
            1482825076, 4078337234, 3264521275, 3344925347, 3161366177, 2196844487, 898054831,
            1438581672, 3352145698,
        ],
        &[
            1849881721, 2092152070, 2177899750, 288752532, 3618364405, 4141821358, 4158419895,
            2146161307, 3308429556, 282226456, 2214705611, 146486422, 1226466729, 2012446110,
            390251911, 3095301941, 1935278338, 1226998912, 102355086, 2004958092, 1379648599,
            833731447, 3784213054,
        ],
        &[
            1615644044, 2866597055, 1417092926, 358845179, 3417066108, 3643757328, 143514487,
            3323797767, 2638762718, 3254012295, 3018153021,
        ],
        &[
            344398095, 226628832, 764911662, 2701340611, 2699247769, 4219726262, 242800476,
            2930885037, 990361216, 3569632382, 93961188,
        ],
        &[
            129918839, 2214278271, 1283256165, 2817801381, 2615176815, 4126216685, 3082478432,
            3280512502, 1853758032, 2634521780, 4223054084,
        ],
        &[
            2590519092, 2716288017, 4272214574, 3155523983, 4104711693, 2762266943, 2745407059,
            2324276365, 326739968, 523560569, 396448235,
        ],
        &[
            3666666516, 547769313, 592311112, 2965277691, 3689426122, 853080706, 1356550839,
            4168566128, 1148520078, 939171181, 4015917617, 683222651, 4090232802, 3632973858,
            1439963708, 1600913535, 187292997, 1577612527, 1212140599, 1900670851, 2445225960,
            2669823530, 0,
        ],
        &[
            1687083814, 3698143323, 3751827194, 3866919242, 2549527161, 4010635967, 461515118,
            2688244606, 3320207170, 3619767327, 434459238, 4066444037, 3805779605, 3312610610,
            735569613, 11746680, 3083700477, 155975832, 1514885535, 1758738984, 2620267716,
            218832355, 0,
        ],
        &[
            3131789211, 3199129533, 2920949518, 53796216, 2859826242, 1820398479, 3973031721,
            2637151743, 3165657662, 2168050566, 1715646418, 2044422416, 1930311216, 3380875453,
            183533660, 2431952327, 2641905274, 4032322429, 1801301919, 3668108775, 2977500729,
            312563773, 1,
        ],
        &[
            1292086439, 3252628865, 3795724813, 3949296317, 3955675466, 2439424685, 4209245455,
            521673263, 3357923169, 728386653, 1761881489, 2103645314, 4058013613, 1659506293,
            3599632745, 96485170, 2801208662, 1691699790, 2077775599, 2126298157, 3250670933,
            280076983, 0,
        ],
    );
    // - *xs01_lo_last != 0
    test(
        14,
        &[
            3187643652, 689861248, 4104971615, 2655959233, 1806333930, 2946646526, 716445782,
            1855643090, 2911585605, 2840897284, 772165124, 4249869367, 2223216112, 535376700,
            620340444, 1645897591, 1333985741, 730570397, 4160730534, 1246271990, 1725573135,
            352253863, 1262570932, 1984889896, 2956051954,
        ],
        &[
            3195311878, 3038033431, 2773970346, 1168697958, 2813281967, 746268626, 2691595086,
            4097287295, 93455850, 492512747, 2710035977, 1223679950, 3946316915, 3578566691,
            2992317089, 3174913905, 2131559101, 1331916743, 3949213449, 80715282, 158737627,
            2988488283, 2028980419, 3830004661, 250424958,
        ],
        &[
            2536403826, 384848984, 1774521292, 2718320818, 310873769, 2046298166, 3228810082,
            1178726899, 1925709392, 178731159, 2832365355, 2836635324, 193074416, 728452613,
            3469205214, 1514937172, 3814853496, 2161355055, 886569678, 4161288761, 2889870661,
            76946453, 1878702060, 1850975782, 1627735969,
        ],
        &[
            1725052145, 898802704, 776971836, 1219924401, 4224517976, 84598208, 2148547128,
            3866703857, 1192216635, 3270964370, 3470989137, 4065537602, 3771846191, 3811480941,
            4062031618, 3552276638, 1926982962, 297558461, 1000365546, 1422440944, 3161036166,
            519129014, 2412823237, 2742512283, 1294289799,
        ],
        &[
            2124565348, 1850594425, 684081640, 3059091438, 3756365771, 3723911794, 1430887131,
            2186106443, 1678377466, 1963700876,
        ],
        &[
            1786516080, 1799763022, 3244164991, 2583605118, 2016537122, 396228546, 1776124622,
            3149228799, 3618737851, 3722388678,
        ],
        &[
            2119342743, 2442803683, 2913523155, 2290813344, 2341544084, 2607906759, 3903931010,
            1623427068, 200668227, 866036225,
        ],
        &[
            154537600, 2783369274, 2784871905, 2923414066, 238748138, 894157397, 425890999,
            2023402367, 2708989084, 1773914772,
        ],
        &[
            714150426, 747474047, 711763591, 2505815901, 1931970168, 1830915645, 132011662,
            1641495062, 3649937934, 1578564850, 2879358541, 2256534044, 375624512, 402134451,
            124437205, 2656206024, 3918226432, 2934327023, 2861387419, 1259632526, 168082530,
            1906995676, 1276986866, 966360812, 0,
        ],
        &[
            2584563904, 501423603, 3179309343, 2650015179, 1354929902, 930009290, 4239032359,
            315773096, 1219190592, 1141774901, 2074216575, 279450192, 2450874439, 247131610,
            26955632, 3117135685, 2632739337, 2367699203, 1269501214, 837130129, 824399760,
            1192414122, 2569424753, 1942029336, 0,
        ],
        &[
            2166600367, 3206417308, 2873245177, 3181806685, 47762280, 4077132431, 3414741771,
            595567128, 1650380454, 3351739639, 1204748131, 4239748557, 4216796226, 1580325602,
            2295006359, 928145182, 2501406243, 2685614796, 1481907821, 3500619179, 3653788019,
            2099044319, 251730683, 1101601776, 0,
        ],
        &[
            327036000, 3129689357, 722252402, 3256719743, 762583270, 1730383711, 2418392723,
            3837009731, 993829843, 531209884, 2056099111, 2266372736, 1988906408, 2154603398,
            2683386624, 826004386, 3819045504, 3703296261, 2407207875, 3346505456, 2182410563,
            1915488498, 41659158, 2205564200, 0,
        ],
    );
}

#[test]
fn test_limbs_gcd_reduced() {
    let test = |xs: &[Limb], ys: &[Limb], actual_out: &[Limb]| {
        let x = Natural::from_limbs_asc(xs);
        let y = Natural::from_limbs_asc(ys);
        let mut out = vec![0; xs.len()];
        let mut xs = xs.to_vec();
        let mut ys = ys.to_vec();
        let out_len = limbs_gcd_reduced(&mut out, &mut xs, &mut ys);
        out.resize(out_len, 0);
        assert_eq!(out, actual_out);
        let gcd = (&x).gcd(&y);
        assert_eq!(Natural::from_owned_limbs_asc(out), gcd);
        assert_eq!(gcd_binary_nz(x, y), gcd);
    };
    #[cfg(feature = "32_bit_limbs")]
    {
        // - n < GCD_DC_THRESHOLD
        // - xs__len > n
        // - !slice_test_zero(&xs[..n])
        // - !mask.get_highest_bit()
        // - x_high >= 2 && y_high >= 2 in limbs_half_gcd_2
        // - x_high <= y_high && (x_high != y_high || x_low <= y_low) in limbs_half_gcd_2
        // - y_high < 2 first time in limbs_half_gcd_2
        // - !limbs_half_gcd_2
        // - xs_len < ys_len first time in limbs_gcd_subdivide_step
        // - xs_len > s in limbs_gcd_subdivide_step
        // - ys_len > s first time in limbs_gcd_subdivide_step
        // - xs_len == ys_len second time in limbs_gcd_subdivide_step
        // - limbs_cmp_same_length(xs_init, ys_init) == Greater second time in
        //   limbs_gcd_subdivide_step
        // - xs_len != 1 first time in limbs_gcd_subdivide_step
        // - ys_len > s second time in limbs_gcd_subdivide_step
        // - n != 0
        // - xs[0].odd()
        // - n != 1
        // - y_0 == 0
        // - y_0.odd()
        test(&[0, 0, 0, 1], &[1, 0, 1], &[1]);
        // - n >= GCD_DC_THRESHOLD
        // - n > s in limbs_half_gcd
        // - n >= HGCD_THRESHOLD in limbs_half_gcd
        // - n < HGCD_REDUCE_THRESHOLD in limbs_half_gcd_matrix_reduce
        // - n < HGCD_THRESHOLD in limbs_half_gcd_approx
        // - n != s + 1 && !mask.get_highest_bit() in limbs_half_gcd_step
        // - y_high >= 2 first time in limbs_half_gcd_2
        // - !subtract_a in limbs_half_gcd_2
        // - x_high != y_high first time in limbs_half_gcd_2
        // - x_high >= HALF_LIMIT_1 in limbs_half_gcd_2
        // - x_high >= 2 second time in limbs_half_gcd_2
        // - x_high > y_high first time in limbs_half_gcd_2
        // - x_high >= 2 third time in limbs_half_gcd_2
        // - x_high != y_high second time in limbs_half_gcd_2
        // - y_high >= HALF_LIMIT_1 in limbs_half_gcd_2
        // - y_high >= 2 second time in limbs_half_gcd_2
        // - x_high < y_high first time in limbs_half_gcd_2
        // - y_high >= 2 third time in limbs_half_gcd_2
        // - x_high <= y_high first time in limbs_half_gcd_2
        // - x_high < HALF_LIMIT_1 in limbs_half_gcd_2
        // - !done in limbs_half_gcd_2
        // - !subtract_a1 in limbs_half_gcd_2
        // - x_high >= HALF_LIMIT_2 first time in limbs_half_gcd_2
        // - x_high > y_high second time in limbs_half_gcd_2
        // - x_high >= HALF_LIMIT_2 second time in limbs_half_gcd_2
        // - y_high >= HALF_LIMIT_2 first time in limbs_half_gcd_2
        // - x_high < y_high in limbs_half_gcd_2 second time
        // - y_high >= HALF_LIMIT_2 second time in limbs_half_gcd_2
        // - x_high >= y_high in limbs_half_gcd_2 second time
        // - x_high <= y_high second time in limbs_half_gcd_2
        // - y_high < HALF_LIMIT_2 second time in limbs_half_gcd_2
        // - !skip && limbs_half_gcd_2 in limbs_half_gcd_step
        // - out[n - 1] | ys[n - 1] != 0 in limbs_half_gcd_matrix_1_mul_inverse_vector
        // - n != 0 fourth time in limbs_half_gcd
        // - x_high >= y_high in limbs_half_gcd_2 first time
        // - y_high < HALF_LIMIT_1 in limbs_half_gcd_2
        // - subtract_a1 in limbs_half_gcd_2
        // - y_high < HALF_LIMIT_2 first time in limbs_half_gcd_2
        // - out[n - 1] | ys[n - 1] == 0 in limbs_half_gcd_matrix_1_mul_inverse_vector
        // - x_high < HALF_LIMIT_2 second time in limbs_half_gcd_2
        // - x_high > y_high || x_high == y_high && a_low > b_low in limbs_half_gcd_2
        // - x_high >= 2 first time in limbs_half_gcd_2
        // - subtract_a in limbs_half_gcd_2
        // - x_high < HALF_LIMIT_2 first time in limbs_half_gcd_2
        // - n == s + 1 in limbs_half_gcd_step
        // - mask >= 4 in limbs_half_gcd_step
        // - x_high < 2 first time in limbs_half_gcd_2
        // - !limbs_half_gcd_2 in limbs_half_gcd_step
        // - xs_len == ys_len first time in limbs_gcd_subdivide_step
        // - limbs_cmp_same_length(xs_init, ys_init) == Greater first time in
        //   limbs_gcd_subdivide_step
        // - ys_len <= s first time in limbs_gcd_subdivide_step
        // - n == 0 fourth time in limbs_half_gcd
        // - new_n != 0 in limbs_half_gcd_matrix_reduce
        // - !x_high && !y_high in limbs_half_gcd_matrix_adjust
        // - new_n != 0 first time in limbs_half_gcd
        // - n != 0 second time in limbs_half_gcd
        // - n > s + 2 in limbs_half_gcd
        // - n != s + 1 && mask.get_highest_bit() in limbs_half_gcd_step
        // - limbs_cmp_same_length(xs_init, ys_init) == Less first time in limbs_gcd_subdivide_step
        // - n != 0 third time in limbs_half_gcd
        // - first zero in limbs_half_gcd_matrix_mul_matrix
        // - second zero in limbs_half_gcd_matrix_mul_matrix
        // - ys_len <= s second time in limbs_gcd_subdivide_step
        // - s != 0 fourth time in limbs_gcd_subdivide_step
        // - ys_len != 0 second time in limbs_gcd_subdivide_step
        // - mask < 4 in limbs_half_gcd_step
        // - new_n != 0 first time
        // - limbs_half_gcd_2
        // - mask.get_highest_bit()
        // - y_0 != 0
        test(
            &[
                4197577227, 3663616140, 2551363557, 471855676, 1132453243, 3890478103, 3255658039,
                924933869, 1182514876, 3443542006, 1628670028, 1920607761, 1774928803, 698481201,
                1688346100, 3912816434, 1546494740, 295495534, 207209302, 1228953098, 3286562513,
                2448096226, 2547931780, 2100456764, 4196096572, 2722413368, 247913388, 3316902282,
                60763307, 2236362858, 2725594164, 774854500, 1708955617, 1129631186, 2093339374,
                3603997370, 1314390085, 2641774231, 385916035, 2355828959, 821901122, 458162478,
                2603441028, 1665224323, 2479737003, 325263924, 3583373830, 1314852315, 3062488937,
                2409141051, 2802944956, 1176409253, 4206372227, 3264669278, 376728881, 3979228915,
                789107765, 1384935912, 2580449282, 3943579021, 1561210927, 2058485109, 2277329329,
                3961162387, 2853000821, 2810633314, 3958004542, 1042177900, 2041876084, 2531779261,
                1110199836, 4268607687, 3910301281, 2843809243, 930485055, 309142224, 2144761027,
                2416055325, 3266084990, 2447991430, 2443964826, 3786182453, 2724967444, 2929403702,
                2871781906, 2093610692, 2665818776, 2885572011, 1023583045, 388434237, 3383760566,
                3331268852, 812997613, 743229208, 707412010, 2045807268, 3029838557, 2734070773,
                1201494085, 2013003250, 3352629986, 2121723344, 2606274703, 446904615, 1986390279,
                777681916, 850787414, 998620529, 3430750455, 4223290015, 850807722, 3597579048,
                1953183408, 2598401455, 4174123707, 2094682680, 3126686508, 741312051, 1249776706,
                3394903770, 1488538657, 86039141, 1122597620, 4088204860, 1965673179, 3854858790,
                1805056514, 712813144, 2223549448, 3340696011, 1057241621, 4123508209, 1178122500,
                3345161830, 1643843465, 3443634290, 1183490760, 3948287560, 1351208937, 1585302010,
                321331177, 1982899829, 4207584480, 428023286, 313173943, 1893554339, 3657870602,
                1184563157, 2577756042, 3633915258, 4107506336, 2000538440, 3462275636, 905594775,
                2288056884, 2174527929, 1967515771, 1889360380, 3647702195, 2766306052, 1692805723,
                2547117089, 2806372808, 1868864235, 377724386, 2108963424, 1243470944, 3419076435,
                215621577, 4277714044, 1847653420, 1364769655, 3409308990, 3221049591, 727661675,
                1355876888, 477454154, 159184771, 47758267, 82226374, 173939358, 3989288267,
                882358143, 314672149, 241143075, 1276781232, 3729530695, 4062922154, 882397039,
                717181562, 1487490701, 2348085968, 3844176796, 3858520706, 2303777194, 1570281243,
                701612807, 1685259689, 44118858, 2403065948, 2475839209, 4243425343, 2409356427,
                3602873673, 2794543251, 1879804284, 986610444, 1780607155, 2709668878, 3714165156,
                3474383994, 52876951, 2738151673, 3272452228, 157949197, 3388390384, 1867287032,
                3397657263, 893351542, 359080279, 1867186583, 2633080366, 1691644564, 2228425159,
                205919408, 2674325620, 408159999, 2614176370, 2954808008, 4288814065, 592989283,
                2421568051, 3947678068, 1076048985, 2725991063, 1466932877, 47765783, 2096755893,
                3871926112, 3067385837, 1115438999, 673485739, 4051915345, 3287777679, 3432949036,
                712556260, 2115195072, 1163581657, 717872497, 1372142050, 1296789715, 3248464426,
                2010337977, 1022298124, 4097244827, 3132735945, 1590029541, 1952666134, 3738758275,
                795022630, 1022963383, 322333376, 3927893537, 1123308885, 3926676584, 110023661,
                1847201588, 3972030290, 1877137386, 2340701232, 2680282095, 171364221, 1255189251,
                4206413144, 3146670074, 2595432946, 2123584779, 421107554, 1661340820, 1717955211,
                1093855779, 2562116708, 311124173, 1286538456, 784915601, 2787458377, 2909859896,
                2079504712, 2482552290, 381311010, 4164814891, 3498767523, 1871402912, 4280107119,
                3555212136, 4263072354, 2428360919, 4030822435, 414962043, 2110229154, 3145804625,
                3419521350, 2300309736, 1037962840, 360477135, 2761975814, 3570812469, 2141362836,
                3744698030, 2917828184, 1604822798, 524785135, 4123189923, 306254809, 2907418601,
                715269380, 2880366359, 85178201, 3315366581, 1234064887, 2261417810, 740136577,
                311877843, 916858922, 701414307, 2440960632, 2877171070, 832859154, 1906376412,
                2470726441, 746564647, 3169727501, 3847690619, 2569989652, 4143797285, 1181804091,
                3467101578, 1620737185, 798590631, 1102046391, 86799227, 1011686392, 2365540922,
                1437914259, 3977803079, 2484908631, 3183729029, 770778157, 2110032689, 642819790,
                3783384202, 2203826388, 179388270, 3069155346, 2279937873, 3949687834, 2292743729,
                1695844312, 1519061556, 333473276, 304895978, 2399722902, 2408889893, 4198515797,
                1815311964, 3633226566, 1221066756, 2016659596, 1969286056, 1186325182, 1718972725,
                228738772, 2598964902, 4168009255, 2340771410, 3236251687, 3295756396, 359898616,
                748792865, 2815036617, 2294605685, 2576581376, 3104446059, 1374922466, 3229588626,
                1342195213, 4121614180, 1287499457, 1644513681, 2251892498, 3519911253, 3676415503,
                3642198693, 764265217, 3084097378, 2745050223, 3246393306, 561565268, 961468834,
                3209498589, 2760753568, 27068759, 1509048774, 1453964033, 1331717633, 2158694102,
                1233260646, 2709419591, 177106547, 2429674784, 1301711456, 2208920832, 176199715,
                3161955920, 3374145530, 4247833566, 4210175938, 2667521985, 2158406637, 28032585,
                3422529443, 2403124753, 4248809640, 1432019389, 2720502195, 3698731935, 3697762717,
                3038810720, 73235983, 1930366289, 1889267718, 1024383025, 3630767897, 2664806680,
                1701068479, 3104152545, 3126610581, 1945171628, 2107750861, 1705151590, 1384917050,
                1013720645, 3504778715, 3696436857, 1705707415, 2189842908, 1689236637, 1657303695,
                544592080, 1136857424, 824561482, 2369780064, 2472849354, 2717401545, 2421800924,
                580975041, 750461869, 3727051247, 3077843833, 18610721, 194041714, 1077076049,
                1259434033, 1763428865, 2241079119, 2783260484, 305088684, 4127815512, 736813644,
                540809689, 3159882452, 1825555986, 1501660759, 3053878326, 2824718405, 958691107,
                1972676940, 501505940, 2507376271, 4221542679, 7397642, 335120812, 4137134683,
                394428968, 867251752, 156156583, 3806282517, 3663693413, 1066638741, 3888387987,
                3416832624, 1874236937, 1816284001, 24943821, 2444851827, 2900828566, 2539594150,
                779015020, 2409648256, 2933935060, 3781965704, 2113602755, 1850649100, 344087483,
                1315113853, 3810878395, 2922986247, 1124979158, 1254362509, 2961155811, 567171845,
                1368984368, 3708291355, 3083282828, 1452928286, 2981316093, 3064523290, 1479238797,
                2292333655, 2448998592, 3524304101, 3649979636, 68492449, 1954531957, 2750371018,
                1841894757, 1097795246, 2048976053, 3845300136, 2399041467, 2960591765, 1149499376,
                625651164, 2037776876, 164166162, 2380013720, 66608184, 3818619585, 2131700282,
                502477679, 2053801901, 2925841131, 1028579949, 762321931, 3017484607, 2336518362,
                2067352515, 3383243930, 3093911248, 1575534888, 4087803856, 1041068290, 1614488293,
                3321708937, 1309929846, 1384017125, 1090379238, 439291539, 3162665576, 344261203,
                3404316428, 550149101, 2430715962, 2274647103, 1656669548, 1081997280, 1328444417,
                2031945767, 1473781932, 2837006571, 1533898454, 3338843219, 3838110065, 1262509474,
                3527801284, 2579686530, 3569205777, 4100363249, 2105425082, 2234328532, 2748351804,
                2673483805, 3924474224, 2636212581, 3463412678, 4098352143, 2368424540, 4175082715,
                2039046172, 3243053164, 1220737144, 1748690383, 2080583256, 1099659816, 1135489475,
                2884856950, 4192473639, 1947310046, 2881220399, 3944101233, 2762458465, 4110783623,
                1673864095, 1791817121, 2629252938, 2351311744, 1835759362, 1831898034, 3637639151,
                2781044911,
            ],
            &[
                2582606328, 3907176038, 1869269403, 1460626031, 582046080, 4111410419, 2516532417,
                146794403, 430764753, 1289757806, 833414363, 1159861655, 1855686035, 660114073,
                2678166363, 976180952, 3948114536, 2765729676, 1405860081, 390581044, 2452693248,
                166981518, 4119389957, 2838988971, 1280548287, 2735105611, 2230034237, 807044950,
                2753901414, 3377733072, 1260163723, 2906031321, 1160378830, 14255505, 2306821236,
                1827788545, 1803661748, 3344014911, 358752723, 3659821753, 1669317560, 230363172,
                1671886371, 1424632611, 2409163050, 508495152, 2520039856, 3298638732, 1428088953,
                2701777018, 3798098801, 1335576077, 2246521165, 3958581687, 1600454442, 3907674130,
                3548795039, 3826552611, 3725833438, 2551041161, 2709616429, 2354699408, 3126481426,
                3099474439, 937375512, 2352825782, 1243847506, 1747475903, 233022035, 728638168,
                2156419178, 1739901706, 842209461, 1678526634, 4177514773, 1117076123, 2650916066,
                3124223161, 1696818673, 1436772296, 3437185289, 261500205, 992897074, 4122050997,
                3281707495, 437035780, 764393933, 702695215, 20994502, 1300788043, 2485247663,
                2129017620, 1314245779, 2080540901, 3382431782, 4177690980, 496213991, 3620296923,
                45820436, 824665660, 1249390567, 2880218577, 793755363, 2099510489, 1127291309,
                3018724734, 4045529792, 3787034562, 2447184228, 4231165855, 2543665948, 2685844279,
                3687498369, 2465827535, 937827718, 3493198617, 1112432160, 682701349, 2407287012,
                261540535, 403665586, 739072430, 4114763090, 276463615, 938788455, 3393306569,
                493459877, 2513680357, 229914576, 2765762116, 3809089681, 1975260832, 630837582,
                512373818, 2415638589, 816524198, 926812996, 2919108012, 1777763031, 2420059151,
                2303215774, 730440645, 2590700033, 4251535370, 4205174050, 4119494394, 960682558,
                2886343983, 3540930563, 711941431, 1626586284, 981652789, 2327139618, 626401545,
                612995811, 1953752764, 2899519720, 1408949156, 4290924666, 3997643096, 1422846953,
                364233241, 2659284800, 965784531, 3136185596, 1794737991, 1004877728, 3355875944,
                2432964630, 4193048101, 1130684469, 3045267236, 1286642177, 2756498771, 932256612,
                961567406, 1201162315, 1773420101, 2567717098, 709597428, 452677858, 4284511418,
                3021868094, 686834336, 268204656, 187140090, 2471076712, 144112752, 793083023,
                3609314649, 1553540924, 1941459858, 94249970, 2264076058, 1658809682, 2185281448,
                357602413, 2940153559, 2210448220, 1292548375, 144668819, 2795072628, 4170611652,
                2951824217, 478980333, 77414522, 2542872381, 310035381, 3503531438, 2774626613,
                1665526812, 1575810865, 3550598577, 2584875090, 14090765, 4084436300, 1767381238,
                452463079, 547596654, 1305057780, 2372702234, 2823750152, 2615637338, 3915249206,
                2017151267, 512190028, 3431028255, 1204341799, 1734623287, 965084563, 2326922875,
                953678590, 2503809985, 1951017332, 2478332356, 3743715904, 3536776438, 2015910466,
                583258098, 1654548120, 3876369464, 1912036404, 1854124724, 3285753258, 4239446578,
                64896010, 3226676422, 709577570, 3605847387, 4233435166, 3578913837, 1305181609,
                1104334009, 535007176, 3368595294, 3400831924, 2658360667, 83892703, 452925944,
                1148434659, 3632651250, 2497302974, 4052931735, 2825332083, 4085939360, 2559084850,
                3079544094, 4119878194, 504432022, 2083265837, 1798431512, 465950253, 2330752139,
                1762670455, 4153406121, 171243945, 2564294901, 3457826969, 3997425378, 3322817525,
                1219343400, 3494114736, 1472311045, 278005611, 3537646006, 1737609178, 745736375,
                1744823525, 271949138, 3132324813, 2299821342, 2740187772, 1351381540, 3548063636,
                1535674335, 1875773947, 1619241799, 1128793993, 1049425265, 3145446141, 3425432576,
                2046178101, 1387398701, 1597786450, 1263128416, 34613519, 4163179935, 1798039392,
                3111216944, 2777264082, 22483696, 1038704995, 3359344737, 3795184360, 1831084665,
                228540483, 690048105, 2440634399, 801439700, 1953285894, 1885782072, 490462137,
                1220610650, 87049442, 3789599385, 3845031019, 2099863453, 2821455657, 1978309953,
                3224772848, 727874845, 120712817, 4094193244, 696826576, 408973252, 2585539339,
                633960302,
            ],
            &[1],
        );
        // - xs_len <= n
        test(
            &[
                1304645631, 3453605495, 2384770191, 4234662844, 193170837, 3507896589, 1570888072,
                3287476780, 3273564387, 2083012415, 2738572262, 216274431,
            ],
            &[
                140631447, 652003847, 539742246, 1688191151, 622459323, 2568104193, 1918906162,
                3927704678, 3061246232, 1898154381, 2239425174, 122041517,
            ],
            &[1],
        );
        // - x_high < 2 third time in limbs_half_gcd_2
        test(
            &[
                2504682851, 2582732736, 385625941, 2382204212, 314045215, 2050447634, 503267526,
                1775048793, 3749152248, 83577313, 3954681023, 2222858364, 2002028732, 3989047480,
                3329901642,
            ],
            &[2374767210, 3305948119, 2775088291, 3859016530, 6636685],
            &[5],
        );
        // - slice_test_zero(&up[..n])
        test(&[0, 1, 1], &[1, 1], &[1, 1]);
        // - an == 1 in limbs_gcd_subdivide_step
        // - s == 0 fourth time in limbs_gcd_subdivide_step
        // - n == 0
        test(&[0, 0, 0, 0, 1], &[1, 0, 1], &[1, 0, 0]);
        // - n == 1
        test(&[0, 1, 1], &[1, 0, 1], &[1]);
        // - limbs_cmp_same_length(xs_init, ys_init) == Less second time in limbs_gcd_subdivide_step
        test(&[0, 0, 0, 0, 0, 1], &[1, 0, 1], &[1]);
        // - limbs_cmp_same_length(xs_init, ys_init) == Equal first time in limbs_gcd_subdivide_step
        // - s == 0 first time in limbs_gcd_subdivide_step
        test(&[1, 0, 1], &[1, 0, 1], &[1, 0, 1]);
        // - x_high == y_high first time in limbs_half_gcd_2
        test(&[0, 0, 2], &[1, 0, 1], &[1]);
        // - y_high < 2 second time in limbs_half_gcd_2
        test(&[0, 0, 2], &[1, 1, 1], &[1]);
        // - x_high == y_high second time in limbs_half_gcd_2
        test(&[0, 0, 0, 0, 1], &[1, 0, 3], &[1]);
        // - xs_len > ys_len first time in limbs_gcd_subdivide_step
        test(&[0, 1, 1, 1], &[1, 0, 0, 1], &[1]);
        // - n == 0 second time in limbs_half_gcd
        // - s != 0 first time in limbs_gcd_subdivide_step
        // - new_n == 0 in limbs_half_gcd_matrix_reduce
        // - new_n == 0 first time in limbs_half_gcd
        // - nn == 0 first time
        // - nn == 0 second time
        test(
            &[
                104199695, 1437842355, 3026540896, 1036691142, 2895760091, 316986459, 3848493166,
                4283163139, 1409731068, 2162161861, 3180981287, 2664973704, 3129778597, 1141759869,
                3423483599, 3304860181, 1499458172, 2283443318, 304208206, 3898070080, 1624380249,
                4217356185, 3984965450, 1989320468, 4046024996, 290564828, 982998126, 714749497,
                1256133149, 1964245093, 1989969476, 2845229677, 75994139, 1534010229, 3551668582,
                437558754, 2400783907, 1104529196, 2628109158, 767576225, 3386003603, 2589581389,
                3197373266, 1119570806, 2388054566, 560919780, 2181214692, 895228346, 2455964730,
                1236092704, 3460190245, 2187848751, 355369047, 1999986258, 3567727382, 1691926629,
                2725212068, 2592492178, 797372755, 1342659622, 1706103226, 2741335822, 683303027,
                2714389761, 3546978425, 1986412131, 762213298, 2333449774, 778670753, 97719623,
                581978613, 3531804453, 1122531574, 1722591132, 2538512178, 1919025659, 3171826961,
                3494777585, 3878057786, 1159079409, 1839284455, 1154718578, 2665661658, 897285327,
                1500853657, 2633085447, 1751356374, 3270858770, 1163694943, 3330045700, 1488678831,
                3027918693, 502928061, 1337476314, 3692361284, 2178683191, 3096849964, 970332766,
                1824923766, 846751495, 2918346590, 70616406, 3836757211, 2684780384, 2463029602,
                3457312899, 887456732, 3244684159, 1709425071, 1650827329, 1114331473, 2300037523,
                47504219, 2349860938, 514226917, 2830450926, 3649882524, 3714094143, 301323195,
                4142782943, 3785993652, 866247992, 2164534671, 2737799129, 3275770468, 215424696,
                4257046307, 1733065842, 2434034911, 1462340609, 3642973392, 3350853644, 4199792744,
                684333086, 217713765, 3411993450, 1792335767, 4171842881, 159579156, 2430618740,
                3131183234, 3163246383, 2128439763, 2401347699, 2936760275, 3236944394, 2200904017,
                2436166477, 4193014474, 3613539040, 2916219306, 4261812949, 115873482, 2180021985,
                4055598752, 1260608904, 2416550157, 1259155018, 1431188781, 1176329051, 2299641742,
                3980571345, 1599289272, 1782362083, 3811356761, 1974756300, 533380582, 3641430046,
                832768712, 672407572, 1502225325, 1016817644, 14093203, 1980541787, 1908462143,
                3913154473, 724095260, 1182316086, 719426427, 75511913, 3523894057, 3990195494,
                2120530771, 3562485018, 1433690638, 2518584164, 1493664787, 3968793063, 3670159139,
                157980348, 1301197374, 3305985080, 3941044403, 612257166, 3771911392, 2763453419,
                2718688747, 3693090746, 589991197, 498197071, 2438022286, 3991964070, 377079193,
                3099592329, 3321736969, 3902471945, 2307760142, 3013676133, 3258936160, 4011810226,
                1053219984, 2837546691, 1856062985, 2092610837, 91102249, 406247891, 890201919,
                2559127506, 338532027, 395154770, 3275565770, 1258530084, 2939576662, 2055294057,
                1883048671, 3836169741, 2026424182, 3339579277, 1542609345, 417648228, 4191728149,
                2895752032, 1515160584, 278532504, 1393858866, 1730994375, 3794676496, 1670361915,
                1258111291, 2422641149, 2347194379, 80193494, 2358530906, 2877787771, 1965418230,
                4205347720, 2666879163, 4215296303, 2916779693, 3935289306, 3244538076, 1632052180,
                4176725684, 992286325, 2739073795, 2079197711, 2797905589, 2449598406, 289423436,
                762177014, 2363661744, 772167940, 2765100761, 198147685, 572172334, 3817619946,
                162288540, 1223981567, 540177774, 2154166987, 2653356928, 3206160568, 1862076861,
                3133558385, 474824123, 2525460202, 3924632969, 1146129872, 1432186566, 737582770,
                3316793636, 3709987676, 3557072993, 567470616, 1129703040, 1521364280, 3566052322,
                916273230, 2293310523, 1377033267, 1508571570, 3382970810, 2996523808, 1581520543,
                2618222496, 3288125825, 2313878894, 349906863, 2219317771, 3176122714, 1013378826,
                4116023745, 1921798849, 3062375628, 2736960803, 2501959999, 1510561246, 1700569625,
                1652836989, 3312274215, 4195830099, 1132143254, 950525497, 2316560086, 375651095,
                2664099671, 455151019, 1159444553, 4198810519, 3127984438, 2144120008, 1969060002,
                562539184, 1602239482, 2852481959, 2095454819, 3210933640, 3500352077, 3422305025,
                2014869566, 2974700896, 3465912392, 2040467605, 4197300028, 3591134045, 3895974116,
                1820422501, 1395818068, 2660152096, 1916184939, 3260276758, 53585995, 927898271,
                2259302740, 963522240, 2856612864, 3450205940, 3367413678, 3535931570, 397649639,
                3700183847, 1270654564, 3500464902, 3997758233, 3130672822, 2562223359, 638317405,
                3181668171, 539650504, 2914453051, 645816251, 4007289298, 3438058969, 2770073590,
                864060896, 2320524273, 3692263013, 3473619051, 3882200592, 499549190, 2692529571,
                1895097838, 3793943961, 3454593367, 1512683055, 852698565, 1995890144, 865090611,
                609912127, 2960073872, 1078887,
            ],
            &[
                3965328483, 312519701, 3403268855, 3606492602, 1902761898, 851349064, 3139262646,
                633064190, 2260801841, 3139511175, 2981429315, 570792856, 3276826258, 3740173343,
                2258820496, 613809729, 241642072, 3441475901, 4294219823, 976984900, 2389777673,
                4191459734, 1003327410, 1302790361, 2451013524, 2641182095, 3622343124, 2806631117,
                3967790700, 581646251, 2019686780, 2032370136, 2076036058, 856161788, 3365012291,
                2890451318, 2127795458, 2187590985, 4093939520, 343783955, 4242308697, 4157888926,
                2564079477, 4260723333, 221747269, 2643548554, 3833528808, 1763870742, 825076470,
                259966747, 1095002154, 931639969, 3831275768, 2810876375, 3231187357, 3440619698,
                2434935986, 612813763, 1378882183, 2162485107, 1140428311, 3695900833, 3397094290,
                4162195979, 2662618085, 3751543150, 3243820436, 3516498989, 707780244, 2079085110,
                448686345, 1895725327, 2371954578, 512402760, 2522669399, 3692228697, 4274764985,
                3682675544, 3852971158, 3189904769, 3141125752, 3990040037, 688067094, 4206021309,
                1411303768, 751179620, 1438168058, 3448172742, 917372130, 1562060869, 515503757,
                2481107086, 2276549351, 4210575064, 1210846101, 3889280108, 1149707690, 3998423549,
                450077793, 3069409039, 1205057532, 4241449813, 4263835068, 777002153, 97116780,
                1862501041, 4173027391, 538950023, 2024750625, 1067091172, 3975089902, 3595618138,
                1162626458, 2892893888, 185770841, 2582369494, 3927002631, 3209126940, 3017762169,
                3322211458, 184442973, 577978350, 564069165, 3126111172, 1558987521, 3158591140,
                2367515315, 3627578322, 3638689514, 2197481806, 387613, 3780351852, 2801314434,
                1919960640, 414138073, 161239652, 5955360, 4204709093, 49134550, 1616981665,
                2152668401, 3606726895, 1857761242, 1299061544, 3099986598, 3846390597, 3355161875,
                595322253, 2252701581, 2283337465, 675228606, 2653548638, 3430616197, 2380374646,
                3784410660, 1379122536, 3178229817, 3972552968, 3447966842, 1961475461, 918042062,
                1365972881, 2129295687, 2448607551, 2884167968, 3568778297, 2408690044, 469704050,
                304852349, 2906236085, 2120391091, 606392149, 1797001614, 3658187294, 2116257019,
                2708625014, 2498918970, 3223950057, 4105341944, 243352585, 3784332349, 522017452,
                1593525595, 4137737627, 167793374, 4087606703, 344557611, 2032896579, 2593781680,
                1431393249, 141052093, 2496087765, 2700336339, 516501742, 1876246139, 1822328287,
                2252547797, 4061227450, 2006777028, 3501565120, 3466396566, 735894510, 255684379,
                303584386, 1336037903, 1323979436, 1064067294, 3553551979, 2832942530, 1268770967,
                1027134407, 545735300, 3866535246, 2986685637, 1613305389, 1267557025, 31509612,
                771698397, 1878270300, 110063942, 4276370389, 3260973919, 2964940734, 1109302596,
                1411099702, 4020146211, 3169853494, 2389480541, 556890992, 1074682663, 1708332475,
                2658991487, 2894729410, 2296152345, 1870102242, 955236482, 2125245922, 1470432646,
                1889719140, 3468723644, 35016220, 2579705046, 2403600204, 2455839109, 738153092,
                361506106, 1501550475, 1570537857, 2512180427, 2079164570, 3284728752, 2525205632,
                1765298251, 324508463, 4289185078, 3262124640, 590432268, 3609992720, 1056480497,
                1521962672, 2902938979, 3633169769, 818640538, 3205732694, 3783707657, 1320287971,
                3427819133, 123698783, 1954647368, 1478731664, 729732421, 3485056101, 3384149754,
                886686555, 3524117645, 3672706053, 3467528186, 1825239229, 3605286168, 4252587265,
                2476737792, 1468148598, 1767076404, 3098062795, 2235174734, 3938633722, 3377437243,
                2055298570, 974103332, 1521104891, 1331281005, 3901922961, 2788866965, 1326440607,
                3830433543, 2349058686, 1891550867, 289592648, 833149915, 4062577431, 300173440,
                3618259462, 2437802253, 1988806398, 1196009062, 1693726935, 4261250055, 3358655485,
                1354639338, 4217909066, 3173323323, 1256774815, 2873946595, 783721831, 3287664797,
                1212500109, 544495281, 3943757093, 3869342488, 3580616897, 2288353173, 1381949567,
                1954109845, 1071065683, 1907373154, 1605862239, 399825655, 1608845741, 1366351932,
                1757744839, 2456500252, 4271660304, 2736932691, 1256196079, 654792440, 3432423047,
                814117095, 3916828311, 1478775709, 2623979306, 3502313771, 1912113011, 826313993,
                2137260095, 4198684443, 54736831, 380668741, 872309827, 3132896276, 1131528234,
                1143308601, 528742460, 3890747269, 2898367605, 3531963195, 2576564587, 2690837364,
                2676539983, 1748034501, 3722941468, 2157585074, 738387831, 161914314, 3886930899,
                805359701, 84535132, 1019698896, 1024816471, 2120663635, 4600563,
            ],
            &[
                436512665, 1372743407, 342109034, 2762999440, 1801741181, 1128075399, 536773409,
                2462344174, 4092132461, 1413817282, 3712114490, 446843050, 2277642284, 924335942,
                250799808, 1300467839, 3708770080, 346693759, 1462440153, 2514050355, 2607744870,
                572189306, 18842802, 85605861, 3453161777, 65109244, 1981480639, 1036404070,
                3325202117, 858703184, 1098140435, 137595022, 1796280951, 4121604160, 927712201,
                2983534703, 1277109179, 1842338521, 684621561, 3003060670, 1334445980, 3716987280,
                2509263629, 1481158588, 101476260, 2332533799, 3920104238, 1799849987, 289899178,
                3790057285, 3090879719, 1847908743, 2289570281, 899352588, 1237257524, 3478208269,
                2864532864, 4237285852, 2021619931, 2299181539, 2990516063, 1146232131, 1906740225,
                365885015, 436254990, 1572217307, 2276239798, 2938321029, 3411408571, 782341583,
                245501560, 1212705343, 3987607268, 2685812792, 3162630759, 525514772, 1738934514,
                4026775891, 675427556, 2638310706, 1872560581, 378254209, 3521602859, 3209772533,
                2171756894, 1916654768, 464606995, 1021746380, 1509799815, 1803124229, 1626950142,
                1757247127, 3463260849, 2059493644, 3328028905, 714292268, 4221410307, 3024039287,
                245477866, 4111629244, 2454388273, 2288605764, 4069258877, 2417991112, 1008037684,
                1584471268, 480491917, 657289197, 740153804, 3304298858, 3549803564, 3389482897,
                2285883880, 3754282415, 3452581852, 1300983995, 2563854813, 3387253149, 3144283055,
                4223958762, 2984826191, 3538501116, 1630274809, 3571211850, 3145706384, 119892266,
                3268130561, 2893468680, 3796377730, 1313792196, 84565900, 2317626527, 1585112821,
                2831830124, 3879773053, 3823642743, 260829176, 113541293, 2698035069, 1162236214,
                56045023, 2572314428, 3303774518, 786546397, 3304627813, 2532190633, 733612212,
                3194392916, 1538645070, 871903777, 346960285, 3824030942, 991737609, 660569608,
                25531145, 3125334834, 2019398137, 136254844, 2601227241, 3740691089, 1400597961,
                758494376, 1289737416, 176798002, 52677595, 1282844193, 2029711214, 1154925510,
                924664133, 2384558292, 2657323714, 3032100806, 2076618856, 3874064840, 2301862464,
                2613824074, 2408544438, 3475871678, 4286594061, 3888658535, 1599077513, 3091123524,
                691764425, 571533226, 643388614, 1946732585, 231085233, 3636702918, 3681581111,
                3187061512, 1902159644, 1741637618, 761538126, 1536383210, 579538109, 2344701437,
                1320685197, 429406282, 614226248, 3064576305, 3913009213, 1408468081, 2953160453,
                1860909301, 1257356077, 2326226229, 512992927, 4173707647, 4227234757, 2055824003,
                1797074658, 723531295, 1055503760, 3723048013, 1019848561, 1813977292, 2245679653,
                4188374262, 2764314541, 3543381795, 700976608, 1790968603, 1926021909, 604162486,
                3885171802, 886529425, 2545372123, 1099009280, 2144253074, 2286027026, 3759545247,
                65262690, 867405492, 478202482, 3860597229, 1950145935, 2338053342, 1301482230,
                1694465994, 665328168, 1639519384, 1246558025, 3758953070, 514219744, 1033396600,
                2585944964, 73215891, 1407268144, 3275177322, 713201364, 1552227337, 3796620920,
                464835142, 2341943175, 3119000857, 2088431307, 3897368907, 878918678, 1090403719,
                3163407018, 1730981592, 3537552166, 2597674257, 4098649693, 1903525249, 274610490,
                2448833117, 3013444547, 2285961959, 1309148292, 4284801003, 1457869563, 1131416271,
                1048930443, 2513500006, 1482041073, 1625081763, 4053050668, 2518621166, 3969148109,
                2761689155, 2164832372, 81475261, 3345442184, 3349299825, 2006252039, 3489857052,
                1658875101, 691754231, 849076446, 3627267661, 317685450, 4140537807, 2104452006,
                3534476970, 2383041805, 3170903305, 738640319, 3255529797, 1446723654, 1118811539,
                2937204107, 1614377664, 2969444277, 2952928746, 4001684504, 1718111127, 2073038934,
                1656108772, 3191239768, 1185532158, 562604415, 3971592886, 3000494224, 2019311576,
                3398207772, 1017177624, 2967316419, 1650769512, 2231731784, 3739468469, 1827669898,
                141217323, 4207733620, 2832371984, 1621792041, 4151096498, 2547492982, 308078925,
                3687686062, 2475718624, 3561209065, 4026891835, 1696468987, 3574384454, 2592420513,
                3300544973, 2850962487, 656760200, 2508381898, 419452052, 626276936, 285665799,
                3657891297, 554297412, 2239031137, 2012129,
            ],
        );
        // - new_n == 0 third time in limbs_half_gcd_approx
        // - new_n == 0 fourth time in limbs_half_gcd_approx
        // - !limbs_half_gcd_approx in limbs_half_gcd_matrix_reduce
        test(
            &[
                3657711640, 998557705, 2673527526, 140583600, 3560526410, 3079349729, 1007803327,
                208825735, 4004755247, 3170035294, 3362415892, 3152985538, 1012526106, 3432209347,
                2014829100, 829200636, 4195462629, 886243751, 4231479532, 3989085076, 697433600,
                4029974025, 2553256073, 2180771906, 589193852, 4067672710, 1710672160, 1276816229,
                1543905950, 784977036, 2190950050, 3053735776, 2278807238, 3135300984, 139130646,
                3731236261, 947882330, 397493027, 2887571315, 998281983, 3936346233, 1563553141,
                963440454, 1318816838, 4291981145, 919167051, 2962354621, 2048072300, 777973943,
                540037849, 3810855422, 3090640862, 821740314, 3090839706, 2989895484, 717348865,
                669874649, 3744157584, 491278687, 2028098573, 1277141730, 808716380, 2793035676,
                4006563562, 1685318670, 3368031079, 2624676111, 1345230900, 2205513093, 3750673160,
                1587034889, 1439572889, 3845769461, 4280964819, 1482349371, 1079078534, 4168075013,
                1635840911, 1957150571, 2178101877, 4115634728, 3398571758, 640182611, 1759062480,
                1075711136, 1010428620, 2276894745, 1504571688, 2596747558, 3855398501, 2838338760,
                2402020270, 2884702413, 1826928686, 2640885437, 2399131896, 3789994626, 4040600525,
                1305079453, 81899822, 2191284405, 1815628758, 2609804524, 758549568, 1578952023,
                518043841, 1501449207, 2578767272, 480891725, 2994717031, 2654624082, 527919434,
                2238167925, 312529994, 718208527, 2007076208, 586720275, 729308642, 1465364144,
                1706713224, 3777764105, 4134322226, 1156823710, 1507266584, 231421089, 144523751,
                1853362091, 1463359998, 305379000, 4240641599, 2721880467, 550468546, 4074779797,
                3504661731, 3933836974, 4064481517, 1555622614, 657068986, 2312879275, 2981257775,
                3082895648, 3388461547, 1981146054, 547361683, 3971819850, 1412877825, 2006992814,
                2288122621, 2206203029, 4226095933, 778245076, 1345823706, 3035370877, 389946835,
                2749124467, 4010410686, 339262243, 1609006241, 2258477917, 3569782595, 3637101384,
                2977823024, 426672744, 1521815787, 1263621653, 499965771, 456736400, 2492983378,
                1224546346, 3806307160, 175211074, 1054659817, 3941472726, 1740350040, 1438650030,
                2119348495, 2293521324, 3942515909, 614998919, 3084032385, 4144460915, 2482956785,
                503211659, 34511204, 1681765654, 3380532341, 3242282393, 551611855, 1154795583,
                201141450, 3394599938, 4032836190, 3946663852, 2007181123, 3278589677, 2773028926,
                1396191139, 1093235858, 2343752245, 3726488552, 4039496728, 1711775405, 2313196824,
                3792644308, 650938588, 2340124756, 2148862037, 1605804899, 2561768954, 4158774056,
                1104406522, 160746998, 3547663955, 2740053054, 297320527, 2288260483, 1727465300,
                1502488472, 191213708, 3240632447, 2763771729, 3316906616, 286363239, 427523346,
                1834992479, 1767311733, 2323090189, 777754150, 2637436451, 3864398159, 3878482105,
                3439516930, 3983770569, 2179596123, 3639743147, 3277930009, 132861371, 3016048414,
                2479346124, 772138261, 103363267, 2106856453, 3363312313, 2104064722, 2211640386,
                2790634720, 1094166731, 472640970, 3707415174, 2379763511, 3660706206, 266699090,
                2825551202, 3816959358, 3686617390, 2405346213, 4064944758, 580087497, 3705304882,
                804128893, 1526085410, 4259014614, 3424830723, 720534290, 546653993, 3743278090,
                1876012283, 1494835352, 4205298074, 165588292, 3110211976, 448529551, 3332834541,
                319442517, 580011988, 2786838094, 2921250286, 3100311441, 532629087, 3433284980,
                2073128881, 2951256114, 2435157505, 1486539685, 2906352365, 3727836664, 162399185,
                4051180248, 1338216875, 3019487912, 857100700, 3261403491, 3401769875, 2049869835,
                4188966169, 1293350219, 1594676200, 3476259820, 2694785494, 45498552, 1043174148,
                4115811420, 3167680253, 2550477339, 1707821053, 4259857856, 211137496, 198623401,
                429308573, 1520075160, 2496533962, 2517139935, 1168536469, 3699138961, 2800251863,
                1390306464, 1692109651, 2356645986, 774325761, 3399926958, 4110738344, 3817779241,
                2013343760, 2250854886, 351243491, 425031366, 2479242665, 3764357544, 2713277117,
                2512350560, 905949970, 117406898, 1332418557, 903482453, 4072632262, 1769369186,
                3494102231, 4127403951, 4081588534, 3220287892, 2278166406, 2040914160, 4090789198,
                1394480913, 1691201115, 1722070834, 3894412056, 1651961652, 3826548170, 3699211264,
                3183549152, 2920880337, 182084175, 102220837, 2221619377, 1758527489, 951379287,
                2855734431, 3213300857, 4263069418, 263677350, 3727218616, 689258763, 1971664130,
                181797961, 326897757, 3105490726, 232500248, 2261802360, 3085707207, 1389245554,
                3666172101, 2146196227, 1681117212, 1108350622, 441387235, 1914462616, 984521004,
                859059782, 2826588020, 2946730104, 1218457794, 245867746, 2189555164, 2126866996,
                1090542083, 3707825999, 2492424754, 2110587015, 9914155, 497858520, 2957494474,
                1624327398, 490541517, 1459216649, 310263141, 933264393, 1522660397, 1285180428,
                1470621088, 1023680737, 720481949, 4214305990, 1259717429, 3353303046, 987419142,
                2214746907, 932909422, 1788637997, 2735466452, 436276719, 1594193462, 2568346174,
                4146642666, 1937705165, 3620182538, 2069795783, 1080910056, 1452556953, 390339039,
                4100737699, 3512365053, 2731125385, 4062819879, 1591306552, 3854740242, 64478911,
                3073372522, 3345312614, 687806150, 1047499489, 2463969406, 1884499861, 2042651399,
                10170802, 936049112, 3939636351, 1291898824, 3075319841, 1421285201, 1306154841,
                1288723177, 2905666609, 2906581743, 4126000750, 1274137379, 2020506513, 837593817,
                3184876278, 4228491086, 1655288190, 2689287128, 556238442, 113142447, 324605392,
                2389209854, 2108685128, 4145207287, 3985615492, 3890009695, 3073605401, 990783580,
                960479727, 2794641498, 1841758262, 2666851594, 2839506601, 921776375, 2622653975,
                1051634901, 4138029960, 287681273, 2867814148, 4236186571, 3664092009, 1127349061,
                1289370427, 1166089919, 3232826100, 1090362915, 2063320477, 324664748, 756968253,
                2079873770, 237416454, 1960074355, 3038229413, 3265986384, 2253499694, 3259818663,
                4216895416, 1546526391, 954740893, 359758052, 2422581369, 2008469231, 270459062,
                2662846020, 198030360, 1699647574, 1189229658, 3064589775, 3086260774, 3492329301,
                95140180, 3679964930, 1719240364, 3308170694, 3900158813, 1824364083, 3912229897,
                2065183143, 1640649275, 210215170, 3595329392, 2936055969, 3380333840, 2729203827,
                1716703032, 3346477528, 2910262660, 1448045650, 3214180907, 3765279970, 2747456398,
                1754123031, 2253034236, 1951705358, 3487056132, 4179006228, 715674883, 1154440593,
                1890254496, 1287645100, 2931442980, 3677760863, 3113867376, 3657457141, 3252835590,
                1969239822, 4012698633, 3027155613, 3969623199, 2878407462, 2876709948, 2688525098,
                3727199561, 2234381059, 3136829525, 3554705692, 1831478139, 1674107447, 2082176817,
                670088203, 728773244, 1632752100, 1848846884, 1114000722, 775704401, 3844163200,
                469678943, 4133910816, 3435744844, 1661656089, 3919969806, 372912331, 3925680842,
                1366571708, 2821510836, 649494, 724284323, 505472189, 2687312723, 2405046872,
                3125301318, 3256049289, 774087973, 3981705603, 3325439184, 2954835675, 2725563058,
                2515754404, 3477776973, 4269454691, 2449194279, 1547625513, 1728063507, 231201570,
                4292877797, 3495163803, 4013580537, 1837665961, 1300013543, 1241432110, 3093642896,
                1691481026, 3921752027, 1059474853, 3218109461, 4272199222, 3090405739, 784116813,
                4280886718, 3547794686, 119045029, 214217180, 1172298704, 272007380, 4168588819,
                1539710510, 4048478839, 3293824373, 1932114687, 2517831124, 502648680, 1599065216,
                235634761, 2931327099, 1806135786, 2748758605, 4286102182, 2218543471, 4159247954,
                2851704864, 959147626, 2366564152, 3019979868, 2655370718, 281510209, 931090423,
                499413544, 64739670, 894424336, 1818208026, 2257234516, 2624162454, 3189431803,
                1134320613, 3166537004, 2926565501, 2669059446, 497238445, 30233754, 2620265220,
                3558625938, 352282231, 3147660599, 1340912220, 3276802486, 3012669529, 3553537679,
                326416777, 4279833819, 348274989, 2055151357, 2126716616, 819698272, 1389794698,
                2264090128, 204730947, 843480400, 3957036142, 1740808497, 3403300417, 493638729,
                3397686538, 2997162679, 4039958112, 3129102367, 4099024232, 4121783672, 3272745379,
                1139179399, 285642431, 648122033, 2734146014, 1523103819, 4033077595, 2558609201,
                3747324433, 2458964514, 4132460622, 1043448320, 10736,
            ],
            &[
                2854126571, 3810922285, 371837910, 2847540454, 486740183, 2160285633, 2017968719,
                1221435102, 1141701073, 1526766552, 1231337987, 1813385239, 2734884018, 235620011,
                3665183150, 1120279822, 2382922525, 3908597230, 2767187339, 4225970635, 2058878812,
                1054451375, 3908013355, 4090049198, 353761630, 943442885, 2028279969, 2755908669,
                3495701523, 2274876514, 7111411, 2322147314, 3166509422, 3631852760, 1999836871,
                650648601, 2896603946, 4278326950, 2519734766, 2042259469, 2175932534, 3377156271,
                2708233165, 1153846353, 569980209, 716406085, 1416372219, 2491866469, 1849390979,
                1657511720, 527987900, 312131542, 1483115611, 3019246366, 1772076445, 2597632097,
                1806628960, 602363143, 828919543, 4054351320, 745399979, 392681753, 729942081,
                3783820494, 958343864, 1891389523, 1916371046, 3957352530, 1940850968, 2999627271,
                2056442667, 3600581748, 739316794, 615343381, 2991048069, 963790214, 3579897498,
                2727465507, 1291721350, 2257779716, 189799771, 249939605, 2313870626, 3298318286,
                2695432027, 1104216368, 858091490, 3612066070, 3242257962, 2819622739, 3621623439,
                1645360864, 1858386419, 3690046232, 1963002266, 3365019901, 709477199, 1869694999,
                1333459016, 3929260115, 1804302107, 1548831440, 2447603949, 3613058855, 2294343811,
                3061585375, 1234689982, 1110502912, 1267669326, 2897966429, 1238959344, 2388962700,
                1545916965, 4015877004, 2291501123, 1151613625, 431826202, 2784246687, 1081585466,
                861824085, 3830712001, 4199076025, 1169739676, 1721601770, 486252424, 2456992908,
                2367988995, 853976555, 2362625785, 847070690, 3892469091, 189183763, 983145128,
                604862255, 1284385769, 1440014575, 1449024264, 1124161398, 992775577, 896708354,
                1499353865, 1733133868, 2463919313, 856157696, 3778499837, 3108165894, 3335953575,
                2577002240, 486391747, 3793776140, 4178059147, 1421326491, 2877075680, 121756272,
                2955126461, 1185519566, 439523848, 534575395, 1560554614, 3182072951, 3886491221,
                3273785996, 1722350380, 1102349772, 3163492322, 855840633, 749300337, 4268787245,
                1616840129, 3057312503, 3760176742, 3207150726, 491867592, 370828011, 1123930175,
                3909514255, 607439111, 864050721, 3004872052, 1802788201, 1624858827, 2832354028,
                72204982, 457231306, 3454382699, 2189149463, 3840204336, 1209241202, 1978336288,
                4169032732, 2876009661, 1082634619, 2497263384, 1326174220, 1253911629, 3737995297,
                3263043078, 645177131, 3799451502, 325159611, 1330703088, 441891512, 4125145352,
                149253662, 186351587, 4080844290, 2196709175, 845328519, 1100994658, 1730443741,
                697455139, 221172934, 284568041, 3633521112, 4012942302, 2503979252, 1982165755,
                1603237111, 3062410273, 2172526016, 3932170374, 746842837, 2685299075, 2226087030,
                2288686577, 3833519356, 1106715479, 803191120, 2945220717, 3462665442, 1851929745,
                3220631793, 825749594, 3850318854, 1995848704, 372995746, 2365949838, 620492561,
                1369591225, 223709365, 1954997080, 3882190656, 136576738, 2495749432, 3002183229,
                4087193741, 799453426, 942624435, 1615021967, 2904816939, 3821110584, 3804731230,
                539775187, 1076068431, 1596180969, 3090422137, 2013680976, 4160179801, 1501779020,
                3485041009, 627578448, 1687656645, 1097144289, 967159869, 1069388948, 3185056534,
                815670796, 358149589, 682430892, 1919067351, 1080492798, 1738690879, 3469244164,
                3048277671, 676563295, 793928921, 778744353, 1648323757, 2221906903, 2287524694,
                1039650177, 1679972277, 449929141, 1905719297, 3917418719, 2749202465, 2635810607,
                2907978358, 1764745285, 4092093759, 2136036160, 3823420732, 3781081204, 789220560,
                76055428, 1068573956, 3496138402, 707619794, 699429176, 1599250365, 1226916168,
                2953556842, 3013778359, 445642445, 3359206573, 1634979966, 32169773, 3986006899,
                2747561339, 2534736220, 62657613, 1511121562, 2122981684, 3246037727, 3781297188,
                1174417153, 2326429527, 2030416725, 4239201915, 1267322727, 4213670773, 3561818989,
                2314510277, 2980523839, 1790535762, 4031301214, 3864863880, 178369327, 518325210,
                2973773473, 1492340216, 2880227942, 1144890024, 2409210141, 3051276713, 1507874226,
                2982534148, 4129621437, 1186413065, 760037878, 152289721, 2233567098, 3431847497,
                2459547367, 3925952526, 1236723769, 785269549, 3929478183, 2566922363, 2036461315,
                2787863332, 2385369768, 1903832984, 2751226282, 2966793376, 4210685335, 110217992,
                2334089437, 832643681, 3824460110, 2884348488, 3553526804, 1696587587, 3077903646,
                3774351568, 118586818, 2535051277, 3044137612, 611368190, 2247957170, 4151739331,
                3032079141, 2398348531, 301800229, 2489467427, 463881677, 286741354, 919821454,
                2514396732, 1049973971, 2885185956, 4278356569, 1106751573, 2780358912, 3165013482,
                682728744, 156457261, 966913152, 1795511616, 1102483004, 47768694, 774401181,
                382571797, 3418933490, 1337271451, 2553642946, 1682481208, 186752586, 402086066,
                1253357008, 636687570, 1174558316, 3075513694, 3581054104, 1520081698, 3462279363,
                3127642491, 3337969953, 851993452, 3859511913, 3125432584, 3565365475, 3639315630,
                1929579809, 1245576721, 3896701628, 3934477337, 4087201647, 3275038136, 1788711123,
                3958067369, 4089205000, 231641077, 2231523824, 2619997727, 441912559, 4072770613,
                4240978164, 2974598062, 2924678733, 2970538057, 2736142575, 1105201139, 970816157,
                2369699954, 1887951773, 4171767668, 4285132008, 1776816709, 3770203858, 2382945,
                2017961118, 2832376676, 2804837768, 3274902934, 3591733883, 3691479138, 52712933,
                2612228989, 3628704902, 3513862898, 478216119, 2726352952, 804977216, 3390084015,
                4081495820, 1671436017, 2848400264, 3656019845, 2136393241, 431834724, 3510727226,
                3071482565, 410753183, 1657700546, 1058987404, 489895964, 1587210753, 4111636005,
                2918121105, 803516926, 638869931, 2234185434, 1507756257, 3908703942, 810746259,
                2117398521, 3273771795, 2327709962, 1345338774, 2380428267, 1886210982, 3272617401,
                2424798876, 2485064558, 3441956046, 4282650102, 1189648611, 161669993, 2174805813,
                46186229, 3870678520, 1725393849, 2725996937, 2285861609, 2855918656, 2085727938,
                3005980273, 1091105108, 2274905128, 3124240963, 3402918292, 3948459386, 3730327554,
                2624673727, 1462493475, 1911926536, 1172198609, 1531781228, 2875744790, 955061915,
                3653212440, 1316441536, 1571847879, 117813941, 724461511, 106145092, 3007920163,
                509355349, 2361882009, 3290661803, 3009193891, 2101167937, 3236175251, 2600604339,
                2409592198, 267656105, 28902868, 3787222075, 3910267520, 518607069, 78112248,
                1231828511, 2237810649, 2039587423, 3258311695, 244193578, 1901807321, 1888614436,
                724431620, 1229615953, 3602175515, 107889915, 1499981622, 915728462, 3185158200,
                3875556208, 1422223958, 74968146, 1782255156, 3690901362, 3148861984, 285887046,
                456047577, 1877634221, 4119417329, 2674004542, 771859946, 3841364442, 2576022811,
                2774717438, 3707475262, 598699796, 4086485722, 2071933255, 254807101, 2713526148,
                4015941391, 590180295, 2462439133, 893362735, 2776258951, 670583777, 3998467192,
                4252701293, 3144240767, 929429555, 3121145790, 2186463475, 595824244, 1196211096,
                2796145353, 2313718767, 2327057115, 2600569013, 1375960388, 2419292403, 3991920167,
                3660567838, 3488689798, 2047799775, 2345831879, 3333113070, 1392141953, 3356864686,
                3683415162, 30718491, 2800440004, 305780227, 2420043106, 1237507888, 1479858831,
                3600187970, 3379229271, 1429268544, 2752590068, 2152897287, 2762740595, 3499503442,
                1412039428, 1580413636, 4255719498, 3321537024, 978632041, 2570103234, 2866926493,
                714621483, 2791004236, 2495536081, 343144324, 378104110, 920987435, 3182143689,
                1200960749, 1263985718, 1422494241, 235668853, 1459938722, 2182331117, 1534070260,
                1037022387, 2359040698, 277760467, 231493066, 4156892106, 3785706492, 1644416142,
                279187889, 4046443858, 3644810161, 2159180570, 313719967, 4093063716, 1479097649,
                3123510773, 2411818036, 1854765137, 6659591,
            ],
            &[
                3522409411, 692426266, 4252687070, 3221722249, 2259982318, 1572230342, 2447667874,
                582309214, 3464041890, 833370466, 2502005176, 866026260, 3434778270, 832555619,
                577419378, 355707174, 2758409995, 539088491, 4165515733, 3525341005, 70966281,
                4112863463, 18093477, 996208336, 2996281883, 243025039, 1724269073, 3456564031,
                2183707540, 2623790820, 4068996778, 2003125751, 3979669707, 2092158727, 1319454084,
                233637378, 3444775191, 2047163977, 1305370049, 4052656807, 3362300485, 2153220367,
                3451943873, 2659951731, 310435720, 240435490, 1283149803, 3779214360, 1819576764,
                3267732950, 2661843307, 2263394946, 1901303373, 1858196903, 798526878, 2800696617,
                3224809646, 982174624, 559354368, 2472934158, 3806202203, 2295203592, 2945471200,
                2663904303, 1357384963, 1650942869, 796482443, 2766778986, 1604012049, 1236473541,
                1483190475, 1132242907, 3470424624, 3256605783, 1372433796, 2572130477, 1005422111,
                3417699569, 1998572418, 2781008338, 131034958, 2366709018, 2158499536, 4239137506,
                2147101299, 3836689278, 1767372944, 198678316, 3099405709, 1642856762, 1359859087,
                1016829435, 1775642165, 3166947118, 1370994118, 3565882683, 2697435592, 2988614686,
                700091191, 2139458915, 293125924, 1677498485, 324444909, 1956567030, 1647523339,
                1288409259, 3433527205, 3488399404, 2692884632, 2629401949, 1865373164, 260321166,
                2452146960, 1510030261, 3309904643, 1821006233, 4092693137, 360274707, 430743857,
                1022845846, 768344651, 870695548, 4244607652, 2799849346, 1897317934, 907494911,
                3494316912, 3319466364, 3831408243, 379446458, 1637847888, 761682818, 2305788385,
                488150578, 2591607856, 3487487084, 3482200336, 3409152717, 4191475557, 2349556031,
                1024188774, 2357767930, 3538056694, 1299385038, 863893778, 3197830576, 3620847361,
                3734458112, 2102579497, 2299090284, 1776637937, 923893333, 3538505849, 1525313260,
                1095090443, 1425332433, 4270279191, 3899256887, 2234522273, 3938133714, 1144096051,
                2771405273, 4039853365, 2164168861, 2220073157, 3305905250, 3547017695, 3028576138,
                3211205289, 2969717301, 1297821450, 3898348269, 536907576, 3526521253, 2984151429,
                325158808, 3634226110, 254237584, 1647629802, 1213935217, 1963529116, 2864575925,
                248508991, 1681221168, 2138605160, 3011363057, 1667991051, 2164842621, 1221146487,
                1896090790, 3584296136, 1197643020, 520819114, 2845411935, 2882123631, 86651362,
                2840429882, 305685636, 812868855, 4270240044, 2258043791, 3681398458, 2125157433,
                3400174505, 52122812, 3977731352, 2960080531, 1752547412, 1789316668, 2086783129,
                2065283260, 474350220, 1392633188, 2642559453, 2623416931, 1213699470, 431976342,
                4191881681, 3989646787, 2555136586, 331973965, 3045351568, 1974527188, 1721946949,
                2420304698, 3018807687, 3912284737, 2323363082, 4175374785, 3073974039, 2019051853,
                3254436577, 4145054805, 2909559648, 1674676821, 618050625, 1701855694, 3844330576,
                2368970459, 1142540548, 19839276, 2809741072, 1358679745, 1616340555, 1321571980,
                4269787203, 2139498091, 1516379787, 4232094406, 1813501895, 3499360571, 108430199,
                2969637702, 953766617, 746993329, 1798819970, 3008070664, 29506047, 3755689428,
                1731853786, 2150972169, 3970252289, 1707195359, 150651832, 4272796881, 3214714265,
                1289674369, 3832009122, 1863256348, 4208461628, 2928731478, 1909075615, 206649607,
                3721679533, 1149140532, 2135144822, 2553592226, 390359104, 2664151758, 4207827335,
                791092034, 1418111790, 3869230526, 1229201988, 3450535823, 3141749817, 4170511491,
                1808234944, 2870573701, 1933266254, 1944630510, 541620618, 2484636520, 3899287047,
                837911245, 318074222, 778344163, 332143424, 3148535492, 3257541807, 2584276077,
                3752887648, 1145669314, 2724633195, 168627417, 2600193851, 2188347881, 315294329,
                1951166123, 3631461291, 2699587438, 3705790078, 97682065, 3212849906, 1644271056,
                2402937425, 1495234346, 2546518537, 3699134635, 1566766581, 4076379355, 1369185061,
                1927352427, 3453847057, 2121073778, 3846264460, 3345029574, 990382258, 1093055554,
                4020639133, 2106873871, 3558358079, 3623019498, 1126463853, 4000595941, 2326522050,
                2496119497, 1177412487, 3737033586, 1970672443, 4124458184, 2182522865, 1950531515,
                2217610303, 1875359075, 887126337, 4187532100, 3050435188, 1382650461, 3459929976,
                1249535657, 4069389042, 3437440772, 2924920125, 1065376385, 3759927287, 1835638851,
                1549066656, 2607123205, 4287679131, 1933130208, 1861085748, 3531803620, 2455707011,
                733561985, 2945180228, 3707942192, 3580180904, 3375170391, 1690619419, 3658664211,
                3096941018, 1587283463, 3941503860, 3163621426, 526869998, 1316208739, 3776069458,
                3759213667, 3343111698, 3552349239, 1698745515, 529667942, 4250086784, 2328783424,
                1942833307, 2532170555, 1407840798, 3929450154, 689766216, 603422970, 2945072222,
                3148401239, 2154639678, 2439164182, 1799459040, 1555139265, 4024153015, 3988650657,
                3734125279, 1894805848, 2147080077, 1860482341, 3275095036, 3629120956, 1637523200,
                2484855322, 1006412, 650874934, 3277326964, 1186385437, 3787893694, 3384961602,
                31727330, 1019763729, 4132753588, 3078074634, 1730195820, 1742210061, 3980710682,
                1476247656, 1537422777, 4040062712, 1425312769, 1848918319, 4087493673, 2365974768,
                1886134839, 3875551706, 1685012340, 41954153, 3951521761, 2564745248, 2519620712,
                3959816490, 2560867112, 344633036, 3691137569, 1256857877, 2291066279, 2054203203,
                1386019941, 4272539404, 3363945588, 2226687874, 3871655974, 1494082150, 1112096467,
                2284368903, 975044169, 523309601, 2133056593, 3313692829, 904329125, 3666815061,
                222106749, 1616520314, 2393855972, 3705259257, 860827649, 37212776, 2059317736,
                3079163663, 932474881, 3394576876, 1706615928, 1819714696, 3820968510, 536368964,
                2910847848, 2135453698, 1350335129, 300212611, 2633354050, 153183294, 1928772583,
                1021126237, 573543692, 3567442837, 1409698831, 3111277745, 581910616, 1979561780,
                396494706, 2463303563, 3166095864, 3105677740, 2400257412, 1913519467, 2416376547,
                1957034146, 556585253, 4210367449, 2539364372, 572061442, 4289883802, 2215328126,
                294061050, 2040995404, 2952991336, 1040316946, 720150954, 2348132965, 3885697361,
                424223697, 3805769810, 554628633, 1894997829, 3738308930, 2890140217, 16184469,
                2112778335, 3492087017, 1528198629, 3156309847, 1672825355, 2320695680, 1770693922,
                4158833400, 111430926, 2523417395, 2654875625, 3108094520, 2689255279, 2038117485,
                2616876984, 2352131353, 3150209189, 3217446658, 836678783, 2785826980, 3252708922,
                964549716, 1140915598, 3033957610, 2356630981, 3514265925, 1278364234, 1083297932,
                4023829085, 3614439116, 680554138, 3997456824, 2252713625, 2055817074, 1234522698,
                3102015221, 3404334905, 112882014, 2007560807, 1487607631, 3501350871, 1111627245,
                980132418, 3206101949, 4130779405, 2111302372, 1968609288, 4204601547, 2874374202,
                2853878438, 2857752228, 274256864, 1817723765, 1905422451, 3363188610, 3333488106,
                463317661, 2351076159, 1477837401, 1418561892, 1866844722, 303232671, 140367367,
                3061206542, 1503639539, 589823218, 1918827275, 2600297211, 4277833048, 2393528802,
                1661440636, 2339324312, 1263101253, 2873025067, 3441016568, 194170142, 2576615806,
                233994937, 2966168464, 3087711309, 1530686231, 2308302598, 3393973525, 2467525628,
                469950892, 4103299607, 1794837893, 1549361853, 3580558511, 12661541, 722609243,
                965007319, 449849784, 3829064245, 3965630564, 3112166475, 3424263194, 195830868,
                3230014161, 201295863, 1892682525, 2243429350, 678386870, 3696010396, 367065208,
                3856657124, 3741113040, 3014123703, 739968874, 413,
            ],
        );
        // - xs_len <= s in limbs_gcd_subdivide_step
        // - s != 0 second time in limbs_gcd_subdivide_step
        // - new_n != 0 second time
        test(
            &[
                1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4294966784, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294901759,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967291, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 2047, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 262144,
            ],
            &[
                1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4294966784, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294901759,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4293918719, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 536870911, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 16,
            ],
            &[
                4294967233, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 32255, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4128768, 0,
            ],
        );
        // - n <= s + 2 in limbs_half_gcd
        test(
            &[
                1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4294959104, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294950911, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 134217727,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 4294965248, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 33554431,
            ],
            &[
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 16383,
            ],
        );
        // - ys_len == 0 second time in limbs_gcd_subdivide_step
        test(
            &[
                1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                4294966272, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967293, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 2047, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4294967232, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 65535,
            ],
            &[
                1, 0, 0, 0, 0, 0, 4294967232, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967293,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 127, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4294967232, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4095,
            ],
            &[
                4294967293, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 192,
            ],
        );
        // - n == 0 third time in limbs_half_gcd
        test(
            &[
                1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 4160749568, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4293918719, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967279, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 2147483647, 0, 0, 0, 0, 0, 0, 0, 0,
                4294967232, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 16777216, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 67108864,
            ],
            &[
                1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                4294967040, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967279, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967231, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4095, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                16384,
            ],
            &[
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 15, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 64,
            ],
        );
        // - xs[new_n - 1] == 0 && ys[new_n - 1] == 0 in limbs_half_gcd_matrix_apply
        test(
            &[
                1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 4294966784, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294966783,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 262143,
            ],
            &[
                1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                4294705152, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294966783, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 134217727,
            ],
            &[
                4294966785, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295, 4294967295,
                4294967295, 4294967295, 4294967295, 4294967295, 261631,
            ],
        );
        // - s + 1 == n || ... in limbs_half_gcd_approx
        // - extra_bits == 0 second time in limbs_half_gcd_approx
        // - n != 2 in limbs_half_gcd_approx
        test(
            &[
                2004596728, 1848887139, 3588545405, 3004392306, 853879983, 1360524256, 1385705184,
                3321803032, 2791968922, 3892438424, 1781322303, 3493068948, 2015405293, 4074669885,
                2900269078, 3823683574, 89252947, 4038834041, 3432017782, 3193194315, 2816970936,
                1259222869, 792476452, 2969025189, 2789393972, 3016742665, 757499223, 11545282,
                3423520861, 3616113932, 3465820395, 1979030055, 2104068595, 3714106365, 1466660150,
                1712602216, 2630941262, 610797776, 4074395197, 3701243311, 3555176782, 562321145,
                3934144323, 2134957404, 1613994064, 2254770945, 844351745, 803798570, 2703265082,
                3640524712, 3907579617, 1740863700, 3961951685, 2756477900, 1800563792, 4006854388,
                1927911032, 4232194641, 2895575388, 1248300427, 324453294, 865512796, 1640531918,
                3337883058, 4127691867, 1251838156, 3755093876, 2638444353, 2277825898, 1079070681,
                2347582744, 4086131979, 1292025334, 1141464465, 2190524620, 3815677488, 3218393458,
                1168612800, 544017320, 2324397425, 3767722505, 2551036255, 749770338, 3505489601,
                3557482154, 3813020533, 446277331, 443847294, 4085812779, 842889652, 494048275,
                2419464445, 1214470431, 1228587913, 4005029074, 1143269930, 2427231859, 2933315483,
                2923633652, 1889084749, 2411174117, 2305971895, 2959154784, 2927615709, 3241695251,
                682460868, 653924800, 3814161525, 2508305181, 1339320678, 3455333037, 4176128090,
                2539010236, 2893402188, 1241794103, 3954344331, 4233454015, 3681840959, 1878852448,
                4262815930, 166349449, 1235728789, 336877896, 1367168363, 2965431878, 3447952070,
                1714138221, 1049768737, 1231199029, 2910381689, 307859398, 899270841, 1606345579,
                1234775041, 189535788, 1984838421, 2087074239, 2867133699, 4209429577, 1098566748,
                83344395, 1521074408, 1698236631, 1907863376, 1923053177, 3422780594, 787031035,
                3522524050, 2028229229, 3031046586, 2851262763, 2888070447, 2834519912, 2885252087,
                177502086, 3275331517, 2442998599, 470146074, 499318359, 4217123754, 3089228461,
                1039758505, 1609847346, 1217631005, 3141899940, 1645363140, 4180648912, 1348926971,
                2872045994, 2516596487, 70751717, 3755810719, 2844208311, 860932668, 3025234253,
                4228898226, 1667541479, 3791359976, 430657658, 2775582388, 3472378627, 2507534646,
                292448923, 166266046, 3517783361, 2602204453, 3143313511, 1907742516, 2093632780,
                3391286313, 2706298791, 1700209801, 646869730, 2296768972, 3445646575, 4251992105,
                2910039672, 2099034896, 82722938, 2958905749, 1059663844, 2996112985, 3875276435,
                3838463324, 3607053228, 1649011648, 1013811619, 526117160, 1012433449, 885794551,
                1704687270, 869082082, 2736112342, 1882492056, 2952317080, 4088973917, 3059951449,
                548119577, 134193431, 4269903209, 719387298, 3039694582, 3312858765, 131546364,
                3768977754, 1008091338, 736625, 100297344, 2441670905, 2960124427, 4187608134,
                1281665081, 347304892, 425777148, 3815705055, 2002210752, 3158348324, 3370719815,
                235350445, 3073024597, 2841606549, 3677106047, 627054432, 1463475055, 2306894862,
                2004510531, 1215369345, 3796695068, 1705709666, 1910826455, 3380653290, 1669034431,
                1326881073, 2691232035, 1345394646, 2505725768, 1643208418, 2278435319, 3975379348,
                936580608, 3558291381, 1578739386, 3360990083, 1367648693, 2174608196, 3784418662,
                4096285583, 1125126761, 1702879279, 2373900154, 2794269811, 892965648, 3431450476,
                1598114090, 518942123, 3942271859, 163233596, 3635705911, 2802569655, 3902026503,
                4172233752, 1343615033, 1443283240, 3214860685, 1830135384, 324092050, 3595506427,
                4006369796, 3094883604, 1337811945, 1590934260, 1405064877, 1629961311, 3917976233,
                2307585818, 1966934539, 1130062423, 1870989582, 1624138122, 363138090, 2804418650,
                2247302844, 3205057981, 1897671604, 1838684004, 1004563089, 3421753356, 558428644,
                1914315130, 3772633601, 2526221813, 2274129897, 877924724, 3120882592, 3985407655,
                3151825377, 3497165637, 4225852164, 3537575972, 1378449691, 3428657854, 917817375,
                527806996, 947800429, 136564017, 2885607105, 73317183, 265413671, 113274788,
                3223321193, 210252458, 2455534679, 730413621, 3248846851, 904264433, 3785997086,
                3749983772, 3451619934, 1253036056, 814792744, 1441772826, 2902751149, 1233888488,
                3433013081, 1418759256, 924600323, 3080242546, 1505445983, 983904026, 3815022296,
                2057785068, 3901544799, 1289567020, 3045444045, 3571691925, 2469383112, 2902682391,
                2637073877, 2288505094, 1098867737, 941340356, 1780640923, 488687629, 3661273470,
                2018461290, 4073895479, 4219286255, 1226075625, 1286996168, 2455678338, 544007097,
                4088835826, 555104214, 2207823194, 3598396023, 742075093, 1931088791, 3231680922,
                2207259058, 1054824417, 2809061923, 3659187103, 667505786, 1038625820, 2546526131,
                100718813, 3971349256, 2689176731, 867921223, 216238397, 2926583018, 2524616945,
                3105675814, 1929564974, 84832194, 2898985519, 1538371806, 2304769426, 3331366264,
                3992871197, 2524313399, 2880073024, 214374078, 3087979230, 3763462972, 1492033090,
                2745547667, 1590203017, 2220884647, 1043643074, 270180523, 3619501678, 913269760,
                432169997, 1560222504, 3606034447, 370004374, 2242741806, 161846260, 1611153441,
                2788329497, 221705084, 1562296329, 2602384573, 2489408324, 26752274, 1012421860,
                2870608198, 3339203935, 461343907, 153553549, 769161583, 33015466, 185839789,
                1238122110, 1955418650, 4034473489, 258622964, 3142360380, 1154801666, 1938695011,
                3543768243, 3169796551, 918987546, 1837828018, 3174207376, 2490048788, 1968348654,
                981031537, 2019290428, 2481414836, 4075697852, 2177287359, 2394194295, 2918955032,
                3990490756, 2721846619, 4831810, 3676889722, 77410574, 89746890, 984555749,
                1700971854, 1157576335, 2282651337, 2161988414, 200058435, 3928266699, 930309134,
                1501872778, 2708771958, 700076188, 406467641, 3056590431, 3360765111, 4031233736,
                3421740854, 977030487, 99478473, 1098197595, 344694600, 579238328, 3538505442,
                2636219428, 2819718878, 1017360877, 4122930426, 1031875816, 893936384, 2187396525,
                2651789727, 1406131552, 2442039378, 3802582287, 2558943236, 1638234317, 1996245621,
                3369039129, 3228698584, 2117350450, 790103060, 1066928337, 2234731599, 2618583709,
                2555385033, 1780760113, 3641524458, 4150990164, 903504072, 4053936972, 3149614166,
                2337971588, 890882883, 356711938, 2263542728, 1364381326, 3810269234, 176857342,
                3484849272, 545650307, 4095119697, 1070867675, 44185550, 3245052385, 512603627,
                3259006013, 1823968935, 820533825, 1968169710, 2512600375, 3565879518, 1297163908,
                49040429, 3182956160, 508986908, 410825498, 3306313947, 993237936, 167382405,
                3237055914, 1596868278, 149136283, 259065124, 2117102457, 2524094604, 4026729163,
                1740524001, 1910248499, 3900973553, 963142543, 1135340442, 1969527786, 4217058880,
                785182291, 4104638176, 2329275390, 3800005728, 3604262878, 2318748672, 2241924274,
                4146333651, 4280666223, 2529283140, 4007762691, 4137420426, 3087334197, 1001749081,
                618981599, 2290019564, 1030626743, 2353440201, 3491060669, 3215041527, 2133818684,
                733955838, 1074731940, 1013602989, 1745515239, 3242057247, 3498788174, 755386347,
                1719467948, 329147897, 2110573052, 2651634797, 2329469633, 1756488056, 86346262,
                3905621145, 2947376796, 3505737560, 1900253843, 2879666387, 2157905472, 1776782776,
                199667884, 2726782520, 373441468, 1768340110, 2462560802, 507615514, 1956022588,
                2384919250, 3412435481, 3320720595, 1941716387, 2023426521, 920190565, 3248821718,
                2537144431, 2247676695, 2897967965, 3195719172, 895846478, 1897877802, 2598950252,
                3385294109, 88060678, 1813045139, 1710194386, 534922353, 4115328300, 1579427575,
                489224725, 3942695965, 336077582, 1683601440, 3996082755, 4004864353, 3956008235,
                3160734247, 220270324, 2211225079, 1810415176, 2384394389, 3863744894, 3644215752,
                1441189974, 3953466009, 49244730, 3292932716, 982515392, 1412561304, 2892516725,
                2889338602, 2133629116, 333644531, 586020549, 3757651544, 3094771642, 3962017540,
                1922803563, 2852357730, 3982002373, 4184765592, 1490335806, 644006294, 3503622441,
                191971406, 1321085451, 2933347886, 2298585985, 3942234867, 2711863789, 4287811909,
                1869646329, 4193678178, 2300694512, 1672898480, 686525422, 29702670, 2048484513,
                3800848401, 1016390476, 2542462976, 1630627170, 2566172213, 3799057710, 826467738,
                2081397119, 3825871484, 1720343261, 2134275403, 582093436, 840829215, 3070534680,
                2452108732, 3836992562, 3614708040, 642988748, 3799027792, 4279250555, 2415353135,
                1431887958, 2718502020, 1396816819, 1619130585, 4148081938, 1571085815, 967105049,
                1826809732, 1669740739, 1082141732, 1397004647, 778435201, 2046386989, 104029467,
                2259777826, 364934737, 3269047887, 2763055117, 230291892, 451391998, 1891814456,
                831038728, 2964787176, 3635108912, 1922422114, 3925407285, 2321391206, 2465927715,
                4166210675, 1197190708, 2294797824, 3526000914, 2502435952, 3203138648, 3372900543,
                733841109, 3114196707, 2959722855, 3974072908, 125712163, 3949935156, 1192064193,
                1085374669, 3982672572, 845787141, 2041667785, 1091167459, 4191929987, 1653040596,
                488154413, 960754299, 1195687062, 1016099064, 3052206013, 1445590992, 2800180480,
                2391011353, 2595121989, 865889910, 3530257661, 2435488454, 2218519781, 3043715764,
                719268352, 4118610915, 2644321404, 2688437598, 3635531668, 1233398325, 421024892,
                777080283, 364599954, 3262345441, 1278614086, 1903481893, 1108104645, 3477418707,
                2459845146, 2161754027, 3572822544, 3213552665, 667392374, 3742233712, 1344617923,
                3673776960, 3333792099, 4171297121, 2361470285, 2456722922, 294864113, 184980340,
                4022019397, 1074091825, 1871736332, 874717959, 3297537109, 2857811966, 2634096492,
                3743762943, 1543744992, 3070296783, 1414890392, 3534170596, 2874392428, 3120304495,
                1883738483, 1248660253, 867059402, 1991084538, 2294883679, 2004356063, 920311728,
                2402442236, 3400179891, 4246588249, 1364011692, 3510199169, 637783028, 3953751884,
                3539712763, 1197923989, 3315532567, 1772695848, 851905130, 119859671, 2150435159,
                709509942, 559611034, 2881395845, 2070876910, 575568326, 1904348566, 3178937360,
                4124642603, 2770017354, 462709647, 3534605840, 1958609374, 4276935967, 377299139,
                4184852657, 2339718928, 3948556371, 3884870239, 847308351, 1567877006, 942676575,
                3013407713, 1391345145, 1561541328, 2032670783, 2115922540, 2840947915, 1424667973,
                2625051425, 142557574, 3608928348, 2857192866, 1885061732, 1032209703, 1670027307,
                619942102, 577354863, 1991193473, 4106125554, 3897353387, 731765970, 2111758919,
                2952189172, 1128764158, 4294956501, 1552565742, 150742947, 1645656939, 2396412695,
                1598935993, 3811058692, 4282926397, 2028513885, 3417133207, 1960806103, 2987468245,
                1261155071, 1955858897, 2337563115, 880430719, 3940851795, 2851862439, 2837360725,
                998053826, 1459954608, 859053755, 1064753371, 2745438489, 3442260013, 2618174,
                3064510597, 1526197768, 1172088455, 368916865, 4081414092, 872579574, 2606125510,
                2393005057, 3895670156, 2773169580, 2784687800, 2713793780, 824327507, 1882474218,
                4007720140, 701440179, 2854102433, 2485031446, 69057080, 1788242193, 717689363,
                3404715444, 2079426393, 2150736275, 2447994641, 2919217268, 3473535129, 3831746162,
                73654495, 2724500951,
            ],
            &[
                2048467029, 459202127, 4051209999, 2641158559, 2529283231, 785114584, 3478830982,
                682417094, 4137477036, 1994886259, 3849756551, 1850485567, 1510562924, 3626191361,
                235960481, 980403409, 1789076459, 723101310, 855767948, 2804845518, 455915126,
                3615072101, 3090232230, 466533522, 1169083525, 1158162624, 1858763072, 3595675614,
                1661517062, 868777641, 871705130, 2861391047, 500497205, 2878901309, 1180001451,
                686942037, 1956533284, 1920395594, 463032751, 1775551241, 2962069853, 1379946573,
                1204600053, 3490263166, 3319964396, 297358647, 3127096944, 2886152999, 54379252,
                3229449581, 1810608641, 3761727243, 374040322, 1887571393, 3146210360, 3708571160,
                2205846452, 1001286939, 4203438216, 1231239932, 1865353368, 3994983472, 1326013992,
                1568675894, 2481470254, 3886943594, 2173153009, 2709989031, 2174267461, 646584830,
                3616102992, 4053133949, 481146194, 650153841, 2339337832, 2740187083, 225432613,
                589865866, 843055701, 3020975970, 265007788, 1116775520, 2919330007, 3346158850,
                3487640422, 1169247910, 997606069, 4010912587, 578436020, 4290033930, 3633599003,
                2580529737, 162726999, 3743750914, 1981881618, 391172274, 3576970038, 3723023159,
                3307886769, 3395118745, 575247113, 415887104, 1534676348, 2823996659, 4269453883,
                760173732, 2138038336, 1086521291, 3199196489, 781423556, 1357123249, 4195648284,
                3759296877, 2958948707, 3878993388, 3483121017, 2443771071, 52664592, 4239525630,
                804231160, 1853250793, 3941273196, 3701311485, 2618716701, 56163857, 352626830,
                3422477586, 3132969079, 3117007699, 181228357, 1045692359, 2223063843, 2343808063,
                3122019574, 1709080652, 1254416114, 2121822259, 2976997459, 1780344599, 272725612,
                3902688101, 2445075321, 3015383251, 2720256116, 1965686532, 3674920947, 2017493119,
                2443555429, 3245745772, 4284800850, 618003105, 2592784982, 3140704668, 1643367542,
                4252514361, 3843724900, 217510595, 3357765030, 3943128900, 3880661463, 369505618,
                4186357727, 3349936627, 1429436215, 3729602835, 3738804567, 1753791419, 405876993,
                2800106006, 3445400414, 2334940713, 365540818, 2000813054, 3402754389, 533916330,
                387189415, 2847853159, 66632878, 1077148, 1396928998, 4089775018, 1213790544,
                1059352194, 2514176069, 4257435125, 1261407550, 3530893728, 2451914750, 3905023603,
                3254919488, 2751582838, 2008398588, 3449851963, 1812431992, 3571267520, 4290833338,
                2474346395, 3378814420, 3351611512, 1303922942, 3283346580, 1489524774, 1395585068,
                91861671, 2943769869, 3412508982, 2507344244, 2586024774, 660828217, 522672644,
                2554574189, 4093332175, 1125997685, 3592656249, 1036705155, 2798764932, 3441576709,
                392339390, 3489310669, 2596866038, 1947787118, 3696312887, 226298054, 472935459,
                505119706, 3783498850, 867650894, 1072853134, 1072568479, 248365064, 3418187906,
                1331699784, 3000650397, 1715193485, 3810874498, 66785355, 3344423276, 1996144563,
                2793928046, 2180574544, 3767096859, 1540549268, 2389926167, 4274062787, 4121401190,
                442584702, 2431090696, 3507775591, 710194452, 1159458607, 2416059535, 1642122860,
                3823480937, 2535544964, 668845548, 1661633586, 2857928640, 2314741510, 377317229,
                2483955636, 2957445644, 3934139513, 238200907, 2969751595, 4212529507, 2281068873,
                1301743382, 1952773150, 1192593223, 3892828784, 4037269943, 1148821772, 391839522,
                3504559806, 4173379790, 3085121003, 1474792696, 628152284, 2698025247, 4076939937,
                2416300665, 2652295225, 365441928, 670206063, 1553108013, 1932716485, 1558658987,
                368881025, 2692123237, 2519631847, 3865945719, 963612609, 2001573437, 161034038,
                1883789309, 2496653544, 1909132409, 3269239063, 510666825, 2562655909, 3241797641,
                1506218500, 2991308436, 944858691, 2394342093, 3501775324, 3134551643, 1655397291,
                1372472782, 1117006576, 3017788232, 568720212, 1600706831, 1262117840, 549706387,
                2176848659, 1793986036, 3712714767, 75684922, 3737373740, 630177306, 343378978,
                2300438472, 1144716485, 859910517, 846443936, 3312516505, 3320108534, 1661760387,
                1112699815, 1474296780, 1964776416, 724080825, 3106608638, 1122986017, 2775133042,
                988101895, 116278673, 4169511163, 1001735315, 487463586, 1705207220, 1191915788,
                4098138272, 2818005531, 3023730900, 2655878164, 3026173793, 1966505099, 374525789,
                1651006603, 3071722751, 1288077227, 2811531267, 3902509872, 356900923, 2715149970,
                2252856041, 1766699937, 430591715, 3645482691, 4287397379, 331671751, 2921221743,
                4164993604, 2710905723, 3151975289, 2670586646, 3232311359, 1487120535, 104450214,
                2011996119, 1378669544, 260533527, 2931634918, 2932587506, 551998172, 2217304186,
                3692803012, 1214595277, 2200712189, 31026051, 1314881730, 4252874943, 3922112067,
                841640375, 2518926569, 4060391300, 4091059600, 829994243, 473526157, 2969598738,
                1569819164, 3269273437, 1687660622, 769790079, 2939858479, 2314982026, 95679934,
                2650280721, 1284506574, 2095382569, 3629187443, 2367705831, 2672087363, 659550851,
                11251546, 2844951430, 4117036159, 3658960043, 3566145978, 2512143790, 2139258524,
                86934552, 267324838, 2221780235, 710419209, 756024368, 323223865, 3778949258,
                3816289645, 2219776149, 388834355, 1068024113, 1222179129, 580874345, 1198868104,
                1817835984, 4094192509, 3826892475, 52655050, 287659039, 1120108646, 2058392690,
                2104203826, 3037610895, 1029726175, 4061743149, 366112992, 346791754, 3821995524,
                536509597, 2325629474, 2066675899, 136719370, 1695618401, 4127537516, 2452460370,
                341169206, 3738496386, 2767987524, 3190043068, 3016991314, 1985040888, 2078594914,
                2139671587, 3776602980, 2341476692, 2570282486, 930489583, 277160411, 1626294789,
                3691686855, 4262789355, 1212444490, 2072677275, 3457253017, 2170703287, 666683948,
                3333015377, 504864707, 487969583, 1288636011, 1567777164, 1530159893, 3815252539,
                2666881166, 3965829243, 1041099897, 656342032, 1028191164, 3915206067, 3690757289,
                1762777065, 3803846698, 1266228114, 4076868833, 2399530576, 195016210, 1445885502,
                1995001751, 3704589774, 3772778297, 1868741945, 256483377, 467868342, 1640547407,
                225595970, 1128094955, 14460666, 4066078010, 1531108057, 2276851512, 3961921308,
                1169342968, 374165010, 2374552173, 1981313800, 2852786477, 4281055282, 1639781930,
                1322826308, 3962989569, 2251429162, 3816598409, 4202374414, 4064929266, 4168284369,
                213598332, 1878654815, 4012592889, 2331662084, 1902718261, 2647610456, 352521621,
                220018838, 1635198976, 870981338, 1528262232, 4278709709, 2883394367, 3811471744,
                101869306, 1737965113, 5642069, 4097501724, 1259821990, 3670562669, 1974253307,
                3684053909, 2405311762, 2106561791, 3003266622, 3144376081, 3347409339, 3029518983,
                1386943455, 1931470855, 2440160434, 3815143890, 1950218632, 2764905171, 2625123447,
                1837241813, 2329259261, 481118439, 3173225256, 2962727463, 1945440488, 3710596738,
                4277813522, 859909239, 3649480114, 3527231066, 1319170120, 2057720939, 1491180621,
                453126850, 1871773559, 1890800972, 3318027331, 178171345, 898881946, 585192043,
                2316224458, 320531880, 3894139506, 887621194, 1336067791, 3934764232, 2500940598,
                753297159, 2270381915, 617165818, 3189476410, 1156481495, 2662137263, 2316205098,
                2461186746, 1238818742, 2671302457, 3259197258, 2457707232, 2069915454, 45049564,
                4167698335, 1201886883, 582951017, 231483990, 126335446, 3076808505, 912812759,
                2501874082, 540273866, 547753220, 1170424177, 3189829517, 708808880, 3129622972,
                2002579360, 1870645386, 390169448, 2894707900, 3908460579, 2350579381, 1331709602,
                443459340, 3712574777, 1224504425, 206884931, 2378720574, 1020032309, 1560294846,
                3882569460, 1581034183, 2420521192, 3533022079, 1115021897, 629573578, 3657301365,
                277294527, 553080215, 3028363420, 1291561052, 1328469432, 3338312082, 1338200129,
                1766236727, 3416356554, 2978891312, 3082654857, 606657334, 1276316260, 1068554434,
                578254253, 1605030828, 1138302318, 1303706914, 979241233, 2741167388, 4202417433,
                1219394772, 2054300069, 652415613, 4270613519, 2111897808, 96742901, 4165401801,
                3285972425, 510249183, 3584775636, 2783564547, 1562245500, 2730496534, 663355262,
                3500910819, 722790975, 1919819672, 2086954706, 253795458, 2005733999, 2970779153,
                1467732358, 3926882554, 183736355, 2926165427, 3506297134, 1277594353, 2214151876,
                4114889724, 1981267378, 1672821314, 1973092046, 1816211223, 3995013321, 2372975102,
                960109327, 515631621, 1141049977, 2556287867, 1599407251, 1420644713, 2146155618,
                483793231, 3212064205, 1747734295, 3877264223, 2699801323, 1227467313, 3524136023,
                3700565681, 2022634486, 100193147, 976346175, 2047559949, 178497547, 659438932,
                693783143, 1025344114, 3296189331, 4234938364, 3519391869, 741867813, 1210413520,
                1131945841, 3748089817, 611872932, 502118182, 1727700656, 207851418, 2667971549,
                122428996, 1451993464, 2823431360, 1457011744, 3016921746, 3133660679, 199979290,
                2570157746, 3883523853, 2905936686, 4032889764, 1341310325, 3955219349, 3754604265,
                2731331914, 3325586999, 450131610, 3586820947, 759246962, 3907198611, 524872115,
                2765413843, 2883771692, 147272373, 2712175410, 1690171466, 3722496509, 2121634784,
                1639601660, 231013188, 1337728573, 1947365408, 1956500101, 251574525, 858236174,
                1259381993, 36329478, 1435388981, 285326643, 3274056498, 369851, 2209662002,
                3443099831, 2062553205, 1736753981, 3579525503, 3084085035, 406796802, 1551408425,
                4209938819, 3342796945, 653033552, 1255972233, 272582735, 516452148, 2007012629,
                892881275, 2969869353, 3262627654, 822918090, 2317982905, 1913126099, 3381804467,
                3143222829, 798410293, 1697756544, 4038741424, 3291804708, 78220767, 1148496187,
                1458398142, 4048000884, 4101585984, 3148543887, 244432076, 882179301, 720217242,
                2211718549, 531560575, 2338154942, 3742711210, 3961239456, 1640123469, 790017482,
                748840499, 1853629994, 3073906634, 2530890447, 2206461015, 4030211025, 606026839,
                4148432480, 3022974668, 1825681328, 1123927944, 2425759081, 863745138, 3994200333,
                1662692962, 2415180855, 2354782292, 808261716, 4050305196, 216852620, 358006854,
                1426415913, 1552547999, 4131825801,
            ],
            &[1],
        );
    }
    #[cfg(not(feature = "32_bit_limbs"))]
    {
        test(&[0, 0, 0, 1], &[1, 0, 1], &[1]);
        // - xs[0].even()
        // - y_0.even()
        test(
            &[
                7642088996549441618,
                16939115288814344947,
                4158423030205990745,
                8781531266919626472,
                4536426036746990718,
                2759142628546321348,
                2227485206975570894,
                4571191084929603686,
                9433785736036925989,
                1551184301255450043,
                6411840578125486860,
                12800352943910688520,
                17181013235616069321,
                685975652320320677,
                16307888956108230656,
                8336973718541306978,
                8755082279744089483,
                16569752005561764973,
                13002378357019963033,
                6860220932393655015,
                3560357731840892678,
                7070244414620499337,
                8840693397681308044,
                16290086784750574360,
                11733147242613009414,
                6107879515762280778,
                9376837729905597698,
                17040637707038025639,
                5177768201776762198,
                8518342977048446763,
                8898881218619802887,
                10329787183129644470,
                1526673743209372789,
                10123244100180979109,
                8726514720133331124,
                12821051538345717782,
                10882156112794604029,
                14910284544698069484,
                8872234416399843814,
                15937194738982521740,
                8503006142727912361,
                3966017325452025377,
                14700847728548934808,
                14256889243292689986,
                2528249268802686972,
                7286745106660315912,
                1439847309446662533,
                15901896417657589110,
                10477568383848354894,
                4553356972647272600,
                5529939210774917581,
                14963188060984128980,
                5375019716695641294,
                2457481744426644222,
                10485361437428266879,
                5046590988101961772,
                13198446382534172453,
                15426785494942087681,
                16124585910119816388,
                3012068926694215709,
                12864270021749924937,
                11383059477645299313,
                4320116732205951673,
                15221500672825687515,
                2665939181257259992,
                10163551729206894715,
                13893415558679363380,
                5611079099597163943,
                11092151207759951299,
                3992411646432712733,
                7869118229635958729,
                11130808158108336850,
                16931021319887020092,
                10589484424862129300,
                131590704017469183,
                3810734318365068802,
                1788964661632163921,
                5738706636053906772,
                16881608116606909233,
                3243306516763190472,
                12976245142731989716,
                5499515806711091741,
                4710317151847171167,
                6683556421601504929,
                2662551646050148338,
                13562673622930528535,
                9707043649639400158,
                12788296700706707088,
                12642829810299517252,
                5102876644822332232,
                7646892190996074311,
                8078207877741285205,
                14186421431011741661,
                80547895349629461,
                14862011566017513150,
                77115327325659604,
                12406562955874830673,
                17375475859238808323,
                7716894176906818727,
                2802055520431677536,
                2762761117974868489,
                8210654419852490583,
                8128381653654809882,
                14172453553140328425,
                11087897451444868103,
                4396135003820796101,
                14219611943052523516,
                13587332761182504345,
                17711266467744329635,
                14950205666228843352,
                9666645683806989543,
                13020084189388475608,
                7037840100396284962,
                18373412346854186941,
                11498753076197464725,
                469550867963885335,
                15259714004312508653,
                10050610823962070335,
                15262036116174229631,
                7354344829290616378,
                16006636465759715169,
                6784457910192993929,
                12937757685409325171,
                7348103207566204581,
                3244695992856053257,
                3640146490035082661,
                3037244987945809586,
                14532227798128191783,
                12631736369162945929,
                7229210180128729119,
                1338706890404586547,
                4471472703522570809,
                15106263865480071304,
                11243988291566356467,
                13331144267559269179,
                10751519539999242267,
                16921431036638007632,
                6340810572216327297,
                16133593180565445752,
                14463917288006420312,
                15840542126788846182,
                13181419073605341229,
                3946982966135895193,
                10759326015518767201,
                8447608418599844129,
                4353759043775104853,
                6237067423655196872,
                9599085561312857028,
                5365643417931112910,
                8257457675273138042,
                13475654552193991900,
                17137537000886946596,
                9313010620863889699,
                17623954250315027559,
                9114405119196504360,
                5242451382075890968,
                4211561857798749424,
                1420640002606510731,
                9161162479506033523,
                483617176601131098,
                5086158402623829536,
                8760073747436331817,
                5494506533277800577,
                9008744669113942127,
                7906886502348947807,
                460316122106335334,
                11348410017111087645,
                17743092384644894320,
                4543625687376029100,
                17638867588777995538,
                9728026851704705749,
                5787008173309872443,
                3036894650402650039,
                67106435017645771,
                17925489681419424557,
                6389173597207372250,
                7144011805433357127,
                9295381454741344964,
                14053113438378634953,
                12217843843723167499,
                9140869435106275798,
                7517027790294614588,
                3777825091021737118,
                9367707355787166504,
                2718655376583649956,
                17560908054609622633,
                9438653603987629385,
                6976020052495124145,
                14469978948800476053,
                14607968393097793942,
                8465948641441613082,
                8829499402923220026,
                15526274211406668344,
                13263426691391304866,
                13513707111700176688,
                7058023659658993100,
                12343251749665219733,
                8922907808653322519,
                7970367952666929730,
                15924847143147976357,
                3013336538735001280,
                15754712827326735634,
                810163294268081647,
                11862926494972533447,
                17142226521130689167,
                16045131138370367308,
                7810014550740933303,
                11113360618561317801,
                4146999428967559431,
                13627063951311659386,
                14101121524317207001,
                17935556847685451236,
                12893002441589054692,
                4908175619932706667,
                8148134223567687298,
                2797861052729079969,
                14252667124539620702,
                3335552068649661178,
                706647811473458791,
                18284923027330686228,
                1353092748991658850,
                365920444454436210,
                1178722791964549570,
                14053600116413155723,
                4250093264582203735,
                17814207958723924485,
                11616162427315300984,
                7445813626020939704,
                15051244446850726990,
                7657749036148962641,
                17550408898208013781,
                16543784538024296766,
                2175464478760404540,
                17382259944663939998,
                7085519492527115621,
                1948795784424831543,
                10153333277365682256,
                2691142888904508555,
                13042656130996531455,
                3184232577077893193,
                2321647310467391299,
                16266534428918597602,
                10813170547829807049,
                1175296704970889954,
                1336221003418894830,
                6552922275158000133,
                5879504718196828979,
                13617292288202891737,
                9159008209770128895,
                2004574424000467977,
                2728245432858568837,
                17787384979230748037,
                9410299200382955923,
                16609139716214733389,
                8268702623607434688,
                6088812551270982604,
                6106049667290975360,
                7607520078701557430,
                12472853613137360196,
                10887056384527664263,
                5597819467365735087,
                10585294901756958311,
                10513460596082524709,
                4393150075765782543,
                2683577406985854704,
                10995282969723658182,
                11867305236638382656,
                2547454398707374572,
                17029951353545530980,
                6779642703694514142,
                4391116384641001059,
                3196785568538169495,
                14807958476666083273,
                1763112366273599517,
                17875511915567196132,
                18293280485747533701,
                12994911180160214048,
                17806815555469749177,
                2603276529594432585,
                6451629044374772788,
                5131789756208165076,
                6368584533107989340,
                1442599467663039247,
                16334140366206405769,
                953757322567328280,
                7611656048707188221,
                16623430410377913689,
                16263978152439933144,
                1098065150329876092,
                8375896578927982994,
                7235501216156156538,
                10872521053367378484,
                1384698101043873340,
                7546487270974101281,
                16129196974871143949,
                6523889198981083995,
                3955763266387866205,
                223786943533297226,
                14312359200553508909,
                166525052874074138,
                17540833294310979085,
                8682296429517930658,
                11898411716692651263,
                14647731107592123290,
                11418813950965759480,
                16276766779066588942,
                9109613189145192743,
                14365061852139334701,
                2637416103257879045,
                12683017840408065458,
                17251784137128610303,
                9857440911685680274,
                16899637908997360539,
                11676396516728162961,
                12445807799710831976,
                16409119217540007670,
                344530427846566793,
                10122735096389448235,
                2236083202975860367,
                7239130410771478229,
                10419077940654134530,
                3417466820844546773,
                14103102709363050632,
                8515412940073771511,
                16793797293941006006,
                2344964963606541868,
                1770709333553370373,
                11715027939900566010,
                6724317159215710203,
                13382766807638929554,
                6643722639139813184,
                1561112814735278829,
                15883495724862322390,
                16352011717026042684,
                11080163213412729696,
                2601285403880727060,
                17539809965809036336,
                11674296212933365440,
                6987308431764628640,
                8334299117659294638,
                8401136336706455,
                798056208147687026,
                7585403371491023835,
                2768563877851444820,
                18160865562203294879,
                11088989971128700023,
                7958399735371734621,
                905009938091789825,
                3077994674694137685,
                17781743814574321612,
                9696234199814814695,
                14424204686884620898,
                12364803790224779557,
                3680446812855063956,
                10607012459674795622,
                14354213044601441446,
                7201960041306030417,
                2820537475693014763,
                6483412063024289487,
                8894703326452336808,
                6972597094957544888,
                3089113953641583877,
                14077274888152871698,
                11143284357837217212,
                6121624755399734066,
                14754677241106442954,
                14679578243226120309,
                18187552762841862364,
                7408374060988262333,
                17543224593823838775,
                9146972686354667307,
                11578511243258357402,
                7291446516637915248,
                4910731754496429968,
                6446378869968255622,
                5665946125539935002,
                14591605513868934991,
                7360982275667630490,
                12489166270353119720,
                652942929868639081,
                6195275995865002712,
                8128775784532283838,
                3387861524647988828,
                10206768616721991356,
                8310654759157370255,
                12396521032462580315,
                8378642939179207200,
                8913611503978571346,
                8964509316769391998,
                6983721973747822525,
                1620627079682485170,
                3550888596803878384,
                16764327882472918320,
                12418434124596730949,
                6492074094596751971,
                14522169762543900271,
                9643692256647897071,
                10018141547919756836,
                18119186792532188547,
                17605638757524994218,
                15110565499291832585,
                3859111113510276634,
                3608639569011774205,
                16841921385542843558,
                6333586318389785071,
                17712249436612631968,
                13651143305566236494,
                6282590680802993984,
                16962925672131291608,
                17070033603299913704,
                343263679484429433,
                13960595387387109171,
                6824567182512993890,
                16643055854234833912,
                13368136582061640129,
                13407186988250747721,
                12067851383097839439,
                8734440357301460743,
                184374602510479264,
                1980281157180345466,
                18235178046950429926,
                12314150621953387389,
                235240640756161895,
                17191061665928515024,
                1450328693825849226,
                11658139939358930335,
                9661242385016654415,
                6544047620238274887,
                10076084993639475413,
                5484644773132718155,
                8122848295214676448,
                11451997792866817327,
                8014772838878095834,
                6738239678981409331,
                7524197632536086494,
                4887712629149768445,
                12755950013597708083,
                9072347758799333428,
                9892994336906109770,
                6658664089339267041,
                17010445991373202740,
                4749493607011870912,
                15469799747835715648,
                1410642028794185315,
                9678426427615195040,
                14603513578231268629,
                10144994305300497927,
                13627001390827901004,
                3410387289467486309,
                4969353610007528056,
                6899086728498695416,
                15840849162291468831,
                794809021087797410,
                4128395609120946267,
                14603570710676978752,
                18014301493566032939,
                14554900403738652088,
                15417180431811963767,
                8123851550316828449,
                701668417489927407,
                7765872186237343224,
                3766544625359426110,
                11788559991379689715,
                4186838184370360996,
                184810421200681700,
                9708014331923050854,
                2038789855438533791,
                12063495557069443857,
                13840086852848556400,
                13600250597329615331,
                2088286198469966080,
                5800997251215440284,
                16245994826244567566,
                12426146991432044569,
                14174963417820155909,
                15950886475654709571,
                9625392305352933399,
                4560214803933408223,
                1220673912062550801,
                5315118346179673010,
                15305147243304018100,
                8860293966649254041,
                9878806918352789226,
                9948881370845519892,
                8051249634924644385,
                9524148673689679734,
                12489881836910498385,
                9483374649969419117,
                2429047588791105172,
                15948533481260088039,
                6658948169399436464,
                7055385462316586853,
                6444492735173712353,
                12554099647303616570,
                9411408437418751476,
                3119186149878145385,
                11561271469129136960,
                4699043282095308564,
                13664860088579707042,
                11942040945216924956,
                14938303604593577988,
                6584398232469323882,
                12615016316841274906,
                17607035342042230816,
                16299371719180977481,
                9410230100515705546,
                468652639351146679,
                3099794389553133390,
                7426463649192631643,
                17220440064639262443,
                249838037562419695,
                12394557669144716810,
                11891068226864790195,
                6815470691731709960,
                17050323216000491320,
                17149156625587840029,
                2840995912779779709,
                16453929939061252876,
                10865711713754680427,
                17257962606206172361,
                17324377803464599290,
                9988049752693630171,
                5145541150836693762,
                18187548404778944247,
                9897108837353120002,
                17915722858353969777,
                11223487437202128974,
                3382309055960184808,
                4470363898167775120,
                15118823731598210162,
                12451723255995030474,
                12124942937255563460,
                7883323618797267364,
                11734899536152960444,
                10899101530358100831,
                14568463402558702580,
                3589575239343156021,
                9506082963977245397,
                6269914352444216904,
                9641949985442899274,
                9484754331399933603,
                7969595927070810042,
                13794981515289100280,
                9170567771554076837,
                2138421182270581614,
                788654717547257666,
                7557048352969552118,
                10073066668613087627,
                18136047902767483818,
                4883457934069327911,
                6904157286973811264,
                480950235088373409,
                2214051188032927994,
                8438459334885401002,
                3359421640238953931,
                18193289130141312662,
                211796784365007412,
                16550107280222789371,
                9368274586857413302,
                4080579477676412391,
                3903589165729627543,
                14601294347577892827,
                17880615106796162585,
                8399371980131050682,
                17125581143302260392,
                5236347897105180310,
                373897691749512425,
                13424896419381837167,
                7230320496289832265,
                9814225131983506715,
                3257434254689569664,
                15321426119731656450,
                248987497038801136,
                9132125928890530860,
                18068784670509956259,
                3834280479551637249,
                12658772951095410532,
                17093119016378053719,
                14830343788339267922,
                13054493939738291239,
                7771905415521456170,
                1347652221288704059,
                2017715827457192133,
                12475851911823933261,
                5352012646013352728,
                1878283153860510081,
                9615556940143309330,
                15598414798481854843,
                9895563858845368344,
                4068766894729575171,
                13899464982406282747,
                5859039026557422063,
                17961018624926257461,
                11439432659782104572,
                10132383873951727022,
                5668113599918475821,
                5618863243783853619,
                5296033916369028410,
                8363560482197001010,
                17475747004498994800,
                8474583485147508303,
                15911459900369709149,
                12741330330991532249,
                11847973179510376732,
                16000211930346393780,
                4738554376721879132,
                11907835399609080296,
                12646339508581669507,
                12828924555217992033,
                2386658548311295777,
                4342638340328301062,
                2558348042546117915,
                1515754464721209816,
                785688177042406742,
                3053211873647924192,
                14014331870834149810,
                264938968394967629,
                5983753879589540684,
                267963444144320487,
                1239629907294807005,
                12062955612488831436,
                17308841949048522703,
                330482381584177401,
                9514664837122696614,
                6020475790972444309,
                10895879821954360044,
                17337803144194880578,
                924273726463533426,
                16535207821268418094,
                8946410913169111340,
                16060878889863712458,
                15976666724330243304,
                15895420185200619297,
                10599438926533659781,
                16395937774576899123,
                1515688300207881192,
                17269912862206470571,
                4692813434396761996,
                9815072710015129472,
                8355648274808220440,
                13388537609750517047,
                17672336807467300633,
                6893196003223183422,
                1850026996207184577,
                9739088825020961086,
                10065083987816724431,
                4941592917350004042,
                7273197156716880397,
                11764185831933564101,
                3813114017041525493,
                3604220821878800267,
                8830614448851343613,
                15764272711157879112,
                15343520873072014909,
                17833333986739184436,
                11899253527412743432,
                6319626654591461445,
                8345841242755999512,
                13969872924667267068,
                14689808446598116235,
                5690698738679570394,
                16232574524853907406,
                16476305072233006134,
                3837644082952742591,
                18371118811121083512,
                6370089221943156975,
                3758433368774128127,
                12598095657509455351,
                15041255601684463158,
                219849898456894054,
                16347148290535489696,
                2952439398365618718,
                6463887126673004054,
                18006657361418901576,
                15554865575024552259,
                244493720680004046,
                246212287818939397,
                8391531415221668572,
                17158996591507713973,
                1574344884166927127,
                2139609011514148330,
                15894388729950489361,
                14150049620712128466,
                13649351261368190439,
                12818010881632991653,
                15846311295084438832,
                18136287659718690601,
                17778892669314265107,
                17919810337867516752,
                14238683656303095043,
                9767662762617787421,
                8792587365945756712,
                16289324204380888977,
                12145397565479495717,
                13131505318922744869,
                3537402907198034169,
                15773157789160883228,
                13080220658691776576,
                14801888581508580861,
                4455998795552535017,
                11468446901913315682,
                8764904591192407340,
                16992093112328572788,
                14539148855227682162,
                9062932246611746862,
                12092139254667251241,
                16973622956249392518,
                2330107601262713612,
                14104249685295227139,
                7802927139367928835,
                17015267038524278700,
                8076057374240890515,
                15166827418715942805,
                12316102967608917038,
                5048442745443715233,
                14352260581070172244,
                18429764753411235196,
                290907609897930421,
                12832680942495754212,
                17691084325410578048,
                996179388827615558,
                15796651271292023703,
                18430169649438059546,
                12951979241393380884,
                694335279970980851,
                6638967085265053633,
                17595649263141354482,
                4234846945003198207,
                8367354156647080658,
                17456583496028677793,
                17835204410247184698,
                13301357937590556083,
                6786278664181009598,
                3297223935318868177,
                16554057669712072231,
                12522453888866796248,
                1975238565343362470,
                10528848975094257154,
                10553418118214937237,
                13409190148243583780,
                5006922367367242363,
                13556726007351092492,
                7783632275488805468,
                9183676289046305416,
                15469154331779684343,
                13423499673227845431,
                7837948571314497011,
                2029560205070039291,
                16649457177034615332,
                13491227518645198090,
                9993378618273883553,
                8487159310535624111,
                2809530522983207601,
                13575910212973157992,
                2171679949756077498,
                1660890720446202624,
                15646142722510171608,
                16932531575310315514,
                13775138986080338436,
                10735540465035236537,
                7604129283442046332,
                12444305356219135513,
                18201551442306471158,
                11120131523005131855,
                2798231246691818759,
                15317641731597026057,
                8499616562041083838,
                6017032731278071224,
                1273341979567250231,
                6519997694567640478,
                2435912850700538815,
                169204604573595693,
                18269270505596175452,
                17459024018388721732,
                8266385672999361049,
                10680903187132175198,
                17375563191275197987,
                337068133283788118,
                16239907884128954657,
                17641011679183022712,
                2366475531220210250,
                10755304562557576337,
                1869838694309764714,
                8258555369397490418,
                962287818495402559,
                4995654753554910199,
                11059451160990599794,
                10810204279841400750,
                1456068596599679097,
                7305003907755693862,
                14988290547736996217,
                6795308809242518240,
                8763020957272447790,
                3308001334228530973,
                16974667325340586198,
                991537759276291502,
                18008237326885742575,
                7753905923063895580,
                17512543097224696770,
                3595225617413761186,
                15058630498302531631,
                6622110332814961401,
                12488257875553353319,
                9044137245617220882,
                10271290501572288378,
                5585777756799704531,
                8295396987659231737,
                9215167333658814611,
                11104586326674499796,
                3810019579782011422,
                6888760568785282460,
                14532837177301113100,
                977805367496486072,
                14965851872869659150,
                11134550780907874822,
                8067904641271727206,
                16240490643234968341,
                10640644051933721969,
                12823704007775718263,
                13210674225658496740,
                17421304180276102080,
                8863993590195462492,
                5419546058165868071,
                14184558657599665798,
                12923876509266120971,
                6685006207471871309,
                7704595961550622452,
                17285923828421938855,
                3505207992795309843,
                4532252029642664133,
                726816466976184004,
                10742313034328351171,
                15565587799659267510,
                6490375145545887919,
                17152573783466503447,
                15997997319822869691,
                10153243598033053836,
                9623317771768820586,
                15946877188272739767,
                6079621748092093763,
                4005237073110529685,
                10324349804736126549,
                2979775683162494848,
                12357491130084004088,
                6870626476679302958,
                15875069688841775050,
                7544803566766537053,
                6558561331652540514,
                11395728505159480348,
                15150381432396177515,
                1619752167122893719,
                9372894574032628757,
                4772175913517010825,
                16565939614554921583,
                15177791327910370589,
                4034037178193525662,
                14253679894187996439,
                4373435865471950380,
                9110620573126725481,
                15888647814967051075,
                15867477839426284043,
                18263083430840355206,
                3572721603558989476,
                18033901510983288634,
                11740707689919057494,
                1919396080344872985,
                4785878236873911753,
                9518697307553186252,
                2196912231764078547,
                16859172788282672634,
                1294478177294091838,
                3853855925306587869,
                5712176474426944553,
                10655739333433675017,
                16864687770354405683,
                479733567774176546,
                4002525199861551470,
                17884530730667206604,
                13118987544304899067,
                16693510145083648654,
                4348762192364479250,
                15651674952754110111,
                12941615268132778668,
                3413954408450272001,
                6437132444235811404,
                5911780604883628493,
                7027061191226541890,
                4541332504662688699,
                15278256850661601886,
                8783983716478722525,
                6762972941032427523,
                109436002367794738,
                4964075549363170179,
                13998615704165248071,
                15840448835251983621,
                17124026628251673729,
                451007725533524973,
                8689993352441934927,
                8085937255579820677,
                15895079089926267008,
                9033950962733316797,
                16460182791232706419,
                11706751917248926231,
                1518070523274778714,
                12841484353662177921,
                1106444835771962416,
                3781633741335232752,
                7295048037310303538,
                3078350382473633321,
                5513880551571688035,
                14511804401958832310,
                14686668965111665491,
                1334848572346842714,
                13785608726613289908,
                5484295527088560108,
                4677427364593276486,
                10739032089578444339,
                9118829737688197133,
                2599009188037666293,
                6541018502379686045,
                15124406767918971296,
                13898029149245587181,
                7049797326321056181,
                7676359806921760851,
                9359455871357711051,
                13721511461701704493,
                4337948462861507669,
                12182314864880410336,
                18115088039222630086,
                13611666747148849963,
                16979677843375636930,
                13976122741306049418,
                15131996055772953287,
                5019533830870398293,
                17994426892860907985,
                12238472482719268447,
                8485137745688214163,
                10046713214813218322,
                9981765980097884884,
                5203049149404803705,
                5509439206475409607,
                16646013403333676735,
                5721376623514557165,
                17895610288116512634,
                4146710226708894505,
                4180329068470310700,
                15333932579841565344,
                14578826506899561228,
                11125461119487324145,
                7828598169855633640,
                13016918249925390910,
                11596786187817182424,
                3070363472909089566,
                13372064806216785587,
                16483915751021484392,
                3209300474080959275,
                9744396152477868876,
                5286486708506722214,
                13334231276589469146,
                1703296437445864549,
                6990559773807735786,
                4323851084064743622,
                8258191458458530560,
                5487559074143580665,
                14530068425901775586,
                13125628672642465590,
                7919522946763693741,
                11383708583361967263,
                8699024886208260276,
                5911243921359438557,
                2976618942061378858,
                101882160229833275,
                14947300507963600168,
                2624952948637079445,
                13110311834336189786,
                6328067748328586264,
                8026502157595153247,
                13077832359361647005,
                18034926555389756245,
                10932060114415829114,
                8921112613501153906,
                5010993000512607919,
                15729519684045200720,
                4230394140032073806,
                18249470456439233157,
                16698231796621617025,
                4922479776459643659,
                7326057194769023839,
                16485955343145773710,
                9357885960145954762,
                4908643074768640635,
                9426875174076058663,
                6269667367780881933,
                16372434513892687866,
                8166317811376426944,
                14210102553507463054,
            ],
            &[
                1979867317777782919,
                458803043038589105,
                14807455935956247393,
                102755393029177892,
                13661879760043023426,
                12743502586850908641,
                6485502394763823003,
                7006537482591563738,
                13464897952724427622,
                110941783775954905,
                664806907968747258,
                13976172522383104680,
                7081409417990232579,
                11782066704046559999,
                15078720815504238982,
                2791066361698645774,
                1957006820108785098,
                7661621907711528521,
                7129856627939149071,
                12464578713097949843,
                12568595944828213000,
                5103092420263283198,
                15856125363599807684,
                12992729636796940656,
                10878188187148816380,
                2850172728651555239,
                5248002320936640091,
                6759153103556744157,
                14339593489124645610,
                18317489909052141708,
                15949249147341703950,
                9001038595758313121,
                11504712957931727005,
                8187691649773049259,
                12626028481208875277,
                12844051055584824684,
                9111532080412358373,
                11481281377490778456,
                9905077354469164626,
                10848129645301488797,
                12407190902965064104,
                3004380005013311524,
                7260177637980150880,
                6465761219435617351,
                11687902913416861598,
                11582630797299992145,
                5052207193612544256,
                10619895585106656895,
                3782849494513349232,
                1696521729205924429,
                205289556233949352,
                2309142630958879699,
                18013220874964165818,
                8780365515667787572,
                10777210546602678450,
                16689458940952011084,
                8786023929671818084,
                2959945332887060564,
                12595094201729105975,
                10555858515424418250,
                17838802570355306753,
                9768848979007259380,
                6400634551878490492,
                9350070212380397383,
                10825325059459528805,
                16240713222773986945,
                6773516716020300636,
                18379586867066835204,
                13392009122775849047,
                17446114650305191430,
                3443249317411513199,
                16590433388663762614,
                3592506842077891585,
                2547605864889189098,
                8921931064104759283,
                3745288943697428348,
                6683293017020997233,
                7599775167852553507,
                11479579422714965337,
                12242916146470334290,
                15886988537674743805,
                8027185433862293631,
                5037238148478185925,
                10545897273866342364,
                3749333810592645893,
                7567899208857364441,
                17332217733795476186,
                14570089467947298319,
                10687712370800535186,
                8398127082315689224,
                18138318481654121301,
                2550529540627040228,
                174434346470095983,
                6335843839776953618,
                2142199594346057908,
                7492814286032687084,
                4581612999630224238,
                17362182859125741557,
                15332694635712824041,
                305820774489411582,
                11087741421794954545,
                8166298207140013628,
                13663827369108026131,
                11609249890319933551,
                4708689618714214726,
                31696613282352588,
                6229354554698145266,
                4430381785479249873,
                9093084456389745185,
                8620628652860035454,
                11032119279655454850,
                7436388409999188798,
                4508581046303828766,
                9266330334354028474,
                4654094790776695762,
                10477920368471986136,
                3040546382665779497,
                13578021419513941070,
                10874037816916517318,
                209250739982139053,
                5513314962388268387,
                2276087999654805974,
                14116993918737387242,
                10215096397828559307,
                4706886349596564784,
                9496675948395405688,
                2592786526073230109,
                12032023929905220888,
                11439071011556767245,
                8896841450411382115,
                812528333048514747,
                4857507773721570306,
                2346347769853663620,
                893640476967692732,
                6002646205477400496,
                1258104196359799910,
                3298555154777830453,
                9831348292240867254,
                2983777286176645688,
                14012697848669764867,
                3281297577197824417,
                4818243357287928570,
                7599542312723359299,
                16303740951463015372,
                9530417675969610545,
                14307310939799846427,
                9411915044625670774,
                3098269527533968285,
                16636400423591714570,
                13575763948252176199,
                5311264796988234687,
                12230662710526328293,
                15591605225575960535,
                16170866454635856185,
                7154393582831771624,
                9746836435275228439,
                13154372386985050769,
                13955058214451317887,
                468470833133904286,
                719628130899837785,
                14167898993266425767,
                10492242376823879424,
                4695877358022833681,
                14415659841083920049,
                2128453387083764523,
                17254215071478848692,
                3464552401256088318,
                5473802745477872406,
                12278908178174546508,
                10917423561691292478,
                8269556840984708513,
                15451525009059595276,
                14506788958072783738,
                10120241313842466204,
                897911062165781450,
                10413013565287912000,
                5183133159721680857,
                15306474298436567422,
                2072549059193978699,
                4803937018033002282,
                4902316829203079235,
                17182047496719608796,
                16148451990121486069,
                14460165826013687805,
                10861828715214694182,
                12852802503672044251,
                17897814519242488317,
                12830584456010343195,
                5025833892983477930,
                15171169537313884434,
                904875646680899456,
                3891209230657191180,
                4459567895597613958,
                10213064404613271715,
                14464647608623247371,
                16076880745118924462,
                5651027014515376452,
                6845292042507662562,
                4700263749108232739,
                11912023508343584677,
                15164089726347867289,
                7435959069204688586,
                3673845585791184955,
                9008233584340774048,
                623672340235969578,
                15952660561845258442,
                15255827823417955924,
                403762546093465841,
                9023129674293106279,
                15962739700493341062,
                6108426038775323673,
                11423092137795750189,
                1431075448616656956,
                17640578768263514852,
                9182353610308928359,
                17753229409310170357,
                13898814437210093610,
                12126245547088496216,
                10888527319957075641,
                16039715754410307348,
                628180706859646675,
                11343014058184488762,
                13075222209229766794,
                9658840972156173472,
                14805748267851325307,
                5640565628362124114,
                8813449713167462710,
                14750431340071075257,
                7341247484352499222,
                4105134027480979119,
                5063597277155075543,
                10245719632899271147,
                17417102409056111607,
                12327200750635056279,
                11257956633392412563,
                15654204403173941169,
                16887277320373649913,
                17293310148654704869,
                2931320393510833429,
                18265112369378689230,
                16177931273491220730,
                7148441830138947268,
                2542111810786801658,
                13075348024340421114,
                15045402682302144848,
                17729484015706193097,
                14849701582062849098,
                4494999822131844988,
                13853067439918054754,
                11104593802418852948,
                6272834726309696758,
                8176562272197586716,
                11136460055946228459,
                2249625026165448849,
                15192391646155487262,
                16462654425479085870,
                12438135124930472309,
                4231754137003607533,
                7025530078224271576,
                1902066619924498430,
                16798677464948394620,
                16103591627987069524,
                6073783480651248517,
                2301126056878392650,
                525873165339288975,
                3654812541302834250,
                4237734633374880968,
                7013648087246440503,
                2708020996911432507,
                410030825299234570,
                11409023582536021944,
                807125820884538678,
                17085267726135611778,
                9547871053479185317,
                11213262078356721077,
                9638233998727018009,
                6181404851731546121,
                13509648795699246119,
                11626561711623336962,
                2367028127797295412,
                14408238933122914429,
                9501259205390120800,
                5388820411015324736,
                17908911641461106076,
                423710192565734379,
                9911666373654831039,
                7296600371969340330,
                16104674231020469855,
                17273935562508193984,
                13386092434346383645,
                10904128763248841619,
                15660801723929131082,
                1802676326982826038,
                16314898646999153588,
                11051772565381291670,
                9582949690895199194,
                16665669325627731401,
                1675659856826747750,
                13282418735625888151,
                7647864912575263910,
                414802336339416341,
                7456128243593571890,
                7401454041271044889,
                7259681585053156654,
                10297473634673620153,
                6291331063601368350,
                6858202795057797026,
                2331669818605326975,
                12264404420708345107,
                12089542543065457441,
                3633020299658155421,
                18428329446154081698,
                16353160654361289944,
                5081398862696143362,
                6121111800778820791,
                14309612314653801386,
                13306300382927317704,
                3064838612410500446,
                14989006580151485936,
                12725645885424440632,
                5569344928439882604,
                11093264756688684425,
                2529531794792252742,
                7787364529152520016,
                12100725195740947792,
                11015410010827498630,
                12770939750081523925,
                16423905424192297733,
                11305841527114312073,
                16891896558850432433,
                3225706379047965263,
                14166563803489592930,
                2355248555503091815,
                5476494996858779102,
                15360364114708065353,
                2946255509562408083,
                6223737602577045649,
                785528028405825827,
                7297867173505941067,
                4724393265215111794,
                14532649034725999985,
                4217904748666774400,
                11309973650332868442,
                4820265912884088727,
                14053660458609997460,
                3483460907245444911,
                4888448412579236860,
                8902978947161354892,
                1161291241547718844,
                4827742954141797453,
                17959154045662307166,
                12471977193177859473,
                8286419873042983353,
                7559677115080818717,
                6009725396648616438,
                7950724447316580461,
                12082875173598285235,
                3123141200374415061,
                7736739371326140224,
                9033893194019001400,
                17119540472038591893,
                6838993250758979381,
                1756627985014158028,
                11540944119317576836,
                7167619835597129390,
                6344439141314015967,
                2159381790958961231,
                8723275234827128314,
                14671976335382244081,
                7492612374298612857,
                16983115979216019630,
                1137928104103949457,
                17705437622823562319,
                13213748898063588558,
                11565618477779377046,
                4479222060769626300,
                12657805467897134295,
                8578789730380785090,
                3568050214652067765,
                17264603982687227099,
                7554571839983616835,
                1648213353577773226,
                12805556496169022728,
                12400854649388722924,
                12743032680153521053,
                18027191329575892747,
                9029998509699231947,
            ],
            &[3],
        );
        test(
            &[
                7798496952823407824,
                4811470598592046480,
                16634149002718209756,
                2816485763131163737,
                6764931459572097898,
                12135396153038963217,
                4726251464409773649,
                8573439160265568957,
                14269672704267810519,
                7649448254104194910,
                12613439096551355375,
                6638659391603712256,
                9240011195460328922,
                12704753717311876709,
                7176884384156072417,
                13235015177754973669,
                14756487689013468042,
                9212292110542328191,
                2882513161370093736,
                64849663354449995,
                4532026566849740479,
                407205524552954739,
                16398190979531467263,
                5875928748346174530,
                10796369575411951617,
                14293424794963768678,
                9366522120560533977,
                4213967089717524150,
                13763075659281951487,
                13464968737596420936,
                6549141323160628420,
                110915844406990687,
                18075287808720180110,
                16664063901861004820,
                9897000415442332787,
                1486075656392074792,
                7382904500810297215,
                3592293622273342942,
                10008125746032292648,
                18321114734333840332,
                17920212203847604612,
                11217657766511696793,
                11281083380440747080,
                7892713234117558634,
                8902471323128328235,
                11892365124252837749,
                12611508814807140936,
                11815551321438701427,
                18110749687253320811,
                11715728409437156086,
                3339319126985623378,
                6325617411041006209,
                14096883611650388622,
                17687808067973538964,
                168019485471086672,
                15693590891786235960,
                2786725115760255375,
                9052512570921605652,
                14177547685319586247,
                15370386165587184937,
                5970865523433924174,
                8584955417200346406,
                17233739694187471127,
                3635034994393071086,
                12922974184613453425,
                283176399532113592,
                9116715092600674482,
                533178685039585111,
                5221528600415982001,
                16669037965261793667,
                9659393188345721976,
                1619537891333142145,
                13016156940489086061,
                5348093554893488672,
                4125157131954041036,
                17147602453512343883,
                12283659650945459472,
                6073750737922483302,
                4112966256314115607,
                12848281534595730436,
                11698310578341605232,
                12515024339763963627,
                16720191980955784273,
                16331343356415214175,
                55095711946944202,
                470353114021282512,
                18440424164986645711,
                7119069413231791773,
                7496499681661981147,
                8033047773523053512,
                964312760862509573,
                3077830922660773095,
                155832606423296496,
                13447886967389293275,
                8559367474278812445,
                17998261191861118766,
                9631655612257511488,
                4223841702086601452,
                15213562135345686189,
                13421535853495283983,
                14214420071333045458,
                4946777332360956414,
                4831925713285482008,
                11760595051618114799,
                1418683958686722085,
                10071047199952333941,
                9860212880469572414,
                5658526438682259927,
                3208658863323316983,
                13572249994839707422,
                10858577727713550259,
                3187235617013222032,
                5876918919503422475,
                5078114263735950174,
                1124267584430625852,
                12765353279431679058,
                11774222431813768840,
                17690097956396545959,
                11327308747698708585,
                3866755026407601622,
                4133257169702318528,
                10187715330446905330,
                2618131060603979377,
                14909622824179687974,
                3131981152941396860,
                15758444864200993797,
                15877508051463207022,
                17810056601280430664,
                13308597888743200162,
                14765083240575543194,
                8389207789402457515,
                16638724707073855749,
                17831200813330442342,
                885536911921616113,
                15627100060180098369,
                13529643776104497044,
                13063738574432590840,
                3409549155326214534,
                13384107634300554710,
                14621110597375575515,
                10454034361196971254,
                8866597122119704829,
                7214280631032588417,
                2321851294623259950,
                2664600902577903557,
                8206930671630553011,
                11637903730276320737,
                6120862895588039690,
                10119555767934669183,
                15072606566681932076,
                9196021740659767391,
                17668381905754714807,
                14113831610785889688,
                2274519386441753024,
                16941927109776392899,
                8855427208083419550,
                8707706292745593239,
                6509438127169345769,
                7049981796161059741,
                14351405462508747882,
                8739674628274889331,
                17773549691926711863,
                16108213739701189254,
                46349353234791966,
                16335307352245544894,
                3372116134019651210,
                2026650829541426733,
                14482558945555858306,
                12178227313839428552,
                379954005080389912,
                11703446819120251215,
                13961335387977718269,
                13253644918378197554,
                8389008141245407942,
                3757779100833278863,
                16977903525629057562,
                8706100545828799795,
                4980644394686364078,
                16602809003881382792,
                4704649122813882219,
                397591639474341233,
                5691615138943748269,
                11603908706905475525,
                7830467758888891027,
                752954507775914877,
                13742220678588369779,
                5512709449415473472,
                6602314973830649238,
                17294329281209674921,
                16353762684852197024,
                5198076929899652796,
                2451984573929623323,
                971255382782649783,
                1825437488761527279,
                13282135031519430271,
                15671289983850977417,
                3215788959201267209,
                11367230670788525763,
                15158322487047971358,
                16383227556481842658,
                9272710553939316379,
                6940559952311103297,
                8058002679263304409,
                9200541971370090762,
                13026270051739823047,
                14703602147590614957,
                12270036121815530821,
                10856837658238904154,
                13457513436452657022,
                7752098762039414070,
                5159679976569317358,
                7346882116989161530,
                17098000856751349286,
                7528478613810311492,
                14595271732066598782,
                14345174012514521088,
                8917561215963658071,
                12444774771012436781,
                18003493577034559128,
                7401267539893108958,
                10703654441621295266,
                3928975917156990612,
                12068658501017100188,
                14247190926978777640,
                11887379833560086066,
                16388813962188846581,
                381956448283854400,
                14770881628537066339,
                12486489047707489763,
                6358578569341757564,
                14298750548968079945,
                2056466734423394877,
                6368203595950482405,
                3223554386699468954,
                4227760395094299567,
                8534762536962691709,
                12548903028463843885,
                4967345662728519086,
                16943637153763582029,
                14715091124220815237,
                12141001306263104014,
                12175816912849205543,
                1803108872161511343,
                13641105341345940724,
                4467984689873651089,
                1855564851398276223,
                9806860208915759902,
                15434907399817341842,
                6414202967398339339,
                1952032742127311977,
                11903471585924758516,
                1684708809058774974,
                9584411782518918504,
                10773185872250646792,
                8181377602390776035,
                8122154254362200238,
                12902500799759276861,
                6582720162263342292,
                16706580731489248920,
                2000050292502342191,
                13921749765600186584,
                15656774592958567364,
                13515604429385540270,
                12553127158817424021,
                7759293979203731105,
                11524585147783123654,
                2416828035660522180,
                3815782196161663623,
                12068235656126232794,
                9641578300488516781,
                9227047490937534043,
                12288329318600678892,
                14357764900702876515,
                6643728818780923790,
                9169952627439601524,
                16190860163944860685,
                7224449977289312853,
                2426010657612704988,
                7097348057731348819,
                13428085588675193837,
                2243566205788137318,
                3521617463718109980,
                10311601150383225693,
                18028456914623038645,
                10958023041184848012,
                4863849643405996604,
                13982944808354970647,
                5078862720378429165,
                6995084509678946294,
                7623261163534034449,
                7251391284527626801,
                6642144335648839474,
                889957175812482926,
                14115678510823527946,
                10943283669987163106,
                18022097549698166076,
                1064779896422972216,
                260976419678710154,
                11706337798784823402,
                7339908486105356132,
                8990824151888743890,
                5645262432865657530,
                1657496751968765591,
                3530038441891535071,
                11181694072782782766,
                10650389332231278211,
                15390473409517527604,
                13153289830091656667,
                12038556920917300027,
                18066231150944097445,
                1618038226618345054,
                3389192047673882355,
                11082945276561617384,
                6705349877566422413,
                9781054992335109493,
                12253545225617312403,
                16999500068120091746,
                8769791004306726764,
                4768271990176342717,
                16794616123670513863,
                3996402883485570523,
                9211678469009515216,
                14027728220422542365,
                10496749002692321926,
                11703646058430893877,
                12334209370443949878,
                11449604462076360388,
                4396255705900668331,
                14533140968852883773,
                3491798162892333300,
                3038311448490854168,
                13013057516520639140,
                5160377804146514933,
                14399436147471941106,
                11193864615898836432,
                8531481285844220199,
                3654104119756094460,
                14734961005960740209,
                3654191345397549727,
                8388858864047403816,
                17927724813621687727,
                13429016298799125048,
                5367750080313919027,
                6393224854041665242,
                4821520064553474661,
                8442502022517558844,
                7752658698916624934,
                9550072160911665752,
                4540818189505722827,
                5059997612305268209,
                7060253925263482470,
                5083054112761819250,
                5803398198426211912,
                1380106897985489402,
                18071437738740065909,
                1345071843572391414,
                15710434610483386531,
                11071377898640965589,
                17641605384866702714,
                14870360628358138696,
                9827129489073260439,
                8450415892983753145,
                15666761634961775100,
                7270545221532941060,
                12053279433290804257,
                1622313886640544491,
                5340667040115210848,
                926087624946022227,
                7935611017520266364,
                14642870615373560695,
                3125283099898630391,
                2050649978125224472,
                205120195037820803,
                747063854179462342,
                3789699371533579595,
                1035701621096547349,
                16018212365729931952,
                3789866428655158698,
                6388723914616296058,
                16510613621210149584,
                9894647709359168130,
                3013404062090041115,
                189489053932127657,
                10633648435212574812,
                10348107062615836735,
                12002471873905392969,
                4237459592751843708,
                11637939215779621043,
                14922365631690025380,
                11760271887075563159,
                678386638816913540,
                8019936738073295856,
                3836915660118827695,
                8019505309874069847,
                7265558081469259310,
                884417125200105927,
                1753033849914718324,
                12690803762933082738,
                2546869581652302833,
                16955148199618032179,
                11708042465849324633,
                205152477319765645,
                16629796025665189045,
                4790774024455362541,
                17402843893609042859,
                14744403841742504335,
                9084693659612921572,
                3083238898476439769,
                5569669417086302690,
                8634335868370264618,
                17597532536692475916,
                6829124881401627081,
                16057844520727040534,
                4393594275785544998,
                16870174283907099328,
                16864947511372305749,
                7933670409689289709,
                8062243686940958546,
                11511723944420066352,
                5390996783507099517,
                13514845063338313048,
                9120727178683820530,
                7135404489830930274,
                4698074799063558795,
                1336268150592162916,
                3371186837701723352,
                12497753092049419593,
                10662480898239412552,
                17887743751120115746,
                8037614308177933475,
                15269519858742411375,
                10429730734252577378,
                1782248407797168163,
                13511127986090773154,
                9879755090209915206,
                1548237506818739800,
                15336522777265989638,
                16083355574386989716,
                6892661436203042392,
                17708965875006543343,
                12487267807383327705,
                12371079313118864644,
                14239391039731513177,
                9712715537776006647,
                1339505136772159105,
                3012551510428362794,
                12357355653088287352,
                8187824344238681106,
                3206470745685510953,
                16525705376900723725,
                17797473822898581012,
                14891087890601797179,
                3429920644657740961,
                372799842385126583,
                10159920898351373304,
                17084534135671018643,
                13674012061365744215,
                9062521393516717101,
                16249511416435877582,
                770466755139844308,
                9792258604515956754,
                9847259338113774618,
                6524319705326716888,
                1309518254525408764,
                10346103312499662230,
                7796705521616045141,
                5244441786886038342,
                8458019209005484172,
                7382931637777326782,
                11162469257770583764,
                10053536657529816615,
                14155165939639076903,
                3216040867013041656,
                9855256377105714377,
                13333494298177667840,
                13870977529578497762,
                17702198111172052493,
                7063132479007076033,
                15117903718709274386,
                15643124275645359631,
                13246097376953615105,
                13943153081968370799,
                4129477198714818132,
                11857346290084810717,
                6481315132425963863,
                5719683682695494401,
                5296814144172527318,
                760666829981906503,
                5590808134778217760,
                756772015698441472,
                14491844706456542800,
                18082567968363957214,
                9270285920051865537,
                14699652027310128713,
                18248498453132658193,
                11684467957653234109,
                15881769941581835167,
                314546154914222688,
                8114343064128916817,
                15594029378006079537,
                7306033488226269464,
                13428690195826711521,
                9052721018056013484,
                5948173439127948390,
                15052909961655625285,
                7325957567666136697,
                7255216113309866460,
                2339005174917919375,
                3541464599868150096,
                10620807105734506848,
                10401555768719983049,
                3223209184831011265,
                13219238608657336815,
                833402815708396065,
                5409227984281460817,
                9625361525616921089,
                1310345922942938948,
                3164590508354402136,
                13571591791085099481,
                6449583851417093650,
                12132073172938161206,
                8472582943832045347,
                10769099083212939156,
                31772634679058711,
                17768858162967447980,
                3724817912633131560,
                16347858930007720615,
                4581178512905307749,
                14675184379874252691,
                7800880386417268233,
                10500558640555793613,
                10907473822263746966,
                10349360455162450796,
                16243419016207551444,
                7948477362985436355,
                5648370989495638971,
                12554130341333656507,
                5387445954608484822,
                2435984528448136931,
                15926990095133510448,
                6240279474886617484,
                13162027311361639933,
                9845498081224385677,
                15136770857402679488,
                294172832127927540,
                11812753576130759285,
                4714994681116169573,
                16515438329473328309,
                12715644809880958907,
                2687151289233831920,
                705088298937614828,
                286079974305964184,
                9155582999882597057,
                8821011997760107375,
                4417717245202189035,
                12959997704010734603,
                8879211443564867802,
                13288247630269789338,
                17556983875558228264,
                6934174419250934018,
                5626105851946025353,
                4683143168831417573,
                13583545217544294035,
                14621427723839799891,
                10439845563205327853,
                7115341531213749311,
                5705625326650783712,
                6329845201407641639,
                6588043698151966955,
                16484557210962277459,
                15151791142829317538,
                15329622087488955522,
                9042731875468481521,
                11804081118316930516,
                16855488448748462109,
                14875244187197991269,
                10172305946442195983,
                8757636627949073627,
                5243026113735495788,
                8936037042873886159,
                4876890161176869416,
                18006537171731967094,
                12374747388220381150,
                11864668767677461873,
                7189191550084420743,
                11292555383413732769,
                7884526425467136896,
                15623541190026103730,
                11092209719983693999,
                8028450957205620326,
                2499868879825625711,
                10808424434450244851,
                1850120526551312291,
                3579487434391430254,
                7970110833128773015,
                11502636942992378521,
                16957022813958395608,
                6038123073412640652,
                10534237287670598452,
                17692645144952827790,
                5499913016452810923,
                9577924119610418763,
                11827916510345201494,
                5412361981268336080,
                4983789128726775001,
                9907721766352553361,
                7746668222533981953,
                1540831215979961919,
                7169664330498339513,
                7180697286303485988,
                10347276511906245411,
                10823488766645044528,
                6133595352212519820,
                16312710139973589114,
                9648734934782395917,
                6873899491086510519,
                15241958636619718674,
                16002332770379796259,
                11637713949810347145,
                13428135478576143504,
                4025997171210729991,
                5342284361833989558,
                1000822021319843263,
                9261749846705840856,
                3617262093116689162,
                17942289330270390442,
                11385597809028053659,
                7287780710901341369,
                14762598407984080840,
                4264460461385592109,
                14094826370185134517,
                3283046943932850948,
                90170700188501807,
                10674057436346217291,
                5644642641840061204,
                14527433886721542373,
                2131222867340329316,
                196797277728757979,
                5366091626020562492,
                3409153327989827025,
                4841679307319540953,
                17375418154652407166,
                10510576230334042050,
                10924962062840002463,
                15837684901594184503,
                4027939380558138063,
                4777859749711837977,
                10339218989308260901,
                1733730490652215991,
                17672772903076977070,
                4032065712363831295,
                2119394036996489161,
                987475587307386853,
                16359915610191834692,
                2709426785752979104,
                10375088739222959162,
                3980631508144303014,
                7635434081101942188,
                9892236427381386255,
                11126971916211561413,
                18061085022989404170,
                4126100172567117562,
                15208180968378211631,
                6986134894614109495,
                9994988553517585717,
                2632796961456398601,
                12453342373460829884,
                18429381111478672292,
                6111081134345892184,
                11421541247114134041,
                13469814569972052947,
            ],
            &[
                4315916980033521479,
                10449503521530616424,
                4856252820643173925,
                5526086074914510628,
                4004011662776259923,
                5158952861074217646,
                11028260963063447109,
                1944236596442929396,
                12978824640840365242,
                1151930226841764512,
                10613193664134350842,
                3406265646941928560,
                6672407465186936153,
                404800540740440978,
                7124533336742235930,
                1535890670990966696,
                9493802817341566679,
                621347847648491799,
                17912640652451605620,
                2057204868614013785,
                10921553714374066298,
                15047552947027887029,
                7153383190925766965,
                15249704771014948657,
                60519377435496530,
                7590844620858428748,
                2351909720781490663,
                10190678499481197044,
                11234076827730248200,
                8663598726765213238,
                14736154147389138508,
                7450150289749363751,
                9994057649404380563,
                10753782001926929150,
                10644356419589646708,
                15190339138217087552,
                2505074458053073474,
                16648880076747597464,
                7963405054197062708,
                18208284408934866346,
                13858469707329190922,
                15486996602241633122,
                15371317889350309918,
                4743078453820751273,
                14468006621724512200,
                11417572129138578356,
                1945302117073820127,
                15602118317671954659,
                17407209257242841534,
                17548975927464502643,
                13226541172879034674,
                2166519041665030706,
                7724204530219097389,
                10010504212552996397,
                17838743458463889271,
                11013562737265801641,
                17168811270168264857,
                5237040028916263925,
                6323527791308699056,
                15194073900873025387,
                3202913343800201178,
                1168012655630214373,
                9877657453665156045,
                5804139521458303612,
                6595671049679611796,
                6954590572896979451,
                4507247193899927433,
                14712120891718480637,
                5958832049354060597,
                5425095238966069586,
                17880721668223019279,
                13362575027039102816,
                96566741790470098,
                14428275782443426147,
                7864448756177300200,
                2963734043870314563,
                3442157303656685599,
                8099372328576403206,
                5242482823389764537,
                16276205423603762402,
                9018844860545664107,
                8496776552307752745,
                3126198658080841968,
                17584426086604861041,
                1756526742975593168,
                2722838766637822731,
                10403662163231417462,
                18139392105860519830,
                6989823411439348939,
                14393884001575488137,
                1056172704485379305,
                2679761112328917598,
                9221013185085765008,
                8290458915554542198,
                8622981061497119850,
                10021485082278739094,
                10922752463306949341,
                3289501128306314106,
                17397609516531025243,
                6547400758074542842,
                15604289899859356945,
                333978770812190992,
                18218065973916833475,
                16127810661386041132,
                11230122129491639064,
                6981851969231440484,
                7558646420357403778,
                14981775577710827750,
                1746996883223053842,
                13747448642231442753,
                284062948563354937,
                16854308500433009747,
                4830313735216528914,
                9674939829599862099,
                14383486138223897061,
                9922042067933595738,
                12607420209760730250,
                1163951431708057810,
                3418701428468514056,
                3177302917570982047,
                3217945469089509208,
                10238490471876301438,
                5914047890159119576,
                14313932311282384038,
                7873142415009831359,
                4600498391648960475,
                13057538823866330734,
                3217841119719619774,
                14930867195971443144,
                14900443877098569143,
                13613747010984006363,
                8607287334267369143,
                16223902093134349994,
                9855946615528227230,
                17467986127811625595,
                12851343650380341965,
                5268799467297904422,
                4860865339571653719,
                7510950311328510427,
                9121129615704026263,
                9985294510085760109,
                14647944548541534193,
                4617350593903353212,
                8812378543069741404,
                14071575867988825908,
                2197500103128929398,
                17867729371980602795,
                4932436132645521745,
                7924275711546185628,
                12144473368809175292,
                11700647470688753729,
                1944725794374986851,
                17806041080297052928,
                16379940501892332881,
                3902394385929377917,
                8986013537553457661,
                1385017881577471384,
                14464349274173417728,
                2673059619166647089,
                3060973120132903588,
                2757656221899057349,
                3067604880707584831,
                2601234120306252174,
                7813691230943610885,
                13146165512554955859,
                17640598915084737523,
                2582363950458650017,
                14304837210049245270,
                12688683908862842695,
                4386971208506064471,
                8522626829538031462,
                17452505302106071681,
                10361450896110735337,
                10033458480810018017,
                14206004007517991051,
                4493284979823281928,
                9561668643117213356,
                10841903269895892982,
                11911984372297937793,
                5781966583230334816,
                9995084957311531830,
                15667884071677615567,
                17480826442761249432,
                14163687943258512351,
                12105651512320717073,
                10102397412947964263,
                4152524068034934712,
                3440079042684086691,
                16421551598963516241,
                7663030021306742265,
                13641230882631300397,
                16458918411371638964,
                11750365409353937867,
                5836777064043393355,
                11948592356238334408,
                14094063963862587516,
                763866157967517588,
                2553862217811471056,
                4978286061786547408,
                5951252532271028398,
                2814480147432244251,
                8355110176394453512,
                955000046590858084,
                418817683267489635,
                2375669799293008536,
                7741560113544470025,
                12527540215014459606,
                4807648140530199663,
                1917176557989775622,
                17997529450210348748,
                15775937348841841853,
                12596647388874725944,
                13005922925492525774,
                12825247172998644416,
                16941346907459597295,
                5414495758637984503,
                17648512745626862850,
                11441048783055867665,
                2036278509733973065,
                5315013389272641810,
                7534399943553945358,
                13985254988036165342,
                2129886592790964187,
                2867979934993314327,
                13920262515085521981,
                17918001318024984167,
                3987525118516777291,
                9181953991016110928,
                6703466246932157306,
                5900079110331853186,
                9052887944141651668,
                7478034775976637552,
                3278041148445352516,
                12130157979346423214,
                16449898609250782176,
                812026766466081590,
                17656773672307181547,
                3744388208002889360,
                3791099528357775363,
                4068520361909439414,
                1959438401502413602,
                11329035886548464086,
                10221578524418919372,
                12980191277097844908,
                15691976303951341468,
                2706299845840454932,
                12001725521641928149,
                10405356943915977601,
                1550503560697397676,
                18146165230312144992,
                8775540381707621983,
                11230591336406034981,
                6228681596165483174,
                2133581336729418768,
                12129007378508537384,
                12921300519235974025,
                9375678805406296547,
                3613435617709574498,
                13499610044718144807,
                4451336373698242504,
                2534710693960094303,
                506672095186947791,
                4599686050411382715,
                10283437000867486258,
                13944373918161748057,
                10749539207857035250,
                3256947136842373881,
                3968459069819041327,
                10310512027304093296,
                12371354326187333906,
                9649271905569519482,
                18149492230911698719,
                18039253857035728470,
                11544885873395786648,
                13115665719718705181,
                5883323805103076804,
                3340581244109826221,
                2740719998676740975,
                14211314271529644273,
                10971959826690424178,
                17661039620102652400,
                2681458501156387899,
                5479854823106014662,
                5974561045219278043,
                8473215965038242639,
                5583715930653494870,
                3470313942834108737,
                10304624977365929872,
                4990026433557930610,
                7637937049242001677,
                10791009944394783717,
                15219205922075360764,
                9002830569992670547,
                17065169854478898187,
                12114851395722862577,
                1902980232892183122,
                7339703717537327983,
                11311240986114825051,
                8620902745551541255,
                17653580530932288202,
                17446311100802812161,
                16853776905767232590,
                13528658510971153664,
                8133142991621414981,
                9198712440714819892,
                1307139024561145181,
                5651509534538305425,
                15500876791491139019,
                17261865242402821483,
                9959143234495304378,
                11774306485096011083,
                3518307525093596569,
                7341017580248227715,
                14703612145595470399,
                13172639134670980973,
                15947158997811220828,
                4125585181533726245,
                11285884221627271297,
                12648910027228429661,
                3451176029776968001,
                1260658404823488448,
                7386532199423374411,
                6316655514361401347,
                874446896878374135,
                16116423058515004551,
                16702770564150696101,
                15010314591562646705,
                12772639082035296672,
                311476976636214455,
                9575298239012600874,
                6021099187287010095,
                15330804127436984700,
                7874626841370189235,
                4317595748546784391,
                12774690740232998893,
                15545525633486974557,
                6452739508938620757,
                17947552420114643348,
                9263225273262064697,
                12599141493060226685,
                5119406136789734734,
                1051975011747189152,
                70796958152999221,
                8899487974683094789,
                17856135838574362662,
                12228587125188734252,
                2703592016861351703,
                3828901820603468627,
                9226218297991283269,
                17840028174806686402,
                18075865263748024826,
                1008172625769821573,
                17053104036036352842,
                2643037037180017789,
                4043044186653414452,
                704036817524553677,
                3469270838261112340,
                10578260962346328018,
                10307304960872788772,
                11100501830170637887,
                1321067784267748201,
                16370290911883104830,
                7277997026279002914,
                9194370955407098554,
                11697789085283623477,
                10882102725986890548,
                879575900258556559,
                910006071300607678,
                6634060453928245455,
                7018272357233154950,
                11323085799124732661,
                12126704870344489768,
                8719425209364909409,
                16077720234498492898,
                11214722890560245831,
                6014665085557683621,
                16140295979403357547,
                4017919047975150715,
                11618339290071597454,
                8158854164475996114,
                1011200605525249603,
                11363709319397659744,
                1454324493486065313,
                11129023648824172197,
                13020217682594858999,
                9576952898239586398,
                16616477346929370162,
                4371687970173016192,
                3516034006703213050,
                2937269151616655609,
                16747880970476287063,
                669709731153728654,
                6519853652860201133,
                3659032765342780810,
                15864573400056223462,
                6297095765997928564,
                7110555922376283110,
                12550039101483110657,
                11249126606834391837,
                950080327127089799,
                15797663250049566938,
                14397834648452338482,
                1543039640649750930,
                8341556893541845701,
                8116982948323973962,
                11495256912967207206,
                14766071288504085467,
                16705412473803765100,
                15090506094727735986,
                18163874144022445889,
                3916968151577833899,
                1879421924266860262,
                13373480014502105640,
                13041781611237972179,
                4588441522074875825,
                10845619956267400739,
                3765600421600530679,
                2843644382474146543,
                15132659436045892883,
                17171692480556923795,
                5511726102977095602,
                8616390053076116361,
                287094189132243978,
                16223004946374614921,
                6933607314950995247,
                9122079358883202054,
                12052612540225242210,
                17627899458327225659,
                14820053630940743790,
                13081262502511482639,
                9047814330512425859,
                13819374748836811348,
                7420314415541064490,
                14372043945426241322,
                2870595232152592358,
                5825108302965985850,
                3862751306343773746,
                8297143078061668797,
                7623527790129832636,
                14976108347586383938,
                15504660286044115876,
                2036590244330470013,
                2396696097468395360,
                2679032459898693923,
                7254694992336225547,
                14405708262513989917,
                10988568636010211554,
                14393755906624408223,
                564784608828851932,
                8920394545124086885,
                12652866383919054316,
                18214421540006560542,
                12566908496483953419,
                2267263971377502968,
                709404620135786373,
                12523996199173167681,
                17951676580855933198,
                13650747407959441932,
                5824756462266699490,
                2509616646477747837,
                5107467310813142389,
                16298586900092698229,
                12700441253440908190,
                8046451347626774059,
                3010230833910681313,
                10061781544557486512,
                8463916499683381809,
                8747329120239848091,
                17820189950727523713,
                15204506030714195945,
                14056191164190175147,
                5065311670562888038,
                560164632281916303,
                13209321978701212835,
                4258270003760268847,
                9871801017584276954,
                10084189192084678352,
                4825074015064392667,
                9581605763592146729,
                1918163350850726962,
                14567773874300155163,
                12638976926425573615,
                3649380560217639658,
                8018661822408339507,
                11706665997779706276,
                9943212030613917839,
                15457711824935005280,
                18428573928454707729,
                3926330427544149380,
                7530117517842274315,
                10714583863650741692,
                1570545389027504522,
                4106262895329894829,
                17297335240441020752,
                15663839296267508861,
                15407625046661333980,
                14535107337220563598,
                16174049732199757779,
                18176047754016168171,
                11399573902740750349,
                5214080916822086211,
                10430898001880200017,
                8371115051436088011,
                13474730252470615587,
                18200790159037433152,
                1982492498839193382,
                18199849879554882755,
                13198581683370037706,
                9793735608111398723,
                7320054643250375233,
                16550404964849262419,
                5735793832830119704,
                17154806469016832501,
                17937411786820605740,
                10105704169377815273,
                10847289282838740100,
                11160609093695125779,
                1612622088505295863,
                6640550752178901560,
                10559041691527414248,
                8619667673730810344,
                1695749010443858836,
                1362829788780114710,
                4786165899179147947,
                17288418297634566832,
                8293662777453621345,
                8957181791504387593,
                2223163959711304788,
                16096735981526477251,
                7019659893524176984,
                9142198120820553927,
                9989692970304817734,
                5594683658237316423,
                16694216368899223114,
                2065885423694944515,
                13511522537087683010,
                15544029444344603365,
                11829821522884913752,
                13656682188207614177,
                13642072954609141700,
                13491639210730977338,
            ],
            &[1],
        );
        // - n >= HGCD_REDUCE_THRESHOLD in limbs_half_gcd_matrix_reduce
        // - n > 2 in limbs_half_gcd_approx
        // - n >= HGCD_APPR_THRESHOLD in limbs_half_gcd_approx
        // - new_n != 0 third time in limbs_half_gcd_approx
        // - new_n != 0 fourth time in limbs_half_gcd_approx
        // - n > s + 2 in limbs_half_gcd_approx
        // - n < HGCD_APPR_THRESHOLD in limbs_half_gcd_approx
        // - new_n != 0 first time in limbs_half_gcd_approx
        // - lhs <= rhs in limbs_half_gcd_approx
        // - extra_bits == 0 first time in limbs_half_gcd_approx
        // - s + 1 != n && !... in limbs_half_gcd_approx
        // - extra_bits != 0 first time in limbs_half_gcd_approx
        // - lhs > rhs in limbs_half_gcd_approx
        // - new_n == 0 first time in limbs_half_gcd_approx
        // - extra_bits != 0 second time in limbs_half_gcd_approx
        // - new_n != 0 second time in limbs_half_gcd_approx
        // - limbs_half_gcd_approx in limbs_half_gcd_approx
        // - limbs_half_gcd_approx in limbs_half_gcd_matrix_reduce
        // - m_lens[0][1] != 0 && m_lens[1][0] != 0 in limbs_half_gcd_matrix_apply
        // - n > mod_n in limbs_half_gcd_matrix_apply
        // - n + m_lens[1][1] >= mod_n in limbs_half_gcd_matrix_apply
        // - n + m_lens[0][1] >= mod_n in limbs_half_gcd_matrix_apply
        // - n + m_lens[1][0] >= mod_n in limbs_half_gcd_matrix_apply
        // - n + m_lens[0][0] >= mod_n in limbs_half_gcd_matrix_apply
        test(
            &[
                575467399379548741,
                13839490846250736437,
                4628353154773839253,
                3235160437296904652,
                17926538529589113070,
                16254839488328114822,
                415479271776402024,
                701612307177551527,
                11282731308138649969,
                15549163798131727182,
                9275741579154885251,
                17195569079203512713,
                16631195896462029517,
                11189046172373216916,
                11335804956694503413,
                869753568386833458,
                15161202273898364556,
                1685153641796888684,
                6519812799688796045,
                9450660698529279434,
                3976001355746699381,
                8755860526328087486,
                9379845976868252993,
                15449115864235101519,
                6189820805905752291,
                1999656252072726648,
                14614135148721558801,
                9412442668458141131,
                4769782526372985369,
                12603364410326539145,
                13266305682919253319,
                6179093516856398542,
                11728866005263196505,
                591231256352012886,
                10891822735763494610,
                1379643190812867975,
                17915879013573969253,
                15654737824492381077,
                1665878964172006292,
                2193929907146049204,
                3753704553030045025,
                2643617658743102861,
                11826899300652102169,
                10741401269021818528,
                16693959619336879180,
                15822691007875207875,
                10165667636444252806,
                14646332733026063930,
                14294714305812123116,
                3675743870135516810,
                15588872085226591434,
                12249381630026428042,
                14466388694354825190,
                8226637301182652689,
                6338248525489182911,
                8273376484864616169,
                12903702679068332011,
                15684253830538215322,
                120323452105612013,
                15775771470841035677,
                6517337253986486974,
                5057172938462112267,
                11146259301803018301,
                14217857571428393275,
                766104140274144406,
                10403765575570339080,
                15356388955716644394,
                5960156924753840614,
                3826246010691114683,
                5270840977803713435,
                17897998794972987383,
                10234944855653803379,
                14541987689760337937,
                12820656189348873593,
                17473239792591314758,
                10637029214964504494,
                3122622009615558433,
                8796921122235952176,
                15762598310456891225,
                11517241390289576301,
                8995337670147793236,
                16090454643652603796,
                10491545194936292088,
                10581918733379812780,
                15469777875510035481,
                13740696746710275167,
                17697148650091366126,
                4516632692241420113,
                13479012489115713946,
                613181577666567755,
                6543068365906077489,
                7140413973568370039,
                2632693564220941486,
                6548358243833847767,
                3580011453626046476,
                1595786269113597220,
                3589869122329502051,
                7680790515161529472,
                7672816417510866152,
                9617378297316545875,
                16326449292469902699,
                3356438369767744837,
                13626365580692467927,
                14645408437700981054,
                8993455766562141654,
                17100348831087759519,
                8482398362235761502,
                9005320120246465195,
                13985608369059524000,
                4578760409329988524,
                13471973404906324912,
                13767793875782618021,
                13576218399241367176,
                8607505428292681990,
                8640829853824545675,
                12917106086264067348,
                2578206011448221694,
                3353125869856782607,
                17577121109228709335,
                2905314237462392356,
                2331575639922950207,
                4757803209372475504,
                10407523519854888205,
                10427101613638544570,
                10653559167855267862,
                15969139228571015937,
                3229545277738910965,
                14459186466317664575,
                2719373534120139652,
                2311539788396444793,
                919197711709305677,
                12869313012019772102,
                4142489328916681739,
                5563346075417511527,
                17320278674721747488,
                7600057562482304294,
                137102128861993913,
                6729510908740032907,
                2556227730136563134,
                11485309894508721441,
                1975505609886762679,
                7418018451444286248,
                3280800891052345674,
                11300546422663834076,
                12023912948840777770,
                14142233633360973935,
                6536078661153825094,
                6708126313364507897,
                12979314974688288248,
                18110797927987234453,
                13375595970475713854,
                4145852684577155155,
                13282868696425541670,
                1506114690671120983,
                12037668219018850757,
                9552135080245814694,
                12689050870789160910,
                9641464072414519358,
                17404100951282287670,
                584528148396378983,
                16991192139314454706,
                13435398806093646746,
                1806232481237325720,
                10525369207353922053,
                10539096468693644137,
                16823332166945560763,
                13216271451931684405,
                4272527731081782876,
                9174012089272655562,
                4803739085170580403,
                18043308998760359964,
                2466414918609999948,
                299112418477190901,
                9176472152403710719,
                2100151601463253457,
                14066759482920415398,
                5442308713266599475,
                7797794762570542387,
                906654145983003262,
                15049619616558215358,
                13797241706831616579,
                15969242154524883291,
                6664585166752881617,
                456998176808935659,
                6372085307953904476,
                9486438637948175687,
                6172262737033149469,
                15771388893063576724,
                16211053079428108900,
                9836380621269080797,
                13835412758795578440,
                13022293692163806538,
                7291668881817289573,
                10943660404337870608,
                10465252139732007687,
                13584635332527095924,
                5217773945105333432,
                475626079396510382,
                10012875662407091202,
                2294484038393548897,
                17235255650481874514,
                12855466329954989704,
                7640972869677452753,
                604066466212666838,
                14544408987813543082,
                2255330913952415488,
                11487976173612122909,
                2606969224171147592,
                2732228719296019687,
                4805516573329683679,
                12510535481770618043,
                16080547195251234240,
                17158216987147533124,
                16252483889283702887,
                8566122285474978824,
                17992433483153604641,
                9742248099517142775,
                11846307866686350293,
                10183020136820942338,
                17751314439768932469,
                706526091974895358,
                16112150740402760937,
                8208604329380417289,
                4681925112017699576,
                16023990145260210172,
                3147541399400389021,
                3572534572158190145,
                8720556900734060281,
                14350849130427767056,
                14103987208350104747,
                4009902137374730692,
                8642370345566348130,
                1509235310089233775,
                1803584577779548576,
                8103474899679352369,
                7632155887896411012,
                6720505338030753233,
                16056811053006708347,
                20245868804036263,
                7392162968739567715,
                2667133607635776013,
                4088935200661832474,
                11027949378461390904,
                1298518526076811529,
                15048847519112317617,
                12370787403008587786,
                10126193873658259058,
                16077210093715640601,
                339546017124009341,
                3262625473090659650,
                155522514636863280,
                17133870939862804712,
                3881378537141359797,
                3319617006815341131,
                408938678387296349,
                3519452531868310218,
                9151628810034763447,
                9069824551063458035,
                16175438057441830093,
                1025725608441043501,
                5528120455681184714,
                14962174970099838556,
                4628262249268118080,
                7668604952914400894,
                14421245150669261123,
                1475685311148681328,
                17331200229977607,
                7008482707614634954,
                954865196375036588,
                12721130434144713155,
                17772110747932936263,
                14883220430477114612,
                8346701667063923797,
                18346426769719184589,
                511089516893085384,
                1049490684134535594,
                2717769095812545487,
                9939780542757419537,
                178653728154192874,
                13977596259293439969,
                16604267703708773284,
                16824820253288999719,
                5015044192952183202,
                17285062680923371961,
                8633846623132382042,
                11767312170862342174,
                7070173857681433256,
                3195337349333954080,
                10624389626802650454,
                9952227996635558303,
                16103075034219456497,
                7190762260873686947,
                7637572995251515694,
                2880941320441182648,
                12986992855187366282,
                11480249557832197501,
                10992570334494572171,
                15293473353050840400,
                9953353185709986649,
                5473168766758534422,
                2273834546925865179,
                17481576375175014631,
                16007610327482007771,
                10021685763866120088,
                15140422720183585810,
                14599059403330408270,
                10717864391356784188,
                11043776937872366176,
                17231223364983083606,
                9444955609293032145,
                7583866547951445190,
                15822743417483210898,
                12379430402523343421,
                15499380185272809582,
                3872771344819526970,
                10944774603860747612,
                11364607709396745305,
                12930627891344540854,
                12649577033823599142,
                71081790476548864,
                10661442845094180250,
                3320677886965881318,
                14395511939432853712,
                7178607417614281470,
                10608904905068105566,
                12919112359212522946,
                1457156061186787292,
                2630619879897285370,
                10619217959672149607,
                17259595048026584368,
                18382463142617996350,
                18080737559467551761,
                17900172273670393807,
                8256196626665198496,
                7885528354153161605,
                12390361365175881721,
                16121675154134812463,
                7006162423030601567,
                2487450624947473633,
                9170525836949058136,
                17199587696327682411,
                2778440771740106085,
                4214370535755222213,
                10220293195635704465,
                3494084265135192718,
                6536254103726059629,
                15347315270124235902,
                3035309337163064046,
                7593827964874002724,
                6657250220369903762,
                689162306084882817,
                16988617413925292293,
                11542800279639267463,
                1161396193556683066,
                1404354998405277747,
                1863751006442991062,
                10942498162725872124,
                14089151747681183597,
                12287715267573932219,
                5639989966754469874,
                17232323616379757319,
                2169863596718309595,
                15913207380900188862,
                17154942699109451282,
                1054068200576663493,
                8407670488993208585,
                8297969397912590617,
                3923918644147518797,
                4050822028508586151,
                12069177518272170880,
                5457932827045810178,
                4319428590498199749,
                9292628295755403796,
                13596479411267714291,
                18144016285235947108,
                12061613157169340922,
                18436504313134004880,
                10205373579634205498,
                8932673649062542092,
                8602267798898562736,
                8434725739455235221,
                14593112102164617091,
                17742461694491219842,
                1082380225638646367,
                14627119236030669240,
                2111648930588633111,
                474310462095504152,
                13588341283759929919,
                7334643839189036863,
                12082398814720285337,
                8117633276713593164,
                12226067923217154647,
                8613740435987095407,
                1720065821455369500,
                2278123359132376316,
                17520489735769768440,
                6164160529494561583,
                15817606273461591467,
                16954590151677190424,
                15025003150656907625,
                13700051717872658120,
                18267946311324265571,
                7224134171579686436,
                10357746181071061772,
                13702788007820740199,
                10454471612142559644,
                7972010451335281611,
                6566222485717554672,
                17140404477880114834,
                18107075263792008325,
                14436650980335290332,
                7462985729135152620,
                4479257078280106616,
                4963716523662716997,
                12351075264308888998,
                9381613057424655832,
                17915205681210289409,
                1103301032493960092,
                7390733404857910899,
                5646956458723265003,
                6881852031997766673,
                11474462020537223114,
                10074504892595096981,
                18270265530986154110,
                13033898379492425364,
                11329340856110014657,
                6846873962487852009,
                14679284715132950511,
                5762958206899451423,
                16858950997373083608,
                12658415513768184148,
                605947486630544932,
                6969263633905442272,
                3234937740604341173,
                1110148774247830933,
                11668278583677202055,
                10407754098687553297,
                2417686119102800672,
                7530424168582333814,
                3002680222699095327,
                15704675259272891668,
                6690366115936985010,
                16337949845558804413,
                2409247934459548168,
                5864915702415821624,
                8338701638724044373,
                10371049922614905160,
                15037630168919872234,
                16870779050531029337,
                11956871297691917533,
                12868211179118475395,
                14334258502588542830,
                13588326263831172805,
                5710629666230160718,
                15210061214358246229,
                16673563803302426473,
                4360718753401653475,
                3446288193042059415,
                1969211299957512841,
                8046929029837267934,
                15830991705791077436,
                10834502346805259928,
                9453135954245593947,
                6441617549735024559,
                9334325755693495501,
                4869716018763257456,
                2055831415614097075,
                15166689527799496398,
                7485029880680080630,
                6151920967401870361,
                1397222502143791728,
                1505614305862413992,
                1461010374171129069,
                8184035195339364309,
                14399586491732116557,
                12965712207658014223,
                15682106046619552556,
                1797345002211603047,
                5346629789286835515,
                7778106305028050324,
                1706742763731171237,
                10724467824751483089,
                12579419113050252513,
                13399259254836577641,
                11915423084091485936,
                946678182575130804,
                16458121607357953511,
                6990972249370508484,
                890502451259139532,
                4270424368509631593,
                8247545724300513941,
                1801612845852339952,
                17483758041308719540,
                6916086592936859862,
                2737745973138711233,
                2417887524053243596,
                5384664014111896333,
                14943057063528297642,
                12718394952034204173,
                2446982003060425908,
                5936547835318690226,
                17541689392918972185,
                3455600047399472979,
                11462977481241656022,
                7740652593808978357,
                10168544572978624960,
                11513860485512401268,
                3032850452841422431,
                14616088035414255613,
                3419881191634612824,
                1329030780492877420,
                10207424479064655105,
                920157290452308485,
                18230987435407639144,
                7055850666076657796,
                5759861191162096611,
                964791846480760906,
                6298102481822543476,
                9631778414172323400,
                16943337751488106591,
                3139604193492917469,
                12954525486464279958,
                7171879659975320847,
                4855521043517721810,
                17861560950998429622,
                14439556530136032678,
                2146560660676406638,
                2477931805045525478,
                18220630939687468083,
                17480104655582276103,
                9658906588685453758,
                8900401944077573595,
                4023418835120043589,
                16832899872978339110,
                1178686892293228748,
                6270325765175673747,
                8254729669168021969,
                2574138951229186786,
                12307389417966828562,
                15258107791537644761,
                6677177683840339446,
                16181736336365239342,
                9100009209370850781,
                7463319891721190323,
                1942611152231324238,
                14939657768700897026,
                11782521561154004074,
                15467452972304115086,
                4250458310696784935,
                5671021285338647749,
                3705241574832771219,
                7348665909967657559,
                53271397903297119,
                5064947506099634152,
                1740366402276272014,
                17408710193712200177,
                15448498405302092263,
                6743466464363127079,
                13994255903404715641,
                4783992854853242349,
                3539481428896870175,
                13315689057294289634,
                13906894805247831409,
                12514428970190120434,
                2383999110333252405,
                12520070852614084840,
                13941905089326614282,
                3879009791548345242,
                5979410257812776821,
                3636697787393253030,
                5008829046821158587,
                7227070514537338449,
                12200965607507694664,
                7540152508302974608,
                6967838404828435305,
                17393893632848259796,
                13461232236356481701,
                13462754845856901516,
                11871853053738474797,
                17468976883674757298,
                3137741177962244202,
                14775774099242061613,
                6439034393988288499,
                18322704676228151809,
                6195120941011064230,
                5959979991317651909,
                6056847048610236236,
                17042162839256174523,
                11417561881130884824,
                15046047411544517103,
                11443682701040972606,
                8885365268842644015,
                11048448950702337745,
                16247143345753969584,
                12368813636786226982,
                15842543547151376353,
                15534220375707255951,
                9323544182631077818,
                14651867219777215640,
                9676432759358709067,
                11349135523271337221,
                12060786354814026861,
                8455537878716528518,
                2516611220455718552,
                2834697229574266766,
                14026724682567928496,
                16194065834422785902,
                3536210387556942108,
                12326677060567301436,
                9778401785405835160,
                8484860443802595491,
                5813709913322243461,
                8582854521396621439,
                7760384300089398842,
                15214249845203742378,
                13051924444243483780,
                10084811846899799566,
                18409493367524167584,
                13375466455709327221,
                6692904664653654257,
                2974376177859028080,
                3466412890624951601,
                5012652013243004425,
                11419071376621362405,
                10316171806558216583,
                5304122490809205486,
                11763687800639718258,
                16915991620777140302,
                17449952616849033525,
                16529039178071253953,
                8027460945600074954,
                8304586649338777354,
                12754974260089353895,
                13718817585228096432,
                7288624726912044947,
                16364439710625277815,
                3691598768778868311,
                2498599291857236321,
                6150206881275523166,
                879919669932726787,
                3847940289848302727,
                8973108085527702201,
                8835099474718086428,
                1775428154619381056,
                9674646708055315901,
                4804998374748022266,
                15667197068183761268,
                11671080770163037365,
                18183825577612667035,
                5196715115451817409,
                15261256208797059090,
                5927143056185064875,
                17671549775031937034,
                6124967124699154958,
                13687636407913038076,
                12821802172321286627,
                4864176808142239921,
                5229559739743490410,
                6174384646073847274,
                7112492701662019566,
                9769272169645010501,
                13117020385783746137,
                14025298218438666393,
                8279159244498971116,
                2334429611427765108,
                10286834819889245065,
                176437083060235308,
                18255037520240585085,
                7311361854076984296,
                11074214072414559793,
                10912633555496661543,
                17232209863183538075,
                18034444892277254827,
                10491712314983717396,
                3133053357505241959,
                6725182279679382786,
                14409286461548123921,
                11093186142603324357,
                8906125291613303780,
                4630236334690968087,
                9064682117492342230,
                14099842337035371202,
                6795865635352330540,
                9453610425174835156,
                5504251706304899694,
                10745438811936662910,
                14010490525842433630,
                13462566825848569025,
                16518815534025225957,
                2444138423121228954,
                5677161677290594311,
                3025414318278761996,
                6832862260509708497,
                10197340078921197729,
                14469811983250567107,
                14646056929411495838,
                13731529545280703888,
                16319880459285614587,
                12262255476906895937,
                17339208589673690773,
                15977537099031596016,
                17939410429932459674,
                12993182878867114917,
                17238534706082454727,
                2053282981771798329,
                17052484244277786848,
                6511994569524351950,
                8560880924849275780,
                9955809746666973580,
                17440412924022450091,
                17993918571689756735,
                13924657572680467944,
                1305456412826022861,
                5539105772156152375,
                464733736235545179,
                5605508855364158677,
                12934212145390513447,
                15031133559727375097,
                5841364006666456088,
                1367659995759183499,
                42284530567339042,
                16950527371076027928,
                447288040070705023,
                2086500570182007777,
                7164047686888175900,
                9253094623176419046,
                9704031988894964036,
                16219232125061732819,
                134218840105885090,
                15790690030012805981,
                4913504368646859620,
                6677364960496702314,
                17441912887943685967,
                6845050781100696434,
                9944304098024264884,
                2658333825110326706,
                14495710831763589870,
                16816753026986606684,
                6108394693883656217,
                3674825700349441172,
                5455306879888392581,
                11040129116055440406,
                5939138794432433453,
                10354992546645397455,
                8132690955796952994,
                4208614571757937725,
                18389691378136656443,
                9724927671847420016,
                2111877634342327228,
                11060010589057515988,
                7413966364618415182,
                1813137957718448210,
                7364915359267899434,
                5364164985059412672,
                11301900090444374505,
                8657891306154732866,
                15450480317670354391,
                6417954264199031581,
                12414015469094402379,
                4315624036191063136,
                16872774353815857445,
                17184838727437143309,
                3265431588659804865,
                812818791443395933,
                10989782102893895704,
                599436186045403275,
                12236521865769865542,
                5485432756957314907,
                13060869015958528066,
                1409406384233490669,
                7232635343932008180,
                5044656061617487397,
                2740598907970496901,
                5318021266376254330,
                4866687283454287503,
                2013910043464447712,
                4064631991160546597,
                14894022554846975453,
                1002053896740550311,
                1390923179817260512,
                6747621518522632264,
                16874591559626718282,
                17148422615678889623,
                7004449378946284875,
                6029197897777350194,
                312526901452736192,
                8266420277206881650,
                16202883668988805792,
                5182030222897424979,
                3955555814727488768,
                13309653404546689989,
                9715158796448409403,
                15033855213424393775,
                9207460156539719265,
                13871549650556323801,
                10311613446804515333,
                6029742500327767946,
                9239513459549335068,
                5437631879089476615,
                18257107349942274740,
                12524770742693351803,
                7714074761538734976,
                16596296037668501301,
                251038933261335040,
                4226458194164141432,
                9779997902902433133,
                3030602692656943865,
                9331391008102809870,
                17747445712689774610,
                2740918437486283758,
                9665828446473412335,
                13581408964710460718,
                12469303281264099059,
                8893571674672442832,
                1379442413022712016,
                18199326928352063107,
                11320581769377597151,
                14066714968631958958,
                6105762775909832952,
                4189227983184722066,
                4883704839132061778,
                9916861677661798965,
                11462346480649342207,
                14778409334292907777,
                4493884456551625778,
                5976634023115886514,
                10089921274563850061,
                10995886567862393002,
                12514005106594925266,
                13487422864108832690,
                5949940164660002021,
                3798824308267675643,
                51594519353241699,
                14042191609008111878,
                1084198794703441457,
                6187735363348137007,
                2222453422075951471,
                1977733178909367094,
                1855687927268674141,
                13507429231362965288,
                4563427348875562453,
                53052655758196469,
                12248719414065642395,
                4043781334046130258,
                16594510413478092317,
                9909324764498283904,
                10006246414842585456,
                6639900859435585807,
                12538869588808265068,
                671676141501348307,
                154891531251131063,
                407141030039689521,
                14303125054294315595,
                12519957692091445501,
                2575047121342740832,
                4442053191994286549,
                7257403838939964182,
                11301512346343729354,
                5649479332294422836,
                17444069867132128440,
                15039830200779644809,
                969922697460401557,
                8850382822556140468,
                17587617640514588961,
                5792622618189642538,
                2959871208851481004,
                4002969457198766845,
                14733844908986830364,
                16619267403139166511,
                4982510272869901766,
                13969983195542601865,
                6407607444387703463,
                4503458032777136243,
                13276938846993601029,
                9999572350287586232,
                6077554759008956961,
                9914549221908685340,
                9747419718740154953,
                12283917679445155804,
                12506018878600868562,
                504483054041077175,
                6931585527138520093,
                2510695777487713047,
                1217537109908330722,
                6299518530600532997,
                4461483227388303475,
                345473314729953715,
                11577208925864677728,
                13714009284775997558,
                9306564142261809693,
                10493376611772360762,
                12114735030824310442,
                3723726520319678928,
                13936502274367500072,
                13361717531410065438,
                11054299856501427866,
                2346032701355553484,
                16480112851446496738,
                1716080732608071776,
                13906214996195234076,
                5677286823010362674,
                5031920382916298075,
                11682583709157560454,
                14368427604644474820,
                11974313857346928172,
                7875781703666737260,
                16031516164256186906,
                4566375739076671625,
                5097663793007619497,
                12165327848553281827,
                14159539264387775940,
                17711785725905262208,
                746306330982150615,
                10618303353707428560,
                16415694862704526911,
                9426321299626982899,
                203404317597163992,
                9784341849958013566,
                17737510091254724918,
                16467553962565191057,
                14963305126792873612,
                7875246701602917591,
                15395884754391863730,
                7674782145333341935,
                17779948927306734041,
                4692271260276963472,
                9147303984837818022,
                11261482143084613378,
                6771303715480347754,
                1021253236102285456,
                2938757268921792891,
                8110317487741901284,
                5629482782068783477,
                14074967295738352943,
                11830360745467057113,
                16065773322812244383,
                6799648256361227046,
                10881530047107857944,
                17272480164822477419,
                14188187909112608200,
                8492727723243508205,
                1239958563141876800,
                2396597814031561226,
                6365846116588909456,
                1747799098922280685,
                12329761501577087761,
                7103565661859178619,
                12929732902203489138,
                18130721204914820146,
                7404709294083080877,
                4076736505655677441,
                13997485323848566931,
                11355165264923013442,
                6265382391023965751,
                15037801805473945986,
                16283172306662528168,
                10784174118773201917,
                15442466842568320074,
                6792336536532509513,
                6841433016666743757,
                17062933928060878150,
                17995161545501763874,
                398314979814197042,
                5371491177519133552,
                7487291916923596074,
                17726809651563472115,
                4276741984609027067,
                9063884951015483403,
                6604298906930427610,
                17639693268109172166,
                3512896200182209296,
                14176754219633674214,
                13075641158366811356,
                4304608617823480610,
                6827519196509520249,
                10991757902125467581,
                7985198732976208765,
                15286834137228061227,
                58281206065522927,
                13204561433899186751,
                13223435687875504648,
                2520091453248762960,
                1336536536994438540,
                2657040408513167000,
                5767035770652592231,
                17260962340791386954,
                17641333140100333702,
                9700277297324555168,
                6985084747499361442,
                2237371457239959946,
                7946926051156145397,
                14111197867527256838,
                8805745012779252660,
                6198429337106777414,
                2615850794923160576,
                11783140326213116396,
                14893203149537487125,
                9567314701273251044,
                11695128483656617057,
                647988395719472509,
                11422708536691070047,
                13239243760652390256,
                16636779351598291051,
                18437576843798320734,
                13792983340413428549,
                12074737915047871515,
                17066180541625455492,
                15179271581398274415,
                8175824910139502819,
                14714425222088264251,
                1989693470019103578,
                9855573943302266616,
                6236520060938959996,
                13109545139113435074,
                13566528788838365110,
                1410706471600329523,
                17673327292701259018,
                1282280051559055632,
                9017178883540247720,
                15228935123367154482,
                1258177275149976100,
                2018246390800272584,
                14356985980164438273,
                1823604866366992170,
                17903023640589067433,
                9347737820956061500,
                9735146081773156261,
                6689847293073153955,
                11126271016334934544,
                5994793191677038724,
                3604730352973328532,
                17605242569431998414,
                10327617156260919079,
                10682252855809435330,
                9100234729746896760,
                16993079188042570603,
                11254975990606724502,
                16242311059560795961,
                14227003883755270012,
                18285710061812359350,
                7387005544448366058,
                1145109023330252723,
                15022228266755818179,
                18168093935820946866,
                8922807670716322377,
                17623516307873116019,
                1206890136710388934,
                9516372380550133567,
                10133157547696520694,
                4294971167499340423,
                9743061062772065165,
                5377657584522832108,
                14225392737899643394,
                4157013585438895700,
                10125549201612144267,
                4100226799738447148,
                2075172157465998108,
                13195036640815520738,
                6821431496943827604,
                9246396951088662598,
                9751868297178759982,
                9700779142744360518,
                3658815564772137188,
                6538317493999171090,
                11697362137005293086,
                1923505046610352654,
                475014807978174723,
                3630137128781123727,
                12150077442705041079,
                12427124278971727290,
                13479235332914334363,
                5496164611508275158,
                14936501386999605874,
                8882652803543653011,
                10952558904165612665,
                6739207228856725932,
                8456163010594333639,
                10535813937687860141,
                9505036768783170961,
                11099064892618807360,
                169745221320748492,
                4707891234763005567,
                7738425121897682831,
                11404840688507500898,
                8691505940234327527,
                11771261818863120062,
                14723964031553687059,
                15287258300729004005,
                7573783454674360892,
                18179022528685282485,
                13245862641188327159,
                10588353385460229379,
                2735402380843149021,
                16860704493911100849,
                9024623514323424705,
                12333075916853020349,
                18244278992464300971,
                4867301239541398634,
                1314358060529143728,
                7666882040058016129,
                13150073136816015157,
                6301083950105345225,
                13976444210903821837,
                2060352136785153044,
                16200698722576705758,
                6460888606184453894,
                16445752899318815973,
                17179709527494545154,
                15536997874687360034,
                13022268941493329839,
                2696716978218378749,
                3362347068528182905,
                13565362575501294811,
                2079150969445622376,
                4810345717233969931,
                4472152818411820065,
                6453260112485098774,
                13356564228328639839,
                17110313288592456745,
                1435938168541586853,
                7462678282816862243,
                5371246349000737177,
                8429218073787618630,
                5346521300198079286,
                12789353917659187733,
                14442956055750007343,
                15244751088029774551,
                13297251472118828207,
                7215676263905100435,
                6007211881525156123,
                9493592255758432036,
                16602057928234062153,
                16587917920906294320,
                16979836431020046439,
                13094140568345251281,
                13916421500525839258,
                5769668949157229760,
                18086484472032192111,
                15116469247160210173,
                10732312892355549063,
                12164027081808213486,
                3079069950535673423,
                12585139553289800020,
                12875306561259226432,
                8811443258909940002,
                8124174848473646108,
                15670453854511445398,
                803566658286118334,
                13806458323797710500,
                4762982112857169920,
                1859710041066580425,
                15720087592804142456,
                16303182845117023833,
                855870575093385721,
                4703327830937492490,
                3313518290113627482,
                1178633794722110520,
                8354185752905946404,
                14766290794342661211,
                15214154729438679475,
                13729401929312370465,
                9011734493161906949,
                13113532936632755704,
                1345447462867773483,
                6448244176056184701,
                7012477124755342304,
                15580038433286556688,
                17848919842496199431,
                10028316423612011933,
                869027013079958907,
                17009799055507152201,
                18041218704541497506,
                6045145319869314743,
                12377421195637201983,
                8449076291782840438,
                1397849232559267847,
                2187434984921684478,
                3403378911422360969,
                17480693808924740870,
                982483816965115826,
                15888731420268623003,
                5280522824450934391,
                18281748078592209071,
                4719177826249662435,
                1969348133672517676,
                14302905663206015829,
                3255827777415561924,
                2578045746540552666,
                2573896329300556567,
                2391501373699313740,
                4791581547865311205,
                6247840047570325773,
                2469705008517634438,
                3056326695519610316,
                14948296709540615024,
                1062923194288577107,
                16375392771639774947,
                1015529378012280447,
                10746830707201373455,
                7715886963806302343,
                16297406106746590647,
                17715588202572550265,
                13527139072558963929,
                99211582412004024,
                2711459600905895935,
                8569541865276767972,
                7613790287433305305,
                653208029905987655,
                1727854787208424357,
                15748356304420272869,
                9959409749691667969,
                14291894407744446212,
                503703217735428309,
                7876851416985445297,
                6431225480119190090,
                449483234453843792,
                11523298865591649623,
                3435872863223259742,
                9552329090845154230,
                2896972164361149982,
                1269310417738704553,
                18384605444802363468,
                1237780458899379603,
                16910833523727563142,
                4482981074603401198,
                2121371960521454641,
                7796532283059204409,
                25790177388111233,
                16120544267889484355,
                6144232545049390988,
                3521563904857210638,
                556210369885776174,
                8324534100827531805,
                13516226267058770145,
                8747429514104290165,
                8026816217324859006,
                15405579933448964648,
                2482550910924013528,
                6079310715073401336,
                10449962361225046441,
                5524633401100253579,
                16053481108469650499,
                2790101718009979348,
                9301878289303197606,
                997249373872362850,
                27300315815771942,
                17737705294943786804,
                1424935685660120924,
                8424798139585257768,
                16211516871950763803,
                12990358998177017683,
                10249597119118506275,
                18408586087691363166,
                11438404066809034737,
                11012584279891838821,
                6696271295716018307,
                9930416860933484049,
                13215962566134307271,
                15246803897725633075,
                11387065919388119048,
                2466603392488411175,
                681132078373685642,
                5130503773631633718,
                2509149940966576933,
                10411902793773420668,
                3635262312863149371,
                1603062208885790503,
                9360721267289057530,
                2678321596504669944,
                249229520366360442,
                17329668685637718997,
                8915036515108497990,
                490784964147667014,
                14515392651593343599,
                2409201985556015912,
                1611664978682553018,
                11850486551349563879,
                13761189022094302350,
                3698463909621999146,
                12569860844893297150,
                1227995028080714851,
                16075767045086341788,
                10199406988903905473,
                17794651552362356270,
                16092930490238021824,
                9328723035395370944,
                4734866727034661155,
                2657796117079752451,
                13923818243046619790,
                16472136068270228914,
                18058223245533848756,
                15345222547995310311,
                18050028532712597482,
                13589945113509429199,
                11522903866982422064,
                17951848030650642905,
                12633284474117061774,
                15115880423554767983,
                12930423604492456604,
                13329164597703249258,
                18338837626925469553,
                6367228635373808510,
                8371066919516397122,
                2160980766020466019,
                13425362939901273481,
                10449588671756455076,
                13946011226909253684,
                6563076431328948436,
                3467066912321557163,
                13817709282661574385,
                1954831874337376368,
                13247212989673443019,
                4305395659031332051,
                2576959295441697293,
                6896780844531978074,
                4838944567483159792,
                983435118873209304,
                17099666216820037611,
                7618027338181614041,
                12740220343512004286,
                15668003088622161839,
                6307065336155032471,
                12866355547695255339,
                18105505267682088262,
                309702548863399257,
                11431934014429398771,
                16337123462420112626,
                13455974306445806251,
                16619990986003346004,
                7250047582443547679,
                13575492233443252104,
                13911871272758592885,
                17460767687190632452,
                15247929310408202516,
                18331682218295863001,
                261323487582141633,
                7614661128980424817,
                6422108758439359265,
                2081056287942587161,
                7220787374877583349,
                717472904341595891,
                17813723712700157238,
                3241494416439598726,
                12794513487028370881,
                5358721158543461549,
                4452065304298022398,
                11030176201923581521,
                9210001828401987024,
                10912815101479991274,
                7091546400312646488,
                3013240817218288252,
                8910956722230923212,
                9054959438387840764,
                16851843825181072249,
                13414604809457485698,
                12075986842954907009,
                4658854495181538571,
                10995225665179398310,
                10440147773274056063,
                3614430051441949604,
                7271983795362664750,
                3047687937173698630,
                1994619618655862338,
                13179073926046162775,
                4202285970439440298,
                1118379597613009280,
                15519245009495040243,
                4256675708756697669,
                8238649640283351416,
                7342244199313224551,
                12880588809549328523,
                1187652721118103080,
                5424444711048922656,
                4396267379797475032,
                7873701471102341918,
                15699709671604289187,
                5918216801351035784,
                10814352802756983884,
                17646214522162102119,
                16006963122846937167,
                1430365631320738851,
                376457101376932550,
                8871916202315560716,
                13011781717220903069,
                15215597531017339743,
                12289548108447181959,
                7346780333055067602,
                9224378704987626829,
                15268931953997841814,
                14041223036336774753,
                8927234293163765947,
                2897790799008400374,
                14582010918862717675,
                16940686021857041143,
                17809536309397136921,
                16537410647678046250,
                11385770080898495399,
                11058909064571773777,
                902170224533021061,
                6155050419392920377,
                578026483489571474,
                12192209733415829173,
                9800697958307651530,
                5325607519225962561,
                5749788716129163638,
                3339204573835152173,
                11711084023976848626,
                15800866735538093521,
                11602514118015409575,
                11845686896612751865,
                12805561555388014296,
                6833731309656940130,
                1683693695029104238,
                3252505788213114441,
                12861064563156096686,
                17880480561349418723,
                7109337509100178473,
                15672204595961079139,
                12610706960087661501,
                14127013368071208214,
                17303281711208279022,
                11071871840671280390,
                13019666508300875107,
                13450446740352112547,
                5215478633835401279,
                8173215137132965511,
                3385959284242241690,
                1162541066556458857,
                3982176102869150348,
                11033193378737041188,
                17515896641420400261,
                5108442075949478481,
                7390295244279229703,
                13200704351272018481,
                7838787871014850842,
                6967690604184701289,
                17306979495731024219,
                8091038377110761360,
                8807486371208741534,
                13035790401659421556,
                18352767143317290044,
                5794344711151456204,
                11362614524307015732,
                13861808480505787139,
                10664293568977350297,
                12429231307955906550,
                787191891079923458,
                10802046887566427941,
                4479080089699981872,
                10284905844912789930,
                11562290707970545935,
                10020771126265957209,
                4492419641191856378,
                8315733702710074653,
                8179451464749720683,
                11440898133834467541,
                996731441922278873,
                9874517357398478369,
                3191161615687645368,
                2749003557840473686,
                9190738483349545066,
                4941311214960612971,
                7731219348943068526,
                9841834562404157494,
                4980436855570339065,
                1530764681258350086,
                699509151710528021,
                13504108558405070318,
                9218776911300271865,
                194583242318484046,
                1219401915806982494,
                10269323235937088132,
                16474197391285416186,
                12429159956713323486,
                16406837010296707669,
                2096879123910896284,
                9478062389402448298,
                10390582542218348554,
            ],
            &[
                5506952908463110193,
                3806848119604616776,
                15253829699929639786,
                6153710766072962714,
                7724992251147712460,
                18274643156043175664,
                1050131577594375425,
                3075260137230555926,
                527421990694193641,
                7584901852578630990,
                16965489008951068436,
                6638899686914020055,
                5548349519839988674,
                4340131637208080214,
                13231812794129139237,
                5782596599117579099,
                14707631542809621934,
                13005413828836673107,
                5027517060994695773,
                2409119181385285185,
                12072018513207904945,
                4610121114335836886,
                13007992267227833961,
                10768178433595858647,
                5655501324069419836,
                4047945395373106430,
                16148618693961270401,
                14207121723132763208,
                9303659817261703525,
                6150302743552282248,
                6626197242081859424,
                14180443744733228035,
                16111493454316570674,
                14281519617792962564,
                15852632623980128457,
                843715310399868484,
                371586007013897923,
                10616674417688079376,
                15194274051554944848,
                13553160050325440579,
                17628757169066045879,
                5422077471618948581,
                13336098216102409317,
                17615444491978875175,
                7561910121468822316,
                16521387743687265557,
                15893994794718565447,
                14500498851291186914,
                15404349041758659271,
                1033447516621898753,
                6046875549177986318,
                14017892727616312226,
                504532865625997473,
                12327954595628496718,
                16114003010774897368,
                5123840833265554878,
                12135628469575155214,
                10148936972500311133,
                270050642558637886,
                4713725442176248124,
                2061684810683917013,
                5950651629769154975,
                13856635522463752336,
                3060080971213245973,
                15505219872384321749,
                15660711294880766518,
                10399903341728067157,
                6651665398472767861,
                14008720726998708431,
                13564382325331985222,
                12275490277448149552,
                1079407395917348059,
                9571671064864316311,
                10464498765191388869,
                3257185038323220608,
                16947719475258977056,
                9041734436677629237,
                13800437069910936773,
                15077537963562816357,
                8788516953891922746,
                15832918598557989385,
                14579847279540664520,
                12622946476897850698,
                5338253386688349876,
                364868833142768548,
                3894073811511765730,
                10883504329458004367,
                5694806037383475667,
                17592653866771118524,
                16359891716286511995,
                11831511743093412575,
                5346828754811097128,
                11209687597326063736,
                9730095457442822720,
                10190594511925207302,
                16828792882592167118,
                4455664427487546597,
                16063117532002702952,
                9448048817668105636,
                4972988473343674204,
                3912110890459805268,
                11377401507523354142,
                11367029609143622993,
                9108192617717426353,
                1385875158717547342,
                5297004339925541821,
                159593902424474731,
                14192871358313078323,
                15224505595714662032,
                6200287398085215168,
                17286563738527915901,
                9791571204583847414,
                17516325043571924340,
                14366344976190972987,
                9435375229224263329,
                6178661234763053743,
                7945181496515342114,
                9354008202308728070,
                15540756784774251412,
                17860277456202575278,
                14209596746285961883,
                9512088169794924312,
                5267634490833581206,
                1676119196958948766,
                8311957173462535989,
                439611748176266368,
                5925545614682176396,
                16253071308860013431,
                12311940603650557324,
                1541228309549358910,
                15649818561737411196,
                14275602707926342519,
                13975876390445667038,
                1479178557847854141,
                3285270572845634784,
                11593180894357303235,
                1042820108092959158,
                4253569036877676973,
                403560233122940030,
                9510252758318344055,
                12102364779301891941,
                17721965720899710575,
                14089693913291613024,
                11315184887553464912,
                11126194784026669316,
                18349021879540859921,
                17629602484099316111,
                11791433535374809407,
                1403337479180362893,
                1702732204405483129,
                14577210696810087119,
                10119062496336415952,
                3215108150128546038,
                9555329295127310367,
                14813961943202590306,
                3854717936189991211,
                15305281209892145119,
                6753116109099393644,
                5167941309253736764,
                4805407145840605183,
                12052939585848547177,
                13815002107314275347,
                479262239434164167,
                9984369164481245166,
                32617941710885992,
                3534732426571034215,
                8949504569861129446,
                7141129504706409563,
                14837112786678223968,
                12646159468354254402,
                1634895633875002200,
                4501516205577371448,
                9599178568514083396,
                6662855285932438630,
                4249204762219666393,
                4995345425484051611,
                8852943034822615167,
                14664266321575016915,
                2189704958175050818,
                8112056909545221404,
                13405743490067994915,
                12177032774091373498,
                10374323912657667558,
                17137017152638854266,
                2894829133650272487,
                3043154777910298559,
                14556828612298618138,
                10439272111024509541,
                10616042914455131442,
                15806801477164016164,
                15930845540381766256,
                18017643322697943023,
                17619291702832936657,
                9117612259919897577,
                1044226778242844875,
                14631310785412098609,
                10136888858639713266,
                14449349093679970154,
                18069697024764971954,
                7973459274000903425,
                4238333360181338745,
                1025957588977991867,
                3591096666629647249,
                5973242157764053501,
                14633601316455469060,
                16491905444740244101,
                5258795269555777153,
                7917683918101720763,
                6957575575665285083,
                12471933279161455709,
                17374669784850184346,
                10294948521464027513,
                5365059148739477552,
                7931647940471320382,
                3732756674714358060,
                5378482385455801316,
                605630847399035668,
                2662239966283783678,
                15246382162602486789,
                15502092928361596926,
                14822462227958686755,
                4355260933028095719,
                1388811864175835035,
                16111165341634520801,
                3856506329330266599,
                5480941990955834423,
                4066690789803835796,
                3446091266660667833,
                4443837396874554687,
                9382184593756836423,
                3662513746157661468,
                5981768571534548910,
                7224757717208818003,
                14120780684632575241,
                9320634833813385940,
                6667622706595582265,
                16878293759452873978,
                17088541568896866063,
                7372626780022230514,
                436808405518274966,
                14252678732721446921,
                2104869524947222071,
                6631685410375788505,
                2149637305151365415,
                806417526627454955,
                7421434493476719453,
                1465989281562985011,
                563947906496868418,
                4704820607165988411,
                8173652582005143181,
                17819565157595430094,
                9845347207851603114,
                5241769880445368831,
                16575699457001929457,
                5985635078953360403,
                15978649837734735596,
                11510702129036230197,
                5228690908831482531,
                7526030925056215404,
                3287154526100936638,
                12117225562485013813,
                16823232684363470438,
                583977556407730332,
                17697878485894786632,
                3845824921059996004,
                6726269658160672612,
                18123920920510771001,
                11508614555715257982,
                5614311603142419893,
                6606919581194165114,
                10784854988123637320,
                2264767800782756074,
                7796673814299662386,
                10902499857059539779,
                11823788150079598671,
                157113151127610790,
                14352523930925726283,
                8130167232929626062,
                7847535894354429131,
                11565590994444363555,
                8077867159278395820,
                17389179712963819345,
                14935277299146800308,
                14474917870682010210,
                319588319279269778,
                3134118488268119033,
                9681404746624604797,
                11814370242537451152,
                120347778578480442,
                11682962956864175266,
                9032999928833956716,
                8517160786434081542,
                18099054458154837484,
                6305653560187270190,
                13238111375831798679,
                195224219159194139,
                10509000520281398525,
                14440378477979315635,
                10540539637383947317,
                1816835103067453372,
                12292190545991200585,
                11929309142065466032,
                14923664747079208466,
                15293490320152873937,
                9334278390697238298,
                16215015280814728675,
                13898634333978600541,
                3445627973235322308,
                17969481267516901478,
                13335832845120597892,
                18326371021140590109,
                5890488708488729679,
                4406980003802585112,
                5344445953698019450,
                5013673532816678285,
                15961414409746022478,
                18404414171806953543,
                11468650198917792431,
                15531825206300874617,
                14738477539196458932,
                8676622966697723522,
                1368635071662477864,
                6543668723466390118,
                12114558566232943413,
                2303004786445269111,
                14170489424093999995,
                1985671304134573656,
                8331463571610542764,
                3873015007715912993,
                27894306076102606,
                7104654970761334900,
                5686730752548624804,
                8233904160277754285,
                5172053108218104416,
                12113360256208938435,
                10280077545660908320,
                13821974650155757108,
                1742749416226970211,
                11654300655577230702,
                14976077464467803998,
                13890928519824763701,
                13181570432818935900,
                18400222548512349578,
                4900627745620570574,
                9237622473186007483,
                8348806201397375307,
                11644861466959244564,
                7660774625358692180,
                12640800868812968394,
                10126105491751574188,
                7963584681349558312,
                996039405712308551,
                15174300658579374594,
                16133028349079126235,
                15172075459306840070,
                4754453455507644582,
                2082474474919902411,
                2006285022788280185,
                2422722200295769492,
                10019647931754062695,
                16349712694947095249,
                14083930401576374224,
                615386692410750348,
                12397858553466621141,
                1573457883557036777,
                4980959182073659021,
                11443908431408514446,
                6984262965031673234,
                16633481031388421272,
                4161251318834567407,
                12821551347784643969,
                6608155693481660229,
                1611043482167472050,
                15122218494830750408,
                8839500240035607198,
                15384724866021917283,
                3070733572920490549,
                8592993878808367240,
                4271248517927408440,
                10865144033685961973,
                17023530530649284392,
                3353462395956110692,
                18120823811324392964,
                10221129839711512089,
                3227448551694007251,
                1988380654856065045,
                2242021796806927365,
                13590775423321667198,
                10692026012126777875,
                11306555419118442917,
                11316252767034863343,
                18760321466791470,
                1337100093785900848,
                6492339287439306118,
                12125411568967459196,
                1797441814675370229,
                1631938251008432317,
                6985480655598205780,
                2348176182416429294,
                95412213746880721,
                7175549597849762403,
                2094795596458793654,
                7829002790007311465,
                14746732570765651081,
                134877455607034691,
                15638581748054019794,
                7599776457298678362,
                7986432103542221551,
                7500529350917377326,
                15884143360659537974,
                9903748562237486241,
                4664581449394265591,
                2962548990311093356,
                16023325614230646625,
                10426247192951388751,
                11003478360555882777,
                11539013974193753851,
                12270292209530478200,
                7423475458257570239,
                13100711480717389630,
                16388389606841277356,
                15191771423361849007,
                18427249344570625913,
                4348367306971908628,
                7942807882176181032,
                16409042118394465962,
                863438358079742954,
                1124736515135783861,
                16872942745563460973,
                5139090909118018177,
                4669623391383642012,
                5118057586113819116,
                17891190081052445844,
                14867908875153657094,
                9390870292482891489,
                9171235516940002341,
                2343634198674654987,
                10960182345705330093,
                10619370169029979867,
                4571519790281325513,
                12889605798772993499,
                15970143036917147631,
                10001359583966733655,
                4381102045316081715,
                4163914690821122330,
                6468638717365001528,
                17479787593534552160,
                7599949586205261816,
                10593115230897863905,
                10775345788560696421,
                3838567407183537934,
                16971529548430600403,
                4980854155284411004,
                5827837251392141026,
                14830810846144815135,
                3821271250454438306,
                129239671811848792,
                1581976158752737667,
                2612788759361274201,
                12230162761710825594,
                10097159439277024421,
                4401184023778243978,
                1406084492350700231,
                11938842502217979388,
                7158646758876276842,
                11154483509538442887,
                3807887936265348548,
                4411855765274191285,
                14464002811165395280,
                670130159021436363,
                12640940208761414730,
                7728254476113900072,
                8282009448283823101,
                16968891258239181965,
                3688865346582500831,
                1861487375161465963,
                18109147777912453913,
                14054735663838606991,
                5895084269872329312,
                3180503889143088947,
                16735588978438822575,
                8938486803129382823,
                8074810307626673403,
                9199264274209535698,
                16453402421557915146,
                16952664837297462870,
                5161275365544152420,
                3147874599125818621,
                10656989901498080537,
                17283970806496112517,
                11904772904109037898,
                10815600643737317830,
                4190612046695163764,
                12931964669364485889,
                4640956854723597761,
                12010307958665631876,
                1817311075096228914,
                17760293859189071748,
                8658199478093129696,
                5980712071282278126,
                11751139410599894151,
                13230413742965562522,
                2646723965833712773,
                13682876666838619530,
                10800431924235403175,
                13382436438625160191,
                15176810074556266548,
                14653393669910477601,
                3089870430325875498,
                7482067547555813276,
                9577971917293864659,
                6960932610226841152,
                15130393967538609694,
                6648313153230285085,
                10536236724893566776,
                1093917583121871970,
                17300922283598420212,
                15626332485235410632,
                5710388017071160883,
                16425715338132198876,
                10099451213347819493,
                15114188552391671070,
                5219737568429214388,
                10291251738376368665,
                9999550435891258477,
                11261758167548915441,
                2059999575592944296,
                4808217191002697670,
                11144850771283015677,
                11420508211474364970,
                16799732563689238327,
                3497095200015961133,
                18400167689108815834,
                11950366853355105307,
                11210662566186499912,
                7630565784488556621,
                8708124700824936536,
                16451883938293396059,
                3856271038937245480,
                5014991179538099218,
                13960473819858174514,
                125754478403752818,
                6836966187910773427,
                13398538180289167798,
                4342012043427584780,
                10587058069497406375,
                14137841459960663361,
                10659720389989370126,
                6102546442785984287,
                12392236227630837883,
                8332615892803265341,
                11617599899986402733,
                13605114078859281596,
                7114983647870611256,
                12404096710235362863,
                10847060057289621091,
                9048460749990878785,
                9926828906222562343,
                517853205147118455,
                8179968105148709421,
                558104921847325950,
                7022967858240775546,
                8937761758962462261,
                11010345473796135921,
                15162173902777211238,
                7737154465700094829,
                10934536765747213009,
                15409842658458358292,
                5139225699322622015,
                2424051831250781183,
                6113891917550200290,
                175401245396128529,
                12650806140631886299,
                6488833543271648807,
                12206734304450002273,
                15167188033176677277,
                14723303654739655264,
                2179622318393018228,
                9921850291797916605,
                1533691482276731934,
                7193267197928543879,
                7697000401879611136,
                17905282013737611432,
                195708939669136652,
                8568890883842831118,
                541635191945155915,
                5909439847022830310,
                17931009500490606267,
                6813293046347240,
                10373454508133137832,
                2518969817879302450,
                2557266502794028272,
                11482262686846267489,
                9426469967764905872,
                13731315097386886251,
                15575568313782898237,
                13595230256649138562,
                10024833956978279813,
                5285361198933870158,
                17437810250035109171,
                15432785004507430528,
                10310987294856413932,
                7168389759122719634,
                2631016883595934144,
                17529280321808157946,
                6962165250869702643,
                18242038243432348673,
                9407887776756608694,
                17629211158466299816,
                651801541305818097,
                7005455043755295025,
                3693142147562879928,
                13937562557249975083,
                17185080793670498070,
                768121941451659874,
                3094549444790063442,
                13260257407036400094,
                16126489432734117941,
                10272660138986488848,
                15438764110169806984,
            ],
            &[1],
        );
        // - xs_len > ys_len second time in limbs_gcd_subdivide_step
        // - x_high < 2 || y_high < 2 in limbs_half_gcd_2
        // - xs_len < ys_len second time in limbs_gcd_subdivide_step
        // - y_high < 2 third time in limbs_half_gcd_2
        // - done in limbs_half_gcd_2
        // - x_high < 2 second time in limbs_half_gcd_2
        test(
            &[
                18446744073701163008,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                4194303,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709550592,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                2305843009213693951,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073441116160,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                2147483647,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709550592,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
            ],
            &[
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                9223372036854775807,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446743523953737728,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                288230376151711743,
                0,
                0,
                0,
                0,
                0,
                18446744073709551360,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                1023,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744056529682432,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                288230376151711743,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                127,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709027328,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
            ],
            &[1],
        );
        // - !limbs_half_gcd_approx in limbs_half_gcd_approx
        // - n == 0 fifth time in limbs_half_gcd_approx
        test(
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709551488,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                8589934591,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                9223372036854775808,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                8191,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073675997184,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                8388607,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073707454464,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                65535,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18374686479671623680,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
            ],
            &[
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                576460752303423487,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709420544,
                16383,
                18446744072635809792,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                16777215,
                0,
                0,
                18446744073705357312,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                72057594037927935,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073575333888,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                268435455,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446603336221196288,
                18446744073709551615,
                2251799813685247,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709518848,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                31,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446743798831644672,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                2097151,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709547520,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
            ],
            &[1],
        );
        // - n <= mod_n in limbs_half_gcd_matrix_apply
        // - n + m_lens[1][1] < mod_n in limbs_half_gcd_matrix_apply
        // - n + m_lens[0][1] < mod_n in limbs_half_gcd_matrix_apply
        // - n + m_lens[1][0] < mod_n in limbs_half_gcd_matrix_apply
        // - n + m_lens[0][0] < mod_n in limbs_half_gcd_matrix_apply
        test(
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073575333888,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                33554431,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709550592,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                1023,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18158513697557839872,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                1099511627775,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446739675663040512,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                562949953421311,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744004990074880,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                8388607,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073701163008,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                144115188075855871,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073441116160,
                18446744073709551615,
                4611686018427387903,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709543424,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                70368744177663,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073705357312,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                1152921504606846975,
                18410715276690587648,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                2305843009213693951,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709551614,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                131071,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709551488,
            ],
            &[
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                8796093022207,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709486080,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                4611686018427387903,
                0,
                0,
                0,
                18302628885633695744,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                17592186044415,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709547520,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                7,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446741874686296064,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18014398509481983,
            ],
            &[3],
        );
        // - n <= s + 2 in limbs_half_gcd_approx
        // - m_lens[0][1] == 0 in limbs_half_gcd_matrix_apply
        test(
            &[
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                255,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709027328,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                1125899906842623,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709551104,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709551360,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                8191,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073701163008,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
            ],
            &[
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                70368744177663,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709551488,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                36028797018963967,
                0,
                0,
                0,
                17870283321406128128,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                33554431,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18437736874454810624,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                9007199254740991,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709551612,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
                18446744073709551615,
            ],
            &[1],
        );
    }
}

#[test]
fn test_gcd() {
    let test = |s, t, out| {
        let u = Natural::from_str(s).unwrap();
        let v = Natural::from_str(t).unwrap();

        let mut n = u.clone();
        n.gcd_assign(v.clone());
        assert_eq!(n.to_string(), out);
        assert!(n.is_valid());

        let mut n = u.clone();
        n.gcd_assign(&v);
        assert_eq!(n.to_string(), out);
        assert!(n.is_valid());

        let n = u.clone().gcd(v.clone());
        assert_eq!(n.to_string(), out);
        assert!(n.is_valid());

        let n = (&u).gcd(v.clone());
        assert_eq!(n.to_string(), out);
        assert!(n.is_valid());

        let n = u.clone().gcd(&v);
        assert_eq!(n.to_string(), out);
        assert!(n.is_valid());

        let n = (&u).gcd(&v);
        assert_eq!(n.to_string(), out);
        assert!(n.is_valid());

        let n = BigUint::from_str(s)
            .unwrap()
            .gcd(&BigUint::from_str(t).unwrap());
        assert_eq!(n.to_string(), out);

        let n = rug::Integer::from_str(s)
            .unwrap()
            .gcd(&rug::Integer::from_str(t).unwrap());
        assert_eq!(n.to_string(), out);
    };
    test("0", "0", "0");
    test("0", "6", "6");
    test("6", "0", "6");
    test("1", "6", "1");
    test("6", "1", "1");
    test("8", "12", "4");
    test("54", "24", "6");
    test("42", "56", "14");
    test("48", "18", "6");
    test("3", "5", "1");
    test("12", "60", "12");
    test("12", "90", "6");
    test("12345678987654321", "98765432123456789", "1");
    test("12345678987654321", "98765432123456827", "37");
}

#[test]
fn limbs_gcd_limb_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    unsigned_vec_unsigned_pair_gen_var_23::<Limb, Limb>().test_properties_with_config(
        &config,
        |(xs, y)| {
            let gcd = Natural::from(limbs_gcd_limb(&xs, y));
            let n = Natural::from_owned_limbs_asc(xs);
            assert_eq!((&n).gcd(Natural::from(y)), gcd);
            assert_eq!(gcd_binary_nz(n, Natural::from(y)), gcd);
        },
    );
}

#[test]
fn half_gcd_matrix_init_properties() {
    unsigned_gen_var_11().test_properties_with_limit(TINY_LIMIT, |n| {
        let scratch_len = HalfGcdMatrix::min_init_scratch(n);
        assert_eq!(
            half_gcd_matrix_to_naturals(&OwnedHalfGcdMatrix::init(n, vec![0; scratch_len])),
            (Natural::ONE, Natural::ZERO, Natural::ZERO, Natural::ONE)
        );
    });
}

#[test]
fn half_gcd_matrix_update_q_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    large_type_gen_var_5().test_properties_with_config(&config, |(mut m, qs, column)| {
        let (x00, x01, x10, x11) = half_gcd_matrix_to_naturals(&m);
        let mut scratch = vec![0; m.update_q_scratch_len(qs.len())];
        m.update_q(&qs, column, &mut scratch);
        let q = Natural::from_owned_limbs_asc(qs);
        let (y00, y01, y10, y11) = half_gcd_matrix_to_naturals(&m);
        if column == 0 {
            assert_eq!(y01, x01);
            assert_eq!(y11, x11);
            assert_eq!(y00, &x00 + x01 * &q);
            assert_eq!(y10, &x10 + x11 * &q);
        } else {
            assert_eq!(y00, x00);
            assert_eq!(y10, x10);
            assert_eq!(y01, &x01 + x00 * &q);
            assert_eq!(y11, &x11 + x10 * &q);
        }
    });
}

#[test]
fn half_gcd_matrix_mul_matrix_1_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    large_type_gen_var_7().test_properties_with_config(&config, |(mut m, m_1)| {
        let (x00, x01, x10, x11) = half_gcd_matrix_to_naturals(&m);
        let (y00, y01, y10, y11) = half_gcd_matrix_1_to_naturals(&m_1);
        let mut scratch = vec![0; m.n];
        m.mul_matrix_1(&m_1, &mut scratch);
        let (z00, z01, z10, z11) = half_gcd_matrix_to_naturals(&m);
        assert_eq!(z00, &x00 * &y00 + &x01 * &y10);
        assert_eq!(z01, x00 * &y01 + x01 * &y11);
        assert_eq!(z10, &x10 * y00 + &x11 * y10);
        assert_eq!(z11, x10 * y01 + x11 * y11);
    });
}

#[test]
fn half_gcd_matrix_1_mul_vector_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    large_type_gen_var_6().test_properties_with_config(&config, |(m, mut out, xs, mut ys)| {
        let old_out = out.to_vec();
        let old_ys = ys.to_vec();
        let n = xs.len();
        let a = Natural::from_limbs_asc(&xs);
        let b = Natural::from_limbs_asc(&ys[..n]);
        let out_len = limbs_half_gcd_matrix_1_mul_vector(&m, &mut out, &xs, &mut ys);
        let bits = (u64::exact_from(n) + 1) << Limb::LOG_WIDTH;
        assert_eq!(
            Natural::from_limbs_asc(&out[..out_len]),
            (&a * Natural::from(m.data[0][0]) + &b * Natural::from(m.data[1][0]))
                .mod_power_of_2(bits)
        );
        assert_eq!(
            Natural::from_limbs_asc(&ys[..out_len]),
            (a * Natural::from(m.data[0][1]) + b * Natural::from(m.data[1][1]))
                .mod_power_of_2(bits)
        );
        if out.len() > out_len {
            assert_eq!(&out[out_len + 1..], &old_out[out_len + 1..]);
        }
        if ys.len() > out_len {
            assert_eq!(&ys[out_len + 1..], &old_ys[out_len + 1..]);
        }
    });
}

#[test]
fn limbs_matrix_2_2_mul_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    large_type_gen_var_8().test_properties_with_config(
        &config,
        |(mut xs00, mut xs01, mut xs10, mut xs11, xs_len, ys00, ys01, ys10, ys11)| {
            let x00 = Natural::from_limbs_asc(&xs00[..xs_len]);
            let x01 = Natural::from_limbs_asc(&xs01[..xs_len]);
            let x10 = Natural::from_limbs_asc(&xs10[..xs_len]);
            let x11 = Natural::from_limbs_asc(&xs11[..xs_len]);
            let mut scratch = vec![0; limbs_matrix_mul_2_2_scratch_len(xs_len, ys00.len())];
            limbs_matrix_2_2_mul(
                &mut xs00,
                &mut xs01,
                &mut xs10,
                &mut xs11,
                xs_len,
                &ys00,
                &ys01,
                &ys10,
                &ys11,
                &mut scratch,
            );
            let y00 = Natural::from_owned_limbs_asc(ys00);
            let y01 = Natural::from_owned_limbs_asc(ys01);
            let y10 = Natural::from_owned_limbs_asc(ys10);
            let y11 = Natural::from_owned_limbs_asc(ys11);
            let z00 = Natural::from_owned_limbs_asc(xs00);
            let z01 = Natural::from_owned_limbs_asc(xs01);
            let z10 = Natural::from_owned_limbs_asc(xs10);
            let z11 = Natural::from_owned_limbs_asc(xs11);
            assert_eq!(z00, &x00 * &y00 + &x01 * &y10);
            assert_eq!(z01, x00 * &y01 + x01 * &y11);
            assert_eq!(z10, &x10 * y00 + &x11 * y10);
            assert_eq!(z11, x10 * y01 + x11 * y11);
        },
    );
}

#[test]
fn limbs_gcd_div_properties() {
    unsigned_quadruple_gen_var_11::<Limb>().test_properties(|(n1, n0, d1, d0)| {
        let result = limbs_gcd_div_naive(n1, n0, d1, d0);
        assert_eq!(limbs_gcd_div(n1, n0, d1, d0), result);
        assert_eq!(limbs_gcd_div_alt(n1, n0, d1, d0), result);
    });
}

#[test]
fn limbs_gcd_reduced_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 256);
    config.insert("mean_stripe_n", 64 << Limb::LOG_WIDTH);
    unsigned_vec_pair_gen_var_10().test_properties_with_config(&config, |(mut xs, mut ys)| {
        let mut out = vec![0; xs.len()];
        let x = Natural::from_limbs_asc(&xs);
        let y = Natural::from_limbs_asc(&ys);
        let out_len = limbs_gcd_reduced(&mut out, &mut xs, &mut ys);
        out.resize(out_len, 0);
        let g = Natural::from_owned_limbs_asc(out);
        assert_eq!(gcd_binary_nz(x.clone(), y.clone()), g);
        assert_eq!(x.gcd(y), g);
    });
}

fn gcd_properties_helper(x: Natural, y: Natural) {
    let gcd_val_val = x.clone().gcd(y.clone());
    let gcd_val_ref = x.clone().gcd(&y);
    let gcd_ref_val = (&x).gcd(y.clone());
    let gcd = (&x).gcd(&y);
    assert!(gcd_val_val.is_valid());
    assert!(gcd_val_ref.is_valid());
    assert!(gcd_ref_val.is_valid());
    assert!(gcd.is_valid());
    assert_eq!(gcd_val_val, gcd);
    assert_eq!(gcd_val_ref, gcd);
    assert_eq!(gcd_ref_val, gcd);

    let mut mut_x = x.clone();
    mut_x.gcd_assign(y.clone());
    assert!(mut_x.is_valid());
    assert_eq!(mut_x, gcd);

    let mut mut_x = x.clone();
    mut_x.gcd_assign(&y);
    assert_eq!(mut_x, gcd);
    assert!(mut_x.is_valid());

    assert_eq!(
        Natural::from(&(BigUint::from(&x).gcd(&BigUint::from(&y)))),
        gcd
    );
    assert_eq!(
        Natural::exact_from(&(rug::Integer::from(&x).gcd(&rug::Integer::from(&y)))),
        gcd
    );

    assert_eq!(gcd_euclidean_nz(x.clone(), y.clone()), gcd);
    assert_eq!(gcd_binary_nz(x.clone(), y.clone()), gcd);

    assert_eq!((&y).gcd(&x), gcd);
    assert!((&x).divisible_by(&gcd));
    assert!((&y).divisible_by(&gcd));
    assert_eq!(gcd == 0, x == 0 && y == 0);
    if gcd != 0 {
        assert!(((&x).div_exact(&gcd)).coprime_with((&y).div_exact(&gcd)));
    }
    if x != 0 && y != 0 {
        assert_eq!(&x * &y / x.lcm(y), gcd);
    }
}

#[test]
fn gcd_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_bits_n", 2048);
    config.insert("mean_stripe_n", 512 << Limb::LOG_WIDTH);
    natural_pair_gen().test_properties_with_config(&config, |(x, y)| {
        gcd_properties_helper(x, y);
    });

    natural_pair_gen_var_4().test_properties_with_config(&config, |(x, y)| {
        gcd_properties_helper(x, y);
    });

    natural_gen().test_properties(|x| {
        assert_eq!((&x).gcd(&x), x);
        assert_eq!((&x).gcd(Natural::ONE), 1);
        assert_eq!((&x).gcd(Natural::ZERO), x);
    });

    natural_triple_gen().test_properties(|(x, y, z)| {
        assert_eq!((&x).gcd(&y).gcd(&z), x.gcd(y.gcd(z)));
    });

    unsigned_pair_gen_var_27::<Limb>().test_properties(|(x, y)| {
        assert_eq!(Natural::from(x).gcd(Natural::from(y)), x.gcd(y));
    });
}
