// Copyright © 2025 Mikhail Hogrefe
//
// This file is part of Malachite.
//
// Malachite is free software: you can redistribute it and/or modify it under the terms of the GNU
// Lesser General Public License (LGPL) as published by the Free Software Foundation; either version
// 3 of the License, or (at your option) any later version. See <https://www.gnu.org/licenses/>.

use malachite_base::num::arithmetic::traits::DivisibleBy;
use malachite_base::num::basic::integers::PrimitiveInt;
use malachite_base::num::basic::traits::{One, Zero};
use malachite_base::slices::slice_test_zero;
use malachite_base::test_util::generators::common::GenConfig;
use malachite_base::test_util::generators::{
    unsigned_pair_gen_var_27, unsigned_vec_pair_gen_var_15, unsigned_vec_pair_gen_var_16,
    unsigned_vec_unsigned_pair_gen_var_22,
};
use malachite_nz::natural::Natural;
use malachite_nz::natural::arithmetic::divisible_by::{
    limbs_divisible_by, limbs_divisible_by_limb, limbs_divisible_by_ref_ref,
    limbs_divisible_by_ref_val, limbs_divisible_by_val_ref,
};
use malachite_nz::natural::arithmetic::mod_op::{limbs_mod, limbs_mod_limb};
use malachite_nz::platform::{DoubleLimb, Limb};
use malachite_nz::test_util::generators::{
    natural_gen, natural_gen_var_2, natural_pair_gen, natural_pair_gen_var_6,
    natural_pair_gen_var_7, unsigned_vec_pair_gen_var_17,
};
use malachite_nz::test_util::natural::arithmetic::divisible_by::{
    combined_limbs_divisible_by_limb, num_divisible_by,
};
use num::BigUint;
use std::str::FromStr;

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_divisible_by_limb() {
    let test = |ns: &[Limb], d: Limb, divisible: bool| {
        assert_eq!(limbs_divisible_by_limb(ns, d), divisible);
        assert_eq!(limbs_mod_limb::<DoubleLimb, Limb>(ns, d) == 0, divisible);
        assert_eq!(combined_limbs_divisible_by_limb(ns, d), divisible);
    };
    test(&[0, 0], 2, true);
    test(&[6, 7], 1, true);
    test(&[6, 7], 2, true);
    test(&[100, 101, 102], 10, false);
    test(&[123, 456], 789, false);
    test(&[369, 1368], 3, true);
    test(&[u32::MAX, u32::MAX], 2, false);
    test(&[u32::MAX, u32::MAX], 3, true);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_divisible_by_limb_fail() {
    limbs_divisible_by_limb(&[10], 10);
}

fn verify_limbs_divisible_by(ns: &[Limb], ds: &[Limb], divisible: bool) {
    assert_eq!(
        Natural::from_limbs_asc(ns).divisible_by(Natural::from_limbs_asc(ds)),
        divisible
    );
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_divisible_by() {
    let test = |ns: &[Limb], ds: &[Limb], divisible: bool| {
        let mut mut_ns = ns.to_vec();
        let mut mut_ds = ds.to_vec();
        assert_eq!(limbs_divisible_by(&mut mut_ns, &mut mut_ds), divisible);

        let mut mut_ns = ns.to_vec();
        assert_eq!(limbs_divisible_by_val_ref(&mut mut_ns, ds), divisible);

        let mut mut_ds = ds.to_vec();
        assert_eq!(limbs_divisible_by_ref_val(ns, &mut mut_ds), divisible);

        assert_eq!(limbs_divisible_by_ref_ref(ns, ds), divisible);

        verify_limbs_divisible_by(ns, ds, divisible);
    };
    // - n_low & d_mask == 0
    // - d_len == 1
    // - d_len == 1 && n_len < BMOD_1_TO_MOD_1_THRESHOLD
    test(&[6], &[2], true);
    // - n_low & d_mask != 0
    test(&[6], &[4], false);
    // - d_len != 1
    // - d_len != 2
    // - trailing_zeros == 0
    // - rs[n_len - 1] >= ds[d_len - 1]
    // - schoolbook
    test(&[1, 3, 6, 5, 3], &[1; 3], true);
    test(&[1, 3, 7, 5, 3], &[1; 3], false);
    test(&[6, 19, 32, 21], &[1, 2, 3], true);
    // - d_len == 2
    // - d_1 > d_mask
    // - trailing_zeros != 0
    test(&[6, 19, 32, 21], &[6, 7], true);
    test(&[10200, 20402, 30605, 20402, 10200], &[100, 101, 102], true);
    test(
        &[10200, 20402, 30606, 20402, 10200],
        &[100, 101, 102],
        false,
    );
    test(&[u32::MAX], &[u32::MAX], true);
    test(&[1, u32::MAX - 1], &[u32::MAX], true);
    test(
        &[1, 0, 0, u32::MAX - 1, u32::MAX, u32::MAX],
        &[u32::MAX; 3],
        true,
    );
    test(
        &[1, 0, 0, 0xfffffffd, u32::MAX, u32::MAX],
        &[u32::MAX; 3],
        false,
    );
    test(&[0, 0, 0, 6, 19, 32, 21], &[0, 0, 1, 2, 3], true);
    // - rs[n_len - 1] < ds[d_len - 1] && n_len == d_len
    test(&[0, 1], &[1, 2], false);
    // - rs[n_len - 1] < ds[d_len - 1] && n_len != d_len
    test(&[0, 0, 1], &[1, 2], false);
    // - d_1 <= d_mask
    // - d_len == 2 && n_len < BMOD_1_TO_MOD_1_THRESHOLD
    test(&[0, 1], &[2, 1], false);
    // - d_len == 1 && n_len >= BMOD_1_TO_MOD_1_THRESHOLD
    test(
        &[
            4217434641, 2983831158, 957997048, 1656831590, 3000005030, 3397493916, 1524177764,
            1158142389, 874701225, 1462813081, 1367619360, 2801303018, 2789627609, 215719367,
            3216170452, 431698475, 626354765, 1948186092, 2374814145, 3144956155, 601105061,
            1178232240, 2555160313, 2595360911, 4288604684, 4266583859, 125722653, 201817397,
            664583805, 1753913919, 2144999578, 3776810790, 1371327248, 4182636558, 3002281720,
            1871734359, 2973117455, 2090705384, 824744451, 4231117477, 1910500161, 31831336,
            2931836758, 3786849205, 2959137635, 2746351664, 4074106964, 2451006035, 2637785805,
            1672606871, 2076991740, 4073897939, 923576835, 1556160088, 3713754032, 2220129449,
            1665613419, 2890565429, 2007962628, 3653583281, 3958263098, 2634902570, 503308768,
            2860598088, 4267080858, 486522024, 1950214271, 2459522693, 3179388659, 1827961427,
            848836381, 2558276851, 4179490611, 3898139084,
        ],
        &[2987492981],
        false,
    );
    // - d_len == 2 && n_len >= BMOD_1_TO_MOD_1_THRESHOLD
    test(
        &[
            256, 2983831158, 957997048, 1656831590, 3000005030, 3397493916, 1524177764, 1158142389,
            874701225, 1462813081, 1367619360, 2801303018, 2789627609, 215719367, 3216170452,
            431698475, 626354765, 1948186092, 2374814145, 3144956155, 601105061, 1178232240,
            2555160313, 2595360911, 4288604684, 4266583859, 125722653, 201817397, 664583805,
            1753913919, 2144999578, 3776810790, 1371327248, 4182636558, 3002281720, 1871734359,
            2973117455, 2090705384, 824744451, 4231117477, 1910500161, 31831336, 2931836758,
            3786849205, 2959137635, 2746351664, 4074106964, 2451006035, 2637785805, 1672606871,
            2076991740, 4073897939, 923576835, 1556160088, 3713754032, 2220129449, 1665613419,
            2890565429, 2007962628, 3653583281, 3958263098, 2634902570, 503308768, 2860598088,
            4267080858, 486522024, 1950214271, 2459522693, 3179388659, 1827961427, 848836381,
            2558276851, 4179490611, 3898139084,
        ],
        &[128, 10],
        false,
    );
    // - divide-and-conquer
    test(
        &[
            2169104186, 2266546494, 587315734, 3739269241, 3114885960, 3686571368, 1230083221,
            1369517966, 4291553725, 1232833631, 3745253511, 963470524, 3412093994, 715605915,
            3178840986, 1537155511, 2643999117, 1124945145, 1130467184, 505174687, 279956991,
            2029663376, 3099875141, 4078932665, 3507974056, 3764533985, 2928504341, 2999119478,
            2235181988, 8120813, 2049630814, 816255631, 2958134171, 2134573459, 1356483654,
            1353858604, 1377057347, 3050515126, 1981544721, 3471014354, 2173072144, 1505712122,
            2385668585, 1593490077, 92053253, 1525861712, 1903853398, 2590603784, 545440811,
            731662535, 2914628805, 1579744586, 2159061521, 3862444433, 3486035778, 3569159590,
            1798578513, 679586822, 1027034006, 2821087869, 138364116, 3862434540, 3700050828,
            3041238135, 203907970, 2724479836, 666267271, 4276247071, 1210965278, 3212904426,
            3198623953, 2953530971, 254157796, 4263106694, 389863197, 928000244, 489453825,
            3398590663, 3892546275, 132887015, 982972816, 1812112076, 3203181556, 729377309,
            2468997822, 4114200925, 3495511499, 2423156091, 642381751, 2588236007, 508136034,
            2909297716, 2020665371, 2823339182, 4093944876, 1753900312, 2227288358, 3843178420,
            1383018274, 1337754144, 4067699321, 4094839667, 3612619766, 241995072, 43874530,
            1329365494, 1161788309, 1343981322, 2661807206, 3916970745, 3208129678, 2607521742,
            4068076440, 2254858029, 52800437, 2977865502, 2752070351, 840472004, 993872687,
            118561354, 281929094, 241294936, 894356352, 289721033, 658212066, 2618650207,
            997997486, 2912241249, 828533444, 1694023493, 368405691, 2952571418, 1897728140,
            3178382549, 433809308, 1079641707, 913972412, 127906934, 1696809357, 4135380244,
            862031081, 3680249566, 4137097186, 3602972472, 3335499618, 2982048038, 2435876884,
            4160199134, 3332420996, 1469463193, 3949621168, 4019057802, 3431928811, 241891262,
            4190148730, 991409256, 487806424, 685244570, 703993726, 4044357137, 3327796046,
            1784380542, 1000763217, 966313870, 3075333333, 1734085949, 913022609, 4245248443,
            3252643922, 806393395, 2476394527, 3747108106, 392268399, 2776259136, 2995460881,
            3323006399, 2205717476, 3139072617, 3312440544, 2390989321, 3330711647, 280322670,
            3221287591, 3672189452, 2085991953, 2701899535, 944470536, 3459091473, 669958019,
            3341466114, 1375663389, 4178951502, 1887416017, 2960114949, 1805082649, 1897118491,
            1735206477, 1086467373, 3454663842, 3027811502, 2681987258, 3932510644, 1041830424,
            3552076035, 3424876359, 1886621954, 2650649785, 3695516428, 71584725, 168707053,
            740198485, 2245105417, 1249177483, 1722102768, 374173663, 297078497, 2990861606,
            925762985, 2001492, 1646443443, 3583906616, 3312776133, 1157309588, 1335281328,
            2109828349, 34076132, 2168747053, 3778129093, 2613456818, 1560157805, 4195434489,
            2487913229, 1609822712, 2764757271, 4283118804, 91257607, 1080744325, 2493394532,
            326260208, 187395392, 2939059459, 1885652903, 4191985814, 3258074631, 3513150040,
            3580927149, 3407053353, 226470876, 1817235991, 852868000, 214844744, 4207902570,
            2902398660, 1397161252, 4138041363, 757796679, 3031247970, 1048623352, 661015139,
            1415029830, 2485923562, 141373475, 1787794282, 1976681319, 1253461810, 1392275594,
            2429887226, 2877605657, 1263266041, 3484472193, 2640875272, 1801160237, 4069713126,
            1934086126, 192869060, 3283904145, 2763023257, 989311613, 4108856559, 4226641951,
            3128021950, 2529670899, 2772914524, 1201442981, 176669049, 3707600374, 3706141284,
            276672151, 2904754467, 1347188130, 769251759, 3322969014, 4037207402, 89044259,
            1314731683, 1220332916, 438537136, 827090059, 272783156, 38244839, 2305761897,
            3723044685, 644855780, 3551390010, 3080020204, 3622120389, 1104906535, 2333407502,
            1345296288, 3700728088, 1599981349, 3670637655, 3168618533, 1373988250, 2598638204,
            2174612205, 945499273, 1516940873, 1689708547, 3883475244, 1794555293, 969139846,
            319304823, 197055825, 2598587648, 3355480329, 2619765319, 1140474610, 1254318132,
            547674601, 852635669, 2448264593, 1227857672, 2436426747, 1094311339, 1981571015,
            648000603, 4136039667, 1172894412, 2484651184, 1368629311, 1617781838, 1466927120,
            110324911, 650355804, 3280790286, 242161684, 3602009741, 1994089400, 3324228020,
            1722483735, 3385991773, 1733478840, 1986423929, 3118968494, 4214422449, 4257358954,
            3002156517, 3342276360, 3961751029, 1044818187, 1302652353, 3780349809, 208159762,
            3921381160, 3104805287, 701555864, 1353151101, 2110331602, 1065775293, 3138173139,
            2673335412, 1332663987, 2532658640, 1175704936, 3679818302, 2587893347, 1209649581,
            2160090389, 330309148, 812388071, 1856530929, 1621848118, 3595859350, 372670625,
            856076462, 1710799422, 2493451671, 142603177, 1359238753, 3676968997, 3302712549,
            2781924367, 822719137, 1553432401, 2179118972, 2647958472, 4231975017, 3662950204,
            1513113991, 3143506626, 4261352924, 2900864288, 2505227245, 1285289303, 3374032515,
            1343085626, 1298169727, 2712611178, 2519581207, 863950575, 1044705613, 1608844529,
            398557619, 2405597933, 2442595621, 3756322386, 497719075, 958070332, 710843530,
            1361414502, 3604632029, 3231349850, 3610106540, 3750520155, 2250988708, 2584496493,
            3558352960, 847973690, 291227960, 3347729096, 718140050, 2842802593, 45528740,
            1213748814, 823269856, 1820806986, 2274362480, 3357823158, 2054374162, 3650723151,
            1083860767, 3888008023, 2654794365, 1844177589, 2412317894, 665263029, 68335398,
            1937681761, 1826086466, 2497132629, 2053584711, 766478134, 3524809898, 2670148828,
            2393556691, 3637792476, 3212231999, 1742932892, 2227017682, 1415769555, 2646519675,
            4134401864, 4082068803, 1489292682, 1791908612, 1508761188, 2209720169, 4078191455,
            1533239903, 1768397400, 2157145812, 2565607675, 519979525, 4034426142, 1802535614,
            3593889411, 3552848103, 4200389446, 3707485453, 1542526970, 1196075302, 2665590567,
            2739194250, 1399657538, 113759040, 3183789024, 386168896, 3295775459, 1126370167,
            3951610340, 3551153197, 1690373840, 2358923787, 4229534809, 2290337908, 3142294436,
            6187128, 1890727395, 4243456854, 3017228946, 3877724493, 3444342781, 4211602385,
            183761949, 1350782665, 592876667, 433343586, 1817557528, 261342788, 2125735308,
            3252124670, 1044819933, 2708341279, 100284000, 1935238212, 3692459811, 383621158,
            867374712, 1021196147, 530727196, 3915886219, 3705268858, 2857571834, 39817906,
            728954116, 3793147370, 406405776, 3215234366, 2390137796, 1950992336, 1570696989,
            2517131965, 1753011612, 3117489998, 1817087642, 4076432149, 212175604, 4209324605,
            249521560, 23902299, 4101107861, 620892789, 2870257694, 2409554819, 4089101702,
            752496922, 362566394, 3459326044, 2058596304, 2035287414, 1288805897, 3459829946,
            3477025798, 2693442504, 606291269, 3716454206, 2017617901, 739955186, 226702924,
            1506656210, 564432407, 698436937, 254384295, 187952024, 3541301121, 4130651539,
            2092000582, 1462681196, 2024159384, 3025822813, 4206822061, 3305239609, 2483534937,
            1068789172, 3171775935, 4213371381, 2105143064, 1715930888, 1869129579, 1721290855,
            726914406, 1055443769, 4127573087, 1131510133, 3464915074, 3962097334, 4198659475,
            449084358, 1255330246, 2217278306, 3142203879, 1060366183, 1086505252, 2196640428,
            3020011420, 101613015, 3615286211, 3719078597, 512776700, 1670970979, 2918688390,
            539186647, 3899361016, 3899238233, 1503014623, 1487072013, 2554631966, 640480738,
            2906553626, 1264192892, 2306406723, 4030568138, 4056096494, 3608358777, 3240979341,
            2671626653, 3715996998, 983432302, 712348732, 2406877846, 32785806, 3076643355,
            2166512728, 1664977926, 379356343, 2822820700, 2920574885, 1252499465, 3427197294,
            1702237632, 1641655300, 310170158, 1001329555, 562240182, 3713127974, 1400603861,
            4291073476, 485459242, 1624227405, 3436522834, 1213896273, 893370150, 3608415729,
            89540222, 1549103010, 2362944877, 636789508, 2413471028, 2932597865, 3523220925,
            3113130274, 669409867, 948785250, 2353176063, 3868467803, 3484917154, 2551915221,
            1497486585, 4152651850, 473143127, 998729612, 2905656591, 1182540548, 3623426979,
            1806548948, 598537433, 2345218067, 2744900190, 3658378705, 1964774718, 2557442350,
            1799636549, 4153665976, 3300594666, 682645932, 3787913729, 2627374889, 2780368898,
            3053617139, 2751701327, 1952207418, 761193466, 795600024, 460949087, 2883354289,
            2416166804, 2024094265, 2645942993, 1321335098, 4080983696, 3719057466, 2761861073,
            876455883, 2265708436, 1295460078, 4221100394, 327768739, 3015153526, 631331115,
            3428464785, 644806658, 1607685691, 2709965965, 2477974591, 4185293236, 3564199615,
            5024334, 1218905327, 2381178079, 3421492726, 3210172738, 2208193787, 3497539969,
            67679408, 7015329, 1644345354, 1219650877, 3999432182, 787420821, 3125125301,
            3551310228, 1514288276, 4184725579, 3923600454, 342747221, 2581849732, 654180323,
            2328957051, 1611256108, 3331006737, 788693120, 3401041379, 2099076468, 1628705595,
            3859087292, 2072660933, 3257911942, 984362662, 3077084901, 1851298543, 1203131946,
            1027287045, 3054998432, 2653266373, 3019736717, 195024883, 4171454841, 2527872129,
            1247020327, 2238591475, 2136657438, 2340932338, 2934529639, 3272613638, 2185810440,
            165859626, 2304003399, 3993415600, 2437076830, 3780217527, 2707007352, 208843797,
            3155075128, 336707716, 568567552, 2653218265, 561879893, 2244946665, 369107514,
            449372466, 3941906096, 4130057272, 641659453, 4111061561, 812270662, 4279338122,
            470975590, 3494637046, 2762073253, 1757357680, 867114230, 2219045455, 94626870,
            1137750480, 2094083377, 3138049052, 3894576321, 1378390851, 2665729658, 3104615470,
            2936479206, 635788017, 3475293009, 702959182, 4237902922, 313651363, 3029245314,
            2741552996, 3659722846, 2604607686, 534266423, 2704761681, 1711186117, 1592045306,
            305708572, 1642121404, 1597934575, 1265263680, 4290920109, 537137446, 3140075852,
            2898671495, 364470582, 3399686556, 427464562, 1727273367, 4033988735, 2229384209,
            977641014, 3910691764, 2268933684, 3451806563, 204040124, 1396741907, 3978224431,
            2183874291, 541967491, 3026397898, 2065096561, 961793538, 3683498094, 3088514164,
            3743738285, 1346128225, 3719735346, 3228089802, 3009395215, 1568158796, 1870781853,
            728601499, 2727218958, 50783576, 351811718, 1118170454, 2810777608, 1996734855,
            2444030542, 3509123551, 1649933419, 2057107104, 381073891, 1651296068, 3691277418,
            1252885223, 604972451, 3560912853, 3381041205, 97236104, 4018969688, 3285100340,
            906433788, 1674945334, 2291802158, 4143691201, 1121520461, 1907818021, 3817648523,
            2696920177, 2029578019, 3359867876, 4035314026, 814831395, 402590457, 2757255435,
            1617756566, 2573316001, 952926090, 26184857, 1491250719, 479641193, 1246115200,
            2071561116, 1740343273, 3930947184, 849382562, 3146753986, 2079815766, 1124558195,
            706118855, 1624669925, 2345125117, 1104509365, 4253600455, 2178123171, 51311084,
            644240166, 4235897745, 1552583610, 2721864023, 2417820136, 866003296, 3100038290,
            1138399889, 2621877527, 35166564, 2925990087, 676411279, 824163386, 3059725138,
            3436858126, 639114670, 2354917892, 3668885469, 538607516, 1188540552, 2077947642,
            4214121777, 1024066855, 584585760, 3330067560, 3289211050, 4227743647, 1048503204,
            3682815131, 3593095813, 724046659, 177523551, 789173931, 3210826214, 4114794229,
            2001369528, 1922224607, 2449629560, 2990923760, 1590140407, 3066241294, 2032178641,
            1673025735, 728782818, 2805995869, 3173476316, 4095043043, 4188792079, 961420687,
            78547373, 331719917, 3623238477, 4199725886, 2474752488, 1757124968, 837328916,
            1408306379, 3214330220, 3355378613, 389392287, 2212359467, 2671491064, 2411527388,
            3622122999, 1506404106, 3192630979, 135884006, 2398972643, 3221225957, 827475224,
            2670120145, 3556393877, 1151874066, 4001558171, 773853920, 261365390, 2930272664,
            2171104793, 3679192861, 1286480701, 3052752341, 3704064736, 2857905507, 323617567,
            500166971, 3590167140, 46687505, 3603631510, 574954751, 1921803608, 2826015895,
            3138948179, 3471354167, 1087583334, 2872377970, 589736930, 3600161138, 4280276795,
            610467300, 4258855269, 1847560302, 89888737, 3455621079, 1271345555, 446001635,
            1740292174, 1118130455, 1649510985, 1314419877, 2732937708, 749289315, 911962385,
            361189198, 1736019975, 901913856, 1697911811, 512137942, 1863970636, 2127135390,
            2362086418, 1086237732, 509394495, 2529713980, 229310079, 1661114917, 1240908759,
            1028403199, 2183953926, 1127564377, 3529642168, 557447511, 314233677, 1785885521,
            948058680, 3314696870, 3418731379, 1093740990, 1408897230, 1674741555, 2040518897,
            500968889, 2496717966, 1599286253, 927417612, 846393794, 2100310692, 654673974,
            2475673804, 3656332830, 4291477953, 3810417562, 1914048410, 1890554346, 1707870195,
            3321510012, 1070941267, 1096480276, 2946865379, 3751928683, 1860758395, 3419660556,
            4012128602, 517029378, 1439950265, 502253691, 2697781940, 2999363300, 2555052305,
            852916430, 3054976477, 3742735044, 2485908580, 1698982608, 2625028734, 1580539136,
            3300904616, 633144061, 886348308, 111258399, 1439126181, 2276519459, 3959771808,
            1314171199, 2322947590, 706448164, 998967466, 3140589807, 3516861163, 2443112013,
            2783111182, 792015165, 1578481086, 3460528207, 1455852169, 3304443593, 123860242,
            1860448346, 2199267391, 3894670775, 3728263116, 530899313, 3431936766, 497193688,
            3964977529, 2359257224, 1413751341, 953616340, 2193659597, 2870937129, 2560871048,
            3424144210, 2330060314, 476593761, 4111398811, 224492168, 2426790462, 1366265513,
            3547692500, 237014373, 126994210, 1698344909, 3097624535, 2606367774, 2139247575,
            4278291258, 3111057613, 3597265089, 1486559051, 3888932297, 2068681921, 4018333295,
            1237578874, 1394533778, 3019039915, 2435030995, 361267457, 2911469665, 109581803,
            4017370036, 3519506811, 1390346590, 4165412046, 1729882859, 2014283168, 1289243407,
            2679198626, 49392930, 921068681, 903443552, 3928862909, 3914934492, 1656987052,
            432711693, 1092080705, 1009507400, 3442035405, 2796451416, 178836372, 1341085119,
            472247744, 1804872218, 209397624, 3871538735, 2625371000, 275406901, 1757188449,
            2669657607, 3604284276, 1912410848, 4250918838, 1504400908, 2246112278, 2284856099,
            1017170859, 106995822, 2930347834, 545564822, 1065545923, 3484558996, 3235687826,
            1118147092, 564820628, 3004798730, 856304451, 822860516, 3255412373, 398693215,
            2329594629, 3107774102, 3795269757, 1827602725, 2395215194, 3203911581, 1655534638,
            211158317, 3318323836, 1658306711, 3266378120, 967971461, 3700196933, 4059454123,
            641302150, 3758966844, 3400068366, 4198115937, 485338847, 4078693738, 2194750103,
            1916610491, 1608221201, 592771197, 305961735, 3496309244, 458632142, 1436284187,
            1883874902, 362906529, 38314525, 1950785222, 3918473285, 4280158367, 239076663,
            2870128186, 1858894036, 1027494104, 3282242213, 4129496568, 441027295, 2379941678,
            2193414438, 2212796476, 2277788540, 3450917896, 1884715287, 4063547227, 1121537361,
            1032757188, 272161392, 1784943104, 1790206106, 713995873, 1769042326, 535564457,
            178454254, 618864620, 1597799935, 3270467251, 1842808299, 1018709091, 3361794236,
            4221130656, 1039251424, 1054506074, 2666864849, 3685051761, 2437756290, 4019119161,
            832674533, 2148950955, 2940298749, 1756399996, 1105163896, 1271960636, 3581940733,
            4143342134, 3823790830, 3488841246, 3005908556, 1229249490, 2249360558, 1520953855,
            2394117874, 2966743152, 1745813079, 3874896381, 4191350629, 3459390903, 458890909,
            3016227910, 2882279471, 1489028867, 3561423182, 3768844180, 3613799407, 1316785634,
            1110714379, 3825900063, 3005543401, 2578216023, 536834091, 1054735610, 2490400103,
            3824100200, 1622334988, 2973431210, 3511057535, 3411074810, 2839293801, 3615424236,
            1494356926, 2903750858, 2317164586, 798613884, 2208508524, 1120667415, 459634045,
            208258335, 2160960212, 3770793497, 1682162327, 2891358722, 4253581389, 1506921959,
            1520649651, 1728239848, 347197154, 1030568701, 231999862, 87543989, 4039870891,
            3025487032, 4160936744, 824286621, 619876628, 3974227642, 2732935886, 3410258822,
            2406057333, 1165743060, 940866829, 3430451491, 1309520185, 262533243, 947988428,
            640974029, 2310827524, 1568049953, 913937604, 523285756, 4019938271, 997714649,
            2289745882, 2914301529, 2284035199, 1811643961, 1014663687, 2060371185, 3157474020,
            348590874, 2826896064, 1227085771, 547468599, 862186803, 1711960313, 1600420494,
            1479186385, 3931860334, 602002980, 2799154778, 1091430947, 3212154168, 1149219740,
            3667760717, 1338808890, 2090747344, 3732991295, 242914573, 2349118025, 1074929252,
            1578439132, 486000813, 3540915699, 313415647, 1456544360, 2413478967, 1922572049,
            3983685868, 1692333399, 2800078402, 3672468944, 941785200, 2799315558, 3495998609,
            54827483, 3471966058, 1350080114, 1185538001, 1146752796, 2383192228, 816416261,
            1367006476, 824307946, 1275718301, 313125425, 4174575084, 2115741142, 2899939598,
            1437334171, 4036006477, 3402338404, 1622118790, 1903918451, 2195630834, 2817175072,
            1580143842, 10841607, 2218444240, 328726045, 2172285181, 666198999, 47210791,
            217474374, 3425261191, 3792177351, 1102768641, 194832930, 2798628092, 1099023742,
            2169630244, 1006780406, 4040150832, 3495364601, 1795130764, 2557506394, 687172551,
            1437043653, 596022564, 4124841760, 2832726960, 3976626928, 1940701362, 2129400780,
            1335265202, 2623243469, 475218973, 4176790636, 2894028481, 3311380020, 3407731969,
            3440303872, 4282973538, 1107580287, 2913854988, 3832450081, 2983552995, 3234935353,
            1905487397, 1438250438, 199108879, 1165802593, 3979735320, 292406483, 3789152918,
            1728782338, 1538302953, 3095891341, 4067011302, 947270848, 2901984827, 472807909,
            1124867728, 967972503, 1179911480, 905593964, 3081206343, 1396858745, 1962485374,
            660034736, 2323793427, 135073240, 1624297728, 1102603846, 2597005485, 2882403403,
            890183758, 3191082568, 1919982661, 3168083235, 1958438352, 1304175725, 2908775316,
            1173988864, 294889039, 4007902026, 1120126077, 1310854594, 546116031, 2848470459,
            1869698025, 1813333748, 2963438773, 3933908194, 825386399, 2267683410, 2388979811,
            1581641113, 433859520, 3307533954, 3715973826, 3034129662, 1356470162, 2053300467,
            3705599280, 3545214348, 881711657, 3370463532, 2612058211, 3022216886, 166079300,
            3792395371, 771494606, 367422082, 908499242, 1751065177, 3739297223, 2531239191,
            1790567517, 3657254622, 1969671979, 1753491710, 4166832315, 3390847739, 2694228751,
            3731820124, 1556938927, 832291283, 2595430255, 3509265058, 1963314910, 743669390,
            2316110695, 3553152608, 1539708075, 2581957193, 3581782561, 3733834639, 3874869903,
            821452300, 2654378399, 1826247143, 427372641, 2528090922, 3136006832, 716079731,
            3725286491, 3774681077, 2364200620, 520438267, 936152988, 1300229995, 3318827623,
            893385987, 827395790, 1238576928, 1365658130, 4213726033, 68568077, 1878508859,
            2980664309, 2505293566, 647443774, 893999753, 576745315, 3786437080, 408297414,
            4253424139, 56347062, 1531788504, 4279766238, 897659733, 588650771, 2281920358,
            2662121367, 1247856184, 4077376735, 3094366352, 3475315562, 140900543, 2320176873,
            2731282493, 2529550865, 2646680520, 3977230880, 610342446, 1738659355, 2322564480,
            1977463984, 3253404512, 3381161114, 4072345980, 3740051219, 2680106281, 3143028156,
            3832831863, 1270452360, 127171546, 1648356616, 2300166574, 636625275, 1177009521,
            846853350, 1040691569, 2730938915, 71966795, 3559433301, 466808619, 1872187366,
            3586382562, 358150119, 1901267937, 1893851852, 3145952136, 3674931561, 1233478860,
            40233330, 3052854867, 2874428838, 1759762421, 3681648875, 2015813552, 886942698,
            1158156557, 1493555589, 1504676074, 3260453531, 1445680526, 606176540, 3851133425,
            3378052885, 290847531, 608022018, 409268527, 2561487059, 552122872, 3296447559,
            657047951, 1631287800, 3753000079, 4193809942, 884351757, 650107527, 2377568881,
            3347052935, 2286485244, 3723839791, 2320149490, 2621979946, 3594571203, 1034530050,
            1013146660, 4171674723, 701670817, 2422914046, 3977354486, 3004818531, 2002938131,
            3118226044, 1543549044, 4092997654, 3942653186, 1844763496, 2322984163, 3124918738,
            1417239715, 235247237, 1050357214, 2534723786, 4219669748, 4011595633, 2692726487,
            2351983087, 2725167432, 1625523175, 4047392192, 2234302489, 2849441834, 3972066233,
            1791829507, 939116825, 1749334517, 2363503506, 1794949600, 3554843296, 3658516450,
            3792519561, 2851405786, 612179385, 3421837507, 1838122126, 1526446643, 473977778,
            970578022, 4289507606, 3575021665, 4054714975, 1548076097, 3509765025, 1066969507,
            3153376781, 3298819216, 3889250694, 69259617, 1666832803, 210278288, 3699475297,
            3835947114, 3339052234, 612647103, 875803198, 3239057673, 3476910228, 1774715092,
            2997268264, 1885163212, 2535759355, 4037082439, 4131425791, 1431931269, 2344841786,
            589132544, 3925395746, 784033909, 2001561327, 2875535943, 3682912780, 1098615571,
            801194772, 2827770013, 1354890928, 2166517733, 939353793, 872778539, 3631178723,
            1264903893, 4095947484, 4181434456, 4200174517, 1474870556, 1627027654, 151292710,
            368473093, 3942717951, 819142369, 2349507002, 4074749712, 2489107860, 1097307038,
            3020812166, 190306378, 3215127968, 397275036, 2249230863, 443420014, 464013638,
            2524035741, 2741857795, 2086019319, 1734459117, 3393166271, 3474161219, 2289897976,
            4189280756, 901946899, 1802590992, 2188289127, 2913766496, 1889227592, 3897013662,
            1464728618, 2940520918, 2906043873, 2148982467, 2404803587, 894173079, 2495095800,
            1534807391, 3036557983, 3313990361, 3092310990, 3745247632, 2003342030, 4001597438,
            170941187, 3898936460, 1767517480, 2465767943, 2725934507, 905298257, 2054030074,
            1040718227, 130491459, 462307151, 79634941, 39984710, 3394369759, 123808373,
            1976379290, 2760361748, 2906583982, 2953946087, 809471573, 2116054715, 3917916932,
            2343670492, 3233424199, 774812558, 1756949698, 3194486356, 946191489, 3049294096,
            206896940, 2241842324, 2913740804, 2608611645, 2252633590, 1782956542, 2558802876,
            189416278, 2914035148, 53575167, 657929910, 2197973107, 1186173886, 1652038937,
            1667328710, 3827180894, 1328118528, 627951742, 3852917235, 1261933427, 2796832443,
            3909139439, 803119467, 3434493749, 3852387669, 827924266, 4132690149, 261809517,
            898612437, 1360913566, 3963554976, 3291202592, 3510814304, 1941420827, 410368266,
            1914132178, 4062923120, 1846671715, 3606753265, 3301693690, 2422891416, 4171900211,
            1179609418, 3026009073, 1460697318, 81708763, 2241853979, 3184526697, 2433342232,
            3825378463, 1733696008, 1589648295, 130255373, 693921782, 1859354595, 1919721838,
            1186247508, 4108823444, 77734465, 2629607792, 1076377333, 1990488420, 2937396723,
            3004759683, 2318594408, 1150102008, 2213733556, 3220708392, 2308953052, 2749064027,
            2085137171, 1117546275, 1886156676, 98424960, 2074093393, 2154503198, 189860005,
            2916276379, 1952025216, 3574945482, 3027541861, 3394630765, 3511855818, 3454027741,
            2764910997, 3888451786, 1821639450, 208178455, 2500754493, 2116466467, 2562510724,
            2236268494, 3533596053, 1136585978, 1638867361, 2351171373, 1768471085, 1475147430,
            3007554507, 138204569, 355310868, 514355272, 3368854386, 4017757574, 2764261877,
            2324034334, 1307181094, 2699433145, 1799666796, 1658078944, 3051201857, 4064940189,
            2720978395, 1229071670, 571398165, 3644846781, 1901207895, 398691399, 2312366555,
            933751814, 2120985531, 2216142231, 3139218360, 1909928326, 4233628411, 1909827411,
            2995576052, 3808488656, 2242622915, 3933434554, 3788084924, 3334517362, 3266016991,
            1568666969, 1229301195, 880438760, 3568043601, 4141504174, 840071256, 426306577,
            296537409, 3733271638, 617460425, 4156981489, 1596412970, 2310467087, 3835423522,
            4164901542, 2204793257, 644780121, 1452829515, 2477069580, 516109421, 524261758,
            117982878, 1282333694, 2307680806, 3224487996, 1113925619, 2143297676, 3177316878,
            2381663400, 3278164422, 2125512960, 1035639931, 795806882, 3988921620, 3738216448,
            1187094494, 111058657, 1478788191, 1563430022, 3618637339, 247022523, 12616232,
        ],
        &[
            2598813973, 1904197782, 1215006787, 2930851058, 147918011, 1108782873, 2193711651,
            927233146, 4123817534, 3041706771, 2553964352, 2170557862, 2838589053, 2708835560,
            2407955329, 1653207324, 2470552742, 3357058396, 1072990020, 1514783799, 1734243804,
            1048851304, 3193592152, 1383793603, 2063694499, 3485786208, 990960024, 3569185429,
            1155708496, 1416333121, 623402323, 3585705881, 2928088588, 4117621841, 3292316347,
            2217241313, 1950216441, 1391040412, 2926386224, 3240560579, 3973468920, 2847202482,
            2728825575, 2388048350, 3463506691, 3707514927, 865702555, 681215612, 2250616248,
            1724513618, 364443494, 318451745, 1623887866, 1280032334, 35195442, 3837277484,
            3573894817, 1602810477, 2072874171, 3800895590, 4253419440, 3499691102, 3587932863,
            3517441378, 514856216, 1648457744, 1474005402, 700069688, 1803098596, 378514692,
            3068261379, 3834605983, 93188389, 516236732, 2468656032, 4095977127, 3706578522,
            3452719188, 1052956630, 813640425, 3047313546, 754584849, 3584789926, 347367947,
            1332536470, 94171148, 626270691, 665836088, 426974652, 749957497, 1566095325,
            3066600838, 4106843490, 318006871, 162882344, 3632111614, 720131390, 2147721707,
            2217002633, 2466031857, 1087787961, 753159680, 366300446, 3082468114, 3984408901,
            4057095635, 851983865, 4150513631, 1165194321, 2055798628, 2975996128, 2693369845,
            3726562839, 3482086133, 4069043610, 1448967233, 700979569, 1698695640, 2318799208,
            3684286306, 4092668018, 3628416641, 584456626, 1679287620, 1813093480, 177926224,
            998082621, 3120492909, 1996199168, 3491975384, 3147828383, 2087111449, 381360867,
            1127916110, 3472685226, 3293954452, 490729044, 3264678372, 1277191068, 4239255920,
            2343436004, 2451796708, 1844810428, 2031787164, 3768550378, 3743966167, 2637191163,
            3978079759, 4158883864, 1561267833, 539304561, 3435817790, 277956235, 4028949085,
            2643928548, 2755470882, 3796376394, 194519280, 98463778, 3179299980, 2447702226,
            150057260, 3463931601, 906456277, 990239469, 512339370, 2261507614, 2826877099,
            3855453279, 2271503686, 841719073, 1591142177, 2651714174, 3361609137, 3279250024,
            3326254458, 333896144, 3528172547, 3941409760, 1558812882, 3841874000, 3172301956,
            4257828380, 4114850067, 2182891616, 2046840523, 1555755083, 2209766728, 2332501890,
            607500373, 4124706810, 1713777506, 2341598661, 2284959418, 2746608689, 867377978,
            336903700, 2503448208, 701980251, 2377952035, 614377477, 4284955660, 3814775340,
            74129761, 1280559390, 2313692692, 1849232614, 2933185433, 1636271207, 630083065,
            433420841, 471362220, 4210050941, 406797277, 582028044, 528813510, 2576049278,
            540310007, 2789529550, 3534664845, 3236430259, 1506736021, 1977751182, 2006662671,
            3679126532, 3806292929, 322589328, 2243078958, 2178902647, 2189720772, 3791358339,
            1508304160, 1013509043, 3587377598, 3623382349, 1086144534, 1599008554, 1093927956,
            2876914186, 543927302, 2219667262, 3853928343, 2487266159, 2613372592, 3550686437,
            3716813778, 961453654, 2039051889, 4283100852, 3145660844, 1250225831, 198998794,
            3726057928, 2284062625, 2219120593, 2244278486, 3790941761, 4203700163, 3879487839,
            2965718505, 2863698764, 63056401, 475617471, 399914031, 459648012, 3133447831,
            3412742746, 1512219896, 402842103, 4119870394, 2231397283, 1516833787, 1104380754,
            128232714, 296865691, 1663298638, 716369560, 2904885147, 4237273250, 343946912,
            2604094031, 217784841, 1162604926, 1345215097, 1426555576, 3614102008, 3943750939,
            1639058309, 3296139198, 1330339563, 426593850, 1532081057, 1300177557, 1482433791,
            812987015, 3232993095, 1037853411, 645052957, 2928052694, 1226934622, 2336175954,
            1920836876, 2517428765, 3726338359, 4041660846, 2092081260, 3380258824, 1697518166,
            3349439761, 743024530, 1795639086, 1779094250, 503268222, 3554837208, 2986098469,
            2762697287, 232864157, 3768291070, 2983947152, 1577288457, 1050740564, 834028049,
            1274280907, 3769464765, 1910494593, 2197778112, 2813074316, 2881147492, 1559228213,
            1461309572, 1700587844, 2988103759, 1717431083, 951586291, 1707887130, 985013185,
            3532658879, 2597699341, 1911327175, 3080664111, 3484272917, 794091071, 2311407322,
            342113645, 1768703425, 2382063884, 3831832486, 3217975952, 3451120056, 3035392294,
            136312738, 1699608380, 700762772, 3631031559, 1846450824, 1131124755, 1046357323,
            2552828340, 2598371768, 795498120, 3174589062, 3838467339, 2334795665, 3324675939,
            3403942196, 2075108646, 4259667163, 545251542, 749294402, 3324014447, 1729518387,
            3551096145, 1758729590, 2379344936, 2455329109, 1757426268, 1706226190, 902425986,
            2432676044, 2039145952, 410402636, 1639537846, 1305863342, 256740017, 3207555620,
            3925040779, 668173470, 1141011911, 4212893794, 1581027147, 363670779, 2004465281,
            1138726957, 645471449, 3585049231, 2777598154, 2922780411, 2222055902, 3815639175,
            429697628, 4000303790, 3545345324, 3542109929, 509705074, 3722614775, 1753809005,
            1973665455, 3780943261, 2494678174, 2641931684, 3388813526, 3347201186, 619828405,
            2625911693, 3122659421, 3479474745, 3245356132, 1832479544, 3308783684, 1963028441,
            640786937, 1641844190, 211570630, 1733515651, 2171524601, 3396267685, 3914881000,
            996687218, 1617466047, 3343054134, 1737892381, 3544966831, 3416864496, 107959379,
            2249494870, 793869312, 3685840230, 1757287133, 1655716303, 4226516527, 2538565645,
            278467613, 2644886589, 2699974905, 1984329139, 1005957589, 789155264, 3666799939,
            1611583926, 809907316, 3452803056, 1107204019, 2209104411, 1462179962, 2853352142,
            3225410978, 2108213590, 4192481899, 1912371245, 1171525154, 3712460427, 2560727022,
            1086797266, 952026066, 685685805, 1936296543, 3499184244, 196202245, 3078070455,
            2171291372, 2730765878, 1573851700, 2931331934, 2847177501, 3478529515, 2313080814,
            1556316903, 1559216686, 3130885922, 1862119430, 3274978261, 206968409, 2624567354,
            1978029702, 4063454216, 2415324407, 1518621963, 747943965, 1111109939, 4070402813,
            3113587512, 898431887, 316373197, 3046639257, 173632841, 3798588025, 1794547053,
            272903937, 863380298, 2774736392, 2777149271, 3206794646, 32657715, 1025716581,
            1615425071, 3173842612, 2073903119, 2153824733, 2329799226, 2341967952, 473768936,
            3486256934, 2848218304, 2428724257, 3159637305, 2126501230, 765243196, 3425509759,
            1131263732, 2515711177, 3855554546, 1023477320, 787209967, 336492935, 2359162223,
            1128313353, 2209616384, 389804749, 535230248, 2210531774, 3227122449, 1267072928,
            675693308, 1385601874, 1416098547, 4192675028, 2716632450, 4177948782, 13560634,
            3703614488, 3901742123, 2270254077, 3759771486, 581228935, 3949774462, 3458012976,
            4163726158, 1602955605, 3047124020, 1612897320, 2437308530, 2370536309, 1561087937,
            2116794827, 1579558066, 431023695, 689842616, 2342172842, 4203138565, 2160272314,
            4192374041, 3675193896, 1455954294, 3940635994, 4025127357, 4027321239, 1155472117,
            1928138697, 1875048682, 4087308487, 4291614860, 1175537429, 3503698743, 2187471558,
            2673330725, 2702107217, 2763809959, 2274134903, 1077148334, 2166895545, 1831099412,
            3296218829, 2076626325, 1104984380, 1764164194, 3056784327, 1988083103, 3849746178,
            3460231757, 3983590507, 1641801163, 1321487487, 2277192156, 1700397327, 4163993314,
            118570737, 2510013157, 342313019, 328214482, 1456063300, 3771950632, 1226087686,
            2343733178, 1108934200, 3650307976, 118441436, 3857631518, 1808729865, 2826187782,
            2596972674, 1966611164, 988201436, 3909729734, 2034457184, 3137580128, 437202273,
            1431097330, 2630459633, 2423317302, 1422128850, 3011012632, 3611423092, 3531891566,
            1592039994, 734168534, 3492902008, 947200226, 1735249159, 3526990689, 1725556774,
            1693403463, 3317454666, 2042191731, 3608995702, 497650714, 628421679, 2465885637,
            4008542175, 68576338, 1372104957, 1927020046, 3573635655, 3090117252, 1774561696,
            3603424898, 55089895, 2060875413, 116850528, 183816573, 478639013, 3736973628,
            3606174188, 3086193394, 2548388100, 2591332019, 3032618195, 469584388, 1770142568,
            205800329, 174943950, 260521576, 2114953066, 3428041459, 860714084, 1560598053,
            2885505481, 3365355923, 2821218095, 814424502, 94258256, 2857874640, 3213633690,
            239445181, 1741311155, 70754202, 469749979, 469377271, 634558134, 1314018935,
            2663122712, 3425107874, 616125297, 1220520411, 363320242, 1717903421, 2177377494,
            2683338356, 210565640, 3061448891, 1561105018, 3639722552, 2586119577, 264509957,
            1416442444, 572448512, 3316197518, 4101471127, 1909662580, 3289492851, 702940172,
            1747588119, 3288795988, 1441233966, 678854386, 2604723913, 373703246, 2055937039,
            1124872113, 1294098600, 4165350921, 347872377, 4149215174, 3215450946, 3307233717,
            3600358822, 2659616884, 552889201, 1684681790, 1821948043, 2909781107, 917768879,
            2259947602, 2019396153, 3413299181, 603749894, 2467542574, 4017401393, 2519998741,
            1106606352, 2283516402, 4262893045, 931507531, 257221572, 1471379658, 4223521274,
            2958168654, 1643148068, 537294061, 3213657938, 3835022365, 3594404173, 777970857,
            1918635217, 1702205904, 1047564373, 67547546, 3019489110, 483595975, 4130252917,
            3081408781, 436773586, 1982914868, 3527757249, 1553562987, 4080227164, 822023069,
            1427193014, 2565421648, 3942884687, 586018307, 1138884595, 2613035769, 486310535,
            233458001, 106898418, 170926111, 3212999317, 3194995097, 827548055, 1783632131,
            1458445783, 4273893842, 4169283956, 3425898311, 1900133873, 2830934989, 3597188167,
            60014629, 2917912426, 3137954804, 1361341056, 4180775481, 946784052, 4199341441,
            976101959, 1662997785, 1179237245, 372119531, 2815699743, 2114995295, 3901740963,
            3080952004, 938243235, 1717858705, 1880470411, 3714648015, 2698621435, 4276694893,
            327245390, 2823511048, 1178264505, 2316760826, 1757942990, 1568677829, 1199580238,
            3100459677, 1386893348, 1600406349, 3128791813, 874040538, 2965907285, 8663319,
            2919597841, 1169807363, 2130724711, 2817929643, 2594603073, 975728895, 3248945863,
            1506505714, 8259384, 2395186489, 64139190, 254403781, 3179528072, 3318835559,
            872874473, 3221533853, 1278142055, 2865063695, 2096015856, 3893588672, 1650965167,
            852689473, 2456343381, 2227877463, 3533995396, 2567915364, 3260019114, 3354960043,
            317493973, 4088542285, 603804423, 3133721086, 2455680365, 2501599681, 2371144161,
            4224001882, 3900064951, 2295679614, 483545534, 500738921, 259183629, 2958675979,
            3886544014, 3382548497, 1341558649, 764676712, 2686742281, 3585367009, 2470017030,
            280649561, 504360831, 3177630390, 691028027, 3630869823, 1850039925, 693186712,
            2234334464, 3019783104, 1258253876, 2759186239, 2705660533, 1535024585, 16228098,
            3597194253, 3856258376, 2530800309, 1756061934, 3703461702, 1458672824, 3009579190,
            325542092, 974024523, 937809905, 180338019, 728257472, 1334683442, 1531729599,
            4257819380, 3701291761, 2423233504, 3786098144, 776062663, 3861731850, 3717496633,
            2918219359, 2369401162, 1963105504, 2673826226, 3238671093, 568083438, 4121947082,
            3892853652, 1632686679, 2302193583, 3211408318, 1938993158, 4122064270, 311269184,
            2105346675, 1581174365, 2485158976, 2443260070, 1986431168, 1124848188, 1841087451,
            2704521996, 1214274421, 3662035084, 2837679678, 4043497550, 552208260, 1644975825,
            1950150273, 4027848966, 4079004873, 647675052, 223752470, 637576072, 949880156,
            2749078102, 2043391316, 1008133, 253411434, 3799679907, 238227057, 2495985662,
            1475105846, 2961884606, 652180469, 1691635637, 1430944868, 15398684, 2278485165,
            3795142783, 2981896799, 1330890913, 2239691223, 407470744, 3702570774, 1624292191,
            3703366632, 2187395861, 2415878960, 2303069741, 1369122706, 2750340520, 4035673151,
            1425244662, 25186311, 123582970, 2490005251, 936607532, 4115073120, 2730391719,
            2823090688, 396451377, 2182998280, 1365726601, 4033770954, 3047868289, 2336389408,
            688454315, 1299725781, 3304644284, 4238176839, 3914620151, 3029466942, 3172492309,
            1190474536, 569450335, 320580314, 4004925139, 2557991799, 1493351038, 949583033,
            3275441062, 1311907795, 106222620, 396788468, 3186456628, 517037899, 2177653619,
            370023265, 532558109, 3182151426, 954367771, 2865591818, 600183719, 2521622949,
            3079560626, 1906789399, 1612708432, 1801660151, 3565921772, 2688352000, 1101243899,
            888006794, 2233649225, 2671500356, 1504445832, 1199168343, 2043022343, 1250919220,
            1374956710, 1107550604, 957179006, 3319703903, 4110453611, 3311143947, 2774029776,
            1456730624, 1014119593, 3012545941, 1650066434, 3431589287, 3624715486, 1072097915,
            3568695028, 478299405, 362822053, 2632030958, 955747917, 2974153651, 2880247593,
            2837857407, 1533402093, 1550331006, 3833005459, 111580345, 1694562098, 2731241196,
            2659074758, 1347615125, 3711588514, 1514710939, 3788474419, 3668780501, 3266213391,
            4081085519, 1196081228, 1049947080, 1820720234, 3100382458, 2198468097, 3708538645,
            1035729054, 1784655621, 507574130, 2908640623, 3615200865, 1223546227, 3675568182,
            1661330896, 1399689373, 1323389692, 2981922901, 4156332999, 1902241850, 1428680890,
            3049572611, 4253261231, 2148214199, 1998921071, 3899443086, 1478235253, 3791927172,
            1483896530, 3565485437, 915921912, 2258155251, 3426051958, 3307670857, 3029363090,
            3673788309, 4082838999, 624603732, 573907442, 3116664857, 3586999255, 557493850,
            1739716128, 2397076308, 2312498391, 767640384, 2119706819, 2637314051, 1070866648,
            664107404, 616236779, 3020748367, 4122128780, 2009706384, 3245363034, 429196441,
            4195520565, 4209198410, 2158211364, 3659713923, 3985561806, 1747634790, 4115935838,
            2827667115, 3073689925, 1621229363, 2774950774, 2538390550, 3506768465, 718261358,
            564823729, 380006788, 897186389, 3414352487, 1032216622, 1469922191, 2196317056,
            2239269473, 3654751282, 613015420, 4098167098, 1281134924, 849099531, 2523299715,
            3904840545, 1997547696, 1158713208, 1289444182, 757815917, 899581875, 3622748533,
            3063686656, 2979790776, 1553609204, 355408616, 3897614956, 3680334470, 25191644,
            355655308, 111682127, 4053312775, 2921335050, 2200671456, 472639194, 1374183459,
            2738441274, 1596829615, 959522853, 1993763998, 1832301144, 2263515976, 993190648,
            4045308024, 2213735364, 4061773476, 2596920312, 1929252967, 3935609312, 982934883,
            3154772065, 4254233108, 1244999206, 765237459, 2528318166, 1563389556, 2293797399,
            3495759008, 48504250, 3182637462, 3138309477, 2147232621, 4810076, 3999234780,
            2279064264, 2264522982, 3079929466, 1524222187, 1598294545, 2049273389, 2729549735,
            2116985451, 2637892533, 1094214870, 2711414546, 418295062, 98357556, 296649589,
            2636445209, 4030461397, 3073508440, 4062358649, 1293566861, 1534697481, 1343281639,
            3078259374, 386875427, 1145986442, 1790533181, 2987076845, 237202537, 2194958623,
            935119723, 1921262864, 943957815, 2336181341, 2723699978, 1638860845, 2630349960,
            431757538, 4152375793, 3473513084, 2603606861, 4176464622, 3978369596, 981482293,
            929741839, 2111696471, 571254437, 1419237733, 901791748, 3340393111, 221970895,
            2605197016, 22565479, 1000841034, 2192399670, 1207300847, 356824298, 3009617459,
            2651652098, 2420275852, 1631602976, 575809994, 2928806259, 2831092106, 2629452482,
            2842897182, 1909556662, 2496151886, 3385330607, 957152776, 2804092840, 3551246686,
            2730894896, 2027253611, 2531182020, 2258110944, 1391264743, 3902230517, 3835546712,
            3424753892, 1147410244, 3448482025, 319341118, 1798380053, 2742193162, 2106010955,
            3267048655, 3886107336, 3334600393, 3772435523, 3985863328, 2950255062, 1012493134,
            3954831363, 325018235, 2523597002, 2454925521, 2861802043, 4209313262, 3868234463,
            1685691511, 2786322255, 4244960817, 1991229419, 4239301712, 576152977, 1652411241,
            1589394460, 521549473, 266587858, 961791319, 1146831103, 3469211532, 3501779101,
            2312218475, 2321025269, 1346459216, 472196028, 3771324053, 1495430908, 463405457,
            1940232183, 3937148918, 1156878514, 2831556660, 1644483662, 1787817713, 2339616130,
            3815974485, 2298018558, 71674109, 2675299410, 960845405, 3569711116, 1673611570,
            1769369631, 1437416980, 4174063039, 2062712084, 2133844948, 196794643, 1935768058,
            1831573341, 1418980930, 3294487604, 3544538744, 2334730191, 3988003806, 2274350901,
            2023027498, 3447859831, 1671781771, 4023665101, 1455029189, 2419433622, 2943475311,
            4198568768, 1395089836, 1092520718, 2430238564, 437551709, 4097009390, 3375079636,
            4154198223, 92070792, 974607811, 3873908092, 3374473679, 638725898, 1249915990,
            1085664087, 3927733894, 1217984426, 1825247311, 2107887775, 170602951, 1075337232,
            1842870533, 3647067870, 2444592832, 3013106942, 1161476947, 3743427689, 1263482478,
            69206560, 833594813, 3842989721, 337909675, 3392484922, 370017251, 27594915,
            2126835633, 1736323108, 1893331695, 1468455680, 1075562062, 455344215, 969605531,
            294398875, 883453327, 3994410499, 1964955910, 1983678849, 556499140, 2199399388,
            1506753515, 2562511167, 573514453, 1452183180, 3758124304, 2276736219, 3962356320,
            160602272, 2732947819, 1312555719, 808517807, 1646633687, 4057383018, 1177836167,
            2724613618, 724351202, 3082498407, 3292565482, 2024078169, 2339554384, 3963173875,
            2223760730, 2562066565, 2193608401, 1812209945, 1665831972, 1039920028, 3410644888,
            3314911306, 2847575459, 2005962667, 2392982038, 2769384144, 1673231309, 2605033025,
            3097232789, 525491643, 1088675733, 389231889, 4053722369, 1394160319, 920345783,
            1695188025, 1313259843, 2855284945, 1128781302, 1711293636, 2436253183, 2702553100,
            2342544037, 2382554808, 1645824427, 700888148, 3820471891, 2062002896, 1600256482,
            3523617235, 3825142862, 4016136295, 3838255962, 2953276340, 394106186, 956179026,
            3512260850, 3001113638, 1645204518, 555542490, 409962126, 479780262, 4202384502,
            549252742, 1340551826, 190095025, 2842856010, 446626748, 2565781573, 2172050474,
            85422525, 151000070, 2413238086, 224534468, 4102988065, 3524737259, 784974768,
            3778660388, 2573449273, 956921977, 1416256526, 3821996527, 1297167173, 2183926071,
            2745200336, 1745460864, 399929970, 3271657494, 1058524185, 3138265217, 3822962039,
            796872545, 555781465, 3355863387, 4265560398, 2445094472, 546611744, 958064176,
            2278895404, 2100057835, 789578969, 387055963, 1153298400, 4065547056, 3688020091,
            783181103, 1789734962, 1466117990, 1740574947, 1632230912, 4155788342, 1714025679,
            3862124914, 2247239755, 3047057899, 3610673443, 1450313039, 3533121399, 1564221183,
            4056883174, 319992191, 598075724, 2438292429, 901314271, 2157273664, 435275402,
            1774735757, 177332560, 3010394026, 3827445200, 2002098740, 1897812156, 454287770,
            2829500771, 3286056227, 1498747083, 3402858410, 58581019, 1048766033, 1640106396,
            3774636176, 1228719792, 1768725616, 3094025701, 3705595233, 2943515149,
        ],
        false,
    );
    // - Barrett
    test(&[8; 70000], &[3; 60000], false);
    test(&[9; 120000], &[3; 60000], true);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_divisible_by_fail_1() {
    limbs_divisible_by(&mut [1, 2], &mut [3, 4, 5]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_divisible_by_fail_2() {
    limbs_divisible_by(&mut [], &mut []);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_divisible_by_fail_3() {
    limbs_divisible_by(&mut [1, 2, 0], &mut [4, 5]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_divisible_by_fail_4() {
    limbs_divisible_by(&mut [1, 2, 3], &mut [4, 0]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_divisible_by_val_ref_fail_1() {
    limbs_divisible_by_val_ref(&mut [1, 2], &[3, 4, 5]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_divisible_by_val_ref_fail_2() {
    limbs_divisible_by_val_ref(&mut [], &[]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_divisible_by_val_ref_fail_3() {
    limbs_divisible_by_val_ref(&mut [1, 2, 0], &[4, 5]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_divisible_by_val_ref_fail_4() {
    limbs_divisible_by_val_ref(&mut [1, 2, 3], &[4, 0]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_divisible_by_ref_val_fail_1() {
    limbs_divisible_by_ref_val(&[1, 2], &mut [3, 4, 5]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_divisible_by_ref_val_fail_2() {
    limbs_divisible_by_ref_val(&[], &mut []);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_divisible_by_ref_val_fail_3() {
    limbs_divisible_by_ref_val(&[1, 2, 0], &mut [4, 5]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_divisible_by_ref_val_fail_4() {
    limbs_divisible_by_ref_val(&[1, 2, 3], &mut [4, 0]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_divisible_by_ref_ref_fail_1() {
    limbs_divisible_by_ref_ref(&[1, 2], &[3, 4, 5]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_divisible_by_ref_ref_fail_2() {
    limbs_divisible_by_ref_ref(&[], &[]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_divisible_by_ref_ref_fail_3() {
    limbs_divisible_by_ref_ref(&[1, 2, 0], &[4, 5]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_divisible_by_ref_ref_fail_4() {
    limbs_divisible_by_ref_ref(&[1, 2, 3], &[4, 0]);
}

#[test]
fn test_divisible_by() {
    let test = |s, t, divisible| {
        let u = Natural::from_str(s).unwrap();
        let v = Natural::from_str(t).unwrap();

        assert_eq!(u.clone().divisible_by(v.clone()), divisible);
        assert_eq!(u.clone().divisible_by(&v), divisible);
        assert_eq!((&u).divisible_by(v.clone()), divisible);
        assert_eq!((&u).divisible_by(&v), divisible);

        assert_eq!(u == 0 || v != 0 && &u % &v == 0, divisible);

        assert_eq!(
            num_divisible_by(
                &BigUint::from_str(s).unwrap(),
                &BigUint::from_str(t).unwrap()
            ),
            divisible
        );
        assert_eq!(
            rug::Integer::from_str(s)
                .unwrap()
                .is_divisible(&rug::Integer::from_str(t).unwrap()),
            divisible
        );
    };
    test("0", "0", true);
    test("1", "0", false);
    test("1000000000000", "0", false);
    test("0", "1", true);
    test("0", "123", true);
    test("1", "1", true);
    test("123", "1", true);
    test("123", "123", true);
    test("123", "456", false);
    test("456", "123", false);
    test("369", "123", true);
    test("4294967295", "1", true);
    test("4294967295", "4294967295", true);
    test("1000000000000", "1", true);
    test("1000000000000", "3", false);
    test("1000000000002", "3", true);
    test("1000000000000", "123", false);
    test("1000000000000", "4294967295", false);
    test("1000000000000000000000000", "1", true);
    test("1000000000000000000000000", "3", false);
    test("1000000000000000000000002", "3", true);
    test("1000000000000000000000000", "123", false);
    test("1000000000000000000000000", "4294967295", false);
    test("1000000000000000000000000", "1000000000000", true);
    test("1000000000000000000000000", "1000000000001", false);
    test(
        "851673906388325341550957943071111911557800036845129556099360938813259608650267079456739978\
        1156959952275409185911771336067392377245918291754269000751186715279414560474882570499082990\
        4913122978897463970860833616251189242098804876664368441608727895141238953179204529256780277\
        5978105200286025161944212712977056343127682601975191673217459602567633602198262568921008081\
        9448556670912575287371251190800855926311768876808375177446530243635212748346921654224589861\
        0625170426812525829689862407515510419445335472631905610235915226032848323874067128872385291\
        3730739275467227364692195226129501338887049710586931141309357190341064532366013123280106098\
        6468151628797945455179649866890394481799639832540978091736379482964522229064478167730317490\
        8194108506704480750395054067032502530392147690725919399930683143510771646869931527123340650\
        0547649792331568913460415939722111305270588701531404490040034302102101083691706550376288655\
        2667382899390792494118931379237432071316543313379792218794371176529684614085109418328963817\
        0601432767270419229719490809539776535671938041618536196941370647945336401901450921413823163\
        4059991707077834107830876756821880651429748186401020760113859498185638133726165286481741014\
        9079906337286599226335508424466369316294442004040440528589582239717042654541745348050157252\
        3448224036804997350851153108395928780441635856",
        "147502279655636565600250358452694051893980186696958535174009956523855720107322638159749368\
        0808217479494744305876890972595771484769733857514529616096199394092858302265998260483416016\
        5763904522044264005938281072568140883513713255548643044250086110483617215935636533809248102\
        6926590789142079805638445494760177551776636747830014495012489743990407355232286842071418922\
        9921358409573480901624487977319782755422730834468673438076805532952821406024399006814390166\
        6949827530796971086011267864607814906313334525518102221919643040440267323688341889035864376\
        1377246644579088153222669672271414315240318439843720039808993886410874969340996645010795670\
        2133518716987668865936529827437388042190084309005369564717390726257594902619365180097509576\
        6240189037770619308206906414128686856349950952623970023039440323701643457411485666776354448\
        186307133288106956593939073729500658176632828099789",
        true,
    );
    test(
        "851673906388325341550957943071111911557800036845129556099360938813259608650267079456739978\
        1156959952275409185911771336067392377245918291754269000751186715279414560474882570499082990\
        4913122978897463970860833616251189242098804876664368441608727895141238953179204529256780277\
        5978105200286025161944212712977056343127682601975191673217459602567633602198262568921008081\
        9448556670912575287371251190800855926311768876808375177446530243635212748346921654224589861\
        0625170426812525829689862407515510419445335472631905610235915226032848323874067128872385291\
        3730739275467227364692195226129501338887049710586931141309357190341064532366013123280106098\
        6468151628797945455179649866890394481799639832540978091736379482964522229064478167730317490\
        8194108506704480750395054067032502530392147690725919399930683143510771646869931527123340650\
        0547649792331568913460415939722111305270588701531404490040034302102101083691706550376288655\
        2667382899390792494118931379237432071316543313379792218794371176529684614085109418328963817\
        0601432767270419229719490809539776535671938041618536196941370647945336401901450921413823163\
        4059991707077834107830876756821880651429748186401020760113859498185638133726165286481741014\
        9079906337286599226335508424466369316294442004040440528589582239717042654541745348050157252\
        3448224036804997350851153108395928780441635856",
        "147502279655636565600250358452694051893980186696958535174009956523855720107322638159749368\
        0808217479494744305876890972595771484769733857514529616096199394092858302265998260483416016\
        5763904522044264005938281072568140883513713255548643044250086110483617215935636533809248102\
        6926590789142079805638445494760177551776636747830014495012489743990407355232286842071418922\
        9921358409573480901624487977319782755422730834468673438076805532952821406024399006814390166\
        6949827530796971086011267864607814906313334525518102221919643040440267323688341889035864376\
        1377246644579088153222669672271414315240318439843720039808993886410874969340996645010795670\
        2133518716987668865936529827437388042190084309005369564717390726257594902619365180097509576\
        6240189037770619308206906414128686856349950952623970023039440323701643457411485666776354448\
        186307133288106956593939073729500658176632828099788",
        false,
    );
    test("1", "94815577610368829905234938913859", false);
}

#[test]
fn limbs_divisible_by_limb_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    unsigned_vec_unsigned_pair_gen_var_22().test_properties_with_config(&config, |(xs, y)| {
        let divisible = limbs_divisible_by_limb(&xs, y);
        assert_eq!(
            (&Natural::from_limbs_asc(&xs)).divisible_by(Natural::from(y)),
            divisible
        );
        assert_eq!(limbs_mod_limb::<DoubleLimb, Limb>(&xs, y) == 0, divisible);
        assert_eq!(combined_limbs_divisible_by_limb(&xs, y), divisible);
    });
}

#[test]
fn limbs_divisible_by_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 512);
    config.insert("mean_stripe_n", 64 << Limb::LOG_WIDTH);
    unsigned_vec_pair_gen_var_15().test_properties_with_config(&config, |(mut ns, mut ds)| {
        let ns_old = ns.clone();
        let ds_old = ds.clone();
        let divisible = limbs_divisible_by(&mut ns, &mut ds);

        let mut ns = ns_old.clone();
        assert_eq!(limbs_divisible_by_val_ref(&mut ns, &ds_old), divisible);

        let mut ds = ds_old.clone();
        assert_eq!(limbs_divisible_by_ref_val(&ns_old, &mut ds), divisible);

        assert_eq!(limbs_divisible_by_ref_ref(&ns_old, &ds_old), divisible);

        verify_limbs_divisible_by(&ns_old, &ds_old, divisible);
    });

    unsigned_vec_pair_gen_var_17().test_properties_with_config(&config, |(mut ns, mut ds)| {
        let ns_old = ns.clone();
        let ds_old = ds.clone();
        assert!(limbs_divisible_by(&mut ns, &mut ds));

        let mut ns = ns_old.clone();
        assert!(limbs_divisible_by_val_ref(&mut ns, &ds_old));

        let mut ds = ds_old.clone();
        assert!(limbs_divisible_by_ref_val(&ns_old, &mut ds));

        assert!(limbs_divisible_by_ref_ref(&ns_old, &ds_old));
        verify_limbs_divisible_by(&ns_old, &ds_old, true);
    });

    unsigned_vec_pair_gen_var_16().test_properties_with_config(&config, |(ns, ds)| {
        let divisible = limbs_divisible_by_ref_ref(&ns, &ds);
        assert_eq!(slice_test_zero(&limbs_mod(&ns, &ds)), divisible);
    });
}

#[test]
fn divisible_by_properties() {
    natural_pair_gen().test_properties(|(x, y)| {
        let divisible = (&x).divisible_by(&y);
        assert_eq!((&x).divisible_by(y.clone()), divisible);
        assert_eq!(x.clone().divisible_by(&y), divisible);
        assert_eq!(x.clone().divisible_by(y.clone()), divisible);

        assert_eq!(x == 0 || y != 0 && &x % &y == 0, divisible);
        assert_eq!(
            num_divisible_by(&BigUint::from(&x), &BigUint::from(&y)),
            divisible
        );
        assert_eq!(
            rug::Integer::from(&x).is_divisible(&rug::Integer::from(&y)),
            divisible
        );
    });

    natural_pair_gen_var_6().test_properties(|(x, y)| {
        assert!((&x).divisible_by(&y));
        assert!(x == 0 || y != 0 && &x % &y == 0);
        assert!(num_divisible_by(&BigUint::from(&x), &BigUint::from(&y)));
        assert!(rug::Integer::from(&x).is_divisible(&rug::Integer::from(&y)));
    });

    natural_pair_gen_var_7().test_properties(|(x, y)| {
        assert!(!(&x).divisible_by(&y));
        assert!(x != 0 && (y == 0 || &x % &y != 0));
        assert!(!num_divisible_by(&BigUint::from(&x), &BigUint::from(&y)));
        assert!(!rug::Integer::from(&x).is_divisible(&rug::Integer::from(&y)));
    });

    natural_gen().test_properties(|n| {
        assert!(n.divisible_by(Natural::ONE));
    });

    natural_gen_var_2().test_properties(|n| {
        assert!(!(&n).divisible_by(Natural::ZERO));
        assert!(Natural::ZERO.divisible_by(&n));
        if n > 1 {
            assert!(!Natural::ONE.divisible_by(&n));
        }
        assert!((&n).divisible_by(&n));
    });

    unsigned_pair_gen_var_27::<Limb>().test_properties(|(x, y)| {
        assert_eq!(
            Natural::from(x).divisible_by(Natural::from(y)),
            x.divisible_by(y)
        );
    });
}
