// Copyright © 2025 Mikhail Hogrefe
//
// This file is part of Malachite.
//
// Malachite is free software: you can redistribute it and/or modify it under the terms of the GNU
// Lesser General Public License (LGPL) as published by the Free Software Foundation; either version
// 3 of the License, or (at your option) any later version. See <https://www.gnu.org/licenses/>.

use malachite_base::num::arithmetic::traits::DivMod;
use malachite_base::num::arithmetic::traits::{
    DivExact, DivExactAssign, DivRound, EqModPowerOf2, ModPowerOf2, ModPowerOf2Neg,
};
use malachite_base::num::basic::integers::PrimitiveInt;
use malachite_base::num::basic::traits::{One, Zero};
use malachite_base::num::conversion::traits::{ExactFrom, WrappingFrom};
use malachite_base::rounding_modes::RoundingMode::*;
use malachite_base::test_util::common::rle_decode;
use malachite_base::test_util::generators::common::GenConfig;
use malachite_base::test_util::generators::{
    unsigned_gen_var_22, unsigned_pair_gen_var_11, unsigned_vec_pair_gen_var_12,
};
use malachite_nz::integer::Integer;
use malachite_nz::natural::Natural;
use malachite_nz::natural::arithmetic::div_exact::{
    limbs_div_exact, limbs_div_exact_3, limbs_div_exact_3_in_place, limbs_div_exact_3_to_out,
    limbs_div_exact_limb, limbs_div_exact_limb_in_place,
    limbs_div_exact_limb_in_place_no_special_3, limbs_div_exact_limb_no_special_3,
    limbs_div_exact_limb_to_out, limbs_div_exact_limb_to_out_no_special_3, limbs_div_exact_to_out,
    limbs_div_exact_to_out_ref_ref, limbs_div_exact_to_out_ref_val, limbs_div_exact_to_out_val_ref,
    limbs_modular_div, limbs_modular_div_barrett, limbs_modular_div_barrett_scratch_len,
    limbs_modular_div_divide_and_conquer, limbs_modular_div_mod_barrett,
    limbs_modular_div_mod_barrett_scratch_len, limbs_modular_div_mod_divide_and_conquer,
    limbs_modular_div_mod_schoolbook, limbs_modular_div_ref, limbs_modular_div_ref_scratch_len,
    limbs_modular_div_schoolbook, limbs_modular_div_schoolbook_in_place,
    limbs_modular_div_scratch_len, limbs_modular_invert, limbs_modular_invert_limb,
    limbs_modular_invert_scratch_len, test_invert_limb_table,
};
use malachite_nz::platform::{DoubleLimb, Limb};
use malachite_nz::test_util::generators::{
    large_type_gen_var_13, large_type_gen_var_14, large_type_gen_var_15, large_type_gen_var_16,
    natural_gen, natural_gen_var_2, natural_pair_gen_var_6, unsigned_vec_gen_var_5,
    unsigned_vec_pair_gen_var_13, unsigned_vec_quadruple_gen_var_2, unsigned_vec_triple_gen_var_46,
    unsigned_vec_triple_gen_var_47, unsigned_vec_triple_gen_var_48,
    unsigned_vec_unsigned_pair_gen_var_29, unsigned_vec_unsigned_vec_unsigned_triple_gen_var_14,
};
use malachite_nz::test_util::natural::arithmetic::div_exact::{
    limbs_div_exact_3_in_place_alt, limbs_div_exact_3_to_out_alt,
};
use std::str::FromStr;

#[test]
fn test_test_invert_limb_table() {
    test_invert_limb_table();
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_modular_invert_limb() {
    let test = |x, inverse| {
        assert_eq!(limbs_modular_invert_limb(x), inverse);
    };
    test(1, 1);
    test(3, 2863311531);
    test(5, 3435973837);
    test(7, 3067833783);
    test(123, 3945782963);
    test(1000000001, 2211001857);
    test(0x7fffffff, 0x7fffffff);
    test(2863311531, 3);
    test(u32::MAX, u32::MAX);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_invert_limb_fail_1() {
    limbs_modular_invert_limb::<Limb>(0);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_invert_limb_fail_2() {
    limbs_modular_invert_limb::<Limb>(2);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_div_exact_limb_and_limbs_div_exact_limb_in_place() {
    let test = |ns: &[Limb], d: Limb, q: &[Limb]| {
        let ns_old = ns;
        assert_eq!(limbs_div_exact_limb(ns_old, d), q);

        let mut ns = ns_old.to_vec();
        limbs_div_exact_limb_in_place(&mut ns, d);
        assert_eq!(ns, q);

        assert_eq!(limbs_div_exact_limb_no_special_3(ns_old, d), q);

        let mut ns = ns_old.to_vec();
        limbs_div_exact_limb_in_place_no_special_3(&mut ns, d);
        assert_eq!(ns, q);
    };
    test(&[0], 2, &[0]);
    test(&[6], 2, &[3]);
    test(&[0, 0], 2, &[0, 0]);
    test(&[6, 7], 1, &[6, 7]);
    test(&[6, 7], 2, &[0x80000003, 3]);
    test(&[92, 101, 102], 10, &[1288490198, 858993469, 10]);
    test(&[4294966783, 455], 789, &[2482262467, 0]);
    test(&[u32::MAX - 1, u32::MAX], 2, &[u32::MAX, 0x7fffffff]);
    test(&[u32::MAX, u32::MAX], 3, &[0x55555555, 0x55555555]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_limb_fail_1() {
    limbs_div_exact_limb(&[], 10);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_limb_fail_2() {
    limbs_div_exact_limb(&[10, 10], 0);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_limb_in_place_fail_1() {
    limbs_div_exact_limb_in_place(&mut [], 10);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_limb_in_place_fail_2() {
    limbs_div_exact_limb_in_place(&mut [10, 10], 0);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_div_exact_limb_to_out() {
    let test = |out_before: &[Limb], ns: &[Limb], d: Limb, out_after: &[Limb]| {
        let mut out = out_before.to_vec();
        limbs_div_exact_limb_to_out::<DoubleLimb, Limb>(&mut out, ns, d);
        assert_eq!(out, out_after);

        let mut out = out_before.to_vec();
        limbs_div_exact_limb_to_out_no_special_3::<DoubleLimb, Limb>(&mut out, ns, d);
        assert_eq!(out, out_after);
    };
    test(&[10, 10, 10, 10], &[0], 2, &[0, 10, 10, 10]);
    test(&[10, 10, 10, 10], &[6], 2, &[3, 10, 10, 10]);
    test(&[10, 10, 10, 10], &[0, 0], 2, &[0, 0, 10, 10]);
    test(&[10, 10, 10, 10], &[6, 7], 1, &[6, 7, 10, 10]);
    test(&[10, 10, 10, 10], &[6, 7], 2, &[0x80000003, 3, 10, 10]);
    test(
        &[10, 10, 10, 10],
        &[92, 101, 102],
        10,
        &[1288490198, 858993469, 10, 10],
    );
    test(
        &[10, 10, 10, 10],
        &[4294966783, 455],
        789,
        &[2482262467, 0, 10, 10],
    );
    test(
        &[10, 10, 10, 10],
        &[u32::MAX - 1, u32::MAX],
        2,
        &[u32::MAX, 0x7fffffff, 10, 10],
    );
    test(
        &[10, 10, 10, 10],
        &[u32::MAX, u32::MAX],
        3,
        &[0x55555555, 0x55555555, 10, 10],
    );
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_limb_to_out_fail_1() {
    limbs_div_exact_limb_to_out::<DoubleLimb, Limb>(&mut [10, 10], &[], 10);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_limb_to_out_fail_2() {
    limbs_div_exact_limb_to_out::<DoubleLimb, Limb>(&mut [10, 10], &[10, 10], 0);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_limb_to_out_fail_3() {
    limbs_div_exact_limb_to_out::<DoubleLimb, Limb>(&mut [10], &[10, 10], 10);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_div_exact_3_and_limbs_div_exact_3_in_place() {
    let test = |ns: &[Limb], q: &[Limb]| {
        let old_ns = ns;
        assert_eq!(limbs_div_exact_3(ns), q);

        assert_eq!(limbs_div_exact_limb(ns, 3), q);

        let mut ns = old_ns.to_vec();
        limbs_div_exact_3_in_place(&mut ns);
        assert_eq!(ns, q);

        let mut ns = old_ns.to_vec();
        limbs_div_exact_3_in_place_alt(&mut ns);
        assert_eq!(ns, q);

        let mut ns = old_ns.to_vec();
        limbs_div_exact_limb_in_place(&mut ns, 3);
        assert_eq!(ns, q);
    };
    test(&[0], &[0]);
    test(&[6], &[2]);
    test(&[0, 0], &[0, 0]);
    test(&[8, 7], &[1431655768, 2]);
    test(&[100, 101, 102], &[2863311564, 33, 34]);
    test(&[u32::MAX, u32::MAX], &[0x55555555, 0x55555555]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_3_fail() {
    limbs_div_exact_3(&[]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_3_in_place_fail() {
    limbs_div_exact_3_in_place(&mut []);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_div_exact_3_to_out() {
    let test = |out_before: &[Limb], xs: &[Limb], out_after: &[Limb]| {
        let mut out = out_before.to_vec();
        limbs_div_exact_3_to_out::<DoubleLimb, Limb>(&mut out, xs);
        assert_eq!(out, out_after);

        let mut out = out_before.to_vec();
        limbs_div_exact_3_to_out_alt(&mut out, xs);
        assert_eq!(out, out_after);

        let mut out = out_before.to_vec();
        limbs_div_exact_limb_to_out::<DoubleLimb, Limb>(&mut out, xs, 3);
        assert_eq!(out, out_after);
    };
    test(&[10, 10, 10, 10], &[0], &[0, 10, 10, 10]);
    test(&[10, 10, 10, 10], &[6], &[2, 10, 10, 10]);
    test(&[10, 10, 10, 10], &[0, 0], &[0, 0, 10, 10]);
    test(&[10, 10, 10, 10], &[8, 7], &[1431655768, 2, 10, 10]);
    test(
        &[10, 10, 10, 10],
        &[100, 101, 102],
        &[2863311564, 33, 34, 10],
    );
    test(
        &[10, 10, 10, 10],
        &[u32::MAX, u32::MAX],
        &[0x55555555, 0x55555555, 10, 10],
    );
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_3_to_out_fail_1() {
    limbs_div_exact_3_to_out::<DoubleLimb, Limb>(&mut [10, 10], &[]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_3_to_out_fail_2() {
    limbs_div_exact_3_to_out::<DoubleLimb, Limb>(&mut [10], &[10, 10]);
}

fn verify_limbs_modular_invert(ds: &[Limb], is: &[Limb]) {
    let d = Natural::from_limbs_asc(ds);
    let i = Natural::from_limbs_asc(is);
    let pow = u64::wrapping_from(ds.len()) << Limb::LOG_WIDTH;
    assert!((d * i).eq_mod_power_of_2(&Natural::ONE, pow));
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_modular_invert() {
    let test = |is_in: &[Limb], ds: &[Limb], is_out: &[Limb]| {
        let n = ds.len();
        let mut is = is_in.to_vec();
        let mut scratch = vec![0; limbs_modular_invert_scratch_len(n)];
        limbs_modular_invert(&mut is, ds, &mut scratch);
        assert_eq!(&is[..n], is_out);
        verify_limbs_modular_invert(ds, is_out);
    };
    // - rn < DC_BDIV_Q_THRESHOLD
    test(&[10; 3], &[1], &[1]);
    test(&[10; 3], &[3], &[2863311531]);
    test(&[10; 3], &[5], &[3435973837]);
    test(&[10; 3], &[7], &[3067833783]);
    test(&[10; 3], &[1, 1, 1], &[1, u32::MAX, u32::MAX]);
    test(&[10; 4], &[1, 2, 3, 4], &[1, u32::MAX - 1, 0, 0]);
    test(&[10; 4], &[1, 0, 0, 0], &[1, 0, 0, 0]);
    test(&[10; 4], &[u32::MAX; 4], &[u32::MAX; 4]);
    // - rn >= DC_BDIV_Q_THRESHOLD
    test(
        &[10; 162],
        &[
            3228372949, 511037199, 1766844847, 2482988267, 1469636570, 157644737, 1548533671,
            1988920387, 949768135, 3990423112, 2320976801, 2511643497, 3093682328, 3428374192,
            1438878145, 2361210316, 3906032124, 2138110309, 1599933050, 3823943671, 2259360130,
            823468722, 3291657189, 1273080994, 1300170623, 1148822796, 467501062, 3149282203,
            94789064, 4211146202, 938886545, 537979797, 2587941131, 108323731, 2067107670,
            105331677, 1935250512, 1452284826, 1800549345, 723680431, 2794168985, 904704921,
            3016862976, 1287705882, 4132038308, 624608496, 1997184254, 448598152, 1820893106,
            195416787, 651904977, 2955157364, 94342663, 437349382, 377636621, 3709576977,
            1757632127, 4278237941, 34068629, 2454145994, 4156002276, 3961922656, 2767617077,
            3288237973, 4240277740, 1356471757, 9181081, 1636990395, 1662834396, 626816820,
            2875878941, 921881987, 2158368015, 840508787, 3305037379, 3711054769, 1268398179,
            3896061030, 1052597619, 2040268853, 3162123944, 1619537254, 4260249983, 788774918,
            323831598, 70896907, 791692982, 1081190864, 2374371095, 1425732171, 2496438238,
            4066183598, 2164924109, 3370833658, 4001560838, 1910191661, 4219785120, 451095085,
            3421655677, 2108533585, 3424568374, 2891875753, 3180562301, 954785991, 2447015474,
            3930894331, 715625138, 2543510489, 360538720, 3774920082, 2598943309, 2345969301,
            1201240332, 3131955258, 115485203, 1731266015, 3565733667, 3708968152, 3343319896,
            4164543588, 958613110, 460381838, 508487067, 79603149, 4226757805, 2221491775,
            793628682, 2721231001, 1503226493, 3320680646, 536612991, 1986939605, 3925890122,
            953289772, 453639997, 3182429348, 3743006936, 4003692116, 1964545970, 4050648413,
            4076626036, 416351025, 1696831244, 3133590002, 1111664786, 2906324009, 1439094475,
            2214055485, 2798705323, 435684851, 3596001227, 3036344489, 3889898941, 1065048513,
            4016594719, 4180638855, 3698313403, 2650988185, 407962531, 3005112678, 4110843613,
            1265856698,
        ],
        &[
            991539581, 170886811, 1946164502, 4138638160, 4166973165, 3436035151, 3104724201,
            1844757770, 2340741284, 3474459530, 2913154637, 1985603183, 1490032629, 404439462,
            2516898868, 4063787328, 2068346028, 3562917355, 3607751063, 4085533551, 3185010719,
            4194162665, 3987252641, 3919902763, 1349071027, 3498237178, 1424577630, 2755258000,
            1356884618, 432658210, 1133331071, 1971536170, 609062776, 2584940413, 503389674,
            833096403, 2005416443, 873663129, 600036946, 1888391296, 3845065736, 2778213007,
            3174588960, 860483363, 3214125890, 241195760, 4142060378, 3524395685, 2521966985,
            3548302474, 4280709033, 1127351847, 3384545423, 1156416795, 2964727747, 2348926815,
            3088158836, 914441803, 741513463, 43888624, 3905516593, 2467124760, 57532003,
            2507495445, 3391285900, 917840688, 1449210608, 83470763, 2386746834, 518977089,
            2745927519, 1466192189, 1209060587, 1707570892, 1673078302, 921540600, 3312933158,
            503196443, 2601292096, 584811553, 315531478, 2613557634, 1029661810, 1565433701,
            2749963283, 797016668, 2524006881, 4144613766, 342648943, 3634335756, 2886479078,
            3729634162, 2446396883, 2646833057, 3809228466, 1241062391, 1771218854, 1859001695,
            181499665, 2243910284, 3135305341, 1144637960, 377778783, 462714009, 1503150032,
            2671008870, 2934075825, 1452005121, 869697216, 2087204075, 3565024128, 3030614795,
            4198797483, 4039346519, 601848748, 4060938550, 2096365943, 2015345784, 2227066926,
            2589779098, 2064017544, 3930785044, 572495510, 1372658164, 2267700924, 3842986739,
            1865979088, 727501730, 3530720452, 2082196229, 744666215, 1657468106, 1799434953,
            1390303492, 713826320, 92635671, 3263118136, 4104386251, 2272448841, 2859245350,
            3739570419, 2388112539, 2973693102, 2566815653, 909557454, 3719121807, 4178393753,
            2424198070, 1957898281, 1984544495, 1408061170, 3310277440, 1794952171, 452701329,
            3882836884, 3298396236, 3967258537, 162517223, 3293672996, 764489620, 714009670,
            31373948,
        ],
    );
    // - rn >= BINV_NEWTON_THRESHOLD
    // - rn < d_len
    test(
        &[10; 1539],
        &[
            2598813973, 1904197782, 1215006787, 2930851058, 147918011, 1108782873, 2193711651,
            927233146, 4123817534, 3041706771, 2553964352, 2170557862, 2838589053, 2708835560,
            2407955329, 1653207324, 2470552742, 3357058396, 1072990020, 1514783799, 1734243804,
            1048851304, 3193592152, 1383793603, 2063694499, 3485786208, 990960024, 3569185429,
            1155708496, 1416333121, 623402323, 3585705881, 2928088588, 4117621841, 3292316347,
            2217241313, 1950216441, 1391040412, 2926386224, 3240560579, 3973468920, 2847202482,
            2728825575, 2388048350, 3463506691, 3707514927, 865702555, 681215612, 2250616248,
            1724513618, 364443494, 318451745, 1623887866, 1280032334, 35195442, 3837277484,
            3573894817, 1602810477, 2072874171, 3800895590, 4253419440, 3499691102, 3587932863,
            3517441378, 514856216, 1648457744, 1474005402, 700069688, 1803098596, 378514692,
            3068261379, 3834605983, 93188389, 516236732, 2468656032, 4095977127, 3706578522,
            3452719188, 1052956630, 813640425, 3047313546, 754584849, 3584789926, 347367947,
            1332536470, 94171148, 626270691, 665836088, 426974652, 749957497, 1566095325,
            3066600838, 4106843490, 318006871, 162882344, 3632111614, 720131390, 2147721707,
            2217002633, 2466031857, 1087787961, 753159680, 366300446, 3082468114, 3984408901,
            4057095635, 851983865, 4150513631, 1165194321, 2055798628, 2975996128, 2693369845,
            3726562839, 3482086133, 4069043610, 1448967233, 700979569, 1698695640, 2318799208,
            3684286306, 4092668018, 3628416641, 584456626, 1679287620, 1813093480, 177926224,
            998082621, 3120492909, 1996199168, 3491975384, 3147828383, 2087111449, 381360867,
            1127916110, 3472685226, 3293954452, 490729044, 3264678372, 1277191068, 4239255920,
            2343436004, 2451796708, 1844810428, 2031787164, 3768550378, 3743966167, 2637191163,
            3978079759, 4158883864, 1561267833, 539304561, 3435817790, 277956235, 4028949085,
            2643928548, 2755470882, 3796376394, 194519280, 98463778, 3179299980, 2447702226,
            150057260, 3463931601, 906456277, 990239469, 512339370, 2261507614, 2826877099,
            3855453279, 2271503686, 841719073, 1591142177, 2651714174, 3361609137, 3279250024,
            3326254458, 333896144, 3528172547, 3941409760, 1558812882, 3841874000, 3172301956,
            4257828380, 4114850067, 2182891616, 2046840523, 1555755083, 2209766728, 2332501890,
            607500373, 4124706810, 1713777506, 2341598661, 2284959418, 2746608689, 867377978,
            336903700, 2503448208, 701980251, 2377952035, 614377477, 4284955660, 3814775340,
            74129761, 1280559390, 2313692692, 1849232614, 2933185433, 1636271207, 630083065,
            433420841, 471362220, 4210050941, 406797277, 582028044, 528813510, 2576049278,
            540310007, 2789529550, 3534664845, 3236430259, 1506736021, 1977751182, 2006662671,
            3679126532, 3806292929, 322589328, 2243078958, 2178902647, 2189720772, 3791358339,
            1508304160, 1013509043, 3587377598, 3623382349, 1086144534, 1599008554, 1093927956,
            2876914186, 543927302, 2219667262, 3853928343, 2487266159, 2613372592, 3550686437,
            3716813778, 961453654, 2039051889, 4283100852, 3145660844, 1250225831, 198998794,
            3726057928, 2284062625, 2219120593, 2244278486, 3790941761, 4203700163, 3879487839,
            2965718505, 2863698764, 63056401, 475617471, 399914031, 459648012, 3133447831,
            3412742746, 1512219896, 402842103, 4119870394, 2231397283, 1516833787, 1104380754,
            128232714, 296865691, 1663298638, 716369560, 2904885147, 4237273250, 343946912,
            2604094031, 217784841, 1162604926, 1345215097, 1426555576, 3614102008, 3943750939,
            1639058309, 3296139198, 1330339563, 426593850, 1532081057, 1300177557, 1482433791,
            812987015, 3232993095, 1037853411, 645052957, 2928052694, 1226934622, 2336175954,
            1920836876, 2517428765, 3726338359, 4041660846, 2092081260, 3380258824, 1697518166,
            3349439761, 743024530, 1795639086, 1779094250, 503268222, 3554837208, 2986098469,
            2762697287, 232864157, 3768291070, 2983947152, 1577288457, 1050740564, 834028049,
            1274280907, 3769464765, 1910494593, 2197778112, 2813074316, 2881147492, 1559228213,
            1461309572, 1700587844, 2988103759, 1717431083, 951586291, 1707887130, 985013185,
            3532658879, 2597699341, 1911327175, 3080664111, 3484272917, 794091071, 2311407322,
            342113645, 1768703425, 2382063884, 3831832486, 3217975952, 3451120056, 3035392294,
            136312738, 1699608380, 700762772, 3631031559, 1846450824, 1131124755, 1046357323,
            2552828340, 2598371768, 795498120, 3174589062, 3838467339, 2334795665, 3324675939,
            3403942196, 2075108646, 4259667163, 545251542, 749294402, 3324014447, 1729518387,
            3551096145, 1758729590, 2379344936, 2455329109, 1757426268, 1706226190, 902425986,
            2432676044, 2039145952, 410402636, 1639537846, 1305863342, 256740017, 3207555620,
            3925040779, 668173470, 1141011911, 4212893794, 1581027147, 363670779, 2004465281,
            1138726957, 645471449, 3585049231, 2777598154, 2922780411, 2222055902, 3815639175,
            429697628, 4000303790, 3545345324, 3542109929, 509705074, 3722614775, 1753809005,
            1973665455, 3780943261, 2494678174, 2641931684, 3388813526, 3347201186, 619828405,
            2625911693, 3122659421, 3479474745, 3245356132, 1832479544, 3308783684, 1963028441,
            640786937, 1641844190, 211570630, 1733515651, 2171524601, 3396267685, 3914881000,
            996687218, 1617466047, 3343054134, 1737892381, 3544966831, 3416864496, 107959379,
            2249494870, 793869312, 3685840230, 1757287133, 1655716303, 4226516527, 2538565645,
            278467613, 2644886589, 2699974905, 1984329139, 1005957589, 789155264, 3666799939,
            1611583926, 809907316, 3452803056, 1107204019, 2209104411, 1462179962, 2853352142,
            3225410978, 2108213590, 4192481899, 1912371245, 1171525154, 3712460427, 2560727022,
            1086797266, 952026066, 685685805, 1936296543, 3499184244, 196202245, 3078070455,
            2171291372, 2730765878, 1573851700, 2931331934, 2847177501, 3478529515, 2313080814,
            1556316903, 1559216686, 3130885922, 1862119430, 3274978261, 206968409, 2624567354,
            1978029702, 4063454216, 2415324407, 1518621963, 747943965, 1111109939, 4070402813,
            3113587512, 898431887, 316373197, 3046639257, 173632841, 3798588025, 1794547053,
            272903937, 863380298, 2774736392, 2777149271, 3206794646, 32657715, 1025716581,
            1615425071, 3173842612, 2073903119, 2153824733, 2329799226, 2341967952, 473768936,
            3486256934, 2848218304, 2428724257, 3159637305, 2126501230, 765243196, 3425509759,
            1131263732, 2515711177, 3855554546, 1023477320, 787209967, 336492935, 2359162223,
            1128313353, 2209616384, 389804749, 535230248, 2210531774, 3227122449, 1267072928,
            675693308, 1385601874, 1416098547, 4192675028, 2716632450, 4177948782, 13560634,
            3703614488, 3901742123, 2270254077, 3759771486, 581228935, 3949774462, 3458012976,
            4163726158, 1602955605, 3047124020, 1612897320, 2437308530, 2370536309, 1561087937,
            2116794827, 1579558066, 431023695, 689842616, 2342172842, 4203138565, 2160272314,
            4192374041, 3675193896, 1455954294, 3940635994, 4025127357, 4027321239, 1155472117,
            1928138697, 1875048682, 4087308487, 4291614860, 1175537429, 3503698743, 2187471558,
            2673330725, 2702107217, 2763809959, 2274134903, 1077148334, 2166895545, 1831099412,
            3296218829, 2076626325, 1104984380, 1764164194, 3056784327, 1988083103, 3849746178,
            3460231757, 3983590507, 1641801163, 1321487487, 2277192156, 1700397327, 4163993314,
            118570737, 2510013157, 342313019, 328214482, 1456063300, 3771950632, 1226087686,
            2343733178, 1108934200, 3650307976, 118441436, 3857631518, 1808729865, 2826187782,
            2596972674, 1966611164, 988201436, 3909729734, 2034457184, 3137580128, 437202273,
            1431097330, 2630459633, 2423317302, 1422128850, 3011012632, 3611423092, 3531891566,
            1592039994, 734168534, 3492902008, 947200226, 1735249159, 3526990689, 1725556774,
            1693403463, 3317454666, 2042191731, 3608995702, 497650714, 628421679, 2465885637,
            4008542175, 68576338, 1372104957, 1927020046, 3573635655, 3090117252, 1774561696,
            3603424898, 55089895, 2060875413, 116850528, 183816573, 478639013, 3736973628,
            3606174188, 3086193394, 2548388100, 2591332019, 3032618195, 469584388, 1770142568,
            205800329, 174943950, 260521576, 2114953066, 3428041459, 860714084, 1560598053,
            2885505481, 3365355923, 2821218095, 814424502, 94258256, 2857874640, 3213633690,
            239445181, 1741311155, 70754202, 469749979, 469377271, 634558134, 1314018935,
            2663122712, 3425107874, 616125297, 1220520411, 363320242, 1717903421, 2177377494,
            2683338356, 210565640, 3061448891, 1561105018, 3639722552, 2586119577, 264509957,
            1416442444, 572448512, 3316197518, 4101471127, 1909662580, 3289492851, 702940172,
            1747588119, 3288795988, 1441233966, 678854386, 2604723913, 373703246, 2055937039,
            1124872113, 1294098600, 4165350921, 347872377, 4149215174, 3215450946, 3307233717,
            3600358822, 2659616884, 552889201, 1684681790, 1821948043, 2909781107, 917768879,
            2259947602, 2019396153, 3413299181, 603749894, 2467542574, 4017401393, 2519998741,
            1106606352, 2283516402, 4262893045, 931507531, 257221572, 1471379658, 4223521274,
            2958168654, 1643148068, 537294061, 3213657938, 3835022365, 3594404173, 777970857,
            1918635217, 1702205904, 1047564373, 67547546, 3019489110, 483595975, 4130252917,
            3081408781, 436773586, 1982914868, 3527757249, 1553562987, 4080227164, 822023069,
            1427193014, 2565421648, 3942884687, 586018307, 1138884595, 2613035769, 486310535,
            233458001, 106898418, 170926111, 3212999317, 3194995097, 827548055, 1783632131,
            1458445783, 4273893842, 4169283956, 3425898311, 1900133873, 2830934989, 3597188167,
            60014629, 2917912426, 3137954804, 1361341056, 4180775481, 946784052, 4199341441,
            976101959, 1662997785, 1179237245, 372119531, 2815699743, 2114995295, 3901740963,
            3080952004, 938243235, 1717858705, 1880470411, 3714648015, 2698621435, 4276694893,
            327245390, 2823511048, 1178264505, 2316760826, 1757942990, 1568677829, 1199580238,
            3100459677, 1386893348, 1600406349, 3128791813, 874040538, 2965907285, 8663319,
            2919597841, 1169807363, 2130724711, 2817929643, 2594603073, 975728895, 3248945863,
            1506505714, 8259384, 2395186489, 64139190, 254403781, 3179528072, 3318835559,
            872874473, 3221533853, 1278142055, 2865063695, 2096015856, 3893588672, 1650965167,
            852689473, 2456343381, 2227877463, 3533995396, 2567915364, 3260019114, 3354960043,
            317493973, 4088542285, 603804423, 3133721086, 2455680365, 2501599681, 2371144161,
            4224001882, 3900064951, 2295679614, 483545534, 500738921, 259183629, 2958675979,
            3886544014, 3382548497, 1341558649, 764676712, 2686742281, 3585367009, 2470017030,
            280649561, 504360831, 3177630390, 691028027, 3630869823, 1850039925, 693186712,
            2234334464, 3019783104, 1258253876, 2759186239, 2705660533, 1535024585, 16228098,
            3597194253, 3856258376, 2530800309, 1756061934, 3703461702, 1458672824, 3009579190,
            325542092, 974024523, 937809905, 180338019, 728257472, 1334683442, 1531729599,
            4257819380, 3701291761, 2423233504, 3786098144, 776062663, 3861731850, 3717496633,
            2918219359, 2369401162, 1963105504, 2673826226, 3238671093, 568083438, 4121947082,
            3892853652, 1632686679, 2302193583, 3211408318, 1938993158, 4122064270, 311269184,
            2105346675, 1581174365, 2485158976, 2443260070, 1986431168, 1124848188, 1841087451,
            2704521996, 1214274421, 3662035084, 2837679678, 4043497550, 552208260, 1644975825,
            1950150273, 4027848966, 4079004873, 647675052, 223752470, 637576072, 949880156,
            2749078102, 2043391316, 1008133, 253411434, 3799679907, 238227057, 2495985662,
            1475105846, 2961884606, 652180469, 1691635637, 1430944868, 15398684, 2278485165,
            3795142783, 2981896799, 1330890913, 2239691223, 407470744, 3702570774, 1624292191,
            3703366632, 2187395861, 2415878960, 2303069741, 1369122706, 2750340520, 4035673151,
            1425244662, 25186311, 123582970, 2490005251, 936607532, 4115073120, 2730391719,
            2823090688, 396451377, 2182998280, 1365726601, 4033770954, 3047868289, 2336389408,
            688454315, 1299725781, 3304644284, 4238176839, 3914620151, 3029466942, 3172492309,
            1190474536, 569450335, 320580314, 4004925139, 2557991799, 1493351038, 949583033,
            3275441062, 1311907795, 106222620, 396788468, 3186456628, 517037899, 2177653619,
            370023265, 532558109, 3182151426, 954367771, 2865591818, 600183719, 2521622949,
            3079560626, 1906789399, 1612708432, 1801660151, 3565921772, 2688352000, 1101243899,
            888006794, 2233649225, 2671500356, 1504445832, 1199168343, 2043022343, 1250919220,
            1374956710, 1107550604, 957179006, 3319703903, 4110453611, 3311143947, 2774029776,
            1456730624, 1014119593, 3012545941, 1650066434, 3431589287, 3624715486, 1072097915,
            3568695028, 478299405, 362822053, 2632030958, 955747917, 2974153651, 2880247593,
            2837857407, 1533402093, 1550331006, 3833005459, 111580345, 1694562098, 2731241196,
            2659074758, 1347615125, 3711588514, 1514710939, 3788474419, 3668780501, 3266213391,
            4081085519, 1196081228, 1049947080, 1820720234, 3100382458, 2198468097, 3708538645,
            1035729054, 1784655621, 507574130, 2908640623, 3615200865, 1223546227, 3675568182,
            1661330896, 1399689373, 1323389692, 2981922901, 4156332999, 1902241850, 1428680890,
            3049572611, 4253261231, 2148214199, 1998921071, 3899443086, 1478235253, 3791927172,
            1483896530, 3565485437, 915921912, 2258155251, 3426051958, 3307670857, 3029363090,
            3673788309, 4082838999, 624603732, 573907442, 3116664857, 3586999255, 557493850,
            1739716128, 2397076308, 2312498391, 767640384, 2119706819, 2637314051, 1070866648,
            664107404, 616236779, 3020748367, 4122128780, 2009706384, 3245363034, 429196441,
            4195520565, 4209198410, 2158211364, 3659713923, 3985561806, 1747634790, 4115935838,
            2827667115, 3073689925, 1621229363, 2774950774, 2538390550, 3506768465, 718261358,
            564823729, 380006788, 897186389, 3414352487, 1032216622, 1469922191, 2196317056,
            2239269473, 3654751282, 613015420, 4098167098, 1281134924, 849099531, 2523299715,
            3904840545, 1997547696, 1158713208, 1289444182, 757815917, 899581875, 3622748533,
            3063686656, 2979790776, 1553609204, 355408616, 3897614956, 3680334470, 25191644,
            355655308, 111682127, 4053312775, 2921335050, 2200671456, 472639194, 1374183459,
            2738441274, 1596829615, 959522853, 1993763998, 1832301144, 2263515976, 993190648,
            4045308024, 2213735364, 4061773476, 2596920312, 1929252967, 3935609312, 982934883,
            3154772065, 4254233108, 1244999206, 765237459, 2528318166, 1563389556, 2293797399,
            3495759008, 48504250, 3182637462, 3138309477, 2147232621, 4810076, 3999234780,
            2279064264, 2264522982, 3079929466, 1524222187, 1598294545, 2049273389, 2729549735,
            2116985451, 2637892533, 1094214870, 2711414546, 418295062, 98357556, 296649589,
            2636445209, 4030461397, 3073508440, 4062358649, 1293566861, 1534697481, 1343281639,
            3078259374, 386875427, 1145986442, 1790533181, 2987076845, 237202537, 2194958623,
            935119723, 1921262864, 943957815, 2336181341, 2723699978, 1638860845, 2630349960,
            431757538, 4152375793, 3473513084, 2603606861, 4176464622, 3978369596, 981482293,
            929741839, 2111696471, 571254437, 1419237733, 901791748, 3340393111, 221970895,
            2605197016, 22565479, 1000841034, 2192399670, 1207300847, 356824298, 3009617459,
            2651652098, 2420275852, 1631602976, 575809994, 2928806259, 2831092106, 2629452482,
            2842897182, 1909556662, 2496151886, 3385330607, 957152776, 2804092840, 3551246686,
            2730894896, 2027253611, 2531182020, 2258110944, 1391264743, 3902230517, 3835546712,
            3424753892, 1147410244, 3448482025, 319341118, 1798380053, 2742193162, 2106010955,
            3267048655, 3886107336, 3334600393, 3772435523, 3985863328, 2950255062, 1012493134,
            3954831363, 325018235, 2523597002, 2454925521, 2861802043, 4209313262, 3868234463,
            1685691511, 2786322255, 4244960817, 1991229419, 4239301712, 576152977, 1652411241,
            1589394460, 521549473, 266587858, 961791319, 1146831103, 3469211532, 3501779101,
            2312218475, 2321025269, 1346459216, 472196028, 3771324053, 1495430908, 463405457,
            1940232183, 3937148918, 1156878514, 2831556660, 1644483662, 1787817713, 2339616130,
            3815974485, 2298018558, 71674109, 2675299410, 960845405, 3569711116, 1673611570,
            1769369631, 1437416980, 4174063039, 2062712084, 2133844948, 196794643, 1935768058,
            1831573341, 1418980930, 3294487604, 3544538744, 2334730191, 3988003806, 2274350901,
            2023027498, 3447859831, 1671781771, 4023665101, 1455029189, 2419433622, 2943475311,
            4198568768, 1395089836, 1092520718, 2430238564, 437551709, 4097009390, 3375079636,
            4154198223, 92070792, 974607811, 3873908092, 3374473679, 638725898, 1249915990,
            1085664087, 3927733894, 1217984426, 1825247311, 2107887775, 170602951, 1075337232,
            1842870533, 3647067870, 2444592832, 3013106942, 1161476947, 3743427689, 1263482478,
            69206560, 833594813, 3842989721, 337909675, 3392484922, 370017251, 27594915,
            2126835633, 1736323108, 1893331695, 1468455680, 1075562062, 455344215, 969605531,
            294398875, 883453327, 3994410499, 1964955910, 1983678849, 556499140, 2199399388,
            1506753515, 2562511167, 573514453, 1452183180, 3758124304, 2276736219, 3962356320,
            160602272, 2732947819, 1312555719, 808517807, 1646633687, 4057383018, 1177836167,
            2724613618, 724351202, 3082498407, 3292565482, 2024078169, 2339554384, 3963173875,
            2223760730, 2562066565, 2193608401, 1812209945, 1665831972, 1039920028, 3410644888,
            3314911306, 2847575459, 2005962667, 2392982038, 2769384144, 1673231309, 2605033025,
            3097232789, 525491643, 1088675733, 389231889, 4053722369, 1394160319, 920345783,
            1695188025, 1313259843, 2855284945, 1128781302, 1711293636, 2436253183, 2702553100,
            2342544037, 2382554808, 1645824427, 700888148, 3820471891, 2062002896, 1600256482,
            3523617235, 3825142862, 4016136295, 3838255962, 2953276340, 394106186, 956179026,
            3512260850, 3001113638, 1645204518, 555542490, 409962126, 479780262, 4202384502,
            549252742, 1340551826, 190095025, 2842856010, 446626748, 2565781573, 2172050474,
            85422525, 151000070, 2413238086, 224534468, 4102988065, 3524737259, 784974768,
            3778660388, 2573449273, 956921977, 1416256526, 3821996527, 1297167173, 2183926071,
            2745200336, 1745460864, 399929970, 3271657494, 1058524185, 3138265217, 3822962039,
            796872545, 555781465, 3355863387, 4265560398, 2445094472, 546611744, 958064176,
            2278895404, 2100057835, 789578969, 387055963, 1153298400, 4065547056, 3688020091,
            783181103, 1789734962, 1466117990, 1740574947, 1632230912, 4155788342, 1714025679,
            3862124914, 2247239755, 3047057899, 3610673443, 1450313039, 3533121399, 1564221183,
            4056883174, 319992191, 598075724, 2438292429, 901314271, 2157273664, 435275402,
            1774735757, 177332560, 3010394026, 3827445200, 2002098740, 1897812156, 454287770,
            2829500771, 3286056227, 1498747083, 3402858410, 58581019, 1048766033, 1640106396,
            3774636176, 1228719792, 1768725616, 3094025701, 3705595233, 2943515149,
        ],
        &[
            3303507517, 1737467916, 767592771, 4237693101, 2944498355, 2305367062, 2912464955,
            2039632714, 957753136, 3932250327, 2923226395, 758588986, 3276864736, 4129542782,
            3701423701, 1834120826, 1304646722, 2700953412, 4016654202, 1446197632, 4233056281,
            2795834408, 2358334260, 4031834794, 1761430989, 1322974963, 1041373666, 2212100552,
            2426361770, 4261631544, 3079923997, 556209919, 2565829294, 1158111010, 1485991528,
            4015229549, 3673606504, 611864921, 976174193, 806940030, 2732222596, 2259057471,
            2443961478, 4098098138, 644289655, 180553484, 4059537703, 708282117, 1433746242,
            1823452813, 685407883, 2203888447, 3826783627, 2594937784, 3968973728, 3226845414,
            2669006681, 531830322, 453908823, 3201286311, 919570600, 1858088761, 407253743,
            1432230279, 3303340419, 3210731060, 1388706247, 551779110, 351497524, 3064709178,
            2129368300, 1872141157, 2752241851, 2688864462, 597028566, 3593773250, 2480760169,
            1493938468, 320489468, 1206910552, 2899495800, 1583265563, 2935858872, 126264340,
            2083066625, 2827628121, 3109648374, 3916360359, 389087236, 3766699281, 1333824469,
            1218761052, 2129210405, 2527568435, 3858191140, 3930233522, 1058437374, 1654914995,
            911091073, 1037669921, 2949223555, 2606346852, 1104967551, 701746540, 416874025,
            683350868, 335950780, 486101025, 3696781531, 434131711, 3081485031, 4277564240,
            2555479895, 3850726160, 3227325468, 553377665, 199290481, 3982285635, 3468419008,
            3392035752, 1966268427, 1873702317, 2940911086, 2914228436, 1708633307, 662031149,
            621794577, 2395698305, 2735240556, 1414089965, 3294650435, 479604329, 2426263958,
            847381928, 873722272, 1110653081, 1951740102, 2731402038, 2556525098, 2219924857,
            1852070957, 2449841178, 1210250294, 3699155577, 1784858631, 243884741, 2161178521,
            2410975312, 2681491929, 437235103, 1281548845, 3357384194, 549196434, 4067561207,
            3450283139, 3652169421, 1874377644, 169053482, 1420061127, 395406675, 3332510954,
            3856194671, 3006496664, 809601906, 1922764991, 3111985626, 3390014784, 3048045015,
            2051558581, 2703813358, 3808996437, 3570075083, 3505603444, 35527898, 3996784836,
            4168425592, 1946706469, 856358712, 3661547641, 141395264, 3094553658, 629697921,
            3847603586, 2560897876, 2555855191, 1405066228, 896493515, 3694348082, 3454533550,
            885932680, 3723805569, 493727984, 2574262444, 610088096, 503058120, 2172357146,
            2481205990, 3987005657, 3085722243, 2250687621, 1050336745, 3740028373, 3019114451,
            1642616645, 1137174523, 354355265, 1455093279, 304460919, 4116447569, 2981853747,
            856638318, 3109624356, 3591169837, 4193437247, 443242328, 1666463852, 2097829509,
            2208974333, 1422720757, 3349658507, 2452053895, 2569932614, 47797786, 3752698672,
            2215739699, 1652083417, 3346170449, 585897335, 2075209213, 3518121114, 378885805,
            3744168278, 3903962084, 1064225611, 4197819881, 1867701541, 1058875078, 1828480385,
            35105056, 2154079979, 4006209278, 3153027209, 3695601551, 3461064991, 3223410438,
            3090214664, 4250727661, 167290560, 3088602589, 1392761792, 4016528315, 2899219207,
            2196701677, 1154503712, 810711706, 3932982426, 2420112400, 3293415319, 4134821524,
            3493789883, 3329390364, 3047733519, 1049261264, 3282299496, 482643201, 1333502492,
            3412458538, 3505648168, 1030433939, 3681403988, 4159258448, 821956, 3965405477,
            3688327, 2218271368, 3431171763, 2674850379, 2269778883, 596405244, 2839693344,
            4146102065, 3215467606, 1891293404, 973749441, 3640452446, 2709562759, 459831821,
            460230179, 2910252183, 2014822547, 3590552255, 2390315678, 1031487459, 1213313670,
            2511201602, 1511763753, 3969704570, 2909506049, 1760818720, 1110188569, 154546279,
            577049207, 1753565754, 4085487615, 3287577161, 2028144074, 4111187020, 3004119870,
            900037962, 2806754372, 748875551, 2620104615, 2513455401, 1585914851, 239727281,
            2605810930, 416537433, 2166996582, 1965459949, 758058785, 1510362220, 1824837142,
            3097689711, 3896639991, 4187903962, 372233538, 4010166025, 1907205494, 3036010338,
            1576810188, 3371537329, 2895423119, 3461187657, 1440946377, 2483629064, 2639054882,
            2819542476, 1677376583, 3628809647, 1237686175, 3985351720, 2642412817, 4013664855,
            2065613802, 817459407, 2876445270, 3076582045, 1810603735, 1174549530, 4260651012,
            1973700727, 2194803557, 2360115398, 3069422632, 1295376632, 1002159587, 4162264900,
            963230566, 193817342, 2402613170, 275479630, 116160077, 2599851335, 159870465,
            211768251, 4017718175, 931115106, 1004170356, 2730898374, 932870994, 1082826978,
            2961790318, 3424432922, 1300198410, 2280596007, 2164752687, 2135660458, 2665413409,
            378731924, 3146959946, 1447131084, 624620351, 1626838440, 3569823375, 787344599,
            398683074, 2909969062, 1166082222, 2222839820, 3445152909, 241015893, 3204685501,
            1212401564, 4123206390, 1295604587, 1098987770, 3308585450, 3613934791, 1755566651,
            3657857779, 1263036827, 3678216851, 2402921074, 3493820436, 1704692066, 403162744,
            3170124922, 3563160884, 3160828389, 464965432, 1118230114, 988354534, 356341207,
            3849468154, 686849000, 1566884214, 240572604, 3063337879, 731356349, 820769259,
            941504045, 2706839108, 2723874282, 278246299, 3880298064, 4262421502, 256437639,
            1773319540, 153110521, 565549233, 1853553593, 4289762574, 4098151149, 3144524222,
            4144088665, 2052608329, 1464290562, 2110534105, 1113619720, 2516886130, 2842359662,
            436774705, 2771156018, 1683247173, 3108637805, 2436817839, 4127310701, 3910938906,
            2062961202, 3341755081, 3583281049, 3534420977, 996954658, 4115691732, 1529857101,
            519089315, 1800475325, 2866891326, 3674269922, 4258687645, 1222536664, 119098982,
            3470430550, 2133110377, 10231145, 2797637322, 1573837132, 2948970055, 44978960,
            4025694418, 3346973726, 1067376893, 2502195176, 14223176, 2185770650, 2887537518,
            15571252, 2033910005, 63833522, 787872402, 3088869854, 940048320, 409417410,
            1084110155, 601749701, 914916762, 524011526, 846030561, 4258519014, 2487948410,
            3192173244, 3091400971, 3644666631, 532480229, 2780924927, 604543953, 756270658,
            2063413786, 61217573, 780610452, 955642466, 737073209, 97297839, 4258953485,
            2857641388, 304710700, 3918409840, 2194881876, 196515060, 1845737739, 80004705,
            3864082477, 2417895713, 1575272517, 2071122658, 4181848470, 699244129, 2719978593,
            3969766361, 3331919022, 1806356115, 1674881762, 3140317527, 1020055538, 928717808,
            3657371453, 3576271656, 1661098282, 2787948393, 1612552028, 3154905497, 1703392561,
            843717976, 1621287936, 964546377, 3549430129, 712295080, 2234282678, 4200849568,
            2117828027, 1392560869, 1138816605, 1628721368, 1719186629, 499638295, 2276446667,
            2040220810, 4170119435, 3098091505, 3464708059, 2446105454, 823802794, 561105361,
            2956734110, 85242489, 3594780155, 577528063, 3814286772, 3517308917, 1542560009,
            903130712, 3610028360, 1094281534, 428913785, 2017125716, 4147179238, 2416543280,
            2299224901, 3094092893, 690981050, 1074080982, 3806304984, 42870724, 661112767,
            951312097, 993899402, 1421924279, 3477956487, 2620630825, 421567847, 1135313615,
            1871766543, 2089515143, 3125551624, 2893389248, 1822212994, 608096287, 1209599323,
            1729308570, 3183687973, 4188910249, 4113106401, 3467805891, 421479009, 2067824783,
            1123275745, 3372672466, 1299403729, 2597804990, 4109910992, 3549093152, 1334873992,
            2837764304, 217816190, 2798635274, 1928429025, 3532349620, 1432073554, 894604604,
            1320754530, 2946314029, 1689416007, 4088210274, 3660222258, 1778465861, 2089083512,
            2489444834, 3020220872, 2955906475, 530018361, 3604331239, 266271770, 1093367251,
            2425988385, 756441106, 3810449360, 166826417, 4128383674, 2193768558, 1695641441,
            4163952927, 3844926416, 3179839754, 1485072550, 1035240603, 2716609076, 4025246794,
            1340294165, 3056622704, 1486949742, 582340579, 1925519282, 1050735048, 33509855,
            2342941070, 2747830341, 1845018013, 2064583117, 350099328, 937043595, 1618308339,
            1824995534, 1246976548, 2566004874, 3546659674, 659106533, 2844260619, 3911232451,
            1416070938, 2388309597, 316855261, 1695213219, 856692567, 50305942, 3668671854,
            155369786, 2137812662, 4211140628, 1042632363, 287846104, 2375035513, 1607313275,
            4018768752, 2592097421, 741212831, 2408601799, 903321756, 3509061462, 3142569462,
            4087971745, 3019038954, 1803053183, 2257944916, 1390909767, 2836222866, 371214087,
            3682888114, 2241578583, 3605759723, 963603499, 1055335598, 3601870947, 4058606066,
            377849537, 1843635487, 1835087927, 3393215451, 17492256, 3372299656, 3413139881,
            4264793180, 1928965951, 4021643980, 2269905143, 545062409, 1389035345, 948398528,
            271536224, 3504014149, 567335901, 3413496753, 3071719597, 1445310729, 1770797777,
            1267315070, 2207614182, 3493277503, 4282532855, 2107955232, 133857837, 2087779634,
            4056521208, 818046024, 1890865855, 2126495054, 697548852, 3485729598, 1695030614,
            893769923, 184087075, 1500071319, 2080614980, 2480313210, 3450043314, 1064086870,
            2786617720, 4034347698, 681412032, 2428347460, 3253322495, 3604525345, 1428905795,
            4035094721, 2101041634, 4183013229, 1705836765, 1682316444, 1879660152, 656187483,
            3706455747, 2213541704, 669373128, 1529444811, 2814804687, 908625065, 2227416512,
            2001006578, 607169088, 3393837756, 2682839326, 1093822620, 2249228026, 741792013,
            3650948262, 2073718838, 2268511569, 1643477360, 2804865813, 3375407338, 1318291972,
            1414889752, 2869309697, 1958772002, 4162044589, 344252635, 3226572534, 3577208303,
            1959437424, 899330854, 3892405212, 1622822546, 1644013310, 1300136815, 2640337368,
            2069864152, 1596350405, 3843745327, 1256171531, 870418775, 2393395126, 3876818506,
            3944780828, 4008626102, 3127368256, 3426341771, 2086853545, 3827677693, 313614572,
            3669266463, 4080954414, 1707330242, 2963410257, 3437453487, 2794195219, 3099230302,
            726769739, 2044701195, 1129715603, 1793697677, 2760059361, 2331714255, 2589913453,
            4127324884, 2881798920, 3400611219, 2052671160, 3546649304, 2164589537, 2812204213,
            3337755588, 3983060148, 1899178107, 776922408, 4182115872, 444866608, 1399651129,
            3474508713, 2842510671, 3402478835, 1423128980, 4178402672, 3172508430, 804756202,
            1597189857, 1958174560, 3392832851, 1197650976, 3772223860, 2054821761, 4011804750,
            474742851, 1329523696, 4032274332, 2454461926, 638843849, 2072449514, 764758175,
            3934751812, 4075369298, 3142015673, 3394043201, 1535751977, 1846529809, 959593533,
            2235839818, 2814770736, 247085892, 2629667206, 2957898589, 3933947035, 1295728615,
            3586572901, 4038985747, 1129159296, 1855469111, 1271614337, 1507173551, 332723961,
            1156816569, 1294330789, 4122256011, 299955786, 3523281781, 2491532525, 1960595579,
            1730519156, 499071642, 1575886120, 2620965123, 3289148909, 1154711404, 1451719777,
            2628469322, 1901725555, 823453464, 3761287780, 2261972333, 3596389504, 3137670793,
            82630428, 3159699693, 837229231, 344117878, 2381467908, 726572177, 304723155,
            3208229621, 1458659872, 1794857529, 3755652400, 2395146944, 4102358516, 3277747506,
            1818040372, 3936695066, 2397713693, 1708618418, 1728140834, 1261714327, 1581655946,
            2071676586, 1202217464, 1344775033, 4250653297, 2492875948, 1126916876, 2945410447,
            2941087697, 2560380036, 2546735705, 3178023445, 626886956, 484882355, 3357302024,
            4057372957, 2081549406, 3896175417, 2686162101, 1986295486, 1621237422, 3568031348,
            4006200466, 1442461868, 3844879388, 3501884160, 4038570281, 4208037469, 183139977,
            999125999, 530710734, 2507567639, 3058017326, 4201561530, 2864542147, 2696523106,
            2969212130, 2678586192, 844065498, 2655259915, 1258788749, 2827607705, 2237733350,
            3783297719, 2313239370, 950226942, 1280595939, 1196906959, 1606546303, 1844760556,
            228764146, 744772053, 1358152164, 1193261600, 2565492365, 1225744635, 4233301881,
            2253145769, 1602119788, 1365069640, 52910185, 2636770690, 3298764603, 1878144899,
            4262332977, 3649154366, 3690806265, 231714913, 2913226838, 1554707381, 2963669507,
            2049539773, 1775985343, 3627719803, 433359314, 3418531436, 793315695, 913341557,
            2380241821, 348855994, 234181973, 3722537381, 3525843118, 1275331720, 2537849571,
            2827984585, 1565515383, 642373946, 1576289295, 2688247425, 3614474901, 1917731824,
            2884918146, 1646964474, 2383030418, 2488801475, 1560749769, 139516483, 1142139281,
            3459274279, 2863576242, 2828722138, 717484310, 1564480623, 3172598637, 2368201724,
            2247871922, 419758505, 3015961787, 3098012702, 1041261606, 984280115, 2438811327,
            2578749598, 978582878, 968817491, 2965070269, 2780764697, 792560930, 4229722212,
            4114038484, 817420493, 3041070967, 3191549455, 56820468, 186749905, 784610516,
            1729092057, 1935036496, 3745217089, 631003694, 3845500664, 1776171936, 2105858365,
            1027041956, 1638191578, 3198133424, 3040033996, 3224025446, 2046726489, 3326226799,
            4261577923, 778164631, 4272446112, 967648331, 4223690387, 394608816, 2849103815,
            1132525868, 4239388057, 3710951992, 582192848, 1372476599, 1228786127, 444221576,
            1576598378, 4000181362, 2210214319, 3226877228, 3921989392, 3841275639, 1917309689,
            2860911813, 4290305048, 1796377659, 3485206273, 2654605026, 1122524178, 3609458012,
            1733743381, 2869375618, 3606400920, 2485406505, 617720548, 497728528, 3826929977,
            788489471, 1188098828, 3383811959, 836335995, 104474735, 994749139, 523424077,
            1334471135, 3246261832, 579729621, 539626281, 1917656426, 3649706685, 407844197,
            3736652416, 449727051, 2451735689, 854611863, 3476911840, 973228334, 374181454,
            174850286, 4178627179, 3564929895, 1321454916, 3996342532, 3051609408, 3899526499,
            1406247157, 803997219, 3125646376, 1606457001, 1405170718, 208843195, 1067743855,
            2152262286, 3179865614, 3639249079, 3035981075, 2732174588, 351444382, 2864637808,
            1006363319, 3244329910, 405573136, 562447830, 2108083668, 396995391, 326716392,
            3263476213, 46276814, 1326176087, 1851003306, 2516298844, 3058805571, 2774935984,
            1856888260, 2568955433, 3218178509, 649282170, 2256866690, 3674417033, 1111304414,
            2088271619, 3238685708, 1574098232, 4183705418, 1987306144, 1189150097, 1378845891,
            3654818831, 386424237, 515382663, 1197128686, 2542968522, 3887585643, 73911371,
            4058793301, 1696639371, 2561358933, 2221990016, 727444672, 1973624906, 3853875864,
            174963610, 2695490643, 1066818493, 150873363, 3952837259, 2936691596, 3233597847,
            2506388918, 3711816965, 2536068963, 2311484662, 1093886392, 2704991380, 4059725080,
            1625573608, 2471671232, 1763939042, 2734410153, 4028854994, 2952269163, 1527756733,
            3902245417, 803045644, 1140158788, 3552349570, 235707516, 415130691, 1269910803,
            2620435440, 2226985943, 534205480, 2823709851, 4167053910, 3680035619, 2757219390,
            253917945, 3675657485, 761900957, 921345945, 3266224194, 3148826681, 1686290818,
            1647986984, 495687313, 977442969, 519576012, 2062617190, 2755583385, 2380042959,
            3142598908, 2437799893, 2426856697, 3052553836, 1753471289, 3915929071, 181022453,
            2017995427, 2796049960, 1358824868, 2906668276, 3538494963, 1500035161, 65877569,
            3786619137, 2956889923, 2470576554, 1511345314, 2266296788, 2066914712, 94664780,
            4203735289, 4281330982, 2763817600, 2319245226, 944042376, 3922820514, 3935434053,
            3397738753, 243208651, 4248331561, 1105500069, 2967638830, 1029380469, 1099175028,
            3684769174, 970577857, 824995756, 2937133428, 1925916100, 859262581, 1913695947,
            99275310, 848678738, 1204388593, 2545149594, 4025381218, 2655396831, 4274996806,
            785850435, 2813888154, 2491029176, 729818073, 3512685438, 2347032890, 236189106,
            1548333126, 4226775910, 3531794275, 2605168049, 2258187662, 3684015953, 898250663,
            3771349383, 4093644918, 1624210803, 4281075128, 3093735055, 1291164275, 4037665805,
            400599276, 1273249058, 2375386975, 3439029906, 3333739393, 21660248, 1230861454,
            4062325366, 4236459577, 114385419, 3486611078, 2042004563, 1720831514, 2860273565,
            1626594640, 1874700160, 2999308559, 3363764551, 3500763507, 2853826537, 2479469741,
            1210500932, 2562869107, 880924171, 3384557003, 4168144997, 1148570091, 3116312623,
            3172212634, 3567662417, 2642768241, 2015203625, 322037435, 4195745331, 482368754,
            3667466677, 4219023014, 1434757526, 2822946590, 1936561533, 2947351890, 76179171,
            2209227842, 2435463017, 276550749, 3368341828, 4262885294, 527655253, 3745940749,
            947007746, 3600866533, 4084045218, 3905637169, 702492342, 3383700150, 1661174941,
            1763804378, 961484162, 439776914, 257725152, 2840617852, 2635835773, 2403444459,
            2908195944, 2902712886, 3595015469, 1460138072, 457118198, 2120525120, 2083051232,
            2885301328, 3622650790, 400659000, 184135173, 1137972313, 759714739, 1306898817,
            2898422672, 1499186793, 2809562653, 1117081917, 3344337520, 1802186651, 3627008548,
            3860814929, 3728070085, 2991502661, 1782981707, 2477289364, 3565582296, 4061650811,
            1579851913, 3689665892, 75184626, 2690145835, 1228774440, 781089767, 3218967841,
            4064814684, 3163493596, 2809482905, 208507155, 3756966102, 1779229104, 3680138867,
            1751377038, 2598643358, 741546321, 1303997493, 1306096237, 3769872893, 3911517876,
            1315212882, 2839259600, 1408676647, 4129419272, 26525771, 1163846026, 4159566079,
            3195690749, 3064298320, 2468791883, 2538731242, 3488224592, 840135976, 493847318,
            3680037513, 4066665187, 1669269859, 29113490, 3642824198, 2180773838, 3705727774,
            3741660915, 2916119284, 4126524803, 2531904672, 3689890848, 3760318626, 2463539219,
            2074029166, 3481349310, 834750020, 3864728871, 2085171104, 1481087706, 385893521,
            1753696318, 262108731, 3110985970, 3185711468, 3592333917, 4001141364, 1777140007,
            1668227551, 3014458968, 3982753649, 216294504, 3414213843, 553573571, 4237875195,
            1780838673, 2208517053, 1405463181, 3074428565, 443193338, 3142617921, 426182091,
            3549357900, 3829054100, 3730720095, 3556916232, 3050819311, 395210557, 3290215021,
            1925989323, 1278915506, 2074504769, 788246561, 4036378728, 2546297900, 3576484297,
            881981653, 3844099771, 1100647287, 3275806264, 1545433580, 2973403693, 2010768484,
            58589600, 2055927634, 900275269, 1772160551, 3642278816, 759464205, 3337186197,
            2884437454, 1815406779, 1826092146, 1703588272, 91001350, 785570260, 4114370752,
            2690000574, 4090829518, 3792473390, 2886081610, 712723937, 3835538037, 4094897515,
            2015334309, 3367660714, 1368192432, 1580207087, 2030632060, 1542088491, 1685523972,
            4182209840, 1673405540, 4001729368, 595538569, 1903475145, 3297819296, 946281001,
            1202436672, 4111768072, 412627407, 1029652681, 2792765909, 2040861363, 2987136935,
            22966596, 3410302892, 3001821942, 1981338581, 3480779498, 1173591145, 4071007940,
            1927554934, 1430729301, 2103178216, 3055596527, 3263660591, 4266315882, 3008856523,
            1661503737, 2281231147, 4261532350, 2177588068, 1785850443, 62425813,
        ],
    );
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_invert_fail_1() {
    limbs_modular_invert(&mut [10; 3], &[], &mut [10; 3]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_invert_fail_2() {
    let ds = &[1, 2, 3];
    let mut scratch = vec![0; limbs_modular_invert_scratch_len(ds.len())];
    limbs_modular_invert(&mut [10; 2], ds, &mut scratch);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_invert_fail_3() {
    let ds = &[4, 5];
    let mut scratch = vec![0; limbs_modular_invert_scratch_len(ds.len())];
    limbs_modular_invert(&mut [10; 3], ds, &mut scratch);
}

fn verify_limbs_modular_div_mod(ns: &[Limb], ds: &[Limb], borrow: bool, qs: &[Limb], rs: &[Limb]) {
    let n = Natural::from_limbs_asc(ns);
    let d = Natural::from_limbs_asc(ds);
    let q = Natural::from_limbs_asc(qs);
    let r = Natural::from_limbs_asc(rs);
    let n_len = ns.len();
    let d_len = ds.len();
    let q_len = n_len - d_len;
    let qd = q * d;
    assert_eq!(n < qd, borrow);
    assert!(qd.eq_mod_power_of_2(&n, u64::wrapping_from(q_len) << Limb::LOG_WIDTH));
    let expected_r = (Integer::from(n) - Integer::from(qd))
        .mod_power_of_2(u64::wrapping_from(n_len) << Limb::LOG_WIDTH)
        >> (u64::wrapping_from(q_len) << Limb::LOG_WIDTH);
    assert_eq!(expected_r, r);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_modular_div_mod_schoolbook() {
    let test =
        |qs_in: &[Limb], ns_in: &[Limb], ds: &[Limb], borrow, qs_out: &[Limb], ns_out: &[Limb]| {
            let mut qs = qs_in.to_vec();
            let mut ns = ns_in.to_vec();
            let inverse = limbs_modular_invert_limb(ds[0]).wrapping_neg();
            assert_eq!(
                limbs_modular_div_mod_schoolbook(&mut qs, &mut ns, ds, inverse),
                borrow
            );
            let q_len = ns.len() - ds.len();
            assert_eq!(&qs[..q_len], qs_out);
            assert_eq!(&ns[q_len..], ns_out);
            verify_limbs_modular_div_mod(ns_in, ds, borrow, qs_out, ns_out);
        };
    // - lowest_q && limbs_slice_add_limb_in_place(&mut qs[q_diff..], 1)
    test(&[10; 3], &[0, 0, 0], &[1, 2], false, &[0], &[0, 0]);
    // - !(lowest_q && limbs_slice_add_limb_in_place(&mut qs[q_diff..], 1))
    test(&[10; 3], &[1, 2, 3], &[1, 2], false, &[1], &[0, 3]);
    test(
        &[10; 3],
        &[1, 2, 3],
        &[5, 6],
        true,
        &[3435973837],
        &[858993456, u32::MAX - 1],
    );
    test(&[10; 3], &[1, 2, 3, 4], &[1, 2, 3], false, &[1], &[0, 0, 4]);
    test(
        &[10; 3],
        &[1, 2, 3, 4],
        &[1, u32::MAX, 3],
        false,
        &[1],
        &[3, u32::MAX, 3],
    );
    test(&[10; 3], &[0, 1], &[1], false, &[0], &[1]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_mod_schoolbook_fail_1() {
    limbs_modular_div_mod_schoolbook(&mut [10; 3], &mut [1, 2, 3], &[], 1);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_mod_schoolbook_fail_2() {
    let ds = &[1, 2, 3];
    let inverse = limbs_modular_invert_limb::<Limb>(ds[0]).wrapping_neg();
    limbs_modular_div_mod_schoolbook(&mut [10; 3], &mut [1, 2, 3], ds, inverse);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_mod_schoolbook_fail_3() {
    let ds = &[1, 2];
    let inverse = limbs_modular_invert_limb::<Limb>(ds[0]).wrapping_neg();
    limbs_modular_div_mod_schoolbook(&mut [], &mut [1, 2, 3], ds, inverse);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_mod_schoolbook_fail_4() {
    let ds = &[4, 5];
    let inverse = limbs_modular_invert_limb::<Limb>(ds[0]).wrapping_neg();
    limbs_modular_div_mod_schoolbook(&mut [10; 3], &mut [1, 2, 3], ds, inverse);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_modular_div_mod_divide_and_conquer() {
    let test =
        |qs_in: &[Limb], ns_in: &[Limb], ds: &[Limb], borrow, qs_out: &[Limb], ns_out: &[Limb]| {
            let mut qs = qs_in.to_vec();
            let mut ns = ns_in.to_vec();
            let inverse = limbs_modular_invert_limb(ds[0]).wrapping_neg();
            assert_eq!(
                limbs_modular_div_mod_divide_and_conquer(&mut qs, &mut ns, ds, inverse),
                borrow
            );
            let q_len = ns.len() - ds.len();
            assert_eq!(&qs[..q_len], qs_out);
            assert_eq!(&ns[q_len..], ns_out);
            verify_limbs_modular_div_mod(ns_in, ds, borrow, qs_out, ns_out);
        };
    // - q_len <= d_len
    // - q_len < DC_BDIV_QR_THRESHOLD
    // - q_len != d_len
    test(&[10; 3], &[0, 0, 0], &[1, 2], false, &[0], &[0, 0]);
    test(&[10; 3], &[1, 2, 3], &[1, 2], false, &[1], &[0, 3]);
    test(
        &[10; 3],
        &[1, 2, 3],
        &[5, 6],
        true,
        &[3435973837],
        &[858993456, u32::MAX - 1],
    );
    test(&[10; 3], &[1, 2, 3, 4], &[1, 2, 3], false, &[1], &[0, 0, 4]);
    test(
        &[10; 3],
        &[1, 2, 3, 4],
        &[1, u32::MAX, 3],
        false,
        &[1],
        &[3, u32::MAX, 3],
    );
    // - q_len == d_len
    test(&[10; 4], &[0; 4], &[1, 0], false, &[0, 0], &[0, 0]);
    // - q_len > d_len
    // - q_len_mod_d_len < DC_BDIV_QR_THRESHOLD
    // - q_len_mod_d_len != d_len
    // - lo < DC_BDIV_QR_THRESHOLD in limbs_modular_div_mod_divide_and_conquer_helper
    // - hi < DC_BDIV_QR_THRESHOLD in limbs_modular_div_mod_divide_and_conquer_helper
    test(&[10; 5], &[0; 5], &[1, 0], false, &[0, 0, 0], &[0, 0]);
    test(
        &[10; 4],
        &[1, 2, 3, 4],
        &[5, 6],
        true,
        &[3435973837, 3607772528],
        &[4123168602, u32::MAX - 1],
    );
    test(
        &[10; 5],
        &[1, 2, 3, 4, 5],
        &[7, 8, 9],
        false,
        &[3067833783, 175304787],
        &[1051828726, 2717224210, 4],
    );
    // - q_len_mod_d_len == d_len
    test(
        &[10; 6],
        &[1, 2, 3, 4, 5, 6],
        &[7, 8],
        false,
        &[3067833783, 175304787, 3481052211, 2770888938],
        &[3602692266, 0],
    );
    // - q_len >= DC_BDIV_QR_THRESHOLD
    test(
        &[10; 111],
        &[
            238690007, 851362023, 2059930942, 863267985, 3076614342, 3047275605, 1421974480,
            393459707, 388753961, 1307811244, 1724185886, 1342073382, 3451829757, 511726014,
            460409053, 863948559, 755242682, 1610976003, 1095233439, 3983048866, 3486643678,
            1487854845, 1830823553, 2167088595, 2000609352, 3778875209, 1877274219, 3295032048,
            1477159538, 2048839778, 1451455912, 2109980052, 1750913495, 391343357, 3276142698,
            2641970441, 1571798863, 2401599044, 1467594428, 2128234517, 3438746006, 227485655,
            15222216, 256928874, 3095411638, 111550405, 2230139158, 531435393, 1299420097,
            783687022, 3432104224, 1656185186, 3410451479, 1448536986, 1411134389, 3874476101,
            2765514254, 2128294132, 3954579563, 2329369626, 3269515791, 3855344744, 1503334643,
            1671855141, 3162383277, 1731111055, 3777334519, 1174367318, 1945392580, 966689109,
            97441364, 1465255424, 1616147497, 1637231485, 1811763876, 2884209513, 3019238075,
            1328651869, 840247524, 344122022, 795760234, 289967503, 616630287, 675829910,
            2847875873, 2811380606, 2236228680, 4067593481, 2980091801, 2832971103, 2491091113,
            1521816576, 4075571575, 2543360203, 935927249, 573194417, 1822073845, 1839465906,
            3275095514,
        ],
        &[
            2515301847, 3904775757, 133728302, 3903482605, 3523183307, 1531057916, 1011111105,
            1204888196, 587552558, 2259844136, 2514432920, 3885615695, 1529240459, 2527102686,
            1527502935, 3607833032, 3108651811, 1015333242, 2650209133, 499583440, 2106087248,
            3218570602, 187468843, 2524810763, 3875472062, 3825849708, 2963555251, 2365237686,
            2035410556, 3853326592, 2366649615, 757846500, 2808712542, 1010450631, 2537107077,
            415304475, 2322781433, 1747200466, 3029815769, 2722060855, 3586580760, 2610036947,
            2383069225, 1479189021, 3488202013, 3088351671, 2313154907, 1103886425, 3716175621,
            1172908513,
        ],
        false,
        &[
            3119021825, 2664508330, 2031628331, 478764351, 1877838675, 14612131, 652799697,
            1178144318, 581220955, 1647659598, 1352407033, 2397453730, 3102518439, 248362411,
            760583238, 3183702428, 3503543471, 1633363102, 700510731, 2278494661, 3883161292,
            2875693392, 778768718, 1830608772, 4092108024, 1084731936, 2636788137, 1618403943,
            2120905188, 1304331644, 3129515489, 1701146314, 2248764147, 4096904357, 4179132986,
            1285739406, 4084818677, 3908907080, 2892808133, 3130632865, 1541722253, 754313201,
            49163159, 4048683373, 1840022468, 3666703478, 149207896, 1807686489, 529174445,
        ],
        &[
            222246482, 233545700, 3332794578, 1998197261, 3881642604, 2334420680, 457729581,
            1028684141, 1820939861, 393050552, 3364087427, 980919077, 215964517, 2941392816,
            735903426, 707494399, 1663858662, 1254976633, 2324830169, 279707711, 2388586745,
            1736265304, 1292267389, 1204103850, 1444415893, 646068967, 361893912, 1168453387,
            4162569836, 37496928, 2388108126, 4045054107, 479165313, 2859008180, 3729669518,
            790583185, 1990610890, 653432721, 4260517455, 1533264118, 4071210056, 1925491571,
            2468006372, 1418208775, 3998441757, 2911027604, 1743255069, 4051022203, 3853317804,
            3130583770,
        ],
    );
    // - q_len_mod_d_len >= DC_BDIV_QR_THRESHOLD
    test(
        &[10; 172],
        &[
            119338358, 2430782482, 2540297421, 188973918, 3420087589, 1543598052, 1550450700,
            3792047461, 889269407, 501191530, 2576798720, 2543454220, 3210031718, 485123386,
            106926418, 550954803, 939577053, 4245574922, 3198319089, 3116005666, 2750993558,
            1870348975, 1975807411, 1781898658, 487257459, 1417798881, 2833805593, 3259853663,
            1497754962, 686849710, 2609757048, 3718757183, 57671149, 3248634542, 3521889134,
            4157073590, 177642810, 2320627206, 3417728001, 301744109, 1374315478, 1156075184,
            2442293407, 1057610827, 1297359181, 1747109539, 2141404099, 4281881065, 138386125,
            3063455685, 3257890728, 3858193084, 3304890228, 238881307, 3445664438, 1185407578,
            2888653737, 993706894, 1967766053, 127412966, 2403399672, 1285486798, 908290107,
            1510442679, 4206401596, 3276759710, 11374285, 3959337953, 2236839585, 1565413293,
            2800534730, 3519885942, 3614906874, 2633795417, 3347806654, 569358328, 2189946218,
            1203262343, 2746982062, 62980684, 2586441066, 3124531637, 2381648646, 105584535,
            2941906813, 4638308, 3536520384, 446015359, 345521288, 2337847433, 1716341197,
            3434262596, 327360947, 2195093682, 1901891799, 2053064947, 1866727148, 1933602442,
            1896991511, 2434864863, 2509221232, 4146560978, 1738009642, 1186271375, 1991440446,
            388280769, 169881608, 3374872656, 4866502, 1578994123, 3894433257, 1685621086,
            3090617712, 2446643244, 1027311466, 2913067733, 415217914, 206402492, 3701056394,
            1797709466, 3451818053, 648910037, 1949755852, 3049709007, 1611096612, 271402178,
            2578204568, 481131710, 1944796007, 2365170803, 32134309, 3750170475, 1864104146,
            3460592883, 1303514890, 1133731290, 3830118297, 591382459, 3363797689, 3707044033,
            3781009395, 1183270231, 3773122255, 3243748324, 3121647216, 3297073440,
        ],
        &[
            1049478211, 680469795, 747349000, 2496192539, 3157275628, 2809929985, 1340573873,
            2369546793, 2965031816, 1069643167, 3522649492, 82508657, 3745222809, 2653443896,
            2691224863, 3638169243, 2790760129, 4050384292, 1236545914, 17769303, 3123357666,
            3308586064, 230925596, 2288243675, 526850016, 3478799166, 1359885289, 375398501,
            1996193928, 224867464, 3724522919, 3776621157, 1721777221, 1941498131, 1319934893,
            335479091, 1206826435, 1008537741, 2525891651, 3027900562, 4088218789, 1755627645,
            2151342900, 1492538444, 3337655012, 1349454, 757064511, 1793808978, 1243123906,
        ],
        false,
        &[
            3523610706, 4095718486, 1445432431, 1673437541, 4020367945, 3745015492, 3016399130,
            2189565108, 2158132813, 1848828598, 2472767424, 878094830, 2680402548, 1260943793,
            2476928489, 1177867634, 1646518182, 2946352334, 2967411271, 795570707, 2492588175,
            1099595857, 2363747444, 2197221897, 2821893422, 2883961834, 3448280420, 786023016,
            2349590684, 2639964406, 1547208182, 3946067756, 3185616768, 3182946836, 2191883192,
            714873272, 792265815, 933013306, 2758420490, 3469512009, 2749660816, 1740114201,
            1648303579, 1226176719, 4026247645, 894127522, 574096753, 1891430834, 2973532085,
            944441129, 1683821712, 681573051, 3596769123, 1196189501, 2202034777, 1230292795,
            3495428083, 3349595990, 3089151840, 3902324247, 1824847451, 783543484, 3845016445,
            1815687272, 3454243, 2400067141, 2812607721, 1192925916, 2545803688, 4259913124,
            2525790266, 1603659073, 2954188152, 3220648056, 4024945180, 1854165635, 2161271037,
            517485723, 3238095654, 89843703, 3273382586, 3798028954, 1204973927, 2456421551,
            1909094133, 2001850210, 3332598500, 2971871953, 2353788999, 1241071170, 422153499,
            3611758239, 1771812426, 1704380646, 2934602124, 2834187644, 518477462,
        ],
        &[
            266027871, 609493494, 1199784782, 1682700206, 4164056974, 2244647064, 14359526,
            1914376228, 1643693158, 449364942, 41001758, 2790190796, 1578020376, 30923228,
            2290083886, 793751679, 3951462642, 4187532857, 4144262715, 2260802306, 203413221,
            3707574023, 576327919, 33186882, 2830664220, 3895250804, 3878634772, 2739709544,
            1332976664, 1238738498, 2733771934, 2370719463, 1598734028, 1315681082, 1037147578,
            3551458133, 3406603032, 2882741411, 196631076, 833893415, 1307502306, 2956084628,
            3466863337, 682644819, 4092006006, 1963236296, 154386710, 1982810036, 3147006703,
        ],
    );
    // - lo >= DC_BDIV_QR_THRESHOLD in limbs_modular_div_mod_divide_and_conquer_helper
    // - hi >= DC_BDIV_QR_THRESHOLD in limbs_modular_div_mod_divide_and_conquer_helper
    test(
        &[10; 704],
        &[
            4185751770, 949164274, 3900519725, 1754790124, 2927570383, 1610053170, 3366859463,
            1066429707, 3501570618, 2367440639, 2328634068, 3704221588, 2322372141, 3380520853,
            4189820685, 485163788, 2195405920, 3117134592, 3277704486, 3921062791, 4106797055,
            3045508412, 661029638, 2678369724, 1467681615, 1851814983, 2960522150, 2294339994,
            489626124, 1242331300, 3385810062, 1149294828, 4088074806, 3724039365, 3770424399,
            2493469584, 806372748, 3771669403, 1111232705, 3478087568, 4029774748, 3551452116,
            3110587082, 2488974879, 47749858, 2223367929, 873250000, 2012398807, 1861118382,
            2706639555, 2880844104, 1868093092, 3743135694, 4074318903, 2930806087, 2281983526,
            1378855744, 3388340416, 3195092613, 3141388671, 3385803629, 2248154961, 716647719,
            408786065, 4189607626, 2706804476, 392451433, 843831220, 2117893506, 3763769755,
            2423989357, 1884254215, 4268634885, 1347756990, 4289680773, 1595972140, 1860322889,
            3365907154, 654256963, 3331834702, 3229143368, 1643589278, 3268087485, 3428966374,
            3734153935, 506918314, 2193830653, 1524006905, 2233546309, 2451094754, 983474066,
            1062775683, 1398440782, 617471652, 233856640, 1491439739, 194893774, 404597210,
            3349886938, 807897566, 1268209008, 3507741641, 434756682, 3067317419, 1955402334,
            3598337478, 4236902625, 3345914967, 2108259867, 4199783792, 3533374250, 690146057,
            2735070107, 3232842523, 3628050608, 139149260, 1670944793, 3256209787, 3411654393,
            1946511496, 2082461349, 2746160530, 1671886353, 2094620437, 1492520252, 3287026452,
            1499589769, 2631443030, 4265952385, 1560009101, 1665017895, 3582256582, 583800370,
            1971617104, 1393694131, 1717803931, 2118915823, 606443859, 76766530, 1028659200,
            3222396398, 491393514, 2650674330, 1420523709, 990196161, 4164467322, 955609751,
            3658315631, 3108133206, 2441918819, 119215334, 1647135728, 1151329712, 2270557882,
            2839636265, 1788114480, 953442631, 220077499, 3739679968, 289913489, 153493659,
            2215060199, 654989994, 2506760104, 3263600596, 2504124177, 3901175056, 3815693428,
            4062899928, 656080429, 1691317505, 2738892449, 2218799091, 254384726, 3472889999,
            452028418, 2323167132, 3160160509, 2790465254, 3506704767, 920776279, 1555295659,
            2088038090, 3587541031, 704460377, 41385058, 117178999, 3630755471, 2068346931,
            1737173706, 943790911, 1174415673, 2975257119, 4104946893, 563605415, 3504496607,
            3460311420, 3481281138, 2065638799, 222198878, 693087774, 2553540537, 3515067895,
            493992216, 2675820307, 1624273500, 537035998, 1538452403, 227298276, 566431973,
            3341420291, 2663602752, 849190806, 517998830, 674919319, 1319125805, 3230314450,
            889723668, 3312911398, 2040926687, 1551018220, 217350907, 1190271637, 2196470418,
            582739501, 574580675, 627521985, 3462620295, 1826546772, 1200183559, 2511667565,
            3321520722, 1760501936, 3401858303, 2056315825, 2956669134, 591282350, 2228097677,
            3953970187, 3763033846, 3981289762, 2843535951, 4035523339, 1074508115, 3233409639,
            314820046, 1519729781, 1729078105, 4201604342, 3380921403, 2184949152, 2897857848,
            212321156, 481361797, 3780255393, 2528871308, 2219779174, 3502413298, 2433932491,
            2088433048, 1119465465, 553056530, 2681509152, 1990025128, 4113937421, 757723844,
            4065055099, 1438167045, 1714067756, 295039437, 3553014346, 1500257339, 121711495,
            1299814419, 1481787451, 466462561, 3450349124, 3354633040, 4197116852, 2300067620,
            2300312798, 734816444, 668372087, 610325644, 1820480332, 312923552, 1371828336,
            386480792, 3231465004, 2119854347, 2503795939, 37544982, 2127724350, 3977310867,
            906274532, 3014798043, 1163507806, 39393504, 3203870929, 2603325424, 601940205,
            1446813021, 4008328844, 1129836361, 2242436794, 1108252363, 710998579, 3822246787,
            4245857572, 227662739, 3162273355, 2718012967, 512837928, 1146047591, 1624412382,
            4172642116, 2479791213, 2246659717, 2115301250, 144504796, 3097248580, 1488044443,
            3235358299, 310016185, 727752929, 3403555569, 1034814907, 572097587, 1799465727,
            3315327068, 1904601168, 1060041227, 95653066, 3179621384, 2784606474, 3635929233,
            3002198637,
        ],
        &[
            925120207, 4225173411, 773590203, 2705088909, 760018119, 3560986781, 3276778645,
            569879745, 275063826, 3553444079, 3591608936, 938243104, 3604596571, 1820901407,
            4022554834, 1440792481, 574650989, 3656262710, 4150032702, 1559596409, 3960538469,
            1809314494, 2152378256, 825148860, 2074258358, 2059483645, 3119970833, 3052492792,
            1083432735, 2994856903, 3287600147, 2502703055, 1752688956, 4149576868, 1709984301,
            3235915685, 487122947, 437538202, 3178502734, 1263854855, 135705911, 2843982074,
            4164166971, 4100829511, 3031022541, 3201076966, 1805387578, 1826440600, 557213027,
            2792934052, 2243352750, 1801883920, 3301321918, 469048417, 2725138727, 2947700289,
            1913798124, 574678647, 2000040094, 2336849657, 3912194680, 1616490328, 2363261019,
            3380210359, 2784609072, 835985245, 1062319853, 2519822447, 4185407058, 3253459482,
            3240288702, 3425190358, 3392304118, 3339858917, 227057671, 2117867538, 1238503500,
            1783611406, 1671436072, 4020871006, 3708283179, 1814419869, 2809845213, 402675383,
            700520900, 4021715245, 3864091313, 1886661387, 2898624118, 2266154557, 1538350152,
            4205425742, 2641391768, 3508390452, 202172288, 2270033614, 787751496, 4225843194,
            2087465714, 718749793, 4160365132, 121686135, 3232242907, 1290170166, 2696610438,
            2043913461, 2304403306,
        ],
        false,
        &[
            4221978630, 1973199499, 4118761517, 1280124308, 1169734844, 4186771056, 1549813668,
            2075241811, 3420909216, 3907043806, 2541993804, 282548280, 1975129143, 3336674855,
            4090776129, 2428071752, 3676342369, 3454430534, 316538936, 243907370, 2971927846,
            251436476, 2221794683, 2822328444, 528428056, 1993265952, 3040794479, 4050913221,
            2509817742, 2793009366, 3346235129, 4179770275, 1329386656, 3069305579, 422911051,
            232216774, 3312143191, 81270597, 779322188, 1011095209, 1055038871, 1975842925,
            1732790895, 733239904, 1557320335, 1710699531, 4221175046, 721245441, 2600219287,
            1818367711, 1883945801, 1722730504, 3667715777, 3514130820, 2079744029, 4097159692,
            8702783, 1634848318, 4003868364, 3189174634, 154328123, 3136083300, 1256793950,
            3908958390, 3995938868, 3505295673, 15682981, 1733079452, 2199765868, 1399865418,
            3931463027, 216129100, 1771409837, 4170117419, 2889309579, 1323366430, 3037738897,
            3024721374, 3698031352, 1843487639, 2640258708, 2514361590, 67538017, 3752890769,
            2377631023, 1117874819, 2034201383, 3233935115, 186520404, 1781753373, 2159589499,
            1457819895, 1955947342, 3294618852, 2583537479, 1780703679, 1295362239, 4288927730,
            3528770822, 290885828, 1022721043, 2379556752, 249111996, 1042924458, 3285072303,
            1799654769, 341780256, 2011707698, 4011728901, 4131230694, 243833190, 3438825130,
            732870287, 3977999178, 1964878050, 1772633237, 3427011489, 4149075167, 3147152113,
            4186542306, 3008212122, 2269822538, 2457396003, 2899887041, 3084745386, 2805864858,
            2636345155, 2806159180, 1448641497, 1724536998, 3602736476, 670315274, 821858587,
            3976166693, 1431302113, 994517893, 3205096696, 734083218, 684810460, 871706632,
            1619463495, 1943812284, 3232191634, 1688008439, 959671513, 861714098, 3632682856,
            4008579961, 1713155910, 621038709, 2171689707, 16974992, 1322572013, 3716365836,
            1503115330, 3655753010, 3148139022, 862756091, 2477797665, 1345732090, 2351929785,
            4244479677, 3338662555, 590731935, 3482514189, 1766212776, 1514121012, 3065000224,
            1126651654, 3402553599, 4032051831, 2997384063, 1732308544, 1185658163, 505658742,
            12068286, 191037370, 3820036984, 1808669698, 4025106072, 1258717124, 2996770408,
            104941276, 3293831485, 3303505858, 1959694112, 2104003029, 513795826, 682626035,
            1611502331, 165846820, 3906438907, 2655225469, 1304471424, 2263663116, 2896749335,
            2359126350, 2213606484, 1429710281, 2842535444, 2328001305, 1030529844, 72684272,
            3885232193, 4190971495, 1802226099, 3077502465, 1399196074, 15975559, 1608426341,
            2824610166, 3242678682, 97752959, 1141976170, 2545270306, 2934382174, 1423201353,
            4009237774, 989437497, 3792380846, 2436069063, 101138334, 3054205179, 476518326,
            45092653, 1388468856, 622609810, 3089605077, 273759253, 2522262643,
        ],
        &[
            1326262193, 4033304858, 3860284891, 3270761392, 301590389, 3160950768, 2103051311,
            421308210, 339700217, 2084975913, 4010857750, 3321647285, 2893234504, 3337934258,
            2470197482, 3426298259, 3795587565, 2533784226, 1029322143, 2753534641, 1002751731,
            1383277453, 1739180040, 4179121877, 686043541, 3404602754, 2524149226, 1961421140,
            739626289, 50422302, 4167603214, 890939538, 544526709, 333127915, 1031745192,
            2528526981, 2519624435, 2088255264, 2561443326, 3976592858, 1904453814, 37801928,
            2483368030, 3723552401, 479916426, 2605772934, 4101701491, 2731241841, 516350289,
            1540979081, 2317342783, 3699898529, 2962115900, 618662503, 360824901, 2483974833,
            3444107029, 23562940, 1137290279, 793246641, 3743331916, 1726283098, 3569199524,
            898653133, 4096842332, 1438551593, 32970915, 3426974921, 444732961, 724921875,
            1414708840, 1717003917, 1924528246, 4171817847, 3372667246, 3830123695, 927406351,
            2027580181, 1664983727, 2533132252, 1161370976, 2395800918, 1493087279, 3712645656,
            216022641, 405129823, 437352819, 2790439088, 3977337552, 1902827120, 3057912539,
            3003163486, 3875702633, 3752739451, 987987508, 2784338852, 840729416, 101169395,
            3198958489, 4287799304, 1745316888, 3524003193, 1226093819, 3034997281, 714903529,
            1035449811, 1648914672,
        ],
    );
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_mod_divide_and_conquer_fail_1() {
    limbs_modular_div_mod_divide_and_conquer(&mut [10; 3], &mut [1, 2, 3], &[], 1);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_mod_divide_and_conquer_fail_2() {
    let ds = &[1, 2, 3];
    let inverse = limbs_modular_invert_limb::<Limb>(ds[0]).wrapping_neg();
    limbs_modular_div_mod_divide_and_conquer(&mut [10; 3], &mut [1, 2, 3], ds, inverse);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_mod_divide_and_conquer_fail_3() {
    let ds = &[1, 2];
    let inverse = limbs_modular_invert_limb::<Limb>(ds[0]).wrapping_neg();
    limbs_modular_div_mod_divide_and_conquer(&mut [], &mut [1, 2, 3], ds, inverse);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_mod_divide_and_conquer_fail_4() {
    let ds = &[4, 5];
    let inverse = limbs_modular_invert_limb::<Limb>(ds[0]).wrapping_neg();
    limbs_modular_div_mod_divide_and_conquer(&mut [10; 3], &mut [1, 2, 3], ds, inverse);
}

#[test]
fn test_limbs_modular_div_mod_barrett() {
    let test = |qs_in: &[Limb],
                rs_in: &[Limb],
                ns: &[Limb],
                ds: &[Limb],
                borrow,
                qs_out: &[Limb],
                rs_out: &[Limb]| {
        let mut qs = qs_in.to_vec();
        let mut rs = rs_in.to_vec();
        let mut scratch = vec![0; limbs_modular_div_mod_barrett_scratch_len(ns.len(), ds.len())];
        assert_eq!(
            limbs_modular_div_mod_barrett(&mut qs, &mut rs, ns, ds, &mut scratch),
            borrow
        );
        let q_len = ns.len() - ds.len();
        assert_eq!(&qs[..q_len], qs_out);
        assert_eq!(&rs[..ds.len()], rs_out);
        verify_limbs_modular_div_mod(ns, ds, borrow, qs_out, rs_out);
    };
    #[cfg(feature = "32_bit_limbs")]
    {
        // - in limbs_modular_div_mod_barrett_balanced
        // - i_len < MUL_TO_MULMOD_BNM1_FOR_2NXN_THRESHOLD in limbs_modular_div_mod_barrett_balanced
        // - q_len < MUL_TO_MULMOD_BNM1_FOR_2NXN_THRESHOLD in limbs_modular_div_mod_barrett_balanced
        // - carry in limbs_modular_div_mod_barrett_balanced
        test(
            &[10; 3],
            &[10; 3],
            &[0, 0, 0, 0],
            &[1, 2],
            false,
            &[0, 0],
            &[0, 0],
        );
        test(
            &[10; 3],
            &[10; 3],
            &[1, 2, 3, 4],
            &[1, 2],
            false,
            &[1, 0],
            &[3, 4],
        );
        test(
            &[10; 3],
            &[10; 3],
            &[1, 2, 3, 4],
            &[5, 6],
            true,
            &[3435973837, 3607772528],
            &[4123168602, u32::MAX - 1],
        );
        test(
            &[10; 3],
            &[10; 3],
            &[1, 2, 3, 4, 5],
            &[1, 2, 3],
            false,
            &[1, 0],
            &[0, 4, 5],
        );
        test(
            &[10; 3],
            &[10; 3],
            &[1, 2, 3, 4, 5],
            &[1, u32::MAX, 3],
            false,
            &[1, 3],
            &[2, 0xfffffff8, 4],
        );
        // - in limbs_modular_div_mod_barrett_unbalanced
        // - q_len_s > i_len in limbs_modular_div_mod_barrett_unbalanced
        // - i_len < MUL_TO_MULMOD_BNM1_FOR_2NXN_THRESHOLD in
        //   limbs_modular_div_mod_barrett_unbalanced
        // - d_len == i_len in limbs_modular_div_mod_barrett_unbalanced
        // - q_len < MUL_TO_MULMOD_BNM1_FOR_2NXN_THRESHOLD in
        //   limbs_modular_div_mod_barrett_unbalanced
        // - d_len != q_len_s in limbs_modular_div_mod_barrett_unbalanced
        // - !carry second time in limbs_modular_div_mod_barrett_unbalanced
        test(
            &[10; 3],
            &[10; 3],
            &[1, 2, 3, 4, 5],
            &[7, 8],
            true,
            &[3067833783, 175304787, 3481052211],
            &[2216353382, u32::MAX - 1],
        );
        // - d_len == q_len_s in limbs_modular_div_mod_barrett_unbalanced
        test(
            &[10; 4],
            &[10; 3],
            &[1, 2, 3, 4, 5, 6],
            &[7, 8],
            false,
            &[3067833783, 175304787, 3481052211, 2770888938],
            &[3602692266, 0],
        );
        // - d_len != i_len in limbs_modular_div_mod_barrett_unbalanced
        // - !carry first time in limbs_modular_div_mod_barrett_unbalanced
        test(
            &[10; 25],
            &[10; 19],
            &[
                1055350223, 3049382586, 4211404571, 3876314835, 4079443157, 4038320272, 788805966,
                3784408476, 1246213405, 2888182970, 2261017929, 1076053788, 2501863035, 4127455271,
                4275509120, 2405873407, 595189898, 734988773, 472621924, 2174894187, 3899735108,
                215445128, 1522826922, 2013231118, 2572793610, 3364892170, 1420194578, 2506221487,
                667648088, 810748791, 2612101093, 2650289028, 4056901105, 2431350237, 37519423,
                2017228158, 1313697295, 2715710214, 1734284446, 2882471675, 3715454439, 1236125406,
                1443396768, 2933508364,
            ],
            &[
                3663635511, 3774569147, 1456029034, 545218027, 3883722449, 4256811234, 4280475551,
                446268340, 1439707019, 3006588287, 3015676595, 2251279938, 3776675540, 112585308,
                2726538330, 1425601246, 1583649315, 3604979549, 1473232315,
            ],
            false,
            &[
                170063657, 2165570113, 2907518307, 1430243918, 2140749125, 3558893223, 2847072687,
                4090009629, 4254439605, 2409969231, 3425274081, 4131046019, 2116325398, 2254079976,
                4252966135, 2463907465, 580945345, 1823337423, 2340467621, 2110070789, 2620914623,
                3150901076, 1253645171, 1967019766, 2932062934,
            ],
            &[
                2089766457, 292821147, 3117586049, 1036271118, 2574097064, 1232271710, 1288296201,
                2511496451, 3125977553, 2778795769, 2999224452, 1311965009, 225728039, 1754024341,
                1242425483, 2864810952, 1157382204, 2754439417, 1927770818,
            ],
        );
        // - carry second time in limbs_modular_div_mod_barrett_unbalanced
        test(
            &[10; 113],
            &[10; 7],
            &[
                1780785185, 786171578, 3157515628, 1592442002, 3380766160, 2287562230, 715389251,
                248693343, 1917127407, 3734172511, 318221247, 353404054, 770901771, 1185388775,
                1026227030, 2159266648, 4115835732, 3372528053, 3300094046, 4237524737, 3734961305,
                3971742075, 2838379294, 4275415020, 3696524563, 4195057627, 1748644269, 3938177066,
                3720834606, 3725787789, 2343496907, 2973877655, 710152359, 3397267742, 2968876692,
                2888566080, 4065287679, 2995290545, 355522630, 1240165026, 2600807130, 468290838,
                38477660, 2658554572, 2915670343, 1795675394, 278501959, 1137682730, 3997261853,
                34007068, 4160869696, 531258143, 1374216847, 1723957411, 277507095, 2287214710,
                508149326, 2515464872, 1732728407, 258712591, 2285022762, 2447676003, 1262780680,
                1199002631, 1579465944, 1049912517, 2621415467, 3593350728, 962641356, 1344046504,
                137232829, 638490247, 3421265303, 2727821003, 2855549593, 827801261, 2202020535,
                2322971763, 360280234, 840962056, 2736221025, 1257442332, 2541827007, 3410395173,
                1257508164, 315829993, 2860519251, 3307878515, 1942955455, 1060223758, 1834897098,
                2073690721, 43780026, 1410663879, 614961912, 2120543497, 2242462397, 2715797106,
                2037137824, 1831803889, 30458973, 707478969, 1483765857, 206823596, 726188538,
                4064151055, 1331924582, 2159077308, 299368864, 644097902, 1927782610, 1274113091,
                780909329, 3276588176, 2622376275, 1944908892, 1548718985, 1478198552, 167729486,
                2805305447,
            ],
            &[2712529337, 3550254038, 912173613, 3471575420, 1584706027, 4183294179, 2032726108],
            false,
            &[
                2145811881, 2853684621, 164754117, 3675200822, 3263953060, 540174416, 540344644,
                3117220624, 74272150, 2433930443, 2738006052, 2816255289, 2203418744, 911786820,
                3177797500, 1226409737, 3793898412, 403951736, 2441100417, 2633166493, 3446817232,
                345329148, 3098851660, 941805631, 3993664945, 1275632198, 508337323, 839461454,
                1530627916, 271645050, 1670009911, 3116428377, 3731487849, 2293244083, 917189423,
                1772472525, 3599863118, 187105349, 2474067907, 2795120425, 630120186, 3544350690,
                3629527923, 759676403, 199697033, 729268782, 314351667, 1979650537, 528328059,
                1369356989, 4006685949, 3314502758, 3919641191, 1499885293, 1421775739, 4058709376,
                549255670, 2433522610, 194681592, 1941767539, 3859348464, 2757354972, 2081397265,
                1728594419, 1355471243, 605531861, 3102620513, 2633037944, 2567352331, 2289344558,
                1636097668, 2472013149, 1966694756, 3680045927, 3351623316, 3953479661, 1034435973,
                2559556196, 2727941371, 996730585, 3729767953, 2471327060, 4216682881, 1386133884,
                3545016228, 2025097022, 2150272500, 2943616975, 4160651382, 4095528727, 3654689124,
                2442799685, 2211976583, 3869553045, 169899811, 1764168626, 2153322769, 2927176214,
                3196475524, 1798404841, 220326463, 216721343, 2602454439, 1608608695, 2299642725,
                685786805, 2299002959, 3664589029, 1143480179, 1493337062, 3847260732, 2243223380,
                1914898063,
            ],
            &[2336930396, 3906180555, 3405422945, 1687575114, 2055613801, 3816276157, 1899020713],
        );
        // - carry first time in limbs_modular_div_mod_barrett_unbalanced
        test(
            &[10; 210],
            &[10; 19],
            &[
                2957922040, 870247152, 910873828, 2706667305, 2006892424, 1634563813, 287970711,
                1133803930, 961388798, 2714428017, 4030274882, 3138804883, 547887122, 1780655515,
                2754780720, 2058686138, 2291641774, 1741543525, 2246044650, 933209107, 4004027900,
                3379557691, 266017900, 2676189357, 1591355793, 3361078936, 2211217877, 2921128900,
                1828179459, 1074012967, 43100052, 2836201490, 1311232425, 4155464115, 1757143636,
                1277755008, 1485841251, 3152888554, 583586772, 3806098743, 782964212, 3226263583,
                1162123370, 3059562578, 2860980942, 479696604, 2270702751, 75758186, 2277793459,
                2408711142, 2815967312, 232354809, 1421580147, 108234715, 3286460793, 841443270,
                1825464019, 3954173086, 4046395553, 1749247097, 1093808724, 3438517259, 359669614,
                601658924, 1342170297, 1428604227, 182377350, 2587642660, 4286173557, 1885226635,
                3025582699, 3966955161, 1368652640, 1973817988, 672984222, 1170853285, 4036364126,
                2637166249, 2697845360, 3566541779, 376105455, 3942586561, 2215680833, 2746461408,
                2651245592, 760249496, 3897056740, 2058897545, 1283445246, 3868587627, 2438849073,
                4047410037, 3367521161, 675390171, 775086206, 3438960005, 144798590, 3473698909,
                836037579, 2050195295, 2470811574, 1614292177, 357128906, 2374914894, 979884995,
                1299595291, 3731376453, 1227867350, 3881033174, 3084409704, 409188904, 4138760620,
                4274774539, 1698773036, 3894254462, 3016352503, 1333890087, 2095418435, 3153732455,
                2176884950, 3593733835, 3605051893, 3005410021, 3887509669, 1018183006, 2630850017,
                2235284844, 98727747, 3537572449, 1659151250, 1804358600, 886148381, 3364035538,
                3284706688, 3368127755, 613336882, 2099319807, 2400145286, 2228643956, 2297667468,
                3984708746, 1630727280, 1283416474, 2286105322, 1331586489, 375863143, 2336260394,
                379345862, 1592268783, 421930024, 791870237, 34696659, 2830390144, 1045783073,
                2327165333, 3737840283, 4280928327, 1608634581, 821500409, 2477616696, 748075940,
                2898867500, 2810718701, 680766084, 1706513434, 2559362270, 2907012739, 557079510,
                761670334, 2936564915, 2799140450, 263855902, 83724051, 3113029528, 710433489,
                63647540, 2392860382, 2849685609, 2191113984, 3344692166, 3650051957, 1283783144,
                679051425, 1874499366, 2982158115, 2233164876, 1068193514, 169526123, 124050420,
                2254129164, 3129272672, 3748280701, 2962809382, 2925481293, 3971297350, 343974506,
                1952488808, 298193824, 3111608248, 3842844698, 1134548982, 2796565349, 439974096,
                661703453, 1670489521, 1331486799, 3296145757, 3970474581, 3498762865, 838199092,
                3353634107, 1628665432, 439108122, 3585259826, 3022889712, 1940250559, 79233496,
                1912286078, 4232848205, 2352720259, 952267727, 1536903505, 39209222, 3912743975,
                3928334222, 2718037985, 1489352573, 2765707939, 598657491,
            ],
            &[
                4281128173, 3027906753, 2001755931, 2775065157, 1187751336, 596325169, 2779056912,
                170261384, 831473261, 3013867871, 518058824, 3100518656, 3459929010, 2119404241,
                1145055308, 743036299, 35681442, 2984784309, 2394592752,
            ],
            false,
            &[
                3545195480, 2830193270, 335420053, 794287831, 2741916634, 2863209122, 3474537493,
                756302532, 2390999417, 4294477763, 1189935988, 1859141045, 2587320393, 948338846,
                4082107915, 2964828974, 2707598527, 3327111998, 3063462536, 2651416679, 905396265,
                2388020551, 593710429, 449874296, 2226775585, 7979097, 1540449503, 3250233597,
                1312158539, 1991615641, 335908038, 3520510682, 2746466872, 914186511, 1932273631,
                861083548, 3035810408, 2344742126, 659982098, 2860585760, 2634369596, 2813832754,
                3952630841, 2075380109, 309440464, 1523149574, 3192186814, 999974815, 2341598038,
                3017399438, 862035203, 4054680737, 2201990712, 397181202, 1647559963, 1267523856,
                1199238444, 3731680308, 3790641117, 709354153, 2258539734, 1749483122, 2853840203,
                1578892991, 2727550048, 2884219240, 3732650659, 2280125842, 2227946823, 2743139282,
                150244596, 3009923444, 4262641312, 2587401277, 1068934326, 2162268682, 500752745,
                2478246253, 657149362, 727272162, 1138165722, 3053293624, 622175917, 1883556266,
                995622659, 3180371798, 2641139734, 636602958, 3974225725, 643270566, 3594175694,
                1815151046, 181929979, 634258268, 1988561038, 1406268435, 1255929703, 2047200162,
                327297893, 2089111137, 2465221310, 1515525444, 2651009463, 1627761087, 2816949718,
                371898069, 3623581394, 2408374459, 852283718, 743629996, 1406343073, 1288623951,
                1390517443, 3612531871, 1908188719, 20588564, 1991288974, 2134681310, 3976054820,
                2112398386, 2742074585, 2223277780, 2566567369, 4161267340, 3521909307, 2707583513,
                771440763, 2374943346, 3684262675, 3129700156, 1702004952, 3829866647, 2745914165,
                544083442, 3125630973, 3396214384, 3490241400, 3855124301, 1918091363, 2827032159,
                1870456944, 3090227886, 3351449380, 3758213391, 1013018528, 2291181284, 3221497475,
                880874956, 4157883302, 3310385525, 1659149070, 1344782497, 1930681950, 237809822,
                665463528, 2594721127, 3171089793, 17771920, 3012888524, 986806237, 1721541891,
                203322900, 2850993174, 2812942792, 3291016822, 793893217, 3551981739, 3725150196,
                3904109269, 908950595, 3445210635, 1732434801, 136778475, 4020163671, 2911572158,
                2619712085, 695817951, 2564865838, 2683591409, 4128013969, 204856464, 3938011158,
                472034484, 2799987675, 4277430554, 1053828060, 2996826767, 1330644500, 1181151222,
                787673069, 1532990849, 432834735, 4011003718, 248480909, 3771359672, 316592679,
                2847657270, 634291083, 1350943800, 1122137552, 489179352, 3994078723, 852486950,
                2165215099, 624250642, 3818338360, 4277937938, 3825066593, 1697585856, 845743932,
            ],
            &[
                1692999357, 2954880473, 990817610, 3627848867, 604557740, 926149699, 3727741567,
                705168250, 257141873, 2721173810, 2517358846, 2668361786, 1527518030, 3882089664,
                221121460, 921437873, 2477571204, 1545760534, 127126010,
            ],
        );
        // - carry in limbs_modular_div_mod_barrett_balanced
        test(
            &[10; 6],
            &[10; 8],
            &[
                1374494479, 2192068018, 2880324652, 42688047, 1794508871, 4043622026, 2888818532,
                3570449488, 135909991, 1834703233, 99771521, 2930887574, 605577774, 1011468183,
            ],
            &[
                2460992677, 1140036614, 1658868868, 366747528, 1860201147, 2709920888, 1959763964,
                1863179050,
            ],
            true,
            &[776020643, 407209981, 1979624516, 357370108, 1022326398, 2383287252],
            &[
                2671332979, 918397726, 698873472, 823603187, 3540180412, 56598308, 2177764749,
                4272553129,
            ],
        );
        // - i_len >= MUL_TO_MULMOD_BNM1_FOR_2NXN_THRESHOLD in
        //   limbs_modular_div_mod_barrett_unbalanced
        // - d_len + i_len > mul_size in limbs_modular_div_mod_barrett_unbalanced
        // - q_len >= MUL_TO_MULMOD_BNM1_FOR_2NXN_THRESHOLD in
        //   limbs_modular_div_mod_barrett_unbalanced
        // - d_len + q_len_s > mul_size in limbs_modular_div_mod_barrett_unbalanced
        test(
            &[10; 139],
            &[10; 84],
            &[
                2646628784, 618896692, 3398943402, 3264540707, 1504960063, 3906164607, 570941654,
                3580995164, 1443736701, 3590824971, 3860070111, 2945656477, 4251307330, 2402685539,
                1846137259, 1189307444, 3386646133, 136368423, 249459846, 784663226, 4146072344,
                2595677890, 733424230, 3453504236, 88571396, 749579536, 918695180, 1666773246,
                4177391250, 4175984066, 2859904653, 3320165100, 314964734, 1227587574, 2960770677,
                2399848571, 430617262, 138749172, 3397335244, 100239326, 1527324818, 3102878375,
                3902410462, 3147053920, 2748042023, 2424421132, 1052940357, 986705203, 3463928015,
                2137344177, 1023713259, 1900812309, 1091750798, 1033535614, 1704349724, 1274392095,
                4220281689, 2312523638, 2550288105, 1603152422, 2468778401, 2362990259, 1771954100,
                2226726386, 2256260198, 1919957004, 3125206342, 1705043648, 275322946, 1578653828,
                3244824726, 568760497, 2157867085, 3722151892, 982962005, 2605569069, 3011932580,
                555973834, 3850227878, 1840125026, 859902026, 3226837879, 803396279, 4091669116,
                3502825211, 2705549345, 2355778569, 2472138162, 2501828872, 1716056509, 2461082272,
                3407718600, 2048730354, 2754501598, 81705455, 2613874586, 3616438498, 743457455,
                866447928, 1549437857, 2823907456, 216297612, 4292914429, 4033666199, 449146474,
                3154061115, 452155000, 160006622, 658415472, 480293619, 2568541254, 523346876,
                1123518434, 782900648, 4045385592, 969181715, 2603815504, 1562047677, 413762781,
                164584514, 3971139354, 389543933, 3094350891, 3173761159, 3798939769, 3469659030,
                2945777042, 2994242422, 3136151730, 2486286949, 2988445086, 1260050525, 1171382725,
                1867967284, 302235847, 1042890122, 3608326628, 2275778352, 1767146951, 2944622700,
                2789083247, 4141708171, 1551586595, 3621216040, 2781452047, 1405040348, 1672588153,
                14666875, 2647122819, 3187518016, 2279001686, 1938063779, 1875081978, 4251256078,
                2572448523, 3800192906, 1289906739, 2632255896, 2152286481, 3897576094, 1708421957,
                4073852972, 3476212556, 1875655706, 3865408865, 1201794786, 4216156085, 1050577862,
                472506848, 2553269926, 840961926, 2090709561, 2675996462, 2446149597, 833756615,
                1758241617, 2274654294, 2386883401, 449558962, 1197625280, 3018484457, 1526642077,
                2525115961, 1104794510, 4179539512, 2334981542, 1704190014, 4245163918, 821915893,
                2958170015, 709350427, 728990040, 1687853788, 786779220, 3852055852, 1751071670,
                3041460477, 352348239, 1868210536, 3695502625, 830421334, 235096442, 19032492,
                3460387365, 2186466609, 161794199, 2560621452, 1382113822, 606335634, 2333561107,
                1707370336, 2209093163, 2998828734, 917854441, 1572665439, 1169967127, 912392167,
                3799501844, 3071223299, 3987504849, 3355880904, 567811128, 949461215,
            ],
            &[
                175531285, 78228064, 2801564496, 304265049, 891912354, 3315401569, 1026356408,
                3942470248, 2223627569, 408866608, 2002970320, 4044159622, 3481083399, 1214787196,
                4080970297, 855215497, 2843678654, 1991914207, 835424234, 3258509780, 1749412212,
                3777869951, 1129376847, 3137205020, 1400979858, 3150368514, 3165530001, 2207061983,
                1161753389, 803214774, 3625753632, 590750112, 259965773, 1326162274, 2653188614,
                26451591, 1562785702, 1235742790, 3331276450, 9510804, 4278241988, 3727506800,
                2781712221, 543113252, 380099502, 710412699, 905024784, 187278197, 587009259,
                1508184302, 2414129863, 4152465703, 2161022364, 4124396813, 2190629393, 906688385,
                102547773, 3659658175, 1420753856, 3400522400, 3636743612, 1520808959, 4198488051,
                2115490366, 1371548286, 1292662256, 1069684821, 2485205762, 3559685229, 2961412573,
                847483721, 410608408, 3530014382, 413744145, 1193871608, 3152782990, 1690455930,
                233301503, 547904857, 2335968082, 845227807, 2094174869, 2149532553, 2291269465,
            ],
            true,
            &[
                2546777840, 2081991539, 2509769500, 1875915406, 1905948192, 602689312, 3268538198,
                613162535, 1767316190, 3835834564, 2622706926, 3323532817, 374980976, 3111700826,
                936860640, 2413052605, 2407595201, 278465408, 3314744140, 4091514460, 2110199799,
                117453367, 1530925758, 309474103, 1279578725, 70909593, 1326551758, 2779708349,
                3712869410, 1897725706, 705261972, 1824847489, 1691398432, 3209177026, 1020273962,
                1167724900, 2567174051, 4116130680, 3208103429, 760573618, 782808697, 2745346818,
                2496095329, 2811545307, 3946467709, 2483594659, 959341815, 4137660153, 997566470,
                1730965450, 4089641998, 4267066391, 2858661809, 600039749, 1399987501, 506879518,
                1186751467, 302620911, 2812551403, 3007055905, 303564811, 4286826331, 788916190,
                2719882990, 4258512233, 1028190872, 1114024691, 3514649924, 3727249082, 841848367,
                3084474590, 1059371415, 2166678521, 1581100579, 2950580743, 1574625618, 1082144357,
                2521093167, 1048084761, 31290854, 2524741152, 808554579, 3189810605, 2992968319,
                243265975, 603472290, 2860489763, 2101153718, 2034388432, 3710181854, 111490823,
                1160951531, 1727109115, 746190508, 1539058660, 2839040461, 859789905, 1109658141,
                2693429224, 364552289, 3793037740, 2280420170, 2901284656, 1614139914, 2372237091,
                3487857543, 3345177295, 3288910636, 1737015121, 2590600286, 724573974, 197769707,
                3550689133, 2635427825, 2347968548, 3296511750, 2601097680, 1138768331, 4046279289,
                2437184198, 938543525, 1764278559, 1965551605, 514794004, 2192820822, 206655186,
                3857560265, 1879744272, 1999335571, 3017015326, 3516909563, 421663640, 3063628317,
                60413188, 359364666, 2997260220, 3206275809, 1623244265, 2111620703,
            ],
            &[
                3415280693, 292226959, 592874699, 3763036729, 1688416787, 1967588907, 1135297545,
                300526407, 488749755, 1494683603, 3165003011, 4282144535, 2476794373, 1988031229,
                913525083, 454665488, 4180669379, 3576137622, 467981361, 2124275570, 3315398077,
                2017816700, 2870440357, 2701664661, 2420523946, 2114772415, 3454473078, 3959609658,
                1750896866, 3529048766, 2526439800, 734193874, 3225577122, 1983738086, 1618614567,
                4019248755, 1615177076, 1150622424, 2861553604, 317586884, 2084839958, 109391139,
                4199849234, 407061349, 413639713, 161276082, 1826515215, 462920784, 2166692927,
                1121252881, 2339049969, 3097449182, 1750060809, 1304728762, 2057029243, 4030798175,
                3626590369, 3782070983, 103174900, 2174112267, 3202455482, 1120937734, 162805295,
                2459403827, 1438730035, 1398853228, 3980589931, 411009274, 295760150, 835365008,
                2046227916, 2834161547, 3061448231, 1463103212, 2485431474, 3580658208, 2859008315,
                864196777, 1789864771, 1239143610, 84264026, 1385174506, 806629248, 4117925861,
            ],
        );
        // - i_len >= MUL_TO_MULMOD_BNM1_FOR_2NXN_THRESHOLD in
        //   limbs_modular_div_mod_barrett_balanced
        // - d_len + i_len > mul_size in limbs_modular_div_mod_barrett_balanced
        // - q_len >= MUL_TO_MULMOD_BNM1_FOR_2NXN_THRESHOLD in
        //   limbs_modular_div_mod_barrett_balanced
        // - d_len + q_len_s > mul_size in limbs_modular_div_mod_barrett_balanced
        test(
            &[10; 101],
            &[10; 102],
            &[
                359503822, 930755077, 1384939963, 726523519, 735855137, 4230638458, 2518447149,
                3461004289, 1461342994, 3713880403, 1117118383, 2325203080, 2382674090, 2102380477,
                2949789537, 2703453484, 86306444, 2133711224, 4243514322, 2016200957, 2466844919,
                2791971265, 1678349265, 503608099, 3435266102, 4137444085, 937001532, 485586778,
                2181285770, 4154702612, 3835162797, 4047003407, 3104468273, 2789877216, 2555048214,
                312826773, 2193261799, 1490099020, 995118624, 4053513367, 342560776, 1032986831,
                1842962421, 1301135762, 1524809911, 902292378, 1239258361, 88884962, 1144616285,
                1492178819, 420992757, 519709698, 2442073331, 1884369937, 26045701, 1835589285,
                2189835185, 2439987082, 3707057329, 2082610449, 999299935, 1541780167, 1690776984,
                112340575, 1698317092, 1643561123, 3408357912, 3288699838, 2673112056, 2870952889,
                3681616841, 2531108070, 1640007944, 506062406, 3401691507, 3344023372, 853239077,
                4105040843, 2886461869, 3452820566, 1667819286, 3128286539, 2625300568, 1756006991,
                3341301070, 1308108861, 1414362786, 2458021424, 9934651, 3098138015, 3107892925,
                2498445889, 3830361110, 379544259, 2568097985, 2246190918, 2856152801, 384822350,
                1724512815, 3991713640, 3817037833, 3722412510, 2895114053, 2728589301, 19358989,
                3717195474, 1165862612, 2362092950, 828816743, 2052326574, 2217965628, 2673072144,
                324120214, 1340520493, 3213377932, 3674602689, 1059080299, 824464261, 2585132071,
                1440667471, 392712257, 3280116872, 2202169969, 398134473, 1073083450, 2311392108,
                3823195150, 3284957997, 2227733574, 1303704641, 1615356863, 3190573039, 2485032610,
                3180590299, 2042457445, 308852594, 690284549, 3598835432, 511022704, 3497853857,
                659940084, 1641894511, 783187155, 2292526793, 562512416, 1637241073, 1083958605,
                4163240058, 604721442, 3780983693, 2998951985, 2603228801, 463730318, 1015654434,
                2651982477, 4118206282, 512096013, 1080397248, 101189089, 2863689109, 354738858,
                1787339697, 2232290920, 534759005, 2119002163, 3794431325, 232343840, 2486173117,
                68089103, 2140825959, 2555962430, 2890402101, 2413815707, 1672311294, 2499213554,
                3468248781, 871538482, 2596129344, 531624012, 2504774829, 1221872721, 1208930227,
                3557575767, 3709013124, 1698819786, 4123261656, 4065987239, 1688610525, 1294770319,
                2785995745, 4198307858, 2777810428, 2269402709, 3351492035, 1694111420, 3998731057,
                2900253048, 1205651103, 396770736, 696872604, 1384642173, 3321829462, 289627277,
            ],
            &[
                2278424439, 3077751259, 1304585622, 1987828284, 3483676582, 456771916, 1922279346,
                2709541120, 2656843747, 12978767, 316240098, 3536722689, 548385414, 724002627,
                2476881616, 2166507279, 202171725, 1147560006, 1880112538, 1756777728, 1508901890,
                1237270434, 2343198647, 2028182871, 1956050418, 3185234762, 2726786236, 1226215383,
                3283321503, 3550779054, 3962623185, 2233581072, 3685362689, 2523061304, 3432992352,
                856774698, 642855332, 1539603917, 3436254526, 3438316948, 3678354762, 2885957136,
                638778530, 3855571391, 738420174, 1718985865, 3616901246, 1722694230, 1556281305,
                3363415828, 3883725652, 1233902038, 492256678, 810883596, 2933458617, 2930892677,
                3831374672, 3234208811, 4293259190, 2352386302, 3873041439, 997918588, 2424427679,
                2782564963, 1728191643, 3952822243, 238626071, 1035613084, 542308994, 2994827340,
                1113278284, 841529778, 4170703109, 3414455081, 896207174, 1979954113, 2035511039,
                85618777, 2727640652, 1344449047, 3337583835, 3623474070, 3820508255, 635242386,
                1085332469, 711665582, 1681624716, 2455915902, 2126072253, 2798572817, 1776815147,
                453484, 3038980850, 1347504630, 372275114, 403210354, 3868566120, 3631747289,
                3548627910, 3178868797, 3753938571, 150270533,
            ],
            false,
            &[
                3465160226, 1253419105, 3124808467, 751085507, 3110124395, 407173395, 2437959899,
                4040430705, 2411492979, 2914523482, 3840204125, 319435997, 3999044619, 4145842866,
                3512313567, 1166909167, 2317449476, 3701079118, 126559811, 90896448, 1455026859,
                2191225041, 3229561661, 1343784046, 2969703260, 2878543582, 273951576, 4116117584,
                781734515, 1973696951, 3647673909, 1542403693, 2180800305, 617680099, 300639543,
                1558155610, 1182878845, 3865689017, 1416306973, 198512632, 2832322478, 3504115888,
                1419290744, 3178356971, 3900449515, 199561130, 64886902, 3940732115, 3447589634,
                1832464907, 1574927461, 435919088, 4113254992, 3460547477, 513876119, 1205361386,
                3044966276, 2248267594, 3776615277, 3623228612, 3932766400, 1166892513, 291542906,
                218374345, 4239344438, 2778233019, 1936059236, 1477081525, 1164400965, 3635983219,
                4153675354, 3092149950, 937401074, 1818500930, 3870615142, 85247866, 3236546484,
                3254512410, 3962391971, 2922578976, 95454893, 3077617251, 1113587582, 2603242994,
                1807246933, 2202312525, 1989071615, 2769213474, 3776552472, 3043026852, 1891888557,
                3886288480, 2532140003, 2035601305, 2317402048, 606214956, 4160105394, 3090211730,
                1733339611, 926487350, 965806220,
            ],
            &[
                3221267380, 424988678, 3461740599, 99497455, 420198800, 867051773, 2503641458,
                2356910464, 401890558, 3995226265, 1173135914, 569117494, 4292931100, 2799167928,
                2695949014, 899794022, 3039093573, 892570103, 1937446217, 2096306730, 3092703233,
                541614723, 1040631696, 3029761757, 2957844148, 1645807647, 291224988, 2152250183,
                1826147730, 3034872166, 3287666699, 1640543162, 3730948302, 1919958117, 1401704167,
                1823651835, 2352158907, 1673084692, 4217391492, 2392397767, 1316530687, 1884365084,
                1664508764, 1242497742, 1700540913, 3262218786, 1465845111, 1581184944, 4138608221,
                3316745971, 4068716277, 1099430011, 3708438587, 2390739279, 2810494457, 2750115918,
                1906243889, 3659100604, 3033098071, 4226373414, 3983457187, 609871527, 3791925678,
                404663530, 1206366873, 3160414140, 388713457, 2942486656, 2004345881, 3595245692,
                2542300575, 2795641800, 2424368091, 424084901, 3116448977, 3602081167, 4024353039,
                1210988544, 3034081293, 957423016, 1582447550, 3269802842, 1711116416, 3752601257,
                296665184, 437398155, 285377319, 623507510, 1699111470, 132189609, 500420363,
                3184770686, 2323445041, 2548878469, 2747066033, 3915257708, 3792192004, 1406510909,
                2945427566, 210394923, 1847865361, 255836050,
            ],
        );
        // - d_len + i_len <= mul_size in limbs_modular_div_mod_barrett_balanced
        // - d_len + q_len_s <= mul_size in limbs_modular_div_mod_barrett_balanced
        test(
            &[10; 90],
            &[10; 1294],
            &rle_decode(&[
                (0, 60),
                (4294966784, 1),
                (u32::MAX, 246),
                (65535, 1),
                (0, 295),
                (3221225472, 1),
                (u32::MAX, 35),
                (31, 1),
                (0, 181),
                (4294967292, 1),
                (u32::MAX, 64),
                (7, 1),
                (0, 20),
                (4293918720, 1),
                (u32::MAX, 143),
                (4095, 1),
                (0, 225),
                (4292870144, 1),
                (u32::MAX, 106),
            ]),
            &rle_decode(&[
                (u32::MAX, 558),
                (31, 1),
                (0, 72),
                (4294967288, 1),
                (u32::MAX, 110),
                (127, 1),
                (0, 435),
                (4261412864, 1),
                (u32::MAX, 115),
            ]),
            false,
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 512, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0,
            ],
            &rle_decode(&[
                (0, 217),
                (65536, 1),
                (0, 295),
                (3221225472, 1),
                (u32::MAX, 14),
                (4294950911, 1),
                (u32::MAX, 20),
                (31, 1),
                (0, 51),
                (4096, 1),
                (0, 110),
                (4294901760, 1),
                (u32::MAX, 18),
                (4294967291, 1),
                (u32::MAX, 64),
                (7, 1),
                (0, 20),
                (4293918720, 1),
                (u32::MAX, 143),
                (4095, 1),
                (0, 187),
                (4, 1),
                (0, 37),
                (4292870144, 1),
                (u32::MAX, 76),
                (4294966783, 1),
                (u32::MAX, 29),
            ]),
        );
    }
    #[cfg(not(feature = "32_bit_limbs"))]
    {
        test(
            &[10; 595],
            &[10; 731],
            &rle_decode(&[
                (u64::MAX, 51),
                (9007199254740991, 1),
                (0, 346),
                (18446603336221196288, 1),
                (u64::MAX, 39),
                (127, 1),
                (0, 170),
                (18446743798831644672, 1),
                (u64::MAX, 380),
                (8191, 1),
                (0, 224),
                (18446744073709551488, 1),
                (u64::MAX, 110),
            ]),
            &rle_decode(&[
                (u64::MAX, 183),
                (1048575, 1),
                (0, 4),
                (18446744071562067968, 1),
                (u64::MAX, 206),
                (134217727, 1),
                (0, 130),
                (18446744072635809792, 1),
                (u64::MAX, 204),
            ]),
            false,
            &rle_decode(&[
                (1, 1),
                (0, 50),
                (18437736874454810624, 1),
                (u64::MAX, 131),
                (1048575, 1),
                (0, 4),
                (18446744071562067968, 1),
                (u64::MAX, 46),
                (18446744073709551103, 1),
                (u64::MAX, 4),
                (1048575, 1),
                (0, 125),
                (1099511627776, 1),
                (0, 4),
                (18442240474082181120, 1),
                (u64::MAX, 4),
                (4611686018427387903, 1),
                (0, 18),
                (134217728, 1),
                (0, 2),
                (140737488355328, 1),
                (0, 19),
                (18446744073172680704, 1),
                (u64::MAX, 4),
                (2199023255551, 1),
                (0, 4),
                (18444492273895866368, 1),
                (u64::MAX, 9),
                (18446744073709551487, 1),
                (u64::MAX, 8),
                (18446744073709486079, 1),
                (u64::MAX, 78),
                (18446744072635809791, 1),
                (u64::MAX, 22),
                (1152921504606846975, 1),
                (0, 5),
                (18446744073709551232, 1),
                (u64::MAX, 4),
                (786431, 1),
                (0, 4),
                (18446744073172680704, 1),
                (u64::MAX, 12),
                (281474977234943, 1),
                (0, 3),
                (8, 1),
                (17870283321406128128, 1),
                (u64::MAX, 3),
                (18446744073709535231, 1),
                (u64::MAX, 7),
            ]),
            &rle_decode(&[
                (1, 1),
                (0, 5),
                (562949953421312, 1),
                (0, 4),
                (14987979559889010688, 1),
                (u64::MAX, 2),
                (18446743798831644671, 1),
                (u64::MAX, 2),
                (383, 1),
                (0, 4),
                (18446744073709289472, 1),
                (u64::MAX, 3),
                (134217727, 1),
                (0, 4),
                (18446743798831644672, 1),
                (u64::MAX, 3),
                (137438953471, 1),
                (0, 4),
                (18446462598732840960, 1),
                (u64::MAX, 73),
                (2251799813685247, 1),
                (0, 4),
                (13835058055282163712, 1),
                (u64::MAX, 16),
                (18446744073709551614, 1),
                (u64::MAX, 1),
                (18446744073709486079, 1),
                (u64::MAX, 4),
                (536870911, 1),
                (0, 4),
                (18446742424442109952, 1),
                (u64::MAX, 4),
                (2251799813685247, 1),
                (0, 4),
                (17293822569102704640, 1),
                (u64::MAX, 7),
                (18446742974197923839, 1),
                (18446744073709551591, 1),
                (u64::MAX, 2),
                (18446744073701163007, 1),
                (2251799813685247, 1),
                (98304, 1),
                (0, 2),
                (34359738368, 1),
                (0, 1),
                (18446744073608888320, 1),
                (u64::MAX, 2),
                (18446708889337462783, 1),
                (u64::MAX, 2),
                (18446744073708503039, 1),
                (u64::MAX, 3),
                (9007199254740991, 1),
                (2147483648, 1),
                (0, 6),
                (18428729675200069632, 1),
                (u64::MAX, 3),
                (18446744073709550591, 1),
                (u64::MAX, 18),
                (72057594037927935, 1),
                (0, 5),
                (18446744073709551600, 1),
                (u64::MAX, 4),
                (16383, 1),
                (0, 8),
                (17179869184, 1),
                (0, 8),
                (8796093022208, 1),
                (0, 50),
                (64, 1),
                (0, 4),
                (18446744073709289472, 1),
                (u64::MAX, 4),
                (268435455, 1),
                (0, 10),
                (18446744073708503040, 1),
                (u64::MAX, 4),
                (2147483647, 1),
                (0, 1),
                (288230376151711744, 1),
                (0, 3),
                (8192, 1),
                (0, 18),
                (17870283321406128128, 1),
                (18446744073701163007, 1),
                (u64::MAX, 4),
                (51539607679, 1),
                (0, 4),
                (18446638520593154048, 1),
                (u64::MAX, 4),
                (72057594037927935, 1),
                (0, 3),
                (18446743936270598144, 1),
                (u64::MAX, 1),
                (511, 1),
                (0, 4),
                (18446744073708503040, 1),
                (u64::MAX, 1),
                (18446603336221196287, 1),
                (18446744073709549567, 1),
                (u64::MAX, 2),
                (18446744072635809791, 1),
                (u64::MAX, 1),
                (4194303, 1),
                (0, 2),
                (2199023255552, 1),
                (0, 7),
                (18446744073575342080, 1),
                (u64::MAX, 61),
                (17293822569102704639, 1),
                (u64::MAX, 23),
                (67108863, 1),
                (0, 4),
                (18446743661392691200, 1),
                (u64::MAX, 4),
                (844424930131967, 1),
                (0, 4),
                (17870283321406128128, 1),
                (u64::MAX, 5),
                (18446742974197923839, 1),
                (u64::MAX, 4),
                (4503599627370495, 1),
                (0, 1),
                (562949953421312, 1),
                (16384, 1),
                (0, 1),
                (13835058055282163712, 1),
                (8589934591, 1),
                (0, 1),
                (18446744073675997184, 1),
                (u64::MAX, 2),
                (18446726481523507199, 1),
                (u64::MAX, 7),
                (1073741823, 1),
                (0, 4),
                (18446744073575333888, 1),
                (u64::MAX, 2),
                (18446603336221196287, 1),
                (u64::MAX, 19),
                (536870911, 1),
                (0, 4),
                (18446741874686296064, 1),
                (u64::MAX, 4),
                (2251799813685247, 1),
                (0, 9),
                (128, 1),
                (0, 8),
                (65536, 1),
                (0, 36),
                (18446744073709551488, 1),
                (u64::MAX, 41),
                (1073741823, 1),
                (0, 22),
                (17293822569102704640, 1),
                (u64::MAX, 5),
                (383, 1),
                (0, 4),
                (18446744073708765184, 1),
                (u64::MAX, 4),
                (536870911, 1),
                (0, 12),
                (18446462598732316672, 1),
                (u64::MAX, 3),
                (18446744073709551607, 1),
                (576460752303423487, 1),
                (0, 3),
                (16384, 1),
                (0, 7),
            ]),
        );
    }
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_mod_barrett_fail_1() {
    let ns = &[1, 2, 3];
    let ds = &[3];
    let mut scratch = vec![0; limbs_modular_div_mod_barrett_scratch_len(ns.len(), ds.len())];
    limbs_modular_div_mod_barrett(&mut [10; 3], &mut [10; 3], ns, ds, &mut scratch);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_mod_barrett_fail_2() {
    let ns = &[1, 2, 3];
    let ds = &[1, 2];
    let mut scratch = vec![0; limbs_modular_div_mod_barrett_scratch_len(ns.len(), ds.len())];
    limbs_modular_div_mod_barrett(&mut [10; 3], &mut [10; 3], ns, ds, &mut scratch);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_mod_barrett_fail_3() {
    let ns = &[1, 2, 3, 4];
    let ds = &[1, 2];
    let mut scratch = vec![0; limbs_modular_div_mod_barrett_scratch_len(ns.len(), ds.len())];
    limbs_modular_div_mod_barrett(&mut [10], &mut [10; 3], ns, ds, &mut scratch);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_mod_barrett_fail_4() {
    let ns = &[1, 2, 3, 4];
    let ds = &[1, 2];
    let mut scratch = vec![0; limbs_modular_div_mod_barrett_scratch_len(ns.len(), ds.len())];
    limbs_modular_div_mod_barrett(&mut [10; 3], &mut [10], ns, ds, &mut scratch);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_mod_barrett_fail_5() {
    let ns = &[1, 2, 3, 4];
    let ds = &[4, 5];
    let mut scratch = vec![0; limbs_modular_div_mod_barrett_scratch_len(ns.len(), ds.len())];
    limbs_modular_div_mod_barrett(&mut [10; 3], &mut [10; 3], ns, ds, &mut scratch);
}

fn verify_limbs_modular_div(ns: &[Limb], ds: &[Limb], qs: &[Limb]) {
    let n = Natural::from_limbs_asc(ns);
    let d = Natural::from_limbs_asc(ds);
    let q = Natural::from_limbs_asc(qs);
    assert_eq!(
        (q * d).mod_power_of_2(u64::wrapping_from(ns.len()) << Limb::LOG_WIDTH),
        n
    );
}

fn verify_limbs_modular_div_neg(ns: &[Limb], ds: &[Limb], qs: &[Limb]) {
    let n = Natural::from_limbs_asc(ns);
    let d = Natural::from_limbs_asc(ds);
    let q = Natural::from_limbs_asc(qs);
    let p = u64::wrapping_from(ns.len()) << Limb::LOG_WIDTH;
    assert_eq!((q * d).mod_power_of_2(p).mod_power_of_2_neg(p), n);
}

#[test]
fn test_limbs_modular_div_schoolbook() {
    let test = |qs_in: &[Limb], ns_in: &[Limb], ds: &[Limb], qs_out: &[Limb]| {
        let mut qs = qs_in.to_vec();
        let mut ns = ns_in.to_vec();
        let inverse = limbs_modular_invert_limb(ds[0]).wrapping_neg();
        limbs_modular_div_schoolbook(&mut qs, &mut ns, ds, inverse);
        assert_eq!(&qs[..ns.len()], qs_out);
        verify_limbs_modular_div_neg(ns_in, ds, qs_out);

        let mut ns = ns_in.to_vec();
        limbs_modular_div_schoolbook_in_place(&mut ns, ds, inverse);
        assert_eq!(ns, qs_out);
    };
    #[cfg(feature = "32_bit_limbs")]
    {
        test(&[10; 3], &[0, 0, 0], &[1, 2], &[0, 0, 0]);
        test(
            &[10; 3],
            &[1, 2, 3],
            &[1, 2],
            &[u32::MAX, u32::MAX, 0xfffffffc],
        );
        test(
            &[10; 3],
            &[1, 2, 3],
            &[3],
            &[1431655765, u32::MAX, 0xfffffffe],
        );
        test(&[10; 3], &[1, 2, 3], &[u32::MAX], &[1, 3, 6]);
        test(
            &[10; 3],
            &[1, 2, 3],
            &[5, 6],
            &[858993459, 687194767, 893353197],
        );
        test(
            &[10; 3],
            &[1, 2, 3],
            &[1, 2, 3],
            &[u32::MAX, u32::MAX, u32::MAX],
        );
        test(
            &[10; 3],
            &[1, 2, 3],
            &[1, u32::MAX, 3],
            &[u32::MAX, 0xfffffffc, 0xfffffffd],
        );
    }
    #[cfg(not(feature = "32_bit_limbs"))]
    {
        test(
            &[10; 3],
            &[1, 2, 3],
            &[1, 2],
            &[u64::MAX, u64::MAX, 0xfffffffffffffffc],
        );
    }
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_schoolbook_fail_1() {
    limbs_modular_div_schoolbook(&mut [10; 3], &mut [1, 2, 3], &[], 1);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_schoolbook_fail_2() {
    let ds = &[1, 2, 3, 4];
    let inverse = limbs_modular_invert_limb::<Limb>(ds[0]).wrapping_neg();
    limbs_modular_div_schoolbook(&mut [10; 3], &mut [1, 2, 3], ds, inverse);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_schoolbook_fail_3() {
    let ds = &[1, 2];
    let inverse = limbs_modular_invert_limb::<Limb>(ds[0]).wrapping_neg();
    limbs_modular_div_schoolbook(&mut [10, 10], &mut [1, 2, 3], ds, inverse);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_schoolbook_fail_4() {
    let ds = &[4, 5];
    let inverse = limbs_modular_invert_limb::<Limb>(ds[0]).wrapping_neg();
    limbs_modular_div_schoolbook(&mut [10; 3], &mut [1, 2, 3], ds, inverse);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_schoolbook_in_place_fail_1() {
    limbs_modular_div_schoolbook_in_place(&mut [1, 2, 3], &[], 1);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_schoolbook_in_place_fail_2() {
    let ds = &[1, 2, 3, 4];
    let inverse = limbs_modular_invert_limb::<Limb>(ds[0]).wrapping_neg();
    limbs_modular_div_schoolbook_in_place(&mut [1, 2, 3], ds, inverse);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_schoolbook_in_place_fail_3() {
    let ds = &[4, 5];
    let inverse = limbs_modular_invert_limb::<Limb>(ds[0]).wrapping_neg();
    limbs_modular_div_schoolbook_in_place(&mut [1, 2, 3], ds, inverse);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_modular_div_divide_and_conquer() {
    let test = |qs_in: &[Limb], ns_in: &[Limb], ds: &[Limb], qs_out: &[Limb]| {
        let mut qs = qs_in.to_vec();
        let mut ns = ns_in.to_vec();
        let inverse = limbs_modular_invert_limb::<Limb>(ds[0]).wrapping_neg();
        limbs_modular_div_divide_and_conquer(&mut qs, &mut ns, ds, inverse);
        assert_eq!(&qs[..ns.len()], qs_out);
        verify_limbs_modular_div(ns_in, ds, qs_out);
    };
    // - q_len > d_len first time
    // - q_len < DC_BDIV_QR_THRESHOLD first time
    // - q_len != d_len
    test(&[10; 3], &[0, 0, 0], &[1, 2], &[0, 0, 0]);
    test(&[10; 3], &[1, 2, 3], &[1, 2], &[1, 0, 3]);
    test(
        &[10; 3],
        &[1, 2, 3],
        &[5, 6],
        &[3435973837, 3607772528, 3401614098],
    );
    // - q_len <= d_len
    // - q_len < DC_BDIV_QR_THRESHOLD second time
    test(&[10; 3], &[1, 2, 3], &[1, 2, 3], &[1, 0, 0]);
    test(&[10; 3], &[1, 2, 3], &[1, u32::MAX, 3], &[1, 3, 2]);
    // - q_len == d_len
    test(
        &[10; 4],
        &[1, 2, 3, 4],
        &[5, 6],
        &[3435973837, 3607772528, 3401614098, 2790010755],
    );
    // - q_len > d_len second time
    test(
        &[10; 5],
        &[1, 2, 3, 4, 5],
        &[7, 8],
        &[3067833783, 175304787, 3481052211, 2770888938, 2968937350],
    );
    // - q_len >= DC_BDIV_QR_THRESHOLD first time
    test(
        &[10; 101],
        &[
            2614433502, 201952551, 1985699938, 3737156448, 1826834370, 390646530, 3862902767,
            3701087729, 3240671658, 1498939256, 405421884, 2500498977, 543951667, 3836845972,
            114681939, 1915271357, 1549193218, 2855181942, 1722136441, 2766310271, 47656978,
            3837638424, 4016489648, 3574908855, 271673898, 401895301, 1488021015, 2138694572,
            2044000169, 958933265, 3144481463, 2779079775, 1326886023, 2406822210, 3541050174,
            2770735565, 2480957566, 2243398174, 2238068283, 311009614, 1882015784, 1325726230,
            997031698, 1581908377, 2605721422, 3189028102, 272460210, 490645275, 2254115078,
            1729448452, 3279102284, 4891191, 3182892978, 4279769790, 76488030, 2146767575,
            3135871036, 2565128600, 1134722432, 4021959710, 2443099174, 3511788218, 3402507407,
            3887545637, 1602118380, 3373080814, 913482070, 2525212842, 458807656, 2191742476,
            189120803, 1571244040, 2065986906, 3536519990, 1106920062, 249073982, 3953429556,
            70634017, 2295215236, 1095935313, 3576698687, 4150987084, 3091466601, 2659048520,
            2421964497, 478008822, 4183767176, 3019500765, 472981990, 4237993573, 2387724271,
            2897482399, 2735157706, 3951571091, 1100488960, 772381751, 3385833448, 243788259,
            2673522310, 2507994216, 422336110,
        ],
        &[
            1465267273, 74481649, 4134700379, 1230170156, 365676293, 3483084975, 1197662581,
            2640006295, 4011823868, 3248823520, 3487252378, 2404109212, 1035566706, 2348272566,
            1563713437, 926167746, 4275574143, 2618001622, 3006090084, 758724212, 2363503275,
            2777706618, 36668516, 1573638162, 826631123, 1003261600, 3834425037, 3917704239,
            380020162, 1200612060, 4053700768, 4029551185, 1269962948, 3135204397, 1191219015,
            2005782900, 600631609, 2788434518, 160830671, 1865310436, 2821311021, 1359155705,
            390596229, 173852699, 2808888002, 845149215, 4091564759, 2997755999, 3196096033,
            2442709791, 2497943703,
        ],
        &[
            2268505070, 3175859475, 3142537387, 3900159731, 1159682617, 4189328550, 3907703309,
            1212735193, 532303389, 2766948169, 2016978110, 2967794961, 2774851624, 3217650683,
            2925509985, 3764969541, 958893059, 3417020824, 1954191584, 3562512044, 2446134389,
            2177043907, 2722738527, 1056354433, 163366464, 2100641128, 2795771616, 981138683,
            2004843724, 421545248, 931541656, 2667221997, 1053518296, 3427347852, 943193977,
            1693724312, 497384180, 1592766692, 3814767396, 496551406, 761630525, 122738561,
            2053011267, 3134192635, 3959716126, 756419570, 4291525522, 52995454, 2355566740,
            1535344262, 3457101895, 1561710520, 629646311, 1264617650, 1304911374, 3953997387,
            101220699, 606111922, 3169531883, 119225490, 2099572326, 3662568011, 2701978464,
            2354203828, 3024227856, 812859784, 3907484219, 895862916, 2367636279, 3049608863,
            3534493707, 303599528, 235895898, 3152156937, 3282734021, 2504903737, 3347451841,
            105947633, 2602282968, 2091238035, 1697885061, 2125538933, 1538362896, 3260245263,
            4185444552, 1173878884, 860722873, 4135695975, 3027742222, 4045316121, 684087923,
            2221029460, 2175787664, 4254578535, 1093169950, 3230867467, 1521099406, 1031502141,
            3310189672, 2767979128, 3491264110,
        ],
    );
    // - n >= DC_BDIV_Q_THRESHOLD in limbs_modular_div_divide_and_conquer_helper
    // - lo < hi in limbs_modular_div_divide_and_conquer_helper
    test(
        &[10; 369],
        &[
            926256687, 2852364395, 3370590143, 2584805050, 1597100583, 529558289, 126580396,
            4262060268, 1358185452, 3688826498, 2880947234, 2448955814, 2642826080, 1444415359,
            2262174529, 4252876126, 1546829518, 1026350339, 461104688, 3572581738, 1379301297,
            1469000044, 1232904583, 1130621172, 3414532348, 2076979854, 415474734, 138424304,
            2741622037, 1356956847, 3700037198, 3056382371, 4003358430, 4114290085, 2632329664,
            2757229707, 942973054, 2844466602, 2536850025, 1562205098, 2369019562, 479441879,
            1133122146, 3033404131, 799055770, 2801165147, 333840674, 1266456054, 3826844211,
            2394399329, 1580344224, 4208515641, 3938843492, 2943006000, 3095897464, 3657696853,
            1895017685, 1174400257, 856895147, 2769847676, 2637358396, 4211838346, 2677238745,
            3116142044, 2687432725, 1110573568, 933169252, 2112520496, 1203633399, 706652780,
            1876380535, 947010911, 2311376766, 3912635689, 3738597172, 2407075297, 1150178856,
            332957826, 2061573893, 1936087771, 2146544892, 3812143813, 4006647140, 1808470881,
            3639090995, 3102762721, 3422984617, 3555513955, 3249465976, 4102963672, 1077718033,
            2477507069, 1259888894, 1319895809, 2840602832, 488930783, 280413795, 1714984325,
            906714553, 4044121361, 2834701307, 353693245, 630667692, 714081891, 689252325,
            2538920747, 1229220355, 1965412395, 2366311709, 3490987487, 4244144248, 3004878982,
            1934198881, 3439419469, 584060055, 3763445399, 1754985895, 3426861913, 3852827741,
            2905990235, 591028154, 3955620274, 2446523634, 2269181228, 2155869554, 2785380931,
            1067038528, 4226586573, 938767979, 3463341351, 729920782, 3672068894, 3277213230,
            3829428878, 78609376, 2725264455, 1334526325, 3852128883, 83033112, 3171538426,
            212483737, 1705821676, 4130869601, 985721852, 4137618486, 3647946653, 2078761135,
            2033567134, 922945141, 2842773827, 2731090851, 3725966427, 346163520, 620359799,
            2807835138, 4251526149, 3329030547, 1785319896, 1412060986, 112477766, 2463195081,
            2269604331, 1667556850, 3069673827, 2451159887, 3531154947, 859465533, 797558571,
            2402685684, 1938998432, 3367592143, 1855629490, 415564883, 1771525305, 2108262482,
            878332057, 3591804465, 742811660, 2001773684, 869241071, 3406728833, 4240156422,
            1145567809, 1952848548, 1700151067, 95712656, 1166986301, 2821041275, 1639679488,
            836739412, 159195524, 2990285642, 1385013141, 4175880469, 612922039, 172085215,
            403018883, 954336314, 686090551, 1892450280, 3747619613, 4144413437, 2234712194,
            2339662600, 2244879528, 1753875098, 1838725682, 3357280280, 2192624967, 91122545,
            4248769883, 3595273608, 1004650549, 3206549498, 3722330732, 2319766803, 3308299148,
            3660385401, 1984746697, 2537428039, 837529793, 3423459313, 3380318451, 3600361999,
            1289811131, 2814141705, 1934106458, 1264409971, 2339500232, 2513439371, 628923995,
            161098739, 97046111, 528398469, 1351706370, 2700997745, 3433492154, 3705626784,
            2281353606, 2351376214, 78178014, 4256447668, 2755024441, 1600945286, 2135856519,
            2332322194, 1336372796, 924574937, 3984145695, 806549767, 1455952508, 1602006824,
            1992184377, 175977061, 2976584306, 1501249388, 3211774572, 2177299115, 2436808205,
            1296153861, 1528208726, 2086869059, 4256091310, 3359280803, 3794379308, 2845006300,
            3370727254, 4191386263, 3868923083, 4024786060, 3356572949, 1309623451, 3404587756,
            799637471, 2539690724, 3538256419, 1789660732, 993651510, 868809315, 4070903506,
            3500550366, 3537383550, 3000705462, 2011729323, 1276336091, 3715263489, 3856442137,
            3078455161, 2394585974, 1731395403, 209285071, 4286415453, 3634201569, 2714865379,
            3923934404, 1590322431, 2827746455, 4104930505, 1849053889, 1110373541, 2098826577,
            3752273612, 1449873597, 1543844883, 3523617712, 4278529692, 3074073157, 32217677,
            2222114104, 1507299609, 4254234641, 619965639, 52325523, 2586116737, 3828937288,
            841926627, 3927959971, 1429216471, 4078179238, 4166937366, 964935962, 846542155,
            3665864344, 1340829030, 3980686425, 1463832311, 2710331305, 4041927928, 287911129,
            1092730462, 1712349041, 332408781, 721925455, 3228519043, 2399858087, 1126502586,
            1239406845, 1890441075, 3602881664, 1720394596, 1136141449, 2550451241, 2108470819,
            2381224270, 196743147, 2289453443, 3352746316, 1442822225, 195272465, 1977411238,
            619069892, 1120443118, 2367263780, 2339580211, 3244857684, 413188862, 3648382353,
            1866426486, 3275669837, 714860356, 2503398064, 3520234361, 2602261672, 142956137,
            3457786845, 1647737711, 3596429731, 98035229, 2134549817,
        ],
        &[
            288812247, 1867980581, 1784848883, 3719376405, 891732341, 4042694263, 2315971843,
            4173934812, 3724384121, 3438649023, 817688419, 3717708150, 392002651, 1406301870,
            3000223143, 3208696401, 3333344771, 2069472443, 2868745797, 2276970520, 4200672618,
            4286576728, 3178657413, 2709499760, 3528379042, 37056359, 713896147, 1464335286,
            999714577, 4261103320, 2842057702, 532870143, 318250036, 454191633, 3585978181,
            3853346096, 232386666, 4153228123, 3001669262, 2644858239, 1372523035, 1638844646,
            221021648, 2633701374, 3889749815, 3749206185, 2682504994, 1691062985, 1564838714,
            2021047227, 3653939067, 375766416, 1676298932, 1632585265, 1987642618, 3216589770,
            2139352178, 3531691061, 3135733087, 2974439634, 3240759359, 1441918073, 2963546690,
            956474789, 1010096752, 3295505885, 1556487598, 2763009589, 2181664193, 3830580006,
            567013630, 1792629848, 3395572099, 3047782274, 849031602, 3372175233, 1238500612,
            3602622896, 1224295906, 514205489, 615596822, 3650298487, 3835144940, 4230075746,
            3563666002, 2171604008, 3860998558, 4030801816, 1966076533, 534702268, 4096165465,
            118025721, 3217620626, 1281269306, 2026131743, 1083491983, 2634322198, 1319079531,
            384610271, 343671134, 1924283281, 4110082962, 1035723460, 7922039, 1864294974,
            3433501319, 612818706, 2074478411, 430050897, 347224817, 3689916278, 431159468,
            2396947295, 2216850314, 2873958193, 1052280319, 3807224736, 3366982378, 445016867,
            801857639, 2770754059, 2166713172, 2421080783, 4293281425, 3315144969, 832988729,
            3203751531, 60602646, 3435936620, 2013384606, 3375674785, 506953530, 4135169500,
            2547839556, 1670695010, 2532819097, 1595078260, 1906874739, 4140805961, 909986610,
            1450206000, 2416600358, 78210121, 2461045671, 3017469740, 1248552863, 2491666236,
            3529749845, 1306097619, 609634854, 1618830814,
        ],
        &[
            2153621097, 2032975874, 1092283530, 2105499851, 2772147020, 797232604, 4197023255,
            2316448335, 231339989, 790127543, 2827859807, 3740953002, 2953654149, 4028925052,
            890100763, 944013012, 3074009738, 2799155639, 1581696516, 421616097, 2763522161,
            460382413, 3452929191, 1491944014, 1268408485, 3706732519, 987986692, 1185168616,
            312846948, 1736211677, 4257673428, 3728427015, 3445998410, 2787875431, 2693980376,
            2270279443, 1915043797, 4081500771, 22182374, 2602808921, 4089111581, 316485891,
            1787500084, 2286785318, 1422270801, 799485298, 870038934, 2055189464, 3431348353,
            2777540456, 426645251, 1311005497, 3234663366, 540942104, 1278767054, 3141018165,
            3997602571, 3885692388, 2906279738, 1880065873, 701408818, 1624620668, 2502792336,
            27895058, 3449141668, 731396337, 2025748336, 3252039671, 2316581185, 1576067456,
            959043581, 4043904286, 3885331851, 1432462822, 3531781850, 3866289759, 1120305997,
            421733220, 2095656685, 306908537, 1829908494, 3026950391, 2753364951, 2446351196,
            3811823100, 2096821518, 2024511440, 1827913918, 540355707, 1677071333, 2189975772,
            1140077230, 2361607088, 3292807914, 1616778383, 1428264564, 4185564524, 4078951710,
            3050933070, 1900914292, 1656040826, 4169852007, 3654163666, 459884159, 1142386374,
            2092770650, 4093216198, 3297930392, 1585102456, 2240909415, 2977419294, 1424884171,
            3131984332, 3677770202, 1103210323, 3732259374, 3405359572, 1239187170, 1233825656,
            1171077559, 3166656449, 3557574007, 2517986752, 298018080, 64725485, 3157564402,
            1895722919, 3711770814, 2165944903, 4108826234, 39205698, 3297059852, 2709772591,
            3472388897, 1795079771, 1014084034, 1690969319, 188496797, 362199350, 1538080346,
            3295087220, 3035112492, 860797619, 2138464630, 893193827, 3209353152, 1892791880,
            3956061449, 2582762448, 123764390, 3853335455, 1109119369, 2810579875, 2557150328,
            194723070, 1511406773, 308580645, 1506805209, 3800717643, 1154701911, 3327639678,
            1742392333, 383847505, 808485729, 1107933974, 1843546476, 3160127489, 461019178,
            2758534633, 3136176283, 4161349001, 1122667843, 529002233, 1756067056, 610266011,
            868502089, 1924153935, 3733910978, 2089140891, 2010741130, 1699263293, 1974321295,
            2337744872, 2401421411, 152242715, 1841948095, 4080892494, 3557450617, 1648335241,
            1739529953, 1965700723, 4159737024, 1826643625, 2624044999, 2662463178, 1453688703,
            2373328295, 1284900613, 166103125, 2950850796, 224937865, 2363667655, 1395721930,
            1038140390, 2227351547, 4027728456, 2456853028, 3392407547, 3558026617, 1022960493,
            324320014, 1741363959, 665626647, 1989801844, 1039225187, 2749242260, 2891585679,
            413980454, 1421911978, 542423805, 1198826717, 3829692439, 2424493878, 1743568705,
            3904138435, 957436945, 3380949485, 3874828753, 3764048544, 2784271009, 3709257819,
            2420168014, 2258006234, 1818204898, 4293192208, 1516897791, 4231562984, 1087919881,
            3298788303, 409554742, 363790612, 1163362152, 1851768229, 3036249655, 1304207802,
            2757649751, 1345091600, 3980437826, 845106654, 1191417084, 789417113, 3069537799,
            4128120253, 1724644264, 1850762321, 440075901, 3081463549, 3296565609, 880610907,
            3604815503, 825119004, 662787316, 2065720314, 1905930059, 3158622573, 2441285185,
            609976312, 3214173289, 1972737948, 1856923900, 3881955749, 147509891, 13256522,
            480264715, 2304077206, 4079499728, 1733454240, 683301825, 2067789682, 357755971,
            4064960581, 880616108, 2567161687, 2143724680, 295233846, 885328588, 3528678059,
            2821762581, 438733817, 1651128958, 266078579, 389905061, 451514054, 1632102041,
            3121845303, 1370227908, 2165682378, 4205795758, 3678759448, 3760360877, 3616397292,
            3496735510, 2181567729, 1760479516, 2226441722, 1169335764, 1685289328, 3980766219,
            584449244, 1198456230, 3262408896, 3506319243, 3461775774, 3985269650, 4207509002,
            2496174343, 4246361431, 2525595216, 505792351, 694109202, 2532334608, 177977694,
            3590514888, 1386331403, 3322919897, 3677035627, 1748970524, 2355331384, 2803452306,
            1684779748, 431295285, 2720257929, 1349292466, 196319200, 721926885, 2699941953,
            2509363172, 2856348470, 971658703, 3158854173, 648569495, 806886112, 894065939,
            1809981633, 2207773884, 2422134450, 2395969450, 154939100, 2258690857, 1558269103,
            3746056343, 3625500344, 1655744570, 3783968316, 2684172985, 1026377946, 2668609092,
            3087350250, 1696799137, 4172271775, 2172265355, 1009592900, 3751494892, 1102570367,
            728659538, 896138123, 62660027, 2953563063, 2225405012,
        ],
    );
    // - lo == hi in limbs_modular_div_divide_and_conquer_helper
    test(
        &[10; 130],
        &[
            3959967020, 604241247, 1023896993, 1094415464, 1559137023, 2965982948, 2156029102,
            1706771649, 894030132, 45266116, 3024397174, 2788172395, 890061141, 3642398662,
            2446922996, 2040815673, 4240388236, 3527485030, 2696135743, 1863211571, 4250621960,
            2587213674, 4041734617, 3833321094, 3255910635, 1391342501, 405445752, 1166990562,
            3729755900, 3449768038, 3995921119, 41398076, 589125403, 1408294985, 2526515658,
            979087391, 1829542521, 2584013937, 86635251, 3289463217, 1129003160, 4215486047,
            2533366306, 2464115291, 3785087535, 2492911607, 1526230004, 1102360497, 3665011293,
            2884421895, 324933681, 708889365, 377966516, 1925322820, 2718776900, 3007014420,
            2694491006, 64698980, 2283839644, 3204232940, 897214742, 3983354864, 1516406610,
            4133557944, 2175029320, 609379471, 564247587, 683039961, 1804280416, 4145184207,
            4080873242, 3101363732, 4184296551, 3096748957, 1158225507, 2896147906, 1844344520,
            2528253875, 2002410714, 265589879, 3121801985, 3170049206, 1924743552, 3357216142,
            1965153305, 4252960808, 2092345429, 2837817119, 4284431393, 4293590240, 654671346,
            442937916, 1896790019, 691824410, 525764843, 1906879578, 1441648208, 101533739,
            1634254350, 3353889676, 3656414560, 1273785308, 3544127466, 3459937121, 870751892,
            1042879338, 266055738, 1555417213, 1241879745, 1883960373, 1942927191, 1838179445,
            1068713063, 2025583273, 531467322, 1876253978, 3303292405, 414426425, 1511200112,
            4106411687, 3527482414, 1961626864, 2704598906, 3866316252, 3265850188, 3581763702,
            1084651282, 3286715315, 3240494590, 2037895289,
        ],
        &[
            831013375, 20762440, 1023050464, 2893474254, 1340969251, 2523392356, 3059428058,
            4273516901, 733612744, 2348114766, 527881926, 1821776884, 342733126, 533682011,
            3273669965, 3813354237, 730319485, 3494597442, 3842303180, 2653483407, 1439592834,
            1040222095, 4178562882, 3903858398, 1448880503, 453957758, 2591332887, 1627251265,
            3845154845, 2189626142, 2509689197, 1600458471, 2116253912, 567231261, 4265711834,
            4030352409, 1804529170, 4286478946, 2730794561, 4214016930, 265468824, 1147813206,
            1770605310, 3394893197, 1654160857, 531943003, 32521814, 4040486896, 1507001215,
            3704501969, 651705156, 969516949, 24702145, 1664997108, 421558584, 372837233,
            1254072677, 531592109, 3829378883, 2113868005, 960337326, 3547746167, 1987930996,
            3779916457, 3178987148, 2246813569, 111988945, 2002604914, 923221962, 1220517471,
            3875731717, 456825253, 4034876422, 444954392, 2767163540, 2138086216, 2109913076,
            4089697693, 1989588591, 1122017014, 722640034, 2406717790, 980978606, 3956008296,
            1092718561, 396048014, 3434190160, 3721687760, 980653819, 2134907925, 1119974854,
            3412876362, 3043790819, 185604051, 2832132094, 3655349262, 2911925829, 377462795,
            2359250992, 3598561591, 1539340059, 4033418662, 2214700100, 3039512018, 3007111049,
            4215728089, 2804550486, 3578434606, 227455404, 265376688, 2221211195, 1024077580,
            4234081405, 2702795924, 1578634809, 696069720, 1613277316, 1828976876, 1070942553,
            2681872839, 2525247432, 352687185,
        ],
        &[
            2214746836, 963790571, 514240474, 3287959011, 3633281888, 331145186, 2070936355,
            1275814903, 649598638, 4048580142, 703307156, 1012761762, 1604206100, 3671704900,
            1413149070, 1684128911, 3864334051, 3517749930, 404335570, 3003402025, 300542403,
            3126191669, 3699252803, 1439364308, 4192415949, 1299197928, 1611984161, 2452019244,
            1490517426, 3629863277, 2995875193, 4190690602, 3357095237, 2125051168, 2745878491,
            133070914, 2983910851, 2032579934, 3884117502, 2707418796, 147936035, 2807568299,
            463241876, 2904771792, 974710149, 1446026448, 1012503772, 1241584448, 1218926543,
            22512568, 3540164426, 3704594926, 1675610321, 4283065258, 1692490953, 1496989584,
            1037241778, 2303280516, 3682225212, 1739214044, 602765408, 4279069132, 4056574424,
            1288231393, 2447355568, 1822142590, 2659661918, 520518691, 877252364, 1172314748,
            2143580045, 2927546685, 1702611733, 3329775201, 265548959, 2886036575, 982951504,
            1898736712, 1808101939, 1001715132, 988772812, 3415931945, 3156345141, 2590276172,
            1147813273, 1551504667, 4036713818, 2703846994, 3378891303, 3501124044, 3973617807,
            2147277907, 1948408686, 3297628811, 4200352081, 73680938, 2715396127, 809994587,
            819313279, 2649935820, 3358862368, 3764689552, 1229451982, 3158810723, 1613280893,
            2296459508, 3037083750, 2496650271, 814520391, 1890901501, 2263419075, 880319244,
            571981604, 4091474119, 4150388764, 3742405101, 2249427916, 4175082452, 1674645979,
            71011260, 2431119031, 1958098093, 3007232024, 389827510, 355733904, 3385289187,
            3022974738, 3499817070, 70324044, 1955894168,
        ],
    );
    // - q_len >= DC_BDIV_QR_THRESHOLD second time
    test(
        &[10; 136],
        &[
            1635505187, 367942507, 585747625, 3259895362, 2551559511, 2891390854, 210701094,
            253381911, 741016565, 1562764630, 647253529, 995079955, 1415112766, 1121555269,
            3264386894, 1432983665, 3084492729, 4197513843, 2107932479, 911748961, 1669179905,
            3062572274, 160542021, 1653853504, 3878824697, 4044369261, 1986777757, 3535266551,
            1866910827, 4275901664, 354208268, 1288584384, 2863182224, 1221901169, 1253657528,
            592130239, 3568722156, 2486101902, 483424561, 1575660483, 720736121, 1101685031,
            2565922629, 196632433, 1033148213, 969310472, 1995520702, 3260631554, 1562557993,
            45802047, 949951688, 3295028866, 2613836943, 4061881640, 2210607847, 14248080,
            1604642540, 3576146202, 1838134557, 4181752638, 2414843751, 3325579689, 2902708971,
            2230053926, 1716268270, 3844071486, 3423382254, 2254614074, 234521767, 3030206003,
            660556467, 4001187396, 3773254173, 2953162211, 943562247, 2352566705, 3238995051,
            2513628232, 1645194089, 4221434207, 298778061, 1395231701, 3223314097, 3370303237,
            1905903471, 1408845004, 3916893951, 3646732666, 162038877, 1235325374, 1651995328,
            4159903692, 716366148, 2592723919, 2588428010, 2841918047, 2236491801, 1439600422,
            268691456, 2330191766, 371571121, 2510936837, 1992243115, 888863283, 68281559,
            2527602368, 113695533, 2735585253, 1909150691, 1102365183, 1743094105, 2335719272,
            958887136, 4011105869, 2588799839, 630693297, 1764561812, 4102816119, 1751306966,
            3347952786, 2985949642, 2087233415, 4171910514, 4184815114, 2269683542, 2467143008,
            2240451464, 3865437579, 205000839, 537874053, 1119318572, 169965588, 2947313176,
            862816324, 3987040697, 1222400428,
        ],
        &[
            3139654297, 2963323009, 2614026787, 2578421476, 2191104461, 2624045036, 3640460632,
            1417210598, 3126930163, 1027257221, 460590703, 4066457807, 3459488871, 576203252,
            1688476041, 2466961360, 1837547601, 376220531, 4141413316, 1803893809, 1761371554,
            3478953436, 130868688, 3859844778, 2980999350, 3309259618, 1059925822, 2182828747,
            2005291310, 4234145897, 1408307661, 3344594438, 3438553526, 2363099253, 2188599822,
            3464723571, 1041189464, 1746045537, 1958228487, 3752307692, 303751983, 1642664874,
            131752811, 1640394963, 2637801305, 3516563951, 1933714212, 1737019176, 2450206053,
            503586131, 126392718, 2688144827, 1242477456, 2295392334, 466349000, 2569462004,
            1318597067, 1300781532, 1655398676, 2050075490, 3903849354, 3120503464, 171096473,
            29581654, 3447431964, 832547958, 3444142819, 3539975343, 4114704783, 521462868,
            251982313, 1371581103, 3750111566, 2076014585, 988831298, 3885431509, 2528566949,
            1332909323, 41017395, 250182185, 190467400, 477317824, 3346975325, 1469734194,
            3287617291, 1797244834, 3716306921, 3132805196, 4130477334, 931614979, 4159691979,
            2278615012, 3033894917, 953086932, 887198557, 3075690000, 189881536, 2562655208,
            2340948573, 1756155681, 1434962210, 1786798745, 2696354320, 466230606, 265495223,
            3772572252, 4060091719, 4091929587, 2602735823, 3377510257, 285339272, 1093341111,
            1082532166, 1615653819, 2657481369, 1946488895, 3281845888, 8959603, 969779240,
            3948725158, 3119648217, 4059928239, 2639117089, 715898173, 3730038209, 2531002727,
            656254619, 4082370365, 4254910891, 2266367871, 2845947517, 91542353, 988318600,
            2012804020, 2917139165, 2687206014,
        ],
        &[
            344182299, 3565607671, 1425615408, 3688127580, 932395430, 2078412097, 712144307,
            2579135290, 1670334397, 3526571898, 527889915, 1847317444, 3706898982, 1670379914,
            3028965264, 1454573587, 1233674887, 1842735039, 673704170, 2755491011, 3144215569,
            2844132685, 3976352845, 3704439681, 2871494744, 563225203, 477319433, 2394374187,
            3361238262, 4097585783, 2091409235, 2989027616, 251393985, 2038664131, 1900105723,
            2726074037, 3903048322, 601506890, 4227623429, 3732979811, 2904090093, 3911663358,
            694119437, 2312158283, 3761860359, 1226110613, 2077730233, 2543223326, 3098049773,
            3187544264, 3680092572, 3815125216, 1990113409, 382920264, 3200154083, 970587848,
            1032771863, 1792989867, 408437052, 2035811594, 2495259018, 423722523, 3044771958,
            1864896105, 2129868689, 188726050, 1810624947, 1128856008, 504663759, 221476670,
            3718864116, 2197791369, 288158522, 1522677473, 4187008757, 3299027054, 3045833372,
            1408455415, 2175274137, 54848700, 3470586256, 2721864863, 3804754388, 1239548951,
            3301368160, 982306786, 3770156486, 3880485126, 2829505820, 2147636394, 2199815955,
            3505488481, 1942842807, 1772747693, 4108779241, 2096541274, 930806315, 421709313,
            3631892223, 1621500079, 1514788551, 2479798345, 4264484840, 4155460021, 258059056,
            2718683630, 3598111667, 3304295306, 3429094861, 516349540, 2653253840, 1683537378,
            3529770422, 112780533, 2171129104, 167784124, 2755669738, 3293537875, 3629382442,
            614431521, 657960320, 376396606, 4098436288, 822064619, 226101057, 1630664330,
            10604005, 2512924628, 3388389178, 3002235032, 485512099, 1009076950, 1391753479,
            286560938, 1910351313, 2438028481,
        ],
    );
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_divide_and_conquer_fail_1() {
    let ds = &[4];
    let inverse = limbs_modular_invert_limb::<Limb>(ds[0]).wrapping_neg();
    limbs_modular_div_divide_and_conquer(&mut [10; 3], &mut [1, 2, 3], ds, inverse);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_divide_and_conquer_fail_2() {
    let ds = &[1, 2, 3, 4];
    let inverse = limbs_modular_invert_limb::<Limb>(ds[0]).wrapping_neg();
    limbs_modular_div_divide_and_conquer(&mut [10; 3], &mut [1, 2, 3], ds, inverse);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_divide_and_conquer_fail_3() {
    let ds = &[1, 2];
    let inverse = limbs_modular_invert_limb::<Limb>(ds[0]).wrapping_neg();
    limbs_modular_div_divide_and_conquer(&mut [10, 10], &mut [1, 2, 3], ds, inverse);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_divide_and_conquer_fail_4() {
    let ds = &[4, 5];
    let inverse = limbs_modular_invert_limb::<Limb>(ds[0]).wrapping_neg();
    limbs_modular_div_divide_and_conquer(&mut [10; 3], &mut [1, 2, 3], ds, inverse);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_modular_div_barrett() {
    let test = |qs_in: &[Limb], ns: &[Limb], ds: &[Limb], qs_out: &[Limb]| {
        let mut qs = qs_in.to_vec();
        let n = ns.len();
        let mut scratch = vec![0; limbs_modular_div_barrett_scratch_len(n, ds.len())];
        limbs_modular_div_barrett(&mut qs, ns, ds, &mut scratch);
        assert_eq!(&qs[..n], qs_out);
        assert_eq!(&qs[n..], &qs_in[n..]);
        verify_limbs_modular_div(ns, ds, qs_out);
    };
    // - in limbs_modular_div_barrett_same_length
    // - i_len < MUL_TO_MULMOD_BNM1_FOR_2NXN_THRESHOLD in limbs_modular_div_barrett_same_length
    test(&[10; 3], &[0, 0], &[1, 2], &[0, 0]);
    test(&[10; 3], &[1, 2], &[1, 1], &[1, 1]);
    test(&[10; 3], &[1, 2], &[1, 2], &[1, 0]);
    test(&[10; 3], &[1, 2], &[5, 6], &[3435973837, 3607772528]);
    // - in limbs_modular_div_barrett_greater
    // - !carry second time in limbs_modular_div_barrett_greater
    test(&[10; 4], &[1, 2, 3, 4], &[1, 2, 3], &[1, 0, 0, 4]);
    test(
        &[10; 4],
        &[1, 2, 3, 4],
        &[1, u32::MAX, 3],
        &[1, 3, 2, 0xfffffffa],
    );
    // - q_len > i_len in limbs_modular_div_barrett_greater
    // - i_len < MUL_TO_MULMOD_BNM1_FOR_2NXN_THRESHOLD in limbs_modular_div_barrett_greater
    // - d_len == i_len in limbs_modular_div_barrett_greater
    test(
        &[10; 5],
        &[1, 2, 3, 4, 5],
        &[7, 8],
        &[3067833783, 175304787, 3481052211, 2770888938, 2968937350],
    );
    // - d_len != i_len in limbs_modular_div_barrett_greater
    // - !carry first time in limbs_modular_div_barrett_greater
    test(
        &[10; 17],
        &[
            0,
            0xfffffff0,
            u32::MAX,
            u32::MAX,
            u32::MAX,
            u32::MAX,
            0,
            0,
            0,
            0,
            0,
            4294836224,
            u32::MAX,
            0x7ffff,
            0,
            0xfffffffc,
            u32::MAX,
        ],
        &[u32::MAX, 0xffff, 4294959104, u32::MAX, 2047, 4294966784, u32::MAX],
        &[
            0, 16, 0x100000, 4294836224, 15, 1074823164, 4294565887, 2684338192, 2282854407,
            605372416, 3137181202, 4166294690, 3073446262, 77886964, 3052808264, 3411277657,
            901277614,
        ],
    );
    // - carry first time in limbs_modular_div_barrett_greater
    test(
        &[10; 98],
        &[
            1642495634, 526967555, 2241173042, 3209550984, 1676167598, 1610173536, 3438803694,
            1164362544, 2678449286, 780912452, 2768403743, 2060815434, 3972640317, 2520341055,
            592620591, 1203487883, 983093889, 890767242, 57706568, 4116094675, 884607251,
            2877607316, 1277464658, 1436231943, 3566448321, 320221382, 3788109081, 1244065662,
            1675904837, 2386486007, 3956992595, 319964407, 202742052, 3806969795, 1515593820,
            3744019160, 2090003288, 3532996659, 1561167831, 1602116916, 624040306, 162765508,
            3520901959, 2560212238, 1836753736, 4193390434, 2748131373, 2815896923, 2183843856,
            717268028, 4057685405, 1038000620, 3031926158, 1351486738, 3081828692, 716238242,
            2133401994, 4055809448, 2426834827, 42249556, 1437875081, 2953322847, 3372525974,
            807912434, 3231438580, 1532794318, 2545826975, 3083637183, 581139768, 2339238221,
            1475394016, 3592198406, 2002134106, 444657767, 2223944228, 1052336361, 284385712,
            2716809682, 457871403, 257228149, 53492436, 3918756724, 902759153, 2804665433,
            3072525061, 1179183759, 1583887668, 2221275350, 3084717885, 3234652545, 2379337844,
            3543607479, 1728198647, 1630260072, 2895054954, 2533669171, 1060868412, 547908407,
        ],
        &[
            14618623, 1746469986, 3422287375, 4133184481, 1570537644, 4262126259, 2806378130,
            3961348569, 2005535219, 3128034624, 2629791421, 2581513752, 3943557466, 4294487356,
            2073524094, 2453727247, 3583353645, 1921128531, 3406066844, 1698548606, 1946999159,
        ],
        &[
            2734842222, 2285689435, 3313291574, 144014201, 4220859605, 2950326491, 2357784357,
            1586509685, 2832228486, 1607316550, 280781708, 4284905083, 511390653, 1180279956,
            1018877648, 2593613350, 2120941422, 3139501780, 905043699, 3767548899, 580952750,
            2178827242, 2551200564, 983629603, 884994054, 211841976, 2433402391, 1021212454,
            819127040, 1904972730, 3433697954, 3446961599, 2967793385, 184760660, 4258736086,
            2575421315, 3702982944, 1090206528, 738361131, 1464668514, 536747506, 1728476312,
            836227767, 3720425025, 1632791665, 3046719088, 3289636949, 4151602118, 3854618415,
            3416186138, 83311182, 2337871634, 3256616827, 250345345, 4152926648, 1505633136,
            1983549218, 3590016829, 2141100866, 3890424583, 3192188136, 3779057143, 3508994270,
            4056653461, 4016652469, 573993530, 498142680, 2568576590, 3123744761, 893793360,
            3037437361, 2515586057, 1419850307, 2367351324, 2046622935, 822851473, 550435958,
            1715769219, 386202898, 2986510451, 1685332339, 1548081751, 2846757228, 2600333304,
            3154407368, 2550424293, 1762563946, 251045310, 2158430157, 2056859408, 1057424536,
            2450050482, 2554796526, 3366193628, 4244612028, 1102253197, 3744549054, 521836615,
        ],
    );
    // - carry second time in limbs_modular_div_barrett_greater
    test(
        &[10; 37],
        &[
            2935526596, 3034554327, 1814238638, 2697925306, 3829736681, 2739165479, 724626090,
            604324966, 4252490674, 2065896555, 3890359639, 4148922800, 3007022887, 846622959,
            2009383236, 1908685339, 1526530451, 2415388932, 338610665, 3714252276, 2553761603,
            3643634643, 496773116, 3423491766, 1075245675, 217851812, 3421283768, 206430974,
            1384123998, 1106562315, 3884178146, 2393734976, 3612774482, 3699430895, 1007237140,
            4205715447, 842811847,
        ],
        &[
            205703789, 2117899933, 1318971308, 3978644775, 3407016997, 1105357630, 1976230147,
            4259422827, 1569463276, 1189711442, 3449406972, 1341407674, 423077568, 1572740530,
            3866269574,
        ],
        &[
            3183291732, 3833024062, 3586222897, 2197803151, 1007086642, 544727792, 1794027376,
            3492679295, 555612395, 977509785, 2253915713, 1050562685, 3689706953, 91679594,
            965381603, 1030563531, 506556705, 1163731842, 1782685644, 1032462892, 838184086,
            2712070683, 101254061, 2126604889, 1455324808, 4284267836, 3246890330, 2786100847,
            4292354129, 3612163869, 4255169090, 1705817867, 2385139426, 3288783439, 280839878,
            1415086525, 3673678347,
        ],
    );
    // - i_len >= MUL_TO_MULMOD_BNM1_FOR_2NXN_THRESHOLD in limbs_modular_div_barrett_same_length
    // - n_len + i_len > mul_size in limbs_modular_div_barrett_same_length
    test(
        &[10; 69],
        &[
            2082197907, 3463552973, 3033841621, 3293308755, 1318302421, 924962436, 4055211147,
            3066037067, 189854397, 360674525, 270528549, 1824708088, 1848896300, 2009286050,
            1723092285, 1962202393, 1691706400, 1228277495, 2169116062, 1340789312, 2688811811,
            2948615768, 3930757643, 232970699, 1674295859, 2771170263, 29024924, 4025863949,
            2350200807, 1685576833, 1697147461, 3283768071, 1970079668, 2868576138, 4067067287,
            2755619479, 1433554174, 570418866, 4193378147, 56846214, 4010333403, 99500968,
            594166316, 2103126565, 3797481093, 1835813191, 3717816485, 1427301989, 2602454977,
            2725793563, 4164737951, 1352512745, 3416783056, 1623413593, 3064676881, 1724024701,
            130815055, 1736837714, 2687511206, 1192745823, 63300791, 2476521677, 2332269618,
            139603363, 1232593938, 2211955807, 1536891080, 1717467933, 803195014,
        ],
        &[
            571177489, 4171067636, 130205627, 385002762, 3710389308, 2495334077, 2555702128,
            3776766701, 2374702786, 4018024315, 3321371413, 1939382625, 1735826724, 2610501056,
            959471237, 3348497660, 3584321092, 3397401589, 1458021574, 3693295416, 1782750542,
            2587986675, 1266401498, 4096921279, 1864600357, 3704089229, 3712721011, 2510009154,
            1625100953, 2753044802, 3523244715, 2048819457, 1999850617, 1841193902, 1889118279,
            2552563105, 37156074, 382481471, 4065787580, 572345799, 140984432, 4192479401,
            1924809764, 1459787400, 2966035238, 819250523, 1301397292, 2037545391, 1858919076,
            1830995686, 1796776602, 1566339969, 2002295704, 3651240066, 2475237044, 904307699,
            3991071672, 1874129794, 4232916985, 755030924, 3876937220, 3359171346, 2215460812,
            1717512044, 3462640226, 1416735647, 2825930507, 1273943241, 1501109004,
        ],
        &[
            3324824931, 3285501135, 2080039886, 1363439285, 4216289514, 170135702, 3339034566,
            3206256933, 2049361690, 3880142493, 2355736055, 3496015895, 269070856, 3612458995,
            3874254584, 2792128956, 3736657578, 2652084650, 3367020175, 1255915505, 1207914599,
            3301647744, 3872751597, 1044752670, 4048018730, 1158138599, 46279244, 374848909,
            3803666978, 1097388174, 4125010214, 3492963129, 3793628443, 1566031349, 1950072987,
            3983338264, 2373811991, 3106070941, 3871479066, 2857657266, 1820484444, 1815087675,
            3736927857, 3638157771, 1043782805, 1853270968, 2202072929, 2227928131, 1445478612,
            4065556003, 1988319175, 1719054009, 2654397327, 263746738, 1572373639, 349867437,
            3114446163, 1211887713, 951929852, 2733483965, 4117156940, 2770305201, 3589296114,
            83156101, 1563034919, 1982884392, 689027156, 3848414016, 2735835897,
        ],
    );
    // - i_len >= MUL_TO_MULMOD_BNM1_FOR_2NXN_THRESHOLD in limbs_modular_div_barrett_greater
    // - d_len + i_len > mul_size in limbs_modular_div_barrett_greater
    test(
        &[10; 369],
        &[
            926256687, 2852364395, 3370590143, 2584805050, 1597100583, 529558289, 126580396,
            4262060268, 1358185452, 3688826498, 2880947234, 2448955814, 2642826080, 1444415359,
            2262174529, 4252876126, 1546829518, 1026350339, 461104688, 3572581738, 1379301297,
            1469000044, 1232904583, 1130621172, 3414532348, 2076979854, 415474734, 138424304,
            2741622037, 1356956847, 3700037198, 3056382371, 4003358430, 4114290085, 2632329664,
            2757229707, 942973054, 2844466602, 2536850025, 1562205098, 2369019562, 479441879,
            1133122146, 3033404131, 799055770, 2801165147, 333840674, 1266456054, 3826844211,
            2394399329, 1580344224, 4208515641, 3938843492, 2943006000, 3095897464, 3657696853,
            1895017685, 1174400257, 856895147, 2769847676, 2637358396, 4211838346, 2677238745,
            3116142044, 2687432725, 1110573568, 933169252, 2112520496, 1203633399, 706652780,
            1876380535, 947010911, 2311376766, 3912635689, 3738597172, 2407075297, 1150178856,
            332957826, 2061573893, 1936087771, 2146544892, 3812143813, 4006647140, 1808470881,
            3639090995, 3102762721, 3422984617, 3555513955, 3249465976, 4102963672, 1077718033,
            2477507069, 1259888894, 1319895809, 2840602832, 488930783, 280413795, 1714984325,
            906714553, 4044121361, 2834701307, 353693245, 630667692, 714081891, 689252325,
            2538920747, 1229220355, 1965412395, 2366311709, 3490987487, 4244144248, 3004878982,
            1934198881, 3439419469, 584060055, 3763445399, 1754985895, 3426861913, 3852827741,
            2905990235, 591028154, 3955620274, 2446523634, 2269181228, 2155869554, 2785380931,
            1067038528, 4226586573, 938767979, 3463341351, 729920782, 3672068894, 3277213230,
            3829428878, 78609376, 2725264455, 1334526325, 3852128883, 83033112, 3171538426,
            212483737, 1705821676, 4130869601, 985721852, 4137618486, 3647946653, 2078761135,
            2033567134, 922945141, 2842773827, 2731090851, 3725966427, 346163520, 620359799,
            2807835138, 4251526149, 3329030547, 1785319896, 1412060986, 112477766, 2463195081,
            2269604331, 1667556850, 3069673827, 2451159887, 3531154947, 859465533, 797558571,
            2402685684, 1938998432, 3367592143, 1855629490, 415564883, 1771525305, 2108262482,
            878332057, 3591804465, 742811660, 2001773684, 869241071, 3406728833, 4240156422,
            1145567809, 1952848548, 1700151067, 95712656, 1166986301, 2821041275, 1639679488,
            836739412, 159195524, 2990285642, 1385013141, 4175880469, 612922039, 172085215,
            403018883, 954336314, 686090551, 1892450280, 3747619613, 4144413437, 2234712194,
            2339662600, 2244879528, 1753875098, 1838725682, 3357280280, 2192624967, 91122545,
            4248769883, 3595273608, 1004650549, 3206549498, 3722330732, 2319766803, 3308299148,
            3660385401, 1984746697, 2537428039, 837529793, 3423459313, 3380318451, 3600361999,
            1289811131, 2814141705, 1934106458, 1264409971, 2339500232, 2513439371, 628923995,
            161098739, 97046111, 528398469, 1351706370, 2700997745, 3433492154, 3705626784,
            2281353606, 2351376214, 78178014, 4256447668, 2755024441, 1600945286, 2135856519,
            2332322194, 1336372796, 924574937, 3984145695, 806549767, 1455952508, 1602006824,
            1992184377, 175977061, 2976584306, 1501249388, 3211774572, 2177299115, 2436808205,
            1296153861, 1528208726, 2086869059, 4256091310, 3359280803, 3794379308, 2845006300,
            3370727254, 4191386263, 3868923083, 4024786060, 3356572949, 1309623451, 3404587756,
            799637471, 2539690724, 3538256419, 1789660732, 993651510, 868809315, 4070903506,
            3500550366, 3537383550, 3000705462, 2011729323, 1276336091, 3715263489, 3856442137,
            3078455161, 2394585974, 1731395403, 209285071, 4286415453, 3634201569, 2714865379,
            3923934404, 1590322431, 2827746455, 4104930505, 1849053889, 1110373541, 2098826577,
            3752273612, 1449873597, 1543844883, 3523617712, 4278529692, 3074073157, 32217677,
            2222114104, 1507299609, 4254234641, 619965639, 52325523, 2586116737, 3828937288,
            841926627, 3927959971, 1429216471, 4078179238, 4166937366, 964935962, 846542155,
            3665864344, 1340829030, 3980686425, 1463832311, 2710331305, 4041927928, 287911129,
            1092730462, 1712349041, 332408781, 721925455, 3228519043, 2399858087, 1126502586,
            1239406845, 1890441075, 3602881664, 1720394596, 1136141449, 2550451241, 2108470819,
            2381224270, 196743147, 2289453443, 3352746316, 1442822225, 195272465, 1977411238,
            619069892, 1120443118, 2367263780, 2339580211, 3244857684, 413188862, 3648382353,
            1866426486, 3275669837, 714860356, 2503398064, 3520234361, 2602261672, 142956137,
            3457786845, 1647737711, 3596429731, 98035229, 2134549817,
        ],
        &[
            288812247, 1867980581, 1784848883, 3719376405, 891732341, 4042694263, 2315971843,
            4173934812, 3724384121, 3438649023, 817688419, 3717708150, 392002651, 1406301870,
            3000223143, 3208696401, 3333344771, 2069472443, 2868745797, 2276970520, 4200672618,
            4286576728, 3178657413, 2709499760, 3528379042, 37056359, 713896147, 1464335286,
            999714577, 4261103320, 2842057702, 532870143, 318250036, 454191633, 3585978181,
            3853346096, 232386666, 4153228123, 3001669262, 2644858239, 1372523035, 1638844646,
            221021648, 2633701374, 3889749815, 3749206185, 2682504994, 1691062985, 1564838714,
            2021047227, 3653939067, 375766416, 1676298932, 1632585265, 1987642618, 3216589770,
            2139352178, 3531691061, 3135733087, 2974439634, 3240759359, 1441918073, 2963546690,
            956474789, 1010096752, 3295505885, 1556487598, 2763009589, 2181664193, 3830580006,
            567013630, 1792629848, 3395572099, 3047782274, 849031602, 3372175233, 1238500612,
            3602622896, 1224295906, 514205489, 615596822, 3650298487, 3835144940, 4230075746,
            3563666002, 2171604008, 3860998558, 4030801816, 1966076533, 534702268, 4096165465,
            118025721, 3217620626, 1281269306, 2026131743, 1083491983, 2634322198, 1319079531,
            384610271, 343671134, 1924283281, 4110082962, 1035723460, 7922039, 1864294974,
            3433501319, 612818706, 2074478411, 430050897, 347224817, 3689916278, 431159468,
            2396947295, 2216850314, 2873958193, 1052280319, 3807224736, 3366982378, 445016867,
            801857639, 2770754059, 2166713172, 2421080783, 4293281425, 3315144969, 832988729,
            3203751531, 60602646, 3435936620, 2013384606, 3375674785, 506953530, 4135169500,
            2547839556, 1670695010, 2532819097, 1595078260, 1906874739, 4140805961, 909986610,
            1450206000, 2416600358, 78210121, 2461045671, 3017469740, 1248552863, 2491666236,
            3529749845, 1306097619, 609634854, 1618830814,
        ],
        &[
            2153621097, 2032975874, 1092283530, 2105499851, 2772147020, 797232604, 4197023255,
            2316448335, 231339989, 790127543, 2827859807, 3740953002, 2953654149, 4028925052,
            890100763, 944013012, 3074009738, 2799155639, 1581696516, 421616097, 2763522161,
            460382413, 3452929191, 1491944014, 1268408485, 3706732519, 987986692, 1185168616,
            312846948, 1736211677, 4257673428, 3728427015, 3445998410, 2787875431, 2693980376,
            2270279443, 1915043797, 4081500771, 22182374, 2602808921, 4089111581, 316485891,
            1787500084, 2286785318, 1422270801, 799485298, 870038934, 2055189464, 3431348353,
            2777540456, 426645251, 1311005497, 3234663366, 540942104, 1278767054, 3141018165,
            3997602571, 3885692388, 2906279738, 1880065873, 701408818, 1624620668, 2502792336,
            27895058, 3449141668, 731396337, 2025748336, 3252039671, 2316581185, 1576067456,
            959043581, 4043904286, 3885331851, 1432462822, 3531781850, 3866289759, 1120305997,
            421733220, 2095656685, 306908537, 1829908494, 3026950391, 2753364951, 2446351196,
            3811823100, 2096821518, 2024511440, 1827913918, 540355707, 1677071333, 2189975772,
            1140077230, 2361607088, 3292807914, 1616778383, 1428264564, 4185564524, 4078951710,
            3050933070, 1900914292, 1656040826, 4169852007, 3654163666, 459884159, 1142386374,
            2092770650, 4093216198, 3297930392, 1585102456, 2240909415, 2977419294, 1424884171,
            3131984332, 3677770202, 1103210323, 3732259374, 3405359572, 1239187170, 1233825656,
            1171077559, 3166656449, 3557574007, 2517986752, 298018080, 64725485, 3157564402,
            1895722919, 3711770814, 2165944903, 4108826234, 39205698, 3297059852, 2709772591,
            3472388897, 1795079771, 1014084034, 1690969319, 188496797, 362199350, 1538080346,
            3295087220, 3035112492, 860797619, 2138464630, 893193827, 3209353152, 1892791880,
            3956061449, 2582762448, 123764390, 3853335455, 1109119369, 2810579875, 2557150328,
            194723070, 1511406773, 308580645, 1506805209, 3800717643, 1154701911, 3327639678,
            1742392333, 383847505, 808485729, 1107933974, 1843546476, 3160127489, 461019178,
            2758534633, 3136176283, 4161349001, 1122667843, 529002233, 1756067056, 610266011,
            868502089, 1924153935, 3733910978, 2089140891, 2010741130, 1699263293, 1974321295,
            2337744872, 2401421411, 152242715, 1841948095, 4080892494, 3557450617, 1648335241,
            1739529953, 1965700723, 4159737024, 1826643625, 2624044999, 2662463178, 1453688703,
            2373328295, 1284900613, 166103125, 2950850796, 224937865, 2363667655, 1395721930,
            1038140390, 2227351547, 4027728456, 2456853028, 3392407547, 3558026617, 1022960493,
            324320014, 1741363959, 665626647, 1989801844, 1039225187, 2749242260, 2891585679,
            413980454, 1421911978, 542423805, 1198826717, 3829692439, 2424493878, 1743568705,
            3904138435, 957436945, 3380949485, 3874828753, 3764048544, 2784271009, 3709257819,
            2420168014, 2258006234, 1818204898, 4293192208, 1516897791, 4231562984, 1087919881,
            3298788303, 409554742, 363790612, 1163362152, 1851768229, 3036249655, 1304207802,
            2757649751, 1345091600, 3980437826, 845106654, 1191417084, 789417113, 3069537799,
            4128120253, 1724644264, 1850762321, 440075901, 3081463549, 3296565609, 880610907,
            3604815503, 825119004, 662787316, 2065720314, 1905930059, 3158622573, 2441285185,
            609976312, 3214173289, 1972737948, 1856923900, 3881955749, 147509891, 13256522,
            480264715, 2304077206, 4079499728, 1733454240, 683301825, 2067789682, 357755971,
            4064960581, 880616108, 2567161687, 2143724680, 295233846, 885328588, 3528678059,
            2821762581, 438733817, 1651128958, 266078579, 389905061, 451514054, 1632102041,
            3121845303, 1370227908, 2165682378, 4205795758, 3678759448, 3760360877, 3616397292,
            3496735510, 2181567729, 1760479516, 2226441722, 1169335764, 1685289328, 3980766219,
            584449244, 1198456230, 3262408896, 3506319243, 3461775774, 3985269650, 4207509002,
            2496174343, 4246361431, 2525595216, 505792351, 694109202, 2532334608, 177977694,
            3590514888, 1386331403, 3322919897, 3677035627, 1748970524, 2355331384, 2803452306,
            1684779748, 431295285, 2720257929, 1349292466, 196319200, 721926885, 2699941953,
            2509363172, 2856348470, 971658703, 3158854173, 648569495, 806886112, 894065939,
            1809981633, 2207773884, 2422134450, 2395969450, 154939100, 2258690857, 1558269103,
            3746056343, 3625500344, 1655744570, 3783968316, 2684172985, 1026377946, 2668609092,
            3087350250, 1696799137, 4172271775, 2172265355, 1009592900, 3751494892, 1102570367,
            728659538, 896138123, 62660027, 2953563063, 2225405012,
        ],
    );
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_barrett_fail_1() {
    let ns = &[1, 2];
    let ds = &[3];
    let mut scratch = vec![0; limbs_modular_div_barrett_scratch_len(ns.len(), ds.len())];
    limbs_modular_div_barrett(&mut [10; 3], ns, ds, &mut scratch);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_barrett_fail_2() {
    let ns = &[1];
    let ds = &[1];
    let mut scratch = vec![0; limbs_modular_div_barrett_scratch_len(ns.len(), ds.len())];
    limbs_modular_div_barrett(&mut [10; 3], ns, ds, &mut scratch);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_barrett_fail_3() {
    let ns = &[1, 2];
    let ds = &[1, 2, 3];
    let mut scratch = vec![0; limbs_modular_div_barrett_scratch_len(ns.len(), ds.len())];
    limbs_modular_div_barrett(&mut [10; 3], ns, ds, &mut scratch);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_barrett_fail_4() {
    let ns = &[1, 2, 3, 4];
    let ds = &[1, 2];
    let mut scratch = vec![0; limbs_modular_div_barrett_scratch_len(ns.len(), ds.len())];
    limbs_modular_div_barrett(&mut [10; 3], ns, ds, &mut scratch);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_barrett_fail_5() {
    let ns = &[1, 2, 3, 4];
    let ds = &[4, 5];
    let mut scratch = vec![0; limbs_modular_div_barrett_scratch_len(ns.len(), ds.len())];
    limbs_modular_div_barrett(&mut [10; 3], ns, ds, &mut scratch);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_modular_div() {
    let test = |qs_in: &[Limb], ns: &[Limb], ds: &[Limb], qs_out: &[Limb]| {
        let mut qs = qs_in.to_vec();
        let n = ns.len();
        let mut scratch = vec![0; limbs_modular_div_scratch_len(n, ds.len())];
        let mut mut_ns = ns.to_vec();
        limbs_modular_div(&mut qs, &mut mut_ns, ds, &mut scratch);
        assert_eq!(&qs[..n], qs_out);

        let mut qs = qs_in.to_vec();
        let n = ns.len();
        let mut scratch = vec![0; limbs_modular_div_ref_scratch_len(n, ds.len())];
        limbs_modular_div_ref(&mut qs, ns, ds, &mut scratch);
        assert_eq!(&qs[..n], qs_out);

        verify_limbs_modular_div(ns, ds, qs_out);
    };
    // - schoolbook branch
    test(&[10; 3], &[0, 0], &[1, 2], &[0, 0]);
    test(&[10; 3], &[1, 2, 3], &[3], &[2863311531, 0, 1]);
    test(
        &[10; 3],
        &[1, 2, 3],
        &[u32::MAX],
        &[u32::MAX, 0xfffffffc, 0xfffffff9],
    );
    test(&[10; 3], &[1, 2], &[1, 1], &[1, 1]);
    test(&[10; 3], &[1, 2], &[1, 2], &[1, 0]);
    test(&[10; 3], &[1, 2], &[5, 6], &[3435973837, 3607772528]);
    test(&[10; 4], &[1, 2, 3, 4], &[1, 2, 3], &[1, 0, 0, 4]);
    test(
        &[10; 4],
        &[1, 2, 3, 4],
        &[1, u32::MAX, 3],
        &[1, 3, 2, 0xfffffffa],
    );
    // - divide-and-conquer branch
    test(
        &[10; 940],
        &[
            1617001812, 406941400, 4290122623, 1353643808, 3101415473, 2776366000, 1481378761,
            1754925789, 2932070541, 2268599150, 3665046852, 1854806905, 1473093751, 1310246577,
            1085278892, 3448341300, 664884543, 894931028, 2986635139, 2227836107, 657956320,
            2780373025, 723708052, 3678557934, 205944971, 3639109391, 1410682902, 273368655,
            137589532, 2752044605, 212717727, 3792430292, 3119625350, 3784798586, 2990465174,
            4284837080, 1139179612, 1590711001, 1736757382, 3899481945, 3054534962, 3715303406,
            2871448595, 2613719261, 3519175216, 1742061049, 687836540, 3035159189, 2430146491,
            3643811245, 1449733326, 909516982, 4131451057, 3824921017, 1370422021, 2664108177,
            2259517283, 1110019077, 2291074089, 3844959301, 2341419713, 1983390487, 2379479711,
            3103336479, 3631463561, 4141601545, 3984814009, 2847757328, 925199080, 2677150178,
            3932146626, 2585985618, 3097165312, 624553107, 3497399511, 2463155636, 1025776088,
            3972587430, 1776770640, 1070718417, 1149088515, 4112473641, 225083923, 3371457763,
            2377652798, 2195213316, 3230992135, 2084781295, 2695452223, 2117879824, 920852431,
            3418533408, 4371131, 2567093115, 3561201955, 3500513345, 1532361440, 2619974821,
            1999143525, 1723126370, 2374780155, 3680488284, 328733252, 1032411542, 1853229941,
            2293678997, 3094028501, 2057689466, 732918339, 449235695, 2135519188, 2885118676,
            1010648119, 2686289702, 3794769615, 3150457436, 896104974, 1933155926, 3334021940,
            944686456, 2364306114, 4136346953, 4200045016, 1969878091, 2569970022, 685844738,
            2612152187, 860994125, 3220153414, 2803393155, 981612648, 3042964600, 1071555038,
            2816042187, 3876557086, 3346011246, 1146392271, 2817040744, 1510282841, 325681574,
            1630649453, 3288209101, 2072179016, 2821749271, 529631819, 4008039627, 3123605697,
            3327357221, 1207815414, 2982053815, 4147131056, 1931822734, 353985760, 2963142283,
            1073494612, 2091657110, 3979899238, 1802218645, 3796622263, 3071022976, 733595921,
            2814034738, 2643673522, 4145539201, 906706184, 1357361031, 2806820925, 924346897,
            4024033817, 4170035271, 751396103, 427807833, 1293458142, 825562756, 749911432,
            2455906530, 2373035821, 2035314458, 523324040, 74749282, 1449696708, 1969230914,
            1493171573, 1513766478, 4210281190, 1335461048, 707211493, 3186784150, 2467802616,
            2058950664, 3722342212, 2509828536, 3802882999, 2600271262, 2092219856, 4157404778,
            3400786142, 155585754, 652633073, 822626032, 3523591903, 1459099206, 2649574877,
            4113736981, 1830792451, 2727586932, 1760483902, 241823374, 2715302907, 2576249518,
            3361586004, 726705922, 3486441743, 16464902, 240807763, 1358538485, 4039078491,
            1424904458, 2071547645, 3899722875, 3750426231, 2344735245, 1643746205, 491168785,
            1384963465, 1750451847, 4078035388, 1770764243, 4052515472, 3241294450, 4243264539,
            3310019665, 2887029156, 714075326, 1226997201, 3444729396, 2832779460, 622387501,
            3969455295, 2045396250, 393223708, 226908064, 4072524790, 714484882, 1907609400,
            3766575797, 478121983, 60272670, 898422298, 32593466, 2569174567, 391827066,
            4035751597, 1908938867, 173327883, 2048138818, 230061596, 1242368902, 969448935,
            1350957584, 1329847677, 1861496005, 381477284, 2549500005, 44785600, 873742504,
            2067021143, 2991092988, 3548588862, 2442888192, 1877980812, 2914948150, 1098468146,
            675021252, 3550660795, 705365138, 874024736, 17542161, 2817527888, 3629705143,
            3195425797, 2175349263, 759180483, 625046999, 323114189, 2947844301, 3753994875,
            2150414476, 3280636325, 148566019, 3393964028, 3188177781, 3019549329, 543709989,
            2636840844, 1091743439, 2478470416, 2011975001, 1970347299, 2912238231, 715625754,
            820695691, 3922781318, 3776236498, 3442057982, 921595229, 1441760440, 161150259,
            1641485394, 1541289533, 3922947949, 3843619638, 3081238601, 3722091136, 1118176273,
            621876688, 3997694905, 2566254497, 1508842638, 2833152011, 2269341704, 3805097671,
            1910097409, 1778233534, 2629818400, 1928875119, 3415886257, 3654500721, 1162300191,
            2855569689, 2679374632, 3480632355, 1697167619, 1916977851, 522312972, 3598627948,
            1453501632, 1251971247, 1389213373, 807549034, 3088783989, 3417566420, 2280710564,
            3403195199, 2495855004, 932408531, 2331161525, 3778497616, 16187086, 3485533486,
            3495411373, 1786709186, 316453771, 1948970731, 2299432775, 3933979962, 4095089804,
            379650624, 3914831055, 441554897, 2871518068, 3616069049, 1511616572, 3359750356,
            4204007612, 2372585636, 3447071366, 2322506117, 272845013, 2251768829, 219574446,
            805900066, 1562309083, 2747023165, 625123816, 3084319726, 1766945764, 730018360,
            2396297375, 3185454773, 30578559, 2118564330, 2970233977, 3864914812, 3673919355,
            387910790, 3961065920, 1279254859, 606858626, 3226772313, 3128243714, 3607492460,
            1703562315, 2007811259, 2317297443, 3003813959, 3582624750, 2460028613, 4007959672,
            3387473962, 2321603229, 2634383176, 1530063181, 653942855, 777144175, 3383143429,
            2938927788, 152348935, 1627498490, 339085549, 1090554742, 566713397, 2625208878,
            3313094379, 3682455164, 825072458, 3450039589, 825880309, 236573086, 225561664,
            1245435637, 1331746952, 1353578371, 1858571383, 1507506405, 4248940778, 625956529,
            3841763541, 1935501192, 113898702, 1717500961, 1803749553, 3698863369, 922793844,
            3945374015, 4243233523, 4250348340, 3804428707, 375844143, 3217202690, 2958736927,
            2390653610, 3913327699, 1445233502, 2564009562, 1981762528, 4082411483, 3027869173,
            1427382679, 1184126161, 818177569, 3869201961, 3896288494, 2991466485, 3711933803,
            569576015, 1555955484, 2471128165, 1976033094, 2785392947, 2319149042, 4011978677,
            2965267321, 4102225036, 3477006125, 647674513, 2105038091, 1803927615, 783856168,
            1324871043, 3615993255, 3431877321, 3971793725, 3934927792, 2179674064, 1248922540,
            743817474, 1502564707, 803921101, 459529729, 3792678523, 2146248483, 4130547052,
            3605002956, 3012317918, 4064731802, 4197909109, 4084469190, 1701485661, 3117730821,
            972406826, 3210336796, 2256307987, 3848993871, 2176735158, 3781051143, 2680104892,
            3305191212, 1603519343, 602901276, 1617518529, 2454120969, 1268075759, 498226963,
            260513853, 1438206225, 3423412289, 2731106726, 234808220, 688404693, 1622003954,
            1671415587, 4289391631, 2811620618, 1159840463, 856151253, 3940102962, 4048700229,
            426530845, 2638052743, 3885492985, 806562813, 2558031618, 2366389147, 826787397,
            1423193924, 819474847, 1921929046, 1831867082, 3600381551, 2477872533, 1029143458,
            769714702, 2814119928, 4191052015, 1167365455, 2457269908, 975781406, 2087117506,
            69469836, 1923853303, 1135113139, 2334628021, 663459911, 3006299219, 538724612,
            3334068941, 2700983994, 1073161914, 3486899373, 1634133651, 1157990333, 1503694793,
            1267044873, 2540754456, 1496511112, 781068794, 3561760800, 3895331222, 2500281154,
            3412563576, 2764489218, 3868392337, 1074848405, 1646812850, 2199258432, 4258601309,
            661369302, 874915103, 2014299592, 1949788135, 24892497, 680092321, 146288879,
            3655059587, 1166284415, 1703237926, 4235856968, 2990205786, 3631372211, 1168518819,
            1358761778, 2444279125, 2533070212, 2255837372, 3511526532, 3043738970, 1234663270,
            2988008723, 4269697291, 4118872953, 910125556, 3167731674, 4285191140, 2460574472,
            3150156454, 1702113899, 313167013, 3377687638, 3851000034, 2336152121, 3957198339,
            1488999987, 3934439945, 654432127, 2247512052, 2811676271, 3497130291, 450665617,
            2913226868, 1369911749, 1685617502, 3084648045, 1382681328, 3265795904, 35118333,
            2698123229, 3632834581, 1779152731, 2940643214, 3013389545, 134571045, 3077109637,
            4216881108, 3149045213, 4287730821, 3037477329, 1627490387, 1543881990, 375058941,
            507557878, 2540666574, 2631980166, 487254845, 4018184736, 834259987, 493887543,
            3678837714, 4006335571, 2894197190, 444039706, 2250255290, 3658855260, 2692485260,
            3067208848, 177991403, 3168488251, 3931497670, 1064352145, 3704612442, 763514580,
            2667497882, 2999748217, 1495439415, 231547898, 355122902, 3257752995, 2277957568,
            3110120058, 857628464, 2568037394, 142755211, 815281764, 2367800630, 1224636898,
            2080180449, 1355348312, 3289292906, 2297669780, 1847374168, 2276368440, 3819739502,
            1791466417, 1102305871, 2354766268, 1338112610, 3463874583, 2021767462, 1066043808,
            2309743709, 2571990210, 3315672796, 2398059936, 365056733, 3447539115, 3746732720,
            3008661963, 3862798665, 2308467583, 1665116459, 1280978829, 3527882851, 3761190132,
            671769688, 266901941, 3154667402, 1038908368, 2515820906, 1271818580, 2962093088,
            3773575104, 2517853396, 4061725672, 3523770889, 2033488750, 2429531509, 2971035542,
            4238011901, 1000531442, 1416882400, 678786363, 1019864239, 2751310982, 442339976,
            55517395, 82583326, 1942516957, 3947856921, 722940461, 728206370, 1744336133,
            2287954803, 292238920, 3911418046, 2155661057, 4159957990, 887247601, 556462431,
            1016030816, 1512907682, 3069626937, 1472354633, 3141549111, 2016546891, 2821933580,
            2770752073, 563151583, 221583615, 289067957, 3908737670, 2004753320, 3931101403,
            883429785, 3689435720, 2987821650, 28050592, 2735311860, 3265507995, 2758493314,
            2655530609, 2545294868, 2470465036, 1271710983, 545536994, 1327776999, 141840027,
            1138841565, 1781544782, 3199919479, 681382208, 1245717786, 2930809728, 3280849083,
            3956551507, 3355385777, 355999081, 2179057718, 882649569, 1825265258, 1442192351,
            3389922361, 1461142241, 1668273192, 1024455283, 1363451707, 3165776423, 1708618405,
            3593748137, 553996297, 2695393796, 1317090480, 2473638063, 3722258345, 2520027887,
            2083034437, 1575395416, 3095210370, 3995459958, 3484130891, 803033008, 9632350,
            3554497809, 3029973822, 706965453, 2956683625, 3768697729, 2368889001, 3579687116,
            1507088502, 44440509, 3722401618, 10979986, 1440335280, 360127666, 3537309717,
            2881783860, 821182819, 2620792872, 1352001982, 4178426943, 4092532482, 2378352391,
            2593011994, 1487230079, 1280656126, 2550113868, 292018470, 3077472516, 3307900686,
            3895849291, 1947913132, 636104659, 231356996, 3069426222, 1616111446, 535734499,
            4152792010, 3810588916, 594013155, 3342644263, 3876779199, 2365417389, 2541878010,
            631827463, 899645198, 1215887929, 2257594209, 3558848715, 3595018839, 2467717482,
            368943385, 202032628, 353961970, 189404791, 2740031059, 1445247256, 3111695158,
            2430510723, 1636469966, 3085197240, 393496208, 1147559660, 3854034067, 144621988,
            901901079, 1157982037, 3309406483, 881900034, 1419066224, 2147885124, 1202171837,
            4028375088, 344523185, 4281695498, 2098200318, 4285280344, 830500235, 2819516821,
            2402660095, 3347692646, 429942249, 118042706, 2413131752, 1801992999, 1419387777,
            917304148, 2405971626, 2514112112, 1486885522, 3561350045, 1391269566, 2312342833,
            2165644886, 3237357298, 529810310, 533161912, 573572810, 2388955822, 1839462779,
            3595785519, 502068002, 1370445852, 2940004806, 3127248122, 402923598, 942165049,
            1322781936, 2267106055, 963050643, 2271136014, 593519182, 997148741, 1007096690,
            2862595233, 2455777215, 87788237, 2537341695, 213203205, 3618812433, 934598945,
            2019611411, 152516526, 2045476718, 4203041946, 1700636518, 2634288824, 957703279,
            438300965, 752868345, 1741551695, 3299805511, 2787468423, 3915443541, 1822431092,
            4184623349, 470943777, 3306758738, 2378813334, 887946583, 3895139974, 3274730894,
            1775157124, 1716784797, 3766822546, 515029397, 366876429, 3967551838, 2567429723,
            1242834401, 3586292863,
        ],
        &[
            3316717475, 3755784115, 621265671, 2021086862, 474517724, 1164498167, 2627801958,
            2170528742, 3104726388, 2515033406, 1702293156, 20581773, 1429119997, 2801585517,
            3698032538, 596509909, 4238480986, 3843806649, 3322888890, 2861086342, 2626735834,
            3405303428, 1508735314, 2650558814, 76337057, 1202565180, 2949817513, 3758629286,
            1042329346, 3168950269, 883013312, 3720930735, 2244494029, 489190350, 4140126449,
            2460687698, 888181262, 1751014525, 1555755626, 2244055025, 763172867, 3257618646,
            345848605, 3619012015, 2273117624, 135239042, 1894337682, 2498817131, 351760917,
            1230693073, 3288131984, 3614737212, 1133719825, 185344371, 2675805123, 1402914899,
            3191767840, 3841311369, 2430249603, 3911615234, 2851144250, 2899717103, 2663597034,
            1253013791, 851947363, 3119663958, 3884604595, 2250953966, 2832866738, 1236465462,
            223567620, 1066146269, 3608374122, 1320851988, 3253186631, 2553947674, 1872191801,
            516096295, 1195634624, 1449552529, 3856722883, 885977930, 3490641956, 2826234306,
            3220379648, 3128438151, 1512748001, 655293586, 3885072026, 2836395257, 2848550792,
            2316139958, 4224898232, 2632024198, 3603593725, 2166290957, 1571114837, 2145825902,
            2043443301, 1891374967, 2046538548, 2488717199, 458651019, 2154445315, 2592455849,
            3266747300, 98563372, 1904966481, 3676400896, 1716053111, 4206308204, 2658208676,
            8001594, 1703645106, 2813029027, 894823261, 3726692608, 2449116480, 816030429,
            3707476830, 1383662560, 2226126219, 3497705382, 1561989294, 2785389389, 3599384314,
            2861958972, 26104257, 2618018317, 3938629738, 2008094819, 2901123170, 2341058897,
            4197167100, 1303127374, 2903445052, 2658544205, 4149209641, 3128590068, 2172777857,
            4127535687, 1057764004, 1621064811, 2989221344, 1862232212, 2949761869, 2620726474,
            1750641652, 791265704, 2280902366, 3677859617, 1079043155, 1923182939, 2931362944,
            123886216, 3043646978, 2297249701, 1035980818, 647873384, 2322301058, 2888236714,
            2426466965, 3992010422, 910268721, 985368295, 1204228737, 3853108154, 2009215835,
            893158799, 3476709186, 1783358605, 2497820589, 1805443650, 3969566710, 275487805,
            3039957375, 514336764, 894146804, 2205031305, 2510265732, 1422658886, 227958265,
            2068145443, 3893567490, 1543534282, 3541044587, 4205612325, 72081868, 3715915655,
            3036960423, 419507608, 227127645, 434408508, 892436563, 162488684, 2166806308,
            2727867269, 4081733001, 1736363129, 1586322132, 1569504753, 3585166014, 3409177960,
            729399842, 600955707, 620409951, 4035728663, 3310125212, 4007687135, 1985227096,
            198425200, 2174110521, 3681529070, 3128924487, 3475835290, 3105816703, 1645162765,
            1329687130, 1438822241, 138742218, 545756349, 1589117794, 2874541073, 3226347249,
            4238593852, 3175238090, 722760824, 2131293683, 359116635, 298494439, 1088923288,
            1968400526, 425105133, 1003272490, 3419770934, 2286548114, 4069139264, 2021547095,
            4252514166, 739676914, 1252440864, 2954914141, 891635266, 2569821890, 2274156960,
            110747191, 10901206, 1932982168, 3445500258, 3634603074, 2076728168, 3765573833,
            1087331828, 1830739116, 3074239474, 3109493948, 2614390121, 4096514773, 675737774,
            1507210057, 3191779296, 487432240, 2260865324, 2080515432, 2883102503, 3886146031,
            2630370868, 4277699658, 2892800514, 2200233033, 4132407774, 1674322414, 2982982405,
            233528548, 1992281682, 4010179274, 132362763, 3820134833, 940222260, 1630667207,
            3398377275, 59636590, 4159603377, 2127038731, 3535020648, 2726744939, 4053613473,
            2694809533, 706296318, 3338739441, 3842009520, 2042421975, 318380771, 2862807993,
            2988149892, 3967129325, 1725278138, 1137034794, 471477310, 2744784585, 3460424855,
            1534357877, 3129901970, 2959751571, 981310577, 1151686479, 3917168837, 973476963,
            1657796170, 57914437, 592921187, 458153097, 1933428006, 3953517828, 3462305086,
            4127665091, 3416337136, 162732228, 1898167698, 1748974110, 1940432124, 4123185775,
            3630680071, 582967365, 2778696182, 1253667008, 3496303486, 4269467104, 2724664452,
            1077655402, 2678165119, 3253518851, 2109204349, 1050417512, 303542903, 1843757381,
            2118812266,
        ],
        &[
            760607388, 1277184530, 2981799216, 2484283466, 1845234507, 4067960649, 1724479507,
            83819652, 3545040829, 3052531934, 420422484, 3643672428, 1349470285, 4056685047,
            2832938505, 1491007438, 178763876, 1111495297, 631150561, 3047444289, 1566356971,
            2488845930, 425164460, 1595773856, 3837001561, 484465035, 2820607142, 3804528360,
            2674871824, 3186224219, 4087693654, 2920476376, 8588869, 558775, 3389974041,
            2969506917, 2965452016, 3676587727, 210868935, 1114006350, 2218193233, 1556179958,
            148150696, 2219106491, 1462994960, 1292243663, 321251749, 2464568116, 253403500,
            3875750499, 2872190504, 163483092, 3519534673, 805076316, 2163404152, 615601509,
            767178352, 1619091211, 1814614561, 1067405571, 1717163796, 3946244485, 3024690426,
            2792967083, 1807679765, 431637400, 322732386, 2839911024, 3368845765, 1517033346,
            2660595988, 2750276183, 4061009359, 1623680793, 757894942, 3706877533, 2218116650,
            2293451319, 569823489, 698352566, 3510886798, 2554837373, 4086146958, 1954697554,
            573960878, 2250331646, 303856091, 2708174687, 4046135943, 1693832138, 3593561197,
            3446967787, 2952095018, 2269948291, 1381823220, 3412641037, 1102668748, 621987569,
            1535069601, 767621348, 2590901967, 3254158348, 3515706407, 733878453, 346561487,
            989792827, 991774293, 2178089063, 3612432705, 1736330716, 4220679880, 4097313986,
            214252218, 399876886, 590654784, 1772229036, 768989885, 944426986, 26251316, 461731150,
            2081388689, 3083458623, 706415427, 1827828472, 3899769203, 4083585540, 4247342897,
            1726610250, 973927807, 2871216474, 3494935804, 2158506902, 1905985267, 3353223634,
            3173570788, 1690463510, 382352574, 2134842019, 3565559088, 249151518, 3370936016,
            2396613398, 3417046586, 3594305304, 3140493812, 1254379168, 698764737, 2933504786,
            1640097237, 3137847969, 147779617, 2340800809, 3271699801, 123654759, 4243613971,
            4091038901, 226197826, 2843561677, 452274519, 1696592386, 510445282, 3118679881,
            85905218, 4025631594, 2028469208, 509528611, 1040836081, 2397186849, 1562223848,
            2694595611, 3893476920, 2309600022, 2399768696, 296649436, 2674754660, 3089905235,
            1453504612, 2462730671, 3620829868, 2445109014, 1608966704, 4064616054, 1895591320,
            4020983935, 2194700822, 810629206, 2473380931, 3571447370, 3670144438, 2241547653,
            1600037379, 1312769092, 4288987007, 2180603515, 2032431482, 3155930440, 2347081206,
            1170375513, 804187424, 1588865075, 1003883189, 1403312026, 2521589509, 3447772884,
            176190746, 2770328111, 1376304849, 3302569355, 2706302415, 771852514, 2551687698,
            2864197933, 647550216, 2115432876, 906064237, 572464402, 3003545732, 864303856,
            389881224, 2127293745, 1745696468, 2876423177, 434301283, 2411003884, 1459296715,
            4116191604, 4123400655, 514083733, 3501677069, 560842328, 3484783459, 3147137690,
            98562603, 1227984143, 3656369715, 1790748622, 207681062, 1858085765, 3042917557,
            15051061, 2662004930, 3234896654, 2201418512, 2261075821, 2414525847, 2115659173,
            1538531010, 3426973282, 1850436548, 3108645332, 3468490861, 103560194, 3847474391,
            2304980299, 646430594, 3231078594, 939320799, 21325968, 297361623, 4235174350,
            3627466917, 2483242091, 4046019218, 2574344522, 3739169162, 2592019430, 288825870,
            4090930955, 4137250565, 3891494555, 3120605826, 2196869924, 3174597170, 1613629883,
            685507332, 1534687890, 908779827, 3295054006, 3756384184, 2861186142, 1659095103,
            1717981410, 1263584044, 1519352335, 2462230874, 2507408901, 1017299565, 1783063087,
            585746985, 3964830516, 3699440600, 3536055582, 2697252588, 2469673820, 962878683,
            4263311381, 3667261985, 3070169507, 2743455086, 1717498452, 1115060528, 3846008407,
            2601161942, 1892836847, 2163612072, 759958512, 3415397520, 1318661582, 988342975,
            1319302705, 2265404538, 1889584918, 1549178474, 3271275522, 2877014458, 290716992,
            1767870320, 3388946514, 13220617, 3925060219, 884929420, 3247704764, 2619847907,
            3003460125, 522300420, 1485636618, 3423826468, 4173033596, 9068662, 1399499354,
            3671771326, 1257731400, 2320092093, 1591017631, 599804719, 155451276, 1685145394,
            1704457903, 547504407, 515106096, 3101264871, 1006089811, 1373639639, 3788559554,
            74017985, 1232649413, 780153206, 1697467997, 2186106110, 3727566152, 1581072277,
            3430739907, 1111233986, 34667293, 1405231508, 1228646292, 1721533476, 476892877,
            3639232096, 812825459, 82472797, 2919035606, 1949528088, 1351453055, 2062781753,
            3793110056, 2099220542, 1317878085, 3082007409, 3316362678, 3051185694, 3814721763,
            4123576632, 299554338, 1863096302, 151300891, 1996738407, 741596005, 1115878980,
            609896904, 1128474784, 3757075843, 2332323233, 4185436105, 2320459296, 3864402168,
            1830711645, 4274682854, 3920215426, 3480904315, 1735518575, 1450876856, 4150198541,
            2232155899, 3944834006, 2497911169, 2383598128, 2094445320, 1342532472, 1789279465,
            1000444294, 11639007, 1290432806, 4207854768, 3606748345, 1655667625, 1766854649,
            440958770, 778291729, 2589089910, 3630396169, 3916516125, 1115783505, 1808920531,
            1936074214, 1379602123, 2109970892, 672050079, 3681625243, 880229580, 1226430335,
            2376783696, 4021974970, 3642478776, 1316164236, 4224587460, 3674694076, 1609544635,
            3597979777, 1507369018, 2871382803, 2147895845, 46634184, 3128655439, 2042987970,
            1565373672, 3820697495, 1861755004, 3965634071, 192091812, 1714797038, 1913602383,
            1284821271, 535686584, 95183224, 2099444647, 3918351823, 2323427836, 3425011013,
            4034948283, 830062458, 3077755706, 2664361966, 3369633219, 685190299, 3773862621,
            3479436350, 2166056948, 1165446941, 597447203, 449505020, 1407500224, 2595439346,
            4163318436, 2061079625, 328423701, 1694015638, 1739710589, 2700176438, 3868944325,
            1880345981, 126900733, 4252410326, 1210883987, 2952395303, 2472131703, 2919687101,
            1151019281, 2923524835, 744738035, 1495025615, 864881767, 3672741575, 3952364434,
            1561011577, 1661355866, 579464773, 430715700, 3967156099, 3638535987, 3314993313,
            39754630, 3015258697, 3965969076, 1226994342, 333446910, 1554778231, 1872069078,
            3265886307, 3661167969, 2545225482, 86455596, 4170100560, 1688415942, 1011498191,
            1731003257, 3624902119, 2939526631, 3035198743, 1225561695, 172905320, 779745656,
            3939303614, 1180230735, 2279433297, 2623909995, 2744984397, 2464197534, 1263906486,
            3664838213, 826505934, 3484580218, 2811919655, 2439097791, 742011920, 3984794087,
            305740292, 4085693322, 1641069773, 2623476948, 1391797660, 2004149167, 3244285446,
            1000729643, 537058827, 2761876864, 3370814500, 1632812628, 2024768271, 10435691,
            1808177540, 3508999569, 4285869266, 3955904362, 999801692, 2701282598, 3234606559,
            1828569180, 2238113468, 2835896689, 398984954, 628112193, 3250276768, 3848427074,
            4046507569, 1227584178, 3624650311, 1983600411, 2035997048, 3406710114, 876143052,
            3576865031, 1808241372, 47133236, 2032404345, 1238579946, 2439947599, 1090627887,
            3162552029, 3743711685, 2946905539, 2462606049, 1934095111, 2581983028, 2108794193,
            537838758, 4276183193, 4118403780, 86207174, 241654991, 980261680, 1783811780,
            3182293354, 2600347842, 3815684348, 2736014977, 1734720333, 4292465690, 296762674,
            1299907137, 2710350606, 2474176175, 4143658256, 3614357927, 3668497288, 1388238213,
            4039173642, 4262328310, 2046246019, 11867698, 1389032783, 3383689984, 203861277,
            1983217572, 66262335, 2302573608, 2106760160, 676069123, 1923174580, 781475291,
            740922128, 1145598272, 1863497642, 4017642908, 1433452599, 2335232697, 2252173876,
            2622261950, 558873943, 1782242773, 4277857496, 2736476832, 4066356669, 2928683215,
            630971298, 1950508576, 1217075937, 657092636, 3655489968, 3906055628, 2142188310,
            3431799034, 1433370594, 3460735328, 844557915, 1205870649, 1403166550, 664366810,
            699119677, 2418845509, 2981098211, 1120979748, 3155458151, 1505569454, 1404478419,
            3580055806, 298814169, 883897064, 1829019390, 1951530736, 450098305, 1015728621,
            1355065591, 957492909, 2448099162, 1099299552, 3160642001, 40991601, 2335441865,
            1242957033, 744421734, 1288340852, 838608359, 3740535760, 2237946750, 3362970437,
            1146604084, 3552223768, 2000474849, 3302747771, 3289090092, 1179750911, 299925783,
            417958743, 3758178331, 3103400639, 510077191, 2978242633, 4234232170, 2462645467,
            3759250684, 2930664089, 1112129962, 2041179483, 538852775, 1881748154, 3282982596,
            692481925, 2461480040, 1321990335, 2603998961, 891524618, 2762291355, 1602130552,
            3063842230, 1512104437, 1311990640, 1124520432, 1943880370, 1383543514, 659628275,
            1498309472, 3514392205, 1588156795, 1101447212, 234163485, 1550295807, 2222040367,
            2338297355, 3634753617, 251867854, 637824295, 3385282073, 2717908232, 3780668880,
            3282410375, 12579023, 3519277677, 3981301653, 2289038755, 522494362, 2949573042,
            786166898, 1057696288, 329580624, 3463380191, 2483531405, 700795614, 2675217847,
            3326945763, 1078245300, 1966168086, 4106783828, 513839480, 452541817, 992723496,
            2322276273, 1610065747, 262720733, 1456710625, 2076208126, 1528480920, 2344925742,
            3245103726, 3530526281, 1227544700, 3828557800, 202954188, 844439261, 4143421330,
            858722987, 791357449, 3651495542, 2338484477, 628819270, 2867425597, 512142937,
            1740596275, 1958985288, 2954923600, 2724470424, 239279551, 3549834366, 4273847436,
            4205746302, 1617654892, 4055107588, 3345065763, 3918815107, 3690347759, 2780847799,
            704192022, 577918201, 2047971125, 199715340, 2815715011, 4100422521, 304495953,
            3524335327, 3863026147, 1090602107, 55579725, 801128139, 1204054409, 750920527,
            3494787127, 1695780971, 3713381555, 3067697193, 3857711920, 4229533126, 1848429928,
            2930893519, 3245738664, 2770202715, 3311103276, 3239813930, 215046932, 768437545,
            3361501036, 668953535, 2523450946, 3426827629, 4294907906, 3567391972, 2118111054,
            3636801154, 235010253, 506820626, 4082694685, 1020596576, 4286162559, 93467349,
            2864513902, 1787418532, 841111654, 3988531733, 1803513923, 2454566170, 2924727696,
            2008920947, 3384148494, 1392082716, 197522545, 1809534525, 2067947619, 1915304315,
            1794320741, 3658475265, 2477556156, 3466391692, 759597459, 3446403109, 4175986942,
            2016300498, 3285587019, 3805614518, 3359773927, 3758992970, 2082537230, 4170722751,
            1924400162, 3120097767, 1817189766, 4215519640, 1640964039, 3460451146, 553294147,
            3993022590, 936189736, 3289831433, 2746878752, 218269093, 3641264302, 3746904227,
            127062817, 3607266633, 3906014892, 3583438340, 1711866360, 1769540234, 8808456,
            692035790, 3582940716, 2726467727, 1377561547, 355273557, 2147522454, 1648952034,
            3996367145, 3973855422, 791111697, 504755645, 3306755904, 3160971610, 1420009713,
            292333945, 1246559702, 502418547, 2839194844, 1755323582, 3824720395, 280335533,
            2486741893, 3748085109, 1510237367, 913162122, 1367015807, 2150739469, 431332632,
            2458778103, 468839323, 631541365, 3621249238, 2195421604, 2309648614, 1907083220,
            1785404891, 1376379485, 1304676200, 1670817555, 1112332257, 2685812955, 3507402533,
            2655556228, 3969960616, 901856719, 2257719427, 4169793859, 3281921750, 20313081,
            605538211, 3389947560, 1288421142, 601129353, 2630934375, 1090529584, 3541228947,
            2135095396, 2428588788, 1720926402, 1972699444, 1936465090, 4196883041, 1513169082,
            1306967955, 3516168954, 2504390827, 3238431030, 936930168, 3824872735, 519589764,
            3699706696, 2222441744, 2342278743, 3084957176, 4290410350, 1778787953, 3015190990,
            2716506129,
        ],
    );
    // - Barrett branch
    test(
        &[10; 1659],
        &[
            890276236, 341226799, 792588053, 367901941, 3412384327, 710303656, 2007116218,
            1732149551, 2749665392, 2733077788, 1892910904, 3903262213, 3368335519, 980935858,
            449583968, 1960964070, 82417675, 2300807023, 737772447, 2848196409, 2168358691,
            2864360576, 177909817, 2598798571, 1444232680, 2011304558, 436734343, 2531093317,
            1833818035, 2701704151, 4079761960, 2060337785, 3666406285, 2332405305, 1694319899,
            2756923912, 2596868778, 3353801212, 2319197408, 949456643, 3210673236, 1574562999,
            1402290323, 875914832, 1808447500, 1400287402, 1870271379, 3193561790, 2631625063,
            422393142, 454330683, 1399511891, 887928993, 1982772955, 4249787009, 1042550992,
            1845718215, 1923499583, 2405586067, 3603194566, 2924605215, 694963825, 1949261922,
            395815473, 2098120598, 2155391008, 3001875867, 2497152933, 2157055549, 1494381162,
            2624433809, 445187226, 1230608566, 2984284721, 1836275894, 3306237100, 1841513512,
            2710642428, 973820593, 514673088, 3144609082, 3051871892, 3428549981, 1521009477,
            3865181386, 3006578727, 4266792118, 1296136965, 2663669369, 2707104363, 1525098552,
            3096200455, 83349968, 440117228, 2452613966, 1926577488, 404744979, 1730348753,
            1281414650, 2753590879, 2501408984, 2436683689, 4018005907, 3151813017, 774894936,
            928812960, 1406876714, 1300626258, 3795254718, 572793086, 504939191, 1858924532,
            3208327188, 2458888708, 2209857693, 3635862246, 4015896456, 2424475729, 2407841828,
            1838875522, 2698758219, 900362220, 3297883135, 2319766547, 3926911411, 2683221001,
            635164605, 1978063173, 1554445660, 4039067308, 3534363633, 228909859, 2698728084,
            1446768153, 3078533766, 1803252867, 810281015, 1437865883, 3136549314, 2523233981,
            1635904205, 2841910210, 2673459917, 302753536, 3721460164, 4235982205, 2610290684,
            3241605472, 3116643475, 1191081198, 2046254645, 101183106, 1647792738, 1559541261,
            3806071610, 1461177358, 3272653778, 1407647063, 2448211251, 4228622147, 3316552027,
            2359816329, 1591928596, 2248518657, 1224679111, 3354288039, 3472204054, 941606258,
            1209690115, 322676832, 2932831862, 2003163428, 3206276298, 993107033, 2629921131,
            3427077956, 4115430705, 18758505, 3427417332, 2001126401, 2108037391, 3665577171,
            1863775255, 943945131, 1612515047, 4135299880, 3120161330, 280157154, 2453614374,
            314225268, 3156105086, 1108918989, 164492394, 3228445085, 159609226, 3542796998,
            3908606313, 737372698, 4020386652, 1541117308, 4079987109, 831583516, 710011798,
            2517260657, 1502443551, 3947743604, 2630448337, 3713488594, 2045169924, 4240999995,
            1008865782, 4189193897, 1658576435, 2238907645, 1755810942, 797651323, 2137409038,
            1033636473, 4124730937, 3073837451, 109594062, 2690701355, 4034943218, 1405033142,
            1440037662, 1082006809, 3569043007, 3728191023, 3761504320, 4122286843, 2321793213,
            1900629490, 2311374908, 3129571201, 3976202280, 1676045885, 544386369, 2901355594,
            3307278941, 2444850863, 774358865, 1135879223, 4061259975, 2102938029, 3898096702,
            3670762822, 2514569341, 3882503461, 1631374873, 2849583118, 3325748020, 2953632965,
            3236697735, 93102960, 765161636, 3854704169, 1573342719, 1310401001, 2041585538,
            3684168483, 1596820840, 3244422616, 3149564815, 3111579294, 2543256788, 285031977,
            1116476069, 1854107100, 2993132022, 2237783497, 1121738400, 1873498161, 3906059657,
            2868037945, 2128298315, 2461778166, 2519976634, 1169495678, 145517031, 1089544388,
            3496922453, 3858416589, 3577132981, 819516733, 360815306, 3662368821, 1541910226,
            2826451779, 1275644110, 1972056710, 1036339248, 3879421912, 168837530, 2778017004,
            508177746, 1418725768, 2362803078, 1537678007, 4273641757, 1157309391, 2027675884,
            2766447468, 1140619111, 280217396, 1977767061, 3847572985, 563089282, 2707116985,
            565542300, 3133606843, 1550118329, 4226756370, 1602547683, 2474372184, 2817100937,
            2304466412, 3401474958, 1765460555, 514706751, 2958558266, 2788304434, 728359185,
            3110910343, 2268078075, 2011266928, 1833177040, 3411609757, 2509364835, 2276646789,
            3090785310, 410706208, 3982922454, 396138284, 1225183095, 2653156018, 1198560357,
            1081851637, 4266095252, 200467546, 2084872625, 3819726737, 1842926566, 2137414674,
            3780793518, 3803338949, 2618440541, 3079036479, 1098784392, 2442578090, 3632573226,
            2172339623, 2589618514, 3908085439, 2879837004, 4287758566, 2751698111, 2126731652,
            2825624276, 2248293625, 2459102047, 727371099, 1762220065, 2035848579, 2684644913,
            1138174417, 1607470803, 1910213315, 3392274014, 2122238927, 799528170, 3993036983,
            3276585428, 3696900606, 3376656588, 3273677417, 3728672429, 2897266369, 1209435480,
            3668943597, 554740320, 1910520674, 2755821121, 2815354947, 628678360, 1972408136,
            2959168464, 2823142531, 2826638433, 2423863843, 1087171516, 3879922452, 1945794416,
            3539137213, 2491723621, 2326706944, 1055221317, 1132982204, 3700164345, 2874108055,
            4229375434, 2069834207, 242639877, 1033996106, 3133990269, 2480736852, 3972172611,
            207742724, 3310273273, 3175731064, 3581837682, 76902028, 4245157123, 4269031394,
            2490069025, 2999666665, 504120169, 191809390, 167412924, 555218984, 1496941280,
            2093783841, 3208530629, 3145850049, 3933461123, 2175208309, 365172205, 2462688963,
            3453497377, 1555304950, 4025179114, 2083241909, 168967519, 4078676060, 4042564952,
            2524740606, 910140135, 2191946661, 36949303, 2750127471, 3088746837, 1513675215,
            2555879390, 718186264, 2118995763, 2901874435, 2945636440, 2863577556, 2968054349,
            2987445655, 1391389309, 3362068597, 420379086, 3549925062, 2452687361, 1408886777,
            2462121091, 3799278878, 3322044177, 4293985015, 1286609586, 2785058305, 4274674252,
            2778156482, 3910962810, 4068513623, 1564413978, 1181250931, 3988825528, 2788859022,
            2134808366, 2286009824, 191534000, 673284258, 943092112, 3792387654, 1636364244,
            1258619608, 2687335657, 2373558306, 1780255272, 192951187, 431701677, 3304549528,
            1704199693, 314302815, 1733754456, 2651178366, 262110118, 2490882199, 3280711290,
            3787049925, 328986136, 1023802009, 1956694362, 2484923034, 2727330053, 2638100006,
            92690660, 3896101961, 1028256658, 3988282321, 2110531875, 1842222098, 2787896255,
            3395969716, 990977104, 2139528143, 1832975942, 4160298612, 2285171947, 1418666556,
            222644271, 919581311, 862681678, 2923905104, 1799774966, 2815043367, 2781755775,
            1331093883, 669689278, 2966365455, 3173764511, 2239898981, 2363266499, 2640372661,
            2127456692, 2279796862, 1776783331, 2548834983, 472218271, 2900695473, 1495706076,
            2657343542, 209220262, 275069056, 3271376502, 3733744095, 3534782378, 2464305853,
            4233994418, 1155007782, 1180251759, 1970459014, 2342844621, 59653648, 2748740588,
            2766398852, 3832978838, 242101716, 1386468541, 1326666089, 3173296092, 2837432655,
            3998816786, 3710871416, 679295186, 2135255711, 1894408962, 2653191387, 3752231861,
            2630930863, 487033484, 1492980844, 3038367576, 4293864058, 2543786315, 3417240532,
            2205003924, 1597945938, 1509635773, 4283299822, 3442672823, 4165828898, 396431869,
            648731431, 38097411, 522913513, 957583354, 4279028565, 2596119340, 2064412533,
            3862275959, 3172491786, 2807778054, 1802160322, 2786925709, 608133700, 156702572,
            3651500279, 2104603013, 2661177081, 618430078, 2315385783, 299901731, 2422105030,
            2329676657, 716074660, 1732980457, 1214771864, 1586076972, 1803988277, 3707411655,
            2659362119, 2373571795, 770730047, 3785576910, 735242862, 2539172759, 2203837147,
            1488005025, 1384867732, 3789824993, 1122099741, 552382082, 1506769429, 4044090493,
            1748889897, 4206809578, 781459036, 3672028809, 2541707199, 840152057, 238216856,
            801669583, 908070471, 3215820340, 277209814, 2800868730, 2623960742, 1086085297,
            1126632972, 4142066460, 1949273163, 2523144053, 2629334386, 3335675532, 4057941621,
            1205937990, 1082630648, 1878674537, 1256320064, 1727330605, 2529007873, 4276070038,
            2148176144, 2447857920, 3472917547, 1158643513, 2671603905, 2829575595, 3450336191,
            982424809, 882567930, 323789646, 3200918344, 2216686077, 3009541444, 2571083160,
            1006292466, 437753632, 3278939834, 3355173103, 3481349388, 857243294, 2364943880,
            3808395135, 3214225862, 3777869070, 3565105852, 2808230263, 221823914, 2758813582,
            2925915811, 414573289, 2532719232, 2409558681, 270315198, 2558439626, 615075030,
            4089479722, 3494552549, 2115205628, 74023539, 2750188165, 1839125170, 583459154,
            3761181274, 2329244963, 1691231890, 507167187, 4046741250, 1026250436, 902121215,
            2344634879, 2988992123, 2984781046, 3518129686, 1917222692, 1968401639, 2543696192,
            3620784874, 4204784824, 983090664, 2003341457, 3457430091, 1995667040, 2478215910,
            1213924330, 902607969, 1674777804, 4280126343, 2726668253, 2097919452, 3522418576,
            33515971, 1894400520, 3029857368, 150356335, 4243001247, 673978253, 2719146244,
            2090227855, 4192329224, 2236320436, 2148835059, 3185966796, 941447358, 779143957,
            2545384906, 340222909, 1163954627, 3190577527, 864905744, 3107306228, 910249975,
            2948352628, 147585477, 959446574, 3474262008, 4035464440, 1309641572, 1843324139,
            594300461, 1004840213, 2224007255, 3852374975, 4120257334, 2723207946, 2794257458,
            1545266608, 1321887969, 2165623498, 4123085504, 1017498034, 2385703284, 1909037055,
            543047021, 3878828835, 3447407518, 1606126047, 1695313677, 836832112, 3506770595,
            2603100919, 3453043698, 199938105, 3434291843, 3309362210, 3420539779, 3881672999,
            3941526979, 1892491834, 1578793338, 2517226952, 2402834890, 5862131, 40700601,
            3714441463, 729961232, 1925856845, 3468241001, 3115568092, 1631854454, 1701899766,
            3027256253, 2894901269, 1124111614, 389357030, 3985153554, 552236945, 2703725197,
            3429926834, 2037737468, 2457575384, 3768369718, 1501137909, 1925317821, 2657749431,
            3797990637, 4187234508, 327539080, 794903321, 2319399609, 2762815786, 976015557,
            1691624096, 3787965011, 1187388905, 2626212695, 3807450271, 3591385293, 1093027360,
            2171140441, 1152487697, 3182952906, 2771269699, 1234596937, 2271936819, 603050394,
            3263289503, 3560700020, 2800995941, 1196053050, 2414391090, 1881373224, 1884483033,
            2568796666, 3824541190, 3858390730, 2040352321, 718151978, 1377901177, 1597261066,
            2356591213, 2419086272, 2844721838, 2553373958, 3487777061, 4036573915, 3960541066,
            646040800, 4243906341, 2572505351, 1801145995, 664327833, 676356931, 2958233650,
            3981016651, 481021340, 418338722, 3399065823, 3030397492, 150785760, 1176367446,
            2939257689, 32665451, 2951976791, 311905711, 1897102562, 3243398501, 1748429176,
            1746479862, 343846941, 2781004348, 2171670176, 309744191, 2415354772, 1041777746,
            2288489926, 4248954013, 1910365690, 1113007596, 3208904334, 323062080, 584001309,
            30088757, 3210313377, 810170431, 1442786337, 854351100, 904966376, 768074292,
            1149027717, 3752792280, 3857132892, 2279768216, 3349859712, 282292503, 56581336,
            2391268817, 2947375479, 1126824647, 4105649938, 3962247540, 519562385, 1963761442,
            395588526, 2431054442, 950162585, 1570376756, 2250607225, 2673908656, 2527346059,
            288669023, 575810459, 2786581872, 3735760778, 4200489389, 680967503, 1713328630,
            1930270331, 2425215981, 3980660137, 4014429658, 2192587396, 3942840373, 1564337241,
            327120779, 4156973813, 3880803765, 3540049821, 1863580480, 3375736133, 4036504068,
            1785184442, 1955521254, 1453896460, 1053996688, 1978861082, 548425572, 302301885,
            1991013158, 22073650, 3921072009, 1928162922, 1082841772, 327037462, 1044813693,
            2938349910, 995698425, 332894183, 1388488396, 3872091479, 4029340893, 3277696405,
            775391931, 1016127394, 2545485682, 2073876466, 1128541985, 1496426020, 80233992,
            1338981471, 3354539036, 3085869246, 1640435947, 3318556546, 3284845655, 2946418668,
            3047158280, 2003184416, 159404031, 2086225997, 3708415056, 820845255, 2941403694,
            1894980986, 4054398500, 354127917, 1759927817, 461854099, 2756970748, 102702694,
            1301323588, 2833690096, 1511747665, 2593308423, 1879972490, 1838875158, 2618211449,
            1797209275, 120683822, 1883611437, 4202360723, 4128014501, 3211136205, 1088772951,
            1828585088, 2938812583, 2284221739, 3823561623, 4077006488, 1029205999, 3672550742,
            343538454, 3044781643, 1237692065, 2928228237, 1830634178, 275610581, 3774294394,
            3361331648, 2196692892, 3255359432, 2329150995, 1274593100, 1328237401, 1721689163,
            1552282513, 3159830746, 3644636753, 1105036069, 355771027, 1576347151, 944409757,
            448354212, 1354862379, 339109623, 3510474598, 3466751984, 20825450, 3821901448,
            2057496944, 2943845147, 3219926439, 1333132401, 183117509, 3498530864, 2182448771,
            3426279410, 1665731931, 1510911964, 1196289032, 2580562014, 1929195017, 2594768969,
            809172016, 2653623722, 2512680195, 217752055, 1721337598, 2483986094, 2361703097,
            2370506525, 1987764412, 517621644, 693022222, 4199398517, 373355871, 3352107025,
            2338083298, 159801450, 1797870121, 2860144576, 3939907735, 3514862832, 431143617,
            1623844364, 1508889091, 2387069842, 1972468120, 736883904, 1753611344, 1661123641,
            2409352944, 461880992, 2038512497, 3457487084, 3961958981, 801429985, 2972366430,
            3338662267, 3327375019, 2105332644, 998675685, 691868141, 870966463, 1279549664,
            3134690576, 3190301255, 1631564609, 1367483321, 2646565011, 634484526, 2537380840,
            2213333282, 472168933, 3498162369, 4174275366, 1543944344, 3557960735, 1147270417,
            3787417027, 214242757, 3877605781, 1792046194, 1247989220, 4226299524, 2026490398,
            2743362165, 3282533732, 1766662214, 3166276549, 3016180603, 2258391278, 3962452054,
            168300410, 2111946937, 713217908, 945581724, 3974119689, 3587010488, 2181601474,
            246735659, 2543921339, 2723353212, 1467436138, 3185503405, 3831196899, 158015188,
            490283336, 2084333291, 2741425836, 655722571, 2045963231, 4174037980, 2501752755,
            148738138, 1307383368, 3787063365, 721414861, 367804370, 3580824327, 4086466283,
            1279991485, 2037844627, 3544424029, 843306486, 2062910422, 2484907808, 2564538263,
            2558157768, 3736247186, 2359778109, 2792113263, 2653179154, 1171565785, 970085568,
            2755189942, 2462193088, 2187804183, 1622505599, 757911218, 3544379371, 4084601374,
            2831197190, 1128632808, 1911943420, 1064662251, 617203970, 3185110184, 3872537015,
            2663673571, 1598938612, 2021095595, 1230226206, 3352634834, 3243340070, 3521473899,
            947764553, 1063048833, 332597817, 1892887725, 885809381, 1232744110, 488758874,
            1219451925, 3879973077, 3087466985, 1317009576, 4096728294, 1647182459, 2603742762,
            3671645026, 2351436637, 2208571110, 772272953, 2835855429, 3069601842, 3507496189,
            956062404, 1530390718, 3749254585, 2207726706, 3111772294, 1060767761, 711759067,
            2176543833, 2198489800, 808754055, 2305333083, 3151555404, 2509458656, 2135135534,
            3230952171, 2571117847, 2341460893, 479051995, 1768657443, 1460137774, 2415002930,
            3360777084, 1122116253, 303290829, 3416566333, 4216166260, 1908146236, 645774129,
            2687866494, 3150948846, 3002877577, 1584197209, 1906233324, 2943774631, 1302474040,
            123233115, 3070944438, 4059478095, 966454883, 1124649660, 459555946, 1739202231,
            220401749, 569942371, 3571881666, 3770440911, 932409766, 2318445287, 414753024,
            1238020551, 1107839477, 659942128, 3483946741, 465817183, 3283253432, 3332782561,
            930288386, 1502477361, 289384294, 2672225570, 656751994, 1938557826, 4263515785,
            3846686615, 1977835682, 3782978161, 3841096638, 2100335863, 1466976524, 905037512,
            3998859595, 2647592476, 1315412287, 2608076472, 2106055663, 3021269968, 3874586748,
            3153686736, 3797040333, 3195133207, 1058452397, 471283566, 3289919590, 2491009200,
            274949631, 1247099012, 4029069224, 2905208231, 2179210118, 3711788704, 3755257973,
            2928888484, 1052560560, 2967164700, 1635564638, 2408974105, 2650877150, 152761664,
            3778129977, 6898243, 3642738916, 2085555548, 4208692126, 3516476673, 3689736495,
            4284496225, 2331191533, 55304992, 3822559904, 2239314561, 371351603, 1372721940,
            2935911010, 2081736209, 3779840280, 1714652721, 3411278140, 2654198957, 4262940016,
            713862676, 3977788884, 1302513367, 1393129292, 3633035689, 2006841374, 3188426778,
            506706704, 4019796909, 2697310122, 2678256017, 4001202363, 4268990548, 1518149683,
            4051379891, 2122891199, 1746178525, 1962304081, 3865021362, 1312381895, 4177647623,
            2434495123, 696907907, 482662266, 540053306, 2380506995, 3725670235, 2171452643,
            3435152939, 4096541777, 3503333448, 3691060485, 1944004703, 898436730, 121854915,
            1588868274, 2860886605, 1300182839, 1693834903, 3031447227, 3217167554, 2922628107,
            4114756146, 2079196834, 2684932909, 363311866, 668991145, 1327347517, 2469568598,
            1913706668, 268108277, 577243285, 1477909197, 1398721081, 4010189391, 2691034717,
            1036016821, 1810355639, 274912831, 1795119756, 1368326430, 2579557677, 913700763,
            1609695519, 2382496942, 1044984785, 303066142, 1822891534, 3029842660, 4197043641,
            584103984, 1997889445, 3604603832, 3119992851, 2675724476, 4061366219, 2925184984,
            2195814396, 429034966, 1102290544, 3100888502, 3392886272, 630724501, 315124612,
            2110860880, 3375452806, 1360073167, 1663964263, 3704594026, 2156765374, 2209318060,
            305106236, 3229984531, 3679679759, 412339952, 3408237957, 4272496758, 1831873985,
            888956855, 2879515229, 2865192273, 757175501, 1423828575, 1369325109, 100316687,
            3388465856, 2777010920, 1042158042, 1817540433, 1115141275, 2632434501, 4015855401,
            4118763832, 942872566, 1619453899, 3946427286, 541394029, 1528755909, 2036673964,
            1456023963, 4059819153, 4121822726, 2025464163, 3230339097, 68415417, 18265310,
            1132496724, 2754635682, 385314119, 1567156044, 940632399, 1831395946, 2469397928,
            541250595, 74019926, 2039901080, 773704780, 590385326, 3107133266, 2143056697,
            3257519203, 3995912435, 2208286868, 1856405930, 1700973893, 1653979838, 2024515776,
            3218710950, 1084145854, 1317505497, 3590817427, 123061081, 2076656022, 4077554585,
            2879482171, 2950794060, 3734348393, 722767161, 3691545699, 1644306454, 333687306,
            2241964536, 3060788682, 2453668285, 1189757337, 1263877652, 569206760, 3846943302,
            4010117854, 4059584053, 4044950226, 3638405845, 3879375082, 2049939648, 1496362390,
            1682303551, 2612201683, 3043389670, 1640230718, 4052525226, 917568678, 863698076,
            993517383, 3674253221, 744083165, 2026633641, 1428745764, 3316111420, 296209737,
            825439661, 1138886542, 4238983059, 1093858251, 623018211, 509903369, 3954989263,
            2875469802, 3163782305, 2068031367, 2440167512, 1590471250, 2577400899, 1401443658,
            438753386, 973022558, 490459931, 4153136453, 2638113298, 1725729658, 1657506325,
            4041198040, 3744818426, 1877561769, 4134538451, 3997643962, 3278471793, 2418908581,
            356962783, 3224136760, 2919949586, 3613171265, 97663361, 3097697220, 3489004921,
            2697784225, 4279173863, 3772958071, 3905886590, 1711013393, 2126838688, 990698143,
            3920056751, 3588492473, 2098301822, 1854964825, 1664184868, 3795158895, 151603425,
            3531429187, 3490656748, 1113504873, 1471018271, 4168869499, 4008256495, 2171869942,
            609135193, 2986518561, 1725952510, 1976303843, 2274910761, 242274490, 3473272380,
            3970326612, 2167449820, 1194973795, 3783506867, 348209623, 3135533248, 3532222693,
            3344736663, 214520997, 1636130783, 2067796675, 2369417375, 2235358466, 86178644,
            2779810734, 4214159539, 3218271482, 3985230716, 2329504243, 147581458, 1474992222,
            1674107989, 3428339487, 2770470340, 1986697826, 2655128050, 2876733066, 1471032373,
            911683545, 2090594993, 3437186916, 3308018952, 533482761, 1086927954, 3676694935,
            3342609410, 1508797862, 2500353221, 3720936876, 2701068861, 642689761, 3643678624,
            290806518, 83395626, 1535005732, 2747794047, 3570146930, 914908213, 1337975084,
            1497808741, 260949154, 4207131444, 593904854, 3888947157, 195239383, 3774140559,
            2853428653, 3136732762, 1248240658, 2234958820, 3762975745, 1162679573, 272129118,
            296868072, 3334585892, 2918794318, 1694070693, 2492485295, 2683814086, 298872871,
            817942615, 1004532076, 2606294117, 4046765057, 2233192255, 3661649470, 2238986852,
            1062091508, 2399311242, 2885215834, 1798967492, 2531479104, 4074434116, 2111742327,
            2896836169, 3250431, 1770836879, 2620395921, 501323762, 1835526774, 2803163141,
            1991155504, 2143149513, 1934058381, 1289076816, 2032493823, 3585211537, 2907068377,
        ],
        &[
            1957536939, 2704233779, 913359386, 2607986801, 2446464778, 489495221, 1977138138,
            199874920, 2066813840, 4078462749, 1016088557, 337324630, 3990125130, 2305453367,
            3378159386, 3362499975, 1892329456, 421235551, 960087932, 2138732375, 1747069721,
            2966299538, 3497725227, 788194103, 2620925095, 3992734827, 2347614949, 1192266871,
            2152140927, 2595902702, 1509482398, 787254206, 2484597728, 1789728733, 1226715026,
            114480829, 987023760, 3496800953, 256020284, 1894092757, 47387107, 2176522500,
            2135704654, 994057186, 2546241901, 4122251778, 3491731236, 384111784, 1236832234,
            3662603701, 2609281585, 1219883919, 2008118687, 2326006512, 2338027654, 2332414668,
            2538669365, 2958308350, 3306252952, 4097401440, 245412822, 1134218707, 3076680142,
            926168655, 451578268, 1698122066, 514466774, 1794890875, 2253188253, 3860209307,
            4125975400, 1233713256, 1331554757, 4108025645, 636835921, 1849327003, 393004812,
            2558555361, 1692156580, 3582970009, 914267920, 2658172248, 4012245606, 854232321,
            1471576996, 3492333793, 2913754320, 2051701466, 1005511236, 3371062639, 3296185640,
            2678783225, 3650455045, 4263496163, 3495211840, 160882721, 3272734255, 3363410960,
            1673492685, 2227170400, 1101828895, 1119793884, 1457840656, 2222657767, 3828794092,
            4237803786, 1329436877, 2830186178, 3860284552, 1337378154, 2783161843, 470651568,
            1823782186, 497979415, 2298463213, 3726533213, 2304335399, 3374908291, 3136539039,
            1011793524, 1434135959, 3546472177, 3334783571, 226156373, 1932074218, 631985614,
            1079925731, 3769906884, 4019194278, 3109451869, 1778729352, 3135074288, 3585179302,
            536686269, 269921815, 75736971, 1251776355, 3758315612, 1656722319, 3842105657,
            309256249, 3658053371, 1366755718, 2879694669, 216747845, 1138851530, 180246577,
            2017139089, 1955595458, 1039824550, 3194217029, 1733371216, 1506550177, 526780059,
            1134528034, 384391871, 984118045, 3841118292, 1290155172, 3325719027, 2046692404,
            549115234, 1571136961, 2903831138, 4237164623, 3384940491, 444561690, 1887009067,
            3032883562, 528649308, 3526917838, 861967137, 849430599, 1174292170, 3363871839,
            3164890264, 4139898798, 2515068538, 2817798053, 1985333569, 3628916450, 4104808900,
            834622482, 2376137274, 570637315, 265226452, 2900200945, 3714687255, 2191892020,
            1444354629, 2162094270, 1794298889, 1027492948, 3254523420, 3973688556, 70444914,
            4247183201, 2558259799, 1382220873, 363937698, 1293687680, 2257401251, 1821850058,
            2709286660, 3142574973, 3132883088, 3836062384, 2080511414, 2488073709, 3338281745,
            158637847, 1975561389, 3788657841, 2878593484, 3932639038, 2471341193, 2976486658,
            4177193258, 3207728388, 3451499189, 1036976564, 4174152173, 1200214586, 1593901807,
            1598813685, 3572817953, 381665761, 23270196, 1802671154, 3457511367, 3684670008,
            4141234293, 4257731307, 1636422959, 740772838, 1988075395, 3741682939, 91025578,
            3019507910, 1239453327, 15530035, 1805537881, 3030234017, 2850226774, 647324122,
            367062563, 997518198, 694008483, 317521938, 1160123804, 725289606, 1178038187,
            1241619590, 3354084342, 2764339200, 624229820, 212334275, 395480330, 1367723402,
            3193328333, 492787846, 2912412117, 769721138, 1070730529, 448893800, 735046630,
            3751288669, 2414740806, 2618411130, 405231076, 2516857660, 464468370, 3258011579,
            1477182041, 493720224, 2850809713, 3137078323, 3502041365, 3173963137, 2479519371,
            167431223, 3658737457, 3772800354, 1986032427, 1987941758, 3573465175, 326371902,
            3433869641, 1504044245, 2797511645, 2359717369, 3995931882, 1634803252, 4253137604,
            4268085428, 3040903829, 2277724443, 128782128, 875783256, 479424864, 528909344,
            3760571120, 3390001080, 3322552131, 2303766273, 2363235433, 1521779839, 71576180,
            1476909520, 2716820223, 3668085110, 2404806181, 3808398898, 551144165, 396216221,
            3351190698, 2567267783, 4253613190, 436775, 1500218837, 1017116170, 4018991981,
            2949611141, 2820025183, 1026140859, 1421949338, 3652746241, 4092056796, 812077816,
            400306669, 3639988628, 3886839943, 1895304730, 426912903, 2799536256, 648522878,
            3154874866, 2278734966, 2267078837, 3248002806, 2570845216, 1716635426, 3324742113,
            2710305693, 2354409016, 209044596, 1052107641, 1275041956, 2683489172, 2218781510,
            3218932590, 367423450, 1139521118, 484065443, 2774672726, 2199688060, 2595994858,
            2766781330, 4010728155, 2471938607, 2443485677, 2512646557, 1783786440, 2586521152,
            1293511168, 2754786394, 843414384, 2733670928, 2673211126, 1831548704, 1507579292,
            300108435, 4168221860, 3056060813, 1074051129, 2443218252, 2786872412, 1521579909,
            3002145596, 1038629976, 333719208, 398037698, 2327562199, 3906306578, 3860291999,
            2708215457, 2994740784, 2444084751, 2051054286, 3721504053, 13639901, 3131442928,
            1665891362, 1004981560, 2672093967, 3849433595, 3693322361, 4157500107, 3275912258,
            939706674, 2775110135, 2772877030, 3937484334, 240094954, 268854799, 358769959,
            3879082861, 990116332, 3844458806, 3358517002, 157654009, 1888197277, 3767715560,
            1992005094, 3000885357, 712308948, 97474400, 1032741665, 1474808743, 1698678514,
            931537527, 1916486634, 2878260947, 3727228108, 2313492286, 3285497655, 3206730275,
            4284678508, 2506474477, 1389954169, 3751888822, 194362906, 2064318045, 4133690442,
            2808582010, 2677686297, 3050279294, 2468404997, 1830004272, 2685594794, 477597531,
            3289330681, 2143814392, 3874622902, 3298844876, 2487277301, 3988787333, 3041371803,
            4049441073, 3767569736, 2542990054, 2425662849, 1897157558, 1338068625, 3204808408,
            2152753507, 3746300240, 2482331983, 3385302403, 2758234439, 2759085438, 508998065,
            1223433126, 2394383885, 132654029, 2315880048, 4098029485, 4024894935, 7497695,
            1562536823, 3137935348, 3016914619, 2109764983, 1252720630, 145849589, 4083512962,
            2437495916, 1112513401, 1406756818, 1950408622, 446872130, 2562784674, 179413856,
            1514287693, 3292189406, 2906659889, 713923011, 3611402990, 1483192135, 624362594,
            1716375666, 2598996983, 3400508352, 532608647, 1064107478, 3678086402, 2971527569,
            373188058, 159485586, 1268240989, 3237585490, 3168342804, 3260235693, 1044471752,
            1794355961, 2708027181, 2350414048, 2578105119, 1701986224, 3287958641, 3731989517,
            2550833186, 1464424680, 1492730967, 788230494, 1536847815, 3539582664, 2684847209,
            3275242595, 1175102357, 812199218, 2945417844, 1324851034, 3657118239, 2080012874,
            2073911306, 348871803, 1385046650, 2256273117, 1079410594, 2685229619, 2682847322,
            2114480861, 646191989, 2203899983, 3969414867, 3538537702, 4068013123, 3232826610,
            596198863, 1046306571, 2865628166, 68987600, 2100190548, 3315046820, 3490220594,
            1096067081, 2941419727, 2558046885, 591489912, 3477689804, 133164194, 2648378976,
            1672712559, 503765043, 472903689, 2389245110, 576018363, 3148922955, 4125524011,
            3286180064, 4030404254, 2604010162, 2997085129, 89056299, 1807554852, 3269841668,
            549895196, 260687385, 3579375620, 187060840, 1942595893, 603206835, 3855746037,
            591224213, 2177212211, 1725264, 3682481866, 1603335098, 2633406770, 2503221238,
            3962816132, 1570827695, 2441815283, 2775443721, 2831622813, 331378838, 2661613345,
            1859324001, 1681417767, 4071306777, 3407361880, 3630093367, 4191434377, 1636246728,
            669176351, 673352253, 3293415547, 2884409096, 14264168, 3758461627, 1716070268,
            3988243278, 1362878027, 3980519306, 536008192, 2678505899, 1811539102, 342405744,
            1073242768, 3131459191, 3387892829, 270841706, 1918541045, 492555376, 4112571650,
            90351315, 2276388252, 3010748516, 3067881057, 2416288684, 2671243984, 1733673053,
            438278887, 3784801996, 1744631357, 948419911, 327397862, 862996580, 3011920294,
            116939684, 2556778847, 3517559828, 3829460311, 4187696223, 1136268214, 4227182561,
            417640562, 947119687, 1552494427, 3914862791, 3660625720, 3269794901, 3983761435,
            1080024760, 3542497606, 734720422, 2130766530, 2052008789, 866179302, 3356984656,
            3353059419, 3601785550, 1473338460, 3554981181, 3284807202, 2537178131, 2660558883,
            1407764537, 2536942750, 1111424643, 197150902, 1986360943, 570578003, 3230731629,
            2880732002, 718229319, 2216239393, 2151519911, 3213233733, 140516899, 2705306189,
            2099103478, 3400320424, 557688725, 1668843985, 873992190, 4038504679, 2235288434,
            3929442420, 1798837796, 1541815340, 866720679, 912630678, 3653176164, 4183404997,
            3783207140, 236751523, 4213981998, 2366690759, 3322048028, 1620537970, 150683096,
            1696323966, 2488037541, 955353889, 3126250139, 2098113031, 996224862, 379286788,
            1454980629, 1330626720, 3254326109, 114141156, 505833530, 776857882, 3230818694,
            1513550233, 2944138982, 1138399014, 1232351884, 3762710836, 2726900776, 1754189655,
            3929691897, 517294527, 1584785665, 3428100574, 493061896, 191275086, 3371640283,
            3691097600, 660743997, 454351402, 2693537492, 224441543, 4083242290, 2042419433,
            1749405749, 3326121761, 1143000188, 528164385, 2093408728, 547444310, 742328728,
            385453463, 1855621240, 693597263, 821097697, 3359254510, 3784781596, 3737350790,
            1858752558, 3289952948, 3844337741, 3017718436, 747204738, 2068341572, 1829338054,
            2656623131, 2621869237, 188997883, 894051007, 2539316328, 720156732, 2857142692,
            479986318, 2311943223, 3461090848, 763159324, 1341957518, 3254977494, 344988894,
            2995353987, 359814922, 3943690309, 4114837915, 4183023494, 1015984176, 2054910037,
            88581401, 2304385752, 936890838, 1910665193, 2253192878, 2257283900, 3264567353,
            3384051258, 2436405862, 3219870836, 739980663, 2626205881, 3700440528, 4116751635,
            123261973, 97170430, 2912685809, 1847392507, 1227630879, 4041294564, 1594273654,
            4133439523, 2273557482, 1852654860, 3619550719, 1702837553, 28059816, 102772606,
            4220040474, 2282708544, 3048829450, 1198037965, 3799925281, 3345026858, 3972819307,
            3424778198, 3055538572, 3844967902, 3860665646, 3364107827, 3065192652, 313761936,
            179089842, 1653844636, 1935859587, 4030697757, 941748694, 2581107832, 2215407717,
            266072062, 151136699, 3862901759, 1913706201, 3703579696, 3175598851, 3518922721,
            1540670487, 1481796449, 2350248159, 1161169395, 1551024445, 2596829351, 908720646,
            510631697, 643247275, 7860017, 1436230977, 1642294503, 3987759281, 3697987608,
            1884149375, 2628751463, 3704486074, 1467870985, 1033168865, 4152476956, 649602304,
            513167376, 826712542, 850605344, 3235955410, 2269970645, 2692348528, 4086685012,
            635138347, 3779838044, 3121512906, 3227378207, 1054819334, 31747484, 458159784,
            4199967375, 1288741089, 1261908606, 1322196907, 66488517, 3887579746, 1119580071,
            3520963022, 3373216749, 223764043, 2861161993, 747227972, 3482785674, 1405771917,
            719369505, 1673984799, 276079513, 2508081261, 2515431484, 2922924126, 3597246716,
            647895476, 3793428410, 3425572118, 1074297825, 821559348, 2839542545, 1897144802,
            3613259736, 1808031746, 1281519681, 3896118069, 4131591488, 296155402, 638550355,
            3052325442, 125220535, 450621007, 2279487792, 3201319320, 3970196322, 1300822796,
            1592875148, 459824311, 3555860014, 37468205, 163861297, 2887286569, 1605580982,
            2267025263, 852924927, 823513980, 2749888610, 2890486427, 1975455355, 4260140080,
            2388308835, 2751260543, 2054257325, 898019685, 3076718156, 2748642166, 4107558681,
            726356635, 2409380553, 4133858015, 2137846929, 1533796622, 3343476816, 2995477096,
            1906524432, 2463273274, 2821225323, 3240034000, 284966169, 1623621895, 155403754,
            3702407866, 2384829979, 3527853746, 3965422206, 3695468695, 4111056268, 2851499724,
            2096098281, 2589036318, 2328285109, 342249683, 1491530029, 1120943322, 2993943428,
            961500646, 3834036750, 1407279773, 2159771508, 2949114548, 1257703137, 112753236,
            2687765617, 3264480915, 541858082, 1905233079, 3741673276, 1506206568, 948437457,
            2072091462, 435343542, 1826721485, 3026216742, 1899272380, 2381726103, 2386973389,
            3851959892, 676110207, 1429488429, 2456962149, 3434188217, 1759358419, 2012996724,
            3832387879, 1306928055, 3879599379, 1855514535, 1278340174, 3895170905, 4036851839,
            1736502179, 902593028, 3041896318, 1088656605, 2205253218, 1400012130, 2261287720,
            175467724, 2113685409, 2927396005, 2790930243, 1527781896, 1808475280, 462065438,
            3808277088, 1996854314, 1227914704, 2532981454, 2751135197, 832474648, 278376997,
            179999604, 3454837790, 4189477161, 3925404576, 724347681, 326641044, 545292031,
            4132073668, 714677745, 3376764287, 4156659605, 1767724141, 1293260683, 3611915310,
            2602914479, 705091580, 3795953319, 104155240, 2107810007, 102476149, 2478337301,
            2728764462, 2324210661, 1662625823, 19728436, 919970704, 408330796, 226002873,
            1224941092, 3253024461, 703084167, 1686344642, 2710447096, 3741823793, 1800137048,
            1728578459, 4073670765, 2856520839, 2518678794, 2111509220, 3046036386, 66768381,
            3409265781, 2031838848, 4139578175, 1919099167, 3614877420, 689717715, 3410494541,
            659505257, 4116537698, 2258414617, 2602261611, 4243664763, 1148504559, 3060144650,
            1821973517, 943353725, 2656223194, 2921874644, 737103001, 2682285832, 3365075677,
            504157948, 2117504770, 1091988882, 138431492, 3089878442, 1947915975, 2861908225,
            625375400, 3456663917, 2919653950, 3589858184, 3706889141, 2982313688, 2375816280,
            2734870666, 4188096235, 3604422039, 983639961, 3301762066, 3181745837, 2721926640,
            3199024837, 1310138874, 1579083497, 2259444203, 3729344966, 3342523607, 3034522197,
            3782108897, 1886708695, 3226663577, 2702137984, 2606528335, 1872012482, 1786733392,
            3448253761, 808696129, 3961046824, 746729737, 2209996950, 249014846, 1666060137,
            3026499663, 1078708282, 370739583, 68889615, 526951028, 3065574510, 263587921,
            691750675, 2941031793, 234389378, 3121098346, 552746304, 1240292787, 2749223196,
            2911200879, 3575079143, 2387877722, 56226183, 2234703068, 4280011272, 2242365487,
            2018017386, 3120559877, 3029127512, 76026914, 3177333864, 3005037212, 779649575,
            3013183545, 2164743696, 931414036, 59423493, 1275368408, 1414803373, 3990230148,
            2088594422, 1188225634, 2944724829, 1746820395, 1685910229, 834306437, 1493402276,
            2360893322, 3553642704, 29771524, 1090675414, 3011165295, 1063371514, 3940303196,
            2577763706, 2185407339, 2491344105, 780054209, 3253655957, 1491625412, 1765967954,
            2859073426, 3732521455, 1293856115, 1892088133, 148995653, 1484716015, 1889017950,
            1269166526, 2838492749, 2418409421, 2209869646, 2522045468, 168513861, 3804997079,
            1587811051, 1897846173, 2879779511, 3528031017, 2591223604, 3214640965, 3550383759,
            2783793874, 2203740552, 3118026720, 2574427919, 3540330003, 3341696766, 3490004383,
            3920451780, 2209488555, 1498679604, 331789812, 2435056342, 3810073792, 1163725363,
            4006114636, 1560879500, 1837741532, 2185704910, 2798311217, 2991940915, 568480613,
            2866397512, 3023365481, 970163083, 4289563056, 2270448642, 1850438485, 3664911059,
            3605978328, 1336002490, 1355682756, 351606621, 3586308334, 1102330800, 3090559563,
            2148719826, 3215451182, 2702159478, 1432967778, 3401990268, 2400799759, 2484974007,
            1353690636, 2753559176, 2640489981, 1036931712, 1903649253, 964220241, 2773991379,
            2133175336, 583322695, 2199489502, 3631891337, 1813009747, 456084772, 2320697906,
            2974016610, 1520270050, 896145014, 2080304689, 1976066498, 2839737822, 1549783855,
            4179969968, 1104386050, 837388826, 3165878801, 814640192, 3604179832, 2665037610,
            3822828834, 2216452945, 2276503837, 634164486, 1412890410, 3261612008, 563171960,
            3566738474, 2754595658, 681116053, 3118865924, 177463739, 139480302, 281558431,
            1690126124, 1897842373, 152100710, 1389253223, 656498764, 465779469, 1700329794,
            1161598335, 4182855549, 2896065503, 1169161019, 988049753, 141540097, 3920962023,
            1414463815, 2649582854, 1247190586, 2345836198, 2096301443, 1735443990, 2410187529,
            3110145792, 4259098967, 3496419344, 1323096534, 2280524624, 2704688879, 814213923,
            2999267246, 3391489910, 3302983115, 105712882, 1051466424, 1317340325, 2155365832,
            3664348767, 41875693, 2574817994, 2417654542, 2345990960, 3248572524, 2232341334,
            2323122825, 3467411353, 2464771648, 3558780344, 2048000656, 2550689605, 3832657853,
            588816494, 1928072148, 882779546, 2585956826, 4115174858, 2041950439, 3851200700,
            3492163676, 2564516100, 4026987647, 2862754352, 11739424, 3152433432, 3038183071,
            86904302, 4194617361, 1533432622, 2267744820, 1709636676, 2268808549, 306348255,
            378915346, 3839473038, 3466138943, 1889711851, 4243725138, 2040287089, 2711698315,
            2512890389, 1443237091, 1788173191, 3483735845, 187126422, 3972604184, 87146887,
            1114058116, 3401707008, 1786906060, 1113751464, 3806893326, 752642051, 3083541814,
            2468915599, 2445639645, 3885031785, 104242888, 3874118728, 2821368929, 2069280853,
            2465759541, 76153213, 2243998971, 60618079, 432802985, 3403462072, 3617268113,
            86582391, 978301034, 3106717053, 879484256, 4191758913, 3233938947, 1333422465,
            4259870922, 878234880, 3752118900, 3493144181, 2923150231, 2387653130, 362351294,
            3725228020, 446282046, 864505264, 819646991, 2142174716, 757107765, 3054689800,
            980736009, 3781683590, 3228301270, 733658545, 44816498, 3764513000, 2648527937,
            1175301170, 689878689, 3284446800, 582872017, 4081139934, 772717500, 2468163649,
            1154957469, 1063689275, 2686616245, 3968673572, 1552104858, 3208534258, 3719381547,
            4293124814, 4185971596, 2822157478, 2226770657, 2722172771, 3204920070, 3153368659,
            3957403777, 3340401452, 358170452, 86873248, 3073864777, 1901787488, 1868606477,
            3135506861, 2969520380, 612214934, 2360443633, 2158036096, 384976614, 1988412094,
            2588500257, 2526449435, 4235741613, 440052124, 972557488, 1421976670, 1592981311,
            1190795166, 123857612, 1450909778, 279599536, 97553494, 2356966581, 2886183120,
            675956446, 1305595169, 1205008613, 3157245850, 1388240997, 2108861482, 2163666121,
            3918485200, 2326199955, 3683465789, 3756179104, 1328336284, 703189534, 3141697951,
            912546716, 3380018405, 2873689339, 1066530264, 272975397, 132319529, 2540398225,
            3521502371, 2761283016, 2471199096, 3521041929, 756789292, 2527658902, 2755972832,
            1885214085, 2175187093, 2258837609, 2586448274, 2964587075, 2168070343, 755385185,
            2820998539, 3455906021, 1853441295, 498516020, 4189729152, 3555989654, 2264955912,
            2663747569, 754868430, 234525495, 3316740819, 1374506476, 3843025136, 3711134757,
            2912660137, 1250443668, 2288121539, 1768013429, 659068459, 1033379830, 3479740778,
            900522158, 1758840019, 1292039331, 692167145, 3562019492, 3793770736, 450238625,
            3629428994, 291785463, 1925186703, 125661517, 2883876802, 324456772,
        ],
        &[
            838935716, 2203587819, 915639214, 3753735741, 3643959629, 380366065, 2800082357,
            479609169, 3344910139, 939862102, 341732586, 2006110852, 3452170213, 3812660487,
            1864783975, 1318854193, 3422317845, 1427228514, 4149153373, 2684083624, 1074989654,
            4073983073, 763800714, 4006319696, 119085884, 60830173, 991131160, 3219809387,
            3231928830, 2959951027, 2787097517, 921471246, 453915876, 3731898283, 3251142762,
            2579466369, 27120149, 2385740750, 1273661758, 3509690455, 3723885200, 4216400843,
            3843558280, 78594259, 3230699712, 3825953413, 1142802181, 3138293244, 2095371158,
            2351682776, 2501576120, 1676985000, 2623958874, 4227586876, 1860325172, 2090337755,
            4281181325, 2213517873, 496143888, 2773301631, 3322452819, 737280293, 940258553,
            320919084, 1595155948, 755694675, 3627876063, 3556466666, 2945032739, 4212566832,
            2489967318, 612979803, 2998019149, 791094903, 2331887068, 3848194214, 1669109803,
            586836653, 17421765, 4174252863, 2502645698, 1463986066, 33539085, 1789626263,
            3342962255, 3824519205, 2026068357, 2765446594, 844159180, 3508825346, 1134575466,
            1545918988, 3138211047, 2308850390, 1340320302, 1500877398, 4234310467, 1983405401,
            1030999951, 2800668723, 4229219814, 3909258449, 3342207809, 3096542283, 1526382630,
            456900343, 1292423164, 2023977467, 3452242468, 697297984, 2512301505, 1854858696,
            2453945880, 475365007, 3487108263, 727126356, 3257286385, 514172720, 2342933557,
            2456659886, 4293261548, 2045491426, 3978264522, 96991830, 728236868, 3356414658,
            858180648, 3186522835, 315467595, 3025301039, 1022091990, 832213724, 1625707968,
            1952252164, 1626399404, 2286098955, 1957714754, 935464737, 344352942, 2655128088,
            2455096079, 3508589253, 1323890983, 18543456, 1120234679, 2368630283, 3464013570,
            414956029, 1608237042, 3123061407, 3048565223, 1337447413, 2725081489, 3544363169,
            3162272209, 1757543970, 3268970159, 2449039760, 3153678826, 666593731, 2303670329,
            4257224320, 977258122, 428570155, 782781929, 3018115039, 453083041, 2998000053,
            3874611711, 244856874, 2209589064, 365052855, 2255276114, 4268430165, 2172011676,
            2202715955, 1929862469, 2117294128, 3787626628, 2347621422, 788538718, 690792725,
            2069984822, 2567516370, 2156040001, 2753711351, 4233613434, 4029666569, 3098679112,
            4024517832, 2132477759, 1327869411, 659072182, 2909455412, 3730675471, 679967738,
            182649740, 194375197, 3841653420, 4027960673, 175135837, 1660264436, 3644680859,
            4007751387, 2899873733, 3767998718, 1048209090, 1751066328, 4286847948, 135738204,
            2614043038, 1982088957, 758427538, 3475803304, 2870686136, 4094121733, 1912601923,
            2108805517, 47636847, 230717174, 1648662744, 3198815387, 919877866, 3677274799,
            599980511, 3198454352, 962418282, 2571908668, 1214590864, 3203010561, 1973877631,
            4097190280, 980645781, 730416487, 1042657050, 3749371087, 2876213668, 1057105900,
            452303833, 936771815, 2711618145, 3682803011, 4148511396, 2871128258, 1670210392,
            363675039, 1226232864, 2545473472, 2847607537, 706216472, 668367486, 3335248515,
            684311493, 1922073430, 2173646273, 1758431003, 553691538, 2135670731, 4237772697,
            1307471006, 1673244127, 475473888, 1489978871, 3151848780, 3727938421, 2510602424,
            1534771587, 1514732764, 3961802829, 1749640392, 3073492043, 4184684257, 2678493181,
            1016148119, 162968394, 1542225613, 3314073214, 742431147, 2655829423, 2077968934,
            3307840731, 3166378296, 3380012342, 1628856437, 2492930090, 1125050986, 3917568307,
            2081901207, 1910847527, 2713093801, 1095695739, 2968092789, 3472722113, 377271708,
            1040508736, 948060390, 2161675070, 385186644, 4184834501, 1657036626, 2397219251,
            2440643845, 440875536, 2390881422, 1456820922, 2584489112, 3663536575, 3123816215,
            2449538096, 3006681818, 2608705815, 4192739282, 1380673229, 3059394456, 1566510861,
            2531146664, 1359283390, 2012589430, 4200074946, 3098609836, 4006992613, 4077565448,
            1974438847, 1221683442, 1719050478, 3161792955, 2578040126, 3061737012, 367079059,
            3046588169, 1563359926, 2224001306, 2791171902, 2289618085, 3388295848, 3287189754,
            2674283012, 2056956826, 3968642769, 3774531757, 3562296786, 3475855578, 618682730,
            3174665899, 4290008806, 1008009435, 1700412634, 2902015788, 2972205524, 2584006496,
            4087976424, 678816164, 2093048627, 2385841515, 846482516, 1762625699, 4062395767,
            1067045848, 4162678883, 3600728522, 1571165888, 3489010385, 787541038, 1248805154,
            2599773530, 115140567, 2919051278, 3369387442, 4107879124, 2428887338, 994007703,
            2665777337, 3130672163, 982959104, 2698714317, 1460766600, 1821697637, 3330816250,
            2095057996, 830788453, 3990976194, 1800223602, 2026811060, 2393136879, 1189765438,
            1109812562, 3311727428, 3224449942, 4237200199, 3873080150, 2683760941, 4088732835,
            614500558, 711266360, 2147434665, 3808624639, 3034313567, 1059395674, 1561883669,
            103784012, 2032335571, 2800686028, 609480554, 1769603386, 4151398145, 2185648870,
            1739535600, 1586519522, 572991424, 1068733442, 2619062803, 4041514271, 1086033525,
            11857992, 3192078073, 1344195581, 1238380717, 4077237295, 3875651208, 292286543,
            619476155, 2161959405, 1203958830, 4290561761, 3693181580, 3900343517, 2797356182,
            4084172378, 261394861, 769278713, 4199742495, 2309375370, 3346948261, 3750248064,
            2567542765, 3972013578, 3992801407, 3056612552, 554436073, 2730615770, 1231284851,
            2255256795, 1587098949, 1645198162, 2603125261, 1127388761, 860864076, 3375951570,
            2228774088, 1597961976, 412935124, 2173190974, 1415442621, 3815450429, 4044940139,
            1867706821, 1053298761, 2092988724, 738251343, 2294030391, 1030772826, 3106409390,
            2170123922, 1863396860, 690798362, 3794558912, 3566843628, 121083434, 994730387,
            574163581, 2016938790, 2148010580, 197642070, 2380205705, 3769045477, 1675706118,
            4120416559, 3559669638, 376897093, 1128063177, 74421326, 761413501, 3392222006,
            1321099529, 47433813, 2130311331, 1584264183, 4078065031, 4056472694, 1048381830,
            2103467739, 1824423056, 2776327151, 434780346, 4017942257, 2398098907, 3022379282,
            21450696, 2803414267, 3336384729, 3655253595, 2013481291, 629727070, 304283403,
            3969058801, 343107726, 3930196577, 1050996951, 1696216281, 4113049883, 2063680700,
            3299802318, 2169413090, 599831066, 265205477, 86598633, 1318950976, 2817019822,
            3523559767, 1192997804, 967006658, 2618135270, 652785590, 1396546377, 85541097,
            1498343279, 1545058670, 1004723554, 3231552447, 628138355, 4033681503, 2114084638,
            3662581523, 2278439544, 3504753352, 1762732636, 1199413890, 2921045813, 2558711012,
            849244440, 3149132172, 2553108318, 1232917889, 3271342783, 3248249426, 1731072265,
            2957986492, 1684696396, 3968320398, 2537362303, 2030936713, 1234679023, 4192065531,
            4222971816, 3064892308, 2590492328, 288251267, 2550089391, 879932618, 784103313,
            3344367287, 676675271, 2000590552, 3777064239, 3282963152, 860961137, 689785050,
            3572163643, 2233769790, 4104660183, 855938732, 2821571786, 2808753272, 1281740498,
            969712731, 474012755, 4001231436, 3985764292, 2862143307, 994442730, 1243356938,
            1775451246, 722781566, 3131106180, 3339337915, 2271184254, 1079900234, 2744647618,
            1638963938, 4263302376, 3563278731, 1946093388, 1809549982, 2256560928, 2960989550,
            1799848335, 3793311156, 4262135533, 2448031337, 319777785, 1925667888, 2774876369,
            2635247551, 2921425304, 1354802634, 2456898309, 1860471049, 2869265862, 3082305168,
            3898824849, 3125463798, 4212194039, 1623533549, 1173751565, 2496120567, 50014129,
            13149499, 4167145366, 1955034137, 1636727570, 404325094, 1751899763, 3718846670,
            2966030995, 3631249081, 850012394, 3102446697, 2042206947, 2248313443, 2785796194,
            2751107788, 3825364911, 839857966, 3593553796, 2978187162, 1963772991, 2535947700,
            3591089575, 972097984, 3139361936, 1671749808, 2397938437, 3496814669, 1899924710,
            750821353, 2796989722, 3330933794, 741406274, 4044115699, 1664390021, 828823849,
            4263496205, 809061833, 1366595084, 1030792229, 1836547249, 3859902111, 1449813785,
            658840733, 2048685768, 4124870169, 2673685491, 2426733455, 2333311310, 3492090101,
            1992837915, 11511403, 2540925625, 2852742586, 2179960638, 2931055419, 3659567067,
            1933074409, 649207452, 4276721791, 4191972082, 539633860, 17233553, 1594779948,
            4215033555, 181714143, 685754269, 1956851130, 589401843, 3102481839, 497723403,
            952363536, 3336958173, 375851460, 3769041100, 2814705111, 503471929, 434207586,
            4273458948, 3293647658, 2353814912, 2821137839, 2876582042, 160572601, 1201368925,
            60357568, 139088744, 721227061, 2623980501, 3943799534, 3041058274, 2977612913,
            3291630599, 349175514, 3961677955, 3235620723, 2972138888, 2079236907, 3547729544,
            3780005092, 1440555729, 2438143802, 4201044301, 2714313685, 3486585540, 2800950188,
            1801147672, 3703751487, 2461295830, 380678753, 3708873777, 40573490, 2437248324,
            198586160, 669549162, 720284965, 2019671626, 1452819480, 495221607, 816889749,
            3963879039, 2755252053, 1699095522, 1122700669, 292682575, 2542299989, 32521186,
            2188233807, 2468163267, 292904686, 3223433980, 2703281150, 1173144272, 3313052362,
            3356734471, 2695828969, 460370984, 3371854267, 351372515, 544848035, 665581896,
            405416947, 1277394067, 1797975604, 937658179, 794722376, 8125493, 3456647328,
            1780985597, 3741333943, 3296469654, 1660046360, 2306936782, 3661850759, 4080591659,
            2501854245, 1384493589, 3663701959, 1729039559, 900961315, 1098594423, 2562550163,
            1148027253, 3616002981, 334019448, 2850279078, 4060960913, 1287642279, 3376572244,
            2534805166, 1377930878, 3804442725, 1383027131, 2082023911, 1793040924, 348666885,
            2304696682, 256389582, 3478053630, 698728310, 2037905177, 3976969740, 3230702660,
            1299868963, 2833628697, 2930863988, 2357656502, 1343551287, 2011573652, 3097268490,
            3399030503, 209755840, 537294183, 3998427081, 1076591702, 1263890468, 1782599039,
            3780229996, 1024431710, 2833602231, 3777979580, 4078024794, 872524254, 352227412,
            1351273981, 852004662, 1161960014, 1906524671, 204792238, 2663768241, 4072664750,
            3414324694, 286128037, 1110040325, 900463961, 963252168, 341367983, 3248675265,
            1219574682, 408882044, 4029283861, 2041191997, 2273854782, 3408304833, 2242938048,
            3436897597, 2980462347, 2514777114, 2857882805, 461470076, 3109832332, 3200488424,
            4074902801, 3969927253, 729904888, 4152956776, 689265400, 3124361813, 2969684080,
            1117396653, 2927468191, 4173382674, 4220933905, 2013331272, 1607793854, 3861336002,
            1873865360, 16258643, 2300509732, 3479887951, 2336999346, 159459282, 3950452492,
            3929460924, 705609132, 526429974, 676661281, 3888522264, 1774544613, 3921273959,
            157436604, 4006393372, 3620960941, 1816027785, 976626829, 1147734086, 4071348376,
            2542528107, 962473538, 1088491988, 1609404762, 914564813, 1410211333, 3779344617,
            4175461012, 824484826, 2980214363, 3955476205, 3424635742, 740023698, 1869598491,
            649848532, 486101202, 2723887281, 2776425384, 682654951, 48707499, 3215265093,
            2091228948, 3882341515, 2914449810, 4145012253, 2500281477, 1192252445, 2720083567,
            3606499354, 2308681139, 3417499332, 2673063768, 2245530872, 1594809199, 2961185232,
            2154329832, 3212262766, 2884845477, 3476290762, 2814732863, 2643163160, 1881713506,
            2949244325, 4196436484, 2317791900, 4235123047, 1093953049, 5453429, 3068787991,
            3345927257, 1386287281, 1495554810, 3092636873, 1484259532, 3600514898, 2957555206,
            3348133074, 2055353028, 681494686, 377599951, 812642228, 1988902688, 2634311151,
            1839068105, 3616435588, 444035475, 3061487513, 55224199, 2434767613, 2830319279,
            473221113, 2719964944, 2466794715, 4145309788, 1222340136, 173399292, 2934551672,
            1337072089, 208573081, 3208261196, 460363797, 2972654949, 4049253252, 3382433428,
            3953728176, 3042257654, 1441354805, 871221561, 1750912796, 3684367914, 969056696,
            1128407825, 1312971320, 350151817, 1913948880, 162760467, 2927001962, 271548687,
            4233867754, 4187064028, 1752223315, 3566855814, 3098906749, 940415226, 1450714578,
            2089508179, 685807392, 1094880779, 3618434588, 1852947202, 317434962, 1659500546,
            3597259289, 2888477399, 2671081767, 2440146053, 2035849436, 3156792370, 4265496433,
            1525348048, 2545566909, 1778313912, 2782108361, 2017683243, 1447595289, 4213477653,
            356590757, 3887333316, 1375294643, 166093975, 3646164777, 2470748042, 2344217837,
            1900424442, 3867136564, 3763279553, 3894629246, 1025417778, 4274768880, 3262354653,
            1368425302, 108316157, 1343309487, 2548223206, 3292698474, 2074720811, 2415701834,
            2319728429, 4253222152, 1823863126, 671442975, 2815690234, 166017875, 4133918798,
            2341656544, 1195071311, 3591080201, 2727319421, 323604173, 3467106257, 1807836963,
            3351025639, 2903513081, 1902463642, 2877952685, 2401135475, 1661402701, 179177133,
            2157408987, 3104539005, 1187176474, 216324254, 3682100936, 47318979, 4266538909,
            1364880663, 3089835665, 2974326632, 136715739, 21460991, 875857069, 3634182054,
            3507421393, 2816493641, 2428086359, 2677298183, 4144250643, 3067655705, 2047511728,
            581965183, 3555378151, 3320505513, 1784970183, 3537100678, 2661089461, 1915923033,
            4016714600, 780795784, 660039721, 1585895448, 4267382956, 788816746, 441129301,
            2642236828, 3236397688, 4098671167, 417254115, 1512419440, 2595207623, 714330442,
            1793107412, 1915812662, 1065761790, 559987752, 2473228903, 2857904446, 176611125,
            1056992200, 3785703632, 1081171201, 352847890, 1749283439, 2145600786, 2611374980,
            2819729772, 1330960860, 237844587, 1927429738, 63223319, 3015994684, 3325055109,
            1399511661, 2815713002, 2466135290, 681412319, 3776472186, 2316275012, 3463722640,
            845546679, 3470095507, 141194431, 4021588401, 723350993, 3043631697, 3504648579,
            2019448585, 3063288730, 581965108, 4096321988, 2137034574, 295040412, 445386599,
            3257986313, 716955156, 1644931397, 55457860, 252470757, 670110185, 1740055863,
            2418200077, 2304754384, 3105055645, 23205783, 1131503062, 216403393, 2353961029,
            3130064620, 3993811637, 1483619628, 649295085, 1095042730, 2656243769, 1223292028,
            582240413, 1206780397, 2973061534, 4292796836, 3537057693, 4187363592, 491905630,
            179086816, 1772613819, 2747068600, 1832416612, 1180763400, 818642060, 36609327,
            1658217467, 2050606159, 3032666431, 3767423337, 1111353105, 1522267728, 1631984380,
            3190473602, 3680487263, 734566855, 2353333066, 986328581, 3681689264, 1595897310,
            3335184558, 2507600517, 2490029559, 2745986115, 4181623162, 3432004504, 2145423842,
            2367982528, 2806960322, 3076340209, 505849942, 533374703, 1257745383, 644213956,
            2962046646, 2201943625, 237256910, 3874792384, 983086135, 1654746256, 3220085267,
            3164213233, 1962620300, 3406864224, 284885940, 2339029442, 1267505911, 3660344615,
            3917330963, 3551145561, 3488278296, 1907092122, 2980179347, 54406137, 1998418578,
            1902516963, 2207644610, 3941412978, 1623928004, 485916323, 3615926304, 523041942,
            115202452, 2022521528, 1490106543, 1835947191, 4260360619, 1424916687, 2528887699,
            3933621604, 3297208940, 2162596225, 1222503210, 2136147383, 2425841507, 3933152399,
            4238982603, 2877219680, 3629789132, 3074841443, 3440850273, 521778548, 197924155,
            209664882, 4244045891, 4041802190, 213808, 1061287415, 4109828330, 3118408869,
            628642373, 537513117, 534884706, 3291471519, 3166304457, 1868482021, 459148534,
            1009990526, 2380010357, 890059218, 604931398, 3306919535, 2774345113, 820980298,
            2438859282, 2852945605, 3422427217, 3698390489, 3313390587, 3080508268, 2767314620,
            3842719809, 3355496804, 2278214546, 1228854785, 3963672337, 3480498117, 1185458174,
            254331697, 4075265013, 1893263033, 1251932761, 2009544780, 559074272, 253120496,
            1936501741, 1726021337, 4053670081, 2881809588, 2485095543, 1685412297, 3542536640,
            2366264687, 637664695, 1837220786, 354783458, 2949531247, 2469565699, 974572166,
            3336819772, 1219492409, 2302227018, 3368888008, 1441596117, 4139261093, 3992023496,
            2186276726, 1337085007, 1297128349, 3345717969, 1502084009, 4279269575, 746110129,
            357443087, 1184174039, 3510875879, 3851501689, 2612516856, 603491753, 797427139,
            852360679, 1117907315, 1427571936, 817235609, 3158982462, 259347609, 1788586925,
            1148039424, 3368411210, 1634605211, 400804429, 936765965, 1742182939, 1903805118,
            1538100182, 2176265557, 2662759467, 984261631, 4020913363, 1487175843, 2947614836,
            1900701443, 4060998848, 4029580532, 2688679851, 1811723482, 1453238253, 3787617091,
            2282686689, 3993782144, 21304404, 507557860, 2944490999, 182972447, 3174956974,
            869715075, 3277689659, 1814259483, 4163075289, 3873534736, 462031943, 3153254355,
            1301198786, 3943710461, 3909340915, 504180560, 2041140274, 3440956310, 3656453113,
            3273349780, 3481111110, 3576620931, 2732393968, 3946434431, 659377962, 677158288,
            670305147, 974547443, 310689727, 94218257, 1724518031, 1905703897, 1286333546,
            2751245199, 5571896, 3457747128, 2179848102, 3894181682, 4124381231, 3410266880,
            709398455, 187319059, 3511629688, 3559400976, 2489398231, 600737488, 3595918510,
            3798083973, 4026752984, 4210193948, 68891514, 3947247567, 3107252542, 1170883420,
            1834752658, 2941121513, 3841122236, 2470855721, 182150380, 3416558885, 1477637651,
            995210324, 588847561, 2552679828, 64224740, 961985977, 1427921858, 3494336204,
            1146890788, 3705393264, 3874804387, 49839043, 1196485418, 4243424919, 215134702,
            1706006595, 423982823, 710751734, 346438174, 2785074971, 3048570439, 3828658776,
            443370961, 4028150602, 942994137, 720074843, 1909180916, 647304255, 1473324775,
            1036517605, 440850580, 3455287996, 2297168269, 2592528575, 323504231, 292579127,
            649004565, 281112587, 2639333091, 2098301904, 2778751588, 2025098146, 1805301626,
            1552474374, 3081073479, 4079139025, 919447806, 2132238174, 3776363222, 1050273203,
            2654179515, 31684454, 55205341, 2546554914, 1088968049, 1402421698, 3682384666,
            1553165183, 115328209, 3405269619, 2255597873, 1379651895, 2310622780, 1996922494,
            1033698934, 3481464610, 1795184151, 152194312, 1061361554, 1284312248, 1338186205,
            4098514812, 2874751651, 2866358430, 412769457, 2483498856, 4198676447, 71720962,
            3683175370, 3989356764, 2846917228, 921604934, 1635603564, 2971093080, 742806196,
            2398879483, 3298081490, 4237003774, 2380113868, 1568001623, 2624936612, 3176752214,
            2867181933, 1917549358, 4044310480, 1307704115, 3767119746, 4201150796, 1450449070,
            744907957, 1114261347, 2636959698, 117707599, 1450886244, 2336471832, 1236404162,
            2912700530, 3665684179, 3640387930, 2966734192, 3208997144, 1270408976, 308398801,
            2329948514, 343477117, 2365820358, 3697864380, 16124765, 968064396, 551610086,
            3261227824, 3845306004, 2839278865, 1710063436, 529765606, 2590339389, 2232645965,
            2220674716, 4175021164, 1253655722, 1585348878, 3251476604, 3936991537, 190532800,
            4099120852, 1806314545, 530090133, 461441516, 3941506885, 3666450612, 3737219241,
            1320249304, 2052469119, 2016428234, 3789822793, 3096501122, 1858994377, 3384814016,
            3282125914, 27976241, 4138503041, 4076911591, 2610665395, 992528352, 1507678034,
            3833126488, 3471104910, 876024133, 4017994422, 3413457336, 534460630, 4014654886,
            4064894491, 2863983481, 92546398, 1436623597, 2255152241, 1523252476, 3763913562,
            2628132090, 1984279468, 2113001554, 2108275221, 1060884588, 2747717543, 1986766057,
            1622949311, 2332772870, 3332840810, 2524006752, 336521846, 4071972450, 324329553,
            3149454512, 187149260, 2031485001, 3480471856, 2265780046, 3792798369, 3177913371,
            2347210476, 4081333345, 4046146024, 2917080827, 590638612, 270185333, 2805124246,
            460831443, 3496962886, 4271837334, 988718092, 1152316522, 3942609672, 3794345353,
            1149696567, 1402909182, 494716490, 1245651130, 1943182059, 1599846823, 3504994553,
            1535906884, 1749731330, 1919915129, 3952459270, 1846223130, 602074928, 1536114851,
            4291196323, 2411210106, 646156071, 636908156, 390344456, 3173016861, 1180977247,
            1710702402, 2756888285, 747439941, 4092116022, 4113279650, 3090618475, 396445650,
            3996355793, 3860756326, 3383911526, 2918629863, 564066118, 1331988394, 3762172819,
            4160630094, 845911615, 4045163162, 2379948201, 4071794973, 3679892002, 1449801312,
        ],
    );
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_fail_1() {
    let ns = &mut [1, 2];
    let ds = &[];
    let mut scratch = vec![0; limbs_modular_div_scratch_len(ns.len(), ds.len())];
    limbs_modular_div(&mut [10; 3], ns, ds, &mut scratch);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_fail_2() {
    let ns = &mut [];
    let ds = &[];
    let mut scratch = vec![0; limbs_modular_div_scratch_len(ns.len(), ds.len())];
    limbs_modular_div(&mut [10; 3], ns, ds, &mut scratch);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_fail_3() {
    let ns = &mut [1, 2];
    let ds = &[1, 2, 3];
    let mut scratch = vec![0; limbs_modular_div_scratch_len(ns.len(), ds.len())];
    limbs_modular_div(&mut [10; 3], ns, ds, &mut scratch);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_fail_4() {
    let ns = &mut [1, 2, 3, 4];
    let ds = &[1, 2];
    let mut scratch = vec![0; limbs_modular_div_scratch_len(ns.len(), ds.len())];
    limbs_modular_div(&mut [10; 3], ns, ds, &mut scratch);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_fail_5() {
    let ns = &mut [1, 2, 3, 4];
    let ds = &[4, 5];
    let mut scratch = vec![0; limbs_modular_div_scratch_len(ns.len(), ds.len())];
    limbs_modular_div(&mut [10; 3], ns, ds, &mut scratch);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_ref_fail_1() {
    let ns = &[1, 2];
    let ds = &[];
    let mut scratch = vec![0; limbs_modular_div_ref_scratch_len(ns.len(), ds.len())];
    limbs_modular_div_ref(&mut [10; 3], ns, ds, &mut scratch);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_ref_fail_2() {
    let ns = &[];
    let ds = &[];
    let mut scratch = vec![0; limbs_modular_div_ref_scratch_len(ns.len(), ds.len())];
    limbs_modular_div_ref(&mut [10; 3], ns, ds, &mut scratch);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_ref_fail_3() {
    let ns = &[1, 2];
    let ds = &[1, 2, 3];
    let mut scratch = vec![0; limbs_modular_div_ref_scratch_len(ns.len(), ds.len())];
    limbs_modular_div_ref(&mut [10; 3], ns, ds, &mut scratch);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_ref_fail_4() {
    let ns = &[1, 2, 3, 4];
    let ds = &[1, 2];
    let mut scratch = vec![0; limbs_modular_div_ref_scratch_len(ns.len(), ds.len())];
    limbs_modular_div_ref(&mut [10; 3], ns, ds, &mut scratch);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_modular_div_ref_fail_5() {
    let ns = &[1, 2, 3, 4];
    let ds = &[4, 5];
    let mut scratch = vec![0; limbs_modular_div_ref_scratch_len(ns.len(), ds.len())];
    limbs_modular_div_ref(&mut [10; 3], ns, ds, &mut scratch);
}

fn verify_limbs_div_exact(ns: &[Limb], ds: &[Limb], qs: &[Limb]) {
    let n = Natural::from_limbs_asc(ns);
    let d = Natural::from_limbs_asc(ds);
    let expected_q = Natural::from_limbs_asc(qs);
    let (q, r) = n.div_mod(d);
    assert_eq!(q, expected_q);
    assert_eq!(r, 0);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_div_exact() {
    let test = |qs_in: &[Limb], ns: &[Limb], ds: &[Limb], qs_out: &[Limb]| {
        let mut qs = qs_in.to_vec();
        let mut mut_ns = ns.to_vec();
        let mut mut_ds = ds.to_vec();
        limbs_div_exact_to_out(&mut qs, &mut mut_ns, &mut mut_ds);
        let q_len = ns.len() - ds.len() + 1;
        assert_eq!(&qs[..q_len], qs_out);

        let mut qs = qs_in.to_vec();
        let mut mut_ns = ns.to_vec();
        limbs_div_exact_to_out_val_ref(&mut qs, &mut mut_ns, ds);
        let q_len = ns.len() - ds.len() + 1;
        assert_eq!(&qs[..q_len], qs_out);

        let mut qs = qs_in.to_vec();
        let mut mut_ds = ds.to_vec();
        limbs_div_exact_to_out_ref_val(&mut qs, ns, &mut mut_ds);
        let q_len = ns.len() - ds.len() + 1;
        assert_eq!(&qs[..q_len], qs_out);

        let mut qs = qs_in.to_vec();
        limbs_div_exact_to_out_ref_ref(&mut qs, ns, ds);
        let q_len = ns.len() - ds.len() + 1;
        assert_eq!(&qs[..q_len], qs_out);

        let qs = limbs_div_exact(ns, ds);
        let qs: &[Limb] = &qs;
        assert_eq!(&qs_out[..qs.len()], qs);

        verify_limbs_div_exact(ns, ds, qs_out);
    };
    // - d_len == 1
    test(&[10; 3], &[6], &[2], &[3]);
    // - leading_zero_limbs == 0
    // - d_len != 1
    // - shift == 0
    // - d_len <= q_len
    test(&[5; 8], &[1, 3, 6, 5, 3], &[1; 3], &[1, 2, 3]);
    // - d_len > q_len
    test(&[0; 5], &[6, 19, 32, 21], &[1, 2, 3], &[6, 7]);
    // - shift != 0
    test(&[0; 5], &[6, 19, 32, 21], &[6, 7], &[1, 2, 3]);
    test(
        &[10; 7],
        &[10200, 20402, 30605, 20402, 10200],
        &[100, 101, 102],
        &[102, 101, 100],
    );
    test(&[10; 3], &[u32::MAX], &[u32::MAX], &[1]);
    test(&[10; 4], &[1, u32::MAX - 1], &[u32::MAX], &[u32::MAX, 0]);
    test(
        &[10; 6],
        &[1, 0, 0, u32::MAX - 1, u32::MAX, u32::MAX],
        &[u32::MAX; 3],
        &[u32::MAX, u32::MAX, u32::MAX, 0],
    );
    // - leading_zero_limbs != 0
    test(
        &[0; 5],
        &[0, 0, 0, 6, 19, 32, 21],
        &[0, 0, 1, 2, 3],
        &[0, 6, 7],
    );
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_fail_1() {
    limbs_div_exact(&[6, 19, 32, 21], &[]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_fail_2() {
    limbs_div_exact(&[6, 19, 32, 21], &[1, 2, 3, 4, 5]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_fail_3() {
    limbs_div_exact(&[6, 19, 32, 21], &[1, 2, 0]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_to_out_fail_1() {
    limbs_div_exact_to_out(&mut [10; 5], &mut [6, 19, 32, 21], &mut []);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_to_out_fail_2() {
    limbs_div_exact_to_out(&mut [10; 5], &mut [6, 19, 32, 21], &mut [1, 2, 3, 4, 5]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_to_out_fail_3() {
    limbs_div_exact_to_out(&mut [10; 5], &mut [6, 19, 32, 21], &mut [1, 2, 0]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_to_out_fail_4() {
    limbs_div_exact_to_out(&mut [10], &mut [6, 19, 32, 21], &mut [1, 2, 3]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_to_out_val_ref_fail_1() {
    limbs_div_exact_to_out_val_ref(&mut [10; 5], &mut [6, 19, 32, 21], &[]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_to_out_val_ref_fail_2() {
    limbs_div_exact_to_out_val_ref(&mut [10; 5], &mut [6, 19, 32, 21], &[1, 2, 3, 4, 5]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_to_out_val_ref_fail_3() {
    limbs_div_exact_to_out_val_ref(&mut [10; 5], &mut [6, 19, 32, 21], &[1, 2, 0]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_to_out_val_ref_fail_4() {
    limbs_div_exact_to_out_val_ref(&mut [10], &mut [6, 19, 32, 21], &[1, 2, 3]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_to_out_ref_val_fail_1() {
    limbs_div_exact_to_out_ref_val(&mut [10; 5], &[6, 19, 32, 21], &mut []);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_to_out_ref_val_fail_2() {
    limbs_div_exact_to_out_ref_val(&mut [10; 5], &[6, 19, 32, 21], &mut [1, 2, 3, 4, 5]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_to_out_ref_val_fail_3() {
    limbs_div_exact_to_out_ref_val(&mut [10; 5], &[6, 19, 32, 21], &mut [1, 2, 0]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_to_out_ref_val_fail_4() {
    limbs_div_exact_to_out_ref_val(&mut [10], &[6, 19, 32, 21], &mut [1, 2, 3]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_to_out_ref_ref_fail_1() {
    limbs_div_exact_to_out_ref_ref(&mut [10; 5], &[6, 19, 32, 21], &[]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_to_out_ref_ref_fail_2() {
    limbs_div_exact_to_out_ref_ref(&mut [10; 5], &[6, 19, 32, 21], &[1, 2, 3, 4, 5]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_to_out_ref_ref_fail_3() {
    limbs_div_exact_to_out_ref_ref(&mut [10; 5], &[6, 19, 32, 21], &[1, 2, 0]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_exact_to_out_ref_ref_fail_4() {
    limbs_div_exact_to_out_ref_ref(&mut [10], &[6, 19, 32, 21], &[1, 2, 3]);
}

#[test]
fn test_div_exact() {
    let test = |s, t, quotient| {
        let u = Natural::from_str(s).unwrap();
        let v = Natural::from_str(t).unwrap();

        let mut x = u.clone();
        x.div_exact_assign(v.clone());
        assert!(x.is_valid());
        assert_eq!(x.to_string(), quotient);

        let mut x = u.clone();
        x.div_exact_assign(&v);
        assert!(x.is_valid());
        assert_eq!(x.to_string(), quotient);

        let q = u.clone().div_exact(v.clone());
        assert!(q.is_valid());
        assert_eq!(q.to_string(), quotient);

        let q = u.clone().div_exact(&v);
        assert!(q.is_valid());
        assert_eq!(q.to_string(), quotient);

        let q = (&u).div_exact(v.clone());
        assert!(q.is_valid());
        assert_eq!(q.to_string(), quotient);

        let q = (&u).div_exact(&v);
        assert!(q.is_valid());
        assert_eq!(q.to_string(), quotient);

        let q = u.div_round(v, Exact).0;
        assert_eq!(q.to_string(), quotient);

        let q = rug::Integer::from_str(s)
            .unwrap()
            .div_exact(&rug::Integer::from_str(t).unwrap());
        assert_eq!(q.to_string(), quotient);
    };
    test("0", "123", "0");
    test("123", "1", "123");
    test("123", "123", "1");
    test("56088", "123", "456");
    test("0", "1000000000000", "0");
    test("1000000000000", "1", "1000000000000");
    test("1000000000000", "1000000000000", "1");
    test("123000000000000", "1000000000000", "123");
    test("123000000000000", "123", "1000000000000");
    test("121932631112635269000000", "123456789000", "987654321000");
    test("8589934590", "4294967295", "2");
    test("18446744065119617025", "4294967295", "4294967295");
    test(
        "851673906388325341550957943071111911557800036845129556099360938813259608650267079456739978\
        1156959952275409185911771336067392377245918291754269000751186715279414560474882570499082990\
        4913122978897463970860833616251189242098804876664368441608727895141238953179204529256780277\
        5978105200286025161944212712977056343127682601975191673217459602567633602198262568921008081\
        9448556670912575287371251190800855926311768876808375177446530243635212748346921654224589861\
        0625170426812525829689862407515510419445335472631905610235915226032848323874067128872385291\
        3730739275467227364692195226129501338887049710586931141309357190341064532366013123280106098\
        6468151628797945455179649866890394481799639832540978091736379482964522229064478167730317490\
        8194108506704480750395054067032502530392147690725919399930683143510771646869931527123340650\
        0547649792331568913460415939722111305270588701531404490040034302102101083691706550376288655\
        2667382899390792494118931379237432071316543313379792218794371176529684614085109418328963817\
        0601432767270419229719490809539776535671938041618536196941370647945336401901450921413823163\
        4059991707077834107830876756821880651429748186401020760113859498185638133726165286481741014\
        9079906337286599226335508424466369316294442004040440528589582239717042654541745348050157252\
        3448224036804997350851153108395928780441635856",
        "147502279655636565600250358452694051893980186696958535174009956523855720107322638159749368\
        0808217479494744305876890972595771484769733857514529616096199394092858302265998260483416016\
        5763904522044264005938281072568140883513713255548643044250086110483617215935636533809248102\
        6926590789142079805638445494760177551776636747830014495012489743990407355232286842071418922\
        9921358409573480901624487977319782755422730834468673438076805532952821406024399006814390166\
        6949827530796971086011267864607814906313334525518102221919643040440267323688341889035864376\
        1377246644579088153222669672271414315240318439843720039808993886410874969340996645010795670\
        2133518716987668865936529827437388042190084309005369564717390726257594902619365180097509576\
        6240189037770619308206906414128686856349950952623970023039440323701643457411485666776354448\
        186307133288106956593939073729500658176632828099789",
        "577397114388109712462006371470162814529304445639807296878809567953200969820156259914159240\
        9106139481288193067515284601342023565222679498917484131095648263181800618990427694244342686\
        4412105186059052689237237088193855584354278755933606296018800151986520872701706693002473648\
        4330061421236425747083307907706860804054565348593527605104495080560663025897787060638154303\
        7631781316565710346299551930891169154491973589315700505458672804104869879731391323700304",
    );
}

#[test]
#[should_panic]
fn div_exact_assign_fail() {
    let mut n = Natural::from(10u32);
    n.div_exact_assign(Natural::ZERO);
}

#[test]
#[should_panic]
fn div_exact_assign_ref_fail() {
    let mut n = Natural::from(10u32);
    n.div_exact_assign(&Natural::ZERO);
}

#[test]
#[should_panic]
fn div_exact_fail() {
    Natural::from(10u32).div_exact(Natural::ZERO);
}

#[test]
#[should_panic]
fn div_exact_val_ref_fail() {
    Natural::from(10u32).div_exact(&Natural::ZERO);
}

#[test]
#[should_panic]
fn div_exact_ref_val_fail() {
    (&Natural::from(10u32)).div_exact(Natural::ZERO);
}

#[test]
#[should_panic]
fn div_exact_ref_ref_fail() {
    (&Natural::from(10u32)).div_exact(&Natural::ZERO);
}

#[test]
fn limbs_modular_invert_limb_properties() {
    unsigned_gen_var_22().test_properties(|x| {
        let inverse = limbs_modular_invert_limb::<Limb>(x);
        assert_eq!(x.wrapping_mul(inverse), 1);
        assert_eq!(limbs_modular_invert_limb::<Limb>(inverse), x);
    });
}

#[test]
fn limbs_div_exact_limb_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    unsigned_vec_unsigned_pair_gen_var_29().test_properties_with_config(&config, |(xs, y)| {
        let expected_result = Natural::from_limbs_asc(&xs).div_exact(Natural::from(y));
        assert_eq!(
            Natural::from_owned_limbs_asc(limbs_div_exact_limb(&xs, y)),
            expected_result
        );
        assert_eq!(
            Natural::from_owned_limbs_asc(limbs_div_exact_limb_no_special_3(&xs, y)),
            expected_result
        );
    });
}

#[test]
fn limbs_div_exact_limb_to_out_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    unsigned_vec_unsigned_vec_unsigned_triple_gen_var_14().test_properties_with_config(
        &config,
        |(mut out, xs, y)| {
            let old_out = out.clone();
            limbs_div_exact_limb_to_out::<DoubleLimb, Limb>(&mut out, &xs, y);
            let len = xs.len();
            let expected_result = Natural::from_limbs_asc(&xs).div_exact(Natural::from(y));
            assert_eq!(Natural::from_limbs_asc(&out[..len]), expected_result);
            assert_eq!(&out[len..], &old_out[len..]);

            let mut out = old_out.to_vec();
            limbs_div_exact_limb_to_out_no_special_3::<DoubleLimb, Limb>(&mut out, &xs, y);
            let len = xs.len();
            let expected_result = Natural::from_owned_limbs_asc(xs).div_exact(Natural::from(y));
            assert_eq!(Natural::from_limbs_asc(&out[..len]), expected_result);
            assert_eq!(&out[len..], &old_out[len..]);
        },
    );
}

#[test]
fn limbs_div_exact_limb_in_place_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    unsigned_vec_unsigned_pair_gen_var_29().test_properties_with_config(&config, |(mut xs, y)| {
        let old_xs = xs.clone();
        limbs_div_exact_limb_in_place(&mut xs, y);
        let expected_result = Natural::from_limbs_asc(&old_xs).div_exact(Natural::from(y));
        assert_eq!(Natural::from_owned_limbs_asc(xs), expected_result);

        let mut xs = old_xs.to_vec();
        limbs_div_exact_limb_in_place_no_special_3(&mut xs, y);
        let expected_result = Natural::from_owned_limbs_asc(old_xs).div_exact(Natural::from(y));
        assert_eq!(Natural::from_owned_limbs_asc(xs), expected_result);
    });
}

#[test]
fn limbs_div_exact_3_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    unsigned_vec_gen_var_5().test_properties_with_config(&config, |xs| {
        let q_limbs = Natural::from_owned_limbs_asc(limbs_div_exact_3(&xs));
        assert_eq!(
            Natural::from_owned_limbs_asc(limbs_div_exact_limb(&xs, 3)),
            q_limbs,
        );
        assert_eq!(
            Natural::from_owned_limbs_asc(xs).div_exact(Natural::from(3u32)),
            q_limbs
        );
    });
}

#[test]
fn limbs_div_exact_3_to_out_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    unsigned_vec_pair_gen_var_13().test_properties_with_config(&config, |(mut out, xs)| {
        let old_out = out.clone();
        limbs_div_exact_3_to_out::<DoubleLimb, Limb>(&mut out, &xs);
        let len = xs.len();
        assert_eq!(
            Natural::from_limbs_asc(&out[..len]),
            Natural::from_limbs_asc(&xs).div_exact(Natural::from(3u32))
        );
        assert_eq!(&out[len..], &old_out[len..]);

        let mut out_alt = old_out.clone();
        limbs_div_exact_limb_to_out::<DoubleLimb, Limb>(&mut out_alt, &xs, 3);
        assert_eq!(out_alt, out);

        let mut out_alt = old_out;
        limbs_div_exact_3_to_out_alt(&mut out_alt, &xs);
        assert_eq!(out_alt, out);
    });
}

#[test]
fn limbs_div_exact_3_in_place_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    unsigned_vec_gen_var_5().test_properties_with_config(&config, |mut xs| {
        let old_xs = xs.clone();
        limbs_div_exact_3_in_place(&mut xs);
        assert_eq!(
            Natural::from_limbs_asc(&xs),
            Natural::from_limbs_asc(&old_xs).div_exact(Natural::from(3u32))
        );

        let mut xs_alt = old_xs.to_vec();
        limbs_div_exact_limb_in_place(&mut xs_alt, 3);
        assert_eq!(xs_alt, xs);

        let mut xs_alt = old_xs.to_vec();
        limbs_div_exact_3_in_place_alt(&mut xs_alt);
        assert_eq!(xs_alt, xs);
    });
}

#[test]
fn limbs_modular_invert_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 512);
    config.insert("mean_stripe_n", 64 << Limb::LOG_WIDTH);
    unsigned_vec_pair_gen_var_12().test_properties_with_config(&config, |(mut is, ds)| {
        let n = ds.len();
        let mut scratch = vec![0; limbs_modular_invert_scratch_len(n)];
        limbs_modular_invert(&mut is, &ds, &mut scratch);
        verify_limbs_modular_invert(&ds, &is[..n]);
    });
}

#[test]
fn limbs_modular_div_mod_schoolbook_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 32);
    config.insert("mean_stripe_n", 16 << Limb::LOG_WIDTH);
    large_type_gen_var_14().test_properties_with_config(
        &config,
        |(mut qs, mut ns, ds, inverse)| {
            let ns_old = ns.clone();
            let borrow = limbs_modular_div_mod_schoolbook(&mut qs, &mut ns, &ds, inverse);
            let q_len = ns.len() - ds.len();
            verify_limbs_modular_div_mod(&ns_old, &ds, borrow, &qs[..q_len], &ns[q_len..]);
        },
    );
}

#[test]
fn limbs_modular_div_mod_divide_and_conquer_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 128);
    config.insert("mean_stripe_n", 64 << Limb::LOG_WIDTH);
    large_type_gen_var_15().test_properties_with_config(
        &config,
        |(mut qs, mut ns, ds, inverse)| {
            let ns_old = ns.clone();
            let borrow = limbs_modular_div_mod_divide_and_conquer(&mut qs, &mut ns, &ds, inverse);
            let q_len = ns.len() - ds.len();
            verify_limbs_modular_div_mod(&ns_old, &ds, borrow, &qs[..q_len], &ns[q_len..]);
        },
    );
}

#[test]
fn limbs_modular_div_mod_barrett_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 512);
    config.insert("mean_stripe_n", 64 << Limb::LOG_WIDTH);
    unsigned_vec_quadruple_gen_var_2().test_properties_with_config(
        &config,
        |(mut qs, mut rs, ns, ds)| {
            let mut scratch =
                vec![0; limbs_modular_div_mod_barrett_scratch_len(ns.len(), ds.len())];
            let borrow = limbs_modular_div_mod_barrett(&mut qs, &mut rs, &ns, &ds, &mut scratch);
            let q_len = ns.len() - ds.len();
            verify_limbs_modular_div_mod(&ns, &ds, borrow, &qs[..q_len], &rs[..ds.len()]);
        },
    );
}

#[test]
fn limbs_modular_div_schoolbook_properties() {
    large_type_gen_var_13().test_properties(|(mut qs, mut ns, ds, inverse)| {
        let ns_old = ns.clone();
        limbs_modular_div_schoolbook(&mut qs, &mut ns, &ds, inverse);
        verify_limbs_modular_div_neg(&ns_old, &ds, &qs);

        let mut ns = ns_old.to_vec();
        limbs_modular_div_schoolbook_in_place(&mut ns, &ds, inverse);
        assert_eq!(ns, &qs[..ns.len()]);
    });
}

#[test]
fn limbs_modular_div_divide_and_conquer_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 512);
    config.insert("mean_stripe_n", 64 << Limb::LOG_WIDTH);
    large_type_gen_var_16().test_properties_with_config(
        &config,
        |(mut qs, mut ns, ds, inverse)| {
            let ns_old = ns.clone();
            limbs_modular_div_divide_and_conquer(&mut qs, &mut ns, &ds, inverse);
            verify_limbs_modular_div(&ns_old, &ds, &qs);
        },
    );
}

#[test]
fn limbs_modular_div_barrett_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 512);
    config.insert("mean_stripe_n", 64 << Limb::LOG_WIDTH);
    unsigned_vec_triple_gen_var_46().test_properties_with_config(&config, |(mut qs, ns, ds)| {
        let mut scratch = vec![0; limbs_modular_div_barrett_scratch_len(ns.len(), ds.len())];
        limbs_modular_div_barrett(&mut qs, &ns, &ds, &mut scratch);
        verify_limbs_modular_div(&ns, &ds, &qs[..ns.len()]);
    });
}

#[test]
fn limbs_modular_div_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 512);
    config.insert("mean_stripe_n", 64 << Limb::LOG_WIDTH);
    unsigned_vec_triple_gen_var_47().test_properties_with_config(
        &config,
        |(mut qs, mut ns, ds)| {
            let qs_old = qs.clone();
            let ns_old = ns.clone();
            let mut scratch = vec![0; limbs_modular_div_scratch_len(ns.len(), ds.len())];
            limbs_modular_div(&mut qs, &mut ns, &ds, &mut scratch);
            let result = qs;

            let mut qs = qs_old.to_vec();
            let ns = ns_old;
            let mut scratch = vec![0; limbs_modular_div_ref_scratch_len(ns.len(), ds.len())];
            limbs_modular_div_ref(&mut qs, &ns, &ds, &mut scratch);
            assert_eq!(qs, result);

            verify_limbs_modular_div(&ns, &ds, &qs[..ns.len()]);
        },
    );
}

#[test]
fn limbs_div_exact_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_length_n", 512);
    config.insert("mean_stripe_n", 64 << Limb::LOG_WIDTH);
    unsigned_vec_triple_gen_var_48().test_properties_with_config(
        &config,
        |(mut qs, mut ns, mut ds)| {
            let qs_old = qs.clone();
            let ns_old = ns.clone();
            let ds_old = ds.clone();
            limbs_div_exact_to_out(&mut qs, &mut ns, &mut ds);
            let result = qs;

            let mut qs = qs_old.to_vec();
            let mut ns = ns_old.to_vec();
            let mut ds = ds_old.to_vec();
            limbs_div_exact_to_out_val_ref(&mut qs, &mut ns, &ds);
            assert_eq!(qs, result);

            let mut qs = qs_old.to_vec();
            let ns = ns_old;
            limbs_div_exact_to_out_ref_val(&mut qs, &ns, &mut ds);
            assert_eq!(qs, result);

            let mut qs = qs_old.to_vec();
            let ds = ds_old;
            limbs_div_exact_to_out_ref_ref(&mut qs, &ns, &ds);
            assert_eq!(qs, result);

            let q_len = ns.len() - ds.len() + 1;
            let qs = limbs_div_exact(&ns, &ds);
            assert_eq!(qs, &result[..q_len]);

            verify_limbs_div_exact(&ns, &ds, &qs[..q_len]);
        },
    );
}

#[test]
fn div_exact_properties() {
    natural_pair_gen_var_6().test_properties(|(x, y)| {
        let mut mut_x = x.clone();
        mut_x.div_exact_assign(&y);
        assert!(mut_x.is_valid());
        let q = mut_x;

        let mut mut_x = x.clone();
        mut_x.div_exact_assign(y.clone());
        assert!(mut_x.is_valid());
        assert_eq!(mut_x, q);

        let q_alt = (&x).div_exact(&y);
        assert!(q_alt.is_valid());
        assert_eq!(q_alt, q);

        let q_alt = (&x).div_exact(y.clone());
        assert!(q_alt.is_valid());
        assert_eq!(q_alt, q);

        let q_alt = x.clone().div_exact(&y);
        assert!(q_alt.is_valid());
        assert_eq!(q_alt, q);

        let q_alt = x.clone().div_exact(y.clone());
        assert!(q_alt.is_valid());
        assert_eq!(q_alt, q);

        let q_alt = (&x).div_round(&y, Exact).0;
        assert_eq!(q_alt, q);

        assert_eq!(
            Natural::exact_from(&rug::Integer::from(&x).div_exact(&rug::Integer::from(&y))),
            q
        );

        assert_eq!(q * y, x);
    });

    natural_gen().test_properties(|n| {
        assert_eq!((&n).div_exact(Natural::ONE), n);
    });

    natural_gen_var_2().test_properties(|n| {
        assert_eq!(Natural::ZERO.div_exact(&n), 0);
        assert_eq!((&n).div_exact(&n), 1);
    });

    unsigned_pair_gen_var_11::<Limb>().test_properties(|(x, y)| {
        assert_eq!(Natural::from(x).div_exact(Natural::from(y)), x.div_exact(y));
    });
}
