// Copyright © 2025 Mikhail Hogrefe
//
// This file is part of Malachite.
//
// Malachite is free software: you can redistribute it and/or modify it under the terms of the GNU
// Lesser General Public License (LGPL) as published by the Free Software Foundation; either version
// 3 of the License, or (at your option) any later version. See <https://www.gnu.org/licenses/>.

use malachite_base::num::arithmetic::traits::{CheckedDiv, DivMod};
use malachite_base::num::basic::integers::PrimitiveInt;
use malachite_base::num::basic::traits::{One, Zero};
use malachite_base::num::conversion::traits::ExactFrom;
use malachite_base::strings::ToDebugString;
use malachite_base::test_util::generators::common::GenConfig;
use malachite_base::test_util::generators::{
    unsigned_pair_gen_var_12, unsigned_vec_unsigned_pair_gen_var_22,
    unsigned_vec_unsigned_vec_unsigned_triple_gen_var_13,
};
use malachite_nz::natural::Natural;
use malachite_nz::natural::arithmetic::div::{
    limbs_div, limbs_div_barrett, limbs_div_barrett_approx, limbs_div_barrett_approx_scratch_len,
    limbs_div_barrett_scratch_len, limbs_div_divide_and_conquer,
    limbs_div_divide_and_conquer_approx, limbs_div_divisor_of_limb_max_with_carry_in_place,
    limbs_div_divisor_of_limb_max_with_carry_to_out, limbs_div_schoolbook,
    limbs_div_schoolbook_approx, limbs_div_to_out, limbs_div_to_out_ref_ref,
    limbs_div_to_out_ref_val, limbs_div_to_out_val_ref,
};
use malachite_nz::natural::arithmetic::div::{
    limbs_div_limb, limbs_div_limb_in_place, limbs_div_limb_to_out,
};
use malachite_nz::natural::arithmetic::div_mod::limbs_two_limb_inverse_helper;
use malachite_nz::platform::{DoubleLimb, Limb};
use malachite_nz::test_util::generators::{
    large_type_gen_var_10, large_type_gen_var_11, large_type_gen_var_12, natural_gen,
    natural_gen_var_2, natural_pair_gen, natural_pair_gen_var_5, natural_pair_gen_var_6,
    unsigned_vec_triple_gen_var_42, unsigned_vec_triple_gen_var_43, unsigned_vec_triple_gen_var_44,
    unsigned_vec_unsigned_unsigned_triple_gen_var_9,
};
use malachite_nz::test_util::natural::arithmetic::div::{
    limbs_div_limb_in_place_alt, limbs_div_limb_to_out_alt,
};
use num::BigUint;
use num::CheckedDiv as NumCheckedDiv;
use rug;
use std::str::FromStr;

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_div_limb_and_limbs_div_limb_in_place() {
    let test = |ns: &[Limb], d: Limb, q: &[Limb]| {
        assert_eq!(limbs_div_limb(ns, d), q);

        let mut ns = ns.to_vec();
        let ns_old = ns.clone();
        limbs_div_limb_in_place(&mut ns, d);
        assert_eq!(ns, q);

        let mut ns = ns_old;
        limbs_div_limb_in_place_alt(&mut ns, d);
        assert_eq!(ns, q);
    };
    test(&[0, 0], 2, &[0, 0]);
    test(&[6, 7], 1, &[6, 7]);
    test(&[6, 7], 2, &[0x80000003, 3]);
    test(&[100, 101, 102], 10, &[1288490198, 858993469, 10]);
    test(&[123, 456], 789, &[2482262467, 0]);
    test(&[u32::MAX, u32::MAX], 2, &[u32::MAX, 0x7fffffff]);
    test(&[u32::MAX, u32::MAX], 3, &[0x55555555, 0x55555555]);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_limb_fail_1() {
    limbs_div_limb(&[10], 10);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_limb_fail_2() {
    limbs_div_limb(&[10, 10], 0);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_limb_in_place_fail_1() {
    limbs_div_limb_in_place(&mut [10], 10);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_limb_in_place_fail_2() {
    limbs_div_limb_in_place(&mut [10, 10], 0);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_div_limb_to_out() {
    let test = |out_before: &[Limb], ns: &[Limb], d: Limb, out_after: &[Limb]| {
        let mut out = out_before.to_vec();
        limbs_div_limb_to_out(&mut out, ns, d);
        assert_eq!(out, out_after);

        let mut out = out_before.to_vec();
        limbs_div_limb_to_out_alt(&mut out, ns, d);
        assert_eq!(out, out_after);
    };
    test(&[10, 10, 10, 10], &[0, 0], 2, &[0, 0, 10, 10]);
    test(&[10, 10, 10, 10], &[6, 7], 1, &[6, 7, 10, 10]);
    test(&[10, 10, 10, 10], &[6, 7], 2, &[0x80000003, 3, 10, 10]);
    test(
        &[10, 10, 10, 10],
        &[100, 101, 102],
        10,
        &[1288490198, 858993469, 10, 10],
    );
    test(
        &[10, 10, 10, 10],
        &[123, 456],
        789,
        &[2482262467, 0, 10, 10],
    );
    test(
        &[10, 10, 10, 10],
        &[u32::MAX, u32::MAX],
        2,
        &[u32::MAX, 0x7fffffff, 10, 10],
    );
    test(
        &[10, 10, 10, 10],
        &[u32::MAX, u32::MAX],
        3,
        &[0x55555555, 0x55555555, 10, 10],
    );
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_limb_to_out_fail_1() {
    limbs_div_limb_to_out(&mut [10], &[10], 10);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_limb_to_out_fail_2() {
    limbs_div_limb_to_out(&mut [10, 10], &[10, 10], 0);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
#[should_panic]
fn limbs_div_limb_to_out_fail_3() {
    limbs_div_limb_to_out(&mut [10], &[10, 10], 10);
}

fn verify_limbs_div(qs_in: &[Limb], ns: &[Limb], ds: &[Limb], q_highest: bool, qs_out: &[Limb]) {
    let n = Natural::from_limbs_asc(ns);
    let d = Natural::from_limbs_asc(ds);
    let expected_q = &n / &d;
    let base_q_len = ns.len() - ds.len();
    let mut qs = qs_out[..base_q_len].to_vec();
    if q_highest {
        qs.push(1);
    }
    let q = Natural::from_owned_limbs_asc(qs);
    assert_eq!(q, expected_q);
    assert_eq!(&qs_in[base_q_len..], &qs_out[base_q_len..]);
    let r = n - q * &d;
    assert!(r < d);
}

#[test]
fn test_limbs_div_schoolbook() {
    let test = |qs_in: &[Limb], ns_in: &[Limb], ds: &[Limb], q_highest, qs_out: &[Limb]| {
        let mut qs = qs_in.to_vec();
        let mut ns = ns_in.to_vec();
        let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
        assert_eq!(limbs_div_schoolbook(&mut qs, &mut ns, ds, d_inv), q_highest);
        assert_eq!(qs, qs_out);
        verify_limbs_div(qs_in, ns_in, ds, q_highest, &qs);
    };
    #[cfg(feature = "32_bit_limbs")]
    {
        // - q_len < d_len_m_1
        // - !highest_q first time
        // - !(!flag || n_1 < Limb::try_from(d_len).unwrap())
        test(&[10], &[1, 2, 3], &[3, 4, 0x80000000], false, &[10]);
        // - q_len >= d_len_m_1
        // - !(n_1 == d_1 && ns[d_len_s_m_1] == d_2)
        // - !carry first time
        // - d_len_s >= 2
        // - !(!flag || n_1 >= d_1) first time
        // - !carry second time
        // - !(!flag || n_1 >= d_1) second time
        test(
            &[10, 10, 10, 10],
            &[1, 2, 3, 4, 5, 6],
            &[3, 4, 0x80000000],
            false,
            &[4294967207, 9, 12, 10],
        );
        // - !flag || n_1 < Limb::try_from(d_len).unwrap()
        // - !highest_q second time
        // - q_len != 0
        // - limbs_sub_limb_in_place(ns_hi, limbs_sub_mul_limb_same_length_in_place_left(ns_lo, qs,
        //   ds[i])) fourth time
        // - n_1 == 0 third time
        test(&[10], &[0, 0, 0, 1], &[1, 0, 0x80000000], false, &[1]);
        // - q_len == 0 second time
        test(&[10], &[0; 3], &[0, 0, 0x80000000], false, &[10]);
        // - *ns_last >= carry
        test(&[10; 3], &[0; 5], &[0, 0, 0x80000000], false, &[0, 0, 10]);
        // - n_1 != 0 third time
        test(
            &[10; 2],
            &[0, 0, 1, 1],
            &[1, 0, 0x80000000],
            false,
            &[2, 10],
        );
        // - highest_q first time
        test(
            &[10, 10],
            &[
                2460989955, 642673607, 1971681331, 2338977519, 475442971, 2516527409, 3470931196,
                4262052990,
            ],
            &[
                1430162564, 1873089879, 3090281851, 2861182896, 2502166555, 2486624904, 960387080,
                3073064701,
            ],
            true,
            &[10, 10],
        );
        // - !flag || n_1 >= d_1 second time
        // - n_1 == carry second time
        test(
            &[10; 2],
            &[0, 0, 1, 0, 1],
            &[0, 1, 0x80000000],
            false,
            &[u32::MAX, 1],
        );
        // - carry second time
        test(
            &[10; 2],
            &[0, 0, 0, 0, 1],
            &[1, 0, 0x80000000],
            false,
            &[u32::MAX, 1],
        );
        // - *ns_last < carry
        // - n_1 != 0 first time
        // - highest_q second time
        // - !carry third time
        test(
            &[10; 20],
            &[
                4294966784,
                16383,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xff000000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                4095,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xfffffffc,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0xfffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xffffff00,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                15,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xf0000000,
                16383,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xfffff000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                4095,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xfffffff0,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                2540339372,
            ],
            true,
            &[
                2788498651, 3973284986, 2817551825, 2648982666, 1871496085, 1183691368, 1920086634,
                2156916676, 2580901671, 858626385, 3833802894, 4255330726, 3612427555, 2966560147,
                10, 10, 10, 10, 10, 10,
            ],
        );
        // - carry third time
        // - n_1 != 0 second time
        test(
            &[10; 8],
            &[
                0,
                0,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                511,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4286578688,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                63,
                0,
                0,
                0,
                0,
                4294966784,
                u32::MAX,
                255,
                4292870144,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[
                u32::MAX,
                31,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xfffff000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                2874223232,
            ],
            true,
            &[
                3459164144, 2642691284, 1062969477, 1392962372, 1677352416, 1850494842, 2123025524,
                10,
            ],
        );
        // - !flag || n_1 >= d_1 first time
        // - n_1 == carry first time
        test(
            &[10; 10],
            &[
                0,
                0,
                0,
                0,
                0,
                0xfffffff8,
                u32::MAX,
                0xfffff,
                0,
                0,
                0xffff0000,
                u32::MAX,
                0x1fffff,
                0,
                0,
                0,
                0,
                4294959104,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[
                4160749568,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x7fffff,
                0,
                0,
                0,
                0,
                u32::MAX,
                3500471700,
            ],
            true,
            &[
                1188137339,
                3726116623,
                2078060623,
                u32::MAX,
                2498443049,
                1067076977,
                2528078370,
                1716103827,
                974820792,
                10,
            ],
        );
        // - n_1 == 0 first time
        test(
            &[10; 20],
            &[
                0,
                0xfffffffc,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x7fffffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294967232,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0xffffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294966784,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xffffff00,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x7ffffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                3828608089,
            ],
            true,
            &[
                2643432265, 2230909280, 3987412564, 1770201633, 2297365160, 1655451772, 3061719465,
                182613124, 3034480808, 3038313569, 3464266718, 249914758, 4270196816, 4141866692,
                91710029, 4264403435, 2579237268, 67618125, 523165989, 10,
            ],
        );
        // - n_1 != carry first time
        // - flag && n_1 < carry first time
        test(
            &[10; 73],
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294966784,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x7fff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294967232,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                1023,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xf0000000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0xfffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                u32::MAX - 1,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294836224,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                4024661822,
            ],
            true,
            &[
                201450768,
                518871072,
                3015445987,
                3944509552,
                208835829,
                3460787832,
                1884898895,
                2547198435,
                186469778,
                3505271765,
                1840399470,
                2685063887,
                2953363253,
                1530364228,
                3130405732,
                893752596,
                3868726834,
                4151054974,
                2514125465,
                3161927869,
                2354681787,
                1962216954,
                2851892627,
                2800023632,
                4197573593,
                3243049600,
                922941226,
                2202547330,
                3061793011,
                1583790204,
                3045500683,
                440439528,
                2631483812,
                1993907261,
                2362030668,
                4006507490,
                u32::MAX - 1,
                2180774078,
                372389172,
                842953352,
                3854387892,
                1420082779,
                4086327226,
                1481011735,
                2795106458,
                2092419963,
                1299755146,
                3455955436,
                2935373316,
                2973368959,
                208681747,
                1883779895,
                599992803,
                1583705426,
                4184839928,
                352155351,
                1499776059,
                3341886577,
                2973508834,
                452082451,
                98512703,
                1051917695,
                3372026069,
                2092419965,
                1233174284,
                2711177091,
                1249466612,
                3854527767,
                1663483483,
                2301060034,
                1932936627,
                288459805,
                10,
            ],
        );
        // - carry first time
        test(
            &[10; 45],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                4294967279,
                u32::MAX,
                u32::MAX,
                0,
                4294966272,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294959104,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0xfffff,
                0,
                0,
                0,
                0,
                0xffffff00,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[0xffff0000, u32::MAX, 16383, 3221225472, u32::MAX, u32::MAX, 2953571387],
            true,
            &[
                1344119085,
                4210855069,
                928035132,
                841350331,
                528284037,
                1219359454,
                3548354897,
                1057752790,
                3847832303,
                1950615424,
                782270048,
                1691663352,
                2130612726,
                2986797999,
                2134900501,
                367200458,
                366133304,
                4140157070,
                1119890783,
                2133610345,
                309344350,
                1553079209,
                39974261,
                4150943609,
                1565848679,
                3244273595,
                2309629507,
                3647061699,
                1476447016,
                2423314465,
                988904489,
                3076842837,
                370106892,
                2669774090,
                81048948,
                2207992586,
                u32::MAX,
                2794855737,
                2919419242,
                2509602689,
                685991398,
                3934508174,
                2079332445,
                1950605148,
                10,
            ],
        );
        // - n_1 != carry second time
        // - flag && n_1 < carry second time
        test(
            &[10; 10],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x7fff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xff000000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0xfffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294836224,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294967264,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                3833725436,
            ],
            true,
            &[
                u32::MAX - 1,
                150659924,
                1536731131,
                3027101341,
                827397342,
                835599031,
                2855838328,
                1016945560,
                516734631,
                10,
            ],
        );
        // - !(flag && n_1 < carry) first time
        test(
            &[0; 3],
            &[
                0xffff0000,
                2047,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294965248,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x1fffffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4286578688,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                8191,
            ],
            &[
                0xffff0000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0xfffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xff000000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                255,
                0,
                0,
                0,
                0,
                0,
                4227858432,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            false,
            &[u32::MAX, u32::MAX, 8191],
        );
        // - q_len == 0 first time
        test(
            &[0],
            &[0, 0, 0x80000000],
            &[0, 0x80000000, 0x80000000],
            false,
            &[0],
        );
    }
    #[cfg(not(feature = "32_bit_limbs"))]
    {
        // - !limbs_sub_limb_in_place(ns_hi, limbs_sub_mul_limb_same_length_in_place_left(ns_lo, qs,
        //   ds[i])) fourth time
        test(
            &[10],
            &[0, 0, 1, 1],
            &[0, 1, 0x8000000000000000],
            false,
            &[2],
        );
        // - n_1 == d_1 && ns[d_len_s_m_1] == d_2
        // - !(flag && n_1 < carry) second time
        test(
            &[10; 518],
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709027328,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
            ],
            &[
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
                u64::MAX,
            ],
            true,
            &[
                u64::MAX,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709027328,
                u64::MAX,
                u64::MAX,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709027328,
                u64::MAX,
                u64::MAX,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709027328,
                u64::MAX,
                u64::MAX,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709027328,
                u64::MAX,
                u64::MAX,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709027328,
                u64::MAX,
                u64::MAX,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709027328,
                u64::MAX,
                u64::MAX,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                18446744073709027328,
                u64::MAX,
                u64::MAX,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
            ],
        );
        test(
            &[10; 3],
            &[
                0x20000000,
                4503599090491408,
                31664835368329200,
                18410716373991817215,
                18157950747604419646,
                2301339547025015295,
            ],
            &[18446744073172680704, 18442240474082189295, 18410716376202215423],
            false,
            &[u64::MAX, 0, 2305843009213431808],
        );
    }
}

#[test]
#[should_panic]
fn limbs_div_schoolbook_fail_1() {
    let ds = &[3, 0x80000000];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_div_schoolbook(&mut [10], &mut [1, 2, 3], ds, d_inv);
}

#[test]
#[should_panic]
fn limbs_div_schoolbook_fail_2() {
    let ds = &[3, 4, 5, 0x80000000];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_div_schoolbook(&mut [10], &mut [1, 2, 3], ds, d_inv);
}

#[test]
#[should_panic]
fn limbs_div_schoolbook_fail_3() {
    let ds = &[3, 4, 0x80000000];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_div_schoolbook(&mut [10], &mut [1, 2, 3, 4, 5], ds, d_inv);
}

#[test]
#[should_panic]
fn limbs_div_schoolbook_fail_4() {
    let ds = &[3, 4, 5];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_div_schoolbook(&mut [10], &mut [1, 2, 3], ds, d_inv);
}

#[cfg(feature = "32_bit_limbs")]
#[test]
fn test_limbs_div_divide_and_conquer() {
    let test = |qs_in: &[Limb], ns: &[Limb], ds: &[Limb], q_highest, qs_out: &[Limb]| {
        let mut qs = qs_in.to_vec();
        let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
        assert_eq!(
            limbs_div_divide_and_conquer(&mut qs, ns, ds, d_inv),
            q_highest
        );
        assert_eq!(qs, qs_out);
        verify_limbs_div(qs_in, ns, ds, q_highest, &qs);
    };
    {
        // - *scratch_2_head != 0
        test(
            &[10; 4],
            &[1, 2, 3, 4, 5, 6, 7, 8, 9],
            &[3, 4, 5, 6, 7, 0x80000000],
            false,
            &[4294967057, 15, 18, 10],
        );
        // - *scratch_2_head == 0
        // - !(highest_q && limbs_slice_add_same_length_in_place_left(&mut scratch_init[q_len..],
        //   ds) || limbs_cmp_same_length(scratch_init, ns) == Greater)
        test(
            &[10; 3],
            &[0; 9],
            &[0, 0, 0, 0, 0, 0x80000000],
            false,
            &[0, 0, 0],
        );
        test(
            &[10; 3],
            &[0, 0, 0, 0, 0, 0, 0, 0, 1],
            &[0, 0, 0, 0, 0, 0x80000000],
            false,
            &[0, 0, 2],
        );
        // - highest_q && limbs_slice_add_same_length_in_place_left(&mut scratch_init[q_len..], ds)
        //   || limbs_cmp_same_length(scratch_init, ns) == Greater
        test(
            &[10; 3],
            &[0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0],
            &[0, 0, 0, 1, 0, 0, 0, 0x80000000],
            false,
            &[1, 0, 0],
        );
    }
}

#[test]
#[should_panic]
fn limbs_div_divide_and_conquer_fail_1() {
    let ds = &[3, 4, 5, 6, 0x80000000];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_div_divide_and_conquer(&mut [10; 4], &[1, 2, 3, 4, 5, 6, 7, 8, 9], ds, d_inv);
}

#[test]
#[should_panic]
fn limbs_div_divide_and_conquer_fail_2() {
    let ds = &[3, 4, 5, 6, 7, 0x80000000];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_div_divide_and_conquer(&mut [10; 4], &[1, 2, 3, 4, 5, 6, 7, 8], ds, d_inv);
}

#[test]
#[should_panic]
fn limbs_div_divide_and_conquer_fail_3() {
    let ds = &[3, 4, 5, 6, 7, 0x80000000];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_div_divide_and_conquer(&mut [10, 10], &[1, 2, 3, 4, 5, 6, 7, 8, 9], ds, d_inv);
}

#[test]
#[should_panic]
fn limbs_div_divide_and_conquer_fail_4() {
    let ds = &[3, 4, 5, 6, 7, 8];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_div_divide_and_conquer(&mut [10; 4], &[1, 2, 3, 4, 5, 6, 7, 8, 9], ds, d_inv);
}

fn verify_limbs_div_approx(
    qs_in: &[Limb],
    ns_in: &[Limb],
    ds: &[Limb],
    q_highest: bool,
    qs_out: &[Limb],
) {
    let n = Natural::from_limbs_asc(ns_in);
    let d = Natural::from_limbs_asc(ds);
    let (expected_q, expected_r) = (&n).div_mod(&d);
    let base_q_len = ns_in.len() - ds.len();
    let mut qs = qs_out[..base_q_len].to_vec();
    if q_highest {
        qs.push(1);
    }
    let q = Natural::from_owned_limbs_asc(qs);
    let q_is_one_too_large = q != expected_q;
    if q_is_one_too_large {
        assert_eq!(q, expected_q + Natural::ONE);
        assert_eq!((q - Natural::ONE) * &d + expected_r, n);
    } else {
        assert_eq!(q * &d + expected_r, n);
    }
    assert_eq!(&qs_in[base_q_len..], &qs_out[base_q_len..]);
}

#[test]
fn test_limbs_div_barrett() {
    let test = |qs_in: &[Limb], ns: &[Limb], ds: &[Limb], q_highest, qs_out: &[Limb]| {
        let mut qs = qs_in.to_vec();
        let mut scratch = vec![0; limbs_div_barrett_scratch_len(ns.len(), ds.len())];
        assert_eq!(limbs_div_barrett(&mut qs, ns, ds, &mut scratch), q_highest);
        assert_eq!(qs, qs_out);
        verify_limbs_div(qs_in, ns, ds, q_highest, &qs);
    };
    #[cfg(feature = "32_bit_limbs")]
    {
        // - q_len < d_len
        // - ghost_limb
        // - *scratch_2_head <= 6
        // - !highest_q third time
        // - !(highest_q && limbs_slice_add_same_length_in_place_left(&mut rs[q_len..], ds) ||
        //   limbs_cmp_same_length(&mut rs, ns) == Greater)
        test(&[10; 3], &[1, 2, 3], &[0, 0x80000000], false, &[6, 10, 10]);
        test(&[10; 3], &[1, 2, 3], &[3, 0x80000000], false, &[6, 10, 10]);
        // - q_len >= d_len
        // - !highest_q first time
        // - !_limbs_div_barrett_approx(&mut scratch_2, &mut rs, ds, scratch)
        // - *scratch_2_head > 4
        test(
            &[10; 3],
            &[1, 2, 3, 4],
            &[0, 0x80000000],
            false,
            &[6, 8, 10],
        );
        test(
            &[10; 3],
            &[1, 2, 3, 4],
            &[3, 0x80000000],
            false,
            &[5, 8, 10],
        );
        // - *scratch_2_head > 6
        test(&[10; 3], &[0, 0, 1], &[0, 0x80000001], false, &[1, 10, 10]);
        // - !ghost_limb
        test(
            &[10; 3],
            &[0, 0, 0, 0],
            &[0, 0, 0x80000000],
            false,
            &[0, 10, 10],
        );
        // - highest_q && limbs_slice_add_same_length_in_place_left(&mut rs[q_len..], ds) ||
        //   limbs_cmp_same_length(&mut rs, ns) == Greater
        // - !limbs_sub_limb_to_out(qs, &tp[1..q_len + 1], 1) second time
        test(&[10; 3], &[0, 0, 1], &[1, 0x80000000], false, &[1, 10, 10]);
        // - *scratch_2_head <= 4
        // - !highest_q second time
        // - !(highest_q && limbs_slice_add_same_length_in_place_left(&mut rs[q_len..d_len], ds) ||
        //   limbs_cmp_same_length(&rs[..n_len], ns) == Greater)
        test(
            &[10; 3],
            &[0, 0, 0, 0],
            &[0, 0x80000000],
            false,
            &[0, 0, 10],
        );
        // - highest_q && limbs_slice_add_same_length_in_place_left(&mut rs[q_len..d_len], ds) ||
        //   limbs_cmp_same_length(&rs[..n_len], ns) == Greater
        // - !limbs_sub_limb_to_out(qs, &tp[1..q_len + 1], 1) first time
        test(
            &[10; 3],
            &[0, 0, 1, 0],
            &[1, 0x80000000],
            false,
            &[1, 0, 10],
        );
        // - highest_q first time
        test(
            &[10; 337],
            &[
                2119562458, 822658147, 3797927043, 3236945010, 3998585665, 1799547707, 694717600,
                81695911, 596552600, 1030767834, 2420116557, 2879241862, 4265568378, 644932084,
                525117944, 1830433723, 3385005819, 1023897437, 3938627008, 3218345466, 471356423,
                1008605584, 192751188, 2084112099, 2804133613, 3079231960, 3746102582, 3887941757,
                2403297368, 3093735039, 2439864684, 2473555841, 2532233722, 2609655906, 3449112520,
                2817574574, 1129609143, 821518979, 991339292, 2760112178, 890698714, 990807460,
                2521797827, 3797612523, 1500537907, 3973412117, 2699035435, 2317986252, 2861160946,
                2101011897, 1732640676, 3352339942, 3286471253, 1371519955, 1379608062, 2699834414,
                3261360638, 226259522, 1897478770, 2788528264, 2309134747, 4024965405, 3976506219,
                1035727997, 3131955545, 3667438412, 384232858, 2419576220, 793788636, 334784748,
                3765760253, 1098631674, 1499085537, 2112219622, 1421552393, 1432628811, 643517209,
                3632058889, 1632079643, 2223695587, 1476932543, 1960156116, 3436427017, 2201635037,
                3182693886, 1548942116, 3157649082, 980646399, 831953531, 2763454966, 3961226196,
                3582012549, 3882861773, 3003921500, 1404369615, 4000431109, 969558169, 660750652,
                4122382799, 2357249946, 3678033512, 1327191217, 2116553842, 22168676, 1216607516,
                2222670817, 4212062746, 3066154963, 4179712741, 406795945, 2599414815, 3647640166,
                4071465241, 883313541, 1105574450, 866562291, 372730105, 970111908, 749995626,
                1253456197, 2374658843, 3555086224, 3636919504, 488985805, 2204950393, 1262942086,
                3204402881, 4275635994, 2310416697, 4135349793, 1226698657, 1188086923, 3222723934,
                1042266586, 4047620517, 3948968819, 4123810886, 1610456917, 556933362, 4197028761,
                59668868, 2993790244, 854419727, 933448524, 897309660, 3037260848, 1562257062,
                4083457628, 981822256, 295330752, 3711901403, 1052560139, 3617703336, 262191843,
                1142114318, 3848091155, 2086394563, 469188180, 3855914481, 154128290, 2738524940,
                1460255288, 1285831461, 534017408, 3101261674, 118239159, 1944080872, 4203935546,
                2653496603, 3202103001, 4264655348, 3809947358, 331166910, 4277075817, 1031408102,
                866321189, 356095977, 1058770275, 3617580770, 2573338110, 1315597113, 1752964646,
                2179362239, 2449996072, 1745253117, 85067425, 347176893, 240744367, 123940072,
                3568364276, 1405760257, 2210945376, 2261469245, 2772215200, 746282058, 568635214,
                1171903297, 1744293446, 1789334502, 1459265780, 2293069614, 319891368, 1979148180,
                627405437, 1184813190, 3346831763, 1106015952, 940476403, 1182911489, 1469568725,
                2364938114, 1157432499, 2372887700, 1043172544, 2250011461, 2085295087, 604304964,
                340825308, 2422766482, 2410385883, 791761071, 2733359762, 1681282344, 1354240087,
                1139722734, 1143256298, 2677141676, 664248460, 1670722978, 3073187653, 4102786625,
                2618886769, 2531436405, 3579964409, 3950031759, 2248680433, 984112565, 3136159147,
                3101912984, 1099692243, 2145841930, 1116532125, 4172191260, 1659871196, 1924994861,
                3162922129, 2658827997, 1059354695, 486907692, 3395508395, 295826664, 2173667478,
                3774792204, 1393285442, 1481797518, 3124974926, 3444355559, 76091140, 368283108,
                4144769582, 1548649424, 2500545816, 2934156349, 324366676, 3648212298, 4072218004,
                2165091138, 1096979546, 1763071718, 1899235026, 803276905, 1146994263, 2575402634,
                2199211565, 757588391, 2673498553, 81023164, 439773287, 3406088797, 2333068608,
                4043446581, 360179846, 2767375326, 69621430, 1199860390, 2642727874, 2693185292,
                3218919242, 2999720961, 1196125438, 1368472471, 992806312, 3414591657, 1991386017,
                1806201055, 3597622657, 998040031, 901561537, 3420061213, 1598740651, 3883097859,
                2560593523, 2144383095, 2844056757, 689661366, 4046234614, 792024425, 2086938139,
                1867234422, 1449908164, 4144743168, 1158803667, 2979094734, 1636380917, 2507967837,
                1146823800, 4089295245, 3701313701, 3893526214, 1323911033, 3620380011, 3952758387,
                3265334125, 2820495010, 2855022691, 1720959672, 825537074, 1695132633, 4264601047,
                3483737293, 3618271080, 2259345874, 806986786, 3597560543, 3609564742, 3752395175,
                1910432839, 2910423130, 2748643710, 2345098346, 3595323045, 1619954943, 4250836093,
                2316914206, 3801483715, 3657868704,
            ],
            &[
                3231068322, 2952358964, 2649323214, 3829109446, 3277451762, 1929830117, 128637820,
                2648688236, 3506019657,
            ],
            true,
            &[
                1863226650, 1957700928, 2397867988, 3361691958, 4165763036, 2541448103, 3330236317,
                390820454, 3199800095, 1741385769, 489727642, 3156882136, 3364833247, 2137995472,
                49695877, 3008476841, 2862420416, 3492919918, 3413332847, 336822005, 1172092721,
                3834063622, 387233639, 4033191572, 825059607, 1739762995, 2020314499, 3839219267,
                693092469, 33950982, 1177221008, 3244207164, 2680526838, 4242912184, 2481840017,
                3724363464, 1448349630, 2835676977, 2435088559, 2020244573, 2206043073, 1215858002,
                3357751071, 3401291562, 3486448975, 1614914078, 3741841610, 3754003433, 545275396,
                696866963, 2296823190, 2173589284, 3887309169, 2170555018, 2354335949, 1942858008,
                2275759948, 238261999, 3409767723, 4050603972, 3841360830, 479906857, 3816482700,
                3132890483, 1442647186, 2664711624, 3255895001, 2304652118, 490441941, 1384930190,
                3781000441, 1197124175, 3673914796, 3264625683, 2934430644, 4012652010, 645313445,
                1342417027, 4259899126, 127557767, 1863840635, 4240034833, 3380128902, 1721040466,
                258484486, 2943320471, 4080044726, 1184529896, 50392490, 99485071, 2656469171,
                394602583, 3236432821, 829680970, 4180640632, 216345893, 794871810, 776950374,
                188852287, 2768051863, 1478264409, 3116222344, 2035264630, 3540143745, 3312627463,
                1243839690, 4283661964, 2255627632, 2493155473, 3422881207, 672998229, 1330645402,
                805858947, 3157155303, 2449760637, 541100830, 3959355989, 2578797546, 1805505456,
                1105314669, 4044058628, 3236890928, 2881753558, 2935150784, 3619155497, 1247576722,
                1057473446, 2602148123, 2591947096, 3098219463, 203480759, 3190196698, 1560456418,
                1737612953, 1525519966, 1833713546, 2974486003, 3554849856, 1570515568, 3538788251,
                3534319124, 3133957142, 2947537950, 1328568927, 2472554746, 3121690099, 3058658055,
                2512921696, 4268236616, 507933028, 1384977378, 2914276309, 2170658818, 2676773599,
                1344588011, 1406282402, 4240165134, 1774763910, 3592097532, 323634811, 4493032,
                2026844160, 4259889667, 1561456992, 1745963453, 3318810240, 2613669069, 3295340381,
                3322882413, 1172726818, 2542871057, 889925562, 2512140586, 1002695611, 3804029809,
                510918143, 2428502323, 1435737298, 2312478558, 800761403, 822185524, 2215463403,
                2284293602, 1855687935, 3929793519, 3135842877, 397151223, 819010071, 3143846345,
                616813752, 2964423320, 2479441994, 2399498982, 3731231486, 1268178985, 637959829,
                1437260196, 1762640607, 2763892037, 3152346415, 130897174, 150170711, 3043757213,
                3263747790, 2268840104, 3897286925, 539186347, 2052996062, 427292218, 4071153750,
                2387542695, 1403176825, 3054171069, 612471588, 1215312135, 1616958897, 1846418285,
                1376844475, 1228696609, 534770698, 3860539144, 3746082038, 3223899771, 2595640187,
                243793487, 698835349, 2226712063, 4113186925, 2899418198, 1544475879, 1323263527,
                3387040179, 3921953055, 1816962743, 3835265005, 512922592, 4131079192, 3966796549,
                586497526, 3324562547, 4163601002, 2280208374, 1649440850, 2888163653, 1932515002,
                2915989787, 1835707588, 661236253, 1521623480, 1468090177, 62176243, 1761654778,
                654002630, 3078945097, 2118465419, 622606572, 1872263161, 371147127, 2407796135,
                1691024627, 1530874911, 2999266525, 3959644478, 2593752318, 673127524, 2516817005,
                1435234490, 2950535738, 3814456529, 2728341971, 1618690437, 2649703585, 1797683058,
                2532611783, 3170196918, 1687541847, 3276477161, 3728293877, 1601226381, 2623169918,
                4277737387, 972038484, 3652503133, 1692819095, 4166924696, 679045210, 2147824742,
                1754890006, 3603422769, 760272390, 2805004613, 1482946566, 2730421710, 638284193,
                1248071107, 1757322486, 2253794113, 446496343, 3072090920, 2265361578, 1244157949,
                703538048, 36778122, 2938030723, 4284977264, 1670946207, 3674544567, 3705855800,
                1533888849, 2783879966, 1509595576, 2692068678, 4286514245, 931483302, 2846849819,
                3750140783, 1937245748, 1537653123, 482103265, 2110925496, 3832643610, 1721135933,
                1078480349, 2178800517, 768852883, 1630173117, 854612116, 4050668505, 3460165548,
                3293238206, 4141790935, 764767464, 3763613607, 3433771841, 2639622, 654168294,
                186019120,
            ],
        );
        // - highest_q third time
        test(
            &[10; 37],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x3ffffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294966784,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0xfffffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4261412864,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294967168,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                3,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294934528,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                1023,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xfffffff0,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                3621246934,
            ],
            true,
            &[
                2097542169, 2497074010, 3595786574, 3396020653, 699180723, 799063684, 1,
                2097542169, 2497074010, 3595786574, 3396020653, 699180723, 799063684, 1,
                2097542169, 2497074010, 3595786574, 3396020653, 699180723, 799063684, 1,
                2097542169, 2497074010, 3595786574, 3396020653, 699180723, 799063684, 1,
                2097542169, 2497074010, 3595786574, 3396020653, 699180723, 799063684, 10, 10, 10,
            ],
        );
        // - limbs_sub_greater_to_out 2 in limbs_div_barrett
        test(
            &[10; 69],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                127,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294967288,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                8191,
                0,
                4294966272,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                15,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                268435455,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4261412864,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            false,
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
        );
        // - !limbs_div_barrett_approx
        test(
            &[10; 8],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 2147483648],
            true,
            &[u32::MAX, u32::MAX, u32::MAX, u32::MAX, u32::MAX, u32::MAX, u32::MAX, u32::MAX],
        );
    }
    #[cfg(not(feature = "32_bit_limbs"))]
    {
        test(
            &[10; 3],
            &[0, 0, 1],
            &[0, 0x8000000000000001],
            false,
            &[1, 10, 10],
        );
        test(
            &[10; 1201],
            &[
                18295077001300582982,
                11318972173113518719,
                9127519662193085094,
                11186232593046463828,
                9967574749124910412,
                9534668194866541873,
                17538998692236597692,
                8856351994537314708,
                15915919282507021309,
                4395337101052053871,
                4573109461445904651,
                7471333999959771232,
                15168214583465292846,
                10112694665499997517,
                7468171880957946669,
                6921645588354592946,
                13601816639536924709,
                15354213944627759328,
                2741837546898925109,
                2808068836502997863,
                11740258706413798883,
                12018890889885101893,
                14950676973818139848,
                11571325165054093751,
                15673140488076698056,
                111792651078725003,
                8462112369261970769,
                3506632968581684254,
                14492060158927869608,
                9683679100633315103,
                8518299955708236612,
                4652093220482746517,
                7043533562034634565,
                12167203625017475462,
                7117384414390535484,
                3466429710589950813,
                17042769017553443224,
                9042817094280175152,
                16844111334058787904,
                3229027329268312555,
                13434137902318146452,
                10307081978533698950,
                16225670675216025408,
                2520628227212852899,
                18240849581864299900,
                1488578585703911609,
                12729292459179553524,
                10245967895896970359,
                13339552018102535488,
                3997937496978780356,
                7658709074265624500,
                1435708185782215934,
                6431277357886612239,
                3790371881169902395,
                6784416241349450757,
                17724596047816212568,
                11815611058096852874,
                18130171813954955659,
                1024780128434872289,
                17963095905370618004,
                122805984702922462,
                4151944930379965325,
                17442615003616391312,
                4994655809588385430,
                3040428539445301559,
                17832402824141992191,
                10165853977855724270,
                2335376537564192245,
                16821936144912130606,
                1159886974868740222,
                12541069017485894294,
                7837219358605440583,
                16446400756537961003,
                126465459086606815,
                2083531762113796389,
                8624382957529422405,
                11784929789278735279,
                11213379749377196194,
                560319428396345164,
                15149217973310924334,
                15463814793141030032,
                198092507050200273,
                8610639264260168818,
                261903620794761859,
                11988086720959071107,
                17592070826426022758,
                3768394046587852857,
                17166555261248835248,
                14963038520790234947,
                2241578438524899796,
                12189060128627683066,
                2262926845306353696,
                13779828760616581286,
                2541235484167862316,
                12643925053624024515,
                17496648690832865541,
                3985970119983125736,
                3095952870801231592,
                11840830080834210856,
                1592987240002085723,
                10982564110995200636,
                4643989547154306181,
                10428500092848201987,
                6198871498072261125,
                7023308239893005307,
                1842762842799324585,
                14071338712339983613,
                226982182212975753,
                10132117640523152816,
                13755737565472281731,
                11632719781566051232,
                15601465236759733965,
                7426599477609205946,
                4611356292343480284,
                8631478323050143434,
                11591955075248152245,
                14677238670216581539,
                8164238761098476942,
                13987621157141456718,
                3767319467475575547,
                12012052271989395919,
                8337005502962563778,
                1906268936538377685,
                3174524835537599456,
                429665364769606677,
                4394657478224842134,
                13121187501762162553,
                14381551623920212050,
                8629282374304273880,
                13263190906733112908,
                5218608522481599043,
                12107632235830044399,
                17543485324435588445,
                5588590375237119557,
                653390775866146799,
                17771172514225072483,
                7528918736871248090,
                282399778929005580,
                2605980929215962661,
                12146324033914854229,
                11592565716384131105,
                18151666657326444699,
                4388203927026263831,
                7341614458621896439,
                8894653933329611005,
                13805725287465815477,
                1961833582231565395,
                7660480270814575592,
                7471493328104234933,
                15044319201246839910,
                4576661672063304734,
                13536449736957685091,
                11860387989360659066,
                14764867249962166936,
                13360409447124464240,
                14221155626952092001,
                17398349115650451905,
                3184465599333245836,
                15707751954179380044,
                12562479861627119924,
                88057213942976,
                1597598702141447884,
                1261772823635792000,
                15958552215247247021,
                12793055150726238784,
                11200918639832307722,
                5597941261295916139,
                4315031833738259815,
                6902838082588292746,
                12728989104884069511,
                16918323087316778932,
                13511521151542209585,
                553843237172766561,
                5043573603672626571,
                17723320900855236842,
                9608995393314215645,
                3240943499004890750,
                2656790654801382241,
                1676457890700011962,
                1762227695284006379,
                7552785576552937293,
                12811313200785996585,
                9898873674914761859,
                17367249174383145486,
                6558113706220556681,
                13049161836673606453,
                518571717053573108,
                15162122533925516408,
                11258823008709425295,
                2435504881057972054,
                14963790284869753122,
                17872120991620222840,
                829886456078946400,
                3111042046099960191,
                12008856441265505258,
                2885441178087651466,
                13689155834809427488,
                1131420088641124682,
                4614200008200617537,
                6047126292996169487,
                10306766196213210408,
                15363972403137500863,
                4835120368572137311,
                6423191714064598283,
                13652834984059479335,
                16393177507860580647,
                235892475830229813,
                6409760052867575115,
                1550734354829621988,
                13575662851619075640,
                718363601454148431,
                7618105344726885101,
                13870052170163128170,
                17099086237059090191,
                9396086948840999204,
                2193052566012551739,
                10987190533735767295,
                15342771743918665873,
                1551527591660837277,
                2481113648650332471,
                12526575541699553066,
                17267384064568181119,
                17289295549595612856,
                2664460126537239716,
                6877918636888277820,
                16082383248389501517,
                14351409001844089162,
                8263128092015238423,
                17346851708062879652,
                311521651703256488,
                3981059575604853696,
                6313669348188551640,
                18099378411031449694,
                15968996741727654128,
                9159799405918354972,
                9462164989236485935,
                3684393721233135450,
                17937865489600192328,
                1638517822102130933,
                17395422161525252689,
                3473709908567621234,
                9193281964841491478,
                10331967326496158302,
                16798696233976578557,
                7807831077297910192,
                12652762876222409301,
                4869797982244064640,
                17701222287597080443,
                14198439585207683091,
                15865499320981581597,
                10334047464234401983,
                3015211655799314399,
                2424108494719176484,
                5856301803897821467,
                6554732345079263181,
                15122890791522644819,
                10545203438810108056,
                9012473640708669972,
                1610970557833154315,
                16035943676789150718,
                2896323099257548494,
                15155422639005018579,
                2259843538120092351,
                10245098502137432793,
                2520646283097234470,
                18012393436345512168,
                7561203372316264305,
                7653799543413918665,
                2140030530743770178,
                9857665885574564324,
                7489977190124210547,
                9998927813057387755,
                2762046115540595154,
                13701876953526406973,
                9166149367362906555,
                15040003763080700518,
                15420365130894777610,
                2514077990148227892,
                8504504167644069065,
                17726177341910187592,
                2449744483621879771,
                1723730756025983988,
                14406403103456534844,
                1454650873709405887,
                18385141654478933772,
                5577491305370650404,
                4566210906549652321,
                12646617965385778707,
                2552078226205003325,
                12355063666203300427,
                12178902896368926706,
                15006718947903716117,
                16372728326443816683,
                10180279671730724459,
                17731483716821982507,
                17200000119948045535,
                16335525811156407438,
                6875979274642807833,
                15168986439371917908,
                15864186564710033224,
                8479084522424050164,
                15994651879769538679,
                17760497144399786164,
                18426614975808400224,
                14704575934872194992,
                15181888226941380306,
                11721908889511702187,
                102617178928138881,
                7447495721105104237,
                7954810306378822535,
                16998786374321127605,
                8090790070484230870,
                16289196329372215488,
                7526820476636601578,
                12572297496289928774,
                113628864092045691,
                6250550022217214406,
                4360142915129731393,
                6501796933114606792,
                11802403139188476967,
                3519002220471562974,
                12904839411442007974,
                17836996140509983390,
                10274485582719436243,
                1702862600039034598,
                14513867867040307158,
                7487625572871740440,
                547811069452644988,
                14936715197880499253,
                13562114970675542784,
                17390100354796730187,
                2650648508100345126,
                2284358746933974591,
                15679445219134526756,
                10812943946121458631,
                3086162675176072976,
                18390684768949838073,
                6075230822143172142,
                6982548968812756053,
                10592236207809317452,
                4646507818793863055,
                12915347565287279717,
                13659140852008443728,
                3573929783088736212,
                8525440095045701760,
                744607937041376677,
                14783015896444558651,
                15131086952103884713,
                5411002486960761278,
                2088437290856887643,
                17144271151733343032,
                13301554666552663294,
                7896467561063988444,
                18211131252066266807,
                18314706540585938015,
                10366120178981009524,
                17459923192069900601,
                4447456409081943565,
                18283667669702090783,
                9450771365947353345,
                4378436544916016463,
                18212457741993594328,
                15198669923619110391,
                1879444567544117559,
                13563147379710886746,
                7674325133378880705,
                3334816774600195478,
                17381122289894552650,
                8606404348019304750,
                11293315989134833536,
                9316500947082049857,
                1266335121593205314,
                13354728313128860089,
                16404400110998945357,
                12189461827482124995,
                8571799438986557328,
                16924412540155584500,
                4066705394775144307,
                8244708734889461660,
                8298262041269740098,
                8485079550935558613,
                10390420936585250554,
                8032986327114538781,
                16386580215630153737,
                1596782152296894307,
                11436174552386854673,
                12281350017506362626,
                1201023713565731644,
                11354800184560792809,
                6244435089778753535,
                14781899671263994812,
                13453856532164887543,
                5205173379729650294,
                6758121452061198317,
                15199577981158799457,
                5097188591555664044,
                17892257447678225776,
                13090788927117929247,
                14729055239625004209,
                4325309995714780542,
                13696424036200737757,
                1273472536475971315,
                2493602685304131114,
                14343892393575517386,
                17316017497393506890,
                7414572135952091508,
                14030461927641410000,
                5832918032885817891,
                5502576889680878828,
                6923748617342471812,
                871116026416752255,
                10966491112110866938,
                2678875367438552346,
                880767707416234664,
                13418969124074860064,
                14901114133194346006,
                13532936137684711970,
                13210870719065554599,
                5680454822125667478,
                18160829213663155836,
                371611573974742026,
                5989253166049701849,
                14180420379914515920,
                15377609392846358665,
                4353627738587754452,
                3173054206132556029,
                7895498175890703618,
                3080756330636658459,
                3760068566220710837,
                6677805614130848551,
                15143994173468972043,
                2776883010742953966,
                7672039583834722626,
                14012333824083176377,
                7996303528193834674,
                11897965609914054208,
                5088662930442171136,
                9775050840076636067,
                155290985412698044,
                7446050331902513644,
                3182373073764824464,
                6333622418059827492,
                3381171406223616986,
                6279035658057293263,
                12457015290254566192,
                9997746010079880598,
                6365931066892340107,
                985007383843617635,
                17683099066034169719,
                11690122858292360173,
                12820337681274901203,
                10119776165009632402,
                8025360517304244421,
                5799209647953164975,
                11923243886776383739,
                4434382877035266302,
                6892316876790904772,
                13311686201719400688,
                13070055340969362249,
                1389161348064595575,
                17317825074198166229,
                12352407812302110965,
                4357096373749880928,
                15322947805519342504,
                3393459926528754710,
                309259942348373780,
                15221863408129710293,
                2184432377148992552,
                5599867762682684625,
                11692237690560353533,
                15617277716526144374,
                16352998506828635464,
                10683295287531537932,
                10174768335502056341,
                9757436149722611038,
                2152351494881547919,
                3149777722784977852,
                8102543602996006449,
                11216609622075916890,
                3579652612346447740,
                16460137215641926135,
                11507203873319504418,
                16154205860053489440,
                2632057472745483271,
                17753479868310580515,
                14766406055562904631,
                8264678782999986871,
                17190366735089617123,
                14132490063308838582,
                5611885333274300040,
                7809615541777044817,
                500715652367171775,
                9987493264173996320,
                841374560193562207,
                6977580041132397357,
                15755392832835879990,
                2396663158056304844,
                6954661831316013111,
                17813876156953598887,
                8398019579977679371,
                15855357483987489279,
                3812760582326753081,
                17273631953943528844,
                9938856188329661065,
                15819226062126522823,
                15903684874086185036,
                17013855101741324873,
                16945130082273335779,
                12554320244359036189,
                13649309719081627643,
                8623047233379225403,
                4386984048641888911,
                10553868952837364802,
                2287715920316141389,
                5449632814028619034,
                14866255022956570263,
                11741122206403462638,
                16053480635563219127,
                5950910649550286503,
                15674410143609173241,
                18052239445521828504,
                17771738773780736596,
                7982624156303669042,
                3523750006668789187,
                11709430973580071492,
                2896965446277645340,
                8080432872087262526,
                6780281993621524078,
                684058261883760969,
                15989257244423388836,
                11589699367052272611,
                10104208263881413964,
                17189453506869451477,
                13916361665060626203,
                2853202431249762367,
                17209296028258297770,
                8683507465402308661,
                9645802663683152169,
                10358479223375842731,
                7829663615722911275,
                16814895244990924256,
                9450557997282764467,
                975077777946948005,
                9552034374331587062,
                16463119689405074774,
                17510823549036664049,
                2497156003024501849,
                8569401863532688038,
                10049279418614386291,
                3587742668108530158,
                6893073803871002466,
                8435788430991711447,
                6027317537544203617,
                7287819136806388073,
                8022714784833611325,
                7097275444704143170,
                9012025202589703553,
                9157332853571289532,
                1993744974767457280,
                1465502656822405877,
                9131312205000720858,
                10325876690137477079,
                6169134302635077705,
                8374996558291634468,
                560822697555683025,
                1908237605140107852,
                3071760787762132773,
                5175055607701420190,
                16301592270910662010,
                13309210291635015714,
                17042047666169332328,
                9526937776970478108,
                12098704706218455302,
                13970362276996674357,
                6796229436389522906,
                15062885882848544518,
                16920623441880575261,
                14229647915216500624,
                8741798560844117715,
                4980134414764225717,
                11277891936160048605,
                8669128257761758831,
                11639866248680237811,
                14320586036388102785,
                8445229725159674642,
                3733656198588769556,
                2734841737258647101,
                3180524601127048731,
                2450617332784964412,
                11426278631729088077,
                6103545629653703721,
                16596235424879528709,
                3083400506423704996,
                3326973352265933113,
                8829291997519615573,
                10934832120864112438,
                2237208881368262020,
                5487788679486358745,
                6433752820775880003,
                17592485008402930155,
                897116185036299103,
                4177675527876951143,
                8991005262989032861,
                12499756317966457681,
                8950539096741137700,
                2847881219307650887,
                15077888665364050191,
                18223594192550309314,
                13486903824928920160,
                388063271439952358,
                10319980883121211184,
                7635019531196725850,
                2427615724424514210,
                7729266079341799689,
                6404444267730632702,
                5433617192107305037,
                4076855819327886411,
                13570092861836669774,
                17637230595629666135,
                3763591644356069904,
                16581684393851086944,
                12542913026947120792,
                900710323095653438,
                10736895612395941051,
                2621023630214846484,
                7378502763289664634,
                13474732295031584723,
                3104494731822189089,
                8271151412175201324,
                12279765059279079874,
                6577258368503226128,
                6763279491725722121,
                11416038652636887266,
                13192060800396327604,
                16237708051063768614,
                7089146474910269889,
                15563073315367269087,
                3957364145608674590,
                5323058129820270810,
                4109735477541683587,
                12355852476209988798,
                15985687144714610011,
                12621726154507263107,
                14640375261881059191,
                16363974953655857698,
                4850702814848069062,
                15272503445030894973,
                16921342410168262313,
                7428955369032396232,
                14045654759586398366,
                1824179654748832763,
                13065290435644832773,
                6620750261665502627,
                3710427655103863956,
                1481622391321253962,
                15738776155042427147,
                5393243462308065801,
                2548893674570156490,
                15446175308782730745,
                13099052830471814150,
                16025011191599670127,
                2890048025568963952,
                15911511151530752849,
                13361640240769822063,
                12104304782171432594,
                961887753450176109,
                15685615589765989640,
                17189497989928671280,
                2831942924382855480,
                4480390308058866189,
                15123214478066113546,
                5319876220060588766,
                7554565254690703099,
                15333258353184365603,
                11213391410117524311,
                15593488040195587753,
                593648809612765970,
                12555350446961062819,
                6229267335263745809,
                14015166800911232090,
                13026142996056874252,
                18183116121398427318,
                15181376709750626113,
                1401960981421676957,
                4293598990318865637,
                530920581688032599,
                6940659528892404741,
                16442234548874017487,
                11187943673940954369,
                5477084646686373295,
                6381208324537451874,
                8061144699238466390,
                8077766508626903071,
                6753688433578836015,
                15372709739771454415,
                14767218468779016882,
                1215884787035963826,
                14382870766506391926,
                2777669072491465441,
                4979926579396211735,
                1768278590436478191,
                15494549597590347998,
                13355843170158138770,
                9534590495526626915,
                11105540785007109613,
                14720148350689650025,
                6017193011436405301,
                16984479834548076496,
                16271637870678307585,
                12326183850782128718,
                6046566507616322174,
                10855966808284447862,
                14486143464576348956,
                11338497252460063850,
                5183772359646668332,
                18243138296691918952,
                13528809746451616015,
                14990329438970768590,
                5382414247629766136,
                17839632985912233567,
                6536591723065785846,
                15159440725656239912,
                13722780980393210799,
                15379786009638462213,
                13680360627088226980,
                157894015409385528,
                5177681992682638634,
                17291176910176600814,
                13010475027988538353,
                1220715812992048990,
                3158791389064101642,
                17418153569888323944,
                14192277962237157065,
                13201221044060083557,
                17197741701763036713,
                1169866928588403159,
                11475502076729221523,
                17842351657425532858,
                8719597950895866531,
                13655118154496174216,
                9324866581514290928,
                6471362086641457443,
                1120540713350762121,
                17250869044224608119,
                10278304262960546351,
                830716653181086159,
                528668968156165816,
                17461167627153646098,
                2954445364658355279,
                4279374933577208791,
                5125629313701256673,
                13688234303349159556,
                6162791048326647760,
                4085886762736458043,
                12797243781587979166,
                4186645783941093387,
                5286754311331113809,
                12228267120751123127,
                13244227501124911724,
                5790695027871396694,
                1179783746683833033,
                547187269099730162,
                3784522572642818792,
                18032448183921975913,
                12457662158631096805,
                15315319400702151227,
                17578423625750785029,
                16343351208792817827,
                12441408408284563210,
                7954314853731713656,
                2984276352836570732,
                13395574438985641870,
                2307482872375012455,
                12980078936529177252,
                10700284809642803759,
                13865447514364266827,
                18230467246784513159,
                10014640149449599139,
                5878477546561163101,
                4136028674985152176,
                13271285149376627150,
                16149599021017368823,
                12810144100834492972,
                10361575929298677228,
                2536873008009811981,
                10725873242024230485,
                6497018066462413733,
                12672985826259352756,
                12621580784482696405,
                3208673605123299314,
                12964686779683735787,
                11198928388252299700,
                9267110518317968851,
                14949846318732066452,
                9541744713692425692,
                15941076237518505589,
                3430239862814579319,
                10612575443053695979,
                13849997225041110710,
                10812240591348788214,
                6393517327149796800,
                3773291149649696828,
                11772973342821893348,
                15336626839762886687,
                9577966728657848673,
                12885121901356197298,
                17815545299295776530,
                6220765473251078270,
                2771097797711300301,
                16970067356480780400,
                1504708658327185321,
                6907611344237613843,
                4075093039543471236,
                13083035552238343340,
                14250200630765174185,
                9672767310783428470,
                11849400399424996986,
                7035900738271398225,
                16082635402379245761,
                16050998072774064345,
                12233157773887550741,
                9123081242185639205,
                10491848564263417153,
                8146725764960278312,
                15575340461645603858,
                4148874142333973054,
                2993956721277198006,
                7451409165297669404,
                11313922544892113187,
                3194232630201899744,
                636408375757237004,
                13608592959958520180,
                1602275071834536905,
                17095805362986103383,
                273804608848916383,
                7791458085242420416,
                4108672307826744184,
                15366005913747176151,
                1911777325643792089,
                7650198163620616024,
                5464577924945397455,
                11373225807219821179,
                11803062818149301122,
                16590458695267155990,
                5042461121399527303,
                17546193017727636334,
                6044754748618261501,
                18027283631246224402,
                5490643452609947204,
                10676364643909038013,
                11630248788877483715,
                13656452259036871197,
                10354144198163268775,
                15527017933877727076,
                7934356641322781866,
                4029898516517466246,
                18052926818605623623,
                3540286841764757111,
                4634551231653920631,
                14122057670778455103,
                6502701997202257971,
                6891458055552018817,
                15456790929915692095,
                4755257324171297175,
                16716123507389519228,
                3651613652886160497,
                726970425978162241,
                1404697533390673853,
                18337661318309196519,
                11286275631912876508,
                16956255978892411559,
                7232996886210695519,
                7015333597196144681,
                2767625396637754039,
                5012454424073998044,
                4313194875364346824,
                10862594252285252036,
                1780047608637690763,
                6465610838655107636,
                14797199105868765000,
                14892726918517585944,
                12886955176299933454,
                12555896806127357876,
                1579402751470265132,
                10039610188105182645,
                13021770347030669513,
                13884058132948290753,
                6203460449771372271,
                17680930315624648890,
                1135636070221202481,
                5887167508778361325,
                6692888760348109187,
                16471454437283076325,
                6790413570336457138,
                4747413456280934549,
                12256524938413768175,
                15683900003187016699,
                7658943070244016799,
                9373611225821015637,
                3097782436024673554,
                3715948327301456793,
                16620556575387636951,
                1144786186914086248,
                2571220709130887133,
                17221925829597224845,
                14639715549529666265,
                2377686026891085475,
                6175736503910047094,
                16254346102373555209,
                2091619280012452304,
                2977400894759148074,
                16973091263321852037,
                18362768537728015363,
                1991299393262221799,
                2234896163405550283,
                18058593386109307972,
                10550027282767151380,
                16780239965370137213,
                799009131584013840,
                10449833190601914997,
                5403901802819318335,
                3399172224735334606,
                2288410999716774322,
                5500021116082973302,
                4679603412526091068,
                17393554539674064705,
                9796061166459144993,
                9467440242924277380,
                11232005203975958356,
                5489824429056622584,
                16693157940463512280,
                15406446911191459754,
                4194365386979719807,
                5203380675405663527,
                4845608829713153827,
                12287647978288669912,
                15124365454705698450,
                13535980942912919723,
                8131237757227102165,
                3174196792035455626,
                5866637449648808112,
                16572436768235757709,
                4014543717417567164,
                10648229607518736806,
                11124454154979311706,
                1769806716148969845,
                487436280093944655,
                8499791320565867237,
                15961111617996045786,
                5068322564632937925,
                15412395235178437129,
                11967627266106331118,
                8045493101565796203,
                13046639311331048399,
                15060301207111006451,
                14565561077561594206,
                4358177937116176854,
                11754827048597359847,
                13428566207941409974,
                4886280109612433539,
                1622891182969078648,
                5631762445567547687,
                1001237725533401134,
                1757759663273866730,
                16057243644130790443,
                1499729696211089292,
                11828414361895128981,
                14654309635396524180,
                11291463832641202452,
                14519357404633231926,
                10577140847149996567,
                3792719835563443387,
                15142788729413823525,
                9678889469185452465,
                8297094875949379007,
                15382627216272468711,
                15925819828335300195,
                14660243428068176769,
                6234311288201569973,
                3098696262152872404,
                16414024924130435474,
                7632710397177488243,
                3027219954282024629,
                13613605315723845518,
                4728284780117539918,
                1526596142713039913,
                13362185372491310310,
                3646060091216202899,
                7134750524336139051,
                10748226811567148106,
                6061127651723295538,
                1977080743741376721,
                8247273669544854710,
                11303352485877072729,
                3212027501425168508,
                3372242849830936750,
                15122856257096582829,
                3657320729410695570,
                1079822070530945640,
                3637761676173672044,
                12937343848915788544,
                14571852791540951516,
                14342466430585531102,
                11728929959200744262,
                9616215636394561656,
                11219596255947364400,
                8537080980244641448,
                5536712989388435927,
                3631171822217372772,
                17865529156187152852,
                5037311271195656389,
                16487502805572224194,
                2147708277645688540,
                5001570498819337775,
                7470084541514784017,
                17492697990745102157,
                8070931416656058009,
                6060950086595339669,
                15013400715742283111,
                2306746444225502304,
                8409359804202256945,
                164372999763571994,
                17018131853777354259,
                12208035939150177086,
                15378350496540915492,
                14788480835646558098,
                10222082122641529550,
                9826197618153251703,
                11243600916118905460,
                11580887407889731032,
                1393539571585322510,
                9935346143997187837,
                7496342550492044399,
                13651431762540088188,
                18099050770585698965,
                7529498299621559528,
                15208919353243962747,
                3726212045576545652,
                4678887453978995265,
                14257385919526698171,
                12560466975991464998,
                1916602273852958082,
                1580725334554883931,
                4225567785880728522,
                11435782961353548865,
                9711420123575671266,
                12525556123102287402,
                2646005433774867481,
                13608235394975640398,
                13551601168317309842,
                16238098975271138187,
                6304934077364865470,
                9171945380172233284,
                12494391509484808505,
                8265808003325748972,
                4965580429357067406,
                2338192348119283399,
                1178551921476923378,
                17792386313089099560,
                9283481247039557200,
                18034555261167084807,
                940432788242421418,
                157287264000433134,
                8930392502298325360,
                14945125834809509178,
                12014926331118784945,
                16300522776195850517,
                3339639261621971347,
                11241360985958214479,
                16436926344603140983,
                10678188202681436261,
                15790390608679380109,
                3932414003149456454,
                5930062229915467643,
                4450937096561861235,
                6454095283937419925,
                9191309540370341069,
                2369208882454404157,
                12058336702238473519,
                6689133296485718676,
                10352121632476532849,
                3457908119433474049,
                8564209740315103396,
                4484492680807237072,
                6272029608511724469,
                14111360500805747905,
                3426639800371632037,
                9193459114578112726,
                7715081428105427735,
                11741225476850793322,
                5763876427542620399,
                10485104516843146360,
                13517901123362979253,
                10971040515178626597,
                10880433648809156464,
                10463748994258517642,
                6039895771661000246,
                18291078422235189340,
                13108581522989000527,
                15970781344230536695,
                14450809554308819328,
                16634698737197679715,
                10907837400337288081,
                4981117528505172335,
                6575031804504949146,
                7418118826254061112,
                5934085700009751031,
                6506575199869110637,
                6840822251681248761,
                11392304369263274998,
                5878363293290930458,
                3935353596127878330,
                13113298309412464813,
                13481516285263684897,
                9985427214718546766,
                6516008243224711903,
                7181072207395196982,
                3447347579103428171,
                7443665624167886719,
                12843482800554270558,
                1283805457628563062,
                17578575987130403567,
                8648960510881174810,
                16654535114846652064,
                16246000708166950031,
                9421890536303267950,
                2811797585537180207,
                5221682992061150871,
                8701033693885846524,
                510866220487737298,
                11014183458125585795,
                7331136142244744171,
                13347933694384243576,
                2971027367826241163,
                16895278402964612481,
                10343481409259870382,
                7637469477289176903,
                14515774696379459873,
                10726339387643525816,
                788506778946106429,
                2659740016403876425,
                16299202570325799651,
                10641347864823085647,
                18351117229710313282,
                11988134638402872271,
                9778312471146927353,
                7381217128978919856,
                11060274181369112531,
                7495561798963235349,
                9055395298714390718,
                16303796496309697333,
                8572341055391067730,
                10552681108574337704,
                778603576632844717,
                9200061608692138973,
                13379830264581686143,
                3558681711903189493,
                15333709760058518468,
                2592042142960702928,
                407067581456916025,
                5641562894408640944,
                12630685466004633336,
                1951314561801616023,
                5079140706956976391,
                3970999389200166383,
                334899398117822722,
                8719953466943676049,
                14359186121933838308,
                8823579651924661407,
                1813976296212826001,
                7289919441079088958,
                6752583945603678883,
                11125294819942902844,
                9024544902550020287,
                16347203117320623082,
                16956773689825861850,
                16095935035076959663,
                5891276384953676887,
                15268407330104588524,
                8413928464863914833,
                8490251354332486177,
                6927106640650584013,
                12126731995366926678,
                3751336702350408676,
                5401550620951546719,
                6931736049013788628,
                12580678497648710244,
                14038988011959255031,
                6324030753850964685,
                10493602541721624812,
                2109892633617514458,
                12146451675883836768,
                13274122909086796977,
                9754756986068039363,
                14495908277997962154,
                9466734708284504026,
                4844085276275930177,
                9796561850244140319,
                1799849561951724621,
                12358897468285161083,
                12177695598696402808,
                7332427622266662258,
                6479138013230677807,
                2934293921862361841,
                10101747402348634287,
                17392449178621812522,
                9076304922506319918,
                11821100168915316422,
                2496048241232164281,
                3879748422895069655,
                16704268052335595321,
                17390288111943870483,
                7693888478144629575,
                76425240623418453,
                16567047119010034390,
                11413373222405397265,
                16470449866739367455,
                11214274131174610225,
                3101065760151847228,
                6960224778085176498,
                13808140043074427665,
                6407627460027904108,
                15896802775139636956,
                7832538022931244434,
                2921401371982230107,
                5173908476677176511,
                10421805820757738988,
                6810895721424324269,
                9500975479118470651,
                2857351951424190284,
                12823190925726244119,
                1508820218642005789,
                8299638370676388531,
                16716319887270456308,
                49159138387135609,
                13569342532624466720,
                13810905518194621132,
                15814726715984180641,
                9733273584351206251,
                2143427140149969534,
                5005663583690258770,
                9498096902345501346,
                8059179870401817189,
                2748099509860782482,
                13393750936162889414,
                8594871683329714285,
                9350309329045037166,
                7894115690084420630,
                16786514420046861964,
                976749382737801867,
                10467038807753661049,
                9339621677902021989,
                402089271605444461,
                4137928429977667667,
                11168137397964197909,
                17118114157506933041,
                2466794684933887081,
                2381322333528956568,
                16060736976005604675,
                7659572533258347452,
                2456177646632167396,
                5581133606107460922,
                9386681170001289422,
                856367678035725915,
                15662913305262005104,
                10182759715493794272,
                6309878850057832646,
                7310497159043440161,
                8801096542582865405,
                17603162305339672061,
                12464396493690551414,
                905835578866578233,
                6335283537487112756,
                18062909181584849860,
                18164643260343932115,
                9611673889224211203,
                11167605540010764967,
                5235785776664953431,
                15827229396414291668,
                15930079830521734622,
                10729843207836263675,
                13661546405175006477,
                16135164248800319186,
                17460395396731193202,
                14180726883841571695,
                15960200487338673589,
                8706962802097008830,
                2160987916800110300,
                1051526548572672844,
                9819378916385658549,
                5286507605709789239,
                13469546533879771266,
                9307969742012845248,
                15615676059776785223,
                13678628123654222573,
                4894676242232977237,
                12463254721383166913,
                16203870605982620330,
                16023651922098038612,
                7232852316950853131,
                8962741390064610435,
                11073871852691286666,
                1951976192995013845,
                981262607320358571,
                16888050122965370896,
                4344384252438731372,
                7412522378253499740,
                883175500149730679,
                9464224172069098591,
                10275279734394232097,
                15403548203346787052,
                9763124462913973107,
                2389315153739146729,
                13323697104831839684,
                5987579865535399885,
                6129762999102774797,
                4347951805136348101,
                2506188997686408591,
                2557923537528328465,
                1907502054327326877,
                11348447673182547440,
                214394370899217588,
                1795825157652904688,
                1079543763001874644,
                7616884667306755371,
                14035899738904279037,
                13603064082194411970,
                4656663014321257673,
                2516633233356574316,
                13157932564393540150,
                11846837651284282888,
                16495943766392305895,
                11420100349017187526,
                10171218756873499208,
                6391422370224144667,
                3018292857029467592,
                3106481628843315850,
                6825790880621561776,
                12921691493010642236,
                17141277638143731380,
                4485397080913431269,
                13546956797638434094,
                17912330572903817378,
                8490274475095362840,
                13555954280504902543,
                14027241222269053491,
                10885497045528349179,
                12019189578091440959,
                8524809140594718081,
                6729564987712259454,
                10294149345473347404,
                14564176145614125180,
                10959232484424234147,
                13881154219512878890,
                11303202608031861845,
                9393772422743641450,
                6070458610875812307,
                16959999555618324305,
                10001897689126693737,
                11648110147126966600,
                11854195848867487269,
                1069111130614492919,
                14992756999092303050,
                14546869344145489447,
                8058351440331963307,
                4274232524574392202,
                11464289163923391246,
                8692996261929543956,
                4508468249049018651,
                14423278340286421655,
                2564508301584730834,
                3859045330752750621,
                2394646001476710533,
                15538310772728613959,
                7119012861567026172,
                14080676356760895921,
                10481316516476920127,
                8953956652625819960,
                15120007614681010256,
                1707764772247280369,
                9418183698843915873,
                8035252867217811158,
                11608754122523476680,
                9711175522978321729,
                4843904770403152351,
                13087602715200118413,
                6994836269212372197,
                17825214555151605811,
                2967734810424283243,
                2387363035965899020,
                17972801638857345955,
                14678563586097685754,
                18332850447185985724,
                4802373133882822931,
                10249883805776628591,
                9771191734968311898,
                12071190072567592143,
                9199739750044147248,
                6156417751384074752,
                9945419763243948060,
                16852715112241242836,
                15498848488019829937,
                8209154894812848656,
                7605360745335808686,
                5813615285131032978,
                63204830640610456,
                13528270312138672012,
                13283862845967502043,
                8050305155218750215,
                14674596502349757008,
                8358424196888559720,
                1146973740112715424,
                14964963965589838478,
                841757193400499440,
                14177680650235797891,
                7631715193722403025,
                5034141147376399263,
                283111841854752856,
                14927674423006354942,
                16964047188773151732,
                6398259345736930740,
                16580537252867656103,
                12301527733370425637,
                12833132833691646344,
                17033427146698681927,
                16008951791940190108,
                17980249132360677006,
                16067599779005199581,
                2081457686236108956,
                5812601273257050732,
                17866101689516824721,
                2684839860280649088,
                854604419444537598,
                8905025564636441116,
                7217301761354685473,
                13930846346011985462,
                2228185093512258877,
                2196889361780157388,
                7373358985538640633,
                4616035087050111997,
                12541255008421421213,
                6023231021806580561,
                5345892880580177162,
                15954974730435085918,
                1459920705275321361,
                10512201752571642839,
                18058356709377865891,
                4833344764102103195,
                2509346903506136425,
                14446963374952835244,
                17897676329582789951,
                1678081788586355622,
                5572980068705774721,
                2292511766532018507,
                12515574856517533810,
                3533113919323294510,
                15790765656655812990,
                1229927139455551951,
                9079999555500778355,
                10714091306623367583,
                16702898494447597729,
                7175066789028512513,
                16316215516083409859,
                1335614351511752127,
                8120563109037646837,
                15529472665463557381,
                4320685394030925984,
            ],
            &[
                1672628060742854697,
                15282246281150199895,
                1805861883825941289,
                13054368594875028248,
                2171479929375595351,
                9116298011409447511,
                17306888266286748641,
                5746239585631076601,
                9725142985408265462,
                8496940287015097744,
                18226505622247538366,
                6266780941044650332,
                6156048087444894443,
                4538057600114721383,
                97265789122852680,
                6274289580297275262,
                5228875786412768441,
                16242628134361946493,
                11190536716838148054,
                14999253618380186569,
                9431412228781264525,
                3720693557798279014,
                16573669939083720392,
                6740694555820849051,
                6524091096770490079,
                3096562044087457385,
                3445205792069137773,
                5879395926345036068,
                11823228934954636111,
                12371688555199950549,
                17297616112085246932,
                2555509880373782106,
                16223058021752764878,
                14999422229046575663,
                4862802819963101806,
                17693697449754524004,
                7451546626575486923,
                16834189484925343792,
                16509933831486099241,
                12631184257434523805,
                17806422081175828224,
                9024323842858863304,
                6033612708074245541,
                14283589057587996336,
                10842466583038220693,
                17663757149762543208,
                17032131808306205091,
                4374016337917105366,
                6505220788410699133,
                5673566217968186629,
                5932755409025155247,
                4554145012174950408,
                17529903423737764658,
                3815725265250996100,
                768030762409375364,
                8101111540473986421,
                514131312877508602,
                8467511376623331898,
                831718339007207491,
                2504859423595920845,
                11166837695738109823,
                14373206768305383478,
                9082192462384414967,
                14242379354001692723,
                12416775352244640482,
                13103976180057994274,
                6653815487356775730,
                4540712387053805867,
                14570815029195900562,
                13110042734243091237,
                3689426930956949722,
                12593196976319748335,
                5082154213541346557,
                8069143451694814354,
                589978278622072544,
                10926961016815911289,
                13766140955627543742,
                4586220875755960736,
                16112246436712502875,
                9204941109851770944,
                5473914245681242802,
                3218880288053733024,
                9536533824552638136,
                10077047347011578942,
                14182303267298662466,
                6026715993374312806,
                10378524287074681936,
                2296780928121215418,
                15754749815480350917,
                3330943091954471356,
                1126865601328103283,
                8951889275386466113,
                7986378594472968517,
                793002098840792529,
                15105672122567239393,
                14635720719820726943,
                7943655256838356552,
                12835998967580492470,
                756113536611343794,
                13521679712064841769,
                6920694928381100485,
                3963584378724408976,
                18168458596975089512,
                6246280487083928698,
                15693439960182266662,
                2204976266736113349,
                17357508962507938891,
                11151653946273121871,
                1703605549588045403,
                16733783926869844505,
                11316772557004760630,
                18286184542860447926,
                17671746225554970780,
                7808746917349768325,
                11269882956337693194,
                12921611154439887602,
                16678620045935729497,
                16402009986336505442,
                12847458604081538511,
                17755399026300991951,
                12686461068405877203,
                14423738039750236538,
                4909229580701064002,
                1469887314647249417,
                13190369397487996719,
                10548255880673058721,
                10653993017413970889,
                15707594908377821293,
                7185571374210148494,
                16675774579122291136,
                6388065414982007363,
                3817450123651794334,
                8635533889608537216,
                6098638948365389616,
                14307458322530931176,
                13765784114239575219,
                14446745783912954054,
                3962804757561248904,
                1162536535886696571,
                8949170256816840738,
                7643084825050542991,
                16599216336031476997,
                17192262166429815739,
                10667971872031883844,
                3121128640608730652,
                10144016131343139900,
                16895209624213388514,
                1069213517692929905,
                11337495802967652212,
                5324596461543864795,
                8246685063522286472,
                18440915663687455364,
                2785075444444891667,
                14642175687044056076,
                7806683982223990578,
                18188589122349697120,
                17807393819655286371,
                9986742896841182780,
                11733628547230256252,
                9608069068769719469,
                2782483964663700175,
                4481784645254463219,
                16640245285459980455,
                7081071197660003592,
                4682789333184064646,
                8699027495672091249,
                2042623593099711310,
                9915351267439399830,
                4529441913968349798,
                17688669638726001150,
                6969233181109699557,
                15041947273170766194,
                17013037482566716685,
                4028641208180307500,
                9766684953326663168,
                16660710246722520539,
                9300789760989405814,
                4619559511775411121,
                11220958186525568895,
                195220445631146614,
                8797196939793783080,
                18091207242784278021,
                14238252978432062697,
                14862253422759731673,
                12947632302042479882,
                15790153148656404089,
                1463807601685839210,
                6696548515759888327,
                5142603015218451719,
                9851870076797225221,
                339341363302195257,
                7704429898409730537,
                11167176554577895085,
                15752522708113359648,
                15301031230056608579,
                2313150267472026543,
                10140513040040392042,
                3537463966687096726,
                15456908874173546317,
                7973357052931918856,
                7110133225280554584,
                1404650061415360805,
                12060443895006083838,
                11121710524180835999,
                17692677335419631161,
                10345142843647975813,
                11682157366762073463,
                2218847055822344923,
                10934859963273441522,
                12441186660408656103,
                822437026043744899,
                2681004617288037103,
                17757286544288847237,
                8711785011821989019,
                2785025474986163397,
                2215183106863227830,
                8059198739628780880,
                571580723319871982,
                16197606749028159993,
                14831503691581153694,
                11539851268300258899,
                524105877649968839,
                17416266841947817353,
                5495778589153513323,
                968960604106612398,
                16678615353805682898,
                11700287878212831351,
                3622606565047473052,
                2647340224033760611,
                6871533508415095757,
                2659198863719376934,
                6931677157599383205,
                5055354256113794098,
                336562919103163745,
                2361059542672889740,
                825783562725949018,
                11642185020259800092,
                10047872728458598237,
                13504093553221294687,
                8602424918317817259,
                15282984454184248861,
                852452176859461039,
                10050340329117425330,
                2701111594842630681,
                3380516875107345403,
                4136143889811526300,
                6268876047837425795,
                2559786284511206847,
                17755488665395878481,
                8001355533630793116,
                14726003870570175656,
                9103515680317108232,
                8435167688649445195,
                14978151287514534341,
                8379493954467012982,
                6052576543057882900,
                15389754048954564920,
                12385734689115419153,
                17777831557332306413,
                9900279278045930351,
                1206609987039670425,
                13611112101506709706,
                12762535112248529447,
                817294020497817885,
                10754700156031583495,
                15389439159096743524,
                14162879890232853094,
                5117425864944557608,
                8604962998164750142,
                13790816172204909738,
                4104758843285603165,
                17930520683724082981,
                8414454756264175755,
                10449954739296855658,
                15479563286299626029,
                303808843612780921,
                1389720105318403852,
                785817621113818440,
                11448825792535874711,
                9736582039937653322,
                12920595361026137481,
                16699975623529765391,
                13749646219160470884,
                11401244124325218336,
                16282041074080967643,
                6553330957169095078,
                2534460029969182943,
                15346714113583882829,
                10322797870140362419,
                8170525298094874241,
                8993837346481500013,
                1592337973614036797,
                4112707017760094378,
                3653698431433437788,
                1164503220725896682,
                4192689923903726560,
                8728431104977525460,
                6286869906994648159,
                15043778140439653995,
                5615728271150547979,
                1988572003934603512,
                15135378657876422165,
                315512758477949858,
                16216116870274501948,
                12504723790971753070,
                1535348898544488365,
                10095860684622701491,
                9132371123926938392,
                10120783865748871929,
                5051411488306545858,
                15543678014917206393,
                15426909057079235480,
                9504015675463012967,
                3066901079497505204,
                6081911170465869341,
                11137007540202157563,
                7450248916723605240,
                16986119260853423951,
                7578352278837776349,
                6751429673444952435,
                13612365479873100530,
                7696425067114400131,
                201953216397633927,
                10830451601162376941,
                5745264538351074871,
                3879748506573515769,
                13854134445751736268,
                12219208888937467580,
                5358318705731611099,
                10986820887645881138,
                5254734320802095761,
                17981689768158240665,
                4420729817422648815,
                18361425798056776288,
                8540074679155734254,
                1731667252659743593,
                6152889871893755218,
                15217521953228771511,
                3567899098692038741,
                5389619645758200151,
                4691008339827948826,
                10713279365488347571,
                9745954968251366598,
                7302191498672369955,
                12626782502849516545,
                14447293561102751675,
                6734094488616523594,
                5731330022735521614,
                1465319249901885331,
                8689750081955043230,
                12950959544528179017,
                8224104597108932838,
                13301534995263852143,
                5645122690858278324,
                17908106229077733178,
                323203859917688448,
                12815293733143521854,
                8031863434623615415,
                15599896533075140903,
                486382747533668075,
                2557885075871780373,
                5639961122845008175,
                10393998329050866773,
                15322005771165766426,
                15578096382762093684,
                14721580303503888873,
                8331810764551543605,
                14134080240781873816,
                5292301859290420205,
                12934613667998296598,
                12759300614679833510,
                11836905953852898472,
                10400210724390354576,
                7895575850616655288,
                329940446534439796,
                6948421725680731847,
                10383034077753168858,
                14251858051847196147,
                13417478805898580488,
                15666356095993822967,
                2649890612422460352,
                3960290467905564799,
                11604699241357916870,
                2707032916751066910,
                17162029134544454899,
                3462167755608084681,
                7708163267151599835,
                6060118632496715447,
                14659850067472343295,
                916231975382177286,
                7887196957456909593,
                16212896135104729851,
                18388686047231092271,
                15188059656466677817,
                10159359915532738686,
                12209557326012167892,
                17104166537381148968,
                8549912606942034263,
                8608254283565330385,
                12199789530349117525,
                17653984748958090856,
                13236243855027491135,
                7285865740076703400,
                15313657680519880233,
                9809529301268841750,
                8322107960211428865,
                5703942948802254530,
                14812716998355005471,
                15136016082127517829,
                7281927702421102653,
                15373636513805545688,
                10377353955277252153,
                2710606888135973500,
                483290790823167088,
                16941949237955644494,
                13114672411012065693,
                16718085808501163359,
                15292113884889789796,
                7981991935175106680,
                16723904260851495386,
                4634513538107365365,
                18324378641217720117,
                18227662801481578598,
                5444832827920199100,
                12893849341139616412,
                11696684010633795560,
                12371761519158834548,
                5017297642292910863,
                18115726883313877717,
                18182028244204009400,
                623586467022528940,
                12093052177881141597,
                12472931752552395239,
                8322821403753201296,
                9468858393454090922,
            ],
            false,
            &[
                5918120123986948981,
                12214866885969252723,
                11609253883777433902,
                5635355260056282089,
                15175771283313717007,
                15492577498145701934,
                17281089998867972767,
                6586417628634894478,
                12935717257603408407,
                213518157147993568,
                17634550085774545221,
                7802961645411480687,
                15474328354975317828,
                3030764988654892480,
                8848149603966882371,
                676892117606100777,
                4196049354216097590,
                1119069746820091103,
                13781529735879690406,
                16275626026273391927,
                8300205860701261916,
                7767984337518977035,
                5352931226469408832,
                12549815516668122057,
                11497059402098551468,
                12238303176301966941,
                17906048896221344372,
                11541304749033810082,
                1087347036356063215,
                10445124749889523059,
                2108226265832637935,
                18326684922164242899,
                17591839094119199053,
                9649362351718252143,
                5419877371634754910,
                7985315133687996015,
                4380072020580579165,
                16785985274039491915,
                17906401277998543325,
                16110490995638091421,
                3199491816914838798,
                16213219991928431786,
                9326535095032375619,
                15738134700481029737,
                60761202193356037,
                14526584003657596983,
                9717506275850427639,
                8872353450872566613,
                17495997723658103586,
                3896491297225969939,
                15875376698593105408,
                9826018841003362522,
                8152616341657954338,
                14943012924993107558,
                7125340071924826005,
                4761034333897027607,
                7251331321451036206,
                4747806146235227299,
                6726751761629566068,
                18178470774825318512,
                18189114841849876379,
                17717832995746573676,
                2573674198725968785,
                5853505062283210675,
                14828791922576430704,
                17468100460396300527,
                16548721553868826459,
                9927275635269960578,
                3174270457173750382,
                11071224255186651932,
                15278979303406616879,
                9257778059115798365,
                14655545247523251097,
                13790831768475677442,
                6071172380973927546,
                14653522655958546738,
                1283852078171252640,
                17755439659741780817,
                7042497028732297491,
                5214827328478127328,
                8543642755744863764,
                8612011501311255315,
                6185713519471448847,
                10532050115036338934,
                4085362692096841984,
                16722145437751649336,
                9095286107166596594,
                13057785857928532741,
                6724302268334378235,
                16685648673426979898,
                8043212514589919457,
                4407985707410781908,
                4857378688754564057,
                13943718795425872569,
                471792017916165335,
                5215960704545918835,
                8898677740944188434,
                16192547849590163757,
                10927045263666149554,
                16012942497443086833,
                17856824427949500892,
                5311983919389056447,
                9076063431281448739,
                2785828113680086598,
                15110707091558662212,
                6583010959630797697,
                1419528267250785056,
                9595683664056411195,
                3249323284298806238,
                9595933966715778670,
                1500902293448815881,
                5874310265457777380,
                5285481081074725330,
                3692318632564707737,
                434746415289817737,
                3244704509795678589,
                5607528670277715164,
                6168231075304184647,
                17820359109262978732,
                15921865578115170464,
                6560400307775181833,
                18092176022757937173,
                2416917950725208618,
                5068383455535678979,
                10996164899950125555,
                2534117342435778418,
                9652711066484768908,
                13461915503471780054,
                8411049298262654215,
                11627045183374435993,
                597713380029364712,
                9914424860026353436,
                6398417476997644577,
                913522648047558044,
                5233560422688152729,
                12269904588172515957,
                2915109300865307211,
                7061777415895646538,
                14428707789216354297,
                1790994874131700041,
                8134975277497787768,
                12979580903727579988,
                11615590867637744738,
                7000312698420541142,
                9784693912853241529,
                10944442549740283787,
                82531176080969023,
                14959951731803161969,
                6668176410821069021,
                4719500064486095694,
                5759785414559226561,
                4970363591469849595,
                3243428337539320431,
                3156898109989998209,
                4247164185502339328,
                8388168007257316831,
                8981232323059064529,
                16860198432014849988,
                3539236745546337501,
                17719922367578025346,
                17210626294345726472,
                8625129527773205281,
                15568690990583000906,
                16537720024170832632,
                11081163673779633734,
                14597737748843479829,
                12286199535506665388,
                7151442784245909806,
                5044864980114186877,
                4391689371296961006,
                6540168485548659477,
                2857514210130326707,
                6969210128397296294,
                185414187382782164,
                9788252758640228255,
                11134472910826455196,
                9826829185177748458,
                10391037584385454968,
                9612702489136097624,
                15745785329372206817,
                11985385919699457287,
                4647950731727578599,
                13474308436818096670,
                7407581385161850827,
                17049401430832148706,
                11185615615288710943,
                11340197387195283249,
                9085306227945895799,
                13048191724100099668,
                8222075024390640547,
                14981122579864582841,
                10723365510805850325,
                15882294117621079731,
                17224307861706260858,
                404752946419267157,
                17455653606541202378,
                11149932181201234911,
                17174493378281359115,
                5742746808088101191,
                6857975670321360348,
                9488183194120086577,
                850756444942468501,
                18251138242152971676,
                5028044743228145865,
                1709652293836541825,
                9336246950855990417,
                12104001557578243874,
                7518552932181414217,
                10953659187130713754,
                8640278306963064150,
                3527996602952080033,
                9724416208794871557,
                9818471899845617626,
                2259597663894735920,
                17196044757056583541,
                2638888046578257240,
                6998509470199156531,
                9192353130659837733,
                567984143948249890,
                9591497398626468338,
                16399018889425267172,
                11561115484479883223,
                13399243786692529429,
                146294187886053462,
                7961591237166713768,
                8208718803163953456,
                1484938287860582687,
                15046852652421580156,
                7169416232431254009,
                5292287501892714727,
                16630324861460976422,
                12475975871299584278,
                13525399861247607934,
                1565336724846468707,
                7276508501752242104,
                11859012083360705479,
                555007877916338240,
                14880605908455542248,
                2705807160288146830,
                14145160047055578292,
                9866403983350364712,
                1195459380772174685,
                11139370143037652187,
                11437022067401833240,
                13682486855606152975,
                3430673715799609998,
                8489219321033686862,
                14395916737962056100,
                14198819344492488794,
                4725627873940009758,
                4377130471348839898,
                132810124294572633,
                10250911241920381171,
                9212291520499301135,
                14290126791732174693,
                709484470312549752,
                13655949074890812034,
                16390659273052951496,
                1962161044085964851,
                6592951073267309884,
                16635386023505459542,
                7521725279768638651,
                10702292265780209953,
                17508035731310744307,
                4849314369727221433,
                3518368935757575829,
                928772043789904943,
                13066503627704932241,
                9170835921766657120,
                1720438369363860482,
                9653343145255079951,
                10806927358101847561,
                11860406035049881649,
                1974385665395717679,
                15975482787244401193,
                3619432381005082811,
                9877997566644094001,
                17053881395186060130,
                10589869650671333670,
                11664760554418654295,
                7590280596977495719,
                10856366592484551548,
                2631368840539712836,
                2839899387286337113,
                8524146102427533222,
                4990365012618727902,
                7151891648819983069,
                11375481945550229603,
                8858696180128338237,
                16384521973744224932,
                17139452207328725464,
                13252654993803597499,
                13434282302115861786,
                14150706912775302393,
                10948307646609458445,
                17388420540008093188,
                6587864291894884444,
                7978190567474696593,
                7928967279204444076,
                15049296655976922413,
                289738567977993423,
                6897438000403266124,
                5964433949215867587,
                16648009539233406755,
                12752972348654131733,
                2033455762505189299,
                11364966258849937493,
                16482570007448457618,
                11388428709157868651,
                7266585693081119187,
                2599499958067785647,
                1720925230213021349,
                17626323535654061769,
                6663193598469739489,
                12671248167045283050,
                12140673778531699982,
                12103201664484858035,
                8004547753695309336,
                6455154238957765328,
                8593204946748750727,
                10957963623834878235,
                17888461227536598973,
                17735403436793241827,
                4585025702858048319,
                9246263998993338794,
                9799856693447240374,
                13046824397398954407,
                3257704574039398043,
                16475559398714306923,
                16151216692843310157,
                7492987706459570361,
                8794687054736704078,
                14498750737487053007,
                6499114153875831088,
                10344627650758003261,
                3933547531509449109,
                9252017834015356297,
                10431927836410739631,
                5636096958617681173,
                17723724381959311922,
                821040401146944398,
                8509491285814488607,
                7142450219324257681,
                2899499047459777930,
                12095042099304891630,
                9090217037821987593,
                3737560355775916991,
                4433293248592466046,
                2579586186392778687,
                5627778192418388032,
                18000377780513451714,
                2643121727684540272,
                13573819511864552202,
                4784246152290536627,
                9420370852195292962,
                712263826591202499,
                15491886233585377779,
                9323035863279061117,
                11045154527883042924,
                7385229880856543696,
                6977873611490450039,
                13639543508335080037,
                13478048328899736023,
                18039135682054740779,
                11521064013981520637,
                12102327419513222106,
                9272869873740159480,
                12775276724893510766,
                5753913366937783318,
                2967349222422898380,
                450790676996347594,
                14991588921019202960,
                4473124084109094254,
                5276806381726757488,
                572213956369142365,
                978390961725309052,
                11077194266341939697,
                9301606295378325073,
                6640701743743293068,
                12862661643802294986,
                11875074059347396257,
                2662510629463159961,
                7518896084004916711,
                5633197821304310891,
                13694031055503341476,
                2390793653044825173,
                1748645157989562610,
                12795800048419142659,
                13894628843525927844,
                2254756627497371142,
                17588489135795917364,
                5359793907722886073,
                14093783225507224946,
                12953198522788916163,
                13706816417396184408,
                17931918665360327727,
                3692987813274069977,
                6331195597561897082,
                10168077267706641283,
                9760779913234776768,
                2948160909961149051,
                17652740296610148356,
                12899526691779881821,
                4089363416315714602,
                15800160433756171036,
                4150896140209606169,
                4332987844457554832,
                1726137208556197778,
                2230907531048364518,
                8037395526181922358,
                11903227707490817951,
                13247419964627176543,
                9656311595299555081,
                11824744075533270637,
                5618559869312497469,
                5589143266997450557,
                18379067610860048886,
                11371445493433813486,
                12614852294401999528,
                15893311235480705385,
                13552674216984930943,
                6331861843109662222,
                562000957286283961,
                15993421123194766567,
                17787868659905745587,
                884898535478444829,
                2284336629817602343,
                13188445592703944671,
                262346945748782371,
                16087606565809361134,
                4452167838802852067,
                17502476454970888321,
                8180837608229621001,
                6395585225706196871,
                11859771797007351562,
                14699457447367466778,
                17084204551310758507,
                7715446538566999362,
                14709025739137466139,
                14530757131065554612,
                14780426661345509038,
                45918674395986635,
                14144922841238297945,
                3642101397043170323,
                12593107591216122208,
                2107492802032040706,
                3969864990100431674,
                10726651770322104822,
                17773265383948264351,
                16680071697289316333,
                12864952677492720002,
                12349506660379881743,
                12343351742867596309,
                12895930201825086535,
                12401346848400595318,
                8155690734768845822,
                11075739107507568490,
                14443945014748180431,
                4606806037355291818,
                8618009589158556699,
                4962909705265320474,
                7936513201613120511,
                6885820758344807063,
                10401668492364333375,
                16727888475202594818,
                4154014773292175466,
                13161203947882521396,
                13919019307994819468,
                9598301910556281548,
                13615929950270046529,
                9814294142614670228,
                388842039453995229,
                3868906497764221215,
                11133917243382513385,
                14431233794630116013,
                13866552635857099237,
                1446694260333969170,
                12634186166034145163,
                5583369938899602777,
                5137033854197759276,
                14066215613504391619,
                3724105431075335405,
                634452116543180599,
                16701028791920674351,
                14689441581249693370,
                11300117837769921445,
                5709070571093483362,
                17979686883221884345,
                4523042923981658057,
                1775748550754504259,
                16344644718339597549,
                17392052235242169852,
                12785196418644852357,
                18211451905528726650,
                2708064947215446363,
                8380100165657891871,
                7137253101551661869,
                8142616943154617917,
                17803746564213147436,
                153826372692538928,
                5723793786021725861,
                10196131183953359834,
                5523825472161035798,
                18267865405460736489,
                3283468354748984180,
                10604208998484587574,
                7393144798381468009,
                14384823378526431848,
                17875404430476094447,
                12734512376384491761,
                12434256692947114698,
                17275896681161726464,
                3991517411918578920,
                3341256985625706194,
                12875923317501906784,
                11553174191068953395,
                9402911263274348037,
                7889147691441176281,
                2938433861208318833,
                16835317070524419869,
                15311222124964640322,
                13191386070347279526,
                2797120364879623301,
                4825023507822429466,
                850800332384380026,
                12566471159773669108,
                10641916485819690941,
                4533772051636675323,
                13280511685797693290,
                1555062755988987659,
                3239647128855404245,
                6726066500947499377,
                6007423757732386662,
                4149243922709316723,
                5132586553438157005,
                15792321766139519460,
                15816926954768067972,
                7294118112451354585,
                17286958838591009277,
                4907287824228034499,
                4053237279580028055,
                15243191806135344763,
                15160933191318572834,
                9399541646460123387,
                10512247032240862385,
                1549392404477822056,
                1553664084033256717,
                8299938816978050250,
                4506406752176376101,
                392106422304255528,
                14325256251728416026,
                9605320355454775897,
                149289209445211694,
                14196672746036566731,
                16390833843583709204,
                7306529835380864755,
                1609617294884241186,
                4376392608007191042,
                4686172428942509254,
                9226100890114720864,
                4817469637170691996,
                11063190997882783988,
                16818806293969511167,
                16753414946544024233,
                11652469874726823984,
                10410834301724561610,
                4496419155373466708,
                14860760715530123421,
                17116153223846910169,
                7192050227650381565,
                17378669812287708918,
                225986433549199368,
                8470401363462512715,
                10770868581319043866,
                9466901035622012241,
                6699024375729741712,
                409036602332432525,
                5386757448559243618,
                8963450891011896500,
                8673144181466249760,
                873094291187559664,
                12664083699094549676,
                5775559671211518891,
                11686008504688202631,
                15839718905540698056,
                13266083552259617867,
                16629163912909716168,
                12951596193960803465,
                16680382695263717715,
                5312112430010155237,
                183580306753984010,
                4620614350630898330,
                146368985100561751,
                8680242838971693339,
                16334891003809230168,
                8296044009054807136,
                11068535921992539488,
                13459068409989801398,
                16330227018753527806,
                12310150645922723889,
                16470777689868351201,
                13885354582351631115,
                10781416202037656061,
                14844375805930616264,
                8719638653924890264,
                9466966915302767475,
                16902517274689764268,
                7637295587685318358,
                10083082213122260465,
                8181103359836753090,
                16962684553940361626,
                7837156732091767324,
                269176569842339446,
                7278095361680012780,
                3942758955593007702,
                4528968938190179194,
                6704240607075070331,
                3303432731279722312,
                2158648768224700327,
                14688749631211044587,
                11272001457898253785,
                2068777615217221600,
                7354581415030465360,
                4115757448706765204,
                9845224061747124764,
                3689237284770699729,
                18290464418127999238,
                7852077070988627012,
                7863151332492806125,
                251628885737301978,
                18042357620611449193,
                14116593607800996151,
                8096141927089169364,
                12647877985059926597,
                8418401584195560472,
                18060685288941472824,
                12955326448341608928,
                17257701815289013872,
                12412692797833535892,
                7581020805942464303,
                1793728477510701265,
                1440048947029875831,
                4001344949088306175,
                4930448054167731848,
                18351422903276209040,
                14203523087498470015,
                4333543995517392762,
                2404299597925046684,
                8665180504038796043,
                4824446420399647636,
                10289758138518593816,
                3607452937085742574,
                5784430632374580381,
                13224009883109730301,
                10433302833369492081,
                8563330309308426093,
                977134692355658252,
                868639596313947660,
                11114845570806284756,
                10553568804667187456,
                828402278919864619,
                8940626136465663544,
                1412605768723256370,
                9626895483501195170,
                16359088278152878811,
                3692614001269046240,
                14789154216532717572,
                4036942485790633503,
                1049261082210626671,
                16218938044785968460,
                85530679118254297,
                13099694691696048080,
                3076142799126365810,
                16178437464286841227,
                12056596164270992336,
                13502907218061644868,
                3588068196299944250,
                6887727249443337211,
                1859310871727038082,
                13661391178199592998,
                13954271532839623639,
                10390958586703587961,
                10379888160727024239,
                4052888005851455694,
                7904956370427672392,
                8519981488846994043,
                15332926568151370053,
                1191060154732214298,
                16959635090494556484,
                11069336977744897600,
                17227745386405237665,
                8089827037565392855,
                13503095650871818986,
                16377926887996418472,
                18093036631333756750,
                12555984050040346394,
                7227273171888696040,
                13299375693247766378,
                17475709327332183680,
                15615295930257614429,
                410338853684361107,
                8259500083186530998,
                2735443689396556788,
                15142914652661121453,
                2197321929993766794,
                17346858321220349784,
                6402361562826954211,
                17682894801076100405,
                10016553667127190501,
                13036488928964213107,
                12660139296995183874,
                17212334112539182701,
                11367734776084342977,
                8665742805841735408,
                16025336783181331773,
                5117698047398356429,
                12735740150828193823,
                1081804134485276729,
                7842045440217572578,
                6019238819083806569,
                8727766917839869670,
                17543657322565662173,
                14412106950578331252,
                14543869151173063856,
                7943364918570979252,
                14084805295748537131,
                13111098905227599465,
                11202865977925716420,
                5996820135817652995,
                11571287622023775421,
                5373524677929215301,
                17153815447258968833,
                10046961949080008556,
                14846860571176635442,
                10338928091935313530,
                2940317764326823227,
                1776405787535180450,
                625809269984863503,
                17769559239543673204,
                3118489222192938565,
                13250086689910946329,
                15923099038962390019,
                10822567853940674338,
                6978627730782203076,
                2595174935535264194,
                9243536850072179353,
                3406470433637984371,
                3073959449450317144,
                15861711889629424487,
                11928324792455649636,
                13550524272193533110,
                15780700931447669545,
                10131935114090648897,
                5340317636333714216,
                15339343608558820412,
                13396278126027078678,
                12344841862313043381,
                12566090228569145515,
                4467783437523316260,
                17567821697166775078,
                9779110602238335971,
                11793438626400388253,
                2497240398513373223,
                10605325543162108671,
                15082988041680062339,
                5183965453682540694,
                6434074123110416631,
                11620219770967033588,
                6206234705311197318,
                9042208974224880901,
                9727369898804305283,
                4647900181451243158,
                9318060116894502304,
                5257285210765950994,
                13361487866051577136,
                3441585901560937350,
                2240964027883718235,
                10026719207153507368,
                17290246452243511459,
                14480168376982470724,
                7629280450957209642,
                14474347826383962832,
                11317012819726516750,
                2962115003543065613,
                12162772852283897343,
                7157384874230657509,
                6851750854456963926,
                15390047842623433503,
                5489411047155759331,
                6332140231414210853,
                11095975516180329805,
                13859585413297997655,
                1668754947299094584,
                7358928682129827290,
                2766327988455105348,
                16708694722996588875,
                2768953098861172595,
                6914880772284915541,
                16557166180686635628,
                3693170667429108001,
                16072312986624846808,
                9896456787099841815,
                17615200765944957338,
                4199047158638921808,
                8630520885961961356,
                12027685245286153560,
                7891368337618414637,
                6404928949401724730,
                18116758261578934492,
                894858370047540031,
                2921629729181210457,
                7745289541745323639,
                11350346089330756111,
                10152653266983330591,
                14798328937825324211,
                8949208387022094213,
                17787392503905898329,
                12486460963965989428,
                651591352658579316,
                1237562200852981434,
                16003240939391023959,
                6420739448387256570,
                4823415470543283046,
                7174592656453798147,
                11424373324798981360,
                737331696961998974,
                4118748058691239361,
                14616473443770035549,
                17650444454043907647,
                17711200809424400157,
                5091501578910147488,
                15550101363878126041,
                4566810755429043240,
                11107983018581197120,
                13554292788177360136,
                3365373296985431135,
                15610147292789460654,
                3564088093422756616,
                10817889322783172996,
                16159632643279358168,
                8597419859706760838,
                11079829268996560033,
                17269507137857304393,
                9947618800967513663,
                462576820491406287,
                841015303753696179,
                15512002278736704978,
                3374877754464709299,
                1154757937962396227,
                9537134004482849819,
                10128571182745135555,
                3704274043469819984,
                15034886915357192640,
                916221862035256124,
                17425598900953186568,
                3947379328146942835,
                3660680652792309828,
                5007284878115686577,
                7522847462384788182,
                10571799915899197168,
                8013565820738950446,
                3966675855215748889,
                129692155915043627,
                5691183223879520601,
                12139669617779071047,
                11814058524451360978,
                6775079055986650672,
                16825782776226623043,
                5104813292662071339,
                1450815249446379483,
                13853175235386092441,
                6694335922759190067,
                7218434162566164191,
                17430813386582913632,
                8498332674216771915,
                17615391640730478338,
                241596204271102644,
                2570852601221578960,
                4599802214241276374,
                10438837029660804288,
                5157548795688936967,
                3798153122426926247,
                2692303022775628041,
                1447520878810147944,
                5855192377386493733,
                6514581604771681776,
                1473660589499588222,
                18307281621256799329,
                11125509395109998281,
                2435145631387070987,
                1285271838144216244,
                4541083490208481993,
                15565142710078764855,
                13071820190366239674,
                10807442981250235320,
                16852181421674849736,
                794202264952966952,
                14267784245260213795,
                13928008029423527207,
                18135353902736935922,
                2114431837901815099,
                6130665714725128892,
                14739842984474439614,
                7454369312678179513,
                15910289572447146154,
                9146362349279587472,
                4658814649895066035,
                4852338178188865614,
                2192949132388551398,
                716457667898608901,
                2399899887449189428,
                13124730724498682787,
                1984867750527727115,
                3299889278515997731,
                8195306963169862777,
                14475785604327151063,
                12724460465724812829,
                6758639346286631138,
                15913784793216142830,
                13171932495254154817,
                3538643548410335706,
                12168540036106852345,
                16425314134821469021,
                331063747268359873,
                18155850216697850278,
                211510699789398426,
                7935466970847300822,
                12229125293409986487,
                8734112355949291277,
                10071615545797653112,
                1793864989578041612,
                11752738563545219357,
                2554337087062880546,
                6784160661654190401,
                685578570200418652,
                2732494065060693551,
                7703734784123999976,
                11477040429265293999,
                17381175706019390228,
                14837293364696097009,
                4833703890663686653,
                2761757739428920605,
                9412696259424618339,
                18336993413001201202,
                5392220203502617737,
                12132875900848494122,
                5897666419764808427,
                10107214772704578306,
                5834665481992921890,
                2244306519766258831,
                5820675501905835926,
                13895182007484584312,
                9733080033395096503,
                17951749141696177588,
                7852600446374176592,
                642107575067795990,
                16064679065784673939,
                6040352419222121316,
                16970428073606505636,
                433090259886128885,
                1549830981278788516,
                3028881619520159752,
                12587143878061487305,
                7306602487976240524,
                6069082702714594855,
                8317005307998462740,
                11586696275428727811,
                10124520395695439236,
                2195436795699758658,
                14913315324971455554,
                5466516946326216931,
                1717042106502750014,
                17693487700744112539,
                17312324646637390229,
                12185008789555994527,
                9030058640985830041,
                10443142281726880983,
                15264441814628501960,
                4146653965712179952,
                8027211328596986571,
                8811720185546566650,
                5690186672540385493,
                251818496025458043,
                10345253390016854955,
                7346199546033069539,
                13632205746357973720,
                14506822790046125209,
                12522294874687963723,
                12449500355426750184,
                16764643529091852379,
                2638044797938755324,
                18319874071056096586,
                14888037590772700254,
                6416648884059549407,
                3741778762412317117,
                3440707741509585117,
                6047470481787558630,
                2492811946256114988,
                17943897517735339646,
                16834876807347850875,
                11693943181588820350,
                9934491754613191232,
                7118638609538842310,
                6721145656345173892,
                7927663645875889614,
                4114384607638700718,
                11775393055826824993,
                12784890791525439800,
                2959297838067480856,
                9238429361931549659,
                15825672608139947803,
                3094848335709726824,
                8091746658688215399,
                7128309512560006037,
                3784265539610231208,
                17653881331317683086,
                13833532758351825783,
                3974699445166882529,
                18126794112732677537,
                3900550060852543519,
                17913516784410626641,
                14238655509846810037,
                17818640735092102056,
                14112661809980388763,
                16347084184144786237,
                7317578804055819219,
                15985432390504555226,
                16103870553153834573,
                3533821776202300649,
                8924951458298615732,
                11378896973645739060,
                8417337589693233698,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
            ],
        );
    }
}

#[test]
#[should_panic]
fn limbs_div_barrett_fail_1() {
    let ns = &[1, 2, 3];
    let ds = &[0x80000000];
    let mut scratch = vec![0; limbs_div_barrett_scratch_len(ns.len(), ds.len())];
    limbs_div_barrett(&mut [10, 10, 10], ns, ds, &mut scratch);
}

#[test]
#[should_panic]
fn limbs_div_barrett_fail_2() {
    let ns = &[1, 2];
    let ds = &[0, 0x80000000];
    let mut scratch = vec![0; limbs_div_barrett_scratch_len(ns.len(), ds.len())];
    limbs_div_barrett(&mut [10, 10, 10], ns, ds, &mut scratch);
}

#[test]
#[should_panic]
fn limbs_div_barrett_fail_3() {
    let ns = &[1, 2, 3];
    let ds = &[0, 1];
    let mut scratch = vec![0; limbs_div_barrett_scratch_len(ns.len(), ds.len())];
    limbs_div_barrett(&mut [10, 10, 10], ns, ds, &mut scratch);
}

#[test]
#[should_panic]
fn limbs_div_barrett_fail_4() {
    let ns = &[1, 2, 3, 4];
    let ds = &[0, 0x80000000];
    let mut scratch = vec![0; limbs_div_barrett_scratch_len(ns.len(), ds.len())];
    limbs_div_barrett(&mut [10], ns, ds, &mut scratch);
}

#[test]
fn test_limbs_div_schoolbook_approx() {
    let test = |qs_in: &[Limb],
                ns_in: &[Limb],
                ds: &[Limb],
                q_highest,
                qs_out: &[Limb],
                ns_out: &[Limb]| {
        let mut qs = qs_in.to_vec();
        let mut ns = ns_in.to_vec();
        let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
        assert_eq!(
            limbs_div_schoolbook_approx(&mut qs, &mut ns, ds, d_inv),
            q_highest
        );
        assert_eq!(qs, qs_out);
        assert_eq!(ns, ns_out);
        verify_limbs_div_approx(qs_in, ns_in, ds, q_highest, &qs);
    };
    #[cfg(feature = "32_bit_limbs")]
    {
        // - q_len + 1 < d_len
        // - !highest_q
        test(
            &[10],
            &[1, 2, 3],
            &[3, 4, 0x80000000],
            false,
            &[10],
            &[1, 2, 3],
        );
        // - !(!flag || n_1 >= d_1) second time
        test(&[0], &[0; 4], &[0, 0, 0x80000000], false, &[0], &[0; 4]);
        // - q_len + 1 >= d_len
        // - !(!flag || n_1 >= d_1) first time
        // - !carry second time
        test(
            &[0, 0],
            &[0; 5],
            &[0, 0, 0x80000000],
            false,
            &[0, 0],
            &[0; 5],
        );
        // - highest_q
        test(
            &[
                2694350779, 1945458005, 4130424186, 4267238024, 2588565427, 561074857, 3031728663,
                2495857464, 1122008234, 2280094424,
            ],
            &[
                2460989955, 642673607, 1971681331, 2338977519, 475442971, 2516527409, 3470931196,
                4262052990,
            ],
            &[
                1430162564, 1873089879, 3090281851, 2861182896, 2502166555, 2486624904, 960387080,
                3073064701,
            ],
            true,
            &[
                2694350779, 1945458005, 4130424186, 4267238024, 2588565427, 561074857, 3031728663,
                2495857464, 1122008234, 2280094424,
            ],
            &[
                2460989955, 642673607, 1971681331, 2338977519, 475442971, 2516527409, 3470931196,
                1188988289,
            ],
        );
        // - !(n_1 == d_1 && ns[j] == d_0)
        // - !carry first time
        test(
            &[
                2678906106, 1133029551, 3498992572, 3960506675, 2156562886, 2180928089, 4027866586,
                3363079195, 2699625088, 10633112, 1776348429, 2657266039, 366882369, 3755249663,
                1513004482, 385462618, 1235118123,
            ],
            &[
                1454965277, 156018042, 1386304234, 3933017003, 1840527397, 4169420651, 3343338814,
                1487329926, 3897306601, 3401241545, 3117994664, 2264466925, 3843062712, 3688462745,
                2751679318, 2963108470, 1519717633, 1232770303,
            ],
            &[3907037329, 1526550202, 2796269294, 2117183522, 529127946, 3581684574],
            false,
            &[
                804693922, 2514269410, 4036020089, 2186548072, 2508423987, 4057814367, 1906755998,
                3240890878, 892733317, 778428899, 2723150360, 1478273149, 366882369, 3755249663,
                1513004482, 385462618, 1235118123,
            ],
            &[
                1454965277, 156018042, 1386304234, 3933017003, 3311017948, 2590855881, 4216998583,
                2363684544, 3240958306, 2015516746, 4229646813, 3042340787, 3637479002, 2452334854,
                2707761200, 1588607308, 1519717633, 1232770303,
            ],
        );
        // - n_1 == d_1 && ns[j] == d_0
        // - !flag || n_1 >= d_1 first time
        // - n_1 == carry
        test(
            &[10, 10, 10, 10],
            &[0, 0, 0, 1, 4, 0x80000000],
            &[3, 4, 0x80000000],
            false,
            &[0xfffffffc, u32::MAX, u32::MAX, 10],
            &[0, 19, 0, 0x7ffffffe, 0x80000000, 0x80000000],
        );
        // - carry first time
        test(
            &[10, 10, 10, 10, 10],
            &[0, 0, 0, 0, 0, 3],
            &[3, 0, 0x80000000],
            false,
            &[u32::MAX, u32::MAX, 5, 10, 10],
            &[0, 3, 2147483630, u32::MAX, 0, 3],
        );
        // - !flag || n_1 >= d_1 second time
        // - !(flag && n_1 < carry)
        test(
            &[10, 10, 10, 10],
            &[0, 0, 0, 3, u32::MAX, 0x80000000],
            &[3, 4, u32::MAX, 0x80000000],
            false,
            &[u32::MAX, u32::MAX, 10, 10],
            &[0, 0, 3, 0x80000000, 0x80000000, 0x80000000],
        );
        // - carry second time
        test(
            &[10; 4],
            &[0, 0, 0, 0, 0, 0, 1, 0],
            &[0, 1, 0, 0x80000000],
            false,
            &[u32::MAX, u32::MAX, 1, 0],
            &[0, 0, 1, 0x7ffffffe, u32::MAX, 0, 1, 0],
        );
        // - n_1 != carry
        // - !flag || n_1 >= carry first time
        test(
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294967232,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                511,
                0,
                0,
                0,
                0,
                0,
                0,
                0x4000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xfffff000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0xfffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294705152,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x3ffffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
            ],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX - 1,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                7,
                0,
                0,
                0,
                0,
                0,
                0,
                256,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
            ],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            false,
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                7,
                0,
                0,
                0,
                0,
                0,
                0,
                256,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294967232,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                511,
                0,
                0,
                0,
                0,
                0,
                0,
                0x4000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xfffff000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0xfffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294705152,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x3ffffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
            ],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                186,
                75,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                u32::MAX,
                7,
                0,
                0,
                0,
                0,
                0,
                0,
                256,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
            ],
        );
        // - limbs_div_schoolbook_approx, flag && n_1 < carry
        test(
            &[10; 115],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                32767,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                2147450880,
            ],
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294934528,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                2147516415,
            ],
            false,
            &[
                4980658, 4718518, 4456378, 4194238, 3932098, 3669958, 3407818, 3145678, 2883538,
                2621398, 2359258, 2097118, 1834978, 1572838, 1310698, 1048558, 786418, 524278,
                262138, 4294967294, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225,
                4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225,
                4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225,
                4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225,
                4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225,
                4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225,
                4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225,
                4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225,
                4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225,
                4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225,
                4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225,
                4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225,
                4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 4294836225,
                4294836225, 4294836225, 4294836225, 4294836225, 4294836225, 10,
            ],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                40271144,
                2621513,
                65535,
                65535,
                65535,
                65535,
                65535,
                65535,
                65535,
                65535,
                65535,
                65535,
                65535,
                65535,
                65535,
                65535,
                65535,
                65535,
                65535,
                65535,
                131071,
                65535,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                2147450880,
            ],
        );
        // - !flag || n_1 >= carry second time
        test(
            &[10; 24],
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294963200,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                32767,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
            ],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                2147483648,
            ],
            false,
            &[
                4294967294, 2147483648, 1073741823, 3758096384, 268435455, 4160749568, 67108863,
                4261412864, 16777215, 4286578688, 4194303, 4292870144, 1048575, 4294443008, 262143,
                4294836224, 65535, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2863267831, 61449, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 4294967295, 32767, 0, 0, 0, 0, 0, 0, 0,
            ],
        );
    }
    #[cfg(not(feature = "32_bit_limbs"))]
    {
        test(
            &[
                13716243670146164789,
                10080982542333195561,
                6537250227633874413,
                12999253343990893001,
                12265098574469062516,
                10364481473864190262,
                3369535821171388227,
                10993808308731343874,
                2872517547503231658,
                12291082106617508077,
                9131160905399738920,
                6775211851169928012,
                8910327134129798117,
                14142305268873431891,
                7124925848807234985,
                2155988926103471984,
                10449767567966035946,
                9346026815749340730,
                1047855496984265723,
                8616868492917905090,
                6520418679719901901,
                7627787833874912128,
                16237358816885147630,
                18338579662632174061,
                13533819409691495819,
                8877538849543441757,
                1515670908092000385,
                11038667469728451796,
                1811062388681013211,
            ],
            &[
                9995257893114397114,
                9401504468144459131,
                558615837638945228,
                10733662027974786928,
                18295107704289976677,
                1814706268673753787,
                12474943759854623335,
                8814778832826774208,
                9159057654048965906,
                4451260977376821357,
                18241701617364042056,
                6169989192350218482,
                15071965537117101028,
                13509168527678537782,
                12224278653171635329,
                16077066393714953826,
                1433938684868066489,
                13014970036232570373,
                899282336249563956,
                3089487642230339536,
                3787737519477527148,
                16667686214395942740,
                8787122953224574943,
                7841835218775877827,
                9693303502025838409,
                16122224776459879427,
                144327425397945219,
            ],
            &[
                2350654041004706911,
                7834348511584604247,
                12756796070221345724,
                3842923787777653903,
                12373799197090248752,
                9712029403347085570,
                1426676505264168302,
                10586232903332693517,
                8387833601131974459,
                6290888746273553243,
                9503969704425173615,
            ],
            false,
            &[
                89235393247566392,
                5198286616477507104,
                15671556528191444298,
                6642842185819876016,
                1703950202232719208,
                6620591674460885314,
                9897211438557358662,
                12382449603707212210,
                13586842887558233290,
                11884313943008627054,
                3205830138969300059,
                4257812936318957065,
                11084100237971796628,
                13937343901544333624,
                11743372027422931451,
                280132530083052382,
                10449767567966035946,
                9346026815749340730,
                1047855496984265723,
                8616868492917905090,
                6520418679719901901,
                7627787833874912128,
                16237358816885147630,
                18338579662632174061,
                13533819409691495819,
                8877538849543441757,
                1515670908092000385,
                11038667469728451796,
                1811062388681013211,
            ],
            &[
                9995257893114397114,
                9401504468144459131,
                558615837638945228,
                10733662027974786928,
                18295107704289976677,
                1814706268673753787,
                12474943759854623335,
                8814778832826774208,
                9159057654048965906,
                10156380351246372681,
                9009005527785483288,
                4330767427200269309,
                11409205475757922767,
                12430752173702915207,
                11990819624778098799,
                4145020291153594556,
                7099200056207569977,
                3510167930325480168,
                4731667122118327121,
                10720310942075546738,
                5799804483118787221,
                17268037247251138479,
                13305947798457087249,
                1405091439325174594,
                13072181651983436371,
                16122224776459879427,
                144327425397945219,
            ],
        );
    }
}

#[test]
#[should_panic]
fn limbs_div_schoolbook_approx_fail_1() {
    let ds = &[3, 0x80000000];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_div_schoolbook_approx(&mut [10], &mut [1, 2, 3], ds, d_inv);
}

#[test]
#[should_panic]
fn limbs_div_schoolbook_approx_fail_2() {
    let ds = &[3, 4, 5, 0x80000000];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_div_schoolbook_approx(&mut [10], &mut [1, 2, 3], ds, d_inv);
}

#[test]
#[should_panic]
fn limbs_div_schoolbook_approx_fail_3() {
    let ds = &[3, 4, 0x80000000];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_div_schoolbook_approx(&mut [10], &mut [1, 2, 3, 4, 5], ds, d_inv);
}

#[test]
#[should_panic]
fn limbs_div_schoolbook_approx_fail_4() {
    let ds = &[3, 4, 5];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_div_schoolbook_approx(&mut [10], &mut [1, 2, 3], ds, d_inv);
}

#[test]
fn test_limbs_div_divide_and_conquer_approx() {
    let test = |qs_in: &[Limb], ns_in: &[Limb], ds: &[Limb], q_highest, qs_out: &[Limb]| {
        let mut qs = qs_in.to_vec();
        let mut ns = ns_in.to_vec();
        let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
        assert_eq!(
            limbs_div_divide_and_conquer_approx(&mut qs, &mut ns, ds, d_inv),
            q_highest
        );
        assert_eq!(qs, qs_out);
        verify_limbs_div_approx(qs_in, ns_in, ds, q_highest, &qs);
    };
    #[cfg(feature = "32_bit_limbs")]
    {
        // - q_len < d_len
        // - q_len < DC_DIVAPPR_Q_THRESHOLD
        test(
            &[10; 4],
            &[1, 2, 3, 4, 5, 6, 7, 8, 9],
            &[3, 4, 5, 6, 7, 0x80000000],
            false,
            &[4294967057, 15, 18, 10],
        );
        // - q_len >= d_len
        // - q_len_mod_d_len != 1
        // - q_len_mod_d_len == 2
        // - q_len_mod_d_len != d_len
        // - hi < DC_DIV_QR_THRESHOLD in limbs_div_divide_and_conquer_approx_helper
        // - carry != 0 in limbs_div_divide_and_conquer_approx_helper
        // - lo < DC_DIVAPPR_Q_THRESHOLD in limbs_div_divide_and_conquer_approx_helper
        test(
            &[
                1745715367, 3242831492, 2189664406, 3695710048, 3400938664, 2983493477, 3450434346,
                3015863128, 3052023755, 2151651529, 1044116780, 1095733406, 3716653807, 297398212,
            ],
            &[
                3142115855, 4096457822, 1432901702, 1952216623, 2397263357, 1868450777, 1021098076,
                367053436, 1889683424, 1528567953, 3087804116, 689228227, 2409781986, 2752631639,
                2960393344, 2301774469, 4157044152, 730723095, 338324989, 783259147, 1430926216,
                364208092, 2781791540, 805329298, 3841626209, 3970169746, 287294352,
            ],
            &[
                1650097037, 2785533476, 2062924184, 1202017958, 802671467, 2646041574, 1723210259,
                3842076078, 3483645026, 3596926305, 1708140396, 44589295, 3278888837,
            ],
            false,
            &[
                3307700692, 2732374665, 3119369275, 1963618313, 3967275780, 1010403672, 2386237047,
                1636315044, 2452324979, 2121986305, 4198504803, 1202996215, 3246857339, 376322562,
            ],
        );
        // - 2 < q_len_mod_d_len < DC_DIV_QR_THRESHOLD
        test(
            &[
                2955158159, 1759170654, 241992668, 2980193182, 840605220, 1045835548, 3208341497,
                2808694346, 3017377112, 355832894, 950019940, 696314608, 1276063535, 1858415265,
                60277520, 2139543466,
            ],
            &[
                3731075771, 2226257702, 854739192, 2333168376, 3831123162, 672459584, 2105255949,
                3941486408, 3063931623, 3785513419, 2454733900, 2814451407, 149252521, 3850018836,
                2493517320, 445200997, 3777868852, 753763226, 4021231089, 1034604267, 2132847427,
                3137530222,
            ],
            &[
                270591029, 2232658820, 2961791111, 2169611462, 3306811456, 395960943, 1347788049,
                792089960, 2698153968, 3485969429,
            ],
            false,
            &[
                1488408081, 248978519, 2376028816, 407178103, 178475908, 1057954502, 1272791110,
                1537402746, 43900650, 1842311580, 1675506505, 3865664908, 1276063535, 1858415265,
                60277520, 2139543466,
            ],
        );
        // - carry != 0
        test(
            &[
                3168925666, 1401331420, 1768122255, 956650685, 3125577762, 275145267, 4000912215,
                2182857050, 353954436, 2524534987, 1528525099, 1007392140, 2880118216, 2063897965,
                1760572146, 1006590899,
            ],
            &[
                3431345542, 1730498888, 4138696524, 681428010, 2575604395, 915393966, 2674130562,
                2622706300, 1008959988, 202234291, 1715673957, 4288131207, 3157817283, 1254299717,
                1844271939, 2429564694, 1102891971, 4218541309, 2925340256, 2466247404, 3527404620,
                1380334293, 3436079542, 200877067, 3607253971,
            ],
            &[
                878527761, 3297204326, 2726502278, 1631019758, 1065904878, 1800983432, 346615437,
                143744864, 473085159, 3630485760,
            ],
            false,
            &[
                181883802, 1500895548, 839791393, 3438225757, 1702652997, 2997889672, 398115167,
                3761202175, 3544971480, 1688702060, 207943299, 1616875104, 3637603520, 3111391940,
                4267483432, 1006590899,
            ],
        );
        // - highest_q second time
        test(
            &[
                2437003791, 645401979, 1378969963, 3859694962, 1591396376, 1173467855, 1717394161,
                2958783214, 3972116836, 3930714191,
            ],
            &[
                844935929, 1399477695, 2543116912, 837902568, 1404832613, 3601679936, 681919524,
                1510676120, 4054159774, 3802723255, 3285008709, 2781902552, 2079211469, 3135384488,
                3124225549,
            ],
            &[4097164548, 2480021577, 1254500038, 3382139236, 615831817, 2805966013, 2977211920],
            true,
            &[
                3611312107, 839029826, 3955154288, 3446643589, 690412234, 2545148908, 2572472581,
                212083904, 3972116836, 3930714191,
            ],
        );
        // - q_len_mod_d_len == 1
        // - !(n_2 == d_1 && n_1 == d_0)
        test(
            &[
                4235270104, 498278423, 1233560634, 4293074805, 853780813, 1015040797, 3070872633,
                2324941085, 3436613312, 2914604270, 3125537558, 1989275393, 2102765265,
            ],
            &[
                2411131391, 1285978457, 4193249971, 1729912476, 3951338248, 4289840260, 894016416,
                2906952374, 3455712241, 1463845617, 1727541890, 1481030343, 1658748062, 1143697602,
                3292965673, 2291275076, 3088692422, 2745736668, 4049914342, 1746048268, 398577563,
                1528894241,
            ],
            &[
                2173999991, 2272873400, 2390524858, 1701140046, 320303811, 2196647430, 90814602,
                3168314224, 2071538040, 267268559, 2660563785,
            ],
            false,
            &[
                3525126782, 1512399222, 2720117630, 206486370, 3387391711, 3483623607, 1368424417,
                1794379685, 2200464811, 3921564486, 2468104993, 1989275393, 2102765265,
            ],
        );
        // - q_len >= d_len
        test(
            &[
                2046213120, 118064180, 676462636, 192405947, 987595058, 3117257430, 2648701979,
                3334098139, 2300211432, 238540576, 1839171291, 3761260665, 1016976338, 464138847,
                2985769256, 4173218084, 198170535, 2181413355,
            ],
            &[
                3268984709, 3890178895, 3714425009, 922229459, 124979838, 1582051975, 1533234197,
                1439049575, 1664144928, 3333243233, 508818207, 1681704030, 573822729, 1791263995,
                978387652, 2703189437, 2825974585, 854916987, 2566023663, 1086830792, 2353550490,
                2091798094, 2573588469, 1760161870,
            ],
            &[
                4063752152, 180981500, 1829379233, 4014641527, 2812579451, 2652403028, 1486280954,
                2616093248,
            ],
            false,
            &[
                918482979, 147806120, 3245881243, 1658946288, 1766790068, 1379341579, 3971780904,
                1671161699, 2162749536, 1257881574, 3103952931, 1923216828, 1012333274, 3304761773,
                243084170, 2889743198, 198170535, 2181413355,
            ],
        );
        // - highest_q first time
        test(
            &[
                2272242430, 2854406627, 1305900137, 1975858742, 4025222745, 212032770, 698550929,
                632227880, 3284689006, 780673660, 395497962, 765997120, 2856842543, 2158993022,
                20701243, 2016479159, 934398704, 1923839427,
            ],
            &[
                2079043963, 3911491338, 447442849, 3956113252, 1222080016, 639120617, 2598755724,
                1080730341, 102550561, 3677835688, 4245198487, 1565695097, 2098176177, 3974320562,
                1241514653, 856068310, 4158641647, 2053124989, 4149048308, 3508898669, 1748966969,
                1026687354, 3392457811, 3901741824, 3279053517, 763547094, 1818065987, 2992419410,
                2814961974, 3367454332, 1015324186, 3391900712,
            ],
            &[
                2456150337, 1126994408, 3067336229, 1544722288, 1285201687, 384279509, 1392445337,
                548923399, 2471923537, 3222966094, 168537260, 3352071531, 2060967172, 2867808805,
                1744855395, 2564913121,
            ],
            true,
            &[
                902145632, 1230897285, 1461544881, 3921469470, 3447138679, 4291295264, 3805913658,
                2179953362, 3669599647, 3982542142, 3582166894, 852760370, 3665299175, 3877964992,
                3092149152, 1384797257, 934398704, 1923839427,
            ],
        );
        // - q_len_mod_d_len >= DC_DIV_QR_THRESHOLD
        // - hi < DC_DIV_QR_THRESHOLD in limbs_div_divide_and_conquer_approx_helper
        test(
            &[
                2108009976, 2838126990, 827008974, 4157613011, 3782799311, 839921672, 879731301,
                1926496935, 3712149888, 2363043408, 3863476261, 1703718215, 499165670, 2993223230,
                4279694017, 2976095205, 2193242813, 3593708162, 586354096, 296757650, 242624274,
                199363552, 2802933005, 2759687879, 3925865458, 4217980786, 621040829, 4008464542,
                3795907109, 270168250, 3625538473, 759155103, 3959828159, 1835721901, 1818377089,
                114690984, 2967220963, 2377698568, 2241245638, 2252300906, 3116551983, 3644576818,
                2608501946, 2181270779, 1815210824, 1268283136, 484429506, 1180516809, 559610343,
                2543044915, 3719740023, 558939794, 3989231279, 1089033996, 3509826828, 3545449066,
                3353643945, 2822587915, 2300900020, 309191900, 2371088678, 1169317833, 2688462322,
                704882387, 1429447898, 2514029776, 1955158460, 3862686818, 1584130288, 482000750,
                4224830364, 2104097699, 1183456674, 2270729014, 2607655660, 2966545947, 2931252595,
                759652385, 3970839358, 2487392344, 1507124762, 758697388, 3840603988, 1293583672,
                252424514, 646352064, 1756992845, 2309291248, 3392597937, 2946557682, 3659552460,
                1720311830, 3395887210, 2674135679, 3196424053, 2895118562, 1256983146, 2482795730,
                1922474903, 2105232035, 3692384401, 2891711374, 3496171341, 515080177, 4073860792,
                2051799135, 2279442367, 3258846627, 70977295, 1937954703, 2800340027, 1760719187,
                646417623, 3158839049, 3176203007, 213391749, 869819866, 3906922929, 1586646665,
                1614123406, 3687984980, 1177302158, 272342927, 1441818617, 2083855650, 1541781406,
                3311904586, 106907162, 4108429222, 662974012, 4140594255, 1984524179, 3259771450,
                3863443649, 3307256449, 3597197635, 3771753969, 3551628070, 3550518548, 2546464879,
                3812123817, 2299164636, 2258249267, 815945886, 3725185601, 738801212, 3862546442,
                207778311, 1535617953, 2150997157, 729581295, 143725958, 1410055499, 4204574368,
                2185442793, 3653861199, 1767186378, 3537808267, 1766021033, 2239408696, 1986025224,
                3625288948, 4046347844, 2490523057, 3491861717, 1312039469, 597857291, 206880668,
                1812928615, 3684502664, 3820772609, 1327777451, 1237379465, 764273090, 3594853086,
                2859643842, 1243019386, 3444655918, 1596721727, 2221238575, 2483346164, 2064518283,
                2282844430, 1960424205, 4028198846, 2342293800, 1645322170, 3508868511, 2376856302,
                1764497492, 2461899059, 3430260828, 1083983032, 1448541077, 2081767380, 1131293272,
                2121152826, 4212689224, 2039367869, 2193844060, 1864700761, 1182989130, 1481869846,
                969270162, 3997775597, 1483663077, 394034269, 1586435653, 2281434261, 1937512315,
                836935557, 59742568, 1854699653, 2116629465, 328821302, 1999851100, 604863759,
                556389113, 2057620108, 1992554416, 427465103, 813735424, 487055225, 402647361,
                3006998394, 2146302389, 2308517156, 1063328348, 2382344201, 2665908320, 481118926,
                2376219143, 1164055989, 128362737, 2711666601, 1898018298, 2928344090, 3862251511,
                749517272, 2790599579, 3506539103, 2592038317, 1682088920, 1332639030, 3053921374,
                3066891587, 242466377, 3536065834, 2016487096, 1321862427, 2409873623, 3088797737,
                2840818323, 2052021858, 4079299898, 452968540, 3880575723, 2680668784, 1502544423,
                1619882634, 1068420458, 3236073318, 1740351058, 1183565882, 2761799103, 246817911,
                2967032269, 4003972840, 969974269, 3193561842, 1002645557, 2347161653, 4272895273,
                1522072178, 972383725, 3026624523, 434293193, 2855690795, 2455822196, 2413168892,
                2999634230, 784531194, 249556803, 2559695464, 3131798530, 668810109, 2114036394,
                2861740558, 370716909, 2126014070, 3913884200, 1577476936, 855642956, 2074597201,
                3461659621, 3292762038, 2153122643, 1658901071, 1562860969, 501980289, 1611843440,
                2433026108, 3791871978,
            ],
            &[
                2325147333, 1406197332, 1502532645, 137754493, 1488912421, 1014222449, 1678300831,
                2078887154, 2311629707, 3855402290, 1904906031, 1350026944, 314061674, 919972212,
                1429252796, 544807358, 2607111216, 3001219316, 160533550, 2875169678, 1402474615,
                3234840532, 1662112023, 1943050521, 4132256985, 2915369430, 699620367, 59590583,
                2461837586, 3986106860, 2659659366, 3589796255, 4168995860, 2337625655, 4292546333,
                1793085295, 2647770730, 2871508164, 2660729400, 3630964395, 3874212033, 1240042620,
                1713051992, 2105061240, 968807145, 3026819276, 2442427316, 2718248469, 2572140009,
                1793916894, 2112926236, 1058005225, 1528978151, 98430379, 3348888463, 2121431457,
                3545708238, 3975070402, 764453596, 192861224, 2698588322, 1493819664, 2502294971,
                3139295505, 1318649192, 1552623529, 2989328476, 1592829691, 1175278450, 182489613,
                1858350437, 2440903999, 1622786339, 162763913, 4106583944, 609833788, 1344925844,
                3910333895, 1369398307, 51929280, 2214052160, 78601532, 642016948, 3966440550,
                4161875340, 3764174238, 2944905670, 832634369, 2082700552, 3444794582, 962611192,
                38510846, 2733538376, 2141542785, 621670923, 3130866640, 1666281368, 3628581169,
                3965100367, 2889091703, 762371524, 3752935086, 656901001, 2374421435, 3084212494,
                1186140919, 3420046356, 775236355, 1337865306, 2051118555, 713071112, 1946850844,
                1427068060, 3949026319, 3075128554, 438166063, 3975561046, 858907746, 26964870,
                3843362808, 1461820702, 3090970200, 1037540471, 289966660, 2768763862, 50890120,
                580189324, 2911422550, 3684738914, 2025034397, 414622319, 2287243961, 3268939669,
                3547523099, 4011963324, 1768695320, 1859343614, 2123856143, 4141054481, 765801396,
                359993985, 3668605792, 3613648266, 1778728280, 2547397231, 2456061149, 2562918666,
                2903450513, 1994190773, 99234624, 3722083920, 4262323306, 202219441, 4201857695,
                3988878636, 1533308334, 401400520, 1069756554, 2457773969, 2892388936, 3423117995,
                1944069442, 492036629, 3426800580, 2282483359, 4006366620, 1695364515, 2555180845,
                1669287836, 349290429, 778467450, 2020203604, 2218159817, 1450404019, 1278304750,
                2412695340, 1592154884, 3868182043, 2240370481, 3859902860, 1008825116, 412233394,
                2475457637, 3664379433, 4204584226, 2750684469, 4113507475, 2916584959, 285955744,
                739598569, 18278051, 3768126932, 2181905109, 2612988076, 1827656088, 1160380415,
                4160443718, 1846086671, 3050604645, 2547108010, 2828666778, 3252702258, 3885923576,
                2331974758, 730724707, 1528859315, 4288784328, 3677151116, 445199233, 3304488688,
                3566979465, 3541025426, 2491779846, 3112990742, 2583249486, 3403111749, 1930721237,
                3428792463, 2896462048, 2985885576, 1819460734, 21206096, 3560441846, 987100555,
                2844904275, 84854892, 1268249628, 3963306788, 3338670067, 2504599089, 65588657,
                321493327, 4249673617, 4150876068, 721566898, 2186945060, 922948272, 1502464627,
                1426914435, 2906888275, 3454987739, 2609132626, 2073366782, 1058809001, 1226951003,
                2624503637, 282722778, 1407178266, 3304346308, 3613017687, 2397594777, 161128076,
                1938083248, 3042822216, 3958145306, 1431161184, 4147630074, 1744618540, 3729847703,
                2366216428, 2101731625, 2812677939, 3587016062, 1626785569, 3117017254, 3720581461,
                4254564252, 2400911475, 750362271, 3062375802, 2182368701, 268751862, 2682134876,
                274793576, 2417658182, 2321240923, 4200603818, 1349142841, 4066976957, 1043247680,
                679021323, 2933095037, 1621487581, 225868669, 1030678959, 1236344244, 972640788,
                1784301152, 578117349, 3509551550, 214420003, 3786992737, 802372148, 3562194086,
                2187046946, 3600373521, 4275090943, 2120016813, 4177241875, 3185774231, 2397692077,
                1015362399, 2178889151, 3433916223, 1688082118, 1971242178, 236388706, 3802829765,
                521309115, 2299816689, 3207614143, 1053195464, 3584561145, 1178690670, 2940812254,
                3321982035, 2754825123, 3073598062, 202404806, 547895545, 1188944547, 1056841779,
                529463805, 204665384, 850370055, 2063320161, 3724100092, 1180272690, 1398467003,
                2814052449, 1311768018, 659771105, 3226477227, 4230080238, 2134344405, 1461172705,
                2728018383, 1816821358, 3231137250, 2012377728, 2206916761, 3121807673, 3037967028,
                3653505029, 4164239058, 1337501466, 2487452346, 428839025, 3321493092, 2920064796,
                2750126813, 3635762100, 2659829309, 1992243088, 2359420187, 861816585, 4182313312,
                1389528266, 2890138969, 2195001695, 576635805, 1974585967, 851480113, 1896534852,
                3143828462, 4276040597, 3921135859, 2394233682, 2692972118, 2103941456, 3931376544,
                3790764632, 2278428793, 3381340311, 3516241824, 2923157557, 1276268033, 3471729699,
                2370137091, 2637023305, 4027814477, 3711009696, 3667995314, 459819000, 1002878772,
                2658190707, 1189561796, 1831813150, 2874008201,
            ],
            &[
                1703326352, 1625259628, 3642322228, 911402341, 2158835226, 939248485, 3607511108,
                2863853568, 1611642161, 1312857772, 1839433327, 567060478, 3139863681, 3642698184,
                3744632443, 712538472, 2692932947, 576185818, 156934113, 518107105, 2803035863,
                2284220097, 3447382922, 2400125006, 3565062840, 160044186, 3644393084, 4196433258,
                3391883838, 1115703759, 2380388002, 962895870, 4001772616, 2311278419, 2620271020,
                3047789793, 3229254302, 3182628087, 2718480927, 2872538422, 1963990826, 2856537226,
                1729736384, 372544200, 1482170753, 3370665422, 305555060, 4060802437, 1456459795,
                2848113958, 1036273297, 3245835346, 3355862097, 3242734427, 3313897419, 4230036415,
                1425594704, 2428756491, 420147714, 3218149930, 1791217142, 2881741659, 3231612869,
                845372829, 1635665046, 1525482197, 3779462557, 2279005731, 626908159, 2963734610,
                215448882, 2517668355, 3294128769, 1167990371, 2344651228, 2882002195, 3350056381,
                3749638142, 476010196, 2304276366, 1088082897, 1927672185, 3316329053, 1174989749,
                1101596429, 3054511229, 1685228241, 2301284206, 2526016461, 684171580, 3869889559,
                2606492401, 2114658735, 1419214215, 2779052449, 2594333668, 2225446358, 1211258458,
                4029343999, 2816277230, 4098142949, 1987363952, 2732004911, 2516355975, 2365350298,
                1747416524, 3951513077, 3526462790,
            ],
            false,
            &[
                1303670916, 3965736473, 654490008, 3337248200, 112312312, 2506678400, 713192736,
                901474194, 2359721047, 1133557120, 3068598661, 3136858413, 4095957211, 3057328754,
                1900588230, 1315930687, 3273808327, 2821746043, 4255148500, 3072995291, 1472014781,
                922081554, 1759317078, 149072645, 2712091402, 2134310281, 708105323, 3703209145,
                2071594694, 8111834, 3766831226, 2182956891, 2525476614, 1478202223, 4199311113,
                3813148212, 316780911, 1396844257, 2321402831, 1151134318, 848673799, 735789140,
                1092813310, 738412079, 1895091662, 2947844274, 31103395, 2559694049, 1194304023,
                3260126654, 1118043028, 1552558623, 3016599055, 1053142289, 446041643, 2615854004,
                2105823819, 3769707039, 2784177611, 9564629, 580911617, 1979035813, 1479556681,
                2599744262, 3097426153, 4195176293, 3497554127, 3484306250, 2114827319, 2314051712,
                1210812068, 4158781103, 478301760, 2027533091, 1682020897, 129914624, 2066167843,
                3989890251, 1378700448, 170057876, 3992582209, 1541347967, 3421610776, 697014638,
                2834243941, 3504973679, 630519426, 1551901326, 3092678418, 1136389639, 1806538485,
                2208151411, 1598302001, 4041410193, 3574334276, 2340664244, 1624426425, 1200740723,
                2912910258, 671197404, 2272603994, 909857936, 792804858, 3083891114, 3143045405,
                2389563439, 115312858, 1445242760, 2065820105, 256586994, 1432132569, 1549619424,
                2631865338, 1674212272, 2599765330, 3713129178, 1590556325, 3610491293, 709987753,
                3230113618, 1606099993, 3453954266, 3501213811, 3278809492, 3770342657, 1534951916,
                1366321311, 3149479124, 3987513652, 2108936446, 1162972378, 279205823, 3711325879,
                2278557694, 2305274058, 3709507652, 3805940202, 3778288085, 1061153771, 491180471,
                479623642, 2960728114, 553156023, 1863065855, 2253455830, 14222934, 2488687289,
                3378176628, 1136227234, 3805644784, 3175842224, 463501844, 2100687360, 2025076951,
                4064921772, 1000043802, 2482142143, 1463585061, 1044328896, 3966669663, 1023595412,
                2689537057, 685044345, 2599439954, 3302022962, 2669841238, 3358991646, 351579450,
                2303934592, 2098132645, 2239011851, 1623838039, 790570033, 2288992150, 2252675624,
                2479618080, 3624986625, 2129338944, 3796999988, 465518072, 620957443, 1836556967,
                2894771224, 2904933718, 287211723, 1386209013, 2288387719, 162551270, 370319691,
                2924564913, 2947669880, 3390862854, 1797811, 3097227838, 2245950593, 1664498395,
                3380635348, 3536484871, 1217287925, 3149066538, 154911037, 1960867817, 4238493382,
                1991791660, 509572665, 3386968719, 3880405816, 2335816696, 2519967410, 3482574698,
                1501302597, 2415671279, 715902305, 1920401004, 3735445496, 1792155851, 2110622431,
                3695395614, 3311496726, 4035668560, 187609524, 3743601469, 2990357125, 3084998227,
                767424914, 1388627321, 1381810466, 3799640235, 1423360242, 3375037167, 979951340,
                3463933333, 387599460, 229837304, 1394766077, 1068156038, 1135935924, 86516134,
                142150630, 448211640, 3205606056, 1420159909, 4203558153, 2816235778, 4188064475,
                1490981561, 2934083684, 2693435736, 1351552613, 962789901, 1071201732, 3244341475,
                1855681940, 4210620238, 2129345127, 2909949483, 3992748944, 3928807766, 2014533910,
                3099391001, 3112195816, 4009296318, 1247610431, 1474512546, 2530880557, 3597268331,
                764359883, 4198267775, 1228639730, 736768887, 396797847, 799410613, 582496441,
                2658423343, 1643853660, 1258585041, 3767032705, 3709454541, 892807168, 1443895160,
                3500326521, 784531194, 249556803, 2559695464, 3131798530, 668810109, 2114036394,
                2861740558, 370716909, 2126014070, 3913884200, 1577476936, 855642956, 2074597201,
                3461659621, 3292762038, 2153122643, 1658901071, 1562860969, 501980289, 1611843440,
                2433026108, 3791871978,
            ],
        );
        // - q_len >= DC_DIVAPPR_Q_THRESHOLD
        test(
            &[
                1253234102, 1407423154, 1496099965, 1695523232, 4087895045, 3633064637, 1918073940,
                1994089224, 2894393019, 3803250781, 3557289532, 855915119, 825886212, 2854426287,
                3305239731, 1558916547, 14680883, 2283482810, 1975429241, 3696056256, 1225143894,
                401826166, 1805660989, 1715696159, 3407333066, 671337274, 3930782284, 615690969,
                1799309743, 3261619710, 2139755586, 1418298860, 1130149390, 2049944334, 3646689909,
                3742182516, 3450706114, 1620481751, 911179844, 2537827243, 1863349000, 2588289405,
                1379393302, 290595733, 3551967170, 597631845, 3116406696, 418522057, 3052889123,
                3602625940, 282818149, 870470676, 2360142740, 122311196, 1285732945, 3338688058,
                2358904627, 1717417860, 1010819896, 1162692117, 745279285, 2988897615, 3882583107,
                1500499444, 868916372, 1937848809, 1694462512, 3596830383, 3102959500, 2042575803,
                1732232437, 2619341801, 1399365095, 1742239855, 3363053286, 2955047192, 2536114731,
                797861027, 3390102223, 1284755330, 1346331125, 3841540834, 224493549, 3323242612,
                1489837409, 424629037, 2887418470, 2691856116, 51264601, 1998453260, 3606432589,
                270475934, 1351120072, 13779558, 1467138526, 1834087227, 4205762576, 1570218025,
                3941564310, 1908199096, 4049815285, 3522198110, 1100468986, 2461406603, 2479269826,
                536560408, 3804672262, 4126365829, 495636874, 1441896026, 1313080761, 3619143141,
                3430175499, 1671147767, 461789206, 1605289369, 3711914422, 3538203337, 2188319161,
                3278799759, 3020736936, 3471537210, 399986340, 1968814758, 969264551, 3530350534,
                1969502380, 147661179, 3137298512, 1364101497, 1404388912, 1303215423, 2286919520,
                3249572742, 3330831913, 3586493901, 3162077775, 2991387015, 3706115087, 169578752,
                1159180539, 3934133734, 558035602, 2925779959, 4063767692, 2885217884, 2752907569,
                1140725464, 749603258, 672416514, 1032385258, 3864779512, 3712562196, 3208066496,
                2643074326, 3543548657, 3720627664, 632092389, 1636851426, 594140763, 69489824,
                2777250607, 2306083065, 3916911258, 3141155998, 1993475481, 68784498, 1770067486,
                1998044447, 2520569326, 1992263322, 3101331801, 140848189, 3764741027, 1978515439,
                2361982733, 3127443736, 2897381492, 1302677523, 887097058, 1952041780, 783608822,
                1097624715, 99755829, 536819342, 2899797741, 3314381803, 4190434837, 928227057,
                3989607704, 3914272902, 3196195572, 1073079639, 3400734903, 3946657062, 3174732283,
                2855002505, 1996931952, 2245019261, 2712989234, 1251678454, 578021470, 933525407,
                1735185706, 3242693894, 757877465, 1570544476, 4054554610, 4232789, 1769248079,
                3386476948, 1651095046, 1722645365, 3742083849, 249564447, 260470985, 926270254,
                1849290911, 936613977, 2807355918, 799601174, 2973430993, 3630517123, 1010630102,
                1596332202, 3693726207, 2696180764, 256508517, 2935861013, 2617574366, 2142295104,
                1294973507, 4038702029, 3944607195, 3841990618, 98499149, 876432870, 2276059480,
                2682515232, 633274451, 3670159674, 2039570585, 3672264107, 3435333464, 722344600,
                590278704, 3403124564, 2414477570, 3921713319, 2561609021, 966607910, 3391894575,
                1125597344, 1769874536, 1258228103, 1862948975, 3973133769, 2533256421, 88292560,
                2813190491, 3533833187, 413327252, 3841705126, 2608177858, 3676328284, 2614193419,
                2483046806, 692717429, 3846808159, 379277707, 1669205079, 4233020303, 1294972508,
                3928564917, 2407908590, 744791264, 3255079725, 3753322493, 472420344, 1166843119,
                2467000669, 4140405991, 623716212, 1139273119, 3354207202, 2025876056, 1890550355,
                3308875843, 4187315065, 568191796, 1415565133, 2687244643, 2241676607, 2580202219,
                1708640561, 390914458, 3510831086, 1856289438, 1473240478, 2516164037, 56978641,
                4018676274, 948004161, 2713281743, 1119094116, 1836707971, 65506862, 2266057879,
                3850031360, 2402902235, 1355795078, 3048720123, 3936358528, 764696513, 3500940987,
                3117850578, 3798438683, 4211733833, 972363431, 1880167313, 253714622, 76302969,
                2144921929, 594588250, 3419391407, 110436200, 3019984992, 1696068166, 1311459144,
                383767145, 4278876264, 1806767552, 478592754, 3307398109, 2106194092, 1374406895,
                512715289, 2870580756, 1992431788, 63343676, 697633985, 890054481, 1959129172,
                2577805590, 2764027866, 2020028848, 3876787343,
            ],
            &[
                861890943, 516245383, 733102822, 3373234349, 110977034, 4271239715, 1739646310,
                3430038451, 2971068856, 1353792561, 273988221, 3727621506, 1880644317, 1169807252,
                2221765190, 645570626, 1827088960, 1940915885, 4161457279, 3100425720, 996251533,
                95565099, 1073046961, 2253764580, 2391616729, 3098700232, 106646536, 853783579,
                1589281984, 1966496578, 2971444515, 4208062631, 3919529557, 3375641417, 1678790027,
                3548362465, 4197401264, 4152189205, 2578742796, 2352527389, 1289840774, 383594740,
                604487604, 3605617314, 1576995811, 2410149432, 990820085, 2546262411, 2154758716,
                618380381, 2206847289, 3469453827, 1341806652, 1019433523, 152426111, 2749431433,
                3607895359, 3011391435, 1478694163, 851990590, 1027992752, 1059272616, 3480493466,
                1488043252, 1392852266, 3113620249, 4146237371, 2130545233, 3935615262, 1894052325,
                3086808567, 1407836003, 3747705872, 2582628925, 4262889138, 874765444, 2961363842,
                680545553, 2488989749, 3345785826, 952704427, 4098093425, 2516922879, 211154154,
                1759131069, 1700311172, 2393378850, 189990019, 3378862840, 1411934279, 4058568121,
                2905294981, 3913230880, 3855904011, 1536608211, 4114811350, 387292488, 691407511,
                2360232735, 1725740094, 50400678, 3408982759, 2861477438, 2945536307, 3193284782,
                2711749342, 658984109, 343184872, 1067289865, 2232608224, 4027855946, 3835625791,
                3914871156, 3647728511, 2766962294, 1158831068, 1928020015, 1802324187, 1465396891,
                2797410329, 3693678708, 1474899877, 1451659497, 2793780175, 3408416157, 698362941,
                1638711078, 2942583266, 3345553229, 3337125778, 3898736345, 3233933201, 2121187878,
                1375122421, 216315377, 285271552, 153543683, 3787450620, 18426473, 2074179790,
                3707951065, 4276245641, 3503168626, 1741007520, 46035775, 1938996412, 2285217270,
                1291171645, 4224874181, 602464202, 1803057980, 3577626393, 355587957, 3077316668,
                2374221701, 2965264159, 1074621373, 4162088640, 2550826771, 2808953809, 1252092668,
                3357187794, 476206830, 3584460445, 4065428738, 633814711, 2880883917, 3884516376,
                1513164555, 3300578942, 1670837060, 1132213662, 3351202257, 2610012791, 3257259539,
                3160483911, 2266079247, 868822128, 4250370567, 1331943301, 1945818535, 1393345318,
                2060415706, 908741824, 4156794744, 2564813613, 905212060, 3218738129, 3262255881,
                2263542766, 1227381935, 3663658012, 2226973450, 4231561520, 2818168164, 2670361969,
                3318293987, 725644295, 4019758576, 998642393, 495827676, 3734906270, 549931550,
                3628965188, 3223395992, 3590294271, 449855696, 1500765851, 2275616292, 2735068879,
                2797134520, 2639621186, 4044997105, 3174110149, 13220550, 3092556343, 1435605571,
                3386705521, 3831842284, 3500054590, 812912044, 2329469982, 1723309995, 2307452017,
                3303104792, 309903689, 2189249934, 3636229896, 1168371322, 2203364016, 56316585,
                1507047047, 3978350651, 1287006021, 4136822779, 1754090185, 2908593163, 1172296646,
                1218980475, 2947600132, 2403727272, 1587640760, 1122215884, 54808095, 1625907344,
                3722900656, 845603689, 1328648304, 1556638437, 3283968864, 3011502249, 112044910,
                3596114169, 1850965401, 272070320, 1659402803, 3430852231, 4069438789, 2244303862,
                651616701, 2182731301, 636443810, 1069166894, 1406346119, 749190002, 1954541375,
                1807738880, 1450920040, 3158039132, 825986184, 4212536016, 1808707022, 2850318233,
                429694363, 4071098452, 1248651330, 2275666509, 838162196, 1424879694, 776592838,
                2740483790, 1284163517, 1280694131, 1134770642, 2077202374, 1206292163, 1284319127,
                4184450872, 562093874, 952059065, 2799693572, 565666121, 156186725, 2849625182,
                1184911783, 1502520806, 633585326, 3853907607, 2739083441, 1951262688, 103314579,
                3831503747, 2993941865, 2470035934, 1778816965, 3674359762, 2878370279, 1350386433,
                2408142731, 3015205620, 3043051127, 1655111158, 3773182311, 3910770118, 2947352886,
                3356936393, 2844618418, 3298179444, 1279762744, 2973165223, 1721543619, 4040357913,
                2673457267, 2733636953, 3004389294, 2106808579, 466071136, 2720455489, 900027010,
                3351752905, 386568249, 3758644029, 66397961, 2459306698, 2672898173, 1655443171,
                3369301491, 1855362811, 1446321007, 2158536501, 1872565755, 3914900675, 1397039644,
                13190889, 514173561, 931870314, 594435363, 1526203696, 2400586492, 2299132127,
                2876601263, 565427241, 1381196042, 2872681002, 3103160824, 405911440, 954740851,
                675181390, 2686349099, 4136963415, 4220549550, 2370055970, 3163147061, 4154464526,
                3976152950, 3281076036, 2145465814, 1952261737, 850385895, 4035222541, 4292853188,
                187378739, 146395110, 3997223188, 2808707041, 983334910, 2236221581, 1884459573,
                2703940487, 4077202970, 3803155817, 1872492268, 2837348007, 3074452453, 479001365,
                3913048191, 3515475086, 4132420828, 3086255747, 4198950954, 3294929815, 811732388,
                2548414145, 1396214061, 2394099307, 4068762174, 526147562, 1076045316, 1123992450,
                3733570622, 4220579043, 889069266, 2894360531, 1971615205, 2766482958, 4265953201,
                2973723720, 3986781638, 529293577, 1884138947, 510069540, 1743552552, 812713578,
            ],
            &[
                2625866541, 2810579078, 3501644219, 2416714925, 1291740196, 3660307741, 2609642196,
                2505392144, 1722898427, 1674316556, 1432469984, 4097335408, 3108223053, 1374639232,
                1851930259, 893716755, 2663933335, 1737930775, 3657985359, 3660388910, 2695083669,
                3386383098, 267519188, 3888348035, 1835595210, 2613536126, 4114884836, 2142604605,
                8565287, 1755951705, 4130425802, 2850158056, 2234922350, 1949213237, 2449412937,
                1402509673, 3986161632, 1988171673, 4272507706, 3260139743, 1807943254, 2734619085,
                740216746, 446923648, 1965334425, 3397454741, 809048263, 494222614, 1484571594,
                3636917813, 1079987980, 3243842717, 3971186898, 338155234, 863804008, 1792493616,
                2522062802, 3816014339, 2741664304, 222474042, 154928734, 2936845361, 1353872451,
                3808072940, 334383012, 1793621170, 2698841688, 3751625796, 3312001535, 1308411945,
                2231283256, 3335332368, 2732745501, 2266806470, 383612527, 3711228074, 2478058102,
                2078994051, 975470800, 4179961686, 2605205960, 4060526006, 2648539390, 992385227,
                1710490965, 385423846, 824778193, 2349766631, 1760192209, 489319807, 2127640163,
                339639147, 2672582271, 1527261122, 3929636146, 3273385907, 51468511, 4261991464,
                1868351312, 413372128, 1083615978, 2931924886, 3886667243, 2763475974, 2738582447,
                2397672301, 2259776571, 2115277994, 2148277873, 2696639925, 1519841204, 811678952,
                628203566, 2552032086, 4011447443, 3160851596, 258941070, 1535778333, 4063703767,
                1926535070, 303663750, 2997548738, 1835822843, 2093980860, 1428491141, 3798948389,
                462487877, 2937763495, 3558060512, 87869235, 3064605488, 295532817, 92123524,
                2923122754, 569558082, 310878410, 2667249293, 1502523772, 700589809, 402354610,
                1860279993, 748794047, 3648640215, 3110702404, 2258574876, 2773396367, 3517237080,
                2990861341, 331348076, 1212514551, 962158293, 1191845582, 2112876079, 2501772177,
                2831655935, 3995296179, 3011183472, 4288287857, 3132719253, 1897663845, 2153050331,
                3461291371, 57799706, 461541578, 1483239302, 28451101, 895953744, 3368918207,
                2594694002, 1268319513, 973441686, 3311571754, 2019274316, 4123248452, 1446014889,
                1417571585, 615477194, 2194462687, 1708163105, 3293326615, 3383426484, 606654074,
                1070117839, 630146344, 2571257218, 3961968468, 1929710282, 1868034367, 403216466,
                3088416046, 449849638, 443975756, 1551823500, 3682150448, 3283842757, 4232686157,
                4164057310, 2186087032, 2166995410, 3181021883, 3318208525, 2768887999, 2877385605,
                469197960, 363931824, 752117713, 3980786726, 4244751154, 3858512024, 4033177219,
                3536702242, 3001143104, 1049662321, 4154276130, 2710845384, 1882050975, 2307442635,
                2303006043, 1063921606, 92351540, 1383930617, 3560089905, 2221914576, 3599852592,
                2796496568, 2234040425, 3492209128, 469541171, 919249352, 4275860868, 3002397499,
            ],
            false,
            &[
                3585575447, 3122160444, 2645674960, 4091549360, 626365202, 1144617016, 3462810264,
                127827394, 3154238169, 3508098827, 4065321164, 2627720920, 3399886577, 3188051673,
                1639089029, 2016161381, 1581970985, 237731508, 3479558677, 1944880307, 2562503556,
                1795700699, 2609445006, 1116190050, 1146777337, 1200401433, 1157695328, 350844679,
                3282708071, 2845229536, 459139289, 3375883464, 2424175740, 3193435758, 432712800,
                3851580195, 993388507, 3458606683, 3233513818, 1868436357, 1353535090, 1991321983,
                1598253594, 3580022384, 1800608735, 3030838485, 1479344427, 2146117321, 3615628302,
                1468724143, 2513216309, 2478400520, 1881309170, 3573465816, 191069649, 2319303987,
                36912024, 1056835502, 4085803346, 2567638548, 3614176540, 1774136604, 2502728501,
                2178987690, 2079375302, 2975418245, 761382377, 3364876440, 3617579499, 1451615440,
                3143305641, 2435105671, 2844406164, 1247246351, 894541250, 3633445851, 214286649,
                1891108578, 2579284088, 307573789, 3098902942, 2276329990, 3682940322, 2840709312,
                3778212298, 2325451725, 3477808517, 31838273, 2670981675, 2270297375, 465284332,
                2697844185, 2622424439, 4051875556, 814914907, 2448623943, 1299744713, 1873124502,
                2036773577, 3689282510, 975141741, 1207881273, 2818435693, 3863231130, 3098155233,
                3073113139, 1724416932, 774815313, 3984444108, 1380084993, 825030359, 1705088193,
                124122371, 3659155851, 2518826336, 528672560, 2665679696, 1402818456, 782001915,
                871778931, 2353626471, 685023982, 1884443856, 4156627926, 64651699, 1586126662,
                1522765556, 3831765770, 3303426700, 2833111857, 928056991, 1307034056, 3386313127,
                3415959770, 3498516705, 1932766589, 2313751714, 366206234, 3192561873, 2943917337,
                1140687467, 1640175340, 3542041198, 2895665827, 154126416, 650854034, 2789160074,
                352460908, 1351655152, 1980048568, 2172281721, 3543770557, 2365332223, 35300985,
                1050292927, 1349692310, 270125821, 2579494270, 3106522063, 4229421146, 471231049,
                13349434, 2438602373, 1915124258, 2193214542, 3087834620, 276889911, 2493806637,
                2402232776, 556133444, 1874277037, 1783512114, 1442646686, 949037422, 4233276648,
                1359566262, 564784663, 939635477, 1679428216, 2650897503, 1481863691, 1162596971,
                1097624715, 99755829, 536819342, 2899797741, 3314381803, 4190434837, 928227057,
                3989607704, 3914272902, 3196195572, 1073079639, 3400734903, 3946657062, 3174732283,
                2855002505, 1996931952, 2245019261, 2712989234, 1251678454, 578021470, 933525407,
                1735185706, 3242693894, 757877465, 1570544476, 4054554610, 4232789, 1769248079,
                3386476948, 1651095046, 1722645365, 3742083849, 249564447, 260470985, 926270254,
                1849290911, 936613977, 2807355918, 799601174, 2973430993, 3630517123, 1010630102,
                1596332202, 3693726207, 2696180764, 256508517, 2935861013, 2617574366, 2142295104,
                1294973507, 4038702029, 3944607195, 3841990618, 98499149, 876432870, 2276059480,
                2682515232, 633274451, 3670159674, 2039570585, 3672264107, 3435333464, 722344600,
                590278704, 3403124564, 2414477570, 3921713319, 2561609021, 966607910, 3391894575,
                1125597344, 1769874536, 1258228103, 1862948975, 3973133769, 2533256421, 88292560,
                2813190491, 3533833187, 413327252, 3841705126, 2608177858, 3676328284, 2614193419,
                2483046806, 692717429, 3846808159, 379277707, 1669205079, 4233020303, 1294972508,
                3928564917, 2407908590, 744791264, 3255079725, 3753322493, 472420344, 1166843119,
                2467000669, 4140405991, 623716212, 1139273119, 3354207202, 2025876056, 1890550355,
                3308875843, 4187315065, 568191796, 1415565133, 2687244643, 2241676607, 2580202219,
                1708640561, 390914458, 3510831086, 1856289438, 1473240478, 2516164037, 56978641,
                4018676274, 948004161, 2713281743, 1119094116, 1836707971, 65506862, 2266057879,
                3850031360, 2402902235, 1355795078, 3048720123, 3936358528, 764696513, 3500940987,
                3117850578, 3798438683, 4211733833, 972363431, 1880167313, 253714622, 76302969,
                2144921929, 594588250, 3419391407, 110436200, 3019984992, 1696068166, 1311459144,
                383767145, 4278876264, 1806767552, 478592754, 3307398109, 2106194092, 1374406895,
                512715289, 2870580756, 1992431788, 63343676, 697633985, 890054481, 1959129172,
                2577805590, 2764027866, 2020028848, 3876787343,
            ],
        );
        // - highest_q in limbs_div_divide_and_conquer_approx_helper
        test(
            &[
                2317066713, 3734873775, 217584598, 4283181214, 1987296898, 2029964730, 2841877055,
                215692782, 869134407, 874390362, 2208433375, 645289786, 1962123143, 1210019992,
                392208793, 479186599, 1403346326, 3121890550, 21351539, 1792046631, 1830759088,
                3009702825, 2064264407, 1425980591, 2915380046, 2661484230, 272407717, 4286945194,
                649139597, 3925921591, 1081687841, 2923866665, 563140201, 1426204510, 3994415326,
                2030662828, 2060697960, 3056670310, 1609787794, 792934567, 2717771124, 4164322768,
                3913458892, 2759459876, 4091351278, 1717819380, 1097243558, 2147464721, 1228826351,
                3209546368, 1237764046, 1059356034, 2760411795, 3026477323, 624675850, 3763684075,
                3540407970, 1913722572, 262369959, 2180102174, 3425380260, 1174673633, 158169541,
                2382293449, 812030319, 3239921986, 484642353, 2919528753, 1658419979, 366878628,
                1718980142, 799549350, 4285110305, 2715036325, 2455015536, 3032041112, 2735249018,
                3310370605, 1181357276, 146624587, 3299107633, 1004501470, 3484807107, 1748985025,
                1773700707, 1386223716, 4087630817, 750649422, 3611015842, 3444400533, 3725662389,
                1109727693, 3037130634, 624478673, 633583116, 2378434635, 2687240280, 2673870544,
                324343858, 731297615, 3614185356, 495482604, 715462793, 3367457094, 2135162069,
                2612684285, 1189448323, 3887861594, 1345391741, 3430076735, 1913449259, 4268745212,
                3848733916, 1785626124, 3506024858, 1223431612, 3120693480, 3802248189, 2144116453,
                557985986, 2604607267, 1018386624, 2418517078, 2442250784, 3582497470, 654976641,
                2599906378, 2211771862, 4102664307, 836270496, 4128006488, 2260892, 277275779,
                535619372, 1868873131, 749436370, 1849951520, 783534973, 2957791640, 1636965908,
                1156916684, 3732442514, 2251346101, 2723065991, 3717891339, 2525911896, 3076617154,
                3265371035, 3292658676, 409114969, 4086635237, 3264336755, 2836953044, 3037161740,
                3543275191, 585786355, 3087011769, 2185871252, 2541194926, 2224439978, 3412364477,
                1350015989, 1311332023, 758561683, 4108017253, 4204509134, 4157141131, 1893317202,
                1224082165, 880878665, 1834575916, 2572831131, 2631286660, 2578509166, 2240326536,
                2609750355, 2798227847, 1197386254, 807604908, 1802810117, 2881797181, 3334708065,
                3583310868, 3599155312, 183678286, 3236382667, 4125313765, 239427309, 3653131096,
                494732271, 3893107141, 1018082495, 1979944519, 4044560358, 4179896605, 3780149973,
                3875190427, 3114449657, 3457335347, 2339734261, 2616033966, 1329204235, 1248786560,
                3467382161, 3028029914, 2536952877, 194748555, 241512194, 3260227272, 2858189566,
                3881365161, 36968705, 581216076, 4188186728, 2833747030, 1972266392, 1374287701,
                3058853674, 4101868185, 3030432843, 3731159238, 3617181750, 1742674464, 3668849596,
                412040791, 406334378, 502702971, 2540238946, 2391871283, 1851296773, 2979189987,
                199738981, 3379464313, 2875483565, 2404227424, 2501388469, 2789274764, 2515008836,
                3408871507, 4145552682, 2991097752, 2338096504, 2325133139, 2082275255, 2834180247,
                537432099, 1257296228, 3446712843, 39458827, 2174956861, 3039757037, 4157335831,
                733488951, 2064219555, 1158696763, 1747438019, 1904327490, 2069625042, 3220261644,
                2535390409, 1866459831, 3174902084, 3763763466, 711704437, 3941981941, 3947689731,
                545125868, 199321616, 195433751, 587082201, 1623079028, 3583449493, 4189372053,
                656157970, 4065268173, 2321947633, 4117128060, 1042761195, 543619922, 4200179141,
                2194761475, 878960867, 3598234187, 3791350851, 346665711, 1360737771, 1930187385,
                2405900037, 3062309031, 2085261831, 707121842, 3173108868, 503413591, 73514434,
            ],
            &[
                1172385474, 2722128840, 3229943889, 318467582, 1270391003, 1178663423, 1974716983,
                3282704963, 3178774681, 3202829277, 3946239996, 2183661896, 756303666, 3877972989,
                4251016295, 3386638802, 4082864246, 2928113767, 1526526769, 2729531320, 933752600,
                3559448071, 3782798998, 249483043, 1511243361, 3564728022, 1594495334, 71444771,
                3603995676, 816918005, 1243632828, 1550377419, 1008068857, 2948396814, 2093904319,
                2147043689, 2566638926, 1425327657, 1162199945, 2960758893, 4198224, 2356436749,
                3424929725, 1080768275, 209858098, 113302786, 3443677754, 3549677964, 3154318213,
                3154103315, 1582401507, 3617714568, 2767982074, 3761726254, 683939930, 192236320,
                1080891479, 1143174553, 2075266572, 1844075493, 3593156363, 1292478494, 1945353284,
                1251725895, 142445996, 3777493501, 3614227130, 2589877580, 1660324871, 652913311,
                2513923073, 104713426, 2880864765, 717044714, 2176657233, 3162919827, 1461924303,
                1335307283, 1089899112, 1463447432, 4286236462, 538631465, 1843988250, 978544087,
                215501900, 93216142, 2947050139, 2693646353, 4213675869, 3765139921, 320058937,
                1970573977, 43732280, 1873575176, 1055975473, 3361081513, 4080049587, 2593310713,
                2546627943, 348433615, 2711034081, 1213059756, 3994279311, 2969515716, 3962824055,
                498958294, 3283335149, 2727270461, 967722336, 1011867961, 3277768890, 1239567734,
                2615827582, 3138350967, 1983810667, 1258771837, 156352868, 3459871732, 3838966901,
                1926475715, 2906860513, 2265591974, 739992556, 4097548896, 4063110249, 2614816895,
                3485800367, 2745680040, 1081661573, 739066934, 1373973331, 1210644481, 2279017196,
                1005551928, 2040599086, 2689171110, 2107331398, 4229586067, 3246733336, 1653467821,
                3803759079, 1396785958, 744631377, 2218132207, 655573009, 2223316795, 2924583188,
                3561375837, 949662445, 1302891749, 1031980062, 864295262, 2438572862, 1504001412,
                933269347, 1253638591, 1534271622, 2530288971, 3587024017, 4130814425, 2573905811,
                2785603748, 3552056080, 4244335838, 1333943369, 2283588299, 2119292262, 1681201617,
                3442164059, 1894009073, 2739198217, 1409299279, 3436643841, 1088847721, 837995626,
                1423764787, 3020531343, 4051404125, 618735582, 2571970479, 3106734771, 2404302700,
                1055667736, 397025181, 252340965, 1296678038, 512661010, 963140047, 1161844858,
                3862111880, 964139113, 3914454392, 2517045255, 3547177225, 2645896340, 2943432060,
                4173143499, 48482790, 1022192902, 500159282, 3637285006, 1101744863, 2363750224,
                312625286, 3984163509, 3118100618, 2790609658, 2907841435, 1239943712, 2072088288,
                159276689, 3267720981, 2856806276, 102734665, 3390376688, 3378951761, 975939361,
                454652442, 1925083628, 2151124106, 1307480526, 3774198893, 3018272285, 272747408,
                1016277798, 3605470575, 531634579, 2961498249, 821488395, 3558633824, 2912444276,
                1657117175, 2153161136, 1884360710, 721291427, 1531829639, 4115435659, 4254348147,
                1506441630, 4046066015, 556451623, 3337814584, 1018537690, 555628783, 1620746271,
                2482837019, 3679282774, 2426930864, 2632326440, 2327928883, 2353267828, 3481986297,
                2112160697, 1022061657, 758309560, 2644660892, 907838024, 1498392732, 528384055,
                1202823703, 1890604888, 1949099119, 3063001275, 685428701, 1780107085, 2619180751,
                2954672471, 2283259797, 1374458453, 1565978557, 1093442961, 3891637052, 780517054,
                503083440, 1375890406, 1386889110, 4237816737, 249672267, 828386282, 1163719531,
                132139434, 391942771, 1096514028, 3006936950, 2283483449, 1662382851, 3602503245,
                4242261323, 2784647410, 2828350825, 1288023746, 1017072535, 2006783410, 122598670,
                1977168703, 4075238695, 1691129759, 2007296448, 1519044036, 1701991670, 1153128956,
                1512900449, 1261204616, 2927982361, 1657752895, 139288437, 140564828, 100283802,
                3070382516, 4089107340, 191147528, 1712401819, 391845701, 1540517289, 1700899625,
                1477882518, 755000874, 480418626, 588196749, 400809532, 2485933533, 2888628826,
                332783138, 3009527268, 3784829804, 2922060404, 2516820570, 2217403977, 2521013993,
                2614722986, 3779307621, 317006621, 1733129556, 1267545638, 3033328419, 2185074347,
                2126485254, 1990828725, 1034359217, 3945891523, 3588995366, 246414315, 2407783722,
                1985580187, 2106721825, 35659189, 204134384, 203035567, 2650819784, 3097513541,
                195622280, 620515605, 2016439173, 304017649, 1791577127, 1576105034, 3535138410,
                3637374201, 3626803115, 1214754525, 1611339531, 1762283227, 2759574029, 32271764,
                2702968983, 1516226210, 160223584, 3507738999, 146234417, 2990154579, 1436112035,
                1948173615, 3890958043, 1222277581, 1349923980, 506218124, 1577016965, 3229129980,
                3912753236, 233704342, 1219563141, 429930162, 2006881030, 3328217266, 3101601662,
                3008827607, 3114439162, 4169647780, 2658687412, 2809028243, 297647962, 1609437431,
                3358143193, 1716156248, 1612652130, 2723039108, 2877712792, 227416933, 3655452320,
                3809918775, 3758975241, 2625261551, 1044900207, 2767833802, 749765276, 1997143904,
                536279565, 1568228199, 764519151, 4215912350, 263962810, 319470773, 3810995202,
                3465177349, 1243659365, 864064918, 4136562751, 3062846208, 708743160, 2187738105,
                3898834484, 2200754536, 4539814, 1789419448, 105721213, 2944213866, 1486345112,
                32430882, 253601522, 3408074780, 2992349041, 2451728307, 226661140, 2460239549,
                2100724593, 3852012222, 743449851, 2967418295, 605498674, 2055548507, 1922946771,
                2632283075, 4086716584, 623236339, 3410294246, 4133057483, 2812524059, 1205446946,
                3043927830, 3482614376, 1275680906, 3676217583, 513988754, 893728600, 1999049758,
                598451747, 3593194658, 609427399, 2673100544, 910865729, 3116238253, 736726549,
                3362221872, 1013768942, 718438609, 2217341124, 3587867839, 2300483002, 1946981080,
                3920103403, 2292895303, 3117736693, 877577485, 2187888016, 1368957453, 4283009664,
                1076391253, 1397877819, 2980178183, 3968837715, 867743243, 4043356977, 2946375050,
                26361492, 3432852562, 1912980855, 130799130, 4033884077, 3349537042, 3182198745,
                2179478628, 4252455628, 825955887, 93466691, 2196075887, 1784475710, 946546654,
                3933870986, 2244605733, 3490402086, 4030429100, 3023300148, 1745265920, 4112142076,
                2903907125, 2758734523, 3896908867, 2564201959, 26549320, 1274768945, 2576126997,
                3923164846, 3736091709, 370046710, 514332769, 4121086903, 2972568269, 4088911695,
                803020494, 3034337812, 317708135, 1334283383, 881113661, 1695276900, 3766800569,
                2061275833, 1227611332, 121744019, 2173946227, 2389688567, 192367735, 3892792894,
                2172535825, 117291679, 1935828849, 2721596143, 823956503, 1586432341, 2448062771,
                4054304197, 864073076, 4252759593, 1258481746, 4285979432, 1026798711, 2713633766,
                114893615, 3134616441, 2817271874, 666109477, 3691375086, 2759951811, 448361256,
                1541295802, 3882571749, 2284391955, 1326400031, 2777223186, 885169913, 2782624297,
                3537147598, 1692556532, 776208409, 3241275549, 1904954149, 2405119603, 3819789804,
            ],
            &[
                225066005, 745131552, 2942567538, 4258778781, 1967392927, 3741613691, 1890932195,
                1506277386, 1146302728, 3715076457, 3577274198, 3953573097, 2085422530, 661098011,
                4236873637, 2930925760, 3967583938, 2852055099, 355730378, 512021169, 2505846300,
                2749910007, 1454559275, 2553714326, 1536856909, 733797684, 656941700, 370199490,
                1949020029, 4091629764, 3452778818, 3434279798, 240547993, 381305151, 534633119,
                3783352536, 1551443097, 2973034337, 2180000100, 2053656589, 3320616670, 1724644295,
                589237508, 2241024734, 412948466, 3347181025, 1812386780, 1524837382, 3656593545,
                501624455, 3029327470, 273718066, 2666953466, 324438884, 2852820080, 120020650,
                3393646374, 2619314651, 3733169374, 3756112094, 1302185220, 3677713422, 2747067315,
                2815017712, 1868682331, 1943529649, 1563990165, 319757768, 2031825250, 2139207520,
                2978578108, 1560356796, 3699549338, 2916893113, 101966731, 1585835576, 2998190164,
                3967243725, 3959463829, 498949725, 3348011234, 3761007557, 302477959, 3992592959,
                2447724633, 2460308753, 1858976258, 3525120157, 384425177, 2978081316, 607592077,
                401773917, 1748638062, 4245512962, 1117346825, 2328817272, 847205838, 3491495030,
                470052326, 1893860493, 2076576843, 2289319719, 1260754313, 4064542122, 1206836886,
                523542952, 894395755, 3043786155, 2337338009, 2752103377, 3251660619, 2215772339,
                3015482729, 2593131420, 3786996532, 3672453772, 651643275, 1988656237, 3347623451,
                356296282, 2406730821, 3093333932, 2660898588, 81938635, 3783019719, 383826519,
                2681656944, 3399824694, 3552203202, 3981865110, 1183621457, 146668152, 3118763304,
                1958273303, 2057041878, 1183522949, 2882248106, 230586931, 680070970, 263636248,
                2634241968, 677653887, 4023567727, 3777994047, 1592901741, 2761934715, 910715025,
                1512493458, 165009600, 3565664162, 1187297321, 2558161419, 2468839453, 887950197,
                220662954, 1244941671, 3095343373, 2063342669, 2090489239, 2010099484, 1948084319,
                1365644807, 1235644818, 2588736961, 1564371721, 411294152, 3045805520, 1843591393,
                3688879242, 2388718380, 2305780976, 3853250176, 1061462164, 3916091720, 1198491889,
                3197591330, 1096143292, 2118906933, 1635799180, 1176220582, 2501989740, 801999335,
                2705462373, 1003645809, 2186975253, 1064745016, 1555323605, 4095122662, 1852179202,
                3337474276, 2971325331, 529480360, 4181951721, 1632451986, 1921251025, 2187957901,
                1550157150, 1055194859, 1178754848, 2551455843, 4253040930, 72428672, 2433250913,
                3380996315, 868858299, 2205090193, 2621190489, 1284715782, 1308118202, 628595114,
                2783487525, 3396644559, 3453264010, 1070669284, 1383523038, 3993112084, 1712356129,
                373012427, 126203575, 2392499470, 607718265, 2253959369, 1040951361, 2590357607,
                2175608952, 2297182602, 4276086375, 1607963583, 1082087029, 3529235816, 326427724,
                3716684434, 1901298488, 37770472, 3247695701, 1067103258, 3117586492, 1266520999,
                1168926910, 2626033388, 630871070, 4243084776, 188878698, 3514192446, 1923550427,
                206270544, 1634209660, 3863690423, 2574699027, 4168121673, 1449879463, 4224275977,
                2227130200, 1884692906, 606301209, 1210083889, 3640641606, 2504768822, 3610799696,
                2178748368, 1708820271, 1819469384, 2802870165, 2246886931, 3567016595, 3555599583,
                1348994237, 3790873838, 3926636967, 1088645573, 3438253657, 3323650855, 828010249,
                618829949, 1504599310, 1908275663, 4270232696, 4036146227, 1856897469, 2998994751,
                2752368926, 870328089, 338595046, 3616029110, 296696765, 1403260870, 1830786965,
                1227812187, 2223459989, 3589209104, 2051289912, 3318866022, 4094335795, 3787444415,
                1234353630, 3808781435, 387466633, 3211872777, 4202021032, 639324231, 1976675272,
                2193056063, 3199669188,
            ],
            true,
            &[
                3923579889, 1687605753, 3377794191, 3322208449, 1203553350, 2154425101, 1359863697,
                3023444870, 776353376, 592836194, 2511309652, 3922934431, 4150213939, 858418424,
                2956152570, 488137268, 931462870, 1642630985, 3149845100, 3318171957, 538998706,
                5191729, 2137745225, 2297747699, 1984407713, 643246259, 2253618591, 523932289,
                191175769, 3642012130, 3400043084, 1707948340, 2992423698, 81890223, 2378233131,
                3079312226, 1066946509, 803135022, 1994948858, 4041732309, 1337312336, 2966488667,
                3428464355, 2737588983, 4228360315, 2146901404, 3059157811, 2476369890, 75140925,
                3251604233, 2881152809, 802794015, 3378093993, 96836131, 3099865764, 811380027,
                133901715, 1047457206, 3609305681, 1771180437, 2461319522, 3515701714, 1066513666,
                1659070392, 3086795089, 3716662526, 559065127, 3237292382, 3631218434, 3795187732,
                704836609, 3786271895, 4061286432, 405232026, 3609248662, 3980760947, 3571087184,
                1847154731, 2094552306, 814159853, 4260485048, 3149179255, 912325889, 3689884426,
                4138663955, 14357881, 2050778168, 3912490158, 1861501789, 814905489, 3254286948,
                2032825464, 3271543587, 2219425366, 4225852701, 1857868075, 1298372166, 1102499025,
                3574133387, 3812312663, 448380962, 3807290049, 4209544034, 2256927203, 412478498,
                1228300916, 2401082596, 350244424, 3441586361, 3136422715, 945678990, 2116172969,
                301198401, 3322856984, 3407592301, 188389305, 2628435530, 597331343, 433647333,
                3063380936, 3817880039, 2901586764, 3507352615, 2352939145, 3490273512, 4090562787,
                1802676852, 514644644, 4124694011, 3639332010, 1115134207, 227756304, 3818731261,
                4222040109, 2430937604, 2156693403, 2184651107, 3551740749, 2132540830, 1150238882,
                1052043654, 571348510, 3172318934, 3673669988, 692769159, 210233084, 2958517959,
                2297993342, 3316672837, 3302014323, 354145219, 2484852144, 3683759209, 3408466568,
                2978225356, 4097994925, 2538336670, 1392418428, 2899544976, 457988587, 3773763501,
                2012444919, 150975722, 2234422639, 1822822763, 551692314, 1014406649, 3374930924,
                1240545416, 2294327444, 3432793083, 1000394478, 2103969650, 1395783166, 4085383479,
                1542514796, 194271393, 145341362, 3655868181, 633459061, 328054273, 3431726030,
                3557285056, 2901104858, 2052553042, 1042550372, 343075862, 4115125230, 3906797654,
                1301043752, 2559925980, 1623793157, 628627946, 1884173596, 1228259826, 2621415081,
                2785923512, 4270141343, 315667031, 282856266, 429519802, 3536348488, 2322956754,
                3194951300, 1510744177, 3616915522, 3309727286, 517266630, 1119984084, 4026073305,
                2238785198, 732618834, 3368263738, 4087015376, 2697003280, 994552013, 4290467065,
                2365091630, 59691114, 169931479, 1498618224, 2382537145, 310192470, 900458915,
                275936028, 3655907299, 1961047155, 943058038, 2333911425, 1578159189, 1847055172,
                2659538736, 1993399472, 2491904542, 1070133929, 2666142065, 2803306446, 284508539,
                2535344546, 798366741, 899869216, 282425661, 1522885207, 1934768011, 1935282035,
                1783703650, 763335394, 1114825113, 4001525101, 3326921253, 1057031491, 231338938,
                505911721, 3073700058, 3881013726, 2622491116, 2434260580, 2104610402, 77048668,
                3840229888, 539240724, 272637566, 3598686634, 832397853, 3941981941, 3947689731,
                545125868, 199321616, 195433751, 587082201, 1623079028, 3583449493, 4189372053,
                656157970, 4065268173, 2321947633, 4117128060, 1042761195, 543619922, 4200179141,
                2194761475, 878960867, 3598234187, 3791350851, 346665711, 1360737771, 1930187385,
                2405900037, 3062309031, 2085261831, 707121842, 3173108868, 503413591, 73514434,
            ],
        );
        // - lo >= DC_DIVAPPR_Q_THRESHOLD in limbs_div_divide_and_conquer_approx_helper
        test(
            &[
                3852468748, 3866732619, 592701608, 3778254743, 1280491238, 2834833773, 550923334,
                1479705073, 498349312, 767409174, 42211279, 2128147526, 2926662909, 141411040,
                1065100366, 1527285446, 2874448851, 3117914313, 1218471999, 4280625327, 2037562894,
                1609603468, 3241376868, 1996556629, 3588969003, 1318580268, 3687594860, 1362481322,
                2944468172, 1695770272, 3083883725, 2377648522, 2496254008, 749263050, 2770685897,
                3512231732, 3023316399, 3384859643, 1220306495, 391138498, 804657736, 44216427,
                1931816623, 339239167, 2662606581, 2648352860, 1235726472, 1626990950, 3004127325,
                1053956347, 808327756, 1442399511, 4286364870, 31595646, 2379491886, 1328294899,
                816658097, 3205321744, 3963029596, 2572812025, 87366995, 3217997432, 1508641500,
                1825064063, 1941897747, 1436358378, 3017514874, 941910582, 3588117902, 135801271,
                2867936354, 2658316194, 407580200, 665652508, 1146163780, 3147022746, 3669479157,
                3697518011, 3942322922, 3433464004, 147802483, 1146857432, 73698519, 3370749655,
                1389534220, 589317299, 177267891, 2688250309, 2128670608, 1265200580, 2019760989,
                1150930096, 2585252733, 15447323, 3088238466, 1783728664, 2811524431, 3891707403,
                4029016169, 2907528558, 1270616768, 166893059, 1545869782, 4275310588, 2441090079,
                2829901749, 3736960597, 3941382642, 2821131110, 564991139, 3126664226, 1998382701,
                1328551552, 1210689595, 2996963417, 1557893584, 43742716, 876455611, 1677218718,
                729180347, 315949329, 1732639163, 1821348164, 151672496, 1329099718, 974261301,
                850419612, 1856037997, 2615159114, 3710750197, 4091633114, 2586774188, 2588577112,
                1339558569, 1735462778, 1262938743, 1891014212, 3269098922, 1992796362, 2808355044,
                3874652326, 2745283316, 1735823559, 3407614800, 370629516, 1152369319, 881914057,
                470647543, 3939410385, 2852748779, 853680636, 3231204925, 296778651, 3315599780,
                970480725, 1570957625, 3243621138, 4081630171, 2137632035, 488321574, 3274673411,
                996742709, 1121917553, 684178420, 2611815203, 1398907392, 3313511725, 2114905089,
                530433742, 1501425765, 3354176655, 3515249223, 3432236694, 2338559491, 3638772717,
                3253118328, 1594238823, 3094732444, 354679189, 1860678321, 86892394, 826940850,
                1314179547, 3978569567, 1699444163, 3462511739, 2591772690, 3726539491, 728424834,
                2932676784, 311068414, 2129104220, 1026553636, 2242765511, 1733810815, 1138330866,
                3652398326, 3074523173, 3839981206, 1015663483, 722242539, 3272711437, 2982099959,
                1938802295, 3022788284, 12930779, 878443915, 2907693541, 1979049428, 593163227,
                3823442786, 3215213583, 3695525644, 1119636738, 3212997654, 3027851617, 1144485880,
                164747928, 2781230452, 1258345532, 2869324653, 3068929942, 1028739729, 3380289029,
                2295840170, 3545102314, 2855402743, 1772822517, 724460983, 2228954158, 3208333919,
                2821114010, 3148029409, 4068546439, 833494094, 2960251966, 160156911, 1983852913,
                2791477758, 3697190934, 153093720, 104144622, 2441461309, 3159646380, 1733652873,
                1562477414, 2555472454, 208737200, 2703626976, 3180538910, 2592414545, 3416739924,
                3356130918, 4069598606, 1949794549, 1771742849, 3756687965, 1452093924, 2443380777,
                1591957147, 1211239165, 1876128705, 4247423793, 398751752, 2564049252, 3016849411,
                379369954, 1500293555, 1436202725, 3000709783, 2380370269, 1733644538, 1525628809,
                13455701, 3296626657, 521910174, 2364967601, 1781968745, 1399663633, 3043350501,
                1368246629, 2392664448, 1283732518, 785313394, 3034454976, 695088212, 1511797594,
                1863608029, 299100351, 2597420190, 989884842, 2042620176, 1636899254, 2378486752,
                402706413, 681449111, 3318018166, 1613374280, 2127967446, 289891229, 191268175,
                2824016875, 1141385569, 1160748059, 2298038638, 2273363044, 1083067452, 2227633073,
                3411833901, 2501670656, 1136686475, 449786314, 1206095141, 1428680228, 373851423,
                1065715962, 3842120141, 1446849052, 142137120, 2503522334, 2890167721, 489029335,
                2904499752, 3583810486, 141821369, 3975802804, 3139899326, 3360723646, 1325094173,
                1458543621, 3135728890, 1722244101, 546577431, 2431082502, 2423442975, 3716191600,
                4106127568, 1535072638, 3490647587, 690007845, 123509837, 3008940800, 2522129913,
                509564148, 3464677431, 2734250447, 2521445361, 2724421404, 1464248423, 4286188561,
                2041779154, 2040896472, 951622461, 2742066946, 3643978111, 3702076160, 3277267213,
                3516427880, 3288032742, 1694027041, 3062496840, 1755611164, 2144317315, 1840150810,
                3659412421, 269788590, 1685549590, 138245102, 3801925635, 2810327973, 3857961937,
                3773132978, 2859864128, 1937028826, 1803313773, 1721171315, 3641809479, 561594878,
                3883288939, 1615573909, 3567576946, 53427597, 3447054169, 924172132, 289810649,
                3166741000, 991317667, 72203781, 985172668, 3013352759, 2107570161, 3250293704,
                4114482787, 1050030772, 1057622762, 3438626748, 2519260970, 159154610, 888655185,
                471817829, 470991380, 2525869562, 257437760, 665926900, 2709862184, 3205674228,
                1381649365, 1803460892, 4134101678, 2543649326, 1209534353, 1226459212, 3664518300,
                2168714963, 3110951542, 2920091469, 280549216, 3763064171, 3139277729, 2961240010,
                3997887886, 2249335682, 4125195068, 2179174964, 4200864330, 1948058108, 1468274578,
                2376503735, 1402764211, 3508183492, 568398904, 3916597152, 1316008321, 3354808565,
                243276170, 383567360, 3854059575, 3533629121, 450497668, 1613229695, 1579942631,
                1479512959, 1708803214, 166236912, 3371361562, 1178374935, 3631449184, 1044231267,
                43160759, 2872462763, 386332526, 1454108301, 463466433, 2881621607, 1902397631,
                3675190003, 3262456635, 3682680678, 3921732891, 1604135368, 3949662129, 3129246190,
                1954536251, 1538666313, 2274928374, 1650180856, 3652524428, 1821056613, 911456961,
                840333318, 764118487, 2652399697, 2580502346, 2438391701, 2513677855, 2640798083,
                2778108413, 3625988619, 3121963132, 2016358782, 2026634297, 2454141008, 3719526468,
                2069512197, 595695222, 1372551058, 2654832919, 1312895472, 3449972978, 1039886392,
                1219309116, 2158257939, 161673294, 1466523266,
            ],
            &[
                1228082563, 3958970957, 2292561045, 1680572099, 795956301, 3199900361, 2775759977,
                470071915, 418570831, 1211920274, 3653536884, 1024252286, 2421732290, 2695210567,
                602247103, 4076240601, 547232254, 3118467901, 373262705, 2538006674, 3879907903,
                484206954, 2539623040, 4205486376, 1910552745, 3038212402, 2654312301, 3586427708,
                954791645, 2348064693, 1497313290, 2715388363, 3338088407, 2319307053, 1080519064,
                4159525507, 1284211779, 4015474700, 3753400642, 388134024, 2776075876, 552537674,
                1195536084, 1000211139, 3608292187, 456221898, 2979823937, 3666331215, 3448143527,
                752704015, 3181775208, 1221668864, 3658472905, 2038458530, 2182904194, 1539979012,
                1086813538, 1185254470, 810158065, 3448748698, 4194931499, 1623694312, 298685602,
                2265178291, 1937541694, 1775046588, 1544843229, 1529059357, 649269555, 2357792903,
                2571831955, 176354978, 1299551755, 830143712, 1015814111, 776566019, 2627651913,
                3960393788, 1268944216, 3414991379, 3922712735, 2519760646, 1906700144, 2590958602,
                353821006, 2218545353, 1375841084, 2092401658, 1639008622, 4082695593, 2141419111,
                3236669066, 1724206670, 419369846, 2835126061, 3041510267, 153991327, 2126200895,
                3461262411, 191209011, 1342692414, 154698036, 563023945, 1510300314, 2248068391,
                3846972081, 1992059708, 2646433655, 1714730200, 2303707509, 3981692650, 3517398169,
                2281282719, 2519619923, 4271369978, 293814410, 438829639, 3337765624, 622782672,
                944352733, 3715313213, 531335520, 113528832, 2419325895, 2984041163, 878536167,
                3214930707, 2146787348, 4917714, 3257269189, 2452274107, 2672157272, 1585645759,
                2131367341, 3585828795, 1333843792, 3923459029, 694088371, 1013702168, 1741059100,
                3997989379, 542853888, 1761645159, 406716499, 1694374453, 1624534824, 3459630459,
                802104445, 128207303, 144400385, 476582731, 3051345902, 1368383669, 1166299002,
                2707219053, 4004021138, 2679504095, 2592982338, 1033724773, 1157182117, 711981576,
                42503937, 2924924484, 3595784172, 3595204272, 2994216676, 4221872019, 3722581147,
                3333988566, 1523164764, 1080358318, 1413433846, 3738351350, 977402259, 1992977980,
                1559146947, 3211129812, 2160246550, 2367537106, 446281713, 2153747232, 397871847,
                3366573642, 2055448941, 41153636, 754933361, 2993598965, 3306066874, 3631977724,
                3123479847, 2585741316, 2994078662, 2509626417, 4066762390, 4211900185, 3742567686,
                3545119766, 949327829, 1947239775, 1596917691, 2428932126, 1055142932, 562149283,
                3199507769, 2908430793, 814804964, 398782122, 1053511625, 2957145782, 4214499480,
                2538107930, 3510188017, 3044050202, 1179016763, 3073719558, 1010197763, 2365480195,
                796718852, 2056040043, 872695428, 645007655, 3691376745, 104700630, 2649843250,
                4122935255, 642238345, 2295983257, 348265112, 2773179245, 898141051, 2579996826,
                2612225547, 3972451489, 2646202581, 3681992299, 181022653, 1485802997, 2440387741,
                2520682234, 51353550, 114076802, 51940680, 2817210994, 452896124, 3254045803,
                2864556183, 81661781, 237542081, 1973800766, 3657654720, 2530129707, 403307022,
                68172699, 332554330, 1181024212, 811837036, 3695906025, 1961935526, 3608053891,
                3566607807, 1505243326, 3952431950, 2229199298, 1258499123, 3907526592, 1202974947,
                4257231754, 383365712, 2370787478, 3966045621, 3709646864, 341562985, 2306743433,
                3087453648, 1119686719, 3010178775, 1966960488, 4005245264, 742100958, 1841989955,
                4073161558, 1387924175, 3136835241, 4153208836, 172275833, 118390136, 3068417534,
                3427994941, 1186924296, 3531763845, 1587276588, 2491453930, 3092663154, 186325597,
                3735008023, 2520603222, 141263570, 2586779772, 1097287985, 1184074419, 2231683844,
                1006316112, 4242375661, 3021145139, 2474020608, 3059475910, 4077499345, 2682549550,
                899267458, 2939020861, 212132728, 4070534479, 899246172, 3206389107, 738644506,
                1196824568, 4135770345, 665964490, 3404106178, 2800460425, 370991796, 6173656,
                989425002, 738558511, 1186448823, 1117986518, 3268400991, 694618816, 743869433,
                1081820528, 1169090980, 2808795201, 2626416762, 1852742088, 3500519845, 1862566642,
                2528685207, 38532995, 3254975580, 2293790115, 278388535, 4214498933, 3355793258,
                2429295061, 3055950106, 1916510220, 408474308, 1993547223, 3283433638, 2876081292,
                3535523705, 2101519467, 4222280709, 1877650206, 1362736623, 1125507149, 2753028888,
                2871670444, 3984235984, 2837413105, 2740068704, 2501544770, 2062713945, 1827003918,
                3387715121, 3978334051, 164380626, 911533483, 3159921130, 3896040725, 2453247705,
                1732359707, 97075111, 1167974376, 1292533086, 3978529628, 1173190459, 1180474283,
                4027781835, 1819164482, 1138188719, 2554004670, 2238689551, 2719133108, 827123216,
                3402746862, 364981183, 1215495008, 3697503228, 2459473144, 794939056, 2850414988,
                3880571940, 3446126388, 387496900, 4012258040, 233036233, 1295469109, 2236990087,
                289904073, 1533599469, 1786729109, 3155911485, 1361493542, 1709668750, 2979939253,
                899656352, 2488414909, 2345651638, 857006573, 2499927665, 3004854698, 2767508418,
                3057934317, 2570296632, 2340731830, 1704320439, 2217465736, 483270163, 433273029,
                3463124152, 2858495981, 2293137585, 3765138357, 2050975825, 1062202999, 2389813229,
                327867315, 3524908521, 1060359725, 2437188567, 4037791947, 265987565, 2531318664,
                3789582609, 1036190383, 359313632, 343266182, 3818075324, 1483429641, 3251308104,
                2231865653, 2088816062, 1363097078, 2619396780, 3922946100, 2087714470, 2714841575,
                313007644, 3053385186, 31524825, 420269932, 2184870970, 2633967376, 2976557270,
                3907847269, 1704768911, 845456817, 3013183512, 673429122, 1088006999, 4254909288,
                3477532731, 2633706423, 30604022, 381299887, 124029095, 2146491520, 810411389,
                1842896686, 1189234944, 490339649, 1115698075, 3923320995, 2822835404, 2863188964,
                2041216097, 3722786391, 1411085302, 643395602, 1892259907, 1276564050, 92834220,
                2113523130, 3750780619, 413882810, 4162989, 2342682023, 2496728931, 3765633586,
                1201219647, 1509483486, 1571999425, 2150766213, 4078947670, 1554603889, 2829344380,
                915004628, 2209136503, 1860939599, 1336190314, 3883277228, 3526987435, 1959059220,
                2644428109, 4106090871, 917923267, 1544296685, 710190597, 1252330700, 445060090,
                3748836645, 2225146599, 1869685424, 2948785336, 396398905, 4121416478, 120806631,
                76766006, 2336284840, 2353857164, 631928490, 2358081096, 2401392404, 1781483930,
                3507012756, 114196273, 761825995, 1747713254, 3582101597, 748625738, 2646455218,
                2905499938, 3230684494, 3213162530, 12062405, 2034846069, 2240074509, 3895726666,
                1559793991, 401560906, 1188895923, 3908766043, 3080529735, 1723996093, 3234235196,
                1524383480, 3006010429, 1021800042, 2598013312, 2404289459, 359714242, 2353150857,
                3214159696, 71259951, 3244867195, 1475212366, 1519569068, 396867142, 1704063457,
                1255619835, 1809747439, 2455947434, 2914124280, 490937656, 1909042102, 2425011492,
                1653816077, 1744426598, 3540268790, 1266922070, 3157358103, 764478931, 143507967,
                4138193217, 1618357750, 4064415341, 915914627, 379606079, 1684722693, 731091210,
                3441411920, 904319643, 4101866043, 1130761207, 2746556498, 2962076734, 2459719176,
                4014414287, 2306516630, 3241923533, 893098894, 155104880, 2462450145, 3698302787,
                2611896162, 3483738489, 1960417753, 3190054039, 3176206866, 2149242647, 241842249,
                3791458236, 266968864, 653002708, 3295229436, 720704829, 668777771, 2920041289,
                1616327007, 2614675119, 3120745852, 3703698323, 642045983, 664497190, 4138076932,
                416801804, 3836460143, 2512697726, 1321118683, 3790475422, 2372618883, 3906311778,
                3636945722, 93045717, 3245150294, 2203703812, 2876544088, 1482347470, 536620593,
                3055935093, 3219276551, 1736510361, 2882369741, 227297457, 2287462988, 1314417701,
                2698591585, 3562176889, 1108866670, 986625362, 3935882051, 2153826689, 4088091186,
                2094026213, 2614688936, 4293577301, 4053305951, 4075217449, 3960924855, 347332803,
                1968625524, 114389475, 430194382, 3252648209, 4018348764, 1840000995, 1589897244,
                258799598, 1737020958, 1683355155, 1473302633, 1904664311, 4282826162, 1231154140,
                4030919851, 1206926359, 3663795445, 2379135003, 303432555, 355307489, 1769635102,
                2552468836, 3490641321, 3099203148, 528813020, 3258352325, 1295334678, 3143722334,
                1978938927, 286181973, 2742706533, 1605086426, 1943464059, 2101290175, 2169564592,
                2507578488, 3835864045, 101762307, 2100748227, 3058433907, 3335629628, 2214316788,
                1268535268, 402136621, 1768704361, 1113090330, 3771703632, 3647202551, 2716086257,
                1380543566, 888137953, 3154815844, 1427117965, 4171112788, 1823391124, 1935813073,
                3498407768, 2295745053, 3209451097, 3427436731, 1760982025, 1197278449, 540450594,
                3647987196, 3703306058, 1018206619, 2476714098, 3182222236, 2163415625, 1564991911,
                377612233, 1240138195, 438211, 2660780499, 3842090366, 2254892076, 770175200,
                24158863, 2695586572, 1592642609, 6566621, 3696323406, 545103586, 1539983422,
                2835737393, 3243834056, 1307518947, 1449819027, 3732497678, 361142215, 560917662,
                1000073399, 2993529240, 2447753259, 277468387, 2097588385, 2296579141, 3491393176,
                593689188, 3981379323, 1199729884, 3155291067, 3484678877, 3241832730, 700414265,
                1991308981, 4120895059, 1897048619, 3149808222, 439364012, 344761676, 3062212838,
                1283609663, 4033828278, 1778135949, 2420787257, 893368411, 602240535, 1331176104,
                3252174301, 2911796787, 3606169310, 3656396589, 3672036583, 2975308991, 3008310252,
                1216481017, 3571455814, 917390807, 3398014016, 3881198175, 3246069043, 2018429161,
                2473741365, 115107297, 3264952157, 2580209049, 36273546, 3206338009, 305508401,
                1376925433, 2056571909, 1019775170, 2706062024, 2901924304, 2190222733, 2740737267,
                21015048, 1250529961, 4189046693, 1010891679, 1852727754, 2130616276, 548460973,
                1990672906, 1047266283, 1377617303, 2745241728, 3579697777, 4061895368, 2785622772,
                2769532205, 3150425811, 210098228, 799917387, 3243975475, 3585288930, 2961830047,
                3048068800, 3653640841, 3600794853, 2717549000, 2701285348, 3943817837, 1093998117,
                2069876461,
            ],
            &[
                1904169149, 2631229056, 3049906843, 1882970882, 1342610636, 348735230, 2030868119,
                308972916, 3291280440, 1713067370, 798189929, 267083738, 1426325271, 1400323459,
                4220917798, 2838399613, 2788016416, 1808463064, 3797033758, 4019522812, 2732377100,
                2865589386, 1858695675, 791228146, 3278919839, 1007346809, 2152870094, 4143135690,
                2156969365, 276568681, 768119249, 3720025676, 3691488609, 2122964639, 1090037461,
                1573203071, 1029758226, 512965488, 1081406960, 3875174920, 4202135603, 1561629768,
                144375534, 824852270, 2139340951, 2521611777, 617945581, 1174393067, 83560822,
                3171084584, 2743715748, 57600123, 2500345583, 509310341, 1466707920, 3965654944,
                1780152208, 2543890327, 3951358979, 4064884222, 1148693723, 731908693, 2303119198,
                2162132555, 3899764661, 4062906460, 2910202154, 3507799436, 3049585561, 1547101764,
                2240591447, 538691646, 3327231079, 4101167722, 3558467927, 632232393, 3069596916,
                792109379, 1443377440, 4063660922, 2403295483, 3242450069, 2054813912, 3224639109,
                2116127467, 3510867445, 4131878961, 554538806, 763384993, 4028933429, 784809516,
                2188214742, 1245086980, 232165771, 3893787601, 1616832299, 3354660858, 3756152253,
                695071424, 3616331020, 3728930728, 2300889664, 1134765807, 2568071856, 2230185436,
                1313073173, 3090697900, 2811549719, 2626884644, 3174004164, 3418927780, 1459400962,
                1170992180, 700259624, 2882894395, 3735934167, 64057275, 4285443281, 3858505044,
                3530830770, 3881898558, 1248666153, 4057784557, 825094518, 3954584560, 1379703941,
                456663582, 1936817375, 2100203590, 2269578911, 1927092754, 2056845231, 64996052,
                3180336408, 3382367590, 1579799629, 369271331, 3393685503, 1678462084, 3580453071,
                3344059212, 2603016988, 1704438831, 2960085271, 3944849185, 1984458477, 3159618129,
                1356105311, 3061973253, 1169634390, 1581525372, 2092399029, 3856589897, 2747272433,
                129370532, 3259112907, 2392093327, 4224720237, 2467960903, 3012864179, 657311189,
                792107047, 3702308235, 2589079343, 1977762240, 2054656079, 365049703, 1136316952,
                1766336156, 1574143327, 1448002399, 1545340743, 3343582998, 4266810491, 885565293,
                1012325697, 4265244667, 4164478481, 3171829328, 3052840557, 1524972073, 3381586754,
                3989036728, 1081734627, 3737298307, 1869630638, 3072103206, 3329790326, 3520758799,
                163463995, 3179586285, 2504600081, 2392849343, 2696495366, 77291649, 1524994513,
                4129364215, 2589017064, 563995388, 2948622336, 3478580619, 1302885151, 1885238682,
                3399987677, 3819885284, 291981185, 2438308465, 1131073201, 2094997380, 4030538680,
                2493835385, 781324886, 780626834, 788047573, 3251459753, 589598543, 530167431,
                691366057, 2997152103, 1895699987, 2067335840, 911810562, 3044146727, 1981642612,
                2744164616, 1268584352, 4091929647, 739635330, 2535146835, 1232134198, 2821166834,
                2694137328, 1845150759, 4145746520, 1570842207, 1877662863, 2624263192, 822008531,
                544782319, 313703935, 2549594582, 1507198513, 1165274204, 3706172942, 1025658022,
                1367415969, 3754039309, 4187050684, 1116191986, 1599395850, 4255758850, 3627566792,
                2013052489, 3678795519, 2570182055, 3933838202, 2878788816, 245983387, 3805875941,
                156152490, 2288310229, 213932492, 2816431456, 3032792377, 4098973047, 1987668121,
                2178466333, 3171514186, 1824097633, 2477633111, 905563455, 2561821790, 1164919073,
                3346223498, 85587270, 388762173, 2830714978, 587416024, 118417479, 985918121,
                2869840425, 1942696561, 1447188301, 1252144775, 1754507324, 2995197874, 348722845,
                1994558236, 3180590158, 3621098260, 2808142053, 664847652, 2216211159, 1812881783,
                3652176793, 1818451392, 1492863756, 4248433045, 1393154835, 2899742381, 1639554680,
                3891616188, 1401178839, 155094399, 3335589331, 3242149447, 1506620207, 4025305548,
                1378564899, 2250748936, 3892206821, 2999250953, 2404072212, 1227092115, 3842156829,
                1175300589, 1366215256, 3655797370, 555970590, 2216476284, 356175930, 2316653842,
                3511159891, 3242198801, 3709284219, 3992609721, 131038128, 561182178, 2366328214,
                2169412954, 1843648386, 1596245006, 590747798, 29988297, 1032233337, 4148540805,
                4287292895, 1784151615, 775442479, 3254055272, 1319462308, 1998762911, 462052502,
                3240867011, 1529884215, 2445458752, 687926411, 320372384, 1993875140, 1103187463,
                1101261370, 434642811, 3921005377, 476413110, 4114908941, 399340969, 550419121,
                2138268102, 1961648534, 2475107221, 812733153, 1013882638, 804382040, 597969578,
                3969389037, 1681594804, 3597283127, 3246865960, 2618922824, 2412709123, 2883379189,
                2268836289, 4041631960, 3932960611, 2819242006, 3434298940, 1965495785, 1801489474,
                2483685757, 525486136, 3103413787, 2524236016, 3297675542, 4250105246, 2163911845,
                868191034, 768077954, 1251877129, 2082725352, 2563716501, 324144134, 2949337083,
                629793518, 3585348714, 2283159799, 1662388577, 3211992651, 1893003272, 367507404,
                597480849, 3530706203, 2180056290, 3040505611, 3393423843, 962401948, 1880350816,
                1059318370, 2185874635, 834999744, 976189277, 2214236591, 1042067189, 3972166345,
                3379537993, 4180308634, 1894580429, 2768917877, 2304053036, 1074132791, 1172385654,
                3558332507, 4118553815, 1144817158, 1932880741, 139994662, 2006231438, 1878531442,
                1451662501, 190676819, 2374814207, 2111662756, 2422642399, 1219123380, 1437107644,
                3325772043, 803974620, 599430876, 2919848672, 1661638336, 1841139086, 1961142950,
                3330180086, 844957898, 2211184671, 1097521602, 488948427, 3209202495, 3087475496,
                2505815200, 1915071794, 989106472, 3460765890, 1592714830, 1577504527, 544612468,
                527148798, 3186590344, 2423036586,
            ],
            false,
            &[
                1274629765, 3141211094, 3066783437, 3586463614, 3484042397, 3524692172, 1849759418,
                639170343, 4042051785, 1786641195, 2781936314, 598559471, 4206822552, 3076839520,
                3426707763, 2541903555, 3985517971, 2924425523, 805503712, 2235352830, 2985073687,
                1986582861, 708824890, 508981633, 2929420962, 2821655200, 1435071022, 2815105969,
                3064101020, 669540051, 64719297, 2298557985, 2663556386, 2872778448, 1920581662,
                3964906867, 3545183326, 3734685799, 965493690, 2511488644, 1289889793, 2716496762,
                1482707283, 1243240515, 1600220728, 3386832682, 1230771057, 1851371335, 962673373,
                3303433511, 2971821380, 465715438, 1476479395, 4210224292, 3326701593, 262885511,
                2623286176, 2785619040, 3631756223, 2689408595, 895184011, 3039724640, 3256518757,
                3157339284, 2736033980, 2623616149, 2262942125, 4066660566, 2010739028, 294414070,
                468536368, 39081857, 488790327, 2978306577, 4076063380, 164990706, 1769246328,
                2404180723, 1910734795, 1212674812, 3651413973, 4107687624, 3562008089, 2495538965,
                1060913944, 700681324, 2213076011, 956710321, 3766900265, 1201745400, 1689241027,
                2023383937, 3309067962, 4290293260, 3262071778, 867200275, 3722256414, 2704572302,
                798356854, 4060154398, 2192265925, 1631346682, 3841392362, 364742914, 1350896170,
                3826205664, 1387850479, 2438144424, 1722720699, 3355734439, 281374639, 4294332167,
                2848655781, 2650822950, 52206106, 1267368119, 3786144635, 3997971973, 3766740222,
                157940553, 874086492, 567057084, 1468392762, 1319013026, 1763094857, 1258351011,
                4219111833, 2069011861, 1508267766, 1742493547, 3709667481, 244137977, 3867904379,
                2749217489, 1901868294, 3141009904, 1334349455, 318703881, 790854628, 2456409186,
                2729035424, 2140696710, 3057979010, 1943182218, 446440787, 3044060750, 3377747126,
                2650171384, 2030143427, 68841182, 3417107910, 1982395041, 213127592, 4263826627,
                3810676045, 2815565679, 4215729650, 2424656131, 2427868488, 3538613128, 2004058276,
                3156863142, 3493891883, 1282367517, 4021690990, 3880291659, 3234626473, 733983061,
                3094727704, 2181100156, 2419795021, 261213792, 1806943490, 2451407303, 2727992941,
                3870486736, 2318523062, 160305854, 3405051241, 3671726857, 3273962892, 3354100289,
                2703913540, 304014061, 2416130602, 1740666524, 1270481586, 829357964, 3389415821,
                1783014510, 3440915536, 2186635473, 1638518047, 2861195291, 1925018981, 267706283,
                1852657873, 3918895473, 3639936236, 1540281136, 3992865037, 1795928884, 535725314,
                2620251512, 3768070603, 342868542, 2807296004, 1944907465, 766931863, 507809586,
                1382623024, 3748022748, 809199766, 2606119767, 440758364, 808682457, 1891522267,
                2798966179, 1932040027, 3544577672, 1613760857, 2497954981, 3761568437, 2673674694,
                2267348907, 2169995118, 1082902734, 1924288552, 2560273581, 1328627426, 500474681,
                2740807968, 1058091958, 4193976709, 2145630875, 2850935242, 3475556484, 1444110200,
                2094948501, 3669224179, 3527535779, 1309727297, 3043349576, 2800927336, 798778876,
                2615924118, 1736199098, 4167626369, 3609129416, 3696542059, 2660961906, 3709573368,
                2742620902, 2002154977, 1840222458, 678539421, 391765530, 3711225631, 2336471984,
                808858486, 2107964482, 2120664214, 3475065201, 3782915227, 2313935531, 1140567569,
                1920486212, 469939343, 4127720462, 1046677730, 3365009246, 2650242485, 79281379,
                3069462836, 1131058884, 1346966266, 2182389628, 3425864155, 487991612, 4274615916,
                3585882986, 3882253255, 3299152123, 3923136894, 4158963676, 1993380396, 3025592126,
                3697510913, 26996074, 1666693894, 4128726650, 1349974160, 1162680197, 1002804513,
                2192715135, 3733784449, 1296058713, 756389024, 985468845, 1910174731, 3512126196,
                2889051038, 896547684, 1052365249, 1712208218, 2895783918, 4015271697, 4194906302,
                1607592000, 488260174, 1247101503, 2635736706, 101416663, 502456820, 3004088397,
                2217260735, 3689763077, 2750526303, 2814008209, 4290490637, 459845084, 194661656,
                3298271035, 499760349, 1457282007, 1720527994, 283878558, 114686420, 207781028,
                2101242696, 2475285413, 4066549966, 3571095625, 1456774526, 1957825881, 644287366,
                859324850, 2429216562, 4069493991, 978462144, 4019464424, 269773692, 4086951598,
                3169184115, 3840130624, 997907757, 3420447554, 999500052, 2700308046, 53871682,
                1856963, 1188612325, 3393695655, 2729360995, 4101441780, 3357241810, 3598434336,
                1257477782, 953991428, 4060500318, 1564349166, 1626434500, 1259685422, 4087690819,
                3327437017, 1630386771, 3761712654, 226730056, 3668971305, 2810327973, 3857961937,
                3773132978, 2859864128, 1937028826, 1803313773, 1721171315, 3641809479, 561594878,
                3883288939, 1615573909, 3567576946, 53427597, 3447054169, 924172132, 289810649,
                3166741000, 991317667, 72203781, 985172668, 3013352759, 2107570161, 3250293704,
                4114482787, 1050030772, 1057622762, 3438626748, 2519260970, 159154610, 888655185,
                471817829, 470991380, 2525869562, 257437760, 665926900, 2709862184, 3205674228,
                1381649365, 1803460892, 4134101678, 2543649326, 1209534353, 1226459212, 3664518300,
                2168714963, 3110951542, 2920091469, 280549216, 3763064171, 3139277729, 2961240010,
                3997887886, 2249335682, 4125195068, 2179174964, 4200864330, 1948058108, 1468274578,
                2376503735, 1402764211, 3508183492, 568398904, 3916597152, 1316008321, 3354808565,
                243276170, 383567360, 3854059575, 3533629121, 450497668, 1613229695, 1579942631,
                1479512959, 1708803214, 166236912, 3371361562, 1178374935, 3631449184, 1044231267,
                43160759, 2872462763, 386332526, 1454108301, 463466433, 2881621607, 1902397631,
                3675190003, 3262456635, 3682680678, 3921732891, 1604135368, 3949662129, 3129246190,
                1954536251, 1538666313, 2274928374, 1650180856, 3652524428, 1821056613, 911456961,
                840333318, 764118487, 2652399697, 2580502346, 2438391701, 2513677855, 2640798083,
                2778108413, 3625988619, 3121963132, 2016358782, 2026634297, 2454141008, 3719526468,
                2069512197, 595695222, 1372551058, 2654832919, 1312895472, 3449972978, 1039886392,
                1219309116, 2158257939, 161673294, 1466523266,
            ],
        );
        test(
            &[10; 6],
            &[1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11],
            &[3, 4, 5, 6, 7, 0x80000000],
            false,
            &[3614, 4294966768, 4294967005, 19, 22, 10],
        );
        // - q_lo
        test(
            &[0; 338],
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                3221225472,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4160749568,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            true,
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x3fffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                3221225472,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x7ffffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
            ],
        );
        // - n_2 == d_1 && n_1 == d_0
        test(
            &[0; 780],
            &[
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xf0000000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0xffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4261412864,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4227858432,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                3,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                u32::MAX - 1,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294959104,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            false,
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                u32::MAX - 1,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x3fffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                2,
                0,
                0x10000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4292870144,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x7ffffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294967232,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                4293918719,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                511,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0x100000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xffffff00,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                16383,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4261412864,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                4294967167,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0xfffffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                128,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4160749568,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4227858432,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x7fff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0x4000000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294950912,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                3,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                8192,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                u32::MAX - 1,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
        );
        // - carry
        // - !limbs_slice_add_same_length_in_place_left(&mut ns_hi[..a], &ds[..a])
        // - q != 0
        test(
            &[10; 10],
            &[1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1],
            &[0, 0, 0, 1, 0, 0x80000000],
            false,
            &[0, 0, 0xfffffffc, u32::MAX, u32::MAX, 1, 10, 10, 10, 10],
        );
        // - limbs_slice_add_same_length_in_place_left(&mut ns_hi[..a], &ds[..a])
        test(
            &[10; 197],
            &[
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                32,
                0,
                0,
                0,
                0,
                0,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                4290772991,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                131071,
            ],
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294967264,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            false,
            &[
                0,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                131071,
                10,
            ],
        );
    }
    #[cfg(not(feature = "32_bit_limbs"))]
    {
        test(
            &[
                1976632771194112527,
                521304770361505402,
                6547459063177486272,
                16111887222588973398,
                11964828684918293857,
                9728747614510340318,
                4267206348672680192,
                10832398719275582210,
                14900216064213533822,
                6380760676681064593,
                2849541272450836814,
                18142033108417998751,
                5641128336349728868,
                2535172385342350806,
                15911904441417898191,
                9084408705773921949,
                15744864820799189887,
                4463837653601785361,
                2987490935632187606,
                11223238737559450353,
                5105094309856727876,
                1009868354752173181,
                13484596105692600309,
                238858926646454289,
                1059132889701728020,
                15279567309856499035,
                517773561239225477,
                13637839145813400305,
                7302869346647374900,
                1151876675162026587,
                9504208398199632979,
                8612638565635453711,
                2164322139068363749,
                15512475459854984520,
                6875534824714840095,
                16955948007577516702,
                6471458936191368235,
                10654427325707436522,
                15337657345435735346,
                15231031327469377800,
                16046593431914105744,
                8712679690677372756,
                15831996438112202870,
                14225387673292899684,
                17347546671830406364,
                7848557880774740728,
                15226813436071672038,
                14096480347669118219,
                16389437176481235757,
                7980211852778477130,
                16446026267287953606,
                7958418492797069881,
                2213403156236979508,
                8758841210279886543,
                5431207818821342199,
                10995901604046539947,
                18335471104239998694,
                13564698255420469178,
                5867947764049999258,
                1175284037301014819,
                12349712146059439945,
                6688979957099199180,
                15522195128263747693,
                11640494562411680106,
                15737920193718148078,
                3117055011815888766,
                13625786892636999921,
                12946533130762629275,
                2401319620356150536,
                15224229761057707791,
                9402116507951822726,
                17081785103310854839,
                18332791217613728290,
                400494137558561035,
                12279775278166901330,
                3814281798348375735,
                16512813422954004929,
                15443554440158194522,
                9668157500068914212,
                6053490056859889986,
                803053277491258716,
                13646927161616864743,
                12206129576276588469,
                8174515000911774339,
                3036848877312472711,
                11063940183681869708,
                6310270877191259165,
                9386179633193645060,
                9559132660384619200,
                8951064714492945869,
                14166616301229473652,
                13071629188826572782,
                11853790173693535149,
                209636673262143030,
                10441446179239151884,
                10420612973148232000,
                11908054025560012662,
                10909003530238644740,
                5352309958024461387,
                6231231241710103495,
                5817566191981122394,
                9966557507113270738,
                2526734084799877406,
                7119546382209824917,
                11004099504334912785,
                17101894178044846313,
                17623127824006159278,
                8521084772968581318,
                13811386822752437212,
                3432971939926452639,
                9600964573924347046,
                17748647615012039077,
                13689755186761681519,
                16574316982652951580,
                16782416719434018360,
                16144163416834277717,
                2756565693629327141,
                8059306003700661181,
                16267666526611427826,
                11909795646086424444,
                943806121593842977,
                13457826244859784108,
                7239747548483360443,
                16749825534693166168,
                14426083390114861929,
                17648467062688582919,
                15758353298227492958,
                8404753209561230712,
                17152753483008477565,
                1709121095849985814,
                9192526565401365977,
                5811560381076741655,
                7570776248350537723,
                10727029289681698856,
                10402933617834546815,
                8318965057148775598,
                13169231870647857793,
                12802647644833789786,
                3875248944583119534,
                12821268270102906356,
                6572470769294941231,
                647608152449604156,
                1871144685026941083,
                4032548946127357216,
                16395335442808140376,
                6736436429481567066,
                1079601411786263878,
                2504995255059825437,
                6763393428021343788,
                7715029394801749110,
                11780543684841254832,
                1383809532223095584,
                14972882189077591253,
                7800448071104479686,
                789117789337480564,
                14240769726663499540,
                11053104839957580124,
                14698906095810695206,
                11960947912669574992,
                12154539497540266947,
                14003595352222612163,
                9605483083077043861,
                1678324429116079639,
                18410093395270606595,
                17901794159659874459,
                4555659564807476615,
                5907892309603612114,
                6524803775567674587,
                341147093193233779,
                12996396640449564521,
                6720226112934991484,
                2388493664120597295,
                4290799651574648025,
                15852556922046266402,
                2246578851767626469,
                2920616190359978,
                1995918845898424965,
                10100917236583320084,
                7326568532476915731,
                15216155190727196226,
                12275242131304351295,
                5999272749153860391,
                16885847068579318746,
                2394902793719864009,
                5326879392776369400,
                15621103453348562573,
                316702208555392711,
                11540860887565649468,
                18366619672720535993,
                12809248756928885140,
                7645410389657248491,
                14015340112675676632,
                2512410450534100064,
                879480154877772022,
                11649289287186823201,
                3763895355294089425,
                6070847093240149275,
                1985095235894032224,
                6801453453040268360,
                2577579383099711197,
                402372101655687448,
                5644686065835595838,
                2514710250566535253,
                1911966914948814254,
                3687058531887005876,
                7434246748483727789,
                14122808689973017488,
                17766089471795036547,
                6666390546144059621,
                9594245591763828251,
                10442127131744522659,
                12010358916814662429,
                11623966019504805712,
                12883088750499797281,
                15648374317449757761,
                7443275395944491151,
                2162931565418953025,
                13144859518518203320,
                14371376174232659599,
                14565951115370759763,
                7113822803057100738,
                791629390390021149,
                17035680947686192637,
                16117534524053580299,
                16569219763811788776,
                12157015834931780538,
                10769346653748666149,
                1546686602678137333,
                13300226369383179632,
                6100259704952035034,
                3072644701407984585,
                4030070721794724661,
                4784297440480155258,
                8207888015979310860,
                7318324394866297710,
                1188255145714469504,
                7497067679833095894,
                831722367483141911,
                3342006775070786183,
                915228369324832271,
                12214752361799071858,
                7220077330608179732,
                13269901002361366384,
                13156705894018513856,
                14069485524218015587,
                16856376646412467543,
                5458014833622408772,
                16052033690985477131,
                3853276807297699899,
                4361733283329901315,
                17976611431135375864,
                7827211709974023187,
                2672179408451719014,
                13336920158316088067,
                13989361937721365460,
                17089134261482111299,
                717462569675369116,
                2677140142520744762,
                12527274888132316130,
                16020437139419307993,
                5109175831372656675,
                6967613605928103302,
                10707316610802880961,
                3751977410812504643,
                3338250566945759407,
                5483469392473217290,
                17852350745219193880,
                4273571006338555810,
                14355529799772896463,
                9097946801457959042,
                6170305947438052682,
                3686031910837362999,
                13659812758484650695,
                1078072280470978659,
                10977423862490226031,
                8329312879062820322,
                4304430845375483198,
                11670535966145436284,
                9843819566701342756,
                5423791105551635377,
                15484617895300773730,
                11737838788253883653,
                17352638972287011361,
                6417215765954491552,
                5210838541415799099,
                14459766465462519223,
                11293780554786569855,
                3913456812295604968,
                10038604499286389380,
                83285032545623676,
                532455486041684132,
                10447557457915823565,
                13091470210751298878,
                12472561867327718392,
                14986754996345557860,
                13353714167702784729,
                14933692721559793075,
                4392496650147061013,
                14233597247368097070,
                13776324136360206128,
                2519461559911558060,
                13447307564192855696,
                17414679919154962748,
            ],
            &[
                13564422861878112234,
                3275713182933072863,
                9034480898755624920,
                2219243711863369707,
                1414795403602489063,
                15894568837862895798,
                5479009852840249103,
                13548176561971981614,
                7335026045847382779,
                8888321385565889284,
                1374250446640957676,
                884800927779782031,
                8670348570372777926,
                13918154801540915789,
                10563048209614776992,
                13969791730415477716,
                3452835315790744420,
                7707311332988117775,
                4057795468009662220,
                503791220336896693,
                11739899866722757790,
                11508632709875839378,
                2046257727824251571,
                14147378955881967537,
                7621816326428068081,
                11992669611664635262,
                4065021093573908389,
                10481004349510473318,
                10475301724256756509,
                2524130821276975109,
                8593134939742261716,
                15623538309287845872,
                12317584420983773918,
                13199980610882090215,
                13443389318588629073,
                803739730193348834,
                17830865805659102381,
                11251151180776149719,
                5892904253365508433,
                17425282708969635181,
                7064424708573252994,
                11602656136970912939,
                9321057698061643115,
                11015344462881106465,
                10461212970348321643,
                14132875754655537304,
                15753277783360549383,
                12320964235023516733,
                15957290977412221921,
                15001559471941651986,
                14955046884332716974,
                10073992907091554325,
                4815480386394792639,
                2369717258966377146,
                9248685362922918367,
                11146342741382934626,
                10126995575484511675,
                10836902283955631137,
                18212311536097636051,
                4203660351079577016,
                7620879273947445884,
                3547199160053636344,
                322234022833908928,
                13570629391930072285,
                14555015649158595870,
                5365173841640051462,
                18270551664241964745,
                16860982171375384296,
                1073487768765270579,
                533079099712269200,
                18402112985722301945,
                5434269089519901126,
                13639392734635113987,
                8006521979690287885,
                1545872313913044683,
                15377554075787870730,
                751964771297808412,
                10798750359921054639,
                11281008050931496235,
                173570012084088260,
                3794273797618100167,
                7040606489685125833,
                2026261595126825272,
                4120208834527743861,
                539222748500084412,
                79750673604550443,
                17475108564705202581,
                7592139295506453445,
                5023893374592386852,
                2824914523935320076,
                3238624402000298552,
                14131756234617118206,
                8788539332133441603,
                12952024446070112298,
                10155295021066454828,
                3941570600498808512,
                6414922728964194522,
                4076629519262102829,
                7536766196851355087,
                6915125713337987783,
                4580280718549432890,
                10168080122054976212,
                15909510579919558189,
                14519226511270844173,
                2083762604858883168,
                13387991133148007718,
                16840836457014280191,
                13080359029893923590,
                11503510372644227919,
                7953484793188318118,
                9854115240624462348,
                5335772307682974862,
                4936183703810019894,
                15994627285462031439,
                10709370317657097612,
                16199196738320076993,
                14938272361213950876,
                15253370694640585418,
                10690065705918307042,
                9549292542903499984,
                8643187064435534254,
                11624928373665837384,
                8023398739766654926,
                17499066444790402375,
                9801044615559621440,
                14552811277630127749,
                13492161609355707245,
                9655752034551803175,
                1755722784425137866,
                11625636701478868329,
                3624227495361674626,
                16165268009822921837,
                8092810235043787525,
                5788572199295079813,
                6854648148487456329,
                14456461148456692547,
                14310121083997049160,
                7059162200334339773,
                14727298439147858639,
                2177197582567289597,
                6545559818086725189,
                10527371808810639250,
                4564586802867503950,
                2652020551780949632,
                6405684903154669518,
                1737731788352731098,
                3469893625756210544,
                15065635631347129418,
                13174028003011531358,
                15454741792218022113,
                14370595360570179099,
                18037934040444240682,
                2964154747013421979,
                13884952913031178416,
                597641522633545753,
                13985314547491780345,
                8360203218690496165,
                11794669667587913233,
                8996326275940748604,
                14117671113926503561,
                11301961759403099265,
                6700188071922378791,
                15385674866154542642,
                8468030983307924915,
                7377911706704156399,
                2604656541341801794,
                4418057625951919598,
                2110519074747192474,
                6101102874337817025,
                17886250954206311063,
                15712345996698736982,
                10487961467211158758,
                7074394084984481200,
                9751971849704663337,
                7679893214057488711,
                945225664530152672,
                1245168954077749403,
                9513611114031241898,
                10766452668861159380,
                10755131449239090448,
                16388278488885030616,
                2817843987791967489,
                11763453497935146995,
                1092574082244810895,
                1941447274495784860,
                13572786039056178918,
                15061182291913076311,
                6679943993103806154,
                15408371401907582553,
                177747470770242167,
                15593976009786423347,
                7461104255684909887,
                5044076910420087327,
                17630612657815416743,
                15051698319468276092,
                14951988637957301647,
                954336914910981662,
                10967373098940345847,
                2121680414874388243,
                6976201562796491998,
                6607602757564910564,
                2432806802785175299,
                11440086710224789398,
                2224788034891182999,
                5665602195014987730,
                3821334059850073126,
                8765713375749646572,
                933515038511209109,
                9433768612524189229,
                2467805208666126785,
                14871840927317419092,
                5154749137613554029,
                14265822875736809648,
                14610870159067374513,
                12698797236213924014,
                9569606658962541579,
                16162107306292390178,
                12212893918580781835,
                4614977216445016679,
                1352141803214945397,
                7426333917406258422,
                14520946858416385440,
                12446204685829260164,
                2067433179439046305,
                10861419565872522342,
                15042750574419434699,
                8969751642165063673,
                2375359711870752032,
                8547092847092151309,
                3254399133444460923,
                6176880426174028076,
                1267184736498266698,
                6443556206710696839,
                5582660421956028475,
                2003441447753754180,
                14408039201078176692,
                9878715208788868318,
                3156012596211387511,
                2621328682710618956,
                1343996423359983728,
                1659922365403643436,
                9104705095152231651,
                161254471946633022,
                17082420100696680164,
                12948458999893309534,
                169193814558716113,
                11181197557527273709,
                6214014612630290060,
                4852610222569086650,
                4759907655510719027,
                16416424951651935524,
                977804021685057099,
                11673776803881765160,
                4922236924628996318,
                17921361428612029549,
                9649330011870916482,
                620643376032400196,
                6391102220914894427,
                1331509376533438689,
                14618159860008486331,
                2457140428084980479,
                14239221334508169296,
                4552842402472365258,
                13656369860726864138,
                15616197149307562605,
                4979886663474976519,
                3721352528085365632,
                10207378704552161011,
                14450395034610232613,
                4394727065169978678,
                6225996635016717724,
                10125465703568602374,
                12627017260126529357,
                6034795013566274191,
                14507135436863373853,
                670092389362258666,
                16892179404337461285,
                17907525342655368510,
                6388033394429406697,
                14608221204689506984,
                9725811389591747512,
                15841826865041066838,
                12726453974670314753,
                5294708137059751589,
                12939898091011681833,
                17545561031992221346,
                12940997953067270813,
                8668645445261571392,
                17465976365740637310,
                3297459757903247617,
                5595819579374915626,
                4255310297204286682,
                13444083690317354200,
                13079687451729978985,
                17936788196509077492,
                4731803836376614325,
                8488258826257443557,
                15872298653522164970,
                6136303961972758860,
                8203587474223770207,
                6640840421159816678,
                5369017699494121011,
                13476518943592512132,
                18148180997156310554,
                16551428108902177352,
                10607422095455667650,
                18070526307796619350,
                592194588928348733,
                9500655832933342555,
                15906694991065813017,
                5672170756115546545,
                5164713544570924308,
                9654099190010511616,
                15467930191553249033,
                15324226386462646310,
                11654051669492811942,
                3845504882613220875,
                17995014637765041183,
                8451419834021139432,
                4831660661140974858,
                1270713005767677247,
                2854249945191704862,
                11336191586387214070,
                6230352080584748033,
                9904554377773225787,
                13646007744312337579,
                12876135209922183651,
                5590478105512035067,
                16683787122041782881,
                2912257455736462263,
                1161124723702057201,
                14599086761545813923,
                13759995150092530098,
                1187669969167762513,
                2025210147021211292,
                8404792573797774696,
                6043699170259360346,
                8567182735917474580,
                11673587056707375436,
                13729589792471955572,
                6040294236876235718,
                14118883879821257022,
                1472762515001555634,
                5772607514220590032,
                12005167700240721094,
                2562574981556592924,
                4322916215476956068,
                2931397736166450113,
                1981334131663903720,
                14008950785795439721,
                2258367698152396200,
                3193592957432138332,
                18147159661113326998,
                7636102688703700127,
                2455873680142820545,
                13327423214878219110,
                12145648468670490988,
                177980942656438245,
                7435164945754514167,
                14611156817208949574,
                9482801842425137904,
                17579649951803679316,
                4395194171697397116,
                2246099311908860608,
                2326078108126869227,
                11810587602203652970,
                16265527186229236462,
                4210697026328134418,
                15046104816787660106,
                5661711221409653164,
                13045157006959773220,
                6999069520359938726,
                5808420710659641279,
                15787347158818842535,
                2851830421850001378,
                5525485631561271262,
                711754543881977867,
                9906329062827899634,
                13595975648907914352,
                15095498981812294732,
                14827890138173702931,
                10114436220688049709,
                7728356353192609902,
                11254468346316497641,
                15153773712001763014,
                10207835043420703649,
                8801897675976972272,
                2550825610727592889,
                4806829881527541342,
            ],
            &[
                1408954216269482756,
                17495363275392015072,
                7569538419880562930,
                3159444535690624422,
                10539713616310108624,
                12287759471187878632,
                10762226695213450386,
                13702928698805429873,
                9308039037015276334,
                2693272575243867415,
                4495731759682297278,
                17957718702553480973,
                1695225019270301796,
                12538720478819195709,
                7355696648235750501,
                14609468359145504152,
                17713212205059712263,
                7067761757765748297,
                4455678910009476432,
                14940946772952633580,
                7353223946372247705,
                1966704175929577612,
                17788828899639927690,
                5873642145421053485,
                9432511011769375502,
                8011456734513040486,
                18383517818628757190,
                14804344175107599111,
                14453706668928035575,
                212174636863869640,
                18318201713407159667,
                15868508625313800988,
                794570724775389143,
                6890250104153732120,
                13215618552493662151,
                15355712080025289288,
                14989269405365678703,
                3440332781177776817,
                1547041711400466409,
                16949408083533983339,
                942645885575525869,
                3593963961417655151,
                4158508676658397508,
                13468348757333127520,
                11862936891535571513,
                4377454676311890507,
                13780065204467937543,
                15803361323352221769,
                953489208716892192,
                17364437917524369478,
                11384216985369504647,
                3360175615399279011,
                7215910295314039412,
                4449007827158713949,
                8690290001027040683,
                772293093052269703,
                6926666182768252467,
                7985255216129545232,
                6665284360882852586,
                15207741179893001845,
                15703767897436764321,
                7914253539478783255,
                5903013543557664007,
                4774214457440373599,
                14080179030683024826,
                9434035144822929636,
                5367813517486193633,
                3304878828952757511,
                10243535705902406930,
                1884627861659715947,
                4978189310017102374,
                1216698964122740709,
                15100941079818973803,
                1749141348527036567,
                8743320843818547513,
                9958778070878908244,
                7858295599866849905,
                15078222426471283656,
                7336868524011917281,
                10555852968409722407,
                6944256015177961916,
                17387864131687949624,
                632732732269768198,
                16900672962472231809,
                13484064257890193159,
                4044769021954392546,
                15294554445007953309,
                330214884931606053,
                8773227351193839544,
                9933428285510948540,
                2903959890557503867,
                6265719520493470019,
                14025203023893391008,
                11647283945890530059,
                13249598051183746227,
                2074546550971361923,
                12247825023511227150,
                2784768583222188866,
                6407637641382682182,
                9752576266849520115,
                15195951757710781232,
                13752087641373844065,
                15476511142100387305,
                14236723356162907578,
                10822337465577716429,
                12814664143156135314,
                10894509357772940167,
                12457195754516214135,
            ],
            false,
            &[
                13415846491005578619,
                13584292933077885365,
                441953663009890135,
                7210335717300495676,
                13057964354000775464,
                10292169585687982086,
                12251907425151490553,
                7386686741811236397,
                291437469999876541,
                1843506473757893895,
                1231339981456887992,
                7336311449497199296,
                7176633324357201366,
                658513238018005674,
                3117977571063681035,
                2261787922928337152,
                4069182574784670800,
                4786347950692670036,
                955338065659184695,
                17893231513983993127,
                15970449235194675852,
                18222079201306580439,
                7381385984359606163,
                13894071053001480008,
                17968185102194547794,
                9652567975385349040,
                4579378768094593374,
                2929413774035294909,
                4858585721157204024,
                432346869077563062,
                16632527599028610709,
                2969760407998231671,
                8723191162795424532,
                2205885763249224226,
                4090893610978417832,
                727335411538814681,
                692739901795118471,
                16185752384798441976,
                9407352218349207860,
                15954006133480937416,
                16002700437312273924,
                1803166211419435125,
                16572732079534009876,
                10103383220489568096,
                14055635057552795086,
                6204301430393608593,
                16304361060342210151,
                1077463458077785016,
                846584174947782921,
                14549143985217156839,
                4794451719446030954,
                15070589344871049989,
                11768766648218861349,
                193778540633917339,
                2069627918507475641,
                17004474480886564748,
                6387833454863576557,
                7403048767127663654,
                6361857152378524096,
                14590329534696373089,
                3264224706753004828,
                15794673518624161617,
                8471728032197957118,
                11123707683472607781,
                5926672570922373756,
                8353224757488864005,
                3920855214133136556,
                4088762124331852387,
                1351340558126331083,
                1064663312281497792,
                6722208927949900059,
                8191142535859454935,
                13932575231193375718,
                13250838834435891833,
                2482941775014508642,
                6361967846756283686,
                12230741962057003367,
                5322339673572617872,
                2461502783752535986,
                15456869581410217577,
                13516762109813035487,
                17722517240817407655,
                12006022535464060948,
                7693156444704391120,
                589139607677986600,
                10992862276542439592,
                15638442660969354281,
                6302046879465388504,
                18186347373545845422,
                17661768623582090594,
                11182202442217703360,
                10323032349274612161,
                351359225361309232,
                17208481651320700686,
                7762799900361196225,
                18130637050502143037,
                3527936976222881348,
                7787458878406549730,
                9785865244475232340,
                9313322536748638436,
                6628693715403354820,
                8155584772586242242,
                4298548228131846405,
                6216995875383402160,
                8745213713616642956,
                6205924554981340524,
                539046468165950413,
                9024163979699199629,
                7137709973749463062,
                4302338034003647906,
                13829701518024075554,
                3020599208667394924,
                2771484371923677669,
                1332418334924131169,
                15030111211658663300,
                8687405539746619117,
                3255425958083223853,
                17607580850311952887,
                6117457031153941598,
                9667238428907972490,
                16866165930381488058,
                9279876560625206616,
                7219817020774185353,
                15408311020361635593,
                10627051636551065581,
                15435960991076989754,
                2125427069228602466,
                14834800748267676770,
                8338000901565415696,
                9843107756445251924,
                11726258180166518722,
                13216133507413853832,
                9548542073000115784,
                4477904860207934342,
                11559138083775928810,
                14261941916295954455,
                2935454921278307407,
                14578760607502424030,
                1741904848044422540,
                10910867960823424013,
                16471597397244360629,
                5831225248121651892,
                12850983616913099920,
                4576475941872797279,
                10819576084838658923,
                9911449990461349025,
                8071318423647144806,
                12934895766920841548,
                1934332324237869529,
                1138130330895714883,
                7749002603757035655,
                6467404987016890639,
                11255316830064030538,
                3018235715458169811,
                15201809744582983527,
                16342498733210722843,
                7708152387679592258,
                9492172974580805134,
                5866422978520553541,
                16455719455662584469,
                13731761368830060542,
                18292973038079145164,
                9007521683809498606,
                11536176422926297746,
                17069970006150149865,
                2103718123709857344,
                9551485804718272266,
                12273976131107017479,
                1653874358479460972,
                10193109839436245173,
                17209319149808494347,
                2304536837571895652,
                6062765286861083817,
                4897491367356012054,
                10921490709194148600,
                7489492882796549046,
                15346457168511978884,
                15415284042926257534,
                3288025229571018975,
                5490937743897191968,
                12030244067941846502,
                6649465014690650122,
                13043829245604474765,
                16104656800587069133,
                15901113890052229676,
                3837090179127712297,
                2184801241947831767,
                14724333584011117924,
                18042748113599866974,
                15635242211549384225,
                11210521843889130992,
                9474616548314628408,
                7072588307383545068,
                7764739347402168434,
                6456269087620448997,
                13468842728337937173,
                10407803768686058347,
                58702129805853206,
                9751180010069185513,
                6064452714620219927,
                13392800693556694325,
                18372122025912817085,
                7516403896475345556,
                336160785595278081,
                16607869224650226242,
                4883984113455670464,
                7621647467206333443,
                7975633505870931502,
                10173870707804084376,
                15734058126402681870,
                13198172970791238175,
                6074102067907439304,
                7289841954562682843,
                16642477840387047386,
                17735102900144975086,
                4380312697910692464,
                17348779887199611248,
                3519767210720195345,
                3200318167890176110,
                8763164776273977859,
                6372373083461058525,
                7012561740996573104,
                16229141474971594905,
                1111767702608658668,
                10378536678592366267,
                9247600318566456250,
                17456358751309498336,
                8398075459962870518,
                398315841023187114,
                10277410686246662542,
                218781901157848211,
                618572494658903822,
                3804414506632334467,
                17482725095963661462,
                11945563631700693151,
                11377949134930807832,
                11388696917522764394,
                12397282866398662460,
                17984588793337249647,
                11852155346818929323,
                6282040579217154095,
                5865043035453987718,
                12490608990412245333,
                8662459463589993440,
                9703534627155947683,
                9083465368456889169,
                17385583284480619421,
                12619315429611258952,
                4780803560699965289,
                9760114025973133776,
                6710382011811335688,
                4756178837305619434,
                11338794508489062066,
                5806084365129173257,
                5858402760317423580,
                16972385428332971155,
                7980041598360573257,
                17539414382812231878,
                2807108495977968496,
                12347721638665260177,
                2657388420803290454,
                18291179742275985307,
                17290651979906283695,
                18248938193230918423,
                9700793153607985827,
                1581581530211316863,
                421641550279827055,
                3550734485990309734,
                8838675277867213091,
                2325841972611425748,
                12896051317905757266,
                2634241676839630629,
                9793375243415420023,
                4956535453535085870,
                1394001153426732084,
                12161423949049475376,
                3204414789776213360,
                3766999954663739543,
                2885444883803065223,
                17796581505208708505,
                7118003311319221436,
                11737838788253883653,
                17352638972287011361,
                6417215765954491552,
                5210838541415799099,
                14459766465462519223,
                11293780554786569855,
                3913456812295604968,
                10038604499286389380,
                83285032545623676,
                532455486041684132,
                10447557457915823565,
                13091470210751298878,
                12472561867327718392,
                14986754996345557860,
                13353714167702784729,
                14933692721559793075,
                4392496650147061013,
                14233597247368097070,
                13776324136360206128,
                2519461559911558060,
                13447307564192855696,
                17414679919154962748,
            ],
        );
        test(
            &[
                3218961335417250355,
                17081672491164411676,
                3707650621435880034,
                12226464081058805510,
                16651181251520783855,
                13640009159697172462,
                4751991872429123135,
                3148650123879180586,
                1277783667166791164,
                8130851378765417486,
                5973994279255982458,
                9850645253643520175,
                7138807895287208194,
                15375995502112115974,
                3398898578710810979,
                17572162751360544244,
                8819660083286717264,
                69135776031233389,
                9998360095371859020,
                1528928997031509791,
                12805160204156629031,
                8189615704171780683,
                2467675461409164019,
                782953727844991423,
                6355321416294201810,
                3548418573913501932,
                1385195069445654350,
                13512602943141297700,
                3524217759662463436,
                15981277851376785202,
                12068598413178807016,
                18233498041601520144,
            ],
            &[
                18165976559003511263,
                9223225080462702306,
                8750698670891994416,
                10120417502780222451,
                7032717053294267869,
                7737301656727013632,
                18225658593706260392,
                7369113716849300883,
                6897596754170972342,
                12718107066356504381,
                9547303833255868241,
                7363473626086637667,
                1289803166471838781,
                11172671997578955149,
                11576917227142009738,
                4808558430339823806,
                14087513123418218680,
                8335805388080937297,
                6817204158716710723,
                6461426422998966208,
                1365262330733891893,
                9676053381450021925,
                13152204186994838090,
                13956491013462127564,
                2322954839434490857,
                6310077930127744700,
                4095927283220333117,
                9197971078622697427,
                16090563028856198822,
                11899164972605215873,
                7553212707478476751,
                1757047503535568327,
                14107358592894281225,
            ],
            &[
                2828382444363234501,
                14019650302598222399,
                7149153771500111721,
                5808789510078375243,
                11746020933581287784,
                13963712573526024090,
                7948303882471336756,
                1363675852130084180,
                6084755510768204882,
                1350344795057047203,
                9774965847686534495,
                3240922473188564283,
                15530555198365657320,
                9503969704425173615,
            ],
            true,
            &[
                3405376264873912531,
                12732472809437054653,
                130818493222944537,
                1430180103580323943,
                11834290994594767473,
                16311544963273773940,
                14852866940109114958,
                11049173384187452405,
                17390023561952283067,
                8415736510262004193,
                7352370979091376738,
                10787900486113600253,
                15465246806979411530,
                12013049514342915482,
                15910934446693912634,
                13226375350164320552,
                12041814675416282230,
                17264487603347577556,
                8934954480948017545,
                1528928997031509791,
                12805160204156629031,
                8189615704171780683,
                2467675461409164019,
                782953727844991423,
                6355321416294201810,
                3548418573913501932,
                1385195069445654350,
                13512602943141297700,
                3524217759662463436,
                15981277851376785202,
                12068598413178807016,
                18233498041601520144,
            ],
        );
    }
}

#[test]
#[should_panic]
fn limbs_div_divide_and_conquer_approx_fail_1() {
    let ds = &[3, 4, 5, 6, 0x80000000];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_div_divide_and_conquer_approx(&mut [10; 4], &mut [1, 2, 3, 4, 5, 6, 7, 8, 9], ds, d_inv);
}

#[test]
#[should_panic]
fn limbs_div_divide_and_conquer_approx_fail_2() {
    let ds = &[3, 4, 5, 6, 7, 0x80000000];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_div_divide_and_conquer_approx(&mut [10; 4], &mut [1, 2, 3, 4, 5, 6], ds, d_inv);
}

#[test]
#[should_panic]
fn limbs_div_divide_and_conquer_approx_fail_3() {
    let ds = &[3, 4, 5, 6, 7, 0x80000000];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_div_divide_and_conquer_approx(&mut [10, 10], &mut [1, 2, 3, 4, 5, 6, 7, 8, 9], ds, d_inv);
}

#[test]
#[should_panic]
fn limbs_div_divide_and_conquer_approx_fail_4() {
    let ds = &[3, 4, 5, 6, 7, 8];
    let d_inv = limbs_two_limb_inverse_helper(ds[ds.len() - 1], ds[ds.len() - 2]);
    limbs_div_divide_and_conquer_approx(&mut [10; 4], &mut [1, 2, 3, 4, 5, 6, 7, 8, 9], ds, d_inv);
}

fn verify_limbs_div_approx_2(
    qs_in: &[Limb],
    ns: &[Limb],
    ds: &[Limb],
    q_highest: bool,
    qs_out: &[Limb],
) {
    let n = Natural::from_limbs_asc(ns);
    let d = Natural::from_limbs_asc(ds);
    let (expected_q, expected_r) = (&n).div_mod(&d);
    let base_q_len = ns.len() - ds.len();
    let mut qs = qs_out[..base_q_len].to_vec();
    if q_highest {
        qs.push(1);
    }
    let q = Natural::from_owned_limbs_asc(qs);
    let q_is_too_large = q != expected_q;
    if q_is_too_large {
        assert!(q > expected_q);
        assert!(q - &expected_q <= 4);
        assert_eq!(expected_q * &d + expected_r, n);
    } else {
        assert_eq!(q * &d + expected_r, n);
    }
    assert_eq!(&qs_in[base_q_len..], &qs_out[base_q_len..]);
}

#[test]
fn test_limbs_div_barrett_approx() {
    let test = |qs_in: &[Limb], ns: &[Limb], ds: &[Limb], q_highest, qs_out: &[Limb]| {
        let mut qs = qs_in.to_vec();
        let mut scratch = vec![0; limbs_div_barrett_approx_scratch_len(ns.len(), ds.len())];
        assert_eq!(
            limbs_div_barrett_approx(&mut qs, ns, ds, &mut scratch),
            q_highest
        );
        assert_eq!(qs, qs_out);
        verify_limbs_div_approx_2(qs_in, ns, ds, q_highest, &qs);
    };
    #[cfg(feature = "32_bit_limbs")]
    {
        // - q_len + 1 < d_len
        // - d_len_s == i_len
        // - !highest_q first time in limbs_div_barrett_approx_preinverted
        // - q_len == 0 first time in limbs_div_barrett_approx_preinverted
        test(&[10; 3], &[1, 2], &[0, 0x80000000], false, &[10, 10, 10]);
        // - q_len + 1 >= d_len
        // - d_len_s != i_len
        // - !limbs_add_limb_to_out(scratch_2, &ds[d_len_s - n..], 1)
        // - q_len != 0 first time in limbs_div_barrett_approx_preinverted
        // - i_len == chunk_len in limbs_div_barrett_approx_preinverted
        // - q_len == 0 second time in limbs_div_barrett_approx_preinverted
        // - !(limbs_slice_add_limb_in_place(qs, 3) || carry) in
        //   limbs_div_barrett_approx_preinverted
        test(&[10; 3], &[1, 2, 3], &[0, 0x80000000], false, &[8, 10, 10]);
        test(&[10; 3], &[1, 2, 3], &[3, 0x80000000], false, &[8, 10, 10]);
        // - q_len != 0 second time in limbs_div_barrett_approx_preinverted
        // - i_len < MUL_TO_MULMOD_BNM1_FOR_2NXN_THRESHOLD in limbs_div_barrett_approx_preinverted
        // - n != 0 in limbs_div_barrett_approx_preinverted
        // - limbs_cmp_same_length(rs, ds) == Less
        // - i_len != chunk_len in limbs_div_barrett_approx_preinverted
        test(
            &[10; 100],
            &[
                1940112928, 3379564708, 4199340399, 1881321211, 3929726949, 459862695, 1350630556,
                2137584308, 1243785894, 983397001, 140271427, 1266031183, 3559919295, 702977622,
                3883080853, 1830599918, 3030922777, 2650241556, 2807154392, 2218698435, 1904442293,
                2992923456, 1481791037, 4079300846, 2680084216, 656285791, 3396057405, 272065717,
                766725345, 1681255263, 1707458816, 1535744308, 3830462997, 1607479268, 3475252676,
                933054715, 4058518411, 1139150315, 1176656025, 2657461768, 3149905151, 2833828072,
                1407836276, 189062495, 3008111084, 3911286362, 3969377587, 520858887, 1921886086,
                2410034665, 1853865087, 1218625232, 3157059817, 1933332942, 1324005415, 2613648167,
                575584498, 1622716448, 2776523679, 1406325147, 1675354862, 559193409, 3342355233,
                2906475403, 692112491, 30493944, 2020871018, 3990740411, 1951624334, 522929143,
                339689612, 1674309350, 1834377980, 548522470, 1069818460, 3188037910, 3867849691,
                141774569, 2470564281, 821388456, 4277724930, 3388680598, 3502532015, 3274512650,
                734688334, 1114445519, 4090718315, 3618580673, 2579392987, 1832470019, 497126331,
                752679886, 3640533944, 1359185510, 2078361523, 766787155, 3847437498, 1508816472,
                3627438612, 1634541191, 742795696, 2960390854, 3550088312, 3226573591, 2822842868,
                3362189143, 3928024536, 2437449313, 2184757656, 3728018796, 3935919847, 948513362,
                2073970870, 135663041, 2352180020, 74067810, 2805347538, 2583013940, 2822521171,
                3940279282, 1038996723, 1890047439, 511776527, 3353284478, 2377633924, 1797661892,
                1357303377, 2035070964, 1462275246, 529813385, 240201619, 2189253653, 736920794,
            ],
            &[
                543800634, 2770754899, 3204213496, 689485059, 1826336873, 1076055628, 4272024858,
                559589670, 2749860031, 1191446963, 568428784, 3084221767, 2956512443, 3015109082,
                4142890122, 1553810981, 1616811544, 2640988818, 4239702102, 3215569246, 4002923809,
                2144877501, 1973532991, 4210632120, 245033969, 3777546533, 2341963108, 3264590407,
                2911498298, 3029909661, 3030243778, 3822448105, 3778161582, 4002513281, 405410583,
                2595147981, 1384025271, 2688968055, 1208777176, 1722256783, 1319267870, 3011578504,
                2249499744, 1245802205, 2315940873, 1685775875, 3654277152, 3113432869, 924331621,
                1024360203, 3411941166, 1488722938, 14894550, 2976848938, 1792408550, 1006468168,
                758547548, 2693569089, 268361906, 3490575936, 2248882501, 3137929560, 2509320863,
                2201587434, 926371577, 1243694325, 1112023631, 2791032478, 4081106682, 3051719441,
            ],
            false,
            &[
                2057668, 2838783492, 4112571934, 2985797927, 3502010766, 249276105, 2839050469,
                4137353790, 2017993048, 1573570544, 2879477808, 2277205117, 3638192330, 748457752,
                3491012018, 1271254053, 513425262, 1211823879, 867012819, 2598440433, 1263651369,
                2021347718, 4249300861, 2804939275, 2973559875, 4064802504, 1895088151, 3564064077,
                4027716297, 3914830075, 1990115176, 1854504344, 4070846044, 9890028, 3874191625,
                3177041731, 3239277327, 1987223315, 3008019786, 2383741649, 1213911302, 1443565130,
                976567504, 3330200204, 1895735680, 141240306, 4207639562, 4048459734, 2216965594,
                4199097617, 2357661853, 3838741723, 3334878567, 3391363083, 1245835070, 263088765,
                3602928895, 211195285, 1878503011, 3634753202, 1476824331, 2380710600, 1037136857,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
            ],
        );
        // - n == 0 in limbs_div_barrett_approx_preinverted
        // - r != 0 in limbs_div_barrett_approx_preinverted
        test(
            &[10; 28],
            &[
                1289672612, 59863601, 2751157317, 2012015018, 753744884, 727617566, 1322732201,
                3978203284, 2280484600, 3096956045, 3846897372, 4039916165, 3268087487, 2544740714,
                3583698031, 3555712249, 1472500374, 3434100279, 100408066, 1521129569, 3806609799,
                552198579, 2644816287, 1159339267, 3543498888, 171157498, 3510108197, 1568117536,
                1428232726, 2356104756, 64268959, 1999542893, 1166391479, 4277110967, 3494387915,
            ],
            &[2774518906, 1389562763, 2693574400, 665477625, 1306831105, 2326033317, 2701983394],
            true,
            &[
                1959935944, 2176722772, 2263227298, 52031251, 3734871887, 3327523565, 110124788,
                3702612244, 3495588882, 1155949119, 2274365757, 1700149681, 157693893, 1287343110,
                1404704510, 3815550465, 1225032011, 1088256607, 2172622843, 4270557118, 3435679666,
                4256455092, 679164912, 4183948729, 591513024, 2958428852, 959546540, 1259575285,
            ],
        );
        // - limbs_slice_add_limb_in_place in limbs_div_barrett_approx_helper
        test(
            &[10; 1],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                524287,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                2147483648,
            ],
            true,
            &[u32::MAX],
        );
        // - limbs_add_limb_to_out in limbs_div_barrett_approx_helper
        test(
            &[10; 119],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            false,
            &[
                3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0,
            ],
        );
    }
    #[cfg(not(feature = "32_bit_limbs"))]
    {
        // - highest_q first time in limbs_div_barrett_approx_preinverted
        // - i_len >= MUL_TO_MULMOD_BNM1_FOR_2NXN_THRESHOLD in limbs_div_barrett_approx_preinverted
        // - limbs_cmp_same_length(rs, ds) >= Equal
        test(
            &[10; 100],
            &[
                2117335661666050915,
                12541399304153348003,
                946298540148951811,
                3706040940807058638,
                8245328193712015182,
                2995778607291858486,
                9642363239685223935,
                6402096640921797670,
                11712805591752250715,
                3201293761729543527,
                2067868456322875388,
                14022668248311848844,
                4632052274282811479,
                14328717146754990094,
                14774171516395200100,
                3932090441073088420,
                257112211266950213,
                8641538702124596212,
                3125093651287853737,
                17086253003900284664,
                13301324934271401692,
                17351307810524827327,
                10929578147013387375,
                15271667824914109078,
                14749348389589883650,
                6533201755640185223,
                2371674840347488671,
                4979324240277110920,
                735115859885293605,
                6735013534832335382,
                10119392873034328735,
                8587971333550618807,
                18370051728122323147,
                9585925514203520684,
                10610241536420552110,
                5468885590163205010,
                9820231198117134549,
                11691363752858795073,
                17305814922233352619,
                859726499782635315,
                191600792977201905,
                6645113919639439987,
                10186139014514132915,
                13892914247169168407,
                6236711305574503596,
                11144904233279320413,
                974401940116314268,
                626033050094744466,
                2363222200699405972,
                5179961716326453662,
                1671413606631599949,
                16140149794420865699,
                4525104777442472653,
                15677954115433676863,
                2638708390995098468,
                11988738016208980482,
                9157465979869425297,
                13700136748402110667,
                12843998927271872117,
                8172056912358636191,
                11114503332573068878,
                12203499984806198697,
                18293693840507282811,
                2347317225764209541,
                7620549999680469515,
                266221208971070149,
                7404318409115354801,
                11821070343886468455,
                16743495774393393347,
                10903651574167871901,
                8120955791898229966,
                6256918112663531036,
                4642094091398991189,
                17666442779793022150,
                6979752362584161816,
                738088006020097171,
                119461506395194534,
                2707265934504496803,
                3596152365672913357,
                14919229675768492176,
                15711892957093044137,
                4388966441873938872,
                7245386709333796615,
                977953205631793823,
                5781210341467685855,
                5753014960288580146,
                15207218610262659062,
                4952531357988639834,
                15654095960314850064,
                16211757785947051034,
                1847823176222065635,
                13267820071825624266,
                7825351897308816877,
                2314230949902785215,
                13036201409691464885,
                6084371089556728596,
                4297442532874624612,
                532433185229873311,
                3442779195445707888,
                2185935771567302024,
                13911571822971487392,
                6709028122157209948,
                8941147419783910451,
                10057960523899590670,
                3902932452596575823,
                18330616827289950879,
                16003030155941114765,
                14460484829934394726,
                3301932590566986553,
                2576945510271437558,
                338533333586368208,
                17133311687376929054,
                15155994049404735760,
                9946561124819597870,
                983339636912873569,
                7933316580780536493,
                10915381430875882064,
                12068808550152064359,
                1144295877076351834,
                6170469241760056335,
                1371470265702819753,
                8726850576136926792,
                10647848489928733096,
                8768829048069168648,
                13771635456188648954,
                17554939941431219329,
                9462385985825033541,
                748159237152854524,
                14199895651244313572,
                9044210482484213648,
                3880401870711113518,
                1694971440240542063,
                13547801197479934494,
            ],
            &[
                6673567349177488920,
                11342960606651398230,
                13810764753596302625,
                9212178722694740287,
                18084527251132181489,
                16224438820695147876,
                14021309035811827770,
                13013362906859690434,
                16417289605410335662,
                17190663644106068759,
                11146075708059454647,
                11549029857422506456,
                7397036559618436638,
                12934631186266104928,
                5350679730075628553,
                7240352255165061152,
                13372092351570783845,
                4399593574620862254,
                6394016331529930198,
                12785468835634740198,
                4322747866783581276,
                11568791147039875250,
                14991909491573471557,
                13477304839860990623,
                9455746029240930041,
                5341626453207818831,
                11987393219164946170,
                4239560960510469879,
                1182652798618482486,
                3574433530016984198,
                14570885001650188449,
                16721218276523736662,
                17582595396154395868,
                10645279465349745085,
                8467600286036697431,
                1335601409485137284,
                5427053117344459552,
                3050086815044747081,
                16544140034308828175,
                16944072159927844572,
                1776035917115670079,
                6854263689321209087,
                1395326973393056556,
                13328497288435313800,
                9104953879994634550,
                13847967313738589938,
                2043265995513605223,
                13193995464379134227,
                5009710316527156329,
                14279264770257730496,
                10408308917163910844,
                5580572910800173671,
                3609059974654018173,
                8878539685629369010,
                15620596077589725028,
                3646029498286920979,
                16172850651808491619,
                15245408082837971190,
                4561308773951735888,
                11261441942590830857,
                14221474405402406790,
                2504490064796610713,
                17165895347857529984,
                12091565718300353127,
                14873026879861808494,
                7208276890744094311,
                13958391986410864884,
                8096376858901544433,
                18161016816080485352,
                9946585096298233506,
            ],
            true,
            &[
                7142549011416014880,
                14684691169135507473,
                1822120729679146161,
                11841933122038769821,
                4526355225639296177,
                8468787586279727777,
                15714230407431964682,
                18333537779008451159,
                16040493219692071303,
                2049960712985714117,
                18060699260844232006,
                11436218371188111964,
                18247749997104652519,
                10208028585116703246,
                6519062707001442120,
                8663593385407998358,
                9429327332580418592,
                11147294715178482267,
                12622325713357218079,
                9917628001464406797,
                8948638694752077597,
                413202514200140930,
                13469381266876650437,
                10995920589784133757,
                10156568832733545237,
                7515628739788185693,
                3313803886141758450,
                16133860033309869788,
                13595249790827433309,
                1761826376674293671,
                9950121844277925507,
                7711011638364466454,
                12239101884168908635,
                9962222649144848616,
                9495651849838993161,
                10726943428780681871,
                3103236698633674082,
                18053635426183322245,
                14277576247420605289,
                17591792639916063380,
                11239385027227950840,
                4103032036605369386,
                266316975199678275,
                4262765348359631356,
                5507126713842059802,
                8339639832375105582,
                9642119792134182293,
                485127249403759007,
                3452585660397695342,
                17804977833202015222,
                1531864037104263565,
                12597964144177527097,
                2015301325553239081,
                1705158706504614015,
                3632321048208706170,
                2599148812642503905,
                18033201044296677048,
                10122148881615284352,
                12336433997173007311,
                8939795914080226824,
                9195323771605999386,
                1966840646595138765,
                6678745632744258779,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
                10,
            ],
        );
    }
}

#[test]
#[should_panic]
fn limbs_div_barrett_approx_fail_1() {
    let ns = &[1, 2];
    let ds = &[0x80000000];
    let mut scratch = vec![0; limbs_div_barrett_approx_scratch_len(ns.len(), ds.len())];
    limbs_div_barrett_approx(&mut [10, 10, 10], ns, ds, &mut scratch);
}

#[test]
#[should_panic]
fn limbs_div_barrett_approx_fail_2() {
    let ns = &[1];
    let ds = &[0, 0x80000000];
    let mut scratch = vec![0; limbs_div_barrett_approx_scratch_len(ns.len(), ds.len())];
    limbs_div_barrett_approx(&mut [10, 10, 10], ns, ds, &mut scratch);
}

#[test]
#[should_panic]
fn limbs_div_barrett_approx_fail_3() {
    let ns = &[1, 2];
    let ds = &[0, 1];
    let mut scratch = vec![0; limbs_div_barrett_approx_scratch_len(ns.len(), ds.len())];
    limbs_div_barrett_approx(&mut [10, 10, 10], ns, ds, &mut scratch);
}

#[test]
#[should_panic]
fn limbs_div_barrett_approx_fail_4() {
    let ns = &[1, 2, 3, 4];
    let ds = &[0, 0x80000000];
    let mut scratch = vec![0; limbs_div_barrett_approx_scratch_len(ns.len(), ds.len())];
    limbs_div_barrett_approx(&mut [10], ns, ds, &mut scratch);
}

fn verify_limbs_div_2(qs_in: &[Limb], ns: &[Limb], ds: &[Limb], qs_out: &[Limb]) {
    let n = Natural::from_limbs_asc(ns);
    let d = Natural::from_limbs_asc(ds);
    let expected_q = &n / &d;
    let base_q_len = ns.len() - ds.len() + 1;
    let q = Natural::from_limbs_asc(&qs_out[..base_q_len]);
    assert_eq!(q, expected_q);
    assert_eq!(&qs_in[base_q_len..], &qs_out[base_q_len..]);
    let r = n - q * &d;
    assert!(r < d);
}

#[test]
fn test_limbs_div() {
    let test = |qs_in: &[Limb], ns: &[Limb], ds: &[Limb], qs_out: &[Limb]| {
        let mut qs = qs_in.to_vec();
        limbs_div_to_out_ref_ref(&mut qs, ns, ds);
        assert_eq!(qs, qs_out);
        verify_limbs_div_2(qs_in, ns, ds, qs_out);

        let mut qs = qs_in.to_vec();
        let mut ns_cloned = ns.to_vec();
        let mut ds_cloned = ds.to_vec();
        limbs_div_to_out(&mut qs, &mut ns_cloned, &mut ds_cloned);
        assert_eq!(qs, qs_out);

        let mut qs = qs_in.to_vec();
        let mut ns_cloned = ns.to_vec();
        limbs_div_to_out_val_ref(&mut qs, &mut ns_cloned, ds);
        assert_eq!(qs, qs_out);

        let mut qs = qs_in.to_vec();
        let mut ds_cloned = ds.to_vec();
        limbs_div_to_out_ref_val(&mut qs, ns, &mut ds_cloned);
        assert_eq!(qs, qs_out);

        let qs = limbs_div(ns, ds);
        let qs: &[Limb] = &qs;
        assert_eq!(&qs_out[..qs.len()], qs);
    };
    #[cfg(feature = "32_bit_limbs")]
    {
        // - q_len + FUDGE >= d_len
        // - bits != 0 in limbs_div_to_out_unbalanced
        // - bits != 0 and two-limb division in limbs_div_to_out_unbalanced
        // - carry == 0 in limbs_div_to_out_unbalanced
        test(&[10; 4], &[1, 2], &[3, 4], &[0, 10, 10, 10]);
        test(
            &[10; 4],
            &[1, 2, 3, 4],
            &[5, 6],
            &[2624702236, 2863311530, 0, 10],
        );
        // - bits == 0 in limbs_div_to_out_unbalanced
        // - bits == 0 and schoolbook division in limbs_div_to_out_unbalanced
        test(
            &[10; 256],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x1ffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                3221225472,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294836224,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                1023,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xfffffffc,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x1ffffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294836224,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                255,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x7fffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294934528,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                63,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                3221225472,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x1fffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                16,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0x80000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0x20000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
            ],
        );
        // - bits != 0 and schoolbook division in limbs_div_to_out_unbalanced
        test(&[10; 3], &[0; 3], &[0, 0, 1], &[0, 10, 10]);
        // - carry != 0 and !highest_q in limbs_div_to_out_unbalanced
        test(&[10; 3], &[0, 0, 2], &[0, 1], &[0, 2, 10]);
        // - q_len + FUDGE < d_len
        // - bits != 0 in limbs_div_to_out_balanced
        // - bits != 0 and schoolbook division in limbs_div_to_out_balanced
        // - carry == 0 in limbs_div_to_out_balanced
        // - *scratch_2_head > 4 in limbs_div_to_out_balanced
        test(
            &[10; 3],
            &[
                2004133373, 2208332663, 2302126873, 2101318142, 1556208712, 845422188, 1055855393,
                3791075321, 2255139682, 1837642188, 3011157525, 2408855870, 3298400249, 965057055,
                1159107131, 881492493,
            ],
            &[
                1319347431, 2633328929, 1132813012, 2337676551, 3505724291, 3258373855, 2627638070,
                4109307798, 2791039276, 3812830525, 922865115, 1132868227, 50383791, 3208398854,
                737302397,
            ],
            &[839942670, 1, 10],
        );
        // - bits != 0 and two-limb division in limbs_div_to_out_balanced
        // - carry != 0 and !highest_q in limbs_div_to_out_balanced
        test(
            &[10; 3],
            &[
                2004133373, 2208332663, 2302126873, 2101318142, 1556208712, 845422188, 1055855393,
                3791075321, 2255139682, 1837642188, 3011157525, 2408855870, 3298400249, 965057055,
                1159107131,
            ],
            &[
                1319347431, 2633328929, 1132813012, 2337676551, 3505724291, 3258373855, 2627638070,
                4109307798, 2791039276, 3812830525, 922865115, 1132868227, 50383791, 3208398854,
                737302397,
            ],
            &[1, 10, 10],
        );
        // - bits == 0 and two-limb division in limbs_div_to_out_unbalanced
        test(
            &[10; 233],
            &[
                1120072065, 20030636, 2023698558, 3410014045, 1693005413, 3940821177, 4061897489,
                3904598150, 1239781075, 2248531977, 1084346740, 3406083359, 715154118, 3499417710,
                3315749667, 3834314996, 3974469907, 2814389907, 3511455607, 3248220325, 909379929,
                1899385878, 877989498, 598994477, 3240673535, 2330636314, 306792275, 1197844499,
                516880032, 2699904331, 945876341, 4162862896, 2209400052, 1190718041, 1558127761,
                743003038, 846606641, 167779957, 442248369, 4254081958, 378292991, 2269632031,
                2106325614, 2903467429, 2325283078, 3655634793, 2243956483, 48899184, 1510394527,
                1220327074, 1986122738, 2771000282, 3801992586, 3807619027, 471210965, 240054629,
                4138891889, 507966513, 2608584883, 3695585074, 1142049536, 1815447345, 878847029,
                2425512616, 3404280273, 1809800602, 2708940436, 2746635100, 4246404208, 2039835981,
                4031917112, 1996943450, 2580411493, 1129753117, 1562335482, 3454161890, 1304546076,
                266633008, 2243463006, 1495416944, 3194149979, 1187229392, 2598096522, 2717656416,
                1723138812, 321773293, 3528634550, 4012356064, 361342150, 1407382791, 281788878,
                387885890, 2614365524, 3674715918, 3735628736, 4055198582, 3441555171, 2873512886,
                881773623, 2426850436, 308355156, 4092323473, 976626532, 3349954092, 3198525613,
                1279820517, 1618078850, 1918325121, 535344811, 988051597, 837895483, 3080823321,
                2284227193, 3889421868, 4104768675, 3163480219, 3024832314, 2949487671, 3104325999,
                4157724952, 4168869313, 721690106, 3562272836, 4007453971, 550776419, 1766890521,
                3691651042, 2002769424, 3017703356, 622145868, 984506967, 872390633, 1401216125,
                2631526378, 3112269305, 432369818, 193905809, 1155233014, 752554312, 145186046,
                92244317, 3561426112, 1231358008, 454785216, 3928879637, 3214356742, 1926040456,
                2162944696, 54812965, 2689940083, 3697347877, 2642017440, 1597744969, 861917368,
                3358318171, 1108217254, 1094946939, 4142346894, 3219881738, 3673252603, 3653156321,
                1364144882, 942313725, 1035715400, 2330944680, 1224099910, 3349983481, 356524863,
                3275906855, 3285206514, 4248893603, 4088970118, 89006304, 23030721, 3809886990,
                2867877807, 2930760156, 2433798057, 3852621687, 3592557226, 3512908344, 1565719632,
                2354066426, 478797990, 1152699515, 318340209, 760174225, 1631571749, 2271463377,
                650468891, 507504586, 2693229321, 3120669668, 2544094891, 3144973242, 3636388382,
                1159049189, 1889357596, 1384924984, 43707224, 3718061345, 2543662513, 3729944648,
                2644670969, 1377639119, 168824066, 2584279623, 4277720720, 1318206723, 1379145863,
                3963963461, 4034653496, 3968013334, 2828714546, 1851135170, 1939714189, 3010931551,
                226867970, 3933697914, 3421927438, 966963996, 1035716399, 790589571, 1188590505,
                555295916, 3048626203, 3577474856, 155510811, 3536545976, 239763597, 2799111567,
                3792587555, 4190649328, 3101199245,
            ],
            &[3290181794, 4048330047],
            &[
                3781336300, 393098052, 601926039, 4290235837, 2822961350, 3264010853, 1395348986,
                1532724636, 1479458209, 2803797702, 2248954690, 1109326360, 868524296, 1622203818,
                2233229102, 3747990662, 555588899, 2749663474, 3197509055, 630575671, 4105739578,
                350883605, 3884847782, 2399406878, 1709869907, 4200326701, 4079556706, 1296998956,
                1096350538, 524761973, 1629603733, 506636222, 2285887205, 2231667772, 4010753639,
                917205307, 45978573, 2604387874, 3703883391, 956577030, 301635887, 1366863825,
                653575304, 4002452771, 1532839774, 365525468, 1207266883, 46605181, 3549725346,
                1181348714, 1401721152, 3079539285, 904310818, 1291426703, 3423900075, 3724908664,
                918900181, 492461520, 2051740414, 861850244, 1441070167, 2409123056, 1923594929,
                2592176152, 762526381, 2608436860, 179967203, 947743965, 3152260497, 3795355036,
                3128372461, 334341652, 2852670346, 525231164, 2707537479, 423546146, 422356576,
                2514936736, 1508996402, 2866854016, 2529118278, 1616687498, 3248910372, 3424501529,
                3509076680, 4206763914, 2036548848, 4062752651, 4244007332, 2701189385, 1278945872,
                1938315563, 341487762, 2245251522, 2614982710, 3077916256, 291100198, 3611957075,
                1995518592, 154652263, 1532772306, 248498549, 3231944766, 628801303, 3344272527,
                1065639153, 1987158587, 1479356917, 2988956252, 4046958565, 2209065419, 477730056,
                2336652721, 1037902218, 2563735377, 4101524640, 3496286466, 893815781, 370165241,
                111642164, 1658058879, 322898205, 2985501252, 4080369932, 313430011, 2711111000,
                1181601656, 555256727, 554722327, 2969317428, 2537636036, 969866577, 3100268542,
                861181567, 2766358890, 3556106295, 2319452091, 2276583134, 3694687721, 2683079950,
                3602799233, 3315957548, 1156121233, 2466768002, 351756246, 1311063094, 607227992,
                2577239074, 3348786927, 3556673019, 1603335686, 2933024191, 2025259240, 1495478557,
                2032170560, 2121476735, 3193544299, 4119114109, 2928661241, 448576207, 1614958740,
                345202029, 2600582520, 2742644459, 92765692, 1372450470, 3781561695, 1568934205,
                145316051, 4006568817, 3666572039, 3335574604, 4261359692, 281814983, 2392237710,
                1491546468, 2408177492, 3815015525, 1277308779, 3604474102, 796103227, 3177970999,
                2372049172, 1292258380, 2260556904, 4006251044, 3037386650, 3097581743, 272132478,
                1428294116, 1610143768, 3421139539, 1124007750, 2568811011, 899340979, 2888377028,
                148246615, 621225076, 1635442139, 179184538, 2565113048, 524982498, 1477029769,
                2443496661, 4238972150, 2111945492, 1826948655, 1631527737, 3293499896, 1592525336,
                1072822530, 398987024, 1403041752, 1154120498, 183009602, 1860553280, 3833836549,
                639383491, 2747751428, 1967745750, 985697863, 4102756978, 3010429932, 2426159954,
                1632866751, 2325250581, 1085491035, 2626066600, 1004650627, 734034293, 2634237232,
                3290134248, 0,
            ],
        );
        // - bits == 0 in limbs_div_to_out_balanced
        // - bits == 0 and two-limb division in limbs_div_to_out_balanced
        test(
            &[10; 3],
            &[
                2974709165, 1408789605, 3865174329, 1359597196, 2297806425, 1241510613, 3227134452,
                325500099, 3798589956, 748039492,
            ],
            &[
                322054190, 2721366349, 534065582, 4284544993, 2403511175, 3934371253, 3610661048,
                2543303641, 4119337658, 3350506258,
            ],
            &[0, 10, 10],
        );
        // - bits == 0 and schoolbook division in limbs_div_to_out_balanced
        test(
            &[10; 5],
            &[
                2333679788, 2356666855, 2364697986, 2341519831, 2855670118, 1220356586, 3077887875,
                2167847389, 1298352907, 2783063425, 343561812, 1466674656, 2828723164, 2244865405,
                3261584816, 323262334, 2146252780, 2244363245, 1140774914,
            ],
            &[
                1638368212, 4269779221, 4096262691, 3276079354, 3835296926, 1228438544, 931207076,
                3941619719, 577971296, 3365897754, 3952671229, 3635074066, 1819466191, 1077726049,
                2724331936,
            ],
            &[985368501, 2737353193, 3370693165, 1798455938, 0],
        );
        // - *scratch_2_head <= 4 in limbs_div_to_out_balanced
        // - !(r_len > n_len || limbs_cmp_same_length(ns, &rs[..n_len]) == Less)
        test(
            &[10; 3],
            &[
                0,
                0,
                4294443008,
                7,
                4286578688,
                u32::MAX,
                u32::MAX,
                4294443071,
                u32::MAX,
                u32::MAX,
                0xffffff,
                0,
                4294950912,
            ],
            &[
                0,
                4294967264,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                16383,
                0,
                0,
                0,
                4294967232,
                u32::MAX,
                u32::MAX,
            ],
            &[4294950912, 0, 10],
        );
        // - bits == 0 and divide-and-conquer division in limbs_div_to_out_balanced
        test(
            &[10; 265],
            &[
                506643067, 674201220, 4077605637, 870747493, 2507358273, 3724638469, 1035009974,
                829046988, 3750301412, 3918208469, 3953784269, 1207127792, 4050829853, 461784483,
                2383763199, 2285448897, 893186049, 2814159946, 583430210, 349181410, 3312529089,
                3040593764, 3505574840, 2245473386, 499733927, 1870266859, 306858541, 2287239467,
                3602186482, 2485354048, 3925520601, 1490580163, 3946071957, 831571085, 2671733075,
                3492322751, 1693084970, 3703683238, 914345023, 1731822012, 971261566, 2899991322,
                1415427842, 2792111536, 3754030226, 2536140100, 2809833781, 255220419, 1494036353,
                1435449211, 3699505245, 3939949425, 3424663004, 1987816971, 872486514, 2304200621,
                3672706956, 3098142146, 4028147917, 3126086923, 3431963845, 2505585471, 2599393784,
                2703989391, 104322041, 31236914, 1102844188, 1685015561, 3414504334, 3166439653,
                3856439354, 3774684706, 4074904284, 605442011, 812212716, 1996651776, 4084653341,
                271581859, 2458893946, 1704814216, 3983060909, 1705321390, 2722920787, 1026280690,
                3548307184, 2796854160, 2478471453, 2069980586, 2327202328, 3179332026, 2188958336,
                2717879675, 130062885, 140536268, 2499125438, 3163111280, 4259661702, 2176278885,
                422519228, 2482586299, 2904549185, 656169575, 2052350629, 1346745024, 2132509288,
                3672720658, 1036389958, 1864007789, 4247227128, 3920036168, 1436562554, 4261984498,
                3509215437, 583752676, 3145348403, 2267709494, 2846186667, 95392897, 3743233716,
                2210401890, 333864866, 4114644153, 3030283850, 2885600773, 209380485, 753945396,
                719327396, 1293498320, 881901364, 2799735404, 3880748109, 2227099476, 2045911493,
                279042015, 1825819541, 1783146691, 2256898093, 2186071881, 3753747971, 3588523011,
                33603847, 1513157507, 3499821617, 3738644542, 1415336880, 2770362204, 2060947257,
                2730479597, 2685626774, 999367629, 3913677028, 1269070210, 71600754, 1301019431,
                2216016590, 146670051, 2663607608, 3377525204, 3629426592, 3008812841, 3146776799,
                2307984538, 3891637717, 2113643890, 191705646, 2335382191, 1140110911, 4020097332,
                1669672424, 2764048323, 131913018, 3294340837, 1179938248, 3518525865, 2275430195,
                3163615882, 1433056069, 3632771998, 945256067, 3112805135, 3793337, 2498643598,
                196475979, 1234934305, 2303169144, 3075871960, 1459180144, 181561979, 3503884769,
                2684065515, 830368038, 3013649837, 2204747214, 1403496840, 3859560760, 2859158714,
                3625515328, 1724787311, 751869756, 1068392634, 3152945440, 166456628, 3912875927,
                1276199042, 3180668929, 2654240450, 3838496263, 1978050764, 1330937637, 3052787309,
                2179574058, 3771594804, 78717414, 2499543381, 2290126169, 3421184223, 3433039199,
                3357845254, 2982215557, 486514084, 3948984473, 748257374, 980199615, 2114791508,
                2263110875, 1438244139, 364374413, 1396459296, 3946812646, 3259529007, 1803026660,
                4176094107, 3107042296, 3690348839, 3454595397, 2121594656, 576668497, 807899954,
                1335323177, 4200325364, 2730879357, 3284030486, 3324267006, 27217584, 3455039500,
                2034759734, 748477743, 1248615399, 1109976687, 476936469, 3606689372, 3993977007,
                2258238141, 1642523191, 2086149347, 3137863504, 3174920716, 199510880, 1099331674,
                704556250, 3389523009, 6232858, 4190352603, 2138785122, 3218278977, 1935766981,
                1255344917, 1933434103, 4229596014, 3581553119, 2147449432, 208926434, 2037430803,
                4143975728, 2356343321, 937192435, 1637432038, 661638621, 1801480924, 3779152128,
                4243491821, 1667774376, 1715755489, 3661813139, 1605971891, 4030695606, 2961165054,
                1368430397, 2222904896, 2817587025, 1714442303, 3822714979, 300305701, 1874484285,
                2601340412, 2275789197, 2695461089, 2246464394, 1119579754, 1646098622, 3280004748,
                33497272, 1940830933, 3780770129, 1587254032, 832573251, 1504418072, 4247592896,
                317874907, 949850421, 2252881736, 3574316069, 3062236166, 1396410954, 3249498785,
                3495392204, 540855070, 1908700137, 1469179505, 4199276220, 953657385, 3056452157,
                2141569526, 2342475731, 3746376146, 3271677606, 2770490239, 2212992129, 1758619376,
                1446549455, 409094501, 767129031, 3284625381, 1887741449, 1134874072, 2988924415,
                1641550007, 856704035, 80648349, 1467185629, 2753807208, 1609415681, 4087676277,
                3276525355, 1530490532, 3475014952, 1971819359, 2190766950, 2667577576, 2404497182,
                4128259693, 2449514447, 4199089872, 2205116036, 4089987616, 457231895, 2931469481,
                3147651033, 2352907189, 876540006, 3743945364, 1472931415, 1553806347, 917186799,
                2698606847, 66937650, 3645382047, 853223241, 3225678178, 3306353588, 4116060170,
                2590602281, 1017313973, 2613704632, 2155925964, 971128116, 2969595931, 2019050340,
                3266635763, 2229237075, 2897700432, 692684809, 2884804723, 548672200, 4148599125,
                2903660719, 3024254547, 3711620919, 3505267307, 3040630725, 2741747953, 1929597653,
                2289604848, 1876399821, 416443208, 113684194, 2458701351, 1129214912, 2220498900,
                2434385125, 383696859, 1361474230, 1163397052, 2669432319, 3058892712, 180222756,
                651218593, 799737686, 149785061, 2171639569, 3195389488, 270082840, 119827887,
                1410384095, 4066783084, 3263839471, 464165531, 3866096625, 2064936052, 221981930,
                3625198339, 3424885311, 1071723748, 409527338, 1676479257, 3093122613, 3729266155,
                1862387444, 1361519384, 1243604327, 1643472480, 611403222, 996523967, 4174256483,
                2326506033, 2418590323, 3423410866, 1529787343, 4205549227, 3997545970, 2223326432,
                380968195, 3506194936, 3910452712, 4038499509, 1845505874, 1017156422, 1929655910,
                2711754908, 635288831, 2247734769, 3304323666, 2179819642, 206674737, 590233515,
                1336492628, 3718075, 3613296921, 3550161900, 1864016712, 114106506, 2398669907,
                2720528985, 3857377519, 3905792607, 3346298226, 462294071, 3119274864, 3972744498,
                2110531614, 2149059199, 3948853307, 864424113, 2523103018, 405972168, 1017288780,
                2875520328, 2155632746, 3241100405, 86049349, 2941679162, 3047748962, 2284186344,
                3132866461, 2331447040, 1003213663, 1873981685, 3371337621, 3796896013, 4144448610,
                2569252563, 2859304641, 1027973602, 3158196152, 4058699545, 2002924383, 3295505824,
                695758308, 544681384, 3452307839, 1190734708, 4232023153, 451772934, 673919865,
                2022672425, 3493426012, 1142609332, 477542383, 1304798841, 461115870, 3268103575,
                2243523508, 606810814, 4235312469, 1885993181, 114475077, 757688489, 1965769398,
                260629125, 2265559181, 2568323569, 4202738507, 422918034, 1258453131, 3552221985,
                1666914845, 4063631552, 1893061685, 1362616670, 3828572660, 3003680479, 119501228,
                2101943449, 1119123129, 2512417484, 4143796978, 4123615401, 867855104, 9080683,
                2989272444, 4071914200, 3126419388, 897981380, 1843424411, 2958654429, 19279568,
                3561481778, 3620717321, 1713789028, 3466124758, 2114531476, 3750356418, 99077566,
                1592097168, 3827583318, 620525513, 290829475, 2167034573, 859199755, 859025849,
                1856340069, 1742571592, 619057918, 448873046, 492980624, 4024378211, 2920022072,
                1315190691, 220327298, 822905603, 862879897, 2816210126, 1919765070, 2058864462,
                697509061, 1633189430, 2245037639, 1891169791, 1490604281, 2749203494, 2727100064,
                3952743771, 745359287, 294665575, 481463143, 2908504060, 3264906873, 4291223436,
                1078483712, 1974128727, 3336164436, 56705038, 3439879863, 2641239652, 915511148,
            ],
            &[
                275357812, 1380366134, 832237659, 2112384134, 3392548533, 2322411681, 3893212013,
                4294409814, 4093767003, 825461980, 2478547269, 3404630461, 1971516824, 3442909527,
                310968935, 3588187524, 1263584270, 1954425632, 710153676, 1490566985, 3851982772,
                3813403663, 3945099227, 487964380, 413515585, 3154361919, 1595882626, 2396609791,
                324874877, 1386033964, 3103282602, 2599529608, 2119912272, 1365578038, 3224231142,
                4103857906, 475734936, 3828952167, 3071966456, 1450111251, 1166414077, 2218130537,
                3324650407, 1559641024, 2423373264, 2283136700, 1299328382, 3345578599, 840299756,
                3797238397, 2067196109, 3055917746, 3636953439, 9994084, 848907394, 3724334355,
                3765535226, 4202522723, 3549598176, 2976719094, 1062012457, 3192129616, 2622008775,
                4140808457, 3311195039, 2219962246, 583122033, 3484577945, 3996746462, 3163823232,
                2815287028, 4187316839, 3462896421, 2231360878, 1678307748, 460684903, 3249941390,
                2444083444, 1885084635, 1384447473, 4228441234, 2992602088, 4233710708, 3252082214,
                3754932745, 974410439, 749693996, 4239737998, 881588329, 1689075190, 292144262,
                1373806909, 960467260, 3106551077, 1249264314, 1210555633, 3882913567, 59286933,
                1593160363, 1969806577, 33237994, 1192096261, 1570929611, 1291838216, 1453595934,
                1481420023, 1722346701, 3005255028, 2671588519, 1129223015, 1988074467, 2499928090,
                3504414178, 2670506578, 798157759, 3859749353, 3404813676, 3070374972, 1362004270,
                388389672, 3227842930, 3141750652, 246994433, 3435384009, 2937163806, 1091298751,
                3219724097, 1663604618, 1547894794, 2034927931, 2077403472, 1183979515, 1312517327,
                1652629779, 2840629951, 1320559340, 3379399676, 1763411889, 2852175138, 1013263636,
                910550154, 529178328, 1437740880, 1545534222, 2859835603, 2271712768, 1100624932,
                108752761, 3521837118, 2043489286, 2482010929, 2526022944, 784503235, 3989653576,
                514018710, 3703716272, 4124099812, 1390681784, 2537781805, 201204889, 971348667,
                2307205686, 1760983313, 2268919139, 3109728863, 1931931512, 1811265792, 2492179233,
                2839289978, 305552922, 2705812278, 487378575, 1148523689, 4211680716, 3055415169,
                512649009, 1499343281, 2379775206, 3398729819, 2885867944, 1989458261, 3168583752,
                3895436961, 435919341, 3229009848, 333401020, 2066985073, 1599311722, 2470446433,
                2036229414, 3754646122, 1602078777, 349281148, 1872253564, 1044135153, 3952751023,
                1908969406, 2526035592, 108266294, 1002947027, 3490519747, 1017767773, 1363582374,
                4236936817, 1095422379, 1476035104, 3938596998, 1896398408, 4013241860, 284937761,
                5850220, 2334057891, 2272377339, 3385837178, 2094910603, 172965460, 2993236410,
                1631768055, 1408498310, 4196305218, 2858075969, 3162013919, 3250561599, 2819691478,
                976316354, 2309193549, 3902122050, 2088052090, 89473947, 3770933220, 1285217956,
                495480326, 2012241013, 1505071862, 2618825681, 148024106, 2211817594, 3486453451,
                3547415938, 4170790841, 710399394, 2988742259, 3003999364, 282857121, 3004330911,
                2445074578, 1614383715, 2436968214, 11188480, 2102332723, 1466315446, 3717470884,
                2603285911, 908917900, 3806165173, 1397056453, 1461836583, 327667425, 1587556539,
                1289291700, 446244259, 3478801927, 758862925, 336152650, 518856100, 804638637,
                2976399236, 3191730168, 3165653830, 4116291332, 1180509997, 787973682, 1833860962,
                1047583821, 156500875, 1287142618, 2903226831, 2477133296, 625749873, 3687467688,
                1470807488, 821205035, 4228089759, 498323515, 59751723, 4138495950, 1453952672,
                2570830070, 521547133, 3428749320, 2363485698, 1810803490, 525920316, 1605961985,
                2230129511, 1733393350, 1424281608, 2365058430, 3533943852, 258572135, 209602549,
                687169735, 746097119, 3883623461, 860878590, 899440424, 1064227214, 2191778986,
                3556832513, 2171344159, 2484630726, 290049039, 2356156351, 740380975, 1960730095,
                142059661, 4018594902, 3106881150, 930323391, 1255739479, 3337449057, 3148783500,
                2253473087, 1071245161, 3447698853, 1197848740, 4060561204, 499397149, 2774518906,
                1389562763, 2693574400,
            ],
            &[
                2892496898, 1447320150, 3590707186, 4252286087, 4099207609, 4105150739, 3969659377,
                3877581106, 3844075082, 1119395228, 4111657348, 1989245533, 2625018871, 1435016728,
                775322113, 2486746233, 411477946, 783741811, 294555322, 874068633, 3432359852,
                2911163611, 1375769638, 3434130437, 985693369, 3510076962, 2092775723, 1201035228,
                812940046, 388585821, 3460410142, 4102284201, 2588728415, 1834489811, 1860072658,
                3206722627, 4070805677, 912012520, 1107950571, 2855449499, 2356110400, 1048165361,
                786166069, 3139410711, 2376418896, 213462974, 2934906474, 4184813074, 2148116146,
                2886707962, 1072776382, 2999802157, 3058489357, 2949636457, 1567339918, 2296903598,
                1471627440, 3093553236, 2902773332, 220749635, 1005913375, 859636387, 1837155708,
                2140012254, 1304301626, 4253207180, 2566725460, 1469242176, 3759058257, 3409005372,
                1238566539, 1398646155, 974517148, 3314067407, 339946221, 924727838, 1957551537,
                680280333, 1281468850, 2209761488, 3116305703, 1340386652, 1621222022, 2320551186,
                862888096, 2491969066, 2767235914, 3271787837, 2762818305, 3533899432, 2143575776,
                3910952894, 14685217, 3463875857, 2036855868, 4258306113, 1030748318, 524321052,
                1852279535, 59479177, 3613680178, 3646237449, 1565986281, 2257075320, 2631569154,
                2895149301, 1938268122, 65582006, 4031119479, 430394105, 3217907221, 543498480,
                1033294445, 2556400020, 767171781, 50312859, 2780724798, 3203289606, 3870927927,
                67083759, 2409696923, 3952601322, 3653157894, 923160622, 3398950378, 1555694690,
                2244182109, 3619744924, 2568706263, 826771561, 640735316, 3641939139, 642767502,
                950596306, 2695574621, 1644028223, 554000152, 486521532, 3326739594, 117306555,
                1841260119, 3938389504, 300582682, 3953482514, 430139137, 455233320, 1037301049,
                3615948425, 136331238, 2572752842, 3241874704, 2846529663, 1268666222, 425076421,
                1387063258, 2890954385, 4096839497, 575163363, 2677416020, 1370013560, 2172542077,
                268835184, 2720518386, 2313195126, 2520731985, 1428843145, 1120133039, 3270957840,
                3315172136, 2870322488, 2870507039, 724984412, 3139237569, 952022623, 3933753186,
                3828757670, 3909874303, 178407797, 3904699386, 1722471536, 1933503701, 2084550100,
                714890658, 2573717352, 962347002, 2899589891, 680105427, 2737432908, 602384873,
                3647759319, 885456319, 2557006269, 2747397933, 2991432384, 2531228735, 235505427,
                3454054340, 324776205, 2920011061, 2039088389, 1071395333, 3752028462, 3264518328,
                2221042825, 2520779553, 3023235163, 3336750037, 2776943124, 698637417, 213422247,
                2253953770, 2289793400, 1099689581, 2183186023, 3600213851, 2987733376, 3240713853,
                2532959736, 3990244023, 935412481, 4003796414, 4044202737, 2701294309, 2774001322,
                1246813030, 3523205316, 2877272264, 2808629070, 3952617986, 575846345, 3887709830,
                2303176257, 2448994062, 1814306890, 1125465602, 1437344932, 57018575, 2882004962,
                257559391, 129924651, 8107674, 2641593151, 3225211500, 3751185810, 4152912940,
                2352009510, 1449645454, 1216509365, 2388967059, 2711793416, 1284040889, 1645864028,
                2051429244, 1605254461, 1917359262, 1805912147, 536698251, 1021142310, 3935443381,
                2696660773, 1238552336, 3193475557, 1525093066, 1459803910, 0,
            ],
        );
        // - bits != 0 and divide-and-conquer division in limbs_div_to_out_unbalanced
        test(
            &[10; 744],
            &[
                3587573856, 4227426142, 3896224682, 1468139892, 1853281713, 3230829722, 3544339255,
                913969399, 3983606726, 181178735, 2741195255, 1539029618, 3261057660, 669185888,
                26774344, 3042294054, 4000320681, 3616777430, 1701761628, 984410729, 4231348353,
                2662674575, 1467572439, 2032481283, 1873156172, 115509983, 2937869423, 973418665,
                1489589927, 1095121637, 506586746, 2535986948, 2719080072, 296096440, 1642945999,
                2339337213, 988268530, 2303133076, 2256242938, 646154972, 1819375956, 2274317751,
                3820060724, 742765560, 3675289356, 3771892301, 3075647720, 1818112062, 3031351147,
                3419055185, 1772759499, 3049965965, 716650100, 3384724295, 3504962640, 3501988647,
                4073738879, 817690323, 306125872, 4262005940, 2328708878, 2185547215, 3604829167,
                1214715318, 405219872, 2470967999, 4165888706, 58640780, 3166122711, 1754011174,
                823366021, 588855436, 539915357, 897646815, 4093176866, 1081711730, 282768489,
                761405028, 3134899797, 3405740924, 322586217, 2056545290, 2437754480, 104247006,
                1236818589, 2491440617, 181378199, 1163012610, 1325379332, 4200723192, 1263362307,
                3234219355, 3292135414, 1507504036, 4161995531, 309710870, 3183147539, 982486451,
                3492150688, 694819593, 2066057564, 1649903936, 1060155149, 1595975750, 2913835322,
                1938985687, 3772153889, 3152752661, 1191763263, 3764287870, 3782997871, 81159186,
                4157520234, 1723748366, 1221210497, 3568223039, 927507384, 4004363262, 1795793243,
                3503626292, 2692130423, 1081516180, 3595063804, 849797265, 1035581542, 770759929,
                2889095772, 2572036064, 3097409730, 2958492178, 2450749829, 2069709085, 2687054597,
                3392070073, 3827943191, 2368917411, 3160758175, 1531480588, 2746326450, 3122359045,
                1995296172, 310451197, 1100329853, 2141732304, 3562890277, 4172526924, 4143185198,
                2397101926, 620168044, 828452963, 3271569899, 2726758945, 1647557222, 2748652901,
                1029871186, 1773072595, 1311795717, 2792240418, 3100733307, 298902661, 2608361440,
                642743765, 3026335712, 1586592828, 1251923561, 2152803283, 3359308047, 3979690761,
                2548031009, 2148504694, 2514911217, 401310800, 1418125404, 2325652800, 716233458,
                422262103, 376190943, 1713380879, 2534508846, 2080413009, 2690193951, 2909422387,
                2435837201, 176977642, 224472737, 1590904934, 1664979624, 1748982641, 1284257790,
                2779881254, 385265989, 1148527382, 800142050, 3595556318, 3950265146, 3931111523,
                3399894595, 4200004994, 3727110364, 1510525540, 429323681, 2586345405, 3441619670,
                94654169, 2266545045, 1451808026, 286040435, 364003484, 2106693078, 1916214330,
                2622096560, 3504008387, 1867458297, 4209615436, 2899684845, 3149003214, 2731979274,
                2481557740, 234269740, 1239420776, 2726912636, 1844267393, 1488044058, 1587005400,
                145015793, 1637696129, 1990069464, 3053970062, 676336554, 625771493, 1492378707,
                1231018082, 3059699722, 4113615249, 2967103567, 3335726303, 2137774253, 273460502,
                1561489611, 1227429414, 778315897, 3061631993, 2437789514, 2566029814, 3017507824,
                3219754602, 35255136, 1934306764, 1655248959, 3655963814, 2821596564, 1431977748,
                3114012833, 3915822059, 343010167, 3704265251, 2691314567, 2602475242, 2459094882,
                58857240, 2230690665, 1357656795, 3025543277, 3488674640, 408985076, 80729036,
                246851391, 1419653386, 530010128, 3529087076, 3980421141, 4129023783, 1630771696,
                2057278583, 2501545742, 790764428, 3984347668, 2562445985, 2262348292, 2515967925,
                1103371818, 3735595401, 3210477145, 3387257181, 538884374, 2695206119, 4211749041,
                1006274735, 2406061865, 2654644297, 1268724640, 2399689302, 157759259, 564853502,
                865205768, 2174783618, 3167863384, 1770172407, 1906775255, 3171669377, 2455086709,
                1355327864, 3351895167, 4184858376, 2699533565, 2939169294, 1702277740, 3447680482,
                3322351998, 4248517760, 4100425921, 1169799041, 524849931, 1743345401, 3987645088,
                3022177240, 3263737545, 3656340460, 239445170, 4089162190, 1208712978, 1453215235,
                3390225374, 3337889603, 717395279, 1066458381, 2162685522, 4009460245, 2575879990,
                2578471337, 4218103221, 3046325563, 1352991515, 1561949281, 3488911174, 295633326,
                1993756395, 3677304657, 796898320, 45622345, 2999482773, 2078651788, 1552765091,
                428170722, 1748054862, 4158222865, 1050292437, 249107, 2651086861, 1540178674,
                1275557298, 651050585, 1895599156, 4150309716, 2064336046, 1021257319, 2422595446,
                3116755258, 2756656575, 2328011578, 3632905157, 2576200202, 830418644, 2430246370,
                3913080080, 2371749061, 2683067461, 3611558701, 1603392737, 796477803, 604567756,
                1376069347, 1491246154, 2728262664, 4138498935, 3008562381, 1061547384, 285679033,
                2358943172, 1884649492, 1783482693, 1010268161, 176114433, 794165875, 2362278477,
                3903204233, 326754905, 1988607850, 3187254334, 1749797209, 2986661384, 1759716588,
                3137467938, 4067743599, 1134210801, 3799848836, 1955405545, 3881788427, 3097574490,
                3844756657, 3183850151, 2496328910, 1468671385, 2888878911, 2306353811, 1498824361,
                4152891378, 1588217107, 79108222, 2883552792, 2390312777, 1587172303, 2070384343,
                2265280181, 4013380367, 2742676878, 2654283484, 1471778694, 970959698, 1006151052,
                1276307400, 2874101774, 3169092608, 244587925, 2402787407, 1635687252, 835856534,
                321407542, 2307278464, 2272745321, 2574317642, 729437319, 1682288870, 1482920833,
                776000268, 3908963888, 214874919, 4233311318, 1441385448, 2358127573, 2753681514,
                467574656, 4139948165, 1538275035, 3244920878, 2576965792, 584068468, 3054546876,
                2629688518, 4253271747, 3723872815, 1652066683, 990821089, 2335421805, 1989570928,
                240486517, 2872315587, 3869991906, 3870517664, 1540804424, 397183643, 3750033565,
                1433260634, 1506168711, 3616651625, 512028445, 3746712828, 3278592880, 2611514549,
                1214563129, 1259227909, 1067976218, 3425169051, 741795595, 893688343, 2674408703,
                3694908868, 2478153735, 2220661625, 1022546736, 3719214155, 3161293211, 4131981986,
                1473264088, 1651777063, 1438502715, 290022167, 59234682, 3458968160, 2552001459,
                3451530289, 3800073253, 717882913, 845719525, 1038699111, 3058303772, 1117505279,
                3682430977, 2869037104, 2562493618, 960519305, 4147639705, 1817463351, 3166022129,
                3200769866, 789666262, 2654485924, 3686362402, 2179867687, 3980226915, 3671542918,
                1896992204, 1514962591, 815867715, 3924270086, 4262628477, 3977258034, 1340257907,
                6618754, 2720861064, 778635062, 682181834, 2891943360, 3002120306, 3399643048,
                3139375492, 865948953, 3273305779, 388881948, 3544744413, 3963050187, 3002594763,
                3339669779, 2722426929, 1246819181, 2786076007, 708438365, 1013683719, 3027751127,
                1766272571, 2839608714, 2866928644, 2107420563, 4035553421, 2376700546, 621608197,
                1993043072, 2666011084, 2265522039, 3230507984, 2869423257, 1776134078, 2413254013,
                3859414865, 193597892, 4255395370, 168637254, 3364100552, 3883433219, 3117797624,
                2738841992, 3052596910, 3280507008, 2860095630, 4031447725, 3454885698, 1783630119,
                3036202894, 3585701130, 4184585287, 1329572188, 2352399996, 3076023682, 2989927975,
                320530428, 2081170907, 933271377, 2974966675, 3452895778, 2331110373, 995864819,
                1177147317, 4084213472, 1179430541, 361665403, 2401303908, 3027157843, 2778759588,
                1031442202, 542151276, 4259656091, 745358488, 2580062497, 2004998882, 2066508478,
                341659477, 958017378, 2415007725, 211645068, 3630737942, 2670158596, 3544834081,
                2043760261, 2149621570, 1287267516, 3353570061, 3758258174, 4171807709, 1363035595,
                2692148345, 3728232161, 2672522097, 3234166892, 1337714504, 2475062988, 902334395,
                3470019951, 1789926953, 39991566, 1071624731, 2480238280, 2010573056, 2975909089,
                2685102208, 1752958961, 2957725128, 2441562510, 1615057382, 2739912075, 962437876,
                1445592393, 750430353, 2848157371, 3515397641, 2140566969, 3080139371, 3564834440,
                561913271, 1812943111, 1349101061, 1627550717, 3467766096, 194766042, 3125120919,
                3021598191, 2389614341, 2536207717, 3687483968, 3746428277, 1304917109, 4262793424,
                1046105397, 103309888, 2808595193, 1896772845, 2625389818, 1425524079, 4245371665,
                1376995745, 1906643058, 4123808395, 4010921636, 3668036324, 538106732, 429893286,
                1473862381, 692485290, 728791765, 4006267410, 2159349173, 1146991809, 1105326804,
                855626330, 2350214961, 3945267379, 4182769713, 1218539569, 2795526933, 508156606,
                1596052577, 4135932990, 4009064452, 3154371819, 1789912473, 3737225773, 2339289640,
                382599364, 2822801808, 1231473766, 3195594892, 3686689017, 2674031129, 2724276086,
                4112764261, 79570030, 1908454621, 270118882, 3204318684, 2240304382, 1923066108,
                3669840087, 3114917464, 57715381, 3015749933, 3183317351, 2563719945, 2409212385,
                2256893938, 718636813, 3965735223, 1345089653, 1264444, 2296052850, 1092210950,
                3468764525, 3967443918, 788417425, 1924956491, 3656370968, 4266402294, 389687964,
                3067575949, 3786278950, 4368934, 3414260125, 1500941491, 4197777812, 1901284905,
                2548021755, 1986057606, 2732888210, 3872664452, 2787539702, 3264559111, 753549553,
                1048190618, 2900005727, 1868077400, 1284542693, 3154799998, 395567255, 2005460208,
                4005052806, 1893310835, 3217932531, 2607307407, 3917316670, 2028218244, 3745680211,
                2397481422, 736482987, 1916844834, 3868328610, 938512555, 1559481864, 729544587,
                3690980706, 1759014647, 2060717833, 2250640148, 3619925046, 2153794810, 4127168634,
                3259374700, 2051907961, 3964686808, 3841055905, 4242264783, 2314742304, 2209077724,
                2577227865, 1487635776, 1585379583, 3475070421, 1683734827, 3363053669, 3722095029,
                3857335408, 2852846850, 456879372, 2473892714, 2928343667, 541075767, 3595876467,
                1688710352, 2071331730, 1142047400, 1817453168, 96871997, 3927306877, 3090061646,
                3474317652, 437148773, 439538568, 324686794, 772632617, 1424328970, 580538580,
                3999279969, 2022469388, 2802303848, 1147488095, 2053949131, 3046702544, 3822972379,
                2920233521, 4031279543, 2356245098, 2951036256, 3287235943, 2760424423, 140913700,
                689952328, 3916658401, 1694797888, 82150998, 4075118605, 1967095926, 1704543314,
                3154572744, 408071699, 844684417, 1174429103, 3583461805, 1015646627, 861970508,
                1906905868, 2272773809, 879277860, 2980820537, 1917774935, 247497916, 2403283458,
                553129122, 3303057196, 4005726052, 1808761740, 1909802116, 964057278, 1586240623,
                3097009405, 2048123311, 2481968244, 3155267854, 555253647, 4027932249, 229358586,
                1015669317, 4112551330, 351151415, 1331401879, 1749898409, 3352469407, 710145444,
                2903798473, 2876271745, 692844392, 2354652850, 100021926, 4212629124, 2971597719,
                2697935131, 445511347, 1636699871, 2524940444, 1303870696, 3634945394, 2398930906,
                1337769794, 3955409228, 2657553814, 1455809030, 701994564, 374320080, 519334058,
                71402463, 2995013099, 1573823285, 2419768029, 4108602983, 4266125692, 3514998795,
                2367509976, 2654621106, 562141353, 3101668250, 2753822172, 406447740, 4132920329,
                3645443797, 4221410098, 1000631411, 2319369935, 3987192941, 609889174, 569928846,
                3471449767, 761399938, 2925981744, 3610481831, 364846710, 2579622933, 1755359875,
                3327963251, 2230753636, 3897751713, 3685870953, 720576993, 4008645094, 2170973511,
                3057371253, 2362087099, 2415801497, 3804893423, 874616900, 4188156090, 2114593709,
                3626784402, 2090293821, 1178445498, 3501583487, 3787814639, 1160994150, 3773261324,
                3438864014, 3474697300, 3232616282, 3082635737, 3290126053, 1041836645, 1497490946,
                2116412677, 78748560, 2610841375, 406515051, 2540982558, 1278151559, 910145724,
                2942229044, 1412896287, 2420692111, 2829066497, 2762072644, 883699073, 4146766932,
                968581437, 2262117978, 4102625453, 753476188,
            ],
            &[
                4049869757, 1602523624, 1387514923, 225246600, 155172862, 3856617978, 2201641309,
                1540516258, 34942771, 4041710783, 3460925118, 1016127034, 1143953610, 3427876200,
                4178328442, 3374908315, 2087712114, 4171526185, 3514588648, 2986361397, 3768375872,
                152142192, 1983892945, 2539876375, 1278133015, 3290367811, 3728704045, 2495609578,
                3383634048, 550401536, 2958764367, 417726361, 3629290307, 3955082917, 3367571078,
                1156977964, 2700212626, 3890522506, 1407330442, 2072012244, 292784856, 2848511017,
                2011019434, 3729188240, 1314875514, 1752114201, 3480385261, 1532349465, 1252987479,
                1748370879, 457308873, 2290488535, 4117600387, 1025979949, 4285742993, 3815216993,
                1825262081, 665711516, 1319233702, 3390016806, 2279278230, 1335733129, 2732993747,
                770076251, 2848793746, 893093476, 350727685, 49933264, 3771423699, 1804029403,
                3544637522, 3176796760, 475781914, 777775753, 2152296620, 1317524444, 3941840558,
                1662743930, 1905993615, 2485835810, 3925643251, 3071436009, 851721712, 1325046168,
                3214018378, 1465803515, 2459667310, 2361559987, 2668552637, 2425633974, 3200812339,
                2594448814, 4170435967, 1112582678, 3198704424, 4028094030, 2482710119, 2990475705,
                708195759, 612294539, 2794828841, 2498141427, 3805184114, 3010938369, 1479667740,
                660767380, 1641177565, 1782849661, 1915222559, 1626388136, 1816788637, 1338361170,
                783877621, 4003339370, 1930607900, 1259399167, 3351643097, 1641708262, 967800396,
                1800752717, 2198926109, 1163817943, 2710351254, 451351637, 1285647338, 865168955,
                645286276, 2685132510, 1773153387, 4273868103, 2604563645, 4105767904, 2556376985,
                158907213, 3579937882, 3059825408, 1920542835, 528717490, 1430681949, 616489338,
                597761261, 3760865497, 963173252, 2915089223, 1441674715, 1717557648, 1819215517,
                3449795284, 844168976, 1574237607, 758725457, 762624299, 533122182, 1201164787,
                1968174784, 896982568, 3419630169, 2247559545, 3983311870, 3975342941, 1112833399,
                2721518545, 2493587613, 3444837338, 3313000598, 751186769, 2970698395, 915811688,
                1206259449, 1340427760, 3844346545, 3762393860, 543253569, 1197933603, 3734607133,
                4037352821, 2263945478, 2831362781, 3363558852, 476952769, 1916745391, 208671986,
                2395250976, 1549715018, 2746690542, 1219103496, 256305249, 358172450, 2072583005,
                3916780941, 4158630723, 1190830416, 3779410961, 103078237, 413254256, 341045330,
                1139999874, 92806975, 1388100212, 1158509802, 2339968162, 1347201015, 784082665,
                1171121798, 1281820962, 528442543,
            ],
            &[
                1082720091, 3476817859, 1893722246, 3080882169, 3417947745, 3340276454, 3885894760,
                1942705317, 675159075, 2356580166, 2812410352, 1542004962, 2479082003, 1559105673,
                1430407422, 3754123073, 3434749261, 3438277207, 3184426987, 2481219277, 3201501212,
                2336166635, 3118471262, 4257588863, 1129103696, 2036867994, 1362270000, 2205796804,
                1619465560, 1334326802, 1654568402, 748744581, 529239522, 352036781, 4164399267,
                339969597, 3948817065, 4118696507, 3167307180, 3299871862, 992173657, 3312501436,
                4234856346, 1472448261, 454903616, 2135585012, 1182478660, 3108573415, 289552136,
                3656397512, 471369369, 621698889, 3506241896, 3050415504, 635379448, 309369370,
                3135812778, 709795745, 3156857644, 1981234127, 777872572, 2069114566, 4227271663,
                1642023978, 4255257763, 4251585692, 4102236891, 61692500, 3829081047, 2974587843,
                3417120535, 3759026709, 543679528, 610505883, 2224107411, 952038760, 1845841012,
                2091165188, 2319077898, 813461061, 4130723132, 819025288, 308176462, 1882359754,
                1123046783, 351488317, 1740084483, 4031432934, 866128405, 3761028110, 153341699,
                1557409832, 3958654248, 9848396, 1343150951, 1655256005, 3758327652, 714608206,
                2476402679, 2222813494, 249341289, 20217846, 3718322691, 1309897917, 1956873127,
                3780340503, 3187273275, 2997743448, 2763864945, 1015229074, 1035154014, 4111197216,
                4129892876, 2463400739, 2137142214, 2932334703, 1212478961, 1404771156, 2403376672,
                1666383249, 2101966267, 2787146653, 3409236902, 382073169, 8092902, 3172754632,
                2362871022, 174894239, 2851299514, 4035565821, 1576086690, 1863466767, 2147230465,
                4012739188, 2573883505, 3496112107, 1561836070, 2714989398, 518249309, 585535272,
                1083806803, 2432909736, 417453124, 3662025813, 3862714644, 2982211872, 3413054134,
                278502998, 1933231828, 1756355548, 4175286263, 1635691203, 1696182335, 3722603401,
                3499515010, 2780239369, 2967447083, 3938392642, 1039437781, 156713901, 202025882,
                3388635620, 4085640754, 2766466060, 3257237392, 1610632422, 2659484643, 759134140,
                2468303426, 3485429206, 391268866, 3602129409, 466642453, 2027942747, 753837398,
                2159887674, 1699077410, 3385420601, 2097219048, 1939536260, 4216211838, 1310649129,
                3770773042, 2367698780, 2182982254, 105876687, 389319130, 956327995, 1042519322,
                3560831866, 2584066309, 2359741951, 3525689940, 2936794464, 4165100291, 464352315,
                1510693553, 1019366466, 1339175695, 2900331968, 1808660155, 3444379929, 844617472,
                594684582, 2551754663, 4184241459, 2541557677, 1600387746, 3461253021, 20795987,
                2007859697, 2294435372, 2967012275, 427466509, 3926685990, 2687323981, 58620064,
                463567707, 2009818918, 2587127815, 335237443, 548349111, 2327100548, 3527232221,
                963188157, 2846759149, 2508857384, 2785887779, 1911224250, 499633849, 339500132,
                1698755389, 3785332522, 1951026064, 1157235411, 2385637652, 70752751, 3581109807,
                1000717338, 367332589, 852120579, 1720305388, 2396130011, 2325323294, 3159509753,
                2304229465, 3412995226, 1302807073, 1047169090, 103900265, 388968533, 1642951622,
                2119771731, 3904510666, 3214555812, 2090353527, 3439045026, 414952145, 1990923465,
                2145341503, 3376393463, 1768247381, 2500689220, 3437741376, 2952550055, 202503815,
                3117369798, 3572610192, 3057469623, 3868128130, 2897590216, 1333004082, 3846323733,
                1145656956, 2957222763, 1612732059, 636089096, 280438137, 4133371531, 2415302446,
                3343573484, 450368029, 4013078578, 1776076354, 4055087576, 2875505329, 956580657,
                1284100217, 484412750, 3403238717, 114812192, 3141957869, 1923373041, 4267447588,
                1848508027, 3578399081, 3602314328, 2015600160, 3169258369, 4225283, 3215758323,
                535488666, 2299743753, 2598839590, 2251925973, 3424293163, 1583565986, 822791938,
                499972304, 3526253756, 3187560030, 839910754, 2473717205, 2920751658, 281393009,
                1675618188, 279356115, 3737356546, 2987221453, 2784535811, 1716035779, 2728300709,
                295755590, 79028394, 2321443009, 3334571157, 2092798399, 776878210, 3880589251,
                3000121191, 895584305, 1972715675, 3933866443, 2459940020, 2535432427, 3442732987,
                1351280251, 2705869134, 4056756488, 642205063, 1335417345, 2925570768, 3272930570,
                1215315775, 2270360900, 3920083766, 1800417946, 2267749997, 4044783727, 1769884599,
                2859569555, 3317860539, 616401729, 639899592, 656298035, 1728919003, 4078302028,
                666745393, 1560472966, 3582178280, 30041205, 1445208285, 1831729364, 3833375730,
                2031873203, 110189422, 4001884424, 1172085689, 434283494, 365118314, 2974799471,
                2419990253, 2254345440, 3303650744, 379954120, 2872690320, 2813488227, 2931609524,
                750879352, 3553708007, 3932935565, 2268402810, 4181556412, 2645378729, 3996572258,
                1392500829, 2476571256, 2592516278, 2941009392, 3567442876, 3295525458, 2364649002,
                1107394009, 3085546406, 3546490308, 3980163356, 3563024751, 1632689827, 2857550583,
                2854525629, 2769744068, 3179245702, 114619536, 1503846887, 3983245950, 4053448614,
                3804164557, 483691484, 4154677001, 3343013834, 3198409629, 2196007920, 2814767401,
                2583091893, 622358751, 1909097055, 4275650466, 2408839673, 975842192, 1807945419,
                530334006, 2193239360, 3573576815, 3375828028, 343495318, 4289026005, 1962467408,
                347337978, 3396953980, 884291722, 1137165202, 220827107, 779266710, 2344445455,
                34856979, 1143045482, 56767308, 1578363872, 3731201669, 3706120565, 3337776817,
                4289277255, 1854266966, 983344215, 1966863762, 367660085, 2557629697, 926188478,
                3333617884, 3810871127, 349652103, 217155142, 1688170224, 3319067823, 2885726973,
                134399180, 5674764, 3973054562, 2273408331, 3833933899, 3417440443, 1041778853,
                1946826410, 3801686902, 1531220405, 1393391045, 331387982, 3898611911, 268525782,
                3769634949, 3016559037, 2445833730, 678007929, 1519682189, 746295463, 499000954,
                1304839962, 2069998292, 572428575, 980128037, 1245074429, 3558109310, 2997264588,
                658214766, 883463509, 3369353012, 1297551772, 2110922247, 2798156905, 955238199,
                1636058159, 4078945672, 1404295386, 3413374279, 1206293582, 2450940328, 3853860296,
                568231531, 3871379428, 801397097, 3548192318, 2198482100, 1425818836, 2226232328,
                2217750444, 1674221667, 575304554, 339577605, 1944985402, 3285623423, 137164688,
                2194970053, 3895173964, 2206988857, 560190101, 4150633681, 2529897699, 2877680493,
                2138142714, 3624049789, 4259353316, 1917725116, 253374220, 1961669318, 524948489,
                3660413785, 2459743863, 609305506, 4234617120, 1880171213, 2231312825, 3930360917,
                3757082546, 1598330816, 2275523468, 2537954969, 2648376719, 3949938840, 1179324615,
                2638061103, 1420005740, 162992051, 2259945784, 1691781876, 2035325287, 3681767229,
                1864052113, 4098159425, 2904470037, 2739041791, 3653434846, 25187402, 743386308,
                1617431581, 1955566361, 1609808798, 612646080, 749092400, 3476155485, 372463231,
                2400385265, 2448935262, 2868736303, 44050485, 1811513622, 244185373, 1058010832,
                3236838304, 1696379775, 1374320342, 893033010, 2184044974, 2103383697, 3678814304,
                3707318863, 2232216478, 2121008938, 3391204665, 114653376, 3539615664, 693415359,
                3948513646, 2472340375, 2689595350, 2911666831, 2862380219, 2277904386, 3052644532,
                3620922554, 1583320802, 989453708, 2848948202, 2802447076, 314144584, 1963582024,
                3054383355, 3230565030, 3198370807, 3232305102, 3533915944, 1159220898, 3409956546,
                739193900, 279482209, 2395025280, 3213913189, 3465386197, 4007322480, 1185701264,
                882190169, 301180834, 3985404529, 775008128, 1929920973, 498148924, 1653387516,
                32770831, 84662291, 1070233280, 420774040, 1845466025, 2660849065, 892513130,
                1220251223, 3462048171, 3639870150, 279428221, 2277386828, 1042423213, 49732010,
                2156548934, 2306505697, 1610786467, 2412885409, 514239305, 3502348066, 3662528922,
                3523842425, 499321566, 1558826766, 1025787712, 750835728, 2151253987, 2771548076,
                1874413816, 1770687295, 242014543, 2896631433, 4270012022, 4154175689, 358416847,
                62726299, 2459979834, 3959988277, 3138565580, 2632517010, 3757121807, 982781859,
                3866772115, 1940202687, 143199844, 3711383957, 3192090943, 192177309, 3414164907,
                2772999412, 3345422423, 3632869399, 298443741, 1301417999, 3779042088, 3129617922,
                2534671226, 3888890090, 3440629680, 860078030, 3705887480, 1118604355, 1986962149,
                2134492511, 4252575925, 1247103708, 2063043343, 4018417608, 1435800387, 3604051252,
                2382645558, 65332182, 359897542, 1245864255, 1609404488, 3987595397, 3239425334,
                187261326, 4123855750, 1200175240, 2149408423, 3981447331, 4266532762, 2991803305,
                904593361, 2972818330, 1285040552, 432629695, 3318003843, 1422945986, 903546290,
                4170273900, 2167679655, 3927128491, 369901225, 3823643528, 3070820575, 153246247,
                3566780970, 3775810437, 100362394, 3451770819, 2165886441, 3421082619, 2757100947,
                3862770948, 991992674, 2507717612, 1238760013, 1566046550, 3362507120, 3516476868,
                591759613, 3252699908, 3659831623, 102934758, 1522937224, 2690593629, 4166523866,
                2815574307, 1262613047, 3171249524, 859033927, 2026114441, 321675781, 4067069202,
                1828982470, 1,
            ],
        );
        // - bits != 0 and divide-and-conquer division in limbs_div_to_out_balanced
        test(
            &[10; 276],
            &[
                108635137, 53042175, 2495789053, 166626339, 2790541996, 3399648116, 704235903,
                1987319994, 3846313422, 3704598543, 3690196699, 3215164091, 1591950078, 4084137444,
                3535469973, 369348781, 3853840741, 38365848, 2204958899, 420812105, 2779552010,
                1548769452, 685824244, 2413919731, 3001005256, 1844653400, 2702666282, 953246091,
                1180609303, 1728187391, 3376808367, 922212062, 2555240, 3091442748, 2779003823,
                1730350709, 3884150184, 1912296588, 1381602302, 1132745285, 2605509609, 907722107,
                764554269, 3991449945, 3268497606, 652777597, 3569835073, 3292533280, 1757212806,
                4083008115, 1567249607, 2029822077, 1300690097, 542854515, 2421776641, 1792015269,
                3385254138, 3324421134, 3135185624, 2936968278, 3252736384, 272507607, 2728141594,
                4200848223, 3001744645, 1397661269, 3892465560, 2335547432, 3245101563, 1241760367,
                93951076, 2656438422, 1056867098, 3937300802, 2762984593, 3306503534, 1748156730,
                1129525575, 654008385, 439540212, 2194484000, 3773978123, 3104733633, 3072629324,
                1956867429, 3192898033, 520613936, 2701719680, 3263467682, 4152521012, 821868901,
                857419957, 794567744, 1966964158, 1188525445, 25386574, 4072516363, 2420189258,
                506921989, 4093279187, 45547769, 513061404, 243072258, 3189135266, 3207683448,
                3842162034, 1133520520, 175840391, 3746219860, 744587725, 660570869, 4002101132,
                522243467, 2303217069, 1674769460, 3791198176, 3339780616, 3458756479, 1979186301,
                896682364, 2308144330, 1378320625, 4274094804, 586903390, 2523245831, 395299569,
                1187786606, 2283803123, 4267141178, 2219595837, 1128697087, 1857876110, 1318499953,
                3165428382, 1575816218, 956482234, 18588216, 3132721083, 47556415, 81611970,
                3910123222, 423589575, 1101413252, 3606836170, 1942712759, 3189437565, 1988019498,
                2092781681, 4236534903, 790511562, 4060480297, 3105894710, 2764365449, 1397438933,
                1196278536, 2058315951, 2838695613, 4281268824, 3826944690, 1181288743, 369591812,
                1309874942, 3790984071, 288341904, 1563879876, 165797805, 858735816, 2072353128,
                685737219, 361047486, 4066070935, 747362930, 2075612928, 247842391, 3759957561,
                4008727703, 3301939318, 2461349415, 1220733209, 3976094938, 4150525547, 950524413,
                615359333, 3819351887, 604594973, 967650734, 3412914931, 940128704, 3084501448,
                3552914752, 1468579018, 3375888808, 272090024, 2102430932, 2231900023, 407815192,
                3133923502, 808329364, 1437980982, 2714133004, 2541977126, 1644097574, 1605051578,
                900680606, 1233529861, 1285360886, 262211605, 3340000454, 3059934898, 261179603,
                1359321373, 2785008571, 1696998009, 4208277335, 1632076295, 1780238612, 1096677031,
                2326184831, 1755582758, 1271079634, 3473819079, 3524297696, 2971249360, 304707349,
                1752064216, 3406003757, 2298730120, 3470434630, 1822263204, 3897394658, 1744777112,
                1474862410, 2619263924, 1883564355, 245003326, 2772487730, 1684833147, 3384787730,
                3432821829, 1659957487, 4143280212, 2945269934, 2715460396, 3109905612, 2183811774,
                423754031, 2103134812, 1722483377, 1623977444, 4207250751, 474534834, 2675551745,
                1065671136, 3814366580, 3372473612, 922981250, 1210468465, 535527770, 590607480,
                2860431542, 509048194, 3860804971, 3161644296, 1119093457, 2017558623, 3882518387,
                2116262688, 1592400489, 1330477596, 1812541649, 3066693344, 1681733293, 2837523711,
                3962139488, 2400483346, 684544040, 618138932, 23933358, 337472067, 4255565569,
                3863904974, 1571272948, 3063467317, 3199661373, 2892202115, 3466650924, 2813788238,
                199916841, 6302159, 3020774987, 2959649954, 3051172815, 385827752, 3500074887,
                1998588949, 740187474, 1537085728, 4040744226, 169795027, 1388002404, 448995184,
                209159058, 2041030169, 3989416757, 2372340142, 610343086, 3895834685, 3130822270,
                2937770030, 354183502, 142181574, 2610105359, 2678801631, 3035570967, 3617741371,
                3969553519, 1367705229, 2107582010, 3358994616, 3893272782, 661536642, 41261552,
                4179523599, 3717595615, 2660616505, 364657719, 1591424432, 2930227288, 1470954796,
                2176631849, 1725166260, 2686775012, 818766763, 1889939384, 3037885089, 3635001611,
                4198592608, 1743285046, 2898224735, 1659026011, 318231763, 4236634945, 2813628609,
                1962109336, 1610396960, 3016705702, 3623307415, 3832212453, 4249700382, 941544661,
                720165462, 4093658500, 3453816013, 2079181823, 523275463, 3269528702, 3019887295,
                3811204518, 159775628, 1722773106, 1469942260, 3026312336, 1636680124, 676646879,
                1243866643, 3046790125, 3838163884, 456070987, 3400235386, 3209376427, 2052709404,
                2975905060, 3314477136, 63144045, 1485059871, 162201390, 186441332, 612262739,
                828760782, 3475105064, 2824483050, 780082694, 2214596670, 79854651, 3785529118,
                3362875978, 2019456835, 3664773598, 2128455541, 1830176771, 3131136264, 385070469,
                2166122015, 4242832149, 1871374367, 2866877895, 565736311, 3669481600, 3732705160,
                920034463, 1676220645, 3589606699, 2654716480, 1934326920, 2849139664, 3961632132,
                3838830143, 3776089649, 3421594363, 3482840282, 2537104160, 3010338838, 2165340101,
                1892478832, 2700868158, 1092691010, 2008876123, 752365533, 628818795, 3526690309,
                1565204, 3539771692, 3129225931, 3999602879, 843176653, 1303696065, 2008779360,
                3311453295, 1156870995, 3738053906, 359090986, 3637794830, 205702249, 3331669407,
                3910416238, 3925253071, 3900326738, 1156846221, 567150051, 379723012, 3421075741,
                2064258650, 2895443535, 753490073, 948868209, 3567651732, 3728641104, 1287032959,
                710808071, 1107267370, 515160531, 1433094102, 3871178973, 3077300132, 2340844017,
                1955407235, 1314652042, 3168268098, 2448320270, 2400447335, 3294295438, 1015553266,
                2155337410, 1202973291, 26721975, 748351815, 3704104726, 493250895, 4196736012,
                260103997, 1216172054, 1878845953, 2517690048, 2952150363, 2853405105, 3643402580,
                173602363, 34441180, 4232560859, 3432794922, 1140120890, 3906522401, 1592024957,
                4187088827, 3565666595, 2036898614, 1513649673, 89930944, 2373451644, 570806865,
                2966068116, 216216149, 901919588, 2525422962, 926283525, 2020466305, 4176067636,
                2062906224, 2749700324, 3770003319, 4243016358, 1536893669, 1734422817, 1903034843,
                2140137740, 2133960349, 2127799686, 3826582112, 219489448, 508455455, 400134475,
                2681095601, 1225367969, 1189240142, 3928151903, 2223436694, 188587138, 2759880580,
                2159086089, 139714886, 4230028328, 2668106965, 197527044, 1076021583, 365687428,
                2429361726, 987702546, 479721329, 370553881, 330849794, 1208154538, 511352639,
                1347185232, 3309753618, 2165973922, 2714839722, 1566744459, 1258272726, 4003842107,
                1592609821, 1602879458, 598396180, 1691880384, 1805556651, 951428589, 1382102411,
                3352671230, 3650598499, 3958924995, 3934211043, 1708901267, 2214291603, 3062058779,
                3334192431, 794991697, 1238615391, 1385289547, 3821895409, 36111771, 1154601572,
                3442859590, 72901265, 204664637, 3567057924, 1929964733, 3447462776, 1857620277,
                1670547089, 3282928074, 1385001633, 905180163, 3027069579, 1563462544, 3261712196,
                2691524721, 2887321824, 532509125, 1250479582, 1736964157, 1752056130, 2411795965,
                2530177666, 1944298739, 4021827042, 235488471, 49513017, 364455717, 3051752647,
                2541367373, 4025312306, 758560501, 2532590036, 2497355449, 2645291172, 4094509748,
                2811604062, 3938869461, 2314766426, 3949981315, 3113301191, 3961932201, 3642942123,
                4042045976, 1080625720, 749107450, 2408837571, 773510366, 408327874, 4043794032,
                2751095877, 4253356392, 1852579882, 2930909909, 22950574, 2455166107, 1518997069,
                3502877484, 3466955328, 2495681100, 1416465050, 2781446012, 3974552369, 3963181882,
                4031801829, 971448688, 3324460697, 3468632379, 596003274, 2001813586, 494254023,
                3111644337, 1214103910, 2547242619, 345292455, 1171889494, 2523157824, 2004995962,
                2793293234, 3357565902, 3618900967, 1122158351, 1840000495, 3714974862, 2372340013,
                3036104311, 1987619821, 1915784172, 187130136, 1032034929, 764058777, 1324555408,
                4258631431, 2565322815, 3622146170, 446425747, 663948352, 3473130059, 3081160831,
                881843540, 2045224772, 2479557013, 3352503836, 1209737991, 3178334862, 1649664345,
                3898477658, 2455734938, 2058141628, 1165813520, 3499004048, 1363642302, 1673253013,
                950040890, 859882680, 3358702534, 2831246639, 3154009927, 3886020609, 4260600294,
                4292328051, 2154852829, 3071909756, 3083791599, 2780514384, 2089946486, 440735136,
                2840096690, 3847114327, 4107780595, 1507811167, 319613030, 771404322, 3267312644,
                343171187, 416166006, 400367691, 354408782, 3664160643, 306037219, 2628548851,
                2490873938, 147038682, 2557865570, 1918482947, 1550632444, 1104237513, 2381980505,
                1257779567, 3978083093, 4188937519, 1450461940, 3573655366, 4007222924, 899058972,
                2762018299, 2973671862, 613016156, 3453147112, 3958949304, 248065424, 750535902,
                2025904997, 2542540928, 3631746189, 1474696231, 3349867745, 1645065489, 1753887057,
                62923614, 3355303085, 1757052497, 4111601366, 4122660071, 3488098722, 968401720,
                2418687776, 3889098632, 1507574874, 1788573805, 930165776, 2563485754, 2283929886,
                361900473, 3829676677, 2920155840, 2690982721, 2647324946, 933444487, 1982178922,
                2334110536, 2878917031, 3486781662, 1761730411, 3024808673, 3873242498, 3895635845,
                1874483821, 1898379794, 4172331028, 3759742822, 339224592, 336709787, 1768365716,
                3768045522, 1226451429, 2705764626, 2090014839, 2715498023, 15765857, 262371622,
                1364974277, 3278122087, 1085595683, 2097096706, 4084807931, 342124999, 3362503915,
                102138122, 222464351, 2367255609, 329998946, 1654125618, 1397194509, 2342407041,
                4218981334, 3043210002, 1820580167, 3532477643, 3698803212, 2459696610, 3760108330,
                590088894, 2109406971, 2851337682, 1821297729, 3479206596, 3141773016, 1887780855,
                3345901315, 3465462131, 1285191121, 3092915327, 141461573, 2073757501, 878536688,
                1772007905, 1193942688, 1553571015, 2179016361, 3232910452, 897070601, 2148852628,
                4215017747, 571464735, 3364559156, 4279542294, 3759036030, 2656683267, 1707236123,
                1356967673, 2580245428, 1451203266, 2299603042, 3068898426, 1148383426, 950626083,
                3457385021, 2591070383, 3181532137, 3347056243, 1495266049, 2210488892, 720796437,
                1413487507, 3065309505, 3696802085, 2734150063, 3600467301, 2529953774, 2679381445,
                4233323864, 1085096641, 1587182738, 4041255127, 891185711, 2340666116, 671752008,
                4136178412, 379067213, 2916836063, 3982716045, 745125821, 3190060915, 2728700228,
                4248229826, 822059760, 2434419301, 2390321192, 1529594228, 2455438975, 1623635782,
                3927813233, 3131415702, 2632189134, 2093836398, 4098626915, 1127324299, 2709110703,
                1267984800, 2156759432, 947735481, 2766133233, 4227965085, 531828395, 3790033202,
                3790793152, 2975006665, 1116356882, 2644453515, 2771505868, 3490730292, 2557842446,
                3338652340, 837555851, 582834001, 3882158027, 384350213, 708918872, 3173032166,
                14955604, 3415856040, 3404370362, 3759099429, 2374207480, 1534379322, 2698542736,
                2164045979, 660913601, 3556971203, 1616608268, 1829936346, 3109981160, 697542468,
                740461237, 1422792083, 3962800663, 3066960795, 1173757342, 769744240, 3750815337,
                3814517080, 385190612, 2953064964, 1871983583, 1649509432, 1554161863, 3185361908,
                2106975424, 1538113731, 2523454661, 143645610, 718612279, 1723847182, 747395397,
                2882688996, 2021890066, 1030716023, 1995073902, 4105305241, 2127982715, 1690451116,
                1956139100, 530256795, 3558366590, 2724310482, 2341665586, 250870045, 3391930252,
                3942088647, 3976895259, 112642770, 3730114995, 1371701615, 3644536213, 2026852217,
                1200740294, 2038780887, 648265626, 1535440607, 782951933, 824579667, 3601202459,
                2314910152, 4263436566, 442006573, 1248351448, 3167704183, 3934018759, 3957471958,
                2504687739, 3253599456, 389758313, 3342046926, 278999028, 2652455147, 3325515388,
                2176380480, 1129683162, 3997989394, 3620086114, 2746229755, 290677810, 1466119856,
                1528967320, 1110833004, 135210629, 4036023385, 82033834, 420677558, 1964407130,
                2702828444, 3920237543, 930065859, 2455503717, 544391994, 38795400, 3389365984,
                2189324221, 3653325550, 2984928969, 985994309, 4212011784, 433292705, 3993514299,
                272538953, 84416207, 3750907699, 4166502555, 3618032189, 1236823026, 4151994611,
                2505418422, 1513527578, 1564158364, 2400104436, 2118135017, 497726437, 3029863178,
                3946017842, 4023841770, 299480378, 3182449534, 3212382324, 3197008823,
            ],
            &[
                4002709364, 963399356, 2782341067, 3945310775, 1640289005, 3723750838, 1254211967,
                2304588540, 3174496786, 2728509707, 255821323, 110353438, 3720272848, 2229377273,
                2152240593, 1976405605, 3700189945, 1261817945, 962162212, 2968380111, 734133596,
                725489232, 1719612084, 3884866627, 28057341, 3961915971, 1325614652, 65176638,
                3323056153, 1428210708, 2748040079, 943478886, 2463128528, 1179615866, 819519877,
                745909987, 2928403710, 428934326, 1802933992, 265877181, 784539883, 3368469872,
                2186856195, 267183160, 711926800, 1784916336, 3062621903, 41142608, 1185902375,
                3289485275, 2769494454, 845407692, 420641321, 1822514821, 1468465938, 3143456289,
                1922484069, 4035995228, 3823561491, 3573485374, 1462883391, 949414736, 1205197438,
                1545720767, 4004962418, 901935939, 1176411707, 3934655491, 1247692219, 3372976809,
                648846849, 561624310, 4213429112, 388574078, 3876368507, 2459110297, 3629548833,
                2842464979, 755880226, 109802803, 410812363, 1884761319, 4167310741, 1416731661,
                4162069084, 3695932529, 2935063704, 1097793995, 2124836831, 724360417, 3061931520,
                3339903277, 3019747519, 303387593, 3800465715, 4021351514, 2617312748, 2101558562,
                212720952, 2948123341, 2761073207, 3424660022, 1745409052, 743791614, 3820519067,
                3495466566, 855725038, 3341655828, 3835113615, 1270021592, 1093683126, 366232505,
                2382273294, 687690291, 2407217871, 3126999218, 2489109394, 2399521074, 1912270761,
                1547900176, 112668330, 2418271838, 164643, 788458686, 1283051461, 502966828,
                2822195605, 3290495578, 1940639423, 1104706107, 334853171, 3663855161, 3267632296,
                1384333597, 2439397119, 360160344, 913951997, 3699617663, 2223491591, 4152571511,
                4127552387, 2384896323, 2719106630, 1968080578, 3047590289, 1349230865, 2632158056,
                417315676, 4047908012, 2058515124, 3962573168, 435657190, 3869063277, 1129208107,
                371028160, 2341806620, 1837536116, 4062395278, 760408526, 489855682, 1341968057,
                3582061385, 567307943, 1459157484, 2793270460, 3765006265, 316630926, 3312280213,
                3038146737, 1954778145, 1255537680, 1872487321, 3078336, 1786349493, 2936757034,
                3157582052, 656659566, 638195606, 2761830593, 796052712, 1835931537, 3594050559,
                2825844970, 1783476497, 1587070264, 582928447, 349530572, 3521756034, 3849814263,
                3898598710, 1487612894, 3546940589, 3702560901, 2571834495, 2432615017, 151334837,
                1929191978, 456717814, 1822026775, 2905791667, 2187809681, 3085861483, 2527039138,
                1461502989, 1674065300, 1427963173, 3931324821, 3503894366, 770079370, 2646944748,
                192300986, 1063095706, 3173456208, 3863439598, 3538143515, 1736463922, 2524154399,
                2134152316, 2565447595, 2642170528, 3536538194, 1250655424, 359773845, 3664161865,
                879705815, 1474928595, 2536605270, 1911087921, 2962799796, 4010905032, 2793477763,
                1959425433, 3713879004, 136714178, 1724676493, 867784003, 3212103591, 453276555,
                1061978135, 1967464120, 3591583971, 1101672396, 1057025284, 2241955674, 442433290,
                2456783152, 793285251, 1178449650, 684670303, 1866273007, 1391022401, 3083722623,
                2493222535, 1189326604, 1199252550, 3969981178, 4060147906, 525899569, 24010191,
                765321740, 2761772368, 3377909206, 1331140315, 3875549784, 4212148483, 2588352295,
                1569174674, 1936659977, 626141956, 2533524762, 239859149, 3534146265, 2997682745,
                71238037, 2732432587, 3908540462, 477053537, 1471594724, 2867685233, 3586191718,
                1273507649, 1161961515, 2329077270, 2132699100, 2632998553, 1299594849, 1554941646,
                3548827963, 3889378581, 467940077, 4211088672, 3371367836, 1620737016, 2233519964,
                2624185462, 843840626, 2427990702, 3297260629, 786361707, 1646501219, 2205482711,
                1032553750, 3122751558, 4227854173, 3810881375, 423778347, 2970196049, 2431725362,
                2079714185, 1671048359, 372886771, 3881108825, 1517433910, 980297672, 460700981,
                1452459198, 1527418894, 1426505982, 3181114286, 1351932282, 2092103967, 2878978239,
                2422613736, 1750361767, 3327673906, 3090894970, 3102624891, 2501611324, 2723100867,
                3663964999, 852336624, 4080907747, 3496497293, 865887813, 929644307, 2488175940,
                838516249, 244558558, 3755455496, 3162262069, 696753816, 1156821200, 2717965945,
                2023685211, 742596168, 4023287277, 869919872, 809477569, 2011079194, 3221489136,
                4266255834, 2419594884, 3939612855, 1706366855, 2428360994, 716879901, 3297635122,
                1769572686, 2873365745, 3831233968, 1661441467, 3919563787, 1226247329, 155760684,
                2127556453, 3650610530, 3912925100, 3397709082, 2605399534, 844200203, 1218697875,
                3747822377, 1899143697, 1732546786, 2288016347, 359886996, 3307751257, 2007742513,
                2748636623, 3533690902, 871243358, 3464803523, 4180212238, 3640833128, 1577830398,
                1892628035, 2306472556, 2996392130, 3006231684, 2296266179, 3151580099, 3594248718,
                298545948, 3539272117, 2749678939, 1874404025, 2584227634, 2959840975, 4165861727,
                959382651, 1009550827, 649210999, 3468285293, 3791505115, 3177835926, 1574975458,
                3712971814, 2268845891, 3027469655, 2612516945, 2767217349, 3437723210, 361909048,
                174803142, 1801608370, 2683396447, 4123855575, 3743115358, 855097355, 389967654,
                3466849183, 62496602, 1570756383, 2619549118, 2702719459, 3205603560, 3517585951,
                1001571916, 3914648094, 586411011, 2203219545, 1991805170, 291462253, 2626091906,
                2500429699, 3658425250, 591275011, 3457521561, 1909516648, 1016647432, 1214423814,
                545389916, 2274937717, 4186397493, 2838711218, 2407104593, 3341370919, 2071188669,
                1056326847, 279675352, 3143136760, 663034604, 2337665064, 1068045341, 3288706086,
                1599970362, 1807582654, 3478852750, 3645690411, 634626702, 3016872041, 4009908955,
                1006051491, 2719021150, 1908698847, 1974558407, 1045151170, 81414730, 1422867232,
                4197239354, 1591983466, 1874189107, 3882070116, 471160619, 3538033652, 1325575575,
                3542883207, 2157625989, 3201577383, 609577155, 1394935989, 3283862577, 898081363,
                1638703961, 2459812069, 2143849566, 1692845986, 191429412, 1947558163, 2654605920,
                362944274, 652627228, 3005443786, 1907584792, 2260991562, 2033179708, 517601169,
                1610423429, 3396730858, 3421003047, 2978082438, 1862441924, 3826508713, 1263943479,
                2041961479, 3942582480, 2128168260, 47267024, 4126534310, 3346534361, 3387023656,
                2882502992, 851762018, 2559987094, 633154678, 2834924154, 68763504, 2517021392,
                1787244558, 476241955, 1869261127, 451025182, 1484615367, 1293374777, 849824408,
                125615889, 34383294, 709860222, 3029061471, 1782241004, 2502486226, 3289755139,
                1884331971, 3585604373, 708130179, 3812432338, 3121673423, 336729898, 3212815979,
                1331640570, 368808354, 3927992910, 2583344309, 1604683896, 2377675664, 1432049895,
                2336708810, 1233966810, 106712876, 1665259905, 4193964217, 192648794, 1498181023,
                947378957, 2402512259, 1416057697, 1026447453, 3945411820, 3451452905, 3553837717,
                3947305034, 902601876, 3942899828, 3516283811, 262422749, 2638031869, 712288220,
                199462641, 3599010749, 3787584036, 1126365450, 4070880751, 3865411772, 1658495826,
                3570711854, 1735726921, 119145303, 3163130848, 1291336833, 2146796315, 1896458910,
                3507258381, 3730474410, 647902664, 119220376, 2833167618, 2741203329, 249470903,
                71327613, 1931728642, 2604757487, 1117219288, 711428972, 3428460616, 322366467,
                3243077053, 3378897681, 1403843932, 1991989918, 2413012613, 1365744141, 492078564,
                2676610402, 3773231972, 1970922906, 2318388358, 1112949420, 3487096476, 2360170040,
                3943418017, 3228760458, 3122193412, 3638387262, 1016961461, 2409174780, 1643889695,
                3848556024, 3685824617, 3512748976, 3284143746, 748779869, 2874184074, 1232281025,
                4144896530, 1069350420, 286933366, 416090620, 2657050801, 3108410259, 4063672733,
                2336832052, 4393947, 1331069304, 1668926341, 3265202467, 169918063, 1585538827,
                1657996381, 1427959009, 2747423595, 433894126, 2042216201, 426611464, 2098273364,
                2681925510, 2347419442, 759472315, 358883472, 3566573061, 566399128, 1133339458,
                3913483015, 1985356395, 2890868444, 306415199, 1693603799, 815436612, 4239331360,
                2080083468, 2674607028, 308976991, 562939988, 142614472, 2751524972, 2580927221,
                4040617817, 2258237186, 2077092459, 1424533819, 1818261836, 2255477956, 2463113859,
                3917783, 3196888747, 1250369919, 2128811027, 4135988753, 2855121231, 2751603541,
                1748944916, 606818116, 891282707, 1280813224, 3409874308, 890595934, 3601429200,
                671048320, 3885753853, 2067382323, 3518602897, 4104376817, 3947164920, 2066705712,
                3805972909, 2493699797, 2966924389, 701802948, 2781603469, 971061938, 1383226196,
                3438743891, 755681190, 3291987780, 2657031748, 2157421567, 2518108674, 3181389973,
                225606787, 3198444098, 651201578, 4077819767, 462870400, 1659941395, 3260365109,
                591607061, 3304541751, 3964864687, 3949358441, 156533293, 798441780, 3281785939,
                1484494452, 1302095638, 251493387, 4009163885, 2395098738, 2328212887, 496298918,
                516574034, 2964917507, 76965044, 3856691333, 864137593, 2447187657, 4258071311,
                3755646436, 4248910684, 2663150304, 1237165667,
            ],
            &[
                1984473199, 3256227249, 2455176569, 2099166325, 4149291216, 3802329550, 4268938044,
                841307730, 1835702977, 1996082126, 3547567838, 773843704, 4016216621, 2681973928,
                2642638628, 756126484, 3711891528, 3333768427, 3023107041, 329078158, 3831084881,
                2250143469, 2740854484, 2065364093, 16475968, 1789063882, 90562755, 2395829638,
                105502248, 2464984171, 352362049, 1671925562, 1672656429, 3912672783, 2976521985,
                1857535053, 912101373, 3599178263, 1475622388, 1679421724, 912451536, 1544347804,
                3409297610, 2780624831, 2956890227, 2759200533, 1377434234, 737077784, 3725690282,
                3427153748, 1794622970, 939859254, 2855863085, 647722556, 354344551, 3826763791,
                1345727351, 288135440, 1036229247, 695718102, 815041410, 2006432724, 28242562,
                1749929211, 875993743, 3126551415, 837166301, 3481584796, 3204894742, 540424284,
                2500473276, 2033531364, 3658774875, 1170881648, 1587377703, 3303215635, 2382459946,
                4261489817, 3503640610, 2312174189, 2085919298, 876781300, 3523968595, 3794900435,
                551562034, 277917899, 4049545585, 1148488459, 3049249256, 3028547737, 385184461,
                2327045056, 326335295, 1526099772, 4222288321, 3335089988, 3223345238, 3867286782,
                814275510, 4052015149, 1619520353, 1451459556, 1906956921, 3352694287, 1890686648,
                4263536151, 3670741827, 3921992654, 2432133033, 3137493217, 587639161, 305609045,
                3928594642, 1789779366, 151593889, 3742404497, 2236095179, 1327594808, 168249194,
                3856434419, 1196489310, 3396059561, 647915695, 3822118736, 3541416683, 1851262272,
                3995631059, 3428808225, 4186414337, 2490990084, 2116008544, 3650220767, 1396560346,
                907120910, 1909963938, 4088722647, 3467954950, 4004624286, 1977459563, 3110163371,
                4093373364, 619921165, 3816233891, 1937989728, 3624382539, 2436122875, 544743061,
                1173132218, 2928770999, 1155101491, 439835796, 978535833, 2767377467, 2446840779,
                3121046201, 1239790209, 1427805909, 3673596610, 3945221930, 1197651601, 1456742497,
                177427399, 1394121276, 372545580, 1590879370, 1356252877, 2819023735, 2147545641,
                3662283870, 4040887354, 3677669405, 1738433639, 291377486, 55300961, 4286101333,
                3829985898, 2667081711, 1422260840, 1198139136, 4157283455, 3655872630, 1182240668,
                2228451189, 1436090123, 289825845, 4160160212, 1255112200, 3852671276, 291873274,
                3097152433, 1700751971, 1521673523, 769906108, 100191651, 909759186, 3738396514,
                63815917, 3764078995, 4283954803, 3644348060, 1146825084, 3992578663, 1098803369,
                4234257585, 2499204008, 2806310476, 1805565067, 2148657240, 3603959232, 1973450338,
                304954722, 187699076, 2886666068, 1565668459, 3119264127, 526428678, 3519310907,
                4144387005, 2133914899, 367483308, 438788794, 2018175717, 152755342, 2289694022,
                1623257904, 3632662641, 1438654875, 1957740844, 3866415111, 4036225294, 1540930636,
                777575895, 1164306998, 905593419, 176596270, 1964891555, 435285470, 427518217,
                341067847, 1226878934, 3817779176, 655247797, 1445754505, 3726676144, 2254233196,
                1355909097, 3364332039, 3448860423, 4203238266, 3244898179, 2238989099, 2550902008,
                20701893, 2266649866, 2508429693, 2270048684, 1797834837, 2026596902, 379667603,
                2096274613, 976774677, 2606058996, 225173995, 3819162877, 1852526611, 2880769412,
                1907061042, 1871814156, 1625893023, 1164237684, 405483535, 4277482961, 2315260153,
                1956694410, 2508860583, 2,
            ],
        );
        // - bits == 0 and divide-and-conquer division in limbs_div_to_out_unbalanced
        test(
            &[10; 166],
            &[
                1459165216, 3821473485, 1527833631, 534557042, 2641251684, 3809609559, 3061112419,
                3967227446, 874369607, 3357113542, 172837185, 3063799945, 809083967, 2114991304,
                3652585499, 3870981027, 260839090, 86387882, 3506805298, 3586414637, 1219151546,
                1074937346, 1869327638, 417941536, 3278188390, 2755475894, 732985043, 3207584010,
                452875381, 1770638312, 2520936487, 429394218, 59073091, 2715600775, 2347478058,
                2065207972, 333851993, 2020557144, 2988851968, 3434603645, 3867986743, 620458350,
                4176081514, 687997565, 2000470173, 207628264, 2939063980, 2887037244, 88858000,
                1979351616, 2144759905, 226043352, 2991906207, 2099593474, 388908813, 605491614,
                917110966, 1440029715, 1265986065, 1353710372, 150747576, 4066821351, 2601543228,
                3022790779, 2756513327, 2301300942, 4180697886, 536617044, 1792525146, 302356039,
                2248737864, 1440314786, 2943873916, 194486140, 2460944396, 2163386634, 663134844,
                1064843562, 683691518, 1346625839, 1429039926, 905082151, 1043984157, 3973238222,
                3954748148, 600018136, 45112318, 3802284763, 599423421, 1288822021, 2339077887,
                2053244537, 2215861254, 3570045286, 1668889163, 3129765729, 1881187635, 2099320920,
                1455508634, 3101359109, 2133360041, 4219848842, 3274735980, 1661020165, 1959002083,
                2476803683, 2995302951, 3007969525, 2952785468, 2735292483, 1339083985, 1425599321,
                963296461, 3835062683, 3441026991, 1009908781, 785678562, 362135934, 221588933,
                2902273739, 3436714793, 672263420, 391421925, 3562850020, 1167915739, 1235366482,
                3241596463, 839741578, 1600792632, 1356868232, 1717207350, 1486509543, 609703091,
                1813709463, 1754127387, 103270828, 2294491827, 1977478773, 3581233061, 2076865607,
                1233918144, 468952816, 3796007953, 765537507, 1015431726, 1729447629, 4075863441,
                1166265824, 1653283772, 2675507303, 1805466182, 2868049045, 393643708, 2395539658,
                1405125898, 3106988834, 1449875350, 1921850832, 1103068145, 3501885274, 2708895753,
                433758320, 3993837559, 3767611255, 312060882, 3998790645, 877231469, 2138719115,
                4142414870, 1018473073, 533189076, 3104708022, 1178372066, 2975216060, 3907507630,
                975237132, 386309107, 902266006, 2644385081, 1477475340, 1858200689, 3285099777,
                1132271153, 2053580658, 1125376417, 874041446, 712345397, 1140230609, 538666680,
                881721324, 1111893069, 2501306270, 2274122960, 1357144116, 2166627910, 577081103,
                1122655088, 392081015, 596573533, 3113340265, 3964396084, 1011820872, 1620966576,
                2716227347, 4149946487, 1583709456, 61899924, 4263575338, 2401528621, 3509028054,
                3666511577, 397386472, 969124122, 4008000629, 2718198502, 1418318388, 2934063395,
                2647641395, 1145485096, 2200369228, 1989216706, 2963238435, 1926386286, 2830877782,
                545778532, 1945014519, 1638724297, 622208887, 1112164416, 2022531771, 2867462821,
                599490439, 1876923470, 950997193, 1684774054, 3263116384, 859638073, 1516033581,
                3344695361, 1785593437, 598067909, 2850637811, 2288401518, 1795058327, 4095583229,
                1222559443, 737107737, 651134240, 7438464, 747824109, 3730428362, 2416724451,
                2565046767, 2620912429, 2561442128, 2361206033, 3310915724, 858282568, 2272790867,
                659862547, 461019210, 268119566, 1687038156, 1409316883, 2126896112, 1753824368,
                2573165648, 3765733919, 1487774113, 2831055305, 21121671, 1366044190, 3078238855,
                4195255404, 1496590230, 1626222644, 3171354920, 1518148632, 2538776379, 1154154541,
                1870543847, 2084780047, 603891666, 524384754, 604772818, 2926191905, 3313865705,
                2210821711, 3918703550, 1119015865, 1362962604, 1749827365, 2045857758, 2562923004,
                1460944271, 3737144414, 3632123919, 2354988860, 2895025884, 3090512056, 3436396245,
                3344468871, 136152018, 3202810570, 2877125824, 3677060161, 747702260, 2458095411,
                1835835611, 1903565435, 1773042151, 1229867663, 1051182633, 1551699740, 46990108,
            ],
            &[
                2983822400, 1175377136, 4004483281, 2235564407, 1822513467, 381114796, 280652026,
                2967591791, 3307355161, 2377064394, 385476518, 822689029, 4246167127, 11318802,
                2302723623, 935991327, 3629372113, 2655890047, 2427850107, 3493543220, 1199218706,
                18464511, 3419074626, 131354313, 1681481125, 1916223295, 3786572616, 4255559076,
                2934355035, 1723151624, 3248974311, 3738736372, 1475008151, 786397519, 2078941349,
                3511637043, 626529836, 3727302624, 3446417866, 1056699846, 1674449049, 3597285965,
                2085431198, 1104070989, 2845556388, 3561365655, 3725037602, 1844214597, 938693814,
                3351033209, 2761175913, 4231374306, 1596758609, 839842001, 1569016951, 2417309445,
                1330306602, 1051752810, 136987586, 4191267361, 12875184, 3448410984, 1907451325,
                2277059171, 4081014716, 1109678253, 3098651582, 2013701166, 3694814849, 2213240606,
                915706703, 466995892, 2849324974, 4088671183, 3041537520, 256167934, 2378660697,
                3026916, 3420187595, 3019264354, 2517539357, 3190771438, 3964084208, 464800745,
                2158354996, 2997461723, 1586829196, 626753654, 1190341385, 32972831, 4211614888,
                3666851480, 4151894977, 3598196315, 2868568182, 3468849454, 1834890649, 3931331938,
                1665539267, 344411925, 2706253979, 4175115668, 3417292439, 2920916773, 1978420139,
                2431824169, 986409380, 1888406655, 3938963826, 1560180445, 3202323406, 1414927418,
                4029028281, 3561590815, 3654051825, 2798678208, 317536782, 2586849954, 3375941666,
                2815670947, 2163397988, 2274010858, 3353427146, 3194776320, 1018812088, 248020252,
                2984616735, 2264657095, 883506886, 1203774994, 3281598423, 4002133977, 2951591465,
                1380559214, 313180293, 2977296703, 943333173, 196114172, 4050138623, 716791187,
                3081577264, 2452658991, 1506085116, 96780562, 3815905806, 894431824, 2606307520,
                4085987609, 286762580, 3059401150,
            ],
            &[
                2808057994, 3606496776, 1638284867, 2965092528, 3070561729, 3046780593, 2492165002,
                2058133462, 1411482622, 1386159493, 1930973158, 2741344048, 991334618, 23694448,
                3645114681, 2489826892, 2652335165, 300868672, 2296390812, 1038735413, 1402107292,
                3134939101, 495678723, 1934871653, 2094500280, 2973473339, 930416921, 1063911356,
                1613559983, 607236505, 3394770573, 1595743960, 2977461902, 3135858186, 3804985197,
                2427992839, 40481157, 1194866742, 2665355479, 1208298331, 264923797, 1533531928,
                589823565, 1623330987, 1821021081, 3936785116, 1670988047, 812003305, 186281380,
                2006301434, 1859818525, 2519784381, 502212272, 4074421773, 1011623398, 2394120437,
                1749077131, 358348440, 3091437650, 2632724490, 750658123, 940819777, 1564325991,
                3192289990, 1922442690, 2098791607, 1410727269, 3997475748, 2046568633, 2328724701,
                594683983, 2946935672, 250929021, 3546617499, 3147415082, 2862801627, 1752972798,
                1777319581, 1049723832, 2195152639, 3483521471, 1734219330, 2576240512, 2409422582,
                2679948137, 3770401512, 1061673858, 1725420873, 718139354, 233062852, 2768861085,
                4124913273, 3554269730, 2872541430, 4278481571, 768499669, 4277599530, 3185103338,
                1954279181, 3422846001, 98074718, 551096407, 3681600913, 2349433618, 3210489357,
                2742261282, 4049115855, 914605405, 3971655743, 62350404, 3980249028, 823758998,
                3662208770, 3596057304, 1896028596, 1563432047, 3924435289, 1454939129, 1062311879,
                1403923520, 56670001, 3483882506, 1501725860, 976633875, 2995625644, 3931841873,
                211374720, 1454175215, 3629938299, 2471336363, 4025409672, 91025627, 421183437,
                2654995201, 1631399822, 552933189, 3700683612, 3350958380, 2564370593, 3463421537,
                3068013499, 2815245398, 1236034626, 2411769049, 3471729333, 1807804732, 3870076218,
                3233217373, 2623595623, 3235588553, 1211586912, 106513608, 1606540242, 735139009,
                3646406666, 1157995439, 990423165, 1032637281, 3009147147, 3249314474, 1109145355,
                2938339245, 1916419568, 824070112, 65967478, 0,
            ],
        );
        // - bits == 0 and Barrett division in limbs_div_to_out_unbalanced
        test(
            &[10; 1770],
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                3758096384,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x1fffffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4286578688,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xfffffffc,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0xfffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4286578688,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                461373439,
                0,
                0,
                0,
                0,
                0,
                0,
                128,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294967024,
                u32::MAX,
                4294945791,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                2047,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4292870144,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x1ffff,
                0,
                54525952,
                0,
                402653184,
                0,
                0,
                0xff000000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                44,
                0,
                0,
                0,
                0,
                0,
                0x10000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294836224,
                u32::MAX,
                4287102975,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                4026531839,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x3ffffff,
                0,
                0,
                6,
                0,
                16,
                0,
                0,
                u32::MAX - 1,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                18431,
                0,
                0,
                0,
                0,
                0,
                0x2000000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4232052736,
                u32::MAX,
                1610612735,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                7,
                0,
                2816,
                0,
                0,
                0,
                0,
                4294966272,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                7340031,
                0,
                0,
                0,
                0,
                0,
                0,
                4,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xfffffff9,
                u32::MAX,
                4294967103,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                4095,
                0,
                1310720,
                0,
                0,
                0,
                0,
                4294443008,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                2818572287,
                0,
                0,
                0,
                0,
                0,
                0,
                2048,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294963968,
                u32::MAX,
                4294918143,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x1fffff,
                0,
                603979776,
                0,
                0,
                0,
                0,
                0xf0000000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                239,
                0,
                0,
                0,
                0,
                0,
                0x100000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4293394432,
                u32::MAX,
                4286578687,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x3fffffff,
                0,
                0,
                64,
                0,
                0,
                0,
                0,
                4294967264,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                81919,
                0,
                0,
                0,
                0,
                0,
                0x20000000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                3556769792,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                127,
                0,
                0x7000,
                0,
                0,
                0,
                0,
                4294950912,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                25165823,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294967216,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0xffff,
                0,
                12582912,
                0,
                0,
                0,
                0,
                4286578688,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x7fffffff,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294930432,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x1ffffff,
                0,
                0x40000000,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                256,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xff000000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                3,
                0,
                512,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0x40000000,
                u32::MAX - 1,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                2047,
                0,
                196608,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294966528,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0xfffff,
                0,
                0x4000000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294639616,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x1fffffff,
                0,
                0,
                4,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4160749568,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                63,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xfffffff4,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x7fff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xfffff000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0xffffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xfff00000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1024,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0x80000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0x10000000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                32,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0x4000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0x800000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
            ],
        );
        // - bits != 0 and Barrett division in limbs_div_to_out_balanced
        test(
            &[10; 1401],
            &[123; 2800],
            &[456; 1410],
            &[
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565,
                2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461,
                226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372,
                3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551,
                904203641, 1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192,
                2486560013, 1130254551, 904203641, 1582356372, 3842865475, 1356305461, 226050910,
                3616814565, 2034458192, 2486560013, 1130254551, 904203641, 1582356372, 3842865475,
                1356305461, 226050910, 3616814565, 2034458192, 2486560013, 1130254551, 904203641,
                1582356372, 3842865475, 1356305461, 226050910, 3616814565, 2034458192, 2486560013,
                1130254551, 904203641, 1582356372, 1158510915, 0, 10, 10, 10, 10, 10, 10, 10, 10,
                10, 10,
            ],
        );
        // - bits == 0 and Barrett division in limbs_div_to_out_balanced
        let mut ds = vec![0; 1409];
        ds.push(u32::MAX);
        test(
            &[10; 1401],
            &[123; 2800],
            &ds,
            &[
                170970, 170847, 170724, 170601, 170478, 170355, 170232, 170109, 169986, 169863,
                169740, 169617, 169494, 169371, 169248, 169125, 169002, 168879, 168756, 168633,
                168510, 168387, 168264, 168141, 168018, 167895, 167772, 167649, 167526, 167403,
                167280, 167157, 167034, 166911, 166788, 166665, 166542, 166419, 166296, 166173,
                166050, 165927, 165804, 165681, 165558, 165435, 165312, 165189, 165066, 164943,
                164820, 164697, 164574, 164451, 164328, 164205, 164082, 163959, 163836, 163713,
                163590, 163467, 163344, 163221, 163098, 162975, 162852, 162729, 162606, 162483,
                162360, 162237, 162114, 161991, 161868, 161745, 161622, 161499, 161376, 161253,
                161130, 161007, 160884, 160761, 160638, 160515, 160392, 160269, 160146, 160023,
                159900, 159777, 159654, 159531, 159408, 159285, 159162, 159039, 158916, 158793,
                158670, 158547, 158424, 158301, 158178, 158055, 157932, 157809, 157686, 157563,
                157440, 157317, 157194, 157071, 156948, 156825, 156702, 156579, 156456, 156333,
                156210, 156087, 155964, 155841, 155718, 155595, 155472, 155349, 155226, 155103,
                154980, 154857, 154734, 154611, 154488, 154365, 154242, 154119, 153996, 153873,
                153750, 153627, 153504, 153381, 153258, 153135, 153012, 152889, 152766, 152643,
                152520, 152397, 152274, 152151, 152028, 151905, 151782, 151659, 151536, 151413,
                151290, 151167, 151044, 150921, 150798, 150675, 150552, 150429, 150306, 150183,
                150060, 149937, 149814, 149691, 149568, 149445, 149322, 149199, 149076, 148953,
                148830, 148707, 148584, 148461, 148338, 148215, 148092, 147969, 147846, 147723,
                147600, 147477, 147354, 147231, 147108, 146985, 146862, 146739, 146616, 146493,
                146370, 146247, 146124, 146001, 145878, 145755, 145632, 145509, 145386, 145263,
                145140, 145017, 144894, 144771, 144648, 144525, 144402, 144279, 144156, 144033,
                143910, 143787, 143664, 143541, 143418, 143295, 143172, 143049, 142926, 142803,
                142680, 142557, 142434, 142311, 142188, 142065, 141942, 141819, 141696, 141573,
                141450, 141327, 141204, 141081, 140958, 140835, 140712, 140589, 140466, 140343,
                140220, 140097, 139974, 139851, 139728, 139605, 139482, 139359, 139236, 139113,
                138990, 138867, 138744, 138621, 138498, 138375, 138252, 138129, 138006, 137883,
                137760, 137637, 137514, 137391, 137268, 137145, 137022, 136899, 136776, 136653,
                136530, 136407, 136284, 136161, 136038, 135915, 135792, 135669, 135546, 135423,
                135300, 135177, 135054, 134931, 134808, 134685, 134562, 134439, 134316, 134193,
                134070, 133947, 133824, 133701, 133578, 133455, 133332, 133209, 133086, 132963,
                132840, 132717, 132594, 132471, 132348, 132225, 132102, 131979, 131856, 131733,
                131610, 131487, 131364, 131241, 131118, 130995, 130872, 130749, 130626, 130503,
                130380, 130257, 130134, 130011, 129888, 129765, 129642, 129519, 129396, 129273,
                129150, 129027, 128904, 128781, 128658, 128535, 128412, 128289, 128166, 128043,
                127920, 127797, 127674, 127551, 127428, 127305, 127182, 127059, 126936, 126813,
                126690, 126567, 126444, 126321, 126198, 126075, 125952, 125829, 125706, 125583,
                125460, 125337, 125214, 125091, 124968, 124845, 124722, 124599, 124476, 124353,
                124230, 124107, 123984, 123861, 123738, 123615, 123492, 123369, 123246, 123123,
                123000, 122877, 122754, 122631, 122508, 122385, 122262, 122139, 122016, 121893,
                121770, 121647, 121524, 121401, 121278, 121155, 121032, 120909, 120786, 120663,
                120540, 120417, 120294, 120171, 120048, 119925, 119802, 119679, 119556, 119433,
                119310, 119187, 119064, 118941, 118818, 118695, 118572, 118449, 118326, 118203,
                118080, 117957, 117834, 117711, 117588, 117465, 117342, 117219, 117096, 116973,
                116850, 116727, 116604, 116481, 116358, 116235, 116112, 115989, 115866, 115743,
                115620, 115497, 115374, 115251, 115128, 115005, 114882, 114759, 114636, 114513,
                114390, 114267, 114144, 114021, 113898, 113775, 113652, 113529, 113406, 113283,
                113160, 113037, 112914, 112791, 112668, 112545, 112422, 112299, 112176, 112053,
                111930, 111807, 111684, 111561, 111438, 111315, 111192, 111069, 110946, 110823,
                110700, 110577, 110454, 110331, 110208, 110085, 109962, 109839, 109716, 109593,
                109470, 109347, 109224, 109101, 108978, 108855, 108732, 108609, 108486, 108363,
                108240, 108117, 107994, 107871, 107748, 107625, 107502, 107379, 107256, 107133,
                107010, 106887, 106764, 106641, 106518, 106395, 106272, 106149, 106026, 105903,
                105780, 105657, 105534, 105411, 105288, 105165, 105042, 104919, 104796, 104673,
                104550, 104427, 104304, 104181, 104058, 103935, 103812, 103689, 103566, 103443,
                103320, 103197, 103074, 102951, 102828, 102705, 102582, 102459, 102336, 102213,
                102090, 101967, 101844, 101721, 101598, 101475, 101352, 101229, 101106, 100983,
                100860, 100737, 100614, 100491, 100368, 100245, 100122, 99999, 99876, 99753, 99630,
                99507, 99384, 99261, 99138, 99015, 98892, 98769, 98646, 98523, 98400, 98277, 98154,
                98031, 97908, 97785, 97662, 97539, 97416, 97293, 97170, 97047, 96924, 96801, 96678,
                96555, 96432, 96309, 96186, 96063, 95940, 95817, 95694, 95571, 95448, 95325, 95202,
                95079, 94956, 94833, 94710, 94587, 94464, 94341, 94218, 94095, 93972, 93849, 93726,
                93603, 93480, 93357, 93234, 93111, 92988, 92865, 92742, 92619, 92496, 92373, 92250,
                92127, 92004, 91881, 91758, 91635, 91512, 91389, 91266, 91143, 91020, 90897, 90774,
                90651, 90528, 90405, 90282, 90159, 90036, 89913, 89790, 89667, 89544, 89421, 89298,
                89175, 89052, 88929, 88806, 88683, 88560, 88437, 88314, 88191, 88068, 87945, 87822,
                87699, 87576, 87453, 87330, 87207, 87084, 86961, 86838, 86715, 86592, 86469, 86346,
                86223, 86100, 85977, 85854, 85731, 85608, 85485, 85362, 85239, 85116, 84993, 84870,
                84747, 84624, 84501, 84378, 84255, 84132, 84009, 83886, 83763, 83640, 83517, 83394,
                83271, 83148, 83025, 82902, 82779, 82656, 82533, 82410, 82287, 82164, 82041, 81918,
                81795, 81672, 81549, 81426, 81303, 81180, 81057, 80934, 80811, 80688, 80565, 80442,
                80319, 80196, 80073, 79950, 79827, 79704, 79581, 79458, 79335, 79212, 79089, 78966,
                78843, 78720, 78597, 78474, 78351, 78228, 78105, 77982, 77859, 77736, 77613, 77490,
                77367, 77244, 77121, 76998, 76875, 76752, 76629, 76506, 76383, 76260, 76137, 76014,
                75891, 75768, 75645, 75522, 75399, 75276, 75153, 75030, 74907, 74784, 74661, 74538,
                74415, 74292, 74169, 74046, 73923, 73800, 73677, 73554, 73431, 73308, 73185, 73062,
                72939, 72816, 72693, 72570, 72447, 72324, 72201, 72078, 71955, 71832, 71709, 71586,
                71463, 71340, 71217, 71094, 70971, 70848, 70725, 70602, 70479, 70356, 70233, 70110,
                69987, 69864, 69741, 69618, 69495, 69372, 69249, 69126, 69003, 68880, 68757, 68634,
                68511, 68388, 68265, 68142, 68019, 67896, 67773, 67650, 67527, 67404, 67281, 67158,
                67035, 66912, 66789, 66666, 66543, 66420, 66297, 66174, 66051, 65928, 65805, 65682,
                65559, 65436, 65313, 65190, 65067, 64944, 64821, 64698, 64575, 64452, 64329, 64206,
                64083, 63960, 63837, 63714, 63591, 63468, 63345, 63222, 63099, 62976, 62853, 62730,
                62607, 62484, 62361, 62238, 62115, 61992, 61869, 61746, 61623, 61500, 61377, 61254,
                61131, 61008, 60885, 60762, 60639, 60516, 60393, 60270, 60147, 60024, 59901, 59778,
                59655, 59532, 59409, 59286, 59163, 59040, 58917, 58794, 58671, 58548, 58425, 58302,
                58179, 58056, 57933, 57810, 57687, 57564, 57441, 57318, 57195, 57072, 56949, 56826,
                56703, 56580, 56457, 56334, 56211, 56088, 55965, 55842, 55719, 55596, 55473, 55350,
                55227, 55104, 54981, 54858, 54735, 54612, 54489, 54366, 54243, 54120, 53997, 53874,
                53751, 53628, 53505, 53382, 53259, 53136, 53013, 52890, 52767, 52644, 52521, 52398,
                52275, 52152, 52029, 51906, 51783, 51660, 51537, 51414, 51291, 51168, 51045, 50922,
                50799, 50676, 50553, 50430, 50307, 50184, 50061, 49938, 49815, 49692, 49569, 49446,
                49323, 49200, 49077, 48954, 48831, 48708, 48585, 48462, 48339, 48216, 48093, 47970,
                47847, 47724, 47601, 47478, 47355, 47232, 47109, 46986, 46863, 46740, 46617, 46494,
                46371, 46248, 46125, 46002, 45879, 45756, 45633, 45510, 45387, 45264, 45141, 45018,
                44895, 44772, 44649, 44526, 44403, 44280, 44157, 44034, 43911, 43788, 43665, 43542,
                43419, 43296, 43173, 43050, 42927, 42804, 42681, 42558, 42435, 42312, 42189, 42066,
                41943, 41820, 41697, 41574, 41451, 41328, 41205, 41082, 40959, 40836, 40713, 40590,
                40467, 40344, 40221, 40098, 39975, 39852, 39729, 39606, 39483, 39360, 39237, 39114,
                38991, 38868, 38745, 38622, 38499, 38376, 38253, 38130, 38007, 37884, 37761, 37638,
                37515, 37392, 37269, 37146, 37023, 36900, 36777, 36654, 36531, 36408, 36285, 36162,
                36039, 35916, 35793, 35670, 35547, 35424, 35301, 35178, 35055, 34932, 34809, 34686,
                34563, 34440, 34317, 34194, 34071, 33948, 33825, 33702, 33579, 33456, 33333, 33210,
                33087, 32964, 32841, 32718, 32595, 32472, 32349, 32226, 32103, 31980, 31857, 31734,
                31611, 31488, 31365, 31242, 31119, 30996, 30873, 30750, 30627, 30504, 30381, 30258,
                30135, 30012, 29889, 29766, 29643, 29520, 29397, 29274, 29151, 29028, 28905, 28782,
                28659, 28536, 28413, 28290, 28167, 28044, 27921, 27798, 27675, 27552, 27429, 27306,
                27183, 27060, 26937, 26814, 26691, 26568, 26445, 26322, 26199, 26076, 25953, 25830,
                25707, 25584, 25461, 25338, 25215, 25092, 24969, 24846, 24723, 24600, 24477, 24354,
                24231, 24108, 23985, 23862, 23739, 23616, 23493, 23370, 23247, 23124, 23001, 22878,
                22755, 22632, 22509, 22386, 22263, 22140, 22017, 21894, 21771, 21648, 21525, 21402,
                21279, 21156, 21033, 20910, 20787, 20664, 20541, 20418, 20295, 20172, 20049, 19926,
                19803, 19680, 19557, 19434, 19311, 19188, 19065, 18942, 18819, 18696, 18573, 18450,
                18327, 18204, 18081, 17958, 17835, 17712, 17589, 17466, 17343, 17220, 17097, 16974,
                16851, 16728, 16605, 16482, 16359, 16236, 16113, 15990, 15867, 15744, 15621, 15498,
                15375, 15252, 15129, 15006, 14883, 14760, 14637, 14514, 14391, 14268, 14145, 14022,
                13899, 13776, 13653, 13530, 13407, 13284, 13161, 13038, 12915, 12792, 12669, 12546,
                12423, 12300, 12177, 12054, 11931, 11808, 11685, 11562, 11439, 11316, 11193, 11070,
                10947, 10824, 10701, 10578, 10455, 10332, 10209, 10086, 9963, 9840, 9717, 9594,
                9471, 9348, 9225, 9102, 8979, 8856, 8733, 8610, 8487, 8364, 8241, 8118, 7995, 7872,
                7749, 7626, 7503, 7380, 7257, 7134, 7011, 6888, 6765, 6642, 6519, 6396, 6273, 6150,
                6027, 5904, 5781, 5658, 5535, 5412, 5289, 5166, 5043, 4920, 4797, 4674, 4551, 4428,
                4305, 4182, 4059, 3936, 3813, 3690, 3567, 3444, 3321, 3198, 3075, 2952, 2829, 2706,
                2583, 2460, 2337, 2214, 2091, 1968, 1845, 1722, 1599, 1476, 1353, 1230, 1107, 984,
                861, 738, 615, 492, 369, 246, 123, 0, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
            ],
        );
        // - bits != 0 and Barrett division in limbs_div_to_out_unbalanced
        test(
            &[10; 1770],
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                3758096384,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x1fffffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4286578688,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xfffffffc,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0xfffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4286578688,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
            ],
            &[
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                461373439,
                0,
                0,
                0,
                0,
                0,
                0,
                128,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294967024,
                u32::MAX,
                4294945791,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                2047,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4292870144,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x1ffff,
                0,
                54525952,
                0,
                402653184,
                0,
                0,
                0xff000000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                44,
                0,
                0,
                0,
                0,
                0,
                0x10000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294836224,
                u32::MAX,
                4287102975,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                4026531839,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x3ffffff,
                0,
                0,
                6,
                0,
                16,
                0,
                0,
                u32::MAX - 1,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                18431,
                0,
                0,
                0,
                0,
                0,
                0x2000000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4232052736,
                u32::MAX,
                1610612735,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                7,
                0,
                2816,
                0,
                0,
                0,
                0,
                4294966272,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                7340031,
                0,
                0,
                0,
                0,
                0,
                0,
                4,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xfffffff9,
                u32::MAX,
                4294967103,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                4095,
                0,
                1310720,
                0,
                0,
                0,
                0,
                4294443008,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                2818572287,
                0,
                0,
                0,
                0,
                0,
                0,
                2048,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294963968,
                u32::MAX,
                4294918143,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x1fffff,
                0,
                603979776,
                0,
                0,
                0,
                0,
                0xf0000000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                239,
                0,
                0,
                0,
                0,
                0,
                0x100000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4293394432,
                u32::MAX,
                4286578687,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x3fffffff,
                0,
                0,
                64,
                0,
                0,
                0,
                0,
                4294967264,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                81919,
                0,
                0,
                0,
                0,
                0,
                0x20000000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                3556769792,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                127,
                0,
                0x7000,
                0,
                0,
                0,
                0,
                4294950912,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                25165823,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294967216,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0xffff,
                0,
                12582912,
                0,
                0,
                0,
                0,
                4286578688,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x7fffffff,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294930432,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x1ffffff,
                0,
                0x40000000,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                256,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xff000000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                3,
                0,
                512,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0x40000000,
                u32::MAX - 1,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                2047,
                0,
                196608,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294966528,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0xfffff,
                0,
                0x4000000,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4294639616,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x1fffffff,
                0,
                0,
                4,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                4160749568,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                63,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xfffffff4,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0x7fff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xfffff000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                0xffffff,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0xfff00000,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                u32::MAX,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1024,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
     