// Copyright © 2025 Mikhail Hogrefe
//
// This file is part of Malachite.
//
// Malachite is free software: you can redistribute it and/or modify it under the terms of the GNU
// Lesser General Public License (LGPL) as published by the Free Software Foundation; either version
// 3 of the License, or (at your option) any later version. See <https://www.gnu.org/licenses/>.

use malachite_base::num::arithmetic::traits::{BinomialCoefficient, DivExact, Gcd};
use malachite_base::num::basic::traits::{One, Zero};
use malachite_base::num::conversion::traits::ExactFrom;
use malachite_base::test_util::generators::common::GenConfig;
use malachite_base::test_util::generators::{unsigned_pair_gen_var_28, unsigned_pair_gen_var_44};
use malachite_nz::natural::Natural;
use malachite_nz::natural::arithmetic::binomial_coefficient::*;
use malachite_nz::platform::Limb;
use malachite_nz::test_util::generators::{
    natural_gen, natural_gen_var_2, natural_pair_gen_var_15, unsigned_pair_gen_var_45,
    unsigned_pair_gen_var_46, unsigned_pair_gen_var_47, unsigned_pair_gen_var_48,
    unsigned_pair_gen_var_49,
};
use malachite_nz::test_util::natural::arithmetic::binomial_coefficient::*;
use std::str::FromStr;

#[cfg(not(feature = "32_bit_limbs"))]
#[test]
fn test_limbs_binomial_coefficient_limb_limb_bdiv() {
    fn test(n: Limb, k: Limb, out: &[Limb]) {
        let xs = limbs_binomial_coefficient_limb_limb_bdiv(n, k);
        assert_ne!(*xs.last().unwrap(), 0);
        assert_eq!(xs, out);
        assert_eq!(
            Natural::from_owned_limbs_asc(xs),
            Natural::binomial_coefficient(Natural::from(n), Natural::from(k))
        );
    }
    // - k_max == 0
    // - ones == 0
    test(26, 26, &[1]);
    // - ones != 0
    test(69, 32, &[10356137869959167254, 2]);
    // - k_max != 0
    test(
        364,
        227,
        &[
            13299750156278230008,
            15204945566537371181,
            1343180567794975186,
            14635289525631461779,
            10747969514869510295,
            9942168,
        ],
    );
    test(
        1000,
        500,
        &[
            2548782591045708352,
            2287006466848960994,
            434054286371261995,
            9307092176803372215,
            1234012513444138811,
            6453284806000209963,
            9510906063553676151,
            16379477204911690877,
            4813923858496575229,
            4008028500528486217,
            13325564105666315328,
            7088459241230905245,
            10525434632971158032,
            12617275459030976558,
            17868910957880245731,
            27735200797,
        ],
    );
}

#[cfg(not(feature = "32_bit_limbs"))]
#[test]
fn test_limbs_binomial_coefficient_limb_limb_small_k() {
    fn test(n: Limb, k: Limb, out: &[Limb]) {
        let xs = limbs_binomial_coefficient_limb_limb_small_k(n, k);
        assert_ne!(*xs.last().unwrap(), 0);
        assert_eq!(xs, out);
        assert_eq!(
            Natural::from_owned_limbs_asc(xs),
            Natural::binomial_coefficient(Natural::from(n), Natural::from(k))
        );
    }
    // - nmax >= k
    test(2, 2, &[1]);
    // - nmax < k
    // - numfac == 0
    // - shift != 0 in limbs_hensel_div_limb
    test(9, 9, &[1]);
    // - numfac != 0
    test(17, 17, &[1]);
    // - shift == 0 in limbs_hensel_div_limb
    test(836, 7, &[55217369185858880]);
    test(
        1000,
        25,
        &[4493608106319329376, 18354803010869864610, 140006850684],
    );
}

#[cfg(not(feature = "32_bit_limbs"))]
#[test]
fn test_limbs_binomial_coefficient_limb_limb_basecase() {
    fn test(n: Limb, k: Limb, out: Limb) {
        assert_eq!(limbs_binomial_coefficient_limb_limb_basecase(n, k), out);
        assert_eq!(
            Natural::from(out),
            Natural::binomial_coefficient(Natural::from(n), Natural::from(k))
        );
    }
    test(10, 5, 252);
    test(67, 65, 2211);
}

#[cfg(not(feature = "32_bit_limbs"))]
#[test]
fn test_limbs_binomial_coefficient_limb_limb_small_k_divide_and_conquer() {
    fn test(n: Limb, k: Limb, out: &[Limb]) {
        let xs = limbs_binomial_coefficient_limb_limb_small_k_divide_and_conquer(n, k);
        assert_ne!(*xs.last().unwrap(), 0);
        assert_eq!(xs, out);
        assert_eq!(
            Natural::from_owned_limbs_asc(xs),
            Natural::binomial_coefficient(Natural::from(n), Natural::from(k))
        );
    }
    // - !BIN_UIUI_RECURSIVE_SMALLDC || hk <= ODD_FACTORIAL_TABLE_LIMIT as Limb
    // - n <= ODD_FACTORIAL_EXTTABLE_LIMIT as Limb
    test(27, 25, &[351]);
    // - n > ODD_FACTORIAL_EXTTABLE_LIMIT as Limb
    // - !BIN_UIUI_RECURSIVE_SMALLDC || k <= ODD_FACTORIAL_TABLE_LIMIT as Limb
    test(
        10000,
        25,
        &[12769553919776416000, 9951314237341865820, 6910797670090152703, 99667982199316897],
    );
    // - BIN_UIUI_RECURSIVE_SMALLDC && hk > ODD_FACTORIAL_TABLE_LIMIT as Limb
    test(54, 52, &[1431]);
    // - BIN_UIUI_RECURSIVE_SMALLDC && k > ODD_FACTORIAL_TABLE_LIMIT as Limb
    test(93, 51, &[10325892988062052036, 28774313]);
}

#[cfg(not(feature = "32_bit_limbs"))]
#[test]
fn test_limbs_binomial_coefficient_limb_limb_goetgheluck() {
    fn test(n: Limb, k: Limb, out: &[Limb]) {
        let xs = limbs_binomial_coefficient_limb_limb_goetgheluck(n, k);
        assert_ne!(*xs.last().unwrap(), 0);
        assert_eq!(xs, out);
        assert_eq!(
            Natural::from_owned_limbs_asc(xs),
            Natural::binomial_coefficient(Natural::from(n), Natural::from(k))
        );
    }
    // - prod <= max_prod first time
    // - ma < mb first time
    // - a >= prime first time
    // - ma >= mb first time
    // - a < prime first time
    // - sieve[index] & mask == 0 first time
    // - prod <= max_prod second time
    // - ma >= mb second time
    // - a >= prime second time
    // - ma < mb second time
    // - a < prime second time
    // - i <= max_i first time
    // - sieve[index] & mask != 0 first time
    // - i > max_i first time
    // - sieve[index] & mask != 0 second time
    // - i <= max_i second time
    // - sieve[index] & mask == 0 second time
    // - n % prime < k % prime
    // - prod <= max_prod third time
    // - n % prime >= k % prime
    // - prod > max_prod third time
    // - i > max_i second time
    // - sieve[index] & mask != 0 third time
    // - i <= max_i third time
    // - sieve[index] & mask == 0 third time
    // - prod <= max_prod fourth time
    // - prod > max_prod fourth time
    // - i > max_i third time
    // - j != 0
    test(
        1024,
        512,
        &[
            9401457825549664838,
            2721813090096631178,
            14255346604713632433,
            3580037932091556548,
            1754384789820018233,
            2635023193750144437,
            11050518185659523387,
            12964790596359511527,
            1445883973524779011,
            12061501847736594246,
            3370187743259284348,
            6342914377781736300,
            5825027960880516300,
            13264895533447765242,
            4401514467996618027,
            459836855626902435,
        ],
    );
    // - prod > max_prod second time
    test(
        1045,
        519,
        &[
            5718966217258963792,
            18109692964617656624,
            12203044925148468153,
            12014411658199454757,
            12073183713056964826,
            10217774024310452016,
            5834695290287326173,
            15738609591983585192,
            1510425106975833791,
            1194950807760887936,
            13372185138508722787,
            1955712450906595670,
            3886453007529083390,
            2635264355471572346,
            11248337219650472692,
            12482562081008645864,
            50551,
        ],
    );
}

#[test]
fn test_binomial_coefficient_limb_limb() {
    fn test(n: Limb, k: Limb, out: &str) {
        let x = binomial_coefficient_limb_limb(n, k);
        assert!(x.is_valid());
        assert_eq!(x.to_string(), out);
        assert_eq!(
            x,
            Natural::binomial_coefficient(Natural::from(n), Natural::from(k))
        );
    }
    // - n >= k
    // - k == 0
    test(0, 0, "1");
    test(1, 0, "1");
    test(1, 1, "1");
    test(2, 0, "1");
    // - k == 1
    test(2, 1, "2");
    test(2, 2, "1");
    test(3, 0, "1");
    test(3, 1, "3");
    test(3, 2, "3");
    test(3, 3, "1");
    test(4, 0, "1");
    test(4, 1, "4");
    // - using limbs_binomial_coefficient_limb_limb_basecase
    test(4, 2, "6");
    test(4, 3, "4");
    test(4, 4, "1");
    // - n < k
    test(1, 2, "0");
    test(10, 5, "252");
    // - using limbs_binomial_coefficient_limb_limb_small_k_divide_and_conquer
    test(100, 50, "100891344545564193334812497256");
    // - using limbs_binomial_coefficient_limb_limb_small_k
    test(68, 2, "2278");
    // - using limbs_binomial_coefficient_limb_limb_bdiv
    test(
        557,
        270,
        "12303897123684661862136414153519313173927533052423815769743887086897271613639625388778793\
        435582088957595545453390198975003546712921290727359277922023155427564241544480",
    );
    // - using limbs_binomial_coefficient_limb_limb_goetgheluck
    test(
        1520,
        604,
        "75854072694680390545636184213535932745981366536099293315554162782484973010323284519943551\
        805370379122919525568608395918396659326939152960648811048376751248471027979484001171948494\
        104610240152756095326405688057789446682785317328585217327221577095099162137370467209392535\
        545646923979679739966748638562974802230089647823656716367175837356234669272446104178944990\
        64852838291828963125461234805108096610500907519032849968173861190511519206171866000",
    );
}

#[test]
fn test_binomial_coefficient() {
    fn test(n: &str, k: &str, out: &str) {
        let n = Natural::from_str(n).unwrap();
        let k = Natural::from_str(k).unwrap();
        let b = Natural::binomial_coefficient(n.clone(), k.clone());
        assert!(b.is_valid());
        assert_eq!(b.to_string(), out);

        let b_alt = Natural::binomial_coefficient(&n, &k);
        assert!(b_alt.is_valid());
        assert_eq!(b_alt, b);

        assert_eq!(binomial_coefficient_naive_1(n.clone(), k.clone()), b);
        assert_eq!(binomial_coefficient_naive_2(n.clone(), k.clone()), b);
        if let Ok(k) = u32::try_from(&k) {
            assert_eq!(rug::Integer::from(&n).binomial(k).to_string(), out);
        }
    }
    // - k <= n
    // - k == 0 || n == k
    test("0", "0", "1");
    test("1", "0", "1");
    test("1", "1", "1");
    test("2", "0", "1");
    // - k != 0 && n != k
    // - double_cmp(&k, &n) != Greater
    // - k < 2 in binomial_coefficient_helper
    test("2", "1", "2");
    test("2", "2", "1");
    test("3", "0", "1");
    test("3", "1", "3");
    // - double_cmp(&k, &n) == Greater
    test("3", "2", "3");
    test("3", "3", "1");
    test("4", "0", "1");
    test("4", "1", "4");
    // - k >= 2 && Limb::convertible_from(&n) in binomial_coefficient_helper
    test("4", "2", "6");
    test("4", "3", "4");
    test("4", "4", "1");
    // - k > n
    test("1", "2", "0");
    test("10", "5", "252");
    test("100", "50", "100891344545564193334812497256");
    // - k >= 2 && !Limb::convertible_from(&n) in binomial_coefficient_helper
    // - k.even() first time in binomial_coefficient_raising_factorial_4
    // - k <= Limb::power_of_2(Limb::WIDTH >> 1) in binomial_coefficient_hmul_nbnpk
    // - k.odd() second time in binomial_coefficient_raising_factorial_4
    // - r == 0 first time in binomial_coefficient_raising_factorial_4
    // - k != 0 in binomial_coefficient_raising_factorial_4
    // - r != 0 second time in binomial_coefficient_raising_factorial_4
    // - k > 1 in binomial_coefficient_raising_factorial_4
    // - k - lk < 5 in binomial_coefficient_raising_factorial_4_rec
    // - k <= lk in binomial_coefficient_raising_factorial_4_rec
    test(
        "1000000000000000000000000",
        "10",
        "27557319223985890652556079144620811287477954168595679012345679012345418595679012345679012\
        347422646604938271604938264183063271604938271604958214285714285714285714253397817460317460\
        3174603457499999999999999999999900000000000000000000000",
    );
    test(
        "1000000000000000000000000",
        "999999999999999999999990",
        "27557319223985890652556079144620811287477954168595679012345679012345418595679012345679012\
        347422646604938271604938264183063271604938271604958214285714285714285714253397817460317460\
        3174603457499999999999999999999900000000000000000000000",
    );
    // - k.odd() first time in binomial_coefficient_raising_factorial_4
    // - k.even() second time in binomial_coefficient_raising_factorial_4
    // - k - lk >= 5 in binomial_coefficient_raising_factorial_4_rec
    // - k > lk in binomial_coefficient_raising_factorial_4_rec
    test(
        "25694718637547264792682404365646",
        "77",
        "24892107841828531181241909544492918596306121771031452631201178559919761115634737951594564\
        134356322099422004184733720911624149235259560430254876738408767931226785921407370175095156\
        586781390423037037354806763474907080465964805824615654538038771653753950919821491697271091\
        377714045381616063058884296191134488706691089783145326802624153519283791876914093506515827\
        833506442633078910514277387531336146210345320293586400210472321425351318080461247076886974\
        768698454873923739768679830132931625134235003968808918478243378730618857978191369012739961\
        051581373447851611688649593583701011088238159758896124868328792895820245726080012240623605\
        641788102744035650422323176408348710487482498610290424711478789115694916288138443827673675\
        370569473426454096853986388197987217558182898826734752119147506194959212385238127156463909\
        327557739453207008435734881763952972072877981513820560067736400661554297872808149726929774\
        009647173110901034073815261251706646282610156215143897705316902472221999807813890716324389\
        059130071475748736042259427029862564233394643788711602637944768761312353268539463729212269\
        415045191305842571853121600306112590657683014528748487348383916892187205977769846675553991\
        678198468541000193638965662990392249005490587691945513633072925439584423085568573063324438\
        190079886080774771402863370193790234632425970224213478091984184852108551225355019635354056\
        486480357281956498614205484617578531125055573838944472215270078882558231706258591409870228\
        048336579336293173152190873375502722426036746069862292026870602177622921149409743274336566\
        057103504313000613515998137450390738545271211718490829689241758733404923221326412878356941\
        435514837492022527605671586407700532750514057135146338845230795592073866772571401329026443\
        608142817184754893131076291570078101560304351288697491914877440785372806706691123696040114\
        923681774613230825471864285137036389227443209862850919338428947675401504411181949854055775\
        200409538150678687818266389944970852915636139634528094736089467929779413965463990220175083\
        710210323999501861734841899661962324188839025051105910476408529885881803658183792799548436\
        079055522933471249257212221461038082674897327598629096995991431788007725162706786103836085\
        868101229756394404428127938624116484159208824390263198961814488490969603019941815506011834\
        656564995524746083712970354462072408140254406415763673750",
    );
    test(
        "18446744073709551615",
        "158",
        "55895522534362171876679545770571780830932656730029153560869974856994653843802569288729672\
        382637511859126481647849454205948292497053990755261689981054438192648753626841630309709195\
        818046531987560211901650810864524918303657683264406997638322847538595795406557688836851113\
        964170617643288481686744802130922248594817017440419821564431902277641567516581995890812918\
        486722065525008580823765669188839888456062057395467503086550870415427259131515369395143321\
        954047801400987772358011644373722286582971360737319736278747048729130702410986775490918876\
        230529996180462561974695082885810128945711270133861913341366432968582985806434359568905664\
        514713947374000003710986140313054418069500150430587294188316434107034410813333644810868109\
        517010015342588756758001231909300692391419214678692034627983296534319977780017419329875838\
        305344927801449612412534953383999840286057067187077005869924788433675067369851522793235730\
        315737008337149771567919123016352557348492008964898963676372074751843928582292062287972645\
        002479275926607204375396328813893195279046674831290420168619360957160719607118724011337751\
        884687312078496400868136122166666403111874431469083483195561561046860576791948708097239448\
        707700121141752690164054210305897535632910540849799510508361519672108494930297513957744916\
        541405888702155034069578507345846043015833055785496095916102636242742597283956473922401079\
        208727407330832207583179935719542716898842632948842003593449284004534387840878675132405353\
        545617559957750742653148569486728382115360807684946620305181043444563683398002670423056641\
        556205884154456331403885923859681242657192316862584038348512448080592213086456602118302523\
        497600139305650208731488402477762993305561030251065134570006992415665014910718509829224625\
        872159298105076966017786257520264468203354024508924179560925218008493771963278229045367865\
        887776645263717204778132848971659200666442585918134220062630814282240784092582922877337752\
        874525729691815214017623817845517096588696914035779725717257987350294068144824840587807588\
        357158318464027597598240198947245829267827323307370357782705746909925183438673236245050493\
        849326967187419243980545566285650601408510053072445047817977219500403018557783113306208827\
        167431080889349651649509675460140965485298062146167041924877070444810030309666624075227137\
        218638063690047219457567724110789914580071868854478163169581354672185562960327685598076336\
        690378854689193418551369074938423269725145368510358579999339101382841905493888240214883154\
        842774151093965701354532381623493383136070753785526556095815078903379059184394232509103407\
        445242908548214164343171473652463157048465973199937898538847219544675249899299702767453227\
        701439662328156741963625896736861699341091781695881895814781993228295591405890355373308533\
        69629590890184209436859538990383412624370182464931679935821512705",
    );
    test(
        "18446744073709551615",
        "67",
        "17975145518991061179479323219129935574158608347154628002363871092506776489199109934521955\
        123099550607217011648682834882233989613437897676528916878068374745801854526845695228287766\
        789754958180819896635236168923015045618525205927314379144644668574770271895365790241559003\
        144822843889333362041474551859621080410036652606791515505353369554207762483251871851266970\
        526017855527470363796741702469836770390720021847155096242244011610914819823947147149757369\
        963787213725621805262787399500529025447928970023245860150872609280717916258813202603295120\
        573904776695107485006469437484809405204869700461623288629055326810940902543920025198724784\
        512861425829561074459250975049818414179892103155718839794288796334197669645772310664090067\
        383176579762499437658976815731335276585886004562625663405112281286580831287539250796636410\
        499555980578478064777424504455782233056592703195498279137626891735967240326084691333116555\
        048351036083482033114827059239588624846987544228990882068278651916209911673445783701818463\
        605412513547801208468857612548719031059286380708408871334262084700310978668806235603467446\
        875803132012584698568224321090559637699882304716272182566747176011138741513605154927341734\
        8337767036988850446297726975",
    );
    // - r == 0 second time in binomial_coefficient_raising_factorial_4
    test(
        "64794141400349148426125344064038245668210",
        "256",
        "66031965476204798455146874331523684080813362175686018331031515911090977873910155832605365\
        024088507121468974811846414037841974044761919328109003377289871853511171575036733766570594\
        891453274802950544969704784506297953600869073325018040500075322154584368142810932546171082\
        852158395260664371405927007989709292505767982485700717462755039001448808389259756963264278\
        906088556027912417831893433509962336148107621718179651346831471790176401806766480380429889\
        091161537666615007387280496039863931909142998822738027792169782639238146079132322198094998\
        802486850137067671945865080411439029612107336301348639088595876320707598941419394424283157\
        143766976110612717751687095427384089847206769596670568078428576165333416386232441693390109\
        663643281983434582393770222667816544269262132334616225677948861772526297326474869185995248\
        147223996638098457580467492870356321781996013113726822987288450823900111773163205636973304\
        380289800405883789689472988397794249318636225811197198716377966054543346740613593561974377\
        093009085172518201919751004048922207546914508128933773326960672960593784186495358724918479\
        796985664038589538200615872665489843329713567750689491887312970295325573546986255121402411\
        084560066792113137699614374211253083694137958737734257446884558418163050219974644457688081\
        676873402993322999032939870868467432529157509565329259639753569836216746833599315763759693\
        829765447474480250973648293550957370797807649367827793436798113040510064141170940749382465\
        206220269485857516368112061908007860073237682581918967583093557269558058823518949817468789\
        084216644967899755535494281973068520476114970926637305610945283948226717762805241122575066\
        678103015124110768502924767744781460648409786391934340118093908438148276389461167601820750\
        057069093773693074665652175457980958723759516515717047910383569619901407819068208673474699\
        453686471361006890630023875323558303493400544302118291489329274887331045875348205800862710\
        175052394281227458757959196016039413945472225266077758335879434496501644524406760148503536\
        777844536674510188616914664398525551534555340932552542038044113796537414101079613456296985\
        563991419519025675361727124584048105966929210844853432544846497675490471504969416848715119\
        092439521396484249151715675927492341087857671196859845186304379535276682803519065353324314\
        753738064545044955708632071615933033701914916065271501916712011810342168742156422262920787\
        990320364401380933898662355120884340792786670753109271316345859012462545725984191058886315\
        780390631728366246215522172383402706101501675956766022277331224839584378620731356043496863\
        785541271891585740098376406528987886519218563576267816719725054923546357372040525223848949\
        127753974390320001547914359669712714875205242660997027993020756142973623600187114692661037\
        463223091304396021296078235892866862323792231314445721295667474988263893282253274196685780\
        474772501116806432261201824061161163113862619883103396549652499049127008949545703101415021\
        959323424435960397307451027052471053244276197436645437020963781472397247593591077373074717\
        094053719991571689422435129889293085327836165576103011597448058754151829033639997737893083\
        842550015423824149716583089263434540508738377195408947652432376611378742646557728539254326\
        476152890041397510536077837901277246692068153800616280367894133910883810668466983308213587\
        221482938745562008958936847061728216788782564518434418288722461230400118273167234421901949\
        895714204179305818559684033777442551682966339667887715092224700434706792933312052909488102\
        079842621982456264706730931310278553720716127067281236666033604620046598324107408026680346\
        412372599646940552339198684588489969280418636356788664147669690402699465461172213740285470\
        809128728317682820264186890847420093051962425795653283884719000382298889430710471760502027\
        229103999961325795946767445375287269359953381217519871307479690576346343309759003193210071\
        556418039142564126112040263001368700411929992559256249934586198220607434498304226202009811\
        851202657216426523574148350721115772654468971702946249623853433692813341850096864575825766\
        875327019841615467863988595861381027218305979988212099705711032765590105239080603592671851\
        263007878964971854611940675180372780784561628267671888044540380421481565399877355853257247\
        904912714553488274268173076162924936684919527723389714147069548613898253656186458224176096\
        650679656664121232090301986651148027643907951818914611914284611587891731441594791289668499\
        732443965790127087948406108626423198713173029654246468017240080444863841518378912082628643\
        724261542806751229219083492033570837867883135531192455131856234586805554416472640371021449\
        626153219324461765285594058845200904698148046455373363116514572144638111797423237086355564\
        247860048288380294928679328053911481495367259163199843147875393924983826306330183640755507\
        721631146048737226709507724059249592648982425588036008000118061247531554850396149937729527\
        930090878598591797914711906297196260037210666173426412078722356587377867788754800643764289\
        349492930056977738160467345399517207696550782762419615728732412876581046942100815806171450\
        348045418018196938450493961876960731452162424045420483182667480966924113555262743123695828\
        842908617606474347446639604774874895056539625426950012998315670654871344246635345053749863\
        496640365576291398308932572744385646669612244790159569747800709995170292663386348773622916\
        561134395120135530744456556349965822317543312742948102629799123576739818344673647025657695\
        985344238813907956117069517497661033396357007152583146264519158459022423403393518704274469\
        555988017487304952556931240339272093145564884487180943346829587730987339974521536847163156\
        576792360583613136094081709906550031136279843694368596413923678731949271836013710356271449\
        750444414580685902236336967295155735991873355394004365947388684449125129801766071721100317\
        754267126693147225832067659102896333959362151819898227685846445795875977111413552941797422\
        553933826223370170202722743530536102426573846057258556374829036318491118228532586723711507\
        306175083430903562551070805871091783816943579841868012387755648345966129650056499655937025\
        786637878092663953415315273277401554564452187856322467035501932551411544392181881298986961\
        977669220229531562675556039911519639525744745226708275593777931260713611128737604054948983\
        139973224611064054789667235394810524925200258714854048451678007812968231907618644701400069\
        416728442304285423001695542417198763520331552551177941831892213963517136796389994665927604\
        459301587378625571860379754039201572427152553572980209384382435194093360849052249292611790\
        503999041541116237688173853723809560117304453500860053652419055303646548732887211219816022\
        640151501792454343959659164109351416330947070094808894238884111531377647551954029108954865\
        820855989915402340973344448073147690851869968726931741577360043521731876511950237869229333\
        859602928143413880913510506371294577002903467954358083654964723356629201256978479953248587\
        837494474562791991946166010566508192581355676686351165394433654073840991635192074086973572\
        163621004682262040173522080229706241781899722662323148146556857674870483185715528972858490\
        341376734230716934752618524152956020461673836080774036785780214798629565634422775934543353\
        476894063593324705839019506068594012535084480289279779876445412055235366951246546684773766\
        212435536276729263154137776171656339214503240937372959067826975944390909279310279488186009\
        284938438825832468869538671600734260099608765379746524963643065452430188106594098479755797\
        093870460951873974971714697637666625870945820997332659684169379272678395704589081533814498\
        140005779826762539275887677973897726706795887769141977524446776813526719526578593420736712\
        954032634210661657840530040608510833332863779404436297002475234120661252662270361338493492\
        578189610080628765357649189060318248893668432942439239400104110762226338091351054740301228\
        267114030372038914676357083876933841022019867273340907399707064313910406654589321612531521\
        602233355695432124690159097553479392454292250308539955098369447489417568755029977263235806\
        737535494742497503685663730366012718441966449290261821980738958389669746636037465845741790\
        748315986037312254641500834573322524472364659503566291792471608054787771836144999713040748\
        466357834461871406015310449476989294914019751283337432367578854283638114561957941268079431\
        890173027032456336882142230048504806653501455520488580583506078652051408716252946680955540\
        372281906409585858036827184877008395810459992480127401354027016126387228520620234335154464\
        821713911693946238556003001085542056042701510153072679687977939864130493222714210229201112\
        642263071677946070197883792071982433883297230356359448021422941875047191241501644623057173\
        683380213277548600830552213704456922714478433917536416960698247481747286784162314893055284\
        885473180947515913242742950779558619216152807571397814191898554257601274089785855209288639\
        494478547781184010543598204017553457188090206965615937213419385871077837956771570536756086\
        174306711357612692470993367865237583867552293229085658581609124490438092837633181262633832\
        224797483199978752769003102249388926725978940121151544263886670246958179931376644535033670\
        467812416073434198259354970151012012393996634623039876371855129029695663871190598266249780\
        985956108718263661078782256910771187915990947315908329250697737849898883453664118270530637\
        992646293212166265938379652583376259988885243320020604733922599818302151475423517827259313\
        745572427630024541315969730421839521723099984520094706630448404321113193582564213335655774\
        981763101020538995438787319788215596029041832485363544281092101855739567987099076742200167\
        364424955142913668223754444009608547667377068092022968659294922490236070017253740709080640\
        315292979235034789700250605885981133239085270171689650328229691156105945528843937717117926\
        974279215959289065294575953759240127500933613923927450217241389655578977813273562315693668\
        620610689781058230089093580999028877435087490580246008336904216212775259586405528318199698\
        927003543457809044748027664785695692314604437532737114872376072006455841723036100877964792\
        160706943287071568418614565943721712313708341206571425251067669284456587123688846181674686\
        565863767558687263852890202721039246395025",
    );
    // - r != 0 first time in binomial_coefficient_raising_factorial_4
    test(
        "525899105584115331831822389398654378975439821235847075538334479459976804361",
        "39",
        "63916028057105686233224647128326757355318526278850191265753818300077123506148592771686251\
        843871031286613741578697393929955287964794759564768972837142349342808068645976054243890830\
        335431299980007761289437669633253213451826328048705819152063003640688010892927140146765994\
        354038412675123498589995067406959494949003877187335117608898667586172913313268407045309326\
        477759661775566817269791782958981335897762329413676812249676003924448031057465002588047731\
        852626643205542869258383826751237297401033366907927123119420545541506268940498743697330488\
        751030122065781855695143246866312361451074604659115580225953639683092992548231210545034231\
        285233852021142144210530482766055167812957716779323951011087624228337943053231819759016008\
        625315621642894259660510209185578346141448798234857352488051062830286245538065408386075147\
        891775165871474412122232335872318403542887945441629517772432566332148611895718898507734193\
        765122244095887688382558927354124421602429208358235005847747779165375960977523729969144623\
        942428302069692001442132909094843144966416850972408319273833346995290959884209879178506231\
        148534376246222578564697565675661378959062018141850826958907520091990573403348600298989654\
        015323124079115828761962218496431800939709402415702347227440683769932960611846922227191597\
        870575184258217562373579180321454906142789445372655817629764513368590547268810214608619866\
        197718738244504054960382041108971143812515663988260257731991609483301668915756882792269602\
        669162926415723894743233548358640390458315117002551455170242787151998722348149039656347608\
        551565127458440090648316733044596038381441744468926633398369489669625808759994489694980764\
        526999955110732774112122334751376885411610502128309282739439676195629420963063203373174208\
        091590129280478685913083169673141792020290565109251623524461277589125997756234920480123074\
        209844165277719752489744793568781301978159825605932386339370565204645057536772220679746768\
        569224291055005731953230004831223439059300914798178296528394239043745002141250499403742249\
        183874569695076545083105825050171871111881712648095404574108051786690297924470700238074897\
        792414511177305601932558389549730072609925691552289753566425338003950881601294728845880132\
        500287043788618527834331584961373617001959960073108393488866008662550518286185355757098004\
        407731116008477326366253892028653451448990656804004447051045116046286082144417019317955065\
        059880762228310370726650525371685367667341518793755754089895626603855406071794521713173191\
        730870597842233077363902687264007764576583562429780462716550361987547856866479132368207653\
        806647693143188635189378544081798845519519486948493126898293329214413593802057692320667583\
        766380661973838437256232890788063096146364826652367871270984406087292750139629458354387512\
        791101502903444068723989625533851827638409364499043290472310636212447688058605971032119472\
        5991269626624579699378058679358476375414026678029422929039835277325354971910400",
    );
    // - k == 0 in binomial_coefficient_raising_factorial_4
    test(
        "55186653679569007523486527767687421937059940360080261894526453",
        "3",
        "28012439529730582769980703531255137397476701127318927388432337037639370308404590341026789\
        244616713573401635136679393282116959418116580481496576838954049642935186063159197881085428\
        921826",
    );
    // - k <= 1 in binomial_coefficient_raising_factorial_4
    test(
        "821407847938765300653812701638851178",
        "5",
        "31161115310790267810728647115818905409309919004871253034511265989952209125820211890996722\
        13934942403111281665464675315630857236685237472673845588842782932266933122933085560371660",
    );

    fn test_large(n: &str, k: &str) {
        let n = Natural::from_str(n).unwrap();
        let k = Natural::from_str(k).unwrap();
        let b = Natural::binomial_coefficient(n.clone(), k.clone());
        assert!(b.is_valid());

        let b_alt = Natural::binomial_coefficient(&n, &k);
        assert!(b_alt.is_valid());
        assert_eq!(b_alt, b);

        assert_eq!(binomial_coefficient_naive_2(n.clone(), k.clone()), b);
        if let Ok(k) = u32::try_from(&k) {
            assert_eq!(Natural::exact_from(&rug::Integer::from(&n).binomial(k)), b);
        }
    }
    // - k > Limb::power_of_2(Limb::WIDTH >> 1) in binomial_coefficient_hmul_nbnpk
    test_large("4294967296", "131076");
}

#[test]
fn limbs_binomial_coefficient_limb_limb_bdiv_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_small_n", 256);
    unsigned_pair_gen_var_45().test_properties_with_config(&config, |(n, k)| {
        let xs = limbs_binomial_coefficient_limb_limb_bdiv(n, k);
        assert_ne!(*xs.last().unwrap(), 0);
        assert_eq!(
            Natural::from_owned_limbs_asc(xs),
            Natural::binomial_coefficient(Natural::from(n), Natural::from(k))
        );
    });
}

#[test]
fn limbs_binomial_coefficient_limb_limb_small_k_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_small_n", 256);
    unsigned_pair_gen_var_46().test_properties_with_config(&config, |(n, k)| {
        let xs = limbs_binomial_coefficient_limb_limb_small_k(n, k);
        assert_ne!(*xs.last().unwrap(), 0);
        assert_eq!(
            Natural::from_owned_limbs_asc(xs),
            Natural::binomial_coefficient(Natural::from(n), Natural::from(k))
        );
    });
}

#[test]
fn limbs_binomial_coefficient_limb_limb_basecase_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_small_n", 256);
    unsigned_pair_gen_var_47().test_properties_with_config(&config, |(n, k)| {
        assert_eq!(
            Natural::from(limbs_binomial_coefficient_limb_limb_basecase(n, k)),
            Natural::binomial_coefficient(Natural::from(n), Natural::from(k))
        );
    });
}

#[test]
fn limbs_binomial_coefficient_limb_limb_small_k_divide_and_conquer_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_small_n", 256);
    unsigned_pair_gen_var_48().test_properties_with_config(&config, |(n, k)| {
        let xs = limbs_binomial_coefficient_limb_limb_small_k_divide_and_conquer(n, k);
        assert_ne!(*xs.last().unwrap(), 0);
        assert_eq!(
            Natural::from_owned_limbs_asc(xs),
            Natural::binomial_coefficient(Natural::from(n), Natural::from(k))
        );
    });
}

#[test]
fn limbs_binomial_coefficient_limb_limb_goetgheluck_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_small_n", 256);
    unsigned_pair_gen_var_49().test_properties_with_config(&config, |(n, k)| {
        let xs = limbs_binomial_coefficient_limb_limb_goetgheluck(n, k);
        assert_ne!(*xs.last().unwrap(), 0);
        assert_eq!(
            Natural::from_owned_limbs_asc(xs),
            Natural::binomial_coefficient(Natural::from(n), Natural::from(k)),
        );
    });
}

#[test]
fn binomial_coefficient_limb_limb_properties() {
    let mut config = GenConfig::new();
    config.insert("mean_small_n", 256);
    unsigned_pair_gen_var_28().test_properties_with_config(&config, |(n, k)| {
        let x = binomial_coefficient_limb_limb(n, k);
        assert!(x.is_valid());
        assert_eq!(
            x,
            Natural::binomial_coefficient(Natural::from(n), Natural::from(k)),
        );
    });
}

#[test]
fn binomial_coefficient_properties() {
    natural_pair_gen_var_15().test_properties(|(n, k)| {
        let b = Natural::binomial_coefficient(n.clone(), k.clone());
        assert!(b.is_valid());

        let b_alt = Natural::binomial_coefficient(&n, &k);
        assert!(b_alt.is_valid());
        assert_eq!(b, b_alt);

        assert_eq!(binomial_coefficient_naive_1(n.clone(), k.clone()), b);
        assert_eq!(binomial_coefficient_naive_2(n.clone(), k.clone()), b);
        assert_eq!(
            Natural::exact_from(&rug::Integer::from(&n).binomial(u32::exact_from(&k))),
            b
        );
        assert_eq!(b == 0u32, n < k);
        if n >= k {
            assert_eq!(Natural::binomial_coefficient(&n, &(&n - &k)), b);
        }
        if n != 0u32 && k != 0u32 {
            let c = Natural::binomial_coefficient(&n - Natural::ONE, &k - Natural::ONE);
            assert_eq!(
                Natural::binomial_coefficient(&(&n - Natural::ONE), &k) + &c,
                b
            );
            let gcd = (&n).gcd(&k);
            assert_eq!(c.div_exact(k.div_exact(&gcd)) * n.div_exact(gcd), b);
        }
    });

    natural_gen().test_properties(|n| {
        assert_eq!(Natural::binomial_coefficient(&n, &Natural::ZERO), 1u32);
        assert_eq!(Natural::binomial_coefficient(&n, &Natural::ONE), n);
    });

    natural_gen_var_2().test_properties(|n| {
        assert_eq!(Natural::binomial_coefficient(&n, &n), 1u32);
        assert_eq!(Natural::binomial_coefficient(&n, &(&n - Natural::ONE)), n);
        assert_eq!(Natural::binomial_coefficient(Natural::ZERO, n), 0u32);
    });

    unsigned_pair_gen_var_44::<Limb>().test_properties(|(n, k)| {
        assert_eq!(
            Natural::binomial_coefficient(Natural::from(n), Natural::from(k)),
            Limb::binomial_coefficient(n, k)
        );
    });
}
