// Copyright © 2025 Mikhail Hogrefe
//
// This file is part of Malachite.
//
// Malachite is free software: you can redistribute it and/or modify it under the terms of the GNU
// Lesser General Public License (LGPL) as published by the Free Software Foundation; either version
// 3 of the License, or (at your option) any later version. See <https://www.gnu.org/licenses/>.

use core::hash::Hash;
use malachite_base::chars::random::random_char_inclusive_range;
use malachite_base::num::random::geometric::geometric_random_unsigneds;
use malachite_base::num::random::random_primitive_ints;
use malachite_base::random::{EXAMPLE_SEED, Seed};
use malachite_base::test_util::vecs::random::random_vecs_helper_helper;
use malachite_base::tuples::random::random_units;
use malachite_base::vecs::random::random_vecs_min_length;
use std::fmt::Debug;

fn random_vecs_min_length_helper<
    T: Clone + Debug + Eq + Hash + Ord,
    I: Clone + Iterator<Item = T>,
>(
    min_length: u64,
    xs_gen: &dyn Fn(Seed) -> I,
    mean_length_numerator: u64,
    mean_length_denominator: u64,
    expected_values: &[&[T]],
    expected_common_values: &[(&[T], usize)],
    expected_median: (&[T], Option<&[T]>),
) {
    random_vecs_helper_helper(
        random_vecs_min_length(
            EXAMPLE_SEED,
            min_length,
            xs_gen,
            mean_length_numerator,
            mean_length_denominator,
        ),
        expected_values,
        expected_common_values,
        expected_median,
    );
}

#[test]
fn test_random_vecs_min_length() {
    random_vecs_min_length_helper(
        0,
        &|_| random_units(),
        4,
        1,
        &[
            &[],
            &[(), (), (), (), (), (), (), (), (), (), (), (), (), ()],
            &[(), (), (), ()],
            &[(), (), (), ()],
            &[()],
            &[],
            &[(), (), (), (), ()],
            &[(), ()],
            &[(), (), (), ()],
            &[],
            &[(), (), (), (), (), ()],
            &[],
            &[],
            &[(), (), (), (), (), (), (), (), (), (), ()],
            &[(), (), (), (), (), (), (), ()],
            &[],
            &[(), (), ()],
            &[],
            &[(), (), (), (), ()],
            &[(), (), (), (), (), (), (), (), ()],
        ],
        &[
            (&[], 199913),
            (&[()], 160173),
            (&[(), ()], 128173),
            (&[(), (), ()], 102460),
            (&[(), (), (), ()], 81463),
            (&[(), (), (), (), ()], 65695),
            (&[(), (), (), (), (), ()], 52495),
            (&[(), (), (), (), (), (), ()], 41943),
            (&[(), (), (), (), (), (), (), ()], 33396),
            (&[(), (), (), (), (), (), (), (), ()], 27035),
        ],
        (&[(), (), ()], None),
    );
    random_vecs_min_length_helper(
        3,
        &|_| random_units(),
        7,
        1,
        &[
            &[(), (), ()],
            &[(), (), (), (), (), (), (), (), (), (), (), (), (), (), (), (), ()],
            &[(), (), (), (), (), (), ()],
            &[(), (), (), (), (), (), ()],
            &[(), (), (), ()],
            &[(), (), ()],
            &[(), (), (), (), (), (), (), ()],
            &[(), (), (), (), ()],
            &[(), (), (), (), (), (), ()],
            &[(), (), ()],
            &[(), (), (), (), (), (), (), (), ()],
            &[(), (), ()],
            &[(), (), ()],
            &[(), (), (), (), (), (), (), (), (), (), (), (), (), ()],
            &[(), (), (), (), (), (), (), (), (), (), ()],
            &[(), (), ()],
            &[(), (), (), (), (), ()],
            &[(), (), ()],
            &[(), (), (), (), (), (), (), ()],
            &[(), (), (), (), (), (), (), (), (), (), (), ()],
        ],
        &[
            (&[(), (), ()], 199913),
            (&[(), (), (), ()], 160173),
            (&[(), (), (), (), ()], 128173),
            (&[(), (), (), (), (), ()], 102460),
            (&[(), (), (), (), (), (), ()], 81463),
            (&[(), (), (), (), (), (), (), ()], 65695),
            (&[(), (), (), (), (), (), (), (), ()], 52495),
            (&[(), (), (), (), (), (), (), (), (), ()], 41943),
            (&[(), (), (), (), (), (), (), (), (), (), ()], 33396),
            (&[(), (), (), (), (), (), (), (), (), (), (), ()], 27035),
        ],
        (&[(), (), (), (), (), ()], None),
    );
    random_vecs_min_length_helper(
        0,
        &random_primitive_ints::<u8>,
        4,
        1,
        &[
            &[],
            &[85, 11, 136, 200, 235, 134, 203, 223, 38, 235, 217, 177, 162, 32],
            &[166, 234, 30, 218],
            &[90, 106, 9, 216],
            &[204],
            &[],
            &[151, 213, 97, 253, 78],
            &[91, 39],
            &[191, 175, 170, 232],
            &[],
            &[233, 2, 35, 22, 217, 198],
            &[],
            &[],
            &[114, 17, 32, 173, 114, 65, 121, 222, 173, 25, 144],
            &[148, 79, 115, 52, 73, 69, 137, 91],
            &[],
            &[153, 178, 112],
            &[],
            &[34, 95, 106, 167, 197],
            &[130, 168, 122, 207, 172, 177, 86, 150, 221],
        ],
        &[
            (&[], 199913),
            (&[146], 693),
            (&[26], 692),
            (&[185], 688),
            (&[58], 683),
            (&[196], 683),
            (&[81], 678),
            (&[229], 675),
            (&[192], 673),
            (&[233], 673),
        ],
        (&[96], None),
    );
    random_vecs_min_length_helper(
        3,
        &random_primitive_ints::<u8>,
        7,
        1,
        &[
            &[85, 11, 136],
            &[200, 235, 134, 203, 223, 38, 235, 217, 177, 162, 32, 166, 234, 30, 218, 90, 106],
            &[9, 216, 204, 151, 213, 97, 253],
            &[78, 91, 39, 191, 175, 170, 232],
            &[233, 2, 35, 22],
            &[217, 198, 114],
            &[17, 32, 173, 114, 65, 121, 222, 173],
            &[25, 144, 148, 79, 115],
            &[52, 73, 69, 137, 91, 153, 178],
            &[112, 34, 95],
            &[106, 167, 197, 130, 168, 122, 207, 172, 177],
            &[86, 150, 221],
            &[218, 101, 115],
            &[74, 9, 123, 109, 52, 201, 159, 247, 250, 48, 133, 235, 196, 40],
            &[97, 104, 68, 190, 216, 7, 216, 157, 43, 43, 112],
            &[217, 24, 11],
            &[103, 211, 84, 135, 55, 29],
            &[206, 89, 65],
            &[191, 51, 9, 79, 148, 34, 22, 22],
            &[62, 3, 114, 118, 20, 47, 194, 50, 32, 120, 176, 166],
        ],
        &[
            (&[65, 71, 68], 3),
            (&[34, 39, 234], 3),
            (&[207, 218, 62], 3),
            (&[89, 175, 228], 3),
            (&[198, 166, 242], 3),
            (&[2, 94, 4], 2),
            (&[3, 91, 9], 2),
            (&[6, 3, 61], 2),
            (&[0, 18, 20], 2),
            (&[1, 48, 93], 2),
        ],
        (&[128, 11, 100, 4, 101, 167, 125], Some(&[128, 11, 104])),
    );
    random_vecs_min_length_helper(
        0,
        &|seed| geometric_random_unsigneds::<u32>(seed, 2, 1),
        4,
        1,
        &[
            &[],
            &[5, 0, 0, 1, 1, 1, 4, 2, 4, 6, 2, 2, 0, 1],
            &[9, 13, 0, 0],
            &[2, 0, 7, 4],
            &[6],
            &[],
            &[7, 6, 0, 0, 0],
            &[1, 3],
            &[5, 1, 2, 1],
            &[],
            &[0, 0, 1, 4, 2, 0],
            &[],
            &[],
            &[12, 0, 0, 2, 3, 1, 1, 1, 2, 3, 3],
            &[9, 1, 0, 2, 1, 11, 1, 0],
            &[],
            &[1, 6, 0],
            &[],
            &[3, 18, 3, 3, 0],
            &[5, 1, 2, 5, 0, 0, 2, 3, 1],
        ],
        &[
            (&[], 199913),
            (&[0], 53462),
            (&[1], 35352),
            (&[2], 23512),
            (&[3], 16118),
            (&[0, 0], 14371),
            (&[4], 10594),
            (&[0, 1], 9566),
            (&[1, 0], 9409),
            (&[5], 7157),
        ],
        (&[1], None),
    );
    random_vecs_min_length_helper(
        3,
        &|seed| geometric_random_unsigneds::<u32>(seed, 2, 1),
        7,
        1,
        &[
            &[5, 0, 0],
            &[1, 1, 1, 4, 2, 4, 6, 2, 2, 0, 1, 9, 13, 0, 0, 2, 0],
            &[7, 4, 6, 7, 6, 0, 0],
            &[0, 1, 3, 5, 1, 2, 1],
            &[0, 0, 1, 4],
            &[2, 0, 12],
            &[0, 0, 2, 3, 1, 1, 1, 2],
            &[3, 3, 9, 1, 0],
            &[2, 1, 11, 1, 0, 1, 6],
            &[0, 3, 18],
            &[3, 3, 0, 5, 1, 2, 5, 0, 0],
            &[2, 3, 1],
            &[0, 2, 1],
            &[2, 3, 1, 3, 3, 0, 0, 7, 2, 0, 0, 4, 2, 1],
            &[0, 3, 1, 4, 2, 2, 1, 5, 0, 2, 0],
            &[2, 0, 0],
            &[1, 4, 1, 3, 3, 1],
            &[0, 0, 0],
            &[1, 4, 2, 1, 1, 0, 1, 0],
            &[0, 0, 1, 0, 2, 0, 7, 0, 0, 0, 2, 12],
        ],
        &[
            (&[0, 0, 0], 7500),
            (&[1, 0, 0], 5048),
            (&[0, 0, 1], 4956),
            (&[0, 1, 0], 4876),
            (&[0, 0, 2], 3336),
            (&[0, 2, 0], 3320),
            (&[1, 1, 0], 3305),
            (&[2, 0, 0], 3303),
            (&[0, 1, 1], 3242),
            (&[1, 0, 1], 3239),
        ],
        (
            &[1, 3, 1, 1, 4, 0, 2, 3],
            Some(&[1, 3, 1, 1, 4, 0, 4, 0, 0]),
        ),
    );
    random_vecs_min_length_helper(
        0,
        &random_primitive_ints::<u8>,
        1,
        4,
        &[
            &[],
            &[],
            &[85],
            &[11],
            &[136, 200],
            &[],
            &[],
            &[],
            &[],
            &[],
            &[],
            &[],
            &[235, 134],
            &[203],
            &[],
            &[223, 38],
            &[],
            &[],
            &[],
            &[],
        ],
        &[
            (&[], 800023),
            (&[8], 704),
            (&[162], 691),
            (&[81], 690),
            (&[211], 690),
            (&[108], 688),
            (&[235], 688),
            (&[35], 687),
            (&[65], 682),
            (&[208], 679),
        ],
        (&[], None),
    );
    random_vecs_min_length_helper(
        3,
        &random_primitive_ints::<u8>,
        13,
        4,
        &[
            &[85, 11, 136],
            &[200, 235, 134],
            &[203, 223, 38, 235],
            &[217, 177, 162, 32],
            &[166, 234, 30, 218, 90],
            &[106, 9, 216],
            &[204, 151, 213],
            &[97, 253, 78],
            &[91, 39, 191],
            &[175, 170, 232],
            &[233, 2, 35],
            &[22, 217, 198],
            &[114, 17, 32, 173, 114],
            &[65, 121, 222, 173],
            &[25, 144, 148],
            &[79, 115, 52, 73, 69],
            &[137, 91, 153],
            &[178, 112, 34],
            &[95, 106, 167],
            &[197, 130, 168],
        ],
        &[
            (&[34, 248, 94], 4),
            (&[155, 89, 108], 4),
            (&[232, 167, 146], 4),
            (&[255, 244, 186], 4),
            (&[0, 70, 96], 3),
            (&[18, 1, 93], 3),
            (&[30, 0, 90], 3),
            (&[50, 3, 37], 3),
            (&[82, 1, 99], 3),
            (&[9, 42, 49], 3),
        ],
        (&[127, 228, 182], Some(&[127, 228, 193])),
    );
    random_vecs_min_length_helper(
        0,
        &|seed| random_char_inclusive_range(seed, 'a', 'z'),
        4,
        1,
        &[
            &[],
            &['v', 'c', 'q', 'i', 'e', 'p', 'g', 's', 'n', 't', 'm', 'z', 'o', 'm'],
            &['f', 'k', 'q', 'y'],
            &['u', 'k', 'x', 'h'],
            &['u'],
            &[],
            &['n', 'n', 'j', 'n', 'j'],
            &['a', 'w'],
            &['z', 'l', 'w', 'b'],
            &[],
            &['l', 'u', 'n', 'e', 'l', 'v'],
            &[],
            &[],
            &['k', 'u', 'h', 'c', 'y', 'i', 'm', 'r', 'm', 'y', 's'],
            &['l', 'e', 'a', 's', 'w', 'k', 'o', 'b'],
            &[],
            &['k', 'w', 'g'],
            &[],
            &['d', 'q', 'e', 'f', 'u'],
            &['z', 'r', 'g', 'j', 'k', 'r', 's', 'y', 'n'],
        ],
        &[
            (&[], 199913),
            (&['o'], 6313),
            (&['y'], 6262),
            (&['q'], 6261),
            (&['j'], 6245),
            (&['p'], 6244),
            (&['g'], 6219),
            (&['x'], 6215),
            (&['e'], 6200),
            (&['t'], 6188),
        ],
        (&['j', 's', 'z'], None),
    );
    random_vecs_min_length_helper(
        3,
        &|seed| random_char_inclusive_range(seed, 'a', 'z'),
        7,
        1,
        &[
            &['v', 'c', 'q'],
            &['i', 'e', 'p', 'g', 's', 'n', 't', 'm', 'z', 'o', 'm', 'f', 'k', 'q', 'y', 'u', 'k'],
            &['x', 'h', 'u', 'n', 'n', 'j', 'n'],
            &['j', 'a', 'w', 'z', 'l', 'w', 'b'],
            &['l', 'u', 'n', 'e'],
            &['l', 'v', 'k'],
            &['u', 'h', 'c', 'y', 'i', 'm', 'r', 'm'],
            &['y', 's', 'l', 'e', 'a'],
            &['s', 'w', 'k', 'o', 'b', 'k', 'w'],
            &['g', 'd', 'q'],
            &['e', 'f', 'u', 'z', 'r', 'g', 'j', 'k', 'r'],
            &['s', 'y', 'n'],
            &['f', 't', 's'],
            &['f', 'e', 's', 'p', 'j', 'n', 'h', 'n', 'r', 'f', 'i', 'u', 'k', 'p'],
            &['p', 'g', 'l', 'd', 'l', 'l', 'z', 's', 'w', 'w', 'l'],
            &['w', 'z', 'j'],
            &['j', 'j', 'y', 'g', 'e', 'z'],
            &['v', 'p', 'y'],
            &['u', 'q', 'l', 'h', 'r', 'r', 's', 'q'],
            &['b', 'n', 'e', 's', 'p', 'r', 'd', 'a', 'k', 'w', 'c', 'y'],
        ],
        &[
            (&['b', 'c', 'j'], 25),
            (&['e', 'k', 'd'], 25),
            (&['a', 'x', 'n'], 24),
            (&['b', 'e', 'z'], 24),
            (&['c', 'c', 'b'], 24),
            (&['d', 'g', 'h'], 24),
            (&['g', 'l', 'i'], 24),
            (&['i', 'w', 'n'], 24),
            (&['j', 'd', 'w'], 24),
            (&['m', 'y', 'a'], 24),
        ],
        (
            &['m', 'z', 'z', 'r', 'e', 'r'],
            Some(&['m', 'z', 'z', 'r', 'g', 'i']),
        ),
    );
}

#[test]
#[should_panic]
fn random_vecs_min_length_fail_1() {
    random_vecs_min_length(EXAMPLE_SEED, 3, &random_primitive_ints::<u32>, 3, 1);
}

#[test]
#[should_panic]
fn random_vecs_min_length_fail_2() {
    random_vecs_min_length(EXAMPLE_SEED, 1, &random_primitive_ints::<u32>, 1, 0);
}

#[test]
#[should_panic]
fn random_vecs_min_length_fail_3() {
    random_vecs_min_length(
        EXAMPLE_SEED,
        0,
        &random_primitive_ints::<u32>,
        u64::MAX,
        u64::MAX - 1,
    );
}
