// Copyright © 2025 Mikhail Hogrefe
//
// This file is part of Malachite.
//
// Malachite is free software: you can redistribute it and/or modify it under the terms of the GNU
// Lesser General Public License (LGPL) as published by the Free Software Foundation; either version
// 3 of the License, or (at your option) any later version. See <https://www.gnu.org/licenses/>.

use malachite_base::num::basic::floats::PrimitiveFloat;
use malachite_base::num::basic::traits::NegativeInfinity;
use malachite_base::num::float::NiceFloat;
use malachite_base::num::random::special_random_primitive_float_inclusive_range;
use malachite_base::random::EXAMPLE_SEED;
use malachite_base::test_util::num::random::special_random_primitive_floats_helper_helper;
use malachite_base::test_util::stats::moments::{CheckedToF64, MomentStats};
use std::panic::catch_unwind;

fn special_random_primitive_float_inclusive_range_helper<T: CheckedToF64 + PrimitiveFloat>(
    a: T,
    b: T,
    mean_exponent_numerator: u64,
    mean_exponent_denominator: u64,
    mean_precision_numerator: u64,
    mean_precision_denominator: u64,
    mean_zero_p_numerator: u64,
    mean_zero_p_denominator: u64,
    expected_values: &[T],
    expected_common_values: &[(T, usize)],
    expected_median: (T, Option<T>),
    expected_moment_stats: MomentStats,
) {
    special_random_primitive_floats_helper_helper(
        special_random_primitive_float_inclusive_range::<T>(
            EXAMPLE_SEED,
            a,
            b,
            mean_exponent_numerator,
            mean_exponent_denominator,
            mean_precision_numerator,
            mean_precision_denominator,
            mean_zero_p_numerator,
            mean_zero_p_denominator,
        ),
        expected_values,
        expected_common_values,
        expected_median,
        expected_moment_stats,
    );
}

#[test]
fn test_special_random_primitive_float_inclusive_range() {
    // f32, a = 1.0, b = 2.0, mean abs of exponent = 1, mean precision = 2, mean zero P = 1/10
    let values = &[
        2.0, 2.0, 2.0, 1.0, 1.5, 2.0, 1.0, 1.0, 1.5, 2.0, 1.28125, 1.0, 2.0, 1.0, 1.4375, 1.8125,
        1.6875, 1.0, 1.75, 1.875, 2.0, 1.0, 2.0, 1.0, 1.75, 2.0, 2.0, 2.0, 2.0, 1.75, 2.0, 1.5,
        1.484375, 2.0, 1.0, 2.0, 1.75, 2.0, 1.25, 1.0, 1.5, 1.25, 1.546875, 1.0, 2.0, 2.0, 2.0,
        1.4921875, 2.0, 2.0,
    ];
    let common_values = &[
        (2.0, 332566),
        (1.0, 222413),
        (1.5, 148427),
        (1.25, 49546),
        (1.75, 49274),
        (1.875, 16717),
        (1.125, 16593),
        (1.375, 16567),
        (1.625, 16551),
        (1.8125, 5566),
        (1.5625, 5553),
        (1.9375, 5543),
        (1.1875, 5451),
        (1.4375, 5383),
        (1.6875, 5367),
        (1.3125, 5361),
        (1.0625, 5322),
        (1.21875, 1873),
        (1.34375, 1871),
        (1.40625, 1865),
    ];
    let sample_median = (1.5, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(1.5553208489836596),
        standard_deviation: NiceFloat(0.3974073963635055),
        skewness: NiceFloat(-0.20042532167244795),
        excess_kurtosis: NiceFloat(-1.5078861004584492),
    };
    special_random_primitive_float_inclusive_range_helper::<f32>(
        1.0,
        2.0,
        1,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = 1.0, b = 2.0, mean abs of exponent = 10, mean precision = 10, mean zero P = 1/100
    let values = &[
        2.0, 2.0, 1.0, 1.640625, 1.375, 2.0, 1.90625, 2.0, 2.0, 1.25, 1.0, 2.0, 2.0, 1.9375, 1.5,
        2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 1.96875, 2.0, 1.1479492, 1.4569855, 1.2559814,
        1.5579834, 2.0, 2.0, 1.2250977, 1.1882324, 1.5, 2.0, 1.6498959, 2.0, 1.1252441, 2.0, 2.0,
        2.0, 1.0336914, 2.0, 2.0, 2.0, 2.0, 1.5, 1.375, 2.0, 1.0, 1.125,
    ];
    let common_values = &[
        (2.0, 476330),
        (1.0, 52792),
        (1.5, 47801),
        (1.25, 22022),
        (1.75, 21892),
        (1.875, 10017),
        (1.625, 9986),
        (1.375, 9985),
        (1.125, 9951),
        (1.8125, 4622),
        (1.4375, 4596),
        (1.1875, 4577),
        (1.9375, 4552),
        (1.5625, 4549),
        (1.3125, 4532),
        (1.0625, 4507),
        (1.6875, 4455),
        (1.96875, 2147),
        (1.15625, 2105),
        (1.28125, 2098),
    ];
    let sample_median = (1.9371719, Some(1.9371796));
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(1.712172784260269),
        standard_deviation: NiceFloat(0.34826759681331443),
        skewness: NiceFloat(-0.7989590190130814),
        excess_kurtosis: NiceFloat(-0.8454122824548973),
    };
    special_random_primitive_float_inclusive_range_helper::<f32>(
        1.0,
        2.0,
        10,
        1,
        10,
        1,
        1,
        100,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = -0.1, b = 0.1, mean abs of exponent = 5, mean precision = 2, mean zero P = 1/10
    let values = &[
        0.09375,
        0.01171875,
        0.01171875,
        -0.0,
        -0.09375,
        -0.01171875,
        -0.01171875,
        -0.03125,
        0.03125,
        0.0625,
        -0.0625,
        -0.07525635,
        -0.0625,
        0.0,
        -0.09375,
        -0.09375,
        -0.09375,
        -0.0,
        0.07525635,
        -0.078125,
        -0.05078125,
        0.0625,
        0.09375,
        0.09375,
        -0.024414062,
        0.09375,
        -0.001953125,
        0.078125,
        0.05078125,
        -0.0,
        -0.0625,
        -0.09375,
        0.0,
        0.024414062,
        -0.03125,
        -0.0043945312,
        -0.09375,
        0.001953125,
        -0.09375,
        -0.003540039,
        -0.03125,
        0.0625,
        -0.046875,
        0.09375,
        -0.0625,
        0.03125,
        0.0,
        -0.0146484375,
        0.0043945312,
        0.09375,
    ];
    let common_values = &[
        (0.0625, 74962),
        (-0.0625, 74736),
        (-0.0, 50351),
        (0.09375, 50309),
        (-0.09375, 50178),
        (0.0, 49873),
        (0.03125, 37782),
        (-0.03125, 37678),
        (0.078125, 33219),
        (-0.078125, 33138),
        (0.046875, 25130),
        (-0.046875, 25053),
        (0.015625, 18743),
        (-0.015625, 18692),
        (0.0234375, 12593),
        (-0.0234375, 12549),
        (0.0703125, 11091),
        (0.0859375, 11088),
        (-0.0859375, 11060),
        (-0.0703125, 11055),
    ];
    let sample_median = (0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(0.0000674789649248169),
        standard_deviation: NiceFloat(0.05672019953597066),
        skewness: NiceFloat(-0.001702872258409426),
        excess_kurtosis: NiceFloat(-1.109977309372344),
    };
    special_random_primitive_float_inclusive_range_helper::<f32>(
        -0.1,
        0.1,
        5,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = -0.1, b = 0.1, mean abs of exponent = 10, mean precision = 10, mean zero P = 1/100
    let values = &[
        0.0041503906,
        0.0625,
        0.00042547443,
        -0.0041503906,
        -0.0625,
        -0.00042547443,
        -0.0027618408,
        0.0027618408,
        0.09375,
        -0.09375,
        -0.083244324,
        -0.076171875,
        -0.00023269653,
        -0.00043609738,
        -0.0029296875,
        0.083244324,
        -0.011459351,
        -0.05419922,
        0.076171875,
        0.00023269653,
        0.00043609738,
        -0.000006198883,
        0.0029296875,
        -0.0008621216,
        0.011459351,
        0.05419922,
        -0.00035095215,
        -0.0625,
        0.000006198883,
        -0.0008010864,
        -0.00030517578,
        -0.00008274312,
        0.0008621216,
        -0.0029296875,
        -0.04892564,
        -0.013671875,
        0.00035095215,
        -0.0009398293,
        0.0625,
        -0.07344723,
        0.0008010864,
        -0.01953125,
        0.00030517578,
        0.00008274312,
        0.0029296875,
        0.04892564,
        -0.006134033,
        -0.0070343018,
        -0.005493164,
        -0.0000029057264,
    ];
    let common_values = &[
        (0.0625, 7060),
        (-0.0625, 7045),
        (0.09375, 6481),
        (-0.09375, 6468),
        (0.03125, 6186),
        (-0.03125, 6171),
        (0.078125, 5831),
        (-0.078125, 5813),
        (0.046875, 5527),
        (-0.046875, 5516),
        (0.015625, 5112),
        (-0.015625, 5103),
        (-0.0, 5098),
        (0.0, 4891),
        (0.0234375, 4821),
        (-0.0234375, 4810),
        (0.0078125, 4681),
        (-0.0078125, 4678),
        (0.01171875, 4123),
        (-0.01171875, 4120),
    ];
    let sample_median = (0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(0.000022248151599295377),
        standard_deviation: NiceFloat(0.035382441406785786),
        skewness: NiceFloat(0.0002929225243710825),
        excess_kurtosis: NiceFloat(1.5823949638933827),
    };
    special_random_primitive_float_inclusive_range_helper::<f32>(
        -0.1,
        0.1,
        10,
        1,
        10,
        1,
        1,
        100,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = e, b = π, mean abs of exponent = 3, mean precision = 2, mean zero P = 1/10
    let values = &[
        3.0, 2.859375, 3.0234375, 3.0, 2.75, 3.0, 3.0, 3.0, 2.75, 2.75, 3.078125, 3.0, 3.0, 3.0,
        2.84375, 2.84375, 2.96875, 3.0, 3.125, 2.8125, 3.0, 3.0, 3.03125, 3.0, 3.125, 3.125, 2.875,
        3.0, 2.75, 2.875, 3.0, 2.75, 2.9140625, 2.734375, 3.0, 3.0, 3.125, 2.9257812, 3.125, 3.0,
        2.75, 3.125, 2.9921875, 3.0, 2.75, 3.0, 3.0, 3.0195312, 3.0, 2.8125,
    ];
    let common_values = &[
        (3.0, 333251),
        (2.75, 222162),
        (3.125, 74149),
        (2.875, 74011),
        (3.0625, 33262),
        (2.8125, 33146),
        (2.9375, 33097),
        (2.90625, 9548),
        (2.78125, 9377),
        (2.71875, 9327),
        (3.03125, 9327),
        (3.09375, 9323),
        (2.84375, 9315),
        (2.96875, 9265),
        (2.859375, 3186),
        (3.015625, 3185),
        (2.796875, 3180),
        (3.046875, 3160),
        (3.140625, 3157),
        (2.765625, 3152),
    ];
    let sample_median = (3.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(2.923236772231216),
        standard_deviation: NiceFloat(0.12616339323242115),
        skewness: NiceFloat(-0.17689588042510748),
        excess_kurtosis: NiceFloat(-1.2738593390011854),
    };
    special_random_primitive_float_inclusive_range_helper::<f32>(
        core::f32::consts::E,
        core::f32::consts::PI,
        3,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = e, b = π, mean abs of exponent = 10, mean precision = 10, mean zero P = 1/100
    let values = &[
        2.9238281, 2.953125, 3.0, 2.8671875, 2.8125, 3.125, 3.015625, 2.8462658, 3.140625, 2.875,
        3.0, 2.75, 3.0, 2.71875, 2.75, 3.0214844, 2.970642, 3.0179443, 2.968872, 2.75, 2.9763536,
        2.875, 2.890625, 2.8388672, 2.7814941, 3.087654, 3.1065063, 2.817566, 2.8125, 2.7746582,
        2.746338, 3.0207214, 2.75, 2.9852295, 3.0, 2.9980469, 3.006958, 2.875, 2.8632812,
        2.9277344, 3.088623, 2.8376465, 3.0, 3.0, 3.0117188, 2.75, 2.8125, 3.0, 3.0, 3.140625,
    ];
    let common_values = &[
        (3.0, 103073),
        (2.75, 94168),
        (2.875, 43152),
        (3.125, 42886),
        (3.0625, 26213),
        (2.8125, 25982),
        (2.9375, 25799),
        (2.90625, 10203),
        (2.71875, 10186),
        (2.84375, 10149),
        (3.09375, 10141),
        (3.03125, 10134),
        (2.78125, 10085),
        (2.96875, 10085),
        (3.046875, 4666),
        (2.734375, 4649),
        (2.765625, 4649),
        (3.109375, 4629),
        (2.859375, 4606),
        (3.140625, 4590),
    ];
    let sample_median = (2.9375, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(2.925577550306893),
        standard_deviation: NiceFloat(0.12565616542944752),
        skewness: NiceFloat(-0.016691773531483878),
        excess_kurtosis: NiceFloat(-1.2405888218124512),
    };
    special_random_primitive_float_inclusive_range_helper::<f32>(
        core::f32::consts::E,
        core::f32::consts::PI,
        10,
        1,
        10,
        1,
        1,
        100,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = 100.0, b = 101.0, mean abs of exponent = 7, mean precision = 2, mean zero P = 1/10
    let values = &[
        100.0, 100.5625, 100.40625, 100.0, 101.0, 100.0, 100.0, 100.0, 101.0, 101.0, 100.8125,
        100.0, 100.0, 100.0, 100.375, 100.875, 100.875, 100.0, 100.5, 100.75, 100.0, 100.0,
        100.875, 100.0, 100.5, 100.5, 100.5, 100.0, 101.0, 100.5, 100.0, 101.0, 100.46875,
        100.1875, 100.0, 100.0, 100.5, 100.546875, 100.5, 100.0, 101.0, 100.5, 100.96875, 100.0,
        101.0, 100.0, 100.0, 100.296875, 100.0, 100.75,
    ];
    let common_values = &[
        (100.0, 333338),
        (101.0, 222241),
        (100.5, 148191),
        (100.75, 49867),
        (100.25, 49679),
        (100.625, 16568),
        (100.875, 16322),
        (100.125, 16318),
        (100.375, 16281),
        (100.6875, 5585),
        (100.3125, 5557),
        (100.5625, 5547),
        (100.1875, 5520),
        (100.9375, 5484),
        (100.0625, 5464),
        (100.8125, 5438),
        (100.4375, 5236),
        (100.71875, 1933),
        (100.09375, 1868),
        (100.34375, 1863),
    ];
    let sample_median = (100.5, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(100.44460313228277),
        standard_deviation: NiceFloat(0.3974843507339211),
        skewness: NiceFloat(0.19977351658386902),
        excess_kurtosis: NiceFloat(-1.5086143196425152),
    };
    special_random_primitive_float_inclusive_range_helper::<f32>(
        100.0,
        101.0,
        7,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = 100.0, b = 101.0, mean abs of exponent = 10, mean precision = 10, mean zero P =
    // 1/100
    let values = &[
        100.82031, 100.5625, 100.0, 100.71875, 100.75, 100.5, 100.9375, 100.88427, 100.9375, 100.5,
        100.0, 101.0, 100.0, 100.125, 101.0, 100.46094, 100.51196, 100.46338, 100.80127, 101.0,
        100.0, 100.5, 100.1875, 100.01172, 100.569336, 100.59958, 100.50122, 100.06665, 100.25,
        100.31543, 100.47949, 100.94539, 101.0, 100.47705, 100.75, 100.72206, 100.59912, 100.5,
        100.796875, 100.44531, 100.44629, 100.42285, 100.0, 100.0, 100.802734, 101.0, 100.25,
        100.75, 100.0, 100.13281,
    ];
    let common_values = &[
        (100.0, 108217),
        (101.0, 98713),
        (100.5, 90150),
        (100.75, 40946),
        (100.25, 40858),
        (100.875, 18753),
        (100.125, 18677),
        (100.625, 18429),
        (100.375, 18424),
        (100.1875, 8582),
        (100.6875, 8505),
        (100.5625, 8500),
        (100.8125, 8480),
        (100.9375, 8342),
        (100.4375, 8341),
        (100.0625, 8329),
        (100.3125, 8269),
        (100.90625, 3936),
        (100.28125, 3920),
        (100.21875, 3910),
    ];
    let sample_median = (100.5, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(100.49535768898696),
        standard_deviation: NiceFloat(0.32868306867221864),
        skewness: NiceFloat(0.008975981158852992),
        excess_kurtosis: NiceFloat(-1.221929691547448),
    };
    special_random_primitive_float_inclusive_range_helper::<f32>(
        100.0,
        101.0,
        10,
        1,
        10,
        1,
        1,
        100,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = 1.0e38, b = Infinity, mean abs of exponent = 129, mean precision = 2, mean zero P =
    // 1/10
    let values = &[
        2.5521178e38,
        1.0633824e38,
        1.0633824e38,
        f32::INFINITY,
        1.2760589e38,
        1.7014118e38,
        1.2349359e38,
        1.7014118e38,
        1.0633824e38,
        1.4887354e38,
        1.0633824e38,
        1.1697206e38,
        1.329228e38,
        f32::INFINITY,
        1.3026434e38,
        1.2760589e38,
        1.2760589e38,
        f32::INFINITY,
        1.4887354e38,
        1.7014118e38,
        1.5419045e38,
        2.5521178e38,
        2.5521178e38,
        3.2964854e38,
        1.7014118e38,
        1.0633824e38,
        1.2760589e38,
        1.2228898e38,
        1.3823971e38,
        f32::INFINITY,
        1.1763668e38,
        1.2760589e38,
        f32::INFINITY,
        1.5950736e38,
        1.4887354e38,
        1.2760589e38,
        2.9774707e38,
        2.179934e38,
        2.3394413e38,
        2.5521178e38,
        2.1267648e38,
        2.9774707e38,
        1.0633824e38,
        1.7014118e38,
        1.7014118e38,
        1.2760589e38,
        f32::INFINITY,
        1.5809505e38,
        1.1697206e38,
        2.7647942e38,
    ];
    let common_values = &[
        (1.2760589e38, 171851),
        (1.7014118e38, 128688),
        (f32::INFINITY, 100224),
        (2.5521178e38, 85928),
        (1.4887354e38, 57285),
        (1.0633824e38, 57158),
        (2.9774707e38, 28779),
        (2.1267648e38, 28453),
        (1.1697206e38, 25497),
        (1.3823971e38, 25299),
        (1.5950736e38, 25002),
        (2.3394413e38, 9529),
        (1.9140883e38, 9502),
        (3.1901472e38, 9420),
        (2.7647942e38, 9406),
        (1.1165515e38, 7388),
        (1.329228e38, 7368),
        (1.2228898e38, 7328),
        (1.4355662e38, 7281),
        (1.6482427e38, 7214),
    ];
    let sample_median = (1.6482427e38, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(f64::NAN),
        standard_deviation: NiceFloat(f64::NAN),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_inclusive_range_helper::<f32>(
        1.0e38,
        f32::INFINITY,
        129,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = -f32::MIN_POSITIVE_SUBNORMAL, b = f32::MIN_POSITIVE_SUBNORMAL, mean abs of exponent
    // = 150, mean precision = 2, mean zero P = 1/10
    let values = &[
        1.0e-45, 1.0e-45, 1.0e-45, -0.0, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, 1.0e-45, 1.0e-45,
        -1.0e-45, -1.0e-45, -1.0e-45, 0.0, -1.0e-45, -1.0e-45, -1.0e-45, -0.0, 1.0e-45, -1.0e-45,
        -1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, -1.0e-45, 1.0e-45, -1.0e-45, 1.0e-45, 1.0e-45, -0.0,
        -1.0e-45, -1.0e-45, 0.0, 1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, 1.0e-45, -1.0e-45,
        -1.0e-45, -1.0e-45, 1.0e-45, -1.0e-45, 1.0e-45, -1.0e-45, 1.0e-45, 0.0, -1.0e-45, 1.0e-45,
        1.0e-45,
    ];
    let common_values = &[(1.0e-45, 450531), (-1.0e-45, 449245), (-0.0, 50351), (0.0, 49873)];
    let sample_median = (0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(1.8020698251216835e-48),
        standard_deviation: NiceFloat(1.3292224464360085e-45),
        skewness: NiceFloat(-0.0025604536978191587),
        excess_kurtosis: NiceFloat(-1.8886053342505447),
    };
    special_random_primitive_float_inclusive_range_helper::<f32>(
        -f32::MIN_POSITIVE_SUBNORMAL,
        f32::MIN_POSITIVE_SUBNORMAL,
        150,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = -0.0, b = f32::MIN_POSITIVE_SUBNORMAL, mean abs of exponent = 150, mean precision =
    // 2, mean zero P = 1/10
    let values = &[
        1.0e-45, 1.0e-45, 1.0e-45, -0.0, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45,
        1.0e-45, 1.0e-45, 1.0e-45, 0.0, 1.0e-45, 1.0e-45, 1.0e-45, -0.0, 1.0e-45, 1.0e-45, 1.0e-45,
        1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, -0.0, 1.0e-45,
        1.0e-45, 0.0, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45,
        1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 0.0, 1.0e-45, 1.0e-45, 1.0e-45,
    ];
    let common_values = &[(1.0e-45, 899776), (-0.0, 50351), (0.0, 49873)];
    let sample_median = (1.0e-45, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(1.2608547270363571e-45),
        standard_deviation: NiceFloat(4.2080794564999146e-46),
        skewness: NiceFloat(-2.662524701673921),
        excess_kurtosis: NiceFloat(5.089037787023615),
    };
    special_random_primitive_float_inclusive_range_helper::<f32>(
        -0.0,
        f32::MIN_POSITIVE_SUBNORMAL,
        150,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = 0.0, b = 0.0, mean abs of exponent = 150, mean precision = 2, mean zero P = 1/10
    let values = &[0.0; 50];
    let common_values = &[(0.0, 1000000)];
    let sample_median = (0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(0.0),
        standard_deviation: NiceFloat(0.0),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_inclusive_range_helper::<f32>(
        0.0,
        0.0,
        150,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // -f32::MIN_POSITIVE_SUBNORMAL, b = -f32::MIN_POSITIVE_SUBNORMAL, mean abs of exponent = 150,
    // mean precision = 2, mean zero P = 1/10
    let values = &[-1.0e-45; 50];
    let common_values = &[(-1.0e-45, 1000000)];
    let sample_median = (-1.0e-45, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(-1.401298464324817e-45),
        standard_deviation: NiceFloat(0.0),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_inclusive_range_helper::<f32>(
        -f32::MIN_POSITIVE_SUBNORMAL,
        -f32::MIN_POSITIVE_SUBNORMAL,
        150,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = -f32::MIN_POSITIVE_SUBNORMAL, b = 0.0, mean abs of exponent = 150, mean precision =
    // 2, mean zero P = 1/10
    let values = &[
        -1.0e-45, -1.0e-45, -1.0e-45, -0.0, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45,
        -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, 0.0, -1.0e-45, -1.0e-45, -1.0e-45, -0.0, -1.0e-45,
        -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45,
        -1.0e-45, -0.0, -1.0e-45, -1.0e-45, 0.0, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45,
        -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, 0.0,
        -1.0e-45, -1.0e-45, -1.0e-45,
    ];
    let common_values = &[(-1.0e-45, 899776), (-0.0, 50351), (0.0, 49873)];
    let sample_median = (-1.0e-45, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(-1.2608547270363571e-45),
        standard_deviation: NiceFloat(4.2080794564999146e-46),
        skewness: NiceFloat(2.662524701673921),
        excess_kurtosis: NiceFloat(5.089037787023615),
    };
    special_random_primitive_float_inclusive_range_helper::<f32>(
        -f32::MIN_POSITIVE_SUBNORMAL,
        0.0,
        150,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = -Infinity, b = Infinity, mean abs of exponent = 1, mean precision = 2, mean zero P =
    // 1/10
    let values = &[
        12.0,
        1.5,
        0.75,
        -0.0,
        -12.0,
        -1.5,
        -0.75,
        -4.0,
        4.0,
        2.0,
        -2.0,
        -1.2041016,
        -2.0,
        f32::NEGATIVE_INFINITY,
        -0.75,
        -0.75,
        -3.0,
        0.0,
        1.2041016,
        -40.0,
        -0.6875,
        2.0,
        0.75,
        0.75,
        -1.3125,
        3.0,
        -2.0,
        40.0,
        0.6875,
        -0.0,
        -0.5,
        -1.5,
        f32::NEGATIVE_INFINITY,
        1.3125,
        -0.125,
        -1.125,
        -0.046875,
        2.0,
        -1.5,
        -1.9375,
        -1.0,
        0.5,
        -0.75,
        1.5,
        -1.0,
        0.125,
        f32::INFINITY,
        -1.875,
        1.125,
        0.046875,
    ];
    let common_values = &[
        (1.0, 49888),
        (-1.0, 49745),
        (1.5, 33700),
        (-1.5, 33609),
        (f32::NEGATIVE_INFINITY, 25322),
        (0.5, 25189),
        (-0.0, 25144),
        (-0.5, 25121),
        (f32::INFINITY, 24924),
        (0.0, 24834),
        (2.0, 24770),
        (-2.0, 24716),
        (3.0, 16990),
        (-3.0, 16941),
        (0.75, 16655),
        (-0.75, 16611),
        (4.0, 12510),
        (0.25, 12465),
        (-4.0, 12461),
        (-0.25, 12430),
    ];
    let sample_median = (0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(f64::NAN),
        standard_deviation: NiceFloat(f64::NAN),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_inclusive_range_helper::<f32>(
        f32::NEGATIVE_INFINITY,
        f32::INFINITY,
        1,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = -0.0, b = -0.0, mean abs of exponent = 1, mean precision = 2, mean zero P = 1/10
    let values = &[-0.0; 50];
    let common_values = &[(-0.0, 1000000)];
    let sample_median = (-0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(0.0),
        standard_deviation: NiceFloat(0.0),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_inclusive_range_helper::<f32>(
        -0.0,
        -0.0,
        1,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = 1.0, b = 2.0, mean abs of exponent = 1, mean precision = 2, mean zero P = 1/10
    let values = &[
        2.0, 2.0, 2.0, 1.0, 1.5, 2.0, 1.0, 1.0, 1.5, 2.0, 1.28125, 1.0, 2.0, 1.0, 1.4375, 1.8125,
        1.6875, 1.0, 1.75, 1.875, 2.0, 1.0, 2.0, 1.0, 1.75, 2.0, 2.0, 2.0, 2.0, 1.75, 2.0, 1.5,
        1.484375, 2.0, 1.0, 2.0, 1.75, 2.0, 1.25, 1.0, 1.5, 1.25, 1.546875, 1.0, 2.0, 2.0, 2.0,
        1.4921875, 2.0, 2.0,
    ];
    let common_values = &[
        (2.0, 332566),
        (1.0, 222394),
        (1.5, 148417),
        (1.75, 49426),
        (1.25, 49385),
        (1.375, 16729),
        (1.625, 16608),
        (1.125, 16560),
        (1.875, 16529),
        (1.8125, 5619),
        (1.5625, 5582),
        (1.6875, 5424),
        (1.9375, 5418),
        (1.4375, 5415),
        (1.0625, 5401),
        (1.3125, 5359),
        (1.1875, 5325),
        (1.09375, 1909),
        (1.65625, 1901),
        (1.96875, 1893),
    ];
    let sample_median = (1.5, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(1.5552951797578527),
        standard_deviation: NiceFloat(0.39732291679015563),
        skewness: NiceFloat(-0.2001600069371136),
        excess_kurtosis: NiceFloat(-1.5071515332005163),
    };
    special_random_primitive_float_inclusive_range_helper::<f64>(
        1.0,
        2.0,
        1,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = 1.0, b = 2.0, mean abs of exponent = 10, mean precision = 10, mean zero P = 1/100
    let values = &[
        2.0,
        2.0,
        1.0,
        1.640625,
        1.375,
        2.0,
        1.90625,
        2.0,
        2.0,
        1.25,
        1.0,
        2.0,
        2.0,
        1.9375,
        1.5,
        2.0,
        2.0,
        2.0,
        2.0,
        2.0,
        2.0,
        2.0,
        1.96875,
        2.0,
        1.14794921875,
        1.4569854736328125,
        1.2559814453125,
        1.5579833984375,
        2.0,
        2.0,
        1.22509765625,
        1.0812369697232498,
        1.5,
        2.0,
        1.1291658878326416,
        2.0,
        1.666259765625,
        2.0,
        2.0,
        2.0,
        1.03857421875,
        2.0,
        2.0,
        2.0,
        2.0,
        1.5,
        1.125,
        2.0,
        1.0,
        1.4247607216238976,
    ];
    let common_values = &[
        (2.0, 476330),
        (1.0, 47627),
        (1.5, 43214),
        (1.75, 19860),
        (1.25, 19849),
        (1.875, 9069),
        (1.625, 9023),
        (1.125, 8975),
        (1.375, 8966),
        (1.5625, 4216),
        (1.4375, 4204),
        (1.3125, 4172),
        (1.1875, 4093),
        (1.9375, 4059),
        (1.8125, 4047),
        (1.6875, 4030),
        (1.0625, 4007),
        (1.09375, 1944),
        (1.28125, 1938),
        (1.53125, 1903),
    ];
    let sample_median = (1.9375, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(1.714502436478215),
        standard_deviation: NiceFloat(0.34632959887726134),
        skewness: NiceFloat(-0.804949565136136),
        excess_kurtosis: NiceFloat(-0.8346071411985774),
    };
    special_random_primitive_float_inclusive_range_helper::<f64>(
        1.0,
        2.0,
        10,
        1,
        10,
        1,
        1,
        100,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = -0.1, b = 0.1, mean abs of exponent = 5, mean precision = 2, mean zero P = 1/10
    let values = &[
        0.09375,
        0.01171875,
        0.01171875,
        -0.0,
        -0.09375,
        -0.01171875,
        -0.01171875,
        -0.03125,
        0.03125,
        0.0625,
        -0.0625,
        -0.07525634765625,
        -0.0625,
        0.0,
        -0.09375,
        -0.09375,
        -0.09375,
        -0.0,
        0.07525634765625,
        -0.078125,
        -0.05078125,
        0.0625,
        0.09375,
        0.09375,
        -0.0244140625,
        0.09375,
        -0.001953125,
        0.078125,
        0.05078125,
        -0.0,
        -0.0625,
        -0.09375,
        0.0,
        0.0244140625,
        -0.03125,
        -0.00439453125,
        -0.09375,
        0.001953125,
        -0.09375,
        -0.0035400390625,
        -0.03125,
        0.0625,
        -0.046875,
        0.09375,
        -0.0625,
        0.03125,
        0.0,
        -0.0146484375,
        0.00439453125,
        0.09375,
    ];
    let common_values = &[
        (0.0625, 74957),
        (-0.0625, 74731),
        (-0.0, 50351),
        (0.09375, 50307),
        (-0.09375, 50176),
        (0.0, 49873),
        (0.03125, 37778),
        (-0.03125, 37674),
        (0.078125, 33218),
        (-0.078125, 33137),
        (0.046875, 25128),
        (-0.046875, 25051),
        (0.015625, 18742),
        (-0.015625, 18691),
        (0.0234375, 12592),
        (-0.0234375, 12548),
        (0.0859375, 11149),
        (-0.0859375, 11121),
        (0.0703125, 11027),
        (-0.0703125, 10991),
    ];
    let sample_median = (0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(0.00006747896492481727),
        standard_deviation: NiceFloat(0.05672798899804289),
        skewness: NiceFloat(-0.001703150825150012),
        excess_kurtosis: NiceFloat(-1.1100263487007116),
    };
    special_random_primitive_float_inclusive_range_helper::<f64>(
        -0.1,
        0.1,
        5,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = -0.1, b = 0.1, mean abs of exponent = 10, mean precision = 10, mean zero P = 1/100
    let values = &[
        0.005245480002486147,
        0.0625,
        0.00037892110412940383,
        -0.005245480002486147,
        -0.0625,
        -0.00037892110412940383,
        -0.0025482177734375,
        0.0025482177734375,
        0.08385447226464748,
        -0.08385447226464748,
        -0.07456207275390625,
        -0.076171875,
        -0.000156402587890625,
        -0.0004252493381500244,
        -0.0029296875,
        0.07456207275390625,
        -0.012820055672818853,
        -0.05615234375,
        0.076171875,
        0.000156402587890625,
        0.0004252493381500244,
        -6.198883056640625e-6,
        0.0029296875,
        -0.00057220458984375,
        0.012820055672818853,
        0.05615234375,
        -0.0003814697265625,
        -0.0625,
        6.198883056640625e-6,
        -0.00092315673828125,
        -0.00030517578125,
        -0.00007461127825081348,
        0.00057220458984375,
        -0.0029296875,
        -0.05444455146789551,
        -0.009765625,
        0.0003814697265625,
        -0.0007852371782064438,
        0.0625,
        -0.07266771433324948,
        0.00092315673828125,
        -0.02317537716589868,
        0.00030517578125,
        0.00007461127825081348,
        0.0029296875,
        0.05444455146789551,
        -0.007110595703125,
        -0.00537109375,
        -0.0072021484375,
        -2.2798776626586914e-6,
    ];
    let common_values = &[
        (0.0625, 6407),
        (-0.0625, 6393),
        (0.09375, 5870),
        (-0.09375, 5858),
        (0.03125, 5626),
        (-0.03125, 5611),
        (0.078125, 5277),
        (-0.078125, 5260),
        (-0.0, 5098),
        (0.046875, 4988),
        (-0.046875, 4978),
        (0.0, 4891),
        (0.015625, 4680),
        (-0.015625, 4672),
        (0.0234375, 4384),
        (-0.0234375, 4374),
        (0.0078125, 4263),
        (-0.0078125, 4260),
        (0.01171875, 3729),
        (-0.01171875, 3726),
    ];
    let sample_median = (0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(0.000022298181036079024),
        standard_deviation: NiceFloat(0.035440202565071834),
        skewness: NiceFloat(0.00030138197742575316),
        excess_kurtosis: NiceFloat(1.5712669731678002),
    };
    special_random_primitive_float_inclusive_range_helper::<f64>(
        -0.1,
        0.1,
        10,
        1,
        10,
        1,
        1,
        100,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = e, b = π, mean abs of exponent = 3, mean precision = 2, mean zero P = 1/10
    let values = &[
        3.0, 2.859375, 3.0234375, 3.0, 2.75, 3.0, 3.0, 3.0, 2.75, 2.75, 3.078125, 3.0, 3.0, 3.0,
        2.84375, 2.84375, 2.96875, 3.0, 3.125, 2.8125, 3.0, 3.0, 3.03125, 3.0, 3.125, 3.125, 2.875,
        3.0, 2.75, 2.875, 3.0, 2.75, 2.9140625, 2.734375, 3.0, 3.0, 3.125, 2.92578125, 3.125, 3.0,
        2.75, 3.125, 2.9921875, 3.0, 2.75, 3.0, 3.0, 3.01953125, 3.0, 2.8125,
    ];
    let common_values = &[
        (3.0, 333198),
        (2.75, 222142),
        (3.125, 74385),
        (2.875, 73751),
        (2.9375, 33356),
        (2.8125, 33246),
        (3.0625, 32891),
        (3.03125, 9479),
        (2.71875, 9381),
        (2.84375, 9352),
        (2.90625, 9344),
        (2.78125, 9342),
        (2.96875, 9313),
        (3.09375, 9258),
        (2.765625, 3221),
        (2.921875, 3214),
        (3.015625, 3204),
        (3.046875, 3175),
        (2.859375, 3165),
        (2.734375, 3163),
    ];
    let sample_median = (3.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(2.9232526079986494),
        standard_deviation: NiceFloat(0.1262248827883599),
        skewness: NiceFloat(-0.17691898540367232),
        excess_kurtosis: NiceFloat(-1.2742881322265622),
    };
    special_random_primitive_float_inclusive_range_helper::<f64>(
        core::f64::consts::E,
        core::f64::consts::PI,
        3,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = e, b = π, mean abs of exponent = 10, mean precision = 10, mean zero P = 1/100
    let values = &[
        2.923828125,
        2.953125,
        3.0,
        2.8671875,
        2.8125,
        3.125,
        3.015625,
        2.8462657928466797,
        3.140625,
        2.875,
        3.0,
        2.75,
        3.0,
        2.71875,
        2.75,
        3.021484375,
        2.97064208984375,
        3.0179443359375,
        2.9688720703125,
        2.75,
        2.97635555267334,
        2.875,
        2.890625,
        2.8388671875,
        2.781494140625,
        3.0876541137695312,
        3.10650634765625,
        2.81756591796875,
        2.8125,
        2.774658203125,
        2.746337890625,
        3.086890615457378,
        2.75,
        2.7857666015625,
        2.9739707708358765,
        2.7529093623161316,
        2.7723388671875,
        3.125,
        2.94140625,
        2.916015625,
        2.776611328125,
        2.823486328125,
        3.0,
        3.0,
        3.135541538707912,
        2.75,
        3.0625,
        2.734408974647522,
        3.0,
        2.789327871054411,
    ];
    let common_values = &[
        (3.0, 91105),
        (2.75, 83336),
        (2.875, 38014),
        (3.125, 37964),
        (3.0625, 23196),
        (2.9375, 22929),
        (2.8125, 22922),
        (3.09375, 9177),
        (2.78125, 9003),
        (2.84375, 8943),
        (2.90625, 8936),
        (2.71875, 8920),
        (3.03125, 8913),
        (2.96875, 8854),
        (2.765625, 4212),
        (2.859375, 4171),
        (2.953125, 4126),
        (2.890625, 4119),
        (3.015625, 4090),
        (2.984375, 4082),
    ];
    let sample_median = (2.9375, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(2.9261256924208183),
        standard_deviation: NiceFloat(0.12519730680807895),
        skewness: NiceFloat(-0.016892719700380622),
        excess_kurtosis: NiceFloat(-1.2338524787684633),
    };
    special_random_primitive_float_inclusive_range_helper::<f64>(
        core::f64::consts::E,
        core::f64::consts::PI,
        10,
        1,
        10,
        1,
        1,
        100,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = 100.0, b = 101.0, mean abs of exponent = 7, mean precision = 2, mean zero P = 1/10
    let values = &[
        100.0, 100.5625, 100.40625, 100.0, 101.0, 100.0, 100.0, 100.0, 101.0, 101.0, 100.8125,
        100.0, 100.0, 100.0, 100.375, 100.875, 100.875, 100.0, 100.5, 100.75, 100.0, 100.0,
        100.875, 100.0, 100.5, 100.5, 100.5, 100.0, 101.0, 100.5, 100.0, 101.0, 100.46875,
        100.1875, 100.0, 100.0, 100.5, 100.546875, 100.5, 100.0, 101.0, 100.5, 100.96875, 100.0,
        101.0, 100.0, 100.0, 100.296875, 100.0, 100.75,
    ];
    let common_values = &[
        (100.0, 333198),
        (101.0, 222142),
        (100.5, 148136),
        (100.75, 49930),
        (100.25, 49563),
        (100.375, 16456),
        (100.875, 16346),
        (100.625, 16343),
        (100.125, 16324),
        (100.6875, 5638),
        (100.5625, 5588),
        (100.1875, 5508),
        (100.8125, 5478),
        (100.4375, 5449),
        (100.3125, 5420),
        (100.9375, 5417),
        (100.0625, 5309),
        (100.71875, 1910),
        (100.21875, 1891),
        (100.09375, 1867),
    ];
    let sample_median = (100.5, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(100.44459040083368),
        standard_deviation: NiceFloat(0.39739652839770573),
        skewness: NiceFloat(0.20000867844534617),
        excess_kurtosis: NiceFloat(-1.5079346005477923),
    };
    special_random_primitive_float_inclusive_range_helper::<f64>(
        100.0,
        101.0,
        7,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = 100.0, b = 101.0, mean abs of exponent = 10, mean precision = 10, mean zero P =
    // 1/100
    let values = &[
        100.8203125,
        100.5625,
        100.0,
        100.71875,
        100.75,
        100.5,
        100.9375,
        100.88426971435547,
        100.9375,
        100.5,
        100.0,
        101.0,
        100.0,
        100.125,
        101.0,
        100.4609375,
        100.511962890625,
        100.46337890625,
        100.80126953125,
        101.0,
        100.27350997924805,
        100.5,
        100.8125,
        100.37109375,
        100.3251953125,
        100.13284301757812,
        100.032470703125,
        100.916748046875,
        100.75,
        100.0087890625,
        100.2373046875,
        100.59933795747929,
        101.0,
        100.89990234375,
        100.71137285232544,
        100.55885925889015,
        100.15478515625,
        100.5,
        100.421875,
        100.3671875,
        100.8330078125,
        100.9619140625,
        100.0,
        100.0,
        100.72769894078374,
        101.0,
        100.25,
        100.2837815284729,
        100.0,
        100.6954902857542,
    ];
    let common_values = &[
        (100.0, 91387),
        (101.0, 83529),
        (100.5, 76188),
        (100.25, 34644),
        (100.75, 34560),
        (100.375, 15817),
        (100.875, 15798),
        (100.125, 15698),
        (100.625, 15638),
        (100.6875, 7188),
        (100.9375, 7169),
        (100.5625, 7154),
        (100.1875, 7093),
        (100.0625, 7081),
        (100.3125, 7073),
        (100.8125, 7055),
        (100.4375, 6967),
        (100.34375, 3371),
        (100.59375, 3321),
        (100.21875, 3291),
    ];
    let sample_median = (100.5, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(100.49597550384374),
        standard_deviation: NiceFloat(0.3229907369466344),
        skewness: NiceFloat(0.007882298769431025),
        excess_kurtosis: NiceFloat(-1.2073173945999414),
    };
    special_random_primitive_float_inclusive_range_helper::<f64>(
        100.0,
        101.0,
        10,
        1,
        10,
        1,
        1,
        100,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = 1.0e38, b = Infinity, mean abs of exponent = 129, mean precision = 2, mean zero P =
    // 1/10
    let values = &[
        4.0833884030512616e39,
        1.0633823966279327e38,
        1.0633823966279327e38,
        f64::INFINITY,
        1.2760588759535192e38,
        2.722258935367508e39,
        2.1121432853022313e39,
        1.0889035741470031e40,
        3.2667107224410092e40,
        1.0633823966279327e38,
        1.0633823966279327e38,
        1.3823971156163125e38,
        3.828176627860558e38,
        f64::INFINITY,
        1.3026434358692176e38,
        3.402823669209385e38,
        1.361129467683754e39,
        f64::INFINITY,
        2.0416942015256308e39,
        6.80564733841877e38,
        1.5419044751105024e38,
        4.0833884030512616e39,
        2.5521177519070385e38,
        1.1484529883581673e39,
        1.7014118346046923e38,
        1.4887353552791058e38,
        1.361129467683754e39,
        6.380294379767596e38,
        4.253529586511731e38,
        f64::INFINITY,
        4.519375185668714e38,
        3.402823669209385e38,
        f64::INFINITY,
        1.3823971156163125e38,
        3.2667107224410092e40,
        3.48449143727041e41,
        2.9774707105582116e38,
        7.443676776395529e38,
        3.190147189883798e38,
        2.5521177519070385e38,
        1.9055812547572554e40,
        3.4028236692093846e39,
        1.0633823966279327e38,
        6.80564733841877e38,
        1.7014118346046923e38,
        1.2760588759535192e38,
        f64::INFINITY,
        1.4879045877817402e38,
        1.595073594941899e38,
        8.166776806102523e40,
    ];
    let common_values = &[
        (f64::INFINITY, 100224),
        (1.2760588759535192e38, 74870),
        (1.7014118346046923e38, 56023),
        (3.402823669209385e38, 42298),
        (2.5521177519070385e38, 37662),
        (6.80564733841877e38, 31926),
        (5.104235503814077e38, 28139),
        (1.0633823966279327e38, 25142),
        (1.4887353552791058e38, 25038),
        (1.361129467683754e39, 23671),
        (1.0208471007628154e39, 21048),
        (2.722258935367508e39, 17641),
        (2.0416942015256308e39, 15983),
        (5.444517870735016e39, 13496),
        (2.1267647932558654e38, 12408),
        (2.9774707105582116e38, 12336),
        (4.0833884030512616e39, 11938),
        (1.169720636290726e38, 11049),
        (1.595073594941899e38, 10950),
        (1.3823971156163125e38, 10907),
    ];
    let sample_median = (5.84860318145363e38, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(f64::NAN),
        standard_deviation: NiceFloat(f64::NAN),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_inclusive_range_helper::<f64>(
        1.0e38,
        f64::INFINITY,
        129,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = -f64::MIN_POSITIVE_SUBNORMAL, b = f64::MIN_POSITIVE_SUBNORMAL, mean abs of exponent
    // = 1075, mean precision = 2, mean zero P = 1/10
    let values = &[
        5.0e-324, 5.0e-324, 5.0e-324, -0.0, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, 5.0e-324,
        5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, 0.0, -5.0e-324, -5.0e-324, -5.0e-324, -0.0,
        5.0e-324, -5.0e-324, -5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, -5.0e-324, 5.0e-324,
        -5.0e-324, 5.0e-324, 5.0e-324, -0.0, -5.0e-324, -5.0e-324, 0.0, 5.0e-324, -5.0e-324,
        -5.0e-324, -5.0e-324, 5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, 5.0e-324, -5.0e-324,
        5.0e-324, -5.0e-324, 5.0e-324, 0.0, -5.0e-324, 5.0e-324, 5.0e-324,
    ];
    let common_values = &[(5.0e-324, 450531), (-5.0e-324, 449245), (-0.0, 50351), (0.0, 49873)];
    let sample_median = (0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(5.0e-324),
        standard_deviation: NiceFloat(0.0),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_inclusive_range_helper::<f64>(
        -f64::MIN_POSITIVE_SUBNORMAL,
        f64::MIN_POSITIVE_SUBNORMAL,
        1075,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = -0.0, b = f64::MIN_POSITIVE_SUBNORMAL, mean abs of exponent = 1075, mean precision =
    // 2, mean zero P = 1/10
    let values = &[
        5.0e-324, 5.0e-324, 5.0e-324, -0.0, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324,
        5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 0.0, 5.0e-324, 5.0e-324, 5.0e-324, -0.0, 5.0e-324,
        5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324,
        5.0e-324, -0.0, 5.0e-324, 5.0e-324, 0.0, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324,
        5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 0.0,
        5.0e-324, 5.0e-324, 5.0e-324,
    ];
    let common_values = &[(5.0e-324, 899776), (-0.0, 50351), (0.0, 49873)];
    let sample_median = (5.0e-324, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(5.0e-324),
        standard_deviation: NiceFloat(0.0),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_inclusive_range_helper::<f64>(
        -0.0,
        f64::MIN_POSITIVE_SUBNORMAL,
        1075,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = 0.0, b = 0.0, mean abs of exponent = 1075, mean precision = 2, mean zero P = 1/10
    let values = &[0.0; 50];
    let common_values = &[(0.0, 1000000)];
    let sample_median = (0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(0.0),
        standard_deviation: NiceFloat(0.0),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_inclusive_range_helper::<f64>(
        0.0,
        0.0,
        1075,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // -f64::MIN_POSITIVE_SUBNORMAL, b = -f64::MIN_POSITIVE_SUBNORMAL, mean abs of exponent = 1075,
    // mean precision = 2, mean zero P = 1/10
    let values = &[-5.0e-324; 50];
    let common_values = &[(-5.0e-324, 1000000)];
    let sample_median = (-5.0e-324, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(-5.0e-324),
        standard_deviation: NiceFloat(0.0),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_inclusive_range_helper::<f64>(
        -f64::MIN_POSITIVE_SUBNORMAL,
        -f64::MIN_POSITIVE_SUBNORMAL,
        1075,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = -f64::MIN_POSITIVE_SUBNORMAL, b = 0.0, mean abs of exponent = 1075, mean precision =
    // 2, mean zero P = 1/10
    let values = &[
        -5.0e-324, -5.0e-324, -5.0e-324, -0.0, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324,
        -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, 0.0, -5.0e-324, -5.0e-324,
        -5.0e-324, -0.0, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324,
        -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -0.0, -5.0e-324, -5.0e-324, 0.0,
        -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324,
        -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, 0.0, -5.0e-324, -5.0e-324,
        -5.0e-324,
    ];
    let common_values = &[(-5.0e-324, 899776), (-0.0, 50351), (0.0, 49873)];
    let sample_median = (-5.0e-324, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(-5.0e-324),
        standard_deviation: NiceFloat(0.0),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_inclusive_range_helper::<f64>(
        -f64::MIN_POSITIVE_SUBNORMAL,
        0.0,
        1075,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = -Infinity, b = Infinity, mean abs of exponent = 1, mean precision = 2, mean zero P =
    // 1/10
    let values = &[
        12.0,
        1.5,
        0.75,
        -0.0,
        -12.0,
        -1.5,
        -0.75,
        -4.0,
        4.0,
        2.0,
        -2.0,
        -1.2041015625,
        -2.0,
        f64::NEGATIVE_INFINITY,
        -0.75,
        -0.75,
        -3.0,
        0.0,
        1.2041015625,
        -40.0,
        -0.6875,
        2.0,
        0.75,
        0.75,
        -1.3125,
        3.0,
        -2.0,
        40.0,
        0.6875,
        -0.0,
        -0.5,
        -1.5,
        f64::NEGATIVE_INFINITY,
        1.3125,
        -0.125,
        -1.125,
        -0.046875,
        2.0,
        -1.5,
        -1.9375,
        -1.0,
        0.5,
        -0.75,
        1.5,
        -1.0,
        0.125,
        f64::INFINITY,
        -1.875,
        1.125,
        0.046875,
    ];
    let common_values = &[
        (1.0, 49883),
        (-1.0, 49740),
        (1.5, 33699),
        (-1.5, 33608),
        (f64::NEGATIVE_INFINITY, 25322),
        (0.5, 25188),
        (-0.0, 25144),
        (-0.5, 25120),
        (f64::INFINITY, 24924),
        (0.0, 24834),
        (2.0, 24765),
        (-2.0, 24711),
        (3.0, 16988),
        (-3.0, 16939),
        (0.75, 16654),
        (-0.75, 16610),
        (4.0, 12510),
        (0.25, 12465),
        (-4.0, 12461),
        (-0.25, 12430),
    ];
    let sample_median = (0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(f64::NAN),
        standard_deviation: NiceFloat(f64::NAN),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_inclusive_range_helper::<f64>(
        f64::NEGATIVE_INFINITY,
        f64::INFINITY,
        1,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = -0.0, b = -0.0, mean abs of exponent = 1, mean precision = 2, mean zero P = 1/10
    let values = &[-0.0; 50];
    let common_values = &[(-0.0, 1000000)];
    let sample_median = (-0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(0.0),
        standard_deviation: NiceFloat(0.0),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_inclusive_range_helper::<f64>(
        -0.0,
        -0.0,
        1,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );
}

fn special_random_primitive_float_inclusive_range_fail_helper<T: PrimitiveFloat>() {
    assert_panic!(special_random_primitive_float_inclusive_range::<T>(
        EXAMPLE_SEED,
        T::ONE,
        T::ZERO,
        1,
        1,
        2,
        1,
        1,
        10
    ));
    assert_panic!(special_random_primitive_float_inclusive_range::<T>(
        EXAMPLE_SEED,
        T::ONE,
        T::NAN,
        1,
        1,
        2,
        1,
        1,
        10
    ));
    assert_panic!(special_random_primitive_float_inclusive_range::<T>(
        EXAMPLE_SEED,
        T::ONE,
        T::TWO,
        0,
        1,
        2,
        1,
        1,
        10
    ));
    assert_panic!(special_random_primitive_float_inclusive_range::<T>(
        EXAMPLE_SEED,
        T::ONE,
        T::TWO,
        1,
        0,
        2,
        1,
        1,
        10
    ));
    assert_panic!(special_random_primitive_float_inclusive_range::<T>(
        EXAMPLE_SEED,
        T::from(100.0),
        T::from(101.0),
        6,
        1,
        2,
        1,
        1,
        10
    ));
    assert_panic!(special_random_primitive_float_inclusive_range::<T>(
        EXAMPLE_SEED,
        T::ONE,
        T::TWO,
        1,
        1,
        1,
        2,
        1,
        10
    ));
    assert_panic!(special_random_primitive_float_inclusive_range::<T>(
        EXAMPLE_SEED,
        T::ONE,
        T::TWO,
        1,
        1,
        1,
        0,
        1,
        10
    ));
    assert_panic!(special_random_primitive_float_inclusive_range::<T>(
        EXAMPLE_SEED,
        T::ONE,
        T::TWO,
        1,
        1,
        2,
        1,
        2,
        1
    ));
    assert_panic!(special_random_primitive_float_inclusive_range::<T>(
        EXAMPLE_SEED,
        T::ONE,
        T::TWO,
        1,
        1,
        2,
        1,
        1,
        0
    ));
}

#[test]
fn special_random_primitive_float_inclusive_range_fail() {
    apply_fn_to_primitive_floats!(special_random_primitive_float_inclusive_range_fail_helper);
}
